#!/usr/bin/env python

#
# Author: Jesus Galaz-Montoya 03/2011, 
# (based on Steven Ludtke's initial implementation [02/15/2011] of Jesus's older scripts, from M.F.Schmid's methods).
# Last modification: July/08/2015
#
# Copyright (c) 2011 Baylor College of Medicine
#
# This software is issued under a joint BSD/GNU license. You may use the
# source code in this file under either license. However, note that the
# complete EMAN2 and SPARX software packages have some GPL dependencies,
# so you are responsible for compliance with the licenses of these packages
# if you opt to use BSD licensing. The warranty disclaimer below holds
# in either instance.
#
# This complete copyright notice must be included in any revised version of the
# source code. Additional authorship citations may be added, but existing
# author citations must be preserved.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  2111-1307 USA
#
#

from EMAN2 import *
import math
import numpy
from copy import deepcopy
import os
import sys
import random
from random import choice
from pprint import pprint
from EMAN2jsondb import JSTask,jsonclasses
import datetime
import gc 	#this will be used to free-up unused memory

#from e2spt_hac import textwriter




def main():
	progname = os.path.basename(sys.argv[0])
	usage = """prog <output> [options]

	This program produces iterative class-averages akin to those generated by e2classaverage, 
	but for stacks of 3-D Volumes.
	Normal usage is to provide a stack of particle volumes and a classification matrix file 
	(if you have more than one class) defining class membership. 
	Members of each class are then iteratively aligned to each other (within the class) and 
	averaged together. 
	It is also possible to use this program on all of the volumes in a single stack without
	providing a classification matrix.

	Specify preprocessing options through --lowpass, --highpass, --mask, --normproc, --thresh, 
	--preprocess and --shrink. These take EMAN2 processors (to see a list, type e2help.py processors at
	the command line).
	
	Notice that the alignment is broken down into two step: 1) Coarse alignment and 2) Fine 
	alignment. This is done for speed optimization. By default, the particles are preprocessed
	THE SAME was for Coarse and Fine alignment, unless you supply --notprocfinelinecoarse.
	In this case, the particles will be preprocessed with default parameters for fine alignment.
	To specify or inactivate any preprocessing for fine alignment, do so through fine 
	alignment parameters:
	--lowpassfine, --highpassfine, --preprocessfine and --shrinkfine.
	
	"""
			
	parser = EMArgumentParser(usage=usage,version=EMANVERSION)
	
	parser.add_header(name="caheader", help="""Options below this label are specific to sptclassaverage""", title="### sptclassaverage options ###", row=3, col=0, rowspan=1, colspan=3, mode='alignment,breaksym')
	
	
	'''
	REQUIRED PARAMETERS
	'''
	parser.add_argument("--input", type=str, default='',help="""Default=None. The name of the input volume stack. MUST be HDF since volume stack support is required.""", guitype='filebox', browser='EMSubTomosTable(withmodal=True,multiselect=False)', row=0, col=0, rowspan=1, colspan=3, mode='alignment,breaksym')

	
	'''
	STANDARD PARAMETERS
	'''
	parser.add_argument("--apix",type=float,default=0.0,help="""Default=0.0 (not used). Use this apix value where relevant instead of whatever is in the header of the reference and the particles.""")

	parser.add_argument("--align",type=str,default="rotate_translate_3d_tree",help="""Default is rotate_translate_3d_tree. See e2help.py aligners to see the list of parameters the aligner takes (for example, if there's symmetry, supply --align rotate_translate_3d_tree:sym=icos). This is the aligner used to align particles to the previous class average. Specify 'None' (with capital N) to disable.""", returnNone=True,guitype='comboparambox', choicelist='re_filter_list(dump_aligners_list(),\'3d\')', row=12, col=0, rowspan=1, colspan=3, nosharedb=True, mode="alignment,breaksym['rotate_symmetry_3d']")
	
	parser.add_argument("--aligncmp",type=str,default="ccc.tomo.thresh",help="""Default=ccc.tomo.thresh. The comparator used for the --align aligner. Do not specify unless you need to use another specific aligner.""",guitype='comboparambox',choicelist='re_filter_list(dump_cmps_list(),\'tomo\')', row=13, col=0, rowspan=1, colspan=3,mode="alignment,breaksym")
	
	parser.add_argument("--clip",type=int,default=0,help="""Default=0 (which means it's not used). Boxsize to clip particles as part of preprocessing to speed up alignment. For example, the boxsize of the particles might be 100 pixels, but the particles are only 50 pixels in diameter. Aliasing effects are not always as deleterious for all specimens, and sometimes 2x padding isn't necessary; still, there are some benefits from 'oversampling' the data during averaging; so you might still want an average of size 2x, but perhaps particles in a box of 1.5x are sufficiently good for alignment. In this case, you would supply --clip=75""")

	parser.add_argument("--iter", type=int, default=1, help="""Default=1. The number of iterations to perform.""", guitype='intbox', row=5, col=0, rowspan=1, colspan=1, nosharedb=True, mode='alignment,breaksym')
	
	parser.add_argument("--path",type=str,default='spt',help="""Default=spt. Directory to store results in. The default is a numbered series of directories containing the prefix 'spt'; for example, spt_02 will be the directory by default if 'spt_01' already exists.""")
		
	parser.add_argument("--npeakstorefine", type=int, help="""Default=1. The number of best coarse alignments to refine in search of the best final alignment.""", default=1, guitype='intbox', row=9, col=0, rowspan=1, colspan=1, nosharedb=True, mode='alignment,breaksym[1]')
			
	parser.add_argument("--preavgproc1",type=str,default='',help="""Default=None. A processor (see 'e2help.py processors -v 10' at the command line) to be applied to the raw particle after alignment but before averaging (for example, a threshold to exclude extreme values, or a highphass filter if you have phaseplate data.)""")
	
	parser.add_argument("--preavgproc2",type=str,default='',help="""Default=None. A processor (see 'e2help.py processors -v 10' at the command line) to be applied to the raw particle after alignment but before averaging (for example, a threshold to exclude extreme values, or a highphass filter if you have phaseplate data.)""")

	parser.add_argument("--parallel",default="thread:1",help="""default=thread:1. Parallelism. See http://blake.bcm.edu/emanwiki/EMAN2/Parallel""", guitype='strbox', row=19, col=0, rowspan=1, colspan=3, mode='alignment,breaksym')
	
	parser.add_argument("--ppid", type=int, help="""Default=-1. Set the PID of the parent process, used for cross platform PPID""",default=-1)
			
	parser.add_argument("--resume",type=str,default='',help="""(Not working currently). sptali_ir.json file that contains alignment information for the particles in the set. If the information is incomplete (i.e., there are less elements in the file than particles in the stack), on the first iteration the program will complete the file by working ONLY on particle indexes that are missing. For subsequent iterations, all the particles will be used.""")
															
	parser.add_argument("--plots", action='store_true', default=False,help="""Default=False. Turn this option on to generate a plot of the ccc scores during each iteration in.png format (otherwise only .txt files will be saved). This option will also produce a plot of mean ccc score across iterations. Running on a cluster or via ssh remotely might not support plotting.""")

	parser.add_argument("--savepreproc",action="store_true",  default=False,help="""Default=False. Will save stacks of preprocessed particles (one for coarse alignment and one for fine alignment if preprocessing options are different).""")
	
	parser.add_argument("--subset",type=int,default=0,help="""Default=0 (not used). Refine only this substet of particles from the stack provided through --input""")
	
	parser.add_argument("--savesteps",action="store_true", default=False, help="""Default=False. If set, this will save the average after each iteration to class_#.hdf. Each class in a separate file. Appends to existing files.""", guitype='boolbox', row=4, col=0, rowspan=1, colspan=1, mode='alignment,breaksym')
	
	parser.add_argument("--saveali",action="store_true", default=False, help="""Default=False. If set, this will save the aligned particle volumes in class_ptcl.hdf. Overwrites existing file.""", guitype='boolbox', row=4, col=1, rowspan=1, colspan=1, mode='alignment,breaksym')
	
	parser.add_argument("--saveallalign",action="store_true", default=False, help="""Default=False. If set, this will save an aligned stack of particles for each iteration""", guitype='boolbox', row=4, col=2, rowspan=1, colspan=1, mode='alignment,breaksym')
	
	parser.add_argument("--saveallpeaks",action="store_true", default=False, help="""Default=False. If set, this will save the alignment information and score for all examined peaks --npeakstorefine during coarse alignment.""")
	
	parser.add_argument("--sym", type=str,dest = "sym", default='', help = """Default=None (equivalent to c1). Symmetry to impose -choices are: c<n>, d<n>, h<n>, tet, oct, icos""", guitype='symbox', row=9, col=1, rowspan=1, colspan=2, mode='alignment,breaksym')
	
	parser.add_argument("--verbose", "-v", dest="verbose", action="store", metavar="n", type=int, default=0, help="""Default=0. Verbose level [0-9], higner number means higher level of verboseness""")

	parser.add_argument("--weighbytiltaxis",type=str,default='',help="""Default=None. A,B, where A is an integer number and B a decimal. A represents the location of the tilt axis in the tomogram in pixels (eg.g, for a 4096x4096xZ tomogram, this value should be 2048), and B is the weight of the particles furthest from the tiltaxis. For example, --weighbytiltaxis=2048,0.5 means that praticles at the tilt axis (with an x coordinate of 2048) will have a weight of 1.0 during averaging, while the distance in the x coordinates of particles not-on the tilt axis will be used to weigh their contribution to the average, with particles at the edge(0+radius or 4096-radius) weighing 0.5, as specified by the value provided for B.""")

	parser.add_argument("--weighbyscore",action='store_true',default=False,help="""Default=False. This option will weigh the contribution of each subtomogram to the average by score/bestscore.""")

	
	'''
	CLASSAVERAGE SPECIFIC PARAMETERS
	'''
	parser.add_argument("--goldstandardoff", action="store_true", default=False, help="""Default=False. This will PREVENT splitting the dataset provided through --input into two groups, and the entire dataset will be refined together. If this parameter is NOT supplied (and thus the refinement is 'gold standard') and --ref is supplied, two copies of the reference will be generated and randomphase-lowpass filtered to the resolution specified through --refrandphase.""")

	parser.add_argument("--classmx", type=str, default='', help="""Default=None. The name of the classification matrix specifying how particles in 'input' should be grouped. If omitted, all particles will be averaged.""")
	
	parser.add_argument("--recompute",action='store_true',default=False,help="""default=False. This parameter requires --classmx and will recompute averages (for example, even and odd) based on the classmx file and the alignment parameters specified therein for each particle. No refinements will follow. This is exclusively for recomputing averages.""")
	
	parser.add_argument("--donotaverage",action="store_true", default=False,help="""Default=False. If e2spt_refinemulti.py is calling e2spt_classaverage.py, the latter need not average any particles, but rather only yield the alignment results.""")
	
	parser.add_argument("--ref", type=str, default='', help="""Default=None. Reference image. Used as an initial alignment reference. The refinements are 'gold standard' by default, and therefore two independent copies of the reference will be generated and randomphase-lowpass filtered to the resolution specified through --refrandphase. To turn dataset splitting and gold standard refinement off, supply --goldstandardoff.""", guitype='filebox', browser='EMBrowserWidget(withmodal=True,multiselect=True)', filecheck=False, row=1, col=0, rowspan=1, colspan=3, mode='alignment')
	
	parser.add_argument("--refpreprocess",action="store_true",default=False,help="""Default=False. This will preprocess the reference identically to the particles. It is off by default, but it is internally turned on when no reference is supplied. It should probably be off when using a crystal structure (with all positive densities) turned to EM density as an initial model, but it should be on when using an EM map.""")
	
	parser.add_argument("--refrandphase", type=float, default=0, help="""Default=0 (off). Resolution to phase randomize the reference to (or the two copies of the reference if --goldstandardoff is NOT supplied [gold standard refinement is on by default].""")
	
	parser.add_argument("--hacref",type=int,default=0,help="""Default=0 (not used by default). Size of the SUBSET of particles to use to build an initial reference by calling e2spt_hac.py which does Hierarchical Ascendant Classification (HAC) or 'all vs all' alignments.""") 
		
	parser.add_argument("--ssaref",type=int,default=0,help="""Default=0 (not used by default). Size of the SUBSET of particles to use to build an initial reference by calling e2symsearch3d.py, which does self-symmetry alignments. You must provide --sym different than c1 for this to make any sense.""")
		
	parser.add_argument("--btref",type=int,default=0,help="""Default=0 (internally turned on and set to 64). Size of the SUBSET of particles to use to build an initial reference by calling e2spt_binarytree.py. By default, the largest power of two smaller than the number of particles in --input will be used. For example, if you supply a stack with 150 subtomograms, the program will automatically select 128 as the limit to use because it's the largest power of 2 that is smaller than 150. But if you provide, say --btref=100, then the number of particles used will be 64, because it's the largest power of 2 that is still smaller than 100.""")
	
	parser.add_argument("--resultmx",type=str,default=None,help="""Default=None. Specify an output image to store the result matrix. This is in the same format as the classification matrix. http://blake.bcm.edu/emanwiki/EMAN2/ClassmxFiles""")
	
	parser.add_argument("--refinemultireftag", type=str, default='', help="""Default=''. DO NOT USE THIS PARAMETER. It is passed on from e2spt_refinemulti.py if needed.""")

	parser.add_argument("--keep",type=float,default=1.0,help="""Default=1.0 (all particles kept). The fraction of particles to keep in each class.""", guitype='floatbox', row=6, col=0, rowspan=1, colspan=1, mode='alignment,breaksym')
	
	parser.add_argument("--keepsig", action="store_true", default=False,help="""Default=False. Causes the keep argument to be interpreted in standard deviations.""", guitype='boolbox', row=6, col=1, rowspan=1, colspan=1, mode='alignment,breaksym')

	
	'''
	====================
	ADVANCED PARAMETERS
	====================
	'''
	
	parser.add_argument("--averager",type=str,default="mean.tomo",help="""Default=mean.tomo. The type of averager used to produce the class average. Default=mean.tomo.""")

	
	'''
	PRE-FFT processing parameters
	'''
	
	#parser.add_argument("--nopreprocprefft",action="store_true",default=False,help="""Turns off all preprocessing that happens only once before alignment (--normproc, --mask, --maskfile, --clip, --threshold; i.e., all preprocessing excepting filters --highpass, --lowpass, --preprocess, and --shrink.""")

	parser.add_argument("--shrink", type=int,default=1,help="""Default=1 (no shrinking). Optionally shrink the input volumes by an integer amount for coarse alignment.""", guitype='shrinkbox', row=5, col=1, rowspan=1, colspan=1, mode='alignment,breaksym')
	
	parser.add_argument("--shrinkfine", type=int,default=1,help="""Default=1 (no shrinking). Optionally shrink the input volumes by an integer amount for refine alignment.""", guitype='intbox', row=5, col=2, rowspan=1, colspan=1, mode='alignment')
	
	parser.add_argument("--threshold",type=str,default='',help="""Default=None. A threshold applied to the subvolumes after normalization. For example, --threshold=threshold.belowtozero:minval=0 makes all negative pixels equal 0, so that they do not contribute to the correlation score.""", guitype='comboparambox', choicelist='re_filter_list(dump_processors_list(),\'filter\')', row=10, col=0, rowspan=1, colspan=3, mode='alignment,breaksym')
	
	parser.add_argument("--mask",type=str,default='', help="""Default=None. Masking processor applied to particles before alignment. IF using --clip, make sure to express outer mask radii as negative pixels from the edge.""", returnNone=True, guitype='comboparambox', choicelist='re_filter_list(dump_processors_list(),\'mask\')', row=11, col=0, rowspan=1, colspan=3, mode='alignment,breaksym')
	
	parser.add_argument("--maskfile",type=str,default='',help="""Default=None. Mask file (3D IMAGE) applied to particles before alignment. Must be in HDF format. Default is None.""")
	
	parser.add_argument("--normproc",type=str, default='',help="""Default=None (see 'e2help.py processors -v 10' at the command line). Normalization processor applied to particles before alignment. If normalize.mask is used, results of the mask option will be passed in automatically. If you want to turn this option off specify \'None\'""")
	
	
	'''
	POST-FFT filtering parameters
	'''
	parser.add_argument("--preprocess",type=str,default='',help="""Any processor (see 'e2help.py processors -v 10' at the command line) to be applied to each volume prior to COARSE alignment. Not applied to aligned particles before averaging.""", guitype='comboparambox', choicelist='re_filter_list(dump_processors_list(),\'filter\')', row=10, col=0, rowspan=1, colspan=3, mode='alignment,breaksym')
	
	parser.add_argument("--preprocessfine",type=str,default='',help="""Any processor (see 'e2help.py processors -v 10' at the command line) to be applied to each volume prior to FINE alignment. Not applied to aligned particles before averaging.""")
	
	parser.add_argument("--lowpass",type=str,default='',help="""Default=None. A lowpass filtering processor (see 'e2help.py processors -v 10' at the command line) to be applied to each volume prior to COARSE alignment. Not applied to aligned particles before averaging.""", guitype='comboparambox', choicelist='re_filter_list(dump_processors_list(),\'filter\')', row=17, col=0, rowspan=1, colspan=3, mode='alignment,breaksym')
	
	parser.add_argument("--lowpassfine",type=str,default='',help="""Default=None. A lowpass filtering processor (see 'e2help.py processors -v 10' at the command line) to be applied to each volume prior to FINE alignment. Not applied to aligned particles before averaging.""")

	parser.add_argument("--highpass",type=str,default='',help="""Default=None. A highpass filtering processor (see 'e2help.py processors -v 10' at the command line) to be applied to each volume prior to COARSE alignment. Not applied to aligned particles before averaging.""", guitype='comboparambox', choicelist='re_filter_list(dump_processors_list(),\'filter\')', row=18, col=0, rowspan=1, colspan=3, mode='alignment,breaksym')
	
	parser.add_argument("--highpassfine",type=str,default='',help="""Default=None. A highpass filtering processor (see 'e2help.py processors -v 10' at the command line) to be applied to each volume prior to FINE alignment. Not applied to aligned particles before averaging.""")

	parser.add_argument("--matchimgs",action='store_true',default=False,help="""Default=False. Applies filter.matchto to one image so that it matches the other's spectral profile during preprocessing for pair-wise alignment purposes.""")
	
	parser.add_argument("--filterbyfsc",action='store_true',default=False,help="""Default=False. If on, this parameter will use dynamic FSC filtering. --lowpass will be used to build initial references if no --ref supplied, then, the FSC between the even and odd initial references will be used to filter the data during preprocessing. If --ref is supplied, --lowpass will be used during the first iteration to align the particles against the reference. Thereafter, the FSC between the most current particle average and the original reference (--ref) will be used in the next iteration.""")

	
	'''
	OTHER advanced parameters
	'''
	parser.add_argument("--radius", type=float, default=0, help="""Default=0 (which means it's not used by default). Hydrodynamic radius of the particle in Angstroms. This will be used to automatically calculate the angular steps to use in search of the best alignment. Make sure the apix is correct on the particles' headers, sine the radius will be converted from Angstroms to pixels. Then, the fine angular step is equal to 360/(2*pi*radius), and the coarse angular step 4 times that.""")
	
	parser.add_argument("--precision",type=float,default=1.0,help="""Default=1.0. Precision in pixels to use when figuring out alignment parameters automatically using --radius. Precision would be the number of pixels that the the edge of the specimen is moved (rotationally) during the finest sampling, --falign. If precision is 1, then the precision of alignment will be that of the sampling (apix of your images) times the --shrinkfine factor specified.""")
	
	parser.add_argument("--search", type=int,default=8,help=""""Default=8. During COARSE alignment, translational search in X, Y and Z, in pixels. This WILL overwrite any search: provided through --align, EXCEPT if you provide --search=8, which is the default. In general, just avoid providing search twice (through here and through the aligner, --align). If you do, just be careful to make them consistent to minimize misinterpretation and error.""")
	
	#parser.add_argument("--searchx", type=int,default=8,help=""""Default=0. Not used. During COARSE alignment translational search in X, Y and Z, in pixels. Default=8. This WILL overwrite any search: provided through --align, EXCEPT if you provide --search=8, which is the default. In general, just avoid providing search twice (through here and through the aligner, --align). If you do, just be careful to make them consistent to minimize misinterpretation and error.""")

	#parser.add_argument("--searchy", type=int,default=8,help=""""Default=8. During COARSE alignment translational search in X, Y and Z, in pixels. Default=8. This WILL overwrite any search: provided through --align, EXCEPT if you provide --search=8, which is the default. In general, just avoid providing search twice (through here and through the aligner, --align). If you do, just be careful to make them consistent to minimize misinterpretation and error.""")

	#parser.add_argument("--searchz", type=int,default=8,help=""""Default=8. During COARSE alignment translational search in X, Y and Z, in pixels. Default=8. This WILL overwrite any search: provided through --align, EXCEPT if you provide --search=8, which is the default. In general, just avoid providing search twice (through here and through the aligner, --align). If you do, just be careful to make them consistent to minimize misinterpretation and error.""")
	
	parser.add_argument("--searchfine", type=int,default=2,help=""""Default=2. During FINE alignment translational search in X, Y and Z, in pixels. Default=2. This WILL overwrite any search: provided through --falign, EXCEPT if you provide --searchfine=2, which is the default. In general, just avoid providing search twice (through here and through the fine aligner --falign). If you do, just be careful to make them consistent to minimize misinterpretation and error.""")
		
	
	parser.add_argument("--postprocess",type=str,default='',help="""Default=None. A processor to be applied to the FINAL volume after averaging the raw volumes in their FINAL orientations, after all iterations are done.""",guitype='comboparambox', choicelist='re_filter_list(dump_processors_list(),\'filter\')', row=16, col=0, rowspan=1, colspan=3, mode='alignment,breaksym')
	
	parser.add_argument("--procfinelikecoarse",action='store_true',default=False,help="""If you supply this parameters, particles for fine alignment will be preprocessed identically to particles for coarse alignment by default. If you supply this, but want specific parameters for preprocessing particles for also supply: fine alignment, nd supply fine alignment parameters, such as --lowpassfine, --highpassfine, etc; to preprocess the particles for FINE alignment differently than for COARSE alignment.""")
	
	
	parser.add_argument("--falign",type=str,default='',help="""Default=None. This is the second stage aligner when the default aligner (rotate_translate_3d_tree) is NOT used. The fine alignment aligner is used to fine-tune the first alignment.""", returnNone=True, guitype='comboparambox', choicelist='re_filter_list(dump_aligners_list(),\'refine.*3d\')', row=14, col=0, rowspan=1, colspan=3, nosharedb=True, mode='alignment,breaksym[None]')
			
	parser.add_argument("--faligncmp",type=str,default='',help="""Default=None. The default will depend on the aligner used (ccc.tomo or ccc.tomo.thresh). The comparator used by the second stage aligner.""", guitype='comboparambox', choicelist='re_filter_list(dump_cmps_list(),\'tomo\')', row=15, col=0, rowspan=1, colspan=3,mode="alignment,breaksym")		

	parser.add_argument("--translateonly",action='store_true',default=False,help="""Default=False. This will force the aligner to not do any rotations and thus serves for translational centering. Specify search values through --search, otherwise its default value will be used.""")	
		
	
	
	parser.add_argument("--inputaliparams",type=str,default="",help="""Default=None. .json file containing a dict of transforms to apply to 'pre-align' the particles.""", guitype='dirbox', dirbasename='spt_|sptsym_', row=7, col=0,rowspan=1, colspan=2, nosharedb=True, mode='breaksym')
	
	parser.add_argument("--breaksym",action="store_true", default=False,help="""Default=False. Break symmetry. Do not apply symmetrization after averaging, even if searching the asymmetric unit provided through --sym only for alignment. Default=False""", guitype='boolbox', row=7, col=2, rowspan=1, colspan=1, nosharedb=True, mode=',breaksym[True]')
	
	#parser.add_argument("--groups",type=int,default=0,help="""Default=0 (not used; data not split). This parameter will split the data into a user defined number of groups. For purposes of gold-standard FSC computation later, select --group=2.""")
		
	parser.add_argument("--randomizewedge",action="store_true",  default=False,help="""Default=False. This parameter is EXPERIMENTAL. It randomizes the position of the particles BEFORE alignment, to minimize missing wedge bias and artifacts during symmetric alignment where only a fraction of space is scanned""")
	
	
	parser.add_argument("--autocenter",type=str, default='',help="""WARNING: Experimental. Default=None. Autocenters each averaged pair during initial average generation with --btref and --hacref. Will also autocenter the average of all particles after each iteration of iterative refinement. Options are --autocenter=xform.centerofmass (self descriptive), or --autocenter=xform.centeracf, which applies auto-convolution on the average.""")
	
	parser.add_argument("--autocentermask",type=str, default='',help="""WARNING: Experimental. Requires --autocenter. Default=None. Masking processor to apply before autocentering. See 'e2help.py processors -v 10' at the command line.""")
	
	parser.add_argument("--autocenterpreprocess",action='store_true', default=False,help="""WARNING: Experimental. Requires --autocenter. Default=False. This will apply a highpass filter at a frequency of half the box size times the apix, shrink by 2, and apply a low pass filter at half nyquist frequency to any computed average for autocentering purposes if --autocenter is provided. Default=False.""")
	
	
	
	
	#parser.add_argument("--automask",action="store_true",help="""Applies a 3-D automask before centering. Can help with negative stain data, and other cases where centering is poor.""")
	
	#parser.add_argument("--resample",action="store_true",help="""If set, will perform bootstrap resampling on the particle data for use in making variance maps.""",default=False)

	
	parser.add_argument("--tweak",action='store_true',default=False,help="""WARNING: BUGGY. This will perform a final alignment with no downsampling [without using --shrink or --shrinkfine] if --shrinkfine > 1.""")
	
	(options, args) = parser.parse_args()
	
	
	
	if options.shrink < options.shrinkfine:
		options.shrink = options.shrinkfine
		print "\n(e2spt_classaverage)(main) WARNING: It makes no sense for shrinkfine to be larger than shrink; therefore, shrink will be made to match shrinkfine"
		
	checksaneimagesize( options, options.input )

	if options.ref:
		checksaneimagesize( options, options.input, options.ref )

	print "Initially, options.goldstandardoff is", options.goldstandardoff, type(options.goldstandardoff)
	
	'''
	Get rootpath to provide absoulute paths to files. 
	Make the directory where to create the database where the results will be stored, 
	if --resume is not provided.
	'''
	
	options.raw = options.input
	
	if options.recompute:
		options.align = 'rotate_translate_3d_tree'
	
	rootpath = os.getcwd()

	if not options.resume:
		options = sptmakepath(options,'spt')	
	else:
		if rootpath not in options.resume:
			options.resume = rootpath + '/' + options.resume
	
		if not options.path:
			print """\nERROR: If you provide --resume, you need to specify which working 
			directory needs to be resumed. Provide it through --path"""			
			sys.exit()

	if rootpath not in options.path:
		options.path = rootpath + '/' + options.path
		
	#abspath= rootpath + '/' + options.path
	#print "\nThus the abs path could be", abspath
	
	nptcl = EMUtil.get_image_count(options.input)
	
	if nptcl < 2:
		options.goldstandardoff = True
	
	if not options.input:
		parser.print_help()
		exit(0)
	elif options.subset:
		#print "there is subset!", options.subset
		
		if options.subset < nptcl:
			#print "smaller than the number of particles", nptcl
			
			if options.goldstandardoff:
				print "goldstandard is off"
				if options.subset < 2:
					if not options.ref:
						print "ERROR: You need at least 2 particles in --input to buidl a reference if --ref is not provided."""
						sys.exit()	
			else:
				print "goldstandard is on"
				if options.subset < 4:
					print "ERROR: You need at least 4 particles in --input for goldstandard refinement if --ref is not provided and --goldstandardoff not provided."""
					sys.exit()
				
				if options.hacref and options.subset < options.hacref * 2:
					print """WARNING: --subset=%d wasn't large enough to accommodate gold standard
					refinement with two independent halves using the specified number of particles
					for initial model generation --hacref=%d. Therefore, --hacref will be reset
					to --subset/2.""" %( options.subset, options.hacref )				
					options.hacref = options.subset / 2			
			
				elif options.ssaref and options.subset < options.ssaref * 2:		
					print """WARNING: --subset=%d wasn't large enough to accommodate gold standard
					refinement with two independent halves using the specified number of particles
					for initial model generation --ssaref=%d. Therefore, --ssaref will be reset
					to --subset/2.""" %( options.subset, options.ssaref )
					options.ssaref = options.subset / 2	
			
				elif options.btref and options.subset < options.btref * 2:			
					print """WARNING: --subset=%d wasn't large enough to accommodate gold standard
					refinement with two independent halves using the specified number of particles
					for initial model generation --btref=%d. Therefore, --btref has been reset
					to --subset/2.""" %( options.subset, options.btref )
					options.btref = options.subset / 2
					
			subsetStack = options.path + '/subset' + str( options.subset ).zfill( len( str( options.subset))) + '.hdf' 
			print "\nSubset to be written to", subsetStack
		
			subsetcmd = 'e2proc3d.py ' + options.input + ' ' + subsetStack + ' --first=0 --last=' + str(options.subset-1) 
			print "Subset cmd is", subsetcmd
		
			runcmd( options, subsetcmd )
			#p=subprocess.Popen( subsetcmd, shell=True,stdout=subprocess.PIPE, stderr=subprocess.PIPE )
			#text=p.communicate()	
			#p.stdout.close()
		
			options.input = subsetStack
			nptcl = EMUtil.get_image_count(options.input)
			print "\nnptcl for subset is now", nptcl
			
		else:
			print """\n(e2spt_classaverage)(main) WARNING: --subset was larger or equal to the number of particles in --input. 
			Therefore, using all particles."""
	else:
		if not options.ref:
			if not options.goldstandardoff:
				if nptcl < 4:
					print "ERROR: You need at least 4 particles in --input for goldstandard refinement if --ref is not provided and --goldstandardoff not provided."""
					sys.exit()
				else:
					pass
			else:
				if nptcl < 2:
					print "ERROR: You need at least 2 particles in --input to build a reference (or to --recompute and average) if --ref is not provided."""
					sys.exit()
		#else:
		#	print "ref is", options.ref
	
	
	options.raw = options.input

					
	if not options.translateonly:
		if options.search or options.searchfine:
			options = sptParseAligner( options )
		
			print "aligner parsed", sptParseAligner
	else:
		options.align = 'rotate_translate_3d_grid:phi0=0:phi1=1:alt0=0:alt1=1:az0=0:az1=1:dphi=2:daz=2:dalt=2'
		
		if options.search:
			options.align += ':search=' + str(options.search)
			
		#if options.searchx:
		#	options.align += ':searchx=' + str(options.searchx)
		
		#if options.searchy:
		#	options.align += ':searchy=' + str(options.searchy)
		
		#if options.searchz:
		#	options.align += ':searchz=' + str(options.searchz)
	
	
	
	print "after fixing searches but before calcali options.falign is", options.falign
	if options.radius and float(options.radius) > 0.0:
		#print "(e2spt_classaverage)(main) before calling calcAliStep, options.input is", options.input
		options = calcAliStep(options)
	
	
	
	'''
	PREPROCESS IMAGE/S IN ADVANCE
	'''
	preprocdone = 0
	
	if options.mask or options.maskfile or options.normproc or options.threshold or options.clip or (options.shrink > 1) or options.lowpass or options.highpass or options.preprocess:		
		
		print "\noptions.mask", options.mask
		print "\noptions.maskfile", options.maskfile
		print "\noptions.normproc", options.normproc
		print "\noptions.threshold", options.threshold
		print "\noptions.clip", options.clip
		print "\noptions.shrink", options.shrink
		print "\noptions.shrinkfine", options.shrinkfine
		print "\noptions.lowpass", options.lowpass
		print "\noptions.highpass", options.highpass
		print "\noptions.preprocess", options.preprocess
		
		print "\ntruth statement", options.mask or options.maskfile or options.normproc or options.threshold or options.clip or (options.shrink > 1) or options.lowpass or options.highpass or options.preprocess
		
		print "\n\n\nPPPPPPP options.input is", options.input
		ret = cmdpreproc( options.input, options, False )
		if ret: 
			preprocdone += 1
			'''
			Use preprocessed particles as input. Flawed, since you can only pass in one stack to
			alignment, and there could be two (fine and coarse) and the alignment function still
			decides internally #fix this later (jan/2016)
			'''
			options.input = options.path + '/' + ret
		
		else:
			print "\n(e2spt_classaverage)(main) preprocessing --input for coarse alignment failed"
		
		"""
		--ref needs to be preprocessed every iteration because it is updated every iteration. no point preprocessing it here
		"""
		
		"""
		if options.ref and options.refpreprocess:
			retref = cmdpreproc( options.ref, options, False )
			if retref: 
				preprocdone += 1
			else:
				print "\n(e2spt_classaverage)(main) preprocessing --ref for coarse alignment failed"
		else: 
			preprocdone += 1
		"""
		
	
	else:
		preprocdone += 1
	
	'''
	PREPROCESS IMAGE/S IN ADVANCE for *"FINE"* alignment IF tree aligner is not used
	'''
	if 'rotate_translate_3d_tree' not in options.align and options.falign:
		if options.mask or options.maskfile or options.normproc or options.threshold or options.clip or (options.shrinkfine > 1) or options.lowpassfine or options.highpassfine or options.preprocessfine:	
			
			ret = cmdpreproc( options.input, options, True ) #True tells the function that particles need to be preprocessed for fine alignment
			if ret: 
				preprocdone += 1
			else:
				print "\n(e2spt_classaverage)(main) preprocessing --input for fine alignment failed"
			
			
			"""
			--ref needs to be preprocessed every iteration because it is updated every iteration. no point preprocessing it here
			"""
			
			"""
			if options.ref and options.refpreprocess:
				retref = cmdpreproc( options.ref, options, True )
				if retref: 
					preprocdone += 1
				else:
					print "\n(e2spt_classaverage)(main) preprocessing --ref for fine alignment failed"
			else:
				preprocdone += 1
			"""
			
	else:
		preprocdone += 1
	

	print "\n(e2spt_classaverage)(main) preprocessing completed"

	'''
	Parse parameters such that "None" or "none" are adequately interpreted to turn of an option
	'''
	options = sptOptionsParser( options )

	print "after parsing options, options.goldstandardoff is", options.goldstandardoff, type(options.goldstandardoff)
	print options.align, type(options.align)
	
	if options.resultmx: 
		print "\nSorry, resultmx not implemented yet"

	#????
	if options.resultmx != None: 
		options.storebad = True
				
	
	

	logger = E2init(sys.argv, options.ppid)

	'''
	Store parameters in parameters.txt file inside --path
	'''
	cmdwp = writeParameters(options,'e2spt_classaverage.py', 'sptclassavg')


	
	if nptcl<1 : 
		print "(e2spt_classaverage)(main) - ERROR : at least 1 particle required in input stack"
		sys.exit(1)
	
	if nptcl==1:
		if options.iter>1 :
			print "(e2spt_classaverage)(main) - Error: makes no sense to have iter>1 with one particle"
			sys.exit(1)
	
		if options.keepsig or options.keep!=1.0 :
			print "(e2spt_classaverage)(main) - Error: do not use --keepsig with one particle, also keep should be 1.0 if specified"
			sys.exit(1)

	'''
	Initialize parallelism if being used
	'''

	if options.parallel :

		if options.parallel == 'none' or options.parallel == 'None' or options.parallel == 'NONE':
			options.parallel = None
			etc = None
	
		else:
			print "\n\n(e2spt_classaverage)(main) - INITIALIZING PARALLELISM!"
			print "\n\n"
			from EMAN2PAR import EMTaskCustomer
			etc=EMTaskCustomer(options.parallel)
			pclist=[options.input]
	
			#if options.ref: 
			#	pclist.append(options.ref)
			etc.precache(pclist)
	else:
		etc=''


	print "\n\n\nCCCC chached this input", options.input

	
	
	

	if options.inputaliparams: 
		preOrientationsDict = js_open_dict(options.inputaliparams)
	
	resumeDict = {}
	actualNumsDict = {}
	actualNumsEven = []
	actualNumsOdd = []

	try:
		if options.resume: 
			#print "The resume dict to open is", options.resume
			resumeDict = js_open_dict( options.resume )
	
			#print "Resume dict is", resumeDict
			for key in resumeDict.keys():
				#print "\n\nKKKKKKey is", key
		
				keyint = int ( key.split('_')[-1] )
				#print "\n\nkeyint is", keyint
		
				if keyint % 2:
					actualNumsOdd.append( keyint )
				else:	
					actualNumsEven.append( keyint )
	
			actualNumsDict.update({ 0:actualNumsEven, 1:actualNumsOdd })
			
			#print "\nActualNumsDict is", actualNumsDict	

			resumeDict.close()
	except:
		pass
		
	ncls = 1

	classmxFile = options.path + '/classmx_' + str( 0 ).zfill( len (str (options.iter))) + '.hdf'

	if not options.goldstandardoff and nptcl > 1:
		ncls = 2

	#C: Read or initialize all classmx images as 2-D EMData files of 2 x N dimensions, where N is the number of particles.

	refsdict = {}

	if options.classmx:
		print "\n--classmx=",options.classmx
		classmxFile = options.classmx
		classmx = EMData.read_images( classmxFile )		# we keep the entire classification matrix in memory, since we need to update it in most cases
		#ncls = int(classmx[0]["maximum"])
		#ncls = int( classmx[0]['nx'] )					#In this program, this will always be 1 for non-goldstandard and 2 for gold standard
	
		scoresImg = classmx[0]
	
		if options.inputaliparams:
			try:
				inputalidict = js_open_dict( options.inputaliparams )
				numdigits = len( inputalidict.keys()[0].split('_')[-1] )	#keys are of the form 'subtomo_XXX'; we determine how many XXX the keys have
			except:
				print "ERROR: Something is wrong with the json file provided". options.inputaliparams
				sys.exit(1)
		
		
		for ic in range(ncls):
			resultsforaverage = {}

			#ptclnums = classmx_ptcls( classmx, ic )
		
		
			#print "x dimension of classmx file is",  scoresImg['nx']
			#ptclnums = []
			#for j in range( scoresImg['nx'] ):				#Loop over the number of classes
			ptclnums=[]									#each of which starts with no particles in it
		
			ptclrange = scoresImg['ny']
			if options.subset:
				ptclrange = options.subset
			
			for i in range( ptclrange ):			#Loop over the number of particles
				score = scoresImg.get_value_at(ic,i)
				print "score in classmx is", score
				if score:
					print "therefore score is TRUE (should not be if it is 0.0!!!!)"
					ptclnums.append(i)
				
		
			for indx in ptclnums:
			
				score = classmx[0].get_value_at(ic,indx)
			
				if float( score ) > -0.000000001:
					print "\nscore in classmx was empty (positive or zero)"
				
					if options.recompute:
						if not options.inputaliparams:
							if options.keepsig or options.keep < 1.0:
								print "\n(e2spt_classaverage)(main) ERROR: if --recompute and --keep (or --keepsig) are provided simulaneously, you must also provide --inputaliparams"
								sys.exit()
						
					if options.inputaliparams:
						numtag = str( indx ).zfill( numdigits )
						key = 'subtomo_' + numtag
						score = inputalidict[ key ][1]
						print "\nfor subtomo %d, getting score from --inputaliparams, score=%f" %( indx, score )
				
				#classnum = classmx[0][indx]
				'''
				weight = classmx[1][indx]
				tx = classmx[2][indx]
				ty = classmx[3][indx]
				tz = classmx[4][indx]
				az = classmx[5][indx]
				alt = classmx[6][indx]
				phi = classmx[7][indx]
				scale = classmx[8][indx]
				#score = classmx[9][indx]
				'''
			
				weight = classmx[1].get_value_at(ic,indx)
				tx = classmx[2].get_value_at(ic,indx)
				ty = classmx[3].get_value_at(ic,indx)
				tz = classmx[4].get_value_at(ic,indx)
				az = classmx[5].get_value_at(ic,indx)
				alt = classmx[6].get_value_at(ic,indx)
				phi = classmx[7].get_value_at(ic,indx)
				scale = classmx[8].get_value_at(ic,indx)
			
				alitransform = Transform({'type':'eman','az':az,'alt':alt,'phi':phi,'tx':tx,'ty':ty,'tz':tz})
			
				if options.inputaliparams:
					key = 'subtomo_' + numtag
					transformjson = inputalidict[ key ][0]
				
					print "\ntransform classmx is", alitransform
					print "\nwhereas transform json is", transformjson
			
				resultsforaverage.update( { indx:[score,alitransform] } )
		
			print "\nic is", ic
			if ic == 0:
				print "\nresults for average EVEN are", resultsforaverage
			
			if ic == 1:
				print "\nresults for average ODD are", resultsforaverage
		
			ret = makeAverage( options, ic, resultsforaverage, 0 )
			ref = ret[0]
			#weights = ret[1]
		
			refsdict.update({ ic : ref })	
		
			if options.recompute:
				if ic == 0:
					ref.write_image( options.path + '/recomp_avg_even.hdf', 0 )
				elif ic == 1:
					ref.write_image( options.path + '/recomp_avg_odd.hdf', 0 )
			else:
				if ic == 0:
					ref.write_image( options.path + '/avgs_even.hdf', 0 )
				elif ic == 1:
					ref.write_image( options.path + '/avgs_odd.hdf', 0 )
		
		#try:
		if refsdict:
			refeven = refsdict[0]
			refodd = refsdict[1]
	
			fscfile = options.path + '/recompute_fsc.txt'
		
		
		
			print "options.align is", options.align
			refsavg = compareEvenOdd( options, refeven, refodd, 2, etc, fscfile, 'initial' ) #We pass on an "iteration number" > 1, that is 2 here, just so that both initial references are preprocessed and treated equally (this is required since --refpreprocessing is off when --ref is supplied 
		
			if options.recompute:
				refsavg.write_image( options.path + '/recomp_avg.hdf', 0 )
		
		else:
			print "\nERROR: building initial references from classmx failed!"

	
		if options.inputaliparams:
			inputalidict.close()
	
		if options.recompute:
			print "recompute finished"
			sys.exit()	
	else:
		options.classmx = classmxFile
		
		#C: classmx images are put into a single stack of 2-D images with 9 images in it (0-8)
		#C: and written out so that the file exists when accessed later by the code
		
		classmxScores = EMData(ncls,nptcl)
		classmxWeights = EMData(ncls,nptcl)
		classmxXs = EMData(ncls,nptcl)
		classmxYs = EMData(ncls,nptcl)
		classmxZs = EMData(ncls,nptcl)
		classmxAzs = EMData(ncls,nptcl)
		classmxAlts = EMData(ncls,nptcl)
		classmxPhis = EMData(ncls,nptcl)
		classmxScales = EMData(ncls,nptcl)

		classmxScores.to_zero()
	
		if not options.goldstandardoff and ncls > 1 and nptcl > 1:
			for i in range(nptcl):
				klassid = 0
				if i % 2:
					klassid = 1
					#print "\n(e2spt_classaverage)(main) - Particle %d will belong to odd class" %( i )
					#classmxScores.set_value_at( 1, i, 1.0 )					
			
				print "\n(e2spt_classaverage)(main) - Particle %d will belong to classid %d" %( i, klassid )
				classmxScores.set_value_at( klassid, i, 1.0 )
				
		else:
			classmxScores.to_one() 		#If there's only one class, all particles are in it
	
		classmxWeights.to_one() 	#Particles contribute entirely and equally to the class to which they are assigned
		classmxXs.to_zero()
		classmxYs.to_zero()
		classmxZs.to_zero()
		classmxAzs.to_zero()
		classmxAlts.to_zero()
		classmxPhis.to_zero()
		classmxScales.to_one()		#One means no scaling

		classmxScores.write_image(classmxFile,0)
		classmxWeights.write_image(classmxFile,1)
		classmxXs.write_image(classmxFile,2)
		classmxYs.write_image(classmxFile,3)
		classmxZs.write_image(classmxFile,4)
		classmxAzs.write_image(classmxFile,5)
		classmxAlts.write_image(classmxFile,6)
		classmxPhis.write_image(classmxFile,7)	
		classmxScales.write_image(classmxFile,8)
		
	print "\n(e2spt_classaverage)(main) - classmx files initialized."
		
	'''		
	Figure out from the classmx file how many classes there are, and what particle indexes are in each
	'''	

	ptclnumsdict = {}			#Each class might have a different list of particle numbers. 
								#We store them in a master dictionary, read only once.
	if ncls == 1: 
		ptclnumsdict.update({ 0 : range(nptcl) })		#If there's only one class, all indexes from 0 to nptcl will be in it
	elif ncls > 1:
		classmx = EMData.read_images( options.classmx )
		if options.verbose:
			#print "\n(e2spt_classaverage)(main) - Read classmx file and its type are", options.classmx, type(classmx)
			pass
		
		#ptclnums = classmx_ptcls(classmx,ic)				# This gets the list from the classmx

		scoresImg = classmx[0]
		#print "x dimension of classmx file is",  scoresImg['nx']
		for j in range( scoresImg['nx'] ):				#Loop over the number of classes
			ptclnums=[]									#each of which starts with no particles in it
			for i in range( scoresImg['ny'] ):			#Loop over the number of particles
				score = scoresImg.get_value_at(j,i)
				if score:								#If a score is non-zero for a [class,ptclindx] position, 
					ptclnums.append(i)					#in scoresImg, add that ptclindx to the ptclnums list for this class
				
			ptclnumsdict.update({ j : ptclnums })


	if not options.goldstandardoff:
		print "\nThe number of classes and their indexes are"
		for klass in ptclnumsdict:
			print "Klass is", klass
			print "Indexes are", ptclnumsdict[ klass ]

	#originalOutput = options.output


	if not refsdict:
		print "\nno classmx provided. Building initial models from scratch or preparing reference (phase randomizing) if --ref provided. --ref is", options.ref
		refsdict = sptRefGen( options, ptclnumsdict, cmdwp )

		if not refsdict:
			print "\nERROR: failed to build initial models from scratch"
			sys.exit()
	
	if not options.goldstandardoff and ncls > 1 and nptcl > 1:
	
		refeven = refsdict[0]
		refodd = refsdict[1] 
		
		if options.filterbyfsc:
			fscfile = options.path + '/initialrefs_fsc.txt'
		
			if not options.ref:	
				refsavg = compareEvenOdd( options, refeven, refodd, 2, etc, fscfile, 'initial' ) #We pass on an "iteration number" > 1, that is 2 here, just so that both initial references are preprocessed and treated equally (this is required since --refpreprocessing is off when --ref is supplied 
			
				if options.savesteps:
					refname = options.path + '/initialrefs_avg.hdf'
					refsavg.write_image(refname, 0)
			
			else:
				calcFsc( options, refeven, refodd, fscfile )
		
			print "options.lowpass and type were", options.lowpass, type( options.lowpass )
			options.lowpass = ('filter.wiener.byfsc',{ 'fscfile':fscfile })
			print "Now it is", options.lowpass, type(options.lowpass)


	'''
	We loop over the number of iterations first (opposed to older versions where you refined
	the classes for a given number of iterations), because at the end of every iteration
	the averages of the classes ought to be compared. E.g., for gold standard refinement 
	purposes and dynamic filtering by FSC, there are only 2 classes (0 and 1), and you measure
	the resolution by aligning the average of class 0 to the average of class 1 and computing the FSC.
	This FSC is then used to filter the particles and the average during preprocessing.
	'''

	#ptclshdr = EMData( options.input, 0, True )

	'''
	This dictionary is used as one measure of convergence. If two consecutive averages have the same values on the header, they are equivalent and the algorithm has converged
	'''
	avgshdrs = { 0:[''] }
	print "Upon initial setup, avgshdrs[0] is",  avgshdrs[0], type( avgshdrs[0] )

	'''
	This dictionary will store the mean score per class across alignment iterations
	'''
	meanScores = { 0:[0] }

	if not options.goldstandardoff and ncls > 1 and nptcl > 1:
		avgshdrs.update({1:['']})
		meanScores.update( { 1:[0] } )
		print "Upon initial setup, avgshdrs[1] is",  avgshdrs[1], type( avgshdrs[1] )


	gc.collect()	#free up unused memory

	from e2spt_preproc import preprocfunc

	for it in range( options.iter ):
		
		if it > 0:
			options.refpreprocess = True
		
		if options.verbose: 
			print "\n(e2spt_classaverage)(main) - ###### Iteration %d/%d"%(it, options.iter)

		classmxFile = options.path + '/classmx_' + str( it ).zfill( len (str (options.iter))) + '.hdf'
	
		#weightsMasterDict = {}
	
		for ic in range( ncls ):
			weights = {}
			if options.verbose:
				print "Processing class %d/%d"%(ic+1,ncls)

			ptclnums = ptclnumsdict[ ic ]
		
		
		
			klassid = '_even'
			if ic == 1:
				klassid = '_odd'
			if options.goldstandardoff:
				klassid = ''
		
		
			print "for klassid", klassid
			print "ptclnums are", ptclnums
			print "\n"
			
			#if options.savesteps:
			#	refname = options.path + '/class_' + str(ic).zfill( len( str(ic) )) + '.hdf'
			#	ref.write_image(refname,-1)

			#print "\nFor class", ic
			#print "Particle numbers are", ptclnums		
			#print "\n"
		
			#options.output = originalOutput.replace('.hdf', '_' + str(ic).zfill( len (str (ncls))) + '.hdf')
			#options.output = originalOutput
		
			#if ncls > 1:
			
		
				#options.output = originalOutput.replace('.hdf', klassid + '.hdf')	
	
			#resNum = 0
			resumeDict = {}
	
			#for it in range(options.iter):
		
			# In 2-D class-averaging, each alignment is fast, so we send each node a class-average to make
			# in 3-D each alignment is very slow, so we use a single ptcl->ref alignment as a task

			tasks=[]
			results=[]
	
		
			#try:
			ref = refsdict[ ic ]
			#if options.parallel: 
			
			tmpref = os.path.join(options.path,"tmpref.hdf")
			
			ref.write_image( tmpref, 0 )
		
			'''
			If the reference is preprocessed, write the preprocessed images. While the coarse alignment preprocessed reference is passed by default 
			(there's no "fine alignment" using the tree aligner), the fine alignment preprocessed reference is accessed internally from the alignment function if needed
			'''
			
			ref2use = tmpref
			ref2usefine = tmpref
			
			if options.refpreprocess:
			
				#ref2use = options.path + '/' + options.ref.replace('.hdf','_preproc.hdf')
				
				
				if options.mask or options.maskfile or options.normproc or options.threshold or options.clip or (options.shrink > 1) or options.lowpass or options.highpass or options.preprocess:	
					
					ref2use = tmpref.replace('.hdf','_preproc.hdf')
					ref = preprocfunc( ref, options, 0, ref2use )
					#ref = preprocfunc( tmpref, options, 0, ref2use )

				if options.falign and 'tree' not in options.align[0]:
					#ref2usefine = options.path + '/' + options.ref.replace('.hdf','_preprocfine.hdf')
					
					if options.mask or options.maskfile or options.normproc or options.threshold or options.clip or (options.shrinkfine > 1) or options.lowpassfine or options.highpassfine or options.preprocessfine:	
						origlowpass = options.lowpass
						orighighpass = options.highpass
						origpreproc = options.preproc
						origshrink = options.shrink
						
						options.lowpass = options.lowpassfine
						options.highpass = options.highpassfine
						options.preprocess = options.preprocessfine
						options.shrink = options.shrinkfine
						
						ref2usefine = tmpref.replace('.hdf','_preprocfine.hdf')
						reffine = preprocfunc( ref, options, 0, ref2usefine)
						#reffine = preprocfunc( tmpref, options, 0, ref2usefine )
						
						options.lowpass = origlowpass
						options.highpass = orighighpass
						options.preprocess = origpreproc
						options.shrink = origshrink
			else:
				#ref2use = options.path + '/' + options.ref.replace('.hdf','_preproc.hdf')
				if options.clip or (options.shrink > 1):
					ref2use = tmpref.replace('.hdf','_preproc.hdf')
					ref = preprocfunc( ref, options, 0, ref2use, False, True ) #False indicates this isn't simulated data, True turns on 'resizeonly' inside the function
					#ref = preprocfunc( tmpref, options, 0, ref2use, False, True )
				
				if options.falign and 'tree' not in options.align[0]:
					#ref2usefine = options.path + '/' + options.ref.replace('.hdf','_preprocfine.hdf')
					if options.clip or (options.shrinkfine > 1):
						
						origshrink = options.shrink
						options.shrink = options.shrinkfine

						ref2usefine = tmpref.replace('.hdf','_preprocfine.hdf')						
						reffine = preprocfunc( ref, options, 0, ref2usefine, False, True) #False indicates this isn't simulated data, True turns on 'resizeonly' inside the function
						#reffine = preprocfunc( tmpref, options, 0, ref2usefine, False, True )
						
						options.shrink = origshrink
				
				
			
		
			#except:
			#	print "reference is empty"
			#	print "classmx is", options.classmx
			
			#	sys.exit()
		
			# ALL CLIPPING and preprocessing should happen inside the preprocessing function
		
			'''
			Code to 'resume' crashed jobs
			'''
			try:
				if options.resume and actualNumsDict[ ic ]:
					resumeDict = js_open_dict(options.resume)
					#resNum += 1
			except:
				pass
				
			for ptclnum in ptclnums:
				nptclsinklass =len( ptclnums )
			
				try:
					if actualNums and ptclnum in actualNums:
						print """Skipping this particle because you provided --resume and the alignment info for this particle is aready present.
						Info for particle loaded into results""", ptclnum
				
						tomoID = "subtomo_" + str(ptclnum).zfill( len(str( len(ptclnums) )) )
				
						if tomoID not in resumeDict.keys():
							print "ERROR: This key is not in the file provided for --resume", tomoID
							sys.exit() 
				
		
						if len(resumeDict.keys()) > 0:
							keys = resumeDict.keys()
					
							for key in keys:
								if type(resumeDict[key]) is not list:					 
									print """ERROR: Your sptali_ir.json file seems to be incomplete. The value for the particle key is a Transform(), but should be a list.
									The file should contain a dictionary where there's a 'key' for each particle, containing the word 'subtomo_' followed by the particle's index 
									(with as many digits as there are orders of magnitude in the set; for example
									the first particle in a set of 10 would be 'subtomo_0', but in a set of 10 to 100 it would be 'subtomo_00', and in a set of 101 to 1000
									it would be 'subtomo_000'), and the 'value' of each key would be a list with two elements, [ Transform(), score ], where Transform
									contains the alignment parameters between a particle and the reference, and score the cross correlation score for that alignment.""" 
									sys.exit()
						
						results.append( [ {'xform.align3d': resumeDict[tomoID][0] , 'score':resumeDict[tomoID][1] } ] )
				except:
					pass	
				
				transform = None					
				if options.inputaliparams:
					tomoID = "subtomo_" + str(ptclnum).zfill( len(str( len(ptclnums) )) )
					transform = preOrientationsDict[tomoID][0]
				
					print transform
					print "Of type", type(transform)
			
			
			
			
				if options.parallel:
					#task=Align3DTask(["cache",os.path.join(options.path,"tmpref.hdf"),0],["cache",options.input,ptclnum],ptclnum,"Ptcl %d in iter %d"%(ptclnum,it),options,transform,it)
					task=Align3DTask(["cache",os.path.join(options.path,ref2use),0],["cache",options.input,ptclnum],ptclnum,"Ptcl %d in iter %d"%(ptclnum,it),options,transform,it)
					tasks.append(task)
				else:
					#print "No parallelism specified"
					#result=align3Dfunc(["cache",os.path.join(options.path,"tmpref.hdf"),0],["cache",options.input,ptclnum],ptclnum,"Ptcl %d in iter %d"%(ptclnum,it),options,transform,it)
					result=align3Dfunc(["cache",os.path.join(options.path,ref2use),0],["cache",options.input,ptclnum],ptclnum,"Ptcl %d in iter %d"%(ptclnum,it),options,transform,it)
					results.append(result['final'])
		
			# start the alignments running
			if options.parallel:
				tids=etc.send_tasks(tasks)
				if options.verbose: 
					print "%d tasks queued in class %d iteration %d"%(len(tids),ic,it) 

				"""Wait for alignments to finish and get results"""
				#results=get_results(etc,tids,options.verbose,jsA,len(ptclnums),1)
				#results=get_results(etc,tids,options.verbose,jsA, nptcl ,1)
				#def get_results(etc,tids,verbose,nptcls,ref=''):
			
				results = filter( None, get_results(etc,tids,options.verbose, nptcl ) )
			
				#results = get_results(etc,tids,options.verbose, nptcl )
			
			
			
			if options.verbose > 2: 
				print "Results:" 
				pprint(results)
					
			if not options.donotaverage:					
				#ref = make_average(options,ic,options.input,options.path,results,options.averager,options.saveali,options.saveallalign,options.keep,options.keepsig,options.sym,options.groups,options.breaksym,options.nocenterofmass,options.verbose,it)
				ref=''
			
				'''
				If there's more than one particle, the reference for the next round will
				be the average of the aligned particles
				'''
				if nptcl > 1:
			
					resultsforaverage = {}
					for r in results:
						ptclindx = r[1]
						score = r[0][0]["score"]
						alitransform = r[0][0]["xform.align3d"]
						resultsforaverage.update( { ptclindx:[score,alitransform] } )
				
			
					ret = makeAverage(options, ic,resultsforaverage,it)
					ref = ret[0]
					weights = ret[1]
				
					#weightsid = 'even'
					#if ic == 1:
					#	weightsid = 'odd'
					#if options.goldstandardoff:
					#	klassid = 'all'
				
					#weightsMasterDict.update({ weightsid: weights })
				
				else:
					'''
					If there's only one particle, the "reference" for the next round will
					be the one aligned particle. However, since it makes no sense to have
					more iterations, the program will be stopped shortly, after the
					aligned particle is written out
					'''
					ref = EMData( options.input, 0 )
					
					ref['xform.align3d']=Transform()
					ref['origin_x'] = 0
					ref['origin_y'] = 0
					ref['origin_z'] = 0
					
					print "results len", len(results)
					print "results[0]", results[0]
					print "results", results
					t = Transform()
					score = 100000000000
					try:
						t = results[0][0][0]["xform.align3d"]
						score = results[0][0][0]['score']
						
						ref.process_inplace("xform",{"transform":t})
						ref['xform.align3d'] = t
						
					
					except:
						t = results[0][0]["xform.align3d"]
						score = results[0][0]['score']
						
						ref.process_inplace("xform",{"transform":t})
						ref['xform.align3d'] = t
				
					if options.postprocess:
						ppref = ref.copy()
						maskPP = "mask.sharp:outer_radius=-2"
						maskPP=parsemodopt(maskPP)

						ppref = postprocess(ppref,maskPP,options.normproc,options.postprocess)

						refnamePP = refname.replace('.hdf', '_postproc.hdf')
			
						#ppref.write_image("%s/class_%02d.hdf"%(options.path,ic),it)
						ppref.write_image(refnamePP,it)
				
					outname = options.path + '/final_avg.hdf'
					#if options.output:
					#	outname = options.path + '/' + options.output
				
					ref.write_image( outname , 0)
					
					jsAliParamsPath = options.path + '/sptali_ir.json'
					jsA = js_open_dict( jsAliParamsPath )
					
					xformslabel = 'subtomo_' + str( 0 ).zfill( len( str( nptcl ) ) )			
					jsA.setval( xformslabel, [ t , score ] )
					jsA.close()
					
					print "Done aligning the only particle in --input to --ref"
					sys.exit()
		
				
			
				refsdict.update( { ic:ref } )

				if options.savesteps:
			
					#refname = options.path + '/class_' + str(ic).zfill( len( str(ic) )) + '.hdf'
					refname = options.path + '/avgs' + klassid + '.hdf'
			
					
					ref.write_image( refname, it )
			
					#print "\n\nFor the final avg going into class_x.hdf, ali params are", ref['xform.align3d']
			
					if options.postprocess:
						ppref = ref.copy()
						maskPP = "mask.sharp:outer_radius=-2"
						maskPP=parsemodopt(maskPP)

						ppref = postprocess(ppref,maskPP,options.normproc,options.postprocess)

						refnamePP = refname.replace('.hdf', '_postproc.hdf')
				
						#ppref.write_image("%s/class_%02d.hdf"%(options.path,ic),it)
						ppref.write_image(refnamePP,it)
			
				if it == options.iter -1:
			
					outname = options.path + '/final_avg.hdf'
					#if options.output:
					#	outname = options.path + '/' + options.output
			
					ref.write_image( outname , 0 )			
			elif options.donotaverage: 				#This is used with e2spt_multirefine, where all you need is the alignment info, but not the averages
				for p in range( nptcl ):
					weights.update( {p:1.0} )
			
			
			
			'''
			Define and open the .json dictionaries where alignment and score values will be stored, for each iteration,
			and for each reference if using multiple model refinement
			'''
		
			jsAliParamsPath = options.path + '/sptali_ir.json'
		
		
			print "options.refinemultireftag and type are", options.refinemultireftag, type(options.refinemultireftag)
		
			if not options.refinemultireftag:
				jsAliParamsPath = jsAliParamsPath.replace('.json', '_' + str(it).zfill( len(str(options.iter))) + '.json')
			
			

			print "(e2spt_classaverage) This is the .json file to write", jsAliParamsPath
			jsA = js_open_dict( jsAliParamsPath ) #Write particle orientations to json database.
		
			iii = 0
		
			classScoresList = [] 
		
			'''
			Iterate over alignment results to write them to classmx.hdf and .json files
			'''
			print "len results is", len(results)
			print "should match nptcl", nptcl
		
			#print "results are", results
		
			for r in results:
				#if r and r[0]:	
			
			
				ptclindx = r[1]
				
				score = r[0][0]['score']
				if options.verbose > 3:
					print "for particle %d score is %.4f" %(ptclindx,score)
			
				#classmxScores.set_value_at(ic,iii,score)
				#posscore = math.fabs(score)
			
				classScoresList.append(score)
			
				#print "\n\n\n\n\n\n\n\nThe appended positive score is", posscore
				#weight=1.0
			
				if options.verbose > 3:
					print "classmxWeights img is", classmxWeights['nx'], classmxWeights['ny'], classmxWeights['nz']
				
				print "weights is", weights
				print "gout to set value for classmxweights at", ic, ptclindx, weights[ptclindx]
				classmxWeights.set_value_at(ic,ptclindx,weights[ptclindx])
			
				print "set value for classmxweights at", ic, ptclindx, weights[ptclindx]
			
				t = r[0][0]['xform.align3d']
			
				if options.verbose > 3:
					print "and transform is",t
		
				xformslabel = 'subtomo_' + str( ptclindx ).zfill( len( str( nptcl ) ) )			
				jsA.setval( xformslabel, [ t , score ] )
		
				if options.verbose > 3:
					print "wrote info to .json file"
			
				if options.saveallpeaks and options.npeakstorefine > 1 and options.falign:
					scoresCoarse = []
				
					rCoarse = r[2]
				
					for peak in rCoarse:
						sc = peak['score']
					
						if float( sc ) < 0.0:
							scoresCoarse.append( sc )
					
					meanPeaksScoresCoarse = numpy.mean( scoresCoarse )
				
					#sigmaPeaksScoresCoarse = 0.0
					#if len( scoresCoarse ) > 1:
				
					sigmaPeaksScoresCoarse = numpy.std( scoresCoarse )
				
				
					if len(scoresCoarse) > 1:
				
				
						jsAliParamsPathCoarseAll = options.path + '/sptali_ir_coarse_allpeaks.json'

						if not options.refinemultireftag:
							jsAliParamsPathCoarseAll = jsAliParamsPathCoarseAll.replace('.json', '_' + str(it).zfill( len(str(options.iter))) + '.json')

						jsAcoarseAll = js_open_dict( jsAliParamsPathCoarseAll )
					
						pi = 0
					
						for peak in rCoarse:
							tc = peak['xform.align3d']
							sc = peak['score']
					
							if float( sc ) < 0.0:
								z = ( sc - meanPeaksScoresCoarse ) / sigmaPeaksScoresCoarse
					
								print "pi, sc, meanPeaksScoresCoarse, sigmaPeaksScoresCoarse, z", pi, sc, meanPeaksScoresCoarse, sigmaPeaksScoresCoarse, z
					
								peaklabel = str( pi ).zfill( len( str(options.npeakstorefine) ) )
					
								jsAcoarseAll.setval( xformslabel + '_peak_' + peaklabel , [ tc , 'ccc='+str(sc), 'zscore='+str(z) ] )
							else:
								print "Empty score for peak", pi
					
							pi += 1	
					
						jsAcoarseAll.close()
				
					else:
						print "WARNING: Not enough successful alignments to compute a z-score"
				else:
					pass #Only one peak from --npeakstorefine was refined
					
			
				if options.verbose > 3:
					print "setting classmx"
					
				trans=t.get_trans()
				print "\n\n\nTranslations were", trans
				print "Therefre the transform was", t
				rots=t.get_rotation()
		
				tx=trans[0]
				print "Translation in x was", tx
				classmxXs.set_value_at(ic,ptclindx,tx)
		
				ty=trans[1]
				print "Translation in y was", ty
				classmxYs.set_value_at(ic,ptclindx,ty)
		
				tz=trans[2]
				print "Translation in z was", tz
				classmxZs.set_value_at(ic,ptclindx,tz)
		
				az=rots['az']
				classmxAzs.set_value_at(ic,ptclindx,az)
		
				alt=rots['alt']
				classmxAlts.set_value_at(ic,ptclindx,alt)
		
				phi=rots['phi']
				classmxPhis.set_value_at(ic,ptclindx,phi)
		
				scale=1.0
				classmxScales.set_value_at(ic,ptclindx,scale)
			#else:
			#	print "Warning!! Aberrant results r[0]", r[0]
			#	print "and r", r
			#	sys.exit()
		
			iii+=1
			
									
			jsA.close()
			print "closed .json file"
		
		
		
			classmxScores.write_image(classmxFile,0)
			classmxWeights.write_image(classmxFile,1)
			classmxXs.write_image(classmxFile,2)
			classmxYs.write_image(classmxFile,3)
			classmxZs.write_image(classmxFile,4)
			classmxAzs.write_image(classmxFile,5)
			classmxAlts.write_image(classmxFile,6)
			classmxPhis.write_image(classmxFile,7)	
			classmxScales.write_image(classmxFile,8)
	
			meanScore = sum( classScoresList ) / len( classScoresList )
		
			if it == 0:
				meanScores[ic] = [ meanScore ]
			else:
				meanScores[ic].append( meanScore )
		
		
			#classScoresList.reverse()
			maxY = max(classScoresList) + 1
	
			plotX = range( len(classScoresList) )
			maxX = max(plotX) + 1
	
			plotName = 'spt_cccs_' + str( it ).zfill( len( str( options.iter ) )) + klassid + '.png'
		
			classScoresList2plot = list(classScoresList)
			classScoresList2plot.sort()
			classScoresList2plot.reverse()
		
			txtname = plotName.replace('.png','.txt')
			textwriter(classScoresList2plot,options,txtname, invert=1)
			
			if options.plots:
				from e2spt_hac import plotter
		
				plotter(plotX, classScoresList2plot, options, plotName, maxX, maxY, 1, sort=1)
		
			gc.collect()	 #free up unused memory
		
		
	
		'''
		Compare even and odd averages if the user did not provide --goldstandardoff
		'''
		if not options.goldstandardoff and len( refsdict ) > 1 and ncls > 1:
			avgeven = refsdict[0]
			avgodd = refsdict[1] 
		
			print "goldstandardoff is", options.goldstandardoff
			print "avgshdrs[0] is and type", avgshdrs[0], type(avgshdrs[0])
			print "avgshdrs[1] is and type", avgshdrs[1], type(avgshdrs[1])
		
			#print "avgeven is", avgeven
			#print "avgodd is", avgodd
		
			avgshdrs[0].append( avgeven.get_attr_dict() )
			avgshdrs[1].append( avgodd.get_attr_dict() )
		
			#avgshdrs.update( { 0:avgshdrs[0].append( avgeven.get_attr_dict() ), 1:avgshdrs[1].append( avgodd.get_attr_dict() ) } )
		
			if it > 0 and len(avgshdrs[0]) > 1 and len(avgshdrs[1]) > 1:
				if avgshdrs[0][-1]['mean'] == avgshdrs[0][-2]['mean'] and avgshdrs[1][-1]['mean'] == avgshdrs[0][-2]['mean']:
					print "Both independent averages have converged!"
					#os.system('rm ' + options.path + '/tmpref.hdf')
					os.remove( options.path + '/tmpref.hdf' )
					sys.exit()
			
			fscfile = options.path + '/fsc_' + str(it).zfill( len( str(options.iter))) + '.txt'
		
			final_avg = compareEvenOdd(options, avgeven, avgodd, it, etc, fscfile, 'goldstandard'  )
		
		
			if options.savesteps:
				final_avg.write_image( options.path + '/avgs_ali2ref.hdf' , it)

			if it == options.iter -1 :
		
				outname = options.path + '/final_avg_ali2ref.hdf'
				#if options.output:
				#	outname = options.path + '/' + options.output
				
				final_avg.write_image( outname , 0)
		
		
		
			try:
				if options.resume and actualNums:
					resumeDict.close()
			except:	
				pass
					
			actualNums = [] 		#Reset this so that when --resume is provided the incomplete jason file is 'fixed' considering the info in actualNums only once

	
		elif options.goldstandardoff and options.ref:
			avg = refsdict[0]
		
			#print "avgshdrs[0] is and type", avgshdrs[0], type(avgshdrs[0])
			avgshdrs[0].append( avg.get_attr_dict() )
		
			#avgshdrs.update( { 0:avgshdrs[0].append( avg.get_attr_dict() ) } )
		
			if it > 0 and len(avgshdrs[0]) > 1 :
				if avgshdrs[0][-1]['mean'] == avgshdrs[0][-2]['mean']:
					print "The average has converged!"
					outname = options.path + '/final_avg_ali2ref.hdf'
					avg.write_image( outname , 0)
					#os.system('rm ' + options.path + '/tmpref.hdf')
					os.remove( options.path + '/tmpref.hdf' )
					sys.exit()

			originalref = EMData( options.ref, 0 )
		
			fscfile = options.path + '/fsc_' + str(it).zfill( len( str(options.iter))) + '.txt'

			#calcFsc( originalref, avg, fscfile )
		
			final_avg = compareEvenOdd( options, originalref, avg, it, etc, fscfile, 'refbased', average=False ) #We pass on an "iteration number" > 1, that is 2 here, just so that both initial references are preprocessed and treated equally (this is required since --refpreprocessing is off when --ref is supplied 
			
			if not options.sym:
				if options.savesteps:
					final_avg.write_image( options.path + '/avgs_ali2ref.hdf' , it)

				if it == options.iter -1 :
		
					outname = options.path + '/final_avg_ali2ref.hdf'
					final_avg.write_image( outname , 0)
		
		
			if options.filterbyfsc:
				print "Options.lowpass was and of type", options.lowpass, type( options.lowpass )
				options.lowpass = ('filter.wiener.byfsc',{ 'fscfile':fscfile })
				print "Now it is", options.lowpass, type(options.lowpass)
			
		ic+=1	
		#import matplotlib as plt
	
	for key in meanScores:
		scores = meanScores[ key ]
		klassid = '_even'
		if key == 1:
			klassid = '_odd'
		if options.goldstandardoff:
			klassid = ''

		maxY = max(scores) + 1
	
		plotX = range( len(scores) )
		maxX = max(plotX) + 1

		plotName = 'spt_meanccc' + klassid + '.png'
	
		scores2plot = list(scores)
		scores2plot.sort()
		scores2plot.reverse()
	
		txtname = plotName.replace('.png','.txt')
		textwriter( scores2plot ,options, txtname, invert=1 )

	if options.plots:
		from e2spt_hac import plotter

		plotter(plotX, scores2plot, options, plotName, maxX, maxY, 1, sort=0)
	
	
	if options.inputaliparams: 
		preOrientationsDict.close()

	#os.system('rm ' + options.path + '/tmpref.hdf')
	os.remove( options.path + '/tmpref.hdf' )
	
	print "Logger ending"	
	E2end(logger)
	
	print "logger ended"
	sys.stdout.flush()
	
	
	
	return



'''
Function to check whether images are cubed and even, and to check whether input and ref match in size
'''
def checksaneimagesize( options, stack1, stack2=''):
	
	stacks = [stack1]
	if stack2:
		stacks.append( stack2 )
		
	for stack in stacks:
	
		try:
			print "(try) stack is", stack
			stackhdr = EMData( stack, 0, True )
			nx = stackhdr["nx"]
			ny = stackhdr["ny"]
			nz = stackhdr["nz"]
		
			if nx!=ny or ny!=nz or nx!=nz:
				print ("\n(e2spt_classaverage)(checksaneimagesize) ERROR: stack %s must contain volumes that are cubes (nx=ny=nz). The size of the images in stack %s is nx=%d, ny=%d, nz=%d" %(stack,nx,ny,nz))
				sys.exit(1)
			
			if nx % 2 or ny % 2 or nz %2:
				print ("\n(e2spt_classaverage)(checksaneimagesize) ERROR: volumes in stack %s must be even sized. The size of the images provided is nx=%d, ny=%d, nz=%d" %(nx,ny,nz))
				print ("while I could resize the images for you on the fly, results will be less murky if you're explicitly aware that the particles need resizing. Please run 'e2proc3d.py input.hdf output_resized.hdf --clip newsize' at the commandline, choosing 'newsize' to be an even number (e.g., 1 pixel smaller than the current boxsize). Then rerun e2spt_classaverage with the resized stack as --input.") 
				print sys.exit()
			elif options.shrink:
				if nx/options.shrink %2 or ny/options.shrink %2 or nz/options.shrink %2:
					print ("\n(e2spt_classaverage)(checksaneimagesize) --shrink is %d; therefore the size of the images in stack %s after shrinking would be x=%d, y=%d, z=%d, which is not even. Please run 'e2proc3d.py input.hdf output_resized.hdf --clip newsize' at the commandline, choosing a box size such that when shrunk by the shrink factor specified it still yields an even box size. " %(options.shrink, nx/options.shrink %2 ,ny/options.shrink %2 ,nz/options.shrink %2 ))
					sys.exit(1)
			
			elif options.shrinkfine:
				if nx/options.shrinkfine %2 or ny/options.shrinkfine %2 or nz/options.shrinkfine %2:
					print ("\n(e2spt_classaverage)(checksaneimagesize) --shrinkfine is %d; therefore the size of the images in stack %s after shrinking would be x=%d, y=%d, z=%d, which is not even. Please run 'e2proc3d.py input.hdf output_resized.hdf --clip newsize' at the commandline, choosing a box size such that when shrunk by the shrink factor specified it still yields an even box size. " %(options.shrinkfine, nx/options.shrinkfine %2 ,ny/options.shrinkfine %2 ,nz/options.shrinkfine %2 ))
					sys.exit(1)
		except:
			print "(except) stack is", stack
			stackhdr = EMData( stack, 0, True )
			print "header stack1hdr is", stack1hdr
			print ("\n(e2spt_classaverage)(checksaneimagesize) ERROR reading stack %s. Check that the file name is correct and the file is sane" %(stack) )
			sys.exit(1)
	
	
	if stack2:
		stack1hdr = EMData( stack1, 0, True )
		stack2hdr = EMData( stack2, 0, True )
		if stack2hdr["nx"]!=stack1hdr["nx"] or stack2hdr["ny"]!=stack1hdr["nx"] or stack2hdr["nz"]!=stack1hdr["nx"]: 
			print """(e2spt_classaverage)(checksaneimagesize). ERROR: the size of stack %s is x1=%d, y1=%d, z1=%d, is not the same as the size of stack %s, x2=%d, y2=%d, z2=%d""" %(stack2, stack2hdr["nx"], stack2hdr["ny"], stack2hdr["nz"], stack1, stack1hdr["nx"], stack1hdr["nx"], stack1hdr["nx"])
			sys.exit(1)
		
	return
	

'''
Function to generate preprocessing command which calls e2spt_preproc.py
'''
def cmdpreproc( fyle, options, finetag=False ):	

	#from e2spt_preproc import preprocfunc	
	
	cmdpreproc = 'e2spt_preproc.py --input ' + fyle + ' --nopath'
	
	#if options.parallel:
	#	cmdpreproc += ' --parallel ' + options.parallel
	
	#if "_preproc.hdf" not in fyle:
	preprocstack = os.path.basename( fyle ).replace('.hdf','_preproc.hdf')
	
	if finetag:
		#if "_preprocfine.hdf" not in fyle:
		preprocstack = os.path.basename( fyle ).replace('.hdf','_preprocfine.hdf')
		
	
	#preprocref = ''
	#cmdpreprocref = ''
	
	#if options.ref and options.refpreprocess:
	#	cmdpreprocref = 'e2spt_preproc.py --input ' + options.ref
	#	preprocref = os.path.basename(options.ref).replace('.hdf','_preproc.hdf')
	
	
	if options.mask:
		cmdpreproc += ' --mask ' + options.mask
		#if options.ref and options.refpreprocess:
		#	cmdpreprocref += ' --mask ' + options.mask
	
	if options.maskfile:
		cmdpreproc += ' --maskfile ' + options.maskfile 
		#if options.ref and options.refpreprocess:
		#	cmdpreprocref += ' --maskfile ' + options.maskfile 
		
	if options.normproc:
		cmdpreproc += ' --normproc ' + options.normproc
		#if options.ref and options.refpreprocess:
		#	cmdpreprocref += ' --normproc ' + options.normproc
		 
	if options.threshold:
		cmdpreproc += ' --threshold ' + options.threshold
		#if options.ref and options.refpreprocess:
		#	cmdpreprocref += ' --threshold ' + options.threshold
	
	if options.clip:
		cmdpreproc += ' --clip ' + str(options.clip)
		#if options.ref and options.refpreprocess:
		#	cmdpreprocref += ' --clip ' + str(options.clip)
	
	
	if not finetag:
		if options.shrink > 1:
			cmdpreproc += ' --shrink ' + str(options.shrink)
			#if options.ref and options.refpreprocess:
			#	cmdpreprocref += ' --shrink ' + str(options.shrink)
	
		if options.lowpass:
			cmdpreproc += ' --lowpass ' + options.lowpass
			#if options.ref and options.refpreprocess:
			#	cmdpreprocref += ' --lowpass ' + options.lowpass
		
		if options.highpass:
			cmdpreproc += ' --highpass ' + options.highpass
			#if options.ref and options.refpreprocess:
			#	cmdpreprocref += ' --highpass ' + options.highpass

		if options.preprocess:
			cmdpreproc += ' --preprocess ' + options.preprocess
			#if options.ref and options.refpreprocess:
			#	cmdpreprocref += ' --preprocess ' + options.preprocess
	
	elif finetag:
		if options.shrinkfine > 1:
			cmdpreproc += ' --shrink ' + str(options.shrinkfine )
			#if options.ref and options.refpreprocess:
			#	cmdpreprocref += ' --shrink ' + str(options.shrink)
	
		if options.lowpassfine :
			cmdpreproc += ' --lowpass ' + options.lowpassfine 
			#if options.ref and options.refpreprocess:
			#	cmdpreprocref += ' --lowpass ' + options.lowpass
		
		if options.highpassfine :
			cmdpreproc += ' --highpass ' + options.highpassfine 
			#if options.ref and options.refpreprocess:
			#	cmdpreprocref += ' --highpass ' + options.highpass

		if options.preprocessfine :
			cmdpreproc += ' --preprocess ' + options.preprocessfine 
			#if options.ref and options.refpreprocess:
			#	cmdpreprocref += ' --preprocess ' + options.preprocess
	
		
	
	ret = runcmd( options, cmdpreproc )
	
	if ret:
		input_preproc = options.path + '/' + preprocstack
		
		print "\npreprocstack %s will be %s" %(preprocstack, input_preproc)

		import time
		time.sleep(5)

		print "renaming preprocstack %s to input_preproc %s" %(preprocstack, input_preproc )

		os.rename( preprocstack, input_preproc )
		options.input = input_preproc
	else:
		print "\n(e2spt_classaverage)(cmdpreproc) preprocessing %s crashed" %( fyle ) 
		sys.exit(1)
		
			
	#if options.ref and options.refpreprocess:
	
	#	retref = runcmd( options, cmdpreprocref )
		
	#	if retref:
	#		ref_preproc = options.path + '/' + preprocref
	#		os.rename( preprocref, ref_preproc )
	#	else:
	#		print "\n(e2spt_classaverage)(main) reference particles for coarase alignment crashed"
					
	return preprocstack
		





def textwriterinfo(ydata,options,name):
	if len(ydata) ==0:
		print "ERROR: Attempting to write an empty text file!"
		sys.exit()
	
	if options.path not in name:
		name=options.path + '/' + name
	
	print "I am in the text writer for this file", name
	
	f=open(name,'w')
	lines=[]
	for i in range(len(ydata)):
		line2write = 'comparison#' + str(i) + ' ' + str(ydata[i][-2]) + ' vs ' + str(ydata[i][-1])+ ' score=' + str(ydata[i][0]) + '\n'
		#print "THe line to write is"
		lines.append(line2write)
	
	#print "Thare are these many lines", len (lines)
	#print "Cause y data is", len(ydata)
	#sys.exit()
	
	f.writelines(lines)
	f.close()

	return()
	
	
def textwriter(ydata,options,name,invert=0):
	
	if options.path not in name:
		name=options.path + '/' + name
	
	print "I am in the text writer for this file", name
	
	f=open(name,'w')
	lines=[]
	for i in range(len(ydata)):
		y=ydata[i]
		if invert:
			y*=-1
			
		line2write = str(i) + '\t' + str(y) + '\n'
		#print "THe line to write is"
		lines.append(line2write)
	
	f.writelines(lines)
	f.close()

	return







'''
This function tries to alleviate the issue of providing parameters both through the aligner,
and outside of it. For example, --sym and --search, vs --align:whatever_alinger:sym=xxx:search=xxx.
Command line options should take precedence.
'''
def sptParseAligner( options ):
	
	if options.align and 'rotate' in options.align:
		#print "There's options.align", options.align
		if options.sym and options.sym is not 'c1' and options.sym is not 'C1' and 'sym' not in options.align and 'grid' not in options.align:
			if 'rotate_translate_3d' in options.align or 'rotate_symmetry_3d' in options.align or 'rotate_translate_3d_tree' in options.align:
				options.align += ':sym=' + str( options.sym )
			#print "And there's sym", options.sym
			
		if 'search' not in options.align:
			if 'rotate_translate_3d' in options.align and options.search and 'tree' not in options.align:	
				options.align += ':search=' + str( options.search )
			
		elif 'rotate_translate_3d' in options.align and 'tree' not in options.align:
			searchA = options.align.split('search=')[-1].split(':')[0]
			searchdefault = 8
			
			if options.search != searchdefault:
						
				prefix = options.align.split('search=')[0]
				trail = options.align.split('search=')[-1].split(':')[-1]
			
				options.align =  prefix + 'search=' + str(options.search)
				if len(trail) > 2 and '=' in trail:
					options.align += ':' + trail 
			
				print """\nWARNING: --search is different from search= provided through
				--align or its default value of 8. There's no need to specify both, but 
				if you did, --search takes precedence :-) ."""
				#sys.exit()
			elif options.search == searchdefault:
				options.search = searchA
			

		if "rotate_translate_3d_grid" in options.align:
			if "alt0" and "alt1" in options.align:
				alt0 = int(options.align.split('alt0')[-1].split(':')[0].replace('=',''))	
				alt1 = int(options.align.split('alt1')[-1].split(':')[0].replace('=',''))
				
				print "alt0 and alt1 are", alt0,alt1, type(alt0), type(alt1)
				print alt1-alt0 == 0
				#sys.exit()
				
				if alt1-alt0 == 0:
					print """\nERROR: alt0 and alt1 cannot be equal for rotate_translate_3d_grid.
					If you want to inactivate searches in this angle, provide a alt0 and alt1
					such that alt1-alt0 is NOT ZERO, and provide a step size for dalt that is larger
					than this difference. For example: 
					alt0=0:alt1=1:dalt=2."""
					sys.exit()
					
			if "phi0" and "phi1" in options.align:
				phi0 = int(options.align.split('phi0')[-1].split(':')[0].replace('=',''))	
				phi1 = int(options.align.split('phi1')[-1].split(':')[0].replace('=',''))
				
				print "phi0 and phi1 are", phi0,phi1, type(phi0), type(phi1)
				print phi1-phi0 == 0
				#sys.exit()
				
				if phi1-phi0 == 0:
					print """\nERROR: phi0 and phi1 cannot be equal for rotate_translate_3d_grid.
					If you want to inactivate searches in this angle, provide a phi0 and phi1
					such that phi1-phi0 is NOT ZERO, and provide a step size for dphi that is larger
					than this difference. For example: 
					phi0=0:phi1=1:dphi=2."""
					sys.exit()
					
			if "az0" and "az1" in options.align:
				az0 = int(options.align.split('az0')[-1].split(':')[0].replace('=',''))	
				az1 = int(options.align.split('az1')[-1].split(':')[0].replace('=',''))
				
				print "az0 and az1 are", az0,az1, type(az0), type(az1)
				print az1-az0 == 0
				#sys.exit()
				
				if az1-az0 == 0:
					print """\nERROR: az0 and az1 cannot be equal for rotate_translate_3d_grid.
					If you want to inactivate searches in this angle, provide a az0 and az1
					such that az1-az0 is NOT ZERO, and provide a step size for daz that is larger
					than this difference. For example: 
					az0=0:az1=1:daz=2."""
					sys.exit()
		
	if options.falign:
		print "\n(e2spt_classaverage)(main) Before adding and fixing searches, options.falign is", options.falign, type(options.falign)	
	else:
		print "\n(e2spt_classaverage)(main) no --falign specified"
	
	if options.falign: 
		if options.falign != 'None' and options.falign != 'none':
			if 'refine_3d_grid' in options.falign:
				if 'search' not in options.falign:
					options.falign += ':search=' + str( options.searchfine )
			
				else:
					searchF = options.falign.split('search=')[-1].split(':')[0]
					searchfinedefault = 2
		
					if options.searchfine != searchfinedefault:
					
						prefix = options.falign.split('search=')[0]
						trail = options.falign.split('search=')[-1].split(':')[-1]
			
						options.falign =  prefix + 'search=' + str(options.searchfine)
			
						if len(trail) > 2 and '=' in trail:
			
							options.falign += ':' + trail 
		
						print """\nWARNING: --searchfine is different from search= provided through
						--falign or its default value of 2. There's no need to specify both, but 
						if you did, --searchfine takes precedence :-) ."""
						#sys.exit()
		
					elif options.searchfine == searchfinedefault:
						options.searchfine = searchF	
	
	
	return options


def compareEvenOdd( options, avgeven, avgodd, it, etc, fscfile, tag, average=True ):
	
	avgeven.process_inplace( 'normalize.edgemean' )
	avgodd.process_inplace( 'normalize.edgemean' )
	
	#from EMAN2PAR import EMTaskCustomer
	#etc=EMTaskCustomer(options.parallel)
	
	#tasks = []
	
	if not options.parallel:
		print "ERROR: Parallelization cannot be turned off unless you supply --goldstandardoff as well"
		sys.exit()
	
	#task = Align3DTask( avgeven, avgodd, 0, "avgeven(ref) vs avgodd", options, None, it, nptclsexception=1)
	#tasks.append( task )

	#tids = etc.send_tasks(tasks)
	
	#resultsAvgs = get_results( etc, tids, options.verbose, 1 )
								#etc,tids,verbose,jsA,nptcls,savealiparams=0,ref=''
								#def get_results(etc,tids,verbose,nptcls,ref=''):
	
	
	if not options.sym:
		resultsAvgs = align3Dfunc( avgeven, avgodd, 0, "avgeven(ref) vs avgodd", options, None, 2 )
								
		##print "r['final'] is", resultsAvgs['final']
		##print "\nr['final'][0] is", resultsAvgs['final'][0]
		#print "\nr[0][0] is", resultsAvgs[0][0]
		#print "\nr[0][0[0] is", resultsAvgs[0][0][0]
		##print "\nr['final'][0]['xform.align3d'] is", resultsAvgs['final'][0]["xform.align3d"]
		#print "\nr[0][-1]", resultsAvgs[0][-1]
	
		transformAliOdd2even = resultsAvgs['final'][0]['xform.align3d']
		scoreAliOdd2even = resultsAvgs['final'][0]['score']
	
		avgsDict = options.path + '/sptali_ir_' + str(it).zfill( len( str(options.iter))) + '_oddali2even.json'
	
		if not average: 
			if tag == 'refbased':
				avgsDict = options.path + '/sptali_ir_' + str(it).zfill( len( str( it ))) + '_avgali2ref.json'
	
		else:
			if tag == 'initial':
				avgsDict = options.path + '/sptali_ir_initialrefs_oddali2even.json'
			if tag == 'recompute':
				 options.path + '/sptali_ir_recompute_oddali2even.json'
			
		jsAvgs = js_open_dict( avgsDict )

		xformslabel = 'subtomo_0'		
	
		jsAvgs.setval( xformslabel, [ transformAliOdd2even , scoreAliOdd2even ] )

		jsAvgs.close()
		
		avgodd.transform( transformAliOdd2even )
	
	finalA = None
	
	if average:
		avgr = Averagers.get( options.averager[0], options.averager[1])
		avgr.add_image( avgeven )
		avgr.add_image( avgodd )
		finalA = avgr.finish()
	
		finalA['origin_x']=0
		finalA['origin_y']=0		#The origin needs to be reset to ZERO to avoid display issues in Chimera
		finalA['origin_z']=0
		finalA['xform.align3d'] = Transform()

		#apix = final_avg['apix_x']
		
	if not average:
		finalA = avgodd.copy()	#avgeven contains the reference. avgodd contains the new average, which needs to be aligned to avgeven (if no symmetry is provided)
	
	if options.sym and not options.breaksym :
		finalA = finalA.process('xform.applysym',{'sym':options.sym})
	
	calcFsc( options, avgeven, avgodd, fscfile )
	
	finalA.process_inplace('normalize.edgemean')
	
	apix = finalA['apix_x']
	nyquist = 2.0 * apix			#this is nyquist 'resolution', not frequency; it's the inverse of nyquist frequency
	halfnyquist = 2.0 * nyquist
	twothirdsnyquist = 3.0 * nyquist / 2.0
	boxsize = finalA['nx']
	halfbox = boxsize/2
	halfnshells = halfbox - int(halfbox/2.0)
	twothirdshells = halfbox - int((2.0/3.0)*halfbox)
	fourfithsshells = halfbox - int((4.0/5.0)*halfbox)
	
	
	
	eventhresh = avgeven['sigma']
	#eventhresh = 0
	avgevenmasked = avgeven.process('mask.auto3d',{'nshells':1,'nshellsgauss':4,'radius':1,'nmaxseed':1,'threshold': eventhresh })
	
	oddthresh = avgodd['sigma']
	#oddthresh = 0
	avgoddmasked = avgodd.process('mask.auto3d',{'nshells':1,'nshellsgauss':4,'radius':1,'nmaxseed':1,'threshold': oddthresh })
	
	fscfilemasked = fscfile.replace('.txt','_mask_supertight.txt')
	
	calcFsc( options, avgevenmasked, avgoddmasked, fscfilemasked )
	
	
	avgevenmasked = avgeven.process('mask.auto3d',{'nshells':1,'nshellsgauss':fourfithsshells,'radius':1,'nmaxseed':1,'threshold': eventhresh })
	avgoddmasked = avgodd.process('mask.auto3d',{'nshells':1,'nshellsgauss':fourfithsshells,'radius':1,'nmaxseed':1,'threshold': oddthresh })
	
	fscfilemasked2 = fscfile.replace('.txt','_mask_lesstight_nshg' + str(fourfithsshells)+ '.txt')
	calcFsc( options, avgevenmasked, avgoddmasked, fscfilemasked2 )
	
	
	avgevenmasked = avgeven.process('mask.auto3d',{'nshells':1,'nshellsgauss':twothirdshells,'radius':1,'nmaxseed':1,'threshold': eventhresh })
	avgoddmasked = avgodd.process('mask.auto3d',{'nshells':1,'nshellsgauss':twothirdshells,'radius':1,'nmaxseed':1,'threshold': oddthresh })
	
	fscfilemasked3 = fscfile.replace('.txt','_mask_itermediate_nshg' + str(twothirdshells)+ '.txt')
	calcFsc( options, avgevenmasked, avgoddmasked, fscfilemasked3 )
	
	
	avgevenmasked = avgeven.process('mask.auto3d',{'nshells':1,'nshellsgauss':halfnshells,'radius':1,'nmaxseed':1 })
	avgoddmasked = avgodd.process('mask.auto3d',{'nshells':1,'nshellsgauss':halfnshells,'radius':1,'nmaxseed':1 })
	
	fscfilemasked4 = fscfile.replace('.txt','_mask_loose_nshg' + str(halfnshells)+ '.txt')
	calcFsc( options, avgevenmasked, avgoddmasked, fscfilemasked4 )
	
	finalthresh = finalA['sigma']
	
	finalAfilt = finalA.copy()
	finalAfiltLp = finalAfilt.process('filter.lowpass.tanh',{'cutoff_freq':twothirdsnyquist})
	finalAfiltLpNormlocal = finalAfiltLp.process('normalize.local',{'radius':2})
	finalAfiltLpNormlocalMasked = finalAfiltLpNormlocal.process('mask.auto3d',{'nshells':1,'nshellsgauss':twothirdshells,'radius':1,'nmaxseed':1,'threshold': finalthresh })
	
	finalAfiltLpNormlocalMasked.write_image( options.path + '/final_avg_postproc.hdf', 0 )
	
	return finalA


def calcFsc( options, img1, img2, fscfile ):
	
	img1fsc = img1.copy()
	img2fsc = img2.copy()
	
	apix = img1['apix_x']
	
	#if options.clip:
		#img1fsc = clip3D( img1fsc, options.clip )
		#img1fsc.process_inpl
		
	#	img1fsc.write_image(options.path +'/vol4fsc1.hdf',0)
		
	#	img2fsc = clip3D( img2fsc, options.clip )
	#	img2fsc.write_image(options.path +'/vol4fsc2.hdf',0)
		
	fsc = img1fsc.calc_fourier_shell_correlation( img2fsc )
	third = len( fsc )/3
	xaxis = fsc[0:third]
	fsc = fsc[third:2*third]
	saxis = [x/apix for x in xaxis]
	Util.save_data( saxis[1], saxis[1]-saxis[0], fsc[1:-1], fscfile )
	
	return




'''
Function to generate the reference either by reading from disk or bootstrapping
'''
def sptRefGen( options, ptclnumsdict, cmdwp, refinemulti=0, method='',subset4ref=0):
	
	import glob, shutil
	
	refsdict = {}
	elements = cmdwp.split(' ')
	
	#print "elements are", elements
	#print "ptclnumsdict received in sptRefGen is", ptclnumsdict
	#print "RECEIVED CMDWP", cmdwp
	#print 'Therefore elemnts are', elements
	
	#current = os.getcwd()
	
	for klassnum in ptclnumsdict:
		
		klassidref = '_even'
		
		if klassnum == 1:
			klassidref = '_odd'
		
		try:
			if options.goldstandardoff:
				klassidref = ''
		except:
			if refinemulti:
				klassidref = ''
				
		
		if refinemulti:
			zfillfactor = len(str( len( ptclnumsdict )))
			
			#if ptclnumsdict[klassnum]:
			#else:
			
			klassidref = '_' + str( klassnum ).zfill( zfillfactor )
			if len( ptclnumsdict ) < 2:
				klassidref = '_' + str( refinemulti ).zfill( zfillfactor )	
		
		if options.ref: 
			ref = EMData(options.ref,0)

			if options.verbose:
				print "\n(e2spt_classaverage)(sptRefGen) - READ reference and its types and min, max, sigma, mean stats are", options.ref, type(ref), ref['minimum'],ref['maximum'],ref['sigma'],ref['mean']

			if not ref['maximum'] and not ref['minimum']:
				print "(e2spt_classaverage)(sptRefGen) - ERROR: Empty/blank reference file. Exiting."
				sys.exit()
			
			if options.apix:
				ref['apix_x'] = options.apix
				ref['apix_y'] = options.apix
				ref['apix_z'] = options.apix
			
			if int(options.refrandphase) > 0:
				filterfreq =  1.0/float( options.refrandphase )
				ref.process_inplace("filter.lowpass.randomphase",{"cutoff_freq":filterfreq,"apix":ref['apix_x']})
						
				refrandphfile = options.path + '/' + os.path.basename( options.ref ).replace('.hdf','_randPH' + klassidref +'.hdf')
				
				if 'final_avg' in refrandphfile:								#you don't want any confusion between final averages produces in other runs of the program and references
					refrandphfile = refrandphfile.replace('final_avg','ref')

				ref['origin_x'] = 0
				ref['origin_y'] = 0
				ref['origin_z'] = 0
				ref.write_image( refrandphfile, 0 )

			if float(ref['apix_x']) <= 1.0:
				print "\n(e2spt_classaverage)(sptRefGen) - WARNING: apix <= 1.0. This is most likely wrong. You might want to fix the reference's apix value by providing --apix or by running it through e2fixheaderparam.py"
			
			refsdict.update({ klassnum : ref })
			
		
		else:
			ptclnums = ptclnumsdict[ klassnum ]
			#print "Therefore for class", klassnum
			#print "ptclnums len and themsvels are", len(ptclnums), ptclnums
		
			if ptclnums:
				ptclnums.sort()		
			
			try:
				if options.hacref:
					method = 'hac'
			except:
				pass
			
			try:
				if options.btref:
					method = 'bt'
			except:
				pass
			
			try:
				if options.ssaref:
					method = 'ssa'
			except:
				pass
				
			if not method and not options.ref:
				method = 'bt'
				print "\n\n\nbt by default!!!!"
				
			#elif options.hacref:
			if method == 'hac':
		
				if options.verbose:
					print "\n(e2spt_classaverage)(sptRefGen) - Generating initial reference using hierarchical ascendant classification through e2spt_hac.py"
			
				subsetForHacRef = 'spthacrefsubset'+ klassidref + '.hdf'
				
				try:
					os.remove(subsetForHacRef)
				except:
					pass
								
				i = 0
				nptclsforref = 10
				try:
					if options.hacref:
						nptclsforref = options.hacref								
				except:
					if subset4ref:
						nptclsforref=subset4ref
			
				if nptclsforref >= len(ptclnums):
					nptclsforref =  len(ptclnums)
			
				print "Hacreflimit is", nptclsforref
				if nptclsforref < 3:
					print """ERROR: You cannot build a HAC reference with less than 3 particles.
					Either provide a larger --hacref number, a larger --subset number, or provide
					--goldstandardoff"""
				
					sys.exit()
			
			
				i = 0
				while i < nptclsforref :
					a = EMData( options.input, ptclnums[i] )
					a.write_image( subsetForHacRef, i )
					i+=1
			
				niterhac = nptclsforref - 1

				hacelements = []
				for ele in elements:
					if 'saveallpeaks' not in ele and 'raw' not in ele and 'btref' not in ele and 'hacref' not in ele and 'ssaref' not in ele and 'subset4ref' not in ele and 'refgenmethod' not in ele and 'nref' not in ele and 'output' not in ele and 'fsc' not in ele and 'subset' not in ele and 'input' not in ele and '--ref' not in ele and 'path' not in ele and 'keep' not in ele and 'iter' not in ele and 'subset' not in ele and 'goldstandardoff' not in ele and 'saveallalign' not in ele and 'savepreproc' not in ele:
						hacelements.append(ele)
			
				cmdhac = ' '.join(hacelements)
				cmdhac = cmdhac.replace('e2spt_classaverage','e2spt_hac')
			
				if refinemulti:
					cmdhac = cmdhac.replace('e2spt_refinemulti','e2spt_hac')
				
			
				hacrefsubdir = 'spthacref' + klassidref
				
				
				try:
					files=glob.glob(hacrefsubdir+'*')		
					print "files are", files
					for path in files: 
						shutil.rmtree(path)
				except:
					pass
					
				cmdhac+=' --path=' + hacrefsubdir
				cmdhac+=' --input='+subsetForHacRef
				
				if options.verbose:
					print "\n(e2spt_classaverage)(sptRefGen) - Command to generate hacref is", cmdhac
				
				runcmd( options, cmdhac )
			
				try:
					print "\nmoving hacrefsubdir %s into path %s" %( hacrefsubdir, options.path )
					os.rename( hacrefsubdir, options.path + '/' + hacrefsubdir )
				except:
					print "\nfirst try moving hacrefsubdir %s into path %s failed" %( hacrefsubdir, options.path )
					hacsubdirstem = '_'.join( hacrefsubdir.split('_')[:-1])

					try:
						hacsubdircount = str( int(hacrefsubdir.split('_')[-1])+1)
					except:
						hacsubdircount = '01'
						hacsubdirstem = hacrefsubdir
					

					newhacrefsubdir = hacsubdirstem + '_' + hacsubdircount #if the subdirectory exists, add one to the tag count at the end of the subdirectory name
					try: 
						print "\nmoving hacrefsubdir %s into path %s" %( hacrefsubdir, options.path )
						os.rename( newhacrefsubdir, options.path + '/' + hacrefsubdir )
					except:
						print "\nsecond and final try moving hacrefsubdir %s into path %s failed" %( newhacrefsubdir, options.path )
						sys.exit(1)

				
				
				try:
					os.rename( subsetForHacRef, options.path + '/' + subsetForHacRef )
				except:
					newsubsetcount = '_'.join( subsetForHacRef.split('_')[:-1]) + '_' + str( int(subsetForHacRef.split('_')[-1])+1 )	#if the subdirectory exists, add one to the tag count at the end of the subdirectory name
					os.rename( subsetForHacRef, options.path + '/' + newsubsetcount )
				
				
				findir = os.listdir(options.path)
				if subsetForHacRef in findir:
					currentdir = os.getcwd()
					findircurrent = os.listdir(currentdir)
					if subsetForHacRef in findircurrent:
						newsubsetcount = '_'.join( subsetForHacRef.split('_')[:-1]) + '_' + str( int(subsetForHacRef.split('_')[-1].split('.hdf')[0]) +1 ) +'.hdf'	#if the subdirectory exists, add one to the tag count at the end of the subdirectory name
						print "\ntrying to move new subsetForHacRef into path", newsubsetcount, options.path
						os.rename( subsetForHacRef, options.path + '/' + newsubsetcount )
					else:
						print "\nWARNING subsetForHacRef does not exist in current directory", subsetForHacRef
						
				else:
					os.rename( subsetForHacRef, options.path + '/' + subsetForHacRef )
					print "\nmoving subsetForHacRef into path", subsetForHacRef, options.path
				
				
				if options.verbose:
					print "\n(e2spt_classaverage)(sptRefGen) - Command to generate hacref is", cmdhac
				
				ref = EMData( options.path + '/' + hacrefsubdir +'/final_avg.hdf', 0 )

				refsdict.update({ klassnum : ref })
				
			
			if method == 'ssa':
				if options.verbose:
					print "\n(e2spt_classaverage)(sptRefGen) - Generating initial reference using self symmetry alignment through e2symsearch3d.py"
			
				if options.sym == 'c1' or options.sym == 'C1':
					print """\n(e2spt_classaverage)(sptRefGen) - ERROR: You must provide at least c2 or higher symmetry to use
					--ssaref"""
			
				subsetForSsaRef = 'sptssarefsubset'+ klassidref + '.hdf'
				
				try:
					os.remove( subsetForSsaRef )
				except:
					pass
								
				nptclsforref = 10
				try:
					if options.ssaref:
						nptclsforref=options.ssaref	
				except:
					if subset4ref:
						nptclsforref=subset4ref
			
				if nptclsforref >= len(ptclnums):
					nptclsforref =  len(ptclnums)
			
				i = 0
				while i < nptclsforref :
					a = EMData( options.input, ptclnums[i] )
					a.write_image( subsetForSsaRef, i )
					i+=1
			
				ssarefsubdir = 'sptssaref' + klassidref
				
				try:
					files=glob.glob(ssarefsubdir+'*')
					for path in files:
						shutil.rmtree(path)
				except:
					pass
					
						
				ssaelements = []
				print "\nelements are", elements
				for ele in elements:
					if 'saveallpeaks' not in ele and 'fine' not in ele and 'raw' not in ele and 'btref' not in ele and 'hacref' not in ele and 'ssaref' not in ele and 'subset4ref' not in ele and 'refgenmethod' not in ele and 'nref' not in ele and 'sfine' not in ele and 'procfine' not in ele and 'fsc' not in ele and 'output' not in ele and 'path' not in ele and 'goldstandardoff' not in ele and 'saveallalign' not in ele and 'savepreproc' not in ele and 'align' not in ele and 'iter' not in ele and 'npeakstorefine' not in ele and 'precision'not in ele and '--radius' not in ele and 'randphase' not in ele and 'search' not in ele and '--save' not in ele and '--ref' not in ele and 'input' not in ele and 'output' not in ele and 'subset' not in ele:
						ssaelements.append(ele)
						print "appending element",ele
		
				cmdssa = ' '.join(ssaelements)
				print "before replacing program name, cmdssa is", cmdssa

				cmdssa = cmdssa.replace('e2spt_classaverage','e2symsearch3d')
				if refinemulti:
					print "should replace refinemulti"
					cmdssa = cmdssa.replace('e2spt_refinemulti','e2symsearch3d')
			
				cmdssa += ' --input=' + subsetForSsaRef 
				cmdssa += ' --path=' + ssarefsubdir
				cmdssa += ' --symmetrize'
				cmdssa += ' --average'
				
				print "\ncmdssa is", cmdssa
				
				if options.verbose:
					print "\n(e2spt_classaverage)(sptRefGen) - Command to generate ssaref is", cmdssa

				runcmd( options, cmdssa )
				
				ssarefname = 'final_avg.hdf'
					
				try:
					print "\nmoving ssarefsubdir %s into path %s" %( ssarefsubdir, options.path )
					os.rename( ssarefsubdir, options.path + '/' + ssarefsubdir )
				except:
					print "\nfirst try moving ssarefsubdir %s into path %s failed" %( ssarefsubdir, options.path )
					hacsubdirstem = '_'.join( ssarefsubdir.split('_')[:-1])

					try:
						ssasubdircount = str( int(ssarefsubdir.split('_')[-1])+1)
					except:
						ssasubdircount = '01'
						ssasubdirstem = ssarefsubdir
					
					newssarefsubdir = ssasubdirstem + '_' + ssasubdircount #if the subdirectory exists, add one to the tag count at the end of the subdirectory name
					try: 
						print "\nmoving ssarefsubdir %s into path %s" %( ssarefsubdir, options.path )
						os.rename( newssarefsubdir, options.path + '/' + ssarefsubdir )
					except:
						print "\nsecond and final try moving ssarefsubdir %s into path %s failed" %( newssarefsubdir, options.path )
						sys.exit(1)




				findir = os.listdir(options.path)
				if subsetForSsaRef in findir:
					print "tried moving subsetForSsaRef into path but failed", subsetForSsaRef, options.path
					newsubsetcount = '_'.join( subsetForSsaRef.split('_')[:-1]) + '_' + str( int(subsetForSsaRef.split('_')[-1].split('.hdf')[0]) +1 ) +'.hdf'	#if the subdirectory exists, add one to the tag count at the end of the subdirectory name
					os.rename( subsetForSsaRef, options.path + '/' + newsubsetcount )
				else:
					os.rename( subsetForSsaRef, options.path + '/' + subsetForSsaRef )
					print "\nmoving subsetForSsaRef into path", subsetForSsaRef, options.path
				
				
				if options.verbose:
					print "\n(e2spt_classaverage)(sptRefGen) - Command to generate ssaref is", cmdssa

				#p=subprocess.Popen( cmdssa, shell=True,stdout=subprocess.PIPE, stderr=subprocess.PIPE)
				#text=p.communicate()	
				#p.stdout.close()
			
				ref = EMData( options.path + '/' + ssarefsubdir +'/' + ssarefname, 0 )

				refsdict.update({ klassnum : ref })

			#elif not options.hacref and not options.ssaref:				
			if method == 'bt':
		
				nptclsforref = 64
				
				#try:
				#	if options.btref:
				#		nptclsforref = options.btref		
				#except:
				#	if subset4ref:
				#		nptclsforref = subset4ref
			
				#if nptclsforref >= len(ptclnums):
				#	nptclsforref =  len(ptclnums)
				
				
				
				#from e2spt_binarytree import binaryTreeRef
				print "\ninput is", options.input
				print "with nimgs", EMUtil.get_image_count( options.input )
				print "--goldstandardoff is", options.goldstandardoff
				print "len ptclnums is", len(ptclnums)
				
				print "log 2 of that is" 
				print log( len(ptclnums), 2 )
			
				niter = int(floor(log( len(ptclnums), 2 )))
				print "and niter is", niter
				nseed = 2**niter
				print "therefore nseed=2**niter is", nseed
				
				
				#try:
				#	if options.btref:
				#		niter = int(floor(log( options.btref, 2 )))
				#		nseed=2**niter			
				#except:
				#	if subset4ref:
				#		niter = int(floor(log( subset4ref, 2 )))
				#		nseed=2**niter	
				
				
				#if not options.goldstandardoff:
				#	nseed /= 2
				
					
				subsetForBTRef = 'sptbtrefsubset'+ klassidref + '.hdf'
				
				try:
					os.remove( subsetForBTRef )
				except:
					pass
								
				i = 0
				
				
				#print "ptclnums are", ptclnums
				#print "with len", len(ptclnums)
				
				while i < nseed :
					print "i is", i
					a = EMData( options.input, ptclnums[i] )
					a.write_image( subsetForBTRef, i )
					print "writing image %d to file %s, which will contain the subset of particles used for BTA reference building" %(i,subsetForBTRef)
					i+=1

				btelements = []
				#print "elements are", elements
				for ele in elements:
					if 'saveallpeaks' not in ele and 'raw' not in ele and 'btref' not in ele and 'hacref' not in ele and 'ssaref' not in ele and 'subset4ref' not in ele and 'refgenmethod' not in ele and 'nref' not in ele and 'output' not in ele and 'fsc' not in ele and 'subset' not in ele and 'input' not in ele and '--ref' not in ele and 'path' not in ele and 'keep' not in ele and 'iter' not in ele and 'goldstandardoff' not in ele and 'saveallalign' not in ele and 'savepreproc' not in ele:
						#print "added ele", ele
						btelements.append(ele)
					else:
						pass
						#print "skipped ele", ele
			
				cmdbt = ' '.join(btelements)
				cmdbt = cmdbt.replace('e2spt_classaverage','e2spt_binarytree')
			
				#print "wildcard is!", wildcard
				#print "BEFORE replacement", cmdbt
			
				if refinemulti:
					cmdbt = cmdbt.replace('e2spt_refinemulti','e2spt_binarytree')
			
			
				btrefsubdir = 'sptbtref' + klassidref		
				
				try:
					files=glob.glob(btrefsubdir+'*')
					for path in files:
						shutil.rmtree(path)
				except:
					pass
					
						
				cmdbt+=' --path=' + btrefsubdir
				#cmdbt+=' --iter=' + str( niter )
				cmdbt+=' --input=' + subsetForBTRef
				#cmdbt+= ' --nopreprocprefft'
				
				runcmd( options, cmdbt )
				
				#cmdbt+= ' && mv ' + btrefsubdir + ' ' + options.path + '/' + ' && mv ' + subsetForBTRef + ' ' + options.path
			
				
				if options.verbose:
					print "\n(e2spt_classaverage)(sptRefGen) - Command to generate btref is", cmdbt

				try:
					print "\nmoving btrefsubdir into path", btrefsubdir, options.path
					os.rename( btrefsubdir, options.path + '/' + btrefsubdir )
				except:
					print "\nfirst try moving btrefsubdir %s into path %s failed" %( btrefsubdir, options.path )
					
					btsubdirstem = '_'.join( btrefsubdir.split('_')[:-1])

					try:
						btsubdircount = str( int(btrefsubdir.split('_')[-1])+1)
					except:
						btsubdircount = '01'
						btsubdirstem = btrefsubdir
					

					newbtrefsubdir = btsubdirstem + '_' + btsubdircount #if the subdirectory exists, add one to the tag count at the end of the subdirectory name
					try: 
						os.rename( newbtrefsubdir, options.path + '/' + btrefsubdir )
					except:
						print "\nsecond and final try moving btrefsubdir %s into path %s failed" %( newbtrefsubdir, options.path )
						sys.exit(1)
						
								
				
				#cmdbt3 = 'mv ' + subsetForBTRef + ' ' + options.path
				#runcmd( options, cmdbt3 )
				
				findir = os.listdir(options.path)
				if subsetForBTRef in findir:
					print "tried moving subsetForBTRef into path but failed", subsetForBTRef, options.path
					newsubsetcount = '_'.join( subsetForBTRef.split('_')[:-1]) + '_' + str( int(subsetForBTRef.split('_')[-1].split('.hdf')[0]) +1 ) +'.hdf'	#if the subset exists, add one to the tag count at the end of the subdirectory name
					os.rename( subsetForBTRef, options.path + '/' + newsubsetcount )
				else:
					os.rename( subsetForBTRef, options.path + '/' + subsetForBTRef )
					print "\nmoving subsetForBTRef into path", subsetForBTRef, options.path
								
				
			
				#if os.getcwd() not in options.path:
				#	options.path = os.getcwd() + '/' + ptions.path
			
				print "\ncmdbt is", cmdbt
			
				#print "\nfindir are"
				#findir=os.listdir(current)
				#for f in findir:
				#	print f
			
				print "The BT reference to load is in file",  options.path+ '/' +btrefsubdir +'/final_avg.hdf'
				ref = EMData( options.path + '/' + btrefsubdir +'/final_avg.hdf', 0 )

				refsdict.update({ klassnum : ref })
	
	refnames={}
	#if options.savesteps:
	#	if options.goldstandardoff and options.ref:
	#		refname = options.path + '/refinitial.hdf'
	#		ref = refsdict[0]
	#		ref.write_image( refname, 0 )
	#
	#	else:
	#		for klass in refsdict:
	#			refname = options.path + '/refinitial_even.hdf'	
	#			if klass == 1:
	#				refname = options.path + '/refinitial_odd.hdf'
	#			ref = refsdict[ klass ]
	#			ref.write_image(refname, 0)

	return refsdict
	

def runcmd(options,cmd):
	if options.verbose > 9:
		print "(e2spt_classaverage)(runcmd) running command", cmd
	
	p=subprocess.Popen( cmd, shell=True,stdout=subprocess.PIPE, stderr=subprocess.PIPE)
	text=p.communicate()	
	p.stdout.close()
	
	if options.verbose > 8:
		print "(e2spt_classaverage)(runcmd) done"
	
	#if options.verbose > 9:
	#	print text
	
	return 1


def sptOptionsParser( options ):
	try:
		if options.align:
			#print "(e2spt_classaverage) --align to parse is", options.align
			options.align=parsemodopt(options.align)
		elif options.align == 'None' or  options.align == 'none':
			options.align=None
	except:
		if options.verbose > 9:
			print "\nWARNING (might not be relevant): --align not provided"
	
	
	#try:
	#	if options.ialign:
	#		#print "(e2spt_classaverage) --align to parse is", options.align
	#		options.ialign=parsemodopt(options.ialign)
	#	elif options.ialign == 'None' or  options.ialign == 'none':
	#		options.ialign=None
	#except:
	#	if options.verbose > 9:
	#		print "\nWARNING (might not be relevant): --ialign not provided"
	
	
		
	try:
		if options.falign and options.falign != None and options.falign != 'None' and options.falign != 'none': 
			options.falign=parsemodopt(options.falign)
		elif options.falign == 'None' or  options.falign == 'none':
			options.falign=None
	except:
		if options.verbose > 9:
			print "\nWARNING (might not be relevant): --falign not provided"
	
	try:
		if options.aligncmp: 
			options.aligncmp=parsemodopt(options.aligncmp)
		elif options.aligncmp == 'None' or  options.aligncmp == 'none':
			options.aligncmp=None
	except:
		if options.verbose > 9:
			print "\nWARNING (might not be relevant): --aligncmp not provided"
	
	try:	
		if options.faligncmp: 
			options.faligncmp=parsemodopt(options.faligncmp)
		elif options.faligncmp == 'None' or  options.faligncmp == 'none':
			options.faligncmp=None
	except:
		if options.verbose > 9:
			print "\nWARNING (might not be relevant): --faligncmp not provided"
		
	try:
		if options.averager: 
			options.averager=parsemodopt(options.averager)
		elif options.averager == 'None' or  options.averager == 'none':
			options.averager=None
	except:
		if options.verbose > 9:
			print "\nWARNING (might not be relevant): --averager not provided"
		
	try:
		if options.autocenter:
			options.autocenter=parsemodopt(options.autocenter)
		elif options.autocenter == 'None' or  options.autocenter == 'none':
			options.autocenter=None
	except:
		if options.verbose > 9:
			print "\nWARNING (might not be relevant): --autocenter not provided"
		
	try:
		if options.autocentermask:
			options.autocentermask=parsemodopt(options.autocentermask)
		elif options.autocentermask == 'None' or  options.autocentermask == 'none':
			options.autocentermask=None
	except:
		if options.verbose > 9:
			print "\nWARNING (might not be relevant): --autocentermask not provided"
	
	try:
		if options.normproc and options.normproc != 'None' and options.normproc != 'none':
			options.normproc=parsemodopt(options.normproc)
		elif options.normproc == 'None' or  options.normproc == 'none':
			options.normproc=None
	except:
		if options.verbose > 9:
			print "\nWARNING (might not be relevant): --normproc not provided"
	
	try:
		if options.mask and options.mask != 'None' and options.mask != 'none':
			#print "parsing mask", sys.exit()
			options.mask=parsemodopt(options.mask)
		elif options.mask == 'None' or  options.mask == 'none':
			options.mask=None
	except:
		if options.verbose > 9:
			print "\nWARNING (might not be relevant): --mask not provided"
	
	try:	
		if options.preprocess and options.preprocess != 'None' and options.preprocess != 'none': 
			options.preprocess=parsemodopt(options.preprocess)
		elif options.preprocess == 'None' or  options.preprocess == 'none':
			options.preprocess=None
	except:
		if options.verbose > 9:
			print "\nWARNING (might not be relevant): --preprocess not provided"
	
	try:	
		if options.threshold and options.threshold != 'None' and options.threshold != 'none': 
			options.threshold=parsemodopt(options.threshold)
		elif options.threshold == 'None' or  options.threshold == 'none':
			options.threshold=None
	except:
		if options.verbose > 9:
			print "\nWARNING (might not be relevant): --threshold not provided"
	
	try:
		if options.preprocessfine and options.preprocessfine != 'None' and options.preprocessfine != 'none': 
			options.preprocessfine=parsemodopt(options.preprocessfine)
		elif options.preprocessfine == 'None' or  options.preprocessfine == 'none':
			options.preprocessfine=None
	except:
		if options.verbose > 9:
			print "\nWARNING (might not be relevant): --preprocessfine not provided"
	
	try:	
		if options.lowpass and options.lowpass != 'None' and options.lowpass != 'none': 
			options.lowpass=parsemodopt(options.lowpass)
		elif options.lowpass == 'None' or  options.lowpass == 'none':
			options.lowpass=None
	except:
		if options.verbose > 9:
			print "\nWARNING (might not be relevant): --lowpass not provided"
	
	try:
		if options.lowpassfine and options.lowpassfine != 'None' and options.lowpassfine != 'none': 
			options.lowpassfine=parsemodopt(options.lowpassfine)
		elif options.lowpassfine == 'None' or  options.lowpassfine == 'none':
			options.lowpassfine=None
	except:
		if options.verbose > 9:
			print "\nWARNING (might not be relevant): --lowpassfine not provided"
	
	try:
		if options.highpass and options.highpass != 'None' and options.highpass != 'none': 
			options.highpass=parsemodopt(options.highpass)
		elif options.highpass == 'None' or  options.highpass == 'none':
			options.highpass=None
	except:
		if options.verbose > 9:
			print "\nWARNING (might not be relevant): --highpass not provided"
	
	try:
		if options.highpassfine and options.highpassfine != 'None' and options.highpassfine != 'none': 
			options.highpassfine=parsemodopt(options.highpassfine)
		elif options.highpassfine == 'None' or  options.highpassfine == 'none':
			options.highpassfine=None
	except:
		if options.verbose > 9:
			print "\nWARNING (might not be relevant): --highpassfine not provided"
	try:
		if options.postprocess and options.postprocess != 'None' and options.postprocess != 'none': 
			options.postprocess=parsemodopt(options.postprocess)
		elif options.postprocess == 'None' or  options.postprocess == 'none':
			options.postprocess=None
	except:
		if options.verbose > 9:
			print "\nWARNING (might not be relevant): --postprocess not provided"
	
	try:
		if options.reconstructor and options.reconstructor != 'None' and options.reconstructor != 'none': 
			options.reconstructor=parsemodopt(options.reconstructor)
		elif options.reconstructor == 'None' or  options.reconstructor == 'none':
			options.reconstructor=None
	except:
		if options.verbose > 9:
			print "\nWARNING (might not be relevant): --reconstructor not provided"
	
	try:
		if options.preavgproc1 and options.preavgproc1 != 'None' and options.preavgproc1 != 'none': 
			options.preavgproc1=parsemodopt(options.preavgproc1)
		elif options.preavgproc1 == 'None' or  options.preavgproc1 == 'none':
			options.preavgproc1=None
	except:
		if options.verbose > 9:
			print "\nWARNING (might not be relevant): --reconstructor not provided"
		
	try:
		if options.preavgproc2 and options.preavgproc2 != 'None' and options.preavgproc2 != 'none': 
			options.preavgproc2=parsemodopt(options.preavgproc2)
		elif options.preavgproc2 == 'None' or  options.preavgproc2 == 'none':
			options.preavgproc2=None
	except:
		if options.verbose > 9:
			print "\nWARNING (might not be relevant): --reconstructor not provided"
	
	return options


'''
Function to write the parameters used for every run of the program to parameters.txt inside the path specified by --path.
*** Imported by many e2spt programs ***
'''
def writeParameters( options, program, tag ):
	print "Tag received in writeParameters is", tag

	names = dir(options)
	
	cmd = program
	lines = []
	now = datetime.datetime.now()
	lines.append(str(now)+'\n')
	
	#print "\nnames are", names
	optionscopy = options
	
	try:
		if options.search == 0 or options.search == 0.0:
			options.search = '0'
	except:
		pass
	try:
		if options.searchfine == 0 or options.searchfine == 0.0:
			options.searchfine = '0'
	except:
		pass
		
	#print "mask in write parameters is", optionscopy.mask, type(optionscopy.mask)
	for name in names:
				
		if getattr(options,name) and "__" not in name and "_" not in name:
		#if "__" not in name and "_" not in name:	
	
			#if "__" not in name and "_" not in name and str(getattr(options,name)) and 'path' not in name and str(getattr(options,name)) != 'False' and str(getattr(options,name)) != 'True' and str(getattr(options,name)) != 'None':			
			line = name + '=' + str(getattr(optionscopy,name))
					
			lines.append(line+'\n')
			
			if str(getattr(optionscopy,name)) != 'True' and str(getattr(optionscopy,name)) != 'False' and str(getattr(optionscopy,name)) != '':
			
				if name != 'parallel':
					if "{" in str( getattr(optionscopy,name) ) or "}" in  str(getattr(optionscopy,name)) or ")" in  str(getattr(optionscopy,name)) or ")"  in str(getattr(optionscopy,name)): 
						
						tail = str( getattr(optionscopy,name) ).replace(':','=').replace('(','').replace(')','').replace('{','').replace('}','').replace(',',':').replace(' ','').replace("'",'')
						if tail[-1] == ':':
							tail = tail[:-1] 
						cmd += ' --' + name + '=' + tail
					else:
						
						tail = str( getattr(optionscopy,name) )
						if tail[-1] == ':':
							tail = tail[:-1]
						cmd += ' --' + name + '=' + tail
						
				else:
					cmd += ' --' + name + '=' + str(getattr(optionscopy,name))
			
			elif str(getattr(optionscopy,name)) == 'True' or str(getattr(optionscopy,name)) == 'False':
				cmd += ' --' + name
	
	parmFile = 'parameters_' + tag + '.txt'
	lines.append('\n'+cmd+'\n')
	#f=open( optionscopy.path + '/' + parmFile,'w')
	pfile = optionscopy.path + '/' + parmFile
	f = open( pfile, 'w')
	f.writelines(lines)
	f.close()
	
	return cmd


def calcAliStep(options):

	print "\n\n(e2spt_classaverage)(calcAliStep) options.radius is", options.radius
	print "(e2spt_classaverage)(calcAliStep) options.input is", options.input

	hdr = EMData( options.input,0,True )
	apix = float( hdr['apix_x'] )
	capix = apix
	fapix =apix
	
	#if options.shrinkfine and float(options.shrinkfine) > 1.0:
	#	apix = 2*apix
	
	factor = factorc = factorf = factorRelative = 1.0
	
	#radPix = radPixC = radPixF = options.radius
	
	if options.shrink and float( options.shrink ) > 1.0:
		print "options.shrink is > 1, see:", options.shrink
		factorc = options.shrink
		capix =  apix*factor
	
	if options.shrinkfine  and float( options.shrinkfine ) > 1.0:
		factorf = options.shrinkfine
		print "options.shrinkfine > 1, see:", options.shrinkfine
		fapix = apix*factorf
	
	#if factorc > 1.0 and factorf > 1.0:										#The relative shrinking factor doesn't really matter
	#	factorRelative = factorc / factorf
	
	print "\n\n\n\nAAAAAAAAAAA\nApix is", apix
	print "\n\n\n\nAAAAAAAAAAA\n"
	
	radPixC = options.radius / capix
	radPixF = options.radius / fapix

	coarseStep1pix =  360.0/(2.0*math.pi*radPixC)
	#coarseStep1pixRounded = math.floor(coarseStep1pix*100.00)/100.00
	
	if options.precision > 1.0:
		coarseStep1pix *= options.precision
	
	fineStep = 360.0/(2.0*math.pi*radPixF)
	if options.precision > 1.0:
		fineStep *= options.precision
	
	fineStepRounded = math.floor(fineStep*100.00)/100.00					#Round fine step DOWN to scan slightly more finally than theoretically needed
	
	rango = 2.0 * fineStep													#Alignment goes from -range (rango) to +range
	#rango = coarseStep / 2.0									
	rangoRounded = math.ceil(rango*100.00)/100.00							#Round the range in fine alignments UP, to cover a slightly larger area than theoretically needed
	
	angularDistance = 2.0*rango												#This is the angular distance between points generated by the sphere aligner for coarse alignment
	angularDistanceRounded = math.floor(angularDistance*100.00)/100.00		#Round angular distance between coarse points down, to sample more tightly
	
	coarseStep = angularDistanceRounded / 2.25								#The 2.25 factor is approximate. The angular distance A between two rotations R1=a1,b1,c1 and R2=a2,b2,c2 
																			#where r1=a1=b1=c1 and r2=a2=b2=c2, for example R1=0,0,0 and R2=12,12,12, is roughly A=(r2-r1)*2.25 
																			#This can be empirically verified with e2spt_transformdistance.py
	
	if coarseStep < coarseStep1pix:
		coarseStep = coarseStep1pix
		print """The coarse step %f was finer than one pixel at the edge of the particle, 
		therefore it will be replaced with %f""" % (coarseStep,coarseStep1pix)
	
	CSrounded = math.floor( coarseStep * 100.00 )/100.00		#Round coarse step DOWN to scan slightly more finally than theoretically needed

	
	print "\n\n*****************"
	print"\n\nThe radius in pixels at size for fine alignment (taking --shrinkfine into account) is", radPixF
	print "Shrink is", options.shrink
	print "Shrink refine is", options.shrinkfine
	print "Therefore, the coarse step and itself rounded are", coarseStep, CSrounded
	print "And the fine step before and after rounding is", fineStep, fineStepRounded
	print "rango and its rounded are", rango, rangoRounded
	print "\n\n*****************\n\n"
	
	searchC = hdr['nx']/2.0 - 2.0
	searchF = 2
	
	if int( options.search ) > 1 :
		searchC = options.search
		print "searchC is", searchC
	else:
		if int(options.search) == 0:
			searchC = 0
			print "searchC is", searchC	

		elif int( options.search ) == 1:
			searchC = 1

		#elif options.mask:
		#	if 'mask.sharp' in options.mask[0]:
		#		if 'outer_radius' in options.mask[1]:
		#			om = options.mask[1]['outer_radius']
		#	
		#			if '-' in str(om):
		#				om = hdr['nx']/2 + om
		#			searchC = om/2.0
		#			print "\nBecause the radius for the mask is", om
		#			print "searchC is", searchC
		#			print "\n"
				
		if options.shrink and float(options.shrink) > 1.0:
			searchC = int( searchC / options.shrink )
	
			print "\nBecause shrink >1.0, searchC is actually", searchC
		
	if options.searchfine:
		searchF = options.searchfine
		print "\nSearchF is", searchF
		print "\n"
	else:
		searchF = 0
	
	options.align = 'rotate_translate_3d:search=' + str(searchC) +':delta=' + str(CSrounded) + ':dphi=' + str(CSrounded)
	if options.sym and options.sym is not 'c1' and options.sym is not 'C1' and 'sym' not in options.align:
		options.align += ':sym=' + options.sym
	
	
	print "inside calcali options.falign received is", options.falign
	if options.falign:
		if options.falign != None and options.falign != 'None' and options.falign != 'none':
			#if options.radius:
			options.falign = 'refine_3d_grid:range=' + str(rangoRounded) + ':delta=' + str(fineStepRounded) + ':search=' + str(searchF)
		else:
			options.falign = 'None'
		
	print "and has been set to", options.falign
	
	
	if options.verbose:
		if options.verbose > 9:
			options.align += ':verbose=1'
			options.falign += ':verbose=1'
	
	return options
	
	

def postprocess(img,mask,normproc,postprocess):
	'''Postprocesses a volume in-place'''
	
	# Make a mask, use it to normalize (optionally), then apply it 
	maskimg=EMData(img["nx"],img["ny"],img["nz"])
	maskimg.to_one()
	if mask:
		maskimg.process_inplace(mask[0],mask[1])
		
	# normalize
	if normproc:
		if normproc[0]=="normalize.mask": 
			normproc[1]["mask"]=maskimg
		img.process_inplace(normproc[0],normproc[1])

	img.mult(maskimg)
	
	# Postprocess filter
	if postprocess: 
		img.process_inplace(postprocess[0],postprocess[1])
	return img


def sptmakepath(options, stem='spt'):
	if options.verbose:
		print "\n(e2spt_classaverage)(sptmakepath), stem is", stem
	
	#if options.path and ("/" in options.path or "#" in options.path):
	#	print "Path specifier should be the name of a subdirectory to use in the current directory. Neither '/' or '#' can be included. "
	#	sys.exit(1)


	if options.path!=None and len(options.path)>0 : stem=options.path
	
	i=1
	while os.path.exists("{}_{:02d}".format(stem,i)): i+=1
	
	options.path="{}_{:02d}".format(stem,i)
	try: os.mkdir(options.path)
	except: pass

	#files=os.listdir(os.getcwd())

	#if not options.path:		
		##options.path = stem + '_01'
		#options.path = stem
		#if options.verbose:
			#print """\n(e2spt_classaverage)(sptmakepath)--path was not specified, 
			#therefore it will have the default value""", options.path 	

	#while options.path in files:
		#if '_' not in options.path:
			#options.path = options.path + '_00'
		#else:
			#jobtag=''
			#components=options.path.split('_')
			#if components[-1].isdigit():
				#components[-1] = str(int(components[-1])+1).zfill(2)
			#else:
				#components.append('00')
						
			#options.path = '_'.join(components)
			##options.path = path
	
	#print "The new options.path is", options.path

	#if options.path not in files:
		#if options.verbose:
			#print "I will make THIS path", options.path
		
		##os.system('mkdir ' + options.path)
		#os.mkdir( options.path )
	
	return options




	

def makeAverage(options,ic,results,it=0):
	
	klassid = '_even'
	if ic == 1:
		klassid = '_odd'
	if options.goldstandardoff:
		klassid = ''
		
	#ptcl_file = options.input
	ptcl_file = options.raw
	path = options.path
	#align_params=results
	averager = options.averager
	saveali = options.saveali
	saveallalign = options.saveallalign
	keep = options.keep
	keepsig = options.keepsig
	symmetry = options.sym
	
	breaksym = options.breaksym
	#options.nocenterofmass
	verbose = options.verbose

	'''Will take a set of alignments and an input particle stack filename and produce a 
	new class-average. Particles may be excluded based on the keep and keepsig parameters. 
	If keepsig is not set, then keep represents an absolute fraction of particles to keep (0-1). 
	Otherwise it represents a sigma multiplier akin to e2classaverage.py.'''
	
	
	print "len of results inside makeAverage is", len(results)
	
	print "(e2pt_classaverage.py)(makeAverage) The results to parse are", 
	
	#for r in results:
	#	print r
	
	#else:
	
	thresh=1.0
	scores = []
	
	#print "results are", results
	#print "\nresults[0] is", results[0]
	#print "\nresults[0][0] is", results[0][0]
	##print "\nresults[0][0[0] is", results[0][0][0]
	#print "\nresults[0][0][0]['xform.align3d'] is", results[0][0][0]["xform.align3d"]
	#print "\nresults[0][-1]", results[0][-1]
	
	
	for indx in results:
		#if 'score' in r[0]:
		#	pass
		#	#print "\nscore!"
		#else:
		#	if r and r[0]:
		#		print "\nIn e2spt_classaverage.py, score not in a non-empty element of p, p[0]", r, r[0]
		#		#print "see, p[0] is", p[0]
		#		sys.exit()
	
		#if r and r[0]:
		#print "\nr is", r
		#print "\nr[0]", r[0]
		#print "\nr[0][0]",r[0][0]
		#print "score in r[0][0]", 'score' in r[0][0]
		#score = r[0][0]['score']
		
		score = results[indx][0]
		scores.append( score )
			#vals.append( score )
				
	if keep < 1.0:
		#print "p[0]['score'] is", align_parms[0]['score']
		print "Len of align_parms is", len(results)
			
			#val=[p[0]["score"] for p in align_parms]
		scores.sort()
		thresh = float( scores[ int( keep * len(scores) ) - 1] )
		if verbose: 
			print "Keep threshold : %f (min=%f  max=%f)"%(thresh,scores[0],scores[-1])
	
	if keepsig:
		# inefficient memory-wise
		
		scores2=[]
		for score in scores:

			scores2.append(score**2)
			
		val=sum(scores)
		val2=sum(scores2)

		mean=val/len(scores)
		sig=sqrt(val2/len(scores)-mean*mean)
		thresh = float( mean + sig * keep )
		if verbose: 
			print "Keep threshold : %f (mean=%f  sigma=%f)"%(thresh,mean,sig)

	"""Make variance image if available"""
	variance = EMData(ptcl_file,0).copy_head()
	if averager[0] == 'mean':
		print "Making variance map"
		averager[1]['sigma'] = variance
	
	print "averager is", averager
	avgr = Averagers.get(averager[0], averager[1])
	included=[]
	
	#print "The path to save the alignments is", path
	#jsdict = path + '/sptali_ir.json'
	#js = js_open_dict(jsdict)
	
	#maxscore = None
	#try:
	maxscore = min( scores )
	
	print "\n(e2spt_classaverage)(makeAverage) maxscore is", maxscore
	
	#except:
	#	print "There are no scores!", scores
		
	writeali = 0
	aliptcls = path + '/aliptcls' + klassid + '.hdf'
	
	weights={}
	
	try:
		if options.saveallalign:
			writeali = 1
			aliptcls = path + '/aliptcls' + klassid + '_' + str(it).zfill( len(str(options.iter)) ) + '.hdf'

		elif saveali and it == options.iter - 1:
			writeali = 1
	except: #The exception should be triggered when e2spt_hac.py is called since it doesn't have the --iter parameter.
		if options.saveali:
			writeali = 1		#This saves the aligned particles across ALL iterations for HAC -probably shouldn't be done.
		
		
	#for i,ptcl_parms in enumerate(align_parms):
	
	ii=0
	for indx in results:
		#ptclindx = r[1]
		ptclindx = indx
		print "ptclindx is", ptclindx
		ptcl = EMData(ptcl_file,ptclindx)
		weight = 1.0
		
		#if r and r[0]:
		
		#ptcl.process_inplace("xform",{"transform":r[0][0]["xform.align3d"]})
		ptcl.process_inplace("xform",{"transform":results[indx][1]})
		#print "I have applied this transform before averaging", ptcl_parms[0]["xform.align3d"]			
	
		#score = r[0][0]["score"]
		score = results[indx][0]
		print "\n!!!score is", score
		print "and thresh is", thresh
		
		if score <= thresh:
			#if thresh != 1.0:
		
			print "particle kept because its score %f is LOWER (which means BETTER, in EMAN2 good scores are more negative) than the threshold %f, when the best score was %f" %( score, thresh, maxscore )
			#else:
			#	print "Particle kept because its score %f is LOWER than the DEFAULT threshold %f, when the best score was %f" %( score, thresh, maxscore )

							
			#print "preavgproc1 and len and type are", options.preavgproc1, len(options.preavgproc1), type(options.preavgproc1)
			#print "preavgproc2 and len are", options.preavgproc2, len(options.preavgproc2),  type(options.preavgproc2)
			
			try:
				if options.preavgproc1:
					ptcl.process_inplace( options.preavgproc1[0], options.preavgproc1[1] )
			except:
				print """ERROR: Preavgproc1 probably requires parameters you did not specify.
					For example, --preavgproc1=threshold.clampminmax.nsigma would fail.
					You need to specify an integer for nsgima, e.g.:
					--preavgproc1=threshold.clampminmax.nsigma:nsimga=2."""	
				sys.exit()		 
			try:
				if options.preavgproc2:
					ptcl.process_inplace( options.preavgproc2[0], options.preavgproc2[1] )
			except:
				print """ERROR: Preavgproc2 probably requires parameters you did not specify.
					For example, --preavgproc2=threshold.clampminmax.nsigma would fail.
					You need to specify an integer for nsgima, e.g.:
					--preavgproc1=threshold.clampminmax.nsigma:nsimga=2."""
				sys.exit()	
					
			if options.weighbytiltaxis:
				print "\n--weighbytiltaxis is", options.weighbytiltaxis
				px = x = int(ptcl['ptcl_source_coord'][0])
				
				tiltaxis = int( options.weighbytiltaxis.split(',')[0] )
				minweight = float( options.weighbytiltaxis.split(',')[1] )
				
				if px > tiltaxis:
					px = -1 *( px - 2*tiltaxis )	#This puts te particle at the same distance from te tilt axis, but to the left of it.
				
				X = tiltaxis				#This models a line in 'weight space' (x, w), that passes through (0, minweight) and ( tiltaxis, maxweight ) 
				W = 1.0 - minweight
				slope = W/X
										#Having the slope of the line and its y-axis (or w-axis in this case) crossing we predict the weight of any particle depending on its dx distance to the tiltaxis
				print "tiltaxis is", X
				print "W is", W
				print "Therefore slope is", slope
				
				dx = tiltaxis - px 
				taweight = slope * px + minweight 
				weight = weight * ( taweight )
				print "tiltaxis weight was %f because it's distance from the tilt axis is %d, because it's x coordinate was %d" % (taweight, dx, x)

			if options.weighbyscore:
				scoreweight = score / maxscore
				print "the score weight is %f because score was %f and the best score was %f" % (scoreweight, score, maxscore )
				weight = weight * scoreweight
			
			weights.update( {ptclindx:weight} )
				
			print "therefore the final weight for particle %d is %f" %(ptclindx, weight )
				
			ptcl.mult( weight )
			avgr.add_image( ptcl )
			included.append( ptclindx )
			print "\nptcl %d added (included in average) because its score %.6f is below (better) the threshold %f" %(ptclindx,score,thresh) 
			
		
		else:
			print "\nptcl %d skipped (not included in average) because its score %.6f is above (worse) the threshold %f" %(ptclindx,score,thresh) 
			weights.update( {ptclindx:0.0} )
		
		#js["subtomo_%04d"%i] = ptcl_parms[0]['xform.align3d']
	
		if writeali:
			ptcl['origin_x'] = 0
			ptcl['origin_y'] = 0		
			ptcl['origin_z'] = 0
			ptcl['spt_score'] = score
		
			ptcl['xform.align3d'] = Transform()
			#ptcl['xform.align3d'] = r[0][0]["xform.align3d"]
			ptcl['xform.align3d'] = results[indx][1]
			
			#originalindex = i*2
			#if ic == 1:
			#	originalindex = i*2 + 1 
			#if options.goldstandardoff:
			#	originalindex = i
				
			ptcl['spt_indx_original']=ptclindx
		
			ptcl.write_image(aliptcls,ii)
		
		ii+=1
			
					
	#js.close()
	
	if verbose: 
		print "Kept %d / %d particles in average"%(len(included),len(results))

	print "Will finalize average"

	avg=avgr.finish()
	print "done"
	
	
	if avg:	
		if symmetry and not breaksym:
			avg = avg.process('xform.applysym',{'sym':symmetry})
	
		avg["class_ptcl_idxs"]=included
		avg["class_ptcl_src"]=ptcl_file
		avg['spt_multiplicity']=len(included)
		avg['spt_ptcl_indxs']=included
	
		if averager[0] == 'mean':
			varmapname = path + '/class' + klassid + '_varmap.hdf'
			variance.write_image( varmapname , it)
				
		#if not nocenterofmass:
		#	avg.process_inplace("xform.centerofmass")
	
	
		if options.autocenter:
			print "\n\n\n\nYou have selected to autocenter!\n", options.autocenter
		
			avgac = avg.copy()
			if options.autocentermask:
				avgac.process_inplace( options.autocentermask[0],options.autocentermask[1] )
			
			if options.autocenterpreprocess:
				apix = avgc['apix_x']
				halfnyquist = apix*4
				highpassf = apix*a['nx']/2.0
			
				avgac.process_inplace( 'filter.highpass.gauss',{'cutoff_freq':highpassf,'apix':apix})
				avgac.process_inplace( 'filter.lowpass.gauss',{'cutoff_freq':halfnyquist,'apix':apix})
				avgac.process_inplace( 'math.meanshrink',{'n':2})
			
			avgac.process_inplace(options.autocenter[0],options.autocenter[1])
		
			tcenter = avgac['xform.align3d']
			print "Thus the average HAS BEEN be translated like this", tcenter
			avg.transform(tcenter)

		avg['origin_x']=0
		avg['origin_y']=0
		avg['origin_z']=0
	
		return [avg,weights]
	else:
		print "\nERROR: for class %d in iteration %d failed to compute average (the average is empty)" %(ic,it)
		sys.exit()
		return [avg,weights]






def get_results(etc,tids,verbose,nptcls,refmethod=''):
	'''This will get results for a list of submitted tasks. Won't return until it has all requested results.
	aside from the use of options["ptcl"] this is fairly generalizable code.'''
	
	import gc
	gc.collect()
	
	# wait for them to finish and get the results
	# results for each will just be a list of (qual,Transform) pairs
	
	#results=[ [ '' ] ]*nptcls
	results=[ 0 ]*nptcls
	
	#print "tids inside get_results are", tids
	
	if refmethod == 'binarytree':
		results=[0]*len(tids)		# storage for results
	
	ncomplete=0
	tidsleft = tids[:]
	print "before loop, in get_results, tidsleft are", tidsleft
	
	while 1:
		time.sleep(5)
		proglist = etc.check_task(tidsleft)
		nwait = 0
		for i,prog in enumerate(proglist):
			if prog == -1: 
				nwait += 1
			
			if prog == 100:
				r = etc.get_results( tidsleft[i] ) 			# results for a completed task
				
				#print "r is", r
				ptcl = r[0].classoptions['ptclnum']			# get the particle number from the task rather than trying to work back to it
				print "ptcl is", ptcl
				#print "results inside get_results are", results
				
				
				if r[1]['final']:
					results[ptcl] = [ filter(None,r[1]['final']) , ptcl, filter(None,r[1]['coarse']) ]					# this will be a list of (qual,Transform)
					
				#print "ptcl and type are", ptcl, type(ptcl)
				#print "results[ptcl] are", results[ptcl]
				#print "because results are", results
				
				'''
				if savealiparams and results and results[ptcl]:
					xformslabel = 'subtomo_' + str( ptcl ).zfill( len( str(nptcls) ) )
			
					AliParams=results[ptcl][0]['xform.align3d']
					score = float(results[ptcl][0]['score'])
					jsA.setval( xformslabel, [ AliParams , score ] )
				'''
				
				ncomplete+=1
		
		tidsleft=[j for i,j in enumerate(tidsleft) if proglist[i]!=100]		# remove any completed tasks from the list we ask about
		if verbose:
			# removed nwait here. It was giving an incorrect perception, since the tasks are apparently not updating progress as they should
			print "  %d tasks, %d complete        \r"%(len(tids),ncomplete)
			sys.stdout.flush()
	
		if len(tidsleft)==0: break
		
	return filter(None,results)


def wedgestats(volume,angle, wedgei, wedgef, options):
	#print "RECEIEVED, in wedge statistics, angle, wedgei and wedgef", angle, wedgei, wedgef
	vfft = volume.do_fft()
	print "Size of vfft is", vfft['nx'],vfft['ny'],vfft['nz']
	wedge = vfft.getwedge(angle, wedgei, wedgef)
	
	mean = vfft.get_attr('spt_wedge_mean')
	sigma = vfft.get_attr('spt_wedge_sigma')

	wedge.process_inplace('xform.phaseorigin.tocenter')
	symwedge = wedge.process('xform.mirror', {'axis':'x'})
	
	#print "Size of symwedge is", symwedge['nx'],symwedge['ny'],symwedge['nz']
	finalwedge = wedge + symwedge
	
	#finalwedge.process_inplace('threshold.binary',{'value':0.0})

	#print "Size of finalwedge is", finalwedge['nx'],finalwedge['ny'],finalwedge['nz']
	#finalwedge_otherhalf = finalwedge.copy()
	#finalwedge_otherhalf.rotate(0,180,0)
	
	'''
	Compute fft amps of the vol and center
	'''
	print "Size of vfft BEFORE real is", vfft['nx'],vfft['ny'],vfft['nz']
	vfft.ri2ap()
	ampsOrig = vfft.amplitude()
	amps = ampsOrig.process('xform.phaseorigin.tocenter')
	symamps = amps.process('xform.mirror', {'axis':'x'})
	finalamps = amps + symamps
		
	#print "Size of amps is", amps['nx'],amps['ny'],amps['nz']
	
	sigmas = options.aligncmp[1]['sigmas']
	print "Sigmas is", sigmas
	
	thresh = math.pow( mean + sigmas * sigma, 2.0 )
	print "Therefore thresh = mean + (sigmas*sigma)^2 is", thresh
	
	#print "Size of symamps is", symamps['nx'],symamps['ny'],symamps['nz']
	
	ampsThresh = finalamps.process('threshold.belowtozero',{'minval':thresh})
	
	#ampsOrigThresh.ap2ri()
	#ampsOrigThresh.do_iff()

	#print "Size of finalamps is", finalamps['nx'],finalamps['ny'],finalamps['nz']
	
	print "\nType of wedge is", type(finalwedge)
	print "\nType of amps is", type(finalamps)
	print "\nType of ampsThresh is", type(ampsThresh)
	
	if options.writewedge:
		
		completePath = os.getcwd() + '/' + options.path
		print "\nThe list of files in path is", os.listdir( completePath )
		 
		wedgename = os.getcwd() + '/' + options.path + '/wedge.hdf'
		finalwedge.write_image(wedgename,0)
		
		ampsname = os.getcwd() + '/' + options.path +'/fftamps.hdf'
		finalamps.write_image(ampsname,-1)
		
		ampsThreshname = os.getcwd() + '/' + options.path + '/fftampsThresh.hdf'
		ampsThresh.write_image(ampsThreshname,-1)	
	
	return(mean,sigma,thresh)








"""
def preprocfilter(image, options, ptclindx=0, tag='ptcls', coarse='yes', round=-1, finetag=''):
	
	simage = image.copy()
	print "\n(e2spt_classaverage) preprocessing"

	if options.lowpass  == 'None' or options.lowpass == 'none':
		options.lowpass = None
	
	if options.highpass == 'None' or options.highpass == 'none':
		options.highpass = None
	
	if options.preprocess == 'None' or options.preprocess == 'none':
		options.preprocess = None
	
	if finetag:
		try:
			if options.lowpassfine  == 'None' or options.lowpassfine == 'none':
				options.lowpassfine = None
			if options.highpassfine == 'None' or options.highpassfine == 'none':
				options.highpassfine = None
			if options.preprocessfine == 'None' or options.preprocessfine == 'none':
				options.preprocessfine = None
		except:
			print "\nWarning: Ignore if parameters --lowpassfine, --highpassfine, preprocessfine don't exist in the program you ran. Otherwise, something went wrong."
				
	if coarse != 'yes':
		#print "lowpassfine received is", options.lowpass	
		pass
		
	apix = simage['apix_x']
	

	'''
	#Preprocess, lowpass and/or highpass
	'''
	
	if not finetag:
		print "(e2spt_classaverage)(preprocessing) trying filters on coarse alignment image"
		if options.lowpass:
			print "(e2spt_classaverage)(preprocessing) --lowpass provided:", options.lowpass
			simage.process_inplace(options.lowpass[0],options.lowpass[1])
			#fimage.write_image(options.path + '/imgPrepLp.hdf',-1)
	
		if options.highpass:
			print "(e2spt_classaverage)(preprocessing) --highpass provided:", options.highpass
			simage.process_inplace(options.highpass[0],options.highpass[1])
			#fimage.write_image(options.path + '/imgPrepLpHp.hdf',-1)
		
		if options.shrink and int( options.shrink  ) > 1:
			print "(e2spt_classaverage)(preprocessing) --shrink provided:", options.shrink
			simage.process_inplace("math.meanshrink",{"n":options.shrink })
		
		if options.preprocess:
			print "(e2spt_classaverage)(preprocessing) --preprocess provided:", options.preprocess
			simage.process_inplace(options.preprocess[0],options.preprocess[1])
			#fimage.write_image(options.path + '/imgPrep.hdf',-1)
		
	else:
		print "(e2spt_classaverage)(preprocessing) trying filters on fine alignment image"
		try:
			if options.lowpassfine:
				print "(e2spt_classaverage)(preprocessing) --lowpass provided:", options.lowpassfine
				simage.process_inplace(options.lowpassfine[0],options.lowpassfine[1])
				#fimage.write_image(options.path + '/imgPrepLp.hdf',-1)
	
			if options.highpassfine:
				print "(e2spt_classaverage)(preprocessing) --highpass provided:", options.highpassfine
				simage.process_inplace(options.highpassfine[0],options.highpassfine[1])
				#fimage.write_image(options.path + '/imgPrepLpHp.hdf',-1)
		
		
			if finetag != 'noshrink':
				if options.shrinkfine and int( options.shrinkfine  ) > 1 :
					print "(e2spt_classaverage)(preprocessing) --shrink provided:", options.shrinkfine
					simage.process_inplace("math.meanshrink",{"n":options.shrinkfine })
				#fimage.write_image(options.path + '/imgPrepLpHpSh.hdf',-1)
			
			if options.preprocessfine:
				print "(e2spt_classaverage)(preprocessing) --preprocess provided:", options.preprocessfine
				simage.process_inplace(options.preprocessfine[0],options.preprocessfine[1])
				#fimage.write_image(options.path + '/imgPrep.hdf',-1)
		
		except:
			print "filters on fine alignment image failed"
	
	return simage
"""




'''
CLASS TO PARALLELIZE ALIGNMENTS
'''
class Align3DTask(JSTask):
	"""This is a task object for the parallelism system. It is responsible for aligning one 3-D volume to another, with a variety of options"""

	#def __init__(self,fixedimage,image,ptcl,label,mask,normproc,preprocess,lowpass,highpass,npeakstorefine,align,aligncmp,falign,faligncmp,shrink,shrinkfine,transform,verbose,randomizewedge,wedgeangle,wedgei,wedgef):
	def __init__(self,fixedimage,image,ptclnum,label,options,transform,currentIter,nptclsexception=0):
	
		"""fixedimage and image may be actual EMData objects, or ["cache",path,number]
		label is a descriptive string, not actually used in processing
		ptcl is not used in executing the task, but is for reference
		other parameters match command-line options from e2spt_classaverage.py
		Rather than being a string specifying an aligner, 'align' may be passed in as a 
		Transform object, representing a starting orientation for refinement"""
		
		
		#data={}
		
		data={"fixedimage":fixedimage,"image":image}
		
		JSTask.__init__(self,"ClassAv3d",data,{},"")

		#self.classoptions={"options":options,"ptcl":ptcl,"label":label,"mask":options.mask,"normproc":options.normproc,"preprocess":options.preprocess,"lowpass":options.lowpass,"highpass":options.highpass,"npeakstorefine":options.npeakstorefine,"align":options.align,"aligncmp":options.aligncmp,"falign":options.falign,"faligncmp":options.faligncmp,"shrink":options.shrink,"shrinkfine":options.shrinkfine,"transform":transform,"verbose":options.verbose,"randomizewedge":options.randomizewedge,"wedgeangle":options.wedgeangle,"wedgei":options.wedgei,"wedgef":options.wedgef}
		self.classoptions={"options":options,"ptclnum":ptclnum,"label":label,"transform":transform,"currentIter":currentIter,'nptclsexception':nptclsexception}
	
	def execute(self,callback=None):
		"""This aligns one volume to a reference and returns the alignment parameters"""
		classoptions=self.classoptions

		#print "classptclnum received", classoptions['ptclnum']
		
		if isinstance(self.data["fixedimage"],EMData):
			fixedimage=self.data["fixedimage"]
		else: 
			fixedimage=EMData(self.data["fixedimage"][1],self.data["fixedimage"][2])
		
		if isinstance(self.data["image"],EMData) :
			image=self.data["image"]
		else: 
			#print "\nImage to align is", self.data["image"][1], self.data["image"][2]
			#print "Inside path", classoptions['options'].path
			image=EMData(self.data["image"][1],self.data["image"][2])
		
		
		#print "With image sizes img and fixedimg", image['nx'],fixedimage['nx']
		
		"""
		CALL the alignment function
		"""
		nptcls = EMUtil.get_image_count(classoptions['options'].input)
		
		if classoptions['nptclsexception']:
			nptcls = classoptions['nptclsexception']
			nptcls = 1
			classoptions['ptclnum'] = 0
			
			#print "classptclnum CHANGED TO", classoptions['ptclnum']
			
		#print "\n(e2spt_classaverage)(Align3DTaks)(execute) nptcls is", nptcls
		#print "classoptions are", classoptions
		
		xformslabel = 'subtomo_' + str(classoptions['ptclnum']).zfill( len( str(nptcls) ) )
		
		refpreprocess=0
		options=classoptions['options']
		
		#Need to use try/except because other alignment programs do not have the --ref option or --refpreprocess
		
		try:	
			if options.refpreprocess:
				refpreprocess=1
			else:
				refpreprocess=0
		except:
			refpreprocess=1
		
		try:
			#print "inside class, options.ref and type and len are", options.ref, type(options.ref), len(str(options.ref))
	
			if not options.ref or options.ref == '':
				refpreprocess=1
				#print "\n\n\n\n(e2spt_classaverage)(Align3DTask) There is no reference; therfore, refpreprocess should be turned on", refpreprocess
				
		except:
			#print "inside class options.ref doesnt exist therefore refpreprocess is 1"
			refpreprocess=1
		
		#After the first iteration, refpreprocess is always on. It should be turned on manually by the user if a non-crystal structure reference is provided.
		currentIter=self.classoptions['currentIter']
		
		try:
			if int(options.iter) > 1 and currentIter > 0:
				refpreprocess=1
		except:
			refpreprocess=1
			
		if options.verbose:
			print "\n\n!!!!!!!!!!!!!!!!!!!!!!!!\n(e2spt_classaverage)(Align3DTask) ",classoptions['label']
			#print "\n\!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!n\n\n\n\n\n\n"
		
		
		#print "SENDING reference image in alignment and values are", fixedimage, fixedimage['minimum'],fixedimage['maximum'],fixedimage['sigma'],fixedimage['mean']
		
		if not fixedimage['maximum'] and not fixedimage['minimum']:
			#print "Error. Empty reference."
			sys.exit()
		
		if nptcls == 1:
			classoptions['ptclnum'] = 0
		
		print "\n in alignment class, before alignment function fixedimage and image sizes are", fixedimage['nx'], fixedimage['ny'], fixedimage['nz'], image['nx'], image['ny'], image['nz']

		ret = alignment(fixedimage,image,classoptions['label'],classoptions['options'],xformslabel,classoptions['currentIter'],classoptions['transform'],'e2spt_classaverage',refpreprocess)
		
		bestfinal=ret[0]
		bestcoarse=ret[1]

		if nptcls == 1:
			#print "\n\n\nCCCCCCCCCCCCCCC\nRet from alignment is", ret
			#print "therefore received from alignment in class"
			#print "best final", bestfinal
			#print "and best coarse", bestcoarse
			
			pass
			
			#sys.exit()
		
		return {"final":bestfinal,"coarse":bestcoarse}

'''
FUNCTION FOR RUNNING ALIGNMENTS WITHOUT PARALLELISM
'''
def align3Dfunc(fixedimage,image,ptclnum,label,options,transform,currentIter):
	"""This aligns one volume to a reference and returns the alignment parameters"""

	if options.verbose: 
		#print "(e2spt_classaverage)(align3Dfunc) Aligning ",label
		pass
		
	#print "In align3Dfunc fixed image and its type are" , fixedimage, type(fixedimage)
	if type(fixedimage) is list:
		fixedimage=EMData(fixedimage[1],fixedimage[2])
	
	if type(image) is list:
		image=EMData(image[1],image[2])

	"""
	CALL the alignment function
	"""
	
	nptcls = EMUtil.get_image_count(options.input)
	xformslabel = 'subtomo_' + str(ptclnum).zfill( len( str(nptcls) ) )
	
	refpreprocess=0
	
	
	#print "Inside func, options.ref and type and len are", options.ref, type(options.ref), len(str(options.ref))
	if not options.ref or options.ref == '':
		refpreprocess=1
		#print "\n(e2spt_classaverage)(align3Dfunc) There is no reference; therfore, refpreprocess should be turned on", refpreprocess

	if options.refpreprocess:
		refpreprocess=1
	
	try:
		if int(options.iter) > 1 and currentIter > 0:
			refpreprocess=1
	except:
		refpreprocess=1
	
	#print "SENDING reference image in alignment and values are", fixedimage, fixedimage['minimum'],fixedimage['maximum'],fixedimage['sigma'],fixedimage['mean']
	

	ret=alignment(fixedimage,image,label,options,xformslabel,currentIter,transform,'e2spt_classaverage',refpreprocess)

	bestfinal=ret[0]
	bestcoarse=ret[1]
	
	#print "\n\n\nReceived from alignment in 3dfunc, bestfinal", bestfinal
	
	if not fixedimage['maximum'] and not fixedimage['minimum']:
		print "Error. Empty reference."
		sys.exit()
	
	
	return {"final":bestfinal,"coarse":bestcoarse}


'''
FUNCTION THAT DOES THE ACTUAL ALIGNMENT OF TWO GIVEN SUBVOLUMES -This is also used by e2spt_hac.py, 
e2spt_binarytree.py and e2spt_refinemulti.py, any modification to it or its used parameters 
should be made with caution
'''
def alignment( fixedimage, image, label, options, xformslabel, iter, transform, prog='e2spt_classaverage', refpreprocess=0 ):
	
	gc.collect() 	#free up unused memory
	
	from operator import itemgetter	
	
	if options.verbose: 
		print "\n\n!!!!\n(e2spt_classaverage)(alignment) Aligning ",label
		#print "\n\!!!!!n\n\n\n\n\n\n"
	
	round=iter
	#print "label is", label
	try:
		ptclindx = int(label.split(' ')[1])
		refindx = -1
	except:
		try:
			ptclindx = int( image['spt_ptcl_indxs'] )							#Deal with --savepreproc for e2spt_hac.py
			refindx = int( fixedimage['spt_ptcl_indxs'] )
			#round = int(xformslabel.split('_')[0].replace('round',''))
		except:
			try:
				ptclindx = int( label.split('#')[-1].split(' ')[0] )			#Deal with --savepreproc for binary tree initial model building (do NOT savepreproc)
				refindx = ptclinx +1
				round = -1
				
				ptclindx = refindx =-1
				
			except:
				ptclindx = refindx = 0
	
	
	#print "Using this ptcl indx in alignment!", ptclindx
	'''	
	PREPROCESSING CALL 
	Currently applied to both volumes. Often 'fixedimage' will be a reference, so may need to rethink whether it should be treated identically. 
	Similar issues in 2-D single particle refinement ... handled differently at the moment
	'''	
	
	#########################################
	#Preprocess the reference or "fixed image", only if rotate_translate_3d_tree is not used
	#########################################
	reffullsize = fixedimage.copy()
	
	sfixedimage = fixedimage.copy()

	s2fixedimage = fixedimage.copy()
	
	if options.falign and 'rotate_translate_3d_tree' not in options.align[0]:
		if options.ref:
			try:
				s2fixedimage = EMData( options.path + '/' + options.ref.replace('.hdf','_preprocfine.hdf'), 0 )
			except:
				s2fixedimage = reffullsize
				print ("\n(e2spt_classaverage)(alignment) preprocessed fine reference not found. Using raw reference")
		else:
			s2fixedimage = reffullsize
			print ("\n(e2spt_classaverage)(alignment) preprocessed particle not found. Using raw version")

	print "\n(e2spt_classaverage)(alignment) s2fixedimage starts with size", s2fixedimage['nx']		
	print "\n(e2spt_classaverage)(alignment) before refpreprocess, refpreprocess, iter", refpreprocess, iter
	
	if 'rotate_translate_3d_tree' not in options.align[0]:	
		
		"""
		if not refpreprocess:
			print "\nthere is NO refpreprocess! But an external reference WAS provided, type, len", options.ref
	
			if options.shrink and int(options.shrink) > 1:
				print "shrinking sfixedimage BEFORE, iter", sfixedimage['nx'], iter
				sfixedimage = sfixedimage.process('math.meanshrink',{'n':options.shrink})
				print "shrinking sfixedimage AFTER, iter", sfixedimage['nx'], iter
		
	
			if options.falign and options.falign != None and options.falign != 'None' and options.falign != 'none':
				print "there's fine alignment"
				if options.procfinelikecoarse:
					print "there's procfinelikecoarse"
					s2fixedimage = sfixedimage.copy()
		
				elif options.shrinkfine and int(options.shrinkfine) > 1:
					s2fixedimage = s2fixedimage.process('math.meanshrink',{'n':options.shrinkfine})
					print "shrinking reference for fine alignment!!!!, iter", options.shrinkfine, s2fixedimage['nx'], iter

			else:
				#s2fixedimage = sfixedimage.copy()
				#s2fixedimage = fixedimage.copy()
				print "no falign, therefore s2image is None"
				s2fixedimage = None


		elif refpreprocess:
			print "there is refpreprocess, iter", iter
			savetag = 'ref'
			if 'odd' in label or 'even' in label:
				savetag = ''
		
			if options.threshold or options.normproc or options.preprocess or options.lowpass or options.highpass:
				#print "\nThere IS refpreprocess!"	
		
				print "BEFORE preprocessing coarse ref, because there is refpreprocess, size is %d, iter %d" %( sfixedimage['nx'],iter )
				
				sfixedimage = preprocfilter(sfixedimage, options, refindx, savetag ,'yes',round)
				print "AFTER preprocessing coarse ref, because there is refpreprocess, size is %d, iter %d" %( sfixedimage['nx'],iter )

			#Only preprocess again if there's fine alignment, AND IF the parameters for fine alignment are different
		
			if options.falign and options.falign != None and options.falign != 'None' and options.falign != 'none':
				#if options.procfinelikecoarse:
				#	s2fixedimage = sfixedimage.copy()
				#	print "REFERENCE fine preprocessing is equal to coarse"
		
				print "procfinelikecoarse is", options.procfinelikecoarse
		
				if options.procfinelikecoarse:
					s2fixedimage = sfixedimage.copy()
		
				elif options.preprocessfine or options.lowpassfine or options.highpassfine or int(options.shrinkfine) > 1:
					print "BEFORE preprocessing fine ref, because there is refpreprocess, size is %d, iter %d" %( s2fixedimage['nx'],iter)
					
					s2fixedimage = preprocfilter(s2fixedimage,options,refindx, savetag ,'no',round,'fine')
	
					print "AFTER preprocessing fine ref, because there is refpreprocess, size is %d, iter %d" %( s2fixedimage['nx'],iter)

			else:
				#s2fixedimage = sfixedimage.copy()
				#s2fixedimage = fixedimage.copy()	
				s2fixedimage = None

		"""
		
		
		if sfixedimage:
			if options.verbose:
				print "\n(e2spt_classaverage)(alignment) COARSE ref is of size %d, in iter %d" %( sfixedimage['nx'], iter)

		if s2fixedimage:
			if options.verbose:
				print "\n(e2spt_classaverage)(alignment) FINE ref is of size %d, in iter %d" %( s2fixedimage['nx'], iter)
	
		if reffullsize:
			if options.verbose:
				print "\n(e2spt_classaverage)(alignment) REFFULLSIZE is of size %d, in iter %d" %( reffullsize['nx'], iter)
		
	
	#########################################
	#Preprocess the particle or "moving image", only if rotate_translate_3d_tree is not used
	#########################################
	imgfullsize = image.copy()
	
	simage = image.copy()
	
	s2image = image.copy()
	
	if options.falign and 'rotate_translate_3d_tree' not in options.align[0]:
		try:
			s2image = EMData( options.path + '/' + options.input.replace('.hdf','_preprocfine.hdf'), 0 )
		except:
			print ("\n(e2spt_classaverage)(alignment) preprocessed fine particle not found. Using raw version")
			s2image = imgfullsize

	
	"""
	if options.clip:
		if simage['nx'] != options.clip or simage['ny'] != options.clip or simage['nz'] != options.clip:
			simage = clip3D( simage, options.clip )
			
			print "\nclipped simage to", options.clip, simage['nx']

		
		if s2image['nx'] != options.clip or s2image['ny'] != options.clip or s2image['nz'] != options.clip:
			s2image = clip3D( s2image, options.clip )
		
		if imgfullsize['nx'] != options.clip or imgfullsize['ny'] != options.clip or imgfullsize['nz'] != options.clip:
			imgfullsize = clip3D( imgfullsize, options.clip )
			
			print "\nclipped s2image to", options.clip, s2image['nx']

	
	
	savetagp = 'ptcls'
	if 'odd' in label or 'even' in label:
		savetag = ''
	
	
	
	if 'rotate_translate_3d_tree' not in options.align[0]:
		print "not using tree aligner, aligner --align is", options.align
		#print "options.threshold",options.threshold
		#print "options.normproc",options.normproc
		#print "options.mask",options.mask
		#print "options.preprocess",options.preprocess
		#print "options.lowpass",options.lowpass
		#print "options.shrink",options.shrink
		
		
		if options.threshold or options.normproc or options.preprocess or options.lowpass or options.highpass:

			print "\n\n\n\n\n\n\n\n\n\n\nSending moving particle to preprocessing. It's size is", simage['nx'],simage['ny'],simage['nz']
			
			if options.verbose > 9:
				print "preprocfilter specified"
				
			simage = preprocfilter(simage,options,ptclindx, savetagp ,'yes',round)
		else:
			pass
		
		print "preprocessed moving particle has size", simage['nx']

		#Only preprocess again if there's fine alignment, AND IF the parameters for fine alignment are different

		print "options.falign is", options.falign

		
			#print "no preproc specified"
			
		print "AAA"
		
		if options.falign and options.falign != None and options.falign != 'None' and options.falign != 'none': 
			#print "somehow there's falign", options.falign
			if options.procfinelikecoarse:
				s2image = simage.copy()
				#print "PARTICLE fine preprocessing is equal to coarse"
			elif options.preprocessfine or options.lowpassfine or options.highpassfine or int(options.shrinkfine) > 1:
			
				s2image = preprocfilter(s2image,options,ptclindx, savetagp ,'no',round,'fine')
				#print "There was fine preprocessing"
			#sys.exit()
		else:
			#s2image = simage.copy()
			#s2image = image.copy()
			s2image = None
			#print "no falign, therefore s2image is None"

		print "BBB"
		"""
		
	if simage:
		if options.verbose:
			print "after all preproc, COARSE ptcl is of size %d, in iter %d" %( simage['nx'], iter)

	if s2image:
		if options.verbose:
			print "after all preproc, FINE ptcl is of size %d, in iter %d" %( s2image['nx'], iter)	

	if imgfullsize:
		if options.verbose:
			print "after all preproc, IMGFULLSIZE is of size %d, in iter %d" %( imgfullsize['nx'], iter)
		
		
		#print "falign is", options.falign
	
	
	
	if sfixedimage['nx'] != simage['nx']:
		print "ERROR: preprocessed images for coarse alignment not the same size, sfixedimage, simage", sfixedimage['nx'], simage['nx']
		#print "ERROR: preprocessed images for coarse alignment not the same size"
		sys.exit()
	
	if 'rotate_translate_3d_tree' not in options.align[0]:
		if options.falign and s2fixedimage and s2image:
			if s2fixedimage['nx'] != s2image['nx']:
				print "ERROR: preprocessed images for fine alignment not the same size, s2fixedimage, s2image", s2fixedimage['nx'], s2image['nx']
				#print "ERROR: preprocessed images for fine alignment not the same size"
				sys.exit()
	
	if transform:
		#print "\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\nThere WAS a transform, see", transform
		#print "And its type is", type(transform)
		#if classoptions.verbose:
		#	print "Moving Xfrom", transform
		#options["align"][1]["inixform"] = options["transform"]
		if options.align:
			#print "There was classoptions.align"
			#print "and classoptions.align[1] is", classoptions.align[1]
			if options.align[1]:
				options.align[1]["transform"] = transform
	
		if int(options.shrink) > 1:
			#options["align"][1]["inixform"].set_trans( options["align"][1]["inixform"].get_trans()/float(options["shrinkfine"]) )
			options.align[1]["transform"].set_trans( options.align[1]["transform"].get_trans()/float(options.shrinkfine) )
	
	elif options.randomizewedge:
		rand_orient = OrientGens.get("rand",{"n":1,"phitoo":1})		#Fetches the orientation generator
		c1_sym = Symmetries.get("c1")								#Generates the asymmetric unit from which you wish to generate a random orientation
		random_transform = rand_orient.gen_orientations(c1_sym)[0]	#Generates a random orientation (in a Transform object) using the generator and asymmetric unit specified 
		if options.align and options.align[1]:
			options.align[1].update({'transform' : random_transform})
		else:
			transform = random_transform
	
	bestcoarse=[{"score":1.0e10,"xform.align3d":Transform()}]
	if not options.align and transform:
		#if not transform:
		#	bestcoarse=[{"score":1.0e10,"xform.align3d":Transform()}]
		#else:
		ccf = sfixedimage.calc_ccf( simage )
		locmax = ccf.calc_max_location()
							
		locmaxX = locmax[0]
		locmaxY = locmax[1]
		locmaxZ = locmax[2]
		
		score = ccf.get_value_at( locmaxX, locmaxY, locmaxZ )
		
		bestcoarse=[{"score":score,"xform.align3d":transform}]	

	elif options.align:
		'''
		Returns an ordered vector of Dicts of length options.npeakstorefine. 
		The Dicts in the vector have keys "score" and "xform.align3d" 
		'''
		
		#print "Will do coarse alignment"
		
		#print "\n\n\nRight before COARSE alignment, the boxsize of image is", simage['nx'],simage['ny'],simage['nz']
		#print "Right before COARSE alignment, the boxsize of FIXEDimage is", sfixedimage['nx'],sfixedimage['ny'],sfixedimage['nz']
		#print "Side adjustments will be attempted\n\n\n\n"
		
	
		#print "\n\n\nRight before COARSE ali, AFTER size adj, the boxsize of image is", simage['nx'],simage['ny'],simage['nz']
		#print "Right before COARSE alignment,  AFTER size adj, the boxsize of FIXEDimage is", sfixedimage['nx'],sfixedimage['ny'],sfixedimage['nz']
		
		#if options.align:
		
		if simage['nx'] != sfixedimage['nx'] or simage['ny'] != sfixedimage['ny'] or simage['nz'] != sfixedimage['nz']:
			print "\n\nERROR: COARSE alignment images not the same size"
			#print "\nThe particle's COARSE size is", simage['nx'],simage['ny'],simage['nz']
			#print "\nThe reference's COARSE size is", sfixedimage['nx'],sfixedimage['ny'],sfixedimage['nz']
			sys.exit()	
		
		#some aligners don't have the ability to return 'nbest' answers
	#	try:
	
		#print "\n\noptions.align is", options.align, type(options.align)
		#print "\noptions.align[0]", options.align[0]
		#print "\nsimage and type", simage, type(simage)
		#print "\nsfixedimage and type", sfixedimage, type(sfixedimage)
		
		bestcoarse = simage.xform_align_nbest(options.align[0],sfixedimage,options.align[1],options.npeakstorefine,options.aligncmp[0],options.aligncmp[1])
		print "aligner was", options.align[0]
		print "with parameters",options.align[1]
		print "npeaks",options.npeakstorefine
		print "comparator",options.aligncmp[0]
		print "results are thus", bestcoarse
		
		#except:
		#	bestcoarse = simage.align(options.align[0],sfixedimage,options.align[1],options.npeakstorefine,options.aligncmp[0],options.aligncmp[1])
		
		
		if 'rotate_translate_3d_tree' not in options.align[0]:	
			# Scale translation
			scaletrans=1.0
			if options.falign and options.falign != None and options.falign != 'None' and options.falign != 'none' and options.shrinkfine:
				scaletrans = options.shrink/float(options.shrinkfine)
			elif options.shrink and not options.falign:
				scaletrans=float(options.shrink)
			
			if scaletrans>1.0:
				#print "\n\n\nShrink or shrinkfine are greater than 1 and not equal, and therefore translations need to be scaled!"
				#print "Before, translations are", bestcoarse[0]['xform.align3d'].get_trans()
				#print "Transform is", bestcoarse[0]['xform.align3d']
			
				for c in bestcoarse:
					c["xform.align3d"].set_trans(c["xform.align3d"].get_trans()*scaletrans)
			
				#print "After scaling, translations are", c['xform.align3d'].get_trans()
				#print "Transform is", c['xform.align3d']

			elif options.shrink > 1.0 and options.shrinkfine > 1.0 and options.shrink == options.shrinkfine:
				pass
				#print "\n\nshrink and shrink refine were equal!\n\n"
		else:
			print "\ntree alignment returned this best score and alignment", bestcoarse
		
			
	# verbose printout
	if options.verbose > 1 :
		for i,j in enumerate(bestcoarse): 
			pass
			#print "coarse %d. %1.5g\t%s"%(i,j["score"],str(j["xform.align3d"]))

	besttweak = bestfinal = bestcoarse
	
	if 'rotate_translate_3d_tree' not in options.align[0]:	
		if options.falign and options.falign[0] and options.falign != 'None' and options.falign != 'none' and options.falign[0] != "None" and options.falign[0] != 'none':
			#print "\n(e2spt_classaverage)(alignment) Will do fine alignment, over these many peaks", len(bestcoarse)
			# Now loop over the individual peaks and refine each
			bestfinal=[]
			#besttweak = []
			peaknum=0
			#print "\n(e2spt_classaverage)(alignment) options.falign is", options.falign, type(options.falign)
			for bc in bestcoarse:
			
				options.falign[1]["xform.align3d"] = bc["xform.align3d"]
			
				#print "\n(e2spt_classaverage)(alignment) s2image['nx'] == s2fixedimage['nx']", s2image['nx'] == s2fixedimage['nx'],  s2image['nx'], type(s2image['nx']), s2fixedimage['nx'], type(s2fixedimage['nx'])
				#print "\n(e2spt_classaverage)(alignment) s2image['ny'] == s2fixedimage['ny']", s2image['ny'] == s2fixedimage['ny'],  s2image['ny'], type(s2image['ny']), s2fixedimage['ny'], type(s2fixedimage['ny'])
				#print "\n(e2spt_classaverage)(alignment) s2image['nz'] == s2fixedimage['nz']", s2image['nz'] == s2fixedimage['nz'],  s2image['nz'], type(s2image['nz']), s2fixedimage['nz'], type(s2fixedimage['nz'])
			
				if int(s2image['nx']) != int(s2fixedimage['nx']) or int(s2image['ny']) != int(s2fixedimage['ny']) or int(s2image['nz']) != int(s2fixedimage['nz']):
					print "\n(e2spt_classaverage)(alignment) ERROR: FINE alignment images not the same size"
					#print "\nThe particle's FINE size is", s2image['nx'],s2image['ny'],s2image['nz']
					#print "\nThe reference's FINE size is", s2fixedimage['nx'],s2fixedimage['ny'],s2fixedimage['nz']
					sys.exit('MIE')
			
				ali = s2image.align(options.falign[0],s2fixedimage,options.falign[1],options.faligncmp[0],options.faligncmp[1])
				
				#if not options.tweak:
				
				try: 					
					bestfinal.append({"score":ali["score"],"xform.align3d":ali["xform.align3d"],"coarse":bc})
					#print "\nThe appended score in TRY is", bestfinal[0]['score']					
				except:
					bestfinal.append({"score":1.0e10,"xform.align3d":bc["xform.align3d"],"coarse":bc})
				#print "\nThe appended score in EXCEPT is", bestfinal[0]['score']
				peaknum+=1
			
			if options.verbose:
				pass
				#print "Best final is", bestfinal
															
			
			#print "\n\n\nAfter fine alignment, before SHRINK compensation, the transform is", bestfinal[0]['xform.align3d']		
			if options.shrinkfine>1 :
				
					#C: you need to sort here to be able to tweak productively
					#C: If you just sort 'bestfinal' it will be sorted based on the 'coarse' key in the dictionaries of the list
					#C: because they come before the 'score' key of the dictionary (alphabetically)
				
				
				for c in bestfinal:
					
					#print "fixing these translations", c
					newtrans = c["xform.align3d"].get_trans() * float(options.shrinkfine)
					#print "New trans and type are", newtrans, type(newtrans)
					c["xform.align3d"].set_trans(newtrans)
					
				if options.tweak:
					print "tweak is on"
					bestfinal = sorted(bestfinal, key=itemgetter('score'))
				
					originalLpFine = options.lowpassfine
					if options.lowpassfine:
						originalLpRes = options.lowpassfine[1]['cutoff_freq']
						newres = originalLpRes*2
						options.lowpassfine[1]['cutoff_freq'] = newres
					
					
					
					if reffullsize['nx'] != imgfullsize['nx'] or  reffullsize['ny'] != imgfullsize['ny'] or  reffullsize['nz'] != imgfullsize['nz']:
						print "ERROR: reffullsize and imgfullsize are not the same size" 
						print "reffullsize", reffullsize['nx'], reffullsize['ny'], reffullsize['nz']
						print "imgfullsize", imgfullsize['nx'], imgfullsize['ny'], imgfullsize['nz']
						sys.exit()
			
					print "\ntweaking alignment!"	
					bestT = bestfinal[0]["xform.align3d"]
					bestScore = bestfinal[0]['score']
				
					print "best alignment was", bestT
					tweakrange = options.falign[1]['delta']+0.5
					print "tweaking range is", tweakrange
					tweakdelta = options.falign[1]['delta']/2.0 - 0.1
					print "tweaking step is", tweakdelta

					tweaksearch = options.shrinkfine
			
					if options.lowpassfine:
						print "new options.lowpassfine is", options.lowpassfine
			
					reffullsizeali = reffullsize.copy()
					imgfullsizeali = imgfullsize.copy()
					
					reffullsizeali = preprocfilter(reffullsizeali,options,ptclindx, 'ref' ,'no',round,'noshrink')
					imgfullsizeali = preprocfilter(imgfullsizeali,options,ptclindx, savetagp ,'no',round,'noshrink')
						
						
					print "before alitweak, sizes of img are and apix", imgfullsizeali['nx'], imgfullsizeali['ny'], imgfullsizeali['nz'],imgfullsizeali['apix_x']
					print "before alitweak, sizes of ref are and apix", reffullsizeali['nx'], reffullsizeali['ny'], reffullsizeali['nz'],reffullsizeali['apix_x']
					
					#print "aligner to tweak is", ['refine_3d_grid',{'xform.align3d':bestT,'range':tweakrange,'delta':tweakdelta,'search':tweaksearch}]
					
					alitweak = imgfullsizeali.align('refine_3d_grid',reffullsizeali,{'xform.align3d':bestT,'range':tweakrange,'delta':tweakdelta,'search':2},options.faligncmp[0],options.faligncmp[1])
					
					#alitweak =imgfullsizeali.align('refine_3d_grid',reffullsizeali,{'xform.align3d':bestT,'range':tweakrange,'delta':tweakdelta,'search':tweaksearch},options.faligncmp[0],options.faligncmp[1])
					
					#reffullsizeali.transform( bestT )
					
					#alitweak =imgfullsizeali.align('rotate_translate_3d_grid',reffullsizeali,{'phi0':0, 'phi1':tweakrange,'dphi':tweakdelta,'az0':0,'az1':360,'daz':tweakdelta,'alt0':0,'alt1':tweakrange,'dalt':tweakdelta,'search':tweaksearch},options.faligncmp[0],options.faligncmp[1])
					#alitweak =imgfullsizeali.align('rotate_translate_3d',reffullsizeali,{'range':tweakrange,'delta':tweakdelta,'search':tweaksearch},options.faligncmp[0],options.faligncmp[1])
					
					
					try: 					
						besttweak.append({"score":ali["score"],"xform.align3d":ali["xform.align3d"],"coarse":bc})
						#print "\nThe appended score in TRY is", bestfinal[0]['score']					
					except:
						besttweak.append({"score":1.0e10,"xform.align3d":bc["xform.align3d"],"coarse":bc})
						#print "\nThe appended score in EXCEPT is", bestfinal[0]['score']
					
					
					#if options.sym and options.sym is not 'c1' and options.sym is not 'C1' and 'sym' not in options.align:
					#	options.align += ':sym=' + options.sym
				
					besttweakT = bestT
					besttweakScore = 1.0e10
					try: 
						besttweakScore = alitweak["score"]
						besttweakT = alitweak["xform.align3d"]					
					
						if float( besttweakScore ) < float( bestScore ) and besttweakT != bestT:
					
							print "tweaking improved score from %.6f to %.6f" %( float( bestScore ), float( besttweakScore ) )
							bestfinal[0]['score'] = besttweakScore
							bestfinal[0]["xform.align3d"] = besttweakT
							print "and changed the transform from, to", bestT, besttweakT
						else:
							print "tweaking did not improve score; therefore, it will be ignored."			

					except:
						print "WARNING: tweaking failed!"

					options.lowpassfine = originalLpFine
				
				else:
					print "NOT tweaking!"								
						
					
				
			#print "AFTER fine alignment, after SHRINK compensation, the transform is", bestfinal[0]['xform.align3d']		
			#print "\n\n\n"
		
			#verbose printout of fine refinement
			if options.verbose>1 :
				
				for i,j in enumerate(bestfinal): 
					pass
					#print "fine %d. %1.5g\t%s"%(i,j["score"],str(j["xform.align3d"]))
		
			bestfinal = bestcoarse
			if options.verbose:
				pass
				#print "\nThere was no fine alignment; therefore, score is", bestfinal[0]['score']
		else:
			print "\nno fine alignment", options.falign
	#else: 
	#	bestfinal = bestcoarse
	#	if options.verbose:
	#		pass

	
	if options.tweak:
		bestfinal = besttweak
	
	from operator import itemgetter							#If you just sort 'bestfinal' it will be sorted based on the 'coarse' key in the dictionaries of the list
															#because they come before the 'score' key of the dictionary (alphabetically). Should be sorted already, except if there was no --falign
	bestfinal = sorted(bestfinal, key=itemgetter('score'))

	
	#print "Best final answer determined"
	if options.verbose:
		#print "\nThe best peaks sorted are"	#confirm the peaks are adequately sorted
		#for i in bestfinal:
		#	print i
		pass
	
	if bestfinal[0]["score"] == 1.0e10 and options.falign:
		print "Error: all fine alignments failed for %s. May need to consider altering filter/shrink parameters. Using coarse alignment, but results are likely invalid."%self.options["label"]
	
	if options.verbose: 
		#print "Best %1.5g\t %s"%(bestfinal[0]["score"],str(bestfinal[0]["xform.align3d"]))
		#print "Inside ALIGNMENT function in e2spt_classaverage, done aligning ",label
		pass	
	
	print "\n(e2spt_classaverage)(alignment) finishing - ", label	
	
	#print "\n\n\nRRRRRRRRR\n Returning from alignment", 
	#print "bestfinal",bestfinal
	#print "and bestcorase", bestcoarse
	
	'''
	bestfinal was sorted, but should also sort bestcoarse in case it's used independently later
	'''
	
	bestcoarse = sorted(bestcoarse, key=itemgetter('score'))
	
	gc.collect() 	#free up unused memory
	
	print "(e2spt_classaverage)(alignment) sorted bestcoarse to return", bestcoarse
	
	return [bestfinal, bestcoarse]
	

jsonclasses["Align3DTask"]=Align3DTask.from_jsondict


def classmx_ptcls(classmx,n):
	"""Scans a classmx file to determine which images are in a specific class. Classmx may be a filename or an EMData object.
	returns a list of integers"""
	print "Classmx and its type received in classmx_ptcls are", classmx, type(classmx)
	if isinstance(classmx,str): 
		classmx=EMData(classmx,0)
	
	plist=[i.y for i in classmx.find_pixels_with_value(float(n))]
	
	return plist


	
if __name__ == "__main__":
    main()
    sys.stdout.flush()
