#!/usr/bin/env python

#
# Author: Jesus Galaz, 10/20/2012; last update July/26/2015
# Copyright (c) 2011 Baylor College of Medicine
#
# This software is issued under a joint BSD/GNU license. You may use the
# source code in this file under either license. However, note that the
# complete EMAN2 and SPARX software packages have some GPL dependencies,
# so you are responsible for compliance with the licenses of these packages
# if you opt to use BSD licensing. The warranty disclaimer below holds
# in either instance.
#
# This complete copyright notice must be included in any revised version of the
# source code. Additional authorship citations may be added, but existing
# author citations must be preserved.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or

# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  2111-1307 USA
#
#

import os
from EMAN2 import *
from time import time

import shutil

import matplotlib
matplotlib.use('Agg',warn=False)	 

#import pyemtbx.options
#from pyemtbx.options import intvararg_callback
#from pyemtbx.options import floatvararg_callback
#from optparse import OptionParser


import matplotlib.pyplot as plt
import sys
import numpy

import math	 
from operator import itemgetter
	
from e2spt_intrafsc import genOddAndEvenVols, fscOddVsEven	
	
def main():
	
	progname = os.path.basename(sys.argv[0])
	usage = """WARNING:  **PRELIMINARY** program, still heavily under development. 
				Autoboxes globular particles from tomograms.
				Note that self-generated spherical templates generated by this program
				are 'white'; which means you have to provide the tomogram with inverted (or 'white') contrast, and the same goes for any stacks
				provided (specimen particles, gold, carbon and/or background)."""
			
	parser = EMArgumentParser(usage=usage,version=EMANVERSION)
	
	
	parser.add_argument("--parallel",type=str,default='',help="""Default=Auto. This program will detect the number of CPU cores on your machine and parallelize some of the tasks using all of them. To disable, provide --parallel=None""")
	parser.add_argument("--pad", action='store_true', help="""Provide this if the particles in the --particlestack used to create a template, or the template supplied 
															through --template are in a tight box. The size""",default=False)
	parser.add_argument("--rotsearch", action='store_true', help="""At each translation position, vary euler angles as well when searching for particles.""",default=False)
	
	parser.add_argument('--tiltangles',type=str,default='',help="""File in .tlt or .txt format containing the tilt angle of each tilt image in the tiltseries.""")
	
	parser.add_argument("--tiltaxis",type=str,default='y',help="""Axis to produce projections about. Default is 'y'; the only other valid option is 'x'.""")

	parser.add_argument("--tiltrange", type=float,default=60,help="""Maximum angular value at which the highest tilt picture will be simulated. Projections will be simulated from -tiltrange to +titlrange. For example, if simulating a tilt series collected from -60 to 60 degrees, enter a --tiltrange value of 60. Note that this parameter will determine the size of the missing wedge.""")

	parser.add_argument("--nslices", type=int,default=61,help="""This will determine the tilt step between slices, depending on tiltrange. For example, to simulate a 2 deg tilt step supply --nslices=61 --tiltrange=60. Recall that --tiltrange goes from - to + the supplied value, and that there is a central slice or projection at 0 deg, for symmetrical tilt series.""")	

	parser.add_argument("--pruneccc", action='store_true', help="""Pruned based on ccc mean and sigma.""",default=False)

	parser.add_argument("--nbins", type=int,default=0,help="""Used for histogram plot. Default=0 (not used). Number of bins for histogram. If not provided, the optimal bin number will be automatically calculated based on bin-width, computed using Scott's normal reference rule, width = (3.5*std)/cuberoot(n), where 'std' is the standard deviation of the distribution of scores and n is the number of values considered. Then, bins will be nbins = (max(scores) - min(scores)) / width.""")

	parser.add_argument("--inverttomo", action='store_true', help="""Multiply --tomogram subsections by -1 to invert the contrast BEFORE looking for particles.""",default=False)
	
	parser.add_argument("--inverttemplate", action='store_true', help="""Multiply --template subsections by -1 to invert the contrast BEFORE looking for particles.""",default=False)
	
	parser.add_argument("--simtemplatesn",type=int,default=0,help="""Number of orientations to simulate templates (that is, --template will be put into --simtemplatesn different orientations, and a 'subtomogram' will be simulated for each through projections by calling e2spt_simulation.py using the information in --angles or --tiltrange and --nslices.""")
	
	parser.add_argument("--savepreprocessed",action="store_true",  default=False,help="""Default=False. Will save the intermediate steps of preprocessing of the tomogram.""")
	
	parser.add_argument("--outmode",type=str, default="float", help="All EMAN2 programs write images with 4-byte floating point values when possible by default. This allows specifying an alternate format when supported (int8, int16, int32, uint8, uint16, uint32). Values are rescaled to fill MIN-MAX range.")
	
	parser.add_argument("--normproc", type=str, default='',help="""Default=None. Normalization processor (e2help.py processors, applied through e2proc3d.py) applied to the tomogram before looking for particles.""")
	
	parser.add_argument("--clip", type=str, default='',help="IMOD required for this option. Default=None. Provide x,y,z sizes to clip about the center of the tomogram. For example, specify --clip=2048,2048,250 to clip a tomogram originally sized 4096x4096x500 in half, around the center of the tomogram. To clip only in x and y but keep the full thickness in z, you would say --clip=2048,2048. If one variable is omitted, the default size from the tomogram will be used for that axis.")
	
	parser.add_argument("--threshold", type=str, default='', help="""Default=None. Thresholding processor (e2help.py processors, applied through e2proc3d.py) applied to the tomogram before looking for particles.""")

	parser.add_argument("--automask", action='store_true', default=False, help="""Default=False. Will automask the tomogram 2 sigmas above mean.""")

	#parser.add_argument("--prunerepeated", action='store_true', help="""Prune potential particles that are too close to eachother and thus are likely repeated hits.""",default=False)

	parser.add_argument("--pruneprj", action='store_true', help="""Generate a projection along the z-axis of potential particles, and compare to a projection of the template.""",default=False)

	parser.add_argument("--keep", type=float, default=0.0,help="""Default=0.0 (not used). Percentage of particles, expressed as a fraction, to keep right before writing the coordinates file and output stack.""")

	parser.add_argument("--nkeep", type=int, default=0,help="""Default=0 (not used). Total number of particles to keep right before writing the coordinates file and output stack. Supersedes --keep""")
	
	parser.add_argument("--tomogram", type=str, default='', help="Name of the tomogram.")
	parser.add_argument("--goldstack", type=str, default='', help="Name of the stack containing a few gold particles picked from the tomogram.")
	parser.add_argument("--ptclstack", type=str, default='', help="""Name of the stack containing a few sample particles picked from the tomogram, used to create an initial template.
															with which to search for particles throughout the tomogram.""")
	
	parser.add_argument("--template", type=str, help="""Default=None. Path to file containing the template 
		to search for particles throughout the tomogram. Alternatively, provide --template=sphere 
		to generate a spherical template from scratch. This requires also setting --boxsize and --ptclradius.
		You can also provide --template=cylinder, which generates a solid or hollow cylinder,
		depending on which of the following options you specify:
		--template=cylinder:radius=r:height=h:radiusinner=ri:heightinner=hi.
		radiusinner and heightinner MUST be specified together, and must be smaller
		than radius and height; otherwise they will be defaulted to radius/2 and height/2.""",default='')

	parser.add_argument("--backgroundstack", type=str, help="""Name of the stack containing a few boxes picked from regions of the tomogram where there where no particles, 
															no gold, and no carbon.""",default=None)
	parser.add_argument("--carbonstack", type=str, help="Name of the stack containing a few boxes picked from the grid hole (or carbon).",default=None)

	#parser.add_argument("--output", type=str, help="Name to output the auto-boxed particles.",default='')
	parser.add_argument("--boxsize", type=int, help="Size of the box to put the extracted particles in, and amount by which the subregions will overlap, when searching for particles in the tomogram.", default=0)

	parser.add_argument("--shrinktomo", type=int,  default=0, help="Requires IMOD. Default=0 (no shrinking). Integer factor by which the tomogram will be shrunk.")
	
	parser.add_argument("--subsettrans", type=int, default=0, help="Default=0 (not used). Subset of particles to keep/consider after translational alignment.")
	
	parser.add_argument("--dilutionfactor", type=int, default=4,help="""Default=4. Determines how many particles will be pre-picked as putative particles. For example, if
																if the tomogram is broken up into subregions of volume V to look for particles in each
																and --dilutionfactor=1, then, the number of best-correlating subvolumes from the subregion
																that will be initially selected as potential particles will be n=V/(pv*D) where 'pv' is the volume of one particle
																calculated based on --particleradius or --boxsize, or the template's boxsize ['nx']; 'D' is the dilution factor;
																therefore, the larger D is, the fewer locations that will be initially picked as potential particles (before any pruning).""")

	parser.add_argument("--apix", type=float, help="The actual apix of the tomogram if for some reason it is wrong on the header.", default=0.0)
	
	parser.add_argument("--ptclradius", type=int, help="The estimated radius of the particle in pixels.", default=0)

	parser.add_argument("--cshrink", type=int, help="""If the tomogram was PREVIOUSLY shrunk, --cshrink is the factor by which the tomogram supplied through --tomogram was shrunk with respect to 
														the raw (unshrunk) tomogram. This CAN work in conjuction with --shrinktomo, so be careful. If both parameters are specified,
														the coordinates found by the autoboxer will be multiplied by BOTH factors.""", default=0)

	#parser.add_argument("--boxsize", type=int, help="Boxsize to resize the template, either provided through --template or computed from --particlestack", default=1)

	parser.add_argument("--preprocess",type=str,help="Any processor (as in e2proc3d.py) to be applied to the tomogram", default=None)
	parser.add_argument("--lowpass",type=str,help="A lowpass filtering processor (as in e2proc3d.py) be applied to the tomogram", default=None)
	parser.add_argument("--highpass",type=str,help="A highpass filtering processor (as in e2proc3d.py) to be applied to the tomogram.", default=None)
	parser.add_argument("--mask",action='store_true',help="""If provided, a cylindrical mask will be created to mask out the carbon and keep only the grid hole.
															--gridradius and --gridoffest must be specified.""", default=False)
	
	parser.add_argument("--gridradius", type=int, help="Radius of the grid in pixels. Supply this parameter only if also supplying --mask.",default=0)
	parser.add_argument("--gridoffset", type=str, help="""x,y amount of pixels to translate the cylindrical mask if the carbon hole in your tomogram is off center. 
								The left bottom corner would be 0,0. Supply this parameter only if also supplying 
								--mask and the grid hole is not centered in the tomogram.""", default='')
	
	parser.add_argument("--ppid", type=int, help="Set the PID of the parent process, used for cross platform PPID",default=-1)
	parser.add_argument("--verbose", "-v", dest="verbose", action="store", metavar="n",type=int, default=0, help="verbose level [0-9], higner number means higher level of verboseness")
	parser.add_argument("--path",type=str,default=None,help="""Directory to store results in. The default is a numbered series of directories containing the prefix 'sptautobox'; 
															for example, sptautobox_01 will be the directory by default if 'sptautobox' already exists.""")
	
	#parser.add_option("--outmode",type=str, default='', help="""Default=All EMAN2 programs 
	#	write images with 4-byte floating point values when possible by default. 
	#	This allows specifying an alternate format when supported (int8, int16, int32, uint8, uint16, uint32). 
	#	Values are rescaled to fill MIN-MAX range.""")

	
	#parser.add_argument("--yshort",action="store_true",default=False,help="This means you have provided a --tomogram in which 'y' is the short axis.")
	parser.add_argument("--test",action="store_true",default=False,help="N.A.")
	parser.add_argument("--templatethreshold",type=float,default=0.0,help="""A binary threshold will be applied to the template which will zero out all the densities below the supplied value, 
												and will make the densities above the supplied value equal to one.""")
	parser.add_argument("--goldthreshtomo",action='store_true',default=False,help="""Zero out all densities above mean of max and min.""")

	(options, args) = parser.parse_args()	#c:this parses the options or "arguments" listed 
											#c:above so that they're accesible in the form of option.argument; 
											#c:for example, the input for --template would be accesible as options.template
		
	logger = E2init(sys.argv, options.ppid)	#c:this initiates the EMAN2 logger such that the execution
											#of the program with the specified parameters will be logged
											#(written) in the invisible file .eman2log.txt
	
	if options.verbose:
		print "THE particle radius at reading is!", options.ptclradius
	
	'''
	c:check that the tomogam format is sane, to prevent crashes later.
	'''
	
	if options.keep:
		if options.keep >= 1.0:
			print "\nERROR: --keep needs to be a decimal smaller than 1.0"
			sys.exit()
	
	if '.hdf' not in options.tomogram and '.mrc' not in options.tomogram and '.rec' not in options.tomogram:
			print "\nERROR: The tomogram must be in .mrc, .rec (which is also just an MRC file) or .hdf format."
			sys.exit()
			
	'''
	c:make a directory to store output generated by this program
	'''
	from e2spt_classaverage import sptmakepath
	options = sptmakepath(options,'sptautobox')
	
	rootpath = os.getcwd()
	
	options.path = rootpath + '/' + options.path

	'''
	c:fix the tomogram's header if needed
	'''
	tomohdr = EMData(options.tomogram,0,True)	#c:an EMData object loads the image data into memory
												#c:when the third argument is False (which is default). 
												#c:'True' as the third argument means we only load the img header (a dictionary), 
												#c:opposed to the entire image data
	tomox = tomohdr['nx']
	tomoy = tomohdr['ny']
	tomoz = tomohdr['nz']
	
	if tomoy < tomoz:
		print """\nERROR: the length of the tomogram in y=%d is smaller than the length in z=%d.
		tomograms MUST be ZSHORT and NOT YSHORT. 
		Use e2proc3d.py input output --tomoprep to convert the tomogram to ZSHORT.
		Alternatively, use IMOD clip command as follows at the command line:
		cliop rotx input output. 
		Alternatively, go back to the IMOD etomo workflow and select 'rotate around the x axis' in the
		postprocessing tab during tomographic reconstruction.""" %( tomoy, tomoz )
		sys.exit()
	
	if options.apix:
		if round(float(options.apix),4) != round(float(tomo['apix_x']),4):
		
			tomogramfile = options.path + '/' + os.path.basename( tomogram ).replace('.','_edED.')	#c:this will be the new location of the modified tomogram
			
			#c:fix the apix if the value is not correct and make an 'edited' copy of the tomogram inside --path
			cmd = 'e2spt_fixheaderparam.py --input=' + options.tomogram + ' --output=' + tomogramfile + ' --stem=apix --stemval=' + str(options.apix) + ' --valtype=float'
			runcmd( options, cmd )
			
			#sc:et the origin to zero and overwrite the new edited tomogram file by not providing output
			cmd = 'e2spt_fixheaderparam.py --input=' + tomogramfile + ' --stem=origin --stemval=' + str(options.apix) + ' --valtype=float'
			runcmd( otpions, cmd )
			
			#c:the edited tomogram is now assigned to --tomogram
			options.tomogram = tomogramfile 
	
	'''
	c:preprocess --tomogram if preprocessing options are specified. --template will later be made to match it
	'''
	if options.shrinktomo or options.inverttomo or options.preprocess or options.lowpass or options.highpass or options.normproc or options.threshold or options.mask:
		options = tomogrampreprocess(options)
	
	'''
	c:if parallelism isn't set, parallelize automatically unless disabled
	'''
	if not options.parallel:
		import multiprocessing
		nparallel = multiprocessing.cpu_count()
		options.parallel = 'thread:' + str(nparallel)
		print "\nfound %d cores" %(nparallel)
		print "setting --parallel to", options.parallel
	elif options.parallel == 'None' or options.parallel == 'none':
		print "\nWARNING: parallelism disabled", options.parallel
	
	
	'''
	c:get the template for template matching (read it in and "prepare it", or just generate it from scratch or from the data itself)
	'''
	options = templategen( options )
	
	'''
	c:preprocesstemplate if necessary		
	'''
	#if options.templatemaskthreshold
	
	#if options.templatethreshold:
	#	options.templatethreshold = parsemodopt( options.templatethreshold )	
	
	if options.inverttemplate:
		cmd = 'e2proc3d.py ' + options.template + ' ' + options.template + " --mult -1"
		runcmd( options, cmd )
	
	boxsize = EMData( options.template,0,True)['nx']
	if options.boxsize:
		boxsize = options.boxsize
	
	'''
	c:determine cubical subregions/subblocks to divide the tomogram into
	'''
	tomochunkscenters = tomosubblocks( options, boxsize )
	

	
	
	'''
	c:simulate the template in different orientations with a missing wedge and missing slices
	'''
	if options.simtemplatesn:
		
		import string
		import random
		
		randompath = ''.join(random.choice(string.lowercase) for _ in range(6))
		
		cmd = 'e2spt_simulation.py --input ' + options.template + ' --nptcls ' + str( options.simtemplatesn ) + ' --path ' + randompath + ' --evenorientations'
		
		if options.tiltangles:
			cmd += ' --tiltangles ' + options.tiltangles
		else:
			if options.tiltrange and options.nslices:
				cmd += ' --tiltrange ' + str(options.tiltrange) + ' --nslices ' + str(options.nslices)
			else:
				print "\nERROR: if --simtemplatesn is provided, either --tiltanges OR --tiltrange AND --nslices must be provided"
				sys.exit()				
		
		runcmd( options, cmd )
		
		os.rename( randompath, options.path + '/simtemplates' )

		options.template = options.path + '/simtemplates/simptcls.hdf'
		
		
		
	'''
	c:do the actual "search" of peaks for potential particles, looping over n templates
	'''
	
	n = EMUtil.get_image_count( options.template )
	
	datal = []
	dataf = []
	
	for i in range( n ):
		datas = scanchunks( options, tomochunkscenters, i )		#c:this should return a list of lists [ [score1,x1,y1,z1],[score2,x2,y2,z2], ...,[score3,xn,yn,zn] ]
		
		#print "scanchunks returned",datan
		#print "scanchunks returned"


		'''
		c:a first easy way to remove IDENTICAL repeated particles is to turn the list into a set
		'''
	
		#datas = sorted(datan, key=itemgetter(0))	#?????
		
		if not datas:
			print "WARNING: no particles found for template", i
			break
		
		print "before any pruning, datalen is", len(datas)
		
		#for d in datas:
		#	print d
			
		setdata = []
		for d in datas:
			if ( d[0],d[1],d[2],d[3] ) not in setdata:
				setdata.append( ( d[0],d[1],d[2],d[3] ) )	
			else:
				print "repeated particle when pruning by set data explicitly!"
		
		
		dataf = list(setdata)

		print "\nafter set pruning by set, len data is", len(dataf)

		dataf = sorted(dataf, key=itemgetter(0))
		#print "the final sorted data is",dataf
		#print "final sorted data"
		
		
		####### START OF NON-TESTED SECTION -JULY 2015
		'''
		c:various pruning attempts based on statistics of background, gold and carbon
		'''
		if options.backgroundstack or options.carbonstack or options.goldstack:
			lendata1=len(dataf)
			if options.backgroundstack:
				print "BG Stack to send is", options.backgroundstack
				ret = meancalc(options,options.backgroundstack)
				tag='background'
				dataf = pruner(dataf,tag,ret[0],ret[1],ret[2],ret[3])
		
				lendata2=len(dataf)
				pruned= lendata1-lendata2
				print "These many particles were pruned using the background stack",pruned
	
			lendata1=len(dataf)
			if options.carbonstack:
				print "CARBON Stack to send is", options.carbonstack
				ret = meancalc(options,options.carbonstack)
				tag='carbon'
				dataf = pruner(dataf,tag,ret[0],ret[1],ret[2],ret[3])
		
				lendata2=len(dataf)
				pruned= lendata1-lendata2
				print "These many particles were pruned using the carbon stack",pruned
	
			lendata1=len(dataf)
			if options.goldstack:
				print "GOLD Stack to send is", options.goldstack
				ret = meancalc(options,options.goldstack)
				tag='gold'
				dataf = pruner(dataf,tag,ret[0],ret[1],ret[2],ret[3])
		
				lendata2=len(dataf)
				pruned= lendata1-lendata2
				print "These many particles were pruned using the gold stack",pruned
	
			print "\nthe len of data AFTER pruning using stacks is", len(dataf)
		
		
		
		'''
		c:prune using autoconvolution of z-projections (unaffected by the missing wedge).
		'''
		if options.pruneprj:
			dataf = prjpruner( options, dataf )
		
		
		print "after pruneprj data is",dataf
		print "after pruneprj"
		#sys.exit()
		####### END OF NON-TESTED SECTION -JULY 2015
		
		
		'''
		c:then prune by distance (RMSD) between particles
		'''
		if len(dataf) > 1:
			dataf = list(rmsdpruner( list(dataf),options))
		#'''
		#c:prune using statistics from the data (max, min, std, etc.)
		#'''
		#if options.prunebystats:
		#	dataf = statspruner( options, dataf )
		
				
		scores = []
		for d in range(len(dataf)):
			score = dataf[d][0] 
			scores.append( score )

		
		plothistogram( options, scores )
		
		'''
		c:prune by correlation score, using statistics from a correlation scores histogram
		'''
		if options.pruneccc:
			dataf = cccpruner( options, scores, dataf )
		
		
		'''
		c:add to results found using other templates
		'''
		datal += list( dataf )
	
	
	if not datal:
		print "\nERROR: no particles found"
		sys.exit()
	
	print "Final data to consider after all special pruning is this long", len(datal)	

	finaldata = set()
	nnn=0
	repeat=0
	for d in datal:
		if (d[0],d[1],d[2],d[3]) not in finaldata:
			finaldata.add( (d[0],d[1],d[2],d[3]) )
			nnn+=1
		else:
			repeat+=1
		
	#print "\nYet, I found these many repeated particles, who knows from where", repeat
	#print "\nTHe total number of particles in the SET data should be", nnn
	#print "lets see", len (finaldata)
	
	finaldata = sorted(finaldata, key=itemgetter(0), reverse=True)
			
	
	#print "The sorted subset of data is", data

	#finaldata=list(rmsdpruner(finaldata,options))
	
	print "options.keep", options.keep
	
	if options.keep:
		print "options.keep is True"
	if not options.nkeep:
		print "options.nkeep is False"
	
	if options.keep and not options.nkeep:
		#print "Keeping these many particles based on --keep", options.keep 
		upperlim = int(round( len(finaldata)*options.keep ))
		print "type of final data is", type(finaldata)
		print "upperlim is", upperlim
		print "len of finaldata is", len(finaldata)
		finaldata = finaldata[:upperlim]	

	elif options.nkeep and not options.keep:
		#print "Keeping these many particles based on --nkeep", options.nkeep 
		finaldata = finaldata[:options.nkeep]
		
		print "final sorted selected data is", finaldata
		print "took subset of %d particles" %(options.nkeep)	

	#if options.subsettrans and options.subsettrans < len(finaldata):
	#	finaldata = finaldata[0:options.subsettrans]
	#	print "taking a subset of the data", options.subsettrans, len(finaldata)
	
	if options.verbose:
		print "\nThe program has finished scanning the tomogram for subvolumes"

	coords=set()	
	for i in finaldata:
		#print "The data element to get coords is", data
		x=round(i[1])
		y=round(i[2])
		z=round(i[3])
		c=(x,y,z)
		if c not in coords:
			coords.add(c)
		#print "Therefore coords are", (x,y,z)

	coords=set(coords)
	coords=list(coords)
	coords=sorted(coords, key=itemgetter(0,1,2))

	print "The sorted coords are",
	for c in coords:
		print c
	
	lines = []
	
	shrinkfactor=1
	if options.shrinktomo > 1:
		shrinkfactor *= options.shrinktomo
	if options.cshrink > 1:
		shrinkfactor *= options.cshrink
		
	
	for c in coords:
		x=c[0] * shrinkfactor
		y=c[1] * shrinkfactor
		z=c[2] * shrinkfactor
		
		line = str(x) + ' ' + str(y) + ' ' + str(z) + '\n'
		lines.append(line)
	
	print "Coordinates for THESE many FINAL particles were written", len(coords) 
	coordsname = options.path + '/coords.txt'

	f = open(coordsname,'w')
	f.writelines(lines)
	f.close()

	if options.verbose:
		print "I have written the coordinates to the following file", coordsname 

	return()


'''
c:function to run commands and the command line
'''
def runcmd( options, cmd ):
	if options.verbose > 9:
		print "\n(e2spt_autoboxer)(runcmd) Running command", cmd
	
	p=subprocess.Popen( cmd, shell=True,stdout=subprocess.PIPE, stderr=subprocess.PIPE)
	text=p.communicate()	
	p.stdout.close()
	
	if options.verbose > 9:
		print "\n(e2spt_autoboxer)(runcmd) Done"
	return
	

'''
c:function to preprocess the tomogram if needed/specified
'''
def tomogrampreprocess(options):
	tomogramfile = options.tomogram

	print "\n(e2spt_autoboxer)(tomogrampreprocess) before preprocessing, --tomogram is located at", tomogramfile
	
	
	'''
	c:clip and shrink tomogram if required; we use IMOD since EMAN2 messes up the range of intensities, unsure why
	'''
	
	if options.clip:
		tomohdr = EMData(options.tomogram,0,True)
	
		tomox = tomohdr['nx']
		tomoy = tomohdr['ny']
		tomoz = tomohdr['nz']
		
		try:
			print "Patience. Clipping the tomogram"
			tomogramfile = options.path + '/' + os.path.basename( tomogramfile ).replace('.','_clip.')
			
			clips=options.clip.split(',')
			
			nx=clips[0]
			ny=0
			nz=0
			if len(clips) > 1:
				ny=clips[1]
			if len(clips) > 2:
				nz=clips[2]
			
			if not ny:
				ny = tomoy
			if not nz:
				nz = tomoz
				
			cmd = 'trimvol ' + options.tomogram + ' ' + tomogramfile + ' -nx ' + nx + ' -ny ' + ny + ' -nz ' + nz
			
			#cmd = 'e2proc3d.py ' + options.tomogram + ' ' + tomogramfile + ' --clip ' + options.clip
		
			#if 'float' not in options.outmode:
			#	cmd+= ' --outmode ' + options.outmode
		
			runcmd( options, cmd )
			options.tomogram = tomogramfile
		except:
			print "\nWARNING: NOT clipping tomogram. IMOD needs to be installed to use 'trimvol', which is the program called when --clip is provided."
		
	
	if options.shrinktomo > 1:
		try:
			print "Patience. Shrinking the tomogram"
			tomogramfile = options.path + '/' + os.path.basename( tomogramfile ).replace('.','_sh.')
			#cmd = 'e2proc3d.py ' + options.tomogram + ' ' + tomogramfile + ' --meanshrinkbig ' + str(options.shrinktomo)
		
			cmd = 'binvol ' + options.tomogram + ' ' + tomogramfile + ' -binning ' + str( options.shrinktomo )
			if 'float' not in options.outmode:
				cmd+= ' --outmode ' + options.outmode
		
			runcmd( options, cmd )
		
			if not options.savepreprocessed and '_clip.' in options.tomogram:
				os.remove( options.tomogram )
			
			options.tomogram = tomogramfile
		except:
			print "\nWARNING: NOT shrinking tomogram. IMOD needs to be installed to use 'binvol', which is the program called when --shrinktomo is provided."
	
	
	'''
	c:load the header only, opposed to the actual image data, by specifying "True" as the third argument of EMData()
	'''
	tomohdr = EMData(options.tomogram,0,True)
	
	tomox = tomohdr['nx']
	tomoy = tomohdr['ny']
	tomoz = tomohdr['nz']
	
	tomogramapix = tomohdr['apix_x']
	
	'''
	c:apply any specified filters (preprocess, lowpass and/or highpass)
	'''
	if options.inverttomo:
		print "Patience. Inverting the tomogram's contrast"
		tomogramfile = options.path + '/' + os.path.basename( tomogramfile ).replace('.','_inv.')
		cmd = 'e2proc3d.py ' + options.tomogram + ' ' + tomogramfile + ' --mult=-1'

		if 'float' not in options.outmode:
			cmd+= ' --outmode ' + options.outmode

		runcmd( options, cmd )
		
		if not options.savepreprocessed and ('_clip.' in options.tomogram or '_sh.' in options.tomogram):
			os.remove( options.tomogram )
		
		options.tomogram = tomogramfile
	
	if options.normproc:	
		print "Patience. Normalizing the tomogram."
		tomogramfile = options.path + '/' + os.path.basename( tomogramfile ).replace('.','_norm.')
		cmd = 'e2proc3d.py ' + options.tomogram + ' ' + tomogramfile + ' --process=' + options.normproc
		
		if 'float' not in options.outmode:
			cmd+= ' --outmode ' + options.outmode

		runcmd( options, cmd )
		
		if not options.savepreprocessed and ('_clip.' in options.tomogram or '_sh.' in options.tomogram or '_inv.' in options.tomogram):
			os.remove( options.tomogram )
		
		options.tomogram = tomogramfile
	
	if options.highpass:
		print "Patience. Applying highpass filter to the tomogram:", options.highpass
		tomogramfile = options.path + '/' + os.path.basename( tomogramfile ).replace('.','_hp.')
		cmd = 'e2proc3d.py ' + options.tomogram + ' ' + tomogramfile + ' --process=' + options.highpass +':apix='+str(tomogramapix)
		
		if 'float' not in options.outmode:
			cmd+= ' --outmode ' + options.outmode
		
		runcmd( options, cmd )
		
		if not options.savepreprocessed and ('_clip.' in options.tomogram or '_sh.' in options.tomogram or '_inv.' in options.tomogram or '_norm.' in options.tomogram):
			os.remove( options.tomogram )
		
		options.tomogram = tomogramfile
	
	if options.preprocess:
		print "Patience. Applying this processor to the tomogram:", options.preprocess
		tomogramfile = options.path + '/' + os.path.basename( tomogramfile ).replace('.','_pr.')
		cmd = 'e2proc3d.py ' + options.tomogram + ' ' + tomogramfile + ' --process=' + options.preprocess
		
		if 'float' not in options.outmode:
			cmd+= ' --outmode ' + options.outmode
		
		runcmd( options, cmd ) 
		
		if not options.savepreprocessed and ('_clip.' in options.tomogram or '_sh.' in options.tomogram or '_inv.' in options.tomogram or '_norm.' in options.tomogram or '_hp.' in options.tomogram):
			os.remove( options.tomogram )
		
		options.tomogram = tomogramfile
		
	if options.lowpass:
		print "Patience. Applying lowpass filter to the tomogram:", options.lowpass		
		tomogramfile = options.path + '/' + os.path.basename( tomogramfile ).replace('.','_lp.')
		cmd = 'e2proc3d.py ' + options.tomogram + ' ' + tomogramfile + ' --process=' + options.lowpass + ':apix='+str(tomogramapix)
		
		if 'float' not in options.outmode:
			cmd+= ' --outmode ' + options.outmode
		
		runcmd( options, cmd )
		
		if not options.savepreprocessed and ('_clip.' in options.tomogram or '_sh.' in options.tomogram or '_inv.' in options.tomogram or '_norm.' in options.tomogram or '_hp.' in options.tomogram or '_pr.' in options.tomogram):
			os.remove( options.tomogram )
		
		options.tomogram = tomogramfile
	
	if options.threshold:	
		print "Patience. Thresholding tomogram."
		tomogramfile = options.path + '/' + os.path.basename( tomogramfile ).replace('.','_thresh.')
		cmd = 'e2proc3d.py ' + options.tomogram + ' ' + tomogramfile + ' --process=' + options.threshold
		
		if 'float' not in options.outmode:
			cmd+= ' --outmode ' + options.outmode
		
		runcmd( options, cmd )
		
		if not options.savepreprocessed and ('_clip.' in options.tomogram or '_sh.' in options.tomogram or '_inv.' in options.tomogram or '_norm.' in options.tomogram or '_hp.' in options.tomogram or '_pr.' in options.tomogram or '_lp.' in options.tomogram):
			os.remove( options.tomogram )
		
		options.tomogram = tomogramfile
		
		tomohdr = EMData( options.tomogram,0,True)
		mean=tomohdr['mean']
		std=tomohdr['sigma']
		
		
		#tomogramfile = options.path + '/' + os.path.basename( tomogramfile ).replace('.','_thresh.')
		
		if "belowtozero" not in options.tomogram:
			cmd = 'e2proc3d.py ' + tomogramfile + ' ' + tomogramfile + ' --process=threshold.belowtozero:minval=0'
			
			if 'float' not in options.outmode:
				cmd+= ' --outmode ' + options.outmode
			
			runcmd( options, cmd )
		
		#options.tomogram = tomogramfile
		
		
		#tomogramfile = options.path + '/' + os.path.basename( tomogramfile ).replace('.','_thresh.')
		#cmd = 'e2proc3d.py ' + tomogramfile + ' ' + tomogramfile + ' --process=threshold.abovetozero:maxval=' +str(mean+mean*0.1)
		#runcmd( options, cmd )
		
		#print "mean to binarize is", mean
		#cmd = 'e2proc3d.py ' + tomogramfile + ' ' + tomogramfile + ' --process=threshold.binary:value' +str(mean)
		#runcmd( options, cmd )
	"""	
	if options.automask:
		print "Patience. automasking tomogram."
		tomogramfile = options.path + '/' + os.path.basename( tomogramfile ).replace('.','_autoM3D.')
		tomohdr = EMData( options.tomogram,0,True)
		mean=tomohdr['mean']
		std=tomohdr['sigma']
		threshval = mean
		print "mean, std, thresval are", mean, std, threshval
		cmd = 'e2proc3d.py ' + options.tomogram + ' ' + tomogramfile + ' --process=mask.auto3d:threshold='+str(threshval) +':nmaxseed=1:radius=1:nshells=1:nshellsgauss=1'
		print "cmd is", cmd
		runcmd( options, cmd )
		options.tomogram = tomogramfile
	"""
	
	'''
	c:apply any masks if there are big carbon regions beyond the grid-hole and masking is specified.
	'''
	xo=0
	yo=0
	
	yshort=False
	
	if tomohdr['ny'] < tomohdr['nz']:
		yshort = True
			
	if options.mask and options.gridradius:
		height = tomoz
		
		if yshort:
			height = tomoy
		
		'''
		c:testimage.cylinder only works on cubical boxes. Therefore, make a cubical box to generate the mask.
		'''
		cubem = max(tomox,tomoy,tomoz)
		masksize = cubem
		if options.gridradius*2 > cubem:
			print "WARNING: --gridradius %d is larger than the largest tomogram dimension %d. --gridradius will be set to %d" %(options.gridradius,cubem,cubem)
			masksize = options.gridradius*2
		
		mask = EMData(masksize,masksize,masksize)
		mask.to_one()
		mask.process_inplace('testimage.cylinder',{'radius':options.gridradius,'height':cubem})
		
		mskfileORIG = options.path + '/mskORIG.rec'
		mask.write_image(mskfileORIG,0)
		
		if yshort:
			print "I will rotate the mask"
			#yo*=-1
			mask.rotate(0,-90,0) #-90 rotates clockwise, towards the screen. +90 rotates counterclockwise, into the screen.
		
		if options.gridoffset:
			xo = int(options.gridoffset.split(',')[0])
			yo = int(options.gridoffset.split(',')[-1])
			mask.translate(float(xo),float(yo),0.0)
			mskfileTRANS = options.path + '/mskTRANS.rec'
			mask.write_image(mskfileTRANS,0)
		
		
		'''
		c:clip mask to actual tomogram size
		'''
		r = Region(0,0,0, cubem, cubem, tomoz)
		#e = EMData()
		#e.read_image(tomogram,0,False,r)
		
		#r=Region(-tomox/2,-tomoy/2,-tomoz/2,tomox,tomoy,tomoz)
		maskc = mask.get_clip(r)
		
		if options.verbose:
			print "The dimensions of the mask are", mask['nx'],mask['ny'],mask['nz']
			print "The dimensions of the tomogram are", tomox,tomoy,tomoz
		
		tomo = EMData( options.tomogram, 0 )
		
		tomo.mult(maskc)
		mskfile = options.path + '/mskC.rec'
		maskc.write_image(mskfile,0)
		
		tomogramfile = options.path + '/' + tomogramfile.split('/')[-1].replace('.','_msk.')
		tomo.write_image(tomogramfile,0)
		options.tomogram = tomogramfile
	
	return options

	
'''
c:fuction to 1) read a template from file and make it match the tomogram, 2) generate a
c:template using a particle stack, or 3) generate a (spherical) template from scratch. 
'''
def templategen( options ):
	print "\n(e2spt_autoboxer)(templategen) - loading/generating template"
	
	tomohdr = EMData( options.tomogram,0,True )
	tomogramapix = tomohdr['apix_x']
	tomox = tomohdr['nx']
	tomoy = tomohdr['ny']
	tomoz = tomohdr['nz']
	
	'''
	If --template provided, check for sanity. If insane, fix it. If sane, make a copy of it 
	inside --path in case it needs to be modified later, so as to not overwrite the
	original template source file
	'''
	if options.template:
		
		if options.verbose:
			print "\n(e2spt_autoboxer)(templategen)\n--template is", options.template
		
		if options.template == 'sphere':
			if options.boxsize and options.ptclradius:
				options = gensph( options, tomogramapix )
			else: 
				print "\nERROR: when --template=sphere, you must specify --boxsize AND --ptclradius"
				sys.exit()
				
		elif 'cylinder' in options.template and '.' not in options.template:
			if options.boxsize:
				options = gencylinder( options )
			else: 
				print "\nERROR: when --template=cylinder, you must specify --boxsize"
				sys.exit()	
		
		elif '.hdf' not in options.template[-4:] and '.pdb' not in options.template[-4:] and '.mrc' not in options.template[-4:]:
			print "\nERROR: The format of the template to use must be .pdb, .mrc or .hdf. Alternatively, provide --template=sphere or --template=cylinder"
			sys.exit()
		
		elif '.hdf' not in options.template[-4:]:
			if '.pdb' in options.template[-4:]:
				pdbmodel = options.template
				hdfmodel = options.path + '/template_' + os.path.basename( pdbmodel ).replace('.pdb','.hdf')
				
				cmd = 'e2pdb2mrc.py ' + pdbmodel + ' ' + hdfmodel
				runcmd( options, cmd )
				
				options.template = hdfmodel

			elif '.mrc' in options.template[-4:]:
				mrcmodel = options.template
				hdfmodel = options.path + '/template_' + os.path.basename( mrcmodel ).replace('.mrc','.hdf')
				
				cmd = 'e2proc3d.py ' + mrcmodel + ' ' + hdfmodel
				runcmd( options, cmd )
				
				options.template = hdfmodel
			
			if options.verbose:
				print "\nconverted the template to .hdf"
		elif '.hdf' in options.template[:-4]:
			hdfmodel = options.path + '/template_' + os.path.basename( options.template )
			
			#a = EMData( options.template )
			#a.write_image( hdfmodel, 0 )
			
			shutil.copy2( options.template, hdfmodel )

			options.template = hdfmodel
			
			if not options.rotsearch:
				'''
				c: make a spherical version of --template with the radius of 
				c: gyration of the actual template for purposes of quick translational template matching 
				c: (no rotational search). --template should have already been made to match --tomogram
				c: in terms of scaling, taking -shrinktomo into account if supplied; and --boxsize as well as 
				c: --ptclradius should have already been considered above
				'''
				template = EMData( options.template, 0 )
		
				transtemplate = template.rotavg_i()

				transtemplatename = options.path + '/' + options.template.replace('.hdf','_sph.hdf')	
				transtemplate.write_image( transtemplatename, 0 )
				
				options.template = transtemplatename
		
		
		
		templatehdr = EMData( options.template, 0, True )
		'''
		c:make all sides of the template box equally sized and even, if they're not
		'''
		nx = templatehdr['nx']
		ny = templatehdr['ny']
		nz = templatehdr['nz']
		
		if nx != ny or nx != nz or ny != nz:
			newsize = max( nx, ny, nz )
			
			if newsize % 2:
				newsize += 1
			
			#template = EMData( options.template, 0 )
			#template = clip3d( template, newsize )
			#options.template = options.template.replace('.','_fixed.')
			#template.write_image( options.template,0 )
			
			cmd = 'e2proc3d.py ' + options.template + ' ' + options.template + ' --clip  ' + str( newsize )
			runcmd( options, cmd )
			
			
	
	elif options.ptclstack:
		print "\n--ptclstack provided %s\n" %( options.ptclstack )
	
		ptclhdr = EMData( options.ptclstack, 0, True ) 
		box = ptclhdr['nx']
		boxsize = box
		if options.boxsize:
			boxsize = options.boxsize
		
		radius = int(boxsize)/4 
		
		cmd = "cd " + options.path + " && e2spt_classaverage.py --path=genref --input=../" + options.ptclstack + " --npeakstorefine=4 -v 0 --mask=mask.sharp:outer_radius=-2 --lowpass=filter.lowpass.gauss:cutoff_freq=.0333 --lowpassfine=filter.lowpass.gauss:cutoff_freq=.05 --search " + str( radius ) + " --radius " + str( radius ) + " --precision=2 --parallel" + options.parllel + " --savesteps --saveali --normproc=normalize.edgemean"
		if options.verbose:
			print "\ngenerating template from --particlestack %s by executing the following command \n%s"%( options.ptclstack, cmd )
	
		runcmd( options, cmd )
		
		shutil.copy2( options.path + "/genref/final_avg.hdf", options.path + '/template_ptclavg.hdf' )
		
		options.template = options.path + '/template_ptclavg.hdf'
		
	elif options.boxsize and options.ptclradius:
		options = gensph( options, tomogramapix )
	
	else:
		print """\nERROR: in the absence of --template and --ptclstack, you must provide --boxsize AND --ptclradius to have a spherical template generated automatically."""
		sys.exit()
	
	
	'''
	c:if the apix of --template and --tomogram don't match, scale --template to match --tomogram.
	c:this assumes both have accurate apix values in the header to begin with.
	'''
	print "\n(e2spt_autoboxer)(templategen) options.template is", options.template
	templatehdr = EMData( options.template, 0, True )
	templateapix = templatehdr['apix_x']
	
	if round(float(templateapix),4) != round(float(tomogramapix),4):
		
		cmd = 'e2match.py --img2process=' + options.template + ' --apix' + str( round(float(tomogramapix), 4))
		runcmd( options, cmd ) 
	
	return options


'''
c:function to generate spherical template
'''
def gensph( options, tomogramapix ):
	
	print "\n(e2spt_autoboxer)(gensph) generating spherical template from scratch"
	
	#tomogramapix = EMData( options.tomogra, 0, True)['apix_x']
		
	a = EMData( options.boxsize, options.boxsize, options.boxsize )
	a.to_one()
	#a.process_inplace('mask.sharp',{'outer_radius':-2})
	a['apix_x'] = tomogramapix
	a['apix_y'] = tomogramapix
	a['apix_z'] = tomogramapix
	
	if options.ptclradius:
		a.process_inplace('mask.sharp',{'outer_radius':options.ptclradius})
		a.process_inplace('normalize.edgemean')
	else:
		print "\nERROR: in the absence of --template and --ptclstack, you must provide --boxsize AND --ptclradius."
		sys.exit()
	
	a.write_image(options.path + '/templatesph.hdf',0)
	#if not options.invert:
	#	a.mult(-1)		
	options.template = options.path + '/templatesph.hdf'
	print "writing spherical template", options.template
	
	return options


'''
c:function to generate cylindrical template
'''
def gencylinder( options ):
		
	options.template = parsemodopt( options.template )	#this parses --template into ("option_name",{"parameter_1":value_1,"parameter_2":value_2, ...,"parameter_n":value_n})
														#e.g., ("cylinder",{"radius":32,"radiusinner":16,"height":50,"heightinner":40})
	params = options.template[1]	
	
	box = options.boxsize
	
	mask = EMData( box, box, box)
	mask.to_one()
	
	radius = int(box)/2
	if 'radius' in params:
		radius = int(params['radius'])
		
	height = int(box)/2	
	if 'height' in params:
		height = int(params['height'])
	
	maskout = mask.process("testimage.cylinder",{'height':height,'radius':radius})
	
	finalmask = maskout
	
	if all (k in params for k in ('heightinner','radiusinner')):
		heightinner = params['heightinner']
		radiusinner = params['radiusinner']
		
		if heightinner > height:
			print "\nWARNING: heightinner %d > height %d; therefore, heightinner will be defaulted to height/2" %( heightinner, height )
			heightinner = height/2
			
		if radiusinner > radius:
			print "\nWARNING: radiusinner %d > radius %d; therefore, heightinner will be defaulted to radius/2" %( radiusinner, radius )
			radiusinner=radius/2
			
		maskin = mask.process("testimage.cylinder",{'height':options.heightinner,'radius':options.radiusinner})
		finalmask = maskout - maskin

	elif 'heightinner' in params:
		print "\nWARNING: heightinner not used since radiusinner is absent. You need to supply BOTH of these when generating a hollow cylyndrical template"

	elif 'radiusinner' in params:
		print "\nWARNING: radiusinner not used since heightinner is absent. You need to supply BOTH of these when generating a hollow cylyndrical template"
	
	a.process_inplace('mask.sharp',{'outer_radius':-2})
	a['apix_x'] = tomogramapix
	a['apix_y'] = tomogramapix
	a['apix_z'] = tomogramapix
	
	finalmask.write_image(options.path + '/templatesph.hdf',0)
	#if not options.invert:
	#	a.mult(-1)		
	options.template = options.path + '/templatesp.hdf'
	
	return options


'''
c:function to clip volume to new cubical size
'''
def clip3D( vol, size ):
	
	volxc = vol['nx']/2
	volyc = vol['ny']/2
	volzc = vol['nz']/2
	
	Rvol =  Region( (2*volxc - size)/2, (2*volyc - size)/2, (2*volzc - size)/2, size , size , size)
	if Rvol:
		vol.clip_inplace( Rvol )
	else:
		print "ERROR!: Empty Region to clip", Rvol
	#vol.process_inplace('mask.sharp',{'outer_radius':-1})
	
	return vol


'''
c:function to divide --tomogram into cubes in which to look for particles
c:(amenable to parallelization, not yet implemented)
'''
def tomosubblocks( options, boxsize ):
	
	tomohdr = EMData(options.tomogram,0,True)
	tomox = tomohdr['nx']
	tomoy = tomohdr['ny']
	tomoz = tomohdr['nz']
	tomogramapix = tomohdr['apix_x']
	
	tomovol = tomox*tomoy*tomoz
	
	cubesize = tomoz
	cubevol = cubesize*cubesize*cubesize

	ncubes = int( tomovol / cubevol )
	
	if options.verbose:
		print "\nThe volume of the tomogram is", tomovol
		print "The volume of the subregions is", cubevol
		print "Therefore, in principle, you could divide the tomogram into at least %d cubes wiht a sidelength of %d" %( ncubes, cubesize )
	
	zc = cubesize/2	#the center in z never changes
		
	'''
	The tomogram ought to be divided into cubes, which will be cubes with side length equal to the 
	tomogram's thickness (z). When scanning each of these cubes for particles, they need to overlap
	by an amount equal to the side length of the particles (--boxsize) to ensure picking particles at the edges of each
	cube without 'breaking' any particles. 
	The first row and column (i=0) of cubes (in the xy plane) need to be centered at cubesize/2,cubesize/2.
	All subsequent cubes need to be centered at [i*(cubesize) - cubesize/2 - i*boxsize, j*(cubesize) - cubesize/2 - j*boxsize].
	'''
		
	if options.verbose:
		print "\nI will look for subvolumes throughout the tomogram, which has these x, y, z dimensions", tomox, tomoy, tomoz
		print "Where the boxsize of SUBREGIONS to examine is", cubesize
		print "And the boxsize of (UNSHRUNK) ptcls to extract is", boxsize
	
	##### The TOMOGRAM SHOULD HAVE BEEN SHRUNK ALREADY, during preprocessing, and the template should have been scaled to match it
	#if options.shrinktomo > 1:
	#	cubesize=cubesize/2
	#	print "Because you chose to shrink, the cubesize for scanning purposes is actually", cubesize
	#	print "And therefore, the outputbox (which determines the effective boxisze of the particles in the tomogram DURING scanning, is"
	#	boxsize = boxsize / 2
	#	print outputbox
	
	'''
	generate lists for the x,y centers of the cubes the tomogram will be divided into; since 
	cubesize might not fit an exact (integer) amount of times into tomox and tomoy, add
	the last cube from the far most edge (this might overlap greatly with the penultimate
	cube depending on the specific tomogram, but will be accounted for later
	'''
	nxs = tomox/(cubesize - boxsize/2)
	print "nxs",nxs
	xcenters = [ cubesize/2 + i*(cubesize - boxsize/2 ) for i in range( nxs ) ] 
	
	lastx = xcenters[-1]+cubesize/2
	unscannedx = tomox-lastx
	print "unscannedx",unscannedx
	if unscannedx > boxsize/2:
		lastxc = tomox - cubesize/2

		if lastxc not in xcenters:
			xcenters += [lastxc]

	print "xcenters",xcenters
	nys = tomoy/(cubesize - boxsize/2)
	print "nys",nys
	ycenters = [ cubesize/2 + i*(cubesize - boxsize/2 ) for i in range( nys ) ] 
	
	lasty = ycenters[-1]+cubesize/2
	unscannedy = tomoy-lasty
	print "unscannedy",unscannedy
	if unscannedy > boxsize/2:
		lastyc = tomoy - cubesize/2
		print "lastyc",lastyc
		if lastyc not in ycenters:
			
			ycenters += [lastyc]

	print "ycenters",ycenters
	#sys.exit()
	'''
	now iterate through rows and columns of tomogram SUBREGIONS or 'cubes'
	'''
	centers = []
	


	for xc in xcenters:
		for yc in ycenters:
			#centers.append( [xc,yc,zc] )		
	
			centers.append( [xc,yc] )
	
	f=open(options.path+'/centers.txt','w')
	f.writelines([ str(center[0])+' '+str(center[1])+'\n' for center in centers ])
	f.close()

	#sys.exit()

	return centers
	

'''
c:function to do cross correlation between template and tomogram subregion/chunk
'''
#def scanposition(options,template,boxsize,yshort,xi,yi,zi,xc,yc,zc,sbn,x,y,xo,yo,count):
def scanchunks( options, centers, templateindx ):
	
	template = EMData( options.template, templateindx )
	templatesphavg = EMData( options.template.replace('.hdf','_sph.hdf', 0 ) )
	
	radius = templatesphavg['nx']/4	#assuming the particle/template is in a box of 2x its diameter, 4x its radius
	if options.ptclradius:
		radius = options.ptclradius
	elif options.boxsize:
		radius = options.boxsize/4.0
	
	ptclvol = (4.0/3.0)* math.pi *math.pow(options.ptclradius,3)
	
	tomohdr = EMData( options.tomogram, 0 )
	tomoz = tomohdr['nz']
	
	cubesize = tomoz
	cubevol = cubesize * cubesize * cubesize
	
	nptclsmax = int( cubevol/ptclvol )
	if options.dilutionfactor:
		nptcls = nptclsmax/options.dilutionfactor
	
	print "\nat maximum concentration, %d particles would fit in each subregion; given dilution factor %d, the estimate is %d" %(nptcls, options.dilutionfactor, nptcls)
	
	#'''
	#preprocesstemplate if necessary		
	#'''
	#if options.templatemaskthreshold
	#
	#if options.templatethreshold:
	#	options.templatethreshold = parsemodopt( options.templatethreshold )	
	
	#transtemplate.write_image(transtemplatename,0)
	
	xo = yo = 0
	if options.gridoffset:
		xo = int(options.gridoffset.split(',')[0])
		yo = int(options.gridoffset.split(',')[-1])
	
	'''
	c:for cross-correlation, the template needs to have the same box size as the tomogram subregions to scan
	'''
	tomohdr = EMData(options.tomogram,0,True)
	tomox = tomohdr['nx']
	tomoy = tomohdr['ny']
	tomoz = tomohdr['nz']
	tomoxc = tomohdr['nx']/2
	tomoyc = tomohdr['ny']/2
	tomozc = tomohdr['nz']/2
	
	templatesphavg.process_inplace('xform.scale',{'scale':1,'clip':cubesize})
	
	print "\ntemplate resized", templatesphavg['nx'],templatesphavg['ny'],templatesphavg['nz']
	
	zc = tomoz/2
	
	results=[]
	


	for i in range(len(centers)):
		xc = centers[i][0]
		yc = centers[i][1]
		
		print "\nscanning SUBREGION %d/%d, centered at x=%d y=%d z=%d" %( i, len(centers),xc, yc, zc )
		
		xi = xc - zc	#the beginning of each subregion to scan will be half the size of 
		yi = yc - zc	#the subregion (zc) left of the center in both x and y. 
	
		r = Region( xi, yi, 0, tomoz, tomoz, tomoz )	#The subregion extends zc pixels from the initial 
											#xi,yi,0 position in all three directions

		print "therefore, the region of data from the tomogram to scan is", r
		
		subbox = EMData()
		subbox.read_image(options.tomogram,0,False,r)	#load only the pixel data from the region r

		subbox['origin_x']=0
		subbox['origin_y']=0
		subbox['origin_z']=0
		subbox['apix_x']=tomohdr['apix_x']
		subbox['apix_y']=tomohdr['apix_y']
		subbox['apix_z']=tomohdr['apix_z']
	
		#if subbox['mean']:
			#subbox.process_inplace('normalize')
		
		if not subbox['sigma']:
			print "\nthere's something wrong with this subregion and it will be skipped. mean=%.4f, std=%.4f" %(subbox['mean'],subbox['sigma'])
			print "\nyou might be scanning empty regions of the tomogram. You might have a --yshort tomogram and haven't realized so."
		else:
			print "subbox['sigma'] is", subbox['sigma']

			if options.test:
				nptcls = 1	
			
			template2use = templatesphavg.copy()
		
			#subboxnorm = subbox.process('normalize.edgemean')
			subboxnorm = subbox.copy()
			
			if options.goldthreshtomo and options.goldstack:
				#print "I will THRESHOLD the subsection based on gold stats"
				#goldstack = os.path.base(options.goldstack)
				#print "Gold stack received is", options.goldstack
				
				#ret = meancalc(options, options.goldstack ,normalize=True)
				ret = meancalc(options, options.goldstack )
				max_m = ret[0]
				max_s = ret[1]
		
				min_m = ret[2]
				min_s = ret[3]
				
				mean_m = ret[4]
				mean_s = ret[5]
				
				sigma_m = ret[6]
				sigma_s = ret[7]
		
				#print "The mean of the gold maxes is", max_m
				#print "The mean of the gold minimums is", min_m
				
				max_val = max_m - 30 * max_s
				min_val = min_m + 5 * math.fabs(min_s)
				#print "Therefore the max val to filter is", max_val
				#print "the min is", min_val
		
				#print "BEFORE CLAMPING, max and min are", subboxnorm['maximum'],subboxnorm['minimum']
				#print "count is", count
				
				subboxnorm.process_inplace('threshold.clampminmax',{'maxval':max_val,'minval':min_val})
				subboxnorm.write_image('clamped.hdf',i)
		
				#print "AFTER CLAMPING, max and min", subboxnorm['maximum'],subboxnorm['minimum']
		
			'''
			c:only calculate the cross correlation between template and subregion once
			'''
			
			print 'subbox size', subbox['nx'],subbox['ny'],subbox['nz']
			print 'template2use size', template2use['nx'],template2use['ny'],template2use['nz']
			template2use.process_inplace('filter.matchto',{'to':subboxnorm} )
			
			ccf = subboxnorm.calc_ccf(template2use)
			ccf.process_inplace("xform.phaseorigin.tocorner") 
			
			ccf.process_inplace('normalize')
			ccf.write_image( options.path+'/ccfs.hdf',i)
			subboxnorm.write_image( options.path+'/subregions.hdf',i)
			coordssubset = set()
			
			print "ccf done"
			
			masks=[]
	
			xmax=0
			ymax=0
			zmax=0
			
			'''
			c:if the tomogram subregion being scanned is at the edge OF THE TOMOGRAM in x or y, 
			don't allow particles to be picked if they're too close to the tomogram's absolute edge.
			you want to avoid introducing empty pixels.
			picked particles need to be padded by at least one radius.
			if the tomogram subregion is NOT at the edge of the tomogram, still, since 
			subregions overlap by the entire boxsize (corresponding to 2*ptcldiameter),
			there's no need to allow particles to be picked closer that a radius away
			from the edge, because they are more likely to be picked again (and with a 
			higher score) in a subregion that contains a larger portion of the particle.
			'''
			edgeminval = radius
			exgemaxval = cubesize - radius
			if xc == cubesize/2 or yc == cubesize/2 or xc == tomox - cubesize/2 or yc == tomoy - cubesize/2:
				edgeminval = 2 * radius
				edgemaxval = cubesize - 2*radius
			
			
			#print "Edge min and max vals are", edgeminval, edgemaxval
			#print "\n\n\nThe number of particles to look for in a subregion is %d\n\n\n" %(nptcls)
			'''
			c:iterate through the likely number of particles
			'''
			print "iterating over %d peaks" %( nptcls )
			
			for p in range(nptcls):
				if options.verbose == 10:
					print "\nexamining peak %d/%d" %(p,nptcls)
				#print "\nAttempt numbr %d to find a particle" %(p)
				#print "in subregion", sbn+1
				#box = ccf.get_zsize()
				#r =  Region((box/2) - int(parameters['searchx']), (box/2) - int(parameters['searchy']), (box/2) - int(parameters['searchz']), 2 * int(parameters['searchx']) + 1, 2 * int(parameters['searchy']) + 1, 2 * int(parameters['searchz']) + 1) 
				#sub_ccf = ccf.get_clip(r)	

				locmax = ccf.calc_max_location()
								
				locmaxX = locmax[0]
				locmaxY = locmax[1]
				locmaxZ = locmax[2]
			
				score = ccf['maximum']
				if options.verbose == 10:
					print "score=",score
					
				#print "Therefore, after subtracting this max from the subbox, the max is at", xmax,ymax,zmax
				#if max < 1.0
				if locmaxX < edgeminval or locmaxX > edgemaxval or locmaxY < edgeminval or locmaxY > edgemaxval or locmaxZ < edgeminval or locmaxZ > edgemaxval:
					#print "Either the max was less than 1; lets see", max
					#print "Or one of the coordinates of the maximum was too close to the edge", locmax
					if options.verbose > 11:
						print "\npotential particle skipped based on EDGE VALUES! cubesize=%d, edgeminval=%d, edgemaxval=%d" %( cubesize, edgeminval, edgemaxval )
						print "locmax", locmax
					#pass			
				else:
					#print "THE max for a potential particle was found at", locmax
					
					#aux=0
					#if yshort:
					#	print "YSHORT IS TURNED ON!"
					#	aux=ymax
					#	ymax=zmax
					#	zmax=aux
					
					'''
					c:add the position where the subregion begins, xi, yi, to get the correct coordinates respect to the entire tomogram 
					'''
					#realxmax = (cubesize - locmaxX) + xi	
					#realymax = (cubesize - locmaxY) + yi
					#realzmax = (cubesize - locmaxZ)
					
					realxmax = locmaxX + xi	
					realymax = locmaxY + yi
					realzmax = locmaxZ
					
					#print "Therefore, the REAL final coordinates are", realxmax,realymax,realzmax	
					
				
					
					
					#ccf.process_inplace('mask.sharp',{'inner_radius':ptclmaskrad,'dx':xmax-subboxsiz/2,'dy':ymax-subboxsiz/2,'dz':zmax-subboxsiz/2})
					#print "\nThe ptclmaskrad used is\n", ptclmaskrad
					#print "\n\n"
					#ccf.write_image('subregion' +str(sbn+1).zfill(3) + '_ccf_mask' + str(p).zfill(3)+ '.hdf',0)	
					
					'''
					if a mask was determined for the ENTIRE tomogram (to try to exclude the 
					carbon areas, for example), only add the coordinates for the putative 
					particle to the coordinates set IF they are within the mask
					'''
					if options.mask and options.gridradius:
						print "\nBecause mask is on, I will see if the real center of the putative particle is within"
						#if (realxmax - (xo + tomoxc)) * (realxmax - (xo + tomoxc)) + (realymax - (yo + tomoyc)) * (realymax - (yo + tomoyc)) > (options.gridradius * options.gridradius) :
						
						dist = math.sqrt( math.pow( realxmax - (xo + tomoxc), 2 ) + math.pow( realymax - (yo + tomoyc), 2 ) ) 
						if dist > options.gridradius:	
							print "\npotential particle excluded because it is outside tomogram mask!"
						else:
							if options.verbose:
								print "\npotential particle included because it is inside tomogram mask!"
						
							results.append([score,realxmax,realymax,realzmax])
							#coordssubset.add(( realxmax, realymax, realzmax )) 
			
							#if xi<80 and p == 2:
							#	print "$$$$$$$$$$$$$$$$$$$ I will write the melon ball!"
							#	ccf.write_image('zzmelonedball.hdf',0)
					else:
						results.append([score,realxmax,realymax,realzmax])
	
				'''
				zero out the maximum just found (and an area as large as the particle around it)
				so that the same maximum (and anything within radius distance) won't be 
				picked as iterations progress.
				'''
				maskx = locmaxX - cubesize/2
				masky = locmaxY - cubesize/2
				maskz = locmaxZ - cubesize/2

				#maskx = (cubesize - locmaxX)
				#masky = (cubesize - locmaxY)
				#maskz = (cubesize - locmaxZ)

				#print "Therefore the mask will be centered at", maskx,masky,maskz
				
				#print "zeroing out mask radius", radius
				#print "at mask dx,dy,dz", maskx,masky,maskz
				
				ccf.process_inplace('mask.sharp',{'inner_radius':radius,'dx':maskx,'dy':masky,'dz':maskz})
				subboxnorm.process_inplace('mask.sharp',{'inner_radius':radius,'dx':maskx,'dy':masky,'dz':maskz})	
				subboxnorm.write_image(options.path+'/masked.hdf',i)


	return results


'''
c:function to prune picked particles by distance to one another (RMSD)
'''
def rmsdpruner( datar, options ):		
	
	elementstoremove = set()
	data = list(datar)     #list of lists with per particle info [ [score1,x1,y1,z1], [score2,x2,y2,z2],...,[scoren,xn,yn,zn] ]
	print "\nin rmsdpruner data length is", len(data)
    #perform all-vs-all comparisons with two nested loops
	numcomp = (len(data)*(len(data)-1))/2
	print "which will require %d comparisons\n" %(numcomp)
	cn=0
	ntoremove = 0
	for d in range( len(data) ):
    	
        #model each particle center as a vector
		dvector = numpy.array( [ data[d][1],data[d][2],data[d][3] ] )
		dcoeff = data[d][0]
	
	
		#print "dvector", dvector
		for e in range(d+1,len(data)):
			evector = numpy.array( [ data[e][1],data[e][2],data[e][3] ] )
			ecoeff = data[e][0]
			print "\rrmsd comparison %d/%d" %(cn,numcomp) 	
			cn+=1
			print'cn',cn
			print "dvector",dvector
			print "evector",evector
			
            #compute the angle and rmsd between the two particle vectors
			angle = float( numpy.degrees( numpy.arccos( numpy.dot(dvector,evector) / ( numpy.dot(evector,evector) * numpy.dot(dvector,dvector) ) )) )
			rmsd = float( numpy.linalg.norm(dvector - evector) )
			print "rmsd is", rmsd
			print "ptclradius, diameters",options.ptclradius,options.ptclradius*2.0
			print "EXCLUDE because rmsd < ptclradius*2.0", float(rmsd) < float(options.ptclradius*2.0)


			diameter = options.ptclradius*2.0
			if rmsd < diameter:
				print "TRUE!"
				if rmsd < diameter:
					print "\nREMOVING a particle that overlapped with another with an RMSD of", rmsd
					ntoremove+=1
					print  rmsd < diameter
					#however, if two particles overlap, keep the one with the highest score
	        		elementoremove = data[e]
	        		#if dcoeff > ecoeff:
	        		if ecoeff > dcoeff:
	        			elementoremove = data[d]
					print "elementoremove",elementoremove
					elementstoremove.add( elementoremove )
				
				
			print '\n'
				

	print "\n(e2spt_autoboxer)(rmsdpruner), removing %d / %d particles, because ntoremove is %d" %( len(elementstoremove), len(data), ntoremove )
	
	print "type of data is",type(data)
	print "len(data)",len(data)
	#print "data list is",data

	#finaldataset = set(data)
	finaldata = data
	if elementstoremove:
		finaldata = list( set(data) - set(elementstoremove) )
	
	return finaldata



def plothistogram(  options, scores ):	
	import matplotlib.pyplot as plt
	import pylab
	import matplotlib
	
	std = numpy.std( scores )
	mean = numpy.mean( scores )
	statistics = ['mean='+str(mean) + ' std='+str(std) + '\n']


	print "\n(e2spt_autoboxer)(plothistogram) number of scores", len(scores)
	print "sigma = %.6f and mean = %.6f" %( std, mean )
	
	if not std:
		print "ERROR: std=0, which means all intensity values are the same."
		sys.exit()
	
	cuberoot = numpy.power(len(scores),1.0/3.0)
	#print "The cuberoot of n is", cuberoot
	width = (3.5*std)/cuberoot
	print "\naccording to Scott's normal reference rule, width = (3.5*std)/cuberoot(n), the width of the histogram bins will be", width
	
	calcbins = (max(scores) - min(scores)) / width
	
	if options.nbins:
		calcbins = options.nbins
	
	print "and the number of bins n = ( max(scores) - min(scores) ) / width, will thus be", calcbins
	calcbins = round(calcbins)
	print "rounding to", calcbins
	
	statistics += ['bins=' + str( calcbins ) + ' , binwidth=' + str( width ) + '\n']
	
	print "statistics for this distribution are", statistics
	
	if not calcbins:
		print "\nWARNING: nins=0, which means max and min intensity are the same, which probably means all scores are zero. Defaulting nbins to number of partilces."
		calcbins = len(scores)
			
	statsfile = options.path + '/histogram_stats.txt'
	f=open(statsfile,'w')
	f.writelines(statistics)
	f.close()
	
	
	#calcbins=50
	plt.hist( scores, calcbins, alpha=0.30 )	
	
	scores.sort()

	'''
	FITTING A CURVE
	binmids = []
	
	for i in range( len(bins) ):
		print bins[i]
		if i < len(bins)-1:
			#print i
			#print "i was smaller, because len bins is", len(bins)
			binmids.append( (bins[i] + bins[i+1])/2 )
	
	
	polyscores = numpy.polyfit(binmids, n, 3)
	yfit = numpy.polyval(polyscores, binmids)
	
	print "\n\nBinmids are\n", binmids
	print "\n\nThe length of binmids is\n", len(binmids)
	
	#hist, bins = numpy.histogram(y, options.nbins)
	'''
	
	plottitle = 'CC scores distribution histogram'
	plt.title( plottitle )

  	matplotlib.rc('xtick', labelsize=16) 
	matplotlib.rc('ytick', labelsize=16) 
  		 	
  	font = {'weight':'bold','size':16}
	matplotlib.rc('font', **font)
  		 	
	pylab.rc("axes", linewidth=2.0)
		
	pylab.xlabel('CC score (au)', fontsize=16, fontweight='bold')
  	pylab.ylabel('Number of particles', fontsize=16, fontweight='bold')
	
	plt.savefig(options.path + '/histogram.png')
	plt.clf()
	
	return [mean,std]




'''
c:prune picked partickes based on behavior/distribution of cc scores
'''
def cccpruner( options, scores, data ):
	
	try:
		from scipy import sparse
	
		scoresmean = numpy.mean(scores)	
		scoressigma = numpy.std(scores)

		#print "scores are", scores
		print "\nThese many ptcls before ccc pruning", len(scores)
		
		nscores = len(scores)
	
	
	

	
	
		topnscores = int(nscores*(0.05))
	
		print "Therefore, 5% are", topnscores
		topscores = scores[:topnscores]
		bottomscores = scores[topnscores:]
		print "The top scores are", topscores
		print "And the len of bottom scores are", len(bottomscores)

		print "The sum of bottom and top scores should equal the len of data", len(data), len(topscores) + len(bottomscores)

		print "\n\n\n\nThe mean and sigma of the datas CCC", scoresmean, scoressigma
		print "\n\n\n\nWhereas the max and the min of the data's ccc are CCC", numpy.max(scores), numpy.min(scores)
		print "\n\n\n\n"
		lowerbound = scoresmean - 2*scoressigma
		upperbound = scoresmean + 2*scoressigma
		
		#upperbound = numpy.max(scores)
		#print "Therefore, one sigma away, the lower and upper bounds are", lowerbound, upperbound

		removed_count=0
		conserved_count=0

		pruneddata = []
		print "Will now do correlation coefficient based pruning"	
		for d in data:
			#print "The data element to examine", d
			coeff = d[0]
			#print "This is the coeff to examine", coeff
	
			if coeff < lowerbound or coeff > upperbound:
				removed_count+=1
				data.remove(d)
		
				try:
					scores.remove(coeff)
					print "One element was removed because it has coeff < lowerbound or coeff > upperbound", coeff, lowerbound, upperbound
				except:
					print "The coeff smaller than lowerbound or larger than upperbound was not in scores"
			
			elif coeff in scores and coeff in bottomscores:
		
				data.remove(d)
				removed_count+=1
		
				try:
					scores.remove(coeff)
					print "Element removed because coeff was in bottomscores", coeff
				except:
					print "The coeff smaller than lowerbound or larger than upperbound was not in scores"
			
			elif coeff in scores and coeff not in topscores:
				scores.remove(coeff)
				data.remove(d)
				removed_count+=1
				print "Element removed because coeff was in top coeff", coeff
			else:
				conserved_count+=1
				print "not removed", coeff
				if d not in pruneddata:
					pruneddata.append(d)
			print "removed count", removed_count
			print "preserved count", conserved_count

		print "I have pruned out these many based on mean and sigma statistics", removed_count
		print "And therefore data now is", len(data)
		print "But pruned data is more accurately", len(pruneddata)

		data = pruneddata

		print "THe len of data AFTER CCC pruning is", len(data)

		return data
		
		
	except:
		print "\nERROR: SciPy not found. Must be installed if using --pruneccc"
		
		return data
			
	



#def meancalc(options,stack,normalize=False):
def meancalc(options,stack):
	print "Stack RECEIVED in meancalc is", stack
	n=EMUtil.get_image_count(stack)
	#print "N is", n
	means=[]
	maxs=[]
	mins=[]
	stds=[]
	
	for i in range(n):
		ahdr = EMData(stack,i,True)
		#if normalize:
		#	a=EMData(stack,i)
			#bx=a['nx']
			
			#mask = EMData(bx,bx,bx)
			#mask.to_one()
			#mask.process_inplace('mask.sharp',{'outer_radius':-2})
			#a.process_inplace('normalize.mask',{'mask':mask})		
			#a.mult(mask) #NEW
			
		meana=a['mean']
		means.append(meana)
		
		max=a['maximum']
		maxs.append(max)
		
		min=a['minimum']
		mins.append(min)
		
		std=a['sigma']
		stds.append(std)
		
	mean_means=numpy.mean(means, dtype=numpy.float64)
	sigma_means=numpy.std(means, dtype=numpy.float64)
	
	mean_maxs=numpy.mean(maxs, dtype=numpy.float64)
	sigma_maxs=numpy.std(maxs, dtype=numpy.float64)
	
	mean_mins=numpy.mean(mins, dtype=numpy.float64)
	sigma_mins=numpy.std(mins, dtype=numpy.float64)
	
	mean_stds=numpy.mean(stds, dtype=numpy.float64)
	sigma_stds=numpy.std(stds, dtype=numpy.float64)
	
	return [mean_maxs,sigma_maxs,mean_mins,sigma_mins,mean_means,sigma_means,mean_stds,sigma_stds]
	
	
def pruner(data,tag,mean_maxs,sigma_maxs,mean_mins,sigma_mins):	
	data=list(data)
	for d in data:
		#print "d in data is", d
		#print "d[1] is ", d[1]
		x=d[1]
		y=d[2]
		z=d[3]
		#print "The actual coordinates used for extraction are", x, y, z
		e = getbox(options,x,y,z,options.boxsize)
		
		e.process_inplace('normalize.edgemean')
		
		min = e['minimum']
		max = e['maximum']
		#print "min and max of ptcl are", min, max
		if tag=='gold':
			#print "tag is", tag
			#print "and mean_maxs and sigma_maxs are", mean_maxs, sigma_maxs
			#print "and mean_mins and sigma_mins are", mean_mins, sigma_mins
			if max > (mean_maxs - (sigma_maxs * 3)) or min < (mean_mins - (sigma_mins * 3)):
				data.remove(d)
				#print "particle REMOVED based on GOLD PRUNING!"
		elif tag == 'background':
			 if max < (mean_maxs + sigma_maxs):
				data.remove(d)
				#print "particle REMOVED based on BACKGROUND PRUNING!"
		elif tag == 'carbon':
			print "NO cabron pruning method yet"			
	
	return data



#NOT WORKING 07/2015
'''
c:function to extract subtomogram from --tomogram based on x,y,z coordinates and a defined boxsize
'''
def getbox( options, x, y, z, size ):
	r = Region( (2*x - size)/2,(2*y- size)/2, (2*z-size)/2, size, size, size )
	img = EMData()
	img.read_image(options.tomogram,0,False,r)

	return img


def statspruner( options, data ):
	
	maxs = {}
	mins = {}
	means = {}
	meansnz = {} 
	stds = {}
	
	maxs2d = {}
	mins2d = {}
	means2d = {}
	meansnz2d = {} 
	stds2d = {}
	
	bad = {}
	
	for d in data:
		score = data[0]
		x = data[1]
		y = data[2]
		z = data[3]
		
		img = getbox( options, x, y, z, options.boxsize )
		
		maxs.update( { score: img['maximum'] } )
		mins.update( { score: img['minimum'] } )
		means.update( { score: img['mean'] } )
		meansnz.update( { score: img['mean_nonzero'] } )
		stds.update( { score: img['sigma'] } )
		
		prj = img.update( 'standard',Transform() )
		
		maxs2d.update( { score: prjmg['maximum'] } )
		mins2d.update( { score: prj['minimum'] } )
		means2d.update( { score: prj['mean'] } )
		meansnz2d.update( { score: prj['mean_nonzero'] } )
		stds2d.update( { score: prj['sigma'] } )
		
	
	'''
	c:MAX based pruning, intended to further get rid of golds
	'''
	MAXSmean = numpy.mean(maxs2d,dtype=numpy.float64)
	MAXSsigma = numpy.std(maxs2d,dtype=numpy.float64)

	print "The mean of MAXS", MAXSmean
	print "The sgima of MAXS", MAXSsigma
	print "\n"

	maxs_thresh = MAXSmean + 2*MAXSsigma
	print "Because MAXSmean is %f and MAXSsigma is %f then maxs_thresh = MAXSmean + 2*MAXSsigma is %f" %(MAXSmean, MAXSsigma, maxs_thresh)
	
	for d in data:
		score = data[0]
		max = maxs[score]
		max2d = maxs2d[score]

		if max > maxs_thresh:
			data.remove(d)
			if score in bad:
				badcount = bad[score] +1
				bad.update( { score:badcount } )
			else:
				bad.add({ score: 1 })
			

	print "These many particles were removed based on max value", co
	print "Len data after max pruning is", len(newestdata)

	'''
	MIN based pruning (not clear so far that min info can be used to prune in addition to max pruning)
	'''

	#newmins=[]
	#for d in newestdata:
	#	newmins.append(d[-4])
	
	MINSmean = numpy.mean( newmins,dtype=numpy.float64 )
	MINSsigma = numpy.std( newmins,dtype=numpy.float64 )

	print "The mean of MINS", MINSmean
	print "The sgima of MINS", MINSsigma
	print "\n"


	'''
	SIGMA based pruning; noise has a smaller std than noise + data; and gold + data has a much larger std
	'''

	newsigmas=[]
	for d in newestdata:
		newsigmas.append(d[-3])
	
	SIGMASmean = numpy.mean(newsigmas,dtype=numpy.float64)
	SIGMASsigma = numpy.std(newsigmas,dtype=numpy.float64)

	print "The mean of SIGMAS", SIGMASmean
	print "The sgima of SIGMAS", SIGMASsigma
	print "\n"

	sigmas_thresh_upper = SIGMASmean + 2*SIGMASsigma
	sigmas_thresh_lower = SIGMASmean - 2*SIGMASsigma
	print "Because SIGMASmean is %f and SIGMASsigma is %f then sgimas_thresh_upper and lower are %f, %f" %(SIGMASmean, SIGMASsigma, sigmas_thresh_upper, sigmas_thresh_lower)

	co=0
	for d in newestdata:
		if d[-3] > sigmas_thresh_upper or d[-3] < sigmas_thresh_lower:
			#print "A particle has been removed because its sgima is larger or smaller than mean_thresh based on PRJ"
			newestdata.remove(d)
			co+=1
	print "These many particles were removed based on sigma pruning", co
	print "Len data after sigma pruning is", len(newestdata)


	'''
	MEAN based pruning; noise has a mean closer to zero than noise + data
	'''

	newmeans=[]
	for d in newestdata:
		newmeans.append(d[-2])

	MEANSmean = numpy.mean(newmeans,dtype=numpy.float64)
	MEANSsigma = numpy.std(newmeans,dtype=numpy.float64)

	print "\nThe mean of MEANS is", MEANSmean
	print "The sigma of MEANS is", MEANSsigma

	means_thresh_lower = MEANSmean - 2*MEANSsigma
	means_thresh_upper = MEANSmean + 2*MEANSsigma

	return data


#NOT WORKING 07/2015
def prjpruner( options, data ):
	print "I will do projection based pruning"
	bxf = options.boxsize
	tempcrop = EMData( options.template, 0 )

	#bxi=tempcrop['nx']
	#byi=tempcrop['ny']
	#bzi=tempcrop['nz']
	#xct=bxi/2
	#yct=byi/2
	#zct=bzi/2

	#r=Region( (2*xct - bxf)/2, (2*yct - bxf)/2, (2*zct - bxf)/2, bxf,bxf,bxf)	

	#tempcrop.clip_inplace(r)
	#print "THe copy if the template has this mean and max and size", tempcrop['mean'], tempcrop['maximum'], tempcrop['nx']

	#tempcrop.write_image(options.path + '/' + 'tempcrop.hdf',0)

	t = Transform({'type':'eman','az':0,'alt':0,'phi':0})	
	tprj = tempcrop.project("standard",t)
	tprj.set_attr('xform.projection',t)
	tprj['apix_x'] = tempcrop['apix_x']
	tprj['apix_y'] = tempcrop['apix_x']
	tprj.write_image(options.path + '/' + 'tprj.hdf',0)

	prjccfs = []
	newdata=[]
	kkk=0
	basicdata=set()
	tside = Transform({'type':'eman','az':0,'alt':-90,'phi':0})
	for d in data:
		#print "d in data is", d
		#print "d[1] is ", d[1]
		x=d[1]
		y=d[2]
		z=d[3]
		#print "The actual coordinates used for extraction are", x, y, z
		r = Region((2*x- options.boxsize)/2,(2*y-options.boxsize)/2, (2*z-options.boxsize)/2, options.boxsize, options.boxsize, options.boxsize)
		e = EMData()
		e.read_image(options.tomogram,0,False,r)
	
		#print "dimensions of read e are", e['nx'],e['ny'],e['nz']
		eb = e['nx']
		mask=EMData(eb,eb,eb)
		mask.to_one()
		mask.process_inplace('mask.sharp',{'outer_radius':-2})
		e.process_inplace('normalize.mask',{'mask':mask})
		#e.process_inplace('mask.sharp',{'outer_radius':-2})
	
		e.mult(mask)
	
		#print "After normalization in 3D and masking, mean and mean nonzero are", e['mean'],e['mean_nonzero']
	
		#print "I have masked e, its mean, meannonzero, min and max are", e['mean'], e['mean_nonzero'],e['minimum'],e['maximum']
		#e.process_inplace('normalize.edgemean')
		pprj=e.project("standard",t)
	
		#if kkk==0:
		#	display(pprj)
	
		mask2d=EMData(eb,eb)
		mask2d.to_one()
		mask2d.process_inplace('mask.sharp',{'outer_radius':-2})
		pprj.process_inplace('normalize.mask',{'mask':mask2d})
	
		#pprj.process_inpace('normalize')
		#pprj.process_inplace('mask.sharp',{'outer_radius':-2})
		pprj.mult(mask2d)
	
		#if kkk==0:
		#	display(pprj)
		#pprj.process_inplace('mask.sharp',{'outer_radius':-2})
	
		#print "After FIRST normalization in 2D and masking, mean and mean nonzero are", pprj['mean'],pprj['mean_nonzero']
	
		#if kkk==0:
		#	display(pprj)
		
		#print "I have projected, masked, normalized and remasked e, whose mean, meannonzero, min, max are", pprj['mean'], pprj['mean_nonzero'],pprj['minimum'], pprj['maximum']
		pprj['apix_x'] = e['apix_x']
		pprj['apix_y'] = e['apix_x']
		pprj.write_image(options.path + '/' + 'pprj.hdf',kkk)
	

		#pprj.process_inplace('normalize.edgemean')
	
		#print "The template prj has a size of", tprj['nx'],tprj['ny']
		#print "The particle prj has a size of", pprj['nx'],pprj['ny']
		ccf = tprj.calc_ccf(pprj)
		ccf.process_inplace("xform.phaseorigin.tocorner") 
		ccf.process_inplace('normalize')
	
		locmax = ccf.calc_max_location()
							
		locmaxX = locmax[0]
		locmaxY = locmax[1]
	
		#print "\nThe peak is at", locmaxX, locmaxY
	
		transx = bxf/2 - locmaxX
		transy = bxf/2 - locmaxY
	
		#print "\nAnd the proposed translations are", transx,transy
	
		x=x+transx
		y=y+transy
	
	
		'''
		Generate side projections to center Z
		'''
		pprjside=e.project('standard',tside)
		pprjside.process_inplace('normalize.mask',{'mask':mask2d})
		pprjside.mult(mask2d)
		pprjside.process_inplace('mask.sharp',{'outer_radius':-2})
	
		pprjside['apix_x'] = e['apix_x']
		pprjside['apix_y'] = e['apix_x']
		pprjside.write_image(options.path + '/' + 'pprjside.hdf',kkk)
	
		ccfside = tprj.calc_ccf(pprjside)
		ccfside.process_inplace("xform.phaseorigin.tocorner") 
		ccfside.process_inplace('normalize')
	
		locmaxside = ccfside.calc_max_location()
							
		locmaxXside = locmaxside[0]
		locmaxZside = locmaxside[1]
	
		transz = bxf/2 - locmaxZside
		#print "Transz is", transz
		z=z-transz
	
	
	
		#newcoords=(x,y,z)
		newccf = ccf['maximum']
		#print "\nThe ccf for this projection is", newccf
		#print "\nTherefore, the new coordinates are", x,y
	
		prjccfs.append(newccf)
	
		#coordset.add(coords)
		#scoreset.add(newcoefftuple)
	
		if (newccf,x,y,z) not in basicdata:
			basicdata.add( (newccf,x,y,z) )		
		
			newdata.append( [newccf,x,y,z] )
	
			r = Region((2*x- options.boxsize)/2,(2*y-options.boxsize)/2, (2*z-options.boxsize)/2, options.boxsize, options.boxsize, options.boxsize)
			e = EMData()
			e.read_image(options.tomogram,0,False,r)
	
			eb = e['nx']
			mask=EMData(eb,eb,eb)
			mask.to_one()
			mask.process_inplace('mask.sharp',{'outer_radius':-2})
			e.process_inplace('normalize.mask',{'mask':mask})
			e.mult(mask)
	
			pprj=e.project("standard",t)
			pprj['apix_x'] = e['apix_x']
			pprj['apix_y'] = e['apix_x']
			pprj.write_image(options.path + '/' + 'pprj_corrected.hdf',kkk)
	
	
			pprjside=e.project("standard",tside)
			pprjside['apix_x'] = e['apix_x']
			pprjside['apix_y'] = e['apix_x']
			pprjside.write_image(options.path + '/' + 'pprj_correctedside.hdf',kkk)
	
	
			kkk+=1
		else:
			print "Trying to add repeated particle in FIRST PRJ loop!"
		newdata.sort()

	#print "\n\n\nsorted newdata are", newdata
	newdata.reverse()
	#print "\n\n\nreversed", newdata 


	setdata = set()
	lll=0
	for d in newdata:
		if (d[0],d[1],d[2],d[3]) not in setdata:
			setdata.add( (d[0],d[1],d[2],d[3]) )	
	else:
		lll+=1

	#print "repeated particles when pruning by FIRST PRJ prune!",lll

	newdata = list(setdata)
	print "Therefore data len is now", len(newdata)


	newdata=sorted(newdata, key=itemgetter(1,2,3))

	mmm=0
	ppp=0
	#prjs=[]
	#means=[]
	#sigmas=[]
	maxs2d=[]
	#mins2d=[]

	newestdata = []
	rrr=0
	basicdata=set()
	for d in newdata:
		x=d[1]
		y=d[2]
		z=d[3]
		r = Region((2*x- options.boxsize)/2,(2*y-options.boxsize)/2, (2*z-options.boxsize)/2, options.boxsize, options.boxsize, options.boxsize)
		e = EMData()
		e.read_image(options.tomogram,0,False,r)
	
		eb = e['nx']
		mask=EMData(eb,eb,eb)
		mask.to_one()
		mask.process_inplace('mask.sharp',{'outer_radius':-2})
		e.process_inplace('normalize.mask',{'mask':mask})
		e.mult(mask)
	
		pprj=e.project("standard",t)
		pprj['apix_x'] = e['apix_x']
		pprj['apix_y'] = e['apix_x']
		pprj.write_image(options.path + '/' + 'pprj_corrected_sorted.hdf',mmm)
	
		pmx=pprj.process('xform.mirror',{'axis':'x'})
		pmy=pprj.process('xform.mirror',{'axis':'y'})
	
		ccfpmx=pprj.calc_ccf(pmx)
		ccfpmy=pprj.calc_ccf(pmy)
	
		ccfpmxC=ccfpmx.process('xform.phaseorigin.tocorner')
		ccfpmyC=ccfpmy.process('xform.phaseorigin.tocorner')
	
		maxccfpmxC=ccfpmxC.calc_max_location()
		maxccfpmyC=ccfpmyC.calc_max_location()
	
		xt=(eb/2.0 - maxccfpmxC[0])/2.0
		yt=(eb/2.0 - maxccfpmyC[1])/2.0

		pfix=pprj.copy()
		pfix.translate( xt, yt, 0)
	
		newx=x+xt
		newy=y+yt
	
		'''
		Generate side projections to center Z
		'''
		pprjside=e.project("standard",tside)
		pprjside['apix_x'] = e['apix_x']
		pprjside['apix_y'] = e['apix_y']
		pprjside.write_image(options.path + '/' + 'pprj_corrected_sorted_side.hdf',mmm)
	
		pmxside=pprjside.process('xform.mirror',{'axis':'x'})
		pmzside=pprjside.process('xform.mirror',{'axis':'y'})
	
		ccfpmxside=pprjside.calc_ccf(pmxside)
		ccfpmzside=pprjside.calc_ccf(pmzside)
	
		ccfpmxCside=ccfpmxside.process('xform.phaseorigin.tocorner')
		ccfpmzCside=ccfpmzside.process('xform.phaseorigin.tocorner')
	
		maxccfpmxCside=ccfpmxCside.calc_max_location()
		maxccfpmzCside=ccfpmzCside.calc_max_location()
	
		#xtside=(eb/2.0 - maxccfpmxCside[0])/2.0
		ztside=(eb/2.0 - maxccfpmzCside[1])/2.0
		#print "zt side is", ztside
		pfixside=pprjside.copy()
		pfixside.translate( 0, 0, ztside)
	
		#newx=x+xt
		newz=z-ztside
	
		if math.fabs(xt) <= eb/8.0 and math.fabs(yt) <= eb/8.0 :
			#newestcoords=(newx,newy,newz)
		
			newestccf = ccfpmy['maximum']
			if ccfpmx['maximum'] > ccfpmy['maximum']:
				newestccf = ccfpmx['maximum']
		
			#print "\n$$$$$$$$$$$$$$$$\nAUTOCORRELATION PARTICLE WRITTEN!\n"
	
			#prjccfs.append(newccf)
	
			#coordset.add(coords)
			#scoreset.add(newcoefftuple)
			#newestdata.append( [newestccf,newestcoords] )
		
			r = Region((2*newx- options.boxsize)/2,(2*newy-options.boxsize)/2, (2*newz-options.boxsize)/2, options.boxsize, options.boxsize, options.boxsize)
			e = EMData()
			e.read_image(options.tomogram,0,False,r)
	
			eb = e['nx']
			mask=EMData(eb,eb,eb)
			mask.to_one()
			mask.process_inplace('mask.sharp',{'outer_radius':-2})
			e.process_inplace('normalize.mask',{'mask':mask})
			e.mult(mask)
	
			pprj=e.project("standard",t)
			pprj['apix_x'] = e['apix_x']
			pprj['apix_y'] = e['apix_x']
			pprj.write_image(options.path + '/' + 'pprj_corrected_sorted_autocorrelated.hdf',ppp)	
						
			mean=pprj['mean_nonzero']
			#means.append(mean)
			#print "The mean_nonzero is %f, compared to mean %f" %( mean, pprj['mean'] )
	
			sigma=pprj['sigma_nonzero']
			#sigmas.append(sigma)
	
			max=pprj['maximum']
			maxs2d.append(max)
	
			min=pprj['minimum']
			#mins2d.append(min)
			if (newestccf,newx,newy,newz) not in basicdata:			
				newestdata.append([newestccf,newx,newy,newz,ppp,max,min,sigma,mean,pprj])
				basicdata.add( (newestccf,newx,newy,newz) )
				ppp+=1
			else:
				print "TRYing to add repeated particle in second PRJ loop!"	
		
		else:
			rrr+=1
			#print "\nParticle eliminated because translation from AUTOCORRELATION were too big!\n"
			pass
		mmm+=1
		#if d[0] < prjccfs_mean - prjccfs_sigma:
		#	newdata.remove(d)
		#	print "I have removed a particle based new PRJ mask"

	print "The number of particles pruned by AUTOCORRELATION is", rrr		
	print "Therefore, AFTER AUTUCORRELATION pruning, the len of data is", len(newestdata)


	#setdata = set()
	#lll=0
	#for d in newestdata:
	#	if (d[0],d[1],d[2],d[3]) not in setdata:
	#		setdata.add( (d[0],d[1],d[2],d[3]) )	
	#else:
	#	lll+=1

	#print "repeated particles when pruning by SECOND PRJ prune!",lll

	#newestdata = list(setdata)
	#print "Therefore data len is now", len(newestdata)

		
	newestdata = sorted(newestdata, key=itemgetter(1,2,3))
	
	return newestdata








		
if '__main__' == __name__:
	main()
	





"""
SCRAPS
	
	#for m in range(len(boxes)):
	#	print 'box', boxes[m]
	#	print '\t center', centers[m]
	
	
	
	
	
	
	
	############################
	
	
	
	
	
	print "!!!!Tomo dimensions are", x,y,z
	
	print "The template is", options.template
	
	transtemplate = EMData(options.template,0)
	
	transtemplateapix = transtemplate['apix_x']
	print "With an apix of", transtemplateapix
	
	'''
	Calculate the scale between the reference model and the data, round to the nearest integer to be able to use math.meanshrink (which preserves features),
	then calculate the scale factor again and use xform.scale, to get the exact scaling, even if features might become a bit distorted by xform.scale
	'''
	
	if tomogramapix != transtemplateapix:

		meanshrinkfactor = tomogramapix/transtemplateapix
		meanshrinkfactor_int = int(round(meanshrinkfactor))
		
		#print "The template's apix is", transtemplateapix
		#print "And the tomogram's apix is", tomogramapix
		#print "Therefore, the meanshrink factor is", meanshrinkfactor
		#print "Which, for the first step of shrinking (using math.meanshrink), will be rounded to", meanshrinkfactor_int
		if meanshrinkfactor_int > 1:
			if options.verbose:
	
				print "About to shrink"
				#print "The type of template is", type(transtemplate)
				print "\n\n\n BBBBBBBB\nANd its boxsize BEFORE shrinking was\nBBBBBBBB\n\n", transtemplate['nx']
		
			transtemplate.process_inplace("math.meanshrink",{"n":meanshrinkfactor_int})
		
			#cubesize = transtemplate['nx']
			#print "\n\n\nCCCCCCCC\nWhereas AFTER shrinking it is\nCCCCCC\n\n", transtemplate['nx']

			#if options.verbose:
			#	print "The template was shrunk, to a first approximation."
		
			transtemplateapix = transtemplate['apix_x']
		
		scalefactor = transtemplateapix/tomogramapix
		#cubesize = round(transtemplate['nx']*scalefactor)
	
		#if options.verbose:
		#print "The finer scale factor to apply is", scalefactor
	
		if float(scalefactor) != 1.0:
			transtemplate.process_inplace("xform.scale",{"scale":scalefactor,"clip":cubesize})
			#cubesize = transtemplate['nx']
		#transtemplate.process_inplace("xform.scale",{"scale":1,"clip":transtemplatebox})
	
		#print "\n\n\n AAAAAAAAAAAA\n after all necessary APIX MATCHING, the expandedboxsize of the template is", cubesize
		#print "\n\n\n AAAAAAAAAAAAA\n"
	
		if options.verbose:
			print "The template has now been precisely shrunk."
	
	#####################################
	
	lines=[]
	
	i=0
	xc = cubesize/2
	otherlong = y
	short = z
	
	#if yshort:
	#	otherlong = z
	#	short = y
	
	data=[]
	coordset = set()
	ret = None
	
	sbn = 0
	zi = 0
	
	print "\ncubesize size is", cubesize
	print "Therefore its half is", cubesize/2
		
	boxes = []
	centers = []
	xi = 0
	factor=1
	
	if regionboxsize == boxsize:
		factor = regionboxsize / 2
		print "Regionboxsize/2"
	
	elif boxsize < regionboxsize/2:
		factor = regionboxsize / 2 + (regionboxsize/2 - boxsize)
		print "regionboxsize - boxsize"
		
	elif boxsize >= regionboxsize/2 and boxsize < regionboxsize:
		factor = regionboxsize/2 + (regionboxsize/2 - boxsize/2)
		print "boxsize/2"
	
	print "\n\n\nFFFFFFF\nFactor is",factor
	print "\nFFFFFFFFFFFFF\n\n\n\n"
	
	coordsset = set()
	scoreset = set()
	rmsdset = set()
	
	sys.exit()
	
	count=0
	finalx=0
	while xi < x and xc <= x -regionboxsize/2 and finalx < 2:
		yi=0
		yc = regionboxsize/2
		#print "xi is", xi
		finaly=0		
		while yi < y and yc <= y -regionboxsize/2 and finaly < 2:
			zi=0
			zc = regionboxsize/2
		#	print "yi is", xi
			while zi < z and zc <= z -regionboxsize/2:
				#print "zi is", xi
				#print "Therefore the initial box coords is", xi,yi,zi
				box=[xi,yi,zi, xi+regionboxsize , yi+regionboxsize , zi+regionboxsize]
				center=[xc,yc,zc]
				boxes.append(box)
				centers.append(center)
				
				if options.mask and options.gridradius:
					print "Because mask is on, I will go to the scanner considering that!"
					criteria=(xc - (xo + x) )*(xc - (xo + x)) + (yc - (yo + y))*(yc - (yo + y))
					print "Criteria is", criteria
					print "And must be less than grdiradius squared which is", options.gridradius * options.gridradius
					if (xc - (xo + x) )*(xc - (xo + x)) + (yc - (yo + y))*(yc - (yo + y)) < options.gridradius * options.gridradius:
						print "The center of the subbox is inside the mask"
						ret = scanposition(options,transtemplate,boxsize,yshort,xi,yi,zi,xc,yc,zc,sbn,x,y,xo,yo,count)
				else:
					ret = scanposition(options,transtemplate,boxsize,yshort,xi,yi,zi,xc,yc,zc,sbn,x,y,xo,yo,count)
				
				if ret:
					print "\nThese many particles have been returned", len(ret)
					
					asdfg=0
					for r in ret:
						#print "I will examine particle", asdfg, r
						#print "This info has been returned", r
						newcoeff = r[0]
						newcoefftuple = tuple( [newcoeff] )
						coords=tuple( [ r[1],r[2],r[3] ] )
						#oldcoeff=0
						
						if coords in coordset:
							#print "The particle was already in coordset see", coordset							
							for kkk in range(len(data)):
								if list(coords) in data[kkk]:
									oldcoeff = data[kkk][0]
									if oldcoeff < newcoeff:
										data[kkk][0]=newcoeff
										#print "The particlce was already in data, but the newcoeff is higher than the old", oldcoeff, newcoeff
										
						elif coords not in coordset:
								
							coordset.add(coords)
							scoreset.add(newcoefftuple)
							data.append( (newcoeff, int(round(coords[0])), int(round(coords[1])), int(round(coords[2])) ) )
							#print "I have appended a new particle!", [newcoeff,coords]
							count+=1
						asdfg+=1
							
				#print "And I have indeed appended such box", box
				sbn+=1
				#zi = zi + (factor + regionboxsize - boxsize)
				#zc = zc + (factor + regionboxsize - boxsize)	
				zi += factor
				zc += factor
			#yi = yi + (factor + regionboxsize - boxsize)
			#yc = yc + ( factor + regionboxsize - boxsize)
			yi += factor
			yc += factor
			if yi > y or yc > (y - regionboxsize/2 ) and y % factor:
				#print "I am at a special last box in Y! Therefore the old yi and yc", yi, yc 
				yi = y - regionboxsize
				yc = y - regionboxsize/2
				#print "will be changed for", yi, yc
				finaly+=1
			
			if options.test:
				yi*=2
				yc*=2

		#xi = xi + ( factor + regionboxsize - boxsize )
		#xc = xc + ( factor + regionboxsize - boxsize )
		xi += factor
		xc += factor
		if xi > x or xc > y - regionboxsize/2 and x % factor:
			#print "I am at a special last box in X! Therefore the old xi and xc", xi, xc 
			xi = x - regionboxsize
			xc = x - regionboxsize/2
			#print "will be changed for", xi, xc
			finalx+=1

		if options.test:
			xi*=2
			xc*=2
	print "\n\nCCCCCCCCCCCC\nThe total number of appended particles was %d\nCCCCCCCCCC\n\n" %(count)
	
	return data
"""
	
	
	
