#!/usr/bin/env python

#
# Author: Steve Ludtke 06/10/2013 (sludtke@bcm.edu)
# Copyright (c) 2013- Baylor College of Medicine
#
# This software is issued under a joint BSD/GNU license. You may use the
# source code in this file under either license. However, note that the
# complete EMAN2 and SPARX software packages have some GPL dependencies,
# so you are responsible for compliance with the licenses of these packages
# if you opt to use BSD licensing. The warranty disclaimer below holds
# in either instance.
#
# This complete copyright notice must be included in any revised version of the
# source code. Additional authorship citations may be added, but existing
# author citations must be preserved.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  2111-1307 USA
#
#

try:
	import matplotlib
	matplotlib.use("AGG")
	import matplotlib.pyplot as plt
	pltcolors=["k","b","g","r","m","c","darkblue","darkgreen","darkred","darkmagenta","darkcyan","0.5"]
except:
	print "Matplotlib not available, some output will not be generated"


from EMAN2 import *
from optparse import OptionParser
from math import *
import os
import sys
import time
import traceback
import numpy as np


# This is used to build the HTML status file. It's a global for convenience
output_html=[]
output_html_com=[]
output_path=None

def append_html(msg,com=False) :
	global output_html,output_html_com
	if com : output_html_com.append(str(msg))
	else : output_html.append(str(msg))
	write_html()


def write_html() :
	global output_html,output_html_com,output_path
	out=file(output_path+"/index.html","w")
	out.write("<html><head><title>EMAN2 Refinement Overview</title></head>\n<body>")
	out.write("\n".join(output_html))
	out.write("<h2>Detailed command log</h2>\n")
	out.write("\n".join(output_html_com))
	out.write("<br></br><hr></hr>Generated by {ver} {date}\n</body></html>".format(ver=EMANVERSION,date=DATESTAMP))

def main():
	progname = os.path.basename(sys.argv[0])
	usage = """e2refine_easy.py [options]

========================================================================
	This is the primary single particle refinement program in EMAN2.1+. It replaces earlier programs such as e2refine.py
and e2refine_evenodd.py. Major features of this program:

 * While a range of command-line options still exist. You should not normally specify more than a few basic requirements. The rest will be auto-selected for you.
 * This program will split your data in half and automatically refine the halves independently to produce a gold standard resolution curve for every step in the refinement.
 * An HTML report file will be generated as this program runs, telling you exactly what it decided to do and why, as well as giving information about runtime, etc while the job is still running.
 * The gold standard FSC also permits us to automatically filter the structure at each refinement step. The resolution you specify is a target, NOT the filter resolution.
 * If --inputavg is specified to use a different stack for alignment and averaging, both stacks must have the same box size and sampling
 * Many of the 'advanced' options are hidden in the e2projectmanager.py GUI, because most users should not need to specify them.

To run this program, you would normally specify only the following options:
  --model=<starting map to seed refinement>
  --input=<lst file referencing phase-flipped particles in HDF format used for alignment>
  --inputavg=<optional lst file with particles to be reconstructed, if not specified --input used>
  OR
  --startfrom=<path to existing refine_xx directory to continue from>

  --targetres=<in A>     Resolution to target in Angstroms in this refinement run. Do not be overoptimistic !
                         Generally begin with something conservative like 25, then use --startfrom and reduce
                         to ~12, only after that try for high (3-8 A). Data permitting, of course. Low resolution
                         attempts will run MUCH faster due to more efficient parameters.
  --speed=<1-7>          Default=5. Larger values will run faster but slightly decrease measured resolution. 7 may
                         be useful for initial refinement runs. 5 is good for routine refinement. You may consider 1
                         when all other possibilities are exhausted and you are ready to push for a final published map.
                         This option modifies several parameters, including the angular step, sep= and even the
                         specific alignment and similarity metric options. The resolution difference between 1 and 7 is
                         generally no more than 10%%.
  --sym=<symmetry>       Symmetry to enforce during refinement (Cn, Dn, icos, oct, cub).
                         Default=c1 (no symmetry)
  --mass=<in kDa>        Putative mass of object in kDa, but as desired volume varies with resolution
                         actual number may vary by as much a ~2x from the true value. The goal is to
                         have a good isosurface in the final map with a threshold of 1.0.
  --parallel=<par spec>  While not strictly required, without this option the refinement will run on a single CPU
                         and you will likely wait a very long time. To use more than one core on a single computer,
                         just say thread:N (eg - thread:4). For other options, like MPI, see:
                         http://blake.bcm.edu/emanwiki/EMAN2/Parallel for details.
  --threads=<ncpu>       For some algorithms, processing in parallel over the network (MPI) works poorly.
                         Running on multiple processors on a single machine may still be worthwhile. If you specify this
                         option, in specific cases it will replace your specified --parallel option. Specify
                         the number of cores that can be used on a single machine.

  Optional:
  --apix=<A/pix>         The value will normally come from the particle data if present. You can override if necessary.
  --sep=<classes/ptcl>   each particle will be put into N classes. Improves contrast at cost of rotational blur.
  --classkeep=<frac>     fraction of particles to use in final average. Default 90%%. Should be >50%%
  --m3dkeep=<frac>       fraction of class-averages to use in 3-D map. Default=auto
  --classautomask        applies an automask when aligning particles for improved alignment
  --m3dpostprocess       <name>:<parm>=<value>:...  An arbitrary processor
                         (e2help.py processors -v2) to apply to the 3-D map after each
                         iteration. Default=none
  --path=<path>          Normally the new directory will be named automatically. If you prefer your own convention
                         you can override, but it may cause minor GUI problems if you break the standard naming
                         convention.

Since many parameters are now selected automatically, if you are curious exactly what the differences are between any
two refinements, on Linux/Mac, you can run, for example, diff refine_01/0_refine_parms.json refine_02/0_refine_parms.json
to compare exactly what changed between the two runs.

========================================================================
  There are numerous additional options based on the original e2refine.py command. These options are not available from
the graphical interface, as it is generally best to let e2refine_easy pick these values for you. Normally you should
not need to specify any of the following other than the ones already listed above:

"""
	parser = EMArgumentParser(usage=usage,version=EMANVERSION)

	#options associated with e2refine.py
	#parser.add_header(name="multirefineheader", help='Options below this label are specific to e2refinemulti', title="### e2refinemulti options ###", row=1, col=0, rowspan=1, colspan=3, mode="refinement")
	#parser.add_header(name="multimodelheader", help='Options below this label are specific to e2refinemulti Model', title="### e2refinemulti model options ###", row=4, col=0, rowspan=1, colspan=3, mode="refinement")
	parser.add_argument("--input", dest="input", default=None,type=str, help="Image stack containing phase-flipped particles used for alignment", guitype='filebox', browser='EMSetsTable(withmodal=True,multiselect=False)', filecheck=False, row=1, col=0, rowspan=1, colspan=3, mode="refinement")
	parser.add_argument("--inputavg", dest="inputavg", default=None,type=str, help="Optional file containing alternate version of the particles to use for reconstruction after alignment", guitype='filebox', browser='EMSetsTable(withmodal=True,multiselect=False)', filecheck=False, row=2, col=0, rowspan=1, colspan=3, mode="refinement")
	parser.add_argument("--model", dest="model", type=str,default=None, help="The map to use as a starting point for refinement", guitype='filebox', browser='EMModelsTable(withmodal=True,multiselect=False)', filecheck=False, row=3, col=0, rowspan=1, colspan=3, mode="refinement")
	parser.add_header(name="orblock", help='Just a visual separation', title="- OR -", row=5, col=0, rowspan=1, colspan=3, mode="refinement")
	parser.add_argument("--startfrom", default=None, type=str,help="Path to an existing refine_xx directory to continue refining from. Alternative to --input and --model.",guitype='filebox', filecheck=False, browser='EMModelsTable(withmodal=True,multiselect=False)', row=7, col=0, rowspan=1, colspan=3, mode="refinement")
	parser.add_argument("--targetres", default=25.0, type=float,help="Target resolution in A of this refinement run. Usually works best in at least two steps (low/medium resolution, then final resolution) when starting with a poor starting model. Usually 3-4 iterations is sufficient.", guitype='floatbox', row=10, col=0, rowspan=1, colspan=1, mode="refinement")
	parser.add_argument("--speed", default=5,type=int,help="(1-7) Balances speed vs precision. Larger values sacrifice a bit of potential resolution for significant speed increases. Set to 1 when really pushing resolution. Set to 7 for initial refinements. default=5", guitype='intbox', row=16, col=1, rowspan=1, colspan=1, mode="refinement")
	parser.add_header(name="required", help='Just a visual separation', title="Required:", row=9, col=0, rowspan=1, colspan=3, mode="refinement")
	parser.add_argument("--sym", dest = "sym", default="c1",help = "Specify symmetry - choices are: c<n>, d<n>, tet, oct, icos.", guitype='strbox', row=10, col=1, rowspan=1, colspan=1, mode="refinement")
	parser.add_argument("--breaksym", action="store_true", default=False,help = "If selected, reconstruction will be asymmetric with sym= specifying a known pseudosymmetry, not an imposed symmetry.", guitype='boolbox', row=11, col=1, rowspan=1, colspan=1, mode="refinement[False]")
	parser.add_argument("--tophat", type=str, default=None,help = "'local' or 'global'. Instead of imposing a final Wiener filter, use a tophat filter (global similar to Relion). local determines local resolution and filters. danger of feature exaggeration", guitype='strbox', row=11, col=0, rowspan=1, colspan=1, mode="refinement['None']")
	parser.add_argument("--treeclassify",default=False, action="store_true", help="Classify using a binary tree.")
	parser.add_argument("--m3dold", action="store_true", default=False,help = "Use the traditional e2make3d program instead of the new e2make3dpar program",guitype='boolbox', row=11, col=2, rowspan=1, colspan=1, mode="refinement")
	parser.add_argument("--iter", dest = "iter", type = int, default=6, help = "The total number of refinement iterations to perform. Default=auto", guitype='intbox', row=10, col=2, rowspan=1, colspan=1, mode="refinement")
	parser.add_argument("--mass", default=0, type=float,help="The ~mass of the particle in kilodaltons, used to run normalize.bymass. Due to resolution effects, not always the true mass.", guitype='floatbox', row=12, col=0, rowspan=1, colspan=1, mode="refinement['self.pm().getMass()']")
	parser.add_header(name="optional", help='Just a visual separation', title="Optional:", row=14, col=0, rowspan=1, colspan=3, mode="refinement")
	parser.add_argument("--apix", default=0, type=float,help="The angstrom per pixel of the input particles. Normally set to 0, which will read the value from the header of the input file", guitype='floatbox', row=16, col=0, rowspan=1, colspan=1, mode="refinement[0]")
	parser.add_argument("--sep", type=int, help="The number of classes each particle can contribute towards (normally 1). Increasing will improve SNR, but produce rotational blurring.", default=-1)
	parser.add_argument("--classkeep",type=float,help="The fraction of particles to keep in each class, based on the similarity score. (default=0.9 -> 90%%)", default=0.9, guitype='floatbox', row=18, col=0, rowspan=1, colspan=1, mode="refinement")
	parser.add_argument("--classautomask",default=False, action="store_true", help="This will apply an automask to the class-average during iterative alignment for better accuracy. The final class averages are unmasked.",guitype='boolbox', row=20, col=1, rowspan=1, colspan=1, mode="refinement")
	parser.add_argument("--prethreshold",default=False, action="store_true", help="Applies a threshold to the volume just before generating projections. A sort of aggressive solvent flattening for the reference.",guitype='boolbox', row=20, col=2, rowspan=1, colspan=1, mode="refinement")
	parser.add_argument("--eulerrefine",default=False, action="store_true", help="Refines Euler angles of class-averages before reconstruction")
	parser.add_argument("--m3dkeep", type=float, help="The fraction of slices to keep in e2make3d.py. Default=0.8 -> 80%%", default=0.8, guitype='floatbox', row=18, col=1, rowspan=1, colspan=1, mode="refinement")
	parser.add_argument("--m3dpostprocess", type=str, default=None, help="Default=none. An arbitrary post-processor to run after all other automatic processing. Maps are autofiltered, so a low-pass filter should not normally be used here.", guitype='comboparambox', choicelist='re_filter_list(dump_processors_list(),"filter.lowpass|filter.highpass|mask")', row=26, col=0, rowspan=1, colspan=3, mode="refinement")
	parser.add_argument("--parallel","-P",type=str,help="Run in parallel, specify type:<option>=<value>:<option>=<value>. See http://blake.bcm.edu/emanwiki/EMAN2/Parallel",default=None, guitype='strbox', row=30, col=0, rowspan=1, colspan=2, mode="refinement[thread:4]")
	parser.add_argument("--threads", default=1,type=int,help="Number of threads to run in parallel on a single computer when multi-computer parallelism isn't useful", guitype='intbox', row=30, col=2, rowspan=1, colspan=1, mode="refinement[4]")
	parser.add_argument("--path", default=None, type=str,help="The name of a directory where results are placed. Default = create new refine_xx")
	parser.add_argument("--verbose", "-v", dest="verbose", action="store", metavar="n", type=int, default=0, help="verbose level [0-9], higner number means higher level of verboseness")
#	parser.add_argument("--usefilt", dest="usefilt", type=str,default=None, help="Specify a particle data file that has been low pass or Wiener filtered. Has a one to one correspondence with your particle data. If specified will be used in projection matching routines, and elsewhere.")

	# options associated with e2project3d.py
#	parser.add_header(name="projectheader", help='Options below this label are specific to e2project', title="### e2project options ###", row=12, col=0, rowspan=1, colspan=3)
	parser.add_argument("--automaskexpand", default=-1, type=int,help="Default=boxsize/20. Specify number of voxels to expand mask before soft edge. Use this if low density peripheral features are cut off by the mask.",guitype='intbox', row=12, col=1, rowspan=1, colspan=1, mode="refinement[-1]" )
	parser.add_argument("--automask3d", default=None, type=str,help="Default=auto. Specify as a processor, eg - mask.auto3d:threshold=1.1:radius=30:nshells=5:nshellsgauss=5.", )
	parser.add_argument("--automask3d2", default=None, type=str,help="Default=none. If specified, this mask will be multiplied by the result of the first mask, eg - using mask.soft to mask out the center of a virus.", )
	parser.add_argument("--projector", dest = "projector", default = "standard",help = "Default=standard. Projector to use with parameters.")
	parser.add_argument("--orientgen", type = str, default=None,help = "Default=auto. Orientation generator for projections, eg - eman:delta=5.0:inc_mirror=0:perturb=1")

	# options associated with e2simmx.py
#	parser.add_header(name="simmxheader", help='Options below this label are specific to e2simmx', title="### e2simmx options ###", row=15, col=0, rowspan=1, colspan=3)
	parser.add_argument("--simalign",type=str,help="Default=auto. The name of an 'aligner' to use prior to comparing the images", default="rotate_translate_tree")
	parser.add_argument("--simaligncmp",type=str,help="Default=auto. Name of the aligner along with its construction arguments",default=None)
	parser.add_argument("--simralign",type=str,help="Default=auto. The name and parameters of the second stage aligner which refines the results of the first alignment", default="auto")
	parser.add_argument("--simraligncmp",type=str,help="Default=auto. The name and parameters of the comparitor used by the second stage aligner.",default=None)
	parser.add_argument("--simcmp",type=str,help="Default=auto. The name of a 'cmp' to be used in comparing the aligned images", default=None)
	parser.add_argument("--simmask",type=str,help="Default=auto. A file containing a single 0/1 image to apply as a mask before comparison but after alignment", default=None)
	parser.add_argument("--shrink", dest="shrink", type = int, default=0, help="Default=auto. Optionally shrink the input particles by an integer amount prior to computing similarity scores. For speed purposes. 0 -> no shrinking", )
	parser.add_argument("--shrinks1", dest="shrinks1", type = int, help="The level of shrinking to apply in the first stage of the two-stage classification process. Default=0 (autoselect)",default=0)
	parser.add_argument("--prefilt",action="store_true",help="Default=auto. Filter each reference (c) to match the power spectrum of each particle (r) before alignment and comparison. Applies both to classification and class-averaging.",default=False)
	parser.add_argument("--cmpdiff",action="store_true",help="Used only in binary tree classification. Use a mask that focus on the difference of two children.",default=False)
	parser.add_argument("--treeincomplete",type=int, help="Used only in binary tree classification. Incompleteness of the tree on each level.Default=0",default=0)

	# options associated with e2classify.py

	# options associated with e2classaverage.py
#	parser.add_header(name="caheader", help='Options below this label are specific to e2classaverage', title="### e2classaverage options ###", row=22, col=0, rowspan=1, colspan=3, mode="refinement")
	parser.add_argument("--classkeepsig", default=False, action="store_true", help="Change the keep (\'--keep\') criterion from fraction-based to sigma-based.")
	parser.add_argument("--classiter", type=int, help="Default=auto. The number of iterations to perform.",default=-1)
	parser.add_argument("--classalign",type=str,default="rotate_translate_tree",help="Default=auto. If doing more than one iteration, this is the name and parameters of the 'aligner' used to align particles to the previous class average.")
	parser.add_argument("--classaligncmp",type=str,help="Default=auto. This is the name and parameters of the comparitor used by the fist stage aligner.",default=None)
	parser.add_argument("--classralign",type=str,help="Default=auto. The second stage aligner which refines the results of the first alignment in class averaging.", default="auto")
	parser.add_argument("--classraligncmp",type=str,help="Default=auto. The comparitor used by the second stage aligner in class averageing.",default=None)
	parser.add_argument("--classaverager",type=str,help="Default=auto. The averager used to generate the class averages. Default is auto.",default=None)
	parser.add_argument("--classcmp",type=str,help="Default=auto. The name and parameters of the comparitor used to generate similarity scores, when class averaging.", default=None)
	parser.add_argument("--classnormproc",type=str,default="normalize.edgemean",help="Default=auto. Normalization applied during class averaging")
	parser.add_argument("--classrefsf",default=False, action="store_true", help="Use the setsfref option in class averaging. This matches the filtration of the class-averages to the projections for easier comparison. Disabled when ampcorrect=flatten is used.",guitype='boolbox', row=20, col=0, rowspan=1, colspan=1, mode="refinement[False]")

	#options associated with e2make3d.py
#	parser.add_header(name="make3dheader", help='Options below this label are specific to e2make3d', title="### e2make3d options ###", row=32, col=0, rowspan=1, colspan=3)
	parser.add_argument("--pad", type=int, dest="pad", default=0, help="Default=auto. To reduce Fourier artifacts, the model is typically padded by ~25 percent - only applies to Fourier reconstruction")
	parser.add_argument("--recon", dest="recon", default="fourier", help="Default=auto. Reconstructor to use see e2help.py reconstructors -v",)
	parser.add_argument("--m3dkeepsig", default=False, action="store_true", help="Default=auto. The standard deviation alternative to the --m3dkeep argument")
#	parser.add_argument("--m3dsetsf", type=str,dest="m3dsetsf", default=None, help="Default=auto. Name of a file containing a structure factor to apply after refinement")
	parser.add_argument("--m3dpreprocess", type=str, default=None, help="Default=auto. Normalization processor applied before 3D reconstruction")

	parser.add_argument("--ampcorrect",choices=['auto','strucfac','flatten','none'],default='auto',help="Will perform amplitude correction via the specified method.  'flatten' requires a target resolution better than 8 angstroms (experimental). 'none' will disable amplitude correction (experimental).", guitype='combobox', row=28, col=0, rowspan=1, colspan=2, mode="refinement['auto']", choicelist="('strucfac','flatten','none','auto')")

	#parser.add_argument("--classweight",choices=['sqrt','count','no_wt'],default='count',help="Alter the weight of each class in the reconstruction (experimental).", guitype='combobox', row=24, col=0, rowspan=1, colspan=2, mode="refinement['count']", choicelist="('sqrt','count','no_wt')")
	#parser.add_argument("--sqrtnorm",action="store_true",default=False, dest="sqrtnorm", help="If set, the sqrt of the number of particles in each class will be used to weight the direct fourier inversion.",guitype='boolbox', row=24, col=0, rowspan=1, colspan=1, mode="refinement")

	#lowmem!
	parser.add_argument("--lowmem", default=True, action="store_true",help="Default=auto. Make limited use of memory when possible - useful on lower end machines")
	parser.add_argument("--ppid", type=int, help="Set the PID of the parent process, used for cross platform PPID",default=-1)

	(options, args) = parser.parse_args()

	if options.threads<=1 :
		if options.parallel!=None and options.parallel[:6]=="thread" :
			options.threads=int(options.parallel[7:])
			print "Note: automatically setting --threads:{}".format(options.threads)
		else: print "WARNING: specifying --threads=<N> (where N is the number of cores to use on a single processor) is strongly recommended, even if already specifying --parallel"

	if options.input!=None and options.model!=None and options.startfrom!=None:
		print "ERROR : You may specify --input and --model  OR  --startfrom, not both"
		sys.exit(1)

	if (options.input==None or options.model==None) and options.startfrom==None :
		print "ERROR : You must specify --input and --model  OR  --startfrom\n"
		parser.print_help()
		sys.exit(1)

	if options.m3dpreprocess==None:
		if not options.m3dold : m3dpreprocess=""
		else: m3dpreprocess="--preprocess normalize.edgemean"
	else:
		m3dpreprocess="--preprocess "+options.m3dpreprocess

	if options.path == None:
		fls=[int(i[-2:]) for i in os.listdir(".") if i[:7]=="refine_" and len(i)==9 and str.isdigit(i[-2:])]
		if len(fls)==0 : fls=[0]
		options.path = "refine_{:02d}".format(max(fls)+1)
	global output_path
	output_path="{}/report".format(options.path)
	try: os.makedirs(output_path)
	except: pass

	# make sure the box sizes match
	if options.input!=None :
		xsize3d=EMData(options.model,0,True)["nx"]
		xsize=EMData(options.input,0,True)["nx"]
		img1 = EMData(options.input,0,True)
		img3 = EMData(options.model,0,True)
		apix1=1.0
		try:
			apix1=img1["apix_x"]
			apix3=img3["apix_x"]
		except:
			apix3=apix1

		if ( xsize3d != xsize or apix3==0 or fabs(fabs(apix1/apix3)-1.0)>.001 ) :
			print "WARNING: the dimensions of the particles (%d @ %1.4f A/pix) do not match the dimensions of the starting model (%d @ %1.4f A/pix). I will attempt to adjust the model appropriately."%(xsize,apix1,xsize3d,apix3)
			try:
				scale=img3["apix_x"]/img1["apix_x"]
				print "Reference is {box3} x {box3} x {box3} at {apix3:1.2f} A/pix, particles are {box2} x {box2} at {apix2:1.2f} A/pix. Scaling by {scale:1.3f}".format(box3=img3["nx"],box2=img1["nx"],apix3=img3["apix_x"],apix2=img1["apix_x"],scale=scale)
			except:
				print "A/pix unknown, assuming scale same as relative box size"
				scale=float(xsize)/xsize3d
			if scale>1 : cmd="e2proc3d.py %s %s/scaled_model.hdf --clip=%d,%d,%d --scale=%1.5f"%(options.model,options.path,xsize,xsize,xsize,scale)
			else :       cmd="e2proc3d.py %s %s/scaled_model.hdf --scale=%1.5f --clip=%d,%d,%d"%(options.model,options.path,scale,xsize,xsize,xsize)
			run(cmd)

			options.model="%s/scaled_model.hdf"%options.path

	if options.speed>7 or options.speed<1 :
		print "ERROR: --speed must be between 1 and 7. Lower numbers will make refinements take longer, but produce slightly better measured resolutions. The default value of 5 is a good balance for typical refinements. When\
satisfied with the results with speed=5 you may consider reducing this number as you try to push for optimal resolution."

	progress = 0.0
	total_procs = 5*options.iter

#	if options.automask3d: automask_parms = parsemodopt(options.automask3d) # this is just so we only ever have to do it
	if options.apix>0 : apix=options.apix
	else:
		if options.startfrom!=None :
			olddb = js_open_dict(options.startfrom+"/0_refine_parms.json")
			apix=options.apix=olddb["apix"]
			if apix<=0 :
				apix=EMData(str(olddb["last_even"]),0,True)["apix_x"]
		else : apix=EMData(options.input,0,True)["apix_x"]

	if options.targetres<apix*2:
		print "ERROR: Target resolution is smaller than 2*A/pix value. This is impossible."
		sys.exit(1)

	logid=E2init(sys.argv,options.ppid)

	###################################
	### This is where we fill in all of the undefined options, and analyse the data
	###################################
	append_html("<h1>e2refine_easy.py report</h1>\n")

	append_html("""<p>If you are curious to see a list of the exact refinement parameters
used, browse to the 0_refine_parms.json file in the refinement directory. You can use 'Info' in the file browser or just read the file directly
(.json files are plain text)""")
	append_html("""<h3>Explantion of Refinement Parameters</h3>""")

	### Prepare initial models
	if options.startfrom!=None:
		try:
			olddb = js_open_dict(options.startfrom+"/0_refine_parms.json")
			run("e2proc3d.py {oldeven} {path}/threed_00_even.hdf".format(oldeven=olddb["last_even"],path=options.path))
			run("e2proc3d.py {oldodd} {path}/threed_00_odd.hdf".format(  oldodd =olddb["last_odd"] ,path=options.path))
			options.input=(str(olddb["input"][0]),str(olddb["input"][1]))
			append_html("<p>Using {oldeven} {oldodd} as starting models without additional randomizing. Input particles are from {infile}</p>".format(oldeven=olddb["last_even"],oldodd=["last_odd"],infile=options.input))

		except:
			print "Error: Cannot find necessary files in ",options.startfrom
			sys.exit(1)
	else:
		try:
			if options.targetres>20 : randomres=options.targetres*1.2
			elif options.targetres>12 : randomres=options.targetres*1.5
			else : randomres=options.targetres*2.0

			run("e2proc3d.py {model} {path}/threed_00_even.hdf --process=filter.lowpass.randomphase:cutoff_freq={freq} --apix={apix}".format(model=options.model,path=options.path,freq=1.0/(randomres),apix=apix))
			run("e2proc3d.py {model} {path}/threed_00_odd.hdf --process=filter.lowpass.randomphase:cutoff_freq={freq} --apix={apix}" .format(model=options.model,path=options.path,freq=1.0/(randomres),apix=apix))
			append_html("""<p>Randomizing the Fourier phases of <i>{model}</i> at resolutions higher than {res:1.1f} &Aring;. If the final achieved resolution is not at least ~{resb:1.1f} &Aring;, then the
gold standard resolution assessment is not valid, and you need to re-refine, starting with a lower resolution target.</p>
<p>Input particles are from <i>{infile}</i></p>""".format(model=options.model,infile=options.input,res=randomres,resb=randomres*0.9))
			options.input=image_eosplit(options.input)
			if options.inputavg!=None : options.inputavg=image_eosplit(options.inputavg)
		except:
			traceback.print_exc()
			print "Error: Unable to prepare input files"
			sys.exit(1)

	repim=EMData(options.input[0],0)		# read a representative image to get some basic info
	if repim.has_attr("ctf") : hasctf=True
	else: hasctf=False
	nx=repim["nx"]

	# Fill in optional parameters

	if hasctf and apix<6.5:
		if os.path.exists("strucfac.txt") :
# 			append_html("<p>Using the standard 3-stage filter on the reconstruction: 'filter.lowpass.autob' which flattens the overall falloff of the structure factor in the 4-15 A range, \
# 'setsf strucfac.fromdata.txt' which will force the low resolution structure factor to match the profile determined from the data, and filter.wiener.byfsc which performs a low-pass Wiener \
# filter based on the computed FSC curve between even/odd maps. Combined, this process produces properly resolution-limited maps with very accurate structure factors. The only real free \
# parameter is the amplitude contrast setting when determining the structure factor.</p>")
# 			postprocess="--postprocess filter.lowpass.autob"
# 			m3dsetsf="--setsf strucfac.fromdata.txt"
			append_html("""<p>Several different methods can be used for final amplitude correction in cryoEM. The most accurate of these is to take the final\
 structure and filter it so its 1-D power spectrum matches a known 1-D power spectrum from X-ray solution scattering or other source. This 'ideally filtered'\
 structure is then low-pass filtered based on the same FSC curve used to measure resolution between the independent even/odd models used in the Gold Standard\
 resolution criterion. This is the normal method EMAN2 will use for final corrections. Other choices are possible as well, however. EMAN2 generates an\
 experimental structure factor directly from your data as part of the CTF fitting procedure. The low resolution portion is drawn directly from your data\
 and the high resolution portion (generally past 14-18 A) is an empirical structure factor for proteins in general with a mix of alpha and beta components.\
 This works quite well for most proteins, however is not completely appropriate for hybrid molecules with nucleotides (like ribosomes), but will generally\
 still produce reasonable structures. Note that this is just a standard linear filter, which is not really changing the 3-D structure at all, it's really\
 just changing how that structure is portrayed.</p>
 <p>The structure factor produced when CTF fitting is stored in <i>strucfac.txt</i> in the project directory. If you replace this file with a structure factor from
 some other source, it will be used to process all subsequent particles.</p>""")
			postprocess=""
			m3dsetsf="--setsf strucfac.txt"
		else :
			append_html("<p>No data-based structure factor was found in the project. Computing one during CTF correction is highly recommended. Falling back to two-stage filtration: \
'filter.lowpass.autob' which flattens the overall falloff of the structure factor in the 4-15 A range, \
and filter.wiener.byfsc which performs a low-pass Wiener filter based on the computed FSC curve between even/odd maps. \
While this filtration can work reasonably well, you may find that it over-exagerates low-resolution terms over multiple iterations. \
To avoid this, compute a structure factor.</p>")
			postprocess="--m3dpostprocess filter.lowpass.autob"
			m3dsetsf=""
	elif apix>=6.5 :
		append_html("<p>A/pix value >6.5. With such a large A/pix some of the post-processing sharpening filters need to be disabled, so you are effectively\
 getting an unfiltered output map other than the automatic CTF correction.</p>")
		postprocess=""
		m3dsetsf=""
	else:
		append_html("<p>No CTF information found in the input data. Note that EMAN2 cannot perform optimal reconstructions using phase-flipped particles from other \
software. Part of EMAN2's CTF correction process is measuring the SSNR of the particle data, an estimate of the low resolution structure factor and other parameters \
which are used to provide more accurate orientations, filters, etc. during processing. Since CTF information isn't present, we are limited to a basic refinement. \
output maps will be low-pass filtered based on resolution, but note that there will be no B-factor correction, so this will result in over-filtration of the final \
maps.")
		postprocess=""
		m3dsetsf=""

	if options.tophat=="global":
		append_html("<p>You are using the --tophat=global option, which modifies the final filter applied to 3-D maps. The default behavior is to apply a final Wiener filter \
based on the FSC curve used to compute the resolution. This Wiener filter gives a map which in theory reduces noise and filters the map to get as close as possible to \
what you should be able to see at the specified resolution. However, this means that some features, such as sidechains and the pitch of alpha-helices may be somewhat \
smoothed out. The alternative is a 'tophat' or 'sharp' filter imposed at the cutoff (0.143) FSC value, and is what Relion imposes on its final maps. This produces \
maps which look prettier, with more apparent side-chains at high resolution, but runs the risk that some of these features may be artifacts of the sharp filter.")
		tophat="--tophat=global"
	elif options.tophat=="local":
		append_html("<p>You are using the --tophat=local option, which modifies the final filter applied to 3-D maps. The default behavior is to apply a final Wiener filter \
based on the FSC curve evenly across the map. This Wiener filter gives a map which in theory reduces noise and filters the map to get as close as possible to \
what you should be able to see at the specified resolution. However, this means that some features, such as sidechains and the pitch of alpha-helices may be somewhat \
smoothed out. This option will compute a local resolution map, by computing local FSCs in different regions of the map. This local resolution map is then applied as a set \
of local tophat filters across the map. If some regions of the map have better resolution than others, these regions will preserve more detail, and vice-versa. This may \
produce some local artifacts, but generally seems to work quite well.")
		tophat="--tophat=local"
	else:
		append_html("<p>You are not using the --tophat option, meaning a final Wiener filter \
based on the FSC curve used to compute the resolution is applied to the reconstruction. This Wiener filter gives a map which in theory reduces noise and filters the map to get as close as possible to \
what you should be able to see at the specified resolution. However, this means that some features, such as sidechains and the pitch of alpha-helices may be somewhat \
smoothed out. The alternative is a 'tophat' or 'sharp' filter imposed at the cutoff (0.143) FSC value, and is what Relion imposes on its final maps. This produces \
maps which look prettier, with more apparent side-chains at high resolution, but runs the risk that some of these features may be artifacts of the sharp filter.")

		tophat=""

	if options.sym.lower()=="c1" and options.targetres>15 :
		append_html("<p>Since you are not imposing symmetry, and your target resolution is low, there is a risk that the even and odd maps produced during refinement \
may become significantly misaligned (more than a few degrees). Small misalignments are automatically corrected after each refinement iteration, but larger misalignments \
are not. Given this risk, you may wish to check the even and odd maps after your refinement to insure that they are oriented identically. If they aren't, you may consider \
running another round of e2refine_easy, using either the even or the odd map (rather than the averaged map) as a starting model, and reducing (smaller number, better \
resolution) --targetres by ~25%. Hopefully this will eliminate the misalignment, and give a correct averaged map.</p>")

	if options.sep<1 :
		options.sep=max(1,6-options.speed)
		if options.sep>1 : append_html("<p>Based on your selected --speed, I am setting --sep {sep}. This puts each particle into its {sep} best orientations. If the angular sampling is finer than required \
to achieve the specified resolution, then a certain amount of rotational 'smearing' of each particle will help improve SNR in the resulting map without the 'smearing' degrading \
the actual map quality. This can achieve maximum liklihood-like effects without the substantial compuations this can entail. If you are concerned by this, or have many more particles than \
are really required to achieve the targeted resolution, you may consider manually specifiying --sep 1, which will override this automatic behavior.</p>".format(sep=options.sep))

	if options.orientgen==None :
		# target resolution worse than 1/2 Nyquist
		if options.targetres>apix*4 :
			effbox=nx*apix*2/options.targetres
#			astep=89.999/ceil(90.0/sqrt(4300/effbox))		# This rounds to the best angular step divisible by 90 degrees. Old way without speed
			astep=89.99/ceil(90.0*9.0/((options.speed+3.0)*sqrt(4300/effbox)))		# This rounds to the best angular step divisible by 90 degrees
			options.orientgen="eman:delta={:1.5f}:inc_mirror=0:perturb=0".format(astep)
			if options.classiter<0 :
				if options.targetres>12.0 :
					classiter=3
					append_html("<p>Your desired resolution is below 1/2 Nyquist, and you requested a resolution of <12 A, so we will initially set --classiter to 3. This will help \
low resolution refinements converge more rapidly. If you run more than 2 iterations and the map seems to have converged fairly well, this will be decreased.</p>")
				elif options.targetres>8.0 :
					classiter=2
					append_html("<p>Your desired resolution is below 1/2 Nyquist, and you requested a resolution of <8 A, so we will initially set --classiter to 2. This balances \
rapid convergence with the subnanometer resolution goal. If you run more than 2 iterations and the map seems to have converged fairly well, this will be decreased.</p>")
				else :
					classiter=1
					append_html("<p>Your desired resolution is below 1/2 Nyquist, and you requested a high resolution, so we will initially set --classiter to 1. This balances \
rapid convergence with the resolution goal. If you run more than 2 iterations and the map seems to have converged fairly well, this will be decreased.</p>")

			append_html("<p>Automatically selecting a good angular spacing for the refinement adjusted for your selected --speed. The resolution you are requesting ({}) is quite conservative given the \
sampling ({}) of your data. If this is simply an initial refinement designed to get the overall shape of the structure correct, this is fine. If this is your final resolution \
target, you may wish to consider downsampling the data and importing into a new project, as too much oversampling is both <i>extremely</i> inefficient, and in some cases will \
even lead to worse structures. Based on your requested resolution and box-size, I will use an angular sampling of {:1.2f} deg. For details, please see \
<a href=http://blake.bcm.edu/emanwiki/EMAN2/AngStep>http://blake.bcm.edu/emanwiki/EMAN2/AngStep</a></p>".format(options.targetres,apix,astep))

		# target resolution between 1/2 and 3/4 Nyquist
		elif options.targetres>apix*8.0/3.0 :
			astep=89.99/ceil(90.0*9.0/((options.speed+3.0)*sqrt(4300/nx)))		# This rounds to the best angular step divisible by 90 degrees
			options.orientgen="eman:delta={:1.5f}:inc_mirror=0:perturb=0".format(astep)
			append_html("<p>Based on your requested resolution and box-size, modified by --speed,  I will use an angular sampling of {:1.2f} deg. For details, please see \
<a href=http://blake.bcm.edu/emanwiki/EMAN2/AngStep>http://blake.bcm.edu/emanwiki/EMAN2/AngStep</a></p>".format(astep))
			if options.classiter<0 :
				classiter=1
				append_html("<p>Your desired resolution is between 1/2 and 3/4 Nyquist, so we will set --classiter to 1. Leaving this above 0 \
will help avoid noise bias in early rounds, but it may be reduced to zero if convergence seems to have been achieved.</p>")

		# target resolution is beyond 3/4 Nyquist
		else :
			if options.classiter<0 :
				classiter=1
				append_html("<p>Your desired resolution is beyond 3/4 Nyquist. Regardless, we will set --classiter to 1 initially. Leaving this above 0 \
will help avoid noise bias, but it may be reduced to zero if convergence seems to have been achieved.</p>")
			astep=89.99/ceil(90.0*9.0/((options.speed+3.0)*sqrt(4300/nx)))		# This rounds to the best angular step divisible by 90 degrees
			options.orientgen="eman:delta={:1.5f}:inc_mirror=0:perturb=0".format(astep)
			append_html("<p>The resolution you are requesting is beyond 2/3 Nyquist. This is normally not recommended, as it represents insufficient sampling to give a good representation of your \
reconstructed map, and resolution can be difficult to accurately assess. The reconstruction will proceed, but generally speaking your A/pix should be less than 1/3 the targeted resolution. \
Based on your requested resolution and box-size, modified by --speed, I will use an angular sampling of {:1.2f} deg. For details, please see \
<a href=http://blake.bcm.edu/emanwiki/EMAN2/AngStep>http://blake.bcm.edu/emanwiki/EMAN2/AngStep</a></p>".format(astep))
	else :
		append_html("<p>Using your specified orientation generator with angular step. You may consider reading this page: <a href=http://blake.bcm.edu/emanwiki/EMAN2/AngStep>http://blake.bcm.edu/emanwiki/EMAN2/AngStep</a></p></p>")
		try:
			astep=float(parsemodopt(options.orientgen)[1]["delta"])
			if astep*floor(90.0/astep)<89.9:
				append_html("<p>WARNING: your specified angular step would not quite reach the equator when generating angles. For higher symmetries (icos or oct) this may not be a problem, but for lower symmetries, it is \
important to use an angular step which is 90/integer.</p>")
		except:
			append_html("<p>Could not extract an angular step from your orientation generator. This means I won't be able to use it for optimal rotational averaging during reconstruction</p>")
			astep=0
		if options.classiter<0 : classiter=1
	if options.breaksym : options.orientgen=options.orientgen+":breaksym=1"

	if options.simaligncmp==None : options.simaligncmp="ccc"
	if options.simralign=="auto" and options.speed<7:
		if options.targetres>=11.0 or options.speed>5:
			options.simralign="refine"
			if options.simraligncmp==None : options.simraligncmp="ccc"
		else :
			options.simralign="refine"
			# previously, the default minres/maxres was 500,10. In testing showed that 50,5 produced noticably improved results on IP3R
			# changing the default to match targetres
			if options.simraligncmp==None : 
				options.simraligncmp="frc:zeromask=1:snrweight=1:minres=80:maxres={}".format(options.targetres)
			
		simralign="--ralign {} --raligncmp {}".format(options.simralign,options.simraligncmp)
	elif options.speed==7 or options.simralign.lower()==None or options.simralign.lower()=="none":
		simralign=" "
	else: simralign="--ralign {} --raligncmp {}".format(options.simralign,options.simraligncmp)

	if options.simcmp==None :
		if options.targetres>18.0 or not hasctf: options.simcmp="frc:maxres={}".format(options.targetres)
		elif options.targetres>11.0 : options.simcmp="frc:snrweight=1:maxres={}".format(options.targetres)	# no zeromask to avoid top/side errors at lower resolutions
		else : options.simcmp="frc:snrweight=1:maxres={}".format(max(7.0,options.targetres))

	if options.shrink==0 : shrink=""
	else : shrink="--shrink {}".format(options.shrink)
	if options.shrinks1==0 :
		if nx>=256 and (nx/4)%2==0 : shrinks1="--shrinks1 4"
		elif nx>=96 and (nx/2)%2==0 : shrinks1="--shrinks1 2"
		else : shrinks1=""
	else: shrinks1="--shrinks1 {}".format(options.shrinks1)

	if options.classaligncmp==None :
		options.classaligncmp="ccc"

	if options.classralign=="auto" :
		if options.targetres>15 or not hasctf or options.speed>5:
			options.classralign="refine"
			if options.classraligncmp==None : options.classraligncmp="ccc"
		else :
			options.classralign="refine"
			if options.classraligncmp==None : options.classraligncmp="frc:snrweight=1:zeromask=1:minres=80:maxres={}".format(options.targetres)
		classralign="--ralign {ralign} --raligncmp {raligncmp}".format(ralign=options.classralign,raligncmp=options.classraligncmp)
	elif options.classralign.lower()==None or  options.classralign.lower()=="none":
		classralign=" "
	else: classralign="--ralign {ralign} --raligncmp {raligncmp}".format(ralign=options.classralign,raligncmp=options.classraligncmp)

	if options.classaverager==None :
#		if hasctf and options.targetres<15 : options.classaverager="ctfw.auto"
		if hasctf and options.targetres<15 : options.classaverager="ctf.weight"		# changed default on 10/23/14
		else : options.classaverager="mean"

	if options.classcmp==None :
		if hasctf : options.classcmp="frc:snrweight=1"
		else : options.classcmp="ccc"

	if options.pad<nx :
		options.pad=good_size(nx*1.7)
		if options.pad>1024 :
			print "Warning: padding for Fourier reconstruction is now {}, meaning quite a lot of memory will \
be required for reconstructions, and they may be very slow. Padding in Fourier space is largely performed to avoid high-radius \
Fourier artifacts, and a gradual radial density falloff. If you feel this value is too large, you can manually specify a value \
with the --pad option.".format(options.pad)
	else :
		if options.pad<nx*1.4 :
			print "Warning: the --pad value you specified is less than 1.4x the box size. We normlly recommend using a --pad \
value at least 1.5x the box size to avoid Fourier artifacts in the reconstruction, particularly at high radius. If you already \
have a box which is large compared to the particle, then a smaller value may be fine. Also, if your particle is very large and there \
are memory concerns, using a smaller pad option may be the only reasonable alternative, though some artifacts will be inevitable."

	##################################
	### prepare for the run
	##################################
	### Convert some of the command-line options to more conventient strings for command generation
	if options.verbose>0 : verbose="--verbose {}".format(options.verbose)
	else: verbose=""

	if options.threads<1 :
		print "WARNING: threads set to an invalid value. Changing to 1, but you should really provide a reasonable number."
		options.threads=1

	if options.parallel!=None : parallel="--parallel {}".format(options.parallel)
	elif options.threads>1: parallel="--parallel thread:{}".format(options.threads)
	else: parallel=""

	if options.prefilt : prefilt="--prefilt"
	else: prefilt=""

	if options.cmpdiff : cmpdiff="--cmpdiff"
	else: cmpdiff=""

	if options.simmask!=None :
		makesimmask=False
		simmask="--mask {}".format(options.simmask)
		append_html("<p>{simmask} was specified, so I will not automatically create a mask for each iteration.</p>".format(simmask=simmask))
	else:
		makesimmask=True
		simmask="--mask {path}/simmask.hdf".format(path=options.path)
		append_html("<p>As particles get translated during alignment, the total amount of noise present in the aligned particle can change\
 significantly. While this isn't a very large effect for similarity metrics like fsc, it can cause a bias in the reconstruction. Similarly\
 if we use a different mask derived for each projection to combat this problem (as with the zeromask=1 option in some comparators, then each\
 projection masks out a different fraction of the image causing some orientations to be preferred. To combat both effects, I will compute a\
 single aggregate mask from all of the projections in each iteration, and use it as <i>{simmask}</i>. The mask is autogenerated and overwritten\
 after each iteration. The only way to completely disable this behavior\
 is to specify --simmask yourself with a file containing all 1.0 pixels.</p>".format(simmask=simmask))

	if options.classrefsf : classrefsf="--setsfref"
	else: classrefsf=""

	if options.inputavg!=None:
		cainput=["--input {} --usefilt {}".format(options.inputavg[ii],options.input[ii]) for ii in (0,1)]
	else:
		cainput=["--input {}".format(options.input[ii]) for ii in (0,1)]


	if options.classautomask : classautomask="--automask"
	else: classautomask=""

	if options.classkeepsig : classkeepsig="--keepsig"
	else: classkeepsig=""

	#if options.classralign!=None : classralign="--ralign {} --raligncmp {}".format(options.classralign,options.classraligncmp)
	#else: classralign=""

	if options.m3dkeepsig : m3dkeepsig="--keepsig"
	else: m3dkeepsig=""

	if options.automask3d==None : amask3d=""
	else : amask3d="--automask3d "+options.automask3d

	if options.automask3d2==None : amask3d2=""
	else : amask3d2="--automask3d2 "+options.automask3d2


	if options.m3dpostprocess==None : m3dpostproc=""
	else : m3dpostproc="--m3dpostprocess "+options.m3dpostprocess

	if options.prethreshold : prethreshold="--prethreshold"
	else : prethreshold=""

	# store the input arguments forever in the refinement directory
	db = js_open_dict(options.path+"/0_refine_parms.json")
	db.update(vars(options))
	db["commandline"]=" ".join(sys.argv)
	db["timestamp"]=str(time.ctime())
	db["astep"]=astep
	db["apix"]=apix

	print "NOTE: you can check the progress of the refinement at any time by opening this URL in your web-browser:  file://{}/index.html".format(os.path.abspath(output_path))

	append_html("""<h3>Analysis of Refinement Results</h3>""")

	append_html("""<h4>Convergence Analysis</h4> <p>The plot below shows the FSC computed between iterations for both even and
odd particle subsets. It is not a measure of resolution, but is a measure of how much the individual even/odd maps are changing
from one iteration to the next. In a perfect world, this plot would eventually be 1.0 indicating no change from one iteration
to the next after the refinement converges. In reality we normally expect only a pseudoconvergence, where the curves approach a
final shape but do not actually become 1.0. This plot is automatically updated after each iteration.</p><br><a href=converge.pdf><img src=converge.png></a><br>""")

	append_html("""<h4>Gold Standard Resolution</h4> <p>e2refine_easy computes a true gold-standard resolution as part of its stanard
processing. This curve is also used as a basis for filtering the maps. Resolution can be gagued as the point at which this curve falls
below a value of 0.143. Note that when computing FSCs in other situations, for example, when comparing the final map produced with
all of the particle data to a higher resolution crystal structure, the more stringent 0.5 (actually 0.4) criterion must be used. Don't
overinterpret these plots. The FSC plots themselves contain some noise, so there is some uncertainty in any resolution value.</p>
<p>Also note that these curves are highly dependent on the mask used prior to FSC computation. See the next plot for more information
on this. This iteration comparison plot uses a relatively tight mask designed to be somewhat comparable to that used in Relion (though Relion is still more aggressive in masking).</p>
<p>These resolution curves should start at ~1 at low resolution, fall smoothly to ~0, then remain at roughly zero. The vertical range
of an FSC plot is -1 to 1. Some negative oscillations after reaching zero are normal. If the FSC curve falls towards zero, but then
rises again at high resolution this is an indication of an artifact, and the resolution values may not be trustworthy. Possible sources
include, but are not limited to: mask too tight, box size too small, reconstruction artifacts, ...  If you observe an "unhealthy" FSC
curve here, please check the next plot, and see if the curve with a looser mask appears "healthier".</p>
<br><a href=resolution.pdf><img src=resolution.png></a><br>""")

	append_html("""<h4>Mask Comparison</h4> <p>As mentioned in the previous section, masking can have a significant impact on cited
resolution. For a while users who followed Relion's suggested procedure would find resolution values slightly better than those
produced by EMAN. This was due to Relion's use of a very tight mask, often extending into the core of the structure.  EMAN2.1 now
generates 3 FSC curves automatically: 1) completely unmasked, 2) a loose, "conservative" mask and 3) a tighter mask which should be
somewhat similar to Relion 1.3 using its suggested parameters (though EMAN is still slightly more conservative). The plot below
compares these three FSC curves for the last completed iteration. The relevant mask files are also stored in the refinement folder as
mask.hdf and mask_tight.hdf. Note: if visualizing 3-D masks, it is generally a good idea to display as 2-D slices (single image view in
browser) rather than using isosurfaces, so any internal features of the mask can be readily observed.</p>
<p>If you see an 'unheathly' looking unmasked curve (rising at high resolution, etc.), particularly if the masked curves look healthy, this can
indicate that your box-size is too small, and you are getting Fourier artifacts at the edge. It could also indicate some other problem, so if
re-extracting your boxed particles with a somewhat larger box doesn't fix the problem, please contact us, and we will help you to debug the problem.</p>
<p>If your goal is to compare EMAN2.1 results to Relion, the only way to reliably accomplish this is to extract the unmasked even and odd
gold-standard maps from both EMAN and RELION, mask them with exactly the same mask, then compute the FSC. For visual comparison of the final
gold-standard maps from both packages, you must insure that they are both identically filtered. This can be done easily by matching the 1-D
power spectrum of one of the maps to the other. For example <i>e2proc3d.py map_eman.hdf map_eman_cmp.hdf --process filter.matchto:to=map_relion.mrc</i></p>

<br><a href=resolution_masks.pdf><img src=resolution_masks.png></a><br>
	""")
	xticklocs=[i for i in (.01,.05,.0833,.125,.1667,.2,.25,.3333,.4,.5) if i<1.0/(2.0*apix)]
	xticklbl=["1/100","1/20","1/12","1/8","1/6","1/5","1/4","1/3","1/2.5","1/2"][:len(xticklocs)]
	yticklocs=(0.0,.125,.143,.25,.375,.5,.625,.75,.875,1.0)
	yticklbl=("0"," ","0.143","0.25"," ","0.5"," ","0.75"," ","1.0")
	yticklocs2=(0.0,.125,.25,.375,.5,.625,.75,.875,1.0)
	yticklbl2=("0"," ","0.25"," ","0.5"," ","0.75"," ","1.0")

	try: initclassiter=classiter
	except: classiter=options.classiter
	### Actual refinement loop ###
	for it in range(1,options.iter+1) :
		append_html("<h4>Beginning iteration {} at {}</h4>".format(it,time.ctime(time.time())),True)

		# adjustments to classiter
		if options.classiter<0 and classiter==initclassiter:
			if (options.iter>3 and it==options.iter-2) or (options.iter>2 and it==options.iter-1) or (options.iter==2 and it==2):
				if initclassiter>1 : classiter=1
				elif initclassiter==1 : classiter=0
			append_html("<p>*** Changing classiter from {} to {} ***</p>".format(initclassiter,classiter),True)


		### 3-D Projections
		# Note that projections are generated on a single node only as specified by --threads
		append_html("<p>* Generating 2-D projections of even/odd 3-D maps",True)
		cmd = "e2project3d.py {path}/threed_{itrm1:02d}_even.hdf  --outfile {path}/projections_{itr:02d}_even.hdf -f --projector {projector} --orientgen {orient} --sym {sym} {prethr} --parallel thread:{threads} {verbose}".format(
			path=options.path,itrm1=it-1,itr=it,projector=options.projector,orient=options.orientgen,sym=options.sym,prethr=prethreshold,threads=options.threads,verbose=verbose)
		run(cmd)
		cmd = "e2project3d.py  {path}/threed_{itrm1:02d}_odd.hdf --outfile {path}/projections_{itr:02d}_odd.hdf -f --projector {projector} --orientgen {orient} --sym {sym} {prethr} --parallel thread:{threads} {verbose}".format(
			path=options.path,itrm1=it-1,itr=it,projector=options.projector,orient=options.orientgen,sym=options.sym,prethr=prethreshold,threads=options.threads,verbose=verbose)
		run(cmd)
		progress += 1.0
		E2progress(logid,progress/total_procs)

		### We may need to make our own similarity mask file for more accurate particle classification
		if makesimmask :
			av=Averagers.get("minmax",{"max":1})
			nprj=EMUtil.get_image_count("{path}/projections_{itr:02d}_odd.hdf".format(path=options.path,itr=it))
			print "Mask from {} projections".format(nprj)
			for i in xrange(nprj):
				a=EMData("{path}/projections_{itr:02d}_even.hdf".format(path=options.path,itr=it),i)
				av.add_image(a)
				a=EMData("{path}/projections_{itr:02d}_odd.hdf".format(path=options.path,itr=it),i)
				av.add_image(a)
			msk=av.finish()
#			msk.process_inplace("threshold.binary",{"value":msk["sigma"]/50.0})
			msk.process_inplace("threshold.notzero")
			msk.write_image("{path}/simmask.hdf".format(path=options.path),0)

		if options.treeclassify:
			### Classify using a binary tree
			append_html("<p>* Classify each particle using a binary tree generated from the projections</p>",True)
			cmd = "e2classifytree.py {path}/projections_{itr:02d}_even.hdf {inputfile} --output={path}/classmx_{itr:02d}_even.hdf  --nodes {path}/nodes_{itr:02d}_even.hdf --cmp {simcmp} --align {simalign} --aligncmp {simaligncmp} {simralign} {cmpdiff} --incomplete {incomplete} {parallel}".format(path=options.path,itr=it,inputfile=options.input[0],simcmp=options.simcmp,simalign=options.simalign,simaligncmp=options.simaligncmp,simralign=simralign,cmpdiff=cmpdiff,incomplete=options.treeincomplete, parallel=parallel)
			run(cmd)
			progress += 1.0

			cmd = "e2classifytree.py {path}/projections_{itr:02d}_odd.hdf {inputfile} --output={path}/classmx_{itr:02d}_odd.hdf  --nodes {path}/nodes_{itr:02d}_odd.hdf --cmp {simcmp} --align {simalign} --aligncmp {simaligncmp} {simralign} {cmpdiff} --incomplete {incomplete} {parallel}".format(path=options.path,itr=it,inputfile=options.input[1],simcmp=options.simcmp,simalign=options.simalign,simaligncmp=options.simaligncmp,simralign=simralign,cmpdiff=cmpdiff,incomplete=options.treeincomplete,parallel=parallel)
			run(cmd)
			progress += 1.0
			E2progress(logid,progress/total_procs)
		else:

			### Simmx
			#FIXME - Need to combine simmx with classification !!!

			append_html("<p>* Computing similarity of each particle to the set of projections using a hierarchical scheme. This will be the basis for classification.</p>",True)
			cmd = "e2simmx2stage.py {path}/projections_{itr:02d}_even.hdf {inputfile} {path}/simmx_{itr:02d}_even.hdf {path}/proj_simmx_{itr:02d}_even.hdf {path}/proj_stg1_{itr:02d}_even.hdf {path}/simmx_stg1_{itr:02d}_even.hdf --saveali --cmp {simcmp} \
	--align {simalign} --aligncmp {simaligncmp} {simralign} {shrinks1} {shrink} {prefilt} {simmask} {verbose} {parallel}".format(
				path=options.path,itr=it,inputfile=options.input[0],simcmp=options.simcmp,simalign=options.simalign,simaligncmp=options.simaligncmp,simralign=simralign,
				shrinks1=shrinks1,shrink=shrink,prefilt=prefilt,simmask=simmask,verbose=verbose,parallel=parallel)
			run(cmd)
			cmd = "e2simmx2stage.py {path}/projections_{itr:02d}_odd.hdf {inputfile} {path}/simmx_{itr:02d}_odd.hdf {path}/proj_simmx_{itr:02d}_odd.hdf {path}/proj_stg1_{itr:02d}_odd.hdf {path}/simmx_stg1_{itr:02d}_odd.hdf --saveali --cmp {simcmp} \
	--align {simalign} --aligncmp {simaligncmp} {simralign} {shrinks1} {shrink} {prefilt} {simmask} {verbose} {parallel}".format(
				path=options.path,itr=it,inputfile=options.input[1],simcmp=options.simcmp,simalign=options.simalign,simaligncmp=options.simaligncmp,simralign=simralign,
				shrinks1=shrinks1,shrink=shrink,prefilt=prefilt,simmask=simmask,verbose=verbose,parallel=parallel)
			run(cmd)
			progress += 1.0
			E2progress(logid,progress/total_procs)

			### Classify
			append_html("<p>* Based on the similarity values, put each particle in to 1 or more classes (depending on --sep)</p>",True)
			cmd = "e2classify.py {path}/simmx_{itr:02d}_even.hdf {path}/classmx_{itr:02d}_even.hdf -f --sep {sep} {verbose}".format(
				path=options.path,itr=it,sep=options.sep,verbose=verbose)
			run(cmd)
			cmd = "e2classify.py {path}/simmx_{itr:02d}_odd.hdf {path}/classmx_{itr:02d}_odd.hdf -f --sep {sep} {verbose}".format(
				path=options.path,itr=it,sep=options.sep,verbose=verbose)
			run(cmd)
			progress += 1.0
			E2progress(logid,progress/total_procs)

		### Class-averaging

		# we need to decide on a postprocessing amplitude correction scheme here, because it may impact class-averaging
		if options.ampcorrect=="auto":
			try:
				# In the first iteration if targetres is 7 or better, we give it the benefit of the doubt and use flatten (changed 5/11/16)
				if options.targetres<=7 and lastres[1]<9.0 : ampcorrect="flatten"
				else: ampcorrect="strucfac"
			except:
				ampcorrect="strucfac"		# first iteration
			append_html("""<p>Auto amplitude correction using mode:{} in this iteration</p>""".format(ampcorrect))
		else: ampcorrect=options.ampcorrect

		if ampcorrect!="strucfac" and classrefsf!="" :
			classrefsf=""
			print "Warning: Not using structure factor amplitude correction, so disabling classrefsf option"
			append_html("<p>Warning: classrefsf option requires 'strucfac' amplitude correction. Since this is not being used either by intent or due to the high resolution of the map, 'classrefsf' has been disabled.</p>")

		append_html("<p>* Iteratively align and average all of the particles within each class, discarding the worst fraction</p>",True)
		cmd="e2classaverage.py {inputfile} --classmx {path}/classmx_{itr:02d}_even.hdf --decayedge --storebad --output {path}/classes_{itr:02d}_even.hdf --ref {path}/projections_{itr:02d}_even.hdf --iter {classiter} \
-f --resultmx {path}/cls_result_{itr:02d}_even.hdf --normproc {normproc} --averager {averager} {classrefsf} {classautomask} --keep {classkeep} {classkeepsig} --cmp {classcmp} \
--align {classalign} --aligncmp {classaligncmp} {classralign} {prefilt} {verbose} {parallel}".format(
			inputfile=cainput[0], path=options.path, itr=it, classiter=classiter, normproc=options.classnormproc, averager=options.classaverager, classrefsf=classrefsf,
			classautomask=classautomask,classkeep=options.classkeep, classkeepsig=classkeepsig, classcmp=options.classcmp, classalign=options.classalign, classaligncmp=options.classaligncmp,
			classralign=classralign, prefilt=prefilt, verbose=verbose, parallel=parallel)
		run(cmd)
		cmd="e2classaverage.py {inputfile} --classmx {path}/classmx_{itr:02d}_odd.hdf --decayedge --storebad --output {path}/classes_{itr:02d}_odd.hdf --ref {path}/projections_{itr:02d}_odd.hdf --iter {classiter} \
-f --resultmx {path}/cls_result_{itr:02d}_odd.hdf --normproc {normproc} --averager {averager} {classrefsf} {classautomask} --keep {classkeep} {classkeepsig} --cmp {classcmp} \
--align {classalign} --aligncmp {classaligncmp} {classralign} {prefilt} {verbose} {parallel}".format(
			inputfile=cainput[1], path=options.path, itr=it, classiter=classiter, normproc=options.classnormproc, averager=options.classaverager, classrefsf=classrefsf,
			classautomask=classautomask,classkeep=options.classkeep, classkeepsig=classkeepsig, classcmp=options.classcmp, classalign=options.classalign, classaligncmp=options.classaligncmp,
			classralign=classralign, prefilt=prefilt, verbose=verbose, parallel=parallel)
		run(cmd)
		progress += 1.0
		E2progress(logid,progress/total_procs)

		### Refine Euler angles of class-averages
		if options.eulerrefine and it>1 :
			cmd="e2euler_refine.py --input {path}/classes_{itr:02d}_even.hdf --ref_volume {path}/threed_{itrm1:02d}_even.hdf --threads {threads} {verbose}".format(
				path=options.path, itr=it, itrm1=it-1,threads=options.threads,verbose=verbose)
			run(cmd)
			
			cmd="e2euler_refine.py --input {path}/classes_{itr:02d}_odd.hdf --ref_volume {path}/threed_{itrm1:02d}_odd.hdf --threads {threads} {verbose}".format(
				path=options.path, itr=it, itrm1=it-1,threads=options.threads,verbose=verbose)
			run(cmd)

		### 3-D Reconstruction
		# FIXME - --lowmem removed due to some tricky bug in e2make3d
		if options.breaksym : m3dsym="c1"
		else : m3dsym=options.sym
		append_html("<p>* Using the known orientations, reconstruct the even/odd 3-D maps from the even/odd 2-D class-averages.</p>",True)

		if not options.m3dold :
			cmd="e2make3dpar.py --input {path}/classes_{itr:02d}_even.hdf --sym {sym} --output {path}/threed_{itr:02d}_even.hdf {preprocess} \
 --keep {m3dkeep} {keepsig} --apix {apix} --pad {m3dpad} --mode gauss_var --threads {threads} {verbose}".format(
			path=options.path, itr=it, sym=m3dsym, recon=options.recon, preprocess=m3dpreprocess,  m3dkeep=options.m3dkeep, keepsig=m3dkeepsig,
			m3dpad=options.pad,fillangle=astep ,threads=options.threads, apix=apix, verbose=verbose)
		else:
			cmd="e2make3d.py --input {path}/classes_{itr:02d}_even.hdf --iter 2 -f --sym {sym} --output {path}/threed_{itr:02d}_even.hdf --recon {recon} {preprocess} \
 --keep={m3dkeep} {keepsig} --apix={apix} --pad={m3dpad} {verbose}".format(
			path=options.path, itr=it, sym=m3dsym, recon=options.recon, preprocess=m3dpreprocess,  m3dkeep=options.m3dkeep, keepsig=m3dkeepsig,
			m3dpad=options.pad, apix=apix, verbose=verbose)

		#if options.classweight == "count":
		#	pass # this is the default.
		#elif options.classweight == "sqrt":
		#	cmd += " --sqrtnorm"
		#elif options.classwright == "no_wt":
		#	cmd += " --no_wt"

		run(cmd)

		if not options.m3dold :
			cmd="e2make3dpar.py --input {path}/classes_{itr:02d}_odd.hdf --sym {sym} --output {path}/threed_{itr:02d}_odd.hdf {preprocess} \
 --keep {m3dkeep} {keepsig} --apix {apix} --pad {m3dpad} --mode gauss_var --threads {threads} {verbose}".format(
			path=options.path, itr=it, sym=m3dsym, recon=options.recon, preprocess=m3dpreprocess, m3dkeep=options.m3dkeep, keepsig=m3dkeepsig,
			m3dpad=options.pad, apix=apix, fillangle=astep ,threads=options.threads, verbose=verbose)
		else:
			cmd="e2make3d.py --input {path}/classes_{itr:02d}_odd.hdf --iter 2 -f --sym {sym} --output {path}/threed_{itr:02d}_odd.hdf --recon {recon} {preprocess} \
 --keep={m3dkeep} {keepsig} --apix={apix} --pad={m3dpad} {verbose}".format(
			path=options.path, itr=it, sym=m3dsym, recon=options.recon, preprocess=m3dpreprocess, m3dkeep=options.m3dkeep, keepsig=m3dkeepsig,
			m3dpad=options.pad, apix=apix, verbose=verbose)

		#if options.classweight == "count":
		#	pass # this is the default.
		#elif options.classweight == "sqrt":
		#	cmd += " --sqrtnorm"
		#elif options.classwright == "no_wt":
		#	cmd += " --no_wt"

		run(cmd)
		progress += 1.0

		### postprocessing
		append_html("""<p>* Finally, determine the resolution, filter and mask the even/odd maps, and then produce the final 3-D map for this iteration.
Note that the next iteration is seeded with the individual even/odd maps, not the final average.</p>""",True)
		evenfile="{path}/threed_{itr:02d}_even.hdf".format(path=options.path,itr=it)
		oddfile="{path}/threed_{itr:02d}_odd.hdf".format(path=options.path,itr=it)
		combfile="{path}/threed_{itr:02d}.hdf".format(path=options.path,itr=it)
		run("e2refine_postprocess.py --even {path}/threed_{it:02d}_even.hdf --odd {path}/threed_{it:02d}_odd.hdf --output {path}/threed_{it:02d}.hdf --automaskexpand {amaskxp} \
--align --mass {mass} --iter {it} {amask3d} {amask3d2} {m3dpostproc} {setsf} {tophat} --sym={sym} --restarget={restarget} --underfilter --ampcorrect={ampcorrect}".format(\
path=options.path, it=it, mass=options.mass, amask3d=amask3d, sym=m3dsym, amask3d2=amask3d2, m3dpostproc=m3dpostproc, setsf=m3dsetsf, restarget=options.targetres, amaskxp=options.automaskexpand,\
ampcorrect=ampcorrect,tophat=tophat))

		db.update({"last_map":combfile,"last_even":evenfile,"last_odd":oddfile})

		##################################################################
		### Generating FSC plots and other output for the report files ###

		### Convergenece plot
		if it>0:
			cmd="e2proc3d.py {path}/threed_{itr:02d}_even.hdf {path}/converge_even_{itrm1:02d}_{itr:02d}.txt --calcfsc {path}/threed_{itrm1:02d}_even.hdf".format(path=options.path,itr=it,itrm1=it-1)
			run(cmd)
			cmd="e2proc3d.py {path}/threed_{itr:02d}_odd.hdf {path}/converge_odd_{itrm1:02d}_{itr:02d}.txt --calcfsc {path}/threed_{itrm1:02d}_odd.hdf".format(path=options.path,itr=it,itrm1=it-1)
			run(cmd)

		try:
			plt.title("Convergence plot (not resolution)")
			plt.xlabel(r"Spatial Frequency (1/$\AA$)")
			plt.ylabel("FSC")
			cnvrg=[i for i in os.listdir(options.path) if "converge_" in i and i[-4:]==".txt"]
			cnvrg.sort(reverse=True)
			nummx=int(cnvrg[0].split("_")[2][:2])
			maxx=0.01
			for c in cnvrg:
				num=int(c.split("_")[2][:2])
				d=np.loadtxt("{}/{}".format(options.path,c)).transpose()
				if c[9:13]=="even" : plt.plot(d[0],d[1],label=c[14:-4],color=pltcolors[(nummx-num)%12])
				else : plt.plot(d[0],d[1],color=pltcolors[(nummx-num)%12])
				maxx=max(maxx,max(d[0]))
			plt.axhline(0.0,color="k")
			plt.axis((0,maxx,-.06,1.02))
			plt.legend(loc="upper right",fontsize="x-small")
			#plt.minorticks_on()
			plt.xticks(xticklocs,xticklbl)
			plt.yticks(yticklocs2,yticklbl2)
			plt.savefig("{}/report/converge.png".format(options.path))
			try: plt.savefig("{}/report/converge.pdf".format(options.path))
			except: pass
			plt.clf()

		except:
			traceback.print_exc()
			append_html("<p>Error generating convergence plot in report. Please look at fsc* files in the refine_xx folder</p>")

		######################
		### Resolution plot
		### broken up into multiple try/except blocks because we need some of the info, even if plotting fails
		try:
			plt.title("Gold Standard Resolution (tight mask)")
			plt.xlabel(r"Spatial Frequency (1/$\AA$)")
			plt.ylabel("FSC")
		except:
			pass

		fscs=[i for i in os.listdir(options.path) if "fsc_maskedtight" in i and i[-4:]==".txt"]
		fscs.sort(reverse=True)
		nummx=int(fscs[0].split("_")[2][:2])
		maxx=0.01

		# iterate over fsc curves
		for fi,f in enumerate(fscs):
			num=int(f.split("_")[2][:2])

			# read the fsc curve
			d=np.loadtxt("{}/{}".format(options.path,f)).transpose()

			# plot the curve
			try: plt.plot(d[0],d[1],label=f[4:],color=pltcolors[(nummx-num)%12])
			except: pass
			maxx=max(maxx,max(d[0]))

			# find the resolution from the first curve (the highest numbered one)
			if f==fscs[0]:
				lastnum=num
				# find the 0.143 crossing
				for si in xrange(2,len(d[0])-2):
					if d[1][si-1]>0.143 and d[1][si]<=0.143 :
						frac=(0.143-d[1][si])/(d[1][si-1]-d[1][si])		# 1.0 if 0.143 at si-1, 0.0 if .143 at si
						lastres=d[0][si]*(1.0-frac)+d[0][si-1]*frac
						try:
							plt.annotate(r"{:1.1f} $\AA$".format(1.0/lastres),xy=(lastres,0.143),
								xytext=((lastres*4+d[0][-1])/5.0,0.2+0.05*fi),arrowprops={"width":1,"frac":.1,"headwidth":7,"shrink":.05})
						except: pass
						break
				else : lastres=0

		try:
			plt.axhline(0.0,color="k")
			plt.axhline(0.143,color="#306030",linestyle=":")
			plt.axis((0,maxx,-.02,1.02))
#			plt.legend(loc="lower left")
			plt.legend(loc="upper right",fontsize="x-small")
#			plt.minorticks_on()
			plt.xticks(xticklocs,xticklbl)
			plt.yticks(yticklocs,yticklbl)
			plt.savefig("{}/report/resolution.png".format(options.path))
			try: plt.savefig("{}/report/resolution.pdf".format(options.path))
			except: pass
			plt.clf()
		except:
			traceback.print_exc()
			append_html("<p>Error generating resolution plot in report. Please look at fsc* files in the refine_xx folder</p>")

		if lastres==0 :
			append_html("<p>No valid resolution found for iteration {}.".format(it))
		else:
			append_html("<p>Iteration {}: Resolution = {:1.1f} &Aring; (gold standard refinement with tight mask, FSC @0.143)</p>".format(it,1.0/lastres))

		######################
		### Resolution plot 2
		### broken up into multiple try/except blocks because we need some of the info, even if plotting fails
		try:
			plt.title("Gold Standard Resolution (mask comparison)")
			plt.xlabel(r"Spatial Frequency (1/$\AA$)")
			plt.ylabel("FSC")
		except:
			pass

		fscs=["fsc_maskedtight_{:02d}.txt".format(lastnum),"fsc_masked_{:02d}.txt".format(lastnum),"fsc_unmasked_{:02d}.txt".format(lastnum)]
		nummx=int(fscs[0].split("_")[2][:2])
		maxx=0.01
		lastres=[]

		# iterate over fsc curves
		for i,f in enumerate(fscs):
			num=int(f.split("_")[2][:2])

			# read the fsc curve
			d=np.loadtxt("{}/{}".format(options.path,f)).transpose()

			# plot the curve
			try: plt.plot(d[0],d[1],label=f[4:],color=pltcolors[i])
			except: pass
			maxx=max(maxx,max(d[0]))

			# find the resolutions
			# find the 0.143 crossing
			for si in xrange(2,len(d[0])-2):
				if d[1][si-1]>0.143 and d[1][si]<=0.143 :
					frac=(0.143-d[1][si])/(d[1][si-1]-d[1][si])		# 1.0 if 0.143 at si-1, 0.0 if .143 at si
					lastres.append(d[0][si]*(1.0-frac)+d[0][si-1]*frac)
					try:
						plt.annotate(r"{:1.1f} $\AA$".format(1.0/lastres[-1]),xy=(lastres[-1],0.143),
							xytext=((lastres[-1]*4+d[0][-1])/5.0,0.2+0.05*i),arrowprops={"width":1,"frac":.1,"headwidth":7,"shrink":.05})
					except: pass
					break

		try:
			plt.axhline(0.0,color="k")
			plt.axhline(0.143,color="#306030",linestyle=":")
			plt.axis((0,maxx,-.02,1.02))
#			plt.legend(loc="lower left")
			plt.legend(loc="upper right",fontsize="x-small")
#			plt.minorticks_on()
			plt.xticks(xticklocs,xticklbl)
			plt.yticks(yticklocs,yticklbl)
			plt.savefig("{}/report/resolution_masks.png".format(options.path))
			try: plt.savefig("{}/report/resolution_masks.pdf".format(options.path))
			except: pass
			plt.clf()
		except:
			traceback.print_exc()
			append_html("<p>Error generating resolution plot in report. Please look at fsc* files in the refine_xx folder</p>")

		if len(lastres)==0 :
			append_html("<p>No valid resolution found for iteration {}".format(it))
		else:
			try :append_html("<p>Iteration {}: Resolution with different masks = {:1.1f}, {:1.1f}, {:1.1f} &Aring;</p>".format(it,1.0/lastres[0],1.0/lastres[1],1.0/lastres[2]))
			except: append_html("<p>Iteration {}: Didn't find resolutions for all 3 curves".format(it))

		E2progress(logid,progress/total_procs)

	if len(lastres)==0 :
		append_html("""<p>I was unable to determine a resolution for your final iteration. This should not normally happen, and generally indicates a problem. 
One exception is when working with heavily downsampled data, the data may be of sufficient quality to achieve resolutions beyond Nyquist. If the resolution curve
falls smoothly to the edge of the plot, this may be what is happening, and you should try continuing the processing with finer A/pix sampling.
Consider consulting the EMAN2 mailing list if you cannot figure out what's going wrong</p>""")
	else:
		try:
			if 1.0/lastres[1]>randomres*.9 :
				append_html("""<p>Unfortunately your final determined resolution of {:1.1f} &Aring; is not sufficiently better than the
phase randomization resolution of {:1.1f} &Aring; for the 'gold standard' resolution method to be valid. Please do not trust this result.
You must rerun the refinement with a more conservative initial target resolution (suggest ~{:1.1f}).</p>""".format(1.0/lastres[1],randomres,1.0/lastres[1]*0.9))
			elif 1.0/lastres[1]<options.targetres*.9:
				append_html("""<p>Your final determined resolution of {:1.1f} &Aring; is higher resolution than your specified target resolution by more than 10%.
This may cause the FSC curves to have an unusual shape, giving invalid numbers. It also means your reconstruction may be rotationally undersampled. Suggest running
another refinement with a target resolution slightly higher than {:1.1f} &Aring; to ensure your results are valid</p>""".format(1.0/lastres[1],1.0/lastres[1]))
			else:
				append_html("""<p>Congratulations, your refinement is complete, and you have a gold standard resolution of {:1.1f} &Aring;.
Note that there is always some variability in these determined values based on masking of the map. If the map is masked too tightly,
the FSC curve will not remain near zero at high resolution and will rise again. If you see an FSC curve that falls to zero then rises again,
it is an indication that something may have gone wrong with the masking. In general, EMAN2.1 is very conservative in its masking to try and
avoid these problems. This means that the resolution may be as much as 10% better with somewhat tighter masking, without getting into
artifact territory. </p>

<p>If you wish to continue this refinement to further improve resolution, the most efficient approach is to use the --startfrom refine_XX option
rather than specifying --input and --model. When you use --startfrom, it will not re-randomize the phases. Since you have already achieved
sufficient resolution to validate the gold-standard approach, continuing to extend this resolution is valid, and more efficient.""".format(1.0/lastres[1]))
		except:
			try:
				append_html("""<p>Congratulations, your refinement is complete, and you have a gold standard resolution of {:1.1f} &Aring (or {:1.1f} &Aring with a more conservative mask);.
Since this refinement continued from an existing refinement (or something funny happened), it is impossible to tell if the gold-standard criteria have been met, but
if they were met in the refinement this run continued, then your resolution should still be valid.""".format(1.0/lastres[0],1.0/lastres[1],options.path))
				traceback.print_exc()
				print "Note: the above traceback is just a warning for debugging purposes, and can be ignored"
			except:
				append_html("""<p> Congratulations, your refinement is complete, but there was some difficulty in assessing the resolution. This could just mean that your data was 
sampling-limited (meaning a smaller A/pix value would have been required to achieve data-limited resolution. If you cannot figure out what is going on, suggest asking for help
on the <a href=https://groups.google.com/forum/#!forum/eman2>Google Group</a>.""")

	append_html("""<h2>Explore your results</h2><p>Here are some useful output files to look at:</p><ul>
<li>Your final 3-D map from this run is {path}/threed_{iter:02d}.hdf</li>
<li>It may be useful to compare {path}/classes_{iter:02d}.hdf to {path}/projections_{iter:02d}.hdf. The images in these
two files should match extremely well, other than more noise being present in the class-averages.</li>
<li>Run <i>e2eulerxplor.py</i> to look at the distribution of particle orientations and interactively compare projections to
class-averages. Click on a specific peak to see the projection and corresponding filtered class-average for any specific orientation.</li>
<li>If you wish to make a single stack where you can look at all projections/averages side-by-side:
<br><i>e2proc2d.py {path}/classes_{iter:02d}_even.hdf clsvsproj.hdf --interlv {path}/projections_{iter:02d}_odd.hdf</i></li>
<li>It may also be worthwhile to compare {path}/threed_{iter:02d}_even.hdf to {path}/threed_{iter:02d}_odd.hdf to observe
the differences responsible for the assessed resolution.</li>
<li>The individual FSC curves are in 2-column text files called fsc_*.txt. Both masked and unmasked FSCs are computed</li>
<li>The reconstruction produced after each iteration (threed_??.hdf) is masked and filtered as part of the reconstruction process.
For the final completed iteration, the unmasked even and odd volumes are also retained: threed_even|odd_unmasked.hdf</li>
</ul>""".format(path=options.path,iter=it))


	E2end(logid)

	print """
***********************************************************{stars}
* REFINEMENT COMPLETE - Please look at {path}/report/index.html *
***********************************************************{stars}

""".format(path=options.path,stars="*"*len(options.path))


def run(command):
	"Mostly here for debugging, allows you to control how commands are executed (os.system is normal)"

	print "{}: {}".format(time.ctime(time.time()),command)
	append_html("<p>{}: {}</p>".format(time.ctime(time.time()),command),True)

	ret=launch_childprocess(command)

	# We put the exit here since this is what we'd do in every case anyway. Saves replication of error detection code above.
	if ret !=0 :
		print "Error running: ",command
		sys.exit(1)

	return

if __name__ == "__main__":
    main()
