#! /usr/bin/env python

# Author: Jesus Galaz, 03/20112
# Copyright (c) 2011 Baylor College of Medicine
#
# This software is issued under a joint BSD/GNU license. You may use the
# source code in this file under either license. However, note that the
# complete EMAN2 and SPARX software packages have some GPL dependencies,
# so you are responsible for compliance with the licenses of these packages
# if you opt to use BSD licensing. The warranty disclaimer below holds
# in either instance.
#
# This complete copyright notice must be included in any revised version of the
# source code. Additional authorship citations may be added, but existing
# author citations must be preserved.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  2111-1307 USA

from EMAN2 import *
import math
import os
import sys
import random
from random import choice

def main():
	progname = os.path.basename(sys.argv[0])
	usage = """prog <output> [options]
	Compute the variance of a stack of .hdf files. If the average is not provided, it will be generated by the program.
	"""
			
	parser = EMArgumentParser(usage=usage,version=EMANVERSION)
	
	parser.add_argument("--stack",type=str,default=None,help="The name of the stack of particles, or a string contained in all the particles you want to put into a stack")
	parser.add_argument("--average",type=str,default=None,help="Path for the refinement, default=auto")
	parser.add_argument("--preprocess",type=str,help="A processor (as in e2proc3d.py; could be masking, filtering, etc.) to be applied to each volume prior to alignment. Not applied to aligned particles before averaging.",default=None)
	parser.add_argument("--lowpass",type=str,help="A processor (as in e2proc3d.py; could be masking, filtering, etc.) to be applied to each volume prior to alignment. Not applied to aligned particles before averaging.",default=None)
	parser.add_argument("--highpass",type=str,help="A processor (as in e2proc3d.py; could be masking, filtering, etc.) to be applied to each volume prior to alignment. Not applied to aligned particles before averaging.",default=None)
	parser.add_argument("--normproc",type=str,help="Normalization processor applied to particles before alignment. Default is to use normalize.mask. If normalize.mask is used, results of the mask option will be passed in automatically. If you want to turn this option off specify \'None\'", default="normalize")
	parser.add_argument("--postprocess",type=str,help="A processor to be applied to the volume after averaging the raw volumes, before subsequent iterations begin.",default=None)
	parser.add_argument("--mask",type=str,help="Mask processor applied to particles before alignment. Default is mask.sharp:outer_radius=-2", default="mask.sharp:outer_radius=-2")
	parser.add_argument("--output",type=str,default=None,help="The name of the output file for the variance.")

	(options, args) = parser.parse_args()
	
	if options.normproc: 
		options.normproc=parsemodopt(options.normproc)
	
	if options.mask: 
		options.mask=parsemodopt(options.mask)
	
	if options.preprocess: 
		options.preprocess=parsemodopt(options.preprocess)
	
	if options.lowpass: 
		options.lowpass=parsemodopt(options.lowpass)

	if options.highpass: 
		options.highpass=parsemodopt(options.highpass)

	if options.postprocess: 
		options.postprocess=parsemodopt(options.postprocess)
	
	hdr=EMData(options.stack,1,True)
	x=hdr['nx']
	y=hdr['ny']
	z=hdr['nz']
	
	average = EMData(x,y,z)
	average.to_zero()
	
	pfs = []
	if '.hdf' not in options.stack:
		f=os.listdir(os.getcwd())
		for i in f:
			if options.stack in i:
				pfs.append(i)
				print "I am appending this file to the stack", i
		for i in range(len(pfs)):
			a=EMData(pfs[i],0)
			a.write_image('stack.hdf',i)
			options.stack='stack.hdf'
	
	n = EMUtil.get_image_count(options.stack)
	ptcls = []
	for i in range(n):
		a=EMData(options.stack,i)
		
		a.process_inplace(options.normproc[0],options.normproc[1])
		
		if options.lowpass:
			a.process_inplace(options.lowpass[0],options.lowpass[1])	
		if options.highpass:
			a.process_inplace(options.highpass[0],options.highpass[1])
		if options.preprocess:
			a.process_inplace(options.preprocess[0],options.preprocess[1])
			
		ptcls.append(a) 
	
	if options.average:
		average=EMData(options.average,0)

		if options.lowpass:
			average.process_inplace(options.lowpass[0],options.lowpass[1])	
		if options.highpass:
			average.process_inplace(options.highpass[0],options.highpass[1])
		if options.preprocess:
			average.process_inplace(options.preprocess[0],options.preprocess[1])
	else:
		average = sum(ptcls)/len(ptcls)
	
	average.process_inplace(options.normproc[0],options.normproc[1])

	variance = EMData(x,y,z)
	variance.to_zero()
	
	k=0
	for p in ptcls:
		variance = variance + (p-average)*(p-average)
		print "I have added this particle to the variance",k
		k+=1	
	variance = variance/len(ptcls)
	
	print "I have calculated the variance"
	
	variance.process_inplace(options.normproc[0],options.normproc[1])
	variance.process_inplace(options.mask[0],options.mask[1])
	
	variance.write_image(options.output,0)

	if options.postprocess:
		variance.process_inplace(options.postprocess[0],options.postprocess[1])
		variance.write_image(options.output.replace('.hdf','_postp.hdf'),0)
	
	return()
	
	
if __name__ == '__main__':
	main()
