/**
 * Allows access to local Smart Sync data. Wraps all Smart Sync APIs 
 * and should make data access easier
 */
package mdkInventory1.dataAccess;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Vector;

import mdkInventory1.Constants;
import mdkInventory1.GetProperties;

import com.sap.ip.me.api.persist.core.PersistenceException;
import com.sap.ip.me.api.persist.query.Condition;
import com.sap.ip.me.api.persist.query.Query;
import com.sap.ip.me.api.persist.query.RelationalOperatorType;
import com.sap.ip.me.api.persist.query.SortOrder;
import com.sap.ip.me.api.services.MeIterator;
import com.sap.ip.me.api.smartsync.BasisFieldType;
import com.sap.ip.me.api.smartsync.CharacterField;
import com.sap.ip.me.api.smartsync.DecimalField;
import com.sap.ip.me.api.smartsync.Field;
import com.sap.ip.me.api.smartsync.FieldDescriptor;
import com.sap.ip.me.api.smartsync.FieldDescriptorIterator;
import com.sap.ip.me.api.smartsync.NumericField;
import com.sap.ip.me.api.smartsync.Row;
import com.sap.ip.me.api.smartsync.RowDescriptor;
import com.sap.ip.me.api.smartsync.SmartSyncException;
import com.sap.ip.me.api.smartsync.SmartSyncQueryFactory;
import com.sap.ip.me.api.smartsync.SmartSyncRuntime;
import com.sap.ip.me.api.smartsync.SmartSyncTransactionManager;
import com.sap.ip.me.api.smartsync.SyncBo;
import com.sap.ip.me.api.smartsync.SyncBoDataFacade;
import com.sap.ip.me.api.smartsync.SyncBoDescriptor;
import com.sap.ip.me.api.smartsync.SyncBoDescriptorFacade;

public class SmartSyncDBAccess implements Constants {
	private String[] tableHeaderNamesToDisplay;
	private String[] tableHeaderNames;
	private int headerFieldCount;
	private static SyncBoDataFacade dataFacade;
	private static SyncBoDescriptorFacade descriptorFacade;

	public SmartSyncDBAccess() {
		dataFacade = SmartSyncRuntime.getInstance().getSyncBoDataFacade();
		descriptorFacade = SmartSyncRuntime.getInstance().getSyncBoDescriptorFacade();
	}

	// Methods 
	 	public MeIterator getAllSyncBoInstances(String syncBoName) {
		// This method is used to get all syncbos or a syncbos that match a certain pattern
			try {
				SyncBoDescriptor sbd = descriptorFacade.getSyncBoDescriptor(syncBoName);
	
					return dataFacade.getSyncBos(sbd).iterator();
			} catch (PersistenceException pex) {
				System.out.println(pex.getMessage());
				return null;
			}
		}

	public MeIterator getRowInstances(String syncBoName, int start, int count, int sortIndex, boolean sortOrder, String filter) {
		try {
			SyncBoDescriptor sbd = descriptorFacade.getSyncBoDescriptor(syncBoName);
			MeIterator iteratorRows = null;

			// regular path. This path sets the sortorder and filter (if set)
			SmartSyncQueryFactory queryFactory = SmartSyncRuntime.getInstance().getQueryFactory();
			RowDescriptor rd = sbd.getTopRowDescriptor();
			FieldDescriptor fd = rd.getFieldDescriptor(tableHeaderNames[sortIndex]);

			Condition cond = queryFactory.createCondition(fd, RelationalOperatorType.STARTS_WITH, filter);
			SortOrder singleSortOrder = queryFactory.createSortOrder(fd, sortOrder);
			Query rowQuery = queryFactory.createQuery(rd, cond, singleSortOrder, start, count);

			iteratorRows = dataFacade.getRows(rowQuery).iterator();
			return iteratorRows;
		} catch (PersistenceException pex) {
			System.out.println(pex.getMessage());
			return null;
		}
	}

	public SyncBo getSyncBoInstance(String syncBoName, String syncKey) {
		SyncBoDescriptor sbd = descriptorFacade.getSyncBoDescriptor(syncBoName);
		try {
			SyncBo sb = dataFacade.getSyncBo(sbd, syncKey);
			return sb;
		} catch (PersistenceException pex) {
			System.out.println(pex.getMessage());
			return null;
		}
	}
	public String[] getHeaderFieldNames(String syncBoName) {
		SyncBoDescriptor sbd = descriptorFacade.getSyncBoDescriptor(syncBoName);
		RowDescriptor trd = sbd.getTopRowDescriptor();
		String[] arrayFieldNames = new String[trd.getFieldCount()];
		tableHeaderNamesToDisplay = new String[trd.getFieldCount()];
		int i = 0;
		if (trd.getFieldCount() > 0) {
			FieldDescriptorIterator fdi = trd.getAllFieldDescriptors();
			while (fdi.hasNext()) {
				// load array with name foer headers defined in the properties file
				arrayFieldNames[i] = fdi.next().getFieldName();
				tableHeaderNamesToDisplay[i] = GetProperties.getString(arrayFieldNames[i]);
				i++;
			}
			return arrayFieldNames;
		} else {
			System.out.println("SmartSyncDBAccess.getHeaderFieldNames - Array of Header Field Names is empty");
			return null;
		}
	}

	public String getHeaderFieldValue(Row header, String headerFieldName) {
		RowDescriptor trd = header.getRowDescriptor();
		FieldDescriptor fd = trd.getFieldDescriptor(headerFieldName);
		if (fd != null) {
			BasisFieldType bft = fd.getFieldType();
			//			Row header = sb.getTopRow();
			try {
				if (bft == BasisFieldType.N) {
					NumericField nf = header.getNumericField(fd);
					return nf.getValueWithLeadingZeros();
				} else {
					Field f = header.getField(fd);
					if (f == null || (f.getValue() == null)) {
						return "";
					} else {
						return f.getValue().toString();
					}
				}
			} catch (SmartSyncException ssex) {
				System.out.println(ssex.getMessage());
				return null;
			}
		} else {
			return null;
		}
	}

	public String getHeaderFieldValue(SyncBo sb, String headerFieldName) {
		SyncBoDescriptor sbd = sb.getSyncBoDescriptor();
		RowDescriptor trd = sbd.getTopRowDescriptor();
		FieldDescriptor fd = trd.getFieldDescriptor(headerFieldName);
		if (fd != null) {
			BasisFieldType bft = fd.getFieldType();
			Row header = sb.getTopRow();
			try {
				if (bft == BasisFieldType.N) {
					NumericField nf = header.getNumericField(fd);
					return nf.getValueWithLeadingZeros();
				} else {
					Field f = header.getField(fd);
					if (f == null || (f.getValue() == null)) {
						return "";
					} else {
						return f.getValue().toString();
					}
				}
			} catch (SmartSyncException ssex) {
				System.out.println(ssex.getMessage());
				return null;
			}
		} else {
			return null;
		}
	}

	// Method to set a value in the row
	public boolean setHeaderFieldValue(SyncBo sb, String headerFieldName, Object value) {
		SyncBoDescriptor sbd = sb.getSyncBoDescriptor();
		RowDescriptor trd = sbd.getTopRowDescriptor();
		FieldDescriptor fd = trd.getFieldDescriptor(headerFieldName);
		if (fd != null) {
			BasisFieldType bft = fd.getFieldType();
			Row header = sb.getTopRow();
			try {
				// Integer operator
				if (bft == BasisFieldType.N) {
					NumericField nf = header.getNumericField(fd);
					if (nf != null) {
						BigInteger ii = new BigInteger(value.toString());
						nf.setValue(ii);
						return true;
					} else {
						return false;
					}
				}
				//	Character operator
				if (bft == BasisFieldType.C) {
					CharacterField cf = header.getCharacterField(fd);
					if (cf != null) {
						cf.setValue(value.toString());
						return true;
					} else {
						return false;
					}

				}
				// Decimal operator
				if (bft == BasisFieldType.P) {
					DecimalField df = header.getDecimalField(fd);
					if (df != null) {
						BigDecimal bd = new BigDecimal(value.toString());
						df.setValue(bd);
						return true;
					} else {
						return false;
					}

				}
				// Similar operation for time and date operator fields
			} catch (SmartSyncException ex) {
				System.out.println(ex.getMessage());
			} catch (PersistenceException e) {
				System.out.println(e.getMessage());
			}
		}
		return false;
	}
	// return the value of a field
	public String getItemFieldValue(Row item, String itemFieldName) {
		RowDescriptor rd = item.getRowDescriptor();
		FieldDescriptor fd = rd.getFieldDescriptor(itemFieldName);
		BasisFieldType bft = fd.getFieldType();

		try {
			if (bft == BasisFieldType.N) {
				NumericField nf = item.getNumericField(fd);
				return nf.getValueWithLeadingZeros();
			} else {
				Field f = item.getField(fd);
				if (f == null) {
					return "";
				} else {
					return f.getValue().toString();
				}
			}
		} catch (SmartSyncException ex) {
			System.out.println(ex.getMessage());
			return null;
		}
	}

	public MeIterator readEntitiesFromDB(int start, int count, int sortIndex, boolean sort_ascending, String filter_string) {
		Vector lineVector = new Vector();
		tableHeaderNames = getHeaderFieldNames(SYNCBO_NAME_INVENTORY);
		setColumns(tableHeaderNames.length);
		return getRowInstances(SYNCBO_NAME_INVENTORY, start, count, sortIndex, sort_ascending, filter_string);
	}

	public int getNumberOfSyncBos() {
		int rows_in_dataset = 0;
		MeIterator syncBos = getAllSyncBoInstances(SYNCBO_NAME_INVENTORY);
		
		while (syncBos.hasNext()) {
			syncBos.next();
			rows_in_dataset++;
		}
		return rows_in_dataset;
	}

	public Vector getEntities(MeIterator syncBos, int fromIndex, int count) {
		// We want usually all items in the MEIterator because the SmartSync API already delivered
		// the correct part of items we want to display.
		// the fromIndex and count variable are only there in case we need a single entry.		
		Vector retVec = new Vector();
		int i = 0;
		if (syncBos != null) {
			syncBos.reset();
			// Fill table header with Field Names of Top Row
			while (syncBos.hasNext()) {
				/*				SyncBo sb = (SyncBo) syncBos.next();
								if (i >= fromIndex) {
									Vector rowData = new Vector();
									for (int col = 0; col < getColumns(); col++) {
										rowData.addElement(getHeaderFieldValue(sb, tableHeaderNames[col]));
									}
									retVec.addElement(rowData);
								} */
				Row row = (Row) syncBos.next();
				if (i >= fromIndex) {
					Vector rowData = new Vector();
					for (int col = 0; col < getColumns(); col++) {
						rowData.addElement(getHeaderFieldValue(row, tableHeaderNames[col]));
					}
					retVec.addElement(rowData);
				}
				i++;
				// return when count has been reached, when count is greater 0.
				// if count less 0 we get all records.
				if (count > 0) {
					if (i >= (fromIndex + count))
						return retVec;
				}
			}
		}
		return retVec;
	}

	public void modifyRowInDB(String syBName, String syncKey, String newValue, int col) throws SmartSyncException, PersistenceException {
		String syncBoName = syBName;
		// Set default Syncboname for this example in case value is null.		
		if (syncBoName == null) {
			syncBoName = SYNCBO_NAME_INVENTORY;
		}
		SyncBo syncBo = getSyncBoInstance(syncBoName, syncKey);
		SmartSyncTransactionManager transactionManager;

		// A transaction manager is valid for one action starting with beginTransaction and ending with commit/rollback
		// In this example we commit (save) every modification - no rollback.			
		transactionManager = dataFacade.getSmartSyncTransactionManager();
		transactionManager.beginTransaction();
		setHeaderFieldValue(syncBo, tableHeaderNames[col], newValue);
		//Commit the transaction		
		transactionManager.commit();
	}

	// add a new item
	// first an empty synbo is created and than the values are set
	public void addRowInDB(String syBName, String[] newValues) throws SmartSyncException, PersistenceException {
		String syncBoName = syBName;
		// Set default Syncboname for this example in case value is null.		
		if (syncBoName == null) {
			syncBoName = SYNCBO_NAME_INVENTORY;
		}
		SyncBoDescriptor sbd = descriptorFacade.getSyncBoDescriptor(syncBoName);
		SmartSyncTransactionManager transactionManager;

		// Create new syncbo
		SyncBo newsyncBo = dataFacade.createEmptySyncBo(sbd);
		// A transaction manager is valid for one action starting with beginTransaction and ending with commit/rollback
		// In this example we commit (save) every row we add - no rollback.			
		transactionManager = dataFacade.getSmartSyncTransactionManager();
		transactionManager.beginTransaction();
		for (int i = 0; i < getColumns(); i++) {
			if (newValues[i] != null) {
				setHeaderFieldValue(newsyncBo, tableHeaderNames[i], newValues[i]);
			}
		}
		dataFacade.insertSyncBo(newsyncBo);
		//Commit the transaction		
		transactionManager.commit();
	}

	// Get and set methods to set columns and headernames.
	public int getColumns() {
		return headerFieldCount;
	}

	public void setColumns(int col) {
		headerFieldCount = col;
	}
	public String[] getTableHeaderNames() {
		return tableHeaderNamesToDisplay;
	}

	public void setTableHeaderNames(String[] strings) {
		tableHeaderNamesToDisplay = strings;
	}

	public String getTableHeaderName(int col) {
		return tableHeaderNamesToDisplay[col];
	}
}
/*	
 * The following methods are not used in this example but demonstrates the access of items and itemnames 
 * 
	 public int getItemTypeCount(String syncBoName) {
		SyncBoDescriptor sbd = descriptorFacade.getSyncBoDescriptor(syncBoName);
		RowDescriptorIterator rdi = sbd.getAllRowDescriptors();
		return rdi.size();
	}
	
	public Row[] getItemInstances(SyncBo sb, String itemName) {
			SyncBoDescriptor sbd = sb.getSyncBoDescriptor();
			RowDescriptor rd = sbd.getRowDescriptor(itemName);
			Row[] arrayRows;
			try {
				RowCollection rc = sb.getRows(rd);
				MeIterator ri = rc.iterator();
				// count the Rows to size the array accordingly
				int i = 0;
				while (ri.hasNext()) {
					ri.next();
					i++;
				}
				ri = rc.iterator();
				arrayRows = new Row[i];
				//start filling the array now
				i = 0;
				while (ri.hasNext()) {
					arrayRows[i] = (Row) ri.next();
					i++;
				}
				return arrayRows;
			} catch (PersistenceException pex) {
				System.out.println(pex.getMessage());
				return null;
			}
	} 

	public String[] getItemNames(String syncBoName) {
		SyncBoDescriptor sbd = descriptorFacade.getSyncBoDescriptor(syncBoName);
		RowDescriptorIterator rdi = sbd.getAllRowDescriptors();
		String[] arrayItemNames = null;
		if (rdi != null) {
			// count the SyncBoDescriptors to size the array accordingly
			int i = 0;
			while (rdi.hasNext()) {
				rdi.next();
				i++;
			}
			rdi = sbd.getAllRowDescriptors();
			arrayItemNames = new String[i];
			// Fill array with item names
			i = 0;
			while (rdi.hasNext()) {
				arrayItemNames[i] = rdi.next().getRowName();
				i++;
			}
			return arrayItemNames;
		} else {
			System.out.println("SmartSyncDBAccess.getItemNames - Array of Item Names is empty");
			return null;
		}
	}
	
	public String[] getItemFieldNames(String syncBoName, String itemName) {
		SyncBoDescriptor sbd = descriptorFacade.getSyncBoDescriptor(syncBoName);
		RowDescriptor rd = sbd.getRowDescriptor(itemName);
		FieldDescriptorIterator fdi = rd.getAllFieldDescriptors();
		String[] arrayItemFieldNames = null;
		if (fdi != null) {
			// count the SyncBoDescriptors to size the array accordingly
			int i = 0;
			while (fdi.hasNext()) {
				fdi.next();
				i++;
			}
			fdi = rd.getAllFieldDescriptors();
			arrayItemFieldNames = new String[i];
			i = 0;
			while (fdi.hasNext()) {
				arrayItemFieldNames[i] = fdi.next().getFieldName();
				i++;
			}
			return arrayItemFieldNames;
		} else {
			System.out.println("SmartSyncDBAccess.getItemFieldNames - Array of Item Field Names is empty");
			return null;
		}
	} */
