/*
 * Olap_7.java
 *
 * Copyright 2004-2006 by SAP AG. All Rights Reserved.
 * SAP, R/3, mySAP, mySAP.com, xApps, xApp, SAP NetWeaver, and other SAP 
 * products and services mentioned herein as well as their respective logos 
 * are trademarks or registered trademarks of SAP AG in Germany and in several 
 * other countries all over the world. All other product and service names 
 * mentioned are the trademarks of their respective companies. Data contained 
 * in this document serves informational purposes only. National product 
 * specifications may vary.
 *
 * These materials are subject to change without notice. These materials are 
 * provided by SAP AG and its affiliated companies ("SAP Group") for 
 * informational purposes only, without representation or warranty of any kind, 
 * and SAP Group shall not be liable for errors or omissions with respect to 
 * the materials. The only warranties for SAP Group products and services are 
 * those that are set forth in the express warranty statements accompanying 
 * such products and services, if any. Nothing herein should be construed as 
 * constituting an additional warranty.
 */
package com.sap.ip.bi.sdk.samples;

import java.io.IOException;
import java.io.PrintWriter;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.sap.exception.IBaseException;
import com.sap.ip.bi.sdk.dac.connector.IBIConnection;
import com.sap.ip.bi.sdk.dac.connector.IBIOlap;
import com.sap.ip.bi.sdk.dac.olap.query.IBICommandProcessor;
import com.sap.ip.bi.sdk.dac.olap.query.main.IBIQuery;
import com.sap.ip.bi.sdk.dac.olap.query.msx.IBIRankingFilter;
import com.sap.ip.bi.sdk.samples.servlet.MinimalServletContainer;

/**
 * Filtering -
 *
 * Illustrates both a ranking filter and a condition-based filter.
 * Renders the result of a query without any filtering, then
 * filters the set of Sold-To parties using a ranking filter and
 * re-renders the result.  Changes the filter to a condition-based
 * filter to restrict by quantity, and re-renders the result for
 * comparison.
 *
 * View the HTML rendered by this servlet in the following file:
 * [SDK archive]/docs/examples/olap_7.result.html
 *
 * @author  SAP
 * @version 3.50
 * @since 3.50
 */
public class Olap_7 extends HttpServlet {
  private static final String CONTENT_TYPE = "text/html";

  public void init(ServletConfig config) throws ServletException {
    super.init(config);
  }

  public void doGet(
    HttpServletRequest request,
    HttpServletResponse response)
    throws ServletException, IOException {

    response.setContentType(CONTENT_TYPE);
    PrintWriter out = response.getWriter();

    out.println(Helpers.getDocTypeDefinition());
    out.println("<html>");
    out.println("<head><title>Olap_7</title>");
    out.println(Helpers.getStyleSheetDefinition());
    out.println("</head><body>");

    try {
      // ********************************************************
      // Connect to data source.
      // ********************************************************
      IBIConnection connection = Helpers.connectToXMLADatasource(out);

      // ********************************************************
      // Retrieve metadata.
      // ********************************************************
      // Get Olap interface
      IBIOlap olap = connection.getOlap();
      Helpers.SampleMetaDataBW sampleMetaData =
        Helpers.getSampleMetaDataBW(connection);

      // ********************************************************
      // Create query and command processor.
      // ********************************************************
      IBIQuery query = olap.createQuery(sampleMetaData.cube);
      IBICommandProcessor commandProcessor = query.getCommandProcessor();

      // ********************************************************
      // Change the layout.
      // ********************************************************
      // Move the "0D_SOLD_TO" Dimension to rows
      commandProcessor.moveDimensionToRows(
        sampleMetaData.soldToPartyDimension);

      // ********************************************************
      // Specify selected members.
      // ********************************************************
      commandProcessor.addMember(sampleMetaData.valueTypeActualMember);
      commandProcessor.addMember(sampleMetaData.versionCurrentMember);
      commandProcessor.addMember(sampleMetaData.measuresQuantityMember);
      commandProcessor.addMember(sampleMetaData.measuresCostMember);

      // ********************************************************
      // Renders the default result.
      // ********************************************************
      Helpers.renderQueryAndDataset(out, query);

      // Create a ranking filter on the customer dimension
      // to display only the top 5 members
      IBIRankingFilter rankingFilter =
        commandProcessor.createTopCountFilter(
          sampleMetaData.soldToPartyDimension,
          5,
          sampleMetaData.measuresQuantityMember);

      // ********************************************************
      // Renders the result of the ranking filter.
      // ********************************************************
      Helpers.renderQueryAndDataset(out, query);

      // Remove the RankingFilter
      rankingFilter.getMemberSet().getMemberSetExpression().
        remove(rankingFilter);

      // Create a ConditionBasedFilter to restrict to Sold-To
      // parties where the quantity is between 500 and 3000
      commandProcessor.createSearchFilterGreaterThan(
        sampleMetaData.soldToPartyDimension,
        500,
        sampleMetaData.measuresQuantityMember);
      commandProcessor.createSearchFilterLessThan(
        sampleMetaData.soldToPartyDimension,
        3000,
        sampleMetaData.measuresQuantityMember);

      // ********************************************************
      // Renders the result of the condition-based filter.
      // ********************************************************
      Helpers.renderQueryAndDataset(out, query);

    } 
    // Catch errors.
    catch (Exception e) {
      // $JL-EXC$
      e.printStackTrace();
      if (e instanceof IBaseException)
        out.println("Error: " +
                ((IBaseException)e).getNestedLocalizedMessage());
      else  
        out.println("Error: " + e.getMessage());        
    }
    out.println("</body>");
    out.println("</html>");
  }

  public void destroy() {
  }

  public static void main(String[] args) {

    if (args.length == 1) {
      MinimalServletContainer.executeServlet(new Olap_7(), args[0]);
    } else {
      MinimalServletContainer.executeServlet(new Olap_7(), System.out);
    }
  }

}
