/*!
    \file    DBMSrvMsg_Error.hpp
    \author  MarcW, TiloH
    \ingroup error, warning and information message handling for DBM Server
    \brief   declaration of a class for error messages
    
    \if EMIT_LICENCE

    ========== licence begin  GPL
    Copyright (c) 2004-2006 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end

    \endif
*/


#if !defined(DBMSrvMsg_Error_hpp)
#define DBMSrvMsg_Error_hpp

#include "DBM/Srv/Message/DBMSrvMsg_Base.hpp"
#include "Messages/Msg_List.hpp"
#include "SAPDBCommon/Messages/SDBMsg_DBMSrv.h"

#include "gcn00.h"

class DBMSrvInfrReply_Reply2MsgList;

/*!
    @brief error message

    A new error message is defined as follows:
    * create a new entry at the end of the enum DBMSrvMsg_Error::ID
    * create a new entry in the static array m_ErrorTextArray,
      that has the new ID as key
*/
class DBMSrvMsg_Error : public Msg_List, public DBMSrvMsg_Base {
public:
    /*! @brief error IDs */
    enum ErrorID {
        SDBMSG_DBMSRV_ENUMERATION,

        /* insert new error message ID's via the Message Monitor */
        LAST_ID         /*!< !!!! this must remain the last enum defined for DBMSrvMsg_Error::ID and must not be used as error ID !!!! */
    };

    /*! @brief default constructor
    
        use this constructor to create a empty error list */
    DBMSrvMsg_Error();

    /*! \brief constructor

        Takes the arguments of the SDB-Define and up to 8 optional tag-
        arguments. For messages of component DBMSrv the message ID is
        changed to a classical DBM-Server error number.
        
        \param aID       [IN] ID of the error message (generated by buildmsg)
        \param component [IN] component reporting the error (generated by buildmsg)
        \param file      [IN] file reporting the error (generated by buildmsg + preprocessor)
        \param line      [IN] line reporting the error (generated by buildmsg + preprocessor)
        \param aText     [IN] message text (generated by buildmsg)
        \param argCheck  [IN] parameter used for aargument check (generated by buildmsg)
        \param messageVersion [IN] version of the message (generated by buildmsg)
        \param arg0      [IN] optional tag 0
        \param arg1      [IN] optional tag 1
        \param arg2      [IN] optional tag 2
        \param arg3      [IN] optional tag 3
        \param arg4      [IN] optional tag 4
        \param arg5      [IN] optional tag 5
        \param arg6      [IN] optional tag 6
        \param arg7      [IN] optional tag 7
        \param arg8      [IN] optional tag 8 */
    DBMSrvMsg_Error(
        int                 aID,
        const char        * component,
        const char        * file,
        int                 line,
        const char        * aText,
        Msg_Has1Args      * argCheck,
        const Msg_IArg    * messageVersion,
        Msg_IOptArg const * arg0=0,
        Msg_IOptArg const * arg1=0,
        Msg_IOptArg const * arg2=0,
        Msg_IOptArg const * arg3=0,
        Msg_IOptArg const * arg4=0,
        Msg_IOptArg const * arg5=0,
        Msg_IOptArg const * arg6=0,
        Msg_IOptArg const * arg7=0,
        Msg_IOptArg const * arg8=0);

    /*! \brief constructor

        Takes the arguments of the SDB-Define and 1 mandatory and up to 7 optional tag-
        arguments. For messages of component DBMSrv the message ID is
        changed to a classical DBM-Server error number.
        
        \param aID       [IN] ID of the error message (generated by buildmsg)
        \param component [IN] component reporting the error (generated by buildmsg)
        \param file      [IN] file reporting the error (generated by buildmsg + preprocessor)
        \param line      [IN] line reporting the error (generated by buildmsg + preprocessor)
        \param aText     [IN] message text (generated by buildmsg)
        \param argCheck  [IN] parameter used for aargument check (generated by buildmsg)
        \param messageVersion [IN] version of the message (generated by buildmsg)
        \param arg0      [IN] mandatory tag 0
        \param arg1      [IN] optional tag 1
        \param arg2      [IN] optional tag 2
        \param arg3      [IN] optional tag 3
        \param arg4      [IN] optional tag 4
        \param arg5      [IN] optional tag 5
        \param arg6      [IN] optional tag 6
        \param arg7      [IN] optional tag 7
        \param arg8      [IN] optional tag 8 */
    DBMSrvMsg_Error(
        int                 aID,
        const char        * component,
        const char        * file,
        int                 line,
        const char        * aText,
        Msg_Has2Args      * argCheck,
        const Msg_IArg    * messageVersion,
        Msg_IArg const    * arg0,
        Msg_IOptArg const * arg1=0,
        Msg_IOptArg const * arg2=0,
        Msg_IOptArg const * arg3=0,
        Msg_IOptArg const * arg4=0,
        Msg_IOptArg const * arg5=0,
        Msg_IOptArg const * arg6=0,
        Msg_IOptArg const * arg7=0,
        Msg_IOptArg const * arg8=0);

    /*! \brief constructor

        Takes the arguments of the SDB-Define and 2 mandatory and up to 6 optional tag-
        arguments. For messages of component DBMSrv the message ID is
        changed to a classical DBM-Server error number.
        
        \param aID       [IN] ID of the error message (generated by buildmsg)
        \param component [IN] component reporting the error (generated by buildmsg)
        \param file      [IN] file reporting the error (generated by buildmsg + preprocessor)
        \param line      [IN] line reporting the error (generated by buildmsg + preprocessor)
        \param aText     [IN] message text (generated by buildmsg)
        \param argCheck  [IN] parameter used for aargument check (generated by buildmsg)
        \param messageVersion [IN] version of the message (generated by buildmsg)
        \param arg0      [IN] mandatory tag 0
        \param arg1      [IN] mandatory tag 1
        \param arg2      [IN] optional tag 2
        \param arg3      [IN] optional tag 3
        \param arg4      [IN] optional tag 4
        \param arg5      [IN] optional tag 5
        \param arg6      [IN] optional tag 6
        \param arg7      [IN] optional tag 7
        \param arg8      [IN] optional tag 8 */
    DBMSrvMsg_Error(
        int                 aID,
        const char        * component,
        const char        * file,
        int                 line,
        const char        * aText,
        Msg_Has3Args      * argCheck,
        const Msg_IArg    * messageVersion,
        Msg_IArg const    * arg0,
        Msg_IArg const    * arg1,
        Msg_IOptArg const * arg2 = 0,
        Msg_IOptArg const * arg3 = 0,
        Msg_IOptArg const * arg4 = 0,
        Msg_IOptArg const * arg5 = 0,
        Msg_IOptArg const * arg6 = 0,
        Msg_IOptArg const * arg7 = 0,
        Msg_IOptArg const * arg8 = 0);

    /*! \brief constructor

        Takes the arguments of the SDB-Define and 3 mandatory and up to 5 optional tag-
        arguments. For messages of component DBMSrv the message ID is
        changed to a classical DBM-Server error number.
        
        \param aID       [IN] ID of the error message (generated by buildmsg)
        \param component [IN] component reporting the error (generated by buildmsg)
        \param file      [IN] file reporting the error (generated by buildmsg + preprocessor)
        \param line      [IN] line reporting the error (generated by buildmsg + preprocessor)
        \param aText     [IN] message text (generated by buildmsg)
        \param argCheck  [IN] parameter used for aargument check (generated by buildmsg)
        \param messageVersion [IN] version of the message (generated by buildmsg)
        \param arg0      [IN] mandatory tag 0
        \param arg1      [IN] mandatory tag 1
        \param arg2      [IN] mandatory tag 2
        \param arg3      [IN] optional tag 3
        \param arg4      [IN] optional tag 4
        \param arg5      [IN] optional tag 5
        \param arg6      [IN] optional tag 6
        \param arg7      [IN] optional tag 7
        \param arg8      [IN] optional tag 8 */
    DBMSrvMsg_Error(
        int                 aID,
        const char        * component,
        const char        * file,
        int                 line,
        const char        * aText,
        Msg_Has4Args      * argCheck,
        const Msg_IArg    * messageVersion,
        Msg_IArg const    * arg0,
        Msg_IArg const    * arg1,
        Msg_IArg const    * arg2,
        Msg_IOptArg const * arg3 = 0,
        Msg_IOptArg const * arg4 = 0,
        Msg_IOptArg const * arg5 = 0,
        Msg_IOptArg const * arg6 = 0,
        Msg_IOptArg const * arg7 = 0,
        Msg_IOptArg const * arg8 = 0);

    /*! \brief constructor

        Takes the arguments of the SDB-Define and 4 mandatory and up to 4 optional tag-
        arguments. For messages of component DBMSrv the message ID is
        changed to a classical DBM-Server error number.
        
        \param aID       [IN] ID of the error message (generated by buildmsg)
        \param component [IN] component reporting the error (generated by buildmsg)
        \param file      [IN] file reporting the error (generated by buildmsg + preprocessor)
        \param line      [IN] line reporting the error (generated by buildmsg + preprocessor)
        \param aText     [IN] message text (generated by buildmsg)
        \param argCheck  [IN] parameter used for aargument check (generated by buildmsg)
        \param messageVersion [IN] version of the message (generated by buildmsg)
        \param arg0      [IN] mandatory tag 0
        \param arg1      [IN] mandatory tag 1
        \param arg2      [IN] mandatory tag 2
        \param arg3      [IN] mandatory tag 3
        \param arg4      [IN] optional tag 4
        \param arg5      [IN] optional tag 5
        \param arg6      [IN] optional tag 6
        \param arg7      [IN] optional tag 7
        \param arg8      [IN] optional tag 8 */
    DBMSrvMsg_Error(
        int                 aID,
        const char        * component,
        const char        * file,
        int                 line,
        const char        * aText,
        Msg_Has5Args      * argCheck,
        const Msg_IArg    * messageVersion,
        Msg_IArg const    * arg0,
        Msg_IArg const    * arg1,
        Msg_IArg const    * arg2,
        Msg_IArg const    * arg3,
        Msg_IOptArg const * arg4 = 0,
        Msg_IOptArg const * arg5 = 0,
        Msg_IOptArg const * arg6 = 0,
        Msg_IOptArg const * arg7 = 0,
        Msg_IOptArg const * arg8 = 0);

    /*! \brief constructor

        Takes the arguments of the SDB-Define and 4 mandatory and up to 4 optional tag-
        arguments. For messages of component DBMSrv the message ID is
        changed to a classical DBM-Server error number.
        
        \param aID       [IN] ID of the error message (generated by buildmsg)
        \param component [IN] component reporting the error (generated by buildmsg)
        \param file      [IN] file reporting the error (generated by buildmsg + preprocessor)
        \param line      [IN] line reporting the error (generated by buildmsg + preprocessor)
        \param aText     [IN] message text (generated by buildmsg)
        \param argCheck  [IN] parameter used for aargument check (generated by buildmsg)
        \param messageVersion [IN] version of the message (generated by buildmsg)
        \param arg0      [IN] mandatory tag 0
        \param arg1      [IN] mandatory tag 1
        \param arg2      [IN] mandatory tag 2
        \param arg3      [IN] mandatory tag 3
        \param arg4      [IN] mandatory tag 4
        \param arg5      [IN] optional tag 5
        \param arg6      [IN] optional tag 6
        \param arg7      [IN] optional tag 7
        \param arg8      [IN] optional tag 8 */
    DBMSrvMsg_Error(
        int                 aID,
        const char        * component,
        const char        * file,
        int                 line,
        const char        * aText,
        Msg_Has6Args      * argCheck,
        const Msg_IArg    * messageVersion,
        Msg_IArg const    * arg0,
        Msg_IArg const    * arg1,
        Msg_IArg const    * arg2,
        Msg_IArg const    * arg3,
        Msg_IArg const    * arg4,
        Msg_IOptArg const * arg5 = 0,
        Msg_IOptArg const * arg6 = 0,
        Msg_IOptArg const * arg7 = 0,
        Msg_IOptArg const * arg8 = 0);

    /*! \brief constructor

        Takes the arguments of the SDB-Define and 5 mandatory and up to 3 optional tag-
        arguments. For messages of component DBMSrv the message ID is
        changed to a classical DBM-Server error number.
        
        \param aID       [IN] ID of the error message (generated by buildmsg)
        \param component [IN] component reporting the error (generated by buildmsg)
        \param file      [IN] file reporting the error (generated by buildmsg + preprocessor)
        \param line      [IN] line reporting the error (generated by buildmsg + preprocessor)
        \param aText     [IN] message text (generated by buildmsg)
        \param argCheck  [IN] parameter used for aargument check (generated by buildmsg)
        \param messageVersion [IN] version of the message (generated by buildmsg)
        \param arg0      [IN] mandatory tag 0
        \param arg1      [IN] mandatory tag 1
        \param arg2      [IN] mandatory tag 2
        \param arg3      [IN] mandatory tag 3
        \param arg4      [IN] mandatory tag 4
        \param arg5      [IN] mandatory tag 5
        \param arg6      [IN] optional tag 6
        \param arg7      [IN] optional tag 7
        \param arg8      [IN] optional tag 8 */
    DBMSrvMsg_Error(
        int                 aID,
        const char        * component,
        const char        * file,
        int                 line,
        const char        * aText,
        Msg_Has7Args      * argCheck,
        const Msg_IArg    * messageVersion,
        Msg_IArg const    * arg0,
        Msg_IArg const    * arg1,
        Msg_IArg const    * arg2,
        Msg_IArg const    * arg3,
        Msg_IArg const    * arg4,
        Msg_IArg const    * arg5,
        Msg_IOptArg const * arg6 = 0,
        Msg_IOptArg const * arg7 = 0,
        Msg_IOptArg const * arg8 = 0);

    /*! \brief constructor

        Takes the arguments of the SDB-Define and 6 mandatory and up to 2 optional tag-
        arguments. For messages of component DBMSrv the message ID is
        changed to a classical DBM-Server error number.
        
        \param aID       [IN] ID of the error message (generated by buildmsg)
        \param component [IN] component reporting the error (generated by buildmsg)
        \param file      [IN] file reporting the error (generated by buildmsg + preprocessor)
        \param line      [IN] line reporting the error (generated by buildmsg + preprocessor)
        \param aText     [IN] message text (generated by buildmsg)
        \param argCheck  [IN] parameter used for aargument check (generated by buildmsg)
        \param messageVersion [IN] version of the message (generated by buildmsg)
        \param arg0      [IN] mandatory tag 0
        \param arg1      [IN] mandatory tag 1
        \param arg2      [IN] mandatory tag 2
        \param arg3      [IN] mandatory tag 3
        \param arg4      [IN] mandatory tag 4
        \param arg5      [IN] mandatory tag 5
        \param arg6      [IN] mandatory tag 6
        \param arg7      [IN] optional tag 7
        \param arg8      [IN] optional tag 8 */
    DBMSrvMsg_Error(
        int                 aID,
        const char        * component,
        const char        * file,
        int                 line,
        const char        * aText,
        Msg_Has8Args      * argCheck,
        const Msg_IArg    * messageVersion,
        Msg_IArg const    * arg0,
        Msg_IArg const    * arg1,
        Msg_IArg const    * arg2,
        Msg_IArg const    * arg3,
        Msg_IArg const    * arg4,
        Msg_IArg const    * arg5,
        Msg_IArg const    * arg6,
        Msg_IOptArg const * arg7 = 0,
        Msg_IOptArg const * arg8 = 0);

    /*! \brief constructor

        Takes the arguments of the SDB-Define and 7 mandatory and up to 1 optional tag-
        arguments. For messages of component DBMSrv the message ID is
        changed to a classical DBM-Server error number.
        
        \param aID       [IN] ID of the error message (generated by buildmsg)
        \param component [IN] component reporting the error (generated by buildmsg)
        \param file      [IN] file reporting the error (generated by buildmsg + preprocessor)
        \param line      [IN] line reporting the error (generated by buildmsg + preprocessor)
        \param aText     [IN] message text (generated by buildmsg)
        \param argCheck  [IN] parameter used for aargument check (generated by buildmsg)
        \param messageVersion [IN] version of the message (generated by buildmsg)
        \param arg0      [IN] mandatory tag 0
        \param arg1      [IN] mandatory tag 1
        \param arg2      [IN] mandatory tag 2
        \param arg3      [IN] mandatory tag 3
        \param arg4      [IN] mandatory tag 4
        \param arg5      [IN] mandatory tag 5
        \param arg6      [IN] mandatory tag 6
        \param arg7      [IN] mandatory tag 7
        \param arg8      [IN] optional tag 8 */
    DBMSrvMsg_Error(
        int                 aID,
        const char        * component,
        const char        * file,
        int                 line,
        const char        * aText,
        Msg_Has9Args      * argCheck,
        const Msg_IArg    * messageVersion,
        Msg_IArg const    * arg0,
        Msg_IArg const    * arg1,
        Msg_IArg const    * arg2,
        Msg_IArg const    * arg3,
        Msg_IArg const    * arg4,
        Msg_IArg const    * arg5,
        Msg_IArg const    * arg6,
        Msg_IArg const    * arg7,
        Msg_IOptArg const * arg8 = 0);

    /*! \brief constructor

        Takes the arguments of the SDB-Define and 8 mandatory and no optional tag-
        arguments. For messages of component DBMSrv the message ID is
        changed to a classical DBM-Server error number.
        
        \param aID       [IN] ID of the error message (generated by buildmsg)
        \param component [IN] component reporting the error (generated by buildmsg)
        \param file      [IN] file reporting the error (generated by buildmsg + preprocessor)
        \param line      [IN] line reporting the error (generated by buildmsg + preprocessor)
        \param aText     [IN] message text (generated by buildmsg)
        \param argCheck  [IN] parameter used for aargument check (generated by buildmsg)
        \param messageVersion [IN] version of the message (generated by buildmsg)
        \param arg0      [IN] mandatory tag 0
        \param arg1      [IN] mandatory tag 1
        \param arg2      [IN] mandatory tag 2
        \param arg3      [IN] mandatory tag 3
        \param arg4      [IN] mandatory tag 4
        \param arg5      [IN] mandatory tag 5
        \param arg6      [IN] mandatory tag 6
        \param arg7      [IN] mandatory tag 7
        \param arg8      [IN] mandatory tag 8 */
    DBMSrvMsg_Error(
        int                 aID,
        const char        * component,
        const char        * file,
        int                 line,
        const char        * aText,
        Msg_Has10Args     * argCheck,
        const Msg_IArg    * messageVersion,
        Msg_IArg const    * arg0,
        Msg_IArg const    * arg1,
        Msg_IArg const    * arg2,
        Msg_IArg const    * arg3,
        Msg_IArg const    * arg4,
        Msg_IArg const    * arg5,
        Msg_IArg const    * arg6,
        Msg_IArg const    * arg7,
        Msg_IArg const    * arg8);

    /*! \brief constructor

        Takes the arguments of the SDB-Define and 1 mandatory and up to 7 optional tag-
        arguments. For messages of component DBMSrv the message ID is
        changed to a classical DBM-Server error number. A supplied Msg_List is
        added as sub message of the newly created info message.
        
        \param subMessage [IN] message to be used as sub message
        \param aID       [IN] ID of the error message (generated by buildmsg)
        \param component [IN] component reporting the error (generated by buildmsg)
        \param file      [IN] file reporting the error (generated by buildmsg + preprocessor)
        \param line      [IN] line reporting the error (generated by buildmsg + preprocessor)
        \param aText     [IN] message text (generated by buildmsg)
        \param argCheck  [IN] parameter used for aargument check (generated by buildmsg)
        \param messageVersion [IN] version of the message (generated by buildmsg)
        \param arg0      [IN] mandatory tag 0
        \param arg1      [IN] optional tag 1
        \param arg2      [IN] optional tag 2
        \param arg3      [IN] optional tag 3
        \param arg4      [IN] optional tag 4
        \param arg5      [IN] optional tag 5
        \param arg6      [IN] optional tag 6
        \param arg7      [IN] optional tag 7
        \param arg8      [IN] optional tag 8 */
    DBMSrvMsg_Error(
        const Msg_List    & subMessage,
        int                 aID,
        const char        * component,
        const char        * file,
        int                 line,
        const char        * aText,
        Msg_Has2Args      * argCheck,
        const Msg_IArg    * messageVersion,
        Msg_IArg const    * arg0,
        Msg_IOptArg const * arg1 = 0,
        Msg_IOptArg const * arg2 = 0,
        Msg_IOptArg const * arg3 = 0,
        Msg_IOptArg const * arg4 = 0,
        Msg_IOptArg const * arg5 = 0,
        Msg_IOptArg const * arg6 = 0,
        Msg_IOptArg const * arg7 = 0,
        Msg_IOptArg const * arg8 = 0);

  private:

    /*! \brief constructor
    
        this private constructor is used by class DBMSrvInfrReply_Reply2MsgList
        to create the DBM Server messages without honoring any tags in the long
        text or action of the message.
        \TODO remove with class DBMSrvInfrReply_Reply2MsgList
        \see DBMSrvInfrReply_Reply2MsgList
        
        \param classicErrorNumber [IN] DBM Server error number
        \param classicErrorText   [IN] DBM Server error text with inserted tag
            values */
    DBMSrvMsg_Error(
        int          classicErrorNumber,
        const char * classicErrorText);

  public:
    /*! @brief copy constructor */
    DBMSrvMsg_Error( DBMSrvMsg_Error const &aError )
         : Msg_List(aError) {}

    /*! @brief empty destructor */
    virtual ~DBMSrvMsg_Error() {}

protected:
    void appendTransportMessage(const char* errID, const char* errText);

  private:
    /*! \brief old operator for easy addition of errors
 
        a+b appended a to b! As this is as unclear as it could be, and
        because Msg_List changed the behaviour of its operator+ more than once
        (before Overrule() was invented) the operator+ is declared private and
        no longer implemented. */
    DBMSrvMsg_Error operator+ (const Msg_List& otherList) const;

  public:
    /*! \brief append a message to this list
        \param other [IN] other message list */
    void append(const DBMSrvMsg_Error & other);

    /*!
        @brief comparing to a DBMSrvMsg_Error::ErrorID

        The range of the ErrorID (0, 1, ...) is converted to the range of
        DBM error numbers (-24999, -24998, ...). In the calling code: in front of the
        == operator there is the DBMSrvMsg_Error, behind it is the ErrorID.

        @param aErrid [IN] the error ID to compare this DBMSrvMsg_Error to
        @return true if the ID in this object equals aErrid, false otherwise
    */
    bool operator==(const DBMSrvMsg_Error::ErrorID& aErrid) const {
       return (Msg_List::ID() + 25000) == int(aErrid);
    }

    /*! \brief comparing to a DBMSrvMsg_Error::ErrorID

        The range of the ErrorID (0, 1, ...) is converted to the range of
        DBM error numbers (-24999, -24998, ...). In the calling code: in front of the
        == operator there is the DBMSrvMsg_Error, behind it is the ErrorID.

        \param aErrid [IN] the error ID to compare this DBMSrvMsg_Error to
        \return false if the ID in this object equals aErrid, true otherwise
    */
    bool operator!=(const DBMSrvMsg_Error::ErrorID& aErrid) const {
       return (Msg_List::ID() + 25000) != int(aErrid);
    }

    /*! \brief static function to compare a calssic tcn00_Error with a new error enum
    
        \param classicError [IN] the classical error
        \param errorEnum    [IN] the new error enum
        \return true if both values represent the same error, false otherwise */
    static bool areEqual(const tcn00_Error classicError, const DBMSrvMsg_Error::ErrorID & errorEnum)
    {
        return ((classicError + 25000) == errorEnum);
    }

    /*! \brief determine, if a Msg_List is a DBM Server Message

        The function checks, wether message has component "DBMSrv", or not. Note,
        that if the message is a message list with more than one message, the
        component of the first message is compared to "DBMSrv".
        \param  message [IN] the message to check
        \return true, if message has component "DBMSrv", false otherwise */
    static bool isDBMSrvMessage(const Msg_List & message);

    /*! \brief Get the name of a Msg_List message
        
        This function determines the name of a Msg_List message, if the message was
        from component "DBMSrv". Otherwise, it will just return a 0 pointer.
        \param  message [IN] the message
        \return the name of the message, if it is a message of component "DBMSrv", 0 otherwise */
    static const char * getErrorName(const Msg_List & message);

    /*! \brief Get the tcn00_Error code for a DBMSRV message
        
        This function computes the tcn00_Error code (-24???) for a given DBMSRV message id.
        \param  ID [IN] the message
        \return the tcn00_Error code for the given message id  */
    static tcn00_Error get_tcn00_Error(ErrorID ID) {return (-25000 + ((int)ID));}

    /*! \brief compute the tcn00_Error code (-24???) of this message.
        
        At the moment, we can just return Msg_List::ID(), because we construct
        the base class with an (illegal) negative ID. After all callers to Msg_List::ID()
        are either removed or changed to call this method, construction can be legalized
        and subtraction of 25000 can be moved here.
        \return tcn00_Error code (-24???) of this message
    */
    tcn00_Error get_tcn00_Error();


    /*! \brief Get the value of a tag as string
        
        This function returns the value of a tag as string
        \param  szTag [IN] the name of the tag
        \return The value of the tag as zero terminated string. A empty string will be
                returned if the tag does not exists. */
    const char * getTagValueAsString(const char * szTag) const;

    /*! \brief Get the value of a tag as string
        
        This function returns the value of a tag integer
        \param  szTag [IN] the name of the tag
        \return The value of the tag as integer. A 0 will be
                returned if the tag does not exists. */
    int getTagValueAsInteger(const char * szTag) const;

    static const char * STRING_MESSAGE_PREFIX_ERROR;        //!< can be made private, as soon as vcn90.cpp is removed
    static const char * STRING_MESSAGE_PREFIX_WARNING;      //!< can be made private, as soon as vcn90.cpp is removed
    static const char * STRING_MESSAGE_PREFIX_INFORMATION;  //!< can be made private, as soon as vcn90.cpp is removed

    /*! @brief to debugging only!! */
    void dumpToStdOut();

  private:
    friend class DBMSrvInfrReply_Reply2MsgList;
};

inline tcn00_Error DBMSrvMsg_Error::get_tcn00_Error() {
    return this->Msg_List::ID(); // it's this long just to clarify what we're doing here...
}


/*!
    @brief comparing a DBMSrvMsg_Error::ErrorID to a Msg_List

    The range of the ErrorID (0, 1, ...) is converted to the range of
    DBM error numbers (-25000, -24999, ...). In the calling code: in front of
    the == operator there is the ErrorID, behind it is the Msg_List.

    @param aErrid [IN] the error ID to compare the message list to 
    @param aMsgList [IN] the message list
    @return true if the id of the message list equals aErrid, false otherwise

*/
bool operator==(const DBMSrvMsg_Error::ErrorID & aErrid, const Msg_List & aMsgList);

/*! \brief comparing a DBMSrvMsg_Error::ErrorID to a Msg_List

    The range of the ErrorID (0, 1, ...) is converted to the range of
    DBM error numbers (-25000, -24999, ...). In the calling code: in front of
    the == operator there is the ErrorID, behind it is the Msg_List.

    \param aErrid [IN] the error ID to compare the message list to 
    \param aMsgList [IN] pointer to the message list
    \return true if the ID of the message list equals aErrid, false otherwise */
bool operator==(const DBMSrvMsg_Error::ErrorID & aErrid, const Msg_List * aMsgList);

/*! \brief comparing a DBMSrvMsg_Error::ErrorID to a SAPDB_UInt4 or tcn00_Error

    This operator causes an ambiguity, if Microsofts's compiler is used, so
    in praxis you will not be able to use it :( (if this changes some day, the
    operator must be implemented too. Hopefully we have eliminated tcn00_Error
    until then)

    The range of the ErrorID (0, 1, ...) is converted to the range of
    DBM error numbers (-24999, -24998, ...). In the calling code: in front of
    the == operator there is the ErrorID, behind it is the SAPDB_UInt4.

    \param aErrid [IN] the error ID as ErrorID  
    \param aError [IN] the integer
    \return true if aError equals aErrid, false otherwise */
bool operator==(const DBMSrvMsg_Error::ErrorID aErrid, const SAPDB_UInt4 aError);

/*! \brief comparing a DBMSrvMsg_Error::ErrorID to another DBMSrvMsg_Error::ErrorID

    This operator causes an ambiguity, if Microsofts's compiler for IA64 is used, so
    in praxis you will not be able to use it :( (if this changes some day, the
    operator must be implemented too.)

    The range of the ErrorID (0, 1, ...) is converted to the range of
    DBM error numbers (-24999, -24998, ...). In the calling code: in front of
    the == operator there is the ErrorID, behind it is another ErrorID.

    \param aErrid      [IN] one error ID as ErrorID 
    \param secondErrid [IN] a second error ID as ErrorID
    \return true if the aErrid equals secondErrid, false otherwise */
bool operator==(const DBMSrvMsg_Error::ErrorID aErrid, const DBMSrvMsg_Error::ErrorID secondErrid);

/*! \brief comparing a DBMSrvMsg_Error::ErrorID to a Msg_List

    The range of the ErrorID (0, 1, ...) is converted to the range of
    DBM error numbers (-25000, -24999, ...). In the calling code: in front of
    the == operator there is the ErrorID, behind it is the Msg_List.

    \param aErrid [IN] the error ID to compare the message list to 
    \param aMsgList [IN] pointer to the message list
    \return false if the ID of the message list equals aErrid, true otherwise */
bool operator!=(const DBMSrvMsg_Error::ErrorID & aErrid, const Msg_List * aMsgList);

/*!
    @brief DBM error message definition

    an error message consists of an ID of class DBMSrvMsg_Error::ID and a text that
    can contain printf formatting instructions.
*/
class DBMSrvMsg_ErrorText {
public:
    /*! default constructor */
    DBMSrvMsg_ErrorText() {}

private:
    /*!
        @brief constructor

        @param aID [IN] the error's ID
        @param aText [IN] the message's text
    */
    DBMSrvMsg_ErrorText(const DBMSrvMsg_Error::ErrorID aID, const char* aText)
        : m_ID(aID),
          m_Text(aText) {}
public:
    /*! @brief constructor

        A constructor that is matching the new #define's created by the
        buildmsg tool. A number of arguments are ignored but are present for
        compatibility with buildmsg.

        @param aID       [IN] the error's ID
        @param component [IN] the component that reports the message (ignored)
        @param file      [IN] the name of the file, where the message is generated (ignored)
        @param line      [IN] the line, where the message is generated (ignored)
        @param aText     [IN] the message's text
        @param argCheck  [IN] pointer used to check number of arguments (ignored)
        @param messageVersion [IN] version number of the message (ignored)
        @param name      [IN] the name of the message */
    DBMSrvMsg_ErrorText(
        int              aID,
        const char     * component,
        const char     * file,
        int              line,
        const char     * aText,
        Msg_HasArgs    * argCheck,
        const Msg_IArg  & messageVersion,
        const char     * name)
      : m_ID(DBMSrvMsg_Error::ErrorID(aID)),
        m_Text(aText),
        m_Name(name)
    {
    }

    /*! @brief return the message's ID */
    const DBMSrvMsg_Error::ErrorID getErrorID() const {return m_ID;}

    /*! @brief return the message's text */
    const char* getText() const {return m_Text;}

    /*! @brief return the message's name */
    const char* getName() const {return m_Name;}

private:
    DBMSrvMsg_Error::ErrorID   m_ID;
    const char               * m_Text;
    const char               * m_Name;

    friend class DBMSrvMsg_ErrorTextArray;
};

/*!
    @brief smart array to avoid long startup time of DBM server
*/
class DBMSrvMsg_ErrorTextArray {
public:
    /*!
        @brief return error text object with passed ID

        Note that there is no check of the rqeuested ID is valid!
        @param aID [IN] ID of requested error text object
        @return requested error text object
    */
    static const DBMSrvMsg_ErrorText& getErrorText(const DBMSrvMsg_Error::ErrorID aID);

    /*! @brief you never know... */
    virtual ~DBMSrvMsg_ErrorTextArray() {}

private:
    DBMSrvMsg_ErrorTextArray() {}

    static void initArray();
    static bool m_IsInitialized;
    static DBMSrvMsg_ErrorText m_ErrorTextArray[DBMSrvMsg_Error::LAST_ID+1];
};

#endif //DBMSrvMsg_Error_hpp
