/*


    ========== licence begin GPL
    Copyright (C) 2002-2003 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


*/

package com.sap.dbtech.jdbcext;

import java.lang.ref.WeakReference;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.Hashtable;
import java.util.Properties;

import javax.sql.XAConnection;
import javax.transaction.xa.XAException;
import javax.transaction.xa.XAResource;
import javax.transaction.xa.Xid;

import com.sap.dbtech.jdbc.exceptions.ObjectIsClosedException;
import com.sap.dbtech.jdbc.exceptions.SQLExceptionSapDB;
import com.sap.dbtech.util.MessageKey;
import com.sap.dbtech.util.MessageTranslator;
// import com.sap.dbtech.util.trace.*;

public class XAConnectionSapDB 
    extends ConnectionListenerBag
    implements XAResource,  XAConnection 
{
    Properties          connectProperties;
    XADataSourceSapDB   ds;
    XATrace             trace;
    WeakReference       lastClientConnection;
    Connection          nonTxConnection;
    XATransaction       currentTransaction;
    int                 transactionTimeout;
    
    XAConnectionSapDB(Properties connectProperties,
                      XADataSourceSapDB ds,
                      XATrace trace)
    {
        this.connectProperties=connectProperties;
        this.ds=ds;
        this.trace=trace;
        this.nonTxConnection=null;
        this.currentTransaction=null;
        this.transactionTimeout=30;
    }

    boolean isTraceEnabled()
    {
        return trace!=null;
    }

    void traceXAError(Xid xid, int code)
    {
        if(trace!=null) trace.traceXAError(xid, code);
    }

    void trace(Xid xid, String message)
    {
        if(trace!=null) trace.trace(xid, message);
    }

    void trace(String message)
    {
        if(trace!=null) trace.trace(message);
    }
    

    public void close() throws SQLException
    {
        // if(isTraceEnabled()) trace("XAConnection.close called from " + TraceUtils.traceCaller(2));
        synchronized(this) {
            closeLastOpenClient();
            if(nonTxConnection!=null) {
                nonTxConnection.close();
            }
        }
    }

    public Connection getConnection() throws SQLException
    {
        // if(isTraceEnabled()) trace("XAConnection.getConnection called from " + TraceUtils.traceCaller(2));
        closeLastOpenClient();
        XAClientConnectionSapDB cc=new XAClientConnectionSapDB();
        lastClientConnection=new WeakReference(cc);
        return cc;
    }


    public XAResource getXAResource() throws SQLException
    {
        // if(isTraceEnabled()) trace("XAConnection.getXAResource called from " + TraceUtils.traceCaller(2));
        return this;
    }
    
    // implementation of javax.transaction.xa.XAResource interface

    public void start(Xid xid, int flags) throws XAException
    {
        synchronized(this) {
            if(isTraceEnabled()) trace(xid, "start");
            if(xid==null) {
                xaerror(xid, XAException.XAER_INVAL);
            }
            if(this.currentTransaction!=null) {
                xaerror(xid, XAException.XAER_OUTSIDE);
            }
            switch(flags) {
            case TMNOFLAGS:
                if(isTraceEnabled()) trace(xid, "start: TMNOFLAGS");
                synchronized(ds) {
                    XATransaction xatx=null;
                    if((xatx=ds.getXATransaction(xid))!=null) {
                        // if(isTraceEnabled()) trace(xatx.xid, "already found -> XAER_DUPID");
                        xaerror(xid, XAException.XAER_DUPID);
                    }
                    try {
                        Connection c;
                        if(nonTxConnection!=null) {
                            nonTxConnection.setAutoCommit(false);
                            c=nonTxConnection;
                            nonTxConnection=null;
                        } else {
                            c=ds.openPhysicalConnection(connectProperties);
                        }
                        long timeout=System.currentTimeMillis() + 1000 * transactionTimeout;
                        this.currentTransaction=new XATransaction(xid, c, timeout);
                        ds.addXATransaction(this.currentTransaction);
                    } catch(SQLException sqlEx) {
                        sqlEx.printStackTrace();
                        xaerror(xid, XAException.XAER_RMERR);
                    }
                }
                break;
            case TMJOIN:
                if(isTraceEnabled()) trace(xid, "start: TMJOIN");
                synchronized(ds) {
                    XATransaction tx=ds.getXATransaction(xid);
                    if(tx==null) {
                        xaerror(xid, XAException.XAER_INVAL);
                    }
                    this.currentTransaction=tx;
                }
                break;
            case TMRESUME:
                if(isTraceEnabled()) trace(xid, "start: TMRESUME");
                synchronized(ds) {
                    XATransaction tx=ds.getXATransaction(xid);
                    if(tx==null) {
                        xaerror(xid, XAException.XAER_INVAL);
                    }
                    this.currentTransaction=tx;
                }
                break;
            default:
                if(isTraceEnabled()) trace(xid, "start: unknown flags");
                xaerror(xid, XAException.XAER_INVAL);
            }
        }
    }
    
    public void end(Xid xid, int flags) throws XAException
    {
        if(isTraceEnabled()) trace(xid, "end");
        if(xid==null) {
            xaerror(xid, XAException.XAER_INVAL);
        }
        if(this.currentTransaction==null) {
            xaerror(xid, XAException.XAER_PROTO);
        }
        switch(flags) {
        case TMSUCCESS:
            if(isTraceEnabled()) trace(xid, "end: TMSUCCESS");
            if(!this.currentTransaction.xid.equals(xid)) {
                xaerror(xid, XAException.XAER_NOTA);
            }
            if(this.currentTransaction.getStatus()==XATransaction.STATUS_ROLLED_BACK) {
                xaerror(xid, XAException.XA_RBTIMEOUT);
            }
               
            this.currentTransaction=null;
            silentCloseLastOpenClient();
            break;
        case TMFAIL:            
            if(isTraceEnabled()) trace(xid, "end: TMFAIL");
            if(!this.currentTransaction.xid.equals(xid)) {
                xaerror(xid,XAException.XAER_NOTA);
            }
            if(this.currentTransaction.getStatus()==XATransaction.STATUS_ROLLED_BACK) {
                xaerror(xid,XAException.XA_RBTIMEOUT);
            }
           
            this.currentTransaction=null;
            silentCloseLastOpenClient();
            break;
        case TMSUSPEND:            
            if(isTraceEnabled()) trace(xid, "end: TMSUSPEND");
            if(!this.currentTransaction.xid.equals(xid)) {
                xaerror(xid,XAException.XAER_NOTA);
            }
            if(this.currentTransaction.getStatus()==XATransaction.STATUS_ROLLED_BACK) {
                xaerror(xid,XAException.XA_RBTIMEOUT);
            }
            this.currentTransaction=null;
            silentCloseLastOpenClient();
            break;
        default:
            xaerror(xid,XAException.XAER_INVAL);
        }

    }

    public void commit(Xid xid, boolean onePhase) throws XAException
    {
        if(isTraceEnabled()) trace(xid, "commit");
        if(xid==null) {
            xaerror(xid,XAException.XAER_INVAL);
        }
        synchronized(ds) {
            XATransaction tx=ds.getXATransaction(xid);
            if(tx==null) {
                xaerror(xid,XAException.XAER_NOTA);
            }
            if(tx.getStatus() == XATransaction.STATUS_ROLLED_BACK) {
                xaerror(xid,XAException.XA_HEURRB);
            }
            if(currentTransaction==tx) {
                this.currentTransaction=null;
                silentCloseLastOpenClient();
            }
            if(onePhase || tx.getStatus()==XATransaction.STATUS_PREPARED) {
                try { 
                    tx.commitRelease(); 
                } catch(SQLException sqlEx) {
                    try { tx.rollbackRelease(); } catch(SQLException ignore) {}
                    ds.removeXATransaction(xid);
                    xaerror(xid,XAException.XA_HEURRB);
                }
                ds.removeXATransaction(xid);
            } else {
                xaerror(xid,XAException.XAER_PROTO);
            }
        }
    }

    public void rollback(Xid xid) throws XAException
    {
        if(isTraceEnabled()) trace(xid, "rollback");
        if(xid==null) {
            xaerror(xid,XAException.XAER_INVAL);
        }
        synchronized(ds) {
            XATransaction tx=ds.getXATransaction(xid);
            if(tx==null) {
                xaerror(xid,XAException.XAER_NOTA);
            }
            if(tx.getStatus() == XATransaction.STATUS_ROLLED_BACK) {
                ds.removeXATransaction(xid);
            }
            if(tx.getStatus() == XATransaction.STATUS_COMMITTED) {
                ds.removeXATransaction(xid);
                xaerror(xid,XAException.XAER_RMERR);
            }
            try { tx.rollbackRelease(); } catch(SQLException ignore) {}
            ds.removeXATransaction(xid);
        }
    }

    public void forget(Xid xid) throws XAException
    {
        if(isTraceEnabled()) trace(xid, "forget");
        if(xid==null) {
            xaerror(xid,XAException.XAER_INVAL);
        }
        synchronized(ds) {
            XATransaction tx=ds.getXATransaction(xid);
            if(tx!=null) {
                if(currentTransaction==tx) {
                    this.currentTransaction=null;
                    silentCloseLastOpenClient();
                }
                ds.removeXATransaction(xid);
                try { tx.rollbackRelease(); } catch(SQLException ignore) {}
            }
        }
    }

    public int getTransactionTimeout() throws XAException
    {
        return this.transactionTimeout;
    }

    public boolean isSameRM(XAResource rm) throws XAException
    {
        boolean result=false;
        if(rm!=null && rm instanceof XAConnectionSapDB) {
            XAConnectionSapDB c=(XAConnectionSapDB)rm;
            result =  c.ds == ds;
        }
        return result;
    }

    public int prepare(Xid xid) throws XAException
    {
        if(isTraceEnabled()) trace(xid, "prepare");
        if(xid==null) {
            xaerror(xid,XAException.XAER_INVAL);
        } 
        // some of these may call this but did not detach correctly.
        synchronized(this) {
            if(currentTransaction!=null && currentTransaction.xid.equals(xid)) {
                silentCloseLastOpenClient();
                currentTransaction=null;
            }
        }
        synchronized(ds) {
            XATransaction tx=ds.getXATransaction(xid);

            if(tx==null) {
                xaerror(xid,XAException.XAER_NOTA);
            }           
//             if(tx.getStatus() != XATransaction.STATUS_DETACHED) {
//                 xaerror(xid,XAER_PROTO);
//             }
            // if it is rolled back, we want to kick the complete
            // transaction!
            if(tx.getStatus() == XATransaction.STATUS_ROLLED_BACK) {
                xaerror(xid,XAException.XA_RBROLLBACK);
            }
            tx.setStatus(XATransaction.STATUS_PREPARED);
        }
        return XA_OK;
    }

    public Xid[] recover(int flags) throws XAException
    {
        synchronized(ds) {
            return ds.getAllPrepared();
        }
    }

    public boolean setTransactionTimeout(int transactionTimeout) throws XAException
    {
        if(transactionTimeout<0) {
            xaerror(null,XAException.XAER_INVAL);
        }
        this.transactionTimeout=transactionTimeout;
        return false;
    }
    
    //----------------------------------------------------------------------
    void sendErrorEvent(SQLException sqlEx)
    {
        // trace("Sending error event to connection listener.");
        super.sendErrorEvent(this, sqlEx);
    }

    void sendCloseEvent()
    {
        // trace("Sending close event to connection listener.");
        super.sendCloseEvent(this);
    }

    Connection getRealConnection(Object client)
        throws SQLException
    {
        Object o=lastClientConnection==null ? null :lastClientConnection.get();
        if(o!=client) {
            return null;
        }
        synchronized(this) {
            // now look where we get the client connection ...
            if(currentTransaction!=null) {
                ((XAClientConnectionSapDB)client).setTransactional(true);
                if(currentTransaction.physicalConnection!=null) {
                    currentTransaction.physicalConnection.setAutoCommit(false); // just to be sure
                }
                return currentTransaction.physicalConnection;
            } else {
                if(nonTxConnection==null) {
                    nonTxConnection=ds.openPhysicalConnection(connectProperties);
                }
                ((XAClientConnectionSapDB)client).setTransactional(false);
                return nonTxConnection;
            }
        }
    }
    
    XAConnectionSapDB self()
    {
        return this;
    }

    void silentCloseLastOpenClient()
    {
        try { closeLastOpenClient(); } catch(SQLException ignore) {}
    }

    void closeLastOpenClient()
        throws SQLException
    {
        Object o = lastClientConnection==null ? null : lastClientConnection.get();
        if(o!=null) {
            synchronized(this) {
                XAClientConnectionSapDB client=(XAClientConnectionSapDB)o;
                client.closeWithoutEvent();
            }
        }
    }
    
    private static Hashtable xacodes;
    
    private void xaerror(Xid xid, int code) 
        throws XAException
    {
        if(isTraceEnabled()) traceXAError(xid, code);
        throw new XAException(code);
    }
    
    public void finalize()
    {
        try { this.close(); } catch(SQLException ignore) {}
    }
    
    //----------------------------------------------------------------------
    class XAClientConnectionSapDB
        extends ClientConnectionSapDB
    {
        boolean closed=false;            
        boolean transactional=false;
        
        public Connection getPhysicalConnection()
            throws SQLException
        {
//             if(isTraceEnabled()) trace(TraceUtils.traceCaller(2) + " acquires real connection. (Called from "
//                                        + TraceUtils.traceCaller(3) + ")");
            Connection result=getRealConnection(this);
            if(result==null) {
                throw new ObjectIsClosedException(this);
            }
            return result;
        }
        
        public void setTransactional(boolean b)
        {
            this.transactional=b;
        }

        public void exceptionOccurred(SQLException sqlEx)
        {
            if(sqlEx instanceof com.sap.dbtech.jdbc.exceptions.SQLExceptionSapDB) {
                com.sap.dbtech.jdbc.exceptions.SQLExceptionSapDB sapEx
                    =(com.sap.dbtech.jdbc.exceptions.SQLExceptionSapDB)sqlEx;
                if(sapEx.isConnectionReleasing()) {
                    sendErrorEvent(sqlEx);
                }
            } else {
                sendErrorEvent(sqlEx);
            }
        }

        public boolean isClosed()
            throws SQLException
        {
            return closed ||
                getRealConnection(this) == null ||
                super.isClosed();
        }

        
        public void setAutoCommit(boolean autocommit)
            throws SQLException
        {
            if(this.transactional) {
                throw new SQLExceptionSapDB(MessageTranslator.translate(MessageKey.ERROR_AUTOCOMMIT_XASESSION));
            } else {
                super.setAutoCommit(autocommit);
            }
        }

        public void rollback()
            throws SQLException
        {
            if(this.transactional) {
                throw new SQLExceptionSapDB(MessageTranslator.translate(MessageKey.ERROR_ROLLBACK_XASESSION));
            } else {
                super.rollback();
            }
        }

        public void commit()
            throws SQLException
        {
            if(this.transactional) {
                throw new SQLExceptionSapDB(MessageTranslator.translate(MessageKey.ERROR_COMMIT_XASESSION));
            } else {
                super.commit();
            }
        }
        
        public void closeWithoutEvent()
            throws SQLException
        {
            if(this.closed==false) {
                synchronized(self()) {
                    try {
                        if(this.transactional) {
                            // if inside a transaction we keep the
                            // hands off
                            this.closed=true;
                        } else {
                            if(!getAutoCommit()) {
                                this.rollback();
                            }
                            this.closed=true;
                        }
                    } catch(ObjectIsClosedException oic) {
                        // good, we can ignore this, but send the close 
                        // event
                    }
                }
            }
        }

        
        public void close()
            throws SQLException
        {
            // if(isTraceEnabled()) trace("ClientConnectionSapDB.close called from " + TraceUtils.traceCaller(2));
            if(this.closed==false) {
                synchronized(self()) {
                    try {
                        if(this.transactional) {
                            // if inside a transaction we keep the
                            // hands off
                            this.closed=true;
                            sendCloseEvent();
                        } else {
                            if(!getAutoCommit()) {
                                this.rollback();
                            }
                            this.closed=true;
                            sendCloseEvent();
                        }
                    } catch(ObjectIsClosedException oic) {
                        // good, we can ignore this, but send the close 
                        // event
                        sendCloseEvent();
                    }
                    
                }
            }
        }
        
        public void finalize()
        {
            try { this.close(); } catch(SQLException ignore) {}
        }
    }


}
