/*!
    \file    DBMTstBToolSimADINT_Main.cpp
    \author  TiloH
    \ingroup ADINT2 simulator for tests
    \brief   implementing the main function for the simulator program

\if EMIT_LICENCE
    ========== licence begin  GPL
    Copyright (c) 2004-2007 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end
\endif
*/

#include <stdio.h>

#include "hcn36.h"
#include "hni33.h"
#include "ToolsCommon/Parsers/ToolsParsers_ProgramParameterParser.hpp"

int main(int argc, char ** argv)
{
    bool         rc=true;
    const char * syntax="(-b|-r) -n <name> [-o <outfile>] [-c <blocksize>] {-p <pipename>} |"
                        "(-q|-d) -n <name> [-o <outfile>]";

    printf("\n");
    printf("+-----------------------------------------------------------------------------+\n");
    printf("|                                                                             |\n");
    printf("|           TiloH's ADINT2 Simulator, Copyright (c) 2004-2006 SAP AG          |\n");
    printf("|                                                                             |\n");
    printf("+-----------------------------------------------------------------------------+\n");
    printf("TDI0074X: The license is valid until >Thu Jun  1 01:00:00 2006<.\n");

    int argumentsStartOfPipes=-1;
    int numberOfPipes=0;

    printf("\nGot the following arguments:");
    for(int i=0; i<argc; i++)
    {
        printf(" \"%s\"", argv[i]);

        if(-1==argumentsStartOfPipes && 0==strcmp(argv[i], "-p"))
            argumentsStartOfPipes=i+1;

        if(0==strcmp(argv[i], "-p") && argc>i+1)
            numberOfPipes++;
    }
    printf("\n");

    ToolsParsers_ProgramParameterParser parser(argc, argv, syntax);

    const ToolsParsers_ParseResult & parseResult=parser.parse();

    if(ToolsParsers_ParseResult::NoError!=parseResult.getResultCode())
    {
        if(parseResult.foundSyntaxError())
            printf("adint2 %s\n", syntax);
        else
            printf("Could not parse parameters\n");

       rc=false;
    }
    else
    {
        char       * dataDir=0;
        const char * adaopt=cn36_GetEnv("ADA_OPT");

        cn36_StrAllocCpy(dataDir, 0==adaopt?"":adaopt);

        if('\0'!=dataDir[0])
            *(cn36_BaseNameFrom(dataDir))='\0'; // dataDir is now a path including a separator or empty

        if(parseResult.getKeyword("-b").isSet())
        {
            if(!parseResult.getVariable("<pipename>").isSet())
                rc=false;
            else
            {
                char                       cmdLinePipe2File[100][2048];
                tni33_ABackgroundProcess * pipe2FileProcs[100];
                int                        numberOfRunningPipe2Files=0;
                int                        i;
                

                for(i=0; i<numberOfPipes; i++)
                    pipe2FileProcs[i]=0;

			    for(i=0; i<numberOfPipes; i++)
			    {
                    char iAsString[21];
                    sprintf(iAsString, "%d", (int)i);

                    sprintf(
                        cmdLinePipe2File[i],
                        "pipe2file -d pipe2file -nowait -p %s -f %sBackupADSM%s_%s",
                        argv[argumentsStartOfPipes+i*2],
                        dataDir,
                        numberOfPipes>1?iAsString:"",
                        parseResult.getVariable("<name>").getValue());

                    pipe2FileProcs[i]=new tni33_ABackgroundProcess(cmdLinePipe2File[i]);

                    if(0==pipe2FileProcs[i] || !pipe2FileProcs[i]->WasStarted())
                        printf("Could not start '%s'.\n", cmdLinePipe2File[i]);
                    else
                    {
                        printf("Started '%s'.\n", cmdLinePipe2File[i]);
                        numberOfRunningPipe2Files++;
                    }
                }

                //wait for the pipe2file commands to end
                do
                {
                    //count number of  running pipe2file-calls
                    numberOfRunningPipe2Files=0;

                    for(i=0; i<numberOfPipes; i++)
                    {
                        if(0!=pipe2FileProcs[i] && pipe2FileProcs[i]->IsRunning())
                            numberOfRunningPipe2Files++;
                    }

                    if(0<numberOfRunningPipe2Files)
                        cn36_Sleep(1);
                }
                while(0<numberOfRunningPipe2Files);

                for(i=0; i<numberOfPipes; i++)
                {
                    if(0!=pipe2FileProcs[i])
                    {
                        printf("'%s' ended with return code %d.\n", cmdLinePipe2File[i], (int)pipe2FileProcs[i]->GiveReturnCode());
                    
                        if(0!=pipe2FileProcs[i]->GiveReturnCode())
                            rc=false;

                        delete pipe2FileProcs[i];
                    }
                }

                // Read COMMIT/ROLLBACK from "stdin":
                char buffer[1024];
                int character;

                i=0;
            do
            {
                character=getchar();

                if(EOF!=character && '\n'!=character)
                    buffer[i++]=(char)character;
            }
            while(i<1024 && EOF!=character && '\n'!=character);

            buffer[i]='\0';

            printf("adint2: Got a '%s'\n", buffer);

                if(0!=cn36_StrUprCmp(buffer, "COMMIT "))
                    rc=false;  
            }
        }

        if(parseResult.getKeyword("-q").isSet())
        {
            tni34_ADirectory dir('\0'==dataDir[0]?".":dataDir);
            const char * nextEntry=0;
            const char * backupNamePart=parseResult.getVariable("<name>").getValue();
            size_t       ignoreLength=0;

            if(0<strlen(backupNamePart) && backupNamePart[strlen(backupNamePart)-1]=='*') //ignore any suffixed '*'
                ignoreLength=1;

            printf("\n");
            printf("Query available backups on TSM server >BERD3000116311A<:\n");
            printf("Savename                                                        [Sessions used]\n");
            printf("===============================================================================\n");
            printf("");

            Tools_Vector<char *> backupList;
            Tools_Vector<int>    backupCount;

            if(dir.Open())
            {
                int i=0;

			    while(dir.Read(nextEntry) && 0!=nextEntry)
                {
                    if(0==strncmp("BackupADSM", nextEntry, strlen("BackupADSM")) &&     //file name starts with "BackupADSM"
                       0<strlen(nextEntry+strlen("BackupADSM")))                        //and is more than just "BackupADSM"
                    {
                        size_t pos=strlen("BackupADSM");
                        size_t sizeNum=0;
                        int    num=1;   //default for part number is 1 part

                        while(isdigit(*(nextEntry+pos+sizeNum))) //count the digits of the optional part number
                            sizeNum++;

                        if(0<sizeNum)
                            cn36_StrNToInt(num, nextEntry+pos, sizeNum);

                        pos+=sizeNum;

                        if(0==strncmp("_", nextEntry+pos, strlen("1")) &&                                       //"BackupADSM and optional part number are followed by _
                           0==strncmp(backupNamePart, nextEntry+pos+1, strlen(backupNamePart)-ignoreLength) &&  //and the backupNamePart
                           0<strlen(nextEntry+pos+1+strlen(backupNamePart)-ignoreLength))                       //after all this there should be other information (this is actually knowledge about the DBM Server
                        {
                            size_t j=0;

                            for(j=0; j<backupList.size(); j++)  //search the backup in all other known backups
                            {
                                if(0==strcmp(backupList[j], nextEntry+pos+1))
                                {
                                    backupCount[j]++;
                                    break;
                                }
                            }

                            if(j==backupList.size()) //If we did not find the backup, add it to the lists
                            {
                                char * name=0;

                                if(cn36_StrAllocCpy(name, nextEntry+pos+1) && backupList.push_back(name))
                                {
                                    if(!backupCount.push_back(1))
                                    {
                                        backupList.pop_back();
                                        delete name;
                                    }
                                }
                            }

                            i++;
                        }
                    }
                }

                if(0==i)
                    printf("Could not find any matching files in directory '%s'\n", dir.GiveName());

                for(i=0; i<(int)backupList.size(); i++)
                {
                    char sessionAsString[21];

                    sprintf(sessionAsString, "%d", (int)backupCount[i]);

                    size_t sizeEntry=strlen(backupList[i]);
                    size_t sizeSession=strlen(sessionAsString);

                    //savename
                    printf(backupList[i]);

                    //sessions used
                    if(sizeEntry<70)
                    {
                        for(int j=70-sizeEntry; j>0; j--)
                            printf(" ");
                    }
                    else
                        printf(" ");

                    printf("[%d]\n", (int)backupCount[i]);
                }

                for(size_t j=0; j<backupList.size(); j++)
                    delete backupList[j];
            }
            else
                printf("Could not open directory '%s'.\n", dir.GiveName());
        }

        if(parseResult.getKeyword("-r").isSet())
        {
            char                       cmdLinePipe2File[100][2048];
            tni33_ABackgroundProcess * pipe2FileProcs[100];
            int                        numberOfRunningPipe2Files=0;
            int                        i;

            for(i=0; i<numberOfPipes; i++)
                pipe2FileProcs[i]=0;

			for(i=0; i<numberOfPipes; i++)
			{
                char iAsString[21];
                sprintf(iAsString, "%d", (int)i);

                sprintf(
                    cmdLinePipe2File[i],
                    "pipe2file -d file2pipe -nowait -p %s -f %sBackupADSM%s_%s",
                    argv[argumentsStartOfPipes+i*2],
                    dataDir,
                    numberOfPipes>1?iAsString:"",
                    parseResult.getVariable("<name>").getValue());

                pipe2FileProcs[i]=new tni33_ABackgroundProcess(cmdLinePipe2File[i]);

                if(0==pipe2FileProcs[i] || !pipe2FileProcs[i]->WasStarted())
                    printf("Could not start '%s'.\n", cmdLinePipe2File[i]);
                else
                {
                    printf("Started '%s'.\n", cmdLinePipe2File[i]);
                    numberOfRunningPipe2Files++;
                }
			}

            while(0<numberOfRunningPipe2Files)
            {
                numberOfRunningPipe2Files=0;

                for(i=0; i<numberOfPipes; i++)
                {
                    if(0!=pipe2FileProcs[i] && pipe2FileProcs[i]->IsRunning())
                    {
                        numberOfRunningPipe2Files++;
                        printf("%d is running\n", (int)i);
                    }
                    else
                        printf("%d is not running\n", (int)i);
                }

                if(0!=numberOfRunningPipe2Files)
                    cn36_Sleep(1);
            }

            for(i=0; i<numberOfPipes; i++)
            {
                if(0!=pipe2FileProcs[i])
                {
                    printf("'%s' ended with return code %d.\n", cmdLinePipe2File[i], (int)pipe2FileProcs[i]->GiveReturnCode());
                
                    if(0!=pipe2FileProcs[i]->GiveReturnCode())
                        rc=false;

                    delete pipe2FileProcs[i];
                }
            }
        }

        if(parseResult.getKeyword("-d").isSet())
        {
            tni34_ADirectory dir('\0'==dataDir[0]?".":dataDir);
            const char * nextEntry=0;
            const char * backupNamePart=parseResult.getVariable("<name>").getValue();
            size_t       ignoreLength=0;

            if(0<strlen(backupNamePart) && backupNamePart[strlen(backupNamePart)-1]=='*') //ignore any suffixed '*'
                ignoreLength=1;

            if(dir.Open())
            {
                int i=0;

			    while(dir.Read(nextEntry) && 0!=nextEntry)
                {
                    if(0==strncmp("BackupADSM", nextEntry, strlen("BackupADSM")) &&
                       0<strlen(nextEntry+strlen("BackupADSM")))
                    {
                        size_t pos=strlen("BackupADSM");
                        size_t sizeNum=0;
                        int    num=1;   //default for part number is 1 part

                        while(isdigit(*(nextEntry+pos))) //allow for digits of the optional part number
                            pos++;

                        if(0==strncmp("_", nextEntry+pos, strlen("1")) &&                                               //"BackupADSM and optional part number are followed by _
                           0==cn36_StrNUprCmp(backupNamePart, nextEntry+pos+1, strlen(backupNamePart)-ignoreLength) &&  //and the backupNamePart
                           0<strlen(nextEntry+pos+1+strlen(backupNamePart)-ignoreLength))                               //after all this there should be other information (this is actually knowledge about the DBM Server
                        {
                            char * absoluteFileName=0;

                            cn36_StrAlloc(absoluteFileName, strlen(dataDir)+strlen(nextEntry));
                            sprintf(absoluteFileName, "%s%s", dataDir, nextEntry);

                            tni34_AFile file(absoluteFileName, tni34_AFile::FromDisk, 0, 0);

                            if(file.IsAFile())
                            {
                                i++;
                                file.Remove();
                            }

                            cn36_StrDealloc(absoluteFileName);
                        }
                    }
                }

                if(0==i)
                    printf("Could not find any matching files in directory '%s'\n", dir.GiveName());
            }
            else
                printf("Could not open directory '%s'.\n", dir.GiveName());
        }

        cn36_StrDealloc(dataDir);
    }

    if(rc)
        printf("\nTDI00074: ADINT2 Simulator has ended successful.\n\n");

    return rc?0:2;
}
