package com.sap.sapdb.DBMTest.eventing;

/*!
  -----------------------------------------------------------------------------
  module: DispatcherTest.java
  -----------------------------------------------------------------------------


    ========== licence begin  GPL
    Copyright (c) 2000-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end

*/


import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Date;
import java.util.StringTokenizer;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import com.sap.sapdb.DBMTest.DBMUtility;
import com.sap.sapdb.testframe.driver.TestDatabaseException;
import com.sap.sapdb.testframe.driver.TestDatabaseSap;
import com.sap.sapdb.testframe.driver.TestProtocol;
import com.sap.sapdb.testframe.testcase.TestCase;
import com.sap.sapdb.testframe.testcase.TestPreparedStatement;
import com.sap.sapdb.testframe.testcase.TestStatement;

/**
 * @author d026948
 *
 * To change the template for this generated type comment go to
 * Window&gt;Preferences&gt;Java&gt;Code Generation&gt;Code and Comments
 */
public class DispatcherTest extends TestCase {

    private static final String CMD_GETDISPATCHERPARAM     = "dbm_configget RunEventDispatcher";
	private static final String CMD_SETDISPATCHERPARAM_YES = "dbm_configset RunEventDispatcher yes";
	private static final String CMD_SETDISPATCHERPARAM_NO  = "dbm_configset RunEventDispatcher no";
	private static final String CMD_GETDISPATCHERPROT      = "file_getfirst EVTDISPPRT";
    private static final String CMD_SHOWTASKS              = "show tasks";
	private static final String VAL_YES                    = "yes";
    private static final String VAL_NO                     = "no";
	private static final String PRT_DISPATCHER             = "dbmevtdisp.prt";
    private static final String PRT_EXTENDER               = "dbmevthndl_extendDB.prt";
    private static final String ZIPPED_LOG_TO_UPLOAD       =  DispatcherTest.class.getName() + "_logs.zip";

    private static final String TEST_DATA                  = "0123456789abcdefghijklmnopqrstuvwxyz";
    private static final String NAME_TABLE                 = "TESTFILLER";
    private static final String NAME_COLUMN                = "Data            (%)";

    private static final String NAME_EVT_TASK              = "EventTs";
    private static final String STATUS_EVT_TASK            = "Stopped";

	private static final int    MAX_FILLING_DEGREE         = 95;
    private static final int    DATA_LENGTH_LOOP           = 120;
    private static final String PARAM_TIMEOUT              = "MSEC_TIMEOUT";
    private static final long   MSEC_TIMEOUT               = 1800000L; // 30 minutes
    private static final long   MSEC_WAIT_FOR_FILE         = 1000L;
    private static final long   MSEC_WAIT_FOR_FILE_TOTAL   = 30000L;
    private static final int    FILL_CHECK_INTERVAL        = 1000;
    private static final int    RECHECK_REPEAT             = 5;
    private static final long   RECHECK_SLEEP_TIME         = 500L;

    private static final int    RES_OKAY = 0;
    private static final int    RES_NO_PROTOCOL = 1;
    private static final int    RES_NO_EVENTTASK = 2;

    private StringBuffer m_DataBuffer = null;

	private TestDatabaseSap m_sapdb = null;
    private DBMUtility m_dbmUtil = null;

    /**
	 * @throws java.sql.SQLException
	 * @throws com.sap.sapdb.testframe.driver.TestDatabaseException
	 */
	public DispatcherTest() throws SQLException, TestDatabaseException {
		super();
	}

	public static boolean isSAPOnly() {
			// if the default is ever changed...
			return true;
	}

    public static long getTimeout() {
        //timeout in milliseconds
        long timeout = MSEC_TIMEOUT;
        try {
            timeout = getParameterLong(PARAM_TIMEOUT, MSEC_TIMEOUT);
        } finally {}
        return timeout;
    }

	/* (non-Javadoc)
	 * @see java.lang.Runnable#run()
	 */
	public void run() {

        m_sapdb = (TestDatabaseSap) super.getDatabase();
        m_dbmUtil = new DBMUtility(m_sapdb,this);
		
        // delete old logfile zip if any exists
        try {
            File upLoadee = new File(m_sapdb.getRunDir() + File.separator + ZIPPED_LOG_TO_UPLOAD);
            if( upLoadee.exists())
                upLoadee.delete();
        }
        catch( Exception e ) {
            // do nothing...
        }
        
        try {
    		if( m_dbmUtil.execCmd(CMD_GETDISPATCHERPARAM,DBMUtility.OK_OK) ) {
    			if( !m_dbmUtil.getAnswer().equals(VAL_YES) ) {
    				// make sure that there is NO dispatcher protocol,
    				// otherwise it's an error
    				if( !m_dbmUtil.execCmd(CMD_GETDISPATCHERPROT,DBMUtility.ERR_RTE,false) ) {
                        throw new DispatcherTestException( "dispatcher protocol found but there should be none. bad." );
    				}
    
    				// set parameter
    				if( m_dbmUtil.execCmd(CMD_SETDISPATCHERPARAM_YES,DBMUtility.OK_OK) ) {
    					// now restart the database
    					try {
    						// the restart() method does no real restart...
    						m_sapdb.stop();
    						m_sapdb.start();
    					}
    					catch (TestDatabaseException e) {
                            throw new DispatcherTestException( "could not restart database, " + e.getMessage() + ". abort." );
    					}
    				}
    				else {
    					// setting parameter failed
                        throw new DispatcherTestException( "could not set parameter 'RunEventDispatcher'. abort." );
    				}
    			}
    		}
    		else {
    			// getting parameter failed
                throw new DispatcherTestException( "could not get parameter 'RunEventDispatcher'. abort." );
    		}
        }
        catch( DispatcherTestException dte ) {
            // something went wrong during the perparation...
            super.addMessage( super.getTestId(), TestProtocol.ERROR, dte.getMessage() );
            return;
        }

		// dispatcher should be running here

		// check if dispatcher is running
        int dispatcherUp = isDispatcherRunning();
        try {
            long start = System.currentTimeMillis();                        
            while( (dispatcherUp != RES_OKAY) && System.currentTimeMillis()-start<MSEC_WAIT_FOR_FILE_TOTAL ) {
                Thread.sleep(MSEC_WAIT_FOR_FILE);
                dispatcherUp = isDispatcherRunning();
            }
        } catch (InterruptedException ie) {}
        
		switch( dispatcherUp ) {
            case RES_NO_EVENTTASK:
                super.addMessage(super.getTestId(),
                                 TestProtocol.ERROR,
                                 "event dispatcher not started after "+String.valueOf(((float)MSEC_WAIT_FOR_FILE_TOTAL)/1000.0)+"s. no event task. abort." );                
                return;
            case RES_NO_PROTOCOL:
                super.addMessage(super.getTestId(),
                                 TestProtocol.ERROR,
                                 "event dispatcher not started after "+String.valueOf(((float)MSEC_WAIT_FOR_FILE_TOTAL)/1000.0)+"s. no protocol. abort." );                
                return;
            default:
		}

        super.addMessage(super.getTestId(), TestProtocol.INFO, "event dispatcher is running." );

		// okay, it's running now fill the database
        Connection con                     = null;
        int count                          = 0;
        TestStatement stmt                 = null;
        TestPreparedStatement prstmtCheck  = null;
        TestPreparedStatement prstmtInsert = null;
		
        // fill data buffer
        StringBuffer testDataBuffer = new StringBuffer();
        for(int i=0; i<DATA_LENGTH_LOOP; i++)
            testDataBuffer.append(TEST_DATA);
        String testData = new String(testDataBuffer);
        
        try {
			con = m_sapdb.connect(m_sapdb.getDBAUser(),m_sapdb.getDBAPassword());
            con.setAutoCommit(false);
			// first, create the table
            stmt = new TestStatement(this, con);
            try {
                stmt.executeUpdate("drop table " + NAME_TABLE);
            }
            catch( SQLException sqle ) {
                // ignore this
            }
			stmt.executeUpdate("create table " + NAME_TABLE + " (cnt int, text char(128), myblob long byte, primary key(cnt))");
			// for checking
			prstmtCheck = new TestPreparedStatement(this, con, "select \""+ NAME_COLUMN +"\" from info_state"); 

			// fill the table
			prstmtInsert = new TestPreparedStatement(this, con, "insert " + NAME_TABLE + " values (?,?,?)");
			boolean goon = true;
			Date myd = new Date();
            int iDegreeCurrent = 0;

            super.addMessage(super.getTestId(), TestProtocol.INFO, "filling database." );
            int initialVols = getDataVolumes();
			while( goon ) {
				prstmtInsert.setInt(1, count++);
				myd.setTime(System.currentTimeMillis());
				prstmtInsert.setString(2,super.getTestId() + " " + myd.toString());
                prstmtInsert.setBytes(3,testData.getBytes());
				prstmtInsert.executeUpdate();
				
				if( (count % FILL_CHECK_INTERVAL) == 0 ) {
					// get current filling degree
					if( getDataVolumes() > initialVols ) {
						// obvisouly, a new volume was added
						goon = false;
						super.addMessage(super.getTestId(), TestProtocol.INFO,
							"at least one data volume was added. good." );
					}
                    else {
                        ResultSet res = prstmtCheck.executeQuery();
                        if( !res.next() )
    						super.addMessage(super.getTestId(), TestProtocol.ERROR,
    							"error executing statement " + prstmtCheck.getSqlString() );
                        if( res.getInt(1) > MAX_FILLING_DEGREE ) {
                            int check = 0;
                            // recheck in loop, handler is asynchronous
                            while(check < RECHECK_REPEAT && !(getDataVolumes() > initialVols)) {
                                check++;
                                Thread.sleep(RECHECK_SLEEP_TIME);
                            }
                            if( !(getDataVolumes() > initialVols) )
                                throw new DispatcherTestException( "a new volume should have been added, but obvisouly wasn't. bad." );
                        }
                    }
				}				
			}
            super.addMessage(super.getTestId(), TestProtocol.INFO, "test finished successfully.");            
		}
        catch (TestDatabaseException e) {
            String message = null;
            String className = null;
            if( e.getOriginalException() != null ) {
                message = e.getOriginalException().getMessage();
                className = e.getOriginalException().getClass().getName();
            }
            else {
                message = e.getMessage();
                className = e.getClass().getName();
            }
            super.addMessage(super.getTestId(), TestProtocol.ERROR,
                className + " caught: " + message + ". abort." );
            createUploadableFile();
        }
        catch( Exception e ) {
            super.addMessage(super.getTestId(), TestProtocol.ERROR,
                e.getClass().getName() + " caught: " + e.getMessage() + ". abort." );
            createUploadableFile();
        }
        finally {
            super.addMessage(super.getTestId(), TestProtocol.INFO,
                "number of records inserted: " + count); 
            try {
                if( stmt != null )
                    stmt.close();
                if( prstmtCheck != null )
                    prstmtCheck.close();
                if( prstmtInsert != null )
                    prstmtInsert.close();

                stmt = new TestStatement(this, con);
                stmt.executeUpdate("drop table " + NAME_TABLE);
                con.commit();
                stmt.close();
                con.close();
            } catch (SQLException e) {
                super.addMessage(super.getTestId(), TestProtocol.ERROR,
                    e.getClass().getName() + " caught while cleaning up: " + e.getMessage() + ". abort." );

            }
        }
	}

    /*
     * not counting the number of volumes, just the number of lines in the
     * answer... good enough.
     */
    private int getDataVolumes() {
        int vols = 0;
        try {
            String res = m_sapdb.executeDBMcmd("param_getvolsall DATA");
            int length = res.length();
            for(int count = 0; count < length; count++) {
                if( res.charAt(count) == '\n' )
                    vols++;            
            }                
        }
        catch (TestDatabaseException e) {
            super.addMessage(super.getTestId(), TestProtocol.ERROR,
                e.getClass().getName() + " counting volumes: " + e.getMessage() );
        }
        finally {}
        return vols;
    }

    private int isDispatcherRunning() {
        m_dbmUtil.execCmd(CMD_SHOWTASKS, DBMUtility.OK_OK);
        StringTokenizer sTok = new StringTokenizer(m_dbmUtil.getAnswer(),"\n");
        while( sTok.hasMoreTokens()) {
            String taskLine = sTok.nextToken();
            if( taskLine.indexOf(NAME_EVT_TASK)>0 && taskLine.indexOf(STATUS_EVT_TASK)>0 ) {
                // dispatcher is listening now check if we can read it's protocol
                if( m_dbmUtil.execCmd(CMD_GETDISPATCHERPROT,DBMUtility.OK_OK) ) {
                    return RES_OKAY;
                }
                else {
                    return RES_NO_PROTOCOL;
                }
            }
        }
        return RES_NO_EVENTTASK;
    }
    
    private void createUploadableFile() {
        // called only if an error occurred during test
        // upload dispatcher protocol and extender protocol
        TestDatabaseSap sapdb = (TestDatabaseSap) super.getDatabase();
        try {
            ZipEntry dispatcherProtEntry = new ZipEntry( sapdb.getRunDir() + File.separator + PRT_DISPATCHER );
            ZipEntry extenderProtEntry = new ZipEntry( sapdb.getRunDir() + File.separator + PRT_EXTENDER );
    
            ZipOutputStream zos = new ZipOutputStream( new FileOutputStream(
                    sapdb.getRunDir() + File.separator + ZIPPED_LOG_TO_UPLOAD));

            File f = null;
                    
            // now put data into the zip
            byte[] data = new byte[1024];
            int byteCount;
            BufferedInputStream bis = null;
            
            // dispatcher protocol
            f = new File(dispatcherProtEntry.getName());
            if( f.exists() ) {
                zos.putNextEntry(dispatcherProtEntry);
                bis = new BufferedInputStream(new FileInputStream(f));
                while ((byteCount = bis.read(data, 0, data.length)) > -1) {
                    zos.write(data, 0, byteCount);
                }
            }                    
            // extender protocol
            f = new File(extenderProtEntry.getName());
            if( f.exists() ) {
                zos.putNextEntry(extenderProtEntry);
                bis = new BufferedInputStream(new FileInputStream(f));
                while ((byteCount = bis.read(data, 0, data.length)) > -1) {
                    zos.write(data, 0, byteCount);
                }
            }
            zos.close();
        }
        catch( IOException ioe ) {
            super.addMessage(super.getTestId(), TestProtocol.ERROR,
                "IOException caught while creating zip for uploading logs: " + ioe.getMessage());
        }
        catch( TestDatabaseException tde ) {
            super.addMessage(super.getTestId(), TestProtocol.ERROR,
                "TestDatabaseException caught while creating zip for uploading logs: " + tde.getMessage());
        }
    }

    /* (non-Javadoc)
     * @see com.sap.sapdb.testframe.testcase.TestCase#getLogFile()
     */
    public String getLogFile() {
        try {
            File upLoadee = new File(m_sapdb.getRunDir() + File.separator + ZIPPED_LOG_TO_UPLOAD);
            if( upLoadee.exists() ) {
                super.addMessage(super.getTestId(), TestProtocol.INFO, upLoadee.getCanonicalPath());
                return upLoadee.getCanonicalPath();
            }
            else
                return null;
        }
        catch( Exception e ) {
            // no so good...
            return null;
        }
    }
}
