/*!
  \file    ToolsDC_GZippedTarCreator.cpp
  \author  MarcW
  \ingroup Tools Common
  \brief   

    ========== licence begin  SAP

    Copyright (c) 2002-2006 SAP AG

    All rights reserved.

    ========== licence end

*/

#include <time.h>
#include <stdio.h>
#include <stdlib.h>

#include "gcn00.h"
#include "heo06.h"

#include "ToolsCommon/DataCompression/ToolsDC_GZippedTarCreator.hpp"


#define TARBLOCKSIZE 512

ToolsDC_GZippedTarCreator::ToolsDC_GZippedTarCreator(const tsp00_Pathc& archiveName)
        : m_archiveName(archiveName) {

    FUNCTION_DBG_MCN00_1("ToolsDC_GZippedTarCreator::ToolsDC_GZippedTarCreator");

    m_ErrorList.eo200_ClearEventList();

    tsp05_RteFileInfo   rteFileInfo;
    tsp05_RteFileError  rteFileErr;
    sqlfinfoc( m_archiveName.asCharp(), &rteFileInfo, &rteFileErr );

    if( rteFileErr.sp5fe_result != vf_ok ) {
        // could not get information about file
        teo200_EventList aRTE(FUNCTION_NAME_MCN00_1, rteFileErr.sp5fe_result, TERR_CN00_1, "DBM", "%.*s", rteFileErr.sp5fe_text.length(), rteFileErr.sp5fe_text.asCharp());
        teo200_EventList aDBM(aRTE, FUNCTION_NAME_MCN00_1, ERR_FILE_CN00_1);
        m_ErrorList = aDBM;
        return;
    }

    if( rteFileInfo.sp5fi_exists && (!rteFileInfo.sp5fi_writeable || !rteFileInfo.sp5fi_readable) ) {
        // cannot use the existing file
        teo200_EventList aTGZ(FUNCTION_NAME_MCN00_1, 1, TERR_CN00_1, "DBM", "open tgz error");
        teo200_EventList aDBM(aTGZ, FUNCTION_NAME_MCN00_1, ERR_FILE_CN00_1);
        m_ErrorList = aDBM;
        return;
    }

    m_tarEntryList.clear();
}

void ToolsDC_GZippedTarCreator::clearEntryList() {
    
    if( m_tarEntryList.empty() )
        return;

    for( TarEntryList::iterator iter = m_tarEntryList.begin(); iter != m_tarEntryList.end(); ++iter )
        delete *iter;

    m_tarEntryList.clear();
}

ToolsDC_GZippedTarCreator::~ToolsDC_GZippedTarCreator() {
    
    clearEntryList();
}

bool ToolsDC_GZippedTarCreator::eraseArchive() {

    FUNCTION_DBG_MCN00_1("ToolsDC_GZippedTarCreator::eraseArchive");

    tsp05_RteFileInfo   rteFileInfo;
    tsp05_RteFileError  rteFileErr;

    sqlfinfoc( m_archiveName.asCharp(), &rteFileInfo, &rteFileErr );

    if( rteFileErr.sp5fe_result != vf_ok ) {
        // could not get information about file
        teo200_EventList aTGZ(FUNCTION_NAME_MCN00_1, 1, TERR_CN00_1, "DBM", "remove tgz error");
        teo200_EventList aDBM(aTGZ, FUNCTION_NAME_MCN00_1, ERR_FILE_CN00_1);
        m_ErrorList = aDBM;
        return false;
    }
    else {
        if( rteFileInfo.sp5fi_exists ) {
            sqlferasec( m_archiveName.asCharp(), &rteFileErr );
            if( rteFileErr.sp5fe_result != vf_ok ) {
                teo200_EventList aTGZ(FUNCTION_NAME_MCN00_1, 1, TERR_CN00_1, "DBM", "remove tgz error");
                teo200_EventList aDBM(aTGZ, FUNCTION_NAME_MCN00_1, ERR_FILE_CN00_1);
                m_ErrorList = aDBM;
                return false;
            }
        }
    }

    return true;
}

bool ToolsDC_GZippedTarCreator::addFile(const tsp00_Pathc& originalName, const tsp00_Pathc& nameInArchive) {
    
    TarEntry* newEntry(0);
    newEntry = new TarEntry(originalName, nameInArchive);
    m_tarEntryList.push_back(newEntry);

    return true;
}

bool ToolsDC_GZippedTarCreator::writeArchive() {

    gzFile tarHandle(0);
    tsp00_Int4 originalHandle(0);
    
    bool rc = writeArchiveImpl(tarHandle, originalHandle);

    if( 0 != tarHandle )
        gzclose(tarHandle);

    if( 0 != originalHandle ) {
        tsp05_RteFileError  rteFileErr;
        sqlfclosec(originalHandle, sp5vf_close_normal, &rteFileErr);
    }

    return rc;
}

const teo200_EventList& ToolsDC_GZippedTarCreator::lastEvent() {
    return m_ErrorList;
}

bool ToolsDC_GZippedTarCreator::writeArchiveImpl(gzFile& tarHandle, tsp00_Int4& originalHandle) {    
    
    FUNCTION_DBG_MCN00_1("ToolsDC_GZippedTarCreator::writeArchiveImpl");

    if( m_tarEntryList.empty() )
        return true;

    // open archive for writing (not appending!)
    tarHandle = gzopen(m_archiveName.asCharp(), "wb");
    if( 0 == tarHandle ) {
        // could not open archive
        // report an error
        return false;
    }

    for( TarEntryList::const_iterator iter = m_tarEntryList.begin(); iter != m_tarEntryList.end(); ++iter ) {

        tsp05_RteFileError  rteFileErr;

        // open original file for reading
        sqlfopenc((*iter)->getOriginalName(), sp5vf_binary, sp5vf_read, sp5bk_buffered, &originalHandle, &rteFileErr );
        if( rteFileErr.sp5fe_result != vf_ok ) {
            // could not open original file for reading
            // we ignore the error and skip this file
            originalHandle = 0;
            continue;
        }

        // create tar header for this file
        TarHeader tarHeader((*iter)->getOriginalName(), (*iter)->getNameInArchive());

        // write tar header for this file to archive
        if( sizeof(tarHeader) != gzwrite(tarHandle, &tarHeader, sizeof(tarHeader)) ) {
            // could not write header to archive
            teo200_EventList aTGZ(FUNCTION_NAME_MCN00_1, 1, TERR_CN00_1, "DBM", "write tgz header error");
            teo200_EventList aDBM(aTGZ, FUNCTION_NAME_MCN00_1, ERR_FILE_CN00_1);
            m_ErrorList = aDBM;
            return false;
        }        

        // finally write the file itself to the archive
        bool goon(true);
        tsp00_Longint requiredPadding(0);
        while( goon ) {
            char transferBuffer[1024];
            tsp00_Longint bytesTransferred(0);
            sqlfreadc( originalHandle, transferBuffer, sizeof(transferBuffer), &bytesTransferred, &rteFileErr );
            if( rteFileErr.sp5fe_result != vf_notok  && 0 != bytesTransferred ) {
                if( (int)bytesTransferred != gzwrite(tarHandle, transferBuffer, (unsigned int)bytesTransferred) ) {
                    // could not write all bytes to archive
                    teo200_EventList aTGZ(FUNCTION_NAME_MCN00_1, 1, TERR_CN00_1, "DBM", "write tgz data error");
                    teo200_EventList aDBM(aTGZ, FUNCTION_NAME_MCN00_1, ERR_FILE_CN00_1);
                    m_ErrorList = aDBM;
                    return false;
                }
            }

            if( rteFileErr.sp5fe_result == vf_notok ) {
                // could not read from original file
                teo200_EventList aRTE(FUNCTION_NAME_MCN00_1, rteFileErr.sp5fe_result, TERR_CN00_1, "DBM", "%.*s", rteFileErr.sp5fe_text.length(), rteFileErr.sp5fe_text.asCharp());
                teo200_EventList aDBM(aRTE, FUNCTION_NAME_MCN00_1, ERR_FILE_CN00_1);
                m_ErrorList = aDBM;
                return false;
            }

            if( rteFileErr.sp5fe_result == vf_eof ) {
                goon = false;
                sqlfclosec(originalHandle, sp5vf_close_normal, &rteFileErr); // we ignore the error here
                originalHandle = 0;
                // pad with 0x00 to a multiple of 512 bytes
                if( 0 < requiredPadding ) {
                    while( TARBLOCKSIZE != requiredPadding ) {
                        gzputc(tarHandle, 0);
                        ++requiredPadding;
                    }
                }
            }
            else {
                // rteFileErr.sp5fe_result is vf_ok, we must pad
                requiredPadding = bytesTransferred % TARBLOCKSIZE;
            }
        }
    }

    // now that all files are in the archive, append tar padding of 2 * 512 0 bytes
    char endpadding[2*TARBLOCKSIZE];
    memset(&endpadding, 0, sizeof(endpadding));

    tsp00_Longint bytesTransferred(0);
    if( (int)sizeof(endpadding) != gzwrite(tarHandle, &endpadding, sizeof(endpadding)) ) {
        // error writing the padding to the archive
        teo200_EventList aTGZ(FUNCTION_NAME_MCN00_1, 1, TERR_CN00_1, "DBM", "write tgz padding error");
        teo200_EventList aDBM(aTGZ, FUNCTION_NAME_MCN00_1, ERR_FILE_CN00_1);
        m_ErrorList = aDBM;
        return false;
    }

    return true;
}

TarHeader::TarHeader(const char* originalName, const char* nameInArchive) {
    
// prepare data fields
    memset(this, 0, sizeof(*this));
    memset(&checksum, 32, sizeof(checksum)); // ASCII space
    sprintf(ustarMagic, "%s", "ustar");

//  set file information
    tsp05_RteFileInfo   rteFileInfo;
    tsp05_RteFileError  rteFileErr;
    sqlfinfoc( originalName, &rteFileInfo, &rteFileErr );
    
    if( rteFileErr.sp5fe_result != vf_ok ) {
        // could not get information about file
        fprintf(stderr, "could not get information about file: %s\n", originalName);
        return;
    }

    if( !rteFileInfo.sp5fi_exists ) {
        // file to add does not exist
        // never happens, caller is responsible for this!
        return;
    }

    // file name
    size_t fileNameLength = strlen(nameInArchive);
    if( fileNameLength > sizeof(fileName)+sizeof(fileNamePrefix) ) {
        // file name is too long (even for ustar)
        fprintf(stderr, "filename too long for TAR: %s\n", nameInArchive);
        return;
    }
    if( fileNameLength <= sizeof(fileName) ) {
        // entire file name fits the filename field
        memcpy(fileName, nameInArchive, strlen(nameInArchive) );
    }
    else {
        // we must use the prefix field
        memcpy(fileNamePrefix, nameInArchive, strlen(nameInArchive)-sizeof(fileName));
        memcpy(fileName, nameInArchive+strlen(nameInArchive)-sizeof(fileName), sizeof(fileName));
    }

    // file dates
    tsp00_Datec dateString;
    tsp00_Timec timeString;
    
    struct tm modificationTimeStruct;
    memset(&modificationTimeStruct, 0, sizeof(modificationTimeStruct));

    dateString.p2c(rteFileInfo.sp5fi_date_modified);
    timeString.p2c(rteFileInfo.sp5fi_time_modified);
    int nDate = atoi(dateString.asCharp());
    int nTime = atoi(timeString.asCharp());
    modificationTimeStruct.tm_year   = (nDate / 10000) - 1900;
    modificationTimeStruct.tm_mon    = ((nDate % 10000) / 100) - 1;
    modificationTimeStruct.tm_mday   = nDate % 100;
    modificationTimeStruct.tm_hour   = nTime / 10000;
    modificationTimeStruct.tm_min    = (nTime % 10000) / 100;
    modificationTimeStruct.tm_sec    = nTime % 100;
    sprintf(modificationTime, "%0.*lo", sizeof(modificationTime)-1, (long)mktime(&modificationTimeStruct));

    // file size
    sprintf(fileSize, "%0.*lo", sizeof(fileSize)-1, (long) rteFileInfo.sp5fi_size);

    // access mode (rw for owner, r for group and others)
    sprintf(fileMode, "%0.*o", sizeof(fileMode)-1, 0644); // 0644 is an octal number

    // type (regular file)
    typeFlag = '0'; // we could also read this from rteFileInfo...

// check sum of header
    int sum(0);
    for(int i=0; i<sizeof(*this); ++i)
        sum += *(((char*)this)+i);

    sprintf(checksum, "%0*o", sizeof(checksum)-2, sum);
    checksum[7] = ' ';
}
