/*


    ========== licence begin GPL
    Copyright (C) 2002-2003 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


*/
package com.sap.dbtech.jdbcext;

import java.io.*;
import java.math.BigDecimal;
import java.sql.*;
import java.util.Calendar;
import java.util.Vector;
import java.util.ArrayList;
import java.util.Map;
import javax.sql.*;
import com.sap.dbtech.jdbcext.translators.*;

public abstract class AbstractRowSetSapDB
    implements Serializable, Cloneable
{
    protected String command = null;
    protected String URL = null;
    protected String dataSource = null;
    protected transient String username = null;
    protected transient String password = null;
    protected int rowSetType = ResultSet.TYPE_SCROLL_SENSITIVE;
    protected boolean showDeleted = false;
    protected int queryTimeout = 0;
    protected int maxRows = 0;
    protected int maxFieldSize = 0;
    protected int concurrency = java.sql.ResultSet.CONCUR_READ_ONLY;
    protected boolean readOnly = true;
    protected boolean escapeProcessing = true;
    protected int isolation = Connection.TRANSACTION_READ_COMMITTED;
    protected int fetchDir = ResultSet.FETCH_FORWARD;
    protected int fetchSize = com.sap.dbtech.jdbc.ResultSetSapDB.DEFAULT_FETCHSIZE;
    protected Map map = null;
    protected ArrayList listeners = null;
    protected Vector params = null;

    public AbstractRowSetSapDB()
    {
        listeners = new ArrayList();
    }

    public void addRowSetListener(RowSetListener rlistener)
    {
        if(rlistener==null) {
            return;
        }
        synchronized(this.listeners) {
            int list_sz=this.listeners.size();
            for(int i=0; i<list_sz; ++i) {
                if(this.listeners.get(i)==rlistener) {
                    return;
                }
            }
            listeners.add(rlistener);
        }
    }

    public void clearParameters()
        throws SQLException
    {
        params.clear();
    }

    public String getCommand()
    {
        return command;
    }

    public int getConcurrency()
        throws SQLException
    {
        return concurrency;
    }

    public String getDataSourceName()
    {
        return dataSource;
    }

    public boolean getEscapeProcessing()
        throws SQLException
    {
        return escapeProcessing;
    }

    public int getFetchDirection()
        throws SQLException
    {
        return fetchDir;
    }

    public int getFetchSize()
        throws SQLException
    {
        return fetchSize;
    }

    public int getMaxFieldSize()
        throws SQLException
    {
        return maxFieldSize;
    }

    public int getMaxRows()
        throws SQLException
    {
        return maxRows;
    }

    public Object[] getParams()
        throws SQLException
    {
        return params.toArray();
    }

    public String getPassword()
    {
        return password;
    }

    public int getQueryTimeout()
        throws SQLException
    {
        return queryTimeout;
    }

    public boolean getShowDeleted()
        throws SQLException
    {
        return showDeleted;
    }

    public int getTransactionIsolation()
    {
        return isolation;
    }

    public int getType()
        throws SQLException
    {
        return rowSetType;
    }

    public Map getTypeMap()
    {
        return map;
    }

    public String getUrl()
        throws SQLException
    {
        return URL;
    }

    public String getUsername()
    {
        return username;
    }

    protected void initParams()
    {
        params = new Vector();
    }

    public boolean isReadOnly()
    {
        return readOnly;
    }

    protected void notifyCursorMoved()
    {
        if(!this.listeners.isEmpty())
        {
            RowSetEvent rowsetevent = new RowSetEvent((RowSet)this);
            synchronized(this.listeners) {
                int list_sz=this.listeners.size();
                for(int i=0; i<list_sz; ++i) {
                    RowSetListener rl=(RowSetListener)this.listeners.get(i);
                    rl.cursorMoved(rowsetevent);
                }
            }
        }
    }

    protected void notifyRowChanged()
    {
        if(!this.listeners.isEmpty())
        {
            RowSetEvent rowsetevent = new RowSetEvent((RowSet)this);
            synchronized(this.listeners) {
                int list_sz=this.listeners.size();
                for(int i=0; i<list_sz; ++i) {
                    RowSetListener rl=(RowSetListener)this.listeners.get(i);
                    rl.rowChanged(rowsetevent);
                }
            }
        }
    }

    protected void notifyRowSetChanged()
    {
        if(!this.listeners.isEmpty())
        {
            RowSetEvent rowsetevent = new RowSetEvent((RowSet)this);
            synchronized(this.listeners) {
                int list_sz=this.listeners.size();
                for(int i=0; i<list_sz; ++i) {
                    RowSetListener rl=(RowSetListener)this.listeners.get(i);
                    rl.rowSetChanged(rowsetevent);
                }
            }
        }
    }

    public void removeRowSetListener(RowSetListener rowsetlistener)
    {
        if(this.listeners==null) {
            return;
        }
        synchronized(this.listeners) {
            int list_sz=this.listeners.size();
            for(int i=0; i<list_sz; ++i) {
                if(this.listeners.get(i)==rowsetlistener) {
                    this.listeners.remove(i);
                    return;
                }
            }
        }
    }

    public void setArray(int i, Array array)
        throws SQLException
    {
        params.add(new ArrayTranslator(i,array));
    }

    public void setAsciiStream(int i, InputStream inputstream, int j)
        throws SQLException
    {
        params.add(new AsciiStreamTranslator(i,inputstream,j));
    }

    public void setBigDecimal(int i, BigDecimal bigdecimal)
        throws SQLException
    {
        params.add(new BigDecimalTranslator(i,bigdecimal));
    }

    public void setBinaryStream(int i, InputStream inputstream, int j)
        throws SQLException
    {
        params.add(new BinaryStreamTranslator(i,inputstream,j));
    }

    public void setBlob(int i, Blob blob)
        throws SQLException
    {
        params.add(new BlobTranslator(i,blob));
    }

    public void setBoolean(int i, boolean flag)
        throws SQLException
    {
        params.add(new BooleanTranslator(i,flag));
    }

    public void setByte(int i, byte byte0)
        throws SQLException
    {
        params.add(new ByteTranslator(i,byte0));
    }

    public void setBytes(int i, byte abyte0[])
        throws SQLException
    {
        params.add(new BytesTranslator(i,abyte0));
    }

    public void setCharacterStream(int i, Reader reader, int j)
        throws SQLException
    {
        params.add(new CharacterStreamTranslator(i,reader,j));
    }

    public void setClob(int i, Clob clob)
        throws SQLException
    {
        params.add(new ClobTranslator(i,clob));
    }

    public void setCommand(String s)
        throws SQLException
    {
        command = s;
        params.clear();
    }

    public void setConcurrency(int i)
    {
        concurrency = i;
    }

    public void setDataSourceName(String s)
    {
        if(s == null)
            dataSource = null;
        else
            dataSource = new String(s);
        URL = null;
    }

    public void setDate(int i, Date date)
        throws SQLException
    {
        params.add(new DateTranslator(i,date));
    }

    public void setDate(int i, Date date, Calendar calendar)
        throws SQLException
    {
        params.add(new DateTranslator(i,date,calendar));
    }

    public void setDouble(int i, double d)
        throws SQLException
    {
        params.add(new DoubleTranslator(i,d));
    }

    public void setEscapeProcessing(boolean flag)
        throws SQLException
    {
        escapeProcessing = flag;
    }

    public void setFetchDirection(int i)
        throws SQLException
    {
       fetchDir = i;
    }

    public void setFetchSize(int i)
        throws SQLException
    {
        fetchSize = i;
    }

    public void setFloat(int i, float f)
        throws SQLException
    {
        params.add(new FloatTranslator(i,f));
    }

    public void setInt(int i, int j)
        throws SQLException
    {
        params.add(new IntTranslator(i,j));
    }

    public void setLong(int i, long l)
        throws SQLException
    {
        params.add(new LongTranslator(i,l));
    }

    public void setMaxFieldSize(int i)
        throws SQLException
    {
        maxFieldSize = i;
    }

    public void setMaxRows(int i)
        throws SQLException
    {
        maxRows = i;
    }

    public void setNull(int i, int j)
        throws SQLException
    {
        params.add(new NullTranslator(i));
    }

    public void setNull(int i, int j, String s)
        throws SQLException
    {
        params.add(new NullTranslator(i));
    }

    public void setObject(int i, Object obj)
        throws SQLException
    {
        params.add(new ObjectTranslator(i,obj));
    }

    public void setObject(int i, Object obj, int j)
        throws SQLException
    {
        params.add(new ObjectTranslator(i,obj,j));

    }

    public void setObject(int i, Object obj, int j, int k)
        throws SQLException
    {
        params.add(new ObjectTranslator(i,obj,j,k));
    }

    public void setPassword(String s)
    {
        password = s;
    }

    public void setQueryTimeout(int i)
        throws SQLException
    {
        queryTimeout = i;
    }

    public void setReadOnly(boolean flag)
    {
        readOnly = flag;
    }

    public void setRef(int i, Ref ref)
        throws SQLException
    {
        params.add(new RefTranslator(i,ref));
    }

    public void setShort(int i, short word0)
        throws SQLException
    {
        params.add(new ShortTranslator(i,word0));
    }

    public void setString(int i, String s)
        throws SQLException
    {
        params.add(new StringTranslator(i,s));
    }

    public void setTime(int i, Time time)
        throws SQLException
    {
        params.add(new TimeTranslator(i,time));
    }

    public void setTime(int i, Time time, Calendar calendar)
        throws SQLException
    {
        params.add(new TimeTranslator(i,time,calendar));
    }

    public void setTimestamp(int i, Timestamp timestamp)
        throws SQLException
    {
        params.add(new TimestampTranslator(i,timestamp));
    }

    public void setTimestamp(int i, Timestamp timestamp, Calendar calendar)
        throws SQLException
    {
        params.add(new TimestampTranslator(i,timestamp,calendar));
    }

    public void setTransactionIsolation(int i)
    {
        isolation = i;
    }

    public void setType(int i)
    {
        rowSetType = i;
    }

    public void setTypeMap(Map map1)
    {
        map = map1;
    }

    public void setUnicodeStream(int i, InputStream inputstream, int j)
        throws SQLException
    {
        params.add(new UnicodeStreamTranslator(i,inputstream,j));
    }

    public void setUrl(String s)
        throws SQLException
    {
        if(s == null)
            URL = null;
        else
            URL = s;
        dataSource = null;
    }

    public void setUsername(String s)
    {
        username = new String(s);
    }
}