/*!
  @file           Log_ActionObjectFile.hpp
  @ingroup        ActionHandling
  @author         UweH
  @brief          defines class Log_ActionObjectFile

\if EMIT_LICENCE
    ========== licence begin  GPL
    Copyright (c) 2000-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end

\endif
*/



#ifndef Log_ActionObjectFile_H
#define Log_ActionObjectFile_H

/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "ggg00.h"  // tgg00_TransContext, tgg00_FileId
#include "SAPDBCommon/SAPDB_Types.hpp"
#include "Logging/Log_Types.hpp"
#include "Logging/Log_IAction.hpp"
#include "liveCache/LVC_Types.hpp"

/*===========================================================================*
 *  DEFINES                                                                  *
 *===========================================================================*/

class Log_AfterImage;
class Log_BeforeImage;
class Data_SplitSpaceWriter;
class Data_SplitSpaceReader;
 
/*===========================================================================*
 *  MACROS                                                                   *
 *===========================================================================*/
/*===========================================================================*
 *  EXTERNAL VARIABLES                                                       *
 *===========================================================================*/
/*===========================================================================*
 *  CLASSES, STRUCTURES, TYPES, UNIONS ...                                   *
 *===========================================================================*/

/*!
   @brief          Handle for manipulating OMS objects.
 */
class Log_ActionObjectFile : public Log_IAction
{

public:

    /// init the creation of a file 
    void InitCreation (tgg00_ObjFileNo            fileno,
                       bool                       forVarObjects,
                       SAPDB_UInt                 bodyLength,
                       SAPDB_UInt                 keyLength,
                       bool                       createSeveralKeyFiles,
                       const LVC_RootPageNoArray &rootPages)
    {
        m_Head.fileno                = fileno;
        m_Head.forVarObjects         = forVarObjects;
        m_Head.createSeveralKeyFiles = createSeveralKeyFiles;
        m_Head.bodyLength            = bodyLength;
        m_Head.keyLength             = keyLength;
        memcpy (m_Head.rootPages, rootPages.GetPersistentFormat(), rootPages.GetPersistentLength());
    }

    /// init the deletion of a file
    void InitDeletion (tgg00_ObjFileNo fileno)
    {
        m_Head.fileno = fileno;
    }

    /// get the FileNumber of the current entry
    tgg00_ObjFileNo GetFileNo () const
    {
        return m_Head.fileno;
    }


    /// get the minimum-entry-size
    static SAPDB_UInt GetMinimalLengthToCopyEntry()
    {
        return sizeof (Head);
    }
    
    /// get the minimum persistent size
    virtual SAPDB_UInt GetMinimalPersistentLength() const
    {
        return sizeof (Head);
    }
    
    /// get the total persistent size
    virtual SAPDB_UInt GetPersistentLength() const
    {
        return sizeof (Head);
    }
    
    /// The Space must be initialized with the value given by GetPersistentLength().
    virtual void WritePersistentFormat (Data_SplitSpaceWriter &Writer,
                                        bool                  &isOK) const;

    /// all internal components are initialized with the reader
    virtual void ReadPersistentFormat (Data_SplitSpaceReader  &Reader,
                                       SAPDBMem_IRawAllocator &Allocator,
                                       bool                   &isOK);

    /// dump this action to the trace
    virtual void WriteToTrace (const char * Title) const;

    /// The action is re-done.
    void Redo (tgg00_TransContext   &Trans,
               const Log_AfterImage &AfterImage);
    
    /// The action is rollbacked.
    void Undo (tgg00_TransContext    &Trans,
               const Log_BeforeImage &BeforeImage)
    {
        Trans.trError_gg00 = e_not_implemented;
    }
    
private:

    /// the physical representation in undo/redo-files
    struct Head
    {
        tgg00_ObjFileNo fileno;                ///< filenumber of the obejectfile (4 bytes)
        SAPDB_Bool1     forVarObjects;         ///< whether it is for variable objects (1 bytes)
        SAPDB_Bool1     createSeveralKeyFiles; ///< (1 bytes) do create not only one file
        SAPDB_UInt2     bodyLength;            ///< (2 bytes) length of the body
        SAPDB_UInt2     keyLength;             ///< (2 bytes) length of the key
        SAPDB_Byte      rootPages[LVC_ROOT_PAGENO_ARRAY_SIZE]; ///< the rootPages of this file
        
        /// the default constructor 
        Head()
        {
            fileno                = 0;
            forVarObjects         = false;
            createSeveralKeyFiles = false;
            bodyLength            = 0;
            keyLength             = 0;
        }
    };

    /// the in-memory representation of the header of this action
    Head m_Head;
};


#endif  /* Log_ActionObjectFile_H */
