#    coding: utf-8
#
#    module:       transport.py
#
#    responsible : d033893
#
#    special area: loader
#
#    description : transport, export or import sdb data/metadata using the 
#                  transformation service provided by sdb loader 
#                  imports, executes and validates default or userdefined
#                  transformation packages via generation and execution of  
#                  sdb loader statements from the relational model of 
#                  sdb database instances.
#
#    prerequisite: install sdb and std python libraries
#
#    ========== licence begin  GPL
#    Copyright (c) 2000-2005 SAP AG
#
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of the GNU General Public License
#    as published by the Free Software Foundation; either version 2
#    of the License, or (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#    ========== licence end

#system
import  os
import  sys
import  stat
import  time
import  socket
import  signal
import  shutil
import  thread
mutex = thread.allocate_lock()
import  getpass
#environment
import  ConfigParser
#string
import  string
import  codecs
import  re
#csv parser
import  csv
#xml parser
from    xml             import sax
#dom
from    xml.dom.minidom import parse
#archive
import  zlib
import  zipfile
import  tarfile
#sdb
import  optlib
from    sdb             import dbm,loader,sql

"""
|-------------------------------------------------------------------------------------------------------------------------------------|
|                                                                                                                                     |
|                                             Loader Transport Parameter                                                              |
|                                                                                                                                     |
|-------------------------------------------------------------------------------------------------------------------------------------|
|                                                                                                                                     |
| ExportID ImportID    Transport Type      LoggingPosition       Package-Part    Medium         Format Export-Import  CharEncoding Lob|
|                                                                                                                                     |
|-------------------------------------------------------------------------------------------------------------------------------------|
|                                                                                                                                     |
|                                            Loader Session Parameter                                                                 |
|                                                                                                                                     |
|-------------------------------------------------------------------------------------------------------------------------------------|
|                                                                                                                                     |
|  Session Attribute       Parameter Value                                 Default Value                                              |
|                                                                                                                                     |
|-------------------------------------------------------------------------------------------------------------------------------------|
|                                                                                                                                     | 
|  SET CONFIGURATION       [EXCLUDE|INCLUDE]                               EXCLUDE                                                    | 
|  SET COMPRESSED          \'/<s>/<d>/\'                                   \'/,/"/\'                                                  |
|  SET SQLMODE             [INTERNAL|ANSI|DB2|ORACLE]                      INTERNAL                                                   | 
|  SET LANGUAGE            [ENG|DEU]                                       \'ENG\'                                                    | 
|  SET CODETYPE            [ASCII|UCS2|UTF8]                               ASCII                                                      |
|  SET CODEPAGE            [http://www.unicode.org/Public/MAPPINGS/]       ISO-8859-1                                                 |
|  SET DATE                [INTERNAL|ISO|USA|EUR|JIS]                      ISO                                                        |
|  SET TIME                [INTERNAL|ISO|USA|EUR|JIS]                      ISO                                                        |
|  SET TIMESTAMP           [INTERNAL|ISO|USA|EUR|JIS]                      ISO                                                        |
|  SET DECIMAL             \'/[<t>]/<d>/\'                                 \'//./\'                                                   |
|  SET BOOLEAN                                                             \'TRUE/FALSE\'                                             |
|  SET NULL                                                                \'?                   \'                                   |
|  SET TRANSACTION SIZE                                                    1000                                                       |
|  SET MAXERRORCOUNT                                                       100                                                        |
|  SET PACKAGETIMEOUT                                                      1000                                                       |
|  SET BACKUPTOOL          [TSM|NETWORKER|BACKINT]                         NETWORKER                                                  | 
|  SET ODBCTRACE           [ON|OFF]                                        OFF                                                        |
|                                                                                                                                     |
|-------------------------------------------------------------------------------------------------------------------------------------|
|                                                                                                                                     |
|                                             Loader Transformation Parameter Values                                                  |
|                                                                                                                                     |
|-------------------------------------------------------------------------------------------------------------------------------------|
|                    |                    |                 |                |               |                       |                |
| What should be     | Transformation of  |  Logging of     | Package of     | Medium of     | Format of             | Mapping of     |
| exported/imported ?| Models und Data    |  Transformation | Transformation | Staging Area  | Models and Data       | Char Encodings |
|                    |                    |                 |                |               |                       |                |
|-------------------------------------------------------------------------------------------------------------------------------------|
|Catalog, Data and   |EXPORT|IMPORT       |[RESTORE]        | DB             |               |                       |                |
|Package of a        |CONFIGURATION       |                 |                | FILE          | DTL, XMI              | UCS2           |
|Database            |CATALOG             |                 |                | PIPE          | DDL, XMI              | UTF8           |
|                    |DATA                |                 |                | TAPE          | PAGES, RECORDS        | ASCII          |
|                    |PACKAGE             |[RESTART]        |                |               | DTL, XMI              | DYNAMIC        |
|-------------------------------------------------------------------------------------------------------------------------------------|
|Catalog, Data and   |EXPORT|IMPORT       |[RESTORE]        | USER           |               |                       |                |
|Package of a        |CONFIGURATION       |                 |                | FILE          | DTL, XMI              | UCS2           |
|Database User       |CATALOG             |                 |                | PIPE          | DDL, XMI              | UTF8           |
|                    |DATA                |                 |                | TAPE          | PAGES, RECORDS        | ASCII          |
|                    |PACKAGE             |[RESTART]        |                |               | DTL, XMI              | DYNAMIC        |
|-------------------------------------------------------------------------------------------------------------------------------------|
|Catalog, Data and   |EXPORT|IMPORT       |[RESTORE]        | TABLE          |               |                       |                |
|Package of a        |CONFIGURATION       |                 |                | FILE          | DTL, XMI              | UCS2           |
|Database Table      |CATALOG             |                 |                | PIPE          | DDL, XMI              | UTF8           |
|                    |DATA                |                 |                | TAPE          | PAGES, RECORDS,       | ASCII          |
|                    |                    |                 |                |               | FORMATTED, COMPRESSED | DYNAMIC        |
|                    |PACKAGE             |[RESTART]        |                |               | DTL, XMI              |                |
|-------------------------------------------------------------------------------------------------------------------------------------|
|Data and Package    |EXPORT|IMPORT       |                 | COLUMNS FIELDS |               |                       |                |
|of a qualified      |                    |                 |                | FILE          |                       | UCS2           |
|Database Columnset  |                    |                 |                | PIPE          |                       | UTF8           |
|                    |DATA                |                 |                | TAPE          | FORMATTED, COMPRESSED | ASCII          |
|                    |PACKAGE             |[RESTART]        |                |               | DTL, XMI              | DYNAMIC        |
|                    |                    |                 |                |               |                       |                |
|-------------------------------------------------------------------------------------------------------------------------------------|
"""
TESTLOADER              = 'FALSE'
#TRANFORMATION LOADER RESOURCES
INDEPPROGPATH           = ''
INDEPDATAPATH           = ''
EXPORTER_VERSION        = ''
IMPORTER_VERSION        = ''
#TRANFORMATION DB RESOURCES
DB_INSTROOT_SOURCE      = ''
DB_INSTROOT_TARGET      = ''
DB_VERSION_SOURCE       = ''
DB_VERSION_TARGET       = ''
DB_INDEPDATAPATH_SOURCE = ''
DB_INDEPDATAPATH_TARGET = ''
DB_DEFAULTCODE_SOURCE   = 'ASCII'
DB_CODE_SOURCE          = 'ASCII'
DB_DEFAULTCODE_TARGET   = 'ASCII'
DB_CODE_TARGET          = 'ASCII'
DBA_SOURCE              = ''
DBA_TARGET              = ''
#TRANSFORMATION CONFIGURATION
LOADERCONFIGURATION     = None
LOADERCONFIGFILE        = None
LOADERPROGPATH          = None
LOADERDATAPATH          = None
#TRANSFORMATION RUNTIME
PACKAGETIMEOUT        = 0
TRANSACTIONSIZE         = 1000
MAXERRORCOUNT           = 150
AUTOCOMMIT              = 'ON'
#TRANSFORMATION PARAMETERS
PACKAGEEXECUTION        = 'EXECUTE'
PACKAGETIMESTAMP        = ''
PACKAGEARCHIVE          = 'TAR'
PACKAGEPATH             = 'HOMEDIR'
PACKAGENAME             = 'TIMESTAMP'
VERBOSE                 = 'ON'
MONITOR                 = 'OFF'
TRANSFORMATIONSYNTAX    = 'UNIFIED' 
SYSTEMATTRIBUTE         = 'OFF'
FIELDCOMPRESSED         = '/,/\"/'
FIELDBOOLVALUE          = 'TRUE/FALSE'
FIELDDECIMALVALUE       = '///'
FIELDNULLVALUE          = '?                   '
FIELDATEVALUE           = 'ISO'
FIELDTIMEVALUE          = 'ISO' 
FIELDTIMESTAMPVALUE     = 'ISO' 
FIELDLOBMASK            = '####'

#DEFINES

#TRANSFORMATION MONITOR
USEDUSERTASKS           = "SELECT COUNT(*)           FROM SESSIONS"
MAXUSERTASKS            = "SELECT VALUE              FROM ACTIVECONFIGURATION  WHERE PARAMETERNAME LIKE 'MAXUSERTASKS'"
DEFAULTCODE             = "SELECT VALUE              FROM ACTIVECONFIGURATION  WHERE PARAMETERNAME LIKE 'DEFAULT_CODE'"
EXISTSDBA               = "SELECT DISTINCT OWNER     FROM DOMAIN.USERS         WHERE USERMODE      LIKE 'SYSDBA'" 
DISTINCTCODE            = "SELECT DISTINCT CODETYPE  FROM DOMAIN.COLUMNS       WHERE OWNER = '%s' AND TABLENAME = '%s' AND CODETYPE <> ''"
PACKAGEMAXSTEP          = "SELECT MAX(STEPID)                         FROM SYSDBA.TRANSFORMATIONMODEL WHERE PACKAGEGUID  = x'%s' AND EXCLUDE = FALSE "
PACKAGEROWCOUNT         = "SELECT STEPID, OWNER, TABLENAME, ROWCOUNT  FROM SYSDBA.TRANSFORMATIONMODEL WHERE PACKAGEGUID  = x'%s' AND EXCLUDE = FALSE AND RESTART = TRUE "
STEPIDROWCOUNT          = "SELECT STEPID, OWNER, TABLENAME, ROWCOUNT  FROM SYSDBA.TRANSFORMATIONMODEL WHERE PACKAGEGUID  = x'%s' AND EXCLUDE = FALSE AND RESTART = TRUE  AND STEPID  = %s "
STEPIDSROWCOUNT         = "SELECT STEPID, OWNER, TABLENAME, ROWCOUNT  FROM SYSDBA.TRANSFORMATIONMODEL WHERE PACKAGEGUID  = x'%s' AND EXCLUDE = FALSE AND RESTART = FALSE AND STEPID >= %s "

#TRANSFORMATION CONSTANTS
DATE                    = 10
DATEINTERNAL            =  8
TIME                    =  8
TIMEINTERNAL            =  8
TIMESTAMP               = 26
TIMESTAMPINTERNAL       = 20
SMALLINT                =  5
INTEGER                 = 10
HEXLENGTH               =  2
SIGN                    =  2
FLOAT                   = 16
FLOATMASK               =  6  
LONGPOSITION            = 21
STEPID                  =  0

def _options ():
     return [
     #Mode
     ('mode','mode'                 , ':', ''                    , 'specifies mode'  ),                          # PARAM,INTER,DEBUG
     ('T', 'transporttype'          , ':', ''                    , 'type          of transport'),                # EXECUTE, GENERATE
     #Transformation Attributes
     ('t', 'transformationtype'     , ':', ''                    , 'type          of transformation '),          # IMPORT , EXPORT, TRANSPORT
     ('p', 'transformationpackage'  , ':', ''                    , 'package       of transformation '),          # DB     , USER,   TABLE
     ('P', 'transformationpart'     , ':', ''                    , 'part          of transformation '),          # DB     , USER,   TABLE
     ('m', 'streammedium'           , ':', ''                    , 'streammedium  of transformation '),          # PIPE   , FILE,   TAPE
     ('f', 'streamformat'           , ':', ''                    , 'streamformat  of transformation '),          # RECORDS, PAGES,  COMPRESSED, FORMATTED, DDL, XML
     ('F', 'streamformat_target'    , ':', ''                    , 'streamformat  of transformation '),          # PAGE   , ROW  ,  DDL, DTL
     ('c', 'streamcharencoding'     , ':', ''                    , 'streamcoding  of transformation '),          # ASCII  , UCS2 ,  UTF8, DYNAMIC
     ('l', 'streamlob'              , ':', ''                    , 'lobmask       of transformation '),          # TABLE  , COLUMN, COLUMNVALUE
     ('s', 'startposition'          , ':', 'START'               , 'startposition of transformation '),          # START  , RESTART
     #Transformation Resources 
     ('n', 'sourcedbnode'           , ':', ''                    , 'specifies source serverdb node'),
     ('d', 'sourcedbname'           , ':', ''                    , 'specifies source serverdb to create'),
     ('u', 'sourcedbuser'           , ':', ''                    , 'specifies source user'),
     ('N', 'targetdbnode'           , ':', ''                    , 'specifies target serverdb node '),
     ('D', 'targetdbname'           , ':', ''                    , 'specifies target serverdb to create'),
     ('U', 'targetdbuser'           , ':', ''                    , 'specifies target user'),
     #Transformation Loader Resources
     ('R', 'loaderinstroot'         , ':', ''                    , 'specifies loader instroot '),                # 
     #Transformation Resources Attributes
     ('X', 'migration'              , ':', 'sdb'                 , 'specifies source database system'),          # SDB, ADABAS
     ('j', 'sourceindepdatapath'    , ':', ''                    , 'specifies source loader independent path'),  # 
     ('J', 'targetindepdatapath'    , ':', ''                    , 'specifies target loader independent path'),  # 
]
"""
     # depreciated or dummy options
     ('x', 'streamconnector'        , ':', 'None'                , 'streamconnector of medium '),                # PIPE2TAPE, PIPE2FILE
     ('T', 'testcase'                , ':', ''                   , 'specifies testcase'),                        # dummy  
     ('i', 'sourceinit'             , ':', ''                    , 'init      sdb  source'),                     # dummy
     ('I', 'targetinit'             , ':', ''                    , 'init      sdb  target'),                     # dummy
     ('v', 'validation'             , ':', ''                    , 'specifies validation scope'),                # dummy
     ('V', 'validationmode'         , ':', ''                    , 'specifies validation type' ),                # dummy
"""

def signal_handler(sig):
    if( sig == signal.SIGINT   or
        sig == signal.SIGTER   or
        sig == signal.SIGABR   or
        sig == signal.SIGBREAK   ):
        try:
           os._exit(0)
        except OSError, err:
           print err

def main (options, args):
    input   = None
    started = False
    stopped = False
    #print "\nLoader transport module:\n'%s'" % os.path.abspath(sys.argv[0])
    get_environment(options)
    try:
       # Transformation Package Design
       design_package  (options) 
       put_environment (options)
       # Transformation Package Transport
       if options.mode == 'PARAM' :
          input = '1'
       if options.mode != 'PARAM' :
          while input not in ['1','2'] :
                print '\nStep %i: Execute Transformation Package:'   % STEPID
                input = raw_input(' [1] OK\n [2] CANCEL\n\nStep %i: ' %STEPID )
       if input in ['1']:       
          package = TransformationPackage(options)
          # Transformation Package Execution
          # Transformation Package Validation
          package.cmd_execute()
          # Transformation Package Release
          package.release()
          # Transformation Package Archive
          package.cmd_archive()
    except EOFError:
       thread.interrupt_main()
    except EnvironmentError:
       thread.interrupt_main()
    except KeyboardInterrupt:
       thread.interrupt_main()
    except:
       sys.exc_traceback
       
class TransformationPackage (sax.ContentHandler):
      # transformationpackage model2xml
      def __init__(self, options = None ):
          self.options                 = options
          self.sourcepackagename       = os.path.split(self.options.sourceindepdatapath)[1]
          self.targetpackagename       = os.path.split(self.options.targetindepdatapath)[1]
          self.starttime               = None
          self.stoptime                = None
          #package   
          self.transformation          = None
          self.packagereader           = None
          self.packagedom              = None
          self.sourceconfiguration     = os.path.join(self.options.sourceindepdatapath,  '%s.CONFIGURATION' % os.path.split(options.sourceindepdatapath)[1])
          self.targetconfiguration     = os.path.join(self.options.targetindepdatapath,  '%s.CONFIGURATION' % os.path.split(options.targetindepdatapath)[1])
          self.transformationpackage   = None
          self.transformationparameter = None
          self.transformation_step     = []
          self.maxerrorcount           = 0
          #db                        
          self.sapdbnode               = None
          self.sapdb                   = None
          #user  
          self.maxstepid               = 0
          self.stepid                  = None                   
          self.user                    = None
          self.pwd                     = None
          self.usercount               = 0
          #table                     
          self.table                   = None
          self.tablecount              = 0
          self.index                   = None
          self.indexStatement          = ''
          self.indexStatements         = []
          self.trigger                 = None
          self.triggerstatement        = None
          #column                    
          self.column                  = ''
          self.columnIsNullable        = ''
          self.columnInitialValue      = ''
          self.columnEncoding          = ''
          self.columnLength            = 0
          self.columncount             = 0
          self.keycolumn               = None
          self.column_long             = []
          #field 
          self.fieldCompressed         = FIELDCOMPRESSED
          self.fieldDecimalValue       = FIELDDECIMALVALUE
          self.fieldNullValue          = FIELDNULLVALUE
          self.fieldBoolValue          = FIELDBOOLVALUE
          self.fieldDateValue          = FIELDATEVALUE
          self.fieldTimeValue          = FIELDTIMEVALUE
          self.fieldTimeStampValue     = FIELDTIMESTAMPVALUE                    
          self.field_startpos          = 1
          self.field_endpos            = 1
          #transformation            
          self.column_field            = ''
          self.column_field_long       = ''
          self.statement               = ''
          if self.options.streamcharencoding == 'DYNAMIC':
             self.options.streamcharencoding  = ''
      
      def release(self):
          self.transformation.release()
      
      # model parser
      def startElement (self, name, attrs):
          #Datasource
          if name == 'dbm:DataSource':
             self.startDataSource()
          #Connection
          if name == 'sdbm:Connection':
             connectString =  map(string.strip, string.split(attrs.get('dbm:ConnectString', None),";"))
             pos           = 1
             for i in connectString:
                 if pos == 1 :
                    pass
                 if pos == 2 :
                    self.sapdbnode  =  ''
                 if pos == 3 :
                    self.sapdb      =  self.options.sourcedbname
                 if pos == 4 :
                    self.user       =  re.split(',',self.options.sourcedbuser)[0]
                 if pos == 5 :
                    self.pwd        = 'initial'
                 pos = pos + 1
          #Catalog
          if name == 'dbm:DeployedCatalog':
             if(self.options.transformationpart == 'DB'):
                self.startCatalog()
          #Schema
          if name == 'sdbm:DeployedSchema' :
             if(self.options.transformationpart == 'USER' or
                self.options.transformationpart == 'TABLE' ):
                self.user             = attrs.get('oim:name',None )
                if self.options.transformationpart != 'DB' and TESTLOADER == 'TRUE':
                   self.options.sourcedbuser  = '%s,%s' % ( self.user, self.pwd )
                   self.options.targetdbuser  = '%s,%s' % ( self.user, self.pwd )
                self.startSchema()
          #Table
          if(self.options.transformationpart == 'TABLE' and (
             name == 'sdbm:DeployedTable' )) :
             self.table            = attrs.get('oim:name', None)
             self.startTable()
          #Index   
          if(self.options.transformationpart == 'TABLE' and (
             name == 'dbm:Index')):
             self.index            = attrs.get('dbm:name', None)
             self.indexStatement   = 'CREATE '
             if attrs.get('dbm:IsUnique',None) == '1':
                self.indexStatement= self.indexStatement + ' UNIQUE '
             self.indexStatement = self.indexStatement   + ' INDEX '   
             self.indexStatement = self.indexStatement   + ' "%s" ON "%s" ' % (self.index, self.table )
             self.indexStatement = self.indexStatement   + ' ( '
          if(self.options.transformationpart == 'TABLE' and (
             name == 'dbm:IndexColumn' )) :
             self.indexStatement   = self.indexStatement + ' "%s" ' % attrs.get('dbm:name', None)
             if attrs.get('dbm:IsAscending',None) == '0':
                self.indexStatement= self.indexStatement + ' DESC,'
             if attrs.get('dbm:IsAscending',None) == '1':
                self.indexStatement= self.indexStatement + ' ASC,'
          if(self.options.transformationpart == 'TABLE' and 
             name == 'dbm:DeployedTrigger'):
             trigger               = attrs.get('oim:id', None)
             self.triggerstatement = attrs.get('dbm:Statements', None )
             string.replace( trigger, '_%s.' % self.user  , '' )
             string.replace( trigger, '.%s'  % self.table , '' )
             self.trigger = trigger
          #Columns
          if(self.options.transformationpart == 'TABLE' and (
             name == 'dbm:DeployedColumn' )): 
             self.column            = attrs.get('dbm:name', None)
             self.columnIsNullable  = attrs.get('dbm:IsNullable'  ,None)
             self.columnInitialValue= attrs.get('dbm:initialValue',None)
             self.columnLength      = attrs.get('dbm:Length',None)
             self.startColumn()
          if self.options.transformationpart == 'TABLE' and name == 'dbm:KeyColumns' :
             self.keycolumn = 'true'
          #Column Types   
          if(self.options.transformationpart == 'TABLE' and (
             name == 'sdbm:BOOLEAN')):
             if self.keycolumn != 'true':
                self.columnTypeBool(name)
          if(self.options.transformationpart == 'TABLE' and (
             name == 'sdbm:CHAR'   or 
             name == 'sdbm:VARCHAR'or
             name == 'sdbm:BINARY' )):
             if self.keycolumn != 'true':
                if(name == 'sdbm:CHAR'    or 
                   name == 'sdbm:VARCHAR'   ):
                   self.columnEncoding = attrs.get('sdbm:Encoding', None) 
                self.columnTypeCharacter(name)
          if(self.options.transformationpart == 'TABLE' and (
             name == 'sdbm:SMALLINT'or
             name == 'sdbm:INTEGER' or
             name == 'sdbm:DECIMAL' or
             name == 'sdbm:FIXED'   or
             name == 'sdbm:FLOAT'   or
             name == 'sdbm:REAL'    or
             name == 'sdbm:DOUBLE_PRECISION' or
             name == 'sdbm:NUMERIC' )):
             if self.keycolumn != 'true':
                self.columnTypeNumber(name)
          if(self.options.transformationpart == 'TABLE' and (
             name == 'sdbm:DATE' or
             name == 'sdbm:TIME' or
             name == 'sdbm:TIMESTAMP' )):
             if self.keycolumn != 'true':
                self.columnTypeDateTime(name)
          if(self.options.transformationpart == 'TABLE' and (                
             name == 'sdbm:LONG' or
             name == 'sdbm:LONG_BYTE'  )):
             if self.keycolumn != 'true':
                self.columnTypeLong(name)

      def endElement(self, name ):
          #DataSource
          if name == 'dbm:DataSource':
             self.endDataSource()
          #Catalog
          if name == 'dbm:DeployedCatalog' :
             self.endCatalog()
          #Schema
          if name == 'sdbm:DeployedSchema':
             self.endSchema()
          #Table
          if(self.options.transformationpart == 'TABLE' and 
             name == 'sdbm:DeployedTable') :
             self.endTable ()
          #Index 
          if(self.options.transformationpart == 'TABLE' and 
             name == 'dbm:Index'):
             self.endIndex()
          #Trigger
          if(self.options.transformationpart == 'TABLE' and 
             name == 'dbm:DeployedTrigger'):
             self.endTrigger()
          #Columns
          if(self.options.transformationpart == 'TABLE' and 
             name == 'dbm:DeployedColumn'):
             self.endColumn()
          if(self.options.transformationpart == 'TABLE' and 
             name == 'dbm:KeyColumns') :
             self.endKeyColumns()
      
      # model export
      def cmd_export(self):
          rc = True
          if self.options.transformationtype == 'TRANSPORT' or self.options.transformationtype == 'EXPORT' :
             if(self.options.transformationpackage  == 'DB'     or
                self.options.transformationpackage  == 'USER'   or
                self.options.transformationpackage  == 'TABLE'     ):
                
                print '\nExport    Transformation Package'
                protocol =  self.transformation.sourceloader.protocol
                self.fieldCompressed     = self.transformation.sourceloader.fieldCompressed     
                self.fieldDecimalValue   = self.transformation.sourceloader.fieldDecimalValue     
                self.fieldNullValue      = self.transformation.sourceloader.fieldNullValue
                self.fieldBoolValue      = self.transformation.sourceloader.fieldBoolValue
                self.fieldDateValue      = self.transformation.sourceloader.fieldDateValue     
                self.fieldTimeValue      = self.transformation.sourceloader.fieldTimeValue     
                self.fieldTimeStampValue = self.transformation.sourceloader.fieldTimeStampValue 
                
                packageuser = 'USE USER %s %s SERVERDB %s ON %s ' % ( re.split(',',self.options.sourcedbuser)[0], re.split(',',self.options.sourcedbuser)[1], self.options.sourcedbname, self.options.sourcedbnode )
                
                self.transformation.sourceloader.cmd  = packageuser
                self.transformation.sourceloader.cmd_execute()
                  
                # Transport Package and Transport Part
                configuration    = "SET CONFIGURATION EXCLUDE " 
                if self.options.transformationpackage == 'DB' :        
                   package       = "EXPORT DB "
                   package       = package + " CONFIGURATION INSTREAM FILE '%s'            %s"    % (os.path.join(self.options.sourceindepdatapath, '%s.CONFIGURATION' % os.path.split(self.options.sourceindepdatapath)[1]) , 'ASCII' ) 
                   package       = package + " CATALOG      OUTSTREAM FILE '%s' XML          \n"  %  os.path.join(self.options.sourceindepdatapath, '%s.xml'           % os.path.split(self.options.sourceindepdatapath)[1])
                   package       = package + " PACKAGE      OUTSTREAM FILE '%s' COMPRESSED %s\n"  % (os.path.join(self.options.sourceindepdatapath, '%s.PACKAGE'       % os.path.split(self.options.sourceindepdatapath)[1]) , 'UTF8' )
                if self.options.transformationpackage == 'USER' :   
                   package       = "EXPORT USER "
                   package       = package + " CONFIGURATION INSTREAM FILE '%s'            %s"    % (os.path.join(self.options.sourceindepdatapath, '%s.CONFIGURATION' % os.path.split(self.options.sourceindepdatapath)[1]) , 'ASCII' )
                   package       = package + " CATALOG      OUTSTREAM FILE '%s' XML           \n" %  os.path.join(self.options.sourceindepdatapath, '%s.xml'           % os.path.split(self.options.sourceindepdatapath)[1])
                   package       = package + " PACKAGE      OUTSTREAM FILE '%s' COMPRESSED % s\n" % (os.path.join(self.options.sourceindepdatapath, '%s.PACKAGE'       % os.path.split(self.options.sourceindepdatapath)[1]) , 'UTF8' )
                if self.options.transformationpackage == 'TABLE' and self.options.transformationpart == 'TABLE':
                   configuration = "SET CONFIGURATION INCLUDE " 
                   package       = "EXPORT USER " 
                   package       = package + " CONFIGURATION INSTREAM FILE '%s'            %s"    % (os.path.join(self.options.sourceindepdatapath, '%s.CONFIGURATION' % os.path.split(self.options.sourceindepdatapath)[1]) , 'ASCII' )
                   package       = package + " CATALOG      OUTSTREAM FILE '%s' XML           \n" %  os.path.join(self.options.sourceindepdatapath, '%s.xml'           % os.path.split(self.options.sourceindepdatapath)[1])
                   package       = package + " PACKAGE      OUTSTREAM FILE '%s' COMPRESSED %s \n" % (os.path.join(self.options.sourceindepdatapath, '%s.PACKAGE'       % os.path.split(self.options.sourceindepdatapath)[1]) , 'UTF8' )
 
                self.transformation.sourceloader.cmd =  configuration
                self.transformation.sourceloader.cmd_execute()
                self.transformation.sourceloader.cmd =  package
                self.transformation.sourceloader.cmd_execute()
                print ''
                if os.path.isfile( os.path.join(self.options.sourceindepdatapath, '%s.CONFIGURATION' % os.path.split(self.options.sourceindepdatapath)[1] )) == False:
                   pass
                if os.path.isfile( os.path.join(self.options.sourceindepdatapath, '%s.xml'           % os.path.split(self.options.sourceindepdatapath)[1] )) == False:
                   print '%s.xml does not exist ' % os.path.split(self.options.sourceindepdatapath)[1]
                   rc = False
                else:
                   self.transformationpackage = os.path.join(self.options.sourceindepdatapath, '%s.xml' % os.path.split(self.options.sourceindepdatapath)[1])                       
             else:
                self.transformationpackage = os.path.join(self.options.sourceindepdatapath, '%s.xml'% os.path.split(self.options.sourceindepdatapath)[1] )
          if self.options.transformationtype == 'IMPORT' :
             if os.path.isfile             ( os.path.join(self.options.sourceindepdatapath, '%s.xml'% os.path.split(self.options.sourceindepdatapath)[1] )) == False:
                print '%s.xml does not exist ' % os.path.split(self.options.sourceindepdatapath)[1]
                rc = False
             else:
                self.transformationpackage = os.path.join(self.options.sourceindepdatapath, '%s.xml' % os.path.split(self.options.sourceindepdatapath)[1])                         
          return rc

      # model import
      def cmd_import (self): 
          rc = True
          if  os.path.isfile(self.transformationpackage) and  os.stat(self.transformationpackage)[stat.ST_SIZE] <= 2000000 :
              print  'Import    Transformation Package\n'
              packagedom = parse( self.transformationpackage )
              schemas    = packagedom.getElementsByTagName('sdbm:DeployedSchema')
              for schema in schemas:
                  print   'Schema    %s' % (schema.getAttribute ('oim:name'))
                  if( self.options.transformationpart == 'USER' or
                      self.options.transformationpart == 'TABLE'  ):
                      tables =  schema.getElementsByTagName('sdbm:DeployedTable')
                      for table in tables :
                          tablename = table.getAttribute('oim:name')
                          print 'Table     %s' % tablename
                          if self.options.transformationtype == 'TRANSPORT' or self.options.transformationtype == 'IMPORT':
                             varchars = table.getElementsByTagName('sdbm:VARCHAR')
                             for varchar in varchars : 
                                 if rc == True:
                                    columnEncoding = varchar.getAttribute('sdbm:Encoding')
                                    if columnEncoding== 'UNICODE' and  DB_CODE_TARGET == 'ASCII':
                                       print 'UNICODE Character Columns in Table %s cannot be imported in ASCII Database' % tablename
                                       rc = False
                             chars = table.getElementsByTagName('sdbm:CHAR')
                             for char in chars : 
                                 if rc == True:
                                    columnEncoding = char.getAttribute('sdbm:Encoding')
                                    if columnEncoding== 'UNICODE' and  DB_CODE_TARGET == 'ASCII':
                                       print 'UNICODE Character Columns in Table %s cannot be imported in ASCII Database' % tablename
                                       rc = False
              packagedom.unlink()
          return rc  
                 
      # model execution
      def cmd_execute  (self):
          global   STEPID
          global   DB_DEFAULTCODE_SOURCE
          global   DB_DEFAULTCODE_TARGET
          try:
             input   = None
             self.transformation = Transformation( self.options, self )
             # Transformation Package Init
             if self.cmd_export() == True and self.cmd_import() == True:
                self.starttime     = time.time()
                self.transformation.cmd_stamp_package() 
                # Transformation Package Execution
                self.options.mode  = 'PARAM' 
                packageparser = sax.make_parser()
                packageparser.setContentHandler( self ) 
                packageparser.parse (self.transformationpackage )
                # Transformation Package Validation
                if self.options.transporttype == 'EXECUTE':
                   if  self.options.transformationtype  == 'TRANSPORT':
                       transformationpackage_dir  = os.path.split(self.options.sourceindepdatapath)[0]
                       transformationpackage_name = os.path.split(self.options.sourceindepdatapath)[1]
                   if  self.options.transformationtype  == 'EXPORT':
                       transformationpackage_dir  = os.path.split(self.options.sourceindepdatapath)[0]
                       transformationpackage_name = os.path.split(self.options.sourceindepdatapath)[1]
                   if  self.options.transformationtype  == 'IMPORT':
                       transformationpackage_dir  = os.path.split(self.options.targetindepdatapath)[0]
                       transformationpackage_name = os.path.split(self.options.targetindepdatapath)[1]
                   #PACKAGE CONTENT
                   shutil.copy(os.path.join(LOADERPROGPATH,'sdb','loader','config','loader.cfg'), os.path.join(transformationpackage_dir,transformationpackage_name,transformationpackage_name + os.extsep + 'cfg'))
                   transformation_source_connect = ''
                   transformation_target_connect = ''
                   if self.options.streamcharencoding == '':
                      streamcharencoding = 'DYNAMIC'
                   if self.options.streamcharencoding != '':
                      streamcharencoding = self.options.streamcharencoding
                   if self.options.streamcharencoding == '':
                      self.options.streamcharencoding  = 'DYNAMIC'
                   if self.options.transformationtype == 'IMPORT':
                      self.options.transformationpackage =  os.path.split(self.options.sourceindepdatapath)[1]
                   if self.options.transformationtype ==  'TRANSPORT' or self.options.transformationtype == 'EXPORT' :
                      transformation_source_connect = " -n %s -d %s -u %s " % ( self.options.sourcedbnode, self.options.sourcedbname, re.split(',',self.options.sourcedbuser)[0]+',*')
                   if self.options.transformationtype ==  'TRANSPORT' or self.options.transformationtype == 'IMPORT':
                      transformation_target_connect = " -N %s -D %s -U %s " % ( self.options.targetdbnode, self.options.targetdbname, re.split(',',self.options.targetdbuser)[0]+',*')   
                   transformationparameter          = " -T %s -t %s -p %s -P %s -s %s -m %s -f %s -F %s -c %s -l %s "  % ( self.options.transporttype, self.options.transformationtype, self.options.transformationpackage, self.options.transformationpart, self.options.startposition, self.options.streammedium,self.options.streamformat,self.options.streamformat_target,streamcharencoding, self.options.streamlob)
                   transformationparameter       = transformationparameter + transformation_source_connect + transformation_target_connect
                   transformationparameters      = file(os.path.join(transformationpackage_dir,transformationpackage_name,transformationpackage_name+os.extsep+'par'),'w+')
                   transformationparameters.write( transformationparameter )                   
                   transformationparameters.close()
                   #PACKAGE CONTENT MANAGEMENT
                   transformation_source_connect    = "| %-15s | %-15s | %-15s " % ( '', '', '' )
                   transformation_target_connect    = "| %-15s | %-15s | %-15s |"% ( '', '', '')
                   if os.path.isfile(os.path.join(LOADERDATAPATH,'sdb','loader','data','transport'+os.extsep+'log')) == False:
                      transformation_id             = "| %-48s | %-16s | %-16s | %-15s | %-15s "   % ( 'PACKAGEGUID'   , 'EXPORTER_VERSION' , 'IMPORTER_VERSION', 'SOURCE_USERKEY', 'TARGET_USERKEY' )
                      transformation_source_connect = "| %-15s | %-15s | %-15s " %( 'SOURCE_NODE -n', 'SOURCE_DB -d'  , 'SOURCE_USER -u')
                      transformation_target_connect = "| %-15s | %-15s | %-15s "% ( 'TARGET_NODE -N', 'TARGET_DB -D'  , 'TARGET_USER -U')
                      source_target_versions        = "| %-15s | %-15s |"       % ( 'SOURCE_VERSION', 'TARGET_VERSION')
                      transformationparameter_1     = "| %-20s | %-15s | %-10s | %-10s | %-15s | %-10s | %-10s | %-10s | %-10s   %-10s | %-10s | %-11s  | %-10s "  % ( 'PACKAGETIMESTAMP', 'PACKAGENAME' , 'TYPE -T','TYPE -t' , 'PACKAGE -p', 'PART -P', 'START -s', 'MEDIUM -m','FORMAT -f','-F ','CODING -c', 'LOB -l','ARCHIVE')
                      transformationparameter_1     = transformationparameter_1 + transformation_id +  transformation_source_connect + transformation_target_connect + source_target_versions + '\n'
                      transformation_id             = "| %-48s | %-16s | %-16s | %-15s | %-15s " % ( '14', '15', '16', '17', '18')
                      transformation_source_connect = "| %-15s | %-15s | %-15s " % ( '19', '20', '21')
                      transformation_target_connect = "| %-15s | %-15s | %-15s " % ( '22', '23', '24' )
                      source_target_versions        = "| %-15s | %-15s |"        % ( '', '')
                      transformationparameter_2     = "| %-20s | %-15s | %-10s | %-10s | %-15s | %-10s | %-10s | %-10s | %-10s | %-10s | %-10s | %-11s  | %-10s "  % ( '1' , '2' , '3', '4', '5', '6','7','8','9','10','11','12','13')
                      transformationparameter_2     = transformationparameter_2 + transformation_id + transformation_source_connect + transformation_target_connect + source_target_versions + '\n'
                      transformationparameter       = transformationparameter_1 + transformationparameter_2 
                      transformationparameters      = file(os.path.join(LOADERDATAPATH,'sdb','loader','data','transport'+os.extsep+'log'),'a+')
                      transformationparameters.write( transformationparameter )
                      transformationparameters.close()
                   transformationpackage_guid = ''
                   if self.options.transformationtype == 'TRANSPORT' or self.options.transformationtype == 'EXPORT':
                      transformationpackage_guid = self.transformation.sourceloader.packageguid 
                   if self.options.transformationtype == 'IMPORT' :
                      transformationpackage_guid = self.transformation.targetloader.packageguid 
                   transformation_id                = "| %-48s | %-16s | %-16s | %-15s | %-15s " % ( transformationpackage_guid, EXPORTER_VERSION, IMPORTER_VERSION, '', '' )
                   if self.options.transformationtype   == 'TRANSPORT' or self.options.transformationtype == 'EXPORT' or self.options.transformationtype == 'IMPORT' :
                      transformation_source_connect     = "| %-15s | %-15s | %-15s " % ( self.options.sourcedbnode, self.options.sourcedbname, re.split(',',self.options.sourcedbuser)[0]+',*')
                      if self.options.transformationtype == 'EXPORT':
                          transformation_target_connect = "| %-15s | %-15s | %-15s "% ('','','')
                   if self.options.transformationtype    == 'TRANSPORT' or self.options.transformationtype == 'IMPORT' :                      
                      transformation_target_connect = "| %-15s | %-15s | %-15s " % ( self.options.targetdbnode, self.options.targetdbname, re.split(',',self.options.targetdbuser)[0]+',*')
                   source_target_versions           = "| %-15s | %-15s |"        % ( DB_VERSION_SOURCE, DB_VERSION_TARGET)
                   transformationparameter          = "| %-20s | %-15s | %-10s | %-10s | %-15s | %-10s | %-10s | %-10s | %-10s | %-10s | %-10s | %-11s  | %-10s "  % ( PACKAGETIMESTAMP, transformationpackage_name, self.options.transporttype, self.options.transformationtype, self.options.transformationpackage, self.options.transformationpart, self.options.startposition, self.options.streammedium,self.options.streamformat,self.options.streamformat_target,self.options.streamcharencoding, self.options.streamlob, PACKAGEARCHIVE)
                   transformationparameter          = transformationparameter + transformation_id + transformation_source_connect + transformation_target_connect + source_target_versions + '\n'
                   transformationparameters         = file(os.path.join(LOADERDATAPATH,'sdb','loader','data','transport'+os.extsep+'log'),'a+')
                   transformationparameters.write( transformationparameter )                   
                   transformationparameters.close()
                   self.stoptime  = time.time()   
                   print 'Transformation Package Time   : %.3fs ' % float( self.stoptime - self.starttime )
                   print 'Transformation Package Errors : %i ' % self.maxerrorcount
          except IOError , (errno,err):
            print err         
          except:
            kind, val, traceback = sys.exc_info()
            raise kind, val, traceback
      
      # model package validation
      def cmd_verify(self):
          self.transformation.cmd_verify_package()
      
      # model package statistic
      def cmd_statistic(self):
          print 'Transformation Package Time   : %.3fs ' % float( self.stoptime - self.starttime )
          print 'Transformation Package Errors : %i '    % self.maxerrorcount
      
      # model package archive    
      def cmd_archive(self):
          if PACKAGEARCHIVE == 'ZIP':
             if( self.options.transformationtype   == 'TRANSPORT' or
                 self.options.transformationtype   == 'EXPORT'   ):
                 try:
                     zipdirname  = os.path.split(self.options.sourceindepdatapath)[0]
                     zipfilename = os.path.split(self.options.sourceindepdatapath)[1]+os.extsep+'zip'
                     zip =  zipfile.ZipFile(os.path.join (zipdirname,zipfilename), 'w', zipfile.ZIP_DEFLATED )  
                     print 'Transformation Package Archive: %s ' % zipfilename
                     os.chdir(self.options.sourceindepdatapath)
                     for filename in os.listdir(os.getcwd()):
                         zip.write(filename)
                     zip.close()
                     #VALIDATE PACKAGE
                     zip =  zipfile.ZipFile(os.path.join (zipdirname,zipfilename), 'r', zipfile.ZIP_DEFLATED )  
                     if zip.testzip() == None:
                        zip.close()
                        os.chdir(self.options.sourceindepdatapath)
                        for filename in os.listdir(os.getcwd()):
                            os.unlink(filename)
                        os.chdir (os.path.split(self.options.sourceindepdatapath)[0])
                        time.sleep(1)
                        os.removedirs(self.options.sourceindepdatapath)
                     else:
                        print 'Not ok'
                 except RuntimeError, err:
                     print err
                 except OSError, (errno, err ):
                     if errno != 13:
                        print err
                 except IOError, (errno, err ):
                     print err
                 except:
                     kind, val, traceback = sys.exc_info()
                     raise kind, val, traceback
             if( self.options.transformationtype  == 'TRANSPORT' or
                 self.options.transformationtype  == 'IMPORT'  ):
                 try:
                     zipdirname  = os.path.split(self.options.targetindepdatapath)[0]
                     zipfilename = os.path.split(self.options.targetindepdatapath)[1]+os.extsep+'zip'
                     zip =  zipfile.ZipFile(os.path.join (zipdirname,zipfilename), 'w', zipfile.ZIP_DEFLATED )  
                     if  self.options.transformationtype  == 'IMPORT':
                         print 'Transformation Package Archive: %s ' % zipfilename
                     os.chdir(self.options.targetindepdatapath)
                     for filename in os.listdir(os.getcwd()):
                         zip.write(filename)
                     zip.close()
                     #VALIDATE PACKAGE
                     zip =  zipfile.ZipFile(os.path.join (zipdirname,zipfilename), 'r', zipfile.ZIP_DEFLATED )  
                     if zip.testzip() == None:
                        zip.close()
                        os.chdir(self.options.targetindepdatapath)
                        for filename in os.listdir(os.getcwd()):
                            os.unlink(filename)
                        os.chdir (os.path.split(self.options.targetindepdatapath)[0])
                        time.sleep(1)
                        os.removedirs(self.options.targetindepdatapath)
                     else:
                        print 'Not ok'
                 except RuntimeError, err:
                     print err
                 except OSError, (errno, err ):
                     if errno != 13:
                        print err
                 except IOError, (errno, err ):
                     print err
                 except:
                     kind, val, traceback = sys.exc_info()
                     raise kind, val, traceback
          if PACKAGEARCHIVE == 'TAR':
             if( self.options.transformationtype   == 'TRANSPORT' or
                 self.options.transformationtype   == 'EXPORT' ):
                 try:
                     zipdirname  = os.path.split(self.options.sourceindepdatapath)[0]
                     zipfilename = os.path.split(self.options.sourceindepdatapath)[1]+os.extsep+'tgz'
                     zip =  tarfile.TarFile.open(os.path.join (zipdirname,zipfilename), 'w:gz' )  
                     print 'Transformation Package Archive: %s ' % zipfilename
                     os.chdir(self.options.sourceindepdatapath)
                     for filename in os.listdir(os.getcwd()):
                         zip.add(filename)
                     zip.close()
                     #VERIFY PACKAGE
                     if tarfile.is_tarfile(os.path.join (zipdirname,zipfilename)) == True:
                        os.chdir(self.options.sourceindepdatapath)
                        for filename in os.listdir(os.getcwd()):
                            os.unlink(filename)
                        os.chdir (os.path.split(self.options.sourceindepdatapath)[0])
                        time.sleep(1)
                        os.removedirs(self.options.sourceindepdatapath)
                     else:
                        print 'Not ok'
                 except RuntimeError, err:
                     print err
                 except OSError, (errno, err ):
                     if errno != 13:
                        print err
                 except IOError, (errno, err ):
                     print err
                 except:
                     kind, val, traceback = sys.exc_info()
                     raise kind, val, traceback      
             if( self.options.transformationtype  == 'TRANSPORT' or
                 self.options.transformationtype  == 'IMPORT'  ):
                 try:
                     zipdirname  = os.path.split(self.options.targetindepdatapath)[0]
                     zipfilename = os.path.split(self.options.targetindepdatapath)[1]+os.extsep+'tgz'
                     zip =  tarfile.TarFile.open(os.path.join (zipdirname,zipfilename), 'w:gz' ) 
                     if  self.options.transformationtype  == 'IMPORT':
                         print 'Transformation Package Archive: %s ' % zipfilename 
                     os.chdir(self.options.targetindepdatapath)
                     for filename in os.listdir(os.getcwd()):
                         zip.add(filename)
                     zip.close()
                     #VERIFY PACKAGE
                     if tarfile.is_tarfile(os.path.join (zipdirname,zipfilename)) == True:
                        zip.close()
                        os.chdir(self.options.targetindepdatapath)
                        for filename in os.listdir(os.getcwd()):
                            os.unlink(filename)
                        os.chdir (os.path.split(self.options.targetindepdatapath)[0])
                        time.sleep(1)
                        os.removedirs(self.options.targetindepdatapath)
                     else:
                        print 'Not ok'
                 except RuntimeError, err:
                     print err
                 except OSError, (errno, err ):
                     if errno != 13:
                        print err
                 except IOError, (errno, err ):
                     print err
                 except:
                     kind, val, traceback = sys.exc_info()
                     raise kind, val, traceback
                
      def cmd_dearchive(self):
          if PACKAGEARCHIVE == 'ZIP':
             if((self.options.transformationtype   == 'TRANSPORT' or
                 self.options.transformationtype   == 'IMPORT')   and
                 self.options.streammedium         == 'FILE'):         
                 try:
                     zipdirname  = os.path.split(self.options.sourceindepdatapath)[0]
                     zipfilename = os.path.split(self.options.sourceindepdatapath)[1]+os.extsep+'zip'
                     zip =  zipfile.ZipFile(os.path.join (zipdirname,zipfilename), 'r', zipfile.ZIP_DEFLATED )
                     if zip.testzip() == None:
                        print 'Transformation Package Archive: %s Ok' % zipfilename
                        os.chdir(self.options.sourceindepdatapath)
                        for filename in zip.namelist():
                            unzip        = file (filename,'wb')
                            unzip.write(zip.read(filename))
                            unzip.close()
                        zip.close()
                        os.chdir(os.path.split(self.options.sourceindepdatapath)[0])
                     else:
                        print 'Not ok'
                 except IOError, err:
                     print err
                 except:
                     kind, val, traceback = sys.exc_info()
                     raise kind, val, traceback

      # model element callback functions      
      def len (self, name):
          length = 0
          if self.options.streamcharencoding == 'ASCII' :
             length = len(name)
          if self.options.streamcharencoding == 'UTF8' :
             length = len(name)
          if self.options.streamcharencoding == 'UCS2' :
             length = 2*len(name)
          return length   
          
      def int (self, name): 
          length = 0
          if self.options.streamcharencoding == 'ASCII' :
             length = int(name,10)
          if self.options.streamcharencoding == 'UTF8' :
             length = int(name,10)   
          if self.options.streamcharencoding == 'UCS2' :
             length = 2*int(name,10)
          return length
          
      def transformation_statement (self ):
          return self.transformation.cmd_statement()    
          
      def startDataSource(self):
          pass    
          
      def startCatalog(self):
          if( self.options.transformationtype == 'TRANSPORT' or
              self.options.transformationtype == 'EXPORT'      ):
              sourcedbuser = 'USE USER %s %s SERVERDB %s ON %s \n' % ( re.split(',',self.options.sourcedbuser)[0], re.split(',',self.options.sourcedbuser)[1], self.options.sourcedbname, self.options.sourcedbnode )
              try:
                  self.transformation.sourceloader.cmd  = sourcedbuser
                  self.transformation.sourceloader.cmd_execute ()
                  self.transformation.sourceloader.user = re.split(',',self.options.sourcedbuser)[0]
              except loader.LoaderError, err:
                  print 'command failed:' , err

          if( self.options.transformationtype == 'TRANSPORT' or
              self.options.transformationtype == 'IMPORT'      ):
              targetdbuser = 'USE USER %s %s SERVERDB %s ON %s \n' % ( re.split(',',self.options.targetdbuser)[0],re.split(',',self.options.targetdbuser)[1], self.options.targetdbname, self.options.targetdbnode )
              try :
                  self.transformation.targetloader.cmd  = targetdbuser
                  self.transformation.targetloader.cmd_execute ()
                  self.transformation.targetloader.user = re.split(',',self.options.targetdbuser)[0]
              except loader.LoaderError, err:
                  print 'command failed:' , err  
              
      def startSchema(self):
          #Source
          if( self.options.transformationtype == 'TRANSPORT' or
              self.options.transformationtype == 'EXPORT'      ):
              self.transformation.sourceloader.cmd = ''
              if( self.options.transformationpart == 'USER'   or
                  self.options.transformationpart == 'TABLE'):
                  #Source
                  cmd = 'USE USER %s %s SERVERDB %s ON %s \n' % ( re.split(',',self.options.sourcedbuser)[0], re.split(',',self.options.sourcedbuser)[1] , self.options.sourcedbname, self.options.sourcedbnode )
                  try :
                      self.transformation.sourceloader.cmd  = cmd
                      self.transformation.sourceloader.cmd_execute ()
                      self.transformation.sourceloader.user = re.split(',',self.options.sourcedbuser)[0]
                  except loader.LoaderError, err:
                      print 'command failed:' , err
          #Target
          if( self.options.transformationtype == 'TRANSPORT' or
              self.options.transformationtype == 'IMPORT'      ):
              self.transformation.targetloader.cmd = ''
              if( self.options.transformationpart == 'USER'   or
                  self.options.transformationpart == 'TABLE'):
                  cmd = 'USE USER %s %s SERVERDB %s ON %s \n' % ( re.split(',',self.options.targetdbuser)[0], re.split(',',self.options.targetdbuser)[1], self.options.targetdbname, self.options.targetdbnode )
                  try :
                      self.transformation.targetloader.cmd  = cmd
                      self.transformation.targetloader.cmd_execute ()
                      self.transformation.targetloader.user = re.split(',',self.options.targetdbuser)[0]
                  except loader.LoaderError, err:
                      print 'command failed:' , err
          time.sleep(1)          
      
      def startTable(self):
          self.statement         = ''
          self.indexStatement    = ''
          self.indexStatements   = []
          self.tablecount        = self.tablecount +1
          self.columncount       = 0
          self.statement_column  = ''
          self.field_startpos    = 1
          self.field_endpos      = 1
          self.column            = ''
          self.column_field      = ''
          self.column_field_long = ''
          self.column_long       = []
          if self.transformation.sourceloader != None:
             self.transformation.sourceloader.cmd = ''
          if self.transformation.targetloader != None:
             self.transformation.targetloader.cmd = ''
          
      def startColumn(self): 
          self.columncount       = self.columncount + 1   
          self.columnEncoding    = ''            
          
      def endDataSource(self):
          if self.options.transformationpart  == 'DB':
             mutex.acquire()                                           #acquire
             self.cmd_verify()                                         #release
          self.transformation.release()
           
      def endCatalog(self):
          if self.options.transformationpart  == 'DB'   :
             mutex.acquire()                                           #acquire
             self.transformation_statement()
             input  = None
             #Format
             if(self.options.streamformat        == 'XML'     or
                self.options.streamformat        == 'DDL'     or
                self.options.streamformat        == 'PAGES'   or
                self.options.streamformat        == 'RECORDS'  ):
                #Type
                if self.options.transformationtype   == 'TRANSPORT':
                    while input not in ["y","n"]:
                          if self.options.mode == 'INTER':
                             input = raw_input('Transport DB %s ? [y]/[n] : ' % ( self.sapdb ) )
                          if self.options.mode == 'PARAM':
                             print             'Transport DB %s\n' % ( self.sapdb )
                             input  = "y"
                          if input == "y":
                             self.transformation.cmd_transport_catalog_package()
                          if input == "n":
                             if mutex.locked() == True:
                                mutex.release()
                if(self.options.transformationtype   == 'EXPORT'  and
                    self.options.streammedium        == 'FILE'     ):
                    while input not in ["y","n"]:
                          if self.options.mode == 'INTER':
                             input = raw_input('Export    DB %s ? [y]/[n] : ' % ( self.sapdb ))
                          if self.options.mode == 'PARAM':
                             print             'Export    DB %s\n' % ( self.sapdb )
                             input = "y"
                          if input == "y":
                             self.transformation.cmd_export_catalog_package()
                          if input == "n":
                             if mutex.locked() == True:
                                mutex.release()
                if(self.options.transformationtype   == 'IMPORT' and
                    self.options.streammedium        == 'FILE'    ):
                    while input not in ["y","n"]:
                          if self.options.mode == 'INTER':
                             input = raw_input('Import    DB %s ? [y]/[n] : ' % (self.sapdb ))
                          if self.options.mode == 'PARAM':
                             print             'Import    DB %s\n' % (self.sapdb )
                             input = "y"
                          if input == "y":
                             self.transformation.cmd_import_catalog_package()
                          if input == "n":
                             if mutex.locked() == True:
                                mutex.release()                        #release
             self.user = '' 
             
      def endSchema(self):
          if(self.options.transformationpart  == 'USER'    and
            self.user                         <> 'DBA'       ) :
            mutex.acquire()                                           #acquire
            self.transformation_statement()
            input  = None
            #Format
            if( self.options.streamformat         == 'XML'     or
                self.options.streamformat        == 'DDL'     or
                self.options.streamformat        == 'PAGES'   or
                self.options.streamformat        == 'RECORDS'  ):
                #Type
                if self.options.transformationtype    == 'TRANSPORT':
                    while input not in ["y","n"]:
                          if self.options.mode == 'INTER':
                             input = raw_input('Transport Schema %s ? [y]/[n] : ' % ( self.user ) )
                          if self.options.mode == 'PARAM':
                             print             '\nTransport Schema %s' % ( self.user )
                             input = "y"
                          if input == "y":
                             self.transformation.cmd_transport_schema_package()
                          if input == "n":
                             if mutex.locked() == True:
                                mutex.release()
                if( self.options.transformationtype    == 'EXPORT'  and
                    self.options.streammedium          == 'FILE'     ):
                    while input not in ["y","n"]:
                          if self.options.mode == 'INTER':
                             input = raw_input('Export    Schema %s ? [y]/[n] :  ' % ( self.user ))
                          if self.options.mode == 'PARAM':
                             print             '\nExport    Schema %s' % ( self.user )
                             input = "y"
                          if input == "y":
                             self.transformation.cmd_export_schema_package()
                          if input == "n":
                             if mutex.locked() == True:
                                mutex.release()
                if( self.options.transformationtype     == 'IMPORT' and
                    self.options.streammedium           == 'FILE'    ):
                    while input not in ["y","n"]:
                          if self.options.mode == 'INTER':
                             input = raw_input('Import    Schema %s ? [y]/[n] :  ' % (self.user ))
                          if self.options.mode == 'PARAM':
                             print             '\nImport    Schema %s' % (self.user )
                             input = "y"
                          if input == "y":
                             self.transformation.cmd_import_schema_package()
                          if input == "n":
                             if mutex.locked() == True :               #release
                                mutex.release()
            self.user = '' 
            
      def endTable(self):
          if self.options.transformationpart   == 'TABLE' :
             mutex.acquire()                                          #acquire
             if self.options.streamformat         != 'SQL':
                self.transformation_statement()
             input  = None
             #Format
             if(self.options.streamformat         == 'XML'       or
                self.options.streamformat         == 'DDL'       or
                self.options.streamformat         == 'PAGES'     or
                self.options.streamformat         == 'RECORDS'   or
                self.options.streamformat[0:9]    == 'FORMATTED' or
                self.options.streamformat         == 'COMPRESSED'or
                self.options.streamformat         == 'SQL'          ):
                #Type
                if self.options.transformationtype    == 'TRANSPORT':
                   while input not in ["y","n"]:
                         if self.options.mode == 'INTER':
                            input = raw_input('Transport Table %s ? [y]/[n] : ' % ( self.table ) )
                         if self.options.mode == 'PARAM':
                            print             'Transport Table %s\n' % ( self.table )
                            input  = "y"
                         if input == "y":
                            self.transformation.cmd_transport_table_package()
                         if input == "n":
                            if mutex.locked() == True:
                               mutex.release()
                if(self.options.transformationtype    == 'EXPORT'  and
                    self.options.streammedium         == 'FILE'     ):
                    while input not in ["y","n"]:
                          if self.options.mode == 'INTER':
                             input = raw_input('Export    Table %s ? [y]/[n] : ' % ( self.table ))
                          if self.options.mode == 'PARAM':
                             print             'Export    Table %s :\n' % ( self.table )
                             input  = "y"
                          if input == "y":
                             self.transformation.cmd_export_table_package()
                          if input == "n":
                             if mutex.locked() == True:
                                mutex.release()
                if(self.options.transformationtype     == 'IMPORT' and
                    self.options.streammedium          == 'FILE'    ):
                    while input not in ["y","n"]:
                          if self.options.mode == 'INTER':
                             input = raw_input('Import    Table %s ? [y]/[n] : ' % (self.table ))
                          if self.options.mode == 'PARAM':
                             print             'Import    Table %s\n' % (self.table )
                             input  = "y"
                          if input == "y":
                             self.transformation.cmd_import_table_package()
                          if input == "n":
                             if mutex.locked() == True:
                                mutex.release()                       #release
                              
      def endIndex(self):
          self.indexStatement = self.indexStatement   + ') '
          self.indexStatement = string.replace(self.indexStatement,',)',')\n')
          self.indexStatements.append(self.indexStatement)
          if(self.options.transformationpart   == 'TABLE' and self.options.streamformat != 'PAGES' and ( self.options.streamformat_target == 'PAGE' or self.options.streamformat_target == 'DTL' ) and  self.column_field_long == '' and self.transformation.targetloader != None) :
             self.transformation.targetloader.cmd  = 'DROP INDEX   \"%s\" ON \"%s\".\"%s\" \n' % (self.index, self.user, self.table)
             self.transformation.targetloader.cmd_execute()
             self.transformation.targetloader.cmd  =  ''  
             
      def endTrigger(self):     
          if self.options.transformationpart   == 'TABLE' and self.options.streamformat_target == 'PAGE' and self.transformation.targetloader != None:
             self.transformation.targetloader.cmd  = 'DROP TRIGGER \"%s\" OF \"%s\".\"%s\" \n' % (self.trigger, self.user, self.table)
             self.transformation.targetloader.cmd_execute()
             self.transformation.targetloader.cmd  =  ''          

      def endColumn(self):
          column_field = ''
          if self.options.streamformat[0:9]   == 'FORMATTED' :
             if self.keycolumn == None :
                column_field      = "%+16s%-15s  %3i - %3i CHAR " % (' ','\"%s\"' % self.column,self.field_startpos, self.field_endpos)
                self.column_field = self.column_field  + column_field 
                if self.columnIsNullable == '0':
                   self.column_field = self.column_field + "\n"
                if self.columnIsNullable != '0':
                   self.column_field = self.column_field + "DEFAULT NULL \n"
             self.field_startpos = self.field_endpos + 1
          if self.options.streamformat        == 'COMPRESSED':   
             if self.keycolumn == None :
                column_field    = "%+16s%-15s  %3i  CHAR "      % (' ','\"%s\"' % self.column, self.columncount )
                self.column_field = self.column_field  + column_field 
                if self.columnIsNullable == '0':
                   self.column_field = self.column_field + "\n"
                if self.columnIsNullable != '0':
                   self.column_field = self.column_field + "DEFAULT NULL \n"

      def endKeyColumns(self):
          self.keycolumn = None                   
          
      def columnTypeBool (self, name):
          if name == 'sdbm:BOOLEAN':
             fieldBoolLength = '1'
             if self.columnIsNullable == '0':
                self.field_endpos      = self.field_startpos + self.int('%s' %self.columnInitialValue) -1
             if self.columnIsNullable == '1':   
                if len(self.fieldNullValue) >  len('%s' %self.columnInitialValue) :
                   self.field_endpos   = self.field_startpos + len(self.fieldNullValue) -1
                if len(self.fieldNullValue) <= len('%s' %self.columnInitialValue) :
                   self.field_endpos   = self.field_startpos + self.int('%s' %self.columnInitialValue) -1
      
      def columnTypeCharacter(self, name):
          if(name == 'sdbm:CHAR'    or 
             name == 'sdbm:VARCHAR'   ):
             if self.columnIsNullable == '0': 
                self.field_endpos      = self.field_startpos + self.int('%s' %self.columnLength) -1
             if self.columnIsNullable == '1':   
                if len(self.fieldNullValue)  > self.int('%s' % self.columnLength):
                   self.field_endpos   = self.field_startpos + len(self.fieldNullValue)-1
                if len(self.fieldNullValue) <= self.int('%s' %self.columnLength):   
                   self.field_endpos   = self.field_startpos + self.int('%s' %self.columnLength) -1
          if name == 'sdbm:BINARY':  
             self.columnEncoding = 'HEX'
             if self.columnIsNullable == '0': 
                self.field_endpos      = self.field_startpos + HEXLENGTH*self.int('%s' %self.columnLength) -1
             if self.columnIsNullable == '1':   
                if len(self.fieldNullValue)  > self.int('%s' % self.columnLength):
                   self.field_endpos   = self.field_startpos + HEXLENGTH*len(self.fieldNullValue)-1
                if len(self.fieldNullValue) <= self.int('%s' %self.columnLength):   
                   self.field_endpos   = self.field_startpos + HEXLENGTH*self.int('%s' %self.columnLength) -1       
      
      def columnTypeDateTime(self, name ):
          if name == 'sdbm:DATE': 
             fieldDateLength    = 0
             if self.fieldDateValue == 'INTERNAL':
                fieldDateLength = DATEINTERNAL
             if self.fieldDateValue != 'INTERNAL':
                fieldDateLength = DATE
             if self.columnIsNullable == '0': 
                self.field_endpos      = self.field_startpos + self.int( '%i' % fieldDateLength) -1 
             if self.columnIsNullable == '1':   
                if len(self.fieldNullValue) >  self.int( '%i' % fieldDateLength): 
                   self.field_endpos   = self.field_startpos + len(self.fieldNullValue) -1
                if len(self.fieldNullValue) <= self.int( '%i' % fieldDateLength):
                   self.field_endpos   = self.field_startpos + self.int( '%i' % fieldDateLength) -1
          if name == 'sdbm:TIME': 
             fieldTimeLength    = 0
             if self.fieldTimeValue == 'INTERNAL':
                fieldTimeLength = TIMEINTERNAL
             if self.fieldTimeValue != 'INTERNAL':
                fieldTimeLength = TIME
             if self.columnIsNullable == '0': 
                self.field_endpos      = self.field_startpos + self.int( '%i' % fieldTimeLength) -1
             if self.columnIsNullable == '1':   
                if len(self.fieldNullValue) >  len(self.fieldTimeValue) : 
                   self.field_endpos   = self.field_startpos + len(self.fieldNullValue) -1
                if len(self.fieldNullValue) <= len(self.fieldTimeValue) :
                   self.field_endpos   = self.field_startpos + self.int(fieldTimeLength) -1
          if name == 'sdbm:TIMESTAMP':
             fieldTimeStampLength    = 0
             if self.fieldTimeStampValue == 'INTERNAL':
                fieldTimeStampLength = TIMESTAMPINTERNAL
             if self.fieldTimeStampValue != 'INTERNAL':
                fieldTimeStampLength = TIMESTAMP
             if self.columnIsNullable == '0': 
                self.field_endpos      = self.field_startpos + self.int( '%i' % fieldTimeStampLength) -1
             if self.columnIsNullable == '1':   
                if len(self.fieldNullValue) >  self.int( '%i' % fieldTimeStampLength) :
                   self.field_endpos   = self.field_startpos + len(self.fieldNullValue) -1 
                if len(self.fieldNullValue) <= self.int( '%i' % fieldTimeStampLength) :
                   self.field_endpos   = self.field_startpos + self.int( '%i' % fieldTimeStampLength) -1
      
      def columnTypeNumber(self, name ):    
          fieldIntegerLength = 0
          signSymbolLength  = SIGN
          ThsdCharNum       = 0
          FracPart          = 0
          IntPart           = 0
          if name == 'sdbm:SMALLINT':
             fieldIntegerLength = SMALLINT
          if(name == 'sdbm:INTEGER' or
             name == 'sdbm:DECIMAL'):
             fieldIntegerLength = INTEGER
          if name == 'sdbm:FIXED'  :
             fieldIntegerLength = int(string.split(self.columnLength,',')[0])
             FracPart           = int(string.split(self.columnLength,',')[1])
          if name == 'sdbm:FLOAT' :
             fieldIntegerLength = int(self.columnLength)
             field_length       = fieldIntegerLength + FLOATMASK
             if self.columnIsNullable == '0':
                self.field_endpos      = self.field_startpos + self.int('%s' % field_length) -1
             if self.columnIsNullable == '1': 
                if len(self.fieldNullValue) >  self.int('%s' % field_length): 
                   self.field_endpos   = self.field_startpos + len(self.fieldNullValue) - 1
                if len(self.fieldNullValue) <= self.int('%s' % field_length):
                   self.field_endpos   = self.field_startpos + self.int('%s' % field_length) -1
          if name != 'sdbm:FLOAT':
             IntPart          = fieldIntegerLength - FracPart
             field_length     = 0  
             if string.split(self.fieldDecimalValue,'/')[1] <> '':
                ThsdCharNum = ( IntPart - 1) / 3
             field_length = IntPart + FracPart+ signSymbolLength + ThsdCharNum
             if   FracPart == 0:
                  field_length = field_length - 1
             elif IntPart  == 0: 
                  field_length = field_length - ( ThsdCharNum - 1 )
             if self.columnIsNullable == '0':
                self.field_endpos      = self.field_startpos + self.int('%s' % field_length) -1
             if self.columnIsNullable == '1': 
                if len(self.fieldNullValue) >  self.int('%s' % field_length): 
                   self.field_endpos   = self.field_startpos + len(self.fieldNullValue) -1
                if len(self.fieldNullValue) <= self.int('%s' % field_length):
                   self.field_endpos   = self.field_startpos + self.int('%s' % field_length) -1
      def columnTypeLong (self, name ):#
          if name == 'sdbm:LONG'      and self.options.streamformat   != 'RECORDS' :
             if self.column != ''     and self.options.streammedium   == 'FILE'    :
                self.column_field_long = self.column_field_long + "%+15s LONGSTREAM FILE \"%s\" '%s' %s \n" % ( ' ', self.column, os.path.join(self.options.sourceindepdatapath, '%s_%s.DATA%s' % (self.table,self.column,FIELDLOBMASK)),self.options.streamcharencoding)
                if  self.options.streamlob == 'COLUMN':
                    self.field_endpos   = self.field_startpos + LONGPOSITION -1
                if  self.options.streamlob == 'COLUMNVALUE':
                    self.field_endpos   = self.field_startpos + self.len(os.path.join(self.options.sourceindepdatapath,  '%s_%s.DATA%s' % (self.table,self.column, FIELDLOBMASK))) -1 
             if self.column != ''       and ( self.options.streammedium == 'PIPE' or  self.options.streammedium  == 'PIPE2FILE' ) :
                self.column_field_long = self.column_field_long + "%+15s LONGSTREAM FILE %s '%s' %s \n" % (' ', self.column, '\\\\.\\PIPE\\%s_%s.DATA' % (self.table,self.column),self.options.streamcharencoding)
                self.column_long.append(self.column)
          if name == 'sdbm:LONG_BYTE' and self.options.streamformat   != 'RECORDS' :
             if self.column != ''     and self.options.streammedium   == 'FILE'    :
                self.column_field_long = self.column_field_long + "%+15s LONGSTREAM FILE \"%s\" '%s' \n" % ( ' ',self.column, os.path.join(self.options.sourceindepdatapath,   '%s_%s.DATA%s' % (self.table,self.column, FIELDLOBMASK)))
                if  self.options.streamlob == 'COLUMN':
                    self.field_endpos  = self.field_startpos + LONGPOSITION -1 
                if  self.options.streamlob == 'COLUMNVALUE':
                    self.field_endpos  = self.field_startpos + self.len(os.path.join(self.options.sourceindepdatapath,   '%s_%s.DATA%s' % (self.table,self.column,FIELDLOBMASK))) -1
             if self.column != ''       and ( self.options.streammedium == 'PIPE' or  self.options.streammedium  == 'PIPE2FILE' ) :
                self.column_field_long = self.column_field_long + "%+15s LONGSTREAM FILE \"%s\" '%s' \n" % ( ' ',self.column, '\\\\.\\PIPE\\%s_%s.DATA' % (self.table,self.column))
                self.column_long.append(self.column)
                   
class Transformation :
      def __init__( self, options = None, package = None ) :
          self.options                = options
          self.package                = package
          self.transformation         = None
          self.exportstatement        = None
          self.importstatement        = None
          self.sourceloader           = None
          self.targetloader           = None
          # stream loader source
          if( options.transformationtype == 'TRANSPORT' or
              options.transformationtype == 'EXPORT'      ):
              self.sourceloader = StreamLoader( self.options, self.package, 'EXPORT' )
              try:
                 if options.mode == 'DEBUG':
                    input('\nStarted   SourceLoader')
              except:
                 pass                                              
          # stream loader target
          if( options.transformationtype == 'TRANSPORT' or
              options.transformationtype == 'IMPORT'      ):
              self.targetloader = StreamLoader( self.options, self.package, 'IMPORT' )
              try:
                 if options.mode == 'DEBUG':
                    input('\nStarted   TargetLoader')
              except:
                 pass  
      
      def release(self):
          if mutex.locked() == True:
             mutex.release()
          if self.sourceloader != None:
             self.sourceloader.release()
          if self.targetloader != None:
             self.targetloader.release()
      
      def cmd_release_execute_package(self):
          # trigger package and release streamloader
          release        = 'FALSE'
          while release == 'TRUE':
                try:
                    cmd  = "SELECT COUNT(*) FROM SYSDBA.TRANSFORMATIONMODEL WHERE RESTART = FALSE "
                    self.sourcesession.cmd = cmd
                    self.sourcesession.cmd_execute()
                    if self.sourcesession.result[0] == 1 :
                       print '\nStop   Package Execution'
                       self.sourceloader.loadersession.release()  # python->cancelbyte
                       self.targetloader.loadersession.release()  # python->cancelbyte
                       release = 'TRUE'
                except sql.SQLError,  err :
                    print 'sql command failed:', err
                except loader.LoaderError, err :
                    print 'loader command failed:', err
                except loader.CommunicationError, err :
                    print 'loader communication failed:', err
                    
      def cmd_statement(self):
          #transformationstreamencoding
          if self.options.streamcharencoding == 'DYNAMIC':
             self.options.streamcharencoding  = ''
          #transformationstreams
          datastream_format = self.options.streamformat
          datastream_field  = ''
          if(datastream_format  == 'COMPRESSED' or datastream_format  == 'FORMATTED' ) and self.options.streamformat_target == 'PAGE' and self.package.column_field_long <> '' :
             datastream_format   = 'RECORDS'
          if(datastream_format  == 'PAGES'   or
             datastream_format  == 'RECORDS' or
             datastream_format  == 'DDL'     or
             datastream_format  == 'XML'   ):
             configurationstream_name_mask = "'%s.CONFIGURATION'" + "%-0s"
             catalogstream_name_mask       = "'%s.CATALOG'"       + "%-6s"
             catalogstream_encoding_mask   = "%-6s %s"
             datastream_name_mask          = "'%s.DATA'"          + "%-9s"
             datastream_encoding_mask      = "%-2s %s"
             packagestream_name_mask       = "'%s.PACKAGE'"       + "%-6s"
             packagestream_format_mask     = "%-0s" + "COMPRESSED"
             packagestream_encoding_mask   = "%-0s %s"
          if(datastream_format  == 'COMPRESSED' or
             datastream_format  == 'FORMATTED'    ) :
             configurationstream_name_mask = "'%s.CONFIGURATION'" + "%-0s"
             catalogstream_name_mask       = "'%s.CATALOG'"       + "%-1s"
             catalogstream_encoding_mask   = "%-6s %s"
             datastream_name_mask          = "'%s.DATA'"          + "%-4s"
             packagestream_name_mask       = "'%s.PACKAGE'"       + "%-1s"
             datastream_encoding_mask      = "%-0s %s"
             packagestream_format_mask     = "%-0s" + "DTL"
             packagestream_encoding_mask   = "%-0s %s"
          
          #transformationstream_names
          export_configurationstream_name    = os.path.join(self.options.sourceindepdatapath)
          import_configurationstream_name    = os.path.join(self.options.targetindepdatapath)
          export_packagestream_name          = os.path.join(self.options.sourceindepdatapath)
          import_packagestream_name          = os.path.join(self.options.targetindepdatapath)
          if self.options.streammedium      =='FILE':
             datastream_name                 = os.path.join(self.options.sourceindepdatapath)
             catalogstream_name              = os.path.join(self.options.sourceindepdatapath)
          if self.options.streammedium      =='PIPE' or self.options.streammedium== 'TAPE':  
             datastream_name                 = os.path.join('\\\\','.','PIPE')
             catalogstream_name              = os.path.join('\\\\','.','PIPE')
          if self.options.transformationpart == 'DB'   :
             export_configurationstream_name = os.path.join(export_configurationstream_name,self.options.sourcedbname)
             import_configurationstream_name = os.path.join(import_configurationstream_name,self.options.targetdbname)
             if self.options.streammedium   == 'PIPE':
                export_catalogstream_name    = os.path.join(catalogstream_name       ,self.options.sourcedbname)
                import_catalogstream_name    = os.path.join(catalogstream_name       ,self.options.sourcedbname)
             if self.options.streammedium   != 'PIPE': 
                export_catalogstream_name    = os.path.join(catalogstream_name       ,self.package.sapdb)
                import_catalogstream_name    = os.path.join(catalogstream_name       ,self.package.sapdb)        
             datastream_name                 = os.path.join(datastream_name          ,self.package.sapdb)
             export_packagestream_name       = os.path.join(export_packagestream_name,self.options.sourcedbname)
             import_packagestream_name       = os.path.join(import_packagestream_name,self.options.targetdbname)
          if self.options.transformationpart == 'USER' :
             export_configurationstream_name = os.path.join(export_configurationstream_name,self.package.user)
             import_configurationstream_name = os.path.join(import_configurationstream_name,self.package.user)
             if self.options.streammedium   == 'PIPE':
                export_catalogstream_name    = os.path.join(catalogstream_name       ,self.package.user)
                import_catalogstream_name    = os.path.join(catalogstream_name       ,self.package.user)
             if self.options.streammedium   != 'PIPE': 
                export_catalogstream_name    = os.path.join(catalogstream_name       ,self.package.user)
                import_catalogstream_name    = os.path.join(catalogstream_name       ,self.package.user)     
             datastream_name                 = os.path.join(datastream_name          ,self.package.user)
             export_packagestream_name       = os.path.join(export_packagestream_name,self.package.user)
             import_packagestream_name       = os.path.join(import_packagestream_name,self.package.user)
          if self.options.transformationpart == 'TABLE':
             export_configurationstream_name = os.path.join(export_configurationstream_name,self.package.table)
             import_configurationstream_name = os.path.join(import_configurationstream_name,self.package.table)
             if self.options.streammedium   == 'PIPE':
                export_catalogstream_name    = os.path.join(catalogstream_name       ,self.package.table)
                import_catalogstream_name    = os.path.join(catalogstream_name       ,self.package.table)
             if self.options.streammedium   != 'PIPE': 
                export_catalogstream_name    = os.path.join(catalogstream_name       ,self.package.table)
                import_catalogstream_name    = os.path.join(catalogstream_name       ,self.package.table)  
             datastream_name                 = os.path.join(datastream_name          ,self.package.table)
             export_packagestream_name       = os.path.join(export_packagestream_name,self.package.table)
             import_packagestream_name       = os.path.join(import_packagestream_name,self.package.table)

          #configurationstream
          export_configurationstream_name = configurationstream_name_mask %( export_configurationstream_name,'' )
          import_configurationstream_name = configurationstream_name_mask %( import_configurationstream_name,'' )
          configurationstream_medium      = 'FILE'
          configurationstream_format      = 'COMPRESSED'
          configurationstream_encoding    = ''
          
          #catalogstream
          export_catalogstream_name      = catalogstream_name_mask %( export_catalogstream_name, '' )
          import_catalogstream_name      = catalogstream_name_mask %( import_catalogstream_name, '' )
          catalogstream_medium           = self.options.streammedium     
          if catalogstream_medium       == 'TAPE' and catalogstream_direction    == 'INSTREAM':
             catalogstream_medium        = catalogstream_medium + ' EBID ' + self.targetloader.ddlpipe_ebid
          if self.options.streamformat  == 'DDL'  or  self.options.streamformat  == 'XML':
             catalogstream_format        = self.options.streamformat
          else:
             catalogstream_format        = 'DDL'
          catalogstream_encoding         = catalogstream_encoding_mask %( '', '' )
          
          #datastream
          datastream_name                = datastream_name_mask % (datastream_name, '')
          datastream_medium              = self.options.streammedium
          if datastream_medium          == 'TAPE' and datastream_direction    == 'INSTREAM':
             datastream_medium           = datastream_medium + 'EBID' + self.targetloader.mappipe_ebid
          datastream_format              = self.options.streamformat
          if(datastream_format          == 'COMPRESSED' or datastream_format  == 'FORMATTED' ) and self.package.column_field_long <> '' and ( self.options.streammedium == 'PIPE' or self.options.streamformat_target == 'PAGE' ):
             datastream_format           = 'RECORDS'
          if datastream_format          == 'PAGES':
             datastream_encoding         = ''
          if datastream_format          == 'RECORDS' and self.options.streamcharencoding == 'UTF8':
             datastream_encoding         = ''
          if datastream_format          == 'RECORDS' and self.options.streamcharencoding != 'UTF8':
             datastream_encoding         = datastream_encoding_mask  % ('', self.options.streamcharencoding )   
          if datastream_format          == 'COMPRESSED' or datastream_format == 'FORMATTED' or datastream_format  == 'DDL' or datastream_format  == 'XML' :   
             datastream_encoding         = datastream_encoding_mask  % ('', self.options.streamcharencoding )
          if datastream_format          == 'COMPRESSED' or datastream_format == 'FORMATTED' :
             if datastream_format       == 'COMPRESSED' :
                datastream_field         = datastream_field + "%-15s SEPARATOR  '%s' \n" % ( '', re.split('/',self.package.fieldCompressed)[1] )
                datastream_field         = datastream_field + "%-15s DELIMITER  '%s' \n" % ( '', re.split('/',self.package.fieldCompressed)[2] )
             datastream_field            = datastream_field + "%-15s NULL       '%s' \n" % ( '', self.package.fieldNullValue  )
             datastream_field            = datastream_field + "%-15s DECIMAL    '%s' \n" % ( '', self.package.fieldDecimalValue) 
             datastream_field            = datastream_field + "%-15s BOOLEAN    '%s' \n" % ( '', self.package.fieldBoolValue) 
             datastream_field            = datastream_field + "%-15s TIMESTAMP   %s  \n" % ( '', self.package.fieldTimeStampValue)
             datastream_field            = datastream_field + "%-15s DATE        %s  \n" % ( '', self.package.fieldDateValue)
             datastream_field            = datastream_field + "%-15s TIME        %s  \n" % ( '', self.package.fieldTimeValue)     
          
          #packagestream
          export_packagestream_name      = packagestream_name_mask %( export_packagestream_name, '') 
          import_packagestream_name      = packagestream_name_mask %( import_packagestream_name, '') 
          packagestream_medium           = 'FILE' 
          packagestream_format           = packagestream_format_mask   %('') 
          packagestream_encoding         = packagestream_encoding_mask %('', '' ) 
          
          #transformationstreams
          export_configurationstream     = "  CONFIGURATION %s %s %s %s %s \n"   % ( 'INSTREAM ', configurationstream_medium, export_configurationstream_name, configurationstream_format, configurationstream_encoding)        
          export_catalogstream           = "  CATALOG       %s %s %s %s %s \n"   % ( 'OUTSTREAM', catalogstream_medium      , export_catalogstream_name      , catalogstream_format      , catalogstream_encoding)
          export_datastream              = "  DATA          %s %s %s %s %s \n"   % ( 'OUTSTREAM', datastream_medium         , datastream_name                , datastream_format         , datastream_encoding )
          import_configurationstream     = "  CONFIGURATION %s %s %s %s %s \n"   % ( 'INSTREAM ', configurationstream_medium, import_configurationstream_name, configurationstream_format, configurationstream_encoding)        
          import_catalogstream           = "  CATALOG       %s %s %s %s %s \n"   % ( 'INSTREAM ', catalogstream_medium      , import_catalogstream_name      , catalogstream_format      , catalogstream_encoding)
          import_datastream              = "  DATA          %s %s %s %s %s \n"   % ( 'INSTREAM ', datastream_medium         , datastream_name                , datastream_format         , datastream_encoding )
          
          export_datastream              = export_datastream + datastream_field
          import_datastream              = import_datastream + datastream_field

          if datastream_format          == 'COMPRESSED' or datastream_format  == 'FORMATTED':
             export_datastream           = export_datastream + '%-15s START  %-4s 1   MAXROWS    \n'  %('','')
             import_datastream           = import_datastream + '%-15s START  %-4s 1   MAXRECORDS \n'  %('','') 
             export_datastream           = export_datastream + ''
             import_datastream           = import_datastream + self.package.column_field
             if self.options.streamlob == 'COLUMN':
                export_datastream        = export_datastream + self.package.column_field_long
                import_datastream        = import_datastream + self.package.column_field_long
             if self.options.streamlob == 'COLUMNVALUE': 
                export_datastream        = export_datastream + self.package.column_field_long  
                import_datastream        = import_datastream + ''
          if datastream_format          == 'RECORDS' and self.options.transformationpart == 'TABLE' : 
             export_datastream           = export_datastream + '%-15s START  %-4s 1   MAXROWS    \n' %('','')
             import_datastream           = import_datastream + '%-15s START  %-4s 1   MAXRECORDS \n' %('','') 
          export_packagestream           = "  PACKAGE       %s %s %s %s %s  \n"   % ('OUTSTREAM', packagestream_medium      , export_packagestream_name      , packagestream_format      , packagestream_encoding )      
          import_packagestream           = "  PACKAGE       %s %s %s %s %s  \n"   % ('OUTSTREAM', packagestream_medium      , import_packagestream_name      , packagestream_format      , packagestream_encoding )      
          
          if((self.options.transformationpart == 'DB'    or  
              self.options.transformationpart == 'USER') and
              self.options.streamformat_target== 'DTL' ):
              export_datastream = ''
              import_datastream = ''
          if(self.options.transformationpart  == 'TABLE'   and
             self.options.streamformat        != 'PAGES'   and
             self.options.streamformat        != 'RECORDS' and 
             self.options.streamformat_target == 'ROW'   ):
             import_datastream           = import_datastream    + '%-15s ROW USAGE \n' %(' ')
             export_packagestream        = export_packagestream + '%-15s ROW USAGE '   %(' ')
             import_packagestream        = import_packagestream + '%-15s ROW USAGE '   %(' ')
          if(self.options.transformationpart  == 'TABLE'   and
             self.options.streamformat        != 'PAGES'   and   
             self.options.streamformat        != 'RECORDS' and  
             self.options.streamformat_target == 'PAGE'    and 
             self.package.column_field_long   == ''     ):
             import_datastream           = import_datastream    + '%-15s PAGE WITH 25 ROWS USAGE \n' %(' ')
             export_packagestream        = export_packagestream + '%-15s PAGE WITH 25 ROWS USAGE \n' %(' ')
             import_packagestream        = import_packagestream + '%-15s PAGE WITH 25 ROWS USAGE \n' %(' ')
          if(self.options.transformationpart  == 'TABLE' and  
             self.options.streamformat_target == 'DTL'   and
            (datastream_format                == 'COMPRESSED' or 
             datastream_format                == 'FORMATTED') ):
             if self.package.column_field_long != '' :
                import_datastream           = import_datastream    + '%-15s ROW USAGE \n' %(' ')
                export_packagestream        = export_packagestream + '%-15s ROW USAGE '   %(' ')
                import_packagestream        = import_packagestream + '%-15s ROW USAGE '   %(' ')
             if self.package.column_field_long == '' :
                import_datastream           = import_datastream    + '%-15s PAGE WITH 25 ROWS USAGE \n' %(' ')
                export_packagestream        = export_packagestream + '%-15s PAGE WITH 25 ROWS USAGE \n' %(' ')
                import_packagestream        = import_packagestream + '%-15s PAGE WITH 25 ROWS USAGE \n' %(' ')
          if(self.options.transformationpart  == 'TABLE' ):
             packagestream_format           = 'COMPRESSED'
             import_packagestream           = "  PACKAGE       %s %s %s %s %s  \n"   % ('OUTSTREAM', packagestream_medium      , import_packagestream_name      , packagestream_format      , packagestream_encoding )      
             
          #transformation
          
          #transformationtype export 
          export_stmt  = 'EXPORT %s ' % self.options.transformationpart
          #transformationpackage
          if(self.options.transformationpart  == 'DB' or 
             self.options.transformationpart  == 'USER'):
             export_stmt = export_stmt  + '\n'
          if(self.options.transformationpart  == 'TABLE'):
             export_stmt = export_stmt + '"%s" ' % self.package.table
             if( self.options.streamformat_target== 'DTL' ):
                 export_stmt = export_stmt + " FOR IMPORT TABLE "
             export_stmt = export_stmt + '\n'    
          #configurationstream   
          if(datastream_format  <> 'DDL' and
            (self.options.transformationpart  == 'DB' or 
             self.options.transformationpart  == 'USER'  )):    
             export_stmt = export_stmt + export_configurationstream 
          #catalogstream 
          export_stmt    = export_stmt + export_catalogstream
          #datastream 
          if(datastream_format == 'PAGES'      or   
             datastream_format == 'RECORDS'    or 
             datastream_format == 'COMPRESSED' or 
             datastream_format == 'FORMATTED'  ):
             export_stmt = export_stmt + export_datastream  
          #packagestream
          if datastream_format     <> 'DDL' :   
             export_stmt = export_stmt + export_packagestream 
             
          #transformationtype import
          import_stmt  = 'IMPORT %s ' % self.options.transformationpart
          #transformationpackage
          if(self.options.transformationpart  == 'DB' or 
             self.options.transformationpart  == 'USER'):
             import_stmt = import_stmt  + '\n'
          if(self.options.transformationpart  == 'TABLE'):
             import_stmt = import_stmt + '"%s" \n' % self.package.table
          #configurationstream   
          if(datastream_format  <> 'DDL' and
            (self.options.transformationpart  == 'DB' or 
             self.options.transformationpart  == 'USER'  )):    
             import_stmt = import_stmt + import_configurationstream 
          #catalogstream 
          import_stmt    = import_stmt + import_catalogstream
          #datastream 
          if(datastream_format == 'PAGES'      or   
             datastream_format == 'RECORDS'    or 
             datastream_format == 'COMPRESSED' or 
             datastream_format == 'FORMATTED'  ):
             import_stmt = import_stmt + import_datastream  
          #packagestream
          if  datastream_format    <> 'DDL':
              import_stmt = import_stmt + import_packagestream    
              
          if self.options.transformationtype == 'EXPORT':
             self.sourceloader.catalogstream_name = export_catalogstream_name
             self.sourceloader.datastream_name    = datastream_name
             self.sourceloader.packagestream_name = export_packagestream_name
             self.sourceloader.cmd = export_stmt
          if self.options.transformationtype == 'IMPORT':
             self.targetloader.catalogstream_name = import_catalogstream_name
             self.targetloader.datastream_name    = datastream_name
             self.targetloader.packagestream_name = import_packagestream_name
             self.targetloader.cmd = import_stmt
          if self.options.transformationtype == 'TRANSPORT': 
             self.sourceloader.catalogstream_name = export_catalogstream_name
             self.sourceloader.datastream_name    = datastream_name
             self.sourceloader.packagestream_name = export_packagestream_name
             self.sourceloader.cmd = export_stmt
             self.targetloader.catalogstream_name = import_catalogstream_name
             self.targetloader.datastream_name    = datastream_name
             self.targetloader.packagestream_name = import_packagestream_name
             self.targetloader.cmd = import_stmt
             
          self.exportstatement  = export_stmt
          self.importstatement  = import_stmt
          
          if VERBOSE == 'ON':
             self.cmd_monitor_statement()
          
      def cmd_monitor_statement(self): 
          export_stmt = self.exportstatement 
          import_stmt = self.importstatement
          export_stmt = string.replace(export_stmt,'\\\\.\\PIPE\\','')
          export_stmt = string.replace(export_stmt,self.options.sourceindepdatapath+os.sep, '')
          import_stmt = string.replace(import_stmt,'\\\\.\\PIPE\\','')
          import_stmt = string.replace(import_stmt,self.options.sourceindepdatapath+os.sep, '')
          import_stmt = string.replace(import_stmt,self.options.targetindepdatapath+os.sep, '')
                    
          if self.options.transformationtype == 'TRANSPORT':
             print '\n%s;'  % export_stmt
             print '\n%s;'  % import_stmt
          if self.options.transformationtype == 'EXPORT':
             print '\n%s;'  % export_stmt
          if self.options.transformationtype == 'IMPORT':   
             print '\n%s;'  % import_stmt 
             
      def cmd_stamp_package(self):
          if self.sourceloader != None:
             self.sourceloader.cmd = 'REMARK "TRANSFORMATIONPACKAGE  %s" \n' % self.package.sourcepackagename
             self.sourceloader.cmd_execute()
          if self.targetloader != None:
             self.targetloader.cmd = 'REMARK "TRANSFORMATIONPACKAGE  %s" \n' % self.package.targetpackagename
             self.targetloader.cmd_execute()
             
      def cmd_transport_catalog_package(self):
          input = None
          # stream connector and stream filter
          self.streamconnector =  StreamConnector(self.sourceloader, self.targetloader )
          self.streamfilter    =  StreamFilter   (self.options , self )

          #stream package
          if self.options.startposition == 'RESTART':
             try:
                 if options.mode == 'INTER':
                    input('Start init Transformationpackages')
                 if options.mode == 'PARAM':
                    print 'Start init Transformationpackages'
                    i = 1/0
             except:
                 restartRecord = self.targetloader.cmd_restart_targetdb_package()
                 self.sourceloader.cmd_restart_sourcedb_package ( restartRecord )

          #stream export
          while input not in ["y"]:
                if self.options.mode == 'INTER':
                   input = raw_input('Export    DB %s ? [y]     : ' % ( self.package.sapdb ))
                if self.options.mode == 'PARAM':
                   input   = "y"
                if input  == "y":
                   #stream loader
                   thread.start_new_thread( self.sourceloader.cmd_use_serverdb_user      ,())
                   time.sleep(1)
                   thread.start_new_thread( self.sourceloader.cmd_export_catalog_package ,()) #export
                   time.sleep(1)
                   #stream package
                   if self.options.streammedium == 'FILE' and self.options.streamformat <> 'XML':
                      while self.sourceloader.packageload  == 'FALSE' :
                            thread.start_new_thread( self.sourceloader.cmd_load_package   ,())    
                            time.sleep(1) 
                      while self.sourceloader.packageunload == 'FALSE' :
                            thread.start_new_thread( self.sourceloader.cmd_monitor_package,())
                            thread.start_new_thread( self.sourceloader.cmd_unload_package ,())#release
                            time.sleep(1)
                      mutex.acquire()                                                         #acquire      
                   if self.options.streammedium == 'TAPE' :
                      while self.sourceloader.packageunload == 'FALSE' :
                            thread.start_new_thread( self.sourceloader.cmd_unload_package,()) #release
                            time.sleep(1)
                      mutex.acquire()                                                         #acquire
                      self.targetloader.ddlpipe_ebid = self.sourceloader.ddlpipe_ebid
                      self.targetloader.datpipe_ebid = self.sourceloader.datpipe_ebid
                      self.targetloader.mappipe_ebid = self.sourceloader.mappipe_ebid

          input = None

          #stream import
          while input not in ["y"] :
                if self.options.mode == 'INTER':
                    input = raw_input('Import    DB %s ? [y]     : ' % ( self.package.sapdb ))
                if self.options.mode == 'PARAM':
                    input  = "y"
                if input  == "y":
                    #stream loader
                    thread.start_new_thread( self.targetloader.cmd_use_serverdb_user      ,())
                    time.sleep(1)
                    thread.start_new_thread( self.targetloader.cmd_import_catalog_package ,())  #import
                    time.sleep(1)
                    #stream package
                    while self.targetloader.packageload  == 'FALSE' :
                          thread.start_new_thread( self.targetloader.cmd_load_package   ,())    
                          time.sleep(1) 
                    while self.targetloader.packageunload == 'FALSE' :
                          thread.start_new_thread( self.targetloader.cmd_monitor_package,())
                          thread.start_new_thread( self.targetloader.cmd_unload_package ,())   #release
                          time.sleep(1)
          self.sourceloader.packageload   = 'FALSE'
          self.sourceloader.packageunload = 'FALSE'
          self.targetloader.packageload   = 'FALSE'
          self.targetloader.packageunload = 'FALSE'
			    
      def cmd_export_catalog_package(self):
          if MONITOR == 'OFF':
             #stream packages
             self.sourceloader.cmd_export_catalog_package()
             if  mutex.locked()  == True :
                 mutex.release()           #release
          if MONITOR == 'ON':
             # stream loader
             thread.start_new_thread( self.sourceloader.cmd_export_catalog_package,())    #export
             # stream package
             while self.sourceloader.packageload  == 'FALSE' :
                   thread.start_new_thread( self.sourceloader.cmd_load_package   ,())    
                   time.sleep(1) 
             while self.sourceloader.packageunload == 'FALSE' :
                   thread.start_new_thread( self.sourceloader.cmd_monitor_package,())
                   thread.start_new_thread( self.sourceloader.cmd_unload_package ,())    #release
                   time.sleep(1)
             self.sourceloader.packageload    = 'FALSE'
             self.sourceloader.packageunload  = 'FALSE'

      def cmd_import_catalog_package(self):
          if MONITOR == 'OFF':
             #stream packages
             self.targetloader.cmd_import_catalog_package()
             if  mutex.locked()  == True :
                 mutex.release()           #release
          if MONITOR == 'ON':
             # stream loader
             thread.start_new_thread( self.targetloader.cmd_import_catalog_package,())    #export
             # stream package
             while self.targetloader.packageload  == 'FALSE' :
                   thread.start_new_thread( self.targetloader.cmd_load_package   ,())    
                   time.sleep(1) 
             while self.targetloader.packageunload == 'FALSE' :
                   thread.start_new_thread( self.targetloader.cmd_monitor_package,())
                   thread.start_new_thread( self.targetloader.cmd_unload_package ,())    #release
                   time.sleep(1)
             self.targetloader.packageload    = 'FALSE'
             self.targetloader.packageunload  = 'FALSE'

      def cmd_transport_schema_package(self):
          input = None
          # stream connector and stream filter
          self.streamconnector =  StreamConnector(self.sourceloader, self.targetloader )
          self.streamfilter    =  StreamFilter   (self.options , self )
          #stream packages
          if self.options.startposition == 'RESTART':
             try:
                 if options.mode == 'INTER':
                    input('Start init Transformationpackages')
             except:
                 restartRecord = self.targetloader.cmd_restart_targetdb_package()
                 self.sourceloader.cmd_restart_sourcedb_package( restartRecord )
          #stream export
          while input not in ["y"]:
                if self.options.mode == 'INTER':
                   input = raw_input('Export    Schema %s ? [y] : ' % ( self.package.user ))
                if self.options.mode == 'PARAM':
                   input   = "y"
                if input  == "y":
                   #stream loader
                   thread.start_new_thread( self.sourceloader.cmd_export_schema_package  ,()) #export
                   #stream package
                   if self.options.streammedium == 'FILE' and self.options.streamformat <> 'XML':
                      while self.sourceloader.packageload  == 'FALSE' :
                            thread.start_new_thread( self.sourceloader.cmd_load_package   ,())    
                            time.sleep(1) 
                      while self.sourceloader.packageunload == 'FALSE' :
                            thread.start_new_thread( self.sourceloader.cmd_monitor_package,())
                            thread.start_new_thread( self.sourceloader.cmd_unload_package ,())#release
                            time.sleep(1)
                      mutex.acquire()                                                         #acquire      
                   if self.options.streammedium == 'TAPE' :
                      mutex.acquire()
                      self.targetloader.ddlpipe_ebid = self.sourceloader.ddlpipe_ebid
                      self.targetloader.datpipe_ebid = self.sourceloader.datpipe_ebid
                      self.targetloader.mappipe_ebid = self.sourceloader.mappipe_ebid

          input = None
          # stream import
          while input not in ["y"]:
                if self.options.mode == 'INTER':
                   input = raw_input('Import    Schema %s ? [y] : ' % ( self.package.user ))
                if self.options.mode == 'PARAM':
                   input   = "y"
                if input  == "y":
                    #stream loader
                    thread.start_new_thread( self.targetloader.cmd_import_schema_package ,())  #import
                    time.sleep(1)
                    #stream package
                    while self.targetloader.packageload  == 'FALSE' :
                          thread.start_new_thread( self.targetloader.cmd_load_package   ,())    
                          time.sleep(1) 
                    while self.targetloader.packageunload == 'FALSE' :
                          thread.start_new_thread( self.targetloader.cmd_monitor_package,())
                          thread.start_new_thread( self.targetloader.cmd_unload_package ,())   #release
                          time.sleep(1)
          self.sourceloader.packageload   = 'FALSE'
          self.sourceloader.packageunload = 'FALSE'
          self.targetloader.packageload   = 'FALSE'
          self.targetloader.packageunload = 'FALSE'

      def cmd_export_schema_package(self):
          if MONITOR == 'OFF':
             #stream package
             self.sourceloader.cmd_export_schema_package()
             if  mutex.locked()  == True :
                 mutex.release()           #release
          if MONITOR == 'ON':
             # stream loader
             thread.start_new_thread( self.sourceloader.cmd_export_schema_package,())    #export
             # stream package
             while self.sourceloader.packageload  == 'FALSE' :
                   thread.start_new_thread( self.sourceloader.cmd_load_package   ,())    
                   time.sleep(1) 
             while self.sourceloader.packageunload == 'FALSE' :
                   thread.start_new_thread( self.sourceloader.cmd_monitor_package,())
                   thread.start_new_thread( self.sourceloader.cmd_unload_package ,())    #release
                   time.sleep(1)
             self.sourceloader.packageload    = 'FALSE'
             self.sourceloader.packageunload  = 'FALSE'

      def cmd_import_schema_package(self):
          if MONITOR == 'OFF':
             #stream package
             self.targetloader.cmd_import_schema_package()
             if  mutex.locked()  == True :
                 mutex.release()           #release
          if MONITOR == 'ON':
             # stream loader
             thread.start_new_thread( self.targetloader.cmd_export_schema_package,())    #export
             # stream package
             while self.targetloader.packageload  == 'FALSE' :
                   thread.start_new_thread( self.targetloader.cmd_load_package   ,())    
                   time.sleep(1) 
             while self.targetloader.packageunload == 'FALSE' :
                   thread.start_new_thread( self.targetloader.cmd_monitor_package,())
                   thread.start_new_thread( self.targetloader.cmd_unload_package ,())    #release
                   time.sleep(1)
             self.targetloader.packageload    = 'FALSE'
             self.targetloader.packageunload  = 'FALSE'

      def cmd_transport_table_package(self):
          input = None
          if( (self.options.sourcedbnode == self.options.targetdbnode)   and
              (self.options.sourcedbname == self.options.targetdbname) and 
               self.options.streamformat == 'SQL' ):
               granted = True
               self.sourceloader.cmd     = "SELECT PRIVILEGE FROM DOMAIN.TABLEPRIVILEGES WHERE GRANTEE = '%s' AND TABLENAME = '%s' AND PRIVILEGE LIKE 'SELECT' " %(re.split(',',self.options.targetdbuser)[0],self.package.table)    
               self.sourceloader.cmd_execute()
               if self.sourceloader.sqlCode == 100: 
                  self.sourceloader.cmd  = "GRANT SELECT ON %s.%s TO     %s"     %(re.split(',',self.options.sourcedbuser)[0],self.package.table, re.split(',',self.options.targetdbuser)[0])
                  self.sourceloader.cmd_execute()
                  self.sourceloader.cmd  = "COMMIT WORK"
                  self.sourceloader.cmd_execute()
                  granted = False
               self.targetloader.cmd = "CREATE TABLE %s.%s LIKE       %s.%s"     %(re.split(',',self.options.targetdbuser)[0],self.package.table,re.split(',',self.options.sourcedbuser)[0],self.package.table)   
               self.targetloader.cmd_execute()
               self.targetloader.cmd = "COMMIT WORK"
               self.targetloader.cmd_execute()
               self.targetloader.cmd = "INSERT INTO  %s.%s SELECT * FROM %s.%s"  %(re.split(',',self.options.targetdbuser)[0],self.package.table,re.split(',',self.options.sourcedbuser)[0],self.package.table)
               self.targetloader.cmd_execute()
               self.targetloader.cmd = "COMMIT WORK"
               self.targetloader.cmd_execute()
               if  granted == False:
                   self.sourceloader.cmd = "REVOKE SELECT ON %s.%s  FROM %s"     %(re.split(',',self.options.sourcedbuser)[0],self.package.table, re.split(',',self.options.targetdbuser)[0])
                   self.sourceloader.cmd_execute()
                   self.sourceloader.cmd = "COMMIT WORK"
                   self.sourceloader.cmd_execute()
               if mutex.locked() == True:
                  mutex.release()

          if self.options.streamformat != 'SQL' : 
             #stream filter
             self.streamfilter   =  StreamFilter   (self.options , self )
             #Medium
             streamformat        = self.options.streamformat
             streamcharencoding  = self.options.streamcharencoding
             #Format
             streamformat_target = self.options.streamformat_target
          
             if self.package.column_field_long <> '' :
                if((self.options.streamformat     == 'FORMATTED' or
                    self.options.streamformat     == 'COMPRESSED'  )):
                    if self.options.streammedium  == 'PIPE' :
                       self.options.streamformat   = 'RECORDS'
                    if self.options.streamformat_target == 'PAGE' :
                       self.options.streamformat_target  = 'ROW'
                if self.options.streamformat      == 'RECORDS':
                   if self.options.streamcharencoding   == 'UTF8':
                      self.options.streamcharencoding    = 'UCS2'
             #stream export
             while input not in ["y"]:
                   if self.options.mode == 'INTER':
                      input = raw_input('Export    Table %s ? [y] : ' % ( self.package.table ))
                   if self.options.mode == 'PARAM':
                      input   = "y"
                   if input  == "y":
                      #stream loader
                      thread.start_new_thread( self.sourceloader.cmd_export_table_package,   ())    #export
                      if  self.options.streammedium == 'FILE'  :
                          #stream package
                          while  self.sourceloader.packageload  == 'FALSE' :
                                 thread.start_new_thread( self.sourceloader.cmd_load_package,())   
                                 time.sleep(1) 
                          while  self.sourceloader.packageunload == 'FALSE' :
                                 thread.start_new_thread( self.sourceloader.cmd_monitor_package,())
                                 thread.start_new_thread( self.sourceloader.cmd_unload_package ,()) #release
                                 time.sleep(1)
             input = None
             #stream import
             while input not in ["y"]:
                   if self.options.mode == 'INTER':
                      input = raw_input('Import    Table %s ? [y] : ' % ( self.package.table ))
                   if self.options.mode == 'PARAM':
                      input   = "y"
                   if input  == "y":
                      #stream loader
                      thread.start_new_thread( self.targetloader.cmd_import_table_package, ())    #import
                      #stream package
                      while self.targetloader.packageload  == 'FALSE' :
                             thread.start_new_thread( self.targetloader.cmd_load_package  ,())    
                             time.sleep(1) 
                      while  self.targetloader.packageunload == 'FALSE':
                             thread.start_new_thread( self.targetloader.cmd_monitor_package,())
                             thread.start_new_thread( self.targetloader.cmd_unload_package ,())   #release
                             time.sleep(1)
             #Medium
             if streamformat != self.options.streamformat :
                self.options.streamformat = streamformat
             if streamcharencoding != self.options.streamcharencoding :
                self.options.streamcharencoding = streamcharencoding
             #Format
             if streamformat_target != self.options.streamformat_target :
                self.options.streamformat_target = streamformat_target
             self.sourceloader.packageload   = 'FALSE'
             self.sourceloader.packageunload = 'FALSE'
             self.targetloader.packageload   = 'FALSE'
             self.targetloader.packageunload = 'FALSE'

      def cmd_export_table_package(self):
          if MONITOR == 'OFF':
             #stream package
             self.sourceloader.cmd_export_table_package()
             if  mutex.locked()  == True :
                 mutex.release()           #release
          if MONITOR == 'ON':
             # stream loader
             thread.start_new_thread( self.sourceloader.cmd_export_table_package, ())    #export
             # stream package
             while self.sourceloader.packageload  == 'FALSE' :
                   thread.start_new_thread( self.sourceloader.cmd_load_package   ,())    
                   time.sleep(1) 
             while self.sourceloader.packageunload == 'FALSE' :
                   thread.start_new_thread( self.sourceloader.cmd_monitor_package,())
                   thread.start_new_thread( self.sourceloader.cmd_unload_package ,())    #release
                   time.sleep(1)
             self.sourceloader.packageload    = 'FALSE'
             self.sourceloader.packageunload  = 'FALSE' 
		  
      def cmd_import_table_package(self):
          if MONITOR == 'OFF':
             #stream package
             self.targetloader.cmd_import_table_package()
             if  mutex.locked()  == True :
                 mutex.release()           #release
          if MONITOR == 'ON':
             # stream connector
             self.streamconnector =  StreamConnector(self.sourceloader, self.targetloader )
             self.streamfilter    =  StreamFilter   (self.options , self )
             #stream filter
             self.streamfilter.cmd_dtlsource2dtltarget()
             #stream loader
             thread.start_new_thread(self.targetloader.cmd_import_table_package, ())    #import
             # stream package
             while self.targetloader.packageload  == 'FALSE' :
                   thread.start_new_thread( self.targetloader.cmd_load_package   ,())    
                   time.sleep(1) 
             while self.targetloader.packageunload == 'FALSE' :
                   thread.start_new_thread( self.targetloader.cmd_monitor_package,())
                   thread.start_new_thread( self.targetloader.cmd_unload_package ,())   #release
                   time.sleep(1)
             self.targetloader.packageload    = 'FALSE'
             self.targetloader.packageunload  = 'FALSE'
                         
      def cmd_import_table_package_index(self):
          for cmd in self.package.indexStatements :
              self.targetloader.cmd = cmd
              self.targetloader.cmd_execute()

      def cmd_verify_package(self):
          input         = None
          sourcepackage = None
          targetpackage = None
          while input not in ["y"]:
                if self.options.mode == 'INTER':
                   input = raw_input('\nVerify    Transformation Package ? [y] : ')
                if self.options.mode == 'PARAM':
                   print '\nVerify    Transformation Package'
                   input   = "y"
                if input  == "y":
                   if  self.options.transformationpart  == 'DB'   :
                       if  self.options.transformationtype == 'TRANSPORT'  :
                           if os.path.isfile(os.path.join(self.options.sourceindepdatapath,'%s.package' % self.options.sourcedbname )):
                              sourcepackage =  'LOAD DATA TABLE TRANSFORMATIONMODEL\n'
                              '''
                              sourcepackage = sourcepackage + "SEPARATOR  '%s'  \n" % re.split('/',FIELDCOMPRESSED)[1]
                              sourcepackage = sourcepackage + "DELIMITER  '%s'  \n" % re.split('/',FIELDCOMPRESSED)[2]
                              sourcepackage = sourcepackage + "NULL       '%s'  \n" % FIELDNULLVALUE
                              sourcepackage = sourcepackage + "DECIMAL    '%s'  \n" % FIELDDECIMALVALUE
                              sourcepackage = sourcepackage + "BOOLEAN    '%s'  \n" % FIELDBOOLVALUE
                              sourcepackage = sourcepackage + "TIMESTAMP   %s   \n" % FIELDTIMESTAMPVALUE
                              sourcepackage = sourcepackage + "DATE        %s   \n" % FIELDATEVALUE
                              sourcepackage = sourcepackage + "TIME        %s   \n" % FIELDTIMEVALUE
                              '''
                              sourcepackage = sourcepackage + 'OWNER                 1 DEFAULT      NULL TABLENAME            2 DEFAULT      NULL \nDATASTREAMNAME          3 DEFAULT      NULL \n'
                              sourcepackage = sourcepackage + 'DATASTREAMHEADERBLOCK 4 DEFAULT      NULL DATASTREAMHEADERPOS  5 DEFAULT      NULL \nDATASTREAMTRAILERBLOCK  6 DEFAULT      NULL DATASTREAMTRAILERPOS  7 DEFAULT      NULL  DATASTREAMDATABLOCK  8 DEFAULT      NULL DATASTREAMDATAPOS    9  DEFAULT    NULL \n'
                              sourcepackage = sourcepackage + '\"EXCLUDE\"          10 DEFAULT      NULL \n'
                              sourcepackage = sourcepackage + 'PACKAGEGUID          11 DEFAULT      NULL TASKID              12 DEFAULT      NULL \nSTEPID                 13 DEFAULT      NULL SERVERNODE           14 DEFAULT      NULL \n'
                              sourcepackage = sourcepackage + '\"SERVERDB\"         15 DEFAULT      NULL USERNAME            16 DEFAULT      NULL \n\"CODETYPE\"           17 DEFAULT      NULL CODEPAGEID           18 DEFAULT      NULL \n'
                              sourcepackage = sourcepackage + '\"STORAGEDEVICE\"    19 DEFAULT      NULL TRANSACTIONSIZE     20 DEFAULT      NULL TRANSFORMATION           21 DEFAULT      NULL \nPART               22 DEFAULT      NULL TABLETYPE            23 DEFAULT      NULL \n'
                              sourcepackage = sourcepackage + 'CATALOGSTREAMNAME    24 DEFAULT      NULL CATALOGSTREAMTYPE   25 DEFAULT      NULL \nCATALOGSTREAMEBID      26 DEFAULT      NULL CATALOGSTREAMFORMAT  27 DEFAULT      NULL DATASTREAMTYPE       28 DEFAULT      NULL  DATASTREAMEBID     29 DEFAULT     NULL \n'
                              sourcepackage = sourcepackage + 'DATASTREAMFORMAT     30 DEFAULT      NULL \"RESTART\"         31 DEFAULT      NULL \nRESTORE                32 DEFAULT      NULL PAGECOUNT            33 DEFAULT      NULL \n'
                              sourcepackage = sourcepackage + 'ROWCOUNT             34 DEFAULT      NULL STARTTIME           35 DEFAULT      NULL \nENDTIME                36 DEFAULT      NULL REQUESTTIME          37 DEFAULT      NULL \n'
                              sourcepackage = sourcepackage + 'RECEIVETIME          38 DEFAULT      NULL IOTIME              39 DEFAULT      NULL \nLOGTIME                40 DEFAULT      NULL '
                              sourcepackage = sourcepackage + " INSTREAM FILE '%s'  COMPRESSED %s \n" % (os.path.join(self.options.sourceindepdatapath,'%s.package' % self.options.sourcedbname ),'' ) 
                              if self.sourceloader.packageguid != None:
                                 print 'Import    Transformation Package of EXPORT '   
                              if self.sourceloader.packageguid == None:   
                                 print 'Import    Transformation Package of EXPORT ' 
                              try :
                                     self.targetloader.loadersession.cmd ("%s" %  sourcepackage  )
                              except loader.CommunicationError, err :
                                     raise loader.CommunicationError, err, sys.exc_traceback
                              except loader.LoaderError, err:
                                     raise loader.LoaderError, err, sys.exc_traceback

                       time.sleep(1)               
                       if  self.options.transformationtype == 'TRANSPORT'  :
                           if os.path.isfile(os.path.join(self.options.targetindepdatapath,  '%s.package' % self.options.targetdbname )) :
                               if self.targetloader.catalog == 'UNICODE' :
                                  self.targetloader.package  = 'UCS2'
                               if self.targetloader.catalog == 'ASCII' :
                                  self.targetloader.package  = 'ASCII'
                               targetpackage =  'LOAD DATA TABLE TRANSFORMATIONMODEL  \n'
                               '''
                               targetpackage = targetpackage + "SEPARATOR  '%s'  \n" % re.split('/',FIELDCOMPRESSED)[1]
                               targetpackage = targetpackage + "DELIMITER  '%s'  \n" % re.split('/',FIELDCOMPRESSED)[2]
                               targetpackage = targetpackage + "NULL       '%s'  \n" % FIELDNULLVALUE
                               targetpackage = targetpackage + "DECIMAL    '%s'  \n" % FIELDDECIMALVALUE
                               targetpackage = targetpackage + "BOOLEAN    '%s'  \n" % FIELDBOOLVALUE
                               targetpackage = targetpackage + "TIMESTAMP   %s   \n" % FIELDTIMESTAMPVALUE
                               targetpackage = targetpackage + "DATE        %s   \n" % FIELDATEVALUE
                               targetpackage = targetpackage + "TIME        %s   \n" % FIELDTIMEVALUE
                               '''
                               targetpackage = targetpackage + 'OWNER                 1 DEFAULT      NULL TABLENAME            2 DEFAULT      NULL \nDATASTREAMNAME          3 DEFAULT      NULL \n'
                               targetpackage = targetpackage + 'DATASTREAMHEADERBLOCK 4 DEFAULT      NULL DATASTREAMHEADERPOS  5 DEFAULT      NULL \nDATASTREAMTRAILERBLOCK  6 DEFAULT      NULL DATASTREAMTRAILERPOS  7 DEFAULT      NULL  DATASTREAMDATABLOCK  8 DEFAULT      NULL DATASTREAMDATAPOS    9  DEFAULT    NULL \n'
                               targetpackage = targetpackage + '\"EXCLUDE\"          10 DEFAULT      NULL \n'
                               targetpackage = targetpackage + 'PACKAGEGUID          11 DEFAULT      NULL TASKID              12 DEFAULT      NULL \nSTEPID                 13 DEFAULT      NULL SERVERNODE           14 DEFAULT      NULL \n'
                               targetpackage = targetpackage + '\"SERVERDB\"         15 DEFAULT      NULL USERNAME            16 DEFAULT      NULL \n\"CODETYPE\"           17 DEFAULT      NULL CODEPAGEID           18 DEFAULT      NULL \n'
                               targetpackage = targetpackage + '\"STORAGEDEVICE\"    19 DEFAULT      NULL TRANSACTIONSIZE     20 DEFAULT      NULL TRANSFORMATION           21 DEFAULT      NULL \nPART               22 DEFAULT      NULL TABLETYPE            23 DEFAULT      NULL \n'
                               targetpackage = targetpackage + 'CATALOGSTREAMNAME    24 DEFAULT      NULL CATALOGSTREAMTYPE   25 DEFAULT      NULL \nCATALOGSTREAMEBID      26 DEFAULT      NULL CATALOGSTREAMFORMAT  27 DEFAULT      NULL DATASTREAMTYPE       28 DEFAULT      NULL  DATASTREAMEBID     29 DEFAULT     NULL \n'
                               targetpackage = targetpackage + 'DATASTREAMFORMAT     30 DEFAULT      NULL \"RESTART\"         31 DEFAULT      NULL \nRESTORE                32 DEFAULT      NULL PAGECOUNT            33 DEFAULT      NULL \n'
                               targetpackage = targetpackage + 'ROWCOUNT             34 DEFAULT      NULL STARTTIME           35 DEFAULT      NULL \nENDTIME                36 DEFAULT      NULL REQUESTTIME          37 DEFAULT      NULL \n'
                               targetpackage = targetpackage + 'RECEIVETIME          38 DEFAULT      NULL IOTIME              39 DEFAULT      NULL \nLOGTIME                40 DEFAULT      NULL '
                               targetpackage = targetpackage + " INSTREAM FILE '%s'  COMPRESSED %s \n" % (os.path.join(self.options.targetindepdatapath,  '%s.package' % self.options.targetdbname ), '' ) 
                               if self.targetloader.packageguid != None:
                                  print 'Import    Transformation Package of IMPORT ' 
                               if self.targetloader.packageguid == None:
                                  print 'Import    Transformation Package of IMPORT ' 
                               try :
                                      self.targetloader.loadersession.cmd ("%s" %  targetpackage )
                               except loader.CommunicationError, err :
                                      raise loader.CommunicationError, err, sys.exc_traceback
                               except loader.LoaderError, err:
                                      raise loader.LoaderError, err, sys.exc_traceback

                       if  self.options.transformationtype == 'TRANSPORT'  :
                           time.sleep(1)
                           cmd = '  SELECT   T1.PACKAGEGUID , T1.TRANSFORMATION, T1.OWNER, T1.TABLENAME , T1.DATASTREAMHEADERBLOCK, T1.DATASTREAMHEADERPOS , T1.DATASTREAMDATABLOCK, T1.DATASTREAMDATAPOS , T1.START '
                           cmd = cmd + '   FROM   TRANSFORMATIONSTEPS T1, TRANSFORMATIONSTEPS T2 '
                           cmd = cmd + '   WHERE  ( T1.start  <> T2.start  or T1.datastreamheaderpos <>  T2.datastreamheaderpos or  T1.datastreamdatapos <> T2.datastreamdatapos  ) '
                           cmd = cmd + '   AND T1.owner = T2.owner and T1.tablename = T2.tablename '
                           self.cmd = cmd
                           try :
                               self.targetloader.loadersession.cmd ("%s" % self.cmd  )
                           except loader.LoaderError, err:
                               if err.sqlCode == 100:
                                  print 'OK'
                               else:
                                  raise loader.LoaderError, err, sys.exc_traceback
                              
          if mutex.locked () == 1:
             mutex.release()  #release                           
             
      def cmd_purify_package(self):
          if  socket.gethostname() == 'p30882' :  #p30882
              self.sourceloader.loadersession.release()
              self.targetloader.loadersession.release()
              try :
                   os.chdir (os.path.join (os.environ ['INSTROOT'], 'wrk'))
              except OSError, (errno, err ):
                   print 'command failed: ', err

              #source
              if  os.path.isfile      ( self.sourceloader.protocol ) :
                  transformationstep = []
                  protocol      = file( self.sourceloader.protocol ,'r')
                  protocolline  = protocol.readline()
                  while protocolline :
                        match_1 = re.search ('// R', protocolline )
                        if match_1 != None :
                           cmd  = string.replace( protocolline , "// R\t" , '//\n')
                           transformationstep.append(cmd)
                        protocolline  = protocol.readline()
                  protocol.close()
                  time.sleep(1)
                  protocol      = file( self.sourceloader.protocol ,'a+')
                  protocol.writelines(transformationstep)
                  protocol.close()

                  sourceloader_batch  = string.replace(self.sourceloader.protocol, 'prt', 'bat' )
                  sourceloader_purify = string.replace(self.sourceloader.protocol, 'prt', 'pur' )
                  try:
                     os.rename(self.sourceloader.protocol, sourceloader_batch )
                  except OSError, err:
                     print 'command failed:', err
                  purify_loadercli  = '\n%s -b %s -E 10\n'   % (os.path.join(self.options.sourceindepdatapath,'bin','loadercli'), sourceloader_batch)
                  #purify_loadercli  = 'purify /SaveTextData=%s /ExcludeSystemDir %s.exe -b %s'   % (sourceloader_purify,os.path.join( self.options.sourceindepdatapath,'bin','loadercli'), sourceloader_batch)
                  print purify_loadercli
                  try:
                      os.system (purify_loadercli)
                  except OSError, err:
                      print 'command failed:', err

              #target
              if  os.path.isfile      ( self.targetloader.protocol ) :
                  transformationstep = []
                  protocol      = file( self.targetloader.protocol ,'r')
                  protocolline  = protocol.readline()
                  while protocolline :
                        match_1 = re.search ('// R', protocolline )
                        if match_1 != None :
                           cmd  = string.replace( protocolline , "// R\t" , '//\n')
                           transformationstep.append(cmd)
                        protocolline  = protocol.readline()
                  protocol.close()
                  time.sleep(1)
                  protocol      = file( self.targetloader.protocol ,'a+')
                  protocol.writelines(transformationstep)
                  protocol.close()

                  targetloader_batch  = string.replace(self.targetloader.protocol, 'prt', 'bat' )
                  targetloader_purify = string.replace(self.targetloader.protocol, 'prt', 'pur' )
                  try:
                     os.rename(self.targetloader.protocol, targetloader_batch )
                  except OSError, err:
                     print 'command failed:', err
                  purify_loadercli  = '\n%s -b %s -E 10\n'   % (os.path.join(self.options.targetindepdatapath,'bin','loadercli'), targetloader_batch)
                  #purify_loadercli  = 'purify /SaveTextData=%s /ExcludeSystemDir %s.exe -b %s'   % (targetloader_purify,os.path.join(self.options.targetindepdatapath,'bin','loadercli'),targetloader_batch)
                  print purify_loadercli
                  try:
                      os.system (purify_loadercli)
                  except OSError, err:
                      print 'command failed:', err

class StreamLoader :
      def __init__(self, options = None, package = None, transformationtype = None):
          self.options                = options
          self.package                = package
          self.transformationtype     = transformationtype
          if self.transformationtype== 'EXPORT':
             self.sapdbnode = options.sourcedbnode
             self.sapdb     = options.sourcedbname
             username = re.split(',',options.sourcedbuser)[0]
             password = re.split(',',options.sourcedbuser)[1]
             if username != '' and username [0] != '"' and username[len(username)-1] != '"':
                username = string.upper(username)
             elif username != '':
                username = string.replace(username,'"','') 
             self.user      = username
             self.pwd       = password
             self.catalog   = DB_DEFAULTCODE_SOURCE
          if self.transformationtype== 'IMPORT':
             self.sapdbnode = options.targetdbnode
             self.sapdb     = options.targetdbname
             username = re.split(',',options.targetdbuser)[0]
             password = re.split(',',options.targetdbuser)[1]
             if username != '' and username [0] != '"' and username[len(username)-1] != '"':
                username = string.upper(username)
             elif username != '':
                username = string.replace(username,'"','') 
             self.user      = username
             self.pwd       = password
             self.catalog   = DB_DEFAULTCODE_TARGET
          self.version                = ''
          self.streammedium           = options.streammedium
          self.streamformat           = options.streamformat
          self.streamcharencoding     = options.streamcharencoding
          self.startposition          = options.startposition
          self.owner                  = None
          self.table                  = None
          self.packagecode            = 'ASCII'
          self.packageguid            = None
          self.packagestart           = None
          self.packageend             = PACKAGETIMEOUT
          self.packageload            = 'FALSE'
          self.packageunload          = 'FALSE'
          self.protocol               = None
          self.cmd                    = None
          self.loaderpackage          = None
          #stream_name
          self.catalogstream_name     = None
          self.datastream_name        = None
          self.packagestream_name     = None 
          # medium pipe
          self.ddlpipe                = "\\\\.\PIPE\\catalog"
          self.datpipe                = "\\\\.\PIPE\\data"
          self.datpipe_source         = "\\\\.\PIPE\\source.data"
          self.datpipe_target         = "\\\\.\PIPE\\target.data"
          self.ddlpipe_source         = os.path.join('\\\\','.','PIPE','%s.CATALOG' % self.options.sourcedbname)
          self.ddlpipe_target         = os.path.join('\\\\','.','PIPE','%s.CATALOG' % self.options.targetdbname)
          # medium tape
          self.ddlpipe_ebid           = None
          self.datpipe_ebid           = None
          self.mappipe_ebid           = None
          # statement
          self.cmd
          self.sqlsession             = None
          self.sqlCode                = 0
          self.loadersession          = None
          self.errorCode              = 0
          # protocol session
          try:
              if self.package != None:
                 if self.transformationtype == 'EXPORT' or self.transformationtype == 'TRANSPORT' :
                    self.loaderpackage  = file(os.path.join(self.options.sourceindepdatapath,os.path.split(self.options.sourceindepdatapath)[1] +os.extsep+"prt"),'w+')
                 if self.transformationtype == 'IMPORT' or self.transformationtype == 'TRANSPORT' :
                    self.loaderpackage  = file(os.path.join(self.options.targetindepdatapath,os.path.split(self.options.targetindepdatapath)[1] +os.extsep+"prt"),'w+')
          except IOError, err:
              print err
          
          # sql session
          try:
              self.sqlsession = sql.connect ( self.user, self.pwd, self.sapdb, self.sapdbnode )
          except sql.CommunicationError, err :
              print err
              raise sql.CommunicationError, err, sys.exc_traceback
          except sql.SQLError, err :
              print err
              raise sql.SQLError, err, sys.exc_traceback
              
          # loader session
          try:
              global EXPORTER_VERSION
              global IMPORTER_VERSION
              self.loadersession =  loader.Loader('', '', '%s' % self.options.loaderinstroot)
              #Module Name
              self.cmd = 'REMARK "Loader Extension Module: %s"'% os.path.abspath(sys.argv[0])
              remark = self.cmd_execute()
              #Configuration Directory
              self.cmd = 'REMARK "Loader Config Directory: %s"'% os.path.join (LOADERPROGPATH,'sdb','loader','config')
              remark = self.cmd_execute()
              #version
              self.cmd = 'VERSION\n'
              version = self.cmd_execute()
              if self.transformationtype== 'EXPORT':
                 EXPORTER_VERSION = string.replace(re.split(',',re.split('\n',version)[1])[0],'"','')
              if self.transformationtype== 'IMPORT':
                 IMPORTER_VERSION = string.replace(re.split(',',re.split('\n',version)[1])[0],'"','')
              #protocol odbc
              self.cmd = 'SET ODBCTRACE        OFF\n'
              self.cmd_execute()
              #protocol loader
              self.cmd = 'SET\n'
              configuration = self.cmd_execute()
              if configuration != 0:
                 self.protocol = string.strip(string.replace(string.replace(configuration,'Loader protocol:',''),'\'',''))
              self.cmd  = "SET CONFIGURATION   EXCLUDE\n" 
              self.cmd_execute()   
              #sqlmode
              self.cmd = 'SQLMODE INTERNAL\n'
              self.cmd_execute()
              self.cmd = 'SET SYSTEMATTRIBUTE  %s\n' % SYSTEMATTRIBUTE
              self.cmd_execute()
              #transaction
              self.cmd = 'AUTOCOMMIT           %s\n' % AUTOCOMMIT
              self.cmd_execute()
              self.cmd = 'SET TRANSACTION SIZE %s\n' % TRANSACTIONSIZE
              self.cmd_execute()
              self.cmd = 'SET MAXERRORCOUNT    %s\n' % MAXERRORCOUNT
              self.cmd_execute()
              #print      'SET PACKAGETIMEOUT  %s\n' % PACKAGETIMEOUT
              #format
              self.cmd = "SET COMPRESSED       '%s'\n" % FIELDCOMPRESSED
              self.fieldCompressed            = FIELDCOMPRESSED
              self.cmd_execute()
              self.cmd = "SET LANGUAGE         'ENG'\n"
              self.cmd_execute()
              #datatypes
              #self.cmd = 'SET CODETYPE         ASCII\n'
              #self.cmd_execute()
              self.cmd = 'SET CODEPAGE         ISO-8859-1\n'
              self.cmd_execute()
              self.cmd = "SET BOOLEAN          '%s'\n" % FIELDBOOLVALUE
              self.fieldBoolValue            =  FIELDBOOLVALUE
              self.cmd_execute()
              #self.cmd = "SET DECIMAL         '%s'\n" % FIELDDECIMALVALUE
              #self.fieldDecimalValue        =  FIELDDECIMALVALUE
              #self.cmd_execute()
              self.fieldDecimalValue         =  FIELDDECIMALVALUE
              self.cmd = "SET NULL              '%s'\n" % FIELDNULLVALUE
              self.fieldNullValue            =  FIELDNULLVALUE
              self.cmd_execute()
              self.cmd = 'SET DATE             %s\n' % FIELDATEVALUE
              self.fieldDateValue            = FIELDATEVALUE
              self.cmd_execute()
              self.cmd = 'SET TIME             %s\n' % FIELDTIMEVALUE
              self.fieldTimeValue            = FIELDTIMEVALUE
              self.cmd_execute()
              self.cmd = 'SET TIMESTAMP        %s\n' % FIELDTIMESTAMPVALUE
              self.fieldTimeStampValue       = FIELDTIMESTAMPVALUE
              self.cmd_execute()
              #medium
              self.cmd = 'SET BACKUPTOOL       NETWORKER\n'
              self.cmd_execute()
          except loader.CommunicationError, err :
                 raise loader.CommunicationError, err, sys.exc_traceback
          except loader.LoaderError, err:
                 raise loader.LoaderError, err, sys.exc_traceback

      def release(self):
          # sql session
          if self.sqlsession != None:
             try:
                self.sqlsession.release()
             except sql.CommunicationError, err :
                pass
             except:
                pass
          # loader session
          if self.loadersession != None:
             try:
                 self.loadersession.release()
             except loader.CommunicationError, err :
                    raise loader.CommunicationError, err, sys.exc_traceback
             except loader.LoaderError, err:
                    raise loader.LoaderError, err, sys.exc_traceback
          # protocol session
          if self.loaderpackage != None:
             try:
                 time.sleep(0.5)
                 self.loaderpackage.close()
             except IOError, err:
                 print err
          
      def cmd_execute(self):
          self.packagestart = time.time()
          rc  = 0
          cmd = self.cmd
          cmd = string.replace(cmd,self.options.sourceindepdatapath + os.sep , '')
          cmd = string.replace(cmd,self.options.targetindepdatapath + os.sep , '')
          cmd = '\n%s//' % cmd
          
          if self.package != None and self.options.transporttype != 'EXECUTE':         
             self.packageload   = 'TRUE'
             self.packageunload = 'TRUE'
             try :
                  if self.loaderpackage != None and self.cmd[0:8] == 'USE USER':
                     cmd  = '\nUSER USER %s %s SERVERDB %s ON %s \n//'% ( self.user,'*', self.sapdb, self.sapdbnode)
                  if self.loaderpackage != None and self.cmd[0:6] != 'REMARK' :
                     self.loaderpackage.write(cmd)
             except IOError, err:
                   print err
             if mutex.locked() == True:
                mutex.release()    #release
          if self.package == None or self.options.transporttype == 'EXECUTE':         
             try :
                 rc = self.loadersession.cmd ("%s" % self.cmd )
             except loader.LoaderError, err:
                 self.cmd_error_handler(err)
             except loader.CommunicationError, err :
                 print 'Received cancel request; stopping execution'
                 self.package.transformation.release()
                 os._exit(1)
             try :
                 if self.loaderpackage != None and self.cmd[0:8] == 'USE USER':
                    cmd  = '\nUSER USER %s %s SERVERDB %s ON %s \n//'% ( self.user,'*', self.sapdb, self.sapdbnode)
                 if self.loaderpackage != None and self.cmd[0:6] != 'REMARK' :
                    self.loaderpackage.write(cmd)
             except IOError, err:
                   print err
          self.cmd = ''
          return rc
      
      def cmd_error_handler(self,err):
          self.sqlCode          = err.sqlCode
          self.errorCode        = err.errorCode
          if( err.sqlCode       ==  100   or
              err.sqlCode       ==  200   or
              err.sqlCode       == -4004  or
              err.sqlCode       == -4011  or
              err.sqlCode       == -5001  or
              err.sqlCode       == -6000  or
              err.sqlCode       == -6008  or
              err.sqlCode       == -7017  or
              err.errorCode     == -25010 or
              err.errorCode     == -25362 or
              err.errorCode     == -25498 or
              err.errorCode     == -25500 or
              err.errorCode     == -25501  ):
              if( err.sqlCode   ==  200   or
                  err.errorCode == -25010 or
                  err.errorCode == -25362 or
                  err.errorCode == -25498 or
                  err.errorCode == -25500 or
                  err.errorCode == -25501 ):
                  if err.sqlCode   ==  200  and self.options.streamformat != 'SQL' :
                     print '\n%s error(s) during execution of package: \n---> See loader server logfile for more information.' % MAXERRORCOUNT
                     self.package.transformation.release()
                     os._exit(1)
                  if err.errorCode == -25010 :
                     pass
                  if err.errorCode == -25362 :
                     pass
                  if err.errorCode == -25498 :
                     pass
                  if err.errorCode == -25500 :
                     print err
                     self.package.transformation.release()
                     os._exit(1)
                  if err.errorCode == -25501 :
                     print '\n%s error(s) during execution of package: \n---> See loader server logfile for more information.' % MAXERRORCOUNT
                     pass
              else:
                 self.package.maxerrorcount = self.package.maxerrorcount + 1
              if self.package.maxerrorcount == 1 :
                 print '\nAn error during execution of package: \n---> See loader server logfile for more information.'
              if self.package.maxerrorcount == MAXERRORCOUNT :
                 self.package.transformation.release()
                 os._exit(1)
          else:
             raise loader.LoaderError, err, sys.exc_traceback

      def cmd_use_serverdb_user(self):
          cmd     = 'USE USER %s %s SERVERDB %s' % ( self.user, self.pwd, self.sapdb )
          if self.sapdbnode != '':
             cmd  = cmd + ' ON %s\n'             % ( self.sapdbnode )
          self.cmd     = cmd
          self.cmd_execute()

      def cmd_export_catalog_package(self):
          #Medium
          if   self.streammedium[0:4] == 'PIPE' :
               streammedium = 'PIPE'
          else :
               streammedium = self.streammedium

          #Format
          if( self.streamformat == 'DDL' or
              self.streamformat == 'XML'    ):
              #Medium
              if self.options.streammedium == 'FILE' :
                 cmd = "CATALOGEXTRACT ALL"
                 cmd = cmd +   " OUTSTREAM FILE '%s' %s          "   % (os.path.join(self.options.sourceindepdatapath,  '%s.catalog' % (self.options.sourcedbname )),self.options.streamformat,)
              #Medium
              if(self.options.streammedium == 'TAPE' or
                 self.options.streammedium == 'PIPE'   ):
                 cmd = 'CATALOGEXTRACT ALL \n'
                 cmd = cmd +   " OUTSTREAM  %s '%s'  %s    %s    "   % (streammedium, self.ddlpipe_source, self.options.streamformat, self.streamcharencoding )

              if self.streamformat  == 'XML':
                 self.packageload    = 'TRUE'
                 self.packageunload  = 'TRUE'
                 mutex.release()

          #Format
          if( self.streamformat == 'PAGES'   or
              self.streamformat == 'RECORDS'    ):
              restartRecord = self.cmd_restart_package()
              #Medium
              if self.options.streammedium == 'FILE' :
                 cmd = 'DBEXTRACT \n'
                 cmd = cmd +   " CONFIGURATION  INSTREAM  FILE '%s' COMPRESSED  %s     \n"  % (os.path.join(self.options.sourceindepdatapath,  '%s.configuration' % self.sapdb ), self.options.streamcharencoding )
                 cmd = cmd +   " CATALOG        OUTSTREAM FILE '%s' DDL         %s     \n"  % (os.path.join(self.options.sourceindepdatapath,  '%s.catalog'       % self.sapdb ), self.streamcharencoding )
                 cmd = cmd +   " DATA           OUTSTREAM FILE '%s' %s          %s     \n"  % (os.path.join(self.options.sourceindepdatapath,  '%s.data'          % self.sapdb ), self.options.streamformat, self.streamcharencoding )
                 cmd = cmd +   " PACKAGE        OUTSTREAM FILE '%s' COMPRESSED  %s       "  % (os.path.join(self.options.sourceindepdatapath,  '%s.package'       % self.sapdb ), self.streamcharencoding )

              #Medium
              if(self.options.streammedium == 'TAPE' or
                 self.options.streammedium == 'PIPE'   ):
                 cmd = 'DBEXTRACT \n'
                 cmd = cmd +   " CONFIGURATION  INSTREAM  FILE '%s'  COMPRESSED %s     \n"  % (os.path.join(self.options.sourceindepdatapath,  '%s.configuration' % self.sapdb ), self.streamcharencoding )
                 cmd = cmd +   " CATALOG        OUTSTREAM %s   '%s'  DDL        %s     \n"  % (streammedium, self.ddlpipe_source                     , self.streamcharencoding )
                 cmd = cmd +   " DATA           OUTSTREAM %s   '%s'  %s         %s     \n"  % (streammedium, self.datpipe_source , self.streamformat , self.streamcharencoding )
                 cmd = cmd +   " PACKAGE        OUTSTREAM FILE '%s'  COMPRESSED %s       "  % (os.path.join(self.options.sourceindepdatapath,  '%s.package'       % self.sapdb ), self.streamcharencoding )
              #Start
              if restartRecord <> [] :
                 cmd = cmd + " RESTART '%s' " % (restartRecord[0])

          if TRANSFORMATIONSYNTAX == 'STRATIFIED':
             self.cmd = cmd
          if TRANSFORMATIONSYNTAX == 'UNIFIED':
             self.cmd = self.package.transformation.exportstatement
          self.cmd_execute()

      def cmd_import_catalog_package(self):
          #Medium
          if  self.streammedium      == 'TAPE' :
              self.ddlpipe_target =  self.ddlpipe_source
              self.datpipe_source =  self.datpipe_source
          if  self.streammedium[0:4] == 'PIPE' :
              streammedium = 'PIPE'
          else:
              streammedium = self.streammedium

          #Encoding
          if self.catalog == 'UNICODE' :
             self.packagecode  = 'UCS2'
          if self.catalog == 'ASCII' :
             self.packagecode  = 'ASCII'

          #Format
          if( self.streamformat == 'DDL' or
              self.streamformat == 'XML'   ):
              cmd = 'CATALOGLOAD ALL '
              #Medium
              if self.streammedium      == 'FILE':
                 cmd = cmd + "                   INSTREAM FILE '%s'             %s          " % (os.path.join(self.options.sourceindepdatapath,   '%s.catalog' % (self.options.sourcedbname )),self.streamformat)
              #Medium
              if self.streammedium[0:4] == 'PIPE' :
                 cmd = cmd + "                   INSTREAM  %s  '%s'             %s  %s      " % (streammedium, self.ddlpipe_target                       , self.streamformat, self.streamcharencoding   )
              #Medium
              if self.streammedium      == 'TAPE':
                 cmd = cmd + "                   INSTREAM  %s  EBID %s  '%s'    %s  %s      " % (streammedium, self.ddlpipe_ebid , self.ddlpipe_target   , self.streamformat, self.streamcharencoding   )

              if self.streamformat  == 'XML':
                 self.packageload    = 'TRUE'
                 self.packageunload  = 'TRUE'

          #Format
          if( self.streamformat == 'PAGES'   or
              self.streamformat == 'RECORDS'    ):
              restartRecord = self.cmd_restart_package()

              #Medium
              if self.streammedium[0:4] == 'FILE' :
                 cmd = 'DBLOAD \n'
                 cmd = cmd +    " CONFIGURATION  INSTREAM FILE  '%s' COMPRESSED  %s   \n" % (os.path.join(self.options.sourceindepdatapath,  '%s.configuration' % self.sapdb), self.streamcharencoding )
                 cmd = cmd +    " CATALOG        INSTREAM FILE  '%s' DDL         %s   \n" % (os.path.join(self.options.sourceindepdatapath,  '%s.catalog'       % self.options.sourcedbname), self.streamcharencoding)
                 cmd = cmd +    " DATA           INSTREAM FILE  '%s' %s          %s   \n" % (os.path.join(self.options.sourceindepdatapath,  '%s.data'          % self.options.sourcedbname), self.options.streamformat, self.streamcharencoding)
              #Medium
              if self.streammedium[0:4] == 'PIPE' :
                 cmd = 'DBLOAD \n'
                 cmd = cmd +    " CONFIGURATION  INSTREAM FILE  '%s' COMPRESSED  %s   \n" % (os.path.join(self.options.sourceindepdatapath,  '%s.configuration' % self.sapdb), self.streamcharencoding )
                 cmd = cmd +    " CATALOG        INSTREAM  %s   '%s' DDL         %s   \n" % (streammedium, self.ddlpipe_target                     , self.streamcharencoding   )
                 cmd = cmd +    " DATA           INSTREAM  %s   '%s' %s          %s   \n" % (streammedium, self.datpipe_source , self.streamformat , self.streamcharencoding   )
              #Medium
              if self.streammedium      == 'TAPE':
                 cmd = 'DBLOAD \n'
                 cmd = cmd +    " CONFIGURATION  INSTREAM FILE  '%s' COMPRESSED  %s        \n" % (os.path.join(self.options.sourceindepdatapath,  '%s.configuration' % self.sapdb), self.streamcharencoding )
                 cmd = cmd +    " CATALOG        INSTREAM  %s   EBID %s  '%s'   DDL   %s   \n" % (streammedium, self.ddlpipe_ebid , self.ddlpipe_target ,                       self.streamcharencoding   )
                 cmd = cmd +    " DATA           INSTREAM  %s   EBID %s  '%s'    %s   %s   \n" % (streammedium, self.mappipe_ebid , self.datpipe_source , self.streamformat   , self.streamcharencoding   )
              cmd = cmd +       " PACKAGE        OUTSTREAM FILE '%s' COMPRESSED  %s          " % (os.path.join(self.options.targetindepdatapath,  '%s.package' % (self.sapdb)), self.packagecode )
              #Start
              if restartRecord <> [] :
                 cmd = cmd +    " RESTART '%s' " % (restartRecord[0])

          if self.streamformat  <> 'XML':
             if TRANSFORMATIONSYNTAX == 'STRATIFIED':
                self.cmd = cmd
             if TRANSFORMATIONSYNTAX == 'UNIFIED':
                self.cmd = self.package.transformation.importstatement
             self.cmd_execute()

      def cmd_export_schema_package(self):
          #Medium
          if   self.streammedium[0:4] == 'PIPE' :
               streammedium = 'PIPE'
          else :
               streammedium = self.streammedium
          #Format
          if( self.streamformat == 'PAGES'   or
              self.streamformat == 'RECORDS'    ):
              cmd = 'TABLEEXTRACT USER \n'
              cmd = cmd +      " CONFIGURATION  INSTREAM  FILE '%s'    COMPRESSED %s \n" % (os.path.join(self.options.sourceindepdatapath,   '%s.configuration' % (self.sapdb )), self.streamcharencoding )
              #Medium
              if(self.options.streammedium == 'TAPE' or
                 self.options.streammedium == 'PIPE'   ):
                 cmd = cmd +   " DATA           OUTSTREAM %s   '%s'    %s         %s \n" % (streammedium, self.datpipe_source , self.streamformat  , self.streamcharencoding )
                 cmd = cmd +   " PACKAGE        OUTSTREAM FILE '%s'    COMPRESSED %s   " % (os.path.join (self.options.sourceindepdatapath,   '%s.package'       % (self.sapdb )), self.streamcharencoding )
              #Medium
              if(self.options.streammedium == 'FILE'     ):
                 cmd = cmd +   " DATA           OUTSTREAM FILE '%s'    %s         %s \n" % (os.path.join(self.options.sourceindepdatapath,   '%s_%s.data'       % (self.options.sourcedbname,self.user )),self.options.streamformat,self.streamcharencoding )
                 cmd = cmd +   " PACKAGE        OUTSTREAM FILE '%s'    COMPRESSED %s   " % (os.path.join(self.options.sourceindepdatapath,   '%s_%s.package'    % (self.options.sourcedbname,self.user)), self.streamcharencoding )
          #Format
          if( self.streamformat == 'DDL' or
              self.streamformat == 'XML'      ):
              cmd = 'CATALOGEXTRACT USER \n'
              #Medium
              if(self.options.streammedium == 'TAPE' or
                 self.options.streammedium == 'PIPE'   ):
                 cmd = cmd +   " OUTSTREAM %s   '%s'  %s   %s  "  % ( streammedium, self.ddlpipe_source, self.options.streamformat, self.streamcharencoding )
              #Medium
              if self.options.streammedium == 'FILE'  :
                 cmd = cmd +   " OUTSTREAM FILE '%s'  %s   %s  "    % (os.path.join(self.options.sourceindepdatapath,   '%s_%s.dtl'  % (self.sapdb,self.user )),self.options.streamformat, self.streamcharencoding)

          if TRANSFORMATIONSYNTAX == 'STRATIFIED':
             self.cmd = cmd
          if TRANSFORMATIONSYNTAX == 'UNIFIED':
             self.cmd = self.package.transformation.exportstatement
          self.cmd_execute()

      def cmd_import_schema_package(self):
          #Medium
          if  self.streammedium      == 'TAPE' :
              self.ddlpipe_target =  self.ddlpipe_source
              self.datpipe_source =  self.datpipe_source
          if  self.streammedium[0:4] == 'PIPE' :
              streammedium = 'PIPE'
          else:
              streammedium = self.streammedium

          if self.catalog == 'UNICODE' :
             self.packagecode  = 'UCS2'
          if self.catalog == 'ASCII' :
             self.packagecode  = 'ASCII'

          #Format
          if( self.streamformat == 'PAGES'   or
              self.streamformat == 'RECORDS'    ):
              cmd = 'TABLELOAD    USER \n'
              cmd =       cmd + " CONFIGURATION INSTREAM  FILE '%s' COMPRESSED %s \n" % (os.path.join(self.options.sourceindepdatapath,  '%s.configuration' % (self.sapdb )), self.streamcharencoding )
              #Medium
              if self.streammedium[0:4] == 'PIPE':
                 cmd =    cmd + " DATA          INSTREAM  %s   '%s' %s         %s \n" % (streammedium, self.datpipe_source , self.streamformat , self.streamcharencoding   )
                 cmd =    cmd + " PACKAGE       OUTSTREAM FILE '%s' COMPRESSED %s   " % (os.path.join(self.options.targetindepdatapath,  '%s.package' % (self.sapdb)), self.packagecode )
              #Medium
              if self.streammedium      == 'TAPE':
                 cmd =    cmd + " DATA          INSTREAM  %s   '%s' EBID %s %s %s \n" % (streammedium, self.datpipe_source , self.mappipe_ebid , self.streamformat , self.streamcharencoding   )
                 cmd =    cmd + " PACKAGE       OUTSTREAM FILE '%s' COMPRESSED %s   " % (os.path.join(self.options.targetindepdatapath,  '%s.package' % (self.sapdb)), self.packagecode )
              #Medium
              if self.streammedium      == 'FILE':
                 if self.options.streamformat   == 'PAGES'   :
                    cmd = cmd + " DATA          INSTREAM  FILE '%s' %s            \n" % (os.path.join(self.options.sourceindepdatapath,  '%s_%s.data'       % (self.options.sourcedbname,self.user )),self.options.streamformat)
                    cmd = cmd + " PACKAGE       OUTSTREAM FILE '%s' COMPRESSED %s   " % (os.path.join(self.options.targetindepdatapath,  '%s_%s.package'    % (self.options.targetdbname,self.user)), self.packagecode)
                 if self.options.streamformat   == 'RECORDS' :
                    cmd = cmd + " DATA          INSTREAM  FILE '%s' %s         %s \n" % (os.path.join(self.options.sourceindepdatapath,  '%s_%s.data'       % (self.options.sourcedbname,self.user )),self.options.streamformat,self.streamcharencoding)
                    cmd = cmd + " PACKAGE       OUTSTREAM FILE '%s' COMPRESSED %s   " % (os.path.join(self.options.targetindepdatapath,  '%s_%s.package'    % (self.options.targetdbname,self.user)), self.packagecode)
          #Format
          if( self.streamformat == 'DDL' or
              self.streamformat == 'XML'      ):
              cmd = 'CATALOGLOAD USER\n'
              #Medium
              if self.streammedium[0:4] == 'PIPE' :
                 cmd = cmd + "                INSTREAM  %s  '%s'          %s  %s  \n" % (streammedium, self.ddlpipe_target                     , self.options.streamformat  , self.streamcharencoding  )
              #Medium
              if self.streammedium      == 'TAPE':
                 cmd = cmd + "                INSTREAM  %s  '%s' EBID %s  %s  %s  \n" % (streammedium, self.ddlpipe_target , self.ddlpipe_ebid , self.options.streamformat , self.streamcharencoding   )
              #Medium
              if self.streammedium      == 'FILE':
                 cmd = cmd + "                INSTREAM FILE '%s'          %s   %s   " % (os.path.join(self.options.sourceindepdatapath,  '%s_%s.dtl'  % (self.options.sourcedbname,self.user )), self.options.streamformat, self.streamcharencoding )

              if self.streamformat  == 'XML':
                 self.packageload    = 'TRUE'
                 self.packageunload  = 'TRUE'

          if self.streamformat  <> 'XML':
             if TRANSFORMATIONSYNTAX == 'STRATIFIED':
                self.cmd = cmd
             if TRANSFORMATIONSYNTAX == 'UNIFIED':
                self.cmd = self.package.transformation.importstatement
             self.cmd_execute()

      def cmd_export_table_package(self):
          sapdb        = self.package.transformation.sourceloader.sapdb
          user         = self.package.user
          table        = self.package.table

          if TRANSFORMATIONSYNTAX == 'STRATIFIED':
             #Medium
             if self.options.streammedium    == 'FILE'  :
                #Format
               if(self.options.streamformat == 'XML' or
                  self.options.streamformat == 'DDL' ) :
                  cmd = 'CATALOGEXTRACT TABLE '
                  cmd = cmd + " \"%s\".\"%s\" \n"                 % (user, table)
                  cmd = cmd + "      OUTSTREAM FILE '%s' %s   \n" % (os.path.join(self.options.sourceindepdatapath, '%s_%s_%s.dtl'  % (sapdb,user,table )),self.options.streamformat)
               #Format
               if(self.options.streamformat == 'PAGES' or
                  self.options.streamformat == 'RECORDS'  ):
                  cmd = 'TABLEEXTRACT TABLE '
                  cmd = cmd + " \"%s\".\"%s\" \n"                 % ( user, table)
                  cmd = cmd + " DATA OUTSTREAM FILE '%s' %s %s\n" % (os.path.join(self.options.sourceindepdatapath, '%s_%s_%s.data' % (sapdb,user,table )),self.options.streamformat,self.options.streamcharencoding)
               #Format
               if(self.options.streamformat[0:9] == 'FORMATTED'  or
                  self.options.streamformat      == 'COMPRESSED'    ):
                  cmd = "DATAEXTRACT "
                  if self.options.streamformat_target == 'ROW':
                     cmd  = cmd + " FOR DATALOAD TABLE "
                  if self.options.streamformat_target == 'PAGE':
                     cmd  = cmd + " FOR FASTLOAD TABLE "
                  cmd = cmd + " \"%s\".\"%s\" \n"                 % (user, table)
                  cmd = cmd + "      OUTSTREAM FILE '%s'      \n" % (os.path.join(self.options.sourceindepdatapath, '%s_%s_%s.dtl'  % (sapdb,user,table )))
                  cmd = cmd + "      OUTSTREAM FILE '%s' %s %s\n" % (os.path.join(self.options.sourceindepdatapath, '%s_%s_%s.data' % (sapdb,user,table )),self.options.streamformat,self.options.streamcharencoding)
                  cmd = cmd + self.package.column_field_long
             #Medium
             if(self.options.streammedium[0:4] == 'PIPE' ):
                #Format
                if( self.options.streamformat == 'PAGES' or
                    self.options.streamformat == 'RECORDS'  ) :
                    cmd  = "TABLEEXTRACT TABLE "
                    cmd  = cmd + " \"%s\".\"%s\" \n"                 % ( user, table )
                    cmd  = cmd + " DATA OUTSTREAM PIPE '%s' %s %s\n" % ('\\\\.\\PIPE\\%s_%s_%s.dat' % ( sapdb, user, table ), self.options.streamformat, self.options.streamcharencoding )
                #Format
                if( self.options.streamformat == 'FORMATTED'  or
                    self.options.streamformat == 'COMPRESSED'      ):
                    cmd  = "DATAEXTRACT "
                    if self.options.streamformat_target == 'ROW':
                       cmd  = cmd + " FOR DATALOAD TABLE "
                    if self.options.streamformat_target == 'PAGE':
                       cmd  = cmd + " FOR FASTLOAD TABLE "
                    cmd  = cmd + " \"%s\".\"%s\" \n"               % (user, table)
                    cmd  = cmd + " OUTSTREAM PIPE '%s'         \n" % ('\\\\.\\PIPE\\%s_%s_%s.dtl' % ( sapdb, user, table ) )
                    cmd  = cmd + " OUTSTREAM PIPE '%s' %s   %s   " % ('\\\\.\\PIPE\\%s_%s_%s.dat' % ( sapdb, user, table ), self.options.streamformat, self.options.streamcharencoding )
                    cmd  = cmd + self.package.column_field_long
             self.cmd = cmd
             self.cmd_execute()       
          
          if TRANSFORMATIONSYNTAX == 'UNIFIED':
             self.cmd = self.package.transformation.exportstatement
             self.cmd_execute()       

      def cmd_import_table_package(self):
          sapdb  = self.options.sourcedbname
          user   = self.package.user
          table  = self.package.table

          if TRANSFORMATIONSYNTAX == 'STRATIFIED':
             #Medium
             if self.options.streammedium  == 'FILE':
                #Format
                if( self.options.streamformat == 'PAGES'   or
                    self.options.streamformat == 'RECORDS'    ) :
                    cmd = 'TABLELOAD TABLE '
                    cmd = cmd + " \"%s\".\"%s\" \n"                % (user, table)
                    cmd = cmd + " DATA INSTREAM FILE '%s' %s %s\n" % (os.path.join(self.options.sourceindepdatapath,   '%s_%s_%s.data' % (sapdb,user,table )),self.options.streamformat,self.options.streamcharencoding)
                    self.cmd = cmd
                    self.cmd_execute()
                #Format
                if( self.options.streamformat == 'DDL'        or
                    self.options.streamformat == 'FORMATTED'  or
                    self.options.streamformat == 'COMPRESSED'   ):
                    for cmd in self.package.transformation_step :
                        self.cmd = cmd
                        self.cmd_execute()
                        #grant and execute
                        if string.find(cmd,'Create') == 0 and string.find(cmd,'Table')  == 0 :
                           self.cmd= 'GRANT ALL ON %s TO PUBLIC' % table
             #Medium
             if(self.options.streammedium[0:4] == 'PIPE' ):
                #Format
                if( self.options.streamformat == 'PAGES'   or
                    self.options.streamformat == 'RECORDS') :
                    cmd      = 'TABLELOAD TABLE '
                    cmd      = cmd + " \"%s\".\"%s\" \n"               % ( user, table )
                    cmd      = cmd + " DATA INSTREAM PIPE '%s' %s %s " % ('\\\\.\\PIPE\\%s_%s_%s.dat' % ( sapdb, user, table ), self.options.streamformat, self.options.streamcharencoding)
                    self.cmd = cmd
                    self.cmd_execute()
                #Format
                if( self.options.streamformat == 'DDL'       or
                    self.options.streamformat == 'FORMATTED' or
                    self.options.streamformat == 'COMPRESSED'   ) :
                    for cmd in self.package.transformation_step :
                        self.cmd = cmd
                        self.cmd_execute()
                        #grant and execute
                        if string.find(cmd,'Create') == 0 and string.find(cmd,'Table')  == 0 :
                           self.cmd= 'GRANT ALL ON %s TO PUBLIC' % table
             self.package.transformation_step = []

          if TRANSFORMATIONSYNTAX == 'UNIFIED':
             self.cmd = self.package.transformation.importstatement
             self.cmd_execute()    
             
          if self.options.streamformat != 'PAGES' and self.options.streamformat_target == 'PAGE' :                 
             self.package.transformation.cmd_import_table_package_index()         

      def cmd_load_package(self):
          if PACKAGEEXECUTION == 'EXECUTE':
             try:
                 self.packagestart = time.time()
                 match_0     = None
                 match_1     = None
                 match_2     = None
                 match_3     = None
                 user        = None
                 packageguid = None
                 if  self.transformationtype == 'EXPORT' or self.transformationtype == 'IMPORT' :
                     protocolfile = os.path.join(LOADERPROGPATH,'sdb','loader','log',self.protocol )
                 if  os.path.isfile      ( protocolfile ) :
                     protocol      = file( protocolfile ,'r')
                     protocolline  = protocol.readline()
                     while protocolline and self.packageload  == 'FALSE' :
                           #timestamp user
                           if match_0 == None:
                              if  'TRANSPORT' == self.transformationtype or 'EXPORT' == self.transformationtype:
                                   if self.options.transformationpackage != self.options.transformationpart:
                                      user = self.package.user
                                   else:
                                      user = re.split(',',self.options.sourcedbuser)[0]
                                   match_0  = re.search ('TRANSFORMATIONPACKAGE  %s' % self.package.sourcepackagename, protocolline )
                              if  'TRANSPORT' == self.transformationtype or 'IMPORT' == self.transformationtype :
                                   if self.options.transformationpackage != self.options.transformationpart:
                                      user = self.package.user
                                   else:
                                      user = re.split(',',self.options.targetdbuser)[0]
                                   match_0  = re.search ('TRANSFORMATIONPACKAGE  %s' % self.package.targetpackagename, protocolline )
                           #user
                           if match_0 != None and match_1 == None:
                              match_1  = re.search ('USE USER %s' % user, protocolline )
                           #table
                           if match_1 != None and match_2 == None:
                              if self.options.transformationpart == 'DB' :
                                 match_2  = re.search ('PORT DB'  ,protocolline)
                              if self.options.transformationpart == 'USER' :
                                 match_2  = re.search ('PORT USER',protocolline)
                              if self.options.transformationpart == 'TABLE' :   
                                 match_2  = re.search ('PORT TABLE "%s"' % self.package.table, protocolline) 
                           if match_2 != None and match_3 == None:
                              match_3  = re.search ('LOAD TRANSFORMATIONPACKAGE x', protocolline)
                           #package
                           if match_3 != None :
                              packageguid       = re.split ( '\'', protocolline)
                              packageguid       = string.replace( packageguid[1] , "\n" , '')
                              self.packageguid  = packageguid
                              self.packageload  = 'TRUE'
                              resultset = self.sqlsession.sql(PACKAGEMAXSTEP % self.packageguid )
                              if resultset != None:
                                 row = resultset.first()
                                 if row != None:
                                    self.package.maxstepid = row[0] 
                              if self.options.mode == 'INTER':
                                 print "Load      Transformation Package x'%s'" % self.packageguid
                           protocolline = protocol.readline()
                     protocol.close()
             except sql.CommunicationError, err:
                 pass
             except sql.SQLError,  err :
                 pass
             except IndexError,err:
                 pass
           
      def cmd_monitor_package(self):
          if MONITOR == 'ON':
             try:
                 if self.options.streamformat == 'PAGES' or  self.options.streamformat_target == 'PAGE':
                    pass
                 if self.options.streamformat != 'PAGES' and self.options.streamformat_target != 'PAGE' :
                    if self.options.transformationpart == 'TABLE':
                       self.table = self.package.table
                    if self.package.stepid == None:
                       resultset  = self.sqlsession.sql(PACKAGEROWCOUNT     % self.packageguid )
                       if resultset != None:
                          row = resultset.first()
                          if row != None:
                             self.package.stepid = row[0] 
                             self.owner          = row[1] 
                             if self.options.transformationpart != 'TABLE':
                                self.table       = row[2]
                    if self.package.stepid != None:
                       resultset = self.sqlsession.sql(STEPIDROWCOUNT      % (self.packageguid, self.package.stepid ))
                       row = resultset.first()
                       if row != None:
                          if self.package.stepid == row[0]:
                             print '\rTable     %-10s : %s  ' %( self.table, row[3]),
                       if row == None and self.package.maxstepid != self.package.stepid:
                          resultset = self.sqlsession.sql(STEPIDSROWCOUNT  % (self.packageguid, self.package.stepid ))
                          if resultset != None:
                             for row in resultset:
                                 if row != None and self.package.stepid == row[0]:
                                    print '\rTable     %-10s : %s ' %( row[2], row[3])  
                                 if row != None and self.package.stepid != row[0]:
                                    print 'Table     %-10s : %s '   %( row[2], row[3])
                          if self.package.stepid    < self.package.maxstepid:
                             self.package.maxstepid = self.package.stepid
                             self.package.stepid    = None
             except sql.CommunicationError, err:
                 self.package.tranformation.release()
                 os._exit(1)
             except sql.SQLError,  err :
                 pass
             except :
                 pass
          if MONITOR == 'OFF':
             pass
     
      def cmd_unload_package(self):
          TRANSFORMATIONSYNTAX = 'STRATIFIED' 
          
          if self.packageend > 0 and ( time.time() - self.packagestart > self.packageend ):
             print 'Processing Time exceeded; stopping execution'
             self.package.tranformation.release()
             os._exit(1)
             
          if TRANSFORMATIONSYNTAX == 'UNIFIED' and PACKAGEEXECUTION == 'EXECUTE' :
             catalogstream_name = None
             datastream_name    = None
             packagestream_name = None
             catalogstream_lock = None
             datastream_lock    = None
             packagestream_lock = None
             if  'EXPORT' == self.transformationtype :
                 catalogstream_name    = string.replace(self.catalogstream_name,"'","")
                 datastream_name       = string.replace(self.datastream_name   ,"'","")
                 packagestream_name    = string.replace(self.packagestream_name,"'","")
             if  'IMPORT' == self.transformationtype :
                 catalogstream_name    = string.replace(self.catalogstream_name,"'","")
                 datastream_name       = string.replace(self.datastream_name   ,"'","")
                 packagestream_name    = string.replace(self.packagestream_name,"'","")
                 catalogstream_name    = string.replace(catalogstream_name, self.options.sourcedbname, self.options.targetdbname)
                 datastream_name       = string.replace(datastream_name   , self.options.sourcedbname, self.options.targetdbname)
                 packagestream_name    = string.replace(packagestream_name, self.options.sourcedbname, self.options.targetdbname)
             try:
                if self.options.streamformat == 'DDL' or self.options.streamformat == 'XML':
                   if os.path.isfile           (catalogstream_name) :
                      catalogstream = os.rename(catalogstream_name, catalogstream_name)
                      catalogstream_lock = 'release'
                      datastream_lock    = 'release' 
                      packagestream_lock = 'release'
                if(self.options.streamformat == 'COMPRESSED' or self.options.streamformat == 'FORMATTED' or
                   self.options.streamformat == 'RECORDS'    or self.options.streamformat == 'PAGES'    ):
                   if os.path.isfile           (catalogstream_name) :
                      catalogstream = os.rename(catalogstream_name, catalogstream_name) 
                      catalogstream_lock = 'release'
                   if os.path.isfile        (datastream_name) : 
                      datastream = os.rename(datastream_name, datastream_name)
                      catalogstream_lock = 'release' 
                      datastream_lock    = 'release' 
                   if os.path.isfile        ( string.replace(datastream_name,'.DATA','.DATA0000')) : 
                      datastream = os.rename( string.replace(datastream_name,'.DATA','.DATA0000') ,string.replace(datastream_name,'.DATA','.DATA0000'))
                      catalogstream_lock = 'release' 
                      datastream_lock    = 'release'
                   if os.path.isfile           (packagestream_name) :      
                      packagestream = os.rename(packagestream_name, packagestream_name)
                      catalogstream_lock = 'release' 
                      datastream_lock    = 'release' 
                      packagestream_lock = 'release'
                if( catalogstream_lock == 'release' and 
                    datastream_lock    == 'release' and 
                    packagestream_lock == 'release'    ):
                    self.packageunload  = 'TRUE'
                    if  DB_DEFAULTCODE_SOURCE == 'ASCII' and DB_DEFAULTCODE_TARGET == 'ASCII' :
                        try:
                           packagefile = file (packagestream_name )
                           resultset   = csv.reader(packagefile)
                           if resultset != None:
                              print 'Transformation Package Rows   :'
                              for row in resultset:
                                  if MONITOR == 'ON':
                                     if self.package.maxstepid < string.atoi( row[12], 10 ) :
                                        print 'Table     %-10s : %s ' % (row[1], row[33])
                                  if MONITOR == 'OFF':
                                     print    'Table     %-10s : %s ' % (row[1], row[33])
                        except:
                          pass
                    if not ( 'EXPORT'  == self.transformationtype and 'PIPE'  == self.streammedium ):
                       if  mutex.locked()  == True :
                           mutex.release()           #release
             except OSError, (errno, err ):
                catalogstream_lock = None
                datastream_lock    = None
                packagestream_lock = None
                pass
                
          if TRANSFORMATIONSYNTAX == 'STRATIFIED' and PACKAGEEXECUTION == 'EXECUTE':
             match_0      = None
             match_1      = None
             match_2      = None
             match_3      = None
             match_4      = None
             packageguid  = None
             user         = None
             try:
                 if  'EXPORT' == self.transformationtype  or 'IMPORT' == self.transformationtype :
                     protocolfile = os.path.join(LOADERPROGPATH,'sdb','loader','log',self.protocol )
                 if  os.path.isfile  ( protocolfile ) :
                     protocol  = file( protocolfile ,'r')
                     protocolline  = protocol.readline()
                     counter       = 0
                     while protocolline and self.packageunload == 'FALSE':
                           #timestamp
                           if match_0 == None:
                              if  'TRANSPORT' == self.transformationtype or 'EXPORT' == self.transformationtype:
                                  if self.options.transformationpackage != self.options.transformationpart:
                                     user = self.package.user
                                  else:
                                     user = re.split(',',self.options.sourcedbuser)[0]
                                  match_0  = re.search ('TRANSFORMATIONPACKAGE  %s' % self.package.sourcepackagename, protocolline )
                              if  'TRANSPORT' == self.transformationtype or 'IMPORT' == self.transformationtype :
                                  if self.options.transformationpackage != self.options.transformationpart:
                                     user = self.package.user
                                  else:
                                     user = re.split(',',self.options.targetdbuser)[0]
                                  match_0  = re.search ('TRANSFORMATIONPACKAGE  %s' % self.package.targetpackagename, protocolline )
                           #user                                  
                           if match_0 != None and match_1 == None:
                              match_1  = re.search ('USE USER %s' % user, protocolline )
                           #part
                           if match_1 != None and match_2 == None :
                              match_2  = re.search ("LOAD TRANSFORMATIONPACKAGE x'%s'" % self.packageguid , protocolline)
                           if match_2 != None and match_3 == None :
                              #monitor
                              if self.options.transformationpart != 'TABLE' :
                                 match_3 = 'ok'
                              if self.options.transformationpart == 'TABLE' :
                                 if self.options.streamformat    == 'PAGES':
                                    match_3 = 'ok'
                                 if self.options.streamformat    == 'RECORDS' or self.options.streamformat == 'COMPRESSED' or self.options.streamformat == 'FORMATTED' :
                                    match_3  = re.search ("Sum of " , protocolline)
                                    if match_3 != None:
                                       packagerows        = re.split (":",protocolline)[1]
                                       packagerows        = re.split (",",packagerows )[0]
                                       print '\rTable     %-10s : %s  \n' %( self.package.table, packagerows),
                                 if self.options.streamformat    == 'DDL' or self.options.streamformat  == 'XML' or self.options.streamformat_target == 'DTL':
                                    match_3 = 'ok'
                           if match_3 != None and match_4 == None :
                              #package
                              if 'EXPORT'  == self.transformationtype and 'TAPE'  == self.streammedium :
                                  #format
                                  if self.streamformat    == 'DDL' :
                                     if protocolline[0:8] == '// M  NST':
                                          if counter == 0:
                                             ddlpipe_ebid = re.split  ('NST',protocolline)
                                             ddlpipe_ebid = string.replace(ddlpipe_ebid[1], "\n" , '')
                                             self.ddlpipe_ebid = "\'NST%s\'" % ddlpipe_ebid
                                  if self.streamformat    ==  'RECORDS' or self.streamformat ==  'PAGES':
                                     if protocolline[0:8] == '// M  NST':
                                        if counter == 0:
                                           ddlpipe_ebid = re.split  ('NST',protocolline)
                                           ddlpipe_ebid = string.replace(ddlpipe_ebid[1], "\n" , '')
                                           self.ddlpipe_ebid = "\'NST%s\'" % ddlpipe_ebid
                                        if counter == 1:
                                           datpipe_ebid = re.split  ('NST',protocolline)
                                           datpipe_ebid = string.replace(datpipe_ebid[1], "\n" , '')
                                           self.datpipe_ebid = "\'NST%s\'" % datpipe_ebid
                                        if counter == 2:
                                           mappipe_ebid = re.split  ('NST',protocolline)
                                           mappipe_ebid = string.replace(mappipe_ebid[1], "\n" , '')
                                           self.mappipe_ebid = "\'NST%s\'" % mappipe_ebid
                                        counter  = counter + 1
                              match_4  = re.search ("UNLOAD TRANSFORMATIONPACKAGE x'%s'" % self.packageguid , protocolline)
                           #rows
                           if match_4 != None :
                              self.packageunload = 'TRUE'
                              #monitor
                              package = None
                              if self.options.transformationtype == 'EXPORT' or self.options.transformationtype == 'TRANSPORT':
                                 if self.options.transformationpart == 'DB' :
                                    package = os.path.join(self.options.sourceindepdatapath,'%s.package' % self.options.sourcedbname)
                                 if self.options.transformationpart == 'USER' :
                                    package = os.path.join(self.options.sourceindepdatapath,'%s.package' % re.split(',', self.options.sourcedbuser)[0])
                              if self.options.transformationtype == 'IMPORT' or self.options.transformationtype == 'TRANSPORT':
                                 if self.options.transformationpart == 'DB' :
                                    package = os.path.join(self.options.targetindepdatapath,'%s.package' % self.options.targetdbname)
                                 if self.options.transformationpart == 'USER' :
                                    package = os.path.join(self.options.targetindepdatapath,'%s.package' % re.split(',', self.options.targetdbuser)[0])
                              if((self.options.transformationpart == 'DB'    or  self.options.transformationpart == 'USER') and self.options.streamformat != 'PAGES' and
                                  DB_DEFAULTCODE_SOURCE == 'ASCII' and DB_DEFAULTCODE_TARGET == 'ASCII' ) :
                                 if os.path.isfile(package) :
                                    packagefile = file (package )
                                    resultset   = csv.reader(packagefile)
                                    if resultset != None:
                                       print 'Transformation Package Rows   :'
                                       for row in resultset:
                                           if MONITOR == 'ON':
                                              if self.package.maxstepid < string.atoi( row[12], 10 ) :
                                                 print 'Table     %-10s : %s ' % (row[1], row[33])
                                           if MONITOR == 'OFF':
                                              print    'Table     %-10s : %s ' % (row[1], row[33])
                              #mutex
                              if not ( 'EXPORT'  == self.transformationtype and 'PIPE'  == self.streammedium ):
                                 if  mutex.locked()  == True :
                                     mutex.release()           #release
          
                           protocolline = protocol.readline()
                     protocol.close()
             except IOError,    (errno, err ):
                pass
                
             TRANSFORMATIONSYNTAX = 'UNIFIED'

      def cmd_restart_package(self):
          restartRecord = []
          if self.startposition == 'RESTART' :
             statement     =  " SELECT DISTINCT PACKAGEGUID, OWNER, TABLENAME, \n DATASTREAMHEADERBLOCK, DATASTREAMHEADERPOS, \n DATASTREAMDATABLOCK, DATASTREAMDATAPOS, ROWCOUNT \n"
             statement     =  statement + "FROM SYSDBA.TRANSFORMATIONMODEL "
             statement     =  statement + "WHERE RESTART = TRUE "

             result = self.sqlsession.sql (statement )
             row    = result.next ()
             if row :
                packageguid = ('%02X' * 24) % tuple (map (ord, row[0] ))
                restartRecord.append (packageguid)
                restartRecord.append (row[1])
                restartRecord.append (row[2])
                restartRecord.append (row[3])
                restartRecord.append (row[4])
                restartRecord.append (row[5])
                restartRecord.append (row[6])
                restartRecord.append (row[7])

          return restartRecord

      def cmd_restart_sourcedb_package(self, targetRecord ):
          #Medium Format Start
          if  self.startposition == 'RESTART' and self.streammedium == 'PIPE' and self.streamformat == 'RECORDS' and targetRecord <> [] :
              sourceRecord  = self.cmd_restart_package()
              statement     = " UPDATE SYSDBA.TRANSFORMATIONMODEL SET \n"
              statement     = statement + " DATASTREAMHEADERBLOCK = %s, DATASTREAMHEADERPOS=%s, \n DATASTREAMDATABLOCK=%s,DATASTREAMDATAPOS=%s,\n ROWCOUNT=%s " % ( targetRecord[3] , targetRecord[4], targetRecord[5], targetRecord[6], targetRecord[7] )
              statement     = statement + " WHERE PACKAGEGUID= x'%s' \n AND OWNER = '%s' AND TABLENAME = '%s' AND RESTART = TRUE \n"                            % ( sourceRecord[0], targetRecord[1], targetRecord[2])
              print  statement
              self.sqlsession.sql (statement )
          #Medium Format Start
          if  self.startposition == 'RESTART' and self.streammedium == 'PIPE' and self.streamformat == 'PAGES'   and targetRecord <> [] :
              pass
          #Medium Format Start
          if  self.startposition == 'RESTART' and self.streammedium == 'FILE' and self.streamformat == 'RECORDS' and targetRecord <> [] :
              pass
          #Medium Format Start
          if  self.startposition == 'RESTART' and self.streammedium == 'FILE' and self.streamformat == 'PAGES'   and targetRecord <> [] :
              pass

      def cmd_restart_targetdb_package(self):
          #Medium Format Start
          if  self.startposition == 'RESTART' and self.streammedium == 'PIPE' and self.streamformat == 'RECORDS':
              restartRecord = self.cmd_restart_package()
          #Medium Format Start
          if  self.startposition == 'RESTART' and self.streammedium == 'PIPE' and self.streamformat == 'PAGES'  :
              restartRecord = self.cmd_restart_package()

              statement     = " UPDATE SYSDBA.TRANSFORMATIONMODEL SET \n"
              statement     = statement + " DATASTREAMHEADERBLOCK = 0, DATASTREAMHEADERPOS=0, \n DATASTREAMDATABLOCK=0,DATASTREAMDATAPOS=0,\n ROWCOUNT=0, RESTART = TRUE \n"
              statement     = statement + " WHERE PACKAGEGUID= x'%s' \n AND OWNER = '%s' AND TABLENAME = '%s' \n"  % (restartRecord[0], restartRecord[1], restartRecord[2])
              print statement
              result        = self.sqlsession.sql (statement)
              statement     = " DELETE %s.%s \n"  % (restartRecord[1], restartRecord[2] )
              print statement
              result        = self.sqlsession.sql (statement)
          #Medium Format Start
          if  self.startposition == 'RESTART' and self.streammedium == 'FILE' and self.streamformat == 'RECORDS':
              restartRecord = self.cmd_restart_package()
          #Medium Format Start
          if  self.startposition == 'RESTART' and self.streammedium == 'FILE' and self.streamformat == 'PAGES'  :
              restartRecord = self.cmd_restart_package()

          return restartRecord
          
#class StreamLoader 

class StreamConnector :
      def __init__(self, source_loader = None, target_loader = None ) :
          self.sourceloader   = source_loader
          self.targetloader   = target_loader
          if self.sourceloader != None:
             self.user           = self.sourceloader.package.user
             self.table          = self.sourceloader.package.table
          if self.targetloader != None:
             self.user           = self.targetloader.package.user
             self.table          = self.targetloader.package.table
          self.longcolumn     = ''
          self.sourceddltime  = None
          self.sourcedattime  = None
          self.ddlpipe_exist  = 1
          self.datpipe_exist  = 1

      def cmd_datpipe2datfile(self):
          datfile = os.path.join (self.sourceloader.sapdbroot, '%s_%s.dat'         % ( self.sourceloader.package.user, self.sourceloader.package.table))
          datpipe2file    = "pipe2file -d pipe2file -p %s           -f %s"         % ( self.sourceloader.datpipe, datfile )
          try:
             print 'Start     StreamConnector to Connect \n FILE %s to\n PIPE %s ' % ( self.sourceloader.datpipe, datfile )
             os.system( datpipe2file )
          except OSError, err:
             print 'command failed:', err

      def cmd_longpipe2longfiles(self):
          for longcolumn in self.sourceloader.package.column_long:
              self.longcolumn = longcolumn
              thread.start_new_thread( self.cmd_longpipe2longfile,())
              time.sleep(1)

      def cmd_longpipe2longfile(self):
          longpipe  = '\\\\.\\PIPE\\%s_%s_%s_%s.long'                                                                                            % (self.sourceloader.package.sapdb, self.user ,self.table, self.longcolumn )
          longfile  = os.path.join (self.sourceloader.sapdbroot, '%s_%s_%s.long' % (self.user, self.table, self.longcolumn ))
          longpipe2file = "pipe2file -d pipe2file -p %s -f %s" % ( longpipe , longfile )
          try:
              print 'Start     StreamConnector to Connect \n FILE %s to\n PIPE %s ' % ( longfile, longpipe )
              os.system( longpipe2file )
          except OSError, err:
              print 'command failed:', err

      def cmd_datfile2datpipe(self):
          print '\nStart     StreamConnector to Connect\n FILE %s to\n PIPE %s '    % ( self.sourceloader.datfile, self.sourceloader.datpipe )
          datfile2pipe    = "pipe2file -d file2pipe -p %s           -f %s"          % ( self.sourceloader.datpipe, self.sourceloader.datfile )
          try:
             os.system( datfile2pipe )
          except OSError, err:
             print 'command failed:', err

      def cmd_pipe2tape(self):
          counter = 0
          protocolfile = os.path.join(self.sourceloader.sapdbroot, 'wrk',self.sourceloader.protocol)
          if  os.path.isfile     ( protocolfile ) :
              protocol     = file( protocolfile ,'r')

              if self.sourceloader.streamformat <> 'DDL' :
                 protocolline = protocol.readline()
                 while protocolline :
                     if ( protocolline[0:8] == '// M    NST'):
                          if counter == 0:
                             ddlpipe_ebid = re.split  ('NST',protocolline)
                             ddlpipe_ebid = string.replace(ddlpipe_ebid[1] , "\n" , '')
                             self.targetloader.ddlpipe_ebid = "\'NST%s\'" % ddlpipe_ebid
                          if counter == 1:
                             datpipe_ebid = re.split  ('NST',protocolline)
                             datpipe_ebid = string.replace(datpipe_ebid[1] , "\n" , '')
                             self.targetloader.datpipe_ebid = "\'NST%s\'" % datpipe_ebid
                          counter  = counter + 1
                     protocolline = protocol.readline()
                 if counter == 2:
                    if mutex.locked() == True:
                       mutex.release() #release

              if self.sourceloader.streamformat == 'DDL' :
                 protocolline = protocol.readline()
                 while protocolline :
                     if ( protocolline[0:8] == '// M    NST'):
                          if counter == 0:
                             ddlpipe_ebid = re.split  ('NST',protocolline)
                             ddlpipe_ebid = string.replace(ddlpipe_ebid[1] , "\n" , '')
                             self.targetloader.ddlpipe_ebid = "\'NST%s\'" % ddlpipe_ebid
                          counter  = counter + 1
                     protocolline = protocol.readline()
                 if counter == 1:
                    if mutex.locked() == True:
                       mutex.release() #release

      def cmd_ddlpipe2ddltape(self):
          print '\nStart     StreamConnector to Connect \n TAPE to\n PIPE %s ' % ( self.sourceloader.ddlpipe_source )
          ddltapesave  = self.sourceloader.sapdbroot + "\\wrk\\" + self.sourceloader.sapdb + ".ddltape_save"
          ddlpipe2tape = "\nsave    -s %s -N %s -l full -LL %s 2> %s"          % ( self.sourceloader.sapdbnode, self.sourceloader.sapdb,
                                                                                   self.sourceloader.ddlpipe_source, ddltapesave )
          try:
             os.system( ddlpipe2tape )
          except OSError, err:
             print 'command failed:', err
          if  os.path.isfile  ( ddltapesave ) :
              tapefile  = file( ddltapesave ,'r')
              tapeline  = tapefile.readline()
              while tapeline  :
                  if ( tapeline[0:19] == 'completed savetime='):
                       savetime = re.split ('=',tapeline)
                       savetime = string.replace( savetime[1] , "\n" , '')
                       self.sourceddltime = savetime
                  tapeline = tapefile.readline()
              tapefile.close()

      def cmd_ddltape2ddlpipe(self):
          print '\nStart     StreamConnector to Connect \n TAPE %s to\n PIPE %s' % ( self.sourceddltime, self.sourceloader.ddlpipe_source )
          ddltape2pipe = "\nrecover -q -s %s -a -t %s -c %s -iY %s "             % ( self.sourceloader.sapdbnode, self.sourceddltime,
                                                                                     self.sourceloader.sapdbnode, self.sourceloader.ddlpipe_source )
          try:
              os.system( ddltape2pipe )
          except OSError, err:
              print 'command failed:', err

      def cmd_datpipe2dattape(self):
          print 'Start     StreamConnector to Connect \n TAPE to\n PIPE %s ' %  self.sourceloader.datpipe_source
          dattapesave  = self.sourceloader.sapdbroot + "\\wrk\\" + self.sourceloader.sapdb + ".dattape_save"
          datpipe2tape = "save    -s %s -N %s -l full -LL %s           2> %s"% ( self.sourceloader.sapdbnode, self.sourceloader.sapdb,
                                                                                 self.sourceloader.datpipe_source , dattapesave  )
          try:
             os.system( datpipe2tape )
          except OSError, err:
             print 'command failed:', err
          if  os.path.isfile  ( dattapesave ) :
              tapefile  = file( dattapesave ,'r')
              tapeline  = tapefile.readline()
              while tapeline :
                  if ( tapeline[0:19] == 'completed savetime='):
                       savetime = re.split ('=',tapeline)
                       savetime = string.replace( savetime[1] , "\n" , '')
                       self.sourcedattime = savetime
                  tapeline = tapefile.readline()
              tapefile.close()

      def cmd_dattape2datpipe(self):
          print '\nStart     StreamConnector to Connect\n TAPE %s to\n PIPE %s ' % ( self.sourcedattime, self.sourceloader.datpipe_source )
          dattape2pipe = "\nrecover -q -s %s -a -t %s -c %s -iY %s       "       % ( self.sourceloader.sapdbnode, self.sourcedattime,
                                                                                     self.sourceloader.sapdbnode, self.sourceloader.datpipe_source )
          try:
              os.system( dattape2pipe )
          except OSError, err:
              print 'command failed:', err

class StreamFilter :
      def __init__(self, options = None, transformation = None ) :
          self.options        = options
          self.transformation = transformation
          self.ddlerror       = 1
          self.dtlerror       = 1
          self.daterror       = 1

      def cmd_dtlsource2dtltarget(self):
          dtlsource    = None
          if self.options.streammedium == 'FILE':
             dtlsource = os.path.join(self.options.sourceindepdatapath,  '%s_%s_%s.dtl'  % (self.options.sourcedbname,re.split(',',self.options.sourcedbuser)[0],self.transformation.package.table ))
          if self.options.streammedium == 'PIPE':
             dtlsource = '\\\\.\\PIPE\\%s_%s_%s.dtl'  % (self.sourceloader.sapdb,self.sourceloader.user,self.sourceloader.package.table )
          try :
             self.dtlerror   = 0
             step            = ''
             dtlsourcestream = file(dtlsource, 'r' )
             sourcedtl       = dtlsourcestream.readline()
             self.targetloader.package.transformation_step = []
             while sourcedtl :
                   targetdtl = string.strip(sourcedtl)
                   if ( targetdtl[0:2] <> "//" ) and targetdtl <> '' :
                        string.replace(targetdtl, 'ASCII', self.options.streamcharencoding)
                        step = step + targetdtl + "\n"
                   if ( targetdtl[0:2] == "//" or targetdtl == '' )  and step <> '' and step <> ' ':
                        self.targetloader.package.transformation_step.append(step)
                        step= ''
                   sourcedtl = dtlsourcestream.readline()
             dtlsourcestream.close()
          except IOError, (errno, err ):
             self.dtlerror  = 1

      def cmd_ddlsource2ddltarget(self):
          try :
              self.ddlerror   = 0
              ddlpipesource   = file(self.transformation.sourceloader.ddlpipe_source, 'rt' )
              ddlpipetarget   = file(self.transformation.targetloader.ddlpipe_target, 'wt' )
              sourceddl       = ddlpipesource.readline()
              while sourceddl :
                    targetddl = string.replace(sourceddl, '%s' % string.upper(self.transformation.sourceloader.catalog),
                                                          '%s' % string.upper(self.transformation.targetloader.catalog) )
                    ddlpipetarget.write(targetddl)
                    sourceddl = ddlpipesource.readline()
              ddlpipesource.close()
              ddlpipetarget.close()
          except IOError, (errno, err ):
              self.ddlerror  = 1
              print 'command failed:', errno, err

      def cmd_datsource2dattarget_bin(self):
          try :
             self.daterror   = 0
             datpipesource   = file(self.sourceloader.datpipe_source, 'rb')
             datpipetarget   = file(self.sourceloader.datpipe_target, 'wb')
             sourcedat       = datpipesource.read(400)
             while sourcedat :
                   targetdat = sourcedat
                   datpipetarget.write(targetdat)
             sourcedat = datpipesource.read(400)
             datpipesource.close()
             datpipetarget.close()
          except IOError, (errno, err ):
             print 'command failed:', errno, err
             self.daterror  = 1

      def cmd_datsource2dattarget_txt(self):
          try :
             self.daterror   = 0
             datpipesource   = file(self.sourceloader.datpipe_source, 'rt')
             datpipetarget   = file(self.sourceloader.datpipe_target, 'wt')
             sourcedat       = datpipesource.readline()
             while sourcedat :
                   targetdat = sourcedat
                   datpipetarget.write(targetdat)
             sourcedat = datpipesource.read()
             datpipesource.close()
             datpipetarget.close()
          except IOError, (errno, err ):
             print 'command failed: %s, %s \n' %( errno, err )
             self.daterror  = 1
             
def  design_package(options):
     global LOADERPROGPATH
     global LOADERDATAPATH
     global MONITOR
     global PACKAGEARCHIVE
     rc   = True
     # Transformation Package Version
     
     loaderdatapath = ''
     packagename    = ''
     sqlsource      = None
     sqltarget      = None
     username       = ''
     userpassword   = ''
     table          = None
     input          = None 
     try:
        if options.mode == 'INTER':
           print "\nTransport, export or import data and metadata using the" 
           print "transformation service provided by the database loader. "
           print "\nThe loader transport wizard guides you through the "
           print "steps to define and execute a transformation package."
        global STEPID
        input = '' 
        if options.transformationtype in ['']:
           STEPID = STEPID+1
           print "\nStep %i: Choose the transformation type:" % STEPID
        while options.transformationtype in [ '' ]:
              input = raw_input(' [1] TRANSPORT\n [2] EXPORT\n [3] IMPORT\n  \nStep %i: ' %STEPID )
              if input == '1':
                 options.transformationtype = 'TRANSPORT'
              if input == '2':
                 options.transformationtype = 'EXPORT'
              if input == '3':
                 options.transformationtype = 'IMPORT'
        input = '' 
        if options.mode == 'INTER' and PACKAGEPATH != 'HOMEDIR' :
           STEPID = STEPID+1
           print "\nStep %i: Choose the transformation package path: " % STEPID
           print " Transformation Package Path: %s " % LOADERDATAPATH
           while loaderdatapath in ['']:
                 input = raw_input(' [1] DEFAULT \n [2] CHANGE DEFAULT \n\nStep %i: ' % STEPID)
                 if input == '1':
                    loaderdatapath = LOADERDATAPATH
                 if input == '2':
                    input = None
                    while input in [None]:
                          input = raw_input(' transformation package path: ' )
                          if input != ''    and os.path.isdir  (input):
                             LOADERDATAPATH   = os.path.abspath(input)    
                             loaderdatapath   = LOADERDATAPATH
        else:
           loaderdatapath = LOADERDATAPATH
        input = ''
        if options.mode == 'INTER' and PACKAGENAME != 'TIMESTAMP':
           if packagename in ['']:                  
              STEPID = STEPID+1
              print "\nStep %i: Enter the transformation package name: " % STEPID
              print " Transformation Package Name: %s " % PACKAGETIMESTAMP
           while packagename in ['']:
                 input = raw_input(' [1] DEFAULT \n [2] CHANGE DEFAULT \n\nStep %i: ' % STEPID)
                 if input == '1':
                    packagename = PACKAGETIMESTAMP
                 if input == '2':
                    input = ''
                    while input in ['']:
                          input = raw_input(' transformation package name: ' )
                          if input != '' :
                             packagename = input
        else:
           packagename = PACKAGETIMESTAMP
        
        input = '' 
        username     = ''
        userpassword = ''
        if options.sourcedbuser not in [''] :
           try:
              username     = re.split(',',options.sourcedbuser)[0]
              userpassword = re.split(',',options.sourcedbuser)[1]
              if userpassword == '*':
                 userpassword  = ''
           except:
              pass
        #DATA SOURCE  DATABASE
        if( options.transformationtype == 'TRANSPORT' or
            options.transformationtype == 'EXPORT'      ):
            if(options.sourcedbnode in [''] or 
               options.sourcedbname in [''] or 
               username             in [''] or
               userpassword         in ['']   ):
               STEPID = STEPID+1
               print "\nStep %i: Enter the data source:" % STEPID 
               if options.sourcedbnode not in [''] :
                  print 'Node : %s' % options.sourcedbnode
               if options.sourcedbname not in [''] :
                  print 'Db   : %s' % options.sourcedbname
               if username             not in [''] :
                  print 'User : %s' % username
            sqlsource        = None
            while sqlsource == None:
                  while options.sourcedbnode in ['']:   
                        input = raw_input('Node : ')
                        if input == '':
                           options.sourcedbnode = socket.gethostname()
                        if input != '':
                           options.sourcedbnode = input
                  while options.sourcedbname in ['']:   
                        input = raw_input('Db   : ') 
                        options.sourcedbname = string.upper(input)
                  while username in ['']:  
                        username = raw_input('User : ')   
                  while userpassword in ['']:   
                        userpassword   = getpass.getpass()
                  options.sourcedbuser = '%s,%s' %( username, userpassword )
                  if options.sourcedbuser not in ['']:
                     username     = re.split(',',options.sourcedbuser)[0]
                     userpassword = re.split(',',options.sourcedbuser)[1]
                  if username != '' and username [0] != '"' and username[len(username)-1] != '"':
                     username  = string.upper(username)
                  elif username != '' :
                     username = string.replace(username,'"','')  
                  options.sourcedbuser = '%s,%s' %( username, userpassword )  
                  try:
                      sqlsource = sql.connect( username, userpassword, options.sourcedbname, options.sourcedbnode )
                  except (sql.CommunicationError ) , err:
                      print 'Connect failed, try again:' , err
                      sqlsource    = None 
                  except (sql.SQLError ) , err:  
                      print 'Connect failed, try again:' , err    
                      sqlsource    = None 
                  if sqlsource    == None:
                     options.sourcedbnode = ''
                     options.sourcedbname = ''
                     options.sourcedbuser = ''
                     username     = ''
                     userpassword = ''
            #DATA PATH
            try:
                options.sourceindepdatapath = os.path.join(LOADERDATAPATH, 'sdb','loader', 'data', 'export', options.sourcedbnode, options.sourcedbname, packagename )
                if os.path.isdir(options.sourceindepdatapath) == False:
                   os.makedirs  (options.sourceindepdatapath)
            except EnvironmentError, err:
                print err

        #DATA SOURCE  PACKAGE
        if( options.transformationtype == 'IMPORT' ):
            if(options.sourcedbnode          == '' or  
               options.sourcedbname          == '' or  
               options.transformationpackage == ''  ):
               STEPID = STEPID+1
               print "\nStep %i: Enter the data source:" % STEPID 
            sqlsource        = None
            while sqlsource == None:
                  input = ''
                  while options.sourcedbnode in ['']:   
                        input = raw_input('Node : ')
                        if input == '':
                           input  = socket.gethostname()
                        options.sourcedbnode = input
                  input = '' 
                  while options.sourcedbname in ['']:   
                        input = raw_input('Db   : ') 
                        options.sourcedbname = string.upper(input)
                  #DATA PATH
                  try:
                     options.sourceindepdatapath = os.path.join(LOADERDATAPATH, 'sdb','loader', 'data', 'export', options.sourcedbnode, options.sourcedbname)
                  except EnvironmentError, err:
                      print err
                  if os.path.isdir(options.sourceindepdatapath) == True :
                     if options.transformationpackage  == '':
                        if PACKAGEARCHIVE == 'ZIP' or PACKAGEARCHIVE == 'TAR':
                           count    =  0
                           packages = []
                           # PACKAGE COLLECTION       
                           for package in os.listdir( options.sourceindepdatapath):
                               if os.path.isfile(os.path.join(options.sourceindepdatapath,package)):
                                  if PACKAGEARCHIVE == 'ZIP' and zipfile.is_zipfile(os.path.join(options.sourceindepdatapath,package)) == True:
                                     if count+1  < 10 :
                                        print " [%i]  %s" % (count+1,package)
                                     if count+1 >= 10 :
                                        print " [%i] %s"  % (count+1,package)
                                     packages.append(package)
                                     count = count + 1
                                  if PACKAGEARCHIVE == 'TAR' and tarfile.is_tarfile(os.path.join(options.sourceindepdatapath,package)) == True:
                                     if count+1  < 10 :
                                        print " [%i]  %s" % (count+1,package)
                                     if count+1 >= 10 :
                                        print " [%i] %s"  % (count+1,package)
                                     packages.append(package)
                                     count = count + 1
                           input = ''
                           # PACKAGE
                           packagedict  = dict([(packages.index(package)+1,package) for package in packages])  
                           while input == '' or string.atoi(input,10) < 0 or string.atoi(input,10) > len (os.listdir( options.sourceindepdatapath)):
                                 input  =  raw_input("\nStep %i: "  % STEPID )
                           if PACKAGEARCHIVE == 'ZIP':
                              options.sourceindepdatapath = string.replace(os.path.join(options.sourceindepdatapath,packagedict[string.atoi(input,10)]),os.extsep+'zip','')
                           if PACKAGEARCHIVE == 'TAR':
                              options.sourceindepdatapath = string.replace(os.path.join(options.sourceindepdatapath,packagedict[string.atoi(input,10)]),os.extsep+'tgz','')
                     if options.transformationpackage != '': 
                        if PACKAGEARCHIVE == 'ZIP':
                           options.sourceindepdatapath = os.path.join(options.sourceindepdatapath, options.transformationpackage)
                        if PACKAGEARCHIVE == 'TAR':
                           options.sourceindepdatapath = os.path.join(options.sourceindepdatapath, options.transformationpackage)
                     try:
                        if os.path.isdir(options.sourceindepdatapath) == False:
                           os.makedirs  (options.sourceindepdatapath)
                     except EnvironmentError, err:
                        print err
                  # PACKAGE PACKAGEARCHIVE
                  if PACKAGEARCHIVE == 'ZIP':
                     try:
                         zipdirname  = os.path.split(options.sourceindepdatapath)[0]
                         zipfilename = os.path.split(options.sourceindepdatapath)[1]+ os.extsep + 'zip'
                         zip =  zipfile.ZipFile(os.path.join (zipdirname,zipfilename), 'r', zipfile.ZIP_DEFLATED )
                         if zip.testzip() == None:
                            os.chdir(options.sourceindepdatapath)
                            for filename in zip.namelist():
                                unzip        = file (filename,'wb')
                                unzip.write(zip.read(filename))
                                unzip.close()
                            zip.close()
                            os.chdir(os.path.split(options.sourceindepdatapath)[0])
                         else:
                            print 'Export Package: %s not Ok' % os.path.split(options.sourceindepdatapath)[1]
                     except IOError, err:
                        print err
                     except:
                        kind, val, traceback = sys.exc_info()
                        raise kind, val, traceback
                  if PACKAGEARCHIVE == 'TAR':
                     try:
                         zipdirname  = os.path.split(options.sourceindepdatapath)[0]
                         zipfilename = os.path.split(options.sourceindepdatapath)[1]+ os.extsep + 'tgz'
                         zip =  tarfile.TarFile.open(os.path.join (zipdirname,zipfilename), 'r:gz' )
                         if tarfile.is_tarfile(os.path.join (zipdirname,zipfilename)) == True:
                            os.chdir(options.sourceindepdatapath)
                            for filename in zip.getnames():
                                zip.extract(filename,options.sourceindepdatapath)
                            zip.close()
                            os.chdir(os.path.split(options.sourceindepdatapath)[0])
                         else:
                            print 'Export Package: %s not Ok' % os.path.split(options.sourceindepdatapath)[1]
                     except IOError, err:
                        print err
                     except:
                        kind, val, traceback = sys.exc_info()
                        raise kind, val, traceback
                  # PACKAGE PARAMETER
                  found = False
                  for   package in os.listdir(options.sourceindepdatapath) :
                        if os.path.isfile(os.path.join(options.sourceindepdatapath,package)) and (string.find(package,'.par') > 0 ) and found == False:
                           global DB_DEFAULTCODE_SOURCE
                           global DB_DEFAULTCODE_TARGET
                           found = True
                           parameters = codecs.open ( os.path.join(options.sourceindepdatapath, '%s' % package ) , 'r')
                           parameter  = parameters.readline()
                           #Transformation Attributes
                           options.transporttype          = string.replace(string.replace(re.split('-',parameter) [1] ,'T ',''),' ','')
                           options.transformationpackage  = string.replace(string.replace(re.split('-',parameter) [3] ,'p ',''),' ','')
                           options.transformationpart     = string.replace(string.replace(re.split('-',parameter) [4] ,'P ',''),' ','')
                           options.startposition          = string.replace(string.replace(re.split('-',parameter) [5] ,'s ',''),' ','')
                           options.streammedium           = string.replace(string.replace(re.split('-',parameter) [6] ,'m ',''),' ','')
                           options.streamformat           = string.replace(string.replace(re.split('-',parameter) [7] ,'f ',''),' ','')
                           options.streamformat_target    = string.replace(string.replace(re.split('-',parameter) [8] ,'F ',''),' ','')
                           options.streamcharencoding     = string.replace(string.replace(re.split('-',parameter) [9], 'c ',''),' ','')
                           options.streamlob              = string.replace(string.replace(re.split('-',parameter) [10],'l ',''),' ','')
                           #Transformation Resources
                           options.sourcedbnode           = string.replace(string.replace(re.split('-',parameter) [11],'n ',''),' ','')
                           options.sourcedbname           = string.replace(string.replace(re.split('-',parameter) [12],'d ',''),' ','')
                           options.sourcedbuser           = string.replace(string.replace(re.split('-',parameter) [13],'u ',''),' ','')
                  if found == True:                              
                     sqlsource = os.path.join(options.sourceindepdatapath,package)
                  if found == False:
                     sqlsource = None
                     print ' Package % s not found' % package

        #DATA TARGET DATABASE
        username     = ''
        userpassword = ''
        if options.targetdbuser not in [''] :
           try:
              username     = re.split(',',options.targetdbuser)[0]
              userpassword = re.split(',',options.targetdbuser)[1]
              if userpassword == '*':
                 userpassword  = ''
           except:
              pass
        if( options.transformationtype == 'TRANSPORT' or
            options.transformationtype == 'IMPORT'      ): 
            if(options.targetdbnode in [''] or 
               options.targetdbname in [''] or 
               username             in [''] or
               userpassword         in ['']      ):
               STEPID = STEPID+1
               print "\nStep %i: Enter the data target:" % STEPID 
               if options.targetdbnode not in [''] :
                  print 'Node : %s' % options.targetdbnode
               if options.targetdbname not in [''] :
                  print 'Db   : %s' % options.targetdbname
               if username             not in [''] :
                  print 'User : %s' % username
            sqltarget        = None
            while sqltarget == None:
                  while options.targetdbnode in ['']:   
                        input = raw_input('Node : ')
                        if input == '':
                           options.targetdbnode = socket.gethostname()
                        if input != '':
                           options.targetdbnode = input
                  while options.targetdbname in ['']:   
                        input = raw_input('Db   : ') 
                        options.targetdbname = string.upper(input)
                  while username in ['']: 
                        username = raw_input('User : ')   
                  while userpassword in ['']:   
                        userpassword   = getpass.getpass()
                  options.targetdbuser = '%s,%s' %( username, userpassword )
                  if options.targetdbuser not in ['']:
                     username     = re.split(',',options.targetdbuser)[0]
                     userpassword = re.split(',',options.targetdbuser)[1]
                  if username != '' and username [0] != '"' and username[len(username)-1] != '"':
                     username  = string.upper(username)
                  elif username != '' :
                     username = string.replace(username,'"','')  
                  options.targetdbuser = '%s,%s' %( username, userpassword )  
                  try:
                      sqltarget = sql.connect( username, userpassword, options.targetdbname, options.targetdbnode )
                  except (sql.CommunicationError ) , err:
                      print 'Connect failed, try again:' , err
                      sqltarget    = None 
                  except (sql.SQLError ) , err:  
                      print 'Connect failed, try again:' , err    
                      sqltarget    = None 
                  if sqltarget    == None:
                     options.targetdbnode = ''
                     options.targetdbname = ''
                     options.targetdbuser = ''
                     username     = ''
                     userpassword = ''
            #DATA PATH
            try:
               options.targetindepdatapath = os.path.join(LOADERDATAPATH, 'sdb','loader', 'data', 'import', options.targetdbnode, options.targetdbname, packagename)
               if os.path.isdir(options.targetindepdatapath) == False:
                  os.makedirs  (options.targetindepdatapath)
            except EnvironmentError, err:
                print err

        input = ''
        if((options.transformationtype == 'TRANSPORT' or
            options.transformationtype == 'EXPORT'  ) and
            exist_db   (options, sqlsource, sqltarget) == True):
           if(exist_dba(options, sqlsource, sqltarget) == True):
              if options.transformationpackage in ['']:
                 STEPID = STEPID+1 
                 print "\nStep %i: Enter the tableset you want transport:" % STEPID
              while options.transformationpackage in [''] :
                    input = raw_input(' [1] DB\n [2] USER\n [3] TABLE\n \nStep %i: ' % STEPID )
                    if input == '1':
                       options.transformationpackage = 'DB'
                       STEPID = STEPID+1 
                       print '\nStep %i: Enter the the full qualified tableset you want exclude : ' % STEPID
                       configuration = codecs.open(os.path.join(options.sourceindepdatapath,  '%s.CONFIGURATION' % os.path.split(options.sourceindepdatapath)[1]) , "wb+","utf-8")
                       count = 1
                       username   = ''
                       tablename  = ''
                       table = raw_input(' [%i] ' %count )
                       while table not in [''] :
                             if string.find( table,'.' ) != -1 :
                                username  = re.split('\.',table)[0]
                                tablename = re.split('\.',table)[1]
                                count = count + 1
                                if tablename [0] == '"' and tablename[len(tablename)-1] == '"':
                                   tablename = string.replace(tablename,'"','')
                                else:
                                   tablename = string.upper(tablename)
                                if username [0] == '"' and username[len(username)-1] == '"':
                                   username = string.replace(username,'"','')
                                else:
                                   username = string.upper(username)
                                configuration.write(u'"%s","%s"\n' % (username,tablename) )
                             table = raw_input(' [%i] ' %count )
                       configuration.close()
                    if input == '2':
                       options.transformationpackage = 'USER'
                       STEPID = STEPID+1 
                       print '\nStep %i: Enter the tableset you want exclude: ' % STEPID
                       configuration = codecs.open(os.path.join(options.sourceindepdatapath, '%s.CONFIGURATION' % os.path.split(options.sourceindepdatapath)[1]), "wb+","utf-8")
                       count = 1
                       table = raw_input(' [%i] ' %count )
                       while table not in [''] :
                             if exist_table( options, sqlsource, sqltarget, table) :
                                count = count + 1
                                if table [0] == '"' and table[len(table)-1] == '"':
                                   table = string.replace(table,'"','')
                                else:
                                   table = string.upper(table)
                                configuration.write(u'"%s","%s"\n' % (re.split(',',options.sourcedbuser)[0],table) )
                             table = raw_input(' [%i] ' %count )
                       configuration.close()
                    if input == '3':
                       options.transformationpackage = 'TABLE'     
                       STEPID = STEPID+1 
                       print '\nStep %i: Enter the tableset you want include: ' % STEPID
                       configuration = codecs.open(os.path.join(options.sourceindepdatapath, '%s.CONFIGURATION' % os.path.split(options.sourceindepdatapath)[1] ), "wb+","utf-8")
                       count = 1
                       table = raw_input(' [%i] ' %count )
                       while table not in [''] :
                             if exist_table( options, sqlsource, sqltarget, table) :
                                count = count + 1
                                if table [0] == '"' and table[len(table)-1] == '"':
                                   table = string.replace(table,'"','')
                                else:
                                   table = string.upper(table)
                                configuration.write(u'"%s","%s"\n' % (re.split(',',options.sourcedbuser)[0],table) )
                             table = raw_input(' [%i] ' %count )
                       configuration.close()
           #if exist_dba(options, sqlsource, sqltarget ) == False:
           else:
              if options.transformationpackage in ['']:
                 STEPID = STEPID+1 
                 print "\nStep %i: Enter the tableset you want transport:" % STEPID
              while options.transformationpackage in ['']:
                    input = raw_input(' [1] USER\n [2] TABLE\n \nStep %i: ' % STEPID)
                    if input == '1':
                       options.transformationpackage = 'USER'
                       STEPID = STEPID+1 
                       print '\nStep %i: Enter the tableset you want exclude: ' %STEPID
                       configuration = codecs.open(os.path.join(options.sourceindepdatapath, '%s.CONFIGURATION' % os.path.split(options.sourceindepdatapath)[1]), "wb+","utf-8")
                       count = 1
                       table = raw_input(' [%i] ' %count )
                       while table not in [''] :
                             if exist_table( options,sqlsource, sqltarget, table) :
                                count = count + 1
                                if table [0] == '"' and table[len(table)-1] == '"':
                                   table = string.replace(table,'"','')
                                else:
                                   table = string.upper(table)
                                configuration.write(u'"%s","%s"\n' % (re.split(',',options.sourcedbuser)[0],table) )
                             table = raw_input(' [%i] ' %count )
                       configuration.close()
                    if input == '2':
                       options.transformationpackage = 'TABLE'     
                       STEPID = STEPID+1 
                       print '\nStep %i: Enter the tableset you want include: ' % STEPID
                       configuration = codecs.open(os.path.join(options.sourceindepdatapath, '%s.CONFIGURATION' % os.path.split(options.sourceindepdatapath)[1]), "wb+","utf-8")
                       count = 1
                       table = raw_input(' [%i] ' %count )
                       while table not in ['']  :
                             if exist_table( options, sqlsource, sqltarget, table) :
                                count = count + 1
                                if table [0] == '"' and table[len(table)-1] == '"':
                                   table = string.replace(table,'"','')
                                else:
                                   table = string.upper(table)
                                configuration.write(u'"%s","%s"\n' % (re.split(',',options.sourcedbuser)[0],table) )
                             table = raw_input(' [%i] ' %count )
                       configuration.close()             
           input = ''
           if options.transformationpackage == 'DB' and exist_dba(options, sqlsource, sqltarget) == True:
              if TESTLOADER == 'FALSE':
                 options.transformationpart  = 'DB'
              MONITOR = 'OFF'
           input = ''
           if options.transformationpackage == 'USER':
              if options.transformationpart in ['']:
                 STEPID = STEPID+1 
                 print "\nStep %i: Choose the packagepart:  " %STEPID
              while options.transformationpart in ['']:
                    input = raw_input(' [1] USER\n [2] TABLE\n \nStep %i: ' %STEPID)
                    if input == '1':
                       options.transformationpart = 'USER'
                    if input == '2':
                       options.transformationpart = 'TABLE'
		
           input = ''
		  
           # TRANSFORMATION PARAMETERS
           if options.streammedium in ['']:
              if options.transformationpackage == 'TABLE' :    
                 options.transformationpart     = 'TABLE'                      
              if options.transformationtype    == 'TRANSPORT' and sys.platform[:3] == 'win':
                 streammedium                   = 'PIPE'
              else:
                 streammedium                   = 'FILE'
           else:
              streammedium = options.streammedium  
           if options.streamformat in ['']:      
              if  (streammedium                 == 'PIPE' ) :
                   streamformat                  = 'RECORDS'
              elif(options.transformationpackage!= 'TABLE' or
                   options.transformationpart   != 'TABLE') :
                   streamformat                  = 'RECORDS'
              elif(options.sourcedbnode == options.targetdbnode and
                   options.sourcedbname == options.targetdbname and
                   options.transformationpart  == 'TABLE'):
                   streamformat                 = 'SQL'
              else:
                   streamformat                 = 'COMPRESSED'
           else:
              streamformat = options.streamformat
           if options.streamformat_target in ['']:
              streamformat_target               = 'ROW'
           else:
              streamformat_target = options.streamformat_target
           if options.streamcharencoding  in ['']:
              streamcharencoding                = 'DYNAMIC'
           else:
              streamcharencoding = options.streamcharencoding
           if options.streamlob in ['']:
              if(options.transformationpackage != 'TABLE' and
                 options.transformationpart    != 'TABLE') :               
                 streamlob                      = 'TABLE'
              else:
                 streamlob                      = 'COLUMNVALUE'
           else:
               streamlob = options.streamlob
          
           # TRANSFORMATION PARAMETERS DEFAULTS
           if( options.streammedium        in [''] or 
               options.streamformat        in [''] or 
               options.streamformat_target in [''] or
               options.streamcharencoding  in [''] or
               options.streamlob           in [''] or
               options.transporttype       in ['']):
               STEPID = STEPID+1 
               print "\nStep %i: Choose the transformation parameters: " %STEPID
               print ' Stream           : %s  ' % 'CATALOG, PACKAGE, DATA'
               print ' Medium           : %s  ' % streammedium
               print ' Format  Transport: %s  ' % streamformat
               print ' Format  Import   : %s  ' % streamformat_target
               print ' Char    Encoding : %s  ' % streamcharencoding
               print ' Lob     Transport: %s  ' % streamlob
               print ' Process Package  : %s  ' % 'EXECUTE'
               while input not in ['1','2']:
                     input = raw_input(' [1] DEFAULT\n [2] CHANGE DEFAULT\n \nStep %i: ' %STEPID)
                     if input == '1':
                        input  = '1'
                     if input == '2':
                        input  = '2'
           if input == '1':
              options.streammedium        = streammedium
              options.streamformat        = streamformat
              options.streamformat_target = streamformat_target
              options.streamcharencoding  = streamcharencoding
              options.streamlob           = streamlob 
              options.transporttype      = 'EXECUTE' 
           if input == '2':
              input =  None
              STEPID = STEPID+1 
              print "\nStep %i: Choose the transformation streams:" %STEPID                   
              while input not in ['1','2'] :  
                    input = raw_input(' [1] [CATALOG, PACKAGE, DATA]\n [2] [CATALOG, PACKAGE]\n \nStep %i: ' %STEPID)  
                    if input == '1':
                       pass
                    if input == '2':
                       options.streamformat_target = 'DTL'
              input  = ''
              STEPID = STEPID+1 
              print "\nStep %i: Choose the transformation medium: "  %STEPID
              if options.transformationpart == 'DB' :  
                 while options.streammedium in ['']:
                       input = raw_input(' [1] FILE\n [2] PIPE\n \nStep %i: ' %STEPID)
                       if input == '1':
                          options.streammedium = 'FILE'
                       if input == '2':
                          options.streammedium = 'PIPE'
                       if input == '3':
                          options.streammedium = 'TAPE' 
              if options.transformationpart == 'USER' or options.transformationpart == 'TABLE':
                 while options.streammedium in ['']:
                       input = raw_input(' [1] FILE\n [2] PIPE\n \nStep %i: ' %STEPID)
                       if input == '1':
                          options.streammedium = 'FILE'
                       if input == '2':
                          options.streammedium = 'PIPE'
              input  = ''
              STEPID = STEPID+1    
              if(options.transformationtype == 'TRANSPORT'  and DB_DEFAULTCODE_SOURCE == DB_DEFAULTCODE_TARGET) or options.transformationtype == 'EXPORT':
                 CASE_1 = ' [1] [RECORDS]\n [2] [PAGES]\n \nStep %i: ' %STEPID
                 CASE_2 = ' [1] [COMPRESSED]\n [2] [FORMATTED]\n [3] [RECORDS]\n [4] [SQL]  \n [5] [PAGES]\n \nStep %i: ' %STEPID
                 CASE_3 = ' [1] [COMPRESSED]\n [2] [FORMATTED]\n [3] [RECORDS]\n [4] [PAGES]\n \nStep %i: ' %STEPID
              if options.transformationtype == 'TRANSPORT'  and DB_DEFAULTCODE_SOURCE != DB_DEFAULTCODE_TARGET:
                 CASE_1 = ' [1] [RECORDS]                 \nStep %i: ' %STEPID
                 CASE_2 = ' [1] [COMPRESSED]\n [2] [FORMATTED]\n [3] [RECORDS]\n [4] [SQL]\n   \nStep %i: ' %STEPID
                 CASE_3 = ' [1] [COMPRESSED]\n [2] [FORMATTED]\n [3] [RECORDS]\n               \nStep %i: ' %STEPID
                 
              print "\nStep %i: Choose the format in which data are transformed during\n export from the data source:" %STEPID       
              if options.transformationpart == 'DB' or  options.transformationpart == 'USER':
                 while options.streamformat in [''] :
                       input = raw_input(CASE_1)
                       if input == '1':
                          options.streamformat = 'RECORDS'
                       if input == '2':
                          options.streamformat = 'PAGES'
                          if options.transformationtype == 'TRANSPORT' and DB_DEFAULTCODE_SOURCE != DB_DEFAULTCODE_TARGET:
                             options.streamformat = ''
              input = ''
              if options.transformationpart == 'TABLE':
                 if( options.sourcedbnode   == options.targetdbnode   and
                     options.sourcedbname   == options.targetdbname ):
                     while options.streamformat in [''] :
                           input = raw_input(CASE_2)
                           if input == '1':
                              options.streamformat = 'COMPRESSED'
                           if input == '2':
                              options.streamformat = 'FORMATTED'
                           if input == '3':
                              options.streamformat = 'RECORDS'
                           if input == '4':
                              options.streamformat = 'SQL'
                           if input == '5':
                              options.streamformat = 'PAGES'
                              if options.transformationtype == 'TRANSPORT' and DB_DEFAULTCODE_SOURCE != DB_DEFAULTCODE_TARGET:
                                 options.streamformat = ''
                 else:
                      while options.streamformat in [''] :
                            input = raw_input(CASE_3)
                            if input == '1':
                               options.streamformat = 'COMPRESSED'
                            if input == '2':
                               options.streamformat = 'FORMATTED'
                            if input == '3':
                               options.streamformat = 'RECORDS'
                            if input == '4':
                               options.streamformat = 'PAGES'
                               if options.transformationtype == 'TRANSPORT' and DB_DEFAULTCODE_SOURCE != DB_DEFAULTCODE_TARGET:
                                  options.streamformat = ''
              input = ''
              if options.streamformat_target != 'DTL':
                 STEPID = STEPID+1                            
                 while options.streamformat_target in [''] and options.streamformat != 'PAGES' and options.streamformat != 'RECORDS' :
                       print "\nStep %i: Choose the format in which data are transformed during\n import into the data target:" %STEPID   
                       input = raw_input(' [1] [ROW]\n [2] [PAGE]\n \nStep %i: ' % STEPID)
                       if input == '1':
                          options.streamformat_target = 'ROW'
                       if input == '2':
                          options.streamformat_target = 'PAGE'
              input  = '' 
              STEPID = STEPID+1    
              while options.streamcharencoding in [''] and options.streamformat == 'COMPRESSED':
                    print "\nStep %i: Choose the charencoding how charcolumns should be transformed:" % STEPID
                    input = raw_input(' [1] [ASCII]\n [2] [UTF8]\n [3] [UCS2]\n [4] [DYNAMIC]\n \nStep %i: ' %STEPID)
                    if input == '1':
                       options.streamcharencoding = 'ASCII'
                    if input == '2':
                       options.streamcharencoding = 'UTF8'
                    if input == '3':
                       options.streamcharencoding = 'UCS2'
                       if DB_DEFAULTCODE_SOURCE  == 'ASCII':
                          print '\nTransformation from ASCII data to UNICODE data. \n This Transformation may change Information if ASCII is not encoded in CODEPAGE ISO-8859-1\n'
                       if DB_DEFAULTCODE_TARGET  == 'ASCII':
                          print '\nTransformation from UNICODE data to ASCII data is not supported; \n This Transformation may lose Information '
                          options.streamcharencoding = ''
                    if input == '4':
                       options.streamcharencoding = 'DYNAMIC'
              input = ''
              while options.streamcharencoding in [''] and options.streamformat == 'RECORDS' :   
                    print "\nStep %i: Choose the charencoding how charcolumns should be transformed:"  %STEPID                
                    input = raw_input(' [1] [ASCII]\n [2] [UCS2]\n [3] [DYNAMIC]\n \nStep %i: ' % STEPID)
                    if input == '1':
                       options.streamcharencoding = 'ASCII'
                    if input == '2':
                       options.streamcharencoding = 'UCS2'
                       if DB_DEFAULTCODE_SOURCE  == 'ASCII':
                          print '\nTransformation from ASCII data to UNICODE data. \n This transformation may change information if ASCII is not encoded in CODEPAGE ISO-8859-1\n'
                       if DB_DEFAULTCODE_TARGET  == 'ASCII':
                          print '\nTransformation from UNICODE data to ASCII data is not supported; \n This Transformation may lose Information '
                          options.streamcharencoding = ''
                    if input == '3':
                       options.streamcharencoding = 'DYNAMIC'
              input = ''
              while options.streamcharencoding in [''] and options.streamformat == 'FORMATTED':
                    print "\nStep %i: Choose the Char Encoding how Char Columns should be transformed:" % STEPID
                    input = raw_input(' [1] [ASCII]\n [2] [UTF8]\n [3] [UCS2]\n \nStep %i: ' %STEPID)
                    if input == '1':
                       options.streamcharencoding = 'ASCII'
                    if input == '2':
                       options.streamcharencoding = 'UTF8'
                    if input == '3':
                       options.streamcharencoding = 'UCS2'
                       if DB_DEFAULTCODE_SOURCE  == 'ASCII':
                          print '\nTransformation from ASCII data to UNICODE data. \n This transformation may change information if ASCII is not encoded in CODEPAGE ISO-8859-1\n'
                       if DB_DEFAULTCODE_TARGET  == 'ASCII':
                          print '\nTransformation from UNICODE data to ASCII data is not supported; \n This Transformation may lose Information '
                          options.streamcharencoding = ''
              input  = ''
              STEPID = STEPID+1  
              while options.streamlob in [''] and options.streamformat != 'PAGES' and options.streamformat != 'RECORDS' :  
                    print "\nStep %i: Choose how lobcolumns should be transformed:" % STEPID               
                    input = raw_input(' [1] [COLUMN]\n [2] [COLUMNVALUE]\n \nStep %i: ' % STEPID)  
                    if input == '1':
                       options.streamlob = 'COLUMN'
                    if input == '2':
                       options.streamlob = 'COLUMNVALUE'

              input = ''
              STEPID = STEPID+1              
              while options.transporttype in [''] :
                    print '\nStep %i: Choose how the transformation package should be processed:' % STEPID
                    while input not in ['1','2']:
                          input = raw_input(' [1] EXCECUTE PACKAGE\n [2] GENERATE PACKAGE\n\nStep %i: ' %STEPID )
                    if input == '1':
                       options.transporttype = 'EXECUTE'
                    if input == '2':
                       options.transporttype = 'GENERATE'
              
           try:
              if sqlsource != None:
                 sqlsource.release()
              if sqltarget != None:   
                 sqltarget.release()
           except loader.CommunicationError, err :
              raise loader.CommunicationError, err, sys.exc_traceback
		
     except EOFError:
         os._exit(1)
      
     '''    
     print 'loadercli -t %s -p %s -P %s -m %s -f %s -F %s -c %s -l %s -n %s -d %s -u %s ' %( 
     options.transformationtype, options.transformationpackage, options.transformationpart, 
     options.streammedium, options.streamformat, options.streamformat_target, options.streamcharencoding, options.streamlob,
     options.sourcedbnode, options.sourcedbname, options.sourcedbuser )
     '''
         
     return rc
     
def exist_packageloader (options): 
    bVersion = True
    try:
       session       = loader.Loader('','')
       versionstring = session.cmd('VERSION')
       match         = re.search ('"(\d+)[.](\d+)[.](\d+)[.](\d+)"', versionstring)
       version       = tuple (map (int, match.group (1, 2, 3, 4)))
       if version < (7,5,0,16):
          print versionstring
          print 'Loader Version 7.5.00.16 or higher required'
          bVersion =  False
          if sys.platform[:3] == 'win':
             print 'Try to start Loader from %s ' % INDEPPROGPATH
       return bVersion
    except loader.LoaderError, err:
           raise loader.LoaderError, err, sys.exc_traceback
    except loader.CommunicationError, err :
           raise loader.CommunicationError, err, sys.exc_traceback
         
def exist_db (options, sqlsource, sqltarget):
    rc = True
    maxusertasks  = 0
    usedusertasks = 0
    global DB_INSTROOT_SOURCE
    global DB_INSTROOT_TARGET
    global DB_VERSION_SOURCE
    global DB_VERSION_TARGET
    global DB_INDEPDATAPATH_SOURCE
    global DB_INDEPDATAPATH_TARGET
    global DB_DEFAULTCODE_SOURCE
    global DB_DEFAULTCODE_TARGET
    
    #Source DBM
    try:
        dbmsource     = dbm.DBM (options.sourcedbnode, options.sourcedbname, '','')
        indepdatapath = dbmsource.cmd('dbm_getpath     INDEPDATAPATH')
        if DB_INDEPDATAPATH_SOURCE == '':
           DB_INDEPDATAPATH_SOURCE = re.split('\n',indepdatapath)[0]
        instroot = dbmsource.cmd('dbm_version  INSTROOT')
        DB_INSTROOT_SOURCE = re.split('\n',instroot)[0]
        version  = dbmsource.cmd('dbm_version BUILD')
        version  = string.replace(version,'DBMServer ','')
        DB_VERSION_SOURCE = '%s.%s' %( re.split('Build',version)[0][0:6],re.split('Build',version)[1][2:4])
        match = re.search ('(\d+)[.](\d+)[.](\d+)', DB_VERSION_SOURCE)
        version = tuple (map (int, match.group (1, 2, 3)))
        if version      >=  (7, 5, 0):
           MAXUSERTASKS  = "SELECT VALUE    FROM ACTIVECONFIGURATION  WHERE PARAMETERNAME LIKE 'MAXUSERTASKS'"
           CODE          = "SELECT VALUE    FROM ACTIVECONFIGURATION  WHERE PARAMETERNAME LIKE '_UNICODE'"
           DEFAULTCODE   = "SELECT VALUE    FROM ACTIVECONFIGURATION  WHERE PARAMETERNAME LIKE 'DEFAULT_CODE'"
           USEDUSERTASKS = "SELECT COUNT(*) FROM SESSIONS"
        else:
           MAXUSERTASKS  = "SELECT VALUE    FROM DBPARAMETERS         WHERE DESCRIPTION   LIKE 'MAXUSERTASKS'"
           CODE          = "SELECT VALUE    FROM DBPARAMETERS         WHERE PARAMETERNAME LIKE '_UNICODE'"
           DEFAULTCODE   = "SELECT VALUE    FROM DBPARAMETERS         WHERE DESCRIPTION   LIKE 'DEFAULT_CODE'"
           USEDUSERTASKS = "SELECT COUNT(*) FROM CONNECTEDUSERS"
        if sqlsource != None: 
           resultset    = sqlsource.sql(MAXUSERTASKS)
           for row in resultset:
               maxusertasks = row[0]
           resultset    = sqlsource.sql(USEDUSERTASKS)
           for row in resultset:
               sessions = row[0]    
           if int(maxusertasks) - int(usedusertasks) + 1  < 2:
              print 'At least 2 free USERTASKS for Data Source Needed'
              rc = False
           resultset    = sqlsource.sql(CODE)
           for row in resultset:
               code = row[0]
               if code[0:2]            == 'NO':
                  DB_CODE_SOURCE        = 'ASCII'
               else:
                  DB_CODE_SOURCE        = 'UNICODE'
           resultset    = sqlsource.sql(DEFAULTCODE)
           for row in resultset:
               defaultcode = row[0]
               if defaultcode[0:5]     == 'ASCII':
                  DB_DEFAULTCODE_SOURCE = 'ASCII'
               if defaultcode[0:7]     == 'UNICODE':
                  DB_DEFAULTCODE_SOURCE = 'UNICODE'
    except dbm.CommunicationError, err :
           pass
    except dbm.DBMServError, err:
           pass
    except (sql.CommunicationError ), err:   
           print 'Connect failed, try again:' , err
    except (sql.SQLError ), err:
           print err
        
    #Target DBM
    try:
        dbmtarget     = dbm.DBM (options.targetdbnode, options.targetdbname, '','')
        indepdatapath = dbmtarget.cmd('dbm_getpath     INDEPDATAPATH')
        if DB_INDEPDATAPATH_TARGET == '':
           DB_INDEPDATAPATH_TARGET = re.split('\n',indepdatapath)[0]
        instroot = dbmtarget.cmd('dbm_version  INSTROOT')
        DB_INSTROOT_TARGET = re.split('\n',instroot)[0]
        version = dbmtarget.cmd('dbm_version BUILD')
        version = string.replace(version,'DBMServer ','')
        DB_VERSION_TARGET = '%s.%s' %( re.split('Build',version)[0][0:6],re.split('Build',version)[1][2:4])
        match   = re.search ('(\d+)[.](\d+)[.](\d+)', DB_VERSION_TARGET)
        version = tuple (map (int, match.group (1, 2, 3)))
        if version      >=  (7, 5, 0):
           MAXUSERTASKS  = "SELECT VALUE    FROM ACTIVECONFIGURATION  WHERE PARAMETERNAME LIKE 'MAXUSERTASKS'"
           CODE          = "SELECT VALUE    FROM ACTIVECONFIGURATION  WHERE PARAMETERNAME LIKE '_UNICODE'"
           DEFAULTCODE   = "SELECT VALUE    FROM ACTIVECONFIGURATION  WHERE PARAMETERNAME LIKE 'DEFAULT_CODE'"
           USEDUSERTASKS = "SELECT COUNT(*) FROM SESSIONS"
        else:
           MAXUSERTASKS  = "SELECT VALUE    FROM DBPARAMETERS         WHERE DESCRIPTION   LIKE 'MAXUSERTASKS'"
           CODE          = "SELECT VALUE    FROM DBPARAMETERS         WHERE PARAMETERNAME LIKE '_UNICODE'"
           DEFAULTCODE   = "SELECT VALUE    FROM DBPARAMETERS         WHERE DESCRIPTION   LIKE 'DEFAULT_CODE'"
           USEDUSERTASKS = "SELECT COUNT(*) FROM CONNECTEDUSERS"
        if sqltarget != None: 
           resultset    = sqltarget.sql(MAXUSERTASKS)
           for row in resultset:
               maxusertasks = row[0]
           resultset    = sqltarget.sql(USEDUSERTASKS)
           for row in resultset:
               sessions = row[0]    
           if int(maxusertasks) - int(usedusertasks) + 1  < 2:
              print 'At least 2 free USERTASKS for Data Source Needed'
              rc = False
           resultset    = sqlsource.sql(CODE)
           for row in resultset:
               code = row[0]
               if code[0:2]            == 'NO':
                  DB_CODE_TARGET        = 'ASCII'
               else:
                  DB_CODE_TARGET        = 'UNICODE'
           resultset    = sqltarget.sql(DEFAULTCODE)
           for row in resultset:
               defaultcode = row[0]
               if defaultcode[0:5]     == 'ASCII':
                  DB_DEFAULTCODE_SOURCE = 'ASCII'
               if defaultcode[0:7]     == 'UNICODE':
                  DB_DEFAULTCODE_SOURCE = 'UNICODE'
    except dbm.CommunicationError, err :
           pass
    except dbm.DBMServError, err:
           pass
    except (sql.CommunicationError ), err:   
           print 'Connect failed, try again:' , err
    except (sql.SQLError ), err:
           print err
    
    return rc

def  exist_dba(options, sqlsource, sqltarget ):
     global DBA_SOURCE
     global DBA_TARGET
     exist_sourcedba = False
     exist_targetdba = False
     if sqlsource == None:  
        exist_sourcedba = True
     if sqlsource != None:  
        try:   
           resultset = sqlsource.sql(EXISTSDBA)
           for row in resultset:
               if re.split(',',options.sourcedbuser)[0] == row[0] :
                  exist_sourcedba = True
        except (sql.CommunicationError ), err:   
            print 'Connect failed, try again:' , err
        except (sql.SQLError ), err:
            exist_sourcedba = False
     if sqltarget == None:
        exist_targetdba = True
     if sqltarget != None:         
        try:   
            resultset = sqltarget.sql(EXISTSDBA)
            for row in resultset:
                if re.split(',',options.targetdbuser)[0] == row[0] :
                   exist_targetdba = True
        except (sql.CommunicationError ), err:   
            print 'Connect failed, try again:' , err
        except (sql.SQLError ), err:
            exist_targetdba = False
     if exist_sourcedba == True and exist_targetdba == True:
        rc = True
        if options.transformationpackage == 'DB':
           DBA_SOURCE = options.sourcedbuser
           DBA_TARGET = options.targetdbuser
     else:
        rc = False      
     return rc    
         
def  exist_table(options, sqlsource, sqltarget, table):
     exist_sourcetable = True
     exist_targettable = True
     if table [0] == '"' and table[len(table)-1] == '"':
        table = string.replace(table,'"','')
     else:
        table = string.upper(table)
     if sqlsource == None:
        exist_sourcetable = True    
     if sqlsource != None:  
        try:   
           cmd = 'EXISTS TABLE "%s" ' % table
           sqlsource.sql(cmd)
           exist_sourcetable = True
        except (sql.CommunicationError ), err:   
            print 'Connect failed, try again:' , err
        except (sql.SQLError ), err:
            if  err.errorCode == -4004: 
                exist_sourcetable = False
                print '%s.%s exists not in Data Source.' %(re.split(',',options.sourcedbuser)[0],table)
     if sqltarget == None:
        exist_targettable = True           
     if sqltarget != None:         
        try: 
            cmd = 'EXISTS TABLE "%s" ' % table  
            sqltarget.sql(cmd)
            exist_targettable = True
            print '%s.%s exists in Data Target. ' %(re.split(',',options.targetdbuser)[0],table )   
        except (sql.CommunicationError ), err:   
            print 'Connect failed, try again:' , err
        except (sql.SQLError ), err:
            if  err.errorCode == -4004: 
                exist_targettable = False
    
     if options.transformationtype == 'TRANSPORT':
        if exist_sourcetable == True and exist_targettable == False:
           rc = True
        else:
           rc = False  
     if options.transformationtype == 'EXPORT' :
        if exist_sourcetable == True:
           rc = True
        else:
           rc = False  
     if options.transformationtype == 'IMPORT' :     
        if exist_targettable == False:
           rc = True
        else:
           rc = False     
     return rc  
     
def get_environment(options):
    global TESTLOADER
    global PACKAGEPATH
    global PACKAGENAME
    global PACKAGETIMESTAMP
    global PACKAGEEXECUTION
    global PACKAGEARCHIVE
    #LOADER ENVIRONMENT
    global INDEPPROGPATH    
    global INDEPDATAPATH        
    
    #LOADER CONFIGURATION
    global LOADERCONFIGURATION
    global LOADERCONFIGFILE
    global LOADERPROGPATH
    global LOADERDATAPATH
    
    # TRANSFORMATION RUNTIME
    global PACKAGETIMEOUT       
    global MAXERRORCOUNT        
    global TRANSACTIONSIZE      
    global AUTOCOMMIT           
    
    # TRANSFORMATION PARAMETER
    global VERBOSE              
    global MONITOR              
    global TRANSFORMATIONSYNTAX 
    global SYSTEMATTRIBUTE
    global FIELDLOBMASK
    
    TESTLOADER  = os.getenv('TESTLOADER')
    if TESTLOADER == None:
       TESTLOADER = 'FALSE'
    INSTROOT = os.getenv('INSTROOT')   
    if INSTROOT == None:
       INSTROOT  = ''
    try:
        session = dbm.DBM ('','',INSTROOT)
        INDEPPROGPATH = string.split(session.cmd('dbm_getpath INDEPPROGPATH'),'\n')[0]
        INDEPDATAPATH = string.split(session.cmd('dbm_getpath INDEPDATAPATH'),'\n')[0]
        if options.loaderinstroot == '':
           options.loaderinstroot  = INDEPPROGPATH
    except dbm.CommunicationError, err :
        print err
    except dbm.DBMServError, err:
        print err       
    
    #LOADER ENVIRONMENT PARAMETER
    if sys.platform[:3] == 'win':
       import _winreg
       try: 
          LOADERPROGPATH = _winreg.QueryValueEx(_winreg.OpenKey(_winreg.HKEY_CURRENT_USER ,"Volatile Environment"),"APPDATA")[0] 
          LOADERDATAPATH =  os.path.join (_winreg.QueryValueEx(_winreg.OpenKey(_winreg.HKEY_CURRENT_USER ,"Volatile Environment"),"HOMEDRIVE")[0], _winreg.QueryValueEx(_winreg.OpenKey(_winreg.HKEY_CURRENT_USER ,"Volatile Environment"),"HOMEPATH")[0])
       except EnvironmentError:
          print "Error reading Registry"
    if sys.platform[:3] != 'win':
       try:
          LOADERPROGPATH = os.getenv('HOME')
          LOADERDATAPATH = os.getenv('HOME')
       except:
          pass
          
    PACKAGETIMESTAMP = time.strftime("%Y%m%d_%H%M%S",time.localtime())
    
    #TRANSPORT  PARAMETER   
    #LOADER CONFIGURATION PARAMETER
    try:
       LOADERCONFIGURATION  = ConfigParser.ConfigParser()
       configfile = None
       if sys.platform[:3] == 'win':
          if(os.path.isfile(os.path.join (LOADERPROGPATH, 'sdb'   ,'loader','config','loader.cfg')) == False and
             os.path.isfile(os.path.join (INDEPPROGPATH , 'config','loader.cfg')) == True ):
             shutil.copy( os.path.join   (INDEPPROGPATH , 'config','loader.cfg'),
                          os.path.join   (LOADERPROGPATH, 'sdb'   ,'loader','config','loader.cfg'))
          LOADERCONFIGFILE = os.path.join(LOADERPROGPATH, 'sdb'   ,'loader','config','loader.cfg')
       if sys.platform[:3] != 'win':
          if(os.path.isfile(os.path.join (LOADERPROGPATH, '.sdb'  ,'loader','config','loader.cfg')) == False and
             os.path.isfile(os.path.join (INDEPPROGPATH , 'config','loader.cfg')) == True ):
             shutil.copy(os.path.join    (INDEPPROGPATH , 'config','loader.cfg'),
                         os.path.join    (LOADERPROGPATH, '.sdb'  ,'loader','config','loader.cfg'))
          LOADERCONFIGFILE = os.path.join(LOADERPROGPATH, '.sdb'  ,'loader','config','loader.cfg')
       configfile          = file(LOADERCONFIGFILE,'r')
       LOADERCONFIGURATION.readfp(configfile)
       # PACKAGE
       PACKAGEPATH           = LOADERCONFIGURATION.get       ('PACKAGE','PACKAGEPATH')
       PACKAGENAME           = LOADERCONFIGURATION.get       ('PACKAGE','PACKAGENAME')
       PACKAGEARCHIVE        = LOADERCONFIGURATION.get       ('PACKAGE','PACKAGEARCHIVE')
       PACKAGEEXECUTION      = LOADERCONFIGURATION.get       ('PACKAGE','PACKAGEEXECUTION')
       PACKAGETIMEOUT        = LOADERCONFIGURATION.getint    ('PACKAGE','TIMEOUT')
       VERBOSE               = LOADERCONFIGURATION.get       ('PACKAGE','VERBOSE')
       MONITOR               = LOADERCONFIGURATION.get       ('PACKAGE','MONITOR')
       # DATABASE
       SYSTEMATTRIBUTE       = LOADERCONFIGURATION.get       ('DATABASE_DESIGNTIME','SYSTEMATTRIBUTE')
       AUTOCOMMIT            = LOADERCONFIGURATION.get       ('DATABASE_CONNECTION','AUTOCOMMIT')
       TRANSACTIONSIZE       = LOADERCONFIGURATION.getint    ('DATABASE_RUNTIME'   ,'TRANSACTIONSIZE')
       # LOADER
       MAXERRORCOUNT         = LOADERCONFIGURATION.getint    ('LOADER','MAXERRORCOUNT')
       TRANSFORMATIONSYNTAX  = LOADERCONFIGURATION.get       ('STREAM','TRANSFORMATIONSYNTAX')
       configfile.close()
       transportparameterfile = None
       if options.transformationtype not in ['']:
          options.transformationtype = string.upper(options.transformationtype)
       if options.sourcedbnode not in ['']:
          options.sourcedbnode       = options.sourcedbnode
       if options.sourcedbname not in ['']:
          options.sourcedbname       = options.sourcedbname
       if options.sourcedbuser not in ['']:
          options.sourcedbuser       = options.sourcedbuser
       if options.targetdbnode not in ['']:
          options.targetdbnode       = options.targetdbnode
       if options.targetdbname not in ['']:
          options.targetdbname       = options.targetdbname
       if options.targetdbuser not in ['']:
          options.targetdbuser       = options.targetdbuser 
       transportparameter            =  []
       TRANSPORTPARAMETERFILE= os.path.join(LOADERDATAPATH,'sdb', 'loader', 'data', 'transport' + os.extsep + 'log')
       if(os.path.isfile      (os.path.join(LOADERDATAPATH,'sdb', 'loader', 'data', 'transport' + os.extsep + 'log')) == True and options.transformationpackage not in ['','DB','USER','TABLE'] ):
          transportparameterfile = file(os.path.join(LOADERDATAPATH,'sdb', 'loader','data', 'transport' + os.extsep + 'log'),'r')
          TRANSPORTPARAMETER = csv.reader(transportparameterfile, delimiter  = '|'  )
          for parameter in  TRANSPORTPARAMETER :
              if options.transformationpackage == string.strip(parameter[1]):
                 transportparameter.append(parameter)
          if len(transportparameter) == 1:
             transformationpackage         = string.strip(transportparameter[0][1] )
             options.transformationpackage = string.strip(transportparameter[0][2] )
             options.transporttype         = string.strip(transportparameter[0][3] )
             options.transformationpart    = string.strip(transportparameter[0][6] )
             options.startposition         = string.strip(transportparameter[0][7] )
             options.streammedium          = string.strip(transportparameter[0][8] )
             options.streamformat          = string.strip(transportparameter[0][9] )
             options.streamformat_target   = string.strip(transportparameter[0][10])
             options.streamcharencoding    = string.strip(transportparameter[0][11])
             options.streamlob             = string.strip(transportparameter[0][12])
             PACKAGEARCHIVE                = string.strip(transportparameter[0][13])
             if options.transformationtype in ['']:
                options.transformationtype = string.strip(transportparameter[0][4] )
             if options.sourcedbnode in ['']:
                options.sourcedbnode       = string.strip(transportparameter[0][19])
             if options.sourcedbname in ['']:
                options.sourcedbname       = string.strip(transportparameter[0][20])
             if options.sourcedbuser in ['']:   
                options.sourcedbuser       = string.strip(transportparameter[0][21])
             if options.targetdbnode in ['']:
                options.targetdbnode       = string.strip(transportparameter[0][22])
             if options.targetdbname in ['']:
                options.targetdbname       = string.strip(transportparameter[0][23])
             if options.targetdbuser in ['']:
                options.targetdbuser       = string.strip(transportparameter[0][24])
             # PACKAGE ARCHIVE
             if options.transformationtype == 'TRANSPORT' or options.transformationtype == 'EXPORT' :
                zipdirname  = os.path.join(LOADERDATAPATH,'sdb','loader','data','export',options.sourcedbnode,options.sourcedbname)
                if os.path.isdir(os.path.join(zipdirname,transformationpackage)) == False:
                   os.makedirs  (os.path.join(zipdirname,transformationpackage))
                if os.path.isdir(os.path.join(zipdirname,PACKAGETIMESTAMP)) == False:
                   os.makedirs  (os.path.join(zipdirname,PACKAGETIMESTAMP))
                if PACKAGEARCHIVE == 'ZIP':
                   zipfilename = transformationpackage + os.extsep + 'zip'
                   zip =  zipfile.ZipFile(os.path.join (zipdirname,zipfilename), 'r', zipfile.ZIP_DEFLATED )
                   if zip.testzip() == None:
                      os.chdir(os.path.join(zipdirname,transformationpackage))
                      for filename in zip.namelist():
                          osext = re.split('\%s' %os.extsep ,filename)[1]
                          if osext == 'cfg' or osext == 'xml' or osext == 'CONFIGURATION':
                             unzip        = file (filename,'wb')
                             unzip.write(zip.read(filename))
                             unzip.close()
                             shutil.copy(filename, os.path.join(zipdirname,PACKAGETIMESTAMP,PACKAGETIMESTAMP + os.extsep + osext))
                             os.unlink  (filename)
                      os.chdir(zipdirname)
                      if os.path.isdir(os.path.join(zipdirname,transformationpackage)) == True:
                         os.removedirs(os.path.join(zipdirname,transformationpackage))
                         options.mode = 'PARAM'
                   zip.close()
                if PACKAGEARCHIVE == 'TAR':
                   zipfilename = transformationpackage + os.extsep + 'tgz'
                   zip =  tarfile.TarFile.open(os.path.join (zipdirname,zipfilename), 'r:gz' )
                   if tarfile.is_tarfile(os.path.join (zipdirname,zipfilename)) == True:
                      os.chdir(os.path.join(zipdirname,transformationpackage))
                      for filename in zip.getnames():
                          osext = re.split('\%s' %os.extsep ,filename)[1]
                          if osext == 'cfg' or osext == 'xml' or osext == 'CONFIGURATION':
                             zip.extract(filename,os.path.join(zipdirname,transformationpackage))
                             shutil.copy(filename,os.path.join(zipdirname,PACKAGETIMESTAMP,PACKAGETIMESTAMP + os.extsep + osext))
                             os.unlink  (filename)
                      os.chdir(zipdirname)
                      if os.path.isdir(os.path.join(zipdirname,transformationpackage)) == True:
                         os.removedirs(os.path.join(zipdirname,transformationpackage))
                         options.mode = 'PARAM'
                   zip.close()
    except EnvironmentError, err:
       print err
    except csv.Error , err :
       print err
    except IOError , (errno,err):												   
       print err
    except ConfigParser.Error ,err:
       print err
    except:
       kind, val, traceback = sys.exc_info()
       raise kind, val, traceback

    #LOADER COMMANDLINE PARAMETER
    
    options.transporttype               = string.upper(options.transporttype)
    options.transformationtype          = string.upper(options.transformationtype)
    options.transformationpackage       = string.upper(options.transformationpackage)
    options.transformationpart          = string.upper(options.transformationpart)
    options.streammedium                = string.upper(options.streammedium)
    options.streamformat                = string.upper(options.streamformat)
    options.streamformat_target         = string.upper(options.streamformat_target)
    options.streamcharencoding          = string.upper(options.streamcharencoding)
    options.streamlob                   = string.upper(options.streamlob)
    options.startposition               = string.upper(options.startposition)
    options.migration                   = string.upper(options.migration)
    
    if options.streamlob == 'TABLE':
       FIELDLOBMASK       = ''
    if options.streamlob == 'COLUMN':
       FIELDLOBMASK       = ''
    if options.streamlob == 'COLUMNVALUE':
       FIELDLOBMASK       = '####'
       
    #INTER
    if options.transformationtype       ==  '' :
       options.mode = 'INTER'
    
    if(options.transformationtype       == 'TRANSPORT' and (
       options.transformationpackage    == '' or
       options.transformationpart       == '' or
       options.streammedium             == '' or
       options.streamformat             == '' or
       options.streamformat_target      == '' or
       options.streamlob                == '' or
       options.sourcedbnode	            == '' or
       options.sourcedbname	            == '' or
       options.sourcedbuser	            == '' or
       options.targetdbnode	            == '' or
       options.targetdbname	            == '' or
       options.targetdbuser             == '' or
       options.mode                     == 'INTER')):
       if options.transformationpackage == 'DB' :
          if TESTLOADER == 'FALSE':
             options.transformationpart  = 'DB'
       if options.streamformat          == 'PAGES':
          options.streamformat_target    = 'PAGE'
          options.streamlob              = 'TABLE'
       if options.startposition         == '' :
          options.startposition          = 'START'
       options.mode = 'INTER'
    
    if(options.transformationtype       == 'EXPORT' and (
       options.sourcedbnode	            == '' or
       options.sourcedbname	            == '' or
       options.sourcedbuser	            == '' or
       options.transformationpackage    == '' or
       options.transformationpart       == '' or
       options.streammedium             == '' or
       options.streamformat             == '' or
       options.streamformat_target      == '' or
       options.streamlob                == '' or
       options.mode                     == 'INTER')):
       if options.transformationpackage == 'DB' :
          if TESTLOADER == 'FALSE':
             options.transformationpart  = 'DB'
       if options.streamformat          == 'PAGES':
          options.streamformat_target    = 'PAGE'
          options.streamlob              = 'TABLE'
       if options.startposition         == '' :
          options.startposition          = 'START'
       options.mode = 'INTER'
    
    if(options.transformationtype       == 'IMPORT' and (
       options.transformationpackage    == '' or
       options.sourcedbnode	            == '' or
       options.sourcedbname	            == '' or
       options.targetdbnode	            == '' or
       options.targetdbname	            == '' or
       options.targetdbuser             == '' or
       options.mode                     == 'INTER')) :
       options.mode = 'INTER'
    #PARAM
    if(options.transformationtype       != '' and
       options.sourcedbnode	            != '' and
       options.sourcedbname	            != '' and
       options.sourcedbuser	            != '' and
       options.transformationpackage    != '' and
       options.transformationpart       != '' and
       options.streammedium             != '' and
       options.streamformat             != '' and
       options.streamformat_target      != '' and
       options.streamcharencoding       != '' and
       options.streamlob                != '' and
       options.startposition            != '' and
       options.mode                     != 'INTER'):
       options.mode = 'PARAM'
       
def put_environment(options):
    # Transformation Resources
    if options.transformationtype == 'TRANSPORT' or options.transformationtype == 'EXPORT' or options.transformationtype == 'IMPORT':
       print '\nData source parameters'
       print 'Node             : %s ' % options.sourcedbnode     
       print 'Db               : %s ' % options.sourcedbname
       if options.transformationtype == 'TRANSPORT' or options.transformationtype == 'EXPORT': 
          username = re.split(',', options.sourcedbuser)[0]
          password = re.split(',', options.sourcedbuser)[1]
          if username != '' and username [0] != '"' and username[len(username)-1] != '"':
             username = string.upper(username)
          elif username != '':
             username = string.replace(username,'"','')    
          print 'User             : %s ' % username
    if options.transformationtype == 'TRANSPORT' or options.transformationtype == 'IMPORT':
       print '\nData target parameters'
       print 'Node             : %s ' % options.targetdbnode
       print 'Db               : %s ' % options.targetdbname
       username = re.split(',', options.targetdbuser)[0]
       password = re.split(',', options.targetdbuser)[1]
       if username != '' and username [0] != '"' and username[len(username)-1] != '"':
          username = string.upper(username)
       elif username != '':
          username = string.replace(username,'"','') 
       print 'User             : %s ' % username
    print '\nTransformation package parameters'
    if options.transformationtype == 'TRANSPORT':
       #print 'Package Path     : %s ' % os.path.split(options.sourceindepdatapath)[0]
       print 'Export  Package  : %s ' % os.path.split(options.sourceindepdatapath)[1]
       print 'Import  Package  : %s ' % os.path.split(options.targetindepdatapath)[1]
    if options.transformationtype == 'EXPORT':
       #print 'Package Path     : %s ' % os.path.split(options.sourceindepdatapath)[0]
       print 'Export  Package  : %s ' % os.path.split(options.sourceindepdatapath)[1]
    if options.transformationtype == 'IMPORT':
       #print 'Package Path     : %s ' % os.path.split(options.targetindepdatapath)[0]
       print 'Import  Package  : %s ' % os.path.split(options.targetindepdatapath)[1]
    print    'Processing       : %s ' % options.transporttype
    print    'Type             : %s ' % options.transformationtype
    print    'Package          : %s ' % options.transformationpackage
    if options.streamformat_target != 'DTL':
       print 'Stream           : %s ' % "CATALOG, PACKAGE, DATA "
    if options.streamformat_target == 'DTL':
       print 'Stream           : %s ' % "CATALOG, PACKAGE "
    print    'Part             : %s ' % options.transformationpart
    print    'Medium           : %s ' % options.streammedium
    print    'Format Transport : %s ' % options.streamformat
    print    'Format Import    : %s ' % options.streamformat_target
    streamcharencoding = ''
    if options.streamcharencoding == '':
       streamcharencoding = 'DYNAMIC'
    if options.streamcharencoding != '':
       streamcharencoding = options.streamcharencoding
    print    'Char   Encoding  : %s ' % streamcharencoding
    print    'Lob    Transport : %s ' % options.streamlob  
    print '\nConfiguration parameters'
    print    'VERBOSE          : %s ' % VERBOSE
    print    'MONITOR          : %s ' % MONITOR
    print    'MAXERRORCOUNT    : %s ' % MAXERRORCOUNT
    
if __name__ == '__main__':
    optlib.optMain2 (main, _options ())
    
       
 
