#!/usr/bin/perl
#    ========== licence begin  GPL
#    Copyright (c) 2005 SAP AG
#
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of the GNU General Public License
#    as published by the Free Software Foundation; either version 2
#    of the License, or (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#    ========== licence end


package SAPDB::Install::Templates::Base;


sub BEGIN {
  @ISA = ('SAPDB::Install::Exporter');
  @EXPORT = qw( createThis  matchPath matchNumber matchUserPassword
                matchStorage getGeneric  parseConfigLine  mymkdir 
                fileparse basename file_name_is_absolute catdir  catfile  path canonpath 
                dirname mkpath rmtree isPathAbsolute  matchString matchBoolean
              );
  my $repo = SAPDB::Install::Repository::GetCurrent ();
  my @neededPackages=(
                      'DBMCmd',
                      'BackupPipe',
                      'Untgz',
                      'StdIO',
                      'InstInfo',
                      'Registry',
                      'Values'

                     );
  foreach my $package (@neededPackages) {
    unless (defined $repo->Eval ("SAPDB::Install::$package", 1.01)) {
      print join ("\n", $repo->GetErr)."\n";
      die ("error loading:  SAPDB::Install::$package\n");
    }
    SAPDB::Install::Exporter::import ("SAPDB::Install::$package"); 
  }
  bootstrap SAPDB::Install::DBMCmd;
}

# Instance-Variabless
#+field(String:Database Name)
my $sdb_name= "sdb_name";

#+field(String:Name of DBM User)
my $sdb_dbmuser= "sdb_dbmuser";

#+field(String:Password of DBM User)
my $sdb_dbmpass= "sdb_dbmpass";

#+field(String:Name of DBA User)
my $sdb_dbauser= "sdb_dbauser";

#+field(String:Password of DBA User)
my $sdb_dbapass= "sdb_dbapass";

#+field(String:Name of  SQL User)
my $sdb_sqluser= "sdb_sqluser";

#+field(String:Password of  SQL User)
my $sdb_sqlpass= "sdb_sqlpass";

#+field(String:Password of  Domain User)
my $sdb_domainpass= "sdb_domainpass";

#+field(HashRef:Database Parameters)
my $sdb_params= "sdb_params";

#+field(ListRefOfListRefs:Database Volumes)
my $sdb_volumes= "sdb_volumes";

#+field(FuncRef:Pointer To Function for Logging)
my $outputter= "outputter";

#+field(FuncRef:Pointer to Function for Status Messages)
my $statusputter= "statusputter";

#+field(boolean:On/Off Auto-Extend)
my $autoextend= "autoextend";

#+field(boolean:On/Off Log Auto Overwrite)
my $log_auto_overwrite= "log_auto_overwrite";

#+field(boolean:NEEDED "Depend"-Path)
my $dependend_path= "dependend_path";

#+field(String:Declaration for a Medium)
my $mediumDeclaration= "mediumDeclaration";

#+field(String:Named pipe for Recover)
my $recoverFromPipe= "recoverFromPipe";

#+field(Path:Path for installation packages)
my $archiveDir='archiveDir';

#+field(Type:Type of the Template)
my $typeKey='typeKey';


#+field(Type:doUpdateStatistics)
my $doUpdateStatistics='doUpdateStatisticsKey';

#+sub():String:Get the type of the template
sub getDoUpdateStatistics{
  my $this = shift;
  return $this->{$doUpdateStatisticsKey};
}
sub setDoUpdateStatistics{
  my $this = shift;
  my $val = shift;
  $this->{$doUpdateStatisticsKey}  = $val;
}


#+field(Type:doUpdateStatistics)
my $dbmcliPath='dbmcliPathKey';
#+sub():String:Get the type of the template
sub getDbmcliPath{
  my $this = shift;
  my $retVal = $this->{$dbmcliPath} ;
  $retVal = $retVal ? $retVal : "dbmcli";
}
sub setDbmcliPath{
  my $this = shift;
  my $val = shift;
  $this->{$dbmcliPath}  = $val;
}



#+field(Type:doUpdateStatistics)
my $xserverPathKey='xserverPathKey';
#+sub():String:Get the type of the template
sub getXServerPath{
  my $this = shift;
  my $retVal = $this->{$xserverPathKey} ;
  $retVal = $retVal ? $retVal : "x_server";
}
sub setXServerPath{
  my $this = shift;
  my $val = shift;
  $this->{$xserverPathKey}  = $val;
}








#+field(HashRef:Generic parameters)
my $generic='genericKey';



#+field(FuncRef:Reference to function for calling dbmcli-commands)
my $scriptHandler='scriptHandler';


#+sub():Hash:Get all generics
sub getAllGenerics($){
  my ($this) = $_[0];
  return $this->{$generic};
}

#ifdef sub:addGeneric
#<p>Type is one of</p>
#<ul>
#<li><code>"Path"</code></li>
#<li><code>"UserPassword"</code></li>
#<li><code>"Number"</code></li>
#<li><code>"Storage"</code></li>
#<li><code>"String"</code></li>
#<li><code>"Boolean"</code></li>
#</ul>
#<p>Format: "<code>!&lt;TYPE&gt;: &lt;KEY&gt;  &lt;VALUE&gt;</code>", beispielsweise</p>
#<pre>
#!String: SLOPPY yes,please
#</pre>
#endif

#+sub(String:Key|String:Value|Type:type of the value):void:Add a generic parameter
sub addGeneric($$$$){
  my ($this,$key,$value,$type) = @_;
  my $genericHash = $this->getAllGenerics();
  $genericHash->{$key}=[$value,$type];
}

#+sub(String:key):List(value,type):Get the generic specified by key
sub getGeneric($$){
  my ($this,$key) = @_;
  my $genericHash = $this->getAllGenerics();
  return  $genericHash->{$key};
}


#+sub(String:key):value:Get the generic value specified by key
sub getGenericValue($$){
  my ($this,$key) = @_;
  my $aListRef = $this->getGeneric($key);
  my $aValue=undef;
  if ( defined $aListRef) {
    $aValue = $aListRef->[0];
  }
  return  $aValue;
}

#+sub(String:key):value:Get the type of a generic  value specified by key
sub getGenericType($$){
  my ($this,$key) = @_;
  my $aListRef = $this->getGeneric($key);
  my $aValue=undef;
  if ( defined $aListRef) {
    $aValue = $aListRef->[1];
  }
  return  $aValue;
}



#+sub():String:Get the type of the template
sub getType($){ return $_[0]{$typeKey}; }
sub setType($$){ $_[0]{$typeKey}=$_[1]; }






#ifdef sub:setRecoverFromPipe
#endif
#+sub(String:Medium Declaration):void:Set the pipe to recover from
sub setRecoverFromPipe(){
  my ($this,$pipeName)=@_;
  $this->{ $recoverFromPipe}=$pipeName;
  return $pipeName;
}
#setRecoverFromPipe()



#Legt ein Verzeichnis an, und wenn ntig die darber befindlichen Unterverzeichnisse.
sub mymkdir($){
  if ($^O =~ /mswin/i) {
    $_[0] =~ /^(\w):\\(.*)$/;
    my $drive = $1;
    my $path = $2;

    unless ( (defined $path) && ($path =~ m/\S+/) ){
      return "Invalid path specified.";
    }
    my @aSplit = reverse (split /\\/,$path);
    my $aSub = (defined $drive)? ($drive . ":\\" .  pop @aSplit) : (pop @aSplit) ;
    while ( 1) {
      unless( -d $aSub){
        mkdir($aSub,0777);
      }
      unless (-d $aSub){
        return "Could not create directory $aSub!";
      }
      my $atom = pop @aSplit;
      unless ($atom){
        last ;
      }
      $aSub = $aSub . "\\" . $atom;
    }
  } else {
    #Momentan nur Windows implementiert.
  }
  return 0;
}
#mkdir()






#+sub():String:Get the named pipe to recover from
sub getRecoverFromPipe(){
  my ($this)=@_;
  return $this->{ $recoverFromPipe};
}







#+sub():String:Get the Medium Declaration.
sub getMediumDeclaration(){
  my ($this) = @_;
  my $aMediumDeclaration = $this->{$mediumDeclaration};
  my $aResult ; 
  my $aBackup = $this->getBackup();
  my $anArchive = $this->getArchive();
  my $pipeName = $this->getRecoverFromPipe();
  if (defined $aMediumDeclaration) {
    $aResult = $aMediumDeclaration;
  } elsif ((defined $aBackup) and (not defined $anArchive)) {
    $aResult = "SDBINSTBACKUP " . $aBackup . " FILE DATA 0 8 YES";
  } elsif ((defined $aBackup) and (defined $anArchive)) {

    if( $^O =~ /mswin/i){
      $aResult =  "$pipeName \\\\.\\pipe\\$pipeName PIPE DATA 0 0 YES"; 
    }
    else{
      $aResult =  "$pipeName /opt/sdb/$pipeName PIPE DATA 0 0 YES"; 
    }

    #Windows-Fall
  }
  return $aResult;
}
#getMediumDeclaration()


my $archive="archive";
#ifdef sub:setArchive
#<p>Set the archive to be used to read the backup from for db_activate RECOVER</p>
#endif
#+sub(String:An Archive):void:Set the archive
sub setArchive(){ 
  my ($this,$anArchive)=  @_;
  $this->{$archive}=$anArchive; 
  if (defined $anArchive) {
    $this->setRecoverFromPipe("SDBINSTPIPE");
  } else {
    $this->setRecoverFromPipe(undef);
  }
}

#+sub():String:Get the Archive.
sub getArchive(){ $_[0]{$archive}; }

my $backup="backup";
#ifdef sub:setBackup
#<p>Set the backup to be used  for db_activate RECOVER ...</p>
#endif
#+sub(String:A Backup):void:Set the backup
sub setBackup(){
  my ($this,$aBackup)=  @_;
  $this->{$backup}=$aBackup; 
}
#setBackup()

#+sub():String:Get the backup
sub getBackup(){ 
  my ($this)=  @_;
  return $this->{$backup}; 
}
#getBackup()





#+sub(bool:autoext):bool:Set On/Off autoextend feature on/off
sub setAutoExtend(){ $_[0]{$autoextend}=$_[1]; }

#+sub():bool:Is autoextend on/off
sub getAutoExtend(){ $_[0]{$autoextend}; }

#+sub(bool:autoext):bool:Set auto log overwrite on/off
sub setLogAutoOverwrite(){ $_[0]{$log_auto_overwrite}=$_[1]; }

#+sub():bool:Is auto log overwrite on/off
sub getLogAutoOverWrite(){ $_[0]{$log_auto_overwrite}; }

#+sub():String:Get the name of the Database
sub getDBName(){ $_[0]{$sdb_name}; }
#+sub(String:name):void:Set the name of the Database
sub setDBName(){ $_[0]{$sdb_name}=$_[1]; }

#+sub():String:Get database manager
sub getDBM(){ $_[0]{$sdb_dbmuser}; }

#+sub(String:name):void:Set database manager 
sub setDBM(){ $_[0]{$sdb_dbmuser}=$_[1]; }

#+sub():String:Get database manager password
sub getDBMPass(){ $_[0]{$sdb_dbmpass}; }
#+sub(String:password):void:Set database manager password
sub setDBMPass(){ $_[0]{$sdb_dbmpass}=$_[1]; }
#+sub():String:Get name of Database Administrator
sub getDBA(){ $_[0]{$sdb_dbauser}; }
#+sub(String:name):void:Set name of Database Administrator
sub setDBA(){ $_[0]{$sdb_dbauser}=$_[1]; }
#+sub():String:Get password of Database Administrator
sub getDBAPass(){ $_[0]{$sdb_dbapass}; }
#+sub(String:password):void:Set name of Database Administrator
sub setDBAPass(){
  $_[0]{$sdb_dbapass}=$_[1];
  #  $_[0]{$sdb_domainpass}=$_[1];
}
#+sub():String:Get password of Domain user
sub getDomainPass(){ $_[0]{$sdb_domainpass}; }
#+sub(String:password):void:Set password of Domain user
sub setDomainPass(){
  $_[0]{$sdb_domainpass}=$_[1]; 
}
#+sub():String:Get name of SQL-User
sub getSQLU(){ $_[0]{$sdb_sqluser}; }
#+sub(String:name):void:Set name of SQL-User
sub setSQLU(){ $_[0]{$sdb_sqluser}=$_[1]; }
#+sub():String:Get password of SQL-User
sub getSQLPass(){ $_[0]{$sdb_sqlpass}; }
#+sub(String:password):void:Set password of SQL-User
sub setSQLPass(){ $_[0]{$sdb_sqlpass}=$_[1]; }




#ifdef sub:setCreationStrategy
#<p>Set strategy for instance creation. Strategies are:</p>
#<ul>
#<li><strong>"1": </strong>Do not check database existance, just create</li>
#<li><strong>"2": </strong>Fail if database exists: 
#   <ul class="c3">
#       <li>Fail if goodness &gt; <strong>1</strong> or <strong>0</strong></li>
#   </ul>
#</li>
#<li><strong>"3": </strong>Delete database, if it exists:
#   <ul class="c3">
#    <li> Create the instance  if goodness is <strong>1</strong> </li>
#    <li> Delete and recreate it, if   goodness &gt;  <strong>3</strong>.</li>
#    <li> Fail if goodness is <strong>2,3</strong> or <strong>0</strong> </li>
#   </ul>
#</li>
#<li><strong>"4": </strong> Try to reuse the database:
#   <ul class="c3">
#    <li> Create the instance  if goodness is <strong>1</strong> </li>
#    <li> Delete and recreate it, if     <strong>4</strong> or  <strong>5</strong>.</li>
#    <li> Try to reuse the instance, if its goodness is &gt;= <strong>6</strong>. </li>
#    <li> Fail if goodness is <strong>2,3</strong> or <strong>0</strong> </li>
#   </ul>
#</li>
#</ul>
#endif
#+field(int:creationStrategy)
my $creationStrategy="creationStrategyKey";
#+sub(int):void:Set creationStrategy
sub setCreationStrategy{
  my ($this, $aValue) = @_;
  $this->{$creationStrategy}=$aValue;
}
#+sub():int:Get creationStrategy
sub getCreationStrategy{
  my ($this ) = @_;
  return $this->{$creationStrategy};
}

#ifdef sub:setDeletionStrategy
#<p>Set strategy for instance deletion. Strategies are:</p>
#<ul>
#<li><strong>"1": </strong>delete without files</li>
#<li><strong>"2": </strong>delete with files</li>
#</ul>
#endif

#+field(int:deletionStrategy)
my $deletionStrategy="deletionStrategyKey";
#+sub(int):void:Set deletionStrategy
sub setDeletionStrategy{
  my ($this, $aValue) = @_;
  $this->{$deletionStrategy}=$aValue;
}
#+sub():int:Get deletionStrategy
sub getDeletionStrategy{
  my ($this ) = @_;
  return $this->{$deletionStrategy};
}



sub writeConfig($$){
  my ($this,$write) = @_;
  my ($sec,$min,$our,$mday,$mon,$year,$wday,$yday,$isdst) = localtime(time);
  $year = 1900 + $year;
  &$write("#############################################################\n");
  &$write("#START OF CONFIGURATION DATA FOR MAXDB INSTANCE CREATION ....\n");
  &$write("#GENERATED ON: $mday.$mon.$year $our:$min:$sec\n");
  &$write("[Template]\n");
  &$write("TYPE " . $this->getType() . "\n");
  &$write("\n");
  &$write("[Software]\n");
  &$write("INSTROOTPATH " . $this->getInstRootPath() . "\n");
  &$write("\n");
  &$write("[DBInstance]\n");
  my $aDependPath = $this->getDependendPath();
  &$write("DBNAME " . $this->getDBName() . "\n");
  &$write("DEPENDPATH " . (defined $aDependPath? $aDependPath : "") . "\n");

  #write configuration info from sub classes
  $this->writeInstanceConfig($write);

  my $autoRestart = $this->getAutoRestart();
  my $isAutoRestart = defined $autoRestart ? $autoRestart :  "NO";
  &$write("AUTORESTART " . $isAutoRestart . "\n");
  &$write("#Users and passwords\n");
  &$write("DBM  " . $this->getDBM() . "," . $this->getDBMPass() .  "\n");
  &$write("DBA  " . $this->getDBA() . "," . $this->getDBAPass() .  "\n");
  my $sqlUser = $this->getSQLU();
  if (defined $sqlUser) {
    &$write("SQLUSER  " . $this->getSQLU() . "," . $this->getSQLPass() .  "\n");
  }
  &$write("DOMAINPASS " . $this->getDomainPass() . "\n");
  &$write("#Paths\n");

  my $anArchive = $this->getArchive();
  my $aBackup = $this->getBackup();

  &$write("ARCHIVE " . (defined $anArchive ? $anArchive : "") . "\n");
  &$write("BACKUP " . (defined $aBackup ? $aBackup : "") . "\n");

  &$write("CREATE_STRATEGY " . $this->getCreationStrategy() . "\n");
  &$write("DELETE_STRATEGY " . $this->getDeletionStrategy() . "\n");

  my $updateStatistics = $this->getDoUpdateStatistics();
  $updateStatistics = $updateStatistics ? $updateStatistics : '';
  &$write("UPDATE_STATISTICS " . $updateStatistics . "\n");



  &$write("\n");
  &$write("[DBParam]\n");
  my %dbparams = $this->getDBParams();
  while ((my $key,my $value) = each %dbparams) {
    &$write("$key   $value\n");
  }
  &$write("\n");
  &$write("[Generic]\n");
  my $generics = $this->getAllGenerics();
  while ((my $key,my $value) = each %{$generics}) {
    my $pair = $this->getGeneric($key);
    &$write("!$pair->[1]: $key $pair->[0]\n");
  }
  &$write("\n");
  &$write("################\n");
  &$write("#GENERATED DATA\n");
  &$write("################\n");
  &$write("\n");
  &$write("#VOLUMES:   \n");
  my @volumes = $this->getDBVolumes();
  foreach my $volume (@volumes) {
    &$write( "#[Nummer=" . @$volume[0]
             . ",Art=" . @$volume[1]
             . ",Ort=" . @$volume[2]
             . ",Typ=" . @$volume[3]
             . ",Groesse=" . @$volume[4]
             . "]\n");
  }
  &$write("\n");
  my $fullDataSize = $this->getFullDataVolumeSize();
  &$write("#Complete datasize: " . $fullDataSize . " pages, (" . $fullDataSize * 8 / 1024 . " MB)\n");
  my $fullLogSize = $this->getFullLogVolumeSize();
  &$write("#Complete log size: " . $fullLogSize . " pages, (" . $fullLogSize * 8 /1024 . " MB)\n");
  my $pipeName = $this->getRecoverFromPipe();
  if ( defined $pipeName) {
    &$write("#Recover from pipe: " . $pipeName . "\n");
  }
  my $mediumDeclaration = $this->getMediumDeclaration();
  if ( defined $mediumDeclaration) {
    &$write("#Medium declaration: \"" . $mediumDeclaration . "\"\n");
  }
  my $backup = $this->getBackup();
  if ( defined $backup) {
    &$write("#Backup: \"" . $backup . "\"\n");
  }
  my $archive = $this->getArchive();
  if ( defined $archive) {
    &$write("#Archive: \"" . $archive . "\"\n");
  }
  
  &$write("#END OF CONFIGURATION DATA FOR MAXDB INSTANCE CREATION ....\n");
  &$write("#############################################################\n");
}
#end writeConfig();











#ifdef sub:isRegistered
#<p>Uses <code>SAPDB::Install::Registry</code> to get info if database is registered
#  Return values are </p>
#<ul>
#<li><code>0</code>: Database not found</li>
#<li><code>1</code>: Database found</li>
#</ul>
#endif

#+sub():int:Is the instance registered (Windows: Windows Registry, Unix: DB Registry)
sub isRegistered{
  my ($this ) = @_;
  my $aDBName =$this->getDBName();
  my $retVal = 0;
  $this->addGeneric("registeredString", "Registration of  database $aDBName not found","String");
  my %instance = getInstances();
  if ( defined $instance{$aDBName}) {
    $retVal=1;
    $this->addGeneric("registeredString", "Database $aDBName is registered","String");
  }
  return  $retVal;
}

#ifdef sub:getGoodness
#<p>Uses <code>SAPDB::Install::InstInfo</code> to get info about the database.
#  Return values are </p>
#<ul>
#<li><code>0</code>: General Info not available </li>
#<li><code>1</code>: Database info not available (Database seems not to exist)</li>
#<li><code>2</code>: db software path does not match</li>
#<li><code>3</code>: Could not  log on  as dbm</li>
#<li><code>4</code>: Not enough data size </li>
#<li><code>5</code>: Not enough log size</li>
#<li><code>6</code>: db is pretty good</li>
#</ul>
#endif

#+sub():int:Get the goodness of the database
sub getGoodness{
  my ($this ) = @_;
  my $aDBName =$this->getDBName();
  my $retVal = 0;
  $this->addGeneric("goodnessString", "General db info  not available, numeric goodness is $retVal","String");
  my $aDependendPath = $this->getDependendPath();
  my $info = SAPDB::Install::InstInfo::new ($aDBName);
  if ( defined $info) {
    $retVal = 1;
    $this->addGeneric("goodnessString", "General db info available","String");
  } else {
    return $retVal;
  }

  my $infoListRef =  $info->{'database'};
  my $dbInfo = $infoListRef->{$aDBName};
  if ( defined $dbInfo) {
    $retVal = 2;
    $this->addGeneric("goodnessString", "Info  on $aDBName available","String");
  } else {
    $this->addGeneric("goodnessString", "Info  on $aDBName unavailable","String");
    return $retVal;
  }
    
  #      my $realDBName = $dbInfo->{'dbname'};
  #      my $dbowner = $dbInfo->{'dbowner'};
  #      my $paramfile = $dbInfo->{'paramfile'};
  my $dbswdir = $dbInfo->{'dbswdir'};
  my $path1= $dbswdir;
  my $path2= $aDependendPath;
  $path1 =~ s/\\/\//g;
  $path2 =~ s/\\/\//g;
  $path1 =~ s/\/$//g;
  $path2 =~ s/\/$//g;
  if ( $^O =~ /mswin/i ) {
    $path1 = uc $path1;
    $path2 = uc $path2;
  }
  if ( $path1 eq $path2) {
    $retVal=3;
    $this->addGeneric("goodnessString", "DB Software path matches for $aDBName, numeric goodness is $retVal)","String");
  } else {
    $this->addGeneric("goodnessString", 
                      "Database $aDBName found, but db software path is \"$dbswdir\" " . 
                      "and not \"$aDependendPath\" (Goodness $retVal)",
                      "String");
    return $retVal;
  }


  my $dbm = $this->createDBMCmd();
  if (defined $this->logonAsDBM($dbm)) {
    $retVal=4;
    $this->addGeneric("goodnessString", "Logged on as database manager to  $aDBName, numeric goodness is $retVal.","String");
  } else {
    return $retVal;
  }
  my $fullDataSize = $this->getFullDataVolumeSize();
  my $realFullDataSize = $this->getRealFullDataVolumeSize($dbm);
  if ( $realFullDataSize >= $fullDataSize) {
    $retVal=5;
    $this->addGeneric("goodnessString", 
                      "i$aDBName: Sufficient data size " . 
                      "(Have $realFullDataSize, need $fullDataSize, numeric goodness is $retVal]","String");
  } else {
    $this->addGeneric("goodnessString", 
                      "$aDBName: Insufficient data size " . 
                      "[Have $realFullDataSize, need $fullDataSize, numeric goodness is $retVal]","String");
    return $retVal;
  }

  my $fullLogSize = $this->getFullLogVolumeSize();
  my $realFullLogSize = $this->getRealFullLogVolumeSize($dbm);
  if ( $realFullLogSize >= $fullLogSize) {
    $retVal=6;
    $this->addGeneric("goodnessString", 
                      "$aDBName is pretty good" . 
                      ", [Numeric goodness is $retVal]","String");
  } else {
    $this->addGeneric("goodnessString", 
                      "$aDBName: Insufficient log size " . 
                      "[Have $realFullLogSize, need $fullLogSize, numeric goodness is $retVal]","String");
    return $retVal;
  }

  return $retVal;
}
#getGoodness

sub getGoodnessString{
  my ($this ) = @_;
  return $this->getGenericValue("goodnessString");
}
#getGoodnessString

sub getRegisteredString{
  my ($this ) = @_;
  return $this->getGenericValue("registeredString");
}
#getRegisteredString





#+sub():Hash:Get all DB-Parameters
sub getDBParams(){
  my ($this) = $_[0];
  %{$_[0]{$sdb_params}};
} 
#ifdef sub:setDBParams
#<pre>my $dbi = SAPDB::Install::DBIDesc::new();
#     $dbi->setDBParams({
#                            "MAXUSERTASKS", 5 ,
#                            "NOCHNKEY", "NOCHWERT" 
#                       });</pre>
#endif
#+sub(HashRef:DB-Params):void:Set all Database parameters
sub setDBParams(){ $_[0]{$sdb_params}=$_[1]; }
#ifdef sub:addDBParam
#<pre>my $dbi = SAPDB::Install::DBIDesc::new();
#  $dbi->addDBParam("MAXUSERTASK", 5);</pre>
#endif
#+sub(String:key|String:value):void:Add one Database Parameter
sub addDBParam(){
  my ($this,$key,$value) = @_;
  $this->{$sdb_params}->{$key} = $value;
}



#ifdef sub:getDBVolumes
#<pre><![CDATA[
#  my $dbi = SAPDB::Install::DBIDesc::new();
#  my @volumes =  $this->getDBVolumes();
#  my @volume_sets;
#  foreach my $volume (@volumes) {
#    print ( "[Anzahl=" . @$volume[0]
#      . ",Art=" . @$volume[1]
#        . ",Name=" . @$volume[2]
#          . ",Typ=" . @$volume[3]
#            . ",Groesse=" . @$volume[4]
#              . "]\n");
#  }]]>
#</pre>
#endif
#+sub():ListOfListRefs:Get all DB-Volumes
sub getDBVolumes(){
  my ($this) = $_[0];
  @{$this->{$sdb_volumes}}; 
}


#ifdef sub:setDBVolumes
#<pre><![CDATA[
#  my $dbi = SAPDB::Install::DBIDesc::new();
#  $dbi->setDBVolumes( [  
#                        #    Anzahl Art   Name      Typ  Groesse
#                        [qw (1      LOG   LOG_001   F    2000)   ],
#                        [qw (1      DATA  DAT_001   F    10000)  ]
#                         ]);
#  }]]>
#</pre>
#endif
#+sub(ListRefOfListRefs:volumes):void:Set all database volumes
sub setDBVolumes(){ $_[0]{$sdb_volumes}=$_[1]; }

#ifdef sub:addDBVolume
#<pre><![CDATA[
# my $dbi = SAPDB::Install::DBIDesc::new();
# $dbi->addDBVolume("1","LOG","LOG_002","F","3000");
#  ]]>
#</pre>
#endif
#+sub(number:anzahl|String:art|String:ort|String:typ|Number:groesse):void:Add one database volume
sub addDBVolume(){
  #              Nummer Art   Name      Typ  Groesse
  my ($this,$anzahl,$art,$name,$typ,$groesse) = @_;
  push @{$_[0]{$sdb_volumes}},[$anzahl,$art,$name,$typ,$groesse];
}



#+sub():Pages:Get the complete data size (sum of size of all data volumes)
sub getFullDataVolumeSize{
  my ($this) = @_;
  return $this->getFullVolumeSizeByType("DATA");
}
#getFullDataVolumeSize


#+sub():Pages:Get the complete data size (sum of size of all data volumes)
sub getFullLogVolumeSize{
  my ($this) = @_;
  return $this->getFullVolumeSizeByType("LOG");
}
#getFullDataVolumeSize


#+sub():Pages:Get the complete data size (sum of size of all data volumes)
sub getFullVolumeSizeByType{
  my ($this,$art) = @_;

  my @volumes = $this->getDBVolumes();
  my $fullSize=0;
  foreach my $volume (@volumes) {
    my $aArt = @$volume[1];
    my $aSize = @$volume[4];
    if ($aArt =~ /^\s*$art\s*$/i && defined $aSize) {
      $fullSize += $aSize;
    }

    #     $this->showOutput( "[Nummer=" . @$volume[0]
    #                        . ",Art=" . @$volume[1]
    #                        . ",Ort=" . @$volume[2]
    #                        . ",Typ=" . @$volume[3]
    #                        . ",Groesse=" . @$volume[4]
    #                        . "]\n");
  }
  #foreach
  return $fullSize;
}
#getFullDataVolumeSize





sub dbmcmdAddAllVolumes($$@){
  my ($this,$dbm) = @_;
  my @volumes =  $this->getDBVolumes();
  foreach my $volume (@volumes) {
    unless (defined $this->dbmcmdAddOneVolume($dbm,@$volume)){
      return undef;
    }
  }
  return 0;
}
#dbmcliAddAllVolumes


sub dbmcmdAddOneVolume($$@){
  my ($this,$dbm, @volume) = @_;
  my $statement = "param_addvolume " . $volume[0]
    . " " . $volume[1]
      . " " . $volume[2]
        . " " . $volume[3]
          . " " . $volume[4]
            . "";

  my $retVal = $this->dbm($dbm,$statement,"ERROR: Could not add volume!\n","Volume $volume[2] added.\n" );
  return $retVal;
}
#dbmcliAddOneVolume 


sub getAllRealVolumes(){
  my ($this,$dbm ) = @_;
  my $volRef = $dbm->param_getvolsall();
  unless(defined $volRef){
    $this-> dbmcmdError($dbm,"Cannot read existing volumes!\n");
  }
  return $volRef;
}



#+sub():Pages:Get the real complete data size (sum of size of all data volumes)
sub getRealFullDataVolumeSize{
  my ($this,$dbm) = @_;
  return $this->getRealFullVolumeSizeByType($dbm,"DATA_VOLUME");
}

#+sub():Pages:Get the real complete log size (sum of size of all log volumes)
sub getRealFullLogVolumeSize{
  my ($this,$dbm) = @_;
  return $this->getRealFullVolumeSizeByType($dbm,"LOG_VOLUME");
}

sub getRealFullVolumeSizeByType{
  my ($this,$dbm,$type ) = @_;
  my $fullDataSize = undef;
  my $realVols = $this->getAllRealVolumes($dbm);
  unless(defined $realVols){
    return $fullDataSize;
  }
  $fullDataSize = 0;
  while (my ($key, $val) = each (%$realVols)) {
    if (ref ($val) eq "HASH") {
      my $aVol = $key;
      my $aSize = $val->{'size'};
      my $aType = $val->{'type'};
      my $aName = $val->{'name'};
      if ($aVol =~ /^$type/i) {
        $fullDataSize+=$aSize;
      }
    } else {
    }
  }
  return $fullDataSize;
}









sub outputTimeStamp{
  my ($this) = $_[0];
  my ($sec,$min,$our,$mday,$mon,$year,$wday,$yday,$isdst) = localtime(time);
  $year = 1900 + $year;
  &{$this->getOutPutter()}("TIME: $mday.$mon.$year $our:$min:$sec\n");
}

sub showOutput{
  my ($this,$message) = @_;
  &{$this->getOutPutter()}($message);
}

sub showStatusMessage{
  my ($this,$message) = @_;
  &{$this->getStatusputter()}($message);
}

sub showOutputAndStatusMessage{
  my ($this,$message) = @_;
  $this->showOutput($message);
  $this->showStatusMessage($message);
}





# Gibt sein Argument nach Stdout:
sub stdOutPutter{
  print( $_[0] );
}

#+sub():FuncRef:Get the function for output (logging) 
sub getOutPutter{ $_[0]{$outputter}; }
#+sub(FuncRef:Function):void:Set the function for output
sub setOutPutter{ $_[0]{$outputter}=$_[1]; }



#+sub():FuncRef:Get the function for status messages
sub getStatusputter{ $_[0]{$statusputter}; }
#+sub(FuncRef:Function):FuncRef:Set the function for status messages
sub setStatusputter{ $_[0]{$statusputter}=$_[1]; }



#+sub():FuncRef:Get the dbmcli-Handler
sub getScriptHandler{ $_[0]{$scriptHandler}; }
#+sub(FuncRef:Function):void:Set the dbmcli-Handler
sub setScriptHandler{ $_[0]{$scriptHandler}=$_[1]; }

#+sub(String:dependend Path):void:Set the dependend path
sub setDependendPath{
  $_[0]{$dependend_path}=$_[1];
}

#+sub():FuncRef:Get the dependend path
sub getDependendPath{
  return $_[0]{$dependend_path};
}





my $dataPathKey='dataPathKey';
#+sub():String:Get path to data volumes
sub getDataPath($){ return $_[0]{$dataPathKey}; }
#+sub(String:path):void:Set path for data volumes
sub setDataPath{
  my ($this,  $aPath) = @_;
  my $aDataPath;
  if (defined $aPath) {
    if (isPathAbsolute($aPath)) {
      $aDataPath = $aPath;
    } elsif ("" eq $aPath) {
      my $aSID = $this->getDBName();
      my $aRootPath = $this->getInstRootPath();
      $aDataPath = catfile($aRootPath ,$aSID ,'data');
    } else {
      my $aSID = $this->getDBName();
      my $aRootPath = $this->getInstRootPath();
      $aDataPath = catfile($aRootPath ,$aSID ,$aPath);
    }
  } 
  $this->{$dataPathKey}=$aDataPath; 
  return  $aDataPath;
}

my $dataVolCountKey='dataVolCountKey';
#+sub():int:Get number of data volumes
sub getDataVolCount($){ return $_[0]{$dataVolCountKey}; }
#+sub(int:number):void:Set the number of data volumes
sub setDataVolCount($$){ $_[0]{$dataVolCountKey}=$_[1]; }

my $dataSizeKey='dataSizeKey';
#+sub():pages:Standard size of data volumes
sub getDataSize($){ return $_[0]{$dataSizeKey}; }
#+sub(pages:size):pages:Get the standard size of data volumes
sub setDataSize($$ ){ $_[0]{$dataSizeKey}=$_[1]; }

my $logPathKey='logPathKey';
#+sub():String:Get the path to log volumes
sub getLogPath($){ return $_[0]{$logPathKey}; }


#+sub(String:path):void:Set the path of log volumes
sub setLogPath{
  my ($this,  $aPath) = @_;
  my $aLogPath;
  if (defined $aPath) {
    if (isPathAbsolute($aPath)) {
      $aLogPath = $aPath;
    } elsif ("" eq $aPath) {
      my $aSID = $this->getDBName();
      my $aRootPath = $this->getInstRootPath();
      $aLogPath = catfile($aRootPath ,$aSID ,'log');
    } else {
      my $aSID = $this->getDBName();
      my $aRootPath = $this->getInstRootPath();
      $aLogPath = catfile($aRootPath ,$aSID ,$aPath);
    }
  } 
  $this->{$logPathKey}=$aLogPath; 
  return  $aLogPath;
}


my $logVolCountKey='logVolCountKey';
#+sub():int:Get number of log volumes
sub getLogVolCount($){ return $_[0]{$logVolCountKey}; }
#+sub(int:number):void:Set the number of log volumes
sub setLogVolCount($$){ $_[0]{$logVolCountKey}=$_[1]; }


my $logSizeKey='logSizeKey';
#+sub():pages:Standard size of log volumes
sub getLogSize($){ return $_[0]{$logSizeKey}; }
#+sub(pages:size):pages:Get the standard size of log volumes
sub setLogSize($$){ $_[0]{$logSizeKey}=$_[1]; }


my $drive='drive';
#+sub():String:Get the drive for the complete installation
sub getDrive($){ 
  my ($this) = @_;
  return $this->{$drive}; 
}
#+sub(String:drive):void:Set the drive for the complete installation
sub setDrive($$){
  my ($this, $aDrive) = @_;
  $this->{$drive}=$aDrive; 
  $this->setInstRootPath($this->rootPrefix($aDrive));
}






sub rootPrefix($$){
  my ($this, $aDrive) = @_;
  if ($^O =~ /mswin/i) {
    return catfile($aDrive . ":" , "sdb");
  } else {
    return catfile($aDrive,"sdb");
  }
}







# Known generics:
my $gnutar = "gnutar";




sub initializeVolumeDefinitions($){
  my ($this ) =  @_;
  my $aDataVolCount = $this->getDataVolCount();
  my $aLogVolCount = $this->getLogVolCount();
  my $aDataPath = $this->getDataPath();
  my $aLogPath = $this->getLogPath();
  my $aDataSize = $this->getDataSize();
  my $aLogSize = $this->getLogSize();
  $this->setDBVolumes([]);
  my $counter=0;
  while ( ++$counter <= $aDataVolCount) {
    my $aDataVolName = "DISKD000" . $counter ;
    my $aDataVolume =  (defined $aDataPath) ? catfile($aDataPath,$aDataVolName ) :$aDataVolName ;
    $this->addDBVolume ($counter, "DATA", $aDataVolume, "F", $aDataSize);
  }
  $counter=0;
  while ( ++$counter <= $aLogVolCount) {
    my $aLogVolName = "DISKL000" . $aLogVolCount ;
    my $aLogVolume =  (defined $aLogPath) ? catfile($aLogPath,$aLogVolName):$aLogVolName;
    $this->addDBVolume ($counter , "LOG", $aLogVolume, "F", $aLogSize);
  }
}
#initializeVolumeDefinitions





#+field(String:Installation Root)
my $instRootPath='instRootPath';
#+sub():String:Get installation root
sub getInstRootPath($){
  my ($this) = @_;
  return $this->{$instRootPath}; 
}


#ifdef sub:setInstRootPath
#<p>Implicitly sets dependend, data path and log path according to default</p>
#endif
#+sub(String:installPath):void:Set installation root path
sub setInstRootPath($$){
  my ($this, $anInstRootPath) = @_;
  $this->{$instRootPath}=$anInstRootPath;
  $this->setDependendPath("");
  $this->setDataPath("");
  $this->setLogPath("");
  return  $anInstRootPath;
}
#setInstRootPath




# Software-Parameter als Key/Value-Werte einlesen.
sub configureSoftwareFromKeyValuePairs($$$){
  my ($this,$key,$value) = @_;
  if ( $key =~ m/instrootpath/i) {
    $this->setInstRootPath(matchPath($value));
  } elsif ( $key =~ m/DRIVE/i) {
    $this->setDrive(matchPath($value));
  }
}
#configureSoftwareFromKeyValuePairs

# Vorkonfigurieren der Datenbankinstanz mit Werten aus der Parameterdatei,
# vorgeparst als Schluessel/Wert-Paare
sub configureInstanceFromKeyValuePairs($$$){
  my ($this,$key,$value) = @_;
  if ( $key =~ m/datavols/i) {
    $this->setDataVolCount(matchNumber($value));
  } elsif ( $key =~ m/datapath/i) {
    $this->setDataPath(matchPath($value));
  } elsif ( $key =~ m/datasize/i) {
    $this->setDataSize(matchStorage($value));
  } elsif ( $key =~ m/logvols/i) {
    $this->setLogVolCount(matchNumber($value));
  } elsif ( $key =~ m/logpath/i) {
    $this->setLogPath(matchPath($value));
  } elsif ( $key =~ m/logsize/i) {
    $this->setLogSize(matchStorage($value));
  }
}
#configureInstanceFromKeyValuePairs






#+sub():void:Create the package for db dump
sub createPackage($){
  my ($this) = @_;
  my $backup = $this->getBackup();
  my $backupFile = SAPDB::Install::Templates::Base::basename($backup);
  my $backupDir =  SAPDB::Install::Templates::Base::dirname($backup);
  my $scriptName = "instinstance";
  my $instanceListFile = "instance.lst";
  my $backupData = catfile("$backupDir" , "PACKAGEDATA");
  my $instanceData = catfile("$backupDir" , "INSTANCEDATA");
  my $instanceList = catfile("$backupDir" , $instanceListFile);
  my $script  =  catfile("$backupDir" , $scriptName);
  my $aDBName = $this->getDBName();
  my $packageName = "DB Instance";  

  my ($sec,$min,$our,$mday,$mon,$year,$wday,$yday,$isdst) = localtime(time);
  $year = 1900 + $year;

  open INSTANCEDATA, ">$instanceData";
  print INSTANCEDATA <<INSTANCEDATAEND;
Generated : $mday.$mon.$year $our:$min:$sec
Backup: $backup
INSTANCEDATAEND

  open PACKAGEDATA, ">$backupData";
  print PACKAGEDATA <<PACKAGEDATAEND;
INTERFACE_VERSION = "0.1"
PACKAGE_NAME = "$packageName"
PACKAGE_VERSION = "0"
MODE = "32"
SOFTWARE_VERSION = "7.5.00.04"
MIN_VERSION = "7.2.4.4"
TYPE = "FULL"
MAGIC_MODE = "NOBIT"
ALLOW_SKIP = "1"
DEFAULT_USER = "sdb"
DEFAULT_GROUP = "sdba"
PART_OF = "$packageName"
IS_TOP_OF = "$packageName"
IS_SUBPACKAGE = "1"
REQUIRE = "Database Kernel >= Release"
FILELIST = "${instanceListFile}"
SCRIPT = "${scriptName}"
PACKAGEDATAEND

  open INSTANCELIST, ">$instanceList";
  print INSTANCELIST <<INSTANCELISTEND;
INSTANCELISTEND

  open SCRIPT, ">$script";
  print SCRIPT <<SCRIPTEND;
#!/usr/bin/perl


sub BEGIN{
	my (\$repo) = \@SAPDB::Install::Repository::AllRepo;
	my \@neededPackages = (
		'Templates::Setup'
	);
	foreach my \$package (\@neededPackages) {
	  	unless (defined \$repo->Eval
		("SAPDB::Install::\$package", 1.01)) {
               	print join ("\n", \$repo->GetErr)."\n";
              	die;
        	}
	}
}





\$path{'name'} = 'Instance  path';
\$path{'mode'}=0775;
\$path{'opt'}='instance_path=s';
\$path{'default'}='sapdb/instance';

\@paths = (\\\%path);
\$main_path=\\\%path;
\%config = ('template', 'mi','config', 'mi.config');


sub framestart{
SAPDB::Install::Templates::Setup::framestart(\\\@paths,\\\%config);
return 1;
}

sub preinstall{
SAPDB::Install::Templates::Setup::preinstall(\\\@paths,\\\%config);
return 1;
}
sub postprepare{
SAPDB::Install::Templates::Setup::postprepare(\\\@paths,\\\%config);
return 1;
}

sub postinstall{
SAPDB::Install::Templates::Setup::postinstall(\\\@paths,\\\%config);
return 1;
}

#registrierung z.B. in Windows-Registry
sub register{
SAPDB::Install::Templates::Setup::register(\\\@paths,\\\%config);
return 1;
}

sub framestop{
SAPDB::Install::Templates::Setup::framestop(\\\@paths,\\\%config);
return 1;
}

sub unregister{
SAPDB::Install::Templates::Setup::unregister(\\\@paths,\\\%config);
return 1;
}
sub preuninstall{
SAPDB::Install::Templates::Setup::preuninstall(\\\@paths,\\\%config);
return 1;
}
sub postuninstall{
SAPDB::Install::Templates::Setup::postuninstall(\\\@paths,\\\%config);
return 1;
}
sub verify{
SAPDB::Install::Templates::Setup::verify(\\\@paths,\\\%config);
return 1;
}





1;
SCRIPTEND



  #  print "Creating package for instance backup \"" . $parts[1] .  "\"\n";
  #  print "Directory:  \"" . $parts[0] .  "\"\n";
}                               #createPackage(..)

#+sub():void:Archive the instance package using gnu tar
sub tgzPackage($){
  my ($this) = @_;
  my $backup = $this->getBackup();
  unless (-f $backup){
    $this->showOutput("ERROR: Backup \"$backup\" not found!\n");
    return undef;
  }
  my $backupFile = SAPDB::Install::Templates::Base::basename($backup);
  my $backupDir =  SAPDB::Install::Templates::Base::dirname($backup);
  my $archive = $this->getArchive();
  unless(defined $archive){
    $this->showOutput("ERROR: Archive name not specified. " . 
                      "Cannot create archive!\nPlease specify archive!\n");
    return undef;
  }

  my $gnuTarGeneric = $this->getGeneric($gnutar);
  my $gnuTarExec = "tar";
  if (defined $gnuTarGeneric) {
    $gnuTarExec = $gnuTarGeneric->[0];
  }                             #if

  $archive   =~ s/\\/\//g;
  $backupDir =~ s/\\/\//g;
  $archive   =~ s/^([a-zA-Z]):/\/cygdrive\/$1/;
  $backupDir =~ s/^([a-zA-Z]):/\/cygdrive\/$1/;
  my $tgzCommand = $gnuTarExec . " vcfz " . $archive .
    #      " -C " . $backupDir . " $backupFile PACKAGEDATA INSTANCEDATA  instance.lst instinstance";
    " -C " . $backupDir . " $backupFile PACKAGEDATA instinstance";
  $this->showOutput("Executing " . $tgzCommand . " ...");
  my $retVal = system( $tgzCommand);
  $this->showOutput("\n... finished with return code: $retVal");
  return $retVal;
}                               #tgzPackage()





sub writeInstanceConfig($$){
  my ($this,$write) = @_;
  &$write("DATAPATH " . $this->getDataPath() . "\n");
  &$write("LOGPATH " . $this->getLogPath() . "\n");
  &$write("DATASIZE " . $this->getDataSize() . "\n");
  &$write("DATAVOLS " . $this->getDataVolCount() . "\n");
  &$write("LOGSIZE " . $this->getLogSize() . "\n");
  &$write("LOGVOLS " . $this->getLogVolCount() . "\n");
}
#end writeInstanceConfig();

























#+field(Boolean:autoRestart)
my $autoRestart="autoRestartKey";
#+sub(Boolean):void:Set if autoRestart database after reboot
sub setAutoRestart{
  my ($this, $aValue) = @_;
  $this->{$autoRestart}=$aValue;
}
#+sub():Boolean:Get if autoRestart database after reboot
sub getAutoRestart{
  my ($this ) = @_;
  return $this->{$autoRestart};
}




# Create a new instance of this object.
#+sub():BaseRef:Default-Konstruktor
sub new(){
  my $this = createThis();
  bless $this ;
  return  $this;
}
#new(..)

sub createThis(){
  return {
          $sdb_name , "MAXDB1",
          $sdb_dbmuser , "dbm",
          $sdb_dbmpass , "dbm",
          $sdb_dbauser , "dba",
          #              $sdb_sqluser , "scott",
          #              $sdb_sqlpass , "tiger",
          $sdb_dbapass , "dba",
          $sdb_domainpass , "dba",
          $sdb_params ,{        #REFERENZ AUF HASH
                        #                            "MAXUSERTASKS", 5 
                       },
          $sdb_volumes , [      #REFERENZ AUF LISTE
                          #                              Nummer Art   Ort      Typ  Groesse
                          [qw (1      LOG   LOG_001   F    2000)   ],
                          [qw (1      DATA  DAT_001   F    10000)  ]
                         ],
          $outputter , \&stdOutPutter,
          $statusputter , \&stdOutPutter,
          $creationStrategy, 1,
          $deletionStrategy, 2,
          $generic , {}
         };
}
#createThis




#+sub(String:fileName):void:Parse String and configure
sub configureFromString($$){
  my ($this,$aString) = @_;
  my @lines = split /\n/ ,$aString;
  $this->configureFromLines(\@lines);
}

#+sub(String:fileName):void:Parse input file and configure
sub configureFromFile($$){
  my ($this,$file) = @_;
  open CONFIG, $file;
  my @lines = <CONFIG>;
  $this->configureFromLines(\@lines);
  close CONFIG;
}


#+sub(String:fileName):void:Configure from an Array of lines
sub configureFromLines($$){
  my ($this,$linesRef) = @_;
  my @lines = @$linesRef;
  my ($section);
  foreach my $line (@lines) {
    my ($key,$value,$type);
    parseConfigLine($line,
                    sub{ $section =$_[0];},
                    sub {$key=$_[0];
                         $value=$_[1];
                         if (defined $_[2]) {
                           $type=$_[2];
                         }
                       }
                   );
    if ( defined $key) {
      if ( $section eq 'DBInstance') {
        $this->configureInstanceBaseFromKeyValuePairs($key,$value);
      } elsif ( $section eq 'Software') {
        $this->configureSoftwareFromKeyValuePairs($key,$value);
      } elsif ( $section eq 'DBParam') {
        $this->configureDBParamsFromKeyValuePairs($key,$value);
      } elsif ( $section eq 'Generic') {
        $this->configureGenericFromKeyValuePairs($key,$value,$type);
      }
    }
    #if
  }
  #foreach
}
#configureFromLines




















# Vorkonfigurieren der Datenbankinstanz mit Werten aus der Parameterdatei,
# vorgeparst als Schluessel/Wert-Paare
sub configureInstanceBaseFromKeyValuePairs($$$){
  my ($this,$key,$value) = @_;
  if ( $key =~ m/dbm/i) {
    my ($dbmname,$dbmpass) = matchUserPassword($value);
    $this->setDBM($dbmname);
    if ( defined $dbmpass) {
      $this->setDBMPass($dbmpass);
    }
  } elsif ( $key =~ m/dba/i) {
    my ($dbaname,$dbapass) = matchUserPassword($value);
    $this->setDBA($dbaname);
    if ( defined $dbapass) {
      $this->setDBAPass($dbapass);
    }
  } elsif ( $key =~ m/domainpass/i) {
    my ($domainpass) = matchUserPassword($value);
    $this->setDomainPass($domainpass);
  } elsif ( $key =~ m/sqluser/i) {
    if ( $value =~ /^\s*$/) {
      $this->setSQLU(undef);
      $this->setSQLPass(undef);
    } else {
      my ($dbuser,$dbpass) = matchUserPassword($value);
      $this->setSQLU($dbuser);
      if ( defined $dbpass) {
        $this->setSQLPass($dbpass);
      }
    }
  } elsif ( $key =~ m/DBNAME/i) {
    $value =~ m/^\s*([\w\d]+)\s*$/;
    my $dbname = $1;
    $this->setDBName($dbname);
  } elsif ( $key =~ m/archive/i) {
    if ( $value =~ /^\s*$/) {
      $this->setArchive(undef);
    } else {
      $this->setArchive(matchPath($value));
    }
  } elsif ( $key =~ m/backup/i) {
    if ( $value =~ /^\s*$/) {
      $this->setBackup(undef);
    } else {
      $this->setBackup(matchPath($value));
    }
  } elsif ( $key =~ m/autorestart/i) {
    $this->setAutoRestart(matchBoolean($value));
  } elsif ( $key =~ m/dependpath/i) {
    $this->setDependendPath(matchPath($value));
  } elsif ( $key =~ m/CREATE_STRATEGY/i) {
    $this->setCreationStrategy(matchNumber($value));
  } elsif ( $key =~ m/DELETE_STRATEGY/i) {
    $this->setDeletionStrategy(matchNumber($value));
  }
   elsif ( $key =~ m/UPDATE_STATISTICS/i) {
    $this->setDoUpdateStatistics(matchBoolean($value));
  }
  $this->configureInstanceFromKeyValuePairs($key,$value);
}
#configureInstanceFromKeyValuePairs




















sub configureGenericFromKeyValuePairs($$$$){
  my ($this,$key,$value,$type)=@_;
  if ( $type eq 'Path') {
    my $aPath = matchPath($value);
    $this->addGeneric($key,$aPath,$type);
  } elsif ( $type eq 'UserPassword') {
    my $aUserPassword = matchUserPassword($value);
    $this->addGeneric($key,$aUserPassword,$type);
  } elsif ( $type eq 'Number') {
    my $aNumber = matchNumber($value);
    $this->addGeneric($key,$aNumber,$type);
  } elsif ( $type eq 'Storage') {
    my $aStorage = matchStorage($value);
    $this->addGeneric($key,$aStorage,$type);
  } elsif ( $type eq 'String') {
    my $aStorage = matchString($value);
    $this->addGeneric($key,$aStorage,$type);
  } elsif ( $type eq 'Boolean') {
    my $aStorage = matchBoolean($value);
    $this->addGeneric($key,$aStorage,$type);
  } else {
    $this->showOutput("WARNING: Invalid type for generic value: \"$type\" (Ignored!)");
  }
}







# Vorkonfigurieren der Datenbankparameter mit Werten aus der Parameterdatei,
# vorgeparst als Schluessel/Wert-Paare
sub configureDBParamsFromKeyValuePairs($$$){
  my ($this,$key,$value) = @_;
  $value =~ m/^\s*(\S.*\S?)\s*$/;
  my $parValue = $1;
  $this->addDBParam($key,$parValue);
  #  print $key,"=","\"",$parValue,"\"\n";
}
#configureDBParamsFromKeyValuePairs






my ($doReuseInstance,$doNotReuseInstance) =(1,0);

sub createInstance{
  my ($this) = @_;
  my $aDBName = $this->getDBName();
  my $creationStrategy = $this->getCreationStrategy();
  
  if ( $creationStrategy == 1) {
    return $this->createInstanceInternal($doNotReuseInstance);
  } elsif ($creationStrategy == 2) {
    my  $goodness = $this->getGoodness();
    if ( $goodness == 0) {
      $this->showStatusMessage("Cannot create $aDBName: " . 
                               $this->getGoodnessString() . "\n(Creation strategy was $creationStrategy.)" 
                              );
      return undef;
    } elsif ( $goodness > 1 ) {
      $this->showStatusMessage("Cannot create $aDBName, because it already exists! (Goodness: " . 
                               $this->getGoodnessString() . ")\n(Creation strategy was $creationStrategy, if " .
                               "you want to delete the database choose another creation strategy!)\n"
                              );
      return undef;
    } else {
      return $this->createInstanceInternal($doNotReuseInstance);
    }
  } elsif ($creationStrategy == 3) {
    my  $goodness = $this->getGoodness();
    if ( $goodness == 0, or $goodness == 2 or $goodness == 3) {
      $this->showStatusMessage("Cannot create $aDBName, because  I cannot delete already existing ". 
                               " instance.\nGoodness: " . 
                               $this->getGoodnessString() . "\n" 
                              );
    } elsif ($goodness == 1) {
      return $this->createInstanceInternal($doNotReuseInstance);
    } elsif ($goodness >= 3) {
      my $deleteSuccess = $this->deleteInstance();
      unless(defined $deleteSuccess){
        $this->showStatusMessage("Cannot create $aDBName instance, because  I cannot delete ". 
                                 "already existing instance.\n"  
                                );
        return undef;
      }
      return $this->createInstanceInternal($doNotReuseInstance);
    }



  } elsif ($creationStrategy == 4) {
    my  $goodness = $this->getGoodness();
    if ( $goodness == 1) {
      return $this->createInstanceInternal($doNotReuseInstance);
    } elsif ($goodness == 4 or $goodness == 5) {
      my $deleteSuccess = $this->deleteInstance();
      unless(defined $deleteSuccess){
        $this->showStatusMessage("Cannot create $aDBName, because  I cannot delete ". 
                                 "already existing instance.\n"  
                                );
        return undef;
      }
      return $this->createInstanceInternal($doNotReuseInstance);
    } elsif ($goodness >= 6) {
      return $this->createInstanceInternal($doReuseInstance);
    } elsif ($goodness == 0 or $goodness == 2 or $goodness == 3) {
      $this->showStatusMessage("Cannot create $aDBName:\n". 
                               "Status of  $aDBName is invalid: " .  $this->getGoodnessString()
                              );
      return undef;
    
    }

  } else {
    $this->showStatusMessage("Cannot create database: " . 
                             "Invalid value \"$creationStrategy\" for creation strategy.\n"
                            );
    return undef;
  }
}
#createInstance




#ifdef sub:createInstance
#<p>
#At the moment, depending on the create strategy, the following steps a performed, when a database
#instance is created.
#(For most of the  steps  the <code>SAPDB::Install::DBMCmd</code>-Modul is used).
#</p>
#<ol>
#<li>
#Create instance using <code>dbmcli db_create</code>
#</li>
#<li>
#Connect to the instance as database manager
#</li>
#<li>
#Set all configured database parameters.
#</li>
#<li>
#Add all configured database volumes.
#</li>
#<li>
#Change the instance to admin state.
#</li>
#<li>
#Do <code>db_activate</code>
#</li>
#<li>
#Do load_systabs
#</li>
#<li>
#Do db_online
#</li>
#<li>
#Open an SQL-Connection as database manager.
#</li>
#<li>
#Create one user.
#</li>
#<li>
#Close the SQL-Connection.
#</li>
#</ol>
#endif
#+sub():void:Create the database instance
sub createInstanceInternal{
  # Initialize configuration
  my ($this,$isReuse) = @_;
  my $dbname = $this->getDBName();
  my $mydependend = $this->getDependendPath();
  my $scriptHandler = $this->getScriptHandler();
  my $autoRestart = $this->getAutoRestart();

  if ( not (defined $mydependend)) {
    $this->showStatusMessage("Cannot create database because of missing parameter!\n\"Dependend " . 
                             "path\" is not set.\n");
    return undef;
  }
  unless(defined $scriptHandler){
    if ( not -d $mydependend) {

      $this->showStatusMessage("Cannot create database!\n" . 
                               "Dependend path $mydependend does not exist or is not a directory !\n" 
                              );
      return undef;
    }
  }


  if ($isReuse) {
    $this->showStatusMessage("Trying to refresh existing database $dbname\n");
  } else {
    my $aFlag = (defined $autoRestart and ($^O =~ /mswin/i)) ? "-a" : "";
    my ($indep_data,$indep_prog)=readIndepPath(); # try to get independent pathes from earlier installation
    my $dbmcliexe =$this->getDbmcliPath();
    $^O =~ /mswin/i and $mydependend =~ s/\//\\/g;
    my $create_command =
      "$dbmcliexe -R \"" . $mydependend . 
        "\" db_create " . $aFlag . " " .  $this->getDBName() . " " . $this->getDBM() . ",". $this->getDBMPass();
    unless( defined $this->dbmcliCall( $create_command, "Start  creating db instance $dbname.\n", "Instance creation failed!\n")){
      return undef;
    }
  }


  my $dbm = $this->createDBMCmd();
  my $dbmuser = $this->getDBM();
  my $dbmpass = $this->getDBMPass();

  unless ( defined $this->logonAsDBM($dbm)) {
    $this->showStatusMessage("ERROR: Cannot create $dbname, logon as $dbmuser failed!\n");
    return undef;
  }

  unless ($isReuse){
    $this->showStatusMessage("Setting database parameters.\n");
    if ( $this->dbmcmdSetDBParams($dbm) != 0) {
      return undef;
    }
    $this->showStatusMessage("Adding volumes.\n");
    unless ( defined  $this->dbmcmdAddAllVolumes($dbm)) {
      return 1;
    }
  }



  $this->showStatusMessage("Change to admin state.\n");
  unless (defined ($this->changeState($dbm,"db_admin"))){
    return undef;
  }

  unless (defined ($this->db_activate($dbm))){
    return undef;
  }
  #  $this->showStatusMessage("$dbname is now activated and online.\n");
  $this->showStatusMessage("Loading system tables.\n");
  unless (defined ($this->load_systab($dbm))){
    return undef;
  }

  my $isLogOverWrite = $this->getLogAutoOverWrite();
  if ( defined $isLogOverWrite) {
    #    $this->showStatusMessage("Setting log overwrite on.\n");
    $this->switchOnLogAutoOverwrite($dbm);
  }


  my $isAutoExtend = $this->getAutoExtend();
  if ( defined $isAutoExtend) {
    #    $this->showStatusMessage("Setting auto extend feature on.\n");
    $this->switchOnAutoExtension($dbm);
  }
  unless($isReuse){
    $this->createSQLUser($dbm);
  }
  my $updateStatistics = $this->getDoUpdateStatistics();
  if($updateStatistics){
    $this->performUpdateStatistics($dbm);
  }
  if ($isReuse) {
    $this->showStatusMessage("Finished refreshing  $dbname.\n");
  } else {
    $this->showStatusMessage("Finished creating   $dbname.\n");
  }
  return 0;
}
#createInstance




sub performUpdateStatistics{
  my ($this,$dbm) = @_;
  my $dbname = $this->getDBName();
  $this->showStatusMessage("Updating statistics.\n");
  unless (defined ($this->sql_connectAsDBA($dbm))){
    return undef;
  }
  $this->dbm($dbm,"sql_updatestat *"   ,
             "Failed to update statistics.\n",
             "Statistics are updated.\n");
  my $dbaname = $this->getDBA();
  unless (defined ($this->dbm
                   ($dbm,
                    "sql_release",
                    "Failed to close sql connection to $dbname.\n",
                    "SQL-Connection  to $dbname as $dbaname is now closed.\n"
                   ))){
    return undef;
  }
  return 1;
}










sub createSQLUser{
  my ($this,$dbm) = @_;
  my $dbname = $this->getDBName();
  my $sqluser = $this->getSQLU();
  my $sqlpassword = $this->getSQLPass();
  unless (defined $sqluser){
    return undef;
  }
  $this->showStatusMessage("Creating default user.\n");
  unless (defined ($this->sql_connectAsDBA($dbm))){
    return undef;
  }



  my $user_statement = "CREATE USER " .  $sqluser . " PASSWORD " . $sqlpassword
    . " DBA NOT EXCLUSIVE";


  $this->dbm($dbm,"sql_execute " . $user_statement ,
             "Failed to create default user \"$sqluser\".\n",
             "User \"$sqluser\" with initial password \"$sqlpassword\" created.\n");

  
  my $dbaname = $this->getDBA();
  unless (defined ($this->dbm
                   ($dbm,
                    "sql_release",
                    "Failed to close sql connection to $dbname.\n",
                    "SQL-Connection  to $dbname as $dbaname is now closed.\n"
                   ))){
    return undef;
  }
  return 1;
}
#createSQLUser

 

sub changeState($$$){
  my ($this,$dbm,$state)=@_;
  my $retVal;
  unless ( defined $this->dbm
           ($dbm, "$state",  "Could not switch to admin level!\n")) {
    return undef;
  }
  my   @stateResult = $this->dbm($dbm, "db_state",  "db_state failed after $state!\n" );
  my $aResultState = $stateResult[2];
  if (defined $aResultState) {
    $this->showOutput("Database state is now $aResultState\n");
    return $aResultState;
  } else {
    return undef;
  }
}
#changeState


#ifdef sub:deleteInstance
#<p>
#Shut down the instance and do db_drop
#</p>
#endif
#+sub():void:Delete the database instance
sub deleteInstance(){
  my ($this) = @_;
  my $dbname = $this->getDBName();
  my $dbm = $this->createDBMCmd();

  unless ( defined $this->logonAsDBM($dbm)) {
    return undef;
  }
  $this->showStatusMessage("Start deleting instance $dbname\n");

  $this->changeState($dbm,"db_offline");
  unless (defined ($this->dbm
                   ($dbm,
                    "db_drop",
                    "Failed to drop instance $dbname.\n",
                    "Instance $dbname is now dropped.\n"))){ 
    return undef;
  } 
  $this->showStatusMessage("Instance $dbname deleted.\n");
  return 0;
}
#deleteInstance(..)



sub logonAsDBM{
  my ($this,$dbm) = @_;
  my $dbname = $this->getDBName();
  my $dbmuser = $this->getDBM();
  my $dbmpass = $this->getDBMPass();
  my $aUserLogon = "user_logon " . $dbmuser . "," . $dbmpass;
  unless ( defined $this->dbm($dbm, $aUserLogon,  "Logon as $dbmuser failed!\n")) {
    return undef;
  }
}















#+sub():void:Create a Backup
sub backup(){
  my ($this) = @_;
  my $dbm = $this->createDBMCmd();
  my $aBackup = $this->getBackup();
  my $dbname = $this->getDBName();


  my $mediumDeclaration =  "SDBINSTBACKUP " . $aBackup . " FILE DATA 0 8 YES";
  my @tags = split (/\s+/,$mediumDeclaration);
  my $mediumName = $tags[0];



  my $dbmuser = $this->getDBM();
  my $dbmpass = $this->getDBMPass();

  unless ( defined $this->logonAsDBM($dbm)) {
    return undef;
  }



  my $mediumPutCommand = "medium_put " . $mediumDeclaration; 
  unless(defined $this->dbm($dbm,$mediumPutCommand)){
    return undef;
  }
  #do util_connect
  unless( defined $this->dbm ($dbm,
                              "util_connect " .  $this->getDBM() . "," . $this->getDBMPass() ,
                              "ERROR:  \"util_connect  \" failed!\n"
                             )){
    $this->dbm($dbm,"util_release","\"util_release\" failed!\n");
    return undef;
  }
  unless( defined $this->dbm ($dbm,
                              "backup_save " . $mediumName,
                              "ERROR:  \"backup failed\" failed!\n",
                              "Backup successfull!\n",
                              "",
                              "Starting backup ...\n"
                             )){
    $this->dbm($dbm,"util_release","\"util_release\" failed!\n");
    return undef;
  }
  #end util session 
  $this->dbm($dbm,"util_release","\"util_release\" failed!\n");
  #delete medium
  $this->dbm($dbm,"medium_delete " . $mediumName,
             "Could not delete medium $mediumName\n");
  return 0;
}
#backup()

#Instance-Method: Needs DBMCmd instance as arg
sub db_drop ($$){
  my ($this, $dbm) = @_;
  return $this->dbm($dbm,"db_drop") ;
}
#db_drop()


sub load_systab($$){
  my ($this, $dbm) = @_;
  my $adba = $this->getDBA();
  my $adbaPass = $this->getDBAPass();
  my $aDomainPass = $this->getDomainPass();
  return $this->dbm
    ($dbm,
     "load_systab -u $adba,$adbaPass -ud $aDomainPass",
     "Failed loading system tables.\n",
     "System tables are loaded.\n") ;
}
#load_systab()


sub medium_put($$){
  my ($this, $dbm) = @_;
  my $mediumDeclaration = $this->getMediumDeclaration();
  my $mediumPutCommand = "medium_put " . $mediumDeclaration; 
  return $this->dbm($dbm,$mediumPutCommand)
}
#medium_put()


sub generateScript{
  my ($this ) = @_;
  #  $this->setOutPutter(sub{});
  #  $this->setOutPutter(sub{});
  #  $this->setOutPutter(sub{});
}

sub dbm{
  my ($this ) = @_;
  my $scriptHandler = $this->getScriptHandler();
  if ( defined $scriptHandler) {
    return &$scriptHandler(@_);
  } else {
    return defaultDbmcliHandler(@_);
  }
}
#dbm()


sub defaultDbmcliHandler{
  my ($this, $dbm,$command, $errorMessage,$successMessage,$showResult,$beforeMessage) = @_;
  if ( defined $beforeMessage) {
    $this->showOutput($beforeMessage);
  }
  $this->showOutput("[$command]\n");
  my $output = $dbm->exec($command);
  my @outputLines = split ('\n', $output);
  my $result = $outputLines[0];
  my $err = $outputLines[1];
  my @retVal =() ;
  my $success=undef;
  if ($result =~ /^OK/) {
    $success=1;
    if ( defined $successMessage) {
      $this-> showOutput($successMessage);
    }
    if ( defined $showResult) {
      $this-> showOutput($output);
    }
    @retVal = @outputLines;
  } else {
    if ( defined $errorMessage) {
      $this-> dbmcmdError($dbm,$errorMessage);
    } else {
      $this->dbmcmdError($dbm,"");
    }
    if ( defined $showResult) {
      $this-> showOutput($output);
    }
  }
  if ( defined $success) {
    return @retVal
  }
  return undef;
}
#defaultDbmcliHandler(..)









#Instance-Method: Needs DBMCmd instance as arg
sub db_activate($$){
  my ($this, $dbm) = @_;
  my $pipeName = $this->getRecoverFromPipe();
  my $aBackup = $this->getBackup();
  my $aArchive = $this->getArchive();
  my $aRetVal;
  if ( defined $pipeName) {
    if ( defined $aBackup and defined $aArchive) {
      $this->showOutput("Reading backup \"$aBackup\" from archive \"$aArchive\".\n");
      $aRetVal = $this->db_activateWithBackupArchive($dbm);
    }
  } elsif ( defined $aBackup) {
    $this->showStatusMessage("Reading backup from file.\n");
    $aRetVal = $this->db_activateFromFile($dbm);
    if (defined  $aRetVal) {
      $aRetVal = $this->changeState($dbm,"db_online");
    }
  } else {
    $this->showStatusMessage("No backup specified: No backup will be loaded!\n");
    $aRetVal =  $this->db_activateNormal($dbm);
  }
  return $aRetVal;
}
#db_activate

#Normal activation!!!
sub db_activateNormal ($$){
  my ($this, $dbm) = @_;

  my $activateCommand  = "db_activate " . $this->getDBA() . "," . $this->getDBAPass();

  $this->showStatusMessage("Now start db activation (This can take a while, be patient!)\n");

  return $this->dbm($dbm,$activateCommand, "ERROR: DB-Activation failed: \"$activateCommand\"\n",
                    undef,undef, "Start db activation (This can take a while, be patient!)\n");
}
#db_activate

#db_activate recover with reading of bacup
sub db_activateWithBackupArchive{
  #Build Strings:
  my ($this, $dbm) = @_;
  my $pipeIdentifier = $this->getRecoverFromPipe();
  my $mediumDeclaration = $this->getMediumDeclaration();
  my @tags = split (/\s+/,$mediumDeclaration);
  my $pipeFile = $tags[1];
  #  my $activateCommand = "db_activate RECOVER " . $pipeIdentifier . " DATA";
  my $recoverCommand = "recover_start " . $pipeIdentifier ;
  my $backup = $this->getBackup();
  my $archive = $this->getArchive();
  my $scriptHandler = $this->getScriptHandler();

  if ( defined $scriptHandler) {
    $this->medium_put($dbm);
    $this->dbm ($dbm, "util_connect");
    &$scriptHandler(undef,undef,$recoverCommand);
    $this->dbm($dbm,"util_release");
    $this->dbm($dbm,"medium_delete " . $pipeIdentifier );
    return 1;
  }

  unless(defined $this->db_activateNormal($dbm)){
    return undef;
  }
  unless(defined $this->changeState($dbm,"db_offline")){
    return undef;
  }
  unless(defined $this->changeState($dbm,"db_admin")){
    return undef;
  }


  #Check some parameters
  unless (defined $backup){
    $this-> showStatusMessage("ERROR: Backup is not specified!\n");
    return undef;
  }
  unless (defined $archive){
    $this-> showStatusMessage("ERROR: Archive is not specified!\n");
    return undef;
  }
  unless (-f $archive){
    $this-> showStatusMessage("ERROR: Archive '$archive ' does not exist!\n");
    return undef;
  }
  #Open archive
  my $untgz = SAPDB::Install::Untgz::new ();
  my $archiveSuccess = $untgz->Open($archive);
  unless (defined $archiveSuccess 
          && $archiveSuccess == 0){
    $this-> showOutput("Cannot open archive!\n");
    return undef;
  }
  #postion tgz cursor on backup
  my $aBackup;
  my @hrd = $untgz->Next();
  while (1) {
    unless(defined $hrd[0]){
      last;
    }
    my $maybeBackup = $hrd[3]; 
    if ( (uc $backup) eq (uc $maybeBackup)) {
      $aBackup = $maybeBackup;
      last;
    }
    @hrd = $untgz->Next();
  }
  #Check if backup was found in archive
  unless (defined $aBackup){
    $this->showStatusMessage ("Backup $backup  not found!\n(Archive was  \"$archive\".)\n");
    return undef;
  }
  #Open pipe
  my $bup = SAPDB::Install::BackupPipe->new ($pipeFile, 'w');
  unless (defined $bup){
    $this->showStatusMessage ("Could not create pipe $pipeFile\n");
    return undef;
  }
  #do medium_put 
  unless(defined $this->medium_put($dbm)){
    return undef;
  }
  #do util_connect
  unless( defined $this->dbm ($dbm,
                              "util_connect",
                              "ERROR:  \"util_connect\" failed!\n"
                             )){
    return undef;
  }

  unless( defined $this->dbm ($dbm,
                              "util_execute CLEAR LOG",
                              "ERROR:  \"util_execute CLEAR LOG\" failed!\n"
                             )){
    return undef;
  }


  # Start db_activate RECOVER ...
  my $activateId = $dbm->send ($recoverCommand);
    
  $this->showOutput ("[$recoverCommand]\n");


  # Open pipe
  unless(defined $bup->Open()){
    $this->showStatusMessage ("ERROR: Could not open pipe $bup\n");
    return undef;
  }

  # Write from TGZ-File into pipe
  $this->showStatusMessage ("Reading backup from archive (This can take a while, be patient!).\n");
  for (;;) {
    my ($got, $want, $buff);

    $want = 32 * 1024;
    $got = $untgz->ExtractRead ($buff, $want);
    unless (defined $got) {
      $this->showStatusMessage ("ERROR: Could not read from $archive\n");
      last;
    }

    if ($got <= 0) {
      last;
    }

    $want = $got;
    $got = $bup->Write ($buff, $want);
    if ($got != $want) {
      $this->showStatusMessage ("ERROR: Write failed to $pipeFile.(Wanted : $want, got $got)!\n");
      last;
    } else {
      #      $this->showOutput ("Wrote $want bytes\n");
    }

    if ($activateId == $dbm->select (0)) {
      $this->showStatusMessage ("ERROR, died before end of write\n: \"$recoverCommand \"\n");
      last;
    }
  }
  #close pipe
  $bup->Close();

  #Get Result of db_activate
  my $activateResult  = $dbm->recv ($activateId);
  my @activateLines =  split ('\n',$activateResult );
  my $activateSuccess = $activateLines[0];
  my $aReturnValue;
  if ($activateSuccess =~ /^OK/) {
    $aReturnValue = $activateSuccess;
    $this->showOutput("Result of \"$recoverCommand\":\n $activateResult\n");
  } else {
    $this->showOutput("Result of \"$recoverCommand\":\n $activateResult\n");
  }


  #end util session 
  $this->dbm($dbm,"util_release","\"util_release\" failed!\n");
  #delete medium
  $this->dbm($dbm,"medium_delete " . $pipeIdentifier,
             "Could not delete medium $pipeIdentifier\n");



  unless(defined $this->changeState($dbm,"db_offline")){
    return undef;
  }
  unless(defined $this->changeState($dbm,"db_online")){
    return undef;
  }
  return $aReturnValue;
}
#db_activateWithBackupArchive

#db_activate recover from file
sub db_activateFromFile ($$){
  #Build Strings:
  my ($this, $dbm) = @_;
  my $mediumDeclaration = $this->getMediumDeclaration();
  my @tags = split (/\s+/,$mediumDeclaration);
  my $mediumName = $tags[0];
  my $activateCommand = "db_activate RECOVER " . $mediumName . "";
  my $backup = $this->getBackup();
  my $scriptHandler = $this->getScriptHandler();

  #Check some parameters
  unless (defined $backup){
    unless(defined $scriptHandler){
      $this-> showOutput("ERROR: Backup is not specified!\n");
      return undef;
    }
  }
  unless (-f $backup){
    unless(defined $scriptHandler){
      $this-> showOutput("ERROR: Backup \"$backup\" not found!\n");
      return undef;
    }
  }

  #do medium_put 
  unless(defined $this->medium_put($dbm)){
    return undef;
  }


  #do util_connect
  unless( defined $this->dbm ($dbm,
                              "util_connect",
                              "ERROR:  \"util_connect\" failed!\n"
                             )){
    return undef;
  }



  #Start db_activate RECOVER
  my $aReturnValue = $this->dbm($dbm,
                                $activateCommand,
                                "Error: DB-Activation failed!");

  #end util session 
  $this->dbm($dbm,"util_release","\"util_release\" failed!\n");
  #delete medium
  $this->dbm($dbm,"medium_delete " . $mediumName,
             "Could not delete medium $mediumName\n");
  return $aReturnValue;
}
#db_activateFromFile()



#Instance-Method: Needs DBMCmd instance as arg
sub sql_connectAsDBA ($$){
  my ($this, $dbm) = @_;
  my $aName = $this->getDBA();
  my $command = "sql_connect " . $this->getDBA() . ",". $this->getDBAPass();

  return $this->dbm
    ($dbm,
     $command,
     "Failed to open sql connection as $aName.\n",
     "SQL-Connection as $aName established.\n");
}


#Instance-Method: Needs DBMCmd instance as arg
#Sets log auto overwrite to on
sub switchOnLogAutoOverwrite($$){
  my ($this, $dbm) = @_;
  my $aName = $this->getDBName();
  return $this->dbm
    ($dbm,
     "util_execute SET LOG AUTO OVERWRITE ON",
     "Failed to switch on automatic log overwrite for database $aName.\n",
     "Automatic log overwrite is now ON!\n");
}
#switchOnLogAutoOverwrite


#Instance-Method: Needs DBMCmd instance as arg
#Sets auto extension to on
sub switchOnAutoExtension($$){
  my ($this, $dbm) = @_;
  my $aName = $this->getDBName();
  return  $this->dbm($dbm,
                     "dbm_configset AutoExtFlag yes",
                     "Failed to switch on the auto extension feature for database $aName.\n",
                     "Auto extension  is now ON!\n");
}
#switchOnAutoExtension






# Fuehrt einen einzelnen DBMCLI-Befehl auf der Kommandozeile aus
sub dbmcliCall{
  my ($this,$command,$title,$errorMessage) = @_;
  my $scriptHandler = $this->getScriptHandler();
  if ( defined $scriptHandler) {
    return &$scriptHandler(undef,undef,$command);
  } else {
    return defaultDbmcliCall(@_);
  }
}
#dbmcliCall(..)

# Fuehrt einen einzelnen DBMCLI-Befehl auf der Kommandozeile aus
sub defaultDbmcliCall{
  my ($this,$command,$title,$errorMessage) = @_;
  my $success=1;
  $this->showStatusMessage( $title  );
  $this->showOutput( "[$command]\n" );
  open PROCOUT, $command . "|";
  while (<PROCOUT>) {
    if (m/^Err/i) {
      $success=undef;
    }
    $this->showOutput($_);
  }
  close PROCOUT;
  if (not defined $success) {
    if (defined $errorMessage) {
      $this->showStatusMessage($errorMessage);
    }
    $this->showStatusMessage("(Error  executing $command)\n");
  }
  return  $success;
  #  $this->showStatusMessage("Ready ..." );
}
#defaultDbmcliCall










sub parseConfigLine($$$){
  my ($line,$section,$keyVal) =@_;
  my $key; my $value;
  #Comment
  if ($line =~ /^\s*#/) {
  } 
  #Section title
  elsif ($line =~ /^\s*\[(.+)\]\s*$/) {
    &$section($1);
  } 
  #Key value pair seperated by =
  elsif ($line =~ /^\s*(\w+)\s*=\s*(.+)$/) {
    &$keyVal($1,$2);
  }
  #Key value pair seperated by spaces
  elsif ($line =~ /^\s*(\w+)\s+(\S.*)$/) {
    &$keyVal($1,$2);
  }
  #Key without value
  elsif ($line =~ /^\s*(\w+)\s*$/) {
    &$keyVal($1,"");
  }
  #Generic type
  elsif ($line =~ /^\s*!(\w+):\s+(\w+)\s+(\S.*)$/) {
    my $type = $1;
    my $key = $2;
    my $value = $3;
    &$keyVal($key,$value,$type);
  }
}






#match for "<USER>,<PASSWORD>"
sub matchUserPassword{
  $_[0] =~ m/^\s*([\w\d]+)(\s*,\s*([\w\d]+))?\s*$/;
  return ($1,$3);
}
#matchUserPassword

sub matchPath{
  $_[0] =~ m/^\s*(.+)\s*$/;
  return $1;
}
#matchPath

sub matchNumber{
  $_[0] =~ m/^\s*([\d]+)\s*$/;
  return $1;
}
#matchNumber

sub matchStorage{
  $_[0] =~ m/^\s*([\d.,]+)\s*$/;
  return $1;
}
#matchNumber

sub matchString{
  $_[0] =~ m/^\s*([^\s].*[^\s])\s*$/;
  return $1;
}
sub matchBoolean{
  my $retVal = undef;
  my $val = $_[0];
  unless ( $val =~ /^\s*$/ or  $val =~ /^\s*false\s*$/i or $val =~ /^\s*no\s*$/i) {
    $retVal = $val;
  }
  return $retVal;
}



#matchNumber





# Set DB parameters via DBMCmd
#
sub dbmcmdSetDBParams($$){
  my ($this,$dbm) = @_;


  unless ( defined $this->dbm($dbm, "param_startsession", 
                              "param_startsession failed!\n",
                              "Parameter session started\n" )) {
    return 1;
  }

  unless ( defined $this->dbm($dbm, "param_init", 
                              "param_init failed!\n"
                             )) {
    return 1;
  }




  my %dbparams = $this->getDBParams();

  while ((my $key,my $value) = each %dbparams) {
    $this->dbm($dbm, 
               "param_put " . $key . " " . $value, 
               "Failed setting DB-Parameter \"" . $key . "\"\n",
               "Set " .$key . " to " . $value . "\n");
  }


  $this->dbm($dbm,"param_checkall", "DB-Parameter check failed.\n", "Checked DB Parameters successfully.\n");
  $this->dbm($dbm,"param_commitsession", "Failed to commit parameter session.\n","Parameter session committed.\n");
  return 0;
}
#dbmcmdSetDBParams()





sub createDBMCmd($){
  my ($this) = @_;
  my $listref = $this->maybeStartX();
  my $id = $listref->[0];
  if (defined $id and $id == 18641  ) {
    #    $this->showOutput("Tried to start XServer, but it is already running.\n"); 
  } elsif (defined $id and $id == 19720 ) {
    #    $this->outputTimeStamp();
    #    $this->showOutput("XServer started.\n"); 
  } else {
    $this->showOutput("WARNING: When trying to initialise SAPDB::Install::DBMCmd,\n" .
                      "I could not figure out, if your XServer is running.\n" .
                      "If errors occur, please check the XServer!\n");
  }
  my $dbm = SAPDB::Install::DBMCmd->new ('', $this->getDBName());
  return $dbm;
}
#createDBMCmd()


=over 1 
=item maybeStartX

Start the X-Server if necessary

 my $listref = maybeStartX();
 print "Message-ID: ", $listref->[0],"\n";
 print "Message-Type: ", $listref->[1],"\n";
 print "Message: ", $listref->[2],"\n";

 --------------------------

 Message-ID: 18641
 Message-Type: ERROR
 Message: XSERVER is already running!

 --------------------------

 Message-ID: 19720
 Message-Type: INFO
 Message: XSERVER started

=back

=cut

sub maybeStartX(){
  my $this = shift;
  my $xserverPath = $this->getXServerPath();
  open PROCOUT,  "$xserverPath start|";
  my $message='';
  while (<PROCOUT>) {
    $message = $message . $_;
  }
  close PROCOUT;
  my ($id, $type, $text);
  $message =~ m/(\d+)\s+(\w+):\s+(.+)/; 
  $id=$1;
  $type=$2;
  $text=$3;
  [$id, $type, $text];
}



# Gibt einen Fehler aus, der durch
# von dbmserver kommt.
sub dbmcmdError($$$){
  my ($dbc,$dbm,$message) = @_;
  my $lastcmd = $dbm->lastcmd();
  my $lastmsg = $dbm->lastmsg();
  my $lasterr = $dbm->lasterr();
  $dbc->showStatusMessage($message);
  $dbc->showOutput($message);
  $dbc->showOutput("DBMCLI-Command: \"" . ((defined $lastcmd) ? $lastcmd : "" ) .  "\"\n");
  $dbc->showOutput("DBMCLI-Message: \"" . ((defined $lastmsg ? $lastmsg : "") . "\"\n"));
  $dbc->showOutput("DBMCLI-Error: \"" .   ((defined $lasterr ? $lasterr : "") . "\"\n"));
  return;
}
#dbmcmdError


#Gibt die Konfigurationsdaten nach stdout
sub testDump(){
  my $this = $_[0];
  $this->showOutput( "DBNAME: \"" . $this->getDBName() .  "\"\n");
  $this->showOutput( "DEPENDEND-PATH: \"" . $this->getDependendPath() .  "\"\n");
  $this->showOutput( "DBM: \"" . $this->getDBM() .  "\"\n");
  $this->showOutput( "DBM PASSWORT: \"" .       $this->getDBMPass() . "\"\n");
  $this->showOutput( "DBA: \"" .     $this->getDBA() .  "\"\n");
  $this->showOutput( "DBA-PASSWORT:\"" .       $this->getDBAPass() . "\"\n");
  $this->showOutput( "DOMAIN-PASSWORT: \"" .      $this->getDomainPass() . "\"\n");
  $this->showOutput( "SQL-User: \"" .      $this->getSQLU() . "\"\n");
  $this->showOutput( "SQL-Passwort: \"" .      $this->getSQLPass() . "\"\n");
  $this->showOutput( "Domain-Passwort: \"" . $this->getDomainPass() . "\"\n");
  $this->showOutput(  "PARAMETERS:   \n");
  my %dbparams = $this->getDBParams();
  while ((my $key,my $value) = each %dbparams) {
    $this->showOutput("$key = $value\n");
  }
  $this->showOutput("VOLUMES:   \n");
  my @volumes = $this->getDBVolumes();
  foreach my $volume (@volumes) {
    $this->showOutput( "[Nummer=" . @$volume[0]
                       . ",Art=" . @$volume[1]
                       . ",Ort=" . @$volume[2]
                       . ",Typ=" . @$volume[3]
                       . ",Groesse=" . @$volume[4]
                       . "]\n");
  }
  my $pipeName = $this->getRecoverFromPipe();
  if ( defined $pipeName) {
    $this->showOutput("Recover from pipe: " . $pipeName . "\n");
  }
  my $mediumDeclaration = $this->getMediumDeclaration();
  if ( defined $mediumDeclaration) {
    $this->showOutput("Medium declaration: \"" . $mediumDeclaration . "\"\n");
  }

  my $backup = $this->getBackup();
  if ( defined $backup) {
    $this->showOutput("Backup: \"" . $backup . "\"\n");
  }
  my $archive = $this->getArchive();
  if ( defined $archive) {
    $this->showOutput("Archive: \"" . $archive . "\"\n");
  }
  $this->showOutput("GENERICS:\n");
  my $generics = $this->getAllGenerics();
  while ((my $key,my $value) = each %{$generics}) {
    my $pair = $this->getGeneric($key);
    $this->showOutput("$key=\"$pair->[0]\" [$pair->[1]]\n");
  }
}
#end testDump();


###################################################
#File::Basename.pm von Perl5.004
###################################################

my $Fileparse_fstype;
my $Fileparse_igncase;


#   fileparse_set_fstype() - specify OS-based rules used in future
#                            calls to routines in this package
#
#   Currently recognized values: VMS, MSDOS, MacOS, AmigaOS, os2, RISCOS
#       Any other name uses Unix-style rules and is case-sensitive

sub fileparse_set_fstype {
  my @old = ($Fileparse_fstype, $Fileparse_igncase);
  if (@_) {
    $Fileparse_fstype = $_[0];
    $Fileparse_igncase = ($_[0] =~ /^(?:MacOS|VMS|AmigaOS|os2|RISCOS|MSWin32|MSDOS)/i);
  }
  wantarray ? @old : $old[0];
}




#Dirty test if a path is absolute
sub isPathAbsolute{
  my $aPath = $_[0];
  return file_name_is_absolute($aPath);
  #$aPath =~ "^[a-zA-Z]:[\\\\/]" or  $aPath =~ "^/";
}



#fileparse_set_fstype

#   fileparse() - parse file specification
#


sub fileparse {
  my($fullname,@suffices) = @_;
  my($fstype,$igncase) = ($Fileparse_fstype, $Fileparse_igncase);
  my($dirpath,$tail,$suffix,$basename);
  my($taint) = substr($fullname,0,0); # Is $fullname tainted?

  if ($fstype =~ /^VMS/i) {
    if ($fullname =~ m#/#) {
      $fstype = '';
    }                           # We're doing Unix emulation
    else {
      ($dirpath,$basename) = ($fullname =~ /^(.*[:>\]])?(.*)/);
      $dirpath ||= '';          # should always be defined
    }
  }
  if ($fstype =~ /^MS(DOS|Win32)/i) {
    ($dirpath,$basename) = ($fullname =~ /^((?:.*[:\\\/])?)(.*)/);
    $dirpath .= '.\\' unless $dirpath =~ /[\\\/]$/;
  } elsif ($fstype =~ /^MacOS/i) {
    ($dirpath,$basename) = ($fullname =~ /^(.*:)?(.*)/);
  } elsif ($fstype =~ /^AmigaOS/i) {
    ($dirpath,$basename) = ($fullname =~ /(.*[:\/])?(.*)/);
    $dirpath = './' unless $dirpath;
  } elsif ($fstype !~ /^VMS/i) { # default to Unix
    ($dirpath,$basename) = ($fullname =~ m#^(.*/)?(.*)#);
    if ($^O eq 'VMS' and $fullname =~ m:/[^/]+/000000/?:) {
      # dev:[000000] is top of VMS tree, similar to Unix '/'
      ($basename,$dirpath) = ('',$fullname);
    }
    $dirpath = './' unless $dirpath;
  }

  if (@suffices) {
    $tail = '';
    foreach $suffix (@suffices) {
      my $pat = ($igncase ? '(?i)' : '') . "($suffix)\$";
      if ($basename =~ s/$pat//) {
        $taint .= substr($suffix,0,0);
        $tail = $1 . $tail;
      }
    }
  }

  $tail .= $taint if defined $tail; # avoid warning if $tail == undef
  wantarray ? ($basename . $taint, $dirpath . $taint, $tail)
    : $basename . $taint;
}
#fileparse


#   basename() - returns first element of list returned by fileparse()

sub basename {
  my($name) = shift;
  (fileparse($name, map("\Q$_\E",@_)))[0];
}
#basename


#    dirname() - returns device and directory portion of file specification
#        Behavior matches that of Unix dirname(1) exactly for Unix and MSDOS
#        filespecs except for names ending with a separator, e.g., "/xx/yy/".
#        This differs from the second element of the list returned
#        by fileparse() in that the trailing '/' (Unix) or '\' (MSDOS) (and
#        the last directory name if the filespec ends in a '/' or '\'), is lost.

sub dirname {
  my($basename,$dirname) = fileparse($_[0]);
  my($fstype) = $Fileparse_fstype;

  if ($fstype =~ /VMS/i) { 
    if ($_[0] =~ m#/#) {
      $fstype = '';
    } else {
      return $dirname || $ENV{DEFAULT};
    }
  }
  if ($fstype =~ /MacOS/i) {
    return $dirname;
  } elsif ($fstype =~ /MSDOS/i) { 
    $dirname =~ s/([^:])[\\\/]*$/$1/;
    unless( length($basename) ) {
      ($basename,$dirname) = fileparse $dirname;
      $dirname =~ s/([^:])[\\\/]*$/$1/;
    }
  } elsif ($fstype =~ /MSWin32/i) { 
    $dirname =~ s/([^:])[\\\/]*$/$1/;
    unless( length($basename) ) {
      ($basename,$dirname) = fileparse $dirname;
      $dirname =~ s/([^:])[\\\/]*$/$1/;
    }
  } elsif ($fstype =~ /AmigaOS/i) {
    if ( $dirname =~ /:$/) {
      return $dirname;
    }
    chop $dirname;
    $dirname =~ s#[^:/]+$## unless length($basename);
  } else { 
    $dirname =~ s:(.)/*$:$1:;
    unless( length($basename) ) {
      $Fileparse_fstype = $fstype;
      ($basename,$dirname) = fileparse $dirname;
      $dirname =~ s:(.)/*$:$1:;
    }
  }

  $dirname;
}
#dirname

fileparse_set_fstype $^O;

#######################################################################
# File::Spec geklaut : (Pseudo-Objektorientiert! Instance methods!!!
########################################################################

sub file_name_is_absolute {
  $^O =~ /mswin/i ? file_name_is_absolute_win32(@_) : file_name_is_absolute_unix(@_);
}
#filename_is_absolute
sub file_name_is_absolute_win32 {
  my($file) = @_;
  $file =~ m{^([a-z]:)?[\\/]}i ;
}
#file_name_is_absolute_win32

sub catdir {
  $^O =~ /mswin/i ? catdir_win32(@_) : catdir_unix(@_);
}
#catdir
sub catdir_win32 {
  my @args = @_;
  for (@args) {
    # append a slash to each argument unless it has one there
    $_ .= "\\" if $_ eq '' or substr($_,-1) ne "\\";
  }
  my $result = canonpath_win32(join('', @args));
  $result;
}
#catdir_win32

sub catfile {
  $^O =~ /mswin/i ? catfile_win32(@_) : catfile_unix(@_);
}
#catfile
sub catfile_win32 {
  my $file = pop @_;
  return $file unless @_;
  my $dir = catdir_win32(@_);
  $dir =~ s/(\\\.)$//;
  $dir .= "\\" unless substr($dir,length($dir)-1,1) eq "\\";
  return $dir.$file;
}
#catfile_win32

sub path {
  $^O =~ /mswin/i ? path_win32(@_) : path_unix(@_);
}
#path
sub path_win32 {
  local $^W = 1;
  my $path = $ENV{'PATH'} || $ENV{'Path'} || $ENV{'path'};
  my @path = split(';',$path);
  foreach (@path) {
    $_ = '.' if $_ eq '';
  }
  @path;
}
#path_win32

sub canonpath {
  $^O =~ /mswin/i ? canonpath_win32(@_) : canonpath_unix(@_);
}
#canonpath

sub canonpath_win32 {
  my($path) = @_;
  $path =~ s/^([a-z]:)/\u$1/;
  $path =~ s|/|\\|g;
  $path =~ s|\\+|\\|g ;         # xx////xx  -> xx/xx
  $path =~ s|(\\\.)+\\|\\|g ;   # xx/././xx -> xx/xx
  $path =~ s|^(\.\\)+|| unless $path eq ".\\"; # ./xx      -> xx
  $path =~ s|\\$|| 
    unless $path =~ m#^([a-z]:)?\\#; # xx/       -> xx
  $path .= '.' if $path =~ m#\\$#;
  $path;
}
#canonpath_win32


sub canonpath_unix {
  my($path) = @_;
  $path =~ s|/+|/|g ;           # xx////xx  -> xx/xx
  $path =~ s|(/\.)+/|/|g ;      # xx/././xx -> xx/xx
  $path =~ s|^(\./)+|| unless $path eq "./"; # ./xx      -> xx
  $path =~ s|/$|| unless $path eq "/"; # xx/       -> xx
  $path;
}
#canonpath_unix
sub catdir_unix {
  my @args = @_;
  for (@args) {
    # append a slash to each argument unless it has one there
    $_ .= "/" if $_ eq '' or substr($_,-1) ne "/";
  }
  my $result = join('', @args);
  # remove a trailing slash unless we are root
  substr($result,-1) = ""
    if length($result) > 1 && substr($result,-1) eq "/";
  $result;
}
#catdir_unix

sub catfile_unix {
  my $file = pop @_;
  return $file unless @_;
  my $dir = catdir_unix(@_);
  for ($dir) {
    $_ .= "/" unless substr($_,length($_)-1,1) eq "/";
  }
  return $dir.$file;
}
#catfile_unix



sub file_name_is_absolute_unix {
  my($file) = @_;
  $file =~ m:^/: ;
}
#file_name_is_absolute_unix

sub path_unix {
  my $path_sep = ":";
  my $path = $ENV{PATH};
  my @path = split $path_sep, $path;
  foreach (@path) {
    $_ = '.' if $_ eq '';
  }
  @path;
}
#path_unix




#############################
# File::Path geklaut
##############################



my $Is_VMS = 0;
my $Is_MacOS = $^O eq 'MacOS';

# These OSes complain if you want to remove a file that you have no
# write permission to:
my $force_writeable = ($^O eq 'os2' || $^O eq 'dos' || $^O eq 'MSWin32' ||
		       $^O eq 'amigaos' || $^O eq 'MacOS' || $^O eq 'epoc');

sub carp_replacement{
  #because SDBINST
  print2stderr $_[0];
  #endbecause
  #ifdef SCRIPTING
  #  print $_[0];
  #endif
}

sub mkpath {
  my($paths, $verbose, $mode) = @_;
  # $paths   -- either a path string or ref to list of paths
  # $verbose -- optional print "mkdir $path" for each directory created
  # $mode    -- optional permissions, defaults to 0777
  local($")=$Is_MacOS ? ":" : "/";
  $mode = 0777 unless defined($mode);
  $paths = [$paths] unless ref $paths;
  my(@created,$path);
  foreach $path (@$paths) {
    $path .= '/' if $^O eq 'os2' and $path =~ /^\w:\z/s; # feature of CRT  
      next if -d $path;
    my $parent = dirname($path);
    unless (-d $parent or $path eq $parent) {
      push(@created,mkpath($parent, $verbose, $mode));
    }
    unless (mkdir($path,$mode)) {
      my $e = $!;
      unless (-d $path){
        #because SDBINST
        print2stderr("ERROR: mkdir \"$path\" failed: $e\n");
        #endbecause
        #ifdef SCRIPTING
        #              print "ERROR: mkdir \"$path\" failed: $e\n";
        #endif
      }
    }
    push(@created, $path);
  }
  @created;
}
#mkpath

sub rmtree {
  my($roots, $verbose, $safe) = @_;
  my(@files);
  my($count) = 0;
  $verbose ||= 0;
  $safe ||= 0;

  if ( defined($roots) && length($roots) ) {
    $roots = [$roots] unless ref $roots;
  } else {
    carp_replacement "No root path(s) specified\n";
    return 0;
  }

  my($root);
  foreach $root (@{$roots}) {
    if ($Is_MacOS) {
      $root = ":$root" if $root !~ /:/;
      $root =~ s#([^:])\z#$1:#;
    } else {
      $root =~ s#/\z##;
    }
    (undef, undef, my $rp) = lstat $root or next;
    $rp &= 07777;           # don't forget setuid, setgid, sticky bits
    if ( -d _ ) {
      # notabene: 0777 is for making readable in the first place,
      # it's also intended to change it to writable in case we have
      # to recurse in which case we are better than rm -rf for 
      # subtrees with strange permissions
      chmod(0777, ($Is_VMS ? $root : $root))
        or carp_replacement "Can't make directory $root read+writeable: $!"
          unless $safe;

      if (opendir my $d, $root) {
        if (!defined ${"\cTAINT"} or ${"\cTAINT"}) {
          # Blindly untaint dir names
          @files = map { /^(.*)$/s ; $1 } readdir $d;
        } else {
          @files = readdir $d;
        }
        closedir $d;
      } else {
        carp_replacement "Can't read $root: $!";
        @files = ();
      }

      # Deleting large numbers of files from VMS Files-11 filesystems
      # is faster if done in reverse ASCIIbetical order 
      @files = reverse @files if $Is_VMS;
      if ($Is_MacOS) {
        @files = map("$root$_", @files);
      } else {
        @files = map("$root/$_", grep $_!~/^\.{1,2}\z/s,@files);
      }
      $count += rmtree(\@files,$verbose,$safe);
      if ($safe &&
          ($Is_VMS ? !-w $root : !-w $root)) {
        next;
      }
      chmod 0777, $root
        or carp_replacement "Can't make directory $root writeable: $!"
          if $force_writeable;
      if (rmdir $root) {
        ++$count;
      } else {
        carp_replacement "Can't remove directory $root: $!";
        chmod($rp, ($Is_VMS ? $root : $root))
          or carp_replacement("and can't restore permissions to "
                              . sprintf("0%o",$rp) . "\n");
      }
    } else { 
      if ($safe && !(-l $root || -w $root)) {
        next;
      }
      chmod 0666, $root
        or carp_replacement "Can't make file $root writeable: $!"
          if $force_writeable;
      # delete all versions under VMS
      for (;;) {
        unless (unlink $root) {
          carp_replacement "Can't unlink file $root: $!";
          if ($force_writeable) {
            chmod $rp, $root
              or carp_replacement("and can't restore permissions to "
                                  . sprintf("0%o",$rp) . "\n");
          }
          last;
        }
        ++$count;
        last unless $Is_VMS && lstat $root;
      }
    }
  }

  $count;
}
#rmtree
1;



# Local Variables: ***
# perl-dbg-flags:"-w      -I../../../../../../wrk/script -I../../../sdbrun  " ***
# sdbinst-defines: "-define SCRIPTING -define PERL5.6.1" ***
# sdbinst-testfile: "MobileInfraTest.pl" ***
# sdbinst-script-directory:  "sys/wrk/script/SAPDB/Install/Templates" ***
# End: ***

