#!/usr/bin/perl
#
# $Header: //sapdb/V74/develop/sys/src/install/perl/SAPDB/Install/Instance/Check/DataCons/LVC.pm#12 $
# $DateTime: 2002/04/05 10:37:05 $
# $Change: 19036 $
#
#    ========== licence begin  GPL
#    Copyright (c) 2005 SAP AG
#
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of the GNU General Public License
#    as published by the Free Software Foundation; either version 2
#    of the License, or (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#    ========== licence end


package SAPDB::Install::Instance::Check::DataCons::LVC;

sub BEGIN {
	@ISA = ('SAPDB::Install::Exporter');
	@EXPORT = ();
	my $repo = SAPDB::Install::Repository::GetCurrent ();
	my @neededPackages = (
		'Instance::Base',
		'Instance::Check::Common'
	);

	foreach my $package (@neededPackages) {
	  	unless (defined $repo->Eval 
		("SAPDB::Install::$package", 1.01)) {
                	print join ("\n", $repo->GetErr)."\n";
                	die;
        	}
		SAPDB::Install::Exporter::import ("SAPDB::Install::$package");
	} 
}

push @ISA, 
	'SAPDB::Install::Instance::Base',
	'SAPDB::Install::Instance::Check::Common';

#
# datacons ()
# check if data are consistent for migration
#
sub datacons {
	my ($self) = @_; 
	my $dbm = $self->{dbm};

	$self->set_errorstate ('ERROR');
	$self->msgbegin ("checking data consitency");
	$self->msg0 ("checking data consistency for liveCache migration...\n");

	#
	# find out if consistence in given by export to an external db
	# usually the apo system will do this by a report
	#
	my $reportrequired = $self->is_reportrequired ();
	unless (defined $reportrequired) {
		$self->msgend ();
		return -1;
	}

	if ($reportrequired != 0) {
		#
		# switch instance to online mode to read consistence flag
		#
		my $state =  $dbm->db_state ();
		unless (defined $state) {
			$self->msg1 ($dbm->lastdialog ());
			$self->msgend ();
			return -1;
		}

		unless (defined $self->{instancetypename}) {
			$self->{instancetypename} =
			$self->{instancetype} =~ /LVC/ ? 'liveCache' :
			'database';
		}
	
		if ($state =~ /WARM|ONLINE/) {
			$self->msg1 ($self->{instancetypename}.
		             " state is ".$state."\n");
		} else {
			$self->msg0 ($self->{instancetypename}.
		             " state is ".$state."\n");
		}

		#
		# if we are in admin mode,
		# we should switch to offline before switch to online again
		#
		if ($state =~ /COLD|ADMIN/) {
			$state = $self->switchto ('OFFLINE', "switch ".
			$self->{instancetypename}." state to OFFLINE\n");

			$state =  $dbm->db_state ();
			unless (defined $state) {
				$self->msg1 ($dbm->lastdialog ());
				$self->msgend ();
				return -1;
			}

			$self->msg0 ($self->{instancetypename}.
		             " state is ".$state."\n");
		}

		#
		# we should switch instance offline from unknown mode
		#
		unless ($state =~ /WARM|ONLINE|OFFLINE/) {
			$state = $self->switchto ('CLEAR', "clear ".
			$self->{instancetypename}." state\n");

			unless ($state eq 'OFFLINE') {
				$self->msgend ();
				return (-1);
			}
		}

		#
		# if the instance is not already online,
		# we have to bring it into online mode
		#
		my $prev_state;
		if ($state =~ /WARM|ONLINE/) {
			$prev_state = $state;
		} else {
			$prev_state = 'OFFLINE';
		
			if ($self->{'can_db_online'}) {
				$state = $self->switchto ('ONLINE', "switch ".
				$self->{instancetypename}." state to ONLINE\n"); 
			} else {
				$state =  $self->switchto ('WARM', "switch ".
				$self->{instancetypename}." state to WARM\n"); 
			}

			unless ($state =~ /WARM|ONLINE/) {
				$self->get_errmsg_from_knldiag ();
				$self->msgend ();
				return (-1);
			}
		}

		my $is_reportconsistent = $self->is_reportconsistent ();
		unless (defined $is_reportconsistent) {
			$state = $self->switchto_prev ($state, $prev_state);
			$self->msgend ();
			return -1;
		}

		unless ($is_reportconsistent == 1) {
			$state = $self->switchto_prev ($state, $prev_state);
			$self->msgend ();
			return -1;
		}

		#
		# switch instance back to prev state
		#
		$state = $self->switchto_prev ($state, $prev_state);
		unless ($state eq $prev_state) {
			$self->msgend ();
			return -1;
		}

		$self->msgend ();
		$self->set_errorstate ('OK');
		return 0;
	}

	#
	# find out if consistence is required for this migration
	#
	my $consistencerequired = $self->is_consistencerequired ();
	unless (defined $consistencerequired) {
		$self->msgend ();
		return -1;
	}
	
	if ($consistencerequired == 0) {
		$self->msgend ();
		$self->set_errorstate ('OK');
		return 0;
	}

	#
	# find out if migration backup is required for this migration
	#
	my ($backuprequired, $required_kernel_version) = 
	$self->is_backuprequired ();

	unless (defined $backuprequired) {
		$self->msgend ();
		return -1;
	}

	#
	# switch instance to cold mode to get 'db_restartinfo'
	#
	my $state = $dbm->db_state ();
	unless (defined $state) {
		$self->msg1 ($dbm->lastdialog ());
		$self->msgend ();
		return -1;
	}

	unless (defined $self->{instancetypename}) {
        $self->{instancetypename} =
		$self->{instancetype} =~ /LVC/ ? 'liveCache' : 'database';
	}
	
	if ($state =~ /COLD|ADMIN/) {
		$self->msg1 ($self->{instancetypename}.
		" state is ".$state."\n");
	} else {
		$self->msg0 ($self->{instancetypename}.
		" state is ".$state."\n");
	}

	#
	# if instance is still online and no migration backup is required
	# we can assume that instance will be consistent after
	# normal shutdown 
	#
	if ($state =~ /WARM|ONLINE/ && $backuprequired == 0) {
		$self->msg1 ($self->{instancetypename}.
		" will be consistent after normal shutdown\n");
		$self->msgend ();
		$self->set_errorstate ('OK');
		return 0;
	}

	#
	# if instance is still online and migration backup is required
	# the current log page is not contained in last backup
	#
	if ($state =~ /WARM|ONLINE/ && $backuprequired != 0) {
		$self->msg1 ("migration backup is out of date\n");
		$self->msgend ();
		return (-1);
	}

	#
	# we should switch instance offline from unknown mode
	#
	unless ($state =~ /WARM|ONLINE|COLD|ADMIN|OFFLINE/) {
		$state = $self->switchto ('CLEAR', "clear ".
		$self->{instancetypename}." state\n");

		unless ($state eq 'OFFLINE') {
			$self->msgend ();
			return (-1);
		}
	}

	#
	# if the instance is not already cold,
	# we have to bring it into cold mode
	#
	my $prev_state;
	if ($state =~ /COLD|ADMIN/) {
		$prev_state = $state;
	} else {
		$prev_state = 'OFFLINE';

		if ($self->{'can_db_admin'}) {
			$state =  
			$self->switchto ('ADMIN', "switch ".
			$self->{instancetypename}." state to ADMIN\n"); 
		} else {
			$state =  
			$self->switchto ('COLD', "switch ".
			$self->{instancetypename}." state to COLD\n"); 

		}

		unless ($state  =~ /COLD|ADMIN/) {
			$self->get_errmsg_from_knldiag ();
			$self->msgend ();
			return (-1);
		}
	}
	
	#
	# get restartinfo and look for normal shutdown with checkpoint
	#
	my $restartinfo = $self->get_restartinfo ();
	
	my $is_consistent = $self->is_consistent ($restartinfo);
	unless (defined $is_consistent) {
		$state = $self->switchto_prev ($state, $prev_state);
		$self->msgend ();
		return -1;
	}

	unless ($is_consistent == 1) {
		$state = $self->switchto_prev ($state, $prev_state);
		$self->msgend ();
		return -1;
	}

	if ($backuprequired == 0) {
		$self->msgend ();
		$self->set_errorstate ('OK');
		return 0;
	}
	
	my $is_backupconsistent =
	$self->is_backupconsistent ($restartinfo, $required_kernel_version);
	unless (defined $is_backupconsistent) {
		$state = $self->switchto_prev ($state, $prev_state);
		$self->msgend ();
		return -1;
	}

	unless ($is_backupconsistent == 1) {
		$state = $self->switchto_prev ($state, $prev_state);
		$self->msgend ();
		return -1;
	}

	$state = $self->switchto_prev ($state, $prev_state);
	unless ($state eq $prev_state) {
		$self->msgend ();
		return -1;
	}

	$self->msgend ();
	$self->set_errorstate ('OK');
	return 0;
}

sub is_consistencerequired {
	my ($self) = @_;

	my $migration_strategy = $self->get_migrationstrategy ();
	return undef unless (defined $migration_strategy);

	return ($migration_strategy =~ /^COMPATIBLE_LOG/ ? 0 : 1);
}

sub is_backuprequired {
	my ($self) = @_;
	
	my $migration_strategy = $self->get_migrationstrategy ();
	return undef unless (defined $migration_strategy);

	my $need_backup = 
	$migration_strategy =~ /^EXTERNAL_CONSISTENT_BACKUP/ ? 1 : 0;

	my $required_kernel_version = undef;
	if ($need_backup != 0) {
		if (($self->{starting_release} =~ /^7.2.(\d+)/ && $1 == 5) &&
		    ($self->{target_release} =~ /^7.4.(\d+)/) && $1 == 2) {
				$required_kernel_version = '7.2.05 Build 015';
		}
	}

	if (defined $required_kernel_version) {	
		return ($need_backup, $required_kernel_version);
	}

	return ($need_backup);
}

sub is_reportrequired {
	my ($self) = @_;
	
	my $migration_strategy = $self->get_migrationstrategy ();
	return undef unless (defined $migration_strategy);

	return
	($migration_strategy =~ /^EXTERNAL_LIVECACHE_CONSISTENCE/ ? 1 : 0);
}

sub is_consistent {
	my ($self, $restartinfo) = @_;

	return undef unless (defined $restartinfo);
	return undef unless (defined ($restartinfo->{'Consistent'}));
	return undef unless (defined ($restartinfo->{'Restartable'}));

	if (($restartinfo->{'Consistent'} &&
	     $restartinfo->{'Restartable'}) == 0) {
		$self->msg1 ("liveCache was stoped without checkpoint\n");
		return 0;
	}

	$self->msg1 ("liveCache was stoped with checkpoint\n");
	return 1;
}

sub is_backupconsistent {
	my ($self, $restartinfo, $required_kernel_version) = @_;

	return undef unless (defined $restartinfo);
	
	#
	# get last used log page number of instance from restartinfo
	#
	my $db_used_log_page = $restartinfo->{'Used LOG Page'};
	return undef unless (defined $db_used_log_page);

	$self->msg1 (
	"last used log page of liveCache is ".$db_used_log_page."\n");

	#
	# look for migration backup in backup history
	# and get utility cmdid for this backup
	#
	my $dbm = $self->{'dbm'};
	unless (defined ($dbm->backup_history_open ())) {
		$self->msg1 ("cannot open backup history\n");
		$self->msg1 ("\n");
		$self->msg1 ($dbm->lastdialog ());
		return undef;
	}

	my $history = 
	$dbm->backup_history_list ('-r LAST -c KEY,LABEL,ACTION,LOG,RC');
	unless (defined $history) {
		$self->msg1 ($dbm->lastdialog ());
		return undef;
	}

	unless (defined ($dbm->backup_history_close ())) {
		$self->msg1 ("cannot close backup history\n");
		$self->msg1 ("\n");
		$self->msg1 ($dbm->lastdialog ());
		return undef;
	}

	$self->msg1 ("looking in backup history for migration backup\n");

	my $history_lines = $#{$history->{'KEY'}} + 1;
	if ($history_lines <= 0) {
		$self->msg1 ("empty backup history\n");
		$self->msg1 ("no migration backup found\n");
		return undef;
	}

	$self->msg1 ("\n");
	$self->msg1 ("  KEY          LABEL     ACTION    LOG  RC\n");
	for (my $i = 0; $i < $history_lines; $i++) {
		my $txt = '  ';
		$txt .= $history->{'KEY'}->[$i]." ";
		$txt .= $history->{'LABEL'}->[$i]." ";
		$txt .= $history->{'ACTION'}->[$i]." ";
		$txt .= $history->{'LOG'}->[$i]." ";
		$txt .= $history->{'RC'}->[$i]." ";
		$self->msg1 ($txt."\n");
	}
	$self->msg1 ("\n");
	
	if ($history_lines != 1) {
		$self->msg1 ("more than one backup needed for reovery\n");
		$self->msg1 ("no migration backup found\n");
		return undef;
	}

	unless ($history->{'ACTION'}->[0] =~ /^SAVE COLD/) {
		$self->msg1 ("backup was not made in cold mode\n");
		$self->msg1 ("no migration backup found\n");
		return undef;
	}

	unless ($history->{'LABEL'}->[0] =~ /^DAT/) {
		$self->msg1 ("backup is no complete data backup\n");
		$self->msg1 ("no migration backup found\n");
		return undef;
	}

	unless ($history->{'LOG'}->[0] =~ /^NO/) {
		$self->msg1 ("backup is not consistent\n");
		$self->msg1 ("no migration backup found\n");
		return undef;
	}

	unless ($history->{'RC'}->[0] == 0) {
		$self->msg1 ("backup was unsuccessfull\n");
		$self->msg1 ("no migration backup found\n");
		return undef;
	}

	#
	# looking for cmdid in dbmutl and get matching utlcmd
	#
	my $cmdid = $history->{'KEY'}->[0];
	$self->msg1 ("looking in dbmutl for cmdid ".$cmdid."\n");

	my $dbmutl = $self->get_dbmutl ($cmdid);
	unless (defined $dbmutl) {
		return undef;
	}

	my $dbmutl_lines = $#{$dbmutl->{'TEXT'}} + 1;
	if ($dbmutl_lines <= 0) {
		$self->msg1 ("cmdid not found in dbmutl\n");
		return undef;
	}

	my @utlcmd = ();
	for (my $i = 0; $i < $dbmutl_lines; $i++) {
		push @utlcmd, $dbmutl->{'TEXT'}->[$i];
	}

	$self->msg1 ("\n");
	foreach (@utlcmd) {
		$self->msg1 ("  ".$_."\n");
	}
	$self->msg1 ("\n");	

	my $backup_kernel_version;	
	my $msg = "backup was ";
	foreach (@utlcmd) {
		if (/^RETURNCODE\s+(\d)/) {
			unless ($1 == 0) {
				$msg .= "unsucessfull";
				$self->msg1 ($msg."\n");
				return 0;
			}
			$msg .= "sucessfull";
			next;
		}

		if (/^IS CONSISTENT[\.\s]+(\w+)/) {
			unless ($1 =~ /TRUE/) {
				$msg .= ", unconsistent";
				$self->msg1 ($msg."\n");
				return 0;
			}
			$msg .= ", consistent";
			next;
		}
	
		if (/^FIRST LPNO[\.\s]+(\d+)/) {
			if ($1 != $db_used_log_page) {
				$msg .= ", out of date";
				$self->msg1 ($msg."\n");
				return 0;
			}
			$msg .= ", up to date";
			next;
		}

		if (/^KERNEL VERSION[\.\s]+Kernel\s+([\w\s\.\-]+)/) {
			$backup_kernel_version = $1;
			next;
		}
	}
	$self->msg1 ($msg."\n");

	#
	# return success if no kernel version for backup is required
	#
	$self->msg1 ("\n");
	unless (defined $required_kernel_version) {
		$self->msg1 (
		"no kernel version for migration backup required\n");
		return 1;
	}

	$self->msg1 (
	"required kernel version for migration backup is\n");
	$self->msg1 ($required_kernel_version." or better\n");
	$self->msg1 (
	"used kernel version for migration backup is\n");
	$self->msg1 ($backup_kernel_version."\n");


	my ($rkv_major, $rkv_minor, $rkv_corr, $rkv_build) = 
	split_buildstring ($required_kernel_version);

	my ($bkv_major, $bkv_minor, $bkv_corr, $bkv_build) = 
	split_buildstring ($backup_kernel_version);

	unless ($bkv_major == $rkv_major && $bkv_minor == $rkv_minor &&
	        $bkv_corr == $rkv_corr && $bkv_build >= $rkv_build) {
		$self->msg1 (
		"used kernel version is not equal or better than required\n");
		return 0;
	}

	$self->msg1 (
	"used kernel version is equal or better than required\n");
	return 1;
}

sub split_buildstring {
	my ($buildstring) = @_; 

	if ($buildstring =~ /^(\d+)\.(\d+)\.(\d+)\s+Build\s(\d+)\-/) {
		return ($1, $2, $3, $4);
	}

	if ($buildstring =~ /^(\d+)\.(\d+)\.(\d+)\s+Build\s(\d+)/) {
		return ($1, $2, $3, $4);
	}
	
	return undef;
}

sub is_reportconsistent {
	my ($self) = @_;
	my $dbm = $self->{dbm};

	$self->msg0 ("looking for ready for update flag...\n");

	my $apo_target_release = $self->{'apo_target_release'};

	my $reportname =
	($apo_target_release =~ /^3\.\d\./) ? 
		"report /SAPAPO/OM_ORDER_LC_TO_DBSTART" :
	($apo_target_release =~ /^4\.0\./) ? 
		"step B1 (download liveCache data) of report /SAPAPO/OM_LC_UPGRADE_3X_40" :
	($apo_target_release =~ /^4\.1\./) ? 
		"step B1 (download liveCache data) of report /SAPAPO/OM_LC_UPGRADE_41" :
		"step B1 (download liveCache data) of report /SAPAPO/OM_LC_UPGRADE_50";

	my $cmd = 'select owner,tablename from tables where '.
	          'tablename = \'__$__LCFLAG\'';

	my $sql = $dbm->sql_execute ($cmd);
	unless (defined $sql) {
		$self->msg1 ("\n");
		$self->msg1 ($dbm->lastdialog ());
		$self->msg0 ("cannot read ready for update flag\n");
		return 0;
	}

	if (defined $sql->{error} && ${$sql->{error}}[0] != 100) {
		$self->msg1 ("\n");
		$self->msg1 ($dbm->lastdialog ());
		$self->msg0 ("cannot read ready for update flag\n");
		return 0;
	}

	if (defined $sql->{error} && ${$sql->{error}}[0] == 100) {
		$self->msg1 ("\n");
		$self->msg1 ($dbm->lastdialog ());
		$self->msg0 ("liveCache is not ready for update\n");
		$self->msg0 ("\n");
		$self->msg0 ("make sure that all liveCache transaction data\n");
		$self->msg0 ("stored in the APO system\n");
		$self->msg0 ("by using ".$reportname."\n");
		$self->msg0 ("\n");
		return 0;
	}

	my $result = ${$sql->{result}}[0];	

	$self->msg1 ("\n");
	$self->msg1 ("  ".$cmd."\n");
	$self->msg1 ("  ".$result."\n");
	$self->msg1 ("\n");

	my ($owner) = ($result =~ /^'(.*)';'.*'$/);

	$cmd = ('select flag, begin, end, version from '.
	        $owner.'."__$__LCFLAG" where flag = \'X\'');

	$sql = $dbm->sql_execute ($cmd);
	unless (defined $sql) {
		$self->msg1 ("\n");
		$self->msg1 ($dbm->lastdialog ());
		$self->msg0 ("cannot read ready for update flag\n");
		return 0;
	}

	if (defined $sql->{error}) {
		$self->msg1 ("\n");
		$self->msg1 ($dbm->lastdialog ());
		$self->msg0 ("cannot read ready for update flag\n");
		return 0;
	}

	$result = ${$sql->{result}}[0];

	$self->msg1 ("  ".$cmd."\n");
	$self->msg1 ("  ".$result."\n");
	$self->msg1 ("\n");

	unless ($result =~ /^'X';/) {
		$self->msg0 ("liveCache is not ready for update\n");
		$self->msg0 ("\n");
		$self->msg0 ("make sure that all liveCache transaction data\n");
		$self->msg0 ("stored in the APO system\n");
		$self->msg0 ("by using ".$reportname."\n");
		$self->msg0 ("\n");
		return 0;
	}

	$self->msg1 ("liveCache is ready for update\n");
	return 1;
}

1;

