#!/usr/bin/perl
#
# $Header: //sapdb/V75/c_00/develop/sys/src/install/perl/SAPDB/Install/Instance/ChVolPerm.pm#3 $
# $DateTime: 2003/12/11 13:40:27 $
# $Change: 59245 $
#
#    ========== licence begin  GPL
#    Copyright (c) 2005 SAP AG
#
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of the GNU General Public License
#    as published by the Free Software Foundation; either version 2
#    of the License, or (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#    ========== licence end


package SAPDB::Install::Instance::ChVolPerm;

#
# usually these function is called in 
# the postinstall step of SDBUPD
#
sub BEGIN {
	@ISA = ('SAPDB::Install::Exporter');
	@EXPORT = ('chvolperm');
	my $repo = SAPDB::Install::Repository::GetCurrent ();
	my @neededPackages = (
		'Instance::Base',
		'InstInfo',
		'GetParam',
		'System::Unix',
		'System::Unix::Dir'
	);

	foreach my $package (@neededPackages) {
		unless (
		defined $repo->Eval ("SAPDB::Install::$package", 1.01)) {
			print join ("\n", $repo->GetErr)."\n";
			die;
		}
		SAPDB::Install::Exporter::import ("SAPDB::Install::$package");
	}
}

push @ISA, 'SAPDB::Install::Instance::Base'; 

sub chvolperm {
	my ($self, $dbname) = @_;

	if ($^O =~ /mswin/i) {
		$self->set_errorstate ('OK');
		return 0;
	}

	$self->set_errorstate ('ERROR');
	$self->msgbegin ("changing permissions of database files");

	if (defined $self->{dbname}) {
		$dbname = $self->{dbname};
	} else {
		$self->{dbname} = $dbname;
	}

	$self->msg0 ("changing permissions of database files...\n");

	my $info = SAPDB::Install::InstInfo::new ();
	my $user = $info->{'user'};
	my $group = $info->{'group'};
	my $datadir = $info->{'datadir'};
	
	foreach my $filename (get_datafiles_of_db ($datadir, $dbname)) {
		unless ($self->chperm_file ($filename, $user, $group) == 0) {
			$self->msg0 ("change permissions of ".$filename." failed\n");
			$self->msgend ();
			return -1;
		}
	}

	foreach my $filename (get_configfiles_of_db ($datadir, $dbname)) {
		unless ($self->chperm_file ($filename, $user, $group) == 0) {
			$self->msg0 ("change permissions of ".$filename." failed\n");
			$self->msgend ();
			return -1;
		}
	}

	foreach my $filename (get_obsoletedfiles_of_db ($datadir, $dbname)) {
		unless ($self->remove_file ($filename) == 0) {
			$self->msg0 ("cannot remove ".$filename."\n");
			$self->msgend ();
			return -1;
		}
	}

	$self->set_errorstate ('OK');
	$self->msgend ();
	return 0;
}

sub getabsname {
	my ($myname, $theirname) = @_;

	return ($theirname) if ($theirname =~ /^\//);

	($myname) = $myname =~ /^(.*)\/+[^\/]*$/;
	return ($myname.'/'.$theirname);
}

sub rmfile {
	my ($filename) = @_;

	return -1 unless (-e $filename);
	my $myname = $filename;
	for (;;) {
		my @statbuff = lstat ($myname);
		my $owner = getpwuid ($statbuff[4]);

		# do not unlink files or links under /dev/...
		last if ($myname =~ /^\/dev\//);

		# follow links
		if (-l $myname) {
			my $theirname = readlink ($myname);
			return -1 unless (defined $theirname);

			# do not delete system links
			unless ($owner =~ /^root|sys|bin|unix$/) {
				$rc = unlink ($myname);
				return -1 unless ($rc == 0);
			}

			$myname = getabsname ($myname, $theirname);
		} else {
			$rc = unlink ($myname);
			return -1 unless ($rc == 1);

			last;
		}
	}

	return 0;
}

sub chperm {
	my ($uid, $gid, $mode, $filename) = @_;
	my $rc;

	return -1 unless (-e $filename);

	my $myname = $filename;
	for (;;) {
		my @statbuff = lstat ($myname);
		my $owner = getpwuid ($statbuff[4]);

		if (-l $myname) {
			my $theirname = readlink ($myname);
			return -1 unless (defined $theirname);

			# do not modify system links
			unless ($owner =~ /^root|sys|bin|unix$/) {
				$rc = lchown ($uid, $gid, $myname);
				return -1 unless ($rc == 1);
			}

			$myname = getabsname ($myname, $theirname);
		} else {
			$rc = chmod ($mode, $myname);
			return -1 unless ($rc == 1);

			$rc = chown ($uid, $gid, $myname);
			return -1 unless ($rc == 1);

			last;
		}
	}
	
	return 0;
}

sub remove_file {
	my ($self, $filename) = @_;
	return 0 unless (-e $filename);
	
	$self->msg1 ("remove ".$filename."\n");

	my $rc = rmfile ($filename);
	return ($rc != 0) ? -1 : 0;
}

sub chperm_file {
	my ($self, $filename, $user, $group) = @_;

	return 0 unless (-e $filename);

	my $uid = getpwnam ($user);
	my $gid = getgrnam ($group);

	my $mode = (-d $filename) ? 0770 : 0660;

	my $msg =
	sprintf ("chperm %o %s:%s %s\n", $mode, $user, $group, $filename);
	$self->msg1 ($msg);

	my $rc = chperm ($uid, $gid, $mode, $filename);
	return -1 unless ($rc == 0);

	$rc = uaccess ($user, $filename);
	unless ($rc == 0) {
		$self->msg1 ("user $user cannot access $filename\n");
		$self->msg1 ("modify permissions of parent directories\n");

		my @ancestors =
		SAPDB::Install::System::Unix::Dir::getancestors ($filename);
		foreach my $dir (@ancestors) {
			next if (-l $dir);

			my ($dev, $ino, $mode) = stat ($dir);
			$self->msg1 ("mode of $dir is ".sprintmode ($mode)."\n");
			next if (($mode & 0111) == 0111);

			$self->msg1 ("chmod $dir ugo+x\n");
			chmod (($mode & 07777) | 00111, $dir);
		}
	}

	$rc = uaccess ($user, $filename);
	unless ($rc == 0) {
		$self->msg1 ("user $user cannot access $filename\n");
		$self->msg1 ("check permissions of all parent directories of $filename\n");
		$self->msg1 ("and restart installation\n");
		return -1;
	}

	$self->msg1 ("user $user can access $filename\n");
	return 0;
}

sub tree {
	my ($dirname) = @_;

	my $rc = opendir (DIR, $dirname);
	my @dir = readdir (DIR);
	close (DIR);

	my @list = ();
	foreach my $filename (@dir) {
		next if ($filename eq '.' || $filename eq '..');

		$filename = $dirname.'/'.$filename;
		push @list, $filename;
		push @list, tree ($filename) if (-d $filename);
	}

	return @list;
}

sub get_obsoletedfiles_of_db {
	my ($datadir, $dbname) = @_;

	my $wrkdir = $datadir.'/wrk';

	return () unless (opendir (DIR, $wrkdir));
	my @dir = readdir (DIR);
	closedir (DIR);

	my @files = ();
	foreach my $filename (@dir) {
		next if (-d $wrkdir.'/'.$filename);

		if (
		$filename eq 'dbmsrv.prt' ||
		$filename eq 'loader.prt' ||
		$filename =~ /(\S+)\..*/ && $1 eq $dbname) {
			push @files, $wrkdir.'/'.$filename;
		}
	}

	return @files;
}

sub get_configfiles_of_db {
	my ($datadir, $dbname) = @_;

	my $configdir = $datadir.'/config';

	return () unless (opendir (DIR, $configdir));
	my @dir = readdir (DIR);
	closedir (DIR);

	my @files = ();
	foreach my $filename (@dir) {
		if (
		$filename eq $dbname ||
		$filename =~ /(\S+)\..*/ && $1 eq $dbname) {
			push @files, $configdir.'/'.$filename;
		}
	}

	return @files;
}

sub get_datafiles_of_db {
	my ($datadir, $dbname) = @_;

	my $param =
	SAPDB::Install::GetParam::GetParam ($dbname);

	my $wrkdir = $param->{'data'}->{'RUNDIRECTORY'};
	
	# keep all known files or directories from paramfile
	my @files = ();
	while (my ($key, $val) = each (%{$param->{'data'}})) {
		if (
		$key eq '_BACKUP_HISTFILE' ||
		$key eq '_BACKUP_MED_DEF' ||
		$key eq '_EVENTFILE' ||
		$key eq '_KERNELDIAGFILE' ||
		$key eq '_KERNELDUMPFILE' ||
		$key eq '_KERNELTRACEFILE' ||
		$key eq '_RTEDUMPFILE' ||
		$key =~ '_UTILITY_PROTFILE' ||
		$key =~ /^SYSDEV_\d+$/ ||
		$key =~ /^M_SYSDEV_\d+$/ ||
		$key =~ /^DATADEV_\d+$/ ||
		$key =~ /^M_DATADEV_\d+$/ ||
		$key =~ /^ARCHIVE_LOG_\d+$/ ||
		$key =~ /^M_ARCHIVE_LOG_\d+$/ ||
		$key =~ /^DATA_VOLUME_NAME_\d+$/ ||
		$key =~ /^LOG_VOLUME_NAME_\d+$/ ||
		$key =~ /^M_LOG_VOLUME_NAME_\d+$/ ||
		$key =~ /^M_DATA_VOLUME_NAME_\d+$/ ||
		$key eq 'RUNDIRECTORY' ||
		$key eq 'DIAG_HISTORY_PATH'
		) {
			push @files, $val;
			next;
		}
	}

	# use absolute file names
	foreach my $file (@files) {
		next if ($file =~ /^\//);

		$file = $wrkdir.'/'.$file;
	}

	# resolve directories
	my @tree = ();
	foreach my $file (@files) {
		push @tree, tree ($file) if (-d $file);
	}

	# create unique list of files and resolved directories
	@files = sort (@files, @tree);
	for (my $i = 0; $i < $#files - 1; $i++) {
		if ($files[$i] eq $files[$i + 1]) {
			splice (@files, $i, 1);
		}
	}

	return @files;
}

#
# main
#
sub main {
	return (SAPDB::Install::Instance::ChVolPerm::Test::main (@_));
}

package SAPDB::Install::Instance::ChVolPerm::Test;

$DEBUG = 0;

sub BEGIN {
	@ISA = ('SAPDB::Install::Exporter');
	@EXPORT = ();
	my $repo = SAPDB::Install::Repository::GetCurrent ();
	my @neededPackages = (
		'Getopt::Long'
	);

	foreach my $package (@neededPackages) {
		unless (
		defined $repo->Eval ("SAPDB::Install::$package", 1.01)) {
			print join ("\n", $repo->GetErr)."\n";
			die;
		}
		SAPDB::Install::Exporter::import ("SAPDB::Install::$package");
	}
}

sub main {
	local @ARGV = @_;

	my ($dbname, $verbose, $help);

	# allow interpretation of '-v' as '--verbose'
	SAPDB::Install::Getopt::Long::Configure ('auto_abbrev');
	GetOptions (
		'dbname=s', \$dbname, 
		'help', \$help,
		'verbose', \$verbose);

	if (defined $help) {
		print "SAPDB::Install::Instance::ChVolPerm::main\n";
		print "usage:\n";
		print "  -d (--dbname) <dbname>\n";
		print "  -v (--verbose)\n";
		print "  -h (--help)\n";
		print "\n";
		return 0;
	}

	$DEBUG = 1 if ($verbose);

	my $ic = SAPDB::Install::Instance::ChVolPerm->new ();

	$ic->{msg0} = \&msg0;
	$ic->{msg1} = \&msg1 if ($DEBUG);

	unless (defined $dbname) {
		printmsg ("missing instance name\n");		
		return 0;
	}

	if ($ic->chvolperm ($dbname) == 0) {
		printmsg ("\n");
		printmsg ("change permissions successfully done\n");
		printmsg ("errorstate = ".$ic->get_errorstate ()."\n");
	} else {
		printmsg ("\n");
		printmsg ("change permissions failed\n");
		printmsg ("\n");
		printmsg ($ic->{msgtxt});
		printmsg ("errorstate = ".$ic->get_errorstate ()."\n");
	}
	return 0;
}

sub msg1 {
	print "INFO>>> ";
	print @_;
}

sub msg0 {
	print "MSG >>> ";
	print @_;
}

sub printmsg {
	my $txt = shift;

	msg0 ($txt) if ($txt eq "\n");

	foreach (split '\n', $txt) {
		msg0 ($_."\n");
	}
}

1;
