#!/usr/bin/perl
#    ========== licence begin  GPL
#    Copyright (c) 2005 SAP AG
#
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of the GNU General Public License
#    as published by the Free Software Foundation; either version 2
#    of the License, or (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#    ========== licence end

package SAPDB::Install::Templates::Setup;
#ifdef SCRIPTING
#use strict;
#use Getopt::Long;
#use SAPDB::Install::Templates::MobileInfra;
#use SAPDB::Install::Templates::Base;
#use SAPDB::Install::FSUsage qw(GetFSSize GetFSName);
#endif

sub BEGIN {
  @ISA = ('SAPDB::Install::Exporter');
  my $repo = SAPDB::Install::Repository::GetCurrent ();
  my @neededPackages=(
                      'Getopt::Long',
                      'Templates::MobileInfra',
                      'Templates::Simple',
                      'StdIO',
                      'Registry',
                      'FSUsage',
                      'Base'
                     );
  @EXPORT = qw (  initializeTemplateOptions traceIsOff getActiveTemplate fsinfo_templates );
  foreach my $package (@neededPackages) {
    unless (defined $repo->Eval ("SAPDB::Install::$package", 1.01)) {
      print join ("\n", $repo->GetErr)."\n";
      die ("error loading:  SAPDB::Install::$package\n");
    }
    SAPDB::Install::Exporter::import ("SAPDB::Install::$package"); 
  }
}
#BEGIN


##############################################################################################
#ERROR HANDLING
##############################################################################################

#Dummy error output
sub myerror{
  print2stderr ( $_[0]);
}


#Dummy warn
sub mywarn{
  print2stderr ($_[0]);
}
#mywarn


sub onInvalidTemplate($){
  my $message=
    "*********************************************************************\n" .
      "INVALID TEMPLATE SPECIFICATION: \"$_[0]\"\n\n" .
        "Specify a legal value!\n" .
          "At the moment, the only legal value is \"mi\",\n"  .
            "which is the template for the mobile infrastructure.\n" .
              "*********************************************************************\n" ;
  print2stdout("\n$message");
}
#onInvalidTemplate
sub logger{
  $SAPDB::Install::Values::log->SetMsg($_[0]); 
}

sub status{
  print2stdout("\n" . $_[0]);
}

##############################################################################################
#Singleton instance vor the active Template
##############################################################################################

#+field(TemplateRef:the active template)
my $activeTemplate ; 
my $setup_opt_instrootpath;
my $setup_opt_configfile;
my $setup_opt_drive;
my $setup_opt_template;
my $setup_opt_dbname;

#Get the active Template as Object instance of type Template
#+sub():TemplateRef:Get the active template
sub getActiveTemplate(){
  return $activeTemplate;
}
#getActiveTemplate


#Set the active template by specifiying a template type as a string
#undef is a legal value and causes no action
sub setActiveTemplateByType($){
  my $aTemplateType = $_[0];
  if (isMobileInfraType($aTemplateType)) {
    $activeTemplate = SAPDB::Install::Templates::MobileInfra::new();
    if (defined $setup_opt_configfile) {
      $activeTemplate->configureFromFile($setup_opt_configfile);
    }
    if (defined $setup_opt_instrootpath) {
      $activeTemplate->setInstRootPath($setup_opt_instrootpath);
    }
    if (defined $setup_opt_drive) {
      $activeTemplate->setDrive($setup_opt_drive);
    }
    $activeTemplate->setStatusputter(\&status);
    $activeTemplate->setOutPutter(\&logger);
  } elsif (isSimpleWorkstationType($aTemplateType)) {
    $activeTemplate = SAPDB::Install::Templates::Simple::new();
    if (defined $setup_opt_configfile) {
      $activeTemplate->configureFromFile($setup_opt_configfile);
    }
    $activeTemplate->setStatusputter(\&status);
    $activeTemplate->setOutPutter(\&logger);
  } elsif (defined $aTemplateType) {
    return undef;
  } else {
    return undef;
  }
}
#setActiveTemplateByType

sub isMobileInfraType($){
  return ( defined $_[0] and $_[0]  eq "mi");
}

sub isSimpleWorkstationType($){
  my $aType = $_[0];
  return ( defined $aType and $aType  eq "simple");
}


sub isSimpleWorkstationTemplate($){
  my $aTemplate = $_[0];
  return (defined $aTemplate and isSimpleWorkstationType($aTemplate->getType()));
}



sub isMobileInfraTemplate($){
  my $aTemplate = $_[0];
  return ( defined $aTemplate and isMobileInfraType(  $aTemplate->getType()));
}







##############################################################################################
#Package script methods
##############################################################################################






sub initializeTemplateOptions(){

  $setup_opt_configfile  = $SAPDB::Install::StartInstall::opt_config;
  $setup_opt_instrootpath = $SAPDB::Install::StartInstall::opt_instrootpath;
  $setup_opt_drive = $SAPDB::Install::StartInstall::opt_drive;
  $setup_opt_template = $SAPDB::Install::StartInstall::opt_template;
  $setup_opt_dbname = $SAPDB::Install::StartInstall::opt_d;

  if ( defined $setup_opt_template) {
    setActiveTemplateByType($setup_opt_template);
  }

  if ( defined $setup_opt_configfile) {
    if ( (not -f $setup_opt_configfile)) {
      $setup_opt_configfile=undef;
      kackAb("Configuration file not found: $setup_opt_configfile\n");
    } else {
      my $section;
      my $templateType;
      open CONFIGFILE, "<$setup_opt_configfile";
      while (<CONFIGFILE>) {
        my ($key,$value,$type);
        SAPDB::Install::Templates::Base::parseConfigLine
            ( $_,
              sub{ $section =$_[0];},
              sub {
                my $key = $_[0];
                my $val = $_[1];
                if ( defined $section and defined $key and $key =~ /type/i and $section eq 'Template') {
                  if ( defined $val and  $val =~ /^\s*(\w+)\s*$/ ) {
                    $templateType=$1;
                  }
                }
              });
      }                         #while
      close CONFIGFILE;
      if (defined $templateType) {
        setActiveTemplateByType($templateType);
      }
    }                           #else
  }                             #if
}
#initializeTemplateOptions


#+sub(HashRef:paths|HashRef:config):void:Called from SDBINST during package installation
sub framestart($$){
  initializeTemplateOptions();
  my $configref = $_[1];
  my %config = %$configref;
  my $aTemplateType =$config{'template'};
  my $anActiveTemplate = getActiveTemplate();
  if ( (defined $aTemplateType) && (defined $anActiveTemplate) ) {
    my $currentType = $anActiveTemplate->getType();
    unless( $currentType eq  $aTemplateType){
      mywarn "framestart: Template type is already set to \"$currentType\" " . 
        "and cannot be changed to \"$aTemplateType\"!\n";
    }
  } elsif (defined $aTemplateType) {
    setActiveTemplateByType($aTemplateType);
  }
  #Now the active template must be there !!!!!
  $anActiveTemplate = getActiveTemplate();
  if (isMobileInfraTemplate($anActiveTemplate)) {
    mobileInfraFramestart($_[0],$_[1]);
  } elsif ( isSimpleWorkstationTemplate($anActiveTemplate)) {
    simpleWorkstationFramestart($_[0],$_[1]);
  }
  return 1;
}
#framestart




sub mobileInfraFramestart{
  my $configref = $_[1];
  my %config = %$configref;
  my $aArchiveName =$config{'archive_name'};
  my $aBackupName =$config{'backup_name'};
  my $aTgzHandler =$config{'tgz_handle'};
  my $anActiveTemplate = getActiveTemplate();

  if ( defined $aArchiveName) {
    $anActiveTemplate->setArchive($aArchiveName);
  }
  if ( defined $aBackupName) {
    $anActiveTemplate->setBackup($aBackupName);
  }
  if ( defined $setup_opt_dbname) {
    $anActiveTemplate->setDBName($setup_opt_dbname);
  }

  my ($indep_data,$indep_prog)=readIndepPath(); # try to get independent paths from earlier installation

  if (defined $indep_prog and $indep_prog =~ /\S/) {
    if ( $^O =~ /mswin/i) {
      checkIndepProgWin($sloppy,$anActiveTemplate,$indep_prog);
    } else {
      checkIndepProgUx($sloppy,$anActiveTemplate,$indep_prog);
    }
  } else {
    $SAPDB::Install::StartInstall::opt_indep_prog =
      SAPDB::Install::Templates::Base::catfile( $anActiveTemplate->getInstRootPath() ,  "programs");
    $indep_prog = $SAPDB::Install::StartInstall::opt_indep_prog;
  }

  my $dbmcliexe ="dbmcli";
  my $xserverexe ="x_server";
  if (  $^O =~ /mswin/i) {
    $dbmcliexe = SAPDB::Install::Templates::Base::catfile($indep_prog,"pgm",$dbmcliexe);
  } else  {
    $dbmcliexe = SAPDB::Install::Templates::Base::catfile($indep_prog,"bin",$dbmcliexe);
  }
  my $xserverexe =SAPDB::Install::Templates::Base::catfile($indep_prog,"bin","x_server");
  $anActiveTemplate->setDbmcliPath($dbmcliexe);
  $anActiveTemplate->setXServerPath($xserverexe);




  
  if (defined $indep_data and $indep_data =~ /\S/) {
  } else {
    my $newIndepData = 
      SAPDB::Install::Templates::Base::catfile( $anActiveTemplate->getInstRootPath() ,  "data");
    $SAPDB::Install::StartInstall::opt_indep_data = $newIndepData;
  }
  $SAPDB::Install::StartInstall::opt_depend= $anActiveTemplate->getDependendPath();
}
#mobileInfraFramestart

sub checkIndepProgUx{
}
sub checkIndepProgWin{
  my ($sloppy,$anActiveTemplate,$indep_prog) = @_;
  my $aFilename = SAPDB::Install::Templates::Base::basename($indep_prog);
  my $aDirname = SAPDB::Install::Templates::Base::dirname($indep_prog);
  unless(defined $sloppy){
    unless( $aFilename =~ /^indep_prog$/i or $aFilename =~ /^programs$/i){
      kackAb("ERROR: Independend Programs path \"$indep_prog\" not supported for this Profile!");
    }
    unless ($aDirname =~ /^\s*([a-z]:)?[\\\/]s(ap)?db[\\\/]?\s*$/i){
      kackAb("ERROR: Independend Programs path \"$indep_prog\" not supported for this Profile!");
    }
  }
  if ( $aFilename =~ /^indep_prog$/i or $aFilename =~ /^programs$/i) {
    if ($aDirname =~ /\S\s\S/) {
      mywarn "WARNING: I accepted independend programs path that does not follow filename conventions:\n\"$indep_prog\"\n";
    } else {
      my $root = $aDirname;
      $anActiveTemplate->setInstRootPath($root); 
    }
  } else {
    mywarn "WARNING: I accepted independend programs path that does not follow filename conventions:\n\"$indep_prog\"\n";
  }
}





sub simpleWorkstationFramestart{
  my $configref = $_[1];
  my %config = %$configref;
  my $anActiveTemplate = getActiveTemplate();
  if ( defined $setup_opt_dbname) {
    $anActiveTemplate->setDBName($setup_opt_dbname);
  }
  my ($indep_data,$indep_prog)=readIndepPath(); # try to get independent paths from earlier installation
  #What will we do here?
  if (defined $indep_prog and $indep_prog =~ /\S/) {
  }
}
#simpleWorkstationFramestart







sub unregister{ return 1; }
sub preuninstall{ return 1; }
sub postuninstall{ return 1; }
sub verify{ return 1; }







sub preinstall{
  #  my ($pathlistref, $configref) = @_;
  #  my ($pathspecref, %config) = (@$pathlistref[0],%$configref);
  #  my %pathspec = %$pathspecref;
  return 1;
}                               #preinstall

#Ab hier dependend path gesetzt.
sub postprepare{
  my $pathlistref = $_[0];
  my $pathspecref = @$pathlistref[0];
  my %pathspec = %$pathspecref;
  my $dependend = $pathspec{'value'};
  my $anActiveTemplate = getActiveTemplate();
  if (defined $anActiveTemplate && defined $dependend) {
    $anActiveTemplate->setDependendPath($dependend);
  }
  return 1;
}                               #postprepare

sub postinstall($$){
}                               #postinstall






#registrierung z.B. in Windows-Registry
#+sub(HashRef:paths|HashRef:config):void:Called from SDBINST during package installation
sub register($$){
  return 1;
}
#register

sub framestop($$){
  my $pathlistref = $_[0];
  my $configref = $_[1];
  my $anActiveTemplate = getActiveTemplate();
  $anActiveTemplate->writeConfig(\&logger);
  my $result = $anActiveTemplate->createInstance();
  unless(defined $result){
    kackAb("Error creating database instance!");
  }
  return $result;
}
#framestop



sub fsinfo_templates{
  my ($hFileSystemInfoRef) =@_;
  my $activeTemplate = getActiveTemplate();
  if (isMobileInfraTemplate($activeTemplate)) {
    my %hFileSystemInfo = %$hFileSystemInfoRef;
    foreach my $fsname (keys(%hFileSystemInfo)) {
      my $needed = ${%{$hFileSystemInfo{$fsname}}}{'NEEDED'};
      my $available = ${%{$hFileSystemInfo{$fsname}}}{'AVAIL'};
      # ${%{$hFileSystemInfo{$fsname}}}{'USED'}=9999;
      my $used = ${%{$hFileSystemInfo{$fsname}}}{'USED'};


      my @volumes =  $activeTemplate->getDBVolumes();
      foreach my $volume (@volumes) {
        my $avolumeKind =  @$volume[1];
        my $avolumePath =  @$volume[2];
        my $avolumeType =  @$volume[3];
        my $avolumeSize =  @$volume[4];
        #
        my $avolumeSizeInKb = ($avolumeSize+1) / 8; 
        #my $fsName = GetFSName($avolumePath);

        if ("F" eq $avolumeType) {
          #Strip fsName to existing
          my $lastPath ;
          my $existingPartOfVolumePath = $avolumePath;
          while ( $existingPartOfVolumePath and not -e $existingPartOfVolumePath) {
            $existingPartOfVolumePath = dirname($existingPartOfVolumePath);
            if (defined $lastPath and $lastPath eq $existingPartOfVolumePath) {
              $existingPartOfVolumePath = undef;
            }
            $lastPath=$existingPartOfVolumePath;
          }
          if (defined $existingPartOfVolumePath) {
            my $fsName = GetFSName($existingPartOfVolumePath);
            #print "Volume (full,existing,fsname) (\"$avolumePath\",\"$existingPartOfVolumePath\",$fsName)\n" . 
            #      "       (needed,available,used): ($needed,$available,$used,)\n";
          } else {
            mywarn("WARNING: Cannot find Filesystem for path \"$avolumePath\"\n" . 
                   "Is this a valid path?\n");
          }
          #else
        }
        #if
      }
      #foreach
    }
    #foreach
  }
  #if
}
#fsinfo_templates()

#+ Anwendung beenden
sub kackAb(){
  diesoft $_[0];	
}                               #kackAb



1;










# Local Variables: ***
# compile-command:  "c:\\Sapdevelop\\V75\\c_00\\develop\\sys\\wrk\\fast\\install56\\sdbrun.exe --tree c:\\SapDevelop\\V75\\c_00\\develop\\sys\\src\\install56\\perl  --main SAPDB::Install::Templates::Test::main" ***
# End: ***
