#!/usr/bin/perl
#
# $Header: //sapdb/V75/c_00/develop/sys/src/install/perl/SAPDB/Install/Instance/Check/Params.pm#3 $
# $DateTime: 2003/12/11 13:40:27 $
# $Change: 59245 $
#
#    ========== licence begin  GPL
#    Copyright (c) 2005 SAP AG
#
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of the GNU General Public License
#    as published by the Free Software Foundation; either version 2
#    of the License, or (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#    ========== licence end


package SAPDB::Install::Instance::Check::Params;

sub BEGIN {
	@ISA = ('SAPDB::Install::Exporter');
	@EXPORT = ();
	my $repo = SAPDB::Install::Repository::GetCurrent ();
	my @neededPackages = (
		'Instance::Base',
		'Instance::Check::Conditions'
	);

	foreach my $package (@neededPackages) {
	  	unless (defined $repo->Eval 
		("SAPDB::Install::$package", 1.01)) {
                	print join ("\n", $repo->GetErr)."\n";
                	die;
        	}
		SAPDB::Install::Exporter::import ("SAPDB::Install::$package");
	  } 
}

push @ISA, 
	'SAPDB::Install::Instance::Base',
	'SAPDB::Install::Instance::Check::Conditions';

sub params {
	my ($self) = @_; 

	#
	# find out instance type if not already known
	#
	unless (defined $self->{'instancetype'}) {
		my $rc = $self->conditions ();
		return (-1) unless (defined $rc && $rc == 0);
	}

	$self->set_errorstate ('ERROR');

	$self->msgbegin ("checking paramfile modifications");
	$self->msg0 ("checking paramfile modifications...\n");

	$rc = $self->check_paramfile_version ();
	unless (defined $rc && $rc == 0) {
		$self->msgend ();
		return -1;
	}

	my ($parammodified, $msg) = $self->is_changed ();
	$self->msg1 ($msg."\n");

	if ($parammodified == 0) {
		$self->msgend ();
		$self->set_errorstate ('OK');
		return 0;
	}

	#
	# liveCache needs target version
	# because you cannot restart without lost of migration backup
	# during migration form 7.2.05 to 7.4.xx
	#
	if ($self->{'instance_type'} =~ /LVC/) {
		unless (defined $self->{'target_release'}) {
			$self->msg1 ("target version is needed for ".
			             "check of liveCache parameters\n");
			return -1;
		}
	}

	my @changedparams = $self->get_changed ();

	# some early 7.2.05 and 7.3.00 kernel mark
	# paramfile as modified after changing form
	# cold to offline by setting _CRASH_SEM from '1' to '0'
	if ($#changedparams == 2) {
		my ($key, $old, $new) = @changedparams;
		if ($key eq '_CRASH_SEM' && $old == 1 && $new == 0) {
			$self->msg1 (
			"broken paramfile handling detected, ".
			"pobably early version of 7.2.05 or 7.3.00\n"
			);
			$self->msgend ();
			$self->set_errorstate ('OK');
			return 0;
		}
	}

	$self->print_chg (@changedparams);
	$self->print_verificationdate ();
	$self->print_hst ($self->dateofchange ($parammodified));

	# check params in paramsession 
	# to find out if param_checkall we fail during next restart
	$self->check_paramfile_after_modifications ();

	$self->msgend ();
	return (-1);
}

sub check_paramfile_version {
	my ($self) = @_;
	my $dbm = $self->{dbm};

	$self->msg1 ("check version of paramfile\n");

	my $kernel_version  = $dbm->param_directget ('KERNELVERSION');
	unless (defined $kernel_version) {
		$self->msg1 ($dbm->lastdialog ());
		$self->set_errorstate ('ERROR');
		return undef;
	}

	$kernel_version =~ s/^\D*//;

	my ($param_major, $param_minor, $param_corr, $param_build) =
	($kernel_version =~ /(\d+)\.(\d+)\.(\d+)\s+\w+\s+(\d+)/);

	my ($start_major, $start_minor, $start_corr, $start_build) =
	($self->{'starting_release'} =~ 
	/(\d+)\.(\d+)\.(\d+)\s+\w+\s+(\d+)/);

	#
	# print "start: ".$start_major." ".$start_minor." ".
	#                $start_corr." ".$start_build."\n";
	#
	# print "param: ".$param_major." ".$param_minor." ".
	#                $param_corr." ".$param_build."\n";
	#

	$kernel_version =~ s/\s+BUILD\s+/ Build  /;

	unless ($param_major == $start_major && $param_minor == $start_minor) {
		$self->msg1	(
		"version of paramfile differs from version of kernel ".
		"in release level\n");
		$self->msg1 ("kernel version   : ".$self->{'starting_release'}."\n");
		$self->msg1 ("paramfile version: ".$kernel_version."\n");
		$self->set_errorstate ('ERROR');
		return undef;
	}

	unless ($param_corr == $start_corr && $param_build == $start_build) {
		$self->msg1	("version from paramfile differs version from kernel\n");
		$self->msg1	("in correction or build level\n");
		$self->msg1 ("kernel version   : ".$self->{'starting_release'}."\n");
		$self->msg1 ("paramfile version: ".$kernel_version."\n");

		# check params in paramsession 
		# to find out if param_checkall we fail during next restart
		$self->check_paramfile_after_modifications ();
		return undef;
	}

	$self->msg1 ("kernel version   : ".$self->{'starting_release'}."\n");
	return 0;
}

sub check_paramfile_after_modifications {
	my ($self, $parammodified) = @_;
	my $dbm = $self->{dbm};
	my $rc;

	$self->set_errorstate ('ERROR');
	$self->msg1 ("paramfile needs checking...\n");

	# get a param session
	$rc = $dbm->param_startsession ();
	unless (defined $rc) {
		$self->msg1 ($dbm->lastdialog ());
		return -1;
	}

	$rc = $dbm->param_checkall ();
	unless (defined $rc && $rc eq 'OK') {
		$self->msg1 ("checking of paramfile failed:\n");
		$self->msg1 ("\n");
		$self->msg1 ($dbm->lastdialog ());
		$dbm->param_abortsession ();
		return -1;
	}

	$self->msg1 
	("checking paramfile without modifications was sucessful\n");

	# drop param session without modifications
	$rc = $dbm->param_abortsession ();
	unless (defined $rc) {
		$self->msg1 ($dbm->lastdialog ());
		return -1;
	}

	#
	# return errorstate == 'ERROR'
	# when we are using a consistent backup for migration
	# and we cannot restart with out losing consistence of the backup
	#
	unless ($self->get_migrationstrategy () =~ 
	/^COMPATIBLE_LOG|^CONSISTENT_DATA|^EXTERNAL_LIVECACHE_CONSISTENCE/) {
		return -1;
	}
	
	$self->set_errorstate ('OFFLINE_RESTART_REQUIRED');
	return 0;
}

sub dateofchange {
	my ($self, $parammodified) = @_;
	my $dbm = $self->{dbm};
 
	# do we know how to get the param history?
	my ($can_param_gethistory) = $dbm->help ('param_gethistory');
	unless ($dbm->lastmsg () =~ /^OK/) {
		$self->msg1 ($dbm->lastdialog ());
		$self->set_errorstate ('ERROR');
		return undef;
	}

	$can_param_gethistory = (defined $can_param_gethistory) ? 1 : 0;
	unless ($can_param_gethistory == 1) {
		$self->msg1 (
		"dbmsrv does not know method param_gethistory\n");
		return undef;
	}

	# ok, we can do that get the history
	my %paramhistory = $dbm->param_gethistory ();	
	unless (defined %paramhistory) {
		$self->msg1 ($dbm->lastdialog ());
		$self->set_errorstate ('ERROR');
		return undef;
	}

	# transform the array
	# DATE is now yyyymmddhhmmss
	# this is usefull as a key for a hash
	for (my $i = 0; $i <= $#{$paramhistory{'DATE'}}; $i++) {
		${$paramhistory{'DATE'}}[$i] *= 1000000;
		${$paramhistory{'DATE'}}[$i] += ${$paramhistory{'TIME'}}[$i];
	}

	# build new hash,
	# where key is the date, 
	# and value is a list of name, oldvalue, newvalue, ...
	my %paramhistorybydate;
	for (my $i = 0; $i <= $#{$paramhistory{'DATE'}}; $i++) {
		my $date = ${$paramhistory{'DATE'}}[$i];

		unless (defined $paramhistorybydate{$date}) {
			$paramhistorybydate{$date} = ();
		}

		push @{$paramhistorybydate{$date}}, 
			${$paramhistory{'NAME'}}[$i],
			${$paramhistory{'OLDVALUE'}}[$i],
			${$paramhistory{'NEWVALUE'}}[$i];
	}

	# sort dates, newest dates on top
	my @paramhistorydates = sort {$b <=> $a} keys (%paramhistorybydate);

	# when $parammodified in defined
	# want only the newest enties since lastknowngood paramfile
	splice (@paramhistorydates, $parammodified - $#paramhistorydates - 1)
		if (defined $parammodified && 
		    $#paramhistorydates + 1 > $parammodified);

	# delete other enties from hash
	foreach my $date (keys (%paramhistorybydate)) {
		my $keep = 0;
		foreach $wanted (@paramhistorydates) {
			if ($date == $wanted) {
				$keep = 1;
				last;
			}
		}
		delete ($paramhistorybydate{$date}) if ($keep == 0);
	}
	return (\%paramhistorybydate);
}

sub is_changed {
	my ($self) = @_;
	my $dbm = $self->{dbm};

	# find out if param file was changed
	my $modified = $dbm->param_directget ('__PARAM_CHANGED___');
	unless (defined $modified) {
		$self->msg1 ($dbm->lastdialog ());
		$self->set_errorstate ('ERROR');
		return undef;
	}
	
	return ($modified, 
		$modified == 0 ? "paramfile is clean" :
		$modified == 1 ? "paramfile has been modified 1 time" :
		"paramfile has been modified ".$modified." times");
}

sub print_verificationdate {
	my ($self) = @_;
	my $dbm = $self->{dbm};

	my $verifiation_date = $dbm->param_directget ('__PARAM_VERIFIED__');
	return unless (defined $verifiation_date);

	$self->msg1 ("last successful verification of paramfile was on ".
	             $verifiation_date."\n");
	$self->msg1 ("\n");
}

sub get_changed {
	my ($self) = @_;
	my $dbm = $self->{dbm};

	# get a param session
	my $rc = $dbm->param_startsession ();
	unless (defined $rc) {
		$self->msg1 ($dbm->lastdialog ());
		$self->set_errorstate ('ERROR');
		return undef;
	}

	# find out changed values
	my $paramall = $dbm->param_directgetall ();	
	unless (defined $paramall) {
		$self->msg1 ($dbm->lastdialog ());
		$dbm->param_abortsession ();
		$self->set_errorstate ('ERROR');
		return undef;
	}
	
	my @paramchg = ();
	my ($key, $value);
	while (($key, $val) = each (%$paramall)) {
		# skip CONTROLPASSWORD, there are always differences
		next if ($key eq 'CONTROLPASSWORD');

		# skip __PARAM_CHANGED___, we know this already
		next if ($key eq '__PARAM_CHANGED___');
	
		my %paramfull = $dbm->param_getfull ($key);
		unless (defined %paramfull) {
			$self->msg1 ($dbm->lastdialog ());
			$dbm->param_abortsession ();
			$self->set_errorstate ('ERROR');
			return undef;
		}

		next if ($paramfull{'VALUE'} eq $paramfull{'LASTKNOWNGOOD'});

		# skip _DIAG_SEM '1' => '0'
		# it is always changed form '1' to '0'
		# when kernel RTE was normaly stoped
		next if ($key eq '_DIAG_SEM' && $paramfull{'VALUE'} == 0);
	
		push @paramchg,
			$key, $paramfull{'LASTKNOWNGOOD'}, $paramfull{'VALUE'};
	}

	# drop param session without modifications
	my $rc = $dbm->param_abortsession ();
	unless (defined $rc) {
		$self->msg1 ($dbm->lastdialog ());
		$self->set_errorstate ('ERROR');
		return undef;
	}
	return @paramchg;
}

sub print_hst {
	my $self = shift;

	return unless (defined $_[0]);

	my %history = %{$_[0]};
	my @dates = sort {$b <=> $a} keys (%history);

	if ($#dates == -1) {
		$self->msg1 ("according to your param history, ".
		       "no changes were made\n");
		return;
	} 

	$self->msg1 ("according to your param history, ".
	      		 "the following changes were made:\n");

	foreach my $date (@dates) {
		$self->msg1 ("".formathstdate ($date)."\n");
		my @paramchg = formatchg (@{$history{$date}});
		foreach $_ (@paramchg) {
			$self->msg1 ($_."\n");
		}
	}
	$self->msg1 ("\n");
}

sub formathstdate {
	my ($year, $month, $day, $hour, $min, $sec) =
		($_[0] =~ /^(\d\d\d\d)(\d\d)(\d\d)(\d\d)(\d\d)(\d\d)$/);
	
	return ($year."-".$month."-".$day." ".$hour.":".$min.":".$sec)
}

sub print_chg {
	my $self = shift;

	return unless (defined $_[0]);

	if ($#_ == -1) {
		$self->msg1 ("no changed parameters\n");
		$self->msg1 ("\n");
		return;
	}

	$self->msg1 ("changed parameters\n");
	my @out = formatchg (@_);
	foreach $_ (@out) {	
		$self->msg1 ($_."\n");
	}
	$self->msg1 ("\n");
}

sub formatchg {
	my @in = @_;
	my @out = ();
	my @max = getmaxlen (@in);

	for (my $i = 0; $i <= $#in; $i += 3) {
		my @line = ($in[$i], $in[$i + 1], $in[$i + 2]);
		my @spaces = (0, 0);
		
		$spaces[0] = $max[0] - length ($line[0]);
		$spaces[1] = $max[1] - length ($line[1]);
		push @out, '  '.
		$line[0].(' ' x $spaces[0]).' '.
		$line[1].(' ' x $spaces[1]).' => '.
		$line[2];
	}	
	return @out;
}

sub getmaxlen {
	my @in = @_;
	my @max = (0, 0, 0);

	for (my $i = 0; $i <= $#in; $i += 3) {
		my @line = ($in[$i], $in[$i + 1], $in[$i + 2]);
		my @len = (length ($line[0]), 
		           length ($line[1]), 
		           length ($line[2]));
		$max[0] = ($len[0] > $max[0]) ? $len[0] : $max[0];
		$max[1] = ($len[1] > $max[1]) ? $len[1] : $max[1];
		$max[2] = ($len[2] > $max[2]) ? $len[2] : $max[2];
	}
	return @max;
}

1;
