/*****************************************************************************/
/*!
  @file         SQLTab_SysUpdStatWanted.hpp
  @author       MartinKi
  @ingroup

  @brief        Header file for class SQLTab_SysUpdStatWanted.

\if EMIT_LICENCE
    ========== licence begin  GPL
    Copyright (c) 2003-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end

\endif
*/
/*****************************************************************************/

/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "SQLManager/SQLMan_Context.hpp"
#include "SQLManager/Tables/SQLTab_SysUpdStatWanted.hpp"
#include "SQLManager/ErrorHandling/SQLManErr_Interface.hpp"
#include "SAPDBCommon/Messages/SDBMsg_SQLMan.h"

#include "SQLManager/Catalog/Catalog_TableDescriptor.hpp"
#include "SQLManager/Catalog/Catalog_ColumnDescriptor.hpp"
#include "SQLManager/SQLMan_SQLStatementContext.hpp"
#include "SQLManager/SQLMan_ChangedUserScope.hpp"
#include "SQLManager/KernelSQL/KSQL_PreparedStatementScope.hpp"
#include "SQLManager/KernelSQL/KSQL_PreparedStatement.hpp"
#include "SQLManager/KernelSQL/KSQL_Connection.hpp"
#include "SQLManager/SQLMan_Tracing.hpp"

#include "SAPDBCommon/Fields/SAPDBFields_AttributedField.hpp"
#include "SAPDBCommon/SAPDB_Types.hpp"


#include "hbd01.h"
#include "hbd02.h"
#include "hak28.h"
#include "hak061.h" // a061identifier_len
#include "ggg00.h"
#include "hgg01_3.h" // g01unicode

/*===========================================================================*
 *  CODE                                                                     *
 *===========================================================================*/

/// SQL return code for cancelled command
const SAPDB_Int cCommandCancelled = -102;

struct RecordLayout {
    SAPDB_Int2 recLen;
    SAPDB_Int2 keyLen;
    SAPDB_Int2 varColCount;
    SAPDB_Int2 varColOffset;
    SAPDB_Byte defByteKey;
    SAPDB_Byte schemaName[ sizeof(SQLMan_Identifier) ];
    SAPDB_Byte defByteKey1;
    SAPDB_Byte ownerName[ sizeof(SQLMan_Identifier) ];
    SAPDB_Byte defByteKey2;
    SAPDB_Byte tableName[ sizeof(SQLMan_Identifier) ];
    SAPDB_Byte defByteKey3;
    SAPDB_Byte columnName[ sizeof(SQLMan_Identifier) ];
};

/**************************************************************************/

SQLTab_SysUpdStatWanted::SQLTab_SysUpdStatWanted(SQLMan_Context& context)
    : m_context( context )
{}

/**************************************************************************/

// SQLTab_SysUpdStatWanted::Iterator
// SQLTab_SysUpdStatWanted::Begin() const
// {
//     return Iterator(m_context.TransContext());
// }

// /**************************************************************************/

// SQLTab_SysUpdStatWanted::Iterator
// SQLTab_SysUpdStatWanted::End() const
// {
//     return Iterator(m_context.TransContext(), true);
// }

// /**************************************************************************/

// SQLTab_SysUpdStatWanted::Iterator::Iterator(
//     SQLMan_TransContext& transContext)
//     : m_trans( transContext )
// {
//     m_curKey.keyLen_gg00() = 0;
// }

// /**************************************************************************/

// SQLTab_SysUpdStatWanted::Iterator::Iterator(
//     SQLMan_TransContext& transContext, bool)
//     : m_trans( transContext )
// {
//     m_curKey.keyLen_gg00() = MAX_KEYLEN_GG00;
//     m_curKey.keyVal_gg00() = b01fullkey;
// }

// /**************************************************************************/

// SQLTab_SysUpdStatWanted::Iterator::Iterator(const Iterator& it)
//     : m_trans( it.m_trans )
// {
//     if ( &it == this ) {
//         return;
//     }

//     m_curKey = it.m_curKey;
// }

// /**************************************************************************/

// SAPDB_Bool
// SQLTab_SysUpdStatWanted::Iterator::GetKey(tgg00_Lkey& key) const
// {
//     if ( (m_curKey.keyLen_gg00() != 0)
//          && (! ((m_curKey.keyLen_gg00() != MAX_KEYLEN_GG00)
//                 && (m_curKey.keyVal_gg00() != b01fullkey))) ) {
//         key = m_curKey;
//         return true;
//     }
//     return false;
// }

// /**************************************************************************/

// SAPDB_Bool
// SQLTab_SysUpdStatWanted::Iterator::Next()
// {
//     tgg00_Rec recBuf;

//     if ( memcmp (
//              a28updstatwanted_tree.fileName_gg00(),
//              cgg_zero_fn,
//              sizeof( a28updstatwanted_tree.fileName_gg00() ) ) == 0 ) {
//         return false;
//     }

//     // retrieve record
//     const pasbool c_inclusive = true;
//     b02next_record(
//         m_trans,
//         a28updstatwanted_tree,
//         m_curKey,
//         c_inclusive,
//         recBuf );

//     if ( (m_trans.trError_gg00 == e_no_next_record)
//          || (m_trans.trError_gg00 == e_key_not_found) ) {
//         m_trans.trError_gg00 = e_ok;
//         return false;
//     }

//     if ( m_trans.trError_gg00 != e_ok ) {
//         // FIXME: set errMsg
//         m_trans.trError_gg00 = e_ok;
//         return false;
//     }

//     m_trans.trError_gg00 = e_ok;

//     // update key
//     SAPDB_MemCopyNoCheck(
//         &m_curKey, recBuf.recBuf_gg00(),
//         recBuf.recKeyLen_gg00() + cgg_rec_key_offset );

//     return true;
// }

// /**************************************************************************/

// SQLTab_SysUpdStatWanted::Iterator&
// SQLTab_SysUpdStatWanted::Iterator::operator++()
// {
//     this->Next();
//     return *this;
// }

/**************************************************************************/
/**************************************************************************/

SQLTab_SysUpdStatWanted::RequestColumnIterator
SQLTab_SysUpdStatWanted::GetRequestColumnIterator(
    const Catalog_TableDescriptor& tableDesc) const
{
    SAPDBTRACE_METHOD_DEBUG (
        "SQLTab_SysUpdStatWanted::GetRequestColumnIterator", SQLMan_Trace, 5);

    return RequestColumnIterator(
        m_context.TransContext(), tableDesc );
}

/**************************************************************************/

SQLTab_SysUpdStatWanted::RequestColumnIterator::RequestColumnIterator(
    SQLMan_TransContext&     transContext,
    const Catalog_TableDescriptor& tableDesc)
    : m_trans( transContext ),
      m_tableDesc( &tableDesc ),
      m_firstCall( true )
{
    SAPDBTRACE_METHOD_DEBUG (
        "SQLTab_SysUpdStatWanted::RequestColumnIterator::RequestColumnIterator",
        SQLMan_Trace, 5);

    if ( ! tableDesc ) {
        m_tableDesc = 0;
        return;
    }

    SQLTab_SysUpdStatWanted::BuildKey(
        m_tableDesc->GetSchemaName(),
        m_tableDesc->GetOwnerName(),
        m_tableDesc->GetTableName(),
        SQLMan_Identifier(),
        m_curKey);
}

/**************************************************************************/

SQLTab_SysUpdStatWanted::RequestColumnIterator::RequestColumnIterator(
    const RequestColumnIterator& it)
    : m_trans( it.m_trans )
{
    SAPDBTRACE_METHOD_DEBUG (
        "SQLTab_SysUpdStatWanted::RequestColumnIterator::copy-const",
        SQLMan_Trace, 5);

    if ( &it == this ) {
        return;
    }

    m_curKey    = it.m_curKey;
    m_tableDesc = it.m_tableDesc;
}

/**************************************************************************/

SAPDB_Bool
SQLTab_SysUpdStatWanted::RequestColumnIterator::Next(
    SQLMan_Identifier& columnName)
{
    SAPDBTRACE_METHOD_DEBUG (
        "SQLTab_SysUpdStatWanted::RequestColumnIterator::Next",
        SQLMan_Trace, 5);

    tgg00_Rec recBuf;

    if ( ( memcmp (
               a28updstatwanted_tree.fileName_gg00(),
               cgg_zero_fn,
               sizeof( a28updstatwanted_tree.fileName_gg00() ) ) == 0 )
         || ( m_tableDesc == 0 ) ) {
        return false;
    }

    // retrieve record, check for exact key during first call

    // do not use a28updstatwanted_tree directly because we do not
    // want its root page set (and get e_old_fileversion later on)
    SAPDBERR_ASSERT_STATE( a28updstatwanted_tree.fileRoot_gg00() == NIL_PAGE_NO_GG00 );
    tgg00_FileId htreeId = a28updstatwanted_tree;

    b02next_record(
        m_trans,
        htreeId,
        m_curKey,
        m_firstCall,
        recBuf );

    if ( m_firstCall && (m_trans.trError_gg00 == e_key_not_found) ) {
        m_trans.trError_gg00 = e_ok;
    }
    m_firstCall = false;

    if ( (m_trans.trError_gg00 == e_no_next_record)
         || (m_trans.trError_gg00 == e_key_not_found) ) {
        m_trans.trError_gg00 = e_ok;
        return false;
    }

    if ( m_trans.trError_gg00 != e_ok ) {
        // FIXME: set errMsg
        m_trans.trError_gg00 = e_ok;
        return false;
    }

    m_trans.trError_gg00 = e_ok;

    // update key
    SAPDB_MemCopyNoCheck(
        &m_curKey, recBuf.recBuf_gg00(),
        recBuf.recKeyLen_gg00() + cgg_rec_key_offset );

    SQLMan_Identifier schemaName;
    SQLMan_Identifier ownerName;
    SQLMan_Identifier tableName;
    SQLTab_SysUpdStatWanted::ParseRecord(
        recBuf,
        schemaName,
        ownerName,
        tableName,
        columnName);

    if ( ( schemaName != m_tableDesc->GetSchemaName() )
         || ( ownerName != m_tableDesc->GetOwnerName() )
         || ( tableName != m_tableDesc->GetTableName() ) ) {
        return false;
    }

    return true;
}

/**************************************************************************/

SAPDB_Bool
SQLTab_SysUpdStatWanted::GetRecord(
    tgg00_TransContext& trans,
    tgg00_Lkey&        key,
    SQLMan_Identifier& schemaName,
    SQLMan_Identifier& ownerName,
    SQLMan_Identifier& tableName,
    SQLMan_Identifier& columnName)
{
    SAPDBTRACE_ROUTINE_DEBUG (
        "SQLTab_SysUpdStatWanted::GetRecord", SQLMan_Trace, 5 );

    tgg00_Rec recBuf;

    if ( memcmp (
             a28updstatwanted_tree.fileName_gg00(),
             cgg_zero_fn,
             sizeof( a28updstatwanted_tree.fileName_gg00() ) ) == 0 ) {
        return false;
    }

    // retrieve record
    SAPDBERR_ASSERT_STATE( a28updstatwanted_tree.fileRoot_gg00() == NIL_PAGE_NO_GG00 );
    tgg00_FileId htreeId = a28updstatwanted_tree;
    const pasbool c_inclusive = true;
    b02next_record(
        trans,
        htreeId,
        key,
        c_inclusive,
        recBuf );

    if ( trans.trError_gg00 != e_ok ) {
        trans.trError_gg00 = e_ok;
        return false;
    }

    SQLTab_SysUpdStatWanted::ParseRecord(
        recBuf, schemaName, ownerName, tableName, columnName );

    return true;
}

/**************************************************************************/

void SQLTab_SysUpdStatWanted::BuildKey(
    const SQLMan_Identifier& schemaName,
    const SQLMan_Identifier& ownerName,
    const SQLMan_Identifier& tableName,
    const SQLMan_Identifier& columnName,
    tgg00_Lkey&              key)
{
    SAPDBTRACE_ROUTINE_DEBUG (
        "SQLTab_SysUpdStatWanted::BuildKey", SQLMan_Trace, 5 );

    SAPDB_Byte defByte;
    if ( g01unicode ) {
        defByte = csp_unicode_def_byte;
    } else {
        defByte = csp_ascii_blank;
    }

    const SAPDB_Int colNameOffset =
        cgg_rec_key_offset + 3 * sizeof( SQLMan_Identifier ) + 4;

    RecordLayout* rec = reinterpret_cast<RecordLayout*>( &key );

    rec->defByteKey  = defByte;
    rec->defByteKey1 = defByte;
    rec->defByteKey2 = defByte;
    rec->defByteKey3 = defByte;

    SAPDB_MemCopyNoCheck(
        rec->schemaName, &schemaName, sizeof( schemaName ) );

    SAPDB_MemCopyNoCheck(
        rec->ownerName, &ownerName, sizeof( ownerName ) );

    SAPDB_MemCopyNoCheck(
        rec->tableName, &tableName, sizeof( tableName ) );

    SAPDB_MemCopyNoCheck(
        rec->columnName, &columnName, columnName.GetLength() );

    rec->keyLen = static_cast<SAPDB_Int2>(
        3 * sizeof( SQLMan_Identifier ) + 4 + columnName.GetLength() );

    rec->recLen = cgg_rec_key_offset + rec->keyLen;
    rec->varColCount  = 0;
    rec->varColOffset = 0;
}

/**************************************************************************/

void SQLTab_SysUpdStatWanted::ParseRecord(
    const tgg00_Rec&    recBuf,
    SQLMan_Identifier&  schemaName,
    SQLMan_Identifier&  ownerName,
    SQLMan_Identifier&  tableName,
    SQLMan_Identifier&  columnName)
{
    SAPDBTRACE_ROUTINE_DEBUG (
        "SQLTab_SysUpdStatWanted::ParseRecord", SQLMan_Trace, 5 );

    const SAPDB_Int colNameOffset =
        cgg_rec_key_offset + 3 * sizeof( SQLMan_Identifier ) + 4;

    const RecordLayout* rec =
        reinterpret_cast<const RecordLayout*>( &recBuf );

    SAPDB_MemCopyNoCheck(
        &schemaName, rec->schemaName, sizeof( schemaName ) );

    SAPDB_MemCopyNoCheck(
        &ownerName, rec->ownerName, sizeof( ownerName ) );

    SAPDB_MemCopyNoCheck(
        &tableName, rec->tableName, sizeof( tableName ) );

    columnName.SetBlank();
    SAPDB_MemCopyNoCheck(
        &columnName, rec->columnName, rec->recLen - colNameOffset );
}

/**************************************************************************/

SQLTab_SysUpdStatWanted::InsertRc
SQLTab_SysUpdStatWanted::InsertRow(
    const Catalog_TableDescriptor&  tableDesc,
    const Catalog_ColumnDescriptor& columnDesc,
    Msg_List&                       errMsg)
{
    if ( tableDesc.IsArchiveTable() ) {
        return irNoStatisticsPossible;
    }

    if ( columnDesc.IsLongDataType() ) {
        return irNoStatisticsPossible;
    }

    SQLMan_Identifier schemaName( tableDesc.GetSchemaName() );
    SQLMan_Identifier ownerName( tableDesc.GetOwnerName() );
    SQLMan_Identifier tableName( tableDesc.GetTableName() );
    SQLMan_Identifier columnName( columnDesc.GetColumnName() );

    if ( columnDesc.IsSysKey() ) {
        // just enter request for table itself
        columnName.SetBlank();
    }

    return
        this->InsertRow( schemaName, ownerName, tableName, columnName, errMsg );
}

/**************************************************************************/

SQLTab_SysUpdStatWanted::InsertRc
SQLTab_SysUpdStatWanted::InsertRow(
    const SQLMan_Identifier&  schemaName,
    const SQLMan_Identifier&  ownerName,
    const SQLMan_Identifier&  tableName,
    const SQLMan_Identifier&  columnName,
    Msg_List&                 errMsg)
{
    SAPDBTRACE_METHOD_DEBUG (
        "SQLTab_SysUpdStatWanted::InsertRow", SQLMan_Trace, 5 );

    SQLMan_SQLStatementContext newStatementScope( m_context );

    if ( ! m_context.IsOk() )
    {
        return irFailed;
    }

    KSQL_Connection con;
    if ( ! con.isConnected() ) {
        errMsg = Msg_List(
            Msg_List::Error,
            SDBMSG_SQLMAN_UPDSTATWANTED_CREATESTATEMENT_FAILED,
            Msg_Arg( SDBMSGTAG_SQLMAN_UPDSTATWANTED_CREATESTATEMENT_FAILED__RETURNCODE,
                     SQLManErr_Interface::GetInstance().GetErrorText(
                         m_context.ReturnCode() ) )
            );
        return irSystemError;
    }

    m_context.SetInDDLTrigger();

    // statement prepare
    KSQL_PreparedStatement stmt = con.createPreparedStatement();

    const int c_maxStmtLen = 500;
    char stmtText[ c_maxStmtLen+1 ] =
        "INSERT SYSDBA.SYSUPDSTATWANTED"
        " (schemaname, owner, tablename, columnname)"
        " VALUES"
        " (?, ?, ?, ?)";

    if ( stmt.prepare( stmtText ) != 0 ) {
        SAPDBERR_ASSERT_STATE( false );
        return irSystemError;
    }

    SAPDB_Int colNo = 0;
    SAPDBFields_AttributedField curField;

    stmt.bindIdentifier4Input(
        ++colNo, const_cast<SQLMan_Identifier*>( &schemaName ) );

    stmt.bindIdentifier4Input(
        ++colNo, const_cast<SQLMan_Identifier*>( &ownerName ) );

    stmt.bindIdentifier4Input(
        ++colNo, const_cast<SQLMan_Identifier*>( &tableName ) );

    stmt.bindIdentifier4Input(
        ++colNo, const_cast<SQLMan_Identifier*>( &columnName ) );

    // statement execution
    KSQL_Statement::SQLCode sqlReturnCode = stmt.execute();
    if ( sqlReturnCode != 0 ) {
        switch ( sqlReturnCode ) {
            case -102:
                return irCancelled;
            case -20:
            case 200:
                // duplicate key can be ignored
                break;
            case 500:
                return irLockRowFailed;
            default:
                errMsg = Msg_List(
                    Msg_List::Warning,
                    SDBMSG_SQLMAN_UPDSTATWANTED_INSERT_FAILED,
                    Msg_Arg( SDBMSGTAG_SQLMAN_UPDSTATWANTED_INSERT_FAILED__ERRORTEXT,
                             SQLManErr_Interface::GetInstance().GetErrorText(
                                 sqlReturnCode ) )
                    );
                return irFailed;
        }
    }

    return irOkay;
}

/**************************************************************************/

SQLTab_SysUpdStatWanted::DeleteRc
SQLTab_SysUpdStatWanted::DeleteRow(
    const SQLMan_Identifier&  schemaName,
    const SQLMan_Identifier&  ownerName,
    const SQLMan_Identifier&  tableName,
    const SQLMan_Identifier&  columnName,
    Msg_List&                 errMsg)
{
    SAPDBTRACE_METHOD_DEBUG (
        "SQLTab_SysUpdStatWanted::DeleteRow-identifiers", SQLMan_Trace, 5 );

    const char cStmtText[] =
        "Delete SysUpdStatWanted where"
        " schemaname = ?"
        " and owner = ?"
        " and tablename = ?"
        " and columnname = ?";

    SQLMan_ChangedUserScope sysdbaScope( m_context, g01glob.sysuser_name );
    KSQL_PreparedStatementScope stmtScope( m_context.GetAllocator() );

    if ( stmtScope.Initialize( m_context, cStmtText, errMsg )
        != KSQL_PreparedStatementScope::initRcOk ) {

        errMsg.Overrule(
            Msg_List(
                Msg_List::Error,
                SDBMSG_SQLMAN_UPDSTATWANTED_INTERNAL_DELETE_STMT_FAILED )
            );
        return delRcSystemError;
    }

    SAPDB_Int colNo = 0;
    SAPDB_Int index = 0;
    bool      ok    = true;

    ok &= stmtScope.GetStatement().bindIdentifier4Input( ++colNo, &schemaName );

    ok &= stmtScope.GetStatement().bindIdentifier4Input( ++colNo, &ownerName );
    ok &= stmtScope.GetStatement().bindIdentifier4Input( ++colNo, &tableName );

    ok &= stmtScope.GetStatement().bindIdentifier4Input( ++colNo, &columnName );

    if ( ! ok ) {

        errMsg = Msg_List(
            Msg_List::Error,
            SDBMSG_SQLMAN_UPDSTATWANTED_INTERNAL_DELETE_PARAM_BIND_ERR
            );

        return delRcSystemError;
    }

    // statement execution
    KSQL_Statement::SQLCode sqlReturnCode = stmtScope.GetStatement().execute();

    if ( sqlReturnCode != 0 ) {

        switch ( sqlReturnCode ) {
            case 100:
            case -4004:
                return delRcRowNotFound;
            case cCommandCancelled:
                return delRcCancelled;
            case 500:
                return delRcLockRequestTimeout;
            default:

                errMsg.Overrule(
                    Msg_List(
                        Msg_List::Error,
                        SDBMSG_SQLMAN_UPDSTATWANTED_DELETE_FAILED,
                        Msg_Arg(SDBMSGTAG_SQLMAN_UPDSTATWANTED_DELETE_FAILED__ERRORTEXT,
                                SQLManErr_Interface::GetInstance().GetErrorText(
                                    sqlReturnCode ) ),

                        Msg_Arg(SDBMSGTAG_SQLMAN_UPDSTATWANTED_DELETE_FAILED__RETURNCODE,
                                sqlReturnCode ) )
                    );

                return delRcSystemError;
        }

    }

    return delRcOkay;
}

/**************************************************************************/
