/*!**************************************************************************

  module      : Kernel_TableStatistic.hpp

  -------------------------------------------------------------------------

  responsible : TorstenS

  special area: Kernel_TableStatistic
  description : This class is used to encapsulate statistic informations
                needed for table statistic

  last changed: 2001-01-20
  see also    :

  -------------------------------------------------------------------------

  copyright:    Copyright (c) 2001-2005 SAP AG


    ========== licence begin  GPL
    Copyright (c) 2001-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end




*****************************************************************************/


#ifndef KERNEL_TABLE_STATISTIC_HPP
#define KERNEL_TABLE_STATISTIC_HPP



/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "ggg00.h"

#include "KernelCommon/Kernel_Common.hpp"

/*===========================================================================*
 *  DEFINES                                                                  *
 *===========================================================================*/


/*===========================================================================*
 *  MACROS                                                                   *
 *===========================================================================*/


/*===========================================================================*
 *  CLASSES, STRUCTURES, TYPES, UNIONS ...                                   *
 *===========================================================================*/

/*!*****************************************************************************

   class: Kernel_TableStatistic

   description: This class is used to encapsulate statistic informations
                needed for table statistic
                 
*******************************************************************************/

class Kernel_TableStatistic
{
public:
    /* -----------------------------------------------------------------------*/
    /*! chapter: Constructors and initialization                              */
    /* -----------------------------------------------------------------------*/

    /*!----------------------------------------------------------------------
    function:     Kernel_TableStatistic()
    description:  Constructs a statistic handle to encapsulate the internal
                  tgg00_TableStatInfo for primary table statistics requests.
    arguments:    tableStatistic [in/out] table statistic structure
    return value: none
    -----------------------------------------------------------------------*/

    Kernel_TableStatistic( tgg00_TableStatInfo  &tableStatistic )
    :
    m_TableStatistic( tableStatistic ),
    m_NumberOfLeafNodes( 0 ),
    m_NumberOfIndexNodes( 0 ),
    m_TotalLeafCovering( 0 ),
    m_TotalIndexCovering( 0 ),
    m_RootCovering( 0 ),
    m_MinIndexCovering( SAPDB_MAX_UINT4 ),
    m_MaxIndexCovering( 0 ),
    m_RootLevel( LEAF_LEVEL_BD00 ),
    m_MinLeafCovering( SAPDB_MAX_UINT4 ),
    m_MaxLeafCovering( 0 ),
    m_RecordCount( 0 ),
    m_MinRecordCount( SAPDB_MAX_UINT4 ),
    m_MaxRecordCount( 0 ),
    m_TotalRecordLength( 0 ),
    m_MinRecordLength( SAPDB_MAX_UINT4 ),
    m_MaxRecordLength( 0 ),
    m_TotalKeyLength( 0 ),
    m_MinKeyLength( SAPDB_MAX_UINT4 ),
    m_MaxKeyLength( 0 ),
    m_NumberOfLongColumns( 0 ),
    m_MinLongColumnLength( SAPDB_MAX_UINT4 ),
    m_MaxLongColumnLength( 0 ),
    m_TotalLongColumnLength( 0 ),
    m_MinLongColumnPages( SAPDB_MAX_UINT4 ),
    m_MaxLongColumnPages( 0 ),
    m_TotalLongColumnPages( 0 ),
    m_NumberOfSeparators( 0 ),
    m_MinSeparatorLength( SAPDB_MAX_UINT4 ),
    m_MaxSeparatorLength( 0 ),
    m_TotalSeparatorLength( 0 )
    {}

    /* -----------------------------------------------------------------------*/
    /*! endchapter: constructors and initialization                           */
    /* -----------------------------------------------------------------------*/

    /* -----------------------------------------------------------------------*/
    /*! chapter: Manipulation                                                 */
    /* -----------------------------------------------------------------------*/

    /*!----------------------------------------------------------------------
    function:     CalculateResult()
    description:  This method is used to transfer the determined statistic
                  values into the given structure tgg00_TableStatInfo so that
                  this informations coud be used by the sql manager pascal
                  code.
    arguments:    none
    return value: none
    -----------------------------------------------------------------------*/

    void CalculateResult()
    {
        SAPDBERR_ASSERT_STATE( 0 < m_NumberOfLeafNodes  );

        if(0 >= m_NumberOfLeafNodes )
            return;

        m_TableStatistic.TabStatNodes_gg00      = m_NumberOfLeafNodes + m_NumberOfIndexNodes;
        m_TableStatistic.TabStatIndexNodes_gg00 = m_NumberOfIndexNodes;
        m_TableStatistic.TabStatLeafnodes_gg00  = m_NumberOfLeafNodes;
        m_TableStatistic.TabStatRootLevel_gg00  = m_RootLevel;

        if( LEAF_LEVEL_BD00 == m_RootLevel )
        {
            m_TableStatistic.TabStatTreeCov_gg00     =
                static_cast< SAPDB_Int4>( m_TotalLeafCovering / m_NumberOfLeafNodes );
            m_TableStatistic.TabStatIndexCov_gg00    = 0;
            m_TableStatistic.TabStatMinIndexCov_gg00 = 0;
            m_TableStatistic.TabStatMaxIndexCov_gg00 = 0;
        }
        else
        {
            m_TableStatistic.TabStatTreeCov_gg00  =
                static_cast< SAPDB_Int4>(( m_TotalLeafCovering + m_TotalIndexCovering )/
                                         (m_NumberOfLeafNodes + m_NumberOfIndexNodes ));

            m_TableStatistic.TabStatIndexCov_gg00 =
                static_cast< SAPDB_Int4>( m_TotalIndexCovering / m_NumberOfIndexNodes );

            m_TableStatistic.TabStatMinIndexCov_gg00 = m_MinIndexCovering;
            m_TableStatistic.TabStatMaxIndexCov_gg00 = m_MaxIndexCovering;

        }

        if( 0 < m_NumberOfSeparators )
        {
            m_TableStatistic.TabStatAveSepLength_gg00 =
                static_cast< SAPDB_Int2>( m_TotalSeparatorLength / m_NumberOfSeparators );
            m_TableStatistic.TabStatMinSepLength_gg00 = m_MinSeparatorLength;
            m_TableStatistic.TabStatMaxSepLength_gg00 = m_MaxSeparatorLength;
        }
        else
        {
            m_TableStatistic.TabStatAveSepLength_gg00 = 0;
            m_TableStatistic.TabStatMinSepLength_gg00 = 0;
            m_TableStatistic.TabStatMaxSepLength_gg00 = 0;
        }

        m_TableStatistic.TabStatLeafCov_gg00 =
            static_cast< SAPDB_Int4>( m_TotalLeafCovering / m_NumberOfLeafNodes );
        m_TableStatistic.TabStatRootCov_gg00 = static_cast< SAPDB_Int4>( m_RootCovering );
        m_TableStatistic.TabStatRecords_gg00 = static_cast< SAPDB_Int4>( m_RecordCount );

        if( 0 < m_RecordCount )
        {
            m_TableStatistic.TabStatMinLeafCov_gg00 = m_MinLeafCovering;
            m_TableStatistic.TabStatMaxLeafCov_gg00 = m_MaxLeafCovering;

            m_TableStatistic.TabStatMaxRecsPerPage_gg00 = m_MaxRecordCount;
            m_TableStatistic.TabStatMinRecsPerPage_gg00 = m_MinRecordCount;
            m_TableStatistic.TabStatRecsPerPage_gg00    =
                static_cast< SAPDB_Int4>( m_RecordCount / m_NumberOfLeafNodes );

            m_TableStatistic.TabStatAveRecLength_gg00   =
                static_cast<SAPDB_Int4>( m_TotalRecordLength / m_RecordCount );

            m_TableStatistic.TabStatMinRecLength_gg00   = m_MinRecordLength;
            m_TableStatistic.TabStatMaxRecLength_gg00   = m_MaxRecordLength;

            m_TableStatistic.TabStatAveKeyLength_gg00 =
                static_cast< SAPDB_Int2>( m_TotalKeyLength / m_RecordCount );

            m_TableStatistic.TabStatMinKeyLength_gg00 = m_MinKeyLength;
            m_TableStatistic.TabStatMaxKeyLength_gg00 = m_MaxKeyLength;
        }
        else
        {
            m_TableStatistic.TabStatMinLeafCov_gg00 = 0;
            m_TableStatistic.TabStatMaxLeafCov_gg00 = 0;

            m_TableStatistic.TabStatMaxRecsPerPage_gg00 = 0;
            m_TableStatistic.TabStatMinRecsPerPage_gg00 = 0;
            m_TableStatistic.TabStatRecsPerPage_gg00    = 0;

            m_TableStatistic.TabStatAveRecLength_gg00   = 0;
            m_TableStatistic.TabStatMinRecLength_gg00   = 0;
            m_TableStatistic.TabStatMaxRecLength_gg00   = 0;

            m_TableStatistic.TabStatAveKeyLength_gg00 = 0;
            m_TableStatistic.TabStatMinKeyLength_gg00 = 0;
            m_TableStatistic.TabStatMaxKeyLength_gg00 = 0;
        }

        if( 0 < m_NumberOfLongColumns )
        {
            m_TableStatistic.TabStatDefinedStringCols_gg00 = m_NumberOfLongColumns;

            m_TableStatistic.TabStatAveLenStringCol_gg00   =
                static_cast<SAPDB_Int4>( m_TotalLongColumnLength / m_NumberOfLongColumns );
            m_TableStatistic.TabStatMinLenStringCol_gg00   = m_MinLongColumnLength;
            m_TableStatistic.TabStatMaxLenStringCol_gg00   = m_MaxLongColumnLength;

            m_TableStatistic.TabStatAveStringColPages_gg00   =
                static_cast< SAPDB_Int4>( m_TotalLongColumnPages / m_NumberOfLongColumns );
            m_TableStatistic.TabStatMinStringColPages_gg00   = m_MinLongColumnPages;
            m_TableStatistic.TabStatMaxStringColPages_gg00   = m_MaxLongColumnPages;
        }
        else
        {
            m_TableStatistic.TabStatDefinedStringCols_gg00 = 0;

            m_TableStatistic.TabStatAveLenStringCol_gg00   = 0;
            m_TableStatistic.TabStatMinLenStringCol_gg00   = 0;
            m_TableStatistic.TabStatMaxLenStringCol_gg00   = 0;

            m_TableStatistic.TabStatAveStringColPages_gg00   = 0;
            m_TableStatistic.TabStatMinStringColPages_gg00   = 0;
            m_TableStatistic.TabStatMaxStringColPages_gg00   = 0;
        }
        m_TableStatistic.TabStatAllStringColPages_gg00   =          // we have an empty short columnfile
            static_cast< SAPDB_Int4>( m_TotalLongColumnPages );
    }

    /*!----------------------------------------------------------------------
    function:     IntegrateShortColumnStatistic()
    description:  This method is used to transfer the determined statistic
                  values of the short column file into the base file statistic.
                  Is is assumed that the short column file statistic are
                  executed before the table statistics.
                  code.
    arguments:    shortColStatistic [in/out] short statistic structure
    return value: none
    -----------------------------------------------------------------------*/

    void IntegrateShortColumnStatistic( const Kernel_TableStatistic  &shortColStat )
    {
        SAPDBERR_ASSERT_STATE( 0 < shortColStat.m_NumberOfLeafNodes  );

        const SAPDB_UInt metaDataLength = cgg_rec_key_offset + sizeof( tgg00_Surrogate );

        m_TotalLongColumnPages = shortColStat.m_NumberOfLeafNodes + shortColStat.m_NumberOfIndexNodes;;
        m_NumberOfLongColumns  = shortColStat.m_RecordCount;

        if( 0 < shortColStat.m_RecordCount )
        {
            m_MinLongColumnPages   = 1;
            m_MaxLongColumnPages   = 1;

            m_MinLongColumnLength   = shortColStat.m_MinRecordLength -  metaDataLength;
            m_MaxLongColumnLength   = shortColStat.m_MaxRecordLength - metaDataLength;
            m_TotalLongColumnLength = shortColStat.m_TotalRecordLength - ( shortColStat.m_RecordCount * metaDataLength );
        }
    }

    /*!----------------------------------------------------------------------
    function:     AddRootCovering()
    description:  Set covering and level of the root page and execute AddIndexCovering.
    arguments:    covering [in] number of used bytes
    return value: none
    -----------------------------------------------------------------------*/

    void AddRootCovering(
        const Data_RecordLength covering,
        const Data_Level        rootLevel )
    {
        m_RootCovering = (( covering * 100 ) / FULLCOVERING_BD00 );
        m_RootLevel    = rootLevel;
        AddIndexCovering( covering );
    }

    /*!----------------------------------------------------------------------
    function:     AddIndexCovering()
    description:  Accumulate the non leaf covering in percent and adjust the 
                  values for minimum and maximum covering. Finaly the number 
                  of non leaf nodes is incremented.
    arguments:    covering [in] number of used bytes
    return value: none
    -----------------------------------------------------------------------*/

    void AddIndexCovering( const Data_RecordLength   covering )
    {
        const SAPDB_UInt percentCovering = (( covering * 100 ) / FULLCOVERING_BD00 );

        m_MinIndexCovering = SAPDB_MIN( m_MinIndexCovering, percentCovering );
        m_MaxIndexCovering = SAPDB_MAX( m_MaxIndexCovering, percentCovering );
        m_TotalIndexCovering += percentCovering;
        m_NumberOfIndexNodes++;
    }

    /*!----------------------------------------------------------------------
    function:     AddLeafCovering()
    description:  Accumulate the leaf covering in percent and adjust the 
                  values for minimum and maximum covering. Finaly the number 
                  of leaf nodes is incremented.
    arguments:    covering [in] number of used bytes
    return value: none
    -----------------------------------------------------------------------*/

    void AddLeafCovering( const Data_RecordLength   covering )
    {
        const SAPDB_UInt percentCovering = (( covering * 100 ) / FULLCOVERING_BD00 );

        m_MinLeafCovering = SAPDB_MIN( m_MinLeafCovering, percentCovering );
        m_MaxLeafCovering = SAPDB_MAX( m_MaxLeafCovering, percentCovering );
        m_TotalLeafCovering += percentCovering;
        m_NumberOfLeafNodes++;
    }

    /*!----------------------------------------------------------------------
    function:     AddRecordCount()
    description:  Accumulate the record count and adjust the values for
                  minimum and maximum records on a leaf page.
    arguments:    recordCount [in] number of records
    return value: none
    -----------------------------------------------------------------------*/

    void AddRecordCount( const Data_RecordCount   recordCount )
    {
        m_MinRecordCount = SAPDB_MIN( m_MinRecordCount, recordCount );
        m_MaxRecordCount = SAPDB_MAX( m_MaxRecordCount, recordCount );
        m_RecordCount += recordCount;
    }

    /*!----------------------------------------------------------------------
    function:     AddRecord()
    description:  Accumulate the length of all records and keys and adjust 
                  the values for the minimum and maximum limit of record and 
                  key length.
    arguments:    recordLength [in] length of the record
                  keyLength [in]    length of the key
    return value: none
    -----------------------------------------------------------------------*/

    void AddRecord(
        const Data_RecordLength   recordLength,
        const Data_RecordLength   keyLength )
    {
        m_MinRecordLength = SAPDB_MIN( m_MinRecordLength, recordLength );
        m_MaxRecordLength = SAPDB_MAX( m_MaxRecordLength, recordLength );
        m_TotalRecordLength += recordLength;

        m_MinKeyLength = SAPDB_MIN( m_MinKeyLength, keyLength );
        m_MaxKeyLength = SAPDB_MAX( m_MaxKeyLength, keyLength );
        m_TotalKeyLength += keyLength;
    }

    /*!----------------------------------------------------------------------
    function:     AddLongColumn()
    description:  Accumulate the length of the long column given in bytes and
                  pages and adjust the values for the minimum and maximum limit 
                  of column length. Finaly the number of long columns will be
                  incremented.
    arguments:    longColumnLength [in] length of long column in bytes
                  longColumnPages  [in] length of long column in bytes
    return value: none
    -----------------------------------------------------------------------*/

    void AddLongColumn(
        const Data_RecordLength longColumnLength,
        const Data_PageCount    longColumnPages )
    {
        m_NumberOfLongColumns++;

        m_MinLongColumnLength = SAPDB_MIN( m_MinLongColumnLength, longColumnLength );
        m_MaxLongColumnLength = SAPDB_MAX( m_MaxLongColumnLength, longColumnLength );
        m_TotalLongColumnLength += longColumnLength;

        m_MinLongColumnPages = SAPDB_MIN( m_MinLongColumnPages, longColumnPages );
        m_MaxLongColumnPages = SAPDB_MAX( m_MaxLongColumnPages, longColumnPages );
        m_TotalLongColumnPages += longColumnPages;
    }

    /*!----------------------------------------------------------------------
    function:     AddSeparator()
    description:  Accumulate the length of all separators and adjust 
                  the values for the minimum and maximum limit of separator  
                  length.
    arguments:    separatorLength [in] length of the separator
    return value: none
    -----------------------------------------------------------------------*/

    void AddSeparator( const Data_RecordLength   separatorLength )
    {
        if( 0 < separatorLength )
        {
            m_MinSeparatorLength = SAPDB_MIN( m_MinSeparatorLength, separatorLength );
            m_MaxSeparatorLength = SAPDB_MAX( m_MaxSeparatorLength, separatorLength );
            m_TotalSeparatorLength += separatorLength;
            m_NumberOfSeparators++;
        }
    }

    /* -----------------------------------------------------------------------*/
    /*! endchapter:  Manipulation                                             */
    /* -----------------------------------------------------------------------*/

private:

    /*-------------------------------------------------------------------------
    declaration: m_TableStatistic
    description: Reference to the given  internal structure tgg00_TableStatInfo
                 containing the statistic information for table statistic
    --------------------------------------------------------------------------*/

    tgg00_TableStatInfo &m_TableStatistic;

    /*-------------------------------------------------------------------------
    declaration: m_NumberOfLeafNodes
    description: Number of leaf nodes
    --------------------------------------------------------------------------*/

    Data_PageCount  m_NumberOfLeafNodes;

    /*-------------------------------------------------------------------------
    declaration: m_NumberOfNonLeafNodes
    description: Number of non leaf nodes
    --------------------------------------------------------------------------*/

    Data_PageCount  m_NumberOfIndexNodes;

    /*-------------------------------------------------------------------------
    declaration: m_TotalLeafCovering
    description: The sum of all percental coverings of all leaf pages
    --------------------------------------------------------------------------*/

    SAPDB_ULong m_TotalLeafCovering;

    /*-------------------------------------------------------------------------
    declaration: m_TotalIndexCovering
    description: The sum of all percental coverings of all non leaf pages
    --------------------------------------------------------------------------*/

    SAPDB_ULong m_TotalIndexCovering;

    /*-------------------------------------------------------------------------
    declaration: m_RootLevel
    description: Number of levels of the tree
    --------------------------------------------------------------------------*/

    Data_Level  m_RootLevel;

    /*-------------------------------------------------------------------------
    declaration: m_RootCovering
    description: The covering of the root page, if the tree is at least of level 1
    --------------------------------------------------------------------------*/

    SAPDB_ULong m_RootCovering;

    /*-------------------------------------------------------------------------
    declaration: m_MinIndexCovering
    description: The lowest covering on one index page found in the given file
    --------------------------------------------------------------------------*/

    Data_RecordLength   m_MinIndexCovering;

    /*-------------------------------------------------------------------------
    declaration: m_MaxIndexCovering
    description: The biggest covering on one index page found in the given file
    --------------------------------------------------------------------------*/

    Data_RecordLength   m_MaxIndexCovering;

    /*-------------------------------------------------------------------------
    declaration: m_MinLeafCovering
    description: The lowest covering on one leaf page found in the given file
    --------------------------------------------------------------------------*/

    Data_RecordLength   m_MinLeafCovering;

    /*-------------------------------------------------------------------------
    declaration: m_MaxLeafCovering
    description: The biggest covering on one leaf page found in the given file
    --------------------------------------------------------------------------*/

    Data_RecordLength   m_MaxLeafCovering;

    /*-------------------------------------------------------------------------
    declaration: m_RecordCount
    description: Total number of records within the complete file
    --------------------------------------------------------------------------*/

    Data_RecordCount    m_RecordCount;

    /*-------------------------------------------------------------------------
    declaration: m_MinRecordCount
    description: The lowest record count on one page found in the given file
    --------------------------------------------------------------------------*/

    Data_RecordCount    m_MinRecordCount;

    /*-------------------------------------------------------------------------
    declaration: m_MaxRecordCount
    description: The hightest record count on one page found in the given file
    --------------------------------------------------------------------------*/

    Data_RecordCount    m_MaxRecordCount;

    /*-------------------------------------------------------------------------
    declaration: m_TotalRecordLength
    description: Is the sum of all record lengths found in the given file.
    --------------------------------------------------------------------------*/

    SAPDB_Real8 m_TotalRecordLength;

    /*-------------------------------------------------------------------------
    declaration: m_MinRecordLength
    description: The smallest record on one page found in the given file
    --------------------------------------------------------------------------*/

    Data_RecordLength   m_MinRecordLength;

    /*-------------------------------------------------------------------------
    declaration: m_MaxRecordLength
    description: The biggest record on one page found in the given file
    --------------------------------------------------------------------------*/

    Data_RecordLength   m_MaxRecordLength;

    /*-------------------------------------------------------------------------
    declaration: m_TotalKeyLength
    description: Is the sum of all key lengths found in the given file.
    --------------------------------------------------------------------------*/

    SAPDB_Real8 m_TotalKeyLength;

    /*-------------------------------------------------------------------------
    declaration: m_MinKeyLength
    description: The smallest key on one page found in the given file
    --------------------------------------------------------------------------*/

    Data_RecordLength   m_MinKeyLength;

    /*-------------------------------------------------------------------------
    declaration: m_MaxKeyLength
    description: The biggest key on one page found in the given file
    --------------------------------------------------------------------------*/

    Data_RecordLength   m_MaxKeyLength;

    /*-------------------------------------------------------------------------
    declaration: m_NumberOfLongColumns
    description: Total number of defined long columns within given file
    --------------------------------------------------------------------------*/

    Data_RecordCount m_NumberOfLongColumns;

    /*-------------------------------------------------------------------------
    declaration: m_MinLongColumnLength
    description: The smallest long column in the given file
    --------------------------------------------------------------------------*/

    Data_RecordLength   m_MinLongColumnLength;

    /*-------------------------------------------------------------------------
    declaration: m_MaxLongColumnLength
    description: The biggest long column in the given file
    --------------------------------------------------------------------------*/

    Data_RecordLength   m_MaxLongColumnLength;

    /*-------------------------------------------------------------------------
    declaration: m_TotalLongColumnLength
    description: Is the sum of all long column lengths found in the given file.
    --------------------------------------------------------------------------*/

    SAPDB_Real8 m_TotalLongColumnLength;

    /*-------------------------------------------------------------------------
    declaration: m_MinLongColumnPages
    description: The number of pages of the smallest long column in the given file
    --------------------------------------------------------------------------*/

    Data_RecordLength   m_MinLongColumnPages;

    /*-------------------------------------------------------------------------
    declaration: m_MaxLongColumnPages
    description:  The number of pages of the biggest long column in the given file
    --------------------------------------------------------------------------*/

    Data_RecordLength   m_MaxLongColumnPages;

    /*-------------------------------------------------------------------------
    declaration: m_TotalLongColumnPages
    description: Is the sum of all long column pages found in the given file.
    --------------------------------------------------------------------------*/

    SAPDB_ULong m_TotalLongColumnPages;

    /*-------------------------------------------------------------------------
    declaration: m_NumberOfSeparators
    description: Is the number of all separators found in the given file.
    --------------------------------------------------------------------------*/

    SAPDB_ULong m_NumberOfSeparators;

    /*-------------------------------------------------------------------------
    declaration: m_MinSeparatorLength
    description: The smallest separator on one page found in the given file
    --------------------------------------------------------------------------*/

    Data_RecordLength   m_MinSeparatorLength;

    /*-------------------------------------------------------------------------
    declaration: m_MaxSeparatorLength
    description: The biggest separator on one page found in the given file
    --------------------------------------------------------------------------*/

    Data_RecordLength   m_MaxSeparatorLength;

    /*-------------------------------------------------------------------------
    declaration: m_TotalSeparatorLength
    description: Is the sum of all separator lengths found in the given file.
    --------------------------------------------------------------------------*/

    SAPDB_Real8 m_TotalSeparatorLength;

};

/*!*****************************************************************************

   endclass: Kernel_TableStatistic

*******************************************************************************/


#endif  /* KERNEL_TABLE_STATISTIC_HPP */

