/*!
  @file           IOMan_DataVolume.cpp
  @author         TorstenS
  @ingroup        IOManagement
  @brief          This module is used to handle a data volume.

\if EMIT_LICENCE
    ========== licence begin  GPL
    Copyright (c)  2000-2006 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end

\endif
*/



/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "IOManager/IOMan_DataVolume.hpp"
#include "KernelCommon/Kernel_FileIO.hpp"
#include "KernelCommon/Kernel_IAdminRestartShutdown.hpp"
#include "IOManager/IOMan_Exception.hpp"
#include "IOManager/IOMan_Messages.hpp"
#include "RunTime/RTE_Crash.hpp"
#include "RunTime/RTE_Message.hpp"


/*===========================================================================*
 *  DEFINES                                                                  *
 *===========================================================================*/



/*===========================================================================*
 *  MACROS                                                                   *
 *===========================================================================*/



/*===========================================================================*
 *  LOCAL CLASSES, STRUCTURES, TYPES, UNIONS ...                             *
 *===========================================================================*/



/*===========================================================================*
 *  STATIC/INLINE FUNCTIONS (PROTOTYPES)                                     *
 *===========================================================================*/



/*===========================================================================*
 *  METHODS                                                                  *
 *===========================================================================*/

bool
IOMan_DataVolume::ReadPages(
    RTETask_ITask&              task,
    IOMan_IKernelPages&         pages,
    const IOMan_ClusterAddress& cluster,
    SAPDB_UInt&                 badPageIndex )
{
    SAPDBERR_ASSERT_STATE( cluster.GetDeviceNo() == GetLogicalDevNo() );

    IOMan_IKernelPagesIterator          iter( pages );
    IOMan_ClusterAddress::BlockIndex    index = 0;

#   ifdef SAPDB_SLOW
    const bool bDoCheck = true;
#   else    
    const bool bDoCheck = IOMan_Check.ChecksLevel( 5 );
#   endif

    if( bDoCheck && false) // TODO handle empty page handler
    {
        // Danger, because it's assumed that the given page frames are
        // in strictly ascending order within the memory!
        const SAPDB_UInt            nextIndex = IOMan_IKernelPages::FirstIndex()+ 1;
        IOMan_IKernelPagesIterator  prevIter( pages );

        for( prevIter.Begin(), iter.Begin( nextIndex ); !iter.End(); ++prevIter, ++iter )
        {
            if( ( *prevIter ).GetFramePtr() + m_BlockSize != ( *iter ).GetFramePtr())
            {
                RTE_Crash( SAPDBErr_Exception(__FILE__, __LINE__,
                                              SAPDBERR_ASSERT_STATE_FAILED, "IOMan_DataVolume::ReadPages" ));
            }
        }
    }
    // find first valid page frame
    SAPDB_UInt startIndex = 0;
    for( iter.Begin(), startIndex = 0; startIndex < cluster.GetBlockCount(); ++iter, ++startIndex ){
        if(( *iter ).IsAssigned()){
            break;
        }
    }
    if( startIndex >= cluster.GetBlockCount()){
        return true; // contains no pages to read
    }
    for( SAPDB_UInt readCount = 1; IOMan_Volume::MaxReadRetries() >= readCount; ++readCount )
    {
        if( ! ReadSimple( task.ID(), *( pages.GetFrameList( startIndex )),
                          cluster.GetBlockNo( startIndex ), cluster.GetBlockCount() - startIndex ))
        {
            IOMan_Exception errMsg( __CONTEXT__, IOMAN_EMERGENCY_MSG1 );
            RTE_Crash( errMsg );
            // Kernel_IAdminRestartShutdown::Instance().Offline(); TODO
        }
        for( index = 0, iter.Begin(); index < cluster.GetBlockCount(); ++iter, ++index )
        {
            // Caused by performance reason it is possible that more data pages are read
            // by this cluster io than marked as io pending in the data cache.
            // The io check must not (!) be executed for the pages which are not marked as
            // io pending in the data cache, because this pages may be changed during this
            // cluster io
            if( ! ( *iter ).IsAssigned()){
                continue;
            }
            if(( *iter ).CheckAfterRead()){
                continue;
            }
            IOMan_Exception errMsg( __CONTEXT__, IOMAN_BAD_PAGE,
                                    GetTypeName(), SAPDB_ToString( GetLogicalDevNo(), _T_d ),
                                    SAPDB_ToString( cluster.GetBlockNo( index ), _T_d ));

            Kernel_FileIO   dumpFile( GetTypeName(), KERNEL_BAD_FILE_EXTENSION,
                                      SAPDB_ToString( GetLogicalDevNo(), _T_d ),
                                      SAPDB_ToString( cluster.GetBlockNo( index ), _T_d ));
            dumpFile.Write( *iter );
            dumpFile.Close();

            badPageIndex = index;
            RTE_Message( errMsg );
            task.Sleep( IOMan_Volume::RetrySleepTime());
            break;
        }
        if( index >= cluster.GetBlockCount()){
            return( true );     // usual case
        }
        //else read the page collection again, because an error has occured
    }
    return false ;
}

/*===========================================================================*
 *  END OF CODE                                                              *
 *===========================================================================*/

