/*!
    @file       IOMan_DataArea.cpp
    @author     TorstenS
    @ingroup    IOManagement
    @brief      This module is used to manage the data area.

\if EMIT_LICENCE
    ========== licence begin  GPL
    Copyright (c) 2001-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end

\endif
*/


/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "hkb57.h"

#include "IOManager/IOMan_DataArea.hpp"
#include "IOManager/IOMan_Messages.hpp"
#include "IOManager/IOMan_MessageList.hpp"

#include "Converter/Converter_IManager.hpp"

#include "DataAccess/Data_PageFrame.hpp"

#include "FreeBlockManagement/FBM_IManager.hpp"

#include "RunTime/RTE_Crash.hpp"
#include "RunTime/RTE_Message.hpp"
#include "RunTime/InputOutput/RTEIO_VolumeInfo.hpp"
#include "RunTime/Tasking/RTETask_ITask.hpp"

#include "SAPDBCommon/SAPDB_MemCopyMove.hpp"
#include "SAPDBCommon/SAPDB_ToString.hpp"
#include "SAPDBCommon/MemoryManagement/SAPDBMem_IRawAllocator.hpp"


/*===========================================================================*
 *  DEFINES                                                                  *
 *===========================================================================*/


/*===========================================================================*
 *  MACROS                                                                   *
 *===========================================================================*/



/*===========================================================================*
 *  LOCAL CLASSES, STRUCTURES, TYPES, UNIONS ...                             *
 *===========================================================================*/



/*===========================================================================*
 *  STATIC/INLINE FUNCTION PROTOTYPES                                        *
 *===========================================================================*/



/*===========================================================================*
 *  METHODS                                                                  *
 *===========================================================================*/

SAPDB_Bool
IOMan_DataArea::AddVolume(
    RTETask_ITask&              task,
    const SAPDB_Bool            bAddVolumeOnly,
    tsp00_VFilename             &devName,
    const IOMan_BlockCount      configuredSize,
    const IOMan_DeviceNo        devNo,
    const IOMan_BlockAddress    &restartPageAddress)
{
    SAPDBTRACE_ROUTINE_DEBUG( "IOMan_DataArea::AddVolume", IOMan_Trace, 6 );

    SAPDBTRACE_WRITELN( IOMan_Trace, 6, "Volume: " << devName <<
                        " VolSize: " << configuredSize << " DevNo: " << devNo );

    if( ! IsDataVolumeNoValid( devNo ))
    {
        IOMan_MessageList errMsg( __CONTEXT__, IOMAN_ERR_DATA_DEVNO_OUT_OF_RANGE,
                                  SAPDB_ToString( devNo, _T_d ),
                                  SAPDB_ToString( MaxVolumes(), _T_d ));
        RTE_Message( errMsg );
        return false;
    }

    IOMan_DeviceNo firstConfiguredVolume = GetFirstConfiguredVolumeNo();

    IOMan_DataVolume&   volume      = Get( devNo );
    IOMan_DataVolume*   pPrevVolume = 0;
    IOMan_DataVolume*   pNextVolume = 0;

    // Add data volume to the control structures of the RTE, because at the
    // time of kernel restart this volume was not stored within data base
    // configuration file. The volume was made persistent by the dbmsrv after
    // starting the db kernel, therefore vcold couldn't find the volume!

    if(! volume.Add( devName, configuredSize ))
        return false;

    const SAPDB_Byte *pDBIdentifier = DetermineLinkage( volume, &pPrevVolume, &pNextVolume );

    if( 0 == pDBIdentifier )
        return false;

    IOMan_ReturnCode retCode = volume.Create( task.ID(), pDBIdentifier );  // time to format volume unlocked

    { // locked for linkage checking and modifaction
        LockedScope lock(*m_pRwLock);

        if( IOMan_Okay == retCode )
        {
            DetermineLinkage( volume, &pPrevVolume, &pNextVolume );

            if( IOMan_Okay == volume.Open( task ))
            {
                if( bAddVolumeOnly || (FBM_IManager::Instance().RegisterVolume( task.ID(),
                        volume.GetLogicalDevNo(), volume.GetInternalSize(), 
                        volume.GetAccessMode(), restartPageAddress ) && FBM_IManager::Instance().ReserveClusterSpace(kb57GetClusterReservedSpace())))
                {
                    if( 0 != pPrevVolume )
                        this->CreateLinkage( task, *pPrevVolume, volume );

                    if( 0 != pNextVolume )
                        this->CreateLinkage( task, volume, *pNextVolume );

                    m_TotalDataPages += volume.GetUsableSize();
                    m_UsedVolumes++;
                    m_ConfiguredVolumes++;

                    // At this moment the blocks of the new volume are usable.
                    // If the converter expansion failed (crash) the new volume
                    // is still valid!
                    {
                        IOMan_MessageList msgList( __CONTEXT__, IOMAN_INFO_ADD_VOLUME,
                                                   volume.GetTypeName(),
                                                   SAPDB_ToString( volume.GetLogicalDevNo(), _T_d ),
                                                   SAPDB_ToString( volume.GetConfiguredSize(), _T_d ));
                        RTE_Message( msgList );
                    }
                    // try to expand converter if data base is online to increase
                    // the number of usable pageNo's. Note that it is not fatal if
                    // the expand fails, because the converter will grow during the
                    // next restart.

                    if( ! bAddVolumeOnly ){
                        Converter_IManager::Instance().Expand( task, volume.GetUsableSize());
                    }

                    if ( firstConfiguredVolume > devNo ) // add in front of first volume -> move restart page address to first InfoPage
                    {
                        this->Get( devNo ).SetRestartPageAddress( task, restartPageAddress ); // new
                        this->Get( firstConfiguredVolume ).SetRestartPageAddress( task, IOMan_BlockAddress( 0,0 )); // invalidate
                    }

                    return true; // normal termination
                }
            }
        }
    }
    volume.Close( task.ID());
    volume.Del();
    return false;
}

/*---------------------------------------------------------------------------*/

SAPDB_Bool
IOMan_DataArea::DropVolume(
    RTETask_ITask&          task,
    const IOMan_DeviceNo    volNo )
{
    SAPDBTRACE_ROUTINE_DEBUG( "IOMan_DataArea::DropVolume", IOMan_Trace, 6 );

    SAPDBTRACE_WRITELN( IOMan_Trace, 6, "Volume: " <<  volNo );

    if( ! IsDataVolumeNoValid( volNo ))
    {
        IOMan_MessageList errMsg( __CONTEXT__, IOMAN_ERR_DATA_DEVNO_OUT_OF_RANGE,
                                  SAPDB_ToString( volNo, _T_d ),
                                  SAPDB_ToString( MaxVolumes(), _T_d ));
        RTE_Message( errMsg );
        return false;
    }
    IOMan_DataVolume  &volume = Get( volNo );

    { // locked for linkage checking and modifaction
        LockedScope lock(*m_pRwLock);

        this->DropLinkage( task, volume );

        m_TotalDataPages -= volume.GetUsableSize();
        m_UsedVolumes--;
        m_ConfiguredVolumes--;

        Converter_IManager::Instance().Shrink( task, volume.GetUsableSize());
    }
    volume.Close( task.ID());
    volume.Del();
    return true;
}

/*---------------------------------------------------------------------------*/

void
IOMan_DataArea::CloseAll( RTETask_ITask&    task )
{
    SAPDBTRACE_ROUTINE_DEBUG( "IOMan_DataArea::CloseAll", IOMan_Trace, 6 );

    ConstIterator   endIter = End();

    for( Iterator iter = Begin(); iter != endIter; ++iter )
        ( *iter ).Close( task.ID());

    m_TotalDataPages = 0;
    m_UsedVolumes    = 0;
}

/*---------------------------------------------------------------------------*/

SAPDB_Bool
IOMan_DataArea::CreateAll(
    RTETask_ITask&      task,
    const SAPDB_Byte*   pDBIdentifier )
{
    SAPDBTRACE_ROUTINE_DEBUG( "IOMan_DataArea::CreateAll", IOMan_Trace, 6 );

    IOMan_MessageList msgList;

    if( 0 != m_TotalDataPages )
    {
        IOMan_MessageList errMsg( __CONTEXT__, IOMAN_ERR_DATA_AREA_ALREADY_ONLINE );
        RTE_Crash( errMsg );
    }

    if( ! FormatAll( task, msgList ))
    {
        msgList.AppendNewMessage( IOMan_MessageList( __CONTEXT__, IOMAN_ERR_ALL_FORMAT, "data" ));
        RTE_Message( msgList );
        return false;
    }
    { // locked for linkage checking and modifaction
        LockedScope lock(*m_pRwLock);

        Iterator        prevIter = End();
        ConstIterator   endIter  = End();

        for( Iterator iter = Begin(); iter != endIter; ++iter )
        {
            IOMan_ReturnCode retCode = ( *iter ).CreateInfoPage( task.ID(), pDBIdentifier );

            if( IOMan_Okay == retCode )
            {
                if( IOMan_Okay != (( *iter ).Open( task ))){
                    return false;
                }
                else
                {
                    if( prevIter != endIter ){
                        this->CreateLinkage( task, *prevIter, *iter );
                    }
                    prevIter          = iter;
                    m_TotalDataPages += ( *iter ).GetUsableSize();
                    m_UsedVolumes++;
                }
            }
            else if( IOMan_NotConfigured == retCode ){
                continue;
            }
            else{
                return false;
            }
        }
        SAPDBERR_ASSERT_STATE( m_ConfiguredVolumes == m_UsedVolumes );
    }
    return true;
}

/*---------------------------------------------------------------------------*/

SAPDB_Bool
IOMan_DataArea::GetVolumeNo(
    tsp00_VFilename&    devName,
    IOMan_DeviceNo&     devNo ) const
{
    SAPDBTRACE_ROUTINE_DEBUG( "IOMan_DataArea::GetVolumeNo", IOMan_Trace, 6 );

    tsp00_VFilename     auxName;
    ConstIterator       endIter = End();

    for( ConstIterator iter = Begin(); iter != endIter; ++iter )
    {
        if(( *iter ).GetName( auxName ))
        {
            if( 0 == memcmp( devName, auxName, sizeof( devName )))
            {
                devNo = (*iter).GetLogicalDevNo();
                return true;
            }
        }
    }
    return false;
}

/*---------------------------------------------------------------------------*/

SAPDB_Bool
IOMan_DataArea::OpenAll( RTETask_ITask& task )
{
    SAPDBTRACE_ROUTINE_DEBUG( "IOMan_DataArea::OpenAll", IOMan_Trace, 6 );

    { // locked for linkage checking and modifaction
        LockedScope lock(*m_pRwLock);

        IOMan_BlockCount    totalDataPages = 0;
        SAPDB_UInt          usedVolumes    = 0;
        Iterator            prevIter       = End();
        ConstIterator       endIter        = End();

        for( Iterator iter = Begin(); iter != endIter; ++iter )
        {
            IOMan_ReturnCode retCode = ( *iter ).Open( task );

            if( IOMan_Okay == retCode )
            {
                if( prevIter == endIter )
                {
                    // iter points to the first volume within configuration file
                    if( ( *iter ).IsPrevLinkageValid())
                    {
                        IOMan_MessageList errMsg( __CONTEXT__, IOMAN_ERR_MISSING_PREDECESSOR,
                                                  ( *iter ).GetTypeName(),
                                                  SAPDB_ToString( ( *iter ).GetLogicalDevNo(), _T_d ),
                                                  SAPDB_ToString( ( *iter ).GetPrevLogicalDevNo(), _T_d ));
                        RTE_Message( errMsg );
                        return false;
                    }
                }
                else
                {
                    if( !( *prevIter ).IsLinkageValid( task.ID(), *iter )){
                        return false;
                    }
                    if( !( *prevIter ).IsDBIdentifierValid( task.ID(), *iter )){
                        return false;
                    }
                }
                prevIter        = iter;
                totalDataPages += ( *iter ).GetUsableSize();
                usedVolumes++;
            }
            else if( IOMan_NotConfigured == retCode ){
                continue;
            }
            else{
                return false;
            }
        }
        if( ( *prevIter ).IsNextLinkageValid())
        {
            IOMan_MessageList errMsg( __CONTEXT__, IOMAN_ERR_MISSING_SUCCESSOR,
                                      ( *prevIter ).GetTypeName(),
                                      SAPDB_ToString( ( *prevIter ).GetLogicalDevNo(), _T_d ),
                                      SAPDB_ToString( ( *prevIter ).GetNextLogicalDevNo(), _T_d ));
            RTE_Message( errMsg );
            return false;
        }

        SAPDBERR_ASSERT_STATE( m_ConfiguredVolumes == usedVolumes );

        // Set the data area capacity after opening all volumes
        m_TotalDataPages = totalDataPages;
        m_UsedVolumes    = usedVolumes;

    }
    return true;
}

/*---------------------------------------------------------------------------*/

IOMan_ReturnCode
IOMan_DataArea::MigrateCapacity(
    RTETask_ITask&      task,
    IOMan_MessageList&  msgList )
{
    SAPDBTRACE_ROUTINE_DEBUG( "IOMan_DataArea::MigrateCapacity", IOMan_Trace, 6 );

    ConstIterator   endIter = End();

    for( Iterator iter = Begin(); iter != endIter; ++iter )
    {
        const IOMan_ReturnCode retCode = ( *iter ).MigrateCapacity( task.ID(), msgList );

        if(( IOMan_Okay == retCode ) || ( IOMan_NotConfigured == retCode ))
            continue;
        else
            return retCode;
    }
    return IOMan_Okay;
}

/*---------------------------------------------------------------------------*/

bool
IOMan_DataArea::Initialize(
    RTETask_ITask&      task,
    const SAPDB_Int     maxDataVolumes,
    const SAPDB_UInt    dataBlockIoCount )
{
    SAPDBTRACE_ROUTINE_DEBUG( "IOMan_DataArea::Initialize", IOMan_Trace, 6 );

    if(! m_Volumes.Resize( maxDataVolumes )){
        return false;
    }
    m_BlockIoCount          = dataBlockIoCount;
    IOMan_DeviceNo  devNo   = IOMan_DataArea::GetFirstDataVolumeNo();
    Iterator        iter    = Begin();
    ConstIterator   endIter = End();

    while( iter != endIter )
    {
        Data_PageFrame frame = m_PageAllocator.New( task.ID());

        if( ! frame.IsAssigned()){
            return false;
        }
        ( *iter ).Initialize( devNo, frame );
        // Determine the number of configured data volumes at this point
        // because same compontents are interested in this information
        // before the corresponding data volumes are started.

        if( ( *iter ).IsConfigured()){
            ++m_ConfiguredVolumes;
        }
        ++iter;
        ++devNo;
    }
    return true;
}

/*---------------------------------------------------------------------------*/

IOMan_DeviceNo
IOMan_DataArea::GetConfiguredVolumeNo( const SAPDB_Int pos ) const
{
    SAPDBTRACE_ROUTINE_DEBUG( "IOMan_DataArea::GetConfiguredVolume", IOMan_Trace, 6 );

    IOMan_DeviceNo  devNo   = IOMan_DataArea::GetFirstDataVolumeNo();
    ConstIterator   iter    = Begin();
    ConstIterator   endIter = End();
    SAPDB_Int       counter = 0;

    while( iter != endIter )
    {
        if( ( *iter ).IsConfigured())
        {
            counter ++;
            if (counter == pos)
            {
                return devNo;
            }
        }
        ++iter;
        ++devNo;
    }
    return IOMan_DeviceNo(); // invalid
}

/*===========================================================================*
 *  PRIVATE METHODS                                                          *
 *===========================================================================*/

SAPDB_Bool
IOMan_DataArea::FormatAll(
    RTETask_ITask&      task,
    IOMan_MessageList&  msgList )
{
    RTEIO_VolumeInfoVector  infoVector( m_Allocator );

    if( ! infoVector.Reserve( m_ConfiguredVolumes )){
        return false;
    }
    ConstIterator   iter;
    ConstIterator   endIter = End();

    for( iter = Begin(); iter != endIter; ++iter )
    {
        if(( *iter ).IsConfigured())
        {
            infoVector.InsertEnd( new (m_Allocator) RTEIO_DataVolumeInfo(
                                      (*iter).GetLogicalDevNo(),
                                      (*iter).GetInternalSize()));
        }
    }
    // format all volumes
    if ( ! RTEIO_FormatVolumeVector( task.ID(), infoVector, msgList )){
        return false;
    }
    return true;
}

/*---------------------------------------------------------------------------*/

void
IOMan_DataArea::CreateLinkage(
    RTETask_ITask&      task,
    IOMan_DataVolume&   prevVolume,
    IOMan_DataVolume&   volume )
{
    if( prevVolume.SetNextLinkage( task.ID(), volume ))
        if( volume.SetPrevLinkage( task.ID(), prevVolume ))
            return;

    IOMan_MessageList errMsg( __CONTEXT__, IOMAN_ERR_CREATE_LINKAGE,
                              volume.GetTypeName(),
                              SAPDB_ToString( prevVolume.GetLogicalDevNo(), _T_d ),
                              SAPDB_ToString( volume.GetLogicalDevNo(), _T_d ));
    RTE_Crash( errMsg );
}

/*---------------------------------------------------------------------------*/

void
IOMan_DataArea::DropLinkage(
    RTETask_ITask&      task,
    IOMan_DataVolume&   volume )
{
    IOMan_DataVolume*   pPrevVolume = 0;
    IOMan_DataVolume*   pNextVolume = 0;

    this->DetermineLinkage( volume, &pPrevVolume, &pNextVolume );

    if( 0 != pPrevVolume )
    {
        if( 0 == pNextVolume )
            pPrevVolume->SetNextLinkageToInvalid( task );
        else
            pPrevVolume->SetNextLinkage( task.ID(), *pNextVolume );
    }
    if( 0 != pNextVolume )
    {
        if( 0 == pPrevVolume )
            pNextVolume->SetPrevLinkageToInvalid( task );
        else
            pNextVolume->SetPrevLinkage( task.ID(), *pPrevVolume );
    }
}

/*---------------------------------------------------------------------------*/

const SAPDB_Byte*
IOMan_DataArea::DetermineLinkage(
    const IOMan_DataVolume& volume,
    IOMan_DataVolume**      pPrevVolume,
    IOMan_DataVolume**      pNextVolume )
{
    ConstIterator   endIter = End();

    for( Iterator iter = Begin(); iter != endIter; ++iter )
    {
        if(( *iter ).IsOnline())
        {
            if(( *iter ).GetLogicalDevNo() < volume.GetLogicalDevNo())
                *pPrevVolume = iter;
            else
                if(( *iter ).GetLogicalDevNo() > volume.GetLogicalDevNo())
                {
                    *pNextVolume = iter;
                    break;
                }
        }
    }
    if( 0 != *pPrevVolume )
        return( **pPrevVolume ).GetDBIdentifier();
    else if( 0 != *pNextVolume )
        return( **pNextVolume ).GetDBIdentifier();
    else
        return 0;
}

/*===========================================================================*
*  END OF CODE                                                              *
*===========================================================================*/
