/*!
  @file           FBM_Manager.hpp
  @author         TorstenS
  @author         AlexanderK
  @ingroup        FBM
  @brief          General header file of the FBM

\if EMIT_LICENCE
    ========== licence begin  GPL
    Copyright (c) 2001-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end

\endif
*/



#ifndef FBM_MANAGER_HPP
#define FBM_MANAGER_HPP

/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "gsp00.h"
#include "ggg00.h"
#include "heo51.h"
#include "heo55k.h"                                // RTE   : vbegexcl,venexcl 
#include "hgg08.h"                                 // region identifier

#include "FreeBlockManagement/FBM_IManager.hpp"
#include "FreeBlockManagement/FBM_DataVolumeArray.hpp"

#include "KernelCommon/Kernel_Common.hpp"
#include "KernelCommon/Kernel_SynchronizedDBFullList.hpp"

#include "RunTime/MemoryManagement/RTEMem_AllocatorWrapper.hpp"
#include "SAPDBCommon/MemoryManagement/SAPDBMem_IRawAllocator.hpp"


/*===========================================================================*
 *  DEFINES                                                                  *
 *===========================================================================*/

/// suspend reason for db full
#define FBM_SUSPEND_HANDLE_DB_FULL                 198

/*===========================================================================*
 *  CLASS DECLARATIONS                                                       *
 *===========================================================================*/

class Kernel_Dump;

/*!
  \class          FBM_Manager
  \brief          FreeBlockManagement / manager for all attached volume

 */


class FBM_Manager : public FBM_IManager
{

public:

    /*!
       @brief          static method to create a singelton of the type FBM_Manager
       @return         (FBM_Manager&amp;) reference to the instance of FBM_Manager
     */
    static        FBM_Manager& CreateInstance();

    /*!
       @brief          returns the reference to the sigelton instance of FBM_Manager
       @return         (FBM_Manager&amp;) reference to the instance of FBM_Manager
     */
    static FBM_Manager& Instance()
    {
        return *m_Instance;
    }

    FBM_ReturnCode Restart(
        const tsp00_TaskId                  TaskId,
        const IOMan_VolumeNo                MaxNumDev,
        const IOMan_ReservedBlockAddress   &ReservedBlocks,
        const IOMan_BlockCount              clusterSize,
        const SAPDB_Int                     volumeGroups);

    /*!
       @brief   add a new volume to the FBM
       @param   TaskId           [in] task id
       @param   VolNo            [in] device number
       @param   VolSize          [in] number of pages the device can accomodate
       @param   VolMode          [in] access mode of the volume 
       @param   restartPageBlock [in] block address of restart record
       @return  true if successful executed else false
     */
    bool RegisterVolume(
        const tsp00_TaskId          TaskId,
        const IOMan_VolumeNo        VolNo,
        const IOMan_BlockCount      VolSize,
        const RTE_VolumeAccessMode  VolMode,
        const IOMan_BlockAddress    &restartPageBlock );

    /*!
       \brief   remove a volume from the fbm
       \param   task [in] current task
       \param   msgList [out] message list for errors
       \param   volNo [in] volume number of the volume to remove
       \return  true if the volume was found and successfully removed
    */

    bool DeRegisterVolume(
        const RTETask_ITask     &task,
        Msg_List                &msgList,
        const IOMan_VolumeNo    volNo);

    /*!
       \brief   prepares dropping of a volume
       \param   task [in] current task
       \param   msgList [out] message list for errors
       \param   volNo [in] volume number of the volume to drop
       \return  true if the volume was found and successfully prepared

                a volume in drop mode prevents writing of new pages to this volume
     */
    bool PrepareVolumeForDrop(
        const RTETask_ITask     &task,
        Msg_List                &msgList,
        const IOMan_VolumeNo    volNo);

    /*!
       \brief   restores normal operation of a volume after an aborted drop volume
       \param   task [in] current task
       \param   msgList [out] message list for errors
       \param   volNo [in] volume number of the volume to restore
       \return  true if the volume was found and successfully restored

           a volume in drop mode prevents writing of new pages to this volume
     */
    bool RestoreVolumeForDrop(
        const RTETask_ITask     &task,
        Msg_List                &msgList,
        const IOMan_VolumeNo    volNo);

    /*!
       \brief   checks the 'drop state' of the volume
       \param   volNo [in] volume number of the volume to restore
       \returns        'drop state' of the volume

            a volume in drop mode prevents writing of new pages to this volume
     */
    bool IsVolumeToDrop(
        const IOMan_VolumeNo    volNo);

    /*!
       @brief          Sets the state of block to 'marked for backup'
       @param          TaskId [in] task id
       @param          block [in] I/O adress of the block to change
       @return         (bool) true if operation was executed successfully

       - If the state could not be changed successfully the kernel aborts, if
      the given parameter bAbortIfErrorOccured is set to true.
     */
    bool SetBlockStateToBackup(
        const tsp00_TaskId        TaskId,
        const IOMan_BlockAddress &block )
    {
        FBM_SynchObject SynchronizeFBM (TaskId);
        return SetBlockStateToBackupUnsynched ( block );
    }

    /*!
       @brief          Sets the state of multiple block to 'marked for backup'
       @param          TaskId [in] task id
       @param          BlockIterator [in] I/O adresses of the blocks to change

       @return         (bool) true if operation was executed successfully

       - If the state could not be changed successfully the kernel aborts, if
      the given parameter bAbortIfErrorOccured is set to true.
     */
    bool SetBlockStateToBackup(
        const tsp00_TaskId            TaskId,
        IOMan_IBlockAddressIterator   &BlockIterator )
    {
        SAPDBTRACE_ROUTINE_DEBUG ("FBM_Manager::SetBlockStateToBackup", FBM_Trace, 5);

        FBM_SynchObject SynchronizeFBM (TaskId);

        while (BlockIterator.hasMoreElements())
        {
            IOMan_BlockAddress block = BlockIterator.getNextElement();

            if (! SetBlockStateToBackupUnsynched ( block ))
            {
                return false;
            }
        }
        return true;
    }

    /*!
       @brief   Sets the state of block to 'occupied'. If the block is already
                marked as occupied a error message is written.
       @param   TaskId        [in] task id
       @param   BlockIterator [in] iterator of I/O adresses with the blocks to change
       @return  (bool) true if operation was executed successful

       - If the state could not be changed successfully the kernel aborts if
      the given parameter bAbortIfErrorOccured is set to true.
     */
    bool SetBlockStateToOccupied(
        const tsp00_TaskId          TaskId,
        IOMan_IBlockAddressIterator &BlockIterator)
    {
        SAPDBTRACE_ROUTINE_DEBUG ("FBM_Manager::SetBlockStateToOccupied", FBM_Trace, 5);

        FBM_SynchObject SynchronizeFBM (TaskId);

        while (BlockIterator.hasMoreElements())
        {
            IOMan_BlockAddress block = BlockIterator.getNextElement();
            if ( ! SetBlockStateToOccupiedUnsynched ( block )){
                return false;
            }
        }
        return true;
    }

    /*!
       @brief   Sets the state of block to 'occupied'. If the block is already
                marked as occupied a error message is written.
       @param   TaskId       [in] task id
       @param   block [in] I/O adress of the block to change
       @return  (bool) true if operation was executed successful

       - If the state could not be changed successfully the kernel aborts if
      the given parameter bAbortIfErrorOccured is set to true.
     */
    bool SetBlockStateToOccupied(
        const tsp00_TaskId        TaskId,
        const IOMan_BlockAddress &block)
    {
        FBM_SynchObject SynchronizeFBM (TaskId);

        return SetBlockStateToOccupiedUnsynched ( block );
    }

    /*!
       @brief          Sets the state of cluster to 'free'
       @param          TaskId [in] task id
       @param          ClusterAddress [in] I/O adress and length of the cluster to change
       @return         none

       - If the state could not be changed successfully the kernel aborts
     */
    void SetClusterStateToFree(
        const tsp00_TaskId         TaskId,
        const IOMan_ClusterAddress &ClusterAddress );

    /*!
       @brief          Sets the state of block to 'free'
       @param          TaskId [in] task id
       @param          block [in] I/O adress of the block to change
       @return         none

       - If the state could not be changed successfully the kernel aborts
     */
    void SetBlockStateToFree(
        const tsp00_TaskId       TaskId,
        const IOMan_BlockAddress &block );

    /*!
       @brief   Sets the state of block to 'free'
       @param   blockAddress [in] I/O adress of the block to change
       @return  none

       - If the state could not be changed successfully the kernel aborts
     */

    inline void SetBlockStateToFree( const IOMan_BlockAddress &blockAddress )
    {
        SetBlockStateToFree( GetTaskId(), blockAddress );
    }

    /*!
       @brief          Sets the state of block to 'free after completion of the next save point'
       @param          TaskId [in] task id
       @param          block [in] I/O adress of the block to change
       @return         none

       - If the state could not be changed successfully the kernel aborts
     */

    void SetBlockStateToFreeAfterSVP(
        const tsp00_TaskId       TaskId,
        const IOMan_BlockAddress &block );

    /*!
       @brief   Sets the state of block to 'free after completion of the next save point'
       @param   blockAddress [in] I/O adress of the block to change
       @return  none

       - If the state could not be changed successfully the kernel aborts
    */

    inline void SetBlockStateToFreeAfterSVP( const IOMan_BlockAddress &blockAddress )
    {
        SetBlockStateToFreeAfterSVP( GetTaskId(), blockAddress );
    }

    /*!
       @brief          all memory resources are released and members set to their initial values
       @param          taskId [in] task id
       @return         one
     */
    void Shutdown( const tsp00_TaskId taskId );

    /*!
       @brief          Supplies a set of free and neighbouring blocks
       @param          taskId [in] task id
       @param          NumFreeBlocksWanted [in] wanted number of blocks
       @param          bReqSequential [in] request access to sequential volume
       @return         (IOMan_ClusterAddress) address of the supplied cluster

       - Supplies a set of free and neighbouring blocks and sets these blocks
      to the state 'occupied'
     */
    IOMan_ClusterAddress GetMultFreeBlocks(
        const tsp00_TaskId        taskId,
        const IOMan_BlockCount    NumFreeBlocksWanted,
        const bool                bReqSequential );

    IOMan_ClusterAddress GetFreeCluster(
        const tsp00_TaskId        taskId,
        const IOMan_BlockCount    clusterSize);

    /*!
       @brief          Supplies a free block
       @param          taskId [in] task id
       @param          bReqSequential [in] request access to sequential volume
       @return         (IOMan_BlockAddress) block address of the supplied block

       - Supplies a single free block and marks this block as 'occupied'
      if no free block could be found than an emergency shutdown will
    be executed.
     */
    IOMan_BlockAddress GetFreeBlock(
        const tsp00_TaskId  taskId ,
        const bool          bReqSequential );

    /*!
       @brief          All Blocks marked as 'free after savepoint' are released
       @param          TaskId [in] task id
       @return         none
     */
    void SetAllBlocksMarkedAsFreeAfterSVPToFree( const tsp00_TaskId TaskId );

    /*!
       @brief          Removes for all blocks the flags indicating that this block must be saved
       @param          TaskId [in] task id
       @return         none

       - The original state of the blocks is restored.
     */
    void RestoreAllBlockStatesMarkedForBackup( const tsp00_TaskId TaskId );

    /*!
       @brief          Removes the flags indicating that a block must be saved
       @param          TaskId [in] task id
       @param          block [in] address of the block to restore
       @return         true if the block was really marked for back up

       - The original state of the block is restored.
       - If the state could not be restored successfully since the block to be restored
      was not at all marked for backup the kernel aborts
     */
    void RestoreBlockStateMarkedForBackup (
        const tsp00_TaskId        TaskId,
        const IOMan_BlockAddress &block );

    /*!
       @brief          Returns the number of all blocks marked for back up
       @param          taskId [in] task id
       @return         number of all blocks marked for back up
     */
    IOMan_BlockCount NumBlocksMarkedForBackup( const tsp00_TaskId taskId ) const;

    /*!
       @brief          The iterator supplying the blocks marked for back up is set to the first block
       @param          TaskId [in] task id
       @return         none

       - This function has to be called prior to the call of the function
      GetNextBlocksForBackUp which returns each time it is executed another
    block marked for backup. The function BeginReadingBlocksMarkedForBackUp
    initializes this iterator and sets it onto the very first block marked for backup
     */
    void BeginReadingBlocksMarkedForBackUp( const tsp00_TaskId TaskId );

    /*!
       @brief          Returns the address of a set of neighbouring blocks marked for back up
       @param          taskId [in] task id
       @param          MaxNumBlocksWanted [in] maximum number of neighbouring blocks supplied 
                                       i.e. SuppliedNumBlocks &lt;= MaxNumBlocksWanted
       @param          SuppliedNumBlocks [out] number of adjacent marked blocks found
       @param          volNo [out] number of the volume of the supplied set
       @param          BlockNo [out] address of the first block of the supplied set
       @param          trError [out] errors state of the function. if everything worked 
                                       fine it is e_ok
       @return         none

       - Each time this function is called it returns another set of blocks marked for backup.
      The function  BeginReadingBlocksMarkedForBackUp initializes this iterator and sets it
    onto the very first block marked for backup. After the initialization this function
    returns by each call an other block untill all marked blocks where returned. That there
    are no more blocks left is displayed by a return value 0 for the parameter SuppliedNumBlocks 
     */
    void  GetNextBlocksForBackUp (
        const tsp00_TaskId taskId,
        const IOMan_BlockCount   MaxNumBlocksWanted,
        IOMan_BlockCount        &SuppliedNumBlocks,
        IOMan_VolumeNo          &volNo,
        IOMan_BlockNo           &BlockNo,
        tgg00_BasisError        &trError);

    /*!
       @brief          inserts all important memory structures of the FBM into the 
                       kernel dump file
       @param          taskId [in] identification of the calling task
       @param          dump [in/out] kernel dump file
       @return         none
     */
    void Dump(
        const tsp00_TaskId  taskId,
        Kernel_Dump         &dump ) const;

    /*!
       @brief          Returns the number of all used blocks on a device
       @param          volNo [in] number of the device for which the number of used blocks is requested
       @return         number of all blocks used

       - Returns the number of all used blocks on a device. A block is considered to be used if it is not in the state free.
     */
    IOMan_BlockCount NumBlocksUsed( const IOMan_VolumeNo volNo ) const
    {
        SAPDBERR_ASSERT_ARGUMENT( m_DataVolumes.IsAvailable( volNo ));

        return (m_DataVolumes [volNo].GetNumBlocksUsed());
    }

    /*!
       @brief          Returns the number of blocks marked for backup on a volume
       @param          volNo [in] number of the volume for which the number of backup blocks is requested
       @return         number of backup blocks

       - Returns the number of all blocks marked for backup a volume.
     */
    IOMan_BlockCount GetNumBlocksMarkedForBackup (const IOMan_VolumeNo volNo ) const
    {
        SAPDBERR_ASSERT_ARGUMENT( m_DataVolumes.IsVolumeNoValid( volNo ));

        if( ! m_DataVolumes.IsRegistered( volNo )) {
            return 0;  // volume is not ready
        }
        else{
            return (m_DataVolumes [volNo].GetNumBlocksMarkedForBackup());
        }
    }

	IOMan_BlockCount GetTotalClusterAreaSize (const IOMan_VolumeNo volNo ) const
	{
		SAPDBERR_ASSERT_ARGUMENT( m_DataVolumes.IsVolumeNoValid( volNo ));

		if( ! m_DataVolumes.IsRegistered( volNo )) {
			return 0;  // volume is not ready
		}
		else{
			return (m_DataVolumes [volNo].GetTotalClusterAreaSize());
		}
	}
	
	IOMan_BlockCount GetReservedClusterAreaSize (const IOMan_VolumeNo volNo ) const
	{
		SAPDBERR_ASSERT_ARGUMENT( m_DataVolumes.IsVolumeNoValid( volNo ));

		if( ! m_DataVolumes.IsRegistered( volNo )) {
			return 0;  // volume is not ready
		}
		else{
			return (m_DataVolumes [volNo].GetReservedClusterAreaSize());
		}
	}
	
	IOMan_BlockCount GetUsedClusterAreaSize (const IOMan_VolumeNo volNo ) const
	{
		SAPDBERR_ASSERT_ARGUMENT( m_DataVolumes.IsVolumeNoValid( volNo ));

		if( ! m_DataVolumes.IsRegistered( volNo )) {
			return 0;  // volume is not ready
		}
		else{
			return (m_DataVolumes [volNo].GetUsedClusterAreaSize());
		}
	}
		
    /*!
       \brief          dump all used blocks to knldiag
       \param          volNo [in] volume number
       \return         none

           used for diagnostics in drop volume
     */
    void DumpUsedBlocks(const IOMan_VolumeNo volNo) const
    {
        m_DataVolumes [volNo].DumpUsedBlocks();
    }

    /*!
       @brief          Returns the volume mode of a device
       @param          volNo [in] number of the device for which the volume mode is requested
       @return         volume mode

       - Returns the volume mode of a device.
     */
    RTE_VolumeAccessMode GetVolMode( const IOMan_VolumeNo volNo ) const
    {
        SAPDBERR_ASSERT_ARGUMENT( m_DataVolumes.IsAvailable( volNo ));

        return (m_DataVolumes [volNo].VolMode());
    }

    /*!
       @brief          Returns the number of all blocks in state freeAfterSavepoint
       @return         (IOMan_BlockCount) Number of pending free blocks
     */

    IOMan_BlockCount GetNumberOfFreeAfterSVPBlocks() const
    {
        return m_TotalNumBlocksFreeAfterSVP;
    }
    
    /*!
       @brief          Returns the number of all blocks to backup in state freeAfterSavepoint
                       which will be deleted after the blocks are written.
       @return         (IOMan_BlockCount) Number of pending free blocks still used for backup
     */

    IOMan_BlockCount GetNumberOfBackupBlocksFreeAfterSVP() const
    {
        return m_TotalNumBlocksBackupFreeAfterSVP;
    }


    /*!
       @brief          Returns the number of blocks in state free. Note that the
                       blocks in state free after savepoint are treated as blocks
                       in state occupied, because the are not available at this 
                       moment.
       @return         (IOMan_BlockCount) Number of free blocks
     */

    IOMan_BlockCount GetNumberOfFreeBlocks() const
    {
        return( m_TotalNumBlocksFree );
    }

    /*!
       @brief          Returns the number of all used blocks. Used means all blocks
                       not in state free. Note that blocks in state free after savepoint 
                       are handled as occupied blocks because there are not available
                       at this moment.
       @return         (IOMan_BlockCount) Number of used blocks
     */
    IOMan_BlockCount GetNumberOfUsedBlocks() const
    {
        return( m_TotalNumBlocks - GetNumberOfFreeBlocks() );
    }

    /*!
       @brief          Checks if the given number of blocks is storable within the
                       data volumes.
       @param          numBlocksRequested [in] number of blocks to be written into
                       the data volumes.
       @return         true is given number of blocks is storable within data
     */
    bool IsSpaceAvailable( const IOMan_BlockCount numBlocksRequested ) const
    {
        return(( numBlocksRequested + GetNumberOfUsedBlocks()) < m_TotalNumBlocks );
    }

    /*!
       @brief          If the data volume filling is about 90 percent and the fbm
                       is in online mode, this method returns true to signalize that 
                       garbage collector support is needed.
       @param          numChangedPages [in] number of changed pages within data cache
                       and converter
       @return         true if garbage collector support is needed; else false
     */
    bool GarbageCollectionNeeded( const SAPDB_Int4 numChangedPages ) const
    {
        if( ! m_Active )
            return( false );

        if( m_SavepointIsRequested )
            return( false );

        // data base filling is under 90 percent inclusive the changed pages of the data cache
        if(( GetNumberOfUsedBlocks() + numChangedPages ) < ( m_TotalNumBlocks * 0.9 ))
            return( false );

        return( true );
    }

    /*!
       @brief          If the number of blocks in state free after savepoint is greater
                       than the number of blocks in state free a savepoint is needed.
       @param          bExtendedCheck [in] check also if more then 10 percent of the
                                           whole database size is in state FreeAFterSavepoint
       @return         true if a savepoint is needed; else false
     */

    bool SavepointNeeded() const
    {
        if( ! m_Active )
            return false;

        if( m_SavepointIsRequested )
            return false;

        // more blocks ready to be free than realy free blocks
        if( m_TotalNumBlocksFree < m_TotalNumBlocksFreeAfterSVP )
            return true;

        return false;
    }

    /*!
        @brief  Returns true if the number of blocks in state free after savepoint is greater than zero.
        @return (bool) 
     */

    bool FreeAfterSVPExist() const
    {
        return 0 < m_TotalNumBlocksFreeAfterSVP;
    }

    /*!
       @brief   Checks whether a task is suspended because of db full
       @return  true is given if task is suspended
     */
    bool IsDBFull() const
    {
        return ! m_DBFullWaitQueue.IsEmpty();
    }


private:

    FBM_Manager(SAPDBMem_IRawAllocator &Allocator);

    IOMan_VolumeNo GetFreeVolumeInGroup(
        const SAPDB_UInt        CurrentVolumeGroup,
        const IOMan_BlockCount  NumRequestedBlocks);

    IOMan_VolumeNo GetDeviceWithFreeBlocks (
        const IOMan_BlockCount  NumRequestedBlocks,
        const bool              bReqSequential);

    IOMan_VolumeNo GetVolumeWithFreeCluster();
    IOMan_VolumeNo SearchVolumeWithFreeCluster();
    bool ReserveAddClusterSpace(const SAPDB_UInt4 numCluster);
    bool ReserveClusterSpace(const SAPDB_UInt4 numCluster);
    
    IOMan_BlockCount OptimumNumUsedBlocksPerDevice ();

    void ChangeDeviceUsagePreferences();

    void ResetMembersToNullAndZero();

   bool SetBlockStateToOccupiedUnsynched( const IOMan_BlockAddress &block );

    bool SetBlockStateToBackupUnsynched( const IOMan_BlockAddress &block );

private:

    tsp00_TaskId GetTaskId()
    {
        tsp00_TaskId taskId;
        vgetpid( taskId );
        return( taskId );
    }


    class FBM_SynchObject
    {
    private:
        tsp00_TaskId m_TaskId;
    public:
        FBM_SynchObject(tsp00_TaskId TaskId) :m_TaskId(TaskId)
        {
            vbegexcl (m_TaskId, g08fbm);
        };

        ~FBM_SynchObject()
        {
            vendexcl (m_TaskId, g08fbm);
        };
    };

private:

    RTEMem_AllocatorWrapper         m_Allocator;
    IOMan_BlockCount                m_TotalNumBlocksFreeAfterSVP;
    IOMan_BlockCount                m_TotalNumBlocksBackupFreeAfterSVP;
    IOMan_BlockCount                m_TotalNumBlocksFree;
    IOMan_BlockCount                m_TotalNumBlocks;
    IOMan_BlockCount                m_ClusterSize;
    IOMan_VolumeNo                  m_NumDev;
    IOMan_VolumeNo                  m_NextDevToUseForNewBlock;
    IOMan_VolumeNo                  m_NextDevToUseForFragBlock;
    IOMan_VolumeNo                  m_NextDevToUseForBackup; // for save data or save pages
    IOMan_VolumeNo                  m_NextVolToUseForNewCluster;
    IOMan_VolumeNo                  m_LastArchiveVolumeUsed;
    IOMan_VolumeNo                  m_ParallelWritingArchiveVolumes;

    /// number of Clusters reserved
    IOMan_ClusterCount              m_reservedClusters;

    FBM_DataVolumeArray             m_DataVolumes;
    static FBM_Manager             *m_Instance;
    bool                            m_SavepointIsRequested;
    bool                            m_Active;
    IOMan_ReservedBlockAddress      m_ReservedBlocks;
    Container_Vector<IOMan_VolumeNo>  m_ArchiveVolumes;
    Container_Vector<SAPDB_UInt>    m_ArchiveVolumeGroups;
    SAPDB_UInt                      m_CurArchiveVolumeGroup;
    SAPDB_UInt                      m_NumberVolumeGroups;

    Kernel_SynchronizedDBFullList   m_DBFullWaitQueue;
};


#endif //FBM_MANAGER_HPP
