/*!

  @file           FBM_ClusteredFreeList.hpp
  @author         Henrik
  @ingroup        FBM

\if EMIT_LICENCE
    ========== licence begin  GPL
    Copyright (c) 2002-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end

\endif
*/

#ifndef FBM_CLUSTEREDFREELIST_H
#define FBM_CLUSTEREDFREELIST_H

/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "IOManager/IOMan_Types.hpp"
#include "Container/Container_AVLTree.hpp"
#include "Container/Container_Vector.hpp"

/*===========================================================================*
 *  CLASSES, STRUCTURES, TYPES, UNIONS ...                                   *
 *===========================================================================*/

/*!
  @class          FBM_ClusteredFreeList
  @brief          simple free cluster management
 */
class FBM_ClusteredFreeList
{

public:
    
    /*!
        \brief     constructor of free cluster management
        \param     Allocator [in] allocator for internal data structures
        \param     VolumeSize [in] number of blocks in volume
        \param     MaxClusterSize [in] maximum size a cluster
    */
    FBM_ClusteredFreeList (SAPDBMem_IRawAllocator &Allocator,
                    IOMan_BlockCount VolumeSize,
                    IOMan_BlockCount MaxClusterSize) :
                    m_ListComparator(),
					m_MaxClusterSize(MaxClusterSize),
					m_ClusterSizeList(Allocator),
                    m_FreeList(&m_ListComparator, &Allocator)
    {
		ClusterList clusterList(Allocator);
		m_ClusterSizeList.Initialize(m_MaxClusterSize+1,clusterList);
	}

    /*!
        \brief   add a single block
        \param   BlockNo [in] number of block to add

        search for neighboring clusters to add the block to, if no clusters are found a new cluster is created
    */
    void AddFreeBlock(IOMan_BlockNo BlockNo);

    /*!
        \brief add a cluster to the cluster management
        \param    Start [in] start address of the cluster
        \param    Len   [in] length of the cluster 
    */
    void AddFreeCluster(IOMan_BlockNo Start, IOMan_BlockCount Len);

    /*!
        \brief     remove a single block from the cluster management
        \param     BlockNo [in] block number
    */
    void RemoveFreeBlock(IOMan_BlockNo BlockNo);

    /*!
        \brief     remove a cluster from the cluster management
        \param     Start [in] starting block number of the cluster
        \param     Len   [in] length of the cluster
    */
    void RemoveFreeCluster(IOMan_BlockNo Start, IOMan_BlockCount Len);

    /*!
        \brief     searches for a cluster of a given length
        \param     WantedLen [in] requested length of a cluster
        \param     BlockNo   [out] starting address of a found cluster
        \param     Len       [out] length of a found cluster, always &lt;= WantedLen
    */
    void SearchCluster(IOMan_BlockCount WantedLen, IOMan_BlockNo &BlockNo, IOMan_BlockCount &Len);
    /*!
        \brief    searches for a single block
        \param    BlockNo [out] number of the found block
    */
    void SearchBlock(IOMan_BlockNo &BlockNo);

    /// remove all clusters / deinit
	void RemoveAll();

private:
    
    typedef Container_AVLNodeComparator<IOMan_BlockNo> ListComparator;
    typedef Container_AVLContentNode<IOMan_BlockNo, IOMan_BlockCount, ListComparator> ListNode;
    typedef Container_AVLContentTree<ListNode, IOMan_BlockNo, IOMan_BlockCount, ListComparator> ListTree;

	typedef Container_Vector<IOMan_BlockNo> ClusterList;  // list of equal sized clusters
	typedef Container_Vector<ClusterList> ClusterSizeList;    // list of ClusterLists, one for each size

	void AddToClusterList(IOMan_BlockNo Start, IOMan_BlockCount Len);
	void RemoveFromClusterList(IOMan_BlockNo Start, IOMan_BlockCount Len);


    ListTree m_FreeList;
    ListComparator m_ListComparator;

	ClusterSizeList m_ClusterSizeList;

	IOMan_BlockCount m_MaxClusterSize;

};

#endif //FBM_CLUSTEREDFREELIST_H
