/*!**************************************************************************

  module      : FBM_ClusteredFreeList.cpp

  -------------------------------------------------------------------------

  responsible : Henrik
  author      : Henrik

  special area: FreeBlockManagement (FBM)
  description : 

  last changed: 2000-03-10  10:00
  see also    : 

  -------------------------------------------------------------------------

  copyright:    Copyright (c) 2000-2005 SAP AG



    ========== licence begin  GPL
    Copyright (c) 2000-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


*****************************************************************************/



/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "FreeBlockManagement/FBM_ClusteredFreeList.hpp"
#include "FreeBlockManagement/FBM_Exception.hpp"

/*===========================================================================*
 *  DEFINES                                                                   *
 *===========================================================================*/

/*===========================================================================*
 *  CLASS METHOD DEFINITION                                                  *
 *===========================================================================*/

/*---------------------------------------------------------------------------*
 *  CLASS FBM_ClusteredFreeList                                              *
 *---------------------------------------------------------------------------*/

/* add cluster

if len == maxlen
  check collision -> bum
     insert()
  return
if len > maxlen
    addCluster(Blockstart, maxlen)
    addCluster(BlockStart+maxlen, len - maxlen)
  return
else // len < maxlen
  check collision -> bum
  check Left neighbor (no gap, neighborLen < 64)
    remove left neighbor
	addCluster (neighborStart, neighborlen+len)
	return
  else check right neighbor (no gap, neighborlen < 64)
    remove righr neighbor
	addCluster (Start, len + neighborlen)
	return
  else (no neighbor)
  insert()

*/

void
FBM_ClusteredFreeList::AddFreeCluster(IOMan_BlockNo Start, IOMan_BlockCount ClusterLen)
{

	SAPDBTRACE_WRITELN( FBM_Trace, 5, "Add Cluster: " << Start << "/" << ClusterLen);

    while (ClusterLen > 0)
    {

	if (ClusterLen >= m_MaxClusterSize) // cluster has optimal length, do not add to left or right neighbor
	{
        IOMan_BlockCount Len = m_MaxClusterSize;
        
		// check for collisions with left&right neighbors
		// search cluster in List
		ListTree::Iterator search = m_FreeList.Locate(Start, AVLTree_DESCENDING);

		if (search) // found same or left node
		{
			if ((*search()->GetKey() + *search()->GetContent()) > Start)  // new cluster overlaps with left neighbor
			{
				FBM_Exception errMsg( __CONTEXT__, FBM_ERR_BLOCKNO_ALREADY_FREE, SAPDB_ToString (Start, _T_d),
				SAPDB_ToString (*search()->GetKey(), _T_d),
				SAPDB_ToString (*search()->GetContent(), _T_d)	);
				RTE_Crash( errMsg );
			}
			else // try next cluster
			{
				search++;
			}
		}
		else // nothing found to the left
		{
			search =  m_FreeList.Locate(Start, AVLTree_ASCENDING);
		}

		if (search) // found right cluster
		{
			if ((Start + Len) > *search()->GetKey())  // new cluster overlaps with right neighbor
			{
				FBM_Exception errMsg( __CONTEXT__, FBM_ERR_BLOCKNO_ALREADY_FREE, SAPDB_ToString (Start, _T_d),
				SAPDB_ToString (*search()->GetKey(), _T_d),
				SAPDB_ToString (*search()->GetContent(), _T_d)	);
				RTE_Crash( errMsg );
			}
		}

		// no collisions -> add to tree
		Container_AVLTreeError errCode;
		m_FreeList.Insert(Start, Len, errCode);
		AddToClusterList(Start, Len);

        ClusterLen -= Len;
        Start += Len;
	}
	else // Len < m_MaxClusterSize / short cluster, try to merge with left and/or right neighbor
	{
        IOMan_BlockCount Len = ClusterLen;

		ListTree::Iterator search = m_FreeList.Locate(Start, AVLTree_DESCENDING);

		if (search) // found same or left node
		{
			if ((*search()->GetKey() + *search()->GetContent()) > Start)  // new cluster overlaps with left neighbor
			{
				FBM_Exception errMsg( __CONTEXT__, FBM_ERR_BLOCKNO_ALREADY_FREE, SAPDB_ToString (Start, _T_d),
				SAPDB_ToString (*search()->GetKey(), _T_d),
				SAPDB_ToString (*search()->GetContent(), _T_d)	);
				RTE_Crash( errMsg );
			}
			else if ((((*search()->GetKey() + *search()->GetContent())) == Start) &&
			         (*search()->GetContent() != m_MaxClusterSize))  // no overlap, direct neighbor -> merge
			{
				IOMan_BlockNo leftCluster = *search()->GetKey();
				IOMan_BlockCount leftClusterLen = *search()->GetContent();
				RemoveFromClusterList(leftCluster, leftClusterLen);
				m_FreeList.Delete(search); // remove left neighbor

				AddFreeCluster( leftCluster, leftClusterLen + Len); // add left + new cluster

                ClusterLen -= Len;
                break;
			}
			// else continue with right neighbor
		}

		search = m_FreeList.Locate(Start, AVLTree_ASCENDING);
		if (search) // found same or right node
		{
			if (Start + Len > *search()->GetKey())  // new cluster overlaps with right neighbor
			{
				FBM_Exception errMsg( __CONTEXT__, FBM_ERR_BLOCKNO_ALREADY_FREE, SAPDB_ToString (Start, _T_d),
				SAPDB_ToString (*search()->GetKey(), _T_d),
				SAPDB_ToString (*search()->GetContent(), _T_d)	);
				RTE_Crash( errMsg );
			}
			else if (( Start + Len == *search()->GetKey()) &&
			         (*search()->GetContent() != m_MaxClusterSize))
			{
				IOMan_BlockCount rightClusterLen = *search()->GetContent();

				RemoveFromClusterList(*search()->GetKey(), rightClusterLen);
				m_FreeList.Delete(search); // remove right neighbor

				AddFreeCluster( Start, Len + rightClusterLen); // add new cluster + right 

                ClusterLen -= Len;
                break;
			}
			// else add cluster direct, no merge done
		}

		Container_AVLTreeError errCode;
		m_FreeList.Insert(Start, Len, errCode);

		AddToClusterList(Start, Len);
        ClusterLen -= Len;
        Start += Len;
	}
    }
}

void
FBM_ClusteredFreeList::AddFreeBlock(IOMan_BlockNo Block)
{
	AddFreeCluster(Block, 1);
}

void
FBM_ClusteredFreeList::RemoveFreeBlock(IOMan_BlockNo BlockNo)
{
// search BlockNo in List
SAPDBTRACE_WRITELN( FBM_Trace, 5, "Rem Block: " << BlockNo);

	ListTree::Iterator search = m_FreeList.Locate(BlockNo,AVLTree_DESCENDING);

	if (search) // found same or left node
	{
		if (*search()->GetKey() == BlockNo) // found BlockNo as first block of a cluster
		{
			// del found cluster
			IOMan_BlockCount Len = *search()->GetContent();
			RemoveFromClusterList(BlockNo, Len);
			m_FreeList.Delete(BlockNo);

			if (Len > 1)
			{
				// add cluster with len-1 and block+1
				Container_AVLTreeError errCode;
				m_FreeList.Insert(BlockNo+1, Len-1, errCode);
				AddToClusterList(BlockNo+1, Len-1);
			}
		}
		else 
		{
			if ((*search()->GetKey() + *search()->GetContent()) > BlockNo)  // block is inside cluster 
			{
				IOMan_BlockNo SearchStart = *search()->GetKey();
				IOMan_BlockCount Len = *search()->GetContent();

				// reduce len of left cluster fragment
				*search()->SetContent( BlockNo - SearchStart);
				RemoveFromClusterList(SearchStart, Len);
				AddToClusterList(SearchStart, BlockNo - SearchStart);

				// check remaining
				if (BlockNo != *search()->GetKey() + Len -1)	// is !last block of cluster?
				{
					// add right fragment of cluster
					Container_AVLTreeError errCode;
					m_FreeList.Insert(BlockNo+1, Len - (BlockNo + 1 - *search()->GetKey()), errCode);
					AddToClusterList(BlockNo+1, Len - (BlockNo + 1 - *search()->GetKey()));
				}
			}
			else 
			{   // block not in freelist
				FBM_Exception errMsg( __CONTEXT__, FBM_ERR_BLOCKNO_NOT_IN_FREELIST, SAPDB_ToString (BlockNo, _T_d));
				RTE_Crash( errMsg );
			}
		}
	}
	else // not found
	{   // block not in freelist
		FBM_Exception errMsg( __CONTEXT__, FBM_ERR_BLOCKNO_NOT_IN_FREELIST, SAPDB_ToString (BlockNo, _T_d));
		RTE_Crash( errMsg );
	}
}

void
FBM_ClusteredFreeList::RemoveFreeCluster(IOMan_BlockNo Start, IOMan_BlockCount Len)
{
	// not implemented
    SAPDBERR_ASSERT_STATE( SAPDB_FALSE );
}

void
FBM_ClusteredFreeList::SearchCluster(IOMan_BlockCount WantedLen, IOMan_BlockNo &BlockNo, IOMan_BlockCount &Len)
{
	SAPDB_Bool found = SAPDB_FALSE;

	ClusterSizeList::Iterator iter = m_ClusterSizeList.Position(WantedLen);
	ClusterSizeList::Iterator iterEnd = m_ClusterSizeList.End();

	// search clusters with equal or longer length 
	while(!found && (iter != iterEnd))
	{
		if ((*iter).GetSize() != 0)
		{
			found = SAPDB_TRUE;
			Len = m_ClusterSizeList.GetIndex(iter);
			BlockNo = (*iter)[(*iter).GetSize()-1];  // last element
		}
		else
		{
			iter++;
		}
	}

	if (!found && (WantedLen >1)) // search downward for a shorter cluster
	{
		iter = m_ClusterSizeList.Position(WantedLen-1);
		iterEnd = m_ClusterSizeList.Begin();

		while(!found && (iter != iterEnd))
		{
			if ((*iter).GetSize() != 0)
			{
				found = SAPDB_TRUE;
				Len = m_ClusterSizeList.GetIndex(iter);
				BlockNo = (*iter)[(*iter).GetSize()-1];  // last element
			}
			else
			{
				iter--;
			}
		}
	}
    SAPDBERR_ASSERT_STATE( found );
	SAPDBTRACE_WRITELN( FBM_Trace, 5, "search Cluster: " << BlockNo << "/" <<Len);
}

void
FBM_ClusteredFreeList::SearchBlock(IOMan_BlockNo &BlockNo)
{
	IOMan_BlockCount Len;

	SearchCluster(1, BlockNo, Len);
}

// doesnt check for duplicates
void
FBM_ClusteredFreeList::AddToClusterList(IOMan_BlockNo Start, IOMan_BlockCount Len)
{
	ClusterList &thisCluster = m_ClusterSizeList[Len];

	// check capacity
	if (thisCluster.IsFull())
	{
		thisCluster.Reserve(thisCluster.GetCapacity() + 5);
	}

	thisCluster.InsertEnd(Start);
}

void
FBM_ClusteredFreeList::RemoveFromClusterList(IOMan_BlockNo Start, IOMan_BlockCount Len)
{
	SAPDB_Bool found = SAPDB_FALSE;
	ClusterList &thisCluster = m_ClusterSizeList[Len];
	ClusterList::Iterator iter = thisCluster.Begin();
	ClusterList::Iterator iterEnd = thisCluster.End();

	while(iter != iterEnd)
	{
		if (*iter == Start)
		{
			thisCluster.Delete(iter);
			found = SAPDB_TRUE;
			break;
		}
		iter++;
	}
    SAPDBERR_ASSERT_STATE( found );
}

void
FBM_ClusteredFreeList::RemoveAll()
{
	ClusterSizeList::Iterator iter = m_ClusterSizeList.Begin();
	ClusterSizeList::Iterator iterEnd = m_ClusterSizeList.End();

	while(iter != iterEnd){
		(*iter).Clear();
		iter++;
	}
	m_FreeList.DeleteAll();

}
