/*!

  @file           FBM_ClusteredDataVolume.hpp
  @author         Henrik
  @ingroup        FBM

\if EMIT_LICENCE
    ========== licence begin  GPL
    Copyright (c) 2002-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end

\endif
*/

#ifndef FBM_CLUSTEREDDATAVOLUME_H
#define FBM_CLUSTEREDDATAVOLUME_H

/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "FreeBlockManagement/FBM_DataVolume.hpp"
#include "FreeBlockManagement/FBM_ClusteredFreeList.hpp"

/*===========================================================================*
 *  CLASSES, STRUCTURES, TYPES, UNIONS ...                                   *
 *===========================================================================*/

/*!
  @class          FBM_ClusteredDataVolume
  @brief          representation of one data volume
 */
class FBM_ClusteredDataVolume : public FBM_DataVolume
{
private: 

    FBM_ClusteredFreeList m_ClusterFreeList;

    void Verify ();

public:
    
    /*!
        \brief   constructor for FBM_ClusteredDataVolume 
        \param   Allocator [in] allocator
        \param   VolumeSize [in] number of blocks in volume
        \param   VolMode    [in] volume mode
        \param   VolNo      [in] volume number (for debugging)
    */
    FBM_ClusteredDataVolume (SAPDBMem_IRawAllocator &Allocator,
                    IOMan_BlockCount        VolumeSize,
                    RTE_VolumeAccessMode    VolMode,
                    IOMan_VolumeNo          VolNo)
        :
        FBM_DataVolume(Allocator, VolumeSize, VolMode, VolNo),
        m_ClusterFreeList                      (Allocator, VolumeSize, 64)
    {}


    void InitVolume();

    void RestoreAllBlockStatesMarkedForBackup (IOMan_BlockCount &NumBlocksRestoredToFreeAfterSVP);

    void RestoreBlockStateMarkedForBackup (
        IOMan_BlockNo  BlockNo,
        SAPDB_Bool &bRestoredStateIsFreeAfterSVP);
    
    void SetAllBlocksMarkedAsFreeAfterSVPToFree ();

    void GetNextBlocksForBackUp (
        const IOMan_BlockCount  MaxNumBlocksWanted, 
        IOMan_BlockCount        &SuppliedNumBlocks, 
        IOMan_BlockNo           &BlockNo);

    IOMan_BlockNo GetMultFreeBlocks (
        const IOMan_BlockCount  NumBlocksWanted,
        IOMan_BlockCount       &NumBlocksSupplied);

    IOMan_BlockNo GetFreeBlock ();

    void SetBlockStateToFree (IOMan_BlockNo BlockNo)
    {
        m_BlockStateList.SetBlockState (BlockNo, BlockState_Free);

        m_ClusterFreeList.AddFreeBlock(BlockNo);

        --m_NumBlocksUsed;
        
#       ifdef SAPDB_SLOW
        if( FBM_Check.ChecksLevel( 5 ))
            Verify ();
#       endif
    }
    
    void SetBlockStateToOccupied (IOMan_BlockNo BlockNo)
    {
        m_BlockStateList.SetBlockState (BlockNo, BlockState_Occupied);
        ++m_NumBlocksUsed;

        if (m_LastUsedBlock < BlockNo)
        {
            m_LastUsedBlock = BlockNo;
        }
        m_ClusterFreeList.RemoveFreeBlock(BlockNo);

#       ifdef SAPDB_SLOW
        if( FBM_Check.ChecksLevel( 5 ))
            Verify ();
#       endif
    }

    // dummy procedure, used only by FBM_SimpleDataVolume
    bool getFreeCluster(
        const IOMan_BlockCount clusterSize,
        IOMan_BlockNo &clusterStart)
    {
        return false;
    };
    
    // dummy procedure, used only by FBM_SimpleDataVolume
    bool reserveClusterSpace(IOMan_ClusterCount numCluster)
    {
        return false;
    }

    // dummy procedure, used only by FBM_SimpleDataVolume
    IOMan_ClusterCount GetNumClustersFree() const
    {
        return 0;
    }


    void Dump( 
        Kernel_Dump          &dump,
        const IOMan_VolumeNo volNo) const;

    
};

#endif //FBM_CLUSTEREDDATAVOLUME_H
