/*!**************************************************************************

  module      : FBM_ClusteredDataVolume.cpp

  -------------------------------------------------------------------------

  responsible : Henrik

  auhtor      : Henrik

  special area: FreeBlockManagement (FBM)
  description : 


  last changed: 2000-03-10  10:00
  see also    : 

  -------------------------------------------------------------------------

  copyright:    Copyright (c) 2000-2005 SAP AG



    ========== licence begin  GPL
    Copyright (c) 2000-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


*****************************************************************************/



/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "FreeBlockManagement/FBM_ClusteredDataVolume.hpp"
#include "FreeBlockManagement/FBM_Dump.hpp"
#include "FreeBlockManagement/FBM_Exception.hpp"
#include "FreeBlockManagement/FBM_Messages.hpp"
#include "KernelCommon/Kernel_Dump.hpp"
#include "RunTime/RTE_Crash.hpp"
#include "RunTime/RTE_Message.hpp"


/*===========================================================================*
 *  DEFINES                                                                   *
 *===========================================================================*/

/*===========================================================================*
 *  CLASS METHOD DEFINITION                                                  *
 *===========================================================================*/

/*---------------------------------------------------------------------------*
 *  CLASS FBM_ClusteredDataVolume                                              *
 *---------------------------------------------------------------------------*/

void
FBM_ClusteredDataVolume::InitVolume ()
{
    SAPDBTRACE_ROUTINE_DEBUG  ("FBM_ClusteredDataVolume::InitVolume", FBM_Trace, 5);

    FBM_DataVolume::InitVolume();

    /* initialize free cluster list */
    m_ClusterFreeList.RemoveAll();
    m_ClusterFreeList.AddFreeCluster(0, m_MaxBlockNo + 1);

}

/*---------------------------------------------------------------------------*/

void
FBM_ClusteredDataVolume::RestoreAllBlockStatesMarkedForBackup (IOMan_BlockCount &NumBlocksRestoredToFreeAfterSVP)
{
    SAPDBTRACE_ROUTINE_DEBUG  ("FBM_ClusteredDataVolume::RestoreAllBlockStatesMarkedforBackup", FBM_Trace, 5);

    IOMan_BlockNo     iBlock;
    SAPDB_Bool     bRestoredStateIsFreeAfterSVP;

    NumBlocksRestoredToFreeAfterSVP = 0;

    for (iBlock=0; iBlock <= m_MaxBlockNo;  ++iBlock)
    {
        if (GetBlockState (iBlock) == BlockState_BackUp)
        {
            /* restore the state to the one it had before the backup */
            RestoreBlockStateMarkedForBackup (iBlock, bRestoredStateIsFreeAfterSVP);
            if (bRestoredStateIsFreeAfterSVP)
                ++NumBlocksRestoredToFreeAfterSVP;

            if (m_NumBlocksMarkedForBackup == 0) break;
        }
    }
    SAPDBERR_ASSERT_STATE (m_NumBlocksMarkedForBackup == 0);

    /* reset marker of last block read for back up */
    m_ActBlockNoForBackup.Invalidate();

    /* check consistency of the device */
#   ifdef SAPDB_SLOW 
    if( FBM_Check.ChecksLevel( 5 ))
        Verify ();
#   endif
}

/*---------------------------------------------------------------------------*/

IOMan_BlockNo
FBM_ClusteredDataVolume::GetMultFreeBlocks  (const IOMan_BlockCount  NumBlocksWanted,
                                    IOMan_BlockCount       &NumBlocksSupplied)
{
    SAPDBTRACE_ROUTINE_DEBUG  ("FBM_ClusteredDataVolume::GetMultFreeBlocks", FBM_Trace, 5);

    /* this routine is called to get  a cluster of free blocks.  */
    /* before the routine is called it is already obvious that   */
    /* there must be at least one free block on this device      */

    IOMan_BlockNo FreeBlockNo;

    /* find free BlockNo */
    m_ClusterFreeList.SearchCluster(NumBlocksWanted, FreeBlockNo, NumBlocksSupplied);

    if(NumBlocksSupplied > NumBlocksWanted)
    {
        NumBlocksSupplied = NumBlocksWanted;
    }

    /* change state of free block to occupied */
    for (IOMan_BlockNo block = FreeBlockNo; block < FreeBlockNo + NumBlocksSupplied; block++)
    {
        m_BlockStateList.SetBlockState (block, BlockState_Occupied);
        m_ClusterFreeList.RemoveFreeBlock(block);
    }

    /* write found block no into vtrace */
    SAPDBTRACE_WRITELN( FBM_Trace, 5, "FreeBlockNo: " << FreeBlockNo
                        << " #WantedFreeB: "  << NumBlocksWanted
                        << " #FoundFreeB: "  << NumBlocksSupplied);

    /* update counter of used blocks */
    m_NumBlocksUsed += NumBlocksSupplied;

    /* update the counter which controlls the distribution of blocks onto devices */
    m_NumBlocksToAddUntilOptIsReached -= NumBlocksSupplied;

    /* check consistency of the device */
#   ifdef SAPDB_SLOW 
    if( FBM_Check.ChecksLevel( 5 ))
        Verify ();
#   endif

    return FreeBlockNo; // and NumBlocksSupplied
}

/*---------------------------------------------------------------------------*/

IOMan_BlockNo 
FBM_ClusteredDataVolume::GetFreeBlock  ()
{
    SAPDBTRACE_ROUTINE_DEBUG  ("FBM_ClusteredDataVolume::GetFreeBlock", FBM_Trace, 5);
    SAPDBERR_ASSERT_STATE ((m_MaxBlockNo + 1 - m_NumBlocksUsed) > 0);

    /* this routine is called to get the adress of a free block. */
    /* before the routine is called it is already obvious that   */
    /* there must be at least one free block on this device      */

    IOMan_BlockNo FreeBlockNo;

    m_ClusterFreeList.SearchBlock(FreeBlockNo);

    /* check that a free block was found */
    SAPDBERR_ASSERT_STATE (m_BlockStateList.GetBlockState(FreeBlockNo) == BlockState_Free);

    /* change state of free block to occupy */
    m_BlockStateList.SetBlockState (FreeBlockNo,BlockState_Occupied);

    m_ClusterFreeList.RemoveFreeBlock(FreeBlockNo);

    /* write found block no into vtrace */
    SAPDBTRACE_WRITELN( FBM_Trace, 5, "FreeSBlockNo: " << FreeBlockNo);

    /* update counter of used blocks */
    ++m_NumBlocksUsed;

    /* update the counter which controlls the distribution of blocks onto devices */
    --m_NumBlocksToAddUntilOptIsReached;

    /* check consistency of the device */
#   ifdef SAPDB_SLOW 
    if( FBM_Check.ChecksLevel( 5 ))
        Verify ();
#   endif

    return FreeBlockNo;
}

/*---------------------------------------------------------------------------*/

void
FBM_ClusteredDataVolume::SetAllBlocksMarkedAsFreeAfterSVPToFree ()
{
    SAPDBTRACE_ROUTINE_DEBUG  ("FBM_ClusteredDataVolume::SetAllBlocksMarkedAsFreeAfterSVPToFree", FBM_Trace, 5);

    IOMan_BlockNo     iBlock;

    for (iBlock=0; iBlock <= m_MaxBlockNo; iBlock++)
    {
        if (m_BlockStateList.GetBlockState (iBlock) == BlockState_FreeAfterSVP)
        {
            m_BlockStateList.SetBlockState (iBlock,BlockState_Free);

            m_ClusterFreeList.AddFreeBlock(iBlock);

            --m_NumBlocksUsed;
            --m_NumBlocksFreeAfterSVP;
        }
    }
    /* check that no blocks remain in the state free after save point */
    SAPDBERR_ASSERT_STATE (m_NumBlocksFreeAfterSVP == 0);

    /* check consistency of the device */
#   ifdef SAPDB_SLOW 
    if( FBM_Check.ChecksLevel( 5 ))
        Verify ();
#   endif
}

/*---------------------------------------------------------------------------*/


void
FBM_ClusteredDataVolume::GetNextBlocksForBackUp (const IOMan_BlockCount   MaxNumBlocksWanted,
                                        IOMan_BlockCount        &SuppliedNumBlocks,
                                        IOMan_BlockNo        &BlockNo)

{
    SAPDBTRACE_ROUTINE_DEBUG  ("FBM_ClusteredDataVolume::GetNextBlocksForBackUp", FBM_Trace, 5);

    /* initialize output parameter */
    SuppliedNumBlocks = 0;

    if (m_ActBlockNoForBackup.IsInvalid())
    {
        BlockNo.Invalidate();
        return;
    }

    /* search next block marked for back up starting from the current backup block no */
    SAPDB_Bool bBlockFound    = false;

    /* go through current section and find the first block which is marked as back up */
    while (m_ActBlockNoForBackup <= m_MaxBlockNo)
    {
        if (GetBlockState (m_ActBlockNoForBackup) == BlockState_BackUp)
        {
            /* check if the found block is really marked for backup and not for compression */
            SAPDBERR_ASSERT_STATE ((GetBlockStateAfterBackup (m_ActBlockNoForBackup) == BlockState_Occupied) ||
                                   (GetBlockStateAfterBackup (m_ActBlockNoForBackup) == BlockState_FreeAfterSVP));
            bBlockFound = true;
            break;
        }
        ++m_ActBlockNoForBackup;
    }

    /* goto next section which contains blocks marked for backup */
    if (!bBlockFound)
    {
        m_ActBlockNoForBackup.Invalidate();
    }

    /* set return value where the next set of neighbouring blocks marked for back up starts */
    BlockNo = m_ActBlockNoForBackup;

    /* check if there is really a next block which is          */
    /* marked for back up and not yet fetched by an datawriter */
    if (bBlockFound)
    {
        /* check out if the following blocks are also in the state BlockState_BackUp */
        SuppliedNumBlocks = 1;
        ++m_ActBlockNoForBackup;

        while ((SuppliedNumBlocks      <  MaxNumBlocksWanted) &&
                (m_ActBlockNoForBackup  <= m_MaxBlockNo)    &&
                (GetBlockState (m_ActBlockNoForBackup) == BlockState_BackUp))
        {

            /* check if the found block is really marked for backup and not for compression */
            SAPDBERR_ASSERT_STATE ((GetBlockStateAfterBackup (m_ActBlockNoForBackup) == BlockState_Occupied) ||
                                   (GetBlockStateAfterBackup (m_ActBlockNoForBackup) == BlockState_FreeAfterSVP));
            ++ SuppliedNumBlocks;
            ++ m_ActBlockNoForBackup;
        }

        /* set marker to display that there are no more blocks for backup */
        if (m_ActBlockNoForBackup > m_MaxBlockNo )
            m_ActBlockNoForBackup.Invalidate();
    }

    /* check consistency of the device */
#   ifdef SAPDB_SLOW 
    if( FBM_Check.ChecksLevel( 5 ))
        Verify ();
#   endif
}

/*---------------------------------------------------------------------------*/

void
FBM_ClusteredDataVolume::Verify ()
{
    SAPDBTRACE_ROUTINE_DEBUG  ("FBM_ClusteredDataVolume::Verify", FBM_Trace, 5);

    SAPDBERR_ASSERT_STATE ((m_MaxBlockNo + 1) == m_BlockStateList.Capacity());

    IOMan_BlockCount NumBlocksFree        = 0;
    IOMan_BlockCount NumBlockFreeAfterSVP = 0;

    IOMan_BlockNo FirstFreeBlock                = m_MaxBlockNo + 1;
    IOMan_BlockNo LastUsedBlock                 = 0;
    IOMan_BlockCount NumBlocksMarkedForBackUp   = 0;

    /* go through all entries in the blocklist and count the occurence of each state */
    for (IOMan_BlockNo iBlockNo=0; iBlockNo <= m_MaxBlockNo; ++iBlockNo)
    {
        FBM_BlockState BlockStateAfterBackUp = GetBlockStateAfterBackup(iBlockNo);

        switch (GetBlockState(iBlockNo))
        {
        case BlockState_Free:
            ++NumBlocksFree;
            if (FirstFreeBlock > iBlockNo)
                FirstFreeBlock = iBlockNo;
            SAPDBERR_ASSERT_STATE (BlockState_Free == BlockStateAfterBackUp);
            break;
        case BlockState_Occupied:
            if (LastUsedBlock < iBlockNo)
                LastUsedBlock = iBlockNo;
            SAPDBERR_ASSERT_STATE (BlockState_Free == BlockStateAfterBackUp);
            break;
        case BlockState_FreeAfterSVP:
            ++NumBlockFreeAfterSVP;
            if (LastUsedBlock < iBlockNo)
                LastUsedBlock = iBlockNo;
            SAPDBERR_ASSERT_STATE (BlockState_Free == BlockStateAfterBackUp);
            break;
        case BlockState_BackUp:
            ++ NumBlocksMarkedForBackUp;
            {
                //blocks are marked for back up
                SAPDBERR_ASSERT_STATE ((BlockState_Occupied     == BlockStateAfterBackUp) ||
                                       (BlockState_FreeAfterSVP == BlockStateAfterBackUp));
            }

            if (LastUsedBlock < iBlockNo)
            {
                LastUsedBlock = iBlockNo;
            }
            break;
        default:
            /* write error message into knldiag and vtrace */
            FBM_IllegalState IllegalState(__CONTEXT__, iBlockNo, GetBlockState (iBlockNo), BlockStateAfterBackUp);
            RTE_Crash(IllegalState);
        }
    }

    /* check counters and pointers */
    SAPDBERR_ASSERT_STATE (NumBlocksMarkedForBackUp == m_NumBlocksMarkedForBackup);
    SAPDBERR_ASSERT_STATE (m_NumBlocksUsed == (m_MaxBlockNo + 1 - NumBlocksFree));
    SAPDBERR_ASSERT_STATE (m_NumBlocksFreeAfterSVP == NumBlockFreeAfterSVP);

}

/*---------------------------------------------------------------------------*/

void
FBM_ClusteredDataVolume::RestoreBlockStateMarkedForBackup (IOMan_BlockNo  BlockNo,
        SAPDB_Bool &bRestoredStateIsFreeAfterSVP)
{
    /* check if the state to be restored is correct */
    SAPDBERR_ASSERT_STATE ((GetBlockState             (BlockNo) == BlockState_BackUp)       &&
                           ((GetBlockStateAfterBackup (BlockNo) == BlockState_Occupied) ||
                            (GetBlockStateAfterBackup  (BlockNo) == BlockState_FreeAfterSVP)));

    /* update counter of blocks marked for back up */
    --m_NumBlocksMarkedForBackup;

    /* restore the state to the one it had before the backup. If  */
    /* the restored state (=return value of the restore function) */
    /* is FreeAfterSVP the appropriate counter is updated         */
    if (m_BlockStateList.RestoreBlockStateMarkedForBackup (BlockNo) != BlockState_FreeAfterSVP)
    {
        bRestoredStateIsFreeAfterSVP = false;
    }
    else
    {
        ++m_NumBlocksFreeAfterSVP;
        --m_NumBlocksBackupFreeAfterSVP;
        bRestoredStateIsFreeAfterSVP = true;
    }

#   ifdef SAPDB_SLOW 
    if( FBM_Check.ChecksLevel( 5 ))
        Verify ();
#   endif
}

/*---------------------------------------------------------------------------*/

void
FBM_ClusteredDataVolume::Dump(
    Kernel_Dump         &dump,
    const IOMan_VolumeNo devNo) const
{
    struct FBMDataDevice   fbmDev;

    fbmDev.dmpDevNo                           = devNo;
    fbmDev.dmpNumBlocks                       = m_MaxBlockNo + 1;
    fbmDev.dmpNumBlocksUsed                   = m_NumBlocksUsed;
    fbmDev.dmpNumBlocksMarkedForBackup        = m_NumBlocksMarkedForBackup;
    fbmDev.dmpNumBlocksFreeAfterSVP           = m_NumBlocksFreeAfterSVP;
    fbmDev.dmpNumBlocksBackupFreeAfterSVP     = m_NumBlocksBackupFreeAfterSVP;
    fbmDev.dmpNumBlocksToAddUntilOptIsReached = m_NumBlocksToAddUntilOptIsReached;
    fbmDev.dmpActBlockNoForBackup             = m_ActBlockNoForBackup;
    fbmDev.dmpStartBlockNoToSearchFreeBlock   = 0;
    fbmDev.dmpVolumeToDrop                    = m_IsToDrop;
    fbmDev.dmpNumberOfSections                = 0;
    fbmDev.dmpLastUsedBlock                   = 0;
    fbmDev.dmpFirstFreeBlock                  = 0;
    fbmDev.dmpFiller3                         = false;
    fbmDev.dmpFiller2                         = 0;
    fbmDev.dmpMaxUnclusteredBlockNo           = 0;
    fbmDev.dmpMinClusteredBlockNo             = 0;
    fbmDev.dmpLastUnclusteredUsedBlockNo      = 0;
    fbmDev.dmpUsedUnclusteredBlocks           = 0;
    fbmDev.dmpUsedClusters                    = 0;
    fbmDev.dmpReservedClusters                = 0;
    
    dump.InsertEntry( Kernel_Dump::DmpFBMDataDevice,
                      Kernel_DumpPage::Entry( &fbmDev, sizeof( fbmDev )));
}
