/*
  @file           Converter_PageCollector.cpp
  @author         TorstenS
  @ingroup        Converter
  @brief          This module is used to ...

\if EMIT_LICENCE
    ========== licence begin  GPL
    Copyright (c) 2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end

\endif
*/



/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "Converter/Converter_Types.hpp"
#include "Converter/Converter_PageCollector.hpp"
#include "Converter/Converter_IPageNoMapping.hpp"
#include "DataAccess/Data_BasePage.hpp"
#include "IOManager/IOMan_IDataInfo.hpp"
#include "IOManager/IOMan_ClusterAddress.hpp"
#include "KernelCommon/Kernel_VTrace.hpp"
#include "RunTime/Tasking/RTETask_ITask.hpp"
#include "SAPDBCommon/MemoryManagement/SAPDBMem_IRawAllocator.hpp"
#include "SAPDBCommon/SAPDB_EnhancedTypes.hpp"


/*===========================================================================*
 *  DEFINES                                                                  *
 *===========================================================================*/

typedef SAPDB_Invalidatable<SAPDB_UInt, SAPDB_MAX_UINT4> StartIndex;

/*===========================================================================*
 *  MACROS                                                                   *
 *===========================================================================*/



/*===========================================================================*
 *  LOCAL CLASSES, STRUCTURES, TYPES, UNIONS ...                             *
 *===========================================================================*/



/*===========================================================================*
 *  STATIC/INLINE FUNCTIONS (PROTOTYPES)                                     *
 *===========================================================================*/



/*===========================================================================*
 *  METHODS                                                                  *
 *===========================================================================*/

Converter_PageCollector::Converter_PageCollector(
    SAPDBMem_IRawAllocator&         allocator,
    const Data_PageRecoveryMode&    recMode )
        :
        m_Allocator( allocator ),
        m_RecMode( recMode ),
        m_DataIOBlockCount( IOMan_IDataInfo::GetInstance().GetDataIOBlockCount()),
        m_ClusterCount( 0 ),
        m_PageNoList( m_Allocator ),
        m_ClusterIDList( m_Allocator ),
        m_BlockCount( m_Allocator ),
        m_BlockList( m_Allocator )
{}

/*---------------------------------------------------------------------------*/

void
Converter_PageCollector::DetermineCluster( RTETask_ITask&   task )
{
    m_ClusterCount = 0;
    if( 0 == m_PageNoList.GetSize()){ // no pageNo => no cluster
        return;
    }
    Converter_IPageNoMapping&   iConv = Converter_IPageNoMapping::Instance();

    SAPDB_UInt    currClusterId = 0;
    StartIndex    startClusterIndex; // SAPDB_Invalidatable !

    Data_PageNoList::Iterator       iter    = m_PageNoList.Begin();
    Data_PageNoList::ConstIterator  endIter = m_PageNoList.End();

    for( ; iter != endIter; iter++ )
    {
        const SAPDB_UInt    index  = m_PageNoList.GetIndex( iter );
        const Data_PageNo   pageNo = *iter;
        IOMan_BlockAddress  block;

        const Converter_ReturnCode  retCode = iConv.GetPositionForReading( task, pageNo, m_RecMode, block );

        if( Converter_Ok != retCode ){ // page  could be reserved or corrupted etc.
            block.Invalidate();
        }
        m_BlockList[ index ] = block;
        if( ! block.IsValid())
        {
            startClusterIndex.Invalidate(); // cluster end
            m_ClusterIDList[ index ] = 0;   // belongs to no cluster
            continue;
        }
        if( startClusterIndex.IsInvalid())
        {
            startClusterIndex        = index;           // first page of a new cluster
            m_ClusterIDList[ index ] = ++currClusterId; // set unique cluster id
            m_BlockCount[ index ]    = 1;               // count number of cluster items
            ++m_ClusterCount;                           // count number of clusters
            continue;
        }
        IOMan_BlockAddress& prevBlock = m_BlockList[ index-1 ];
        SAPDBERR_ASSERT_STATE( prevBlock.IsValid()); // because startClusterIndex is valid
        if(
            ( prevBlock.GetVolumeNo()  != block.GetVolumeNo()) ||
            ( prevBlock.GetBlockNo()+1 != block.GetBlockNo())  ||
            ( m_BlockCount[ startClusterIndex ] >= m_DataIOBlockCount )
        )
        {
            startClusterIndex        = index; // first page of a new cluster
            m_ClusterIDList[ index ] = ++currClusterId;
            m_BlockCount[ index ]    = 1;
            ++m_ClusterCount;
            continue;
        }
        // contains to the same cluster
        m_ClusterIDList[ index ] = currClusterId;
        m_BlockCount[ startClusterIndex ]+=1; // increase number of cluster items
    }
}

/*---------------------------------------------------------------------------*/

bool
Converter_PageCollector::GetCluster(
    const SAPDB_UInt        clusterId,
    Data_PageNoList&        pageNoList,
    IOMan_ClusterAddress&   cluster )
{
    SAPDB_UInt startIndex = 0;
    if( ! this->SearchClusterHead( clusterId, startIndex )){
        return false;
    }
    const IOMan_BlockAddress&   block      = m_BlockList[ startIndex ];
    const IOMan_BlockCount      blockCount = m_BlockCount[ startIndex ];
    const SAPDB_UInt            stopIndex  = startIndex + blockCount;
    cluster.SetCluster( block, blockCount ); // beginning of cluster
    pageNoList.Clear();
    for( SAPDB_UInt index = startIndex; index < stopIndex; index++ )
    { // add pageNo's
        const Data_PageNo pageNo = m_PageNoList[ index ];
        if( ! pageNoList.InsertEnd( pageNo )){
            return false;
        }
    }
    if( Converter_Trace.TracesLevel( 6 ))
    {
        Kernel_VTrace trace;
        trace << "Conv_Cluster::Get - ClusterId: " << clusterId
        << " start block: " << cluster.GetVolumeNo() << "/"
        << cluster.GetBlockNo( 0 ) << "("
        << cluster.GetBlockCount() << ")"
        << " start pageNo: " <<  pageNoList[ 0 ] << FlushLine;
    }
    return true;
}

/*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*/

bool
Converter_PageCollector::Init( const SAPDB_UInt   maxPages )
{
    if(  ! m_PageNoList.Reserve( maxPages )){
        return false;
    }
    if( ! m_ClusterIDList.Resize( maxPages, 0  )){
        return false;
    }
    if( ! m_BlockCount.Resize( maxPages, 0  )){
        return false;
    }
    if( ! m_BlockList.Resize( maxPages, IOMan_BlockAddress())){
        return false;
    }
    return true;
}

/*---------------------------------------------------------------------------*/

void
Converter_PageCollector::Delete()
{
    m_PageNoList.Delete();
    m_ClusterIDList.Delete();
    m_BlockCount.Delete();
    m_BlockList.Delete();
}

/*---------------------------------------------------------------------------*/

void
Converter_PageCollector::Clear()
{
    m_PageNoList.Clear();
    m_ClusterIDList.Clear();
    m_BlockCount.Clear();
    m_BlockList.Clear();
}

/*===========================================================================*
 *  END OF CODE                                                              *
 *===========================================================================*/

