/*!
  \file    DBMSrvBMedia_Medium.cpp
  \author  TiloH
  \ingroup backup media handling by the DBMServer
  \brief   declaring a class for handling backup non parallel media

\if EMIT_LICENCE

    ========== licence begin  GPL
    Copyright (c) 2002-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end

\endif
*/

#include "gcn002.h"
#include "hcn20.h"
#include "hcn29.h"
#include "hcn36.h"
#include "hcn90.h"

#include "DBM/Srv/Backup/Media/DBMSrvBMedia_Medium.hpp"
#include "DBM/Srv/Message/DBMSrvMsg_Error.hpp"

bool DBMSrvBMedia_Medium::makeLocationAbsolute(
        tcn003_Medium        * medium,
        SAPDBErr_MessageList & error)
{
    if(0==cn90Stricmp(medium->szType, ME_TYPE_FILE_CN003))
    {
        bool hasRelativePath=
                    #if defined(WIN32)
                    ('\\'!=medium->szLocation[0] && ':'!=medium->szLocation[1]);
                    #elif defined(UNIX)
                    ('/'!=medium->szLocation[0]);
                    #else
                    #error DBMSrvBMedia_Medium::makeLocationAbsolute() is only coded for WIN32 and UNIX (define one of them)
                    #endif

        if(hasRelativePath)
        {
            char absoluteLocation[ME_LOCATION_LENGTH_MXCN003+1];

            //read run directory
            tcn002_XpValueString parameterValue;
            tsp00_DbNamec        DBName;
            tcn00_Error          rcParamGetValue=OK_CN00;

            parameterValue.Init();
            DBName.rawAssign((char *)cn00DBMServerData::vcontrol()->dbname);

            rcParamGetValue=cn20XParamGetValue(DBName, PAN_RUNDIR, parameterValue);

            if(OK_CN00!=rcParamGetValue)
            {
                SAPDBErr_MessageList oMsg("DBM", __FILE__, __LINE__, SAPDBErr_MessageList::Error, cn00_1_ErrId( ERR_CN00_1 ), cn00_1_ErrTxt( ERR_CN00_1 ), 0);
                error=oMsg;
                return false;
            }
            else
            {
                //construct absolute location (run directory + relative path)
                SAPDB_strcpy(absoluteLocation, parameterValue);

                #if defined (WIN32)
                strcat(absoluteLocation, "\\");
                #else
                strcat(absoluteLocation, "/");
                #endif

                strcat(absoluteLocation, medium->szLocation);

                //replace relative location with absolute location
                strcpy(medium->szLocation, absoluteLocation);
            }
        }
    }

    return true;
}

bool DBMSrvBMedia_Medium::getBlocksizeInByte(
        size_t              & blockSize,
        const tcn003_Medium & medium,
        SAPDBErr_MessageList & error)
{
    bool rc=false;

    //get block size in pages
    long pagesPerBlock;

    if(cn36_StrToLong(pagesPerBlock, medium.szBlocksize) &&
       0<=pagesPerBlock)
    {
        if(0==pagesPerBlock)    //if medium has block size 0, use the DBM Server default (db kernel parameter BACKUP_BLOCK_CNT is no longer existent starting with 7.6.00)
        {
            tcn002_XpValueString parameterValue;
            tsp00_DbNamec        DBName;
            tcn00_Error          rcParamGetValue=OK_CN00;

            parameterValue.Init();
            DBName.rawAssign((char *)cn00DBMServerData::vcontrol()->dbname);

            rcParamGetValue=cn20XParamGetValue(DBName, PAN_BACKUP_BLOCK_CNT, parameterValue);

            if(OK_CN00==rcParamGetValue)
                cn36_StrToLong(pagesPerBlock, (char *)parameterValue);
            else
            {
                SAPDBErr_MessageList oMsg("DBM", __FILE__, __LINE__, SAPDBErr_MessageList::Error, cn00_1_ErrId( ERR_CN00_1 ), cn00_1_ErrTxt( ERR_CN00_1 ), 0);
                error=oMsg;

                return rc;
            }       
        }
    }
    else
    {
        SAPDBErr_MessageList oMsg(DBMSrv_DBMError(PARAM), 0);
        error=oMsg;

        return rc;
    }

    //get page size in bytes
    tsp00_DbNamec        dbName;
    tcn002_XpValueString pageSize;
    tcn00_Error          rcParamGetValue=OK_CN00;

    dbName.rawAssign(cn00DBMServerData::vcontrol()->dbname.asCharp());

    rcParamGetValue=cn20XParamGetValue(dbName, PAN_PAGE_SIZE, pageSize);

    if(OK_CN00==rcParamGetValue)
    {
        long bytesPerPage;

        if(cn36_StrToLong(bytesPerPage, (char *)pageSize) &&
           0<bytesPerPage)
        {
            blockSize=bytesPerPage*pagesPerBlock;
            rc=true;
        }
    }
    else
    {
        SAPDBErr_MessageList oMsg("DBM", __FILE__, __LINE__, SAPDBErr_MessageList::Error, cn00_1_ErrId( ERR_CN00_1 ), cn00_1_ErrTxt( ERR_CN00_1 ), 0);
        error=oMsg;
    }

    return rc;
}
