#!/usr/bin/perl
#
# $Header$
# $DateTime$
# $Change$
#
# Desc: 
#    ========== licence begin  GPL
#    Copyright (c) 2005 SAP AG
#
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of the GNU General Public License
#    as published by the Free Software Foundation; either version 2
#    of the License, or (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#    ========== licence end


package SAPDB::Install::Tools;

$VERSION = 1.01;

sub BEGIN {
      @ISA = ('SAPDB::Install::Exporter');
      @EXPORT = ('getRelease','unpackTGZ','readFileList', 'writeFileList','printTable','deltree',
		'checkUnpackedArchive','getMagicTGZ','getBuildInfoTGZ','getMagic', 'readFileInTGZ', 'unpackDirsInTGZ',
		'compareReleases','compareMagics','ask4any','ask4update', 'ask4delete','getcurmd5sums',
		'getnewmd5sums','minus','intersect','union','comparePathes','release2num','deleteFiles',
		'renameModFiles','addHashes','removeDoublesFromList','removeUndefinedFromList','xserverInfo');

	my $repo = SAPDB::Install::Repository::GetCurrent ();
	my @neededPackages=(
		'BuildInfo',
		'StdIO',
		'MD5Sum',
		'Values',
		'System',
		'Trace',
		'ProcState'
	);
	foreach my $package (@neededPackages) {
	  	unless (defined $repo->Eval ("SAPDB::Install::$package", 1.01)) {
                print join ("\n", $repo->GetErr)."\n";
                die;
        	}
		SAPDB::Install::Exporter::import ("SAPDB::Install::$package");
	  } 
}


# compare two path strings

sub comparePathes{
	my ($path1,$path2)=@_;
	if($^O=~/mswin/i){
		$path1=~s/\\/\//g;
		$path2=~s/\\/\//g;
		$path1=~/^$path2$/i and return 1;		
	}
	else{
		($path1 eq $path2) and return 1;
	}
	return 0;
}

sub deleteFiles{
	my ($path,@files)=@_;
	foreach my $file (@files){
		if($file =~ /\.py$/){
			foreach my $ext ('o','c'){
				if(-f "$path/$file$ext"){
					unlink("$path/$file$ext");
				}
			}	
		}
		if(-f "$path/$file"){
			$file=~/^runtime.*libpcr\.[ds]/ and next; # dont delete old precompiler runtime, if same corr. level it will be overwritten by new one else it stay there
			if(unlink("$path/$file")){
				$SAPDB::Install::Values::log->SetMsg("MSG: file \"$path/$file\" deleted\n");
			} 
			else{
				print2stderr("deleteFiles(): cannot delete file \"$path/$file\"\n");
			}
		}
		else{
			$SAPDB::Install::Values::log->SetMsg("WRN: cannot delete file \"$path/$file\" - file not exist\n");
		}
	}
}


sub renameModFiles{
	my($path,@files)=@_;
	my @returnvalue;
	-d $path or print2stderr("renameModFiles(): directory \"$path\" not found\n") and diesoft("$SAPDB::Install::Values::diemsg");
	foreach $file (@files){
		if($file =~ /\.py$/){
			foreach my $ext ('o','c'){
				if(-f "$path/$file$ext"){
					unlink("$path/$file$ext");
				}
			}	
		}
		my $count=0;
		if(-f "$path/$file"){
			while(1){
				my $name = "$file.mod$count";
				unless(-f "$path/$name"){
					 rename("$path/$file","$path/$name") and push(@returnvalue,$name) and $SAPDB::Install::Values::log->SetMsg("MSG: rename \"$path/$file\" to \"$path/$name\"\n");
					 last;
				}
				$count++;
			}
		}
		else{
			$SAPDB::Install::Values::log->SetMsg("MSG: to rename file \"$path/$file\" not found\n");
		}		
	}
	return @returnvalue;
}




sub getRelease{
    my $binary = $_[0];
	my $release;
    -f $binary or $SAPDB::Install::Values::log->SetMsg("getRelease():$binary not found\n") and return 0;
    @build = GetBuild ($binary);
	if (defined @build && $#build > 0) {
		$release=join (".", @build);
	} else {
		$SAPDB::Install::Values::log->SetMsg("getRelease() what on \"$binary\": failed\n");
		return 0;
	}
    return $release;
}


sub minus{
	my($arref1,$arref2)=@_;
	my @list1=@{$arref1};
	my @list2=@{$arref2};
	my @returnvalue;
	foreach my $element1 (@list1){
		my $exist=0;
		foreach my $element2 (@list2){
			if($element1 eq $element2){
				$exist=1;
			}			
		}
		unless($exist){
			push(@returnvalue,$element1);
		}	
	}
	return @returnvalue;
}

sub intersect{
	my($arref1,$arref2)=@_;
	my @list1=@{$arref1};
	my @list2=@{$arref2};
	my @returnvalue;
	foreach my $element1 (@list1){
		my $exist=0;
		foreach my $element2 (@list2){
			if($element1 eq $element2){
				$exist=1;
			}			
		}
		if($exist){
			push(@returnvalue,$element1);
		}	
	}
	return @returnvalue;
}

sub union{
	my($arref1,$arref2)=@_;
	my @list1=@{$arref1};
	my @list2=@{$arref2};
	my @returnvalue;
	my @tmp=minus($arref2,$arref1);
	
	foreach my $element (@list1){
		push(@returnvalue,$element);
	}
	foreach my $element (@tmp){
		push(@returnvalue,$element);
	} 
	return @returnvalue;
}

sub removeDoublesFromList{
	my ($arref) = @_;
	my @returnvalue;
	foreach my $element (@$arref){
		my $found = 0;
		foreach my $result_elememt (@returnvalue){
			$result_elememt eq $element and $found =1 and last; 
		}
		$found or push @returnvalue,$element;
	}
	return @returnvalue;
}



sub addHashes{
	my ($hrHash1,$hrHash2) = @_;
	my @keys1 = keys(%$hrHash1);
	my @keys2 = keys(%$hrHash2);
	my @same = intersect(\@keys1,\@keys2);
	$#same == -1 or return;  # error: hash1 and hash2 have same keys -> cannot add 
	my %result;
	foreach my $key (@keys1){
		$result{$key} = ${%$hrHash1}{$key};
	}
	foreach my $key (@keys2){
		$result{$key} = ${%$hrHash2}{$key};
	}
	return %result;
}


sub removeUndefinedFromList{
	my ($list) = @_;
	ref($list) ne 'ARRAY' and return undef;
	local *array = $list;
	my @list_copy = @$list;
	my $offset = 0;
	my $cur_id = 0;
 	foreach my $element (@list_copy){
		if(defined $element){
			unless($offset == 0){
				$array[$cur_id - $offset] = $element;
			}	
		}
		else{
			pop @array;
			$offset++;
		}
		$cur_id++;
	}
	return 1;
}


sub release2num{
    #sub return a number to compare release strings
	my $cor_digit_num = 5;
	$_=$_[0]; # release in format like this 7.2.4.9 (without leading nulls)
	my $log = $SAPDB::Install::Values::log;
	unless(/^(\d)+\.(\d+)\.(\d+)\.(\d+)$/ || /^(\d)\.(\d)([A-Z|\d]+)\.(\d+)$/){
		defined $log and $log->SetMsg("WRN: \"$_\" is not a valid release format\n");
		return 0;
	} 
	my $major=$1;
	my $minor=$2;
	my $cor_lev=$3;
	my $pat_lev=$4;
	
	if($cor_lev =~ /\D/){
		$cor_lev = ord($cor_lev) - ord('A');
	}
	
	if($cor_digit_num < length($cor_lev)){
		defined $log and $log->SetMsg("WRN: invalid build number: correction level has more than $cor_digit_num digits\n");
		return 0;
	}


	length($minor)==1 and $minor="0$minor";
    
	$cor_lev = ('0' x ($cor_digit_num - length($cor_lev))) . $cor_lev;
	if(length($pat_lev) == 1){
		$pat_lev="00$pat_lev";
    }
	elsif(length($pat_lev) == 2){
		$pat_lev="0$pat_lev";
	}
	elsif(length($pat_lev) > 3){
		defined $log and $log->SetMsg("WRN: invalid build number: build level has more than three digits\n");
		return 0;
	}
	my $num="$major$minor$cor_lev$pat_lev";
	return $num;
}


sub compareReleases{
	# return true -> can update
	# return false -> update not allowed  
	my ($filename,$release,$minrelease) =@_; # file of existig installation 
	my $rc=0;
	-f $filename or print2stderr("compareReleases(): file \"$filename\" not found\n") and diesoft($SAPDB::Install::Values::diemsg);
	my $filerelease = getRelease($filename);
	if(release2num($release) >= release2num($filerelease)){
		# allow update
		$SAPDB::Install::Values::log->SetMsg("MSG: update test: release equal or higher than installed\n");
		$rc = 1;
	}
	else{
		$SAPDB::Install::Values::log->SetMsg("WRN: try to install release \"$release\" over existing \"$filerelease\"\n");
		$SAPDB::Install::Values::log->SetMsg("MSG: update test: installed release newer\n");
	}
	if($minrelease){
		if(release2num($minrelease) > release2num($filerelease) or release2num($minrelease)==0){
			#update not allowed, because of to old installed version or invalid format of $minrelease
			$SAPDB::Install::Values::log->SetMsg("WRN: try to overwrite release \"$filerelease\", but min release is \"$minrelease\"\n");
			$SAPDB::Install::Values::log->SetMsg("MSG: update test: min release not reached\n");
			$rc=undef;
		}
	}
	$rc and $SAPDB::Install::Values::log->SetMsg("MSG: update from \"$filerelease\" to \"$release\" allowed \n");
	$rc or $SAPDB::Install::Values::log->SetMsg("MSG:  update from \"$filerelease\" to \"$release\" not allowed \n");
	return $rc;
}

sub getMagic{
	# sub read and return magic of file
	my $file=$_[0]; # file of existing installation
	-f $file or print2stderr("getMagic(): file \"$file\" not found\n") and return 0;
	my $buildinfo = GetBuildInfo ($file);
	my $text=$buildinfo->{'magic'};
	unless($text){
		print2stderr("getMagic(): cannot read magic of file \"$file\"\n");
		return 0;
	}
	if($text=~/cannot/i){
		print2stderr("getMagic(): cannot get magic of file \"$file\"\n");
		return 0;
	}
	if($text=~/unknown/i){
		print2stderr("getMagic(): file \"$file\" has unknown magic \n");
		return 0;
	}
	return $text;	
}


sub getMagicTGZ{
	#sub read und return magic of file in tgz archive
	my $handle = $_[0]; # untgz object handle
	my $filename = $_[1]; # filename in archive
	my $returnvalue;
	$handle->Rewind();
	while(1){
		my $text=$handle->Next();
		if($handle->GetErr){
			print2stderr("getMagicTGZ(): cannot find file \"$filename\" in package ".$handle->GetArchivename.": ".join(': ',$handle->GetErr)."\n");
			return 0;
		} 
		if($text=~/.*$filename$/){
			my $text=$handle->ExtractMagic();
			$returnvalue = $text;
			last;
		}
	}
	return $returnvalue;
}


sub getBuildInfoTGZ{
	my ($tgzhandle,$file) = @_;
	$tgzhandle->Rewind();
	my $buildinfo;
	while(1){
		my $text=$tgzhandle->Next();
		if($tgzhandle->GetErr){
			print2stderr("getBuildInfoTGZ(): cannot find file \"$file\" in package ".$tgzhandle->GetArchivename.": ".join(': ',$tgzhandle->GetErr)."\n");
			return undef;
		} 
		if($text=~/.*$file$/){
			$buildinfo=$tgzhandle->ExtractBuildInfo();
			if($tgzhandle->GetErr){
				print2stderr("getBuildInfoTGZ(): cannot read BuildInfo in \"$file\" in package ".$tgzhandle->GetArchivename.": ".join(': ',$tgzhandle->GetErr)."\n");
				return undef;
			}
			last;
		}
	}
	return $buildinfo;
}



sub getcurmd5sums{
	my ($path,@files)=@_;
	my %returnvalue;
	-d $path or print2stderr("getcurmd5sum(): directory \"$path\" not found\n") and return 0;
	foreach $file (@files){
		my $fullpath="$path/$file";
		-f $fullpath and my $sum=MD5Sum($fullpath);
		if($sum){
			$returnvalue{$file}=$sum;
		}
		else{
			$SAPDB::Install::Values::log->SetMsg("getcurmd5sum(): cannot get md5sum of file \"$fullpath\": $!\n");
		} 
	}
	return %returnvalue;
}


sub getnewmd5sums{
	my($tgzhandle,$filelist)=@_;
	my %returnvalue;
	my $liststring=readFileInTGZ($tgzhandle,$filelist);
	@lines = split(/\n/,$liststring);
	foreach my $line (@lines){
		$_=$line;
		/^"*([^"]*)"*\ ([0-9,a-f]{32}).*$/ and $returnvalue{$1}=$2;
	}
	return %returnvalue;
}

sub compareMagics{
	# return true -> can update
	# return false -> update not allowed
	my $new_magic=$_[0]; 
	my $rootdir=$_[1]; # directory containing installed file
	my $filename=$_[2]; # name of to checking file  
	my $mode=$_[3]; # compare mode
	$SAPDB::Install::Values::log->SetMsg("magic test mode: $mode\n");
	if($mode=~/32to64/i){
		$mode=SAPDB::Install::FileMagic::AllowMagic32To64();
	}
	elsif($mode=~/64to32/i){
		$mode=SAPDB::Install::FileMagic::AllowMagic64To32();
	}
	elsif($mode=~/NOBIT/i){
		$mode = SAPDB::Install::FileMagic::AllowMagic64To32() |
		        SAPDB::Install::FileMagic::AllowMagic32To64();	
	}
	elsif($mode=~/EQUAL/i){
		# magics must be equal
		$mode=0;
	}
	else{
		#unkown mode -> no check
		return 0;
	}
	-d $rootdir or print2stderr("compareMagics(): directory \"$rootdir\" not found\n") and return 0;
	my $old_magic = getMagic("$rootdir/$filename");
	my ($rc,$errtext)=
	SAPDB::Install::FileMagic::CompareMagic($old_magic,$new_magic,$mode);

	unless($rc == 0){
		print2stderr("magic test failed: $errtext\n");
		print2stderr("old magic: \"$old_magic\" \n");
		print2stderr("new magic: \"$new_magic\"\n");
		return 0;
	}
	$SAPDB::Install::Values::log->SetMsg("MSG: magic test: ok\n");
	return 1;
}




sub unpackTGZ{
	#sub extract archive
    my ($packobj,$destdir,$testrun,@list)=@_;
	# $tgzname: full parth to archive
    # $destdir: destination directory
    # $testrun: if true no files will be extracted, only try to open for writing
	# @list: empty -> all files of archive will be extracted; set -> only files in @list will be extracted
	my $cwd;
    my $tgzhandle = $packobj->Untgz;
	$tgzhandle->SetOwner($packobj->User,$packobj->Group) unless($^O =~ /mswin/i);
	my @mapped_files = ();
	my @busy_files = ();
	$tgzhandle->Rewind();
	
	if($testrun){
		$tgzhandle->SetReadOnly(1);
	}
	else{
		$tgzhandle->SetReadOnly(0);
	}

	if($testrun){print2stdout("start extraction test run of \"".$tgzhandle->GetArchivename."\"\n");}	
	else{print2stdout("start real extraction of \"".$tgzhandle->GetArchivename."\"\n");}
	-d $destdir or print2stderr("destination directory \"$destdir\" not found\n") and diesoft($SAPDB::Install::Values::diemsg);
	
	unless($tgzhandle->SetDestDir($destdir)==0){
		print2stderr("cannot set destination directory \"$destdir\": ".join(': ',$tgzhandle->GetErr)." \n");
		diesoft($SAPDB::Install::Values::diemsg);
	}
	my $info = "start"; 
	while($info){
		$info = $tgzhandle->Next();
		unless($info){last;}
		$_=$info;
		if(/^d/){
			# directory 
			unless($testrun){
				print2stdout("extracting: $info\n");
			}
			$tgzhandle->ExtractFile;
			next;
		}
		/([^\ ]*)$/	and my $file=$1;
		$file=~s/^\.\///; #remove "./" at begin of filename
		my ($rc,$errtext);
		if(@list){
			# @list is set -> extract only files in list
			foreach my $extfile (@list){ 
				if($extfile eq $file){
					unless($testrun){
						print2stdout("extracting: $info\n");
					}
					$tgzhandle->ExtractFile;
				}
			}
		}
		else{
			# @list is unset -> extract all files in archive
			unless($testrun){
						print2stdout("extracting: $info\n");
			}
			$tgzhandle->ExtractFile($testrun);	
		}
		if($tgzhandle->GetErr){
			if($testrun){
				my $err_msg = join(': ',$tgzhandle->GetErr);
				my $ignore_file = 0;
				if(defined $packobj->IgnoreBusyFiles){
					foreach my $ibfile (@{$packobj->IgnoreBusyFiles}){
						if(normalizePath($file) eq normalizePath($ibfile)){
							$ignore_file = 1;
							last;
						}
					} 
				}
				next if($ignore_file);

				if($^O =~ /mswin/i && $err_msg =~ /file\smapped\sby\suser\sapplication/ ){
					defined $SAPDB::Install::Values::log && 
						$SAPDB::Install::Values::log->SetMsg("file \"$destdir/$file\" is mapped\n"); 
					push @mapped_files,$file;
					next;
				}
				else{
					push @busy_files,normalizePath("$destdir/$file");			
					#print2stderr("test run failed: ".join(': ',$tgzhandle->GetErr)." - no file(s) of \"".$tgzhandle->GetArchivename."\" extracted!\n");
					print2stderr("test $file failed: ".join(': ',$tgzhandle->GetErr)."\n");
					next;
				}
			}
			else{
				print2stderr("real extraction failed after successful test run: ".join(': ',$tgzhandle->GetErr)."\n");
				print2stderr("extraction of archive \"".$tgzhandle->GetArchivename."\" not completed\n");
				print2stderr("cannot unpack file \"$destdir/$file\" error: ".join(': ',$tgzhandle->GetErr)."\n");
			}
			print2stderr("maybe any sap db software is running... please stop all\n");
			diesoft($SAPDB::Install::Values::diemsg);
			
		}
	}
	
	if($testrun){
		my @fsusage = $tgzhandle->GetFSUsage; 
		return (\@fsusage,\@mapped_files,\@busy_files);
	}
	
	return 1;

}


sub unpackDirsInTGZ{
	my ($tgzhandle,$destdir,$packobj)=@_;
	-d $destdir or print2stderr("destination directory \"$destdir\" not found\n") and diesoft($SAPDB::Install::Values::diemsg);
	
	$tgzhandle->SetOwner($packobj->User,$packobj->Group) unless($^O =~ /mswin/i);
	
	$tgzhandle->Rewind();
	
	while(1){
		my $info = $tgzhandle->Next();
		if($tgzhandle->GetErr){
			print2stderr("error reading tgz archive \"".$tgzhandle->GetArchivename."\": ".join(': ',$tgzhandle->GetErr)."\n");
			diesoft($SAPDB::Install::Values::diemsg);
		}
		unless($info){last;}
		$_=$info;
		/^d.*/ and /([^\ ]*)$/	and my $dir=$1;
		if(not -d "$destdir/$dir" and $dir){
			makedir("$destdir/$dir",$SAPDB::Install::Values::chmod_exe,$packobj->UID,$packobj->GID);
			$SAPDB::Install::Values::log->SetMsg("created directory \"$destdir/$dir\" from tgz which contains no unpacked file\n");
		}
	}
}



sub readFileInTGZ{
	my($tgzhandle,$readfile)=@_;
	my $returnvalue;
	my $info = "start";
	$tgzhandle->Rewind(); 
	while($info){
		$info = $tgzhandle->Next();
		if($tgzhandle->GetErr){$SAPDB::Install::Values::log->SetMsg("WRN: readFileInTGZ(): file \"$readfile\" not found in archive \"".$tgzhandle->GetArchivename."\": ".join(': ',$tgzhandle->GetErr)."\n");return undef;}
		$_=$info;
		/([^\ ]*)$/	and my $file=$1;
		$file=~s/^\.\///; #remove "./" at begin of filename 
		if($file eq $readfile){
			$tgzhandle->ExtractScalar($returnvalue);
			if($tgzhandle->GetErr){
				print2stderr("ExtractScalar() in archive \"".$tgzhandle->GetArchivename."\" : ".join(': ',$tgzhandle->GetErr)."\n");
			}
			last;
		}
	}
	return $returnvalue;
}


sub readFileList{
	# sub read file with filelists and md5sums
	# return such a hash: $md5sum{$file}=235b3f2324c...
	my $file=$_[0]; # full path to file containing filelist
	my %returnvalue;
	-f $file or print2stderr("readFileList(): file \"$file\" not found\n") and return 0;
	open(LST,$file) or print2stderr("readFileList(): cannot open file \"$file\"\n") and return 0;
	while(my $line=<LST>){
		chomp($line);
		$_=$line;
		/^"*([^"]*)"*\ ([0-9,a-f]{32})$/ and $returnvalue{$1}=$2;
	}
	close(LST);
	unless(%returnvalue){
		print2stderr("WARNING: filelist \"$file\" is empty\n");
	}
	return %returnvalue;
}

sub writeFileList{
	my($filename,%md5sums)=@_;
	open(LST,">$filename") or print2stderr("writeFileList(): cannot create file \"$filename\"\n") and diesoft($SAPDB::Install::Values::diemsg);
	foreach my $file (keys(%md5sums)){	
		print LST "\"$file\" $md5sums{$file}\n";
	}
	close(LST);
}

#sub checks all files of unpacked archive
sub checkUnpackedArchive{
	# sub check unpacked files with md5sums
	print2stdout("checking unpacked archive... ");    
	my ($tree,$ref_filelist)=@_;
	
	#check dir
	-d $tree or print2stderr("checkUnpackedArchive():tree root \"$tree\" not found\n") and diesoft($SAPDB::Install::Values::diemsg);

	#check files from filelists
	my %checksum = %$ref_filelist;
	my $status = 1;
	my @msgs = ();
	foreach my $file (keys(%checksum)){
			my $fullpath="$tree/$file";
			unless(-f $fullpath){
				push @msgs, "file \"$fullpath\" not found\n";
				$status = 0;
			}
			if($checksum{$file} ne MD5Sum($fullpath)){
				push @msgs, "mutation of file \"$fullpath\"\n";
				$status = 0;
			}
			else{
				$SAPDB::Install::Values::log->SetMsg("MSG: check file \"$file\": ok\n");
			}	
	}
	unless($status){
		print2stdout("failed\n");
		foreach my $msg (@msgs){
			print2stderr($msg);
		}
		diesoft($diemsg);
	}
	print2stdout("ok\n");
	return 1;  
}

sub ask4update{
	$SAPDB::Install::StartInstall::opt_b and print2stderr("run in batch mode - cant ask user!\n") and diesoft($SAPDB::Install::Values::diemsg);
	my $cnt=$_[0]; # number of existing installations
	unless($cnt){return -1;} # no existing installation
	while(1){
		print2stdout("update existing installation (y/n): ");
		$_ = readstdin();
		/^n$/i and return -1; # no update 
		/^y$/i and last;
	}
	if($cnt==1){return 0;} # only one existing installation - no user choice needed
	while(1){
		print2stdout("please enter installation id: ");
		$_ = readstdin();
		if(/^[0-9]{1,}$/){
			if($_ >= 0 and $_ < $cnt){ return $_;}
		}
	}	
}

sub ask4any{
	my ($arPossibles,$key_name,$verb,$batch)=@_;
	my @possibles=@$arPossibles;
	return -1 if $#possibles == -1;
	my $id = 0;
	my $s = 's';
	$s='' if  $#possibles == 0;
	print2stdout("existing $key_name$s:\n");
	my @table;
	foreach my $elem (@possibles){
		my @row;
		if (ref($elem) eq 'ARRAY'){
			@row = ("$id:");
			push(@row,@$elem);
		}
		else{
			@row = ("$id:",$elem);
		}
		push(@table,\@row);
		$id++;
	}
	
	push(@table,["$id:",'none']) && $id++;

	printTable(\@table,'  ');
	
	if($batch){
		$#possibles == 0 and return 0;
		print2stderr("run in batch mode - cant ask user!\n");
		diesoft($SAPDB::Install::Values::diemsg);
	}
	
	while(1){
		print2stdout("please enter $key_name id: ");
		$_ = readstdin();
		if(/^[0-9]{1,}$/){
			return -1 if $_ == ($id - 1);
			if($_ >= 0 and $_ < $id){ return $_;}
		}
	}
}

sub ask4delete{
	my $cnt=$_[0]; # number of existing installations
	unless($cnt){return -1;} # no existing installation
	while(1){
		print2stdout("WARNING: DELETE existing SAP DB installation\n");
		print2stdout("please enter installation id: ");
		$_ = readstdin();
		if(/^[0-9]{1,}$/){
			if($_ >= 0 and $_ < $cnt){ return $_;}
		}
	}	
}


sub printTable{
	my ($rrows,$separator) = @_;
	ref($rrows) eq 'ARRAY'  or print2stderr("printTable(): wrong usage\n") and return undef; 
	my @rows = @$rrows;
	my @width; # array of column widths
	
	# find column widths
	foreach (@rows){
		$_ == undef and next;
		ref eq 'ARRAY' or print2stderr("printTable(): wrong usage\n") and return undef;
		my @columns = @$_;
		my $index = 0;
		foreach my $col (@columns){
				$width[$index] = length($col) if(not defined $width[$index] or length($col) > $width[$index]);		
				$index++;
		} 
	}
	
	# print table
	foreach (@rows){
		my @columns = @$_;
		my $index = 0;
		my $printBuf = '';
		foreach my $col (@columns){
				my $fill = ' ' x ($width[$index] - length($col)); 
				$printBuf .= $col.$fill;
				$printBuf .= $separator if $index != $#columns;
				$index++;
		}
		print2stdout("$printBuf\n");	 
	}

	return 1;	
}


my $x_ping_count = 0;

sub xserverInfo{
	my ($config) =@_;
	
	#load Comm and ProcState module during runtime if needed
		
	my $repo = SAPDB::Install::Repository::GetCurrent();
	
	my @neededPackages = ('Comm','ProcState');
	
	foreach my $package (@neededPackages){
		unless(defined %{"SAPDB::Install::${package}::"}){
			unless (defined $repo->Eval ("SAPDB::Install::$package", 1.01)) {
				print join ("\n", $repo->GetErr)."\n";
				die ("error loading:  SAPDB::Install::$package\n");
			}
			SAPDB::Install::Exporter::import ("SAPDB::Install::$package"); 
		}
	}
		
	my %result;
		
	my $xsrv = SAPDB::Install::Comm->new($config->{'node'});
	my ($rc,%xsrv_data) = $xsrv->ping();
	if(	(defined  $rc && $rc == 0) || 
			
			# aix x_server 7.2.05.15 unable to send ping reply 
			
			($xsrv->lasterr() eq 'connection closed by counterpart')
			 ){
		#success
		if(defined $SAPDB::Install::Values::log){
			$SAPDB::Install::Values::log->SetMsg("x_server is running\n");
			if(defined %xsrv_data){
				$SAPDB::Install::Values::log->SetMsg("got following x_server data:\n");
				foreach my $key (keys(%xsrv_data)){
					$SAPDB::Install::Values::log->SetMsg("$key = $xsrv_data{$key}\n");
				}	
			}
		}
		$x_ping_count = 0;
		$result{'data'} = \%xsrv_data;
		$result{'is_running'} = 1;
		
		unless(defined $config->{'node'}){
		
			my $ps = SAPDB::Install::ProcState::new();
						
			foreach $proc (@{$ps->{'procs'}}) {
				if($proc->{'cmd'} =~ /vserver/ and ($proc->{'ppid'} == 1 or $ps->GetCmd($proc->{'ppid'} !~ /vserver/))){
					$result{'pid'} = $proc->{'pid'};
					$result{'uid'} = $proc->{'uid'}; 		
					last;
				}
			}
		}

		if(exists $xsrv_data{'version'}){
			my ($sys,$version,$build,$state_id,$cl) = ($xsrv_data{'version'} =~ /([UWX]\d\d[\/\|].+\S)\s+(\d+\.\d+\.\d+)\s+Build\s0(\d\d)-(\d{3})-(\d{3}-\d{3})/);
			$result{'sys'} = $sys;
			my ($maj,$min,$cor) = ($version =~ /(\d+)\.(\d+)\.(\d)/);
			$cor = "0$cor" if length($cor) == 1;
			$result{'version'} ="$maj.$min.$cor.$build";
			my $relnum = release2num($version.'.0');			
			if(	$relnum ==  release2num('7.3.0.0') and $build >= 38 or 
				$relnum ==  release2num('7.4.2.0') and $build >= 20 or 
				$relnum ==  release2num('7.4.3.0') and $build >= 23 or 
				$relnum ==  release2num('7.4.4.0') and $build >= 1 or
				$relnum >= release2num('7.4.5.0')){
				$result{'is_updatable'} = 1;	
			}
			else{
				$result{'is_updatable'} = 0;
			}
			if($state_id){
				$cl =~ s/-//g;
				$result{'changelist'} = $cl;
				my($prod_make,$state,$purpose) = ($state_id =~ /(\d)(\d)(\d)/);
				$result{'is_prod_make'} = $prod_make;
				$result{'qa_level'} = ($state == 2 ? 'RAMP' : ($state == 1 ? 'COR' : 'DEV') );  		
				$result{'purpose'} = '';
				$result{'purpose'} .= ' OLTP' if (($purpose & 1) != 0);
				$result{'purpose'} .= ' liveCache' if (($purpose & 2) != 0);
				$result{'purpose'} = 'unspecified' unless $result{'purpose'} =~ /\S/;
				$result{'purpose'} =~ s/^\s//;
			}
		}
	}
	else{
		$x_ping_count++;
		$result{'is_running'} = 0;
		$result{'lasterr'} = $xsrv->lasterr();
		$max_ping = 1;
		$max_ping = $config->{'max_ping'} if defined $config->{'max_ping'};
		
		if($x_ping_count >= $max_ping){
			if(defined $SAPDB::Install::Values::log){
				$SAPDB::Install::Values::log->SetMsg("x_server is not running\n");
			}
			$x_ping_count = 0;
		}
		else{
			sleep(1);
			return xserverInfo($config);
		}
	}
	$SAPDB::Install::Values::xserver_info = \%result;
	return \%result;
}

sub deltree{
        my ($dir) = @_;
        -d $dir || return 0;
        opendir(DH,$dir) || return 0;
        my @content = readdir(DH);
        closedir(DH);

        my @files = grep { -f "$dir/$_" } @content;
        my @dirs = grep { -d "$dir/$_" && ! /^\.{1,2}$/} @content;

        
		#
		#  enable write permission in directory to delete entries
		#

		
		unless($^O =~ /mswin/i){
			if($#files > -1 || $#dirs > -1){
				my @statbuf = stat($dir);
				my $mask;
				if($> == $statbuf[4]){
					$mask = 0200;
				}
				else{
					my $found;
					foreach my $gid (split(' ',$) )){
						if($gid == $statbuf[5]){
							$found = 1;
							last;
						} 
					}
					if($found){
						$mask = 020;
					}
					else{
						$mask = 002;
					}
				}
				if(($statbuf[2] & $mask) == 0){
					chmod($statbuf[2] | $mask,$dir) || print2stderr("cannot enable write permissions for $dir\n");
				}
			}
		}

		foreach (@dirs){
                -l "$dir/$_" and next;
                deltree("$dir/$_") || return 0;
        }
        foreach (@files){
                -l "$dir/$_" and next;
		    -f "$dir/$_" and (unlink("$dir/$_") or print2stderr("cannot delete $dir/$_\n") and return 0);
        }
        rmdir($dir) or print2stderr("cannot delete $dir\n") and return 0;
        return 1;
}



1;