#!/usr/bin/perl
#
# $Header: //sapdb/V75/c_00/develop/sys/src/install/perl/SAPDB/Install/Instance/Base.pm#3 $
# $DateTime: 2003/12/11 13:40:27 $
# $Change: 59245 $
#
#    ========== licence begin  GPL
#    Copyright (c) 2005 SAP AG
#
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of the GNU General Public License
#    as published by the Free Software Foundation; either version 2
#    of the License, or (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#    ========== licence end


#
# SAPDB::Install::Instance::Base.pm
#
# base package for SAPDB::Install::Instance session
#

#some new text

package SAPDB::Install::Instance::Base;

sub BEGIN {
	@ISA = ('SAPDB::Install::Exporter');
	@EXPORT = ();
	my $repo = SAPDB::Install::Repository::GetCurrent ();
	my @neededPackages = (
		'DBMCmd', 'BuildInfo', 'InstInfo', 'System::Unix', 'Base' 
	);

	foreach my $package (@neededPackages) {
		unless (defined $repo->Eval
		("SAPDB::Install::$package", 1.01)) {
			print join ("\n", $repo->GetErr)."\n";
			die;
		}
		SAPDB::Install::Exporter::import ("SAPDB::Install::$package");
	}
}

$VERSION = 1.01;

$DEBUG = 0;

#
# constructor
#
sub new {
	my $class = shift;
	my $self = {};

	bless $self, $class;

	# set member of this object
	$self->{msg0} = undef;
	$self->{msg1} = undef;
	$self->{msgtxt} = "";
	$self->{msgheadline} = "";

	$self->{dbmcmds} = {};

	$self->{dbm} = undef;
	$self->{dbname} = undef;
	$self->{userpwd} = undef;
	$self->{dbowner} = undef;
	$self->{instancetype} = undef;
	$self->{instancetypename} = undef;
	$self->{running_instances} = undef;
	$self->{target_release} = undef;
	$self->{starting_release} = undef;
	$self->{apo_target_release} = undef;
	$self->{apo_starting_release} = undef;
	$self->{migration_strategy} = undef;
	$self->{can_db_admin} = undef;
	$self->{can_db_online} = undef;

	$self->{logpos} = undef;
	$self->{backup_logpos} = undef;
	$self->{additional} = undef;

	$self->{errorstate} = 'OK';
	$self->{keep_instance_data} = 0;
	return ($self);
}

#
# output method for info messages
#
sub msg0 {
	my ($self, $txt) = @_;

	my $msg0 = $self->{msg0};
	&$msg0 ($txt) if (defined $msg0);
	# my $msg1 = $self->{msg1};
	# &$msg1 ($txt) if (defined $msg1);
	$self->{msgtxt} .= $txt;
}

#
# output method for detailed messages
#
sub msg1 {
	my ($self, $txt) = @_;
	my $msg1 = $self->{msg1};

	&$msg1 ($txt) if (defined $msg1);
	$self->{msgtxt} .= $txt;
}

#
# reset buffer for detailed  messages
#
sub msgbegin {
	my ($self, $headline) =  @_;
	$self->{msgtxt} = "";
	$self->{msgheadline} = "";

	if (defined $headline) {
		$self->{msgheadline} = $headline;
	}
}

sub msgend {
}

sub set_errorstate {
	my ($self, $errorstate) =  @_;
	$self->{'errorstate'} = $errorstate;
}

sub get_errorstate {
	my ($self) =  @_;
	return ($self->{'errorstate'});
}

#
# helper function to get migration strategy
#
# currently (2001-10-01) we know three migration strategies
# COMPATIBLE_LOG, CONSISTENT_DATA, EXTERNAL_CONSISTENT_BACKUP
#
sub get_migrationstrategy {
	my ($self) =  @_;
	return ($self->{'migration_strategy'});
}

#
# helper function to get keep instance data flag
#
sub get_keep_instance_data {
	my ($self) =  @_;
	return ($self->{'keep_instance_data'});
}

#
# helper function find other running instances
#
sub get_running_instances {
	my ($self) =  @_;
	return ($self->{'running_instances'});
}


#
# change permissions of config and data files
# during migration from 7.4 or below to 7.5 or higher
#
sub chvolperm {
	my ($self, $dbname, $starting_release, $target_release) = @_;

	$self->set_errorstate ('ERROR');

	return undef unless (defined $starting_release);
	return undef unless (defined $target_release);
	return undef unless (defined $dbname);
	return undef unless (defined $self);

	if (
	$starting_release =~ /7\.(\d+)\.\d+/ && $1 <= 4 &&
	$target_release =~ /7\.(\d+)\.\d+/ && $1 >= 5) {
		my ($prev) = ref ($self);

		bless ($self, 'SAPDB::Install::Instance::ChVolPerm');

		$self->chvolperm ($dbname);
		$state = $self->get_errorstate ();
		bless ($self, $prev);

		unless (defined $state && $state eq 'OK') {
			$self->msgend ();
			return undef;
		}
	}

	$self->set_errorstate ('OK');
	$self->msgend ();
	return 0;
}

#
# establish dbm session for dbm user
#
sub session {
	my ($self, %arg) = @_;

	sub ismemberof {
		my ($a, @list) = @_;

		foreach my $b (@list) {
			return 1 if ($a eq $b);
		}
		return 0;
	}

	#
	# get parameter from argument hash
	#
	my $target_release = 
	$arg{'target_release'} ? $arg{'target_release'} : undef;

	my $starting_release = 
	$arg{'starting_release'} ? $arg{'starting_release'} : undef;

	my $apo_starting_release = 
	$arg{'apo_starting_release'} ? $arg{'apo_starting_release'} : undef;

	my $apo_target_release = 
	$arg{'apo_target_release'} ? $arg{'apo_target_release'} : undef;

	my $migration_strategy = 
	$arg{'migration_strategy'} ? $arg{'migration_strategy'} : undef;

	my $dbhost = $arg{'dbhost'} ? $arg{'dbhost'} : undef;

	my $dbname = $arg{'dbname'} ? $arg{'dbname'} : undef;

	my $userpwd = $arg{'userpwd'} ? $arg{'userpwd'} : undef;

	my $logpos = $arg{'logpos'} ? $arg{'logpos'} : undef;

	my $backup_logpos = $arg{'backup_logpos'} ? $arg{'backup_logpos'} : undef;

	my $additional = $arg{'additional'} ? $arg{'additional'} : undef;

	$self->msgbegin ("creating instance session");
	$self->set_errorstate ('ERROR');

	if (defined $apo_starting_release) {
		$apo_starting_release =	normalize_aporel ($apo_starting_release);
	}

	if (defined $apo_target_release) {
		$apo_target_release = normalize_aporel ($apo_target_release);
	}

	#
	# if there is already 7.5 security,
	# make sure that db process owner has sufficient permissions
	# to run a database
	#
	unless ($^O =~ /mswin/i) {
		my $instinfo = SAPDB::Install::InstInfo::new ($dbname);
		
		unless (defined $instinfo) {
			$self->msg1	("cannot get installation info\n");
			$self->msg1	(
			"may be /usr/spool/sql/ini/SAP_DBTech.ini ".
			"or /etc/opt/sdb corrupted\n");
			$self->msg1 ("check these files and restart installation\n");
			$self->msgend ();
			return undef;			
		}

		unless (
		defined $instinfo->{database} &&
		defined $instinfo->{database}->{$dbname}) {
			$self->msg1
			("cannot get information about instance ".$dbname."\n");
			$self->msg1 ("may be paramfile, user profile container or \n");
			$self->msg1 ("registration of instance corrupted\n");
			$self->msg1 ("check this and restart installation\n");
			$self->msgend ();
			return undef;
		}

		if (defined $instinfo->{user}) {
			# 7.5 permissions are usesd

			$self->msg1
			("permissions for databases as of version 7.5.00 ".
			"are used on your system\n");
			$self->msg1 ("database owner is ".$instinfo->{user}."\n");
			$self->msg1 ("administrators group is ".$instinfo->{group}."\n");

			my $dbowner = $instinfo->{database}->{$dbname}->{dbowner};
			my ($uid, @groups) = ugetids ($dbowner);

			foreach (my $i = 0; $i <= $#groups; $i++) {
				$groups[$i] = scalar (getgrgid ($groups[$i]));
			}

			$self->msg1 (
			"owner of ".$dbname." is ".$dbowner." (".
			join (' ', @groups).")\n");
			
			unless (
			$dbowner eq $instinfo->{user} ||
			ismemberof ($instinfo->{group}, @groups)
			) {
				$self->msg1	(
				$dbowner." is not a member of group ".$instinfo->{group}."\n");
				$self->msg1	("add ".$dbowner." to group ".$instinfo->{group}."\n");
				$self->msg1 ("and restart installation\n");
				$self->msgend ();
				return undef;
			}
		}
	}

	#
	# create dbm session, use local host if no dbhost is known
	#
	$dbhost = '' unless (defined $dbhost);
	my $dbm = SAPDB::Install::DBMCmd->new ($dbhost, $dbname);
	if (defined $dbm->lasterr ()) {
		$self->msg1 ($dbm->lasterr ()."\n");
		$self->msgend ();
		return undef;
	}

	#
	# log on as dbm user
	#
	unless (defined ($dbm->user_logon ($userpwd))) {
		# no good idea to use lastcmd
		# if you do not want to show the password
		$self->msg1 ("user_logon failed\n");
		$self->msg1 ($dbm->lastmsg ()."\n");
		$self->msg1 ($dbm->lasterr ()."\n");
		$self->msgend ();
		return undef;
	}

	#
	# we try to find out what our dbmsrv speaks
	#
	my @cmds = $dbm->help ();
	unless ($dbm->lastmsg () =~ /^OK/) {
		$self->msg1 ($dbm->lastdialog ());
		$self->msgend ();
		return undef;
	}

	#
	# output contains empty lines and following lines,
	# known commands are starting at the first column,
	# get rid of parameters after first whitespace
	#
	$self->{dbmcmds} = {};
	foreach my $cmd (@cmds) {
		next unless ($cmd =~ /^(\S+)/);

		$self->{dbmcmds}->{$1} = 1;
		$i++;
	}

	#
	# beginning with 7.40.03
	# the dbmsrv uses ONLINE and ADMIN instead of WARM and COLD
	#
	my $can_db_online = $self->candbm ('db_online');
	if ($can_db_online == 1) {
		$self->msg1 ("dbmsrv uses db_online\n");
	} else {
		$self->msg1 ("dbmsrv uses db_warm\n");
	}

	my $can_db_admin = $self->candbm ('db_admin');
	if ($can_db_admin == 1) {
		$self->msg1 ("dbmsrv uses db_admin\n");
	} else {
		$self->msg1 ("dbmsrv uses db_cold\n");
	}

	#
	# set member variables
	#
	$self->{dbm} = $dbm;
	$self->{dbname} = $dbname;
	$self->{userpwd} = $userpwd;
	$self->{starting_release} =
		$starting_release if (defined $starting_release);
	$self->{target_release} =
		$target_release if (defined $target_release);
	$self->{apo_starting_release} =
		$apo_starting_release if (defined $apo_starting_release);
	$self->{apo_target_release} =
		$apo_target_release if (defined $apo_target_release);
	$self->{migration_strategy} =
		$migration_strategy if (defined $migration_strategy);
	$self->{can_db_online} = $can_db_online;
	$self->{can_db_admin} = $can_db_admin;
	
	$self->{logpos} = $logpos if (defined $logpos);
	$self->{backup_logpos} = $backup_logpos if (defined $backup_logpos);
	$self->{additional} = $additonal if (defined $additional);

	$self->msgend ();
	$self->set_errorstate ('OK');
	return ($dbm);
}

sub candbm {
	my ($self, $cmd) = @_;

	return 0 unless (defined $self->{dbmcmds} &&
	                 defined $self->{dbmcmds}->{$cmd});

	return ($self->{dbmcmds}->{$cmd});
}

#
# getdbprocessowner ()
#
sub getdbprocessowner {
	my ($self) = @_;
	return undef if ($^O =~ /mswin/i);

	# do we know it already
	return ($self->{dbowner}) if (defined $self->{dbowner});

	unless (defined $self->{instancetypename}) {
		$self->{instancetypename} =  
		$self->{instancetype} =~ /LVC/ ? 'liveCache' : 'database';
	}

	my $dbname = $self->{dbname};
	my $instinfo = SAPDB::Install::InstInfo::new ($dbname);
	unless (
	defined $instinfo &&
	defined $instinfo->{database} &&
	defined $instinfo->{database}->{$dbname}) {
		$self->msg1
		("cannot get ".$self->{instancetypename}." process owner\n");
		return undef;
	}

	$self->{dbowner} = $instinfo->{database}->{$dbname}->{dbowner};

	# new interface
	# returns name of process owner
	return ($self->{dbowner}) unless (wantarray ());

	# old interface for backward compatibility for packages until 7.4.3.12
	# returns array of uid, gid
	my @pwent = getpwnam ($dbowner);
	return ($pwdent[2], $pwent[3]);
}

#
# load_systab
# works like SAPDB::Install::DBMCmd::load_systab,
# but it uses 'show active us' and 'sysdd.ctrl_configuration' 
# every 30 seconds to find when something went wrong
#
sub load_systab {
	my ($self, $passwd) = @_;
	my ($id, $operation, $console, $sql);
	my ($rc, $padding, $state);
	my $dbm = $self->{dbm};

	my $cmd = (defined $passwd && $passwd =~ /\S/) ? 
	          'load_systab -ud '.$passwd : 'load_systab';

	$sql = $console = 0;
	$operation = $dbm->send ($cmd);
	for (;;) {
		# read console output every 30 seconds
		$id = $dbm->select (30);

		unless (defined $id) {
			# nothing to select or select error
			return undef;
		} elsif ($id == 0) {
			# timeout, send console command
			if ($console == 0 && $sql == 0) {
				$console = $dbm->send ('show active us');
			}
		} elsif ($id == $operation) {
			# operation finished, read result
			($rc) = split ('\n', $dbm->recv ($operation));
			return undef unless ($rc =~ /^OK/);

			# wait for results of pending commands
			while ($id = $dbm->select ()) {
				$dbm->recv ($id);
			}
			
			last;
		} elsif ($id == $console) {
			# got console output
			($rc, $padding, my @data) = 
			split ('\n', $dbm->recv ($console));

			# we are no longer waiting for console response
			$console = 0;

			#
			# look for console error
			#
			# cba 2001-11-07
			# I have seen that UNIX uses 'ERR <errid>'
			# and Windows uses '<errid> ERROR' 
			#
			if ($data[0] =~ /^ERR\s+\d+|^\d+\s+ERROR/) {
				$self->msg1 ("got console error:\n");
				foreach (@data) {
					$self->msg1 ("  ".$_."\n");
				}
				
				#
				# sometimes the output of console
				# failed because we cannot attach 
				# to shared memory (mostly on sun)
				# then we get an error message
				# 'Invalid argument'
				#	
				unless ($data[0] =~ /Invalid argument/i) {
					$self->msg1 ("dbm command was:\n");
					$self->msg1 ("\n");
					$self->msg1 ($dbm->lastdialog ($id));
					$self->msg1 ("cancel load systab\n");
					$dbm->cancel ($operation);
					return undef;
				}
			}

			# look for dbfull or logfull
			$sql = $dbm->send ('sql_execute select '.
			'DATABASEFULL,LOGFULL from SYSDD.CTRL_CONFIGURATION');
		} elsif ($id == $sql) {
			# got sql output
			($rc, $padding, my @data) = 
			split ('\n', $dbm->recv ($sql));

			# we are no longer waiting for sql response
			$sql = 0;
			
			unless ($rc =~ /^OK/) {
				$self->msg1 ("got sql error:\n");
				$self->msg1 ("dbm command was:\n");
				$self->msg1 ("\n");
				$self->msg1 ($dbm->lastdialog ($id));
			} else {
				my ($dbfull, $logfull) =
				($data[0] =~ /^\'(Y|N)\'\;\'(Y|N)\'$/);

				if ($dbfull =~ /Y/) {
					$self->msg1
					("detected database full\n");
				} elsif ($logfull =~ /Y/) {
					$self->msg1
					("detected log full\n");
				}

				if ($dbfull =~ /Y/ || $logfull =~ /Y/) {
					$self->msg1 ("dbm command was:\n");
					$self->msg1 ("\n");
					$self->msg1 ($dbm->lastdialog ($id));
					$self->msg1 ("cancel load systab\n");
					$dbm->cancel ($operation);
					return undef;
				}
			}
		}
	}

	return undef unless ($rc =~ /^OK/);
	return $rc;	
}

#
# db_online
# works like SAPDB::Install::DBMCmd::db_online,
# but it uses 'show active us' every 30 seconds
# to find when something wents wrong
#
# db_warm is the same as db_online
#

*SAPDB::Install::Instance::Base::db_warm =
\&SAPDB::Install::Instance::Base::db_online;

sub db_online {
	my ($self) = @_;
	my ($id, $operation, $console);
	my ($rc, $padding, $state);
	my $dbm = $self->{dbm};

	my $cmd = $self->{can_db_online} ? 'db_online' : 'db_warm';

	$console = 0;
	$operation = $dbm->send ($cmd);
	for (;;) {
		# read console output every 30 seconds
		$id = $dbm->select (30);

		unless (defined $id) {
			# nothing to select or select error
			return undef;
		} elsif ($id == 0) {
			# timeout, send console command
			if ($console == 0) {
				$console = $dbm->send ('show active us');
			}
		} elsif ($id == $operation) {
			# operation finished, read result
			($rc) = split ('\n', $dbm->recv ($operation));
			return undef unless ($rc =~ /^OK/);

			# wait for results of pending commands
			while ($id = $dbm->select ()) {
				$dbm->recv ($id);
			}
			
			last;
		} elsif ($id == $console) {
			# got console output
			($rc, $padding, my @data) = 
			split ('\n', $dbm->recv ($console));

			# we are no longer waiting for console response
			$console = 0;

			#
			# look for console error
			#
			# cba 2001-11-07
			# I have seen that UNIX uses 'ERR <errid>'
			# and Windows uses '<errid> ERROR' 
			#
			if ($data[0] =~ /^ERR\s+\d+|^\d+\s+ERROR/) {
				$self->msg1 ("got console error:\n");
				foreach (@data) {
					$self->msg1 ("  ".$_."\n");
				}

				#
				# sometimes the output of console
				# failed because we cannot attach 
				# to shared memory
				# the we get an error message
				# 'Invalid argument'
				#	
				unless ($data[0] =~ /Invalid argument/i) {
					$self->msg1 ("dbm command was:\n");
					$self->msg1 ("\n");
					$self->msg1 ($dbm->lastdialog ($id));
					$self->msg1 ("cancel ".$cmd."\n");
					$dbm->cancel ($operation);
					return undef;
				}
			}
		}
	}

	#
	# get db state to make the output look like
	# SAPDB::Install::DBMCmd::db_warm
	#
	($rc, $padding, $state) = split ('\n', $dbm->exec ("db_state"));

	return undef unless ($rc =~ /^OK/);
	return $state;	
}

#
# switchto ()
# switch instance state and check success
#
sub switchto {
	my ($self, $want, $txt) = @_;
	my $dbm = $self->{dbm};

	$self->msg0 ($txt) if (defined ($txt));
	$got = 
	$want =~ /^WARM$/i    ? $self->db_warm () :
	$want =~ /^ONLINE$/i  ? $self->db_online () :
	$want =~ /^COLD$/i    ? $dbm->db_cold () :
	$want =~ /^ADMIN$/i   ? $dbm->db_admin () :
	$want =~ /^OFFLINE$/i ? $dbm->db_offline () :
	$want =~ /^CLEAR$/i   ? $dbm->db_clear () : 
	undef;
	
	$want = 'OFFLINE' if ($want =~ /^CLEAR$/i);

	unless (defined $self->{instancetypename}) {
		$self->{instancetypename} =  
		$self->{instancetype} =~ /LVC/ ? 'liveCache' : 'database';
	}

	unless (defined $got && $got eq $want) {
		$self->msg1 ("cannot switch ".$self->{instancetypename}.
		             " mode, error during dbm command:\n");
		$self->msg1 ($dbm->lastdialog ());
	}

	return ($got);
}
	
#
# switchto_prev
#
sub switchto_prev {
	my ($self, $state, $prev) = @_;

	return ($state) unless ($prev eq 'OFFLINE' && $state ne 'OFFLINE');

	unless (defined $self->{instancetypename}) {
		$self->{instancetypename} =  
		$self->{instancetype} =~ /LVC/ ? 'liveCache' : 'database';
	}

	$state = $self->switchto ('OFFLINE', "switch ".
	         $self->{instancetypename}." state back to OFFLINE\n"); 

	$self->set_errorstate ('ERROR') if ($state ne 'OFFLINE');
	return ($state);
}

#
# read error messages from knldiag after unsuccessful startup
#
sub get_errmsg_from_knldiag {
	my ($self) = @_;
	my $dbm = $self->{'dbm'};
	
	$self->msg1 ("looking for error messages in knldiag\n");
	my $dbmgetf = $dbm->dbmgetf ('KNLDIAG');
	unless (defined $dbmgetf) {
		$self->msg1 ($dbm->lastdialog ());
		return undef;
	}	

	#
	# read the first three header lines
	#
	my $line;
	$line = $dbmgetf->readline ();
	unless ($line =~ /^-+/) {
		$self->msg1 ("received unexpected first line in knldiag\n");
		return undef;
	}

	$line = $dbmgetf->readline ();
	unless ($line =~ /^Date/) {
		$self->msg1 ("received unexpected second line in knldiag\n");
		return undef;
	}
	my $header = $line;

	$line = $dbmgetf->readline ();
	unless ($line =~ /^-+/) {
		$self->msg1 ("received unexpected third line in knldiag\n");
		return undef;
	}

	#
	# split colum names and positions from header line
	#
	my @name = ();
	my @pos = ();
	foreach my $colname (split (' ', $header)) {
		$header =~ /$colname/;
		push @pos, length ($`);
		push @name, $colname;		
	}

	#
	# read every line
	#
	my $found_errmsg = 0;
	while ($line = $dbmgetf->readline ()) {
		if ($found_errmsg == 1) {
			$self->msg1 ($line."\n");
			next;
		}

		for (my $i = 0; $i <= $#pos; $i++) {
			# split it into colums
			my $val = ($i + 1 <= $#pos) ?
			substr ($line, $pos[$i], $pos[$i + 1] - $pos[$i]) :
			substr ($line, $pos[$i]);

			# get rid of spaces, except in message column
			unless ($name[$i] =~ /^Message/) {
				($val, my $space) = ($val =~ /^(.*\S)(\s+)$/);
			}
			
			# look for error messages
			if ($name[$i] =~ /^Typ/ && $val =~ /^ERR/) {
				$found_errmsg = 1;
				$self->msg1 ("...\n");
				$self->msg1 ($line."\n");
			} 			
		}
	}

	#
	# this is the end of knldiag
	#
	if ($found_errmsg == 1) {
		$self->msg1 ("...\n");
		$self->msg1 ("end of knldiag\n");
		return 0;
	}
	
	#
	# tell them that we cannot find any error messages
	#
	$self->msg1 ("no error messages found in knldiag\n");
	return 0;
}

#
# try to get kernel build string
# use 'show version' in cold or warm mode
# use what string or version resource in offline mode
#
sub get_kernelversion {
	my ($self) = @_; 
	my $dbm = $self->{dbm};

	my $state = $dbm->db_state ();
	unless (defined $state) {
		$self->msg1 ($dbm->lastdialog ());
		return (undef);
	}

	my $current = undef;
	if ($state =~ /WARM|COLD/) {
		# use 'show version' if kernel is running
		my @data = $dbm->show_version ();
		if (defined @data) {
			foreach (@data) {
				if ($_ =~
				/^Kernel Version  \'Kernel    (.+)\'$/) {
					$current = $1;
					last;
				}
			}
		}
	}
	
	if (defined $current) {
		$self->msg1 (
		"kernel version by 'show version' ".$current."\n");	
		return ($current);
	}

	#
	# get buildnumber for executable
	#
	my %data = $dbm->dbm_version ();
	unless (defined %data) {
		$self->msg1 ($dbm->lastdialog ());
		return (undef);
	}

	my $file = $data{'INSTROOT'};
	$file .= ($^O =~ /mswin/i) ? '/pgm/kernel.exe' : '/pgm/kernel';
	$self->msg1 ("kernel executable is ".$file."\n");	

	unless (-r $file) {
		$self->msg1 ("no read permission for ".$file."\n");
		return (undef);
	}

	my $current = SAPDB::Install::BuildInfo::GetBuildString ($file);
	unless (defined $current) {
		$self->msg1 ("cannot GetBuildString from ".$file."\n");	
		return (undef);
	}

	if ($current =~ /Rel\.\s+(.+)/) {
		$current = $1;
		$self->msg1
		("kernel version by GetBuildString ".$current."\n");
		return ($current);
	}

	$self->msg1 ("GetBuildString returned not a kernel release\n");	
	$self->msg1 ($current."\n");	
	return (undef);
}

sub get_apoversion () {
	my ($self) = @_; 
	my $dbm = $self->{dbm};

	#
	# get buildnumber for shared object
	#
	my %data = $dbm->dbm_version ();
	unless (defined %data) {
		$self->msg1 ($dbm->lastdialog ());
		return (undef);
	}

	my $sysinfo = SAPDB::Install::SysInfo::GetSystemInfo ();

	my $file = $data{'INSTROOT'};
	$file .= ($^O =~ /mswin/i) ? '/sap/sapapo.dll' : 
	         ($^O =~ /hpux/ && $sysinfo->{architecture} ne 'IA64') ?
			 '/sap/libSAPAPO.sl' :
	         '/sap/libSAPAPO.so';

	$self->msg1 ("sapapo lib is ".$file."\n");	

	unless (-r $file) {
		$self->msg1 ("no read permission for ".$file."\n");
		return (undef);
	}

	my $info = SAPDB::Install::BuildInfo::GetBuildInfo ($file);
	unless (exists %$info->{'Build'} &&
	ref (%$info->{'Build'}) eq 'ARRAY') {
		$self->msg1 ("GetBuildInfo could not get apo version\n");
		return (undef);
	}
	
	my $apo_version = join ('.', @{%$info->{'Build'}});
	$self->msg1 ("apo version by GetBuildInfo ".$apo_version."\n");
	
	return ($apo_version);
}

sub normalize_aporel {
	my ($buildstr) = @_;

	if (my ($a, $b, $c, $d) = 
	($buildstr =~ /^(\d+)\.(\d)([\d|A-Z])\s+Build\s+(\d+)/)) {
		$c = ($c =~ /[A-Z]/) ? ord ($c) - ord ('A') : $c;
		$buildstr = $a.'.'.$b.'.'.$c.'.'.$d;
	}
	
	return ($buildstr);
}

1;

