
/* @(#) $Id: //sapdb/V75/c_00/b_34/sys/src/sap/640/nixx.h#1 $ SAP*/
/*####################################################################*/
/*
 *     SAP AG Walldorf
 *     Systeme, Anwendungen und Produkte in der Datenverarbeitung
 *
 *     (C) Copyright SAP AG 1991 - 1997
 */
/*####################################################################*/



/*include-------------------------------------------------------------*/
/*                                                                    */
/* Beschreibung:                                                      */
/* ============                                                       */
/*                                                                    */
/* nixx.h:   ni = Projekt Network-Interface                           */
/*           xx = OS-unabhaengiges Coding                             */
/*                                                                    */
/* nixx.h enthaelt die Deklarationen fuer das Network-Interface.      */
/* Das Network-Interface bietet Dienste fuer die Interprozess-        */
/* Kommunikation ueber Rechnergrenzen hinweg an.                      */
/*                                                                    */
/* Ein einfaches Beispielprogramm fuer Client und Server ist          */
/* niexample (Modul niexampl.c). Hier findet man auch einfuehrende    */
/* Worte in die Programmierung mit NI.                                */
/*                                                                    */
/* Die ni-Bibliothek (nilib.o, Makefile ni.mak) kann mit              */
/* dem Programm 'niping' getestet werden (Modul nixxtst.c, Makefile   */
/* ni.mak)                                                            */
/*                                                                    */
/*-NiQue--------------------------------------------------------------*/
/*                                                                    */
/* Verbergen der Fragmentierung mit interner Queue (NiBuf):           */
/* ========================================================           */
/*                                                                    */
/* Die alte NI-Schicht verbirgt die Fragmentierung der                */
/* Pakete durch die Netzwerkschicht nicht.                            */
/* Dies fuehrt zu immer gleichem Coding in den Anwendungen.           */
/* Dies kann jetzt umgangen werden. Dafuer gibt es                    */
/* verschiedene Moeglichkeiten:                                       */
/*                                                                    */
/* 1) Man ersetzt alle Aufrufe von NiInit, NiConnect und              */
/*    NiListen durch die entsprechenden NiBuf*                        */
/*    Aufrufe. Ansonsten braucht man nichts zu aendern.               */
/* 2) In einer zweiten Stufe kann man die NiRead und                  */
/*    NiWrite-Aufrufe durch andere Calls ersetzen, die                */
/*    verhindern, dass die Daten kopiert werden muessen.              */
/*    Siehe unten "Neue Puffer-API"                                   */
/*                                                                    */
/* Es wird dann sichergestellt, dass entweder das ganze Paket oder    */
/* gar nichts verarbeitet wird.                                       */
/* Durch setzen der Parameter NIBUFP_MAXQUEUE und NIBUFP_MAXHEAP kann */
/* man die interne Queue aktivieren. Hierbei bedeutet NIBUFP_MAXQUEUE */
/* die maximale Queuelaenge pro handle und NIBUFP_MAXHEAP die         */
/* maximale Groesse aller Queues.                                     */
/* Falls die Queue voll ist wird zunaechst NIEQUE_FULL                */
/* zurueckgegeben, das Paket aber verarbeitet. Erst beim naechsten    */
/* Versuch wird mit NIETIMEOUT signalisiert, dass das Paket nicht     */
/* verarbeitet wurde. Nach NIEQUE_FULL sollte man mit NiWriteOK       */
/* sicherstellen, dass die Queue abgearbeitet wurde.                  */
/* Hinweis: Bei Ueberschreiten von NIBUFP_MAXHEAP wird immer nur      */
/* NIEQUE_FULL zurueckgegeben, aber die Queues trotzdem               */
/* erweitert. Dies ist nur ein Softlimit.                             */
/* NIEPING wird nur noch signalisiert, wenn als Antwort auf ein       */
/* NICheck direkt NI_PONG empfangen wird. D.h., wer NiCheck nicht     */
/* aufruft, bekommt NIEPING nie als Fehler zurueck                    */
/*                                                                    */
/* NiBufInit           Initialisierung der nilib und initialisierung  */
/*                     der internen Pufferung. Diese Funktion muss vor*/
/*                     den anderen NiBuf-Aufrufen gerufen werden      */
/* NiBufListen         Vorbereitungen fuer Connect-Requests von       */
/*                     Clients treffen (Server). Die per NiAccept auf */
/*                     diesem Handle erzeugten NI-Handles verbergen   */
/*                     die Fragmentierung                             */
/* NiBufConnect        Verbindung zu Server aufbauen. Der             */
/*                     resultierende NI-Handle verbirgt die           */
/*                     Fragmentierung                                 */
/*                                                                    */
/* Nonblocking Connects:                                              */
/* ====================                                               */
/* Ab Version 31 unterstuetzt NiBuf nonblocking Connects auf          */
/* Plattformen, die das bereitstellen:                                */
/* Nach NiBuf*Connect kann NIECONN_PENDING ignoriert werden und die   */
/* Pakete werden dann in die Queue gestellt. (Achtung: Wer            */
/* NIECONN_PENDING _nicht_ ignoriert, muss den Handle selber          */
/* schliessen!)                                                       */
/* Es ist damit moeglich Connects mit timeout 0 (NI_ASYNC) aufzurufen.*/
/*                                                                    */
/*-NiBuf--------------------------------------------------------NiQue-*/
/*                                                                    */
/* Neue Puffer-API:                                                   */
/* ===============                                                    */
/*                                                                    */
/* Diese API basiert auf der NiBuf-ERweiterung und bietet ein paar    */
/* neue Funktionen, die NiRead/Write ersetzen, bzw. Hilfsfunktionen   */
/* dazu.                                                              */
/* Diese Funktionen arbeiten nicht mit beliebigen Datenbereichen, in  */
/* die und aus denen dann kopiert werden muss, sondern es werden      */
/* Pointer auf Strukturen vom Typ NIBUFFER uebergeben. Dies fuehrt zu */
/* deutlich kuerzeren Laufzeiten als bei NiRead/Write                 */
/*                                                                    */
/* Die Structur NIBUFFER hat zwei oeffentliche Variablen:             */
/* struct NIBUFFER                                                    */
/* {                                                                  */
/*    SAP_RAW *data;                                                  */
/*    SAP_INT  len;                                                   */
/*    ...                                                             */
/* }                                                                  */
/* data ist ein Zeiger auf den Anfang des Datenbereichs und len ist   */
/* die Laenge desselben.                                              */
/*                                                                    */
/* Man erhaelt solche Puffer implizit ueber NiBufReceive und kann sie */
/* explizit ueber NiBufAlloc anfordern                                */
/* Diese Puffer muessen dann aber auch wieder freigegeben werden.     */
/* Dies geschieht wieder implizit ueber NiBufSend oder explizit ueber */
/* NiBufFree.                                                         */
/* Genauso,wie bei NiBufRead/Write wird garantiert, dass ein NiBufSend*/
/* nach einem NiBufReceive nicht mit NIETIMEOUT zurueckkehrt.         */
/* Diese Funktionen koennen auch mit NiRead/Write vermischt werden.   */
/*                                                                    */
/* Funktionen:                                                        */
/*                                                                    */
/* NiBufReceive    Empfange einen NIBUFFER                            */
/* NiBufSend       Sende einen NIBUFFER                               */
/*                                                                    */
/* NiBufFree       Gebe einen NIBUFFER wieder frei                    */
/* NiBufAlloc      Allokiere einen NIBUFFER                           */
/* NiBufWrap       'Verpacke' einen Datenbereich in einem NI-Puffer   */
/* NiBufUnWrap     'Entpacke' einen mit NiBufWrap                     */
/*                            erzeugten NI-Puffer                     */
/* NiBufDup        Dupliziere einen NIBUFFER                          */
/*                                                                    */
/* NiBufPush       Lege einen Puffer auf einen Stack                  */
/* NiBufPop        Hole einen Puffer vom Stack                        */
/*                                                                    */
/* Ueber den Handle-Parameter NIHP_ALLOC_FUNC kann man eine Funktion  */
/* registrieren, die beim Empfangen eines Pakets gerufen wird, um     */
/* einen Puffer zu allokieren.                                        */
/* Dies ermoeglicht zusammen mit NiBufWrap, auch z.B. direkt aus      */
/* shared Memory heraus zu senden, ohne kopieren zu muessen           */
/*                                                                    */
/*--------------------------------------------------------------NiBuf-*/
/*-NiSel--------------------------------------------------------------*/
/*                                                                    */
/* Um die folgenden  Funktionen nutzen zu koennen, muss zusaetzlich   */
/* zur nilib die nisellib und die filib gebunden werden.              */
/*                                                                    */
/*--------------------------------------------------------------------*/
/*                                                                    */
/* Neue Select-API:                                                   */
/* ===============                                                    */
/*                                                                    */
/* Das alte NiSelect ist dadurch sehr langsam, dass es einen          */
/* eigenen Typ fuer die selectmasken definiert. Diese muessen         */
/* bei jedem Aufruf in die socket-masken umgesetzt                    */
/* werden.                                                            */
/*                                                                    */
/* Die neue API verwaltet jetzt nur noch intern                       */
/* socket-Masken. Ausserdem kann der Aufrufer eigene Daten            */
/* hinterlegen, die den Zugriff auf seine Daten beschleunigt.         */
/*                                                                    */
/* Dafuer muessen die Selectsets jetzt angefordert und                */
/* explizit wieder freigegeben werden.                                */
/*                                                                    */
/* Ein typischer Ablauf waere:                                        */
/* {                                                                  */
/*   NISEL_HDL set;                                                   */
/*   SAP_BOOL  read;                                                  */
/*                                                                    */
/*   NiSelNewSet (&set);                                              */
/*                                                                    */
/*   NiSelSet (set, nihdl, NI_CONNECT, NULL);                         */
/*   [...]                                                            */
/*   while (something_to_do)                                          */
/*   {                                                                */
/*     NiSelSelect (set, timeout);                                    */
/*     while (NiSelNext (set, &read, NULL, &conn, &myclient)          */
/*            != NI_INVALID_HDL)                                      */
/*     {                                                              */
/*       if (conn)                                                    */
/*       {                                                            */
/*         MyConnect(); // hier wird irgendwo                         */
/*            // NiSelSet (set, newnihdl, NI_READ, myclient)          */
/*            // gerufen                                              */
/*       }                                                            */
/*       if (read)                                                    */
/*         Myread (myclient);// hier wird evtl.                       */
/*            // NiSelSet (set, myclient->nihdl, NI_WRITE,            */
/*            //           myclient)                                  */
/*            // gerufen                                              */
/*       if (write)                                                   */
/*         MyWrite (myclient);                                        */
/*     }                                                              */
/*   }                                                                */
/*   NiSelDestroySet (&set);                                          */
/* }                                                                  */
/*                                                                    */
/*-NiSnc--------------------------------------------------------NiSel-*/
/*                                                                    */
/* Transparente Integration von SNC (Secure Network Communication):   */
/* ===============================================================    */
/*                                                                    */
/* Es ist moeglich, NI-vebindungen verschluesseln zu lassen. Hierzu   */
/* wurde NI so erweitert, dass die SNC-Library transparent gerufen    */
/* wird.                                                              */
/* Die Loesung basiert auf NiBuf (s.o.). Es muessen NiInit, NiConnect */
/* und NiListen durch NiSncInit, NiSncConnect und NiBufListen ersetzt */
/* werden.                                                            */
/* NiSncInit erhaelt als Parameter den eigenen SNC-Namen.             */
/* NiConnect erhaelt den SNC-Namen des Partners. Falls Dieser leer    */
/* ist, verhaelt sich NiSncConnect wie NiBufConnect                   */
/* Nach dem NiAccept kann man, nachdem der Handle Schreib-oder        */
/* Lesebereit ist, per NiSncGetPeer erfragen, ob die Verbindung       */
/* verschluesselt ist und wie der Partner heisst                      */
/* Um diese Funktionen nutzen zu koennen, muss zusaetzlich zur nilib  */
/* die nisnclib gebunden werden. Diese braucht zusaetzlich die filib, */
/* snclib und sappar0/sapparam.                                       */
/*                                                                    */
/* NiSncInit          Ersetzt NiInit (muss gerufen werden)            */
/* NiSncConnect       Ersetzt NiConnect                               */
/* NiSnc2Connect      Ersetzt Ni2onnect                               */
/* NiSncGetPeer       Gibt den evtuellen SNC-Account des Partners     */
/*                    zurueck                                         */
/* NiSncRequest       Fordert eine SNC-Verschluesselung zu            */
/*                    bestehender Verbindung an                       */
/* NiSncAclStr        Gibt einen lesbaren String fuer                 */
/*                    einen SNC-Account zurueck                       */
/* NiSncNameToAcl     Macht aus einem lesbaren SNC-Namen einen        */
/*                    internen                                        */
/* NiSncAclCmp        Vergleicht zwei interne SNC-Namen               */
/* NiSncFinHS         Versucht den Austausch von SNC-Overhead         */
/*                    anzustossen                                     */
/*                                                                    */
/*--------------------------------------------------------------NiSnc-*/
/*-NiSendHandle-------------------------------------------------------*/
/*                                                                    */
/* Mit der Funktion NiSendHandle kann ein Ende einer socket-Verbindung*/
/* ueber eine zweite lokale socket-Verbindung verschickt werden.      */
/* Diese Funktionalitaet ist fuer Unix und Windows NT (mit WinSock2)  */
/* implementiert.                                                     */
/* Die Uebertragung erfolgt in den mehreren Einzelschritten. Der      */
/* client ruft NiSendHandle, um den socket zu uebertragen.            */
/* Die empfangende Seite muss den Socket, ueber den empfangen werden  */
/* soll, in der read-Maske haben. Die Uebertragung wird in der select */
/* loop des servers mit dem status 'connect' angezeigt. Der server    */
/* kann die Verbindung mit NiAccept annehmen und anschliesend ueber   */
/* die empfangene Verbindung kommunizieren.                           */
/*                                                                    */
/*-------------------------------------------------------NiSendHandle-*/
/*--------------------------------------------------------------------*/
/*                                                                    */
/* allgemeine Dienste                                                 */
/* ==================                                                 */
/* NiInit              Initialisierung der nilib                      */
/*                     Sollte immer vor Benutzen von Ni-Funktionen    */
/*                     gerufen werden.                                */
/* NiThrInit           Thread specific initialization                 */
/* NiExit              Aufraeumen der Resourcen                       */
/* NiThrExit           Thread specific cleanup                        */
/* NiCloseHandle       Handle freigeben, evtl. Verbindung abbauen     */
/* NiShutdownHandle    Verbindung einseitig schliessen                */
/* NiCloseAll          alle offenen Handles freigeben                 */
/* NiSelect            Auf Daten warten                               */
/* Ni2Select           Auf Daten und freie Sendeleitungen warten      */
/* NiPeek              Testen, ob Daten fuer Handle angekommen        */
/* NiWriteOK           Testen, ob Write auf Handle blockieren wuerde  */
/* NiRouteToTable      Zerlege einen Routestring                      */
/* NiHostToAddr        Hostnamen in Adresse umwandeln                 */
/* NiAddrToHost        Adresse in Hostnamen umwandeln                 */
/* NiAddrToHost2       Adresse in Hostnamen umwandeln (reentrant)     */
/* NiServToNo          Service-Namen in (Port-)Nummer umwandeln       */
/* NiNoToServ          (Port-)Nummer in Service-Namen umwandeln       */
/* NiNoToServ2         (Port-)Nummer in Service-Namen umw. (reentrant)*/
/* NiMyHostName        eigenen Hostnamen ermitteln                    */
/* NiMyAddrVerify      eigene Host-Adresse verifizieren               */
/* NiReadPending       noch zu lesende Bytes (letztes Paket)          */
/* NiWritePending      noch zu schreibende Bytes (letztes Paket)      */
/* NiWait              n Millisekunden warten                         */
/* NiGetParam          allg. NI-Parameter lesen                       */
/* NiSetParam          allg. NI-Parameter setzen                      */
/* NiGetParamEx        allg. NI-Parameter lesen (beliebige Datentypen)*/
/* NiSetParamEx        allg. NI-Parameter setzen(beliebige Datentypen)*/
/* NiHBufPrintBuf      Dump the Ni-Hostbuffer to a (UC) Char - buffer */
/* NiGetSapDllRelease  gibt SAP Release der NIDLL zurueck             */
/* NiErrSet            Sap-Error-Info zu NI-Returncode bereitstellen  */
/* NiErrStr            Namens-String  zu Ni-Error-Konstante liefern   */
/* NiSecStr            SapRouter-Passwort aus String ausblenden       */
/* NiSecStr2           SapRouter-Passwort aus String ausb. (reentrant)*/
/* NiHdlToSock         Gebe den Socket zu einem NI-Handle zurueck     */
/* NiSockToHdl         Gebe den NiHandle zu einem Socket zurueck      */
/* NiHdlGetPeer        Gebe den Remote-Peer zu einem Handle zurueck   */
/* NiHdlGetName        Gebe die lokale Adresse zu einem Handle zurueck*/
/* NiHdlSetBlockmode   Hdl blocking oder nonblocking setzen           */
/* NiHdlGetBlockmode   pruefen ob Hdl blockierend oder nicht ist      */
/* NiHdlGetStatus      Gebe den Verbindungsstatus eines Handle zurueck*/
/* NiLocalCheck        Check ob NI_HOST_ADR lokal ist                 */
/* NiGet<level>_<opt>  Get socket option for specified protocol level */
/* NiSet<level>_<opt>  Set socket option for specified protocol level */
/* NiSetTraceAll       NI-Trace fuer alle activen Handles aktivieren  */
/* NiClearTraceAll     NI-Trace fuer alle activen Handles loeschen    */
/* NiHdlDump           Informationen fuer ein Handle ins Trace dumpen */
/* NiDumpNiHdls        Informationen fuer alle aktiven Handles dumpen */
/*                                                                    */
/* Dienste f. verbindungsorientierte Kommunikation (Stream, "Telefon")*/
/* ===================================================================*/
/* NiListen            Vorbereitungen fuer Connect-Requests von       */
/*                     Clients treffen (Server)                       */
/* NiConnect           Verbindung zu Server aufbauen                  */
/* NiAccept            Verbindung zu Client aufbauen                  */
/* NiWrite             Daten an Partner senden                        */
/* NiRead              Daten vom Partner empfangen                    */
/* NiCheck             Verbindung zum Partner ueberpruefen            */
/* Ni2Listen           wie NiListen, aber Service-Nummer als Input    */
/* Ni2Connect          wie NiConnect, Host-Adr + Serv-Nr als Input    */
/*                                                                    */
/* Alle Funktionen fuer Native TCP/IP Verbindungen nutzen direkt      */
/* die Systemfunktionen und benutzen NICHT die Pufferung (NiBuf):     */
/* NiRawConnect        wie NiConnect, Talkmode nach Verbindungsaufbau */
/*                     Native statt NI (fuer SapTelnet ..)            */
/* NiRawWrite          Daten an "Native-TCP-Partner" senden           */
/* NiRawRead           Daten von "Native-TCP-Partner" empfangen       */
/* NiRawPeek           Testen, ob Daten fuer Handle angekommen        */
/* NiRawWriteOK        Testen, ob Write auf Handle blockieren wuerde  */
/* NiHdlGetOutOctets   Anzahl der ueber Handle gesendeten Bytes       */
/* NiHdlGetInOctets    Anzahl der ueber Handle empfangenen Bytes      */
/* NiHdlGetOutMsgs     Anzahl der ueber Handle gesendeten Ni-Messages */
/* NiHdlGetInMsgs      Anzahl der ueber Handle empfangenen Ni-Messages*/
/* NiHdlGetStartTime   Startzeit der Verbindung zurueckgeben          */
/*                                                                    */
/*                                                                    */
/* Dienste fuer verbindungslose Kommunikation (Datagramm, "Brief")    */
/* ===============================================================    */
/* NiDgBind            Vorbereitungen fuer Empfang von Datagrammen    */
/*                     treffen                                        */
/* NiDgReceive         Datagramm empfangen                            */
/* NiDgSend            Datagramm senden                               */
/* NiDgGetHdl          intern benutzten Handle abfragen               */
/* NiDgFreeResrc       belegte "Datagramm-Resourcen" wieder freigeben */
/* NiDg2Send           wie NiDgSend, aber Host-Adresse als Input      */
/*-NiDatagramAPI------------------------------------------------------*/
/*                                                                    */
/* NiDgHdlCreate       Datagramm Handle erzeugen                      */
/* NiDgHdlBindName     Datagramm-Handle an IP/Port binden             */
/* NiDgHdlSendToName   Datagramm an fernen IP/Port schicken           */
/* NiDgHdlSendTo       Datagramm an fernen IP/Port schicken           */
/* NiDgHdlRecvFrom     Datagramm vom Netz empfangen                   */
/* NiDgHdlConnect      Datagramm-Handle an IP-Adresse verbinden       */
/* NiDgHdlConnectName  Datagramm-Handle an Hostnamen  verbinden       */
/* NiDgHdlSend         Daten an verbundenen Datagram-Handle schicken  */
/* NiDgHdlClose        Datagramm-Handle schliessen                    */
/*                                                                    */
/* -----------------------------------------------------NiDatagramAPI-*/
/* NiWakeupOpen        Wakeup Server erstellen                        */
/* NiWakeupClose       Wakeup Server schliessen                       */
/* NiWakeupGetNiHdl    Listen Handle des Wakeup Servers lesen         */
/* NiWakeupGetCntNo    Anzahl Clients lesen                           */
/* NiWakeupCollect     Empfangene Wakeup Requests lesen               */
/* NiWakeupAttach      Wakeup Client an Server binden                 */
/* NiWakeupDetach      Wakeup Client von Server trennen               */
/* NiWakeupExec        Wakeup Request an Server schicken              */
/*                                                                    */
/*-NiMulticast--------------------------------------------------------*/
/*                                                                    */
/* You may use NI in order to receive (IP-)multicast streams.         */
/* IP multicast is a mechanism to send data to multiple peers         */
/* simultaneously. Any peer wishing to receive data for a particular  */
/* group has to join that group. With IP groups are identified with   */
/* special IP-Addresses                                               */
/* (Class D-Addresses 224.0.0.1 - 239.255.255.255                     */
/*  high order byte starting with 1110 )                              */
/*                                                                    */
/* In order order to receive Multicast you have to:                   */
/*     Create a  Datagram Handle               (NiDgHdlCreate)        */
/*     Bind that handle to a port              (NiDgHdlBind/-Name)    */
/*     Join the group                          (NiDgHdlJoinGroup)     */
/*     Receive Packets for that group          (NiDgHdlRecvFrom)      */
/*                                                                    */
/* The sender may use the NiDgHdl API to send datagrams to a          */
/* particular Multicast Group:                                        */
/*     Create a  Datagram Handle               (NiDgHdlCreate)        */
/*     Send to  the group                      (NiDgHdlSendTo)        */
/*                                                                    */
/* Example: see nidgtst.c  (mapro krn/ni nidgtst)                     */
/*--------------------------------------------------------NiMulticast-*/
/* Dienste fuer 1-n Kommunikation (Multicast)                         */
/*====================================================================*/
/*                                                                    */
/* NiIsMulticastAddress pruefen ob Multicastaddress vorliegt          */
/* NiDgHdlJoinGroup     einer (IP-) Multicastgruppe beitreten         */
/* NiDgHdlLeaveGroup    eine  (IP-) Multicastgruppe verlassen         */
/*                                                                    */
/*end-----------------------------------------------------------------*/

#ifndef NIXX_H
#define NIXX_H

#ifdef __cplusplus
extern "C"
{
#endif


/*--------------------------------------------------------------------*/
/* Version der NI-Schicht                                             */
/*                                                                    */
/* NI_VERSION wird bei allen Aenderungen der NI-Schicht (inklusive    */
/* SapRouter) inkrementiert und bei NI-internen Messages uebers Netz  */
/* mitgesendet                                                        */
/*                                                                    */
/* 13  02.12.94 hq   Interface Erweiterung NiSetParam(), NiGetParam() */
/* 14  21.12.94 hq   Host/Serv-Buf VALID/UNKNOWN/alloc/flush/dump     */
/*                   NiSetParam() BUF_OPCOD, HSBUF_MAX, LIB_VERS      */
/*                   NiAdrToStr(), NiSrvToStr() speichern 10 Strings  */
/*                   SapRouter Queue Pre-Alloc                        */
/* 15  03.01.95 hq   Interface Erweiterung NiMyAddrVerify ()          */
/* 16  16.03.95 hq   Interface Erweiterung NiRawConnect()             */
/*                   fragmentierte Header jetzt erlaubt               */
/* 17  25.04.95 hq   niping client delay wg. keepalive Tests          */
/*                   SapRouter Def-Werte fuer -B /-Q von 100KB / 10MB */
/*                   auf 500 KB / 20 MB                               */
/*     22.05.95 hq   Def. fuer Routtab-Laenge von 1000 auf 2000       */
/* 18  06.09.95 hq   Maske von 21 auf 32 Int's erweitert              */
/*     18.09.95 hq   NiErrStr () aufgenommen                          */
/*     05.10.95 hq   NiSecStr () aufgenommen                          */
/*     16.10.95 hq   NiGetSapDllRelease () aufgenommen                */
/* 19  17.10.95 hq   NiPBind () und NiBind () erweitert (log-Param)   */
/*                   SapRouter Portrange-Feature                      */
/* 20  30.10.95 hq   SapRouter Bug unter NT gefixt (close in NiRAdmin)*/
/* 21  13.11.95 hq   ErrInfo-Makros ueberall eingebaut                */
/*                   NiPing SNC-Schicht integriert                    */
/*                   SAProuter RT_ROUTTAB_MAX_NO von 2000 auf 3000    */
/*                   SAProuter Permission Field Gross / Klein         */
/*     07.12.95 hq   Patch wg. ErrInfo in NiICloseHandle ...          */
/* 22  11.12.95 hq   Vorbereitung fur Umstellung auf Handshake-Protok.*/
/*                   NiIConnect als Zustandsautomat                   */
/* 23  20.12.95 hq   Bug in Connect-Fktn behoben (INVALID_HDL setzen) */
/* 24  27.02.96 hq   ErrInfo analog errno-Konzept, ni_compname global */
/*                   API's setzen bei Fehlern immer ErrInfo, machen   */
/*                   nie ErrClear, ErrSet ueberschreibt immer         */
/*                   SAProuter bug (Routtab vorm Neueinlesen loeschen)*/
/* 25  01.04.96 hq   SAProuter kommt runter, wenn keine Routtab da    */
/*                   oder wenn kein Log geschrieben werden kann       */
/* 26  06.05.96 cr   NiTcpConnect realisiert                          */
/*                   Passwordverfahren fuer SAPSERVX                  */
/*                   Saprouter von sapparam unabhaengig               */
/*     15.10.96	cr   NiRouteToTable                                   */
/*     29.10.96 cr   Saprouter laesst sich auch bei maximaler Anzahl  */
/*                   von clients administrieren		              */
/*                   Default fuer die maximale Anzahl erhoeht auf 800 */
/*     05.11.96 cr   Mehr Sicherheit bei der maximalen Anzahl	      */
/*		     Dynamische Anpassung der Routtab		      */
/*		     NI_MAX_HDLS auf 2048			      */
/*     02.12.96 cr   Saprouter gibt Statusmeldung in Admin-requests   */
/*                   zurueck					      */
/*     04.02.97 cr   Saprouter ersetzt in der iroute seinen Hostname  */
/*                   durch die IP-Addresse des Vorgaengers	      */
/* 27  12.11.96 rw   NiPRead, NiPWrite bekommen jetzt Flag fuer	      */
/*                   nonblocking				      */
/*     02.01.97 cr   Threadsafe: NiRouteAdd, NiRouteConnect gestrichen*/
/*     31.01.97 hq   fragmentierte Header werden sauber behandelt     */
/*	             Fragment-Test-Modus gebaut (siehe NiSetParam (), */
/*	             niping -Q)                                       */
/*     12.02.97 cr   Threadsafe als Compileroption		      */
/* 28  21.03.97 hq   SNC in SAProuter eingebaut (Alpha Version !)     */
/*     25.04.97 cr   SNC in Saprouter erweitert                       */
/*                   NiSel-API realisiert       (Alpha Version !)     */
/*     20.06.97 cr   Administration des Saprouter von Remote im Server*/
/*                   verhindern                                       */
/* 29  11.07.97 cr   NiBuf und NiSnc realisiert und im Saprouter      */
/*                   integriert                                       */
/* 30  23.07.97 cr   Soft Shutdown im Router                          */
/*                   Administration ueber neue Requests anfordern     */
/*                   Workingdirectory und saprouttab im INFO-request  */
/*                   ausgeben                                         */
/*     29.07.97 cr   NiSnc, Raw-Modus und Test von NiBuf in niping    */
/*     13.08.97 cr   NiRawRead gab zu frueh NIECONN_BROKEN zurueck    */
/*     15.08.97 cr   Mehrfache FD_SET's vermeiden fuer NT             */
/*                   NiBuf mit in die nilib                           */
/*     03.09.97 cr   NI_FINDSERV_OS implementiert fuer sapftp         */
/*                   REUSEADDR unter NT abgeschaltet                  */
/*     21.10.97 cr   Parsing der Source-Addresse im Router repariert  */
/* 31  31.10.97 cr   NiQueue implementiert                            */
/*     27.11.97 cr   Nonblocking Connect                              */
/*     03.12.97 cr   Server koennen jetzt auch Verschluesselung       */
/*                   nachfordern.                                     */
/*                   Hostnamenpuffer kann das Retry fuer revers       */
/*                   lookups abschalten                               */
/*                   Local bind ist jetzt moeglich.                   */
/*                   (diese drei Features werden im Saprouter jetzt   */
/*                   genutzt)                                         */
/*     11.12.97 cr   NiBufWrap/NiBufDup                               */
/*                   Nonblocking connect mit extra Funktion           */
/*     12.01.98 cr   EWOULDBLOCK auch pruefen                         */
/* 32  17.09.98 cr   Local port selection fixed                       */
/*                   Saprouter uses NT-Eventlog for fatal errors      */
/* 33  19.02.98 cr   NiDgSend optimiert                               */
/*                   AF_UNIX fuer localhost-Kommunikation             */
/*     11.05.98 cr   NiDgSend war nicht optimal                       */
/*                   Ni{Host,Net}INT in nixx.h                        */
/*     16.07.98 cr   Interne Handles auf Pointer umgestellt           */
/*                   Queuelimit jetzt per Packet und nicht per Byte   */
/*     18.02.99 ol   NiGetHostAddrList eingebaut                      */
/*      7.06.99 ol   NiCheck2 implementiert                           */
/*      8.07.99 ol   Konfigurierbarer Pfad fuer Unix Domain Sockets   */
/* 34  30.08.99 ol   SNC Fehler mit SAProuter (NIVERSION 34)          */
/*      9.08.99 ol   NiRawPeek und NiRawWriteOK implementiert. NiPeek */
/*                   und NiWriteOK fuellen intern die Puffer, die dann*/
/*                   fuer NiRawRead verloren gehen.                   */
/*     11.10.99 ol   NiShutdownHandle implementiert                   */
/*     30.03.00 cps  NiSncFinHS implementiert und NiDgSend            */
/*                   erneut verbessert                                */
/* 35  10.04.00 cps  SAProuter kann jetzt Prozesse nachstarten,       */
/*                   falls keine Slots mehr frei sind (UNIX only)     */
/*              ol   FI fuer die Suche nach freien slots in nitab     */
/*              cps  Poll statt Select unter UNIX                     */
/* 36  12.11.01 cps  queue Test als Parameter                         */
/*              cps  maximale Message Laenge  als Parameter           */
/*                   im saprouter beim Verbindungsaufbau realisiert   */
/*              cps  Versionierung des saprouter                      */
/*              cps  Erweitertes tracing mit NiHandles                */
/* 37  26.02.02 cps  UNIX jetzt ohne select                           */
/*                   nitab dynamisch allokiert, #hdls > 2048 moeglich */
/*              cps  Ermitteln der maximalen Anzahl von sockets, die  */
/*                   man maximal  oeffnen kann implementiert          */
/*              cps  NI_MAX_HDLS auf 16K, maximale Anzahl der Handles */
/*                   jetzt als Parmameter in NiInit2                  */
/*     09.04.02 cps  Datagramm Schnittstelle/ Multicast implementiert */ 
/*     17.05.02 ms   Ni*Raw2Connect eingebaut                         */
/*     22.05.02 ms   Neuer Parameter-Typ fuer HostAdressen            */
/*     26.07.02 ms   Wakeup mechanism                                 */
/*     30.08.02 ms   Functions to get / set socket options            */
/*     11.09.02 ms   Reentrant hostname / address caching             */
/*     12.11.02 ms   Update function specification                    */
/*     13.08.03 ms   New function parameter for NiSncAclStr           */
/*                                                                    */
/*--------------------------------------------------------------------*/

#define NI_COMPNAME		cU("NI (network interface)")
#define NI_VERSION		37
#define NI_LAST_MODIFIED	cU("Aug 13 2002")

/* ################################################################## */
/* # constants                                                      # */
/* ################################################################## */

/* Fehler-Konstanten--------------------------------------------------*/
#define NIEINTERN            -1    /* NI-interner Fehler              */
#define NIEHOST_UNKNOWN      -2    /* Hostname unbekannt              */
#define NIESERV_UNKNOWN      -3    /* Service unbekannt               */
#define NIESERV_USED         -4    /* Service already used            */
#define NIETIMEOUT           -5    /* Zeitlimit ueberschritten        */
#define NIECONN_BROKEN       -6    /* Verbindung zum Partner abgebr.  */
#define NIETOO_SMALL         -7    /* Daten-bereich zu klein          */
#define NIEINVAL             -8    /* ungueltiger Parameter           */
#define NIEWAKEUP            -9    /* Wake-Up (ohne Daten)            */
#define NIECONN_REFUSED      -10   /* Verbindungsaufbau misslungen    */
#define NIEPING              -11   /* PING/PONG-Signal erhalten       */
#define NIECONN_PENDING      -12   /* Verbindung zum Partner ueber    */
				   /* NiRouter noch nicht aufgebaut   */
#define NIEVERSION           -13   /* ungueltige Version              */
#define NIEMYHOSTNAME        -14   /* eigener Hostname nicht ermittelb*/
#define NIENOFREEPORT        -15   /* kein freier Port innerhalb Range*/
#define NIEMYHOST_VERIFY     -16   /* eigener Hostname ungueltig      */
#define NIESNC_FAILURE       -17   /* Fehler in der SNC Schicht       */
				   /* im saprouter ==>                */
#define NIEOPCODE            -18   /* Opcode received                 */
#define NIEQUE_FULL          -19   /* Queue limit erreicht, naechstes */
                                   /* Paket wird nicht akzeptiert     */ 
#define NIETOO_BIG           -20   /* Angefordertes Paket zu gross    */
#define NIEACCEPT            -21   /* Accepted handle pending         */
#define NIEPASSED            -22   /* Handle passed to other process  */
#define NIENO_ANSWER         -23   /* Partner hat nicht geantwortet   */
#define NIECALL              -24   /* Ungueltiger Funktionsaufruf     */

#define NIEROUT_HOST_UNKNOWN -90   /* Hostname unbekannt              */
#define NIEROUT_SERV_UNKNOWN -91   /* Service unbekannt               */
#define NIEROUT_CONN_REFUSED -92   /* Verbindungsaufbau misslungen    */
#define NIEROUT_INTERN       -93   /* NI-interner Fehler              */
#define NIEROUT_PERM_DENIED  -94   /* Connect von Source an           */
				   /* Destination verboten            */
#define NIEROUT_CONN_BROKEN  -95   /* Verbindung abgebrochen          */
#define NIEROUT_VERSION      -96   /* ungueltige Client-Version       */
#define NIEROUT_CANCELED     -97   /* Verbindung von Administrator    */
				   /* ge"canceled"                    */
#define NIEROUT_SHUTDOWN     -98   /* saprouter shutdown              */
#define NIEROUT_INFO_DENIED  -99   /* Informations Request abgelehnt  */
#define NIEROUT_OVERFLOW    -100   /* max. Anzahl Clients erreicht    */
#define NIEROUT_MODE_DENIED -101   /* talkmode not allowed            */
#define NIEROUT_NOCLIENT    -102   /* client nicht vorhanden          */
#define NIEROUT_EXTERN      -103   /* error in external library       */
#define NIEROUT_SNC_FAILURE -104   /* Fehler in der SNC Schicht       */

#define NI_ISROUTERR(nirc)   ((nirc) <= NIEROUT_HOST_UNKNOWN)
#define NI_ISLIBERR(nirc)    ((nirc) == NIEROUT_SNC_FAILURE || \
			      (nirc) == NIESNC_FAILURE)
#define NI_ISNONFATAL(nirc)  (NI_ISLIBERR(nirc) ||    \
                              (nirc) == NIEPING ||    \
                              (nirc) == NIEVERSION || \
                              (nirc) == NIEROUT_VERSION)

/* Konstanten fuer timeout-Parameter----------------------------------*/
#define NI_BLOCK            -1
#define NI_ASYNC             0

/* Konstanten fuer servNo-Parameter-----------------------------------*/
#define NI_FINDSERV         (SAP_USHORT) -1 /* Hier suchen wir selber */
#define NI_FINDSERV_OS      (SAP_USHORT)  0 /* Hier sucht das OS      */

/* Konstante fuer handle-Parameter------------------------------------*/
#define NI_INVALID_HDL      (NI_HDL) -1
#define NI_INVALID_SOCK     -1

/* Default fuer SapRouter-Service-------------------------------------*/
#define NI_ROUTER_SERVICE   cU("sapdp99")
#define NI_ROUTER_SERVNO    cU("3299")
#define NI_ROUTER_SERVNO_N  (SAP_USHORT) 3299

/* Separatoren fuer Routen im Hostnamen-------------------------------*/
#define NI_HOSTSEP     cU("/H/")  /* Separator fuer Hostname in Route */
#define NI_SERVSEP     cU("/S/")  /* Separator fuer Service in Route  */
#define NI_PASSSEP     cU("/P/")  /* Separator fuer Passwort in Route */
#define NI_PAS2SEP     cU("/W/")  /* Separator fuer Passwort in Route */
#define NI_KRPTSEP     "/K/"      /* Separator fuer SNC-Name          */
/* reservierte Separatoren fuer zukuenftige Entwicklung, momentan     */
/* 'extern' in commandline parser NT sapgui, JAVAgui implementiert    */
/* -> M.Tewes                                                         */
#define NI_SYSTSEP     "/R/" /* Separator fuer Systemname R/3 ('MLP') */
#define NI_GRUPSEP     "/G/" /*                Gruppenname ('PUBLIC') */
#define NI_ALIASEP     "/A/" /*                Aliasname f. Saprouter */
#define NI_MSSVSEP     "/M/" /*                Messageserver name     */

#define NI_SEPSIZ      3

/* common handle trace format */
#define NI_HANDLE_TRC           cU("hdl %d")
#define NI_HANDLE_SOCKET_TRC    NI_HANDLE_TRC cU(" / ") SI_SOCKET_TRC

/* Parameter fuer NiSel/NiPeek----------------------------------------*/
#define NI_READ             (SAP_RAW) 1
#define NI_WRITE            (SAP_RAW) 2
#define NI_CONNECT          (SAP_RAW) 4

/* Parameter fuer NiShutdownHandle------------------------------------*/
#define NI_SHUT_RD          (SAP_INT) 0
#define NI_SHUT_WR          (SAP_INT) 1
#define NI_SHUT_RDWR        (SAP_INT) 2


/* Parameter fuer Ni?etHdlParam---------------------------------------*/
/* Beschreibung: siehe Ni?GetParameterEx                              */
typedef enum {
    NIHP_MODE,                  /* I/O Modus (SAP_RAW)                */
    NIHP_SINGLE_READ,           /* NT only: Einzelschritt (SAP_INT)   */
    NIHP_CHECK_WAIT,            /* PING/PONG Synchronisation (SAP_INT)*/
    NIHP_CHECK_QUE,             /* check queue flag (SAP_RAW)         */
    NIHP_ALLOC_FUNC,            /* NIBUF_ALLOC_FUNC                   */
    NIHP_MAXQUEUE,              /* maximale Anzahl Pakete  (SAP_UINT) */
    NIHP_WARNQUEUE,             /* minimale Anzahl verf. Pakete       */
                                /* (SAP_UINT)                         */
    NIHP_BUFSTAT,               /* Bufferstatus: ON / OFF (SAP_RAW)   */
    NIHP_MAXMLEN                /* maximal NI-Message length (SAP_INT)*/
} NIHDL_PARAM;

typedef enum {
    NIBUFP_MAXHEAP = 100        /* maximaler HEAP-Bedarf (SAP_UINT)   */
} NIBUF_PARAM;

/* Parameter fuer NIHP_MODE in Ni?etHdlParam                          */
#define NI_MESG_IO          (SAP_RAW) 0  /* message-orientierte Komm. */
#define NI_RAW_IO           (SAP_RAW) 1  /* stream-orient. Kommunik.  */
#define NI_ROUT_IO          (SAP_RAW) 2  /* spez. Mode fuer Saprouter */

/* Parameter fuer NIHP_BUFSTAT in Ni?etHdlParam                       */
#define NI_BUFSTAT_OFF      (SAP_RAW) 0  /* Pufferung explizit aus    */
#define NI_BUFSTAT_ON       (SAP_RAW) 1  /* Pufferung an              */

/* Datentypen fuer die Parameter                                      */
/* In C++ waeren dies einfach ueberladene Funktionen...               */
#define NI_DT_UNKNOWN           0
#define NI_DT_CHAR              1
#define NI_DT_INT               2
#define NI_DT_UINT              3
#define NI_DT_SHORT             4
#define NI_DT_USHORT            5
#define NI_DT_RAW               6
#define NI_DT_BOOL              7
#define NI_DT_A7                8
#define NI_DT_RAWPTR            9
#define NI_DT_UCPTR             10
#define NI_DT_VOIDPTR           11
#define NI_DT_ALLOCFUNC         12
#define NI_DT_HOSTADR           13


/*--------------------------------------------------------------------*/
/* Neue Puffer-API:                                                   */
/*--------------------------------------------------------------------*/

struct NIBUFFER;
typedef SAPRETURN NIBUF_ALLOC_FUNC (struct NIBUFFER **buf,
                                    SAP_INT           len,
                                    SAP_UC           *text);
typedef void      NIBUF_FREE_FUNC  (struct NIBUFFER **buf);

/* Groessere Packete kann NiBuf nicht verarbeiten                     */
#define NIBUF_MAXALLOC (8*1024*1024)

enum NIBUFFER_TYPE
{
    NIBUF_NORMAL,
    NIBUF_REFERENZ,
    NIBUF_WRAPPED
};


typedef struct NIBUFFER
{
    /* public data */
    void       *data;           /* Zeiger auf Userdaten               */
    SAP_INT     len;            /* Laenge der Userdaten               */

    /* private data */
#if defined(NI_PRIVATE_DATA)
    void       *space;          /* Zeiger auf Datenbereich            */
    SAP_INT     spacelen;       /* Laenge des Datenbereichs           */
# ifdef LST_NODE
    enum NIBUFFER_TYPE type;
    SAP_USHORT  refcount;
    union {
        struct NIBUFFER *ref;
        NIBUF_FREE_FUNC *freef;
    } u;
    void *uptr;
    LST_NODE (struct NIBUFFER, queue); /* Queueliste                  */
# endif /* LST_NODE */
#endif /* NI_INTERNAL */
} NIBUFFER;


/*--------------------------------------------------------------------*/
/* Struktur fuer Host-Adresse (nur um bei den Funktions-Parametern    */
/* Typ-Pruefungen zu ermoeglichen)                                    */
/*--------------------------------------------------------------------*/

#if !defined  (NI_IADR_LN) && !defined (__NTUTIL_H_)
#define NI_IADR_LN           4

typedef struct /* NI_HOSTADR */
{
    SAP_RAW       addr[NI_IADR_LN];
}   NI_HOSTADR;

#define NI_HOSTADR_SIZ       sizeofR (NI_HOSTADR)
#endif

/*---------------------------------------------------------------------*/
/* Maximal hostname lenght inclusive terminating NULL                  */
/* (most os: MAXHOSTNAMELEN)                                           */
/*     e.g. Linux, HP-UX    64                                         */
/*          SunOS, AIX     256                                         */ 
/*---------------------------------------------------------------------*/
#ifdef MAXHOSTNAMELEN
    #define NI_MAX_HOSTNAME_LEN   MAXHOSTNAMELEN
#else
    #define NI_MAX_HOSTNAME_LEN   64
#endif

/*---------------------------------------------------------------------*/
/* Maximal servicename lenght inclusive terminating NULL               */
/*---------------------------------------------------------------------*/
#define NI_MAX_SERVNAME_LEN       32

/*---------------------------------------------------------------------*/
/* Optimal string length for host address and service number           */
/* translation (lenght inclusive terminating NULL)                     */
/* cf. NiAdrToStr2, NiSrvToStr2 and NiSrvToStrL2                       */
/*---------------------------------------------------------------------*/
#define NI_HOSTADDR_STR_LEN       16
#define NI_SERVNO_STR_LEN          6
#define NI_SERVNO_STR_L_LEN       12

/*---------------------------------------------------------------------*/
/* maximale Laenge des Pfades fuer Unix Domain Sockets                 */
/*---------------------------------------------------------------------*/
#define NI_UDS_PATH_LEN     108

/*---------------------------------------------------------------------*/
/* Struktur fuer Routenanalyse                                         */
/*---------------------------------------------------------------------*/
#define NI_HOSTLN (size_t) 60
#define NI_SERVLN (size_t) 30
#define NI_PASSLN (size_t) 20

typedef struct /* NI_ROUTE_TABLE */
{
    SAP_UC hostname[NI_HOSTLN];
    SAP_UC service [NI_SERVLN];
} NI_ROUTE_TABLE;


/*--------------------------------------------------------------------*/
/* Struktur fuer Select-Masken                                        */
/*--------------------------------------------------------------------*/
#define NI_IMASK_LN          512     /* number of SAP_INT's in a mask */

#define BITS_PER_INT         (sizeofR (SAP_INT) * 8)
#define NI_FD_SETSIZE        (NI_IMASK_LN * BITS_PER_INT)

#define NI_DEF_HDLS           2048   /*  2K handles */
#define NI_MAX_HDLS          16384   /* 16K handles */


typedef struct /* NI_MASK_T */
{
    SAP_INT       hdl_count;
    SAP_INT       mask[NI_IMASK_LN];
}   NI_MASK_T;

#define NI_MASK_SIZ          sizeofR (NI_MASK_T)

/*--------------------------------------------------------------------*/
/* Parameter fuer NiP?etEx                                            */
/*--------------------------------------------------------------------*/
typedef struct /* NI_PARAM_DATA */
{
    SAP_RAW                     param;  /* betroffener Parameter      */
    SAP_RAW                     datatype;/* Datentyp des Parameters   */
    union {                             /* angehaengte Daten          */
        SAP_CHAR                char_val;
        SAP_INT                 int_val;
        SAP_UINT                uint_val;
        SAP_SHORT               short_val;
        SAP_USHORT              ushort_val;
        SAP_RAW                 raw_val;
        SAP_BOOL                bool_val;
        SAP_A7                  a7_val;
        SAP_RAW                 *rawptr_val;
        SAP_UC                  *ucptr_val;
        void                    *voidptr_val;
        NIBUF_ALLOC_FUNC        *allocfunc_val; /* SAPUNICODEOK       */
        NI_HOSTADR              hostaddr_val;
    } value;
}   NI_PARAM_DATA;

#define NI_PARAM_DATA_LN  sizeofR (NI_PARAM_DATA)

/*--------------------------------------------------------------------*/
/* Struktur fuer interne SNC-Namen                                    */
/*--------------------------------------------------------------------*/

#define NISNC_ACLKEY_MAXLEN 1024

typedef struct NISNC_ACLKEY {
    SAP_INT  len;
    SAP_RAW  key[NISNC_ACLKEY_MAXLEN];
} NISNC_ACLKEY;


/* ################################################################## */
/* # typedefs                                                       # */
/* ################################################################## */

typedef struct NISEL_SET *NISEL_HDL;
typedef SAP_INT           NI_HDL;


/* ################################################################## */
/* # macros                                                         # */
/* ################################################################## */

/*--------------------------------------------------------------------*/
/* Macros zum Manipulieren von Select-Masken                          */
/*--------------------------------------------------------------------*/

/* Inhalt einer Maske auf HEX 0 setzen */
#define NI_CLEAR_MASK(M) \
    (memsetR (&(M), 0, sizeofR(M)))

/* Anzahl der gesetzten Bits in Maske liefern */
#define NI_COUNT_MASK(M)      ((M).hdl_count)

/* in Maske 'mask' Bit fuer Handle 'hdl' setzen */
#define NI_ADDTO_MASK(M, hdl) \
    { if (NI_HDL_OK (hdl) && (! NI_ISIN_MASK (M, hdl))) \
      { (M).mask [hdl / BITS_PER_INT] |= (1 << (hdl % BITS_PER_INT)); \
	(M).hdl_count++; } }

/* in Maske 'mask' Bit fuer Handle 'hdl' loeschen */
#define NI_DELFROM_MASK(M, hdl) \
    { if (NI_ISIN_MASK (M, hdl)) \
      { (M).mask [hdl / BITS_PER_INT] &= ~(1 << (hdl % BITS_PER_INT)); \
	(M).hdl_count--; } }

/* Ist in Maske 'mask' Bit fuer Handle 'hdl' gesetzt ? */
#define NI_ISIN_MASK(M, hdl) \
    (NI_HDL_OK (hdl) ? \
       ((M).mask [hdl / BITS_PER_INT] & (1 << (hdl % BITS_PER_INT))) \
       : FALSE )

/*--------------------------------------------------------------------*/
/* Pruefen, ob Handle zu gross fuer Maske                             */
/*--------------------------------------------------------------------*/
extern SAP_INT ni_max_hdls;

#define NI_HDL_OK(hdl) \
    (((hdl >= 0) && (hdl < ni_max_hdls)) ? TRUE :FALSE)

/*--------------------------------------------------------------------*/
/* Masken zum Pruefen und setzten von Host-Adressen                   */
/*--------------------------------------------------------------------*/
#define NI_SET_NO_ADR(adr)       (memsetR (&(adr), 0, NI_HOSTADR_SIZ))
#define NI_IS_NO_ADR(adr)        (((adr).addr[0] == 0) && \
				  ((adr).addr[1] == 0) && \
				  ((adr).addr[2] == 0) && \
				  ((adr).addr[3] == 0))


/* ################################################################## */
/* # function declarations                                          # */
/* ################################################################## */


/*--------------------------------------------------------------------*/
/* NiInit  ()                                                         */
/* NiInit2 ()                                                         */
/*                                                                    */
/* NiHSBufInit ()                                                     */
/*                                                                    */
/* Initialisierung der nilib                                          */
/*                                                                    */
/* INIT    Die Initialisierungs Funktion muss vom Haupt Thread auf-   */
/*         gerufen werden                                             */
/*                                                                    */
/* INPUT   maxHandles    maximale Anzahl Handles                      */
/*                       (0 < maxHandles <= NI_MAX_HDLS;              */
/*                       default: NI_DEF_HDLS)                        */
/*         hsBufType     Host- / Service Namen Puffer Typ             */
/*                       (NULL fuer "default")                        */
/*         hsBufIndexSize      Index Groesse des HS Buf               */
/*                       (NULL fuer default)                          */
/*         hsBufRetryTimeout   Zeitdauer, nachder ein ungueltiger     */
/*                       Eintrag verfaellt und somit neu vom System   */
/*                       gelesen werden muss (vgl. Parameter          */
/*                       HSBUF_RETRY; NULL fuer default)              */
/*                                                                    */
/* RETURN                                                             */
/*                                                                    */
/* SAP_O_K               alles klar (Daten da)                        */
/* NIEINVAL              unbekannter Buffer Typ oder maxHandles Wert  */
/*                       ungueltig                                    */
/* NIEINTERN             interner Fehler, (Err-Trace) Speicher        */
/*                       Allokation fehlgesch.                        */
/* NIECALL               nilib bereits initiatisiert (NiInit3)        */
/*                                                                    */
/* Einige NI Funktionen setzten eine bereits initialisierte Library   */
/* voraus. Diese rufen intern NiInit auf und Initialisieren NI. Wird  */
/* spaeter vom Benutzer NiInit2 oder NiInit3 aufgerufen, koennen die  */
/* uebergebenen Parameter nicht beruecksichtigt werden! Ein Trace-    */
/* Eintrag wird in diesem Falle geschrieben. Es empfiehlt sich daher, */
/* die Library beim Start des Programmes zu initialisieren.           */
/*                                                                    */
/*--------------------------------------------------------------------*/

SAPRETURN API_DF NiInit	   (void);
SAPRETURN API_DF NiInit2   (SAP_INT      maxHandles);

SAPRETURN API_DF NiHSBufInit (SAP_BOOL   * pLinearImpl,
                              SAP_UINT     indexSize,
                              SAP_UINT     expTValHost,
                              SAP_UINT     expTUnkHN,
                              SAP_UINT     expTUnkHA,
                              SAP_UINT     expTValServ,
                              SAP_UINT     expTUnkSN,
                              SAP_UINT     expTUnkSP);

#define NiBufInit() NiInit()
#define NiBufReInit() SAP_O_K
#define NiBufInit2(a) NiInit2((a))

SAPRETURN API_DF NiSncInit (SAP_UC * my_name);

/*--------------------------------------------------------------------*/
/* NiThrInit ()                                                       */
/*                                                                    */
/* Thread specific initialization                                     */
/*                                                                    */
/* INIT    Library muss zum Zeitpunkt des Aufrufes bereits            */
/*         initialisiert sein!                                        */
/*                                                                    */
/* RETURN                                                             */
/*                                                                    */
/* SAP_O_K               alles klar (Daten da)                        */
/* NIEINTERN             interner Fehler (Err-Trace)                  */
/* NIECALL               nilib noch nicht initialisiert (vgl. NiInit) */
/*                                                                    */
/* Werden von einer Funktion Thread Spezifische Daten benoetigt, so   */
/* wird intern NiThrInit aufgerufen. Ein Aufruf der Funktion nach     */
/* kreieren des Threads ist somit nicht notwendig.                    */
/*                                                                    */
/*--------------------------------------------------------------------*/

SAPRETURN API_DF NiThrInit (void);

/*--------------------------------------------------------------------*/
/* NiExit ()                                                          */
/*                                                                    */
/* Verlassen der nilib                                                */
/*                                                                    */
/* INIT    Muss von Haupt Thread aufregrufen werden!                  */
/*                                                                    */
/* RETURN                                                             */
/*                                                                    */
/* SAP_O_K               alles klar                                   */
/* NIEINTERN             interner Fehler (Err-Trace)                  */
/* NIECALL               nilib wurde nicht initialisiert (vgl. NiInit)*/
/*                                                                    */
/*--------------------------------------------------------------------*/

SAPRETURN API_DF NiExit	 (void);

/*--------------------------------------------------------------------*/
/* NiThrExit ()                                                       */
/*                                                                    */
/* Thread specific cleanup                                            */
/*                                                                    */
/* INIT    Der vorgaengige Aufruf von NiThrInit ist nicht notwendig   */
/*                                                                    */
/* Aufruf notwendig, um Thread spezifische Ressourcen freizugeben.    */
/*                                                                    */
/*--------------------------------------------------------------------*/

void API_DF NiThrExit (void * data);

/*--------------------------------------------------------------------*/
/* NiListen  ()                                                       */
/* Ni2Listen ()                                                       */
/* Ni3Listen ()                                                       */
/* NiBufListen  ()                                                    */
/* NiBuf2Listen ()                                                    */
/* NiBuf3Listen ()                                                    */
/*                                                                    */
/* Vorbereitungen fuer Connect-Requests von Clients treffen (Server)  */
/*                                                                    */
/* INIT    Die NI Library wird initialisiert, falls sie es nicht      */
/*         schon ist.                                                 */
/*                                                                    */
/* INPUT   servName      log. Name des Service                        */
/*         servNo        Service Nummer (Host-Byte-Order) oder        */
/*                       *servNo = NI_FINDSERV um automatisch zu ver- */
/*                       geben                                        */
/*         localAddr     lokate Hostadresse (oder NULL fuer beliebige)*/
/*                                                                    */
/* OUTPUT  handle        Handle fuer NiAccept- und NiSelect- Aufrufe  */
/*         servNo        Nummer des Service                           */
/*                                                                    */
/* RETURN                                                             */
/*                                                                    */
/* SAP_O_K               alles klar                                   */
/* NIESERV_UNKNOWN       log. Servicename unbekannt                   */
/* NIESERV_USED          Service schon benutzt                        */
/* NIEMYHOST_VERIFY      localAddr ist keine lokale Addresse          */
/* NIEINVAL              servName = NULL, servName zu lange,          */
/*                       handle = NULL oder servNo = NULL             */
/* NIEINTERN             interner Fehler (Err-Trace)                  */
/*                                                                    */
/*--------------------------------------------------------------------*/

SAPRETURN API_DF NiListen      (SAP_UC     * servName,
                                NI_HDL     * handle);

SAPRETURN API_DF Ni2Listen     (SAP_USHORT * servNo,
                                NI_HDL     * handle);

SAPRETURN API_DF Ni3Listen     (SAP_USHORT * servNo,
                                NI_HOSTADR * localAddr,
                                NI_HDL     * handle);

SAPRETURN API_DF NiBufListen   (SAP_UC     * servName,
                                NI_HDL     * handle);

SAPRETURN API_DF NiBuf2Listen  (SAP_USHORT * servNo,
                                NI_HDL     * handle);

SAPRETURN API_DF NiBuf3Listen  (SAP_USHORT * servNo,
                                NI_HOSTADR * localAddr,
                                NI_HDL     * handle);


/*--------------------------------------------------------------------*/
/* NiAccept ()                                                        */
/*                                                                    */
/* Connect-Request von Client entgegennehmen und Verbindung aufbauen  */
/* (Server)                                                           */
/*                                                                    */
/* INIT    NI sollte bereits initialisiert sein von NiListen          */
/*                                                                    */
/* INPUT   inHandle      Handle von NiListen                          */
/*         timeout       in MilliSekunden (oder NI_BLOCK)             */
/*                                                                    */
/* OUTPUT  outHandle     Handle fuer Kommunikation mit Client         */
/*         hostAddr      Adresse des Client-Host (oder NULL)          */
/*                                                                    */
/* RETURN                                                             */
/*                                                                    */
/* SAP_O_K               alles klar                                   */
/* NIETIMEOUT            Zeitlimit ueberschritten                     */
/* NIEINVAL              inHandle ungueltig, outHandle = NULL oder    */
/*                       outHandle zu gross fuer NI-Maske             */
/* NIEINTERN             interner Fehler (Err-Trace)                  */
/*                                                                    */
/*--------------------------------------------------------------------*/

SAPRETURN API_DF  NiAccept       	 (NI_HDL       inHandle,
					  SAP_INT      timeout,
					  NI_HDL     * outHandle,
					  NI_HOSTADR * hostAddr);


/*--------------------------------------------------------------------*/
/* Ni*Connect  ()                                                     */
/* Ni*2Connect ()                                                     */
/*                                                                    */
/* Verbindung zum Server aufbauen (Connect-Request senden)            */
/*                                                                    */
/* Ni*Raw*Connect  Talkmode nach Verbindungsaufbau Native statt NI    */
/*                 (fuer SapTelnet..)                                 */
/* NiBuf*Connect   Gebufferte Verbindung                              */
/* NiSnc*Connect   Verbindung mit SNC                                 */
/*                                                                    */
/* INIT    Die NI Library wird initialisiert, falls sie es nicht      */
/*         schon ist.                                                 */
/*                                                                    */
/* INPUT   hostName      Name des Server-Host                         */
/*         hostAddr      Adr des Server-Host                          */
/*         servName      Name des Service                             */
/*         servNo        Nummer des Service (Host-Byte-Order)         */
/*         timeout       in MilliSekunden > 0 oder NI_BLOCK           */
/*                                                                    */
/* OUTPUT  handle        Handle fuer Kommunikation mit Server         */
/*                                                                    */
/* RETURN                                                             */
/*                                                                    */
/* SAP_O_K               alles klar                                   */
/* NIEHOST_UNKNOWN       log. Hostname unbekannt                      */
/* NIESERV_UNKNOWN       log. Servicename unbekannt                   */
/* NIECONN_REFUSED       Verbindungsaufbau misslungen                 */
/* NIETIMEOUT            Zeitlimit ueberschritten                     */
/* NIEINVAL              Handle zu gross fuer NI-Maske,               */
/*                       hostName = NULL, servName = NULL,            */
/*                       hostAddr = NULL oder handle = NULL           */
/* NIENOFREEPORT         kein freier Port innerhalb Range             */
/*                       (siehe NiLocPortRange)                       */
/* NIEINTERN             interner Fehler (Err-Trace)                  */
/*                                                                    */
/*--------------------------------------------------------------------*/

SAPRETURN API_DF  NiConnect       (SAP_UC       * hostName,
			           SAP_UC       * servName,
			           SAP_INT        timeout,
			           NI_HDL       * handle);
SAPRETURN API_DF  Ni2Connect      (NI_HOSTADR   * hostAddr,
				   SAP_USHORT     servNo,
				   SAP_INT        timeout,
				   NI_HDL       * handle);

SAPRETURN API_DF  NiRawConnect    (SAP_UC       * hostName,
				   SAP_UC       * servName,
				   SAP_INT        timeout,
				   NI_HDL       * handle);
SAPRETURN API_DF  NiRaw2Connect   (NI_HOSTADR   * hostAddr,
				   SAP_USHORT     servNo,
				   SAP_INT        timeout,
				   NI_HDL       * handle);

SAPRETURN API_DF NiBufConnect     (SAP_UC       * hostName,
                                   SAP_UC       * servName,
                                   SAP_INT        timeout,
                                   NI_HDL       * handle);
SAPRETURN API_DF NiBuf2Connect    (NI_HOSTADR   * hostAddr,
                                   SAP_USHORT     servNo,
                                   SAP_INT        timeout,
                                   NI_HDL       * handle);

SAPRETURN API_DF NiBufRawConnect  (SAP_UC       * hostName,
                                   SAP_UC       * servName,
                                   SAP_INT        timeout,
                                   NI_HDL       * handle);
SAPRETURN API_DF NiBufRaw2Connect (NI_HOSTADR   * hostAddr,
                                   SAP_USHORT     servNo,
                                   SAP_INT        timeout,
                                   NI_HDL       * handle);

SAPRETURN API_DF NiSncConnect     (SAP_UC       * hostName,
                                   SAP_UC       * servName,
                                   SAP_UC       * peerName,
                                   SAP_INT        timeout,
                                   NI_HDL       * handle);
SAPRETURN API_DF NiSnc2Connect    (NI_HOSTADR   * hostAddr,
                                   SAP_USHORT     servNo,
                                   NISNC_ACLKEY * peer,
                                   SAP_INT        timeout,
                                   NI_HDL       * handle);

/*--------------------------------------------------------*/
/* NiSelect ()                                            */
/*                                                        */
/* Auf ankommende Daten warten                            */
/*                                                        */
/* INPUT   timeout       in MilliSekunden (oder NI_BLOCK) */
/*                                                        */
/* IN/OUT  readmask      Maske der zu ueberwachenden      */
/*                       Handles (Masken-Macros verwenden)*/
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

#define NiSelect(mask, timeout) Ni2Select (mask, NULL, timeout)

/*--------------------------------------------------------*/
/* Ni2Select ()                                           */
/*                                                        */
/* Auf ankommende Daten warten und freie Sendeleitungen   */
/* warten                                                 */
/*                                                        */
/* INPUT   timeout       in MilliSekunden (oder NI_BLOCK) */
/*                                                        */
/* IN/OUT  readmask      Maske der zu ueberwachenden      */
/*                       Read-Handles                     */
/*         writemask     Maske der zu ueberwachenden      */
/*                       Write-Handles                    */
/*                       (Masken-Macros verwenden)        */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/


SAPRETURN API_DF  Ni2Select      	 (NI_MASK_T *  readmask,
					  NI_MASK_T *  writemask,
					  SAP_INT      timeout);

/*--------------------------------------------------------*/
/* NiPeek ()                                              */
/*                                                        */
/* Testen, ob Daten auf einer best. Leitung angekommen    */
/*                                                        */
/* INPUT   handle        zu ueberpruefender Handle        */
/*         timeout       in MilliSekunden (oder NI_BLOCK) */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar (Daten da)            */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiPeek         	 (NI_HDL   handle,
					  SAP_INT  timeout);


/*--------------------------------------------------------*/
/* NiWriteOK ()                                           */
/*                                                        */
/* Testen, ob Write auf Handle blockieren wuerde          */
/*                                                        */
/* INPUT   handle        zu ueberpruefender Handle        */
/*         timeout       in MilliSekunden (oder NI_BLOCK) */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar (Daten da)            */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiWriteOK      	 (NI_HDL   handle,
					  SAP_INT  timeout);


/*--------------------------------------------------------*/
/* NiWrite ()                                             */
/*                                                        */
/* Daten an Partner senden                                */
/*                                                        */
/* INPUT   handle        Handle fuer Verbindungskanal     */
/*                       zum Partner                      */
/*         buf           Daten-Puffer                     */
/*         buflen        Laenge der zu versendenden Daten */
/*         timeout       in MilliSekunden (oder NI_BLOCK) */
/*                       Bezieht sich auf das Senden der  */
/*                       internen Datenpakete, aus denen  */
/*                       sich die Nachricht zusammensetzt)*/
/*                                                        */
/* OUTPUT  writelen      Laenge der gesendeten Daten      */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIECONN_BROKEN        Verbindung zum Partner abgebr.   */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/* NIEINVAL              handle ungueltig                 */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiWrite        	 (NI_HDL      handle,
					  void     *  buf,
					  SAP_INT     buflen,
					  SAP_INT     timeout,
					  SAP_INT  *  writelen);


/*--------------------------------------------------------*/
/* NiRead ()                                              */
/*                                                        */
/* Daten vom Partner empfangen                            */
/*                                                        */
/* INPUT   handle        Handle fuer Verbindungskanal     */
/*                       zum Partner                      */
/*         buf           Daten-Puffer                     */
/*         maxlen        Groesse des Daten-Puffers        */
/*         timeout       in MilliSekunden (oder NI_BLOCK) */
/*                       Bezieht sich auf den Empfang der */
/*                       internen Datenpakete, aus denen  */
/*                       sich die Nachricht zusammensetzt)*/
/*                                                        */
/* OUTPUT  readlen       Laenge der empfangenen Daten     */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIECONN_BROKEN        Verbindung zum Partner abgebr.   */
/*                       Ist readlen = 0, hat die Gegen-  */
/*                       seite die Verbindung geschlossen,*/
/*                       ansonsten wurde sie abgebrochen. */
/* NIETOO_SMALL          Puffer zu klein fuer Nachricht   */
/*                       (nachfolgende NiRead-Calls       */
/*                       liefern Rest des NI-Pakets)      */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/*                       (nachfolgende NiRead-Calls       */
/*                       liefern Rest des NI-Pakets)      */
/* NIEPING               Partner hat Ping-Signal gesendet */
/*                       (keine Daten, readlen = 0)       */
/* NIEINVAL              handle ungueltig                 */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/* NIEROUT_HOST_UNKNOWN  im nirouter: log. Hostname unbek.*/
/* NIEROUT_SERV_UNKNOWN  im nirouter: log. Service unbek. */
/* NIEROUT_CONN_REFUSED  im nirouter: Verb.aufbau missl.  */
/* NIEROUT_CONN_BROKEN   im nirouter: Verbindung abgebr.  */
/* NIEROUT_PERM_DENIED   im nirouter: route nicht erlaubt */
/* NIEROUT_VERSION       im nirouter: unguelt. Clnt-Vers. */
/* NIEROUT_CANCELED      im nirouter: Route von Administr.*/
/*                                    ge"canceled"        */
/* NIEROUT_SHUTDOWN      im nirouter: Router Shutdown     */
/* NIEROUT_INTERN        im nirouter: interner Fehler     */
/*                                                        */
/*                                                        */
/* Auf Betriebssystemen, bei denen der Abbruch einer      */
/* Verbindung zu einem Signal (SIGPIPE) fuehrt, muss der  */
/* Aufrufer selbst fuer die Signalbehandlung sorgen (z.B. */
/* auf SIG_IGN setzen)                                    */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiRead         	 (NI_HDL      handle,
					  void     *  buf,
					  SAP_INT     maxlen,
					  SAP_INT     timeout,
					  SAP_INT  *  readlen);


/*--------------------------------------------------------*/
/* NiCheck ()                                             */
/*                                                        */
/* Verbindung zum Partner ueberpruefen.                   */
/* NiCheck sendet ein "Ping"-Signal zum Partner. Ist die  */
/* Verbindung o.k., bekommen beide Partner bei einem der  */
/* naechsten NiRead-Calls den Returncode NIEPING (ohne    */
/* Daten).                                                */
/* (NiRead sendet automatisch Antwort auf Ping)           */
/*                                                        */
/* INPUT   handle        Handle fuer Verbindungskanal     */
/*                       zum Partner                      */
/*         timeout       in MilliSekunden (oder NI_BLOCK) */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIECONN_BROKEN        Verbindung zum Partner abgebr.   */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiCheck        	 (NI_HDL      handle,
					  SAP_INT     timeout);

/*--------------------------------------------------------*/
/* NiCheck1 ()						  */
/*                                                        */
/* Wie Nicheck, nur, dass keine Antwort zurueckkommt      */
/*                                                        */
/* INPUT   handle        Handle fuer Verbindungskanal     */
/*                       zum Partner                      */
/*         timeout       in MilliSekunden (oder NI_BLOCK) */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIECONN_BROKEN        Verbindung zum Partner abgebr.   */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiCheck1       	 (NI_HDL      handle,
					  SAP_INT     timeout);

/*--------------------------------------------------------------------*/
/* NiCheckEx ()                                                       */
/*                                                                    */
/* Check connection with ping messages                                */
/*                                                                    */
/* INPUT   handle          NI handle                                  */
/*         timeout         timeout in milliseconds (ms) or negative   */
/*                         (NI_BLOCK) for blocking indefinitely       */
/*         recvResp        should the partner send an response; if    */
/*                         value is TRUE, a ping message is sent and  */
/*                         the partner will respond with a pong; if   */
/*                         value is FALSE, a pong message is sent and */
/*                         the partner won't reply.                   */
/*         pWritten        number of bytes already written; parameter */
/*                         is important if a fragmentation of the     */
/*                         message is possible (non-buffered handles);*/
/*                         integer value must be 0 for the first call,*/
/*                         following calls passes the returned value  */
/*                         of previous call; NULL if not interested   */
/*                         in fragmentation.                          */
/*                                                                    */
/* OUTPUT  pWritten        number of bytes written; value is just set */
/*                         if the write operation timed out           */
/*                         (NIETIMEOUT returned)                      */
/*                                                                    */
/* RETURN  SAP_O_K         successful                                 */
/*         NIEINVAL        handle invalid                             */
/*         NIETIMEOUT      write operation timed out, call function   */
/*                         function again (c.f. parameter pWritten)   */
/*         NIECONN_BROKEN  connection to partner broken               */
/*         NIEINTERN       internal error (check err-trace)           */
/*                                                                    */
/*--------------------------------------------------------------------*/

SAPRETURN API_DF  NiCheckEx              (NI_HDL      handle,
					  SAP_INT     timeout,
                                          SAP_BOOL    recvResp,
                                          SAP_UINT  * pWritten);


/*--------------------------------------------------------*/
/* NiCheck2 ()						  */
/*                                                        */
/* Wie Nicheck, nur wird intern der Status gespeichert    */
/* und nicht NIEPING zurueckgegeben. Beim naechsten Aufruf*/
/* wird der Status zurueckgeliefert. Funktioniert nur     */
/* mit Verwendung der gepufferten Schnittstelle!          */
/*                                                        */
/* INPUT   handle        Handle fuer Verbindungskanal     */
/*                       zum Partner                      */
/*         timeout       in MilliSekunden (oder NI_BLOCK) */
/*                                                        */
/* RETURN                                                 */
/*         count         Anzahl Tests ohne Antwort        */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIENO_ANSWER          bisher keine Antwort vom Partner */
/* NIECONN_BROKEN        Verbindung zum Partner abgebr.   */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/* NIEINVAL              illegale Parameter               */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiCheck2       	 (NI_HDL      handle,
					  SAP_INT     timeout,
                                          SAP_INT     *count);


/*--------------------------------------------------------*/
/* NiCheck2Cancel ()                                      */
/*                                                        */
/* Diese Funktion setzt den Status von NiCheck2 zurueck.  */
/* Damit werden bereits eingegangene Antworten vom        */
/* Partner verworfen. Diese Funktion kann verwendet       */
/* werden, wenn nach einem NiCheck2 richtige Daten ueber  */
/* das Netz kommen.                                       */
/*                                                        */
/* INPUT   handle        Handle fuer Verbindungskanal     */
/*                       zum Partner                      */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiCheck2Cancel       	 (NI_HDL      handle);


/*--------------------------------------------------------*/
/* NiCloseHandle ()                                       */
/*                                                        */
/* Mit NiListen oder NiConnect erzeugten Handle freigeben */
/* und evtl. bestehende Verbindung abbauen                */
/*                                                        */
/* INPUT   handle                                         */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINVAL              ungueltiger Handle               */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiCloseHandle  	 (NI_HDL  handle);

/*--------------------------------------------------------*/
/* NiCloseHandleKP ()                                     */
/*                                                        */
/* Mit NiListen oder NiConnect erzeugten Handle freigeben */
/* und evtl. bestehende Verbindung abbauen. Evtl.         */
/* vorhandener UDS Pfad wird nicht geloescht.             */
/*                                                        */
/* INPUT   handle                                         */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINVAL              ungueltiger Handle               */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiCloseHandleKP	 (NI_HDL  handle);


/*--------------------------------------------------------*/
/* NiShutdownHandle ()                                    */
/*                                                        */
/* Schliessen von Lese- und/oder Schreibseite einer Ver-  */
/* bindung. Der Handle muss mit NiCloseHandle noch        */
/* ansckliessend geschlossen werden.                      */
/*                                                        */
/* INPUT   handle        NI handle                        */
/*         howto         Verbindungsseite, die geschlossen*/
/*                       werden soll: NI_SHUT_RD,         */
/*                       NI_SHUT_WR oder NI_SHUTRDWR      */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINVAL              ungueltiger Handle               */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiShutdownHandle  	 (NI_HDL  handle,
                                          SAP_INT howto);


/*--------------------------------------------------------*/
/* NiCloseAll ()                                          */
/*                                                        */
/* Alle offenen Handles freigeben                         */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiCloseAll     	(void);


/*--------------------------------------------------------*/
/* NiRouteToTable ()                                      */
/*                                                        */
/* Routingstring in eine Tabelle von NI_ROUTE-Eintraegen  */
/* konvertieren                                           */
/*                                                        */
/* INPUT   route	Routestring (null-terminiert)     */
/*         size         Groesse des NI_ROUTE-Arrays       */
/*                                                        */
/* IN/OUT  table        Tabelle mit Routingeintraegen     */
/*                                                        */
/* OUTPUT  cnt          Anzahl der gefundenen Eintaege    */
/*                                                        */
/* RETURN  SAP_O_K	alles klar                        */
/* 	   NIEINTERN	Interner Fehler                   */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiRouteToTable (int            * cnt,
				 const SAP_UC	* routestring,
				 NI_ROUTE_TABLE * table,
				 int              size);
/*--------------------------------------------------------*/
/* NiHBufPrintBuf ()                                      */
/*                                                        */
/* Dump the NI-Hostbuffer into an UC-Buffer               */
/*                                                        */
/*                                                        */
/* INPUT   buf           UC-Buffer to dump to             */
/*         buflen        initial size (must be >= 1)      */
/*                                                        */
/* OUTPUT  buf           pretty-printed nihostbuf         */
/*         buflen        length of the UC-buffer          */
/*                       including trailing cU('\0')      */
/*         done          all entries printed ?            */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               ok                               */
/* NIEINVAL              buffer too small etc.            */
/*                                                        */
/*--------------------------------------------------------*/


SAPRETURN NiHBufPrintBuf(SAP_UC *buf, int *buflen, SAP_BOOL *done);

/*--------------------------------------------------------*/
/* NiGetHostAddrList ()                                   */
/*                                                        */
/* gibt die aktiven Interfaces der Maschine als IP-Adresse*/
/* zur"uck.                                               */
/*                                                        */
/* INPUT   hostaddr      array von bufsiz NI_HOSTADR      */
/*         bufsiz        maximale Anzahl an Eintraegen    */
/*                                                        */
/* OUTPUT  if_found      Anzahl der wirklich gefundenen   */
/*                       Interfaces                       */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/*                       sonst interner Fehler            */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiGetHostAddrList  (NI_HOSTADR * hostAddr,
                                      SAP_INT      bufLen,
                                      SAP_INT    * found);


/*--------------------------------------------------------*/
/* NiHostToAddr ()                                        */
/*                                                        */
/* log. Hostname in interne Adresse umwandeln             */
/* Achtung: Hostnamen und IP-Addr. werden beim 1. Zugriff */
/*          gepuffert. Dump/Flush siehe NiSetParam ()     */
/*                                                        */
/* INPUT   hostName      Hostname (null-terminiert),      */
/*                       Adress-String, z.B. "155.56.2.3" */
/*                       oder NULL fuer localhost         */
/*                                                        */
/* OUTPUT  hostAddr      interne Host-Adresse             */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEHOST_UNKNOWN       log. Hostname unbekannt          */
/* NIEINVAL              hostAddr == NULL oder hostName   */
/*                       zu lange                         */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiHostToAddr   	 (SAP_UC     * hostName,
					  NI_HOSTADR * hostAddr);

/*--------------------------------------------------------*/
/* NiAddrToHost  ()                                       */
/* NiAddrToHost2 ()   (reentrant)                         */
/*                                                        */
/* interne Adresse in log. Hostnamen umwandeln            */
/* Achtung: Hostnamen und IP-Addr. werden beim 1. Zugriff */
/*          gepuffert. Dump/Flush siehe NiSetParam ()     */
/*                                                        */
/* INPUT   hostAddr      interne Host-Adresse oder NULL   */
/*                       fuer localhost                   */
/*         hostName      Hostnamen Puffer                 */
/*         bufLen        Groesse des Puffers in SAP_UC    */
/*                       (optimal NI_MAX_HOSTNAME_LEN)    */
/*                                                        */
/* RETURN                Hostname (null-terminiert)       */
/*                       Host-Adr-String, falls unbekannt */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEHOST_UNKNOWN       Host Adresse unbekannt           */
/* NIEINVAL              hostName == NULL                 */
/* NIETOO_SMALL          Puffer fuer Hostnamen zu klein   */
/*                                                        */
/*--------------------------------------------------------*/

SAP_UC  * API_DF  NiAddrToHost           (NI_HOSTADR * hostAddr);
SAPRETURN API_DF  NiAddrToHost2          (NI_HOSTADR * hostAddr,
                                          SAP_UC     * hostName,
                                          SAP_UINT     bufLen);

/*--------------------------------------------------------*/
/* NiAddrToHostCanon ()                                   */
/*                                                        */
/* -> Funktion mit NiAddrToHost identisch                 */
/* (                                                      */
/* interne Adresse in log. Hostnamen umwandeln            */
/* Achtung: Es wird immer der kanonische Name des Rechners*/
/*          zurueckgegeben. Ist dieser nicht voll         */
/*          qualifiziert, wird auch nur der Rechnernamen  */
/*          zurueckgegeben. Die Eintraege werden nicht    */
/*          gepuffert.                                    */
/* )                                                      */
/*--------------------------------------------------------*/

SAP_UC  * API_DF  NiAddrToHostCanon      (NI_HOSTADR * hostAddr);
SAPRETURN API_DF  NiAddrToHostCanon2     (NI_HOSTADR * hostAddr,
                                          SAP_UC     * hostName,
                                          SAP_UINT     bufLen);


/*--------------------------------------------------------*/
/* NiServToNo ()                                          */
/*                                                        */
/* Service-Namen in (Port-)Nummer umwandeln               */
/* Achtung: Servicenamen und Ports werden beim 1. Zugriff */
/*          gepuffert. Dump/Flush siehe NiSetParam ()     */
/*                                                        */
/* INPUT   servName      Servicename (null-terminiert)    */
/*                       oder Service-Nummer-String, z.B. */
/*                       "3200"                           */
/*                                                        */
/* OUTPUT  servNo        Service-Nummer (Host-Byte-Order) */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIESERV_UNKNOWN       log. Servicename unbekannt       */
/* NIEINVAL              servName == NULL, servName zu    */
/*                       lange oder servNo == NULL        */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiServToNo     	 (SAP_UC     * servName,
					  SAP_USHORT * servNo);

/*--------------------------------------------------------*/
/* NiNoToServ  ()                                         */
/* NiNoToServ2 ()   (reentrant)                           */
/*                                                        */
/* (Port-)Nummer in Service-Namen umwandeln               */
/* Achtung: Servicenamen und Ports werden beim 1. Zugriff */
/*          gepuffert. Dump/Flush siehe NiSetParam ()     */
/*                                                        */
/* INPUT   servNo        Service-Nummer (Host-Byte-Order) */
/*         servName      Servicenamen Puffer              */
/*         bufLen        Groesse des Puffers in SAP_UC    */
/*                       (optimal NI_MAX_SERVNAME_LEN)    */
/*                                                        */
/* RETURN                Service-Name (null-terminiert)   */
/*                       Service-Nummer-String, falls     */
/*                       unbekannt                        */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIESERV_UNKNOWN       (Port-)Nummer unbekannt          */
/* NIEINVAL              servName == NULL                 */
/* NIETOO_SMALL          Puffer fuer Servicenamen zu klein*/
/*                                                        */
/*--------------------------------------------------------*/

SAP_UC  * API_DF  NiNoToServ     	 (SAP_USHORT   servNo);
SAPRETURN API_DF  NiNoToServ2            (SAP_USHORT   servNo,
                                          SAP_UC     * servName,
                                          SAP_UINT     bufLen);

/*--------------------------------------------------------*/
/* NiMyHostName ()                                        */
/*                                                        */
/* eigenen Hostnamen ermitteln                            */
/*                                                        */
/* OUTPUT  hostname      eigener Hostname (null-termin.)  */
/*                                                        */
/* INPUT   maxlen        maximale Laenge fuer Hostnamen   */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIETOO_SMALL          Hostname wurde abgeschnitten     */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiMyHostName   	 (SAP_UC     *  hostname,
					  SAP_INT_U     maxlen);


/*--------------------------------------------------------*/
/* NiMyAddrVerify ()                                      */
/* NiMyAddrCheck ()                                       */
/*                                                        */
/* eigene Host-Adresse verifizieren (mittels NiIlisten,   */
/* NiConnect () und NiAccept ())                          */
/* NiMyAddrVerify laesst 127.0.0.1 nicht zu               */
/*                                                        */
/* INPUT   myhostname    eigener Hostname (null-termin.)  */
/*                       oder Adress-String, z.B.         */
/*                       "155.56.2.3"                     */
/*                       oder (SAP_UC   *) 0 (Funktion    */
/*                       ermittelt eig. Hostadr. selbst)  */
/*                                                        */
/* OUTPUT  myhostaddr    eigene Host-Adresse              */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEMYHOST_VERIFY      Fehler beim Verifizieren der     */
/*                       eigenen Host-Adresse             */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiMyAddrVerify	 (SAP_UC     *  myhostname,
                                  NI_HOSTADR *  myhostaddr);

SAPRETURN API_DF NiMyAddrCheck   (SAP_UC     *  myhostname,
				  NI_HOSTADR *  myhostaddr);


/*--------------------------------------------------------*/
/* NiLocalCheck ()                                        */
/*                                                        */
/* pruefen ob eine NI_HOST_ADR lokal ist                  */
/*                                                        */
/* OUTPUT    hostaddr    Host-Adresse                     */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* TRUE                  hostadr ist lokal                */
/* FALSE                 hostadr ist nicht lokal          */
/*                                                        */
/*--------------------------------------------------------*/

SAP_BOOL NiLocalCheck(NI_HOSTADR *ha);

/*---------------------------------------------------------*/
/* NiHdlToSock ()                                          */
/*                                                         */
/* Gebe den Socket zu einem NI-Handle zurueck              */
/*                                                         */
/* INPUT   handle       NI-Handle                          */
/*                                                         */
/* OUTPUT  sock         korrespondierender Socket          */
/*                                                         */
/* RETURN                                                  */
/*                                                         */
/* SAP_O_K              alles klar                         */
/* NIEINVAL             handle ist kein NI-Handle          */
/*                                                         */
/*---------------------------------------------------------*/

SAPRETURN API_DF NiHdlToSock 	 (NI_HDL    handle,
				  int    *  sock);


/*---------------------------------------------------------*/
/* NiSockToHdl ()                                          */
/*                                                         */
/* Gebe den NiHandle zu einem Socket zurueck               */
/*                                                         */
/* INPUT   sock         Socket                             */
/* OUTPUT  handle       korrespondierender NI-Handle       */
/*                                                         */
/* RETURN                                                  */
/*                                                         */
/* SAP_O_K              alles klar                         */
/* NIEINVAL             sock hat keinen korrespondierenden */
/*                      NI-Handle                          */
/*                                                         */
/*---------------------------------------------------------*/

SAPRETURN API_DF NiSockToHdl 	 (int       sock,
				  NI_HDL *  handle);

/*---------------------------------------------------------*/
/* NiHdlGetPeer ()                                         */
/*                                                         */
/* Gebe die Adresse des Kommunikationspartners eines       */
/* NI_HDL zurueck                                          */
/*                                                         */
/* INPUT   handle         Handle                           */
/* OUTPUT  niha           Host-Adresse                     */
/* OUTPUT  port           Port-Nummer (net-byte-order)     */
/*                        (== 0 fuer UDS Sockets)          */
/* OUTPUT  unix_domain_pathname   sic                      */
/*                        (Speicher dafuer muss            */
/*                         reserviert sein,                */
/*			   NI_UDS_PATH_LEN Bytes)          */
/*                                                         */
/* RETURN                                                  */
/*                                                         */
/* SAP_O_K              alles klar                         */
/* NIEINTERN            Socket-Fehler oder unbekannte      */
/*                      Protokol Domain                    */
/* NIEINVAL             Hdl ist an Unix Domain Namen       */
/*                      gebunden, aber der entsprechende   */
/*                      Parameter war NULL                 */ 
/*                                                         */
/*---------------------------------------------------------*/


SAPRETURN API_DF NiHdlGetPeer  	 (NI_HDL handle,
				  NI_HOSTADR *niha,
				  SAP_USHORT *port,
				  SAP_UC *unix_domain_pathname);

/*---------------------------------------------------------*/
/* NiHdlGetName ()                                         */
/*                                                         */
/* Gebe die Adresse des Komunikationspartners eines NI_HDL */
/* zurueck                                                 */
/*                                                         */
/* INPUT   handle         Handle                           */
/* OUTPUT  niha           Host-Adresse                     */
/* OUTPUT  port           Port-Nummer (net-byte-order)     */
/*                        (== 0 fuer UDS Sockets)          */
/* OUTPUT  unix_domain_pathname   sic                      */
/*                        (Speicher dafuer muss            */
/*                         reserviert sein,                */
/*			   NI_UDS_PATH_LEN Bytes)          */
/*                                                         */
/*                                                         */
/* RETURN                                                  */
/*                                                         */
/* SAP_O_K              alles klar                         */
/* NIEINTERN            Socket-Fehler oder unbekannte      */
/*                      Protokol Domain                    */
/* NIEINVAL             Hdl ist an Unix Domain Namen       */
/*                      gebunden, aber der entsprechende   */
/*                      Parameter war NULL                 */ 
/*                                                         */
/*---------------------------------------------------------*/

SAPRETURN API_DF NiHdlGetName  	 (NI_HDL handle,
				  NI_HOSTADR *ha, 
				  SAP_USHORT *port, 
				  SAP_UC *unix_domain_pathname);

/*---------------------------------------------------------*/
/* NiHdlIsUDSock()                                         */
/*                                                         */
/* Prueft ob ein NI_HDL einen Unix Domain Socket offen hat */
/*                                                         */
/*                                                         */
/* INPUT   handle         Handle                           */
/*                                                         */
/* RETURN                                                  */
/*                                                         */
/* TRUE                 es ist UDS offen                   */
/* FALSE                es ist kein UDS offen              */
/*---------------------------------------------------------*/

SAP_BOOL        NiHdlIsUDSock (NI_HDL handle);

/*---------------------------------------------------------*/
/* NiHdlSetBlockmode ()                                    */
/*                                                         */
/* Setze Blockmode fuer einen Handle (und seine sockets)   */
/* an oder aus.                                            */
/* !!!!! Vorsicht: Dieser Call ist nur fuer diejenigen     */
/*       Benutzer der nilib gedacht, die sockets an der    */
/*       Library vorbei benutzen (muessen) und daher den   */
/*       Blockmode zwischen handle und socket konsistent   */
/*       halten muessen.                                   */
/* !!!!!                                                   */
/*                                                         */
/* INPUT   handle         Handle                           */
/*         mode           TRUE/FALSE (block/nonblock)      */
/*                                                         */
/* RETURN                                                  */
/*                                                         */
/* SAP_O_K              Alles OK                           */
/* NIEINTERN            Fehler                             */
/*                                                         */
/*                                                         */
/*---------------------------------------------------------*/

SAPRETURN API_DF   NiHdlSetBlockmode(NI_HDL handle, SAP_BOOL blocking);


/*---------------------------------------------------------*/
/* NiHdlGetBlockmode ()                                    */
/*                                                         */
/* INPUT   handle         Handle                           */
/*                                                         */
/* RETURN                                                  */
/*                                                         */
/* TRUE                 Blockierende handle                */
/* FALSE                sonst                              */
/*                                                         */
/*                                                         */
/*---------------------------------------------------------*/
SAP_BOOL           NiHdlGetBlockmode ( NI_HDL hdl );

/*---------------------------------------------------------*/
/* NiHdlGetStatus ()                                       */
/*                                                         */
/* Gebe den Verbindungsstatus  eines NI_HDL                */
/*                                                         */
/* INPUT   handle         Handle                           */
/*                                                         */
/* RETURN                                                  */
/*                                                         */
/* SAP_O_K              Es liegen Daten vor                */
/* NIETIMEOUT           Hdl ok, es liegen keine Daten vor  */
/* NIEINTERN            keine Info ueber hdl Status        */
/* NIEINVAL             Es liegt ein Socketfehler vor      */
/* NIECONN_BROKEN       Der Socket zum Hdl wurde           */
/*                      geschlossen                        */
/*                                                         */
/*---------------------------------------------------------*/

SAPRETURN API_DF NiHdlGetStatus  	 (NI_HDL handle);

/*---------------------------------------------------------*/
/* NiHdlGetStatusNB ()                                     */
/* wie oben, setzt den Socket definitiv vorher nonblocking */
/* und nachher !!definitiv!! auf blocking.                 */
/* ---> NICHT BENUTZEN <-----------                        */
/*---------------------------------------------------------*/
SAPRETURN API_DF NiHdlGetStatusNB        (NI_HDL handle);

/*---------------------------------------------------------*/
/* NiHdlSetTrace ()                                        */
/*                                                         */
/* NiHandle-Trace an- oder ausschalten                     */
/*                                                         */
/* INPUT   handle - fuer den der trace an- oder            */
/*                  ausgeschaltet werden soll              */
/*         mode   - TRUE = ON, FALSE = OFF                 */
/*                                                         */
/* RETURN  SAP_O_K oder NIEINVAL (handle ungueltig)        */
/*                                                         */
/*                                                         */
/*---------------------------------------------------------*/

SAPRETURN  API_DF NiHdlSetTrace(NI_HDL handle, SAP_BOOL mode);

/*---------------------------------------------------------*/
/* NiHdlGetTrace ()                                        */
/*                                                         */
/* NiHandle-Trace ermitteln                                */
/*                                                         */
/* INPUT   handle - fuer den der trace                     */
/*                  ermittelt werden soll                  */
/*                                                         */
/*                                                         */
/* RETURN  TRUE    oder FALSE (auch bei handle ungueltig)  */
/*                                                         */
/*                                                         */
/*---------------------------------------------------------*/

SAP_BOOL  API_DF NiHdlGetTrace(NI_HDL handle);

/*---------------------------------------------------------*/
/* NiSetTraceAll ()                                        */
/*                                                         */
/* NiHandle-Trace fuer alle aktiven Handles aktivieren     */
/*                                                         */
/* INPUT                                                   */
/*                                                         */
/*                                                         */
/*                                                         */
/* RETURN                                                  */
/*                                                         */
/*                                                         */
/*---------------------------------------------------------*/

void      API_DF NiSetTraceAll ();

/*---------------------------------------------------------*/
/* NiClearTraceAll ()                                      */
/*                                                         */
/* NiHandle-Trace fuer alle aktiven Handles  leoschen      */
/*                                                         */
/* INPUT                                                   */
/*                                                         */
/*                                                         */
/*                                                         */
/* RETURN                                                  */
/*                                                         */
/*                                                         */
/*---------------------------------------------------------*/

void      API_DF NiClearTraceAll();

/*---------------------------------------------------------*/
/* NiHdlDump  ()                                           */
/* NiHdlDump2 ()                                           */
/*                                                         */
/* Dumpt Informationen ueber einen Handle ins Trace        */
/* der Dump erfolgt bei Tracelevel >= 1                    */
/*                                                         */
/* INPUT   handle         Handle                           */
/*         buffer         Puffer fuer Ausgabe              */
/*         bufLen         Groesse des Puffers              */
/*                                                         */
/* RETURN                 nichts                           */
/*                                                         */
/* SAP_O_K                alles klar                       */
/* NIEINVAL               handle ungueltig oder            */
/*                        pBuffer = NULL                   */
/* NIETOO_SMALL           Puffer zu klein                  */
/*                                                         */
/*---------------------------------------------------------*/

void          API_DF NiHdlDump (NI_HDL     hdl);
SAPRETURN     API_DF NiHdlDump2(NI_HDL     hdl,
                                SAP_UC   * pBuffer,
                                SAP_UINT   bufLen);

/*---------------------------------------------------------*/
/* NiDumpNiHdls ()                                         */
/*                                                         */
/* Dumpt Informationen ueber alle aktiven Handles ins Trace*/
/* der Dump erfolgt bei Tracelevel >= 1                    */
/*                                                         */
/* INPUT                  nichts                           */
/*                                                         */
/* RETURN                 nichts                           */
/*                                                         */
/*                                                         */
/*---------------------------------------------------------*/

void          API_DF NiDumpNiHdls();

/*---------------------------------------------------------*/
/* NiHdlGetOutOctets ()                                    */
/*                                                         */
/* liefere die von einem Handle gesendeten Bytes           */
/*                                                         */
/* INPUT   handle         Handle                           */
/*                                                         */
/* RETURN  gesendete Bytes                                 */
/*                                                         */
/*                                                         */
/*---------------------------------------------------------*/

unsigned long API_DF NiHdlGetOutOctets(NI_HDL handle);

/*---------------------------------------------------------*/
/* NiHdlGetInOctets ()                                     */
/*                                                         */
/* liefere die von einem Handle empfangenen Bytes          */
/*                                                         */
/* INPUT   handle         Handle                           */
/*                                                         */
/* RETURN  empfangene Bytes                                */
/*                                                         */
/*                                                         */
/*---------------------------------------------------------*/

unsigned long API_DF NiHdlGetInOctets(NI_HDL handle);

/*---------------------------------------------------------*/
/* NiHdlGetOutMsgs   ()                                    */
/*                                                         */
/* liefere die von einem Handle gesendeten Ni-Messages     */
/*                                                         */
/* INPUT   handle         Handle                           */
/*                                                         */
/* RETURN  gesendete Messages                              */
/*                                                         */
/*                                                         */
/*---------------------------------------------------------*/

unsigned long API_DF NiHdlGetOutMsgs  (NI_HDL handle);

/*---------------------------------------------------------*/
/* NiHdlGetInMsgs   ()                                     */
/*                                                         */
/* liefere die von einem Handle empfangenen Ni-Messages    */
/*                                                         */
/* INPUT   handle         Handle                           */
/*                                                         */
/* RETURN  empfangene Messages                             */
/*                                                         */
/*                                                         */
/*---------------------------------------------------------*/

unsigned long API_DF NiHdlGetInMsgs  (NI_HDL handle);

/*---------------------------------------------------------*/
/* NiHdlGetStartTime   ()                                  */
/*                                                         */
/* liefere die Sekunden die zwischen dem 1.1. 1970 und der */
/* Zeit als der Handle geoeffnet wurde verstrichen sind    */
/*                                                         */
/* INPUT   handle         Handle                           */
/*                                                         */
/* RETURN  Startzeit                                       */
/*                                                         */
/*                                                         */
/*---------------------------------------------------------*/

long API_DF NiHdlGetStartTime  (NI_HDL handle);

/*--------------------------------------------------------*/
/* NiDgBind ()                                            */
/*                                                        */
/* Vorbereitungen fuer Empfang von Datagrammen treffen    */
/*                                                        */
/* INPUT   servname      Name des Service                 */
/*                       Falls (SAP_UC   *) 0, wird       */
/*                       automatisch ein freier Port      */
/*                       gesucht                          */
/*                                                        */
/* OUTPUT  servno        Nummer des Service               */
/*                       (Host-Byte-Order)                */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIESERV_UNKNOWN       log. Servicename unbekannt       */
/* NIESERV_USED          Service schon benutzt            */
/* NIEINVAL              Handle zu gross fuer NI-Maske    */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiDgBind       	 (SAP_UC     *  servname,
					  SAP_USHORT *  servno);


/*--------------------------------------------------------*/
/* NiDgReceive ()                                         */
/*                                                        */
/* Datagramm empfangen                                    */
/*                                                        */
/* OUTPUT  buf_p_p       Zeiger auf Daten-Puffer          */
/*                       Laenge der Datagramme ist auf    */
/*                       max. 1024 begrenzt               */
/*         len_p         Laenge des Datagramms            */
/*                                                        */
/* INPUT   timeout       in MilliSekunden (oder NI_BLOCK) */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINVAL              NiDgBind nicht vorher aufgerufen */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/* NIEWAKEUP             Weck-Datagramm empfangen         */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/* Datagramme der Laenge <= 1 werden als "Weck-Datagramme"*/
/* behandelt. Mehrere Weck-Datagramme werden beim Receive */
/* zu einem zusammengefasst (Wer wach ist, braucht nicht  */
/* geweckt zu werden).                                    */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiDgReceive    	 (void     **  buf_p_p,
					  SAP_INT   *  len_p,
					  SAP_INT      timeout);


/*--------------------------------------------------------*/
/* NiDgSend ()                                            */
/* NiDg2Send ()                                           */
/*                                                        */
/* Datagramm senden                                       */
/*                                                        */
/* INPUT   buf_p         Daten-Puffer                     */
/*         len           Laenge des Datagramms            */
/*         hostname      Name des Ziel-Host (NiDgSend)    */
/*                       (null-terminiert)                */
/*         hostaddr      Adr des Server-Host (NiDg2Send)  */
/*         servname      log. Name des Service            */
/*                       (null-terminiert)                */
/*         servno        Service-Nummer, falls servname   */
/*                       == (SAP_UC   *) 0                */
/*                       (Host-Byte-Order)                */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINVAL              buf_p oder len ungueltig         */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiDgSend       	 (void       *  buf_p,
					  SAP_INT       len,
					  SAP_UC     *  hostname,
					  SAP_UC     *  servname,
					  SAP_USHORT    servno);

SAPRETURN API_DF  NiDg2Send      	 (void       *  buf_p,
					  SAP_INT       len,
					  NI_HOSTADR *  hostaddr,
					  SAP_UC     *  servname,
					  SAP_USHORT    servno);


/*--------------------------------------------------------*/
/* NiDgGetHdl ()                                          */
/*                                                        */
/* intern benutzten Handle abfragen (fuer NiSelect-       */
/* Aufrufe)                                               */
/*                                                        */
/* OUTPUT  handle                                         */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINVAL              Handle wurde noch nicht angelegt */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiDgGetHdl     	 (NI_HDL *  handle);


/*--------------------------------------------------------*/
/* NiDgFreeResrc ()                                       */
/*                                                        */
/* alle belegten "Datagramm-Resourcen" wieder freigeben   */
/* Aufrufe)                                               */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiDgFreeResrc  	(void);


/*--------------------------------------------------------*/
/* NiRawWrite ()                                          */
/*                                                        */
/* Daten an "Native-TCP-Partner" senden                   */
/*                                                        */
/* INPUT   handle        Handle fuer Verbindungskanal     */
/*                       zum Partner                      */
/*         buf           Daten-Puffer                     */
/*         buflen        Laenge der zu versendenden Daten */
/*         timeout       in MilliSekunden (oder NI_BLOCK) */
/*                       Bezieht sich auf das Senden der  */
/*                       internen Datenpakete, aus denen  */
/*                       sich die Nachricht zusammensetzt)*/
/*                                                        */
/* OUTPUT  writelen      Laenge der gesendeten Daten      */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIECONN_BROKEN        Verbindung zum Partner abgebr.   */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/* NIEINVAL              timeout == 0 oder handle unguelt.*/
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiRawWrite     	 (NI_HDL      handle,
					  void     *  buf,
					  SAP_INT     buflen,
					  SAP_INT     timeout,
					  SAP_INT  *  writelen);


/*--------------------------------------------------------*/
/* NiRawRead ()                                           */
/*                                                        */
/* Daten von "Native-TCP-Partner" empfangen               */
/*                                                        */
/* INPUT   handle        Handle fuer Verbindungskanal     */
/*                       zum Partner                      */
/*         buf           Daten-Puffer                     */
/*         maxlen        Groesse des Daten-Puffers        */
/*         timeout       in MilliSekunden (oder NI_BLOCK) */
/*                       Bezieht sich auf den Empfang der */
/*                       internen Datenpakete, aus denen  */
/*                       sich die Nachricht zusammensetzt)*/
/*                                                        */
/* OUTPUT  readlen       Laenge der empfangenen Daten     */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIECONN_BROKEN        Verbindung zum Partner abgebr.   */
/*                       Ist readlen = 0, hat die Gegen-  */
/*                       seite die Verbindung geschlossen,*/
/*                       ansonsten wurde sie abgebrochen. */
/* NIETOO_SMALL          Puffer zu klein fuer Nachricht   */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/*                       (nachfolgende NiRead-Calls       */
/*                       liefern restliche Daten des      */
/*                       NI-Pakets)                       */
/* NIEPING               Partner hat Ping-Signal gesendet */
/*                       (keine Daten, readlen = 0)       */
/* NIEINVAL              handle ungueltig                 */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/* NIEROUT_HOST_UNKNOWN  im nirouter: log. Hostname unbek.*/
/* NIEROUT_SERV_UNKNOWN  im nirouter: log. Service unbek. */
/* NIEROUT_CONN_REFUSED  im nirouter: Verb.aufbau missl.  */
/* NIEROUT_CONN_BROKEN   im nirouter: Verbindung abgebr.  */
/* NIEROUT_PERM_DENIED   im nirouter: route nicht erlaubt */
/* NIEROUT_VERSION       im nirouter: unguelt. Clnt-Vers. */
/* NIEROUT_CANCELED      im nirouter: Route von Administr.*/
/*                                    ge"canceled"        */
/* NIEROUT_SHUTDOWN      im nirouter: Router Shutdown     */
/* NIEROUT_INTERN        im nirouter: interner Fehler     */
/*                                                        */
/*                                                        */
/* Auf Betriebssystemen, bei denen der Abbruch einer      */
/* Verbindung zu einem Signal (SIGPIPE) fuehrt, muss der  */
/* Aufrufer selbst fuer die Signalbehandlung sorgen (z.B. */
/* auf SIG_IGN setzen)                                    */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiRawRead      	 (NI_HDL      handle,
					  void     *  buf,
					  SAP_INT     maxlen,
					  SAP_INT     timeout,
					  SAP_INT  *  readlen);


/*--------------------------------------------------------*/
/* NiRawPeek ()                                           */
/*                                                        */
/* Testen, ob Daten auf einer best. Leitung angekommen    */
/* sind. Dabei wird nicht die Pufferung benutzt. Diese    */
/* wuerde interne Puffer fuellen, die fuer NiRawRead      */
/* nicht nutzbar sind.                                    */
/*                                                        */
/* INPUT   handle        zu ueberpruefender Handle        */
/*         timeout       in MilliSekunden (oder NI_BLOCK) */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar (Daten da)            */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiRawPeek         	 (NI_HDL   handle,
					  SAP_INT  timeout);

/*--------------------------------------------------------*/
/* NiRawWriteOK ()                                        */
/*                                                        */
/* Testen, ob Write auf Handle blockieren wuerde.         */
/* Dabei wird nicht die Pufferung benutzt. Diese          */
/* wuerde interne Puffer fuellen, die fuer NiRawRead      */
/* nicht nutzbar sind.                                    */
/*                                                        */
/* INPUT   handle        zu ueberpruefender Handle        */
/*         timeout       in MilliSekunden (oder NI_BLOCK) */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar (Daten da)            */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiRawWriteOK      	 (NI_HDL   handle,
					  SAP_INT  timeout);


/*--------------------------------------------------------*/
/* NiReadPending ()                                       */
/* NiWritePending ()                                      */
/*                                                        */
/* Info ueber noch zu lesende / schreibende Bytes liefern */
/* (fuer das letzte NI-Paket)                             */
/*                                                        */
/* INPUT   handle        Handle fuer Verbindungskanal     */
/*                       zum Partner                      */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* < 0                   handle ungueltig                 */
/* sonst                 noch zu lesende / schreibende    */
/*                       Bytes                            */
/*                                                        */
/*--------------------------------------------------------*/

SAP_INT API_DF    NiReadPending  	 (NI_HDL  handle);
SAP_INT API_DF    NiWritePending 	 (NI_HDL  handle);


/*--------------------------------------------------------*/
/* NiWait ()                                              */
/*                                                        */
/* n Millisekunden warten                                 */
/*                                                        */
/* INPUT   millisec      Zeitspanne in Millisekunden      */
/*                                                        */
/*--------------------------------------------------------*/

void API_DF       NiWait         	 (SAP_INT  millisec);


/*--------------------------------------------------------*/
/* NiSetParam ()                                          */
/* NiSetParamEx ()                                        */
/*                                                        */
/* Allgemeinen NI-Parameter setzen                        */
/* NiSetParamEx mit beliebigen Datentypen                 */
/* Fuer komplexe Datentypen muss NiSetParamEx benutzt     */
/* werden, da sonst beim cast Daten verloren gehen        */
/* koennen.                                               */
/*                                                        */
/* INPUT   param         NI-Parameter (s.u.)              */
/*                                                        */
/* IN/OUT  value         Wert                             */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINVAL              unzulaessiger Parameter / Wert   */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF  NiSetParam     	 (SAP_RAW       param,
					  SAP_INT       value);
SAPRETURN API_DF  NiSetParamEx     	 (NI_PARAM_DATA *pdata);

/*--------------------------------------------------------*/
/* NiGetParam ()                                          */
/*                                                        */
/* Allgemeinen NI-Parameter lesen (s.u.)                  */
/*                                                        */
/* INPUT   param         NI-Parameter                     */
/*                                                        */
/* RETURN                Wert des Parameters              */
/*                                                        */
/*--------------------------------------------------------*/
/*--------------------------------------------------------*/
/* NiGetParamEx ()                                        */
/*                                                        */
/* Allgemeinen NI-Parameter lesen (s.u.)                  */
/* NiGetParamEx kann mit beliebigen Datentypen augerufen  */
/* werden. Der Speicher fuer die Daten muss vom Aufrufer  */
/* zur Verfuegung gestellt werden (bei Zeigerwerten).     */
/* Vor dem Aufruf muss der Datentyp in der Struktur auf   */
/* den richtigen Wert gesetzt werden (NI_DT_???).         */
/* Der Wert des Parameters wird im Erfolgsfall in der     */
/* in der Struktur zurueckgeliefert.                      */
/*                                                        */
/* INPUT/OUTPUT                                           */
/*         pdata         Zeiger auf NI-Parameterstruktur  */
/*                                                        */
/* RETURN                Wert                             */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINVAL              unzulaessiger Parameter / Wert   */
/*                                                        */
/*--------------------------------------------------------*/

SAP_INT   API_DF  NiGetParam     	 (SAP_RAW       param);
SAPRETURN API_DF  NiGetParamEx     	 (NI_PARAM_DATA *pdata);


#define NIP_PORTRANGE       (SAP_RAW) 1
#define NIP_MAXPORT         (SAP_RAW) 2
#define NIP_MINPORT         (SAP_RAW) 3
#define NIP_BUF_OPCOD       (SAP_RAW) 4
#define NIP_HSBUF_MAX       (SAP_RAW) 5  
#define NIP_LIB_VERS        (SAP_RAW) 6  
#define NIP_HSBUF_RETRY     (SAP_RAW) 7
#define NIP_HSBUF_REV_RETRY (SAP_RAW) 8  
#define NIP_FRAGTEST        (SAP_RAW) 9
#define NIP_CONNLOCAL       (SAP_RAW) 10
#define NIP_UDS_PATH        (SAP_RAW) 11
#define NIP_QUEUE_LEN       (SAP_RAW) 12
#define NIP_QTEST           (SAP_RAW) 13
#define NIP_LOCAL_ADDR      (SAP_RAW) 14
#define NIP_DNS_TRC_LIMIT   (SAP_RAW) 15
#define NIP_DNS_TIME_FUNC   (SAP_RAW) 16
#define NIP_TRACE_ALL       (SAP_RAW) 17

/*--------------------------------------------------------------------*/
/* allgemeine NI-Parameter NIP_...                                    */
/*                                                                    */
/* PORTRANGE   Bereich fuer lokale Connect-Port Auswahl einschraenken */
/*             0 = nein, sonst ja, Default 0                          */
/*                                                                    */
/* MAXPORT     Obergrenze Port-Range (s. NIP_PORTRANGE, Default 1023) */
/*                                                                    */
/* MINPORT     Untergrenze Port-Range (s. NIP_PORTRANGE, Default 0)   */
/*                                                                    */
/* BUF_OPCOD   NI-Buffer Opcodes (nur NiSetParam)                     */
/*             NI-Buffer ins Trace-File dumpen (0), flushen (1)       */
/*                                                                    */
/* HSBUF_MAX   Anzahl der max. pufferbaren Host/Serv-Eintraege setzen */
/*             (Default 100)                                          */
/*                                                                    */
/* LIB_VERS    NI_VERSION der NI-Library (nur NiGetParam)             */
/*                                                                    */
/* HSBUF_RETRY Anzahl in Sekunden, nach denen ein "Invalid"-Eintrag   */
/*             im Host/Serv-Buffer verfaellt (0 = nie)                */
/*                                                                    */
/* HSBUF_REV_RETRY Schalter, ob "Invalid"-Eintraege fuer              */
/*             IP->host-Aufloesungen verfallen sollen                 */
/*                                                                    */
/* FRAGTEST    Schaltet Fragment-Test-Modus ein (es wird jeweils nur  */
/*             1 Byte gelesen / geschrieben und dann ein Timeout      */
/*             zurueckgegeben (falls nicht NI_BLOCK))                 */
/*             (0 = aus, sonst ein, Default 0)                        */
/*                                                                    */
/* CONNLOCAL   Steuert, ob Connect ueber lokale Domain (AF_UNIX)      */
/*             erfolgen soll                                          */
/*                                                                    */
/* UDS_PATH    Pfadname fuer UNIX DOMAIN SOCKETS. Der Pfadname muss   */
/*             immer absolut angegeben werden!!!                      */
/*             Der Pfad muss ein %d enthalten. Dieser Platzhalter     */
/*             wird durch die socketnummer ersetzt.                   */
/*                                                                    */
/* QUEUE_LEN   Laenge der backlog queue beim listen call.             */
/*             Default Wert: 20                                       */
/*                                                                    */
/* QTEST       Test fuer das NI-Queuing. Ist der Parameter > 0,       */
/*             so errechnet sich die Wahrscheinlichkeit mit der       */
/*             beim Write ein timeout zurueckgegeben wird aus         */
/*             (NIP_QTEST & 0xff) (muss zwischen 0 und 100 sein).     */
/*             Es werden dann immer ((NIP_QTEST) >> 8 & 0xffff),      */
/*             Pakete hintereinander blockiert. Zum erzeugen des      */
/*             Wertes NIP_QTEST aus den einzelparametern benutzt      */
/*             man das Makro NI_QTEST_VAL.                            */
/*                                                                    */
/* LOCAL_ADDR  IP-Adresse fuer ausgehende Verbindungen. Bei 0.0.0.0   */
/*             wird default Adresse verwendet.                        */
/*                                                                    */
/* DNS_TRC_LIMIT Trace Eintrag schrieben, falls die Hostnamen- oder   */
/*             Servicenamen-Aufloesung laenger als <value> ms dauert. */
/*             Trace ausschalten mit <value> = 0.                     */
/*             Daten-Typ: NI_DT_UINT; Defaut-Wert: 0                  */
/*                                                                    */
/* NIP_DNS_TIME_FUNC Time Funktion setzen, mit welcher die Zeit fuer  */
/*             DNS_TRC_LIMIT berechnet wird. Ist die Funktion NULL,   */
/*             wird eine interne verwendet, welche jedoch ungenau     */
/*             sein kann (Sekundenbereich).                           */
/*             Format: SAP_UINT (function *)(void)                    */
/*             Rueckgabe in Mikrosekunden (wie pfclock*())            */
/*             Daten-Typ: NI_DT_VOIDPTR; Defaut-Wert: NULL            */
/* NIP_TRACE_ALL  bestimmt ob alle neuen Handles den Trace            */
/*             eingeschaltet haben                                    */
/*                                                                    */
/*--------------------------------------------------------------------*/

#define NI_QTEST_VAL(_count, _prop) (((_count) & 0xffff) << 8) | ((_prop) & 0xff)


/*--------------------------------------------------------------------*/
/* Pufferspezifische NI-Parameter NIHP_...                            */
/*                                                                    */
/* MODE        Sprachmodus der Verbindung: MESG, ROUT, RAW            */
/*                                                                    */
/* CHECK_QUE   Statusabfrage/Setzen einer Verbindung bei der          */
/*             Pufferverwaltung.                                      */
/*             Mit SetParam wird das Check-flag gesetzt               */
/*             Mit GetParam wird der Status abgefragt                 */
/*             (0: Outqueue ist leer, 1: queue voll und check flag 0  */
/*              2: outqueue voll und check flag 1)                    */
/*                                                                    */
/* ALLOC_FUNC  Funktion zur Erfuellung von Speicheranfragen           */
/*                                                                    */
/* MAXQUEUE    Maximale Anzahl von Paketen in der outqueue. Der       */
/*             Sender wird abgebremst, bis wieder Pakete in die       */
/*             queue passen.                                          */
/*                                                                    */
/* WARNQUEUE   Schwellenwert fuer die Anzahl an Paketen, die noch     */
/*             in die Queue passen sollen. Wird dieser Wert unter-    */
/*             schritten, wird der Fehler NIEQUEFULL zurueckgegeben.  */
/*                                                                    */
/* MAXHEAP     Maximale Speichernutzung durch die Queues. Wird dieser */
/*             Schwellenwert ueberschritten, wird der Fehler          */
/*             NIEQUEFULL zurueckgegeben.                             */
/*                                                                    */
/*--------------------------------------------------------------------*/


/*--------------------------------------------------------*/
/* NiSetHdlParam ()					  */
/* NiGetHdlParam ()					  */
/*                                                        */
/* Handlespezifischen NI-Parameter setzen/lesen		  */
/*                                                        */
/* INPUT   hdl           NI-Handle			  */
/*         param         NI-Parameter (s.u.)              */
/*                                                        */
/* IN/OUT  value         Wert                             */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINVAL              unzulaessiger Parameter / Wert   */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiSetHdlParam (NI_HDL    hdl,
				SAP_RAW   param,
				void     *value);
SAPRETURN API_DF NiGetHdlParam (NI_HDL    hdl,
				SAP_RAW   param,
				void     *value);

/*--------------------------------------------------------*/
/* NiGetSapDllRelease()                                   */
/*                                                        */
/* SAP Release zurueckgeben.                              */
/* Die Funktion kann verwendet werden um das SAP Release  */
/* der Bibliothek dem Aufrufer bekannt zu geben           */
/*                                                        */
/* INPUT   len           maximale Laenge Release string   */
/*                                                        */
/* OUTPUT  release       Release string                   */
/*                                                        */
/*--------------------------------------------------------*/

void API_DF       NiGetSapDllRelease (SAP_UC * release,
				      int    len);


/*--------------------------------------------------------*/
/* NiErrSet ()                                            */
/*                                                        */
/* Sap-Error-Info ueber Ni-Returncode bereitstellen.      */
/* Die Sap-Error-Info kann anschliessend ueber die        */
/* Funktionen der Sap-Error-Schnittstelle abgeholt werden */
/* (siehe err.h)                                          */
/*                                                        */
/* INPUT   nirc          Ni-Returncode                    */
/*                                                        */
/*--------------------------------------------------------*/

void API_DF       NiErrSet       	 (SAPRETURN  nirc);


/*--------------------------------------------------------*/
/* NiErrStr ()                                            */
/*                                                        */
/* Namens-String  zu Ni-Error-Konstante liefern           */
/*                                                        */
/* INPUT   nirc          Ni-Returncode                    */
/*                                                        */
/* RETURN                Namensstring                     */
/*                                                        */
/*--------------------------------------------------------*/

SAP_UC   * API_DF NiErrStr       (SAPRETURN nirc);


/*--------------------------------------------------------------------*/
/* NiSecStr  ()                                                       */
/* NiSecStr2 ()   (reentrant)                                         */
/*                                                                    */
/* SapRouter-Passwort aus String (Route) ausblenden                   */
/*                                                                    */
/* INIT    bei der reentrant library (nirlib) wird beim Aufruf der    */
/*         nicht reentrant Funktion NI initialisiert.                 */
/*                                                                    */
/* INPUT   inStr         Inputstring                                  */
/*         secStr        String Puffer fuer String ohne Passwort      */
/*         bufLen        Groesse des Puffers in SAP_UC                */
/*                       (optimal: strlenU(inStr) + 1)                */
/*                                                                    */
/* RETURN                String mit ausgeblendetem Passwort (falls    */
/*                       String groesser ist als interner Puffer,     */
/*                       wird der String unveraendert zurueckgegeben, */
/*                       bei internem Fehler wird NULL zurueckgegeben)*/
/*                                                                    */
/* SAP_O_K               alles klar                                   */
/* NIEINVAL              inStr = NULL bzw. secStr = NULL              */
/* NIETOO_SMALL          Puffer zu klein                              */
/*                                                                    */
/* Bei der nicht reentrant Funktionen wird threadeigener statischer   */
/* Speicher zurueckgegeben. Die Nutzung der reentrant Funktionen wird */
/* empfohlen!                                                         */
/*                                                                    */
/*--------------------------------------------------------------------*/

SAP_UC   * API_DF NiSecStr       (const SAP_UC * inStr);
SAPRETURN  API_DF NiSecStr2      (const SAP_UC * inStr,
                                  SAP_UC       * secStr,
                                  SAP_UINT       bufLen);


/*--------------------------------------------------------------------*/
/* Ni???ToStr  ()                                                     */
/* Ni???ToStr2 ()   (reentrant)                                       */
/*                                                                    */
/* Konvertiert Hostadresse bzw. Service Nummer in String              */
/*                                                                    */
/* INIT    bei der reentrant library (nirlib) wird beim Aufruf einer  */
/*         nicht reentrant Funktion NI initialisiert.                 */
/*                                                                    */
/* INPUT   hostAddr      Hostadresse (NULL -> "0.0.0.0")              */
/*         hostAddrStr   String Puffer fuer Hostadresse               */
/*         netServNo     Service-Nummer (Net-Byte-Order)              */
/*         servNoStr     String Puffer fuer Service-Nummer            */
/*         bufLen        Groesse des Puffers in SAP_UC (optimal fuer  */
/*                       hostAddr: 16 bzw. ServNo: 6; ServNo L: 12;   */
/*                       vgl. NI_HOSTADDR_STR_LEN etc.)               */
/*                                                                    */
/* RETURN                formatierter String                          */
/*                                                                    */
/* SAP_O_K               alles klar                                   */
/* NIEINVAL              hostAddrStr = NULL bzw. servNoStr = NULL     */
/* NIETOO_SMALL          Puffer zu klein                              */
/*                                                                    */
/* Bei den nicht reentrant Funktionen wird threadeigener statischer   */
/* Speicher zurueckgegeben. Bei den Funktionen NiSrvToStr und         */
/* NiSrvToStrL handelt es sich um den selben Speicher, NiAdrToStr hat */
/* eigenen. Die Nutzung der reentrant Funktionen wird empfohlen!      */
/*                                                                    */
/*--------------------------------------------------------------------*/

SAP_UC  * API_DF NiAdrToStr    (NI_HOSTADR * hostAddr);
SAPRETURN API_DF NiAdrToStr2   (NI_HOSTADR * hostAddr,
                                SAP_UC     * hostAddrStr,
                                SAP_UINT     bufLen);

SAP_UC  * API_DF NiSrvToStr    (SAP_USHORT   netServNo);
SAPRETURN API_DF NiSrvToStr2   (SAP_USHORT   netServNo,
                                SAP_UC     * servNoStr,
                                SAP_UINT     bufLen);

SAP_UC  * API_DF NiSrvToStrL   (SAP_USHORT   netServNo);
SAPRETURN API_DF NiSrvToStrL2  (SAP_USHORT   netServNo,
                                SAP_UC     * servNoStr,
                                SAP_UINT     bufLen);


/*--------------------------------------------------------------------*/
/* NiNetINT  ()                                                       */
/* NiHostINT ()                                                       */
/*                                                                    */
/* Konvertiere SAP_INTs von, bzw. nach Netzbyteorder                  */
/*                                                                    */
/* INIT    NI muss nicht initialisirt sein.                           */
/*                                                                    */
/* INPUT   hostInt       Integer in Host-Byte-Order                   */
/*         netInt        Integer in Net-Byte-Order                    */
/*                                                                    */
/* RETURN                konvertierter Interegr                       */
/*                                                                    */
/*--------------------------------------------------------------------*/

/* SAPUNICODEOK_MIXEDLEN */
SAP_INT API_DF    NiNetINT     	 (SAP_INT  hostInt);

/* SAPUNICODEOK_MIXEDLEN */
SAP_INT API_DF    NiHostINT    	 (SAP_INT  netInt);

/*--------------------------------------------------------------------*/
/* NiNetServno  ()                                                    */
/* NiHostServno ()                                                    */
/*                                                                    */
/* Konvertiere Servicenummern von, bzw. nach Netzbyteorder            */
/*                                                                    */
/* INIT    NI muss nicht initialisirt sein.                           */
/*                                                                    */
/* INPUT   hostServNo    Servicenummer in Host-Byte-Order             */
/*         netServNo     Servicenummer in Net-Byte-Order              */
/*                                                                    */
/* RETURN                konvertierte Servicenummer                   */
/*                                                                    */
/*--------------------------------------------------------------------*/

SAP_USHORT API_DF NiNetServno 	 (SAP_USHORT  hostServNo);

SAP_USHORT API_DF NiHostServno 	 (SAP_USHORT  netServNo);


/*--------------------------------------------------------*/
/* NiSetSingleRead ()                                     */
/*                                                        */
/* Setze Socket auf 'Single Read'                         */
/* `Single Read' heisst, dass in jedem NiRead nur einmal  */
/* versucht wird zu lesen. Normalerweise wird nach        */
/* erfolgreichem Lesen, nocheinmal versucht zu lesen,     */
/* falls das NI-Paket nicht komplett ist. Dies wird       */
/* hiermit verhindert. (Ist unter Windows notwendig.)     */
/*                                                        */
/* INPUT  handle    NI-Handle                             */
/*        mode      TRUE/FALSE                            */
/*                                                        */
/* RETURN SAP_O_K   alles klar                            */
/*        NIEINVAL  kein gueltiger NI-Handle              */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiSetSingleRead (SAP_INT  handle,
				  SAP_BOOL  mode);

/*--------------------------------------------------------*/
/* NiGetSingleRead ()                                     */
/*                                                        */
/* Abfrage auf 'Single Read'                              */
/*                                                        */
/* INPUT  handle    NI-Handle                             */
/*                                                        */
/* RETURN modus                                           */
/*                                                        */
/*--------------------------------------------------------*/

SAP_BOOL  API_DF NiGetSingleRead (SAP_INT  handle);

/*--------------------------------------------------------*/
/* NiGetRawSocket                                         */
/*                                                        */
/* Gebe den socket an den Aufrufer zurueck. Nur RAW       */
/* sockets koennen (sinnvoll) genutzt werden. Socket      */
/* darf nicht von extern veraendert (z.B. geschlossen)    */
/* werden!                                                */
/*                                                        */
/* IN                                                     */
/*      handle           NI-Handle                        */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINTERN             interner Fehler aufgetreten      */
/* NIEINVAL              kein valider Parameter           */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiGetRawSocket    (NI_HDL      handle,
                                    int         *socket);

/*--------------------------------------------------------*/
/* NiSelNewSet ()                                         */
/*                                                        */
/* Fordert eine neues SelectSet an                        */
/*                                                        */
/* IN/OUT  set           selectset                        */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINTERN             interner Fehler aufgetreten      */
/*                                                        */
/*--------------------------------------------------------*/

/*-NiSel--------------------------------------------------*/

SAPRETURN NiSelNewSet     (NISEL_HDL *sel_hdl_ptr);

/*--------------------------------------------------------*/
/* NiSelNewSet2 ()                                        */
/*                                                        */
/* Fordert eine neues SelectSet an                        */
/*                                                        */
/* IN/OUT  set           selectset                        */
/*         max_sockets   maximum number of sockets        */
/*                       in the select set                */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINTERN             interner Fehler aufgetreten      */
/*                                                        */
/*--------------------------------------------------------*/

/*-NiSel--------------------------------------------------*/

SAPRETURN NiSelNewSet2    (NISEL_HDL *sel_hdl_ptr, int max_sockets);

/*--------------------------------------------------------*/
/* NiSelDestroySet ()                                     */
/*                                                        */
/* Gibt eine altes SelectSet wieder frei                  */
/*                                                        */
/* IN/OUT  set           selectset                        */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINVAL              kein valider selectset           */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN NiSelDestroySet (NISEL_HDL *sel_hdl_ptr);

/*--------------------------------------------------------*/
/* NiSelClearSet ()                                       */
/*                                                        */
/* Loescht alle Eintaege in einem SelectSet               */
/*                                                        */
/* INPUT   set           selectset                        */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINVAL              kein valider selectset           */
/* NIEINTERN             interner Fehler                  */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN NiSelClearSet   (NISEL_HDL sel_hdl);

/*--------------------------------------------------------*/
/* NiSelSet/Clear ()                                      */
/*                                                        */
/* Setzt/Loescht die flags fuer diesen NI-Handle flags    */
/* kann hier aus einer Bit-Kombination von NI_WRITE,      */
/* NI_READ und NI_CONNECT bestehen. NiSelSet setzt        */
/* ausserdem die data zu diesem Handle.                   */
/*                                                        */
/* INPUT   set           selectset                        */
/*         hdl           NI-Handle                        */
/*         flags         Bitkombination aus NI_CONNECT,   */
/*                       NI_READ und NI_WRITE             */
/*         void          beliebige Applikationsdaten      */
/*                                                        */
/* IN/OUT  value         Wert                             */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINTERN             interner Fehler                  */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN NiSelSet        (NISEL_HDL sel_hdl,
                           NI_HDL hdl,
                           SAP_RAW flags,
                           void *data);

SAPRETURN NiSelClear      (NISEL_HDL sel_hdl,
                           NI_HDL hdl,
                           SAP_RAW flags);

/*--------------------------------------------------------*/
/* NiSelSelect ()                                         */
/*                                                        */
/* Ersatz fuer NiSelect                                   */
/* Setzt den Iterator fuer NiSelNext implizit zurueck     */
/*                                                        */
/* INPUT   set           selectset                        */
/*         timeout       timeout fuers select             */
/*                                                        */
/* RETURN                                                 */
/*  siehe NiSelect                                        */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN NiSelSelect     (NISEL_HDL sel_hdl, SAP_INT timeout);

/*--------------------------------------------------------*/
/* NiSelNext  ()                                          */
/*                                                        */
/* Geben ersten/naechsten NI-Handle, der laut mode bereit */
/* ist, zurueck. Direkt nach NiSelect mit diesem Set sind */
/* beide aequivalent                                      */
/*                                                        */
/* INPUT   set           selectset                        */
/* IN/OUT  read/                                          */
/*         write/                                         */
/*         connect       Wenn NULL, modus interessiert    */
/*                                  nicht                 */
/*                       sonst Pointer auf Variable       */
/*                                                        */
/* OUT     data          Userdata                         */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* Ni_INVALID_HDL        kein Handle mehr verfuegbar      */
/*                                                        */
/*--------------------------------------------------------*/

NI_HDL    NiSelNext       (NISEL_HDL sel_hdl,
                           SAP_BOOL *read,
                           SAP_BOOL *write,
                           SAP_BOOL *connect,
                           void **data);

/*--------------------------------------------------------*/
/* NiSelStartLoop ()                                      */
/*                                                        */
/* Setzt den Iterator fuer NiSelNext wieder auf den ersten*/
/* Eintrag                                                */
/*                                                        */
/* INPUT   set           selectset                        */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINVAL              kein valider selectset           */
/* NIEINTERN             interner Fehler                  */
/*                                                        */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN NiSelStartLoop  (NISEL_HDL sel_hdl);

/*--------------------------------------------------------*/
/* NiSelMode ()                                           */
/* NiSelStat ()                                           */
/*                                                        */
/* Gib den Status/Modus zu hdl zurueck                    */
/* Status ist dabei der Status, den das letzte Select     */
/* zurueckgeliefert hat, waehrend modus der in diesem     */
/* Set angeforderte Modus fuer das naechste Select ist.   */
/*                                                        */
/* INPUT   set           selectset                        */
/*         hdl           NI-Handle                        */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* modus des Handles in der maske                         */
/*                                                        */
/*--------------------------------------------------------*/

SAP_RAW API_DF NiSelMode     (NISEL_HDL sel_hdl, NI_HDL hdl);
SAP_RAW API_DF NiSelStat     (NISEL_HDL sel_hdl, NI_HDL hdl);

/*-NiBuf--------------------------------------------NiSel-*/

/*--------------------------------------------------------*/
/* NiBufReceive                                           */
/*                                                        */
/* Empfange einen NI-Puffer                               */
/*                                                        */
/* INPUT   hdl           Handle fuer Verbindungskanal     */
/*                       zum Partner                      */
/*         timeout       Zeitlimit                        */
/*                                                        */
/* OUTPUT  buffer        Pointer auf Pufferpointer        */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIECONN_BROKEN        Verbindung zum Partner abgebr.   */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/* NIEINVAL              Puffer oder handle unguelt.      */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiBufReceive  (NI_HDL      hdl,
                                NIBUFFER  **buffer,
                                SAP_INT     timeout);

/*--------------------------------------------------------*/
/* NiBufSend                                              */
/*                                                        */
/* Sende einen NI-Puffer                                  */
/*                                                        */
/* INPUT   hdl           Handle fuer Verbindungskanal     */
/*                       zum Partner                      */
/*         timeout       Zeitlimit                        */
/* IN/OUT  buffer        NI-Puffer zum senden. Wird bei   */
/*                       Erfolg freigegeben.              */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIECONN_BROKEN        Verbindung zum Partner abgebr.   */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/* NIEINVAL              Puffer oder handle unguelt.      */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiBufSend     (NI_HDL      hdl,
                                NIBUFFER  **buffer,
                                SAP_INT     timeout);

/*--------------------------------------------------------*/
/* NiSendHandleEx                                         */
/*                                                        */
/* Sende einen NI-Handle ueber einen anderen              */
/*                                                        */
/* INPUT   hdl           Handle fuer Verbindungskanal     */
/*                       zum Partner                      */
/*         sendhdl       Handle, der gesendet werden soll */
/*         timeout       Zeitlimit                        */
/*         saveModes     handle Eigenschaften uebernehmen */
/*                       - buffered   (def. TRUE )        */
/*                       - talkmode   (def. NI   )        */
/*                       - blocking   (def. FALSE)        */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIECONN_BROKEN        Verbindung zum Partner abgebr.   */
/* NIEINVAL              Puffer oder handle unguelt.      */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

#define NiSendHandle(hdl, sendhdl, timeout) \
        NiSendHandleEx(hdl, sendhdl, timeout, FALSE)

SAPRETURN NiSendHandleEx (NI_HDL   hdl,
                          NI_HDL   sendhdl,
                          SAP_INT  timeout,
                          SAP_BOOL saveModes);

/*--------------------------------------------------------*/
/* NiBufFree                                              */
/*                                                        */
/* Dies ist die Funktion, ueber die beliebige NI-Puffer   */
/* freigegeben werden. Sie wird intern dazu aufgerufen    */
/* und kann von der Applikation gerufen werden.           */
/*                                                        */
/* IN/OUT  buffer        Alter Puffer                     */
/*                                                        */
/*--------------------------------------------------------*/

void     API_DF NiBufFree     (NIBUFFER  **buffer);

/*--------------------------------------------------------*/
/* NiBufAlloc                                             */
/*                                                        */
/* Allokiere einen neuen NI-Puffer                        */
/*                                                        */
/* INPUT   len           Laenge des Datenbereichs         */
/*         txt           Erkennungstext fuer den Puffer   */
/*                                                        */
/* OUTPUT  buffer        Neuer Puffer                     */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINVAL              Puffer oder handle unguelt.      */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiBufAlloc    (NIBUFFER  **buffer,
                                SAP_INT     len,
                                SAP_UC     *txt);

/*--------------------------------------------------------*/
/* NiBufAlloc2                                            */
/*                                                        */
/* Allokiere einen neuen NI-Puffer, bei dem eine externe  */
/* Free-Funktion vorhanden ist, die von NiBufFree         */
/* aufgerufen wird, anstatt den Puffer sofort wieder      */
/* freizugeben. Diese Freefunktion kann dann NiBufFree2   */
/* aufrufen, um den Puffer endgueltig freizugeben         */
/*                                                        */
/* INPUT  len           Laenge des Datenbereichs          */
/*        txt           Erkennungstext fuer den Puffer    */
/* OUTPUT buffer        Neuer Puffer                      */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINVAL              Puffer oder handle unguelt.      */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/
/* NiBufFree2                                             */
/*                                                        */
/* Gibt einen NI-Puffer frei, ohne evtuelle externe       */
/* freefunktion zu rufen (siehe NiBufAlloc2)              */
/*                                                        */
/* IN/OUT  buffer        Alter Puffer                     */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiBufAlloc2   (NIBUFFER  **buffer,
                                SAP_INT     len,
                                SAP_UC     *text,
                                NIBUF_FREE_FUNC *func);

void      API_DF NiBufFree2    (NIBUFFER  **buffer);

/*--------------------------------------------------------*/
/* NiBufDup                                               */
/*                                                        */
/* Dupliziere einen Puffer, um ihn an mehrere Clients zu  */
/* senden. Nach dem Senden einer Referenz darf der Inhalt */
/* des Puffers nicht mehr veraendert werden.              */
/*                                                        */
/* INPUT   from          Bestehender Puffer               */
/*                                                        */
/* OUTPUT  to            Neue Referenz auf Puffer from    */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINVAL              Puffer oder handle unguelt.      */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiBufDup      (NIBUFFER  **to,
                                NIBUFFER   *from);

/*--------------------------------------------------------*/
/* NiBufWrap                                              */
/*                                                        */
/* Erstelle einen Puffer zu einem bestehenden             */
/* Datenbereich:                                          */
/* Wenn der Puffer NiBufFree ruft dann die den Parameter  */
/* 'freef', um den Datenbereich freizugeben. Diese        */
/* Funktion muss dann NiBufUnwrap aufrufen.               */
/*                                                        */
/* INPUT   data          Anfang auf den Datenbereich      */
/*         len           Laenge des Datenbereichs         */
/*         spacelen      Laenge des gesammten Puffers     */
/*         freef         Freigabefunktion                 */
/*                                                        */
/* OUTPUT  to            Neue Referenz auf Puffer from    */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINVAL              Puffer oder handle unguelt.      */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/
/* NiBufUnwrap                                            */
/*                                                        */
/* Entpacke einen mit NiBufWrap erzeugten NI-Puffer.      */
/* Diese Funktion sollte nur aus der in NiBufWrap         */
/* angegebenen Freefunktion aufgerufen werden             */
/*                                                        */
/* IN/OUT  buffer        NI-Puffer                        */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* NULL                  Puffer war nicht mit NiBufWrap   */
/*                       erzeugt                          */
/* sonst                 Datenbereich                     */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiBufWrap     (NIBUFFER  **to,
                                SAP_RAW    *data,
                                SAP_INT     len,
                                SAP_INT     spacelen,
                                NIBUF_FREE_FUNC *freef,
                                void       *uptr);

void *    API_DF NiBufUnWrap   (NIBUFFER  **buffer);

/*--------------------------------------------------------*/
/* NiBufGetPtr                                            */
/*                                                        */
/* Gebe den Userptr zurueck                               */
/*                                                        */
/* IN      buf           Puffer fuer den stack            */
/*                                                        */
/* RETURN  userpointer                                    */
/*                                                        */
/*--------------------------------------------------------*/

void * API_DF NiBufGetPtr      (NIBUFFER  *buf);


/*--------------------------------------------------------*/
/* NiBufPush                                              */
/*                                                        */
/* Baue einen Stack aus NI-Puffern                        */
/*                                                        */
/* IN/OUT  stack         stack-pointer                    */
/*                                                        */
/* IN      buf           Puffer fuer den stack            */
/*                                                        */
/*--------------------------------------------------------*/
/*                                                        */
/* NiBufPop                                               */
/*                                                        */
/* Hole einen Puffer vom stack                            */
/*                                                        */
/* IN/OUT  stack         stack-pointer                    */
/*                                                        */
/* RETURN  NULL          Stack leer                       */
/*         sonst         Puffer                           */
/*                                                        */
/*--------------------------------------------------------*/

void      API_DF NiBufPush     (NIBUFFER **stack,
                                NIBUFFER *buf);
NIBUFFER *API_DF NiBufPop      (NIBUFFER **stack);

/*-NiSnc--------------------------------------------NiBuf-*/

/*--------------------------------------------------------------------*/
/* NiSncAclStr ()                                                     */
/*                                                                    */
/* Converts a binary ACL key into a printable name                    */
/*                                                                    */
/* INPUT   pAcl            ACL key                                    */
/*         pBuffer         buffer to save printable name              */
/*         bufLen          length of buffer in characters             */
/*                                                                    */
/* OUTPUT  pBuffer         printable name of ACL key (NULL terminated)*/
/*                                                                    */
/* RETURN  SAP_O_K         successful                                 */
/*         NIEINVAL        pAcl or pBuffer is NULL                    */
/*         NIETOO_SMALL    buffer to small to save printable name     */
/*                         (smaller than SNC_NAME_MAXLEN)             */
/*         NIESNC_FAILURE  failure in convertion; error info is set   */
/*                                                                    */
/*--------------------------------------------------------------------*/

SAPRETURN API_DF NiSncAclStr  (const NISNC_ACLKEY * pAcl,
                               SAP_UC             * pBuffer,
                               SAP_UINT             bufLen);

/*--------------------------------------------------------*/
/* NiSncGetPeer                                           */
/*                                                        */
/*                                                        */
/*                                                        */
/* INPUT   hdl           Handle fuer Verbindungskanal     */
/*                       zum Partner                      */
/*                                                        */
/* OUTPUT  peer          SNC-Account                      */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIETIMEOUT            Der Handle ist noch nicht lese-  */
/*                       oder schreibbereit               */
/* NIESNC_FAILURE        Verbindung ist nicht             */
/*                       SNC-gesichert                    */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiSncGetPeer  (NI_HDL        hdl,
                                NISNC_ACLKEY *peer);

/*--------------------------------------------------------*/
/* NiSncRequest                                           */
/*                                                        */
/* Fordere eine SNC-Verschluesselung zum Client mit       */
/* SNC-Name peer:                                         */
/* Hiermit kann ein Server Verschluesselung nach dem      */
/* Accept einer normalen Verbindung anfordern.            */
/* SNC-Fehler werden aber erst waehrend der normalen      */
/* Communication gemeldet                                 */
/*                                                        */
/* INPUT   hdl          NI_HDL                            */
/*         timeout      timeout                           */
/*         peer         Client-SNC-Name                   */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiSncRequest (NI_HDL        hdl,
                               SAP_INT       timeout,
                               NISNC_ACLKEY *peer);

/*--------------------------------------------------------*/
/* NiSncNameToAcl                                         */
/*                                                        */
/*                                                        */
/*                                                        */
/* INPUT   handle        Handle fuer Verbindungskanal     */
/*                       zum Partner                      */
/*                                                        */
/* OUTPUT                                                 */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIECONN_BROKEN        Verbindung zum Partner abgebr.   */
/* NIETIMEOUT            Zeitlimit ueberschritten         */
/* NIEINVAL              Puffer oder handle unguelt.      */
/* NIEINTERN             interner Fehler (Err-Trace)      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiSncNameToAcl (SAP_UC       *name,
                                 NISNC_ACLKEY *acl);

/*--------------------------------------------------------*/
/* NiSncAclCmp                                            */
/*                                                        */
/*                                                        */
/*                                                        */
/* INPUT   acl1, acl2    SNC-Accounts                     */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* TRUE                  SNC-Namen sind gleich            */
/* FALSE                 sonst                            */
/*                                                        */
/*--------------------------------------------------------*/

SAP_BOOL  API_DF NiSncAclCmp   (const NISNC_ACLKEY *acl1,
                                const NISNC_ACLKEY *acl2);
/*--------------------------------------------------------*/
/* NiSncFinHS                                             */
/*                                                        */
/*                                                        */
/*                                                        */
/* INPUT                 handle fuer Verb zum Partner     */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               Austausch von Overhead war noetig*/
/* NIESNC_FAILURE        kein Overhead  zu verarbeiten    */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN  API_DF NiSncFinHS(NI_HDL handle);

/*--------------------------------------------------------*/
/* NiGetMaxSockets ( void ) 	                          */
/*                                                        */
/* return the maximal number of sockets                   */
/*							  */
/*                  					  */
/*                                                        */
/*--------------------------------------------------------*/

int NiGetMaxSockets ( void );

/*--------------------------------------------------------*/
/* NiDgHdlCreate                                          */
/*                                                        */
/* Create a Datagram Handle                               */
/*                                                        */
/* INPUT   ptr           ptr to the handle                */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               ok                               */
/* NIEINTERN             OS-error or out of NI-Handles    */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiDgHdlCreate      (NI_HDL *ptr);

/*--------------------------------------------------------*/
/* NiDgHdlBindName                                        */
/*                                                        */
/* Bind hdl to a local hostname and port                  */
/*                                                        */
/* INPUT   hdl           valid NI_HDL                     */
/*         name          hostname of an interface         */
/*         service       port                             */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               ok                               */
/* NIEINVAL              either name or service could     */
/*                       not be resolved  or              */
/*                       hdl invalid                      */
/* NIESERV_USED          service already in use           */
/* NIEMYHOST_VERIFY      non-local address specified      */
/* NIEINTERN             internal error                   */
/*                                                        */
/* (USE NiHdlGetName TO VERIFY THE BOUND NAME ADDRESS)    */
/*                                                        */
/*--------------------------------------------------------*/
SAPRETURN API_DF NiDgHdlBindName    (NI_HDL hdl,
                                     SAP_UC *name,
                                     SAP_UC *service) ;
/*--------------------------------------------------------*/
/* NiDgHdlBind                                            */
/*                                                        */
/* Bind hdl to a local address and service                */
/*                                                        */
/* INPUT   hdl           valid NI_HDL                     */
/*         addr          NI_HOSTADR of an interface       */
/*                       (or {{0,0,0,0}})                 */
/*         port          port in net-byte-order (or -1)   */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               ok                               */
/* NIEINVAL              either name or service could     */
/*                       not be resolved  or              */
/*                       hdl invalid                      */
/* NIESERV_USED          service already in use           */
/* NIEMYHOST_VERIFY      non-local address specified      */
/* NIEINTERN             internal error                   */
/*                                                        */
/* (USE NiHdlGetName TO VERIFY THE BOUND NAME ADDRESS)    */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiDgHdlBind        (NI_HDL hdl,
                                     NI_HOSTADR addr,
                                     SAP_USHORT port);
/*--------------------------------------------------------*/
/* NiDgHdlRecvFrom                                        */
/*                                                        */
/* Receive on a DgHdl                                     */
/*                                                        */
/* INPUT   hdl           Handle (created and bound)       */
/*         buffer        where to place data              */
/*         buflen        length of buffer                 */
/* OUTPUT  buffer        filled buffer                    */
/*         recvlen       contains the number of bytes     */
/*                       received                         */
/*         addr          if given, contains address of    */
/*                       remote peer                      */
/*         port          if given, contains port-number   */
/*                       of remote peer (net-byte-order)  */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               ok                               */
/* NIEINVAL              handle invalid                   */
/* NIEINTERN             socket error                     */
/*                                                        */
/* (USE NiHdlGetName TO VERIFY THE BOUND ADDRESS)         */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiDgHdlRecvFrom    (NI_HDL hdl,
                                     NI_HOSTADR *addr,
                                     SAP_USHORT *port,
                                     SAP_RAW *buffer,
                                     int buflen,
                                     int *recvlen);

/*--------------------------------------------------------*/
/* NiDgHdlSendTo                                          */
/*                                                        */
/* Send datagram to a peer, using a DgHdl                 */
/*                                                        */
/* INPUT   hdl           Handle (created)                 */
/*         buffer        output buffer                    */
/*         buflen        length of buffer                 */
/*         addr          address of remote peer           */
/*         port          port-number of remote peer       */
/*                       (in net-byte-order)              */
/* OUTPUT                                                 */
/*         sentlen       contains the number of bytes     */
/*                       actualy  sent                    */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               ok                               */
/* NIEINVAL              handle invalid                   */
/* NIEINTERN             socket error                     */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiDgHdlSendTo      (NI_HDL hdl,
                                     NI_HOSTADR addr,
                                     SAP_USHORT port,
                                     SAP_RAW *buffer,
                                     int buflen,
                                     int *sentlen);
/*--------------------------------------------------------*/
/* NiDgHdlSendToName                                      */
/*                                                        */
/* Send datagram to a peer, using a DgHdl                 */
/*                                                        */
/* INPUT   hdl           Handle (created)                 */
/*         buffer        output buffer                    */
/*         buflen        length of buffer                 */
/*         hostname      name    of remote peer           */
/*         service       service     of remote peer       */
/* OUTPUT                                                 */
/*         sentlen       contains the number of bytes     */
/*                       actualy  sent                    */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               ok                               */
/* NIEINVAL              handle invalid                   */
/* NIEINTERN             socket error                     */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiDgHdlSendToName  (NI_HDL hdl,
                                     SAP_UC *hostname,
                                     SAP_UC *service,
                                     SAP_RAW *buffer,
                                     int buflen,
                                     int *sentlen);

/*--------------------------------------------------------*/
/* NiDgHdlSend                                            */
/*                                                        */
/* Send datagram to a connected peer, using a DgHdl       */
/* you must call NiDgHdlConnect before using this call    */
/*                                                        */
/* INPUT   hdl           Handle (created)                 */
/*         buffer        output buffer                    */
/*         buflen        length of buffer                 */
/* OUTPUT                                                 */
/*         sentlen       contains the number of bytes     */
/*                       actualy  sent                    */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               ok                               */
/* NIEINVAL              handle invalid (eg unconnected)  */
/* NIEINTERN             socket error                     */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiDgHdlSend      (NI_HDL hdl,
                                   SAP_RAW *buffer,
                                   int buflen,
                                   int *sentlen);

/*--------------------------------------------------------*/
/* NiDgHdlConnect                                         */
/*                                                        */
/* Connect a datagram handle to a remote peer             */
/*                                                        */
/* INPUT   hdl           Datagram Handle                  */
/*         host          Address of remote peer           */
/*         service       Remote service                   */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINVAL              hdl or host or service invalid   */
/* NIEINTERN             socket error                     */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiDgHdlConnect     (NI_HDL      hdl,
                                     NI_HOSTADR  host,
                                     SAP_USHORT  port);

/*--------------------------------------------------------*/
/* NiDgHdlConnectName                                     */
/*                                                        */
/* Connect a datagram handle to a remote peer             */
/*                                                        */
/* INPUT   hdl           Datagram Handle                  */
/*         host          Name of remote peer              */
/*         service       Name of remote service           */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               alles klar                       */
/* NIEINVAL              hdl or host or service invalid   */
/* NIEINTERN             socket error                     */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiDgHdlConnectName (NI_HDL      hdl,
                                     SAP_UC     *host,
                                     SAP_UC     *port);
/*--------------------------------------------------------*/
/* NiDgHdlClose                                           */
/*                                                        */
/* Close a DgHdl                                          */
/*                                                        */
/* INPUT   hdl           handle                           */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               ok                               */
/* NIEINVAL              hdl invalid                      */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiDgHdlClose       (NI_HDL hdl);

/*--------------------------------------------------------*/
/* NiDgHdlJoinGroup/NiDgHdlLeaveGroup                     */
/*                                                        */
/* Join or leave a multicast group                        */
/*                                                        */
/* INPUT   hdl           datagram handle                  */
/*         addr          address of multicast group       */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               success                          */
/* NIEINVAL              hdl or group address invalid     */
/* NIEINTERN             socket error                     */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiDgHdlJoinGroup   (NI_HDL hdl,
                                     NI_HOSTADR addr);
SAPRETURN API_DF NiDgHdlLeaveGroup  (NI_HDL hdl,
                                     NI_HOSTADR addr);

/*--------------------------------------------------------*/
/* NiIsMulticastAddress                                   */
/*                                                        */
/* Test whether a given hostaddress is a Multicast address*/
/*                                                        */
/* INPUT   addr          address to test                  */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* TRUE                  addr is a multicast address      */
/* FALSE                 addr is not a multicast address  */
/*                                                        */
/*--------------------------------------------------------*/

SAP_BOOL  API_DF NiIsMulticastAddress(NI_HOSTADR addr);


/*--------------------------------------------------------*/
/* Client and server handle for wakeup mechanism.         */
/*--------------------------------------------------------*/

struct NIWU_SERVER;
struct NIWU_CLIENT;

typedef struct NIWU_SERVER * NIWU_S_HDL;
typedef struct NIWU_CLIENT * NIWU_C_HDL;

/*--------------------------------------------------------*/
/* NiWakeupOpen ()                                        */
/*                                                        */
/* Create wakeup server. The handle sould be finally      */
/* closed by the NiWakeupClose function to release the    */
/* recources.                                             */
/*                                                        */
/* INPUT   handle        must be initialized by NULL      */
/*                                                        */
/* OUTPUT  handle        server handle                    */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               success                          */
/* NIENOFREEPORT         no free port within range        */
/* NIEINVAL              parameter invalid                */
/* NIEINTERN             internal error (out of virt. mem)*/
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiWakeupOpen    (NIWU_S_HDL * handle);

/*--------------------------------------------------------*/
/* NiWakeupClose ()                                       */
/*                                                        */
/* Close wakeup server                                    */
/*                                                        */
/* INPUT   handle        server handle                    */
/*                                                        */
/* OUTPUT  handle        NULL handle                      */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               success                          */
/* NIEINVAL              parameter invalid                */
/* NIEINTERN             internal error (NiCloseHandle)   */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiWakeupClose   (NIWU_S_HDL * handle);

/*--------------------------------------------------------*/
/* NiWakeupGetNiHdl ()                                    */
/*                                                        */
/* Get ni listen handle                                   */
/*                                                        */
/* INPUT   handle        server handle                    */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* NiHandle              success                          */
/* NI_INVALID_HDL        parameter invalid                */
/*                                                        */
/*--------------------------------------------------------*/

NI_HDL    API_DF NiWakeupGetNiHdl(NIWU_S_HDL   handle);

/*--------------------------------------------------------*/
/* NiWakeupGetCntNo ()                                    */
/*                                                        */
/* Get number of attached clients                         */
/*                                                        */
/* INPUT   handle        server handle                    */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* referenc count        success                          */
/* -1                    parameter invalid                */
/*                                                        */
/*--------------------------------------------------------*/

SAP_INT   API_DF NiWakeupGetCntNo(NIWU_S_HDL   handle);

/*--------------------------------------------------------*/
/* NiWakeupCollect ()                                     */
/*                                                        */
/* Receive wakeup request to clean server handle buffer   */
/*                                                        */
/* INPUT   handle        server handle                    */
/*                                                        */
/* OUTPUT  wuReqNo       number of received requests      */
/*                       (or NULL)                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               success                          */
/* NIETOO_BIG            received datagram isn't a wakeup */
/*                       request                          */
/* NIETIMEOUT            no datagram received             */
/* NIEINVAL              parameter invalid                */
/* NIEINTERN             internal error                   */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiWakeupCollect (NIWU_S_HDL   handle,
                                  SAP_INT    * wuReqNo);

/*--------------------------------------------------------*/
/* NiWakeupAttach ()                                      */
/*                                                        */
/* Attach client handle to server. The handle sould be    */
/* finally detached by the NiWakeupDetach function to     */
/* release the resources.                                 */
/*                                                        */
/* INPUT   serverHandle  server handle                    */
/* INPUT   clientHandle  must be initialized by NULL      */
/*                                                        */
/* OUTPUT  handle        client handle                    */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               success                          */
/* NIECONN_REFUSED       build up connection failed       */
/* NIETIMEOUT            timeout occurred                 */
/* NIEINVAL              parameter invalid                */
/* NIEINTERN             internal error (out of virt. mem)*/
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiWakeupAttach  (NIWU_S_HDL   serverHandle,
                                  NIWU_C_HDL * clientHandle);

/*--------------------------------------------------------*/
/* NiWakeupDetach ()                                      */
/*                                                        */
/* Detach client handle from server and release resources */
/*                                                        */
/* INPUT   handle        client handle                    */
/*                                                        */
/* OUTPUT  handle        NULL                             */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               success                          */
/* NIEINVAL              parameter invalid                */
/* NIEINTERN             internal error (NiCloseHandle)   */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiWakeupDetach  (NIWU_C_HDL * handle);

/*--------------------------------------------------------*/
/* NiWakeupExec ()                                        */
/*                                                        */
/* Send wakeup request to attached server                 */
/*                                                        */
/* INPUT   handle        client handle                    */
/*                                                        */
/* RETURN                                                 */
/*                                                        */
/* SAP_O_K               success                          */
/* NIECONN_BROKEN        server closed                    */
/* NIECONN_REFUSED       build up connection failed       */
/* NIEINVAL              hdl invalid                      */
/* NIEINTERN             internal error                   */
/*                                                        */
/*--------------------------------------------------------*/

SAPRETURN API_DF NiWakeupExec    (NIWU_C_HDL   handle);


/*--------------------------------------------------------------------*/
/* NiGet<level>_<option> ()                                           */
/*                                                                    */
/* Get socket option                                                  */
/*                                                                    */
/* INIT    NI should be already initialized by previous calls         */
/*                                                                    */
/* INPUT   handle        handle                                       */
/*         level         protocol level                               */
/*                                                                    */
/* OUTPUT  value         option value (see below)                     */
/*                                                                    */
/* RETURN                                                             */
/*                                                                    */
/* SAP_O_K               success                                      */
/* NIEINVAL              handle invalid or value = NULL               */
/* NIEINTERN             internal error (system call)                 */
/*                                                                    */
/*--------------------------------------------------------------------*/
/*                                                                    */
/* SOCKET LEVEL OPTIONS:                                              */
/*   ReuseAddr:                                                       */
/*     Allow the socket to be bound to an address that is already in  */
/*     use                                                            */
/*         value         allow reuse (TRUE / FALSE)                   */
/*                                                                    */
/*   KeepAlive: (option only supported by stream sockets)             */
/*     Send keepalives                                                */
/*         value         send keepalive probe (TRUE / FALSE)          */
/*                                                                    */
/*   Linger: (option only supported by stream sockets)                */
/*     Linger on close if unsent data is present.                     */
/*         onOff         option on / off (TRUE / FALSE)               */
/*                       if FALSE: close immediately                  */
/*         timout        expire time (in clock ticks / seconds;       */
/*                       platform dependent)                          */
/*                                                                    */
/*   SndBuf: (Linux will double the buffer size)                      */
/*   RcvBuf:                                                          */
/*     Specifies the total per-socket buffer space reserved for       */
/*     sends / receives                                               */
/*         value         size in bytes                                */
/*                                                                    */
/* IP LEVEL: (options only supported by datagram sockets)             */
/*   MulticastIF:                                                     */
/*     Address of the outgoing multicast-capable interface            */
/*         addr          address                                      */
/*                                                                    */
/*   MulticastTTL:                                                    */
/*     Time-to-live for a multicast packet                            */
/*         hops          number of surviving hops                     */
/*                                                                    */
/* TCP LEVEL: (options only supported by TCP sockets)                 */
/*   NoDelay:                                                         */
/*     Disables the Nagle algorithm for send coalescing               */
/*         value         disable algorithm (TRUE / FALSE)             */
/*                                                                    */
/*                                                                    */
/* SOCKET DEFAULT VALUES:                                             */
/*   SO_REUSEADDR        1 (TRUE)                                     */
/*   SO_KEEPALIVE        0 (FALSE)                                    */
/*   SO_LINGER           { 0 (FALSE), 0 }                             */
/*   SO_SNDBUF           depending on platform and type               */
/*                       e.g. Linux TCP 16384                         */
/*                            Linux UDP 65535                         */
/*                            AIX   UDP  9216                         */
/*   SO_RCVBUF           0                                            */
/*   IP_MULTICAST_IF     0.0.0.0                                      */
/*   IP_MULTICAST_TTL    1                                            */
/*   TCP_NODELAY         1 (TRUE)                                     */
/*                                                                    */
/*--------------------------------------------------------------------*/

/* SO */
SAPRETURN API_DF NiGetSO_ReuseAddr   (NI_HDL       handle,
                                      SAP_BOOL   * value);
SAPRETURN API_DF NiGetSO_KeepAlive   (NI_HDL       handle,
                                      SAP_BOOL   * value);
SAPRETURN API_DF NiGetSO_Linger      (NI_HDL       handle,
                                      SAP_BOOL   * onOff,
                                      SAP_INT    * timeout);
SAPRETURN API_DF NiGetSO_SndBuf      (NI_HDL       handle,
                                      SAP_INT    * value);
SAPRETURN API_DF NiGetSO_RcvBuf      (NI_HDL       handle,
                                      SAP_INT    * value);
/* IP */
SAPRETURN API_DF NiGetIP_MulticastIF (NI_HDL       handle,
                                      NI_HOSTADR * addr);
SAPRETURN API_DF NiGetIP_MulticastTTL(NI_HDL       handle,
                                      SAP_INT    * hops);
/* TCP */
SAPRETURN API_DF NiGetTCP_NoDelay    (NI_HDL       handle,
                                      SAP_BOOL   * value);
SAPRETURN API_DF NiGetTCP2_NoDelay   (NI_HDL       handle,
                                      SAP_INT      level,
                                      SAP_BOOL   * value);

/*--------------------------------------------------------------------*/
/* NiSet<level>_<option> ()                                           */
/*                                                                    */
/* Set socket option                                                  */
/*                                                                    */
/* INIT    NI should be already initialized by previous calls         */
/*                                                                    */
/* INPUT   handle        handle                                       */
/*         level         protocol level                               */
/*         value         option value (cf. get function)              */
/*                                                                    */
/* RETURN                                                             */
/*                                                                    */
/* SAP_O_K               success                                      */
/* NIEINVAL              hdl invalid                                  */
/* NIEINTERN             internal error (system call)                 */
/*                                                                    */
/*--------------------------------------------------------------------*/

/* SO */
SAPRETURN API_DF NiSetSO_ReuseAddr   (NI_HDL       handle,
                                      SAP_BOOL     value);
SAPRETURN API_DF NiSetSO_KeepAlive   (NI_HDL       handle,
                                      SAP_BOOL     value);
SAPRETURN API_DF NiSetSO_Linger      (NI_HDL       handle,
                                      SAP_BOOL     onOff,
                                      SAP_INT      timeout);
SAPRETURN API_DF NiSetSO_SndBuf      (NI_HDL       handle,
                                      SAP_INT      value);
SAPRETURN API_DF NiSetSO_RcvBuf      (NI_HDL       handle,
                                      SAP_INT      value);
/* IP */
SAPRETURN API_DF NiSetIP_MulticastIF (NI_HDL       handle,
                                      NI_HOSTADR   addr);
SAPRETURN API_DF NiSetIP_MulticastTTL(NI_HDL       handle,
                                      SAP_INT      hops);
/* TCP */
SAPRETURN API_DF NiSetTCP_NoDelay    (NI_HDL       handle,
                                      SAP_BOOL     value);
SAPRETURN API_DF NiSetTCP2_NoDelay   (NI_HDL       handle,
                                      SAP_INT      level,
                                      SAP_BOOL     value);


/*--------------------------------------------------------------------*/

#ifdef __cplusplus
}
#endif

#endif /* NIXX_H */
