/*!
  -----------------------------------------------------------------------------
  module: vls20_dataload.cpp
  -----------------------------------------------------------------------------

  responsible:  d033893 and SteffenS

  special area: SAP DB LOADER

  description:  Implementation of functions for dataload

  version:      7.5.
  -----------------------------------------------------------------------------

    ========== licence begin  GPL
    Copyright (c) 1999-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end

*/

/*
  -----------------------------------------------------------------------------
  includes
  -----------------------------------------------------------------------------
*/

#include <time.h>

#include "gls00.h"
#include "hls04.h"
#include "hls041.h"
#include "hls05.h"
#include "hls11.h"
#include "hls15.h"
#include "hls20_dataload.h"
#include "hls20.h"
#include "hls21.h"
#include "hls30.h"
#include "hls99.h"
#include "hls98msg.h"

#include "hsp77.h"


/*
  -----------------------------------------------------------------------------
  BeginChapter: Local functions declaration
  -----------------------------------------------------------------------------
*/

/*!
  -----------------------------------------------------------------------------

  function:     ls20_InitDatLoadRec

  -----------------------------------------------------------------------------

  description:  Creates and initializes DATA load record structure

  arguments:    pDatLoadRec [IN] - structure with information on processing the
                                   dataload command
                DBInfo      [IN] - structure with information on the running
                                   session and the database connected to.
  returnvalue:  No

  -----------------------------------------------------------------------------
*/

static tsp00_Int4
ls20_InitDatLoadRec( tls00_DBInfo            *pDBInfo,
                     tls00_TransformationCmd *pTransformationCmd,
                     tls00_DloadRec          *&pDatLoadRec,
                     tsp00_Addr              ErrText  );

/*
  -----------------------------------------------------------------------------
  function:     ls20_FreeDatLoadRec
  -----------------------------------------------------------------------------

  description:

  arguments:    pTransformationCmd [IN] -  structure with user information for
                                    processing dataload command.
                pDatLoadRec  [IN] - structure with information on processing the
                                    dataload command
  returnvalue:  No

  -----------------------------------------------------------------------------
*/
static void
ls20_FreeDatLoadRec(tls00_TransformationCmd *pTransformationCmd,
                    tls00_DloadRec          *&pDatLoadRec,
                    tsp00_Addr              ErrText);

/*
  -----------------------------------------------------------------------------

  function: ls20_InitStreams

  -----------------------------------------------------------------------------

  description:

  arguments:

  returnvalue:

  -----------------------------------------------------------------------------
*/
static tsp00_Int4
ls20_InitStreams ( tls00_DBInfo            *pDBInfo,
                   tls00_TransformationCmd *pTransformationCmd,
                   tls00_DloadRec          *&pDatLoadRec,
                   tsp00_Addr              ErrText      );

/*
  -----------------------------------------------------------------------------

  function:     ls20_SwitchStreams - local function

  -----------------------------------------------------------------------------

  description:


  arguments:    pDBInfo      [IN]   - structure with all information on the running
                                      session and the db connected to.
                pTransformationCmd  [IN]   - structure with information for dataload supplied by the user
                pDatLoadRec  [IN]   - dataload record structure that must be initialized.

  returnvalue:

  -----------------------------------------------------------------------------
*/
static tsp00_Int4
ls20_SwitchStreams( TransformationDef  *&pTransformationDef,
                    tls00_DloadRec     *pDatLoadRec,
                    tsp00_Addr         ErrText);
/*
  -----------------------------------------------------------------------------
  function: ls20_FreeStreams
  -----------------------------------------------------------------------------

  description:

  arguments:

  returnvalue:
  -----------------------------------------------------------------------------
*/
static void
ls20_FreeStreams(tls00_TransformationCmd *pTransformationCmd,
                 tls00_DloadRec          *pDatLoadRec,
                 tsp00_Addr               ErrText);

/*
  -----------------------------------------------------------------------------
  Local functions: implementation
  -----------------------------------------------------------------------------
*/

/*
  -----------------------------------------------------------------------------
  function:     ls20_InitDatLoadRec
  -----------------------------------------------------------------------------
*/
static tsp00_Int4
ls20_InitDatLoadRec(tls00_DBInfo             *pDBInfo,
                    tls00_TransformationCmd  *pTransformationCmd,
                    tls00_DloadRec           *&pDatLoadRec,
                    tsp00_Addr               ErrText )
{
    ROUTINE_DBG_MEO00 ("ls20_InitDatLoadRec");

    tsp00_Int4 rc = errOK_els00;

    pDatLoadRec = new tls00_DloadRec;
    if (NULL == pDatLoadRec)
    {
        rc = errNoMoreMemory_els98;
    }
    else
        memset(pDatLoadRec, 0, sizeof(tls00_DloadRec));

    pDatLoadRec->dlrLRejected_ls00 = 0;
    pDatLoadRec->dlrSwapKind_ls00  = pDBInfo->dbiOwnSwap_ls00;
    pDatLoadRec->dlrPartSpec_ls00  = pTransformationCmd->trPartSpec_ls00;

    if ( true == pTransformationCmd->trRestart_ls00.rstRestart_ls00 )
    {
         pDatLoadRec->dlrRestart_ls00.rstRestart_ls00 = true;
         pDatLoadRec->dlrTaskSwitched_ls00            = false;
    }
    else
         pDatLoadRec->dlrTaskSwitched_ls00 = false;

    if ( rc == errOK_els00 )
         rc = ls20_InitStreams(pDBInfo, pTransformationCmd, pDatLoadRec, ErrText);

    return rc;
}// ls20_InitDatLoadRec


/*
  -----------------------------------------------------------------------------
  function:     ls20_FreeDatLoadRec
  -----------------------------------------------------------------------------
*/
static void
ls20_FreeDatLoadRec(tls00_TransformationCmd   *pTransformationCmd,
                    tls00_DloadRec    *&pDatLoadRec,
                    tsp00_Addr          pszErrText)
{
    ROUTINE_DBG_MEO00 ("ls20_FreeDatLoadRec");

    if ( pDatLoadRec != NULL )
    {
        ls20_FreeStreams(pTransformationCmd, pDatLoadRec, pszErrText);

        delete pDatLoadRec ;
        pDatLoadRec = NULL;
    }   // end if ( pDatLoadRec != NULL )
}
//  ls20_FreeDatLoadRec()

/*
  -----------------------------------------------------------------------------
  BeginChapter:  Global functions implementation
  -----------------------------------------------------------------------------
*/

/*
  -----------------------------------------------------------------------------
  function:     ls20ExecColumnLoad
  -----------------------------------------------------------------------------
*/
tsp00_Int4
ls20ExecColumnLoad( MetaDataDef *pMetaDataDef, tls00_DBInfo *pDBInfo, tsp00_Addr ErrText)
{
    ROUTINE_DBG_MEO00 ("ls20ExecColumnLoad");

    LoaderMessages *pMsgInstance = LoaderMessages::Instance();
    tsp00_Longint    rc      = errOK_els00;

    TransformationDef       *pTransformationDef = pMetaDataDef->getTransformationModel();
    tls00_TransformationCmd *pTransformationCmd = pTransformationDef->getTransformationCmd();

    //*
    //*  check external encoding and database encoding.
    //*  not supported is loading UNICODE DATA into ASCII databases.
    //*
    if (pTransformationCmd->trPartSpec_ls00.psPartType_ls00 > ptTable_els00)
    {
        if (ctUnknown_els00 != pTransformationCmd->trDataFile_ls00.fsFormat_ls00.ffoCodeType_ls00)
        {
            rc = ls05AreEncodingsCompatible(pTransformationCmd->trDataFile_ls00.fsFormat_ls00.ffoCodeType_ls00,
                                            ls00IsDBUnicode(pDBInfo) ? ctUCS2_els00 : ctAscii_els00);
            if (errOK_els00 != rc)
                pMsgInstance->ls98Msg(layDataLoad_els98, ErrText, errIncompatibleEncodings_els98);
        }
    }

    //*
    //*  check if user is sysdba
    //*
    if ( (errOK_els00 == rc) && (ptAll_els00 == pTransformationCmd->trPartSpec_ls00.psPartType_ls00) )
    {
        rc = ls15CheckUserMode(pDBInfo,
                               pDBInfo->dbiSession_ls00.siUserName_ls00.asCharp(),
                               dbuSysdba_els00,
                               ErrText);
        if (dbRowNotFound_els00 == rc)
        {
             pMsgInstance->ls98Msg(layTableLoad_els98, ErrText, errRestrictedStmt_els98);
        }
    }

    //*
    //*  all|user|table records, formatted, compressed
    //*
    if ( (errOK_els00 == rc) && (NULL != pTransformationCmd->trDataFile_ls00.fsFileName_ls00) )
    {
        if ( true  == pTransformationCmd->trUsage_ls00.uFastMode_ls00 )
        {
            //*
            //* record in pages
            //*
			
			//*
            //*  execute transformationpackage
            //*
            if (ptTable_els00 <= pTransformationCmd->trPartSpec_ls00.psPartType_ls00 )
            {
                pMsgInstance->ls98PrintMessage(layDataExtract_els98, msgCommandExecuting_els98,
                                                                     "EXECUTE PACKAGE TO TRANSFORM DATA");
                pMsgInstance->ls98PrintMessage(layDataExtract_els98, msgNumTabToTransform_els98,1);
            }
            rc = ls21ExecFastload (pMetaDataDef, pDBInfo, ErrText);
        }
        else
        {
            //*
            //* record in rows
            //*
            StudioOAL_String  transformationpackage;
			tls00_DloadRec   *pDatLoadRec  = NULL;

            rc = ls20_InitDatLoadRec(pDBInfo, pTransformationCmd, pDatLoadRec, ErrText);
            if ( errOK_els00 == rc )
            {
                //*
                //*  configure transformationpackage
                //*
				rc = pTransformationDef->LoadTransformationPackage(pMetaDataDef, ErrText);
                if (errOK_els00 == rc)
                {
                    //*
                    //*  execute transformationpackage
                    //*
                    if (ptTable_els00 <= pTransformationCmd->trPartSpec_ls00.psPartType_ls00 )
                    {
                        pMsgInstance->ls98PrintMessage(layDataExtract_els98, msgCommandExecuting_els98,
                                                                             "EXECUTE PACKAGE TO TRANSFORM DATA");
                        pMsgInstance->ls98PrintMessage(layDataExtract_els98, msgNumTabToTransform_els98,
                                                                             pTransformationDef->m_nTableCount);
                    }
                    //*
                    //*  update  transformationpackage
                    //*
                    if (ptAll_els00   == pTransformationCmd->trPartSpec_ls00.psPartType_ls00 ||
                        ptUser_els00  == pTransformationCmd->trPartSpec_ls00.psPartType_ls00  )
                    {
                        rc = ls20LoadPackageTables( pMetaDataDef, pDatLoadRec, ErrText);
                    }
                    else if (ptTable_els00 == pTransformationCmd->trPartSpec_ls00.psPartType_ls00)
                    {
                        rc = ls20LoadTableColumns ( pMetaDataDef, pDatLoadRec, ErrText );
                    }
                    //*
                    //*  unload  transformationpackage
                    /*
                    if ( errOK_els00 == rc || -25362 == rc )
                        rc = pTransformationDef->UnloadTransformationPackage( pMetaDataDef, ErrText);*/
                }
            }   // end if ( errOK_els00 == (rc = ls20_InitDatLoadRec(pDBInfo, pTransformationCmd, pDatLoadRec, ErrText))

            //*
            //*  unload  transformationpackage
            //*
            if ( errOK_els00 == rc || msgEmptyDataFile_els98 == rc )
            {
                tsp00_Int4 rcMETA = pTransformationDef->UnloadTransformationPackage( pMetaDataDef, ErrText);
                if ( (errOK_els00 == rc) && (errOK_els00 != rcMETA) ) {
                    rc = rcMETA;
                }
            }

            if ( NULL != pDatLoadRec )
            {
                ls20_FreeDatLoadRec(pTransformationCmd, pDatLoadRec, ErrText);
                pDatLoadRec = NULL ;
            }
        }// if ( false == pTransformationCmd->trUsage_ls00.uFastMode_ls00 )
    }// if ( NULL != pTransformationCmd->trDataFile_ls00.fsFileName_ls00 )

    return STATIC_CAST(tsp00_Int4, rc);
}
//  ls20ExecColumnLoad()

/*
  -----------------------------------------------------------------------------
  function:     ls20LoadPackageTables
  -----------------------------------------------------------------------------
*/
tsp00_Int4
ls20LoadPackageTables(MetaDataDef      *&pMetaDataDef,
                      tls00_DloadRec   *pDatLoadRec,
                      tsp00_Addr        ErrText)
{
    ROUTINE_DBG_MEO00 ("ls20LoadPackageTables");

    LoaderMessages *pMsgInstance = LoaderMessages::Instance();
    tsp00_Int4      rc           = errOK_els00;

    TransformationDef       *pTransformationDef = pMetaDataDef->getTransformationModel();
    tls00_TransformationCmd *pTransformationCmd = pTransformationDef->getTransformationCmd();
    tls00_DBInfo            *pDBInfo            = pMetaDataDef->getConnection();

    StudioOAL_String  select;
    _TCHAR            szselect[2048];
    szselect[0] = 0;

    sp77sprintf(szselect, 2048, SEL_MAP_TABLE_ALL_76_LS00, pTransformationCmd->trPackageid_ls00.StrPtr());
    select.Assign(REINTERPRET_CAST(Tools_DynamicUTF8String::ConstPointer, szselect));

    if ( !pTransformationDef->executeSQL( select ) )
    {
        StudioOAL_String NativeErrorAsString = pTransformationDef->getNativeErrorAsString();
        pMsgInstance->ls98PrintMessage(layDataLoad_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());
        rc = pTransformationDef->getReturnCode();
    }

    StudioOAL_String sessionUser;
    StudioOAL_String currentUser;

    if (errOK_els00 == rc)
    {
        sessionUser.Assign(REINTERPRET_CAST(Tools_DynamicUTF8String::ConstPointer, pDBInfo->dbiSession_ls00.siUserName_ls00.asCharp()));
        currentUser = sessionUser;
    }

    StudioOAL_String record[17];

    /*-------------------------------------------------------------------------------*
     * Read/Write the CATALOG definition of the specified object from/to resources   *
     *-------------------------------------------------------------------------------*/
    while ( errOK_els00 == rc &&  pTransformationDef->initTransformationStep( record )  )
    {
        rc = pTransformationDef->initTransformationCmd();

        if (errOK_els00 == rc)
        {
            pDatLoadRec->dlrTableSwitched_ls00 = true;

            rc = pTransformationDef->switchUser( currentUser, pTransformationDef->m_owner, ErrText);

            if (errOK_els00 == rc)
            {
                rc = ls20_SwitchStreams( pTransformationDef, pDatLoadRec,ErrText);

                if (errOK_els00 == rc)
                {
                    rc = ls20LoadTableColumns(pMetaDataDef, pDatLoadRec, ErrText);
                    if (errOK_els00 == rc)
                    {
                        ++pDatLoadRec->dlrNumTabLoaded_ls00;
                        ErrText[0] = 0;
                    }
                    else
                    {
                        if ( (errInternal_els98 != rc) && (errCancelRequest_els98 != rc) && (errTableToLoadNotCurrentTable_els98 != rc) )
                        {
                            if ( EOF_LS00 == rc )
                            {
                                pMsgInstance->ls98Msg(layDataLoad_els98, ErrText, msgEmptyDataFile_els98, pDatLoadRec->dlrInfile_ls00->vfFileName_ls00);
                            }
                            // Here the 'real' loading MUST be separated from its preparation to adequately react on return codes; furthermore
                            // - we must log failures on single tables and summarize those in a single message that must be returned to the client.
                            // - the loader should jump to the next table DATA in infile if the max error count is not yet reached.

                            // Rejected lines are counted while inserting DATA in vls20.cpp
                            if (pDatLoadRec->dlrLRejected_ls00 < pDBInfo->dbiSession_ls00.siMaxAdmissibleErrors)
                            {
                                rc = errOK_els00;
                                ErrText[0] = 0;
                            }
                            else
                            {
                                pMsgInstance->ls98Msg(layDataLoad_els98, ErrText, errMaxErrorCountReached_els98);
                                rc = errMaxErrorCountReached_els98;
                            }
                        }
                    }
                }// ls20_SwitchStreams

            }// pTransformationDef->switchUser

            currentUser = pTransformationDef->m_owner;
        }
        pTransformationDef->freeTransformationCmd ( srFctColLoad_els00 );

    }// while pTransformationDef->initTransformationStep

    if (errOK_els00 == rc)
    {
        pMsgInstance->ls98PrintMessage(layDataLoad_els98, msgNumTabExcludedExtr_els98,
                                                          pDatLoadRec->dlrNumTabExcludedExtr_ls00);

        pMsgInstance->ls98PrintMessage(layDataLoad_els98, msgNumTabTransformed_els98,
                                                          pDatLoadRec->dlrNumTabLoaded_ls00 -
                                                          pDatLoadRec->dlrNumTabExcludedLoad_ls00 -
                                                          pDatLoadRec->dlrNumTabExcludedExtr_ls00);
    }

    //*
    //* switch user back to session user
    //*
    tsp00_Int4 rc2 = pTransformationDef->switchUser( currentUser, sessionUser, ErrText);

    if ( errOK_els00 == rc && errOK_els00 != rc2 )
        rc = rc2;

    return rc;
}
// ls20LoadPackageTables()


/*
  -----------------------------------------------------------------------------
  function:     ls20UpdateTransformationPackage
  -----------------------------------------------------------------------------
*/
tsp00_Int4
ls20UpdateTransformationPackage(tls00_DBInfo*            pDBInfo,
                                MetaDataDef      *&      pMetaDataDef,
                                tls00_TransformationCmd* pTransformationCmd,
                                tls00_DloadRec*          pDatLoadRec,
                                tsp00_Addr               ErrText)
{
    ROUTINE_DBG_MEO00 ("ls20UpdateTransformationPackage");

    LoaderMessages *pMsgInstance = LoaderMessages::Instance();

    tsp00_Int4 rc = errOK_els00;

    // dataload    table records|compressed|formatted
    // dataextract table records|compressed|formatted

    if (pDatLoadRec->dlrPartSpec_ls00.psPartType_ls00 >= ptTable_els00 && NULL == getenv ("maxdbloaderserver"))
    {
         TransformationDef* pTransformationDef = pMetaDataDef->getTransformationModel();

         SAPDB_Int4       ErrorPos   = 0;
         SQLRETURN        ReturnCode = 0;
         StudioOAL_String NativeErrorAsString;

         //*
         //* Performance
         //*
         extern tls00_Performance performance;
         clock_t        start, finish;

         //*
         //*  update transformation table columns:
         //*  taskid, storagedevice, servernode, serverdb, username, codetype, codepage, transformation, part, streamformat
         //*
         if (true == pDatLoadRec->dlrTaskSwitched_ls00)
         {
              pDatLoadRec->dlrTaskSwitched_ls00 =  false;

              StudioTD_String  servernode;
              StudioTD_String  serverdb;
              StudioTD_String  username;

              StudioOAL_String parameter[13];

              SAPDB_Char   destBeg[1000] ;
              SAPDB_UInt4  destAt  = 1000;
              // stepid
              if ( pTransformationCmd->trDataFile_ls00.fsDeviceType_ls00 == dtypePipe_ls00 ||
				   pTransformationCmd->trPartSpec_ls00.psPartType_ls00   == ptTable_els00 )
                   parameter[0]  = " + ";
              else
                   parameter[0]  = " - ";
              // servernode
              parameter[1]  = "'";
              memset(destBeg,0,sizeof(destBeg));
              pMetaDataDef->getConnection()->dbiOdbcConnection.GetServerNode( REINTERPRET_CAST(unsigned char*, destBeg ), destAt);
              servernode.ConvertFromASCII_Latin1( destBeg, destBeg+strlen(destBeg));
              parameter[1] += servernode;
              parameter[1] += "'";
              // serverdb
              parameter[2]  = "'";
              memset(destBeg,0,sizeof(destBeg));
              pMetaDataDef->getConnection()->dbiOdbcConnection.GetServerDb ( REINTERPRET_CAST(unsigned char*,destBeg ),destAt);
              serverdb.ConvertFromASCII_Latin1( destBeg, destBeg+strlen(destBeg));
              parameter[2] += serverdb;
              parameter[2] += "'";
              // username
              parameter[3]  = "'";
              memset(destBeg,0,sizeof(destBeg));
              pMetaDataDef->getConnection()->dbiOdbcConnection.GetUser( REINTERPRET_CAST(unsigned char*,destBeg ),destAt);
              username.ConvertFromASCII_Latin1( destBeg, destBeg+strlen(destBeg));
              parameter[3] += username;
              parameter[3] += "'";
              // codepage
              if (ctAscii_els00 == pDBInfo->dbiSession_ls00.siCodeType_ls00)
              {
                  parameter[4]  = "'";
                  parameter[4] += pDBInfo->dbiSession_ls00.siCodePage_ls00.name.asCharp();
                  parameter[4] += "'";
              }
              else
              {
                  parameter[4]  = "NULL";
              }
              // storagedevice
              parameter[5]  = pszBackupToolValue_ls00[pDBInfo->dbiSession_ls00.siBackupTool_ls00];
              // transformation
              parameter[6]  = "'LOAD'";
              // part
              if (ptAll_els00 == pTransformationCmd->trPartSpec_ls00.psPartType_ls00)
                  parameter[7]  = "'ALL'";
              if (ptUser_els00 == pTransformationCmd->trPartSpec_ls00.psPartType_ls00)
                  parameter[7]  = "'USER'";
              if (ptUser_els00 == pTransformationCmd->trPartSpec_ls00.psPartType_ls00)
                  parameter[7]  = "'TABLE'";
              // datastreamtype
              parameter[8]  = pszDeviceTypeValue_ls00[pTransformationCmd->trDataFile_ls00.fsDeviceType_ls00];
              // datastreamebid
              if ( pTransformationCmd->trDataFile_ls00.fsDeviceType_ls00 == dtypeTape_ls00 )
                   parameter[9].ConvertFromASCII_Latin1(pTransformationCmd->trDataFile_ls00.fsBackupId_ls00.bBackupIdName_ls00,
                                                        pTransformationCmd->trDataFile_ls00.fsBackupId_ls00.bBackupIdName_ls00 +
                                                _tcslen(pTransformationCmd->trDataFile_ls00.fsBackupId_ls00.bBackupIdName_ls00));
              else
                   parameter[9] = "";
              // datastreamformat
              if  ( true == pTransformationCmd->trDataFile_ls00.fsFormat_ls00.ffoPage_ls00 )
                    parameter[10]   = "'PAGES'";
              if  ( true == pTransformationCmd->trDataFile_ls00.fsFormat_ls00.ffoCompact_ls00 )
                    parameter[10]  =  "'RECORDS'";
              if  ( true == pTransformationCmd->trDataFile_ls00.fsFormat_ls00.ffoFormatted_ls00 )
                    parameter[10]   = "'FORMATTED'";
              if  ( true == pTransformationCmd->trDataFile_ls00.fsFormat_ls00.ffoCompress_ls00 )
                    parameter[10]  =  "'COMPRESSED'";
              // codetype
              parameter[11] = pszCodeTypeValue_ls00[pTransformationCmd->trDataFile_ls00.fsFormat_ls00.ffoCodeType_ls00];
              // transactionsize
              _TCHAR _transactionsize [256];
              _transactionsize        [0] = 0;
              _itot(pDBInfo->dbiSession_ls00.siTASize_ls00,_transactionsize,10);
              parameter[12].ConvertFromASCII_Latin1(_transactionsize,_transactionsize + _tcslen(_transactionsize));

              if ( !pTransformationDef->updateDatTask(parameter))
              {
                  ReturnCode          = pTransformationDef->getReturnCode();
                  NativeErrorAsString = pTransformationDef->getNativeErrorAsString();
                  pMsgInstance->ls98PrintMessage(layTableLoad_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());
                  return ReturnCode;
              }
        }     //if ( pDatLoadRec->dlrTaskSwitched_ls00 == true  )
        else // if ( pDatLoadRec->dlrTaskSwitched_ls00 == false  )
        {
             StudioOAL_String parameter[5];

             _TCHAR RECORD_LOADED   [256];
             _TCHAR HEADER_BLOCK    [256];
             _TCHAR HEADER_POS      [256];
             _TCHAR DATA_BLOCK      [256];
             _TCHAR DATA_POS        [256];
             _TCHAR TRAILER_BLOCK   [256];
             _TCHAR TRAILER_POS     [256];
             RECORD_LOADED   [0] = 0;
             HEADER_BLOCK    [0] = 0;
             HEADER_POS      [0] = 0;
             DATA_BLOCK      [0] = 0;
             DATA_POS        [0] = 0;
             TRAILER_BLOCK   [0] = 0;
             TRAILER_POS     [0] = 0;

             if (0 != pTransformationCmd->trTable_ls00.tsTabOwnr_ls00.length())
             {
                parameter[3].Assign(REINTERPRET_CAST(Tools_DynamicUTF8String::ConstPointer,
                                                     pTransformationCmd->trTable_ls00.tsTabOwnr_ls00.asCharp()));
             }
             else
             {
                 parameter[3].Assign(REINTERPRET_CAST(Tools_DynamicUTF8String::ConstPointer,
                                                      pDBInfo->dbiSession_ls00.siUserName_ls00.asCharp()));
             }

             parameter[4].Assign(REINTERPRET_CAST(Tools_DynamicUTF8String::ConstPointer,
                                                  pTransformationCmd->trTable_ls00.tsTabName_ls00.asCharp()));

             //*
             //*  datastreamheaderpos,datastreamheaderblock,datastreamdatapos
             //*
             if ( pDatLoadRec->dlrTableSwitched_ls00 == true  )
             {
                  pDatLoadRec->dlrTableSwitched_ls00  = false;

                  // <EXCLUDE>

                  // TIMESTAMP BEGIN
                  if ( !pTransformationDef->updateBegin( parameter[3], parameter[4] ) )
                  {
                       ReturnCode          = pTransformationDef->getReturnCode();
                       NativeErrorAsString = pTransformationDef->getNativeErrorAsString();
                       pMsgInstance->ls98PrintMessage(layDataLoad_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());
                       return ReturnCode;
                  }
                  // <DATASTREAMNAME>
                  parameter[0]  = "'";
                  parameter[0] += pDatLoadRec->dlrInfile_ls00->vfName_ls00;
                  parameter[0] += "'";
                  parameter[1]  = parameter[3];
                  parameter[2]  = parameter[4];

                  if ( !pTransformationDef->updateDatStream( parameter ) )
                  {
                       ReturnCode          = pTransformationDef->getReturnCode();
                       NativeErrorAsString = pTransformationDef->getNativeErrorAsString();
                       pMsgInstance->ls98PrintMessage(layTableExtract_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());
                       return ReturnCode;
                  }

                  parameter[0].Erase();
                  parameter[1].Erase();
                  parameter[2].Erase();

                  rc = ls15RSCommit(pDBInfo, ErrText);

                   // ROWCOUNT
                   parameter[0] = "0";
                   // <HEADER_BLOCK>:<HEADER_POS>
                   _itot(STATIC_CAST(tsp00_Int4, pDatLoadRec->dlrInfile_ls00->vfFilePos_ls00.lnu2GB_Blocks_ls00),HEADER_BLOCK,10);
                   _itot(STATIC_CAST(tsp00_Int4, pDatLoadRec->dlrInfile_ls00->vfFilePos_ls00.lnuRemainder_ls00),HEADER_POS,10);

                   parameter[1].ConvertFromASCII_Latin1(HEADER_BLOCK,HEADER_BLOCK + _tcslen(HEADER_BLOCK));
                   parameter[2].ConvertFromASCII_Latin1(HEADER_POS  ,HEADER_POS   + _tcslen(HEADER_POS  ));

                   if ( !pTransformationDef->updateHeader( parameter ) )
                   {
                       ReturnCode          = pTransformationDef->getReturnCode();
                       NativeErrorAsString = pTransformationDef->getNativeErrorAsString();
                       pMsgInstance->ls98PrintMessage(layDataLoad_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());
                       return ReturnCode;
                   }

                   parameter[0].Erase();
                   parameter[1].Erase();
                   parameter[2].Erase();

             }// if ( pDatLoadRec->dlrTableSwitched_ls00 == true  )

             // RECORD_LOADED
             _itot(STATIC_CAST(tsp00_Int4, pDatLoadRec->dlrLastCommLine_ls00),RECORD_LOADED,10);
             // <DATA_BLOCK>:<DATA_POS>
             _itot(STATIC_CAST(tsp00_Int4, pDatLoadRec->dlrInfile_ls00->vfFilePos_ls00.lnu2GB_Blocks_ls00),DATA_BLOCK,10);
             _itot(STATIC_CAST(tsp00_Int4, pDatLoadRec->dlrInfile_ls00->vfFilePos_ls00.lnuRemainder_ls00 ),DATA_POS,10);

             parameter[0].ConvertFromASCII_Latin1(RECORD_LOADED, RECORD_LOADED + _tcslen(RECORD_LOADED));
             parameter[1].ConvertFromASCII_Latin1(DATA_BLOCK   , DATA_BLOCK    + _tcslen(DATA_BLOCK));
             parameter[2].ConvertFromASCII_Latin1(DATA_POS     , DATA_POS      + _tcslen(DATA_POS));

             start = clock();

             // LOAD
             if ( !pTransformationDef->updateLoadData( parameter ) )
             {
                   ReturnCode          = pTransformationDef->getReturnCode();
                   NativeErrorAsString = pTransformationDef->getNativeErrorAsString();
                   pMsgInstance->ls98PrintMessage(layDataLoad_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());
                   return ReturnCode;
             }

             finish = clock();

             performance.perLog_ls00 += (double)(finish - start)/CLOCKS_PER_SEC;

             pDatLoadRec->dlrLastCommLine_ls00  = pDatLoadRec->dlrInsertedRows_ls00;

             if (  pDatLoadRec->dlrTabEmpty_ls00      == true ||
                   pDatLoadRec->dlrTabLoaded_ls00     == true ||
                   pTransformationCmd->trExclude_ls00 == true    )
             {
                   // <EXCLUDE>
                   if ( pTransformationCmd->trExclude_ls00 == true )
                   {
                        if ( !pTransformationDef->updateExclude( parameter[3], parameter[4] ) )
                        {
                           ReturnCode          = pTransformationDef->getReturnCode();
                           NativeErrorAsString = pTransformationDef->getNativeErrorAsString();
                           pMsgInstance->ls98PrintMessage(layTableExtract_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());
                           return ReturnCode;
                        }
                   }

                   parameter[0].Erase();
                   parameter[1].Erase();
                   parameter[2].Erase();

                   // <TRAILER_BLOCK>:<TRAILER_POS>
                   _itot(STATIC_CAST(tsp00_Int4, pTransformationDef->m_trailerpos.lnu2GB_Blocks_ls00),TRAILER_BLOCK,10);
                   _itot(STATIC_CAST(tsp00_Int4, pTransformationDef->m_trailerpos.lnuRemainder_ls00 ),TRAILER_POS  ,10);

                   parameter[0] = "0";
                   parameter[1].ConvertFromASCII_Latin1(TRAILER_BLOCK, TRAILER_BLOCK + _tcslen(TRAILER_BLOCK));
                   parameter[2].ConvertFromASCII_Latin1(TRAILER_POS  , TRAILER_POS   + _tcslen(TRAILER_POS));

                   if ( !pTransformationDef->updateTrailer( parameter ) )
                   {
                         ReturnCode          = pTransformationDef->getReturnCode();
                         NativeErrorAsString = pTransformationDef->getNativeErrorAsString();
                         pMsgInstance->ls98PrintMessage(layTableExtract_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());
                         return ReturnCode;
                   }

                   parameter[0].Erase();
                   parameter[1].Erase();
                   parameter[2].Erase();

                   // RESTART
                   if ( !pTransformationDef->updateRestart( parameter[3], parameter[4]) )
                   {
                       ReturnCode          = pTransformationDef->getReturnCode();
                       NativeErrorAsString = pTransformationDef->getNativeErrorAsString();
                       pMsgInstance->ls98PrintMessage(layDataLoad_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());
                       return ReturnCode;
                   }
                   // TIMESTAMP END
                   if ( !pTransformationDef->updateEnd( parameter[3], parameter[4]) )
                   {
                       ReturnCode          = pTransformationDef->getReturnCode();
                       NativeErrorAsString = pTransformationDef->getNativeErrorAsString();
                       pMsgInstance->ls98PrintMessage(layDataLoad_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());
                       return ReturnCode;
                   }
                   // SYSTEM TIME
                   if ( !pTransformationDef->updatePerformance( parameter[3], parameter[4]) )
                   {
                       ReturnCode          = pTransformationDef->getReturnCode();
                       NativeErrorAsString = pTransformationDef->getNativeErrorAsString();
                       pMsgInstance->ls98PrintMessage(layTableExtract_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());
                       return ReturnCode;
                   }
             }// if ( pDatLoadRec->dlrTabEmpty_ls00   == true )

       }// if ( pDatLoadRec->dlrTaskSwitched_ls00 == false  )
	   if ( !pTransformationDef->update( "COMMIT WORK" ))
       {
            ReturnCode          = pTransformationDef->getReturnCode();
            NativeErrorAsString = pTransformationDef->getNativeErrorAsString();
            pMsgInstance->ls98PrintMessage(layTableExtract_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());
            return ReturnCode;
       }

    }   // end if (pDatLoadRec->dlrPartSpec_ls00.psPartType_ls00 > ptTable_els00)

    return rc;

}//ls20UpdateTransformationPackage

/*
  -----------------------------------------------------------------------------
  function:     ls20_InitStreams
  -----------------------------------------------------------------------------
*/
static tsp00_Int4
ls20_InitStreams(tls00_DBInfo*            pDBInfo,
                 tls00_TransformationCmd* pTransformationCmd,
                 tls00_DloadRec   *&      pDatLoadRec,
                 tsp00_Addr               ErrText)
{
    ROUTINE_DBG_MEO00 ("ls20_InitStreams");

    tsp00_Int4 rc = errOK_els00;
    //*
    //*  initialize DATA stream
    //*
    tls00_FileSpec *pFileSpec  = &pTransformationCmd->trDataFile_ls00;
    tls00_VFile    *pVDataFile = NULL;

    if ( NULL == pDatLoadRec->dlrInfile_ls00 )
    {
        pDatLoadRec->dlrInfile_ls00  = new tls00_VFile;
        memset(pDatLoadRec->dlrInfile_ls00, 0, sizeof(tls00_VFile));
        pDatLoadRec->dlrInfile_ls00->vfFileName_ls00     = new _TCHAR[PATH_MXSP00];
        pDatLoadRec->dlrInfile_ls00->vfFileName_ls00[0]  = 0;
        pDatLoadRec->dlrInfile_ls00->vfFileType_ls00     = pFileSpec->fsDeviceType_ls00;
        pDatLoadRec->dlrInfile_ls00->vfFileEncoding_ls00 = pFileSpec->fsFormat_ls00.ffoCodeType_ls00;
    }

    pVDataFile = pDatLoadRec->dlrInfile_ls00;

    //*
    //* medium
    //*
    if ( dtypeTape_ls00 == pFileSpec->fsDeviceType_ls00 )
    {
        _tcscpy( pVDataFile->vfFileName_ls00, pTransformationCmd->trDataFile_ls00.fsFileName_ls00 );

        pVDataFile->vfFileType_ls00   = dtypeTape_ls00;
        pVDataFile->vfBackupTool_ls00 = pDBInfo->dbiSession_ls00.siBackupTool_ls00;
        pVDataFile->vfNodeName_ls00.Assign(REINTERPRET_CAST(Tools_DynamicUTF8String::ConstPointer,
                                                            pDBInfo->dbiSession_ls00.si_pDBNode_ls00));
        pVDataFile->vfDBName_ls00.Assign(REINTERPRET_CAST(Tools_DynamicUTF8String::ConstPointer,
                                                          pDBInfo->dbiSession_ls00.siDBName_ls00.asCharp()));
        pVDataFile->vfBackupId_ls00.Assign(REINTERPRET_CAST(Tools_DynamicUTF8String::ConstPointer,
                                                            pFileSpec->fsBackupId_ls00.bBackupIdName_ls00));

        pVDataFile->vfName_ls00.Assign(REINTERPRET_CAST(Tools_DynamicUTF8String::ConstPointer, pFileSpec->fsFileName_ls00));
    }
    else
    {
        if ( ptTable_els00 < pTransformationCmd->trPartSpec_ls00.psPartType_ls00 )
        {
            pDatLoadRec->dlrMetaData_ls00.pmdMetaFile_ls00 = pVDataFile;
            pDatLoadRec->dlrRestart_ls00.rstRestart_ls00 = pTransformationCmd->trRestart_ls00.rstRestart_ls00;
        }
        if ( dtypeFile_ls00 == pFileSpec->fsDeviceType_ls00 )
        {
            //*
            //* part
            //*
            if ( ptTable_els00 < pTransformationCmd->trPartSpec_ls00.psPartType_ls00 )
            {
                sp77sprintf(pVDataFile->vfFileName_ls00, PATH_MXSP00, "%s%0*d", pFileSpec->fsFileName_ls00,
                                                                                FILE_EXTENSION_SIZE_LS20, 1);
            }
            else
            {
                _tcscpy(pVDataFile->vfFileName_ls00, pFileSpec->fsFileName_ls00);
            }

            //*
            //* streamname
            //*
            _TCHAR *pszStreamName = strrchr(pVDataFile->vfFileName_ls00, SLASH_LS00);
            if (NULL == pszStreamName)
            {
                pszStreamName = strrchr(pVDataFile->vfFileName_ls00, BACKSLASH_LS00);
                if (NULL == pszStreamName)
                    pszStreamName = pVDataFile->vfFileName_ls00;
            }
            pszStreamName = (pszStreamName == pVDataFile->vfFileName_ls00 ) ? pszStreamName : pszStreamName + 1;
            pVDataFile->vfName_ls00.Assign(REINTERPRET_CAST(Tools_DynamicUTF8String::ConstPointer, pszStreamName));
            //*
            //* streampath
            //*
            pVDataFile->vfPathName_ls00.AssignRaw(REINTERPRET_CAST(unsigned char*, pVDataFile->vfFileName_ls00),
                                                  _tcslen(pVDataFile->vfFileName_ls00)-_tcslen(pszStreamName));
        }
        //*
        //* medium
        //*
        else if ( dtypePipe_ls00 == pFileSpec->fsDeviceType_ls00 )
        {
            _tcscpy(pVDataFile->vfFileName_ls00, pFileSpec->fsFileName_ls00);
            pVDataFile->vfName_ls00.Assign(REINTERPRET_CAST(Tools_DynamicUTF8String::ConstPointer,
                                                            pFileSpec->fsFileName_ls00));
        }

        //*
        //*     Now initialize runtime data file structure and open data file
        //*
        tsp05_RteDataKind_Param FileDataType = sp5vf_binary;
        if ( (ptTable_els00 == pTransformationCmd->trPartSpec_ls00.psPartType_ls00 ) &&
             (false == pFileSpec->fsFormat_ls00.ffoBinary_ls00) && (false == pFileSpec->fsFormat_ls00.ffoCompact_ls00) )
        {
            FileDataType = sp5vf_text;
        }
        ls30VFInitialize(*pVDataFile,
                         pVDataFile->vfFileName_ls00,
                         FileDataType, sp5vf_read,
                         pFileSpec->fsFormat_ls00.ffoCodeType_ls00);
        //*
        //* open stream
        //*
        rc = ls30VFOpen(*pDatLoadRec->dlrInfile_ls00, ErrText);
        if ( (errOK_els00 == rc) && (ctUnknown_els00 == pFileSpec->fsFormat_ls00.ffoCodeType_ls00) )
        {
            pFileSpec->fsFormat_ls00.ffoCodeType_ls00 = pVDataFile->vfFileEncoding_ls00;
        }
    }   // end else of if ( dtypeTape_ls00 == pFileSpec->fsDeviceType_ls00 )

    return rc;

}
// ls20_InitStreams()


/*
  -----------------------------------------------------------------------------
  function:     ls20_SwitchStreams
  -----------------------------------------------------------------------------
*/
static tsp00_Int4
ls20_SwitchStreams(TransformationDef  *&pTransformationDef,
                   tls00_DloadRec     *pDatLoadRec,
                   tsp00_Addr          ErrText)
{
    ROUTINE_DBG_MEO00 ("ls20_SwitchStreams");

    tsp00_Int4 rc = errOK_els00;

    tls00_TransformationCmd *pTransformationCmd = pTransformationDef->getTransformationCmd();
    tls00_DBInfo            *pDBInfo            = pTransformationDef->getMetaData()->getConnection();
    //*
    //* medium
    //*
    if ( dtypeTape_ls00 == pTransformationCmd->trDataFile_ls00.fsDeviceType_ls00 )
    {
         if ( 0   != pTransformationDef->m_datastreamebid.Compare("?") &&
              0   != pTransformationDef->m_datastreamebid.Compare( pDatLoadRec->dlrInfile_ls00->vfBackupId_ls00 ))
        {
            pDatLoadRec->dlrInfile_ls00->vfFileType_ls00   = pTransformationCmd->trDataFile_ls00.fsDeviceType_ls00;
            pDatLoadRec->dlrInfile_ls00->vfBackupTool_ls00 = pDBInfo->dbiSession_ls00.siBackupTool_ls00;
            pDatLoadRec->dlrInfile_ls00->vfNodeName_ls00.Assign(REINTERPRET_CAST(Tools_DynamicUTF8String::ConstPointer,
                                                                pDBInfo->dbiSession_ls00.si_pDBNode_ls00));
            pDatLoadRec->dlrInfile_ls00->vfDBName_ls00.Assign(REINTERPRET_CAST(Tools_DynamicUTF8String::ConstPointer,
                                                              pDBInfo->dbiSession_ls00.siDBName_ls00.asCharp()));
            pDatLoadRec->dlrInfile_ls00->vfBackupId_ls00   = pTransformationDef->m_datastreamebid;
            //*
            //* initialize stream
            //*
            ls30VFInitialize(*pDatLoadRec->dlrInfile_ls00, pTransformationCmd->trDataFile_ls00.fsFileName_ls00 ,sp5vf_binary, sp5vf_read);
            //*
            //* open stream
            //*
            rc = ls30VFOpen(*pDatLoadRec->dlrInfile_ls00, ErrText);
        }
    }
    //*
    //* medium
    //*
    if (dtypeFile_ls00 == pTransformationCmd->trDataFile_ls00.fsDeviceType_ls00 )
    {
        if (false == pTransformationCmd->trInternal_ls00 &&
            UNDEFINED_LS00 != pTransformationCmd->trDataFile_ls00.fsExtract_ls00.feStart_ls00.lnuRemainder_ls00  )
        {
            //*
            //*  seek to right position
            //*
            if (  pTransformationCmd->trDataFile_ls00.fsExtract_ls00.feStart_ls00.lnu2GB_Blocks_ls00 !=  pDatLoadRec->dlrInfile_ls00->vfFilePos_ls00.lnu2GB_Blocks_ls00 ||
                  pTransformationCmd->trDataFile_ls00.fsExtract_ls00.feStart_ls00.lnuRemainder_ls00  !=  pDatLoadRec->dlrInfile_ls00->vfFilePos_ls00.lnuRemainder_ls00     )
            {
                tls00_LargeNumber lFilePos;
                lFilePos.lnu2GB_Blocks_ls00 =   pTransformationCmd->trDataFile_ls00.fsExtract_ls00.feStart_ls00.lnu2GB_Blocks_ls00 ;
                lFilePos.lnuRemainder_ls00  =   pTransformationCmd->trDataFile_ls00.fsExtract_ls00.feStart_ls00.lnuRemainder_ls00  ;
                rc = ls30VFMoveToStartPosition(*pDatLoadRec->dlrInfile_ls00, &lFilePos, ErrText);
            }

        }
        if (true  == pTransformationCmd->trInternal_ls00 )
        {
            if ( 0 != pTransformationDef->m_datastreamname.Compare("?") &&
                 0 != pTransformationDef->m_datastreamname.Compare(pDatLoadRec->dlrInfile_ls00->vfName_ls00 ))
            {
                //*
                //* close old file if open
                //*
                ls30VFClose(*pDatLoadRec->dlrInfile_ls00, ErrText);
                //*
                //* concatenate file name
                //*
                tsp00_Addr pszFileName = NULL;
                tsp00_Addr pPos = NULL;
                pPos = strrchr(pDatLoadRec->dlrInfile_ls00->vfFileName_ls00, BACKSLASH_LS00);
                if (NULL == pPos)
                    pPos = strrchr(pDatLoadRec->dlrInfile_ls00->vfFileName_ls00, SLASH_LS00);
                if (NULL == pPos)
                    pPos = pDatLoadRec->dlrInfile_ls00->vfFileName_ls00;
                else
                    pPos = &pDatLoadRec->dlrInfile_ls00->vfFileName_ls00[pPos - pDatLoadRec->dlrInfile_ls00->vfFileName_ls00 + 1];

                memcpy(pPos,
                       pTransformationDef->m_datastreamname.CharPtr(),
                       _tcslen(pTransformationDef->m_datastreamname.CharPtr()));
                pPos[_tcslen(pTransformationDef->m_datastreamname.CharPtr())] = 0;

                //*
                //* open new file
                //*
                rc = ls30VFOpen(*pDatLoadRec->dlrInfile_ls00, ErrText);

                //*
                //*  read current filename
                //*
                if (errOK_els00 == rc)
                    pDatLoadRec->dlrInfile_ls00->vfName_ls00 = pTransformationDef->m_datastreamname;
            }
            //*
            //*  seek to right position
            //*
            if ( errOK_els00 == rc &&
                 (pTransformationDef->m_headerpos.lnu2GB_Blocks_ls00 !=  pDatLoadRec->dlrInfile_ls00->vfFilePos_ls00.lnu2GB_Blocks_ls00 ||
                  pTransformationDef->m_headerpos.lnuRemainder_ls00  !=  pDatLoadRec->dlrInfile_ls00->vfFilePos_ls00.lnuRemainder_ls00     ))
            {
                tls00_LargeNumber lFilePos;
                lFilePos.lnu2GB_Blocks_ls00 =   pTransformationDef->m_headerpos.lnu2GB_Blocks_ls00 ;
                lFilePos.lnuRemainder_ls00  =   pTransformationDef->m_headerpos.lnuRemainder_ls00  ;
                rc = ls30VFMoveToStartPosition(*pDatLoadRec->dlrInfile_ls00, &lFilePos, ErrText);
            }
            //*
            //*  seek to right restart position
            //*
            if ( (errOK_els00 == rc) && (true == pDatLoadRec->dlrRestart_ls00.rstRestart_ls00) )
            {
                rc = ls30VFSeekLong(*pDatLoadRec->dlrInfile_ls00, &pTransformationDef->m_headerpos, ErrText, sp5vf_seek_begin);
                //*
                //* reset restart flag
                //*
                pDatLoadRec->dlrRestart_ls00.rstRestart_ls00 = false;
            }
        }
    }   // end if (dtypeFile_ls00 == pTransformationCmd->trDataFile_ls00.fsDeviceType_ls00 )
    return rc;
}
//ls20_SwitchStreams()


/*
  -----------------------------------------------------------------------------
  function:     ls20_FreeStreams
  -----------------------------------------------------------------------------
*/
static void
ls20_FreeStreams( tls00_TransformationCmd *pTransformationCmd,
                  tls00_DloadRec   *pDatLoadRec,
                  tsp00_Addr        ErrText)
{
    ROUTINE_DBG_MEO00 ("ls20_FreeStreams");

    tsp00_Int4 rc = errOK_els00;

    //*
    //* close datafile
    //*
    if (NULL != pDatLoadRec->dlrInfile_ls00)
    {
        if (0 != pDatLoadRec->dlrInfile_ls00->vfFileHndl_ls00)
        {
            ls30VFClose(*pDatLoadRec->dlrInfile_ls00, ErrText);
        }
        //*
        //* delete infile structure
        //*
        if (NULL != pDatLoadRec->dlrInfile_ls00)
        {
            delete [] pDatLoadRec->dlrInfile_ls00->vfFileName_ls00;
            pDatLoadRec->dlrInfile_ls00->vfFileName_ls00 = NULL;

            delete pDatLoadRec->dlrInfile_ls00;
            pDatLoadRec->dlrInfile_ls00 = NULL;
        }
    }
    //*
    //* delete meta DATA
    //*
    if (NULL != pDatLoadRec->dlrMetaData_ls00.pmdMetaFile_ls00)
    {
        if (NULL != pDatLoadRec->dlrMetaData_ls00.pmdLengthArray_ls00)
        {
            delete [] pDatLoadRec->dlrMetaData_ls00.pmdLengthArray_ls00;
            pDatLoadRec->dlrMetaData_ls00.pmdLengthArray_ls00 = NULL;
        }
    }
}
//ls20_FreeStreams()


/*===========================================================================*
 *     END
 *===========================================================================*/
