/*!
  -----------------------------------------------------------------------------

  module: vcn28.cpp

  -----------------------------------------------------------------------------

  responsible:  BerndV

  special area: DBMServer Parameter File Migration

  description:  DBMServer Parameter File Migration - Implementation

  version:      7.4.*

  -----------------------------------------------------------------------------

  copyright: (c) 2001-2004 SAP AG


    ========== licence begin  GPL
    Copyright (c) 2001-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end




  -----------------------------------------------------------------------------
*/

/* 
  -------------------------------------------------------------------------
  includes
  -------------------------------------------------------------------------
 */

#include <stdlib.h>

#include "hsp100.h"
#include "hcn28.h"
#include "hcn29.h"
#include "hcn90.h"

#include "KernelCommon/ParameterNames/KernelParam_IOManager.hpp"
#include "KernelCommon/ParameterNames/KernelParam_Converter.hpp"
/*
  -------------------------------------------------------------------------
  function:     cn28_FindParamNew
  purpose:      param search
  -------------------------------------------------------------------------
 */
static tcn002_XParameter * cn28_FindParamNew
      ( tcn002_XParamData * pXParamData,
        const char        * szNewName )
{
  tcn00_Error                  nFuncReturn = OK_CN00;
  tcn002_XParameter          * pParameter  = NULL;
  tcn002_XParameter          * pDynHelp    = NULL;
  bool                         bMallocOK   = true;

  pParameter = cn29SearchParameter(pXParamData->pFirst, szNewName);
  if (pParameter == NULL) {

    /* No match -> check for dynamic */
    pDynHelp = cn29SearchDynamic(pXParamData->pFirst, szNewName);
    if (pDynHelp != NULL) {

      cn00MyMalloc (sizeof (tcn002_XParameter), &pParameter, &bMallocOK);
      if ((pParameter != NULL) && (bMallocOK)) {
        /* copy data from dynamic record */
        nFuncReturn = cn29CopyParameter(pParameter, pDynHelp);
        /* clear the dynamic/intern property */
        pParameter->xpProperties[XPPropClassDynamic] = XPPropValueNo;
        pParameter->xpProperties[XPPropClassIntern] = XPPropValueNo;

       SAPDB_strcpy(pParameter->szParameterName, szNewName);

        /* list management (insert after dynamic) */
        pParameter->pNext = pDynHelp->pNext;
        pDynHelp->pNext   = pParameter;
        pParameter->pPrev = pDynHelp;
        if (pParameter->pNext != NULL) {
          pParameter->pNext->pPrev = pParameter;
        } else {
          pXParamData->pLast = pParameter;
        } // end if

      } // end if

    } // end if
  } // end if

  return pParameter;
} // end cn28_FindParamNew
 
/*
  -------------------------------------------------------------------------
  function:     cn28_CopyParamInteger
  purpose:      param copy
  -------------------------------------------------------------------------
 */
static tcn00_Error cn28_CopyParamInteger
      ( tsp00_DbNamec       szDbName,
        tcn002_XParamData * pXParamData,
        RTEConf_Parameter & oConfigFile,
        const char        * szOldName,
        const char        * szNewName,
        char              * replyData,
        int               * replyLen,
        bool                bIgnoreNotFound)
{
  FUNCTION_DBG_MCN00_1("cn28_CopyParamInteger");

  tcn00_Error                  nFuncReturn = OK_CN00;
  tcn002_XParameter          * pParameter  = NULL;
  RTEConf_Parameter::Integer   nValue      = 0;
  SAPDBErr_MessageList         oMsg;

  // 
  if (nFuncReturn == OK_CN00) {
    if (!oConfigFile.GetValue ((const SAPDB_UTF8 *) szOldName, nValue, oMsg)) {
      if (!bIgnoreNotFound) {
        SAPDBErr_MessageList aDBM(DBMSrv_DBMError(RTE), 0);
        aDBM.AppendNewMessage(oMsg);
        nFuncReturn = cn90AnswerMessage (replyData, replyLen, aDBM);
      } // end if
    } else {
      pParameter = cn28_FindParamNew(pXParamData, szNewName);
      if (pParameter != NULL) {
        pParameter->xpRecord.xpValue.xpValueInt = nValue;
        if (!oConfigFile.SetValue ((const SAPDB_UTF8 *) pParameter->szParameterName.asCharp(), pParameter->xpRecord.xpValue.xpValueInt, oMsg)) {
          SAPDBErr_MessageList aDBM(DBMSrv_DBMError(RTE), 0);
          aDBM.AppendNewMessage(oMsg);
          nFuncReturn = cn90AnswerMessage (replyData, replyLen, aDBM);
        } else {
          cn29ParamValueToString(pParameter->szNewValue, &(pParameter->xpRecord));
          cn29WriteParamHistory(szDbName, pParameter, PAHI_CHANGE_CN002);
        } // end if
      } else {
        nFuncReturn = cn90AnswerIError(replyData, replyLen, ERR_XPNOTFOUND_CN00);
      } // end if
    } // end if
  } // end if

  return nFuncReturn;
} // end cn28_CopyParamInteger

/*
  -------------------------------------------------------------------------
  function:     cn28_CopyParamString
  purpose:      param copy
  -------------------------------------------------------------------------
 */
static tcn00_Error cn28_CopyParamString
      ( tsp00_DbNamec       szDbName,
        tcn002_XParamData * pXParamData,
        RTEConf_Parameter & oConfigFile,
        const char        * szOldName,
        const char        * szNewName,
        char              * replyData,
        int               * replyLen,
        bool                bIgnoreNotFound)
{
  FUNCTION_DBG_MCN00_1("cn28_CopyParamString");

  tcn00_Error                  nFuncReturn = OK_CN00;
  tcn002_XParameter          * pParameter  = NULL;
  SAPDBErr_MessageList         oMsg;
  RTEConf_Parameter::StringElement szValue[RTECONF_MAXSTRINGLENGTH + 1];

  // 
  if (nFuncReturn == OK_CN00) {
    if (!oConfigFile.GetValue ((const SAPDB_UTF8 *) szOldName, szValue, oMsg)) {
      if (!bIgnoreNotFound) {
        SAPDBErr_MessageList aDBM(DBMSrv_DBMError(RTE), 0);
        aDBM.AppendNewMessage(oMsg);
        nFuncReturn = cn90AnswerMessage (replyData, replyLen, aDBM);
      } // end if
    } else {
      pParameter = cn28_FindParamNew(pXParamData, szNewName);
      if (pParameter != NULL) {
        SAPDB_strcpy((char *) pParameter->xpRecord.xpValue.xpValueString, (char *) szValue);
        if (!oConfigFile.SetValue ((const SAPDB_UTF8 *) pParameter->szParameterName.asCharp(), (RTEConf_Parameter::String) pParameter->xpRecord.xpValue.xpValueString.asCharp(), oMsg)) {
          SAPDBErr_MessageList aDBM(DBMSrv_DBMError(RTE), 0);
          aDBM.AppendNewMessage(oMsg);
          nFuncReturn = cn90AnswerMessage (replyData, replyLen, aDBM);
        } else {
          cn29ParamValueToString(pParameter->szNewValue, &(pParameter->xpRecord));
          cn29WriteParamHistory(szDbName, pParameter, PAHI_CHANGE_CN002);
        } // end if
      } else {
        nFuncReturn = cn90AnswerIError(replyData, replyLen, ERR_XPNOTFOUND_CN00);
      } // end if
    } // end if
  } // end if

  return nFuncReturn;
} // end cn28_CopyParamString

/*
  -------------------------------------------------------------------------
  function:     cn20_MigrateFrom_7_2_5_or_7_3_0
  purpose:      CACHE_SIZE handling
  -------------------------------------------------------------------------
 */
static tcn00_Error cn20_MigrateFrom_7_2_5_or_7_3_0 
      ( tsp00_DbNamec       szDbName,
        tcn002_XParamData * pXParamData,
        char              * replyData,
        int               * replyLen)

{
  FUNCTION_DBG_MCN00_1("cn20_MigrateFrom_7_2_5_or_7_3_0");

  tcn00_Error          nFuncReturn = OK_CN00;
  SAPDBErr_MessageList oMsg;
  SAPDB_Bool           bFileExist  = false;
  RTEConf_Parameter    oConfigFile(szDbName.asCharp(), 
                                   0, 
                                   RTECONF_MAXNAMELENGTH,
                                   RTECONF_MAXSTRINGLENGTH);

  // open param file
  if (!oConfigFile.Read(bFileExist, oMsg)) {
    SAPDBErr_MessageList aDBM(DBMSrv_DBMError(RTE), 0);
    aDBM.AppendNewMessage(oMsg);
    nFuncReturn = cn90AnswerMessage (replyData, replyLen, aDBM);
  } else {

    // Parameter to write
    tcn002_XParameter * pParameter = NULL;
    
    // CACHE_SIZE = DATA_CACHE + CONVERTER_CACHE
    RTEConf_Parameter::Integer nDatacache;
    RTEConf_Parameter::Integer nConvertercache;

    // read DATA_CACHE
    if (nFuncReturn == OK_CN00) {
      if (!oConfigFile.GetValue ((const SAPDB_UTF8 *) PAN_DATA_CACHE_PAGS, nDatacache, oMsg)) {
        SAPDBErr_MessageList aDBM(DBMSrv_DBMError(RTE), 0);
        aDBM.AppendNewMessage(oMsg);
        nFuncReturn = cn90AnswerMessage (replyData, replyLen, aDBM);
      } // end if
    } // end if

    // read CONVERTER_CACHE
    if (nFuncReturn == OK_CN00) {
      if (!oConfigFile.GetValue ((const SAPDB_UTF8 *) PAN_CONV_CACHE_PAGES, nConvertercache, oMsg)) {
        SAPDBErr_MessageList aDBM(DBMSrv_DBMError(RTE), 0);
        aDBM.AppendNewMessage(oMsg);
        nFuncReturn = cn90AnswerMessage (replyData, replyLen, aDBM);
      } // end if
    } // end if

    // write CACHE_SIZE
    if (nFuncReturn == OK_CN00) {
      pParameter = cn29SearchParameter(pXParamData->pFirst, PAN_CACHE_SIZE);
      if (pParameter != NULL) {
        pParameter->xpRecord.xpValue.xpValueInt = nDatacache + nConvertercache;
        if (!oConfigFile.SetValue ((const SAPDB_UTF8 *) pParameter->szParameterName.asCharp(), pParameter->xpRecord.xpValue.xpValueInt, oMsg)) {
          SAPDBErr_MessageList aDBM(DBMSrv_DBMError(RTE), 0);
          aDBM.AppendNewMessage(oMsg);
          nFuncReturn = cn90AnswerMessage (replyData, replyLen, aDBM);
        } else {
          cn29ParamValueToString(pParameter->szNewValue, &(pParameter->xpRecord));
          cn29WriteParamHistory(szDbName, pParameter, PAHI_CHANGE_CN002);
        } // end if
      } else {
        nFuncReturn = cn90AnswerIError(replyData, replyLen, ERR_XPNOTFOUND_CN00);
      } // end if
    } // end if

    // ready stamp kernelversion
    if (nFuncReturn == OK_CN00) {
      pParameter = cn29SearchParameter(pXParamData->pFirst, PAN_PARAM_KERNELVERSION);
      if (pParameter != NULL) {
        // get own version
        tsp00_Versionc      VersionString;
        sp100_GetVersionString(COMP_NAME_KERNEL_SP100, s100buildnumber, VersionString);
        
        cn29StringToParamValue(&(pParameter->xpRecord), VersionString.asCharp());

        RTEConf_Parameter::String pString = (RTEConf_Parameter::String) pParameter->xpRecord.xpValue.xpValueString.asCharp();

        if (!oConfigFile.SetValue ((const SAPDB_UTF8 *) pParameter->szParameterName.asCharp(), pString, oMsg)) {
          SAPDBErr_MessageList aDBM(DBMSrv_DBMError(RTE), 0);
          aDBM.AppendNewMessage(oMsg);
          nFuncReturn = cn90AnswerMessage (replyData, replyLen, aDBM);
        } else {
          cn29ParamValueToString(pParameter->szNewValue, &(pParameter->xpRecord));
          cn29WriteParamHistory(szDbName, pParameter, PAHI_CHANGE_CN002);
        } // end if
      } else {
        nFuncReturn = cn90AnswerIError(replyData, replyLen, ERR_XPNOTFOUND_CN00);
      } // end if
    } // end if

    // close param file
    if (nFuncReturn == OK_CN00) {
      if (!oConfigFile.Write(oMsg)) {
        SAPDBErr_MessageList aDBM(DBMSrv_DBMError(RTE), 0);
        aDBM.AppendNewMessage(oMsg);
        nFuncReturn = cn90AnswerMessage (replyData, replyLen, aDBM);
      } // end if
    } // end if

  } // end if
  
  return nFuncReturn;
} // end cn20_MigrateFrom_7_2_5_or_7_3_0
/*
  -------------------------------------------------------------------------
  function:     cn20_MigrateFrom_7_4_3_or_2_or_1
  purpose:      DEVSPACE_TO_VOLUME Handling and several renamings
  -------------------------------------------------------------------------
 */
static tcn00_Error cn20_MigrateFrom_7_4_3_or_2_or_1
      ( tsp00_DbNamec       szDbName,
        tcn002_XParamData * pXParamData,
        char              * replyData,
        int               * replyLen)

{
  FUNCTION_DBG_MCN00_1("cn20_MigrateFrom_7_4_3_or_2_or_1");

  tcn00_Error          nFuncReturn = OK_CN00;
  SAPDBErr_MessageList oMsg;
  SAPDB_Bool           bFileExist  = false;
  RTEConf_Parameter    oConfigFile(szDbName.asCharp(), 
                                   0, 
                                   RTECONF_MAXNAMELENGTH,
                                   RTECONF_MAXSTRINGLENGTH);

  // open param file
  if (!oConfigFile.Read(bFileExist, oMsg)) {
    SAPDBErr_MessageList aDBM(DBMSrv_DBMError(RTE), 0);
    aDBM.AppendNewMessage(oMsg);
    nFuncReturn = cn90AnswerMessage (replyData, replyLen, aDBM);
  } else {

    // Parameter to write
    tcn002_XParameter * pParameter = NULL;

    // handle MAXDATAVOLUMES
    if (nFuncReturn == OK_CN00) {
      nFuncReturn = cn28_CopyParamInteger(szDbName, 
                                          pXParamData, 
                                          oConfigFile,
                                          PAN_MAXDATADEVSPACES_OLD, 
                                          PAN_MAXDATADEVSPACES, 
                                          replyData, 
                                          replyLen,
                                          false);
    } // end if
    // handle FBM_VOLUME_COMPRESSION
    if (nFuncReturn == OK_CN00) {
      nFuncReturn = cn28_CopyParamInteger(szDbName, 
                                          pXParamData, 
                                          oConfigFile,
                                          PAN_FBM_DEVSP_COMPR_OLD, 
                                          PAN_FBM_DEVSP_COMPR, 
                                          replyData, 
                                          replyLen,
                                          false);
    } // end if
    // handle FBM_VOLUME_BALANCE
    if (nFuncReturn == OK_CN00) {
      nFuncReturn = cn28_CopyParamInteger(szDbName, 
                                          pXParamData, 
                                          oConfigFile,
                                          PAN_FBM_DEVSP_BALANCE_OLD, 
                                          PAN_FBM_DEVSP_BALANCE, 
                                          replyData, 
                                          replyLen,
                                          false);
    } // end if
    // handle MAXVOLUMES
    if (nFuncReturn == OK_CN00) {
      nFuncReturn = cn28_CopyParamInteger(szDbName, 
                                          pXParamData, 
                                          oConfigFile,
                                          PAN_MAXDEVSPACES_OLD, 
                                          PAN_MAXDEVSPACES, 
                                          replyData, 
                                          replyLen,
                                          false);
    } // end if
    // handle MAXPAGER
    if (nFuncReturn == OK_CN00) {
      nFuncReturn = cn28_CopyParamInteger(szDbName, 
                                          pXParamData, 
                                          oConfigFile,
                                          PAN_MAXDATAWRITER_OLD, 
                                          PAN_MAXDATAWRITER, 
                                          replyData, 
                                          replyLen,
                                          false);
    } // end if
    // handle MAXLOGVOLUMES
    if (nFuncReturn == OK_CN00) {
      nFuncReturn = cn28_CopyParamInteger(szDbName, 
                                          pXParamData, 
                                          oConfigFile,
                                          PAN_MAXARCHIVELOGS_OLD, 
                                          PAN_MAXARCHIVELOGS, 
                                          replyData, 
                                          replyLen,
                                          false);
    } // end if
    // handle VOLUMENO_BIT_COUNT
    if (nFuncReturn == OK_CN00) {
      nFuncReturn = cn28_CopyParamInteger(szDbName, 
                                          pXParamData, 
                                          oConfigFile,
                                          PAN_DEVNO_BIT_COUNT_OLD, 
                                          PAN_DEVNO_BIT_COUNT, 
                                          replyData, 
                                          replyLen,
                                          true);
    } // end if
    // handle TRACE_PAGES_LW
    if (nFuncReturn == OK_CN00) {
      nFuncReturn = cn28_CopyParamInteger(szDbName, 
                                          pXParamData, 
                                          oConfigFile,
                                          PAN_TRACE_PAGES_AL_OLD, 
                                          PAN_TRACE_PAGES_AL, 
                                          replyData, 
                                          replyLen,
                                          false);
    } // end if
    // handle TRACE_PAGES_PG
    if (nFuncReturn == OK_CN00) {
      nFuncReturn = cn28_CopyParamInteger(szDbName, 
                                          pXParamData, 
                                          oConfigFile,
                                          PAN_TRACE_PAGES_DW_OLD, 
                                          PAN_TRACE_PAGES_DW, 
                                          replyData, 
                                          replyLen,
                                          false);
    } // end if
    // handle LOG_MODE/LOG_MIRRORED
    RTEConf_Parameter::StringElement szLogMode[RTECONF_MAXSTRINGLENGTH + 1];
    bool  bDual = false;
    if (oConfigFile.GetValue (KERNELPARAM_LOG_MODE, szLogMode, oMsg)) {
      if (STRCMP_UTF8(szLogMode, "DUAL") == 0) {
        bDual = true;
        STRCPY_UTF8(szLogMode, "YES");
        if (!oConfigFile.SetValue (KERNELPARAM_LOG_MIRRORED, szLogMode, oMsg)) {
          SAPDBErr_MessageList aDBM(DBMSrv_DBMError(RTE), 0);
          aDBM.AppendNewMessage(oMsg);
          nFuncReturn = cn90AnswerMessage (replyData, replyLen, aDBM);
        }  // end if
      }  // end if
    } // end if

    // read DATA Volumes
    if (nFuncReturn == OK_CN00) {
      RTEConf_Parameter::Integer nVolumes;
      int                        nVolumeIndex;
      tsp00_XpKeyTypec           szOld;
      tsp00_XpKeyTypec           szNew;
      
      if (!oConfigFile.GetValue ((const SAPDB_UTF8 *) PAN_MAXDATADEVSPACES, nVolumes, oMsg)) {
        SAPDBErr_MessageList aDBM(DBMSrv_DBMError(RTE), 0);
        aDBM.AppendNewMessage(oMsg);
        nFuncReturn = cn90AnswerMessage (replyData, replyLen, aDBM);
      } // end if

      for (nVolumeIndex = 1; (nVolumeIndex <= nVolumes) && (nFuncReturn == OK_CN00); ++nVolumeIndex) {
        sprintf(szOld, PAN_DATADEV_N_OLD"%04d", nVolumeIndex);
        sprintf(szNew, PAN_DATADEV_N"%04d"    , nVolumeIndex);
        nFuncReturn = cn28_CopyParamString(szDbName, 
                                           pXParamData, 
                                           oConfigFile,
                                           szOld, 
                                           szNew, 
                                           replyData, 
                                           replyLen,
                                           true);
        if (nFuncReturn == OK_CN00) {
          sprintf(szOld, PAN_DATADEV_N_SIZE_OLD"%04d", nVolumeIndex);
          sprintf(szNew, PAN_DATADEV_N_SIZE"%04d"    , nVolumeIndex);
          nFuncReturn = cn28_CopyParamInteger(szDbName, 
                                              pXParamData, 
                                              oConfigFile,
                                              szOld, 
                                              szNew, 
                                              replyData, 
                                              replyLen,
                                              true);
        } // end if
        if (nFuncReturn == OK_CN00) {
          sprintf(szOld, PAN_DATADEV_N_TYPE_OLD"%04d", nVolumeIndex);
          sprintf(szNew, PAN_DATADEV_N_TYPE"%04d"    , nVolumeIndex);
          nFuncReturn = cn28_CopyParamString(szDbName, 
                                             pXParamData, 
                                             oConfigFile,
                                             szOld, 
                                             szNew, 
                                             replyData, 
                                             replyLen,
                                             true);
        } // end if

      } // end for

    } // end if

    // read LOG Volumes
    if (nFuncReturn == OK_CN00) {
      RTEConf_Parameter::Integer nVolumes;
      int                        nVolumeIndex;
      tsp00_XpKeyTypec           szOld;
      tsp00_XpKeyTypec           szNew;
      
      if (!oConfigFile.GetValue ((const SAPDB_UTF8 *) PAN_MAXARCHIVELOGS, nVolumes, oMsg)) {
        SAPDBErr_MessageList aDBM(DBMSrv_DBMError(RTE), 0);
        aDBM.AppendNewMessage(oMsg);
        nFuncReturn = cn90AnswerMessage (replyData, replyLen, aDBM);
      } // end if

      for (nVolumeIndex = 1; (nVolumeIndex <= nVolumes) && (nFuncReturn == OK_CN00); ++nVolumeIndex) {
        sprintf(szOld, PAN_ARCHIVE_LOG_N_OLD"%03d", nVolumeIndex);
        sprintf(szNew, PAN_ARCHIVE_LOG_N"%03d"    , nVolumeIndex);
        nFuncReturn = cn28_CopyParamString(szDbName, 
                                           pXParamData, 
                                           oConfigFile,
                                           szOld, 
                                           szNew, 
                                           replyData, 
                                           replyLen,
                                           true);
        if (nFuncReturn == OK_CN00) {
          sprintf(szOld, PAN_ARCHIVE_LOG_N_SIZE_OLD"%03d", nVolumeIndex);
          sprintf(szNew, PAN_ARCHIVE_LOG_N_SIZE"%03d"    , nVolumeIndex);
          nFuncReturn = cn28_CopyParamInteger(szDbName, 
                                              pXParamData, 
                                              oConfigFile,
                                              szOld, 
                                              szNew, 
                                              replyData, 
                                              replyLen,
                                              true);
          // Hack!!! 
          // http://pts:1080/webpts?wptsdetail=yes&ErrorType=0&ErrorID=1125994
          sprintf(szOld, PAN_ARCHIVE_LOG_N_SIZE_OLD"%02d", nVolumeIndex);
          sprintf(szNew, PAN_ARCHIVE_LOG_N_SIZE"%03d"    , nVolumeIndex);
          nFuncReturn = cn28_CopyParamInteger(szDbName, 
                                              pXParamData, 
                                              oConfigFile,
                                              szOld, 
                                              szNew, 
                                              replyData, 
                                              replyLen,
                                              true);
        } // end if
        if (nFuncReturn == OK_CN00) {
          sprintf(szOld, PAN_ARCHIVE_LOG_N_TYPE_OLD"%03d", nVolumeIndex);
          sprintf(szNew, PAN_ARCHIVE_LOG_N_TYPE"%03d"    , nVolumeIndex);
          nFuncReturn = cn28_CopyParamString(szDbName, 
                                             pXParamData, 
                                             oConfigFile,
                                             szOld, 
                                             szNew, 
                                             replyData, 
                                             replyLen,
                                             true);
        } // end if
        if (bDual) {
          sprintf(szOld, (SAPDB_Char *)KERNELPARAM_M_ARCHIVE_LOG_"%03d",     nVolumeIndex);
          sprintf(szNew, (SAPDB_Char *)KERNELPARAM_M_LOG_VOLUME_NAME_"%03d", nVolumeIndex);
          nFuncReturn = cn28_CopyParamString(szDbName, 
                                             pXParamData, 
                                             oConfigFile,
                                             szOld, 
                                             szNew, 
                                             replyData, 
                                             replyLen,
                                             true);
          if (nFuncReturn == OK_CN00) {
            sprintf(szOld, (SAPDB_Char *)KERNELPARAM_M_ARCHIVE_TYPE_"%03d",     nVolumeIndex);
            sprintf(szNew, (SAPDB_Char *)KERNELPARAM_M_LOG_VOLUME_TYPE_"%03d", nVolumeIndex);
            nFuncReturn = cn28_CopyParamString(szDbName, 
                                               pXParamData, 
                                               oConfigFile,
                                               szOld, 
                                               szNew, 
                                               replyData, 
                                               replyLen,
                                               true);
          } // end if
        } // end if

      } // end for

    } // end if

    // ready stamp kernelversion
    if (nFuncReturn == OK_CN00) {
      pParameter = cn29SearchParameter(pXParamData->pFirst, PAN_PARAM_KERNELVERSION);
      if (pParameter != NULL) {
        // get own version
        tsp00_Versionc      VersionString;
        sp100_GetVersionString(COMP_NAME_KERNEL_SP100, s100buildnumber, VersionString);
        
        cn29StringToParamValue(&(pParameter->xpRecord), VersionString.asCharp());

        RTEConf_Parameter::String pString = (RTEConf_Parameter::String) pParameter->xpRecord.xpValue.xpValueString.asCharp();

        if (!oConfigFile.SetValue ((const SAPDB_UTF8 *) pParameter->szParameterName.asCharp(), pString, oMsg)) {
          SAPDBErr_MessageList aDBM(DBMSrv_DBMError(RTE), 0);
          aDBM.AppendNewMessage(oMsg);
          nFuncReturn = cn90AnswerMessage (replyData, replyLen, aDBM);
        } else {
          cn29ParamValueToString(pParameter->szNewValue, &(pParameter->xpRecord));
          cn29WriteParamHistory(szDbName, pParameter, PAHI_CHANGE_CN002);
        } // end if
      } else {
        nFuncReturn = cn90AnswerIError(replyData, replyLen, ERR_XPNOTFOUND_CN00);
      } // end if
    } // end if

    // close param file
    if (nFuncReturn == OK_CN00) {
      if (!oConfigFile.Write(oMsg)) {
        SAPDBErr_MessageList aDBM(DBMSrv_DBMError(RTE), 0);
        aDBM.AppendNewMessage(oMsg);
        nFuncReturn = cn90AnswerMessage (replyData, replyLen, aDBM);
      } // end if
    } // end if

  } // end if

  return nFuncReturn;
} // end cn20_MigrateFrom_7_4_3_or_2_or_1
/*
  -------------------------------------------------------------------------
  function:     cn28_AnalyzeVersion
  -------------------------------------------------------------------------
 */
static bool cn28_AnalyzeVersion 
      ( const tsp00_C256c  &  szVersionString,
        tsp100_VersionID1  &  VersionID1)
{
  tsp00_C64c szVersion;
  tsp00_C64c szBuild;
  tsp00_C64c szTemp;
  char *     pBegin;
  char *     pEnd;
  
  // extract version information from string
  strncpy(szVersion, &szVersionString[BEG_OF_1ST_VERS_SUBSTR_SP100], END_OF_1ST_VERS_SUBSTR_SP100 - BEG_OF_1ST_VERS_SUBSTR_SP100 + 1);
  szVersion[END_OF_1ST_VERS_SUBSTR_SP100 - BEG_OF_1ST_VERS_SUBSTR_SP100 + 1] = 0;

  // extract build information from string
  strncpy(szBuild, &szVersionString[BEG_OF_2CND_VERS_SUBSTR_SP100], END_OF_2CND_VERS_SUBSTR_SP100 - BEG_OF_2CND_VERS_SUBSTR_SP100 + 1);
  szBuild[END_OF_2CND_VERS_SUBSTR_SP100 - BEG_OF_2CND_VERS_SUBSTR_SP100 + 1] = 0;

  // major version  
  pBegin = &szVersion[0];
  pEnd   = strchr(pBegin, '.');
  if (pBegin != NULL && pEnd != NULL) {
    strncpy(szTemp, pBegin, pEnd - pBegin);
    szTemp[(int) (pEnd - pBegin)] = 0;
    VersionID1.MajorVersion_sp100 = (unsigned char) atoi(szTemp);
    pBegin = pEnd + 1;
    pEnd   = strchr(pBegin, '.');
  } else {
    return false;
  } // end if

  // minor version  
  if (pBegin != NULL && pEnd != NULL) {
    strncpy(szTemp, pBegin, pEnd - pBegin);
    szTemp[(int) (pEnd - pBegin)] = 0;
    VersionID1.MinorVersion_sp100 = (unsigned char) atoi(szTemp);
    pBegin = pEnd + 1;
  } else {
    return false;
  } // end if

  // correction level
  if (pBegin != NULL) {
    VersionID1.CorrLevel_sp100 = (unsigned char) atoi(pBegin);
  } else {
    return false;
  } // end if
  
  // build
  VersionID1.BuildNumberPrefix_sp100 = (unsigned char) atoi(szBuild);

  return true;
} // end cn28_AnalyzeVersion

/*
  -------------------------------------------------------------------------
  function:     cn28MigrateParamFile
  -------------------------------------------------------------------------
 */
tcn00_Error cn28MigrateParamFile 
      ( tsp00_DbNamec       szDbName,
        tcn002_XParamData * pXParamData,
        char              * replyData,
        int               * replyLen)

{
  FUNCTION_DBG_MCN00_1("cn28MigrateParamFile");

  tcn00_Error nFuncReturn = OK_CN00;


  SAPDBErr_MessageList oMsg;
  SAPDB_Bool           bFileExist  = false;
  tsp00_C256c          szVersion;
  RTEConf_Parameter    oConfigFile(szDbName.asCharp(), 
                                   0, 
                                   RTECONF_MAXNAMELENGTH,
                                   RTECONF_MAXSTRINGLENGTH);
  tcn002_XPRecord      xpRecord;

  // open param file
  if (!oConfigFile.Read(bFileExist, oMsg)) {
    // key not found or paramfile not exist 
    // don't migrate but success
    return nFuncReturn;
  } else {
    if (!oConfigFile.GetValue((SAPDB_UTF8 *) PAN_PARAM_KERNELVERSION , (RTEConf_Parameter::String)xpRecord.xpValue.xpValueString.asCharp(), oMsg)) {
      // key not found or paramfile not exist 
      // don't migrate but success
      return nFuncReturn;
    } else {
      SAPDB_strcpy(szVersion, (char *) xpRecord.xpValue.xpValueString);
    } // end if
  } // end if

  // analyse version string
  tsp100_VersionID1   ParamVersionID1;
  if (!cn28_AnalyzeVersion(szVersion, ParamVersionID1)) {
    teo200_EventList aEvent(FUNCTION_NAME_MCN00_1, ERR_BADVERSION_CN00_1, szVersion.asCharp());
    nFuncReturn = cn90AnswerEvent(replyData, replyLen, aEvent);
  } // end if
  
  if (nFuncReturn == OK_CN00) {
    // get own version
    tsp100_VersionID1   DBMVersionID1;
    sp100_GetVersionID ( VersionIDType1_esp100, s100buildnumberEx, &DBMVersionID1 );

    // I am should be 7.5.0 (target)
    if ((DBMVersionID1.MajorVersion_sp100        == 7) &&
        (DBMVersionID1.MinorVersion_sp100        == 5) &&
        (DBMVersionID1.CorrLevel_sp100           == 0)    ) {
   
      // check valid sources
      if ((ParamVersionID1.MajorVersion_sp100        == 7) &&
          (ParamVersionID1.MinorVersion_sp100        == 5) &&
          (ParamVersionID1.CorrLevel_sp100           == 0)    ) {
        // OK can I can use all 7.5.0 with no change
        // -> Do Nothing!
      } else if ((ParamVersionID1.MajorVersion_sp100        ==  7) &&
                 (ParamVersionID1.MinorVersion_sp100        ==  2) &&
                 (ParamVersionID1.CorrLevel_sp100           ==  5) &&
                 (ParamVersionID1.BuildNumberPrefix_sp100   >  14)    ) {
        // Ooops 7.2.5 I love it!
        // place here a conversion 
        nFuncReturn = cn20_MigrateFrom_7_2_5_or_7_3_0(szDbName, pXParamData, replyData, replyLen);
        if (nFuncReturn == OK_CN00) {
          nFuncReturn = cn20_MigrateFrom_7_4_3_or_2_or_1(szDbName, pXParamData, replyData, replyLen);
        } // end if

      } else if ((ParamVersionID1.MajorVersion_sp100        ==  7) &&
                 (ParamVersionID1.MinorVersion_sp100        ==  3) &&
                 (ParamVersionID1.CorrLevel_sp100           ==  0) &&
                 (ParamVersionID1.BuildNumberPrefix_sp100   >  20)    ) {
        // 7.3.0 same conversion like 7.2.5
        // place here a conversion 
        nFuncReturn = cn20_MigrateFrom_7_2_5_or_7_3_0(szDbName, pXParamData, replyData, replyLen);
        if (nFuncReturn == OK_CN00) {
          nFuncReturn = cn20_MigrateFrom_7_4_3_or_2_or_1(szDbName, pXParamData, replyData, replyLen);
        } // end if

      } else if ((ParamVersionID1.MajorVersion_sp100        ==  7) &&
                 (ParamVersionID1.MinorVersion_sp100        ==  4) &&
                 (ParamVersionID1.CorrLevel_sp100           <   4)    ) {
        nFuncReturn = cn20_MigrateFrom_7_4_3_or_2_or_1(szDbName, pXParamData, replyData, replyLen);
      } else if ((ParamVersionID1.MajorVersion_sp100        ==  7) &&
                 (ParamVersionID1.MinorVersion_sp100        ==  4) &&
                 (ParamVersionID1.CorrLevel_sp100           ==  4)    ) {
        // OK can I can use all 7.4.4 with no change
        // -> Do Nothing!
      } else {
        // go away
        teo200_EventList aEvent(FUNCTION_NAME_MCN00_1, ERR_MIGRATE_CN00_1, szVersion.asCharp());
        nFuncReturn = cn90AnswerEvent(replyData, replyLen, aEvent);
      } // 
    
    } else {

      // If I am not 7.4.4 make a default check
      if ((DBMVersionID1.MajorVersion_sp100        == ParamVersionID1.MajorVersion_sp100) &&
          (DBMVersionID1.MinorVersion_sp100        == ParamVersionID1.MinorVersion_sp100)    ) {
        // OK I can use my own minor release
        // -> Do Nothing!
      } else {
        // But nothing else
        teo200_EventList aEvent(FUNCTION_NAME_MCN00_1, ERR_MIGRATE_CN00_1, szVersion.asCharp());
        nFuncReturn = cn90AnswerEvent(replyData, replyLen, aEvent);
      } // end if
    } // end if

  } // end if

  if (nFuncReturn == OK_CN00) {
    cn90AnswerOK(replyData, replyLen, NULL);
  } // end if

  return nFuncReturn;
} // end cn28MigrateParamFile
