/*!
  @file           IFR_UpdatableRowSet.h
  @author         TTK
  @ingroup        IFR_Fetch
  @brief          Implements class for rowset processing
  @see            

\if EMIT_LICENCE

    ========== licence begin  GPL
    Copyright (c) 2001-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end



\endif
*/
#ifndef IFR_UPDATABLEROWSET_H
#define IFR_UPDATABLEROWSET_H

#include "SAPDB/Interfaces/Runtime/IFR_RowSet.h"

/**
   @brief A class for representing a database rowset
 **/
class IFR_UpdatableRowSet : public IFR_RowSet
{
friend class IFR_RowSet;

public:
  /**
   * Creates a new updatable row set object.
   * @param connection the current connection.
   * @param resultSet  the (parent) result set
   */
    IFR_UpdatableRowSet (IFR_ResultSet *resultSet, IFR_Bool& memory_ok);

  /**
   * Deletes an updatable row set.
   */
  ~IFR_UpdatableRowSet ();

  /**
   * Inserts all rows of the row array
   */
  IFR_Retcode insertAllRows ();

  /**
   * Inserts the first row of the row array
   */
  IFR_Retcode insertOneRow ();

  /**
   * Updates the row on a given position in the row array
   */
  IFR_Retcode updateRow (int position);

  /**
   * Deletes the row on a given position in the row array
   */
  IFR_Retcode deleteRow (int position);

protected:
  /**
   * Checks if update or delete is allowed (i.e. if statement was
   * executed with updatable cursor concurrency).
   */
    IFR_Bool isResultSetUpdatable();

  /**
   * Builds the internal insert or update statement
   */
    IFR_Retcode buildCommand(IFR_String& command,
                             const char *commandprefix,
                             IFR_Bool& memory_ok);

  /**
   * Bind all columns of the result set as input parameters
   * @return <code>IFR_OK</code> if ok, otherwise <code>IFR_NOT_OK</code>
   */
  IFR_Retcode bindColumnsAsParameters (IFR_PreparedStmt *);

    /**
     * Inserts rows. Depending on the parameter @c single,
     * this performas an execute or a batch execute.
     */
    IFR_Retcode insertRows(IFR_Bool single);
    
  /**
   * Pointer to prepared statement for update
   */
  IFR_PreparedStmt *m_updateStmt;
  /**
   * Pointer to prepared statement for insert
   */
  IFR_PreparedStmt *m_insertStmt;
  /**
   * Pointer to prepared statement for delete
   */
  IFR_PreparedStmt *m_deleteStmt;

};

#endif // IFR_UPDATABLEROWSET_H
