#!/usr/bin/perl
#
# $Header: //sapdb/V75/c_00/develop/sys/src/install/perl/SAPDB/Install/Instance/Check/Conditions.pm#3 $
# $DateTime: 2003/12/11 13:40:27 $
# $Change: 59245 $
#
#    ========== licence begin  GPL
#    Copyright (c) 2005 SAP AG
#
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of the GNU General Public License
#    as published by the Free Software Foundation; either version 2
#    of the License, or (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#    ========== licence end


package SAPDB::Install::Instance::Check::Conditions;

sub BEGIN {
	@ISA = ('SAPDB::Install::Exporter');
	@EXPORT = ();
	my $repo = SAPDB::Install::Repository::GetCurrent ();
	my @neededPackages = (
		'Instance::Base'
	);

	foreach my $package (@neededPackages) {
		unless (defined $repo->Eval
		("SAPDB::Install::$package", 1.01)) {
			print join ("\n", $repo->GetErr)."\n";
			die;
		}

		SAPDB::Install::Exporter::import ("SAPDB::Install::$package");
	}
}

push @ISA,
	'SAPDB::Install::Instance::Base';

#
# check starting conditions
#
sub conditions {
	my ($self) = @_;
	my $rc;

	$rc = $self->instance_type ();
	return (-1) unless (defined $rc && $rc == 0);

	$rc = $self->starting_release ();
	return (-1) unless (defined $rc && $rc == 0);

	if ($self->{'instancetype'} eq 'LVC') {
		$rc = $self->apo_starting_release ();
		return (-1) unless (defined $rc && $rc == 0);
	}

	$rc = $self->migration_strategy ();
	return (-1) unless (defined $rc && $rc == 0);

	return 0;
}

#
# check instance type
#
sub instance_type {
	my ($self) = @_; 

	$self->msgbegin ("finding instance type");
	$self->set_errorstate ('ERROR');

	if (exists $self->{instancetype} &&
	    defined $self->{instancetype} &&
	    $self->{instancetype} ne '') {
		$self->msg0 ("instance type already known\n");
		$self->msg1 ("instance type is ".$self->{instancetype}."\n");
		$self->msgend ();
		$self->set_errorstate ('OK');
		return 0;
	}

	$self->msg0 ("finding instance type...\n");

	my $dbm = $self->{dbm};
	my $instancetype = $dbm->param_directget ('INSTANCE_TYPE');

	#
	# maybe instance type is not set in paramfile
	# use OLTP as default
	#
	unless (defined $instancetype) {
		my $msg = $dbm->lastmsg ();

		unless (defined $msg) {
			$self->msg1 ($dbm->lastdialog ());
			$self->msgend ();
			return undef;
		}

		foreach my $line (split ('\n', $msg)) {
			if ($line =~ /ERR_XPNOTFOUND/) {
				$instancetype = 'OLTP';
				$self->msg1 ("using default instance type OLTP\n");
				last;
			}
		}
	}

	unless (defined $instancetype) {
		$self->msg1 ($dbm->lastdialog ());
		$self->msgend ();
		return undef;
	}

	#
	# if there is any other instance type then OLTP or LVC
	# use instance type OLTP
	#
	$self->msg1 ("instance type is ".$instancetype."\n");
	unless ($instancetype =~ /^OLTP|LVC$/) {
		$instancetype = 'OLTP';
		$self->msg1 ("using default instance type OLTP\n");
	}

	$self->{'instancetype'} = $instancetype;
	$self->msgend ();
	$self->set_errorstate ('OK');
	return 0;
}

#
# get starting release
#
sub starting_release {
	my ($self) = @_; 
	
	$self->msgbegin ("finding starting release");
	$self->set_errorstate ('ERROR');

	if (exists $self->{starting_release} &&
	    defined $self->{starting_release} &&
	    $self->{starting_release} ne '') {
		$self->msg0 ("starting release already known\n");
		$self->msg1
		("starting release is ".$self->{starting_release}."\n");
		$self->msgend ();
		$self->set_errorstate ('OK');
		return 0;
	}
	
	$self->msg0 ("finding starting release...\n");

	my $starting_release = $self->get_kernelversion ();
	unless (defined $starting_release) {
		$self->msgend ();
		return undef;
	}

	$self->msg1 ("starting release is ".$starting_release."\n");
	$self->{'starting_release'} = $starting_release;
	$self->msgend ();
	$self->set_errorstate ('OK');
	return 0;
}

#
# get apo starting release
#
sub apo_starting_release {
	my ($self) = @_;

	$self->msgbegin ("finding apo starting release");
	$self->set_errorstate ('ERROR');

	if (exists $self->{apo_starting_release} &&
	    defined $self->{apo_starting_release} &&
	    $self->{apo_starting_release} ne '') {
		$self->msg0 ("apo starting release already known\n");
		$self->msg1 ("apo starting release is ".
		             $self->{apo_starting_release}."\n");
		$self->msgend ();
		$self->set_errorstate ('OK');
		return 0;
	}
	
	$self->msg0 ("finding apo starting release...\n");
	my $apo_starting_release = $self->get_apoversion ();

	unless (defined $apo_starting_release) {
		#
		# in some case we can accept that
		# apo starting release in unknown
		# e.g. when livecache version is 7.2.05
		# so this is not an error
		#
		$self->msg1
		("apo starting release is unknown\n");
		$self->set_errorstate ('OK');
		return 0;
	}

	$self->msg1 ("apo starting release is ".$apo_starting_release."\n");
	$self->{'apo_starting_release'} = $apo_starting_release;
	$self->msgend ();
	$self->set_errorstate ('OK');
	return 0;
}

#
# find migration strategy
#
sub migration_strategy {
	my ($self) = @_;

	$self->set_errorstate ('ERROR');
	$self->msgbegin ("finding migration strategy");

	if (exists $self->{migration_strategy} &&
	    defined $self->{migration_strategy} &&
	    $self->{migration_strategy} ne '') {
		$self->msg0 ("migration strategy already known\n");
		$self->msg1 ("migration strategy is ".
		             $self->{migration_strategy}."\n");
		$self->msgend ();
		$self->set_errorstate ('OK');
		return 0;
	}

	$self->msg0 ("finding migration strategy...\n");

	$self->{migration_strategy} = undef;

	my $instance_type = $self->{instancetype};
	my $starting_release = $self->{starting_release};
	my $target_release = $self->{target_release};	
	my $apo_target_release = $self->{'apo_target_release'};

	#
	# without starting release or instance type we are completly lost
	# and somthing went terribly wrong 
	# 
	unless (defined $starting_release && defined $instance_type) {
		$self->msgend ();
		return undef;		
	}

	if ($instance_type eq 'OLTP') {
		#
		# target release is absolutly required to
		# find LVC migration strategy, cause worst case
		# is using extenal consistent backup
		#
		unless (defined $target_release) {
			$self->msg1 ("cannot find migration strategy ".
			             "cause of unknown target release\n");
			$self->msgend ();
			return undef;
		}
	
		#
		# apo target release is absolutly forbidden 
		# for OLTP upgrade
		#
		if (defined $apo_target_release && $apo_target_release ne '') {
			$self->msg1 ("cannot find migration strategy ".
			             "for OLTP instance to APO release\n");
			$self->msgend ();
			return undef;
		}
	
		#
		# target release is 7.2
		#
		if ($target_release =~ /^7\.2\.(\d+)/ && $1 >= 4) {
			#
			# using compatible log from 7.2 to 7.2
			#
			if ($starting_release =~ /7\.2\.(\d+)/ && $1 >= 4) {
				$self->{migration_strategy} = 'COMPATIBLE_LOG';
				$self->msg_strategy
				("migration is using compatible log\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}
		}

		#
		# target release is 7.3
		#
		if ($target_release =~ /^7\.3\.\d+/) {
			#
			# using compatible log from 7.2 to 7.3 
			#
			if ($starting_release =~ /7\.2\.(\d+)/ && $1 >= 4) {
				$self->{migration_strategy} = 'COMPATIBLE_LOG';
				$self->msg_strategy
				("migration is using compatible log\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			} 

			#
			# using compatible log from 7.3 to 7.3 
			#
			if ($starting_release =~ /7\.3\.\d+/) {
				$self->{migration_strategy} = 'COMPATIBLE_LOG';
				$self->msg_strategy
				("migration is using compatible log\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			} 
		}

		#
		# target release is 7.4.x, x >= 3
		#
		if ($target_release =~ /^7\.4\.(\d+)/ && $1 >= 3) {
			my $corr = $1;

			#
			# using consistent data from 7.3 to 7.4 
			#
			if ($starting_release =~ /7\.3\.\d+/) {
				if ($starting_release =~
				/^7\.3\.(\d+)\s+Build\s+(\d+)/ && $1 == 0 && $2 < 25) {
					$self->{migration_strategy} = 'NONE';
					$self->msg_strategy	("migration not allowed\n");
					$self->msg1
					("database starting release ".
				     "7.3.00 Build >= 25 required\n");
					$self->msgend ();
					$self->set_errorstate ('ERROR');
					return undef;
				}

				$self->{migration_strategy} = 'CONSISTENT_DATA';
				$self->msg_strategy
				("migration is using consistent data\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			} 

			#
			# using compatible log from 7.4 to 7.4 
			# this is simply a patch
			#
			if ($starting_release =~ /7\.4\.(\d+)/ && $1 >= 3 && $1 <= $corr) {
				$self->{migration_strategy} = 'COMPATIBLE_LOG';
				$self->msg_strategy
				("migration is using compatible log\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			} 
		}

		#
		# target release is 7.5
		#
		if ($target_release =~ /^7\.5\./) {

			#
			# using consistent data from 7.3 to 7.5 
			#
			if ($starting_release =~ /7\.3\./) {
				if (
				$starting_release =~ /^7\.3\.(\d+)\s+Build\s+(\d+)/ &&
				$1 == 0 && $2 < 25) {
					$self->{migration_strategy} = 'NONE';
					$self->msg_strategy	("migration not allowed\n");
					$self->msg1
					("database starting release ".
				     "7.3.00 Build >= 25 required\n");
					$self->msgend ();
					$self->set_errorstate ('ERROR');
					return undef;
				}

				$self->{migration_strategy} = 'CONSISTENT_DATA';
				$self->msg_strategy
				("migration is using consistent data\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			} 

			#
			# using compatible log from 7.4.x x >= 3 to 7.5
			# this is simply a patch
			#
			if ($starting_release =~ /7\.4\.(\d+)/ && $1 >= 3) {
				$self->{migration_strategy} = 'COMPATIBLE_LOG';
				$self->msg_strategy
				("migration is using compatible log\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			} 

			#
			# using compatible log from 7.5 to 7.5
			# this is simply a patch
			#
			if ($starting_release =~ /7\.5\./) {
				$self->{migration_strategy} = 'COMPATIBLE_LOG';
				$self->msg_strategy
				("migration is using compatible log\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			} 
		}

		#
		# target release is 7.6
		#
		if ($target_release =~ /^7\.6\./) {
			#
			# using compatible log from 7.6 to 7.6
			# this is simply a patch
			#
			if ($starting_release =~ /7\.6\./) {
				$self->{migration_strategy} = 'COMPATIBLE_LOG';
				$self->msg_strategy
				("migration is using compatible log\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}
		}

		#
		# in any other case we do not know migration strategy
		#
		$self->msg_strategy ("cannot find migration strategy\n");
		$self->msgend ();
		return undef;
	} elsif ($instance_type eq 'LVC') {
		#
		# target release is absolutly required
		# for liveCache upgrade to find migration strategy
		#
		unless (defined $target_release) {
			$self->msg1 ("cannot find migration strategy ".
			             "cause of unknown target release\n");
			$self->msgend ();
			return undef;
		}
		
		#
		# apo target release is absolutly required
		# for liveCache upgrade to find migration strategy
		#
		unless (defined $apo_target_release) {
			$self->msg1 ("cannot find migration strategy ".
			             "cause of unknown apo target release\n");
			$self->msgend ();
			return undef;
		}
		

		#
		# apo starting release is required
		# for liveCache upgrade to find migration strategy
		# if apo starting release is unknown and
		# liveCache starting release is 7.2.05,
		# apo starting release 3.0A is supposed
		#
		my $apo_starting_release = $self->{'apo_starting_release'};
		unless (defined $apo_starting_release) {
			unless ($starting_release =~ /7\.2\.(\d+)/ && $1 == 5) {
				$self->msg1
				("cannot find migration strategy ".
			         "cause of unknown apo starting release\n");
				$self->msgend ();
				return undef;
			}

			$self->msg1 ("suppose apo starting release 3.0A ".
			             "for liveCache 7.2.05\n");

			$apo_starting_release = '3.0.0.0';
			$self->{'apo_starting_release'} =
				$apo_starting_release;
		}

		#
		# target is 7.2.xx/3.0A
		#
		# the only supported upgrade is from 7.2.xx/3.0A to 7.2.xx/3.0A
		# we are using COMPATIBLE_LOG,
		# because no log is used at all and datadevs are compatible
		#
		if (($target_release =~ /^7\.2\.\d+/) &&
		    ($apo_target_release =~ /^3\.0\.0/)) {
			#
			# this is simply a patch
			#
			if (($starting_release =~ /^7\.2\.\d+/) &&
			    ($apo_starting_release =~ /^3\.0\.0/)) {
				$self->{migration_strategy} = 'COMPATIBLE_LOG';

				$self->msg_strategy
				("migration is using compatible log\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}
		}

		#
		# target is 7.5/5.00
		#
		if (($target_release =~ /^7\.5\./) &&
		    ($apo_target_release =~ /^5\.0\.0/)) {

			#
			# migration from 7.5/5.00 to 7.5/5.00
			# is simply a patch
			#
			if (($starting_release =~ /^7\.5\./) &&
			    ($apo_starting_release =~ /^5\.0\.0/)) { 
				$self->{migration_strategy} = 'COMPATIBLE_LOG';

				$self->msg_strategy
				("migration is using compatible log\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}

			#
			# migration from 7.5/5.00 to 7.5/5.00
			# is simply a patch
			#
			if (($starting_release =~ /^7\.5\./) &&
			    ($apo_starting_release =~ /^4\.1\.0/)) { 
				$self->{migration_strategy} =
				'EXTERNAL_LIVECACHE_CONSISTENCE';

				$self->msg_strategy
				("external livecache consistency\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}

			#
			# migration from 7.4.>=03/4.10 to 7.5/5.00
			# is simply a patch
			#
			if (($starting_release =~ /^7\.4\.(\d+)/ && $1 >= 3) &&
			    ($apo_starting_release =~ /^4\.1\.0/)) { 
				$self->{migration_strategy} =
				'EXTERNAL_LIVECACHE_CONSISTENCE';

				$self->msg_strategy
				("external livecache consistency\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}

			#
			# migration from 7.4.>=03/4.00 to 7.5/5.00
			# is simply a patch
			#
			if (($starting_release =~ /^7\.4\.(\d+)/ && $1 >= 3) &&
			    ($apo_starting_release =~ /^4\.0\.0/)) {
				$self->{migration_strategy} =
				'EXTERNAL_LIVECACHE_CONSISTENCE';

				$self->msg_strategy ("migration is using ".
				"external livecache consistency\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}

			#
			# migration from 7.4.>=2/3.10 to 7.5/5.00 is using
			# external livecache consistency,
			# that means any data form livecache are stored
			# inside of the apo database and
			# we can send all data to hell
			#
			if (($starting_release =~ /^7\.4\.(\d+)/ && $1 >= 2) && 
			    ($apo_starting_release =~ /^3\.1\.0/)) {
				$self->{migration_strategy} =
				'EXTERNAL_LIVECACHE_CONSISTENCE';

				$self->msg_strategy ("migration is using ".
				"external livecache consistency\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}

			#
			# migration from 7.4.>=2/3.0A to 7.5/5.00 is using
			# external livecache consistency,
			# that means any data form livecache are stored
			# inside of the apo database and
			# we can send all data to hell
			#
			if (($starting_release =~ /^7\.4\.(\d+)/ && $1 >= 2) && 
			    ($apo_starting_release =~ /^3\.0\.0/)) {
				$self->{migration_strategy} =
				'EXTERNAL_LIVECACHE_CONSISTENCE';

				$self->msg_strategy ("migration is using ".
				"external livecache consistency\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}

			#
			# migration from 7.2.05/3.0A to 7.5/5.00 is using
			# external livecache consistency,
			# that means any data form livecache are stored
			# inside of the apo database and
			# we can send all data to hell
			#
			if (($starting_release =~ /^7\.2\.(\d+)/ && $1 == 5) && 
			    ($apo_starting_release =~ /^3\.0\.0/)) {

				if ($starting_release =~
				/^7\.2\.\d+\s+Build\s+(\d+)/ && $1 < 15) {
					$self->{migration_strategy} = 'NONE';
					$self->msg_strategy
					("migration not allowed\n");
					$self->msg1 ("LVC starting release ".
				             "7.2.05 Build >= 15 required\n");
					$self->msgend ();
					$self->set_errorstate ('ERROR');
					return undef;
				}

				$self->{migration_strategy} =
				'EXTERNAL_LIVECACHE_CONSISTENCE';

				$self->msg_strategy ("migration is using ".
				"external livecache consistency\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}
		}

		#
		# target is 7.5/4.10
		#
		if (($target_release =~ /^7\.5\./) &&
		    ($apo_target_release =~ /^4\.1\.0/)) {

			#
			# migration from 7.5/4.10 to 7.5/4.10
			# is simply a patch
			#
			if (($starting_release =~ /^7\.5\./) &&
			    ($apo_starting_release =~ /^4\.1\.0/)) { 
				$self->{migration_strategy} = 'COMPATIBLE_LOG';

				$self->msg_strategy
				("migration is using compatible log\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}

			#
			# migration from 7.4.>=03/4.10 to 7.5/4.10
			# is simply a patch
			#
			if (($starting_release =~ /^7\.4\.(\d+)/ && $1 >= 3) &&
			    ($apo_starting_release =~ /^4\.1\.0/)) { 
				$self->{migration_strategy} = 'COMPATIBLE_LOG';

				$self->msg_strategy
				("migration is using compatible log\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}
			#
			# migration from 7.5/4.00 to 7.5/4.10 is using
			# external livecache consistency,
			# that means any data form livecache are stored
			# inside of the apo database and
			# we can send all data to hell
			#
			if (($starting_release =~ /^7\.5\./ ) &&
			    ($apo_starting_release =~ /^4\.0\.0/)) {
				$self->{migration_strategy} =
				'EXTERNAL_LIVECACHE_CONSISTENCE';

				$self->msg_strategy ("migration is using ".
				"external livecache consistency\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}

			#
			# migration from 7.4.>=03/4.00 to 7.5/4.10 is using
			# external livecache consistency,
			# that means any data form livecache are stored
			# inside of the apo database and
			# we can send all data to hell
			#
			if (($starting_release =~ /^7\.4\.(\d+)/ && $1 >= 3) &&
			    ($apo_starting_release =~ /^4\.0\.0/)) {
				$self->{migration_strategy} =
				'EXTERNAL_LIVECACHE_CONSISTENCE';

				$self->msg_strategy ("migration is using ".
				"external livecache consistency\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}

			#
			# migration from 7.4.>=2/3.10 to 7.5/4.10 is using
			# external livecache consistency,
			# that means any data form livecache are stored
			# inside of the apo database and
			# we can send all data to hell
			#
			if (($starting_release =~ /^7\.4\.(\d+)/ && $1 >= 2) && 
			    ($apo_starting_release =~ /^3\.1\.0/)) {
				$self->{migration_strategy} =
				'EXTERNAL_LIVECACHE_CONSISTENCE';

				$self->msg_strategy ("migration is using ".
				"external livecache consistency\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}

			#
			# migration from 7.4.>=2/3.0A to 7.5/4.00 is using
			# external livecache consistency,
			# that means any data form livecache are stored
			# inside of the apo database and
			# we can send all data to hell
			#
			if (($starting_release =~ /^7\.4\.(\d+)/ && $1 >= 2) && 
			    ($apo_starting_release =~ /^3\.0\.0/)) {
				$self->{migration_strategy} =
				'EXTERNAL_LIVECACHE_CONSISTENCE';

				$self->msg_strategy ("migration is using ".
				"external livecache consistency\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}

			#
			# migration from 7.2.05/3.0A to 7.5/4.10 is using
			# external livecache consistency,
			# that means any data form livecache are stored
			# inside of the apo database and
			# we can send all data to hell
			#
			if (($starting_release =~ /^7\.2\.(\d+)/ && $1 == 5) && 
			    ($apo_starting_release =~ /^3\.0\.0/)) {

				if ($starting_release =~
				/^7\.2\.\d+\s+Build\s+(\d+)/ && $1 < 15) {
					$self->{migration_strategy} = 'NONE';
					$self->msg_strategy
					("migration not allowed\n");
					$self->msg1 ("LVC starting release ".
				             "7.2.05 Build >= 15 required\n");
					$self->msgend ();
					$self->set_errorstate ('ERROR');
					return undef;
				}

				$self->{migration_strategy} =
				'EXTERNAL_LIVECACHE_CONSISTENCE';

				$self->msg_strategy ("migration is using ".
				"external livecache consistency\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}
		}


		#
		# target is 7.5/4.00
		#
		if (($target_release =~ /^7\.5\./) &&
		    ($apo_target_release =~ /^4\.0\.0/)) {

			#
			# migration from 7.5/4.00 to 7.5/4.00
			# is simply a patch
			#
			if (($starting_release =~ /^7\.5\./) &&
			    ($apo_starting_release =~ /^4\.0\.0/)) { 
				$self->{migration_strategy} = 'COMPATIBLE_LOG';

				$self->msg_strategy
				("migration is using compatible log\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}


			#
			# migration from 7.4.>=03/4.00 to 7.5/4.00
			#
			if (($starting_release =~ /^7\.4\.(\d+)/ && $1 >= 3) &&
			    ($apo_starting_release =~ /^4\.0\.0/)) {
				$self->{migration_strategy} = 'COMPATIBLE_LOG';

				$self->msg_strategy
				("migration is using compatible log\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}

			#
			# migration from 7.4.>=2/3.10 to 7.5/4.00 is using
			# external livecache consistency,
			# that means any data form livecache are stored
			# inside of the apo database and
			# we can send all data to hell
			#
			if (($starting_release =~ /^7\.4\.(\d+)/ && $1 >= 2) && 
			    ($apo_starting_release =~ /^3\.1\.0/)) {
				$self->{migration_strategy} =
				'EXTERNAL_LIVECACHE_CONSISTENCE';

				$self->msg_strategy ("migration is using ".
				"external livecache consistency\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}

			#
			# migration from 7.4.>=2/3.0A to 7.5/4.00 is using
			# external livecache consistency,
			# that means any data form livecache are stored
			# inside of the apo database and
			# we can send all data to hell
			#
			if (($starting_release =~ /^7\.4\.(\d+)/ && $1 >= 2) && 
			    ($apo_starting_release =~ /^3\.0\.0/)) {
				$self->{migration_strategy} =
				'EXTERNAL_LIVECACHE_CONSISTENCE';

				$self->msg_strategy ("migration is using ".
				"external livecache consistency\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}

			#
			# migration from 7.2.05/3.0A to 7.5/4.00 is using
			# external livecache consistency,
			# that means any data form livecache are stored
			# inside of the apo database and
			# we can send all data to hell
			#
			if (($starting_release =~ /^7\.2\.(\d+)/ && $1 == 5) && 
			    ($apo_starting_release =~ /^3\.0\.0/)) {

				if ($starting_release =~
				/^7\.2\.\d+\s+Build\s+(\d+)/ && $1 < 15) {
					$self->{migration_strategy} = 'NONE';
					$self->msg_strategy
					("migration not allowed\n");
					$self->msg1 ("LVC starting release ".
				             "7.2.05 Build >= 15 required\n");
					$self->msgend ();
					$self->set_errorstate ('ERROR');
					return undef;
				}

				$self->{migration_strategy} =
				'EXTERNAL_LIVECACHE_CONSISTENCE';

				$self->msg_strategy ("migration is using ".
				"external livecache consistency\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}
		}#end target release 7.5/4.0





		#
		# target is 7.4.>=3/4.10
		#
		if (($target_release =~ /^7\.4\.(\d+)/ && $1 >= 3) &&
		    ($apo_target_release =~ /^4\.1\.0/)) {

			#
			# migration from 7.4.>=03/4.10 to 7.4.>=3/4.10
			# is simply a patch
			#
			if (($starting_release =~ /^7\.4\.(\d+)/ && $1 >= 3) &&
			    ($apo_starting_release =~ /^4\.1\.0/)) { 
				$self->{migration_strategy} = 'COMPATIBLE_LOG';

				$self->msg_strategy
				("migration is using compatible log\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}
		}

		#
		# target is 7.4.>=3/4.00
		#
		if (($target_release =~ /^7\.4\.(\d+)/ && $1 >= 3) &&
		    ($apo_target_release =~ /^4\.0\.0/)) {

			#
			# migration from 7.4.>=03/4.00 to 7.4.>=3/4.00
			# is simply a patch
			#
			if (($starting_release =~ /^7\.4\.(\d+)/ && $1 >= 3) &&
			    ($apo_starting_release =~ /^4\.0\.0/)) { 
				$self->{migration_strategy} = 'COMPATIBLE_LOG';

				$self->msg_strategy
				("migration is using compatible log\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}

			#
			# migration from 7.4.>=2/3.10 to 7.4.>=3/4.00 is using
			# external livecache consistency,
			# that means any data form livecache are stored
			# inside of the apo database and
			# we can send all data to hell
			#
			if (($starting_release =~ /^7\.4\.(\d+)/ && $1 >= 2) && 
			    ($apo_starting_release =~ /^3\.1\.0/)) {
				$self->{migration_strategy} =
				'EXTERNAL_LIVECACHE_CONSISTENCE';

				$self->msg_strategy ("migration is using ".
				"external livecache consistency\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}

			#
			# migration from 7.4.>=2/3.0A to 7.4.>=3/4.00 is using
			# external livecache consistency,
			# that means any data form livecache are stored
			# inside of the apo database and
			# we can send all data to hell
			#
			if (($starting_release =~ /^7\.4\.(\d+)/ && $1 >= 2) && 
			    ($apo_starting_release =~ /^3\.0\.0/)) {
				$self->{migration_strategy} =
				'EXTERNAL_LIVECACHE_CONSISTENCE';

				$self->msg_strategy ("migration is using ".
				"external livecache consistency\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}

			#
			# migration from 7.2.05/3.0A to 7.4.>=3/4.00 is using
			# external livecache consistency,
			# that means any data form livecache are stored
			# inside of the apo database and
			# we can send all data to hell
			#
			if (($starting_release =~ /^7\.2\.(\d+)/ && $1 == 5) && 
			    ($apo_starting_release =~ /^3\.0\.0/)) {

				if ($starting_release =~
				/^7\.2\.\d+\s+Build\s+(\d+)/ && $1 < 15) {
					$self->{migration_strategy} = 'NONE';
					$self->msg_strategy
					("migration not allowed\n");
					$self->msg1 ("LVC starting release ".
				             "7.2.05 Build >= 15 required\n");
					$self->msgend ();
					$self->set_errorstate ('ERROR');
					return undef;
				}

				$self->{migration_strategy} =
				'EXTERNAL_LIVECACHE_CONSISTENCE';

				$self->msg_strategy ("migration is using ".
				"external livecache consistency\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}
		}

		#
		# target is 7.4.>=2/3.10
		#
		if (($target_release =~ /^7\.4\.(\d+)/ && $1 >= 2) &&
		    ($apo_target_release =~ /^3\.1\.0/)) {
			#
			# migration from 7.4.>=02/3.10 to 7.4.>=02/3.10
			# is simply a patch
			#
			if (($starting_release =~ /^7\.4\.(\d+)/ && $1 >= 2) &&
			    ($apo_starting_release =~ /^3\.1\.0/)) { 
				$self->{migration_strategy} = 'COMPATIBLE_LOG';

				$self->msg_strategy
				("migration is using compatible log\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}

			#
			# migration form 7.4.01/3.10 to 7.4.>=02/3.10
			# is like a patch
			# 
			if (($starting_release =~ /^7\.4\.(\d+)/ && $1 == 1) &&
			    ($apo_starting_release =~ /^3\.1\.0/)) { 
				$self->{migration_strategy} = 'COMPATIBLE_LOG';

				$self->msg_strategy
				("migration is using compatible log\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}

			#
			# migration from 7.4.>=02/3.0A to 7.4.02/3.10 is using
			# external livecache consistency,
			# that means any data form livecache are stored
			# inside of the apo database and
			# we can send all data to hell
			#
			if (($starting_release =~ /^7\.4\.(\d+)/ && $1 >= 2) && 
			    ($apo_starting_release =~ /^3\.0\.0/)) {
				$self->{migration_strategy} =
				'EXTERNAL_LIVECACHE_CONSISTENCE';

				$self->msg_strategy ("migration is using ".
				"external livecache consistency\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}

			#
			# migration from 7.4.01/3.0A to 7.4.>=02/3.10 is using
			# external livecache consistency,
			# that means any data form livecache are stored
			# inside of the apo database and
			# we can send all data to hell
			#
			if (($starting_release =~ /^7\.4\.(\d+)/ && $1 == 1) && 
			    ($apo_starting_release =~ /^3\.0\.0/)) {
				$self->{migration_strategy} =
				'EXTERNAL_LIVECACHE_CONSISTENCE';

				$self->msg_strategy ("migration is using ".
				"external livecache consistency\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}

			#
			# migration from 7.2.05/3.0A to 7.4.>=02/3.10 is using
			# external livecache consistency,
			# that means any data form livecache are stored
			# inside of the apo database and
			# we can send all data to hell
			#
			if (($starting_release =~ /^7\.2\.(\d+)/ && $1 == 5) && 
			    ($apo_starting_release =~ /^3\.0\.0/)) {

				if ($starting_release =~
				/^7\.2\.\d+\s+Build\s+(\d+)/ && $1 < 15) {
					$self->{migration_strategy} = 'NONE';
					$self->msg_strategy
					("migration not allowed\n");
					$self->msg1 ("LVC starting release ".
				             "7.2.05 Build >= 15 required\n");
					$self->msgend ();
					$self->set_errorstate ('ERROR');
					return undef;
				}

				$self->{migration_strategy} =
				'EXTERNAL_LIVECACHE_CONSISTENCE';

				$self->msg_strategy ("migration is using ".
				"external livecache consistency\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}
		}

		#
		# target is 7.4.>=02/3.0A
		#
		if (($target_release =~ /^7\.4\.(\d+)/ && $1 >= 2) &&
		    ($apo_target_release =~ /^3\.0\.0/)) {
			#
			# this is simply a patch
			#
			if (($starting_release =~ /^7\.4\.(\d+)/ && $1 >= 2) &&
			    ($apo_starting_release =~ /^3\.0\.0/)) { 
				$self->{migration_strategy} = 'COMPATIBLE_LOG';

				$self->msg_strategy
				("migration is using compatible log\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}

			#
			# migration form 7.4.01/3.0A to 7.4.>=02/3.0A
			# is like a patch
			# 
			if (($starting_release =~ /^7\.4\.(\d+)/ && $1 == 1) &&
			    ($apo_starting_release =~ /^3\.0\.0/)) { 
				$self->{migration_strategy} = 'COMPATIBLE_LOG';

				$self->msg_startegy
				("migration is using compatible log\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}

			#
			# migration from 7.2.05/3.0A to 7.4.>=02/3.0A is using
			# external consistent backup,
			# that means we require a consistent backup
			# for migration made with DBMGUI in cold mode
			#
			if (($starting_release =~ /^7\.2\.(\d+)/ && $1 == 5) && 
			    ($apo_starting_release =~ /^3\.0\.0/)) {

				if ($starting_release =~
				/^7\.2\.\d+\s+Build\s+(\d+)/ && $1 < 15) {
					$self->{migration_strategy} = 'NONE';
					$self->msg_strategy
					("migration not allowed\n");
					$self->msg1 ("LVC starting release ".
				             "7.2.05 Build >= 15 required\n");
					$self->msgend ();
					$self->set_errorstate ('ERROR');
					return undef;
				}

				$self->{migration_strategy} =
				'EXTERNAL_CONSISTENT_BACKUP';
				$self->msg_strategy ("migration is using ".
				"external consistent backup\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}
		}

		#
		# target is 7.4.01/3.10
		#
		if (($target_release =~ /^7\.4\.(\d+)/ && $1 == 1) &&
		    ($apo_target_release =~ /^3\.1\.0/)) {
			#
			# this is simply a patch
			#
			if (($starting_release =~ /^7\.4\.(\d+)/ && $1 == 1) &&
			    ($apo_starting_release =~ /^3\.1\.0/)) { 
				$self->{migration_strategy} = 'COMPATIBLE_LOG';

				$self->msg_strategy
				("migration is using compatible log\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}
		}

		#
		# target is 7.4.01/3.0A
		#
		if (($target_release =~ /^7\.4\.(\d+)/ && $1 == 1) &&
		    ($apo_target_release =~ /^3\.0\.0/)) {
			#
			# this is simply a patch
			#
			if (($starting_release =~ /^7\.4\.(\d+)/ && $1 == 1) &&
			    ($apo_starting_release =~ /^3\.0\.0/)) { 
				$self->{migration_strategy} = 'COMPATIBLE_LOG';

				$self->msg_strategy
				("migration is using compatible log\n");
				$self->msgend ();
				$self->set_errorstate ('OK');
				return 0;
			}
		}

		#
		# in any other case we do not know migration strategy
		#
		$self->msg_strategy ("cannot find migration strategy\n");
		$self->msgend ();
		return undef;
	} else {
		#
		# we have some doubt about instance type
		#
		$self->msg1 ("cannot find migration strategy\n");
		$self->msg1 ("for instance type ".$instance_type."\n");
		$self->msgend ();
		return undef;
	}

	$self->msgend ();
	return undef;
}

sub msg_strategy {
	my ($self, $msg) = @_;

	if (defined $msg && $msg ne '') {
		$self->msg1 ($msg);
	}

	if ($self->{instancetype} eq 'LVC') {
		$self->msg1 ("from LVC ".$self->{starting_release}."\n");
		$self->msg1 ("and  APO ".$self->{apo_starting_release}."\n");
		$self->msg1 ("to   LVC ".$self->{target_release}."\n");
		$self->msg1 ("and  APO ".$self->{apo_target_release}."\n");
	} else {
		$self->msg1 ("from ".$self->{starting_release}."\n");
		$self->msg1 ("to   ".$self->{target_release}."\n");
	}
}

1;
