/**********************************************************************/
/* $Id: //tools/src/SAPSource/VSA/sap/vssap.c#17 $
 **********************************************************************/
/* 
 *  (C) Copyright SAP AG, Walldorf 2001-2013, All Rights reserved
 *
 * SAP AG DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL SAP AG BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
 * OF THIS SOFTWARE.
 */

/*source--------------------------------------------------------------*/
/*                                                                    */
/* Description:                                                       */
/* ============                                                       */
/*                                                                    */
/*                                                                    */
/*                                                                    */
/*  Author:        Markus Strehle, SAP AG                             */
/*  Reviewer:      SAP Security Department, SAP AG                    */
/*  Version:       2.00                                               */
/*                                                                    */
/*  Created:                                                          */
/*    01 Nov 2003  Markus Strehle                                     */
/*                                                                    */
/*  Modified:                                                         */
/*    20 Nov  2004  Markus Strehle                                    */
/*    06 June 2005  Markus Strehle                                    */
/*                  (added VSA_SCANERROR for wrong action type)       */
/*    15 Aug  2008  Markus Strehle                                    */
/*                  (changed defaults)                                */
/*    09 Feb  2013  Markus Strehle                                    */
/*                  Return VSA_CONTENTINFO for VSI 2                  */
/*                                                                    */
/*  Remarks:                                                          */
/*      Modifications are done for NW-VSI Version 2.00                */
/*      Bugs can be reported directly to                              */
/*      (mailto:markus.strehle@sap.com)                               */
/*                                                                    */
/*                                                                    */
/**********************************************************************/
/*                                                                    */
/*                                                                    */
/* --- public functions ----------------------------------------------*/
/*                                                                    */
/*      VsaStartup                                                    */
/*      VsaInit                                                       */
/*      VsaGetConfig                                                  */
/*      VsaScan                                                       */
/*      VsaReleaseScan                                                */
/*      VsaEnd                                                        */
/*      VsaCleanup                                                    */
/*                                                                    */
/* --- private functions ---------------------------------------------*/
/*      addScanError                                                  */
/*      addVirusInfo                                                  */
/*      addContentInfo                                                */
/*      scanCompressed                                                */
/*      scanBuffer                                                    */
/*      vsaSetScanConfig                                              */
/*      registerCallback                                              */
/*      freevirusinfo                                                 */
/*      freevirusinfo2                                                */
/*      freescanerror                                                 */
/*      freescanerror2                                                */
/*      freecontentinfo                                               */
/*      freecontentinfo2                                              */
/*      freeVSA_INIT                                                  */
/*      freeVSA_CONFIG                                                */
/*      getFileSize                                                   */
/*      getFileType                                                   */
/*      getByteType                                                   */
/*      check4ActiveContent                                           */
/*      checkContentType                                              */
/*      vsaSetContentTypeParametes                                    */
/*                                                                    */
/**********************************************************************/
/*--------------------------------------------------------------------*/
/* Adapter defines                                                    */
/*--------------------------------------------------------------------*/
#ifdef SAPwithUNICODE
#undef SAPwithUNICODE
#undef UNICODE 
#undef _UNICODE
#endif

/*--------------------------------------------------------------------*/
/* system includes (OS-dependent)                                     */
/*--------------------------------------------------------------------*/
#ifdef _WIN32
#ifndef WIN32_MEAN_AND_LEAN
#define WIN32_MEAN_AND_LEAN
#include <windows.h>
#endif
#endif

#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <sys/stat.h> 

/*--------------------------------------------------------------------*/
/* SAP includes                                                       */
/*--------------------------------------------------------------------*/
#include "vsaxxtyp.h"
#include "vsaxxvir.h"
#include "vssap.h"
#include "csdecompr.h"

/*--------------------------------------------------------------------*/
/* static globals                                                     */
/*--------------------------------------------------------------------*/
static Bool           bgInit        = FALSE;
static size_t         lgRefCounter  = 0;
static const PChar    pszEICAR      = (PChar) EICAR_STRING;
static const PChar    pszShkp       = (PChar) SHAKESPEARE_STRING;

/* all known messages, provided also in VSA_CONFIG */
static const UInt     uigVS_SAP_ALL = VS_M_ERROR              |
                                      VS_M_ABORTSCAN          |
                                      VS_M_VIRUS              |
                                      VS_M_CLEAN              |
                                      VS_M_NOTSCANNED         |
                                      VS_M_OBJECTFOUND;

/*--------------------------------------------------------------------*/
/* helper functions                                                   */
/*--------------------------------------------------------------------*/
static VSA_RC scanBuffer(UInt            uiJobID,
                         PChar           pszObjectName,
                         PByte           pObject,
                         size_t          lObjectSize,
                         USRDATA        *pUsrData);

static VSA_RC scanCompressed(UInt            uiJobID,
                             PChar           pszObjectName,
                             PByte           pObject,
                             size_t          lObjectSize,
                             USRDATA        *pUsrData);

static VSA_RC addContentInfo(UInt              uiJobID,
                             PChar             pszObjectName,
                             size_t            lObjectSize,
                             VS_OBJECTTYPE_T   tContentType,
                             PChar             pszExtension,
                             PChar             pszContentType,
                             PChar             pszCharSet,
                             UInt              lContent,
                             PPVSA_CONTENTINFO pp_content);

static VSA_RC addScanError(UInt            uiJobID,
                           PChar           pszObjectName,
                           size_t          lObjectSize,
                           Int             iErrorRC,
                           PChar           pszErrorText,
                           UInt            lError,
                           PPVSA_SCANERROR pp_scanerror);

static VSA_RC addVirusInfo(UInt            uiJobID,
                           PChar           pszObjectName,
                           size_t          lObjectSize,
                           Bool            bRepairable,
                           VS_DETECTTYPE_T tDetectType,
                           VS_VIRUSTYPE_T  tVirusType,
                           VS_OBJECTTYPE_T tObjectType,
                           VS_ACTIONTYPE_T tActionType,
                           UInt            uiVirusID,
                           PChar           pszVirusName,
                           PChar           pszFreeTextInfo,
                           UInt            lInfected,
                           PPVSA_VIRUSINFO pp_virusinfo);

static VSA_RC check4ActiveContent(UInt            uiJobID,
                                  PChar           pszObjectName,
                                  PByte           pObject,
                                  size_t          lObjectSize,
                                  USRDATA        *pUsrData);

static VSA_RC checkContentType(UInt            uiJobID,
                               PChar           pszObjectName,
                               size_t          lObjectSize,
                               PChar           pExtension,
                               PChar           pMimeType,
                               USRDATA        *pUsrData);

static char* memstr(char* input,
                    const unsigned char* search,
                    int len_input,
                    int len_search);
/*
 *  Assign the VSA_OPTPARAM values to
 *  internal configuration.
 */
static VSA_RC vsaSetScanConfig(VSA_SCANPARAM *,
                               VSA_OPTPARAMS *, 
                               USRDATA *
                               );

static VSA_RC registerCallback( VSA_CALLBACK *, 
                                USRDATA *
                              );

static VSA_RC vsaSetContentTypeParametes(VSA_OPTPARAM *,
                                         USRDATA *
                                         );

static void freevirusinfo(VSA_VIRUSINFO **);
static void freevirusinfo2(VSA_VIRUSINFO *);
static void freescanerror(VSA_SCANERROR **);
static void freescanerror2(VSA_SCANERROR *);
static void freecontentinfo(VSA_CONTENTINFO **);
static void freecontentinfo2(VSA_CONTENTINFO *);

static void freeVSA_INIT(  VSA_INIT **);
static void freeVSA_CONFIG(VSA_CONFIG **);

static VSA_RC getFileSize(PChar, size_t *);
static VSA_RC getFileType(PChar, PChar, PChar, VS_OBJECTTYPE_T *);
static VSA_RC getByteType(PByte , size_t , PChar, PChar, PUSRDATA);


/*--------------------------------------------------------------------*/
/* VSA public functions                                               */
/*--------------------------------------------------------------------*/
/**********************************************************************
 *  VsaStartup()
 *
 *  Description:
 *     Global initialization of the adapter. This function will be called
 *     once after loading the VSA.
 *
 *  Returncodes: 
 *  VSA_OK                   |      Success
 *  VSA_E_NOT_INITIALISED    |      Global initialization not successful
 *  VSA_E_NO_SPACE           |      Any resource allocation failed 
 *
 **********************************************************************/
VSA_RC
DLL_EXPORT VsaStartup( void )
{
    /*--------------------------------------------------------------------*/
    /* The startup will be called process global                          */
    /*--------------------------------------------------------------------*/
    if(bgInit == FALSE)
    {    
        bgInit = TRUE;
        /* 
        * Comment: 
        * Initialize the table for CRC check
        */
        InitializeTable();
    }
    /* Comment to the Return Code:
     * The VSA Specification allows here return codes <> VSA_OK, however
     * this function cannot transport error messages, therefore you should
     * do following: If you have global initialization inside this function
     * keep the error information and return it back in VsaInit->pszErrorText
     */
    return VSA_OK;
}


/**********************************************************************
 *  VsaGetConfig() 
 *
 *  Description:
 *     This call allows the SAP system to know which type of VSA was loaded and
 *     which parameters and other features it has. An AV vendor of a VSA
 *     can also define a default profile of initial and optional 
 *     parameters here. <nl>
 *     "REQ" (required) parameters are mandatory. <nl>
 *     "OPT" (optional) parameters are not mandatory.
 *
 *  Returncodes:
 *  VSA_OK                |         Success
 *  VSA_E_NOT_INITIALISED |         Global initialization not successful
 *  VSA_E_NO_SPACE        |         Any resource allocation failed 
 *  VSA_E_NULL_PARAMETER  |         NULL pointer provided
 *
 **********************************************************************/
VSA_RC 
DLL_EXPORT VsaGetConfig(PPVSA_CONFIG pp_config)
{
    size_t len = 0;

    /*
     * Comment:
     * This structure helps to define a list of parameters. This adapter
     * supports following of the VSA parameters.
     * The length is set to 0 because there are only BOOL parameters. In
     * case of Char parameters you have to set the length of the string.
     */
    static MY_OPTPARAMS _optparams[] = {
        { VS_OP_SCANBESTEFFORT         ,   VS_TYPE_BOOL   ,      0,     (void*)1},
        { VS_OP_SCANALLFILES           ,   VS_TYPE_BOOL   ,      0,     (void*)1},
        { VS_OP_SCANEXTRACT            ,   VS_TYPE_BOOL   ,      0,     (void*)1},
        { VS_OP_SCANMIMETYPES          ,   VS_TYPE_CHAR   ,      0,     (void*)""},
        { VS_OP_SCANEXTENSIONS         ,   VS_TYPE_CHAR   ,      0,     (void*)""},
        { VS_OP_BLOCKMIMETYPES         ,   VS_TYPE_CHAR   ,      0,     (void*)0}, /* only to demonstrate, that both is possible */
        { VS_OP_BLOCKEXTENSIONS        ,   VS_TYPE_CHAR   ,      0,     (void*)""}
    };

    VSA_RC      rc      =   VSA_OK;
    int         x       =   0,
                opar    =   sizeof(_optparams)/sizeof(_optparams[0]);
 
    if(bgInit == FALSE)
        return VSA_E_NOT_INITIALISED; /* no successful VsaStartup */

    /*--------------------------------------------------------------------*/
    /* VSA_CONFIG allocation. Return a filled structure in the function   */
    /*--------------------------------------------------------------------*/
    if(pp_config == NULL)
        return VSA_E_NULL_PARAM; /* no handle */
    (*pp_config)                           = (PVSA_CONFIG)calloc(1, sizeof(VSA_CONFIG));
    if ((*pp_config) == NULL)
        CLEANUP(VSA_E_NO_SPACE);
    /* init params */
    (*pp_config)->pInitParams              = (PVSA_INITPARAMS)calloc(1, sizeof(VSA_INITPARAMS));
    if ((*pp_config)->pInitParams == NULL)
        CLEANUP(VSA_E_NO_SPACE);

    /* no init params supported with this adapter */
    (*pp_config)->pInitParams->pInitParam  = NULL;
    
    /* option params */
    (*pp_config)->pOptParams               = (PVSA_OPTPARAMS)calloc(1, sizeof(VSA_OPTPARAMS));
    if ((*pp_config)->pOptParams == NULL)
        CLEANUP(VSA_E_NO_SPACE);
    (*pp_config)->pOptParams->pOptParam    = (PVSA_OPTPARAM)calloc(opar, sizeof(VSA_OPTPARAM));
    if ((*pp_config)->pOptParams->pOptParam == NULL)
        CLEANUP(VSA_E_NO_SPACE);


    (*pp_config)->pAdapterInfo             = (PVSA_ADAPTERINFO)calloc(1, sizeof(VSA_ADAPTERINFO));
    if ((*pp_config)->pAdapterInfo == NULL)
        CLEANUP(VSA_E_NO_SPACE);

    (*pp_config)->struct_size              =    sizeof(VSA_CONFIG);

    /*--------------------------------------------------------------------*/
    /* loop to fill the supported parameter structure                     */
    /*--------------------------------------------------------------------*/
    for(x=0; x<opar;x++)
    {
       /*  opar : number of entries in optparams
           x    : actual index
           we add parameters from the default structure and set this as default configuration structure
        */
        if(_optparams[x].tType == VS_TYPE_CHAR && _optparams[x].tCode != VS_OP_BLOCKMIMETYPES)
        {
            VSAddOPTParameter (  (*pp_config)->pOptParams->pOptParam,
                                 (*pp_config)->pOptParams->usOptParams,
                                 _optparams[x].tCode,
                                 _optparams[x].tType,
                                 _optparams[x].lLength,
                                  strdup((const char*)_optparams[x].pvValue)
                              );
        }
        else
        {
            VSAddOPTParameter (  (*pp_config)->pOptParams->pOptParam,
                                 (*pp_config)->pOptParams->usOptParams,
                                 _optparams[x].tCode,
                                 _optparams[x].tType,
                                 _optparams[x].lLength,
                                 _optparams[x].pvValue
                              );
        }
     
    }

    /*--------------------------------------------------------------------*/
    /* set adapter features   */
    /*--------------------------------------------------------------------*/
    (*pp_config)->uiVsaActionFlags =     VSA_AP_CHECKMIMETYPE | VSA_AP_SCAN | VSA_AP_BLOCKACTIVECONTENT;

    (*pp_config)->uiVsaScanFlags   =     VSA_SP_FILE             |
                                         VSA_SP_BYTES;

    (*pp_config)->uiVsaEvtMsgFlags =     uigVS_SAP_ALL;
    /* No client I/O callback supported for this VSA version
     * Otherwise you should set here the callback flags for CIO support.
     */
    (*pp_config)->uiVsaCIOMsgFlags =     0;

    /*--------------------------------------------------------------------*/
    /* set adapter info constants                                         */
    /*--------------------------------------------------------------------*/
    (*pp_config)->pAdapterInfo->struct_size          = sizeof(VSA_ADAPTERINFO);
    (*pp_config)->pAdapterInfo->usVsiVersion         = VSI_VERSION; /* stay here with VSI 1 */
    (*pp_config)->pAdapterInfo->usVsaMajVersion      = VSA_ADAPTER_MAJVER;
    (*pp_config)->pAdapterInfo->usVsaMinVersion      = VSA_ADAPTER_MINVER;
    (*pp_config)->pAdapterInfo->tAdapterID           = VS_AD_SAP;
    (*pp_config)->pAdapterInfo->tThreadingModel      = VS_THREAD_FREE; 
    SETERRORTEXT((*pp_config)->pAdapterInfo->pszVendorInfo , VSA_VENDORINFO );
    SETERRORTEXT((*pp_config)->pAdapterInfo->pszAdapterName ,VSA_ADAPTERNAME );

cleanup:
    if (rc == VSA_OK)
    {
        return (VSA_OK);
    }
    else
    {
        freeVSA_CONFIG(pp_config);
        return (rc);
    }

} /* VsaGetConfig */


/**********************************************************************
 *  VsaInit()
 *
 *  Description:
 *     Initializes or creates a new scan engine instance.
 *     Then assigns all the vendor specific data structures from
 *     actual AV product to VSA data structure.<nl>
 *     "REQ" (required) parameters are mandatory. <nl>
 *     "OPT" (optional) parameters are not mandatory.
 *
 *  Returncodes:
 *  VSA_OK                   |      Success
 *  VSA_E_EXPIRED            |      Engine or driver expired
 *  VSA_E_NO_SPACE           |      Any resource allocation failed 
 *  VSA_E_LOAD_FAILED        |      Load failed (such as external process)
 *  VSA_E_BAD_EXPRESSION     |      Regular expression syntax is invalid
 *  VSA_E_NULL_PARAM         |      NULL pointer provided
 *  VSA_E_INVALID_PARAM      |      At least one parameter is invalid
 *  VSA_E_DRIVER_FAILED      |      At least one driver failed
 *  VSA_E_NOT_SUPPORTED      |      At least one parameter or object is not supported
 *  VSA_E_CIO_FAILED         |      Client I/O request action failed.
 *  VSA_E_NOT_INITIALISED    |      Global initialization not successful
 *  VSA_E_CBC_TERMINATED     |      Action was terminated during callback
 *
 **********************************************************************/
VSA_RC 
DLL_EXPORT VsaInit( const PVSA_CALLBACK p_callback, 
                    const PVSA_INITPARAMS p_initparams, 
                          PPVSA_INIT pp_init )
{
    VSA_RC                      rc      =   VSA_OK;
    Int                         i       =   0;
    size_t                      len     =   0;
    struct tm                 _utc_date = { 0,      /* hours                    */
                                            0,      /* daylight                 */
                                            0,      /* day of month (1-31)      */
                                            0,      /* minutes after hour (0-59)*/
                                            0,      /* month (0-11, January=0)  */
                                            0,      /* seconds after min (0-59) */
                                            0,      /* day of week, 0-6,Sun=0   */
                                            0,      /* day of year, 0-365,Jan1=0*/
                                            0};     /* year (current minus 1900 */
        
    if(bgInit == FALSE)
        return VSA_E_NOT_INITIALISED; /* no successful VsaStartup */
    if(pp_init == NULL)
        return VSA_E_NULL_PARAM; /* no handle */
    
    /* set date 06.2.2013 */
    _utc_date.tm_mon = 1;
    _utc_date.tm_mday= 6;
    _utc_date.tm_year= 113;

    /* Comment:
     * In the VsaInit function you should either connect/contact your
     * internal engine or allocate memory for the scan instance.
     * The structure VSA_INIT contains several flags which should help you
     * to pass your scan handle, internal license information, flags and so on to
     * the VsaScan function.
     */
    /* Initialize VSA_INIT structure */
    (*pp_init)          =   (PVSA_INIT)calloc(1,sizeof(VSA_INIT));
    if ((*pp_init) == NULL)
        CLEANUP(VSA_E_NO_SPACE);
        /* sizeof */
    (*pp_init)->struct_size = sizeof(VSA_INIT);

    /* Important hint:
     * Set hEngine with a value != NULL,
     * here you should set the HANDLE of your internal
     * engine instance.
     */
#ifdef __i386__
    (*pp_init)->hEngine     = (PVoid)0xaaaaaaaa; 
#else
    (*pp_init)->hEngine     = (PVoid)0xaaaaaaaaaa; 
#endif
    (*pp_init)->uiViruses   = 1;
    (*pp_init)->uiExtensions= 0;
    (*pp_init)->uiIntRevNum = 0;
    (*pp_init)->uiSignature = 0xbbbbbbbb; /* add here an own magic */
    (*pp_init)->usEngineMajVersion = 2;
    (*pp_init)->usEngineMinVersion = 0;
    SETSTRING( (*pp_init)->pszEngineVersionText, VSA_VERSION_STRING );
     /* convert date to calendar date *//*CCQ_CLIB_LOCTIME_OK*/
    (*pp_init)->utcDate     = mktime(&_utc_date);
    (*pp_init)->iErrorRC   = 0;
    SETSTRING( (*pp_init)->pszErrorText, "No error occured" );
    /* set VSA_DRIVERINFO structure */
    (*pp_init)->usDrivers = MAX_DRIVERS;
    (*pp_init)->pDriver = (PVSA_DRIVERINFO)calloc(MAX_DRIVERS,sizeof(VSA_DRIVERINFO));
    if ((*pp_init)->pDriver == NULL)
        CLEANUP(VSA_E_NO_SPACE);

    /*
     * Comment: 
     * We allocate one driver info to demonstrate the use case.
     * The VSA specification does not dictate the usage of VSA_DRIVERINFO.
     * The information here will only be displayed at SAP internal applications,
     * but this information should help you later to anaylse with which pattern
     * files your engine runs.
     * This information also helps any customer to see which version of pattern
     * files are loaded.
     */
    for (i=0; i<MAX_DRIVERS; i++)
    {
        (*pp_init)->pDriver[i].struct_size      = sizeof(VSA_DRIVERINFO);
        SETSTRING( (*pp_init)->pDriver[i].pszName, VSA_DRIVER_NAME );
        (*pp_init)->pDriver[i].usDrvMajVersion  = 1;
        (*pp_init)->pDriver[i].usDrvMinVersion  = 0;
        /* convert date to calendar date *//*CCQ_CLIB_LOCTIME_OK*/
        (*pp_init)->pDriver[i].utcDate = mktime(&_utc_date);
        (*pp_init)->pDriver[i].uiViruses        = 1;
        (*pp_init)->pDriver[i].uiVariants       = 0;  
        (*pp_init)->pDriver[i].iDriverRC        = 0;          
        
    }
    
cleanup:

    if (rc != VSA_OK)
    {
        if (pp_init && (*pp_init) && (*pp_init)->iErrorRC == 0)
            freeVSA_INIT(pp_init);
    }
    else
    {   /* increase the reference counter */
        lgRefCounter++;
    }
    return (rc);
} /* VsaInit */



/**********************************************************************
 *  VsaScan()
 *
 *  Description:
 *     Performs the scan. Requires a valid instance handle VSA_INIT from
 *     <f VsaInit>. If the VSA should not perform any callback during the
 *     scan action, then the parameter pVsaScanparam can be set to NULL.
 *     The address of a handle to a <t VSA_SCANINFO> structure can optionally
 *     be provided. This means if "ppVsaScaninfo" is set to NULL, the VSA
 *     will not return any information about the scan action.
 *     This allows the caller to decide, if [only] callback [or \| and]
 *     [only] the VSA_SCANINFO structure should be returned. <nl>
 *     "REQ" (required) parameters are mandatory. <nl>
 *     "OPT" (optional) parameters are not mandatory.
 *
 *  Returncodes:
 *  Virus error codes (negative values):
 *  VSA_E_CLEAN_FAILED         |    Removing/replacing infection failed
 *  VSA_E_PATTERN_FOUND        |    Pattern was found
 *  VSA_E_MACRO_FOUND          |    Macro was found
 *  VSA_E_VIRUS_FOUND          |    Virus was found
 *  VSA_E_CLEAN_OK             |    The clean action was successful
 *  No error, no virus:
 *  VSA_OK                     |    Success
 *  Program error codes (positive values):
 *  VSA_E_NO_SPACE             |    Any resource allocation failed 
 *  VSA_E_NULL_PARAM           |    NULL pointer provided
 *  VSA_E_INVALID_PARAM        |    At least one parameter is invalid
 *  VSA_E_INVALID_HANDLE       |    The provided handle is invalid
 *  VSA_E_NOT_INITIALISED      |    The adapter was not successfully initialized
 *  VSA_E_NOT_SUPPORTED        |    At least one parameter or object is not supported
 *  VSA_E_INVALID_SCANOBJECT   |    See VSA_SCANPARAM, object is invalid
 *  VSA_E_CIO_FAILED           |    Client I/O failed. Scan could not be performed.
 *  VSA_E_SCAN_FAILED          |    The scan action failed
 *  VSA_E_NOT_SCANNED          |    At least one object was not scanned
 *  VSA_E_CBC_TERMINATED       |    Action was terminated during callback
 *
 **********************************************************************/
VSA_RC 
DLL_EXPORT VsaScan(
                 const PVSA_INIT      p_init, 
                 const PVSA_CALLBACK  p_callback,
                 const PVSA_SCANPARAM p_scanparam,
                 const PVSA_OPTPARAMS p_optparams, 
                 PPVSA_SCANINFO pp_scinfo
                )
{
    VSA_RC              rc              = VSA_OK;
    VS_CALLRC           _vsa_rc         = VS_CB_OK;
    PChar               pszBuffer       = NULL;
    PChar               pszReason       = NULL;
    PChar               pszObjName      = NULL;
    size_t              lBuffer         = 0;
    PVSA_SCANINFO       p_scaninfo      = NULL;
    FILE                *_fp            = NULL;
    Char                 szExt[EXT_LN]  = ".*";
    Char                 szExt2[EXT_LN] = ".*";
    Byte                _buffer[65536];
    Char                 szMimeType[MIME_LN] = "unknown/unknown";
    size_t              lLength         = 0;
    USRDATA             usrdata;
    PUSRDATA            pUsrData        = &usrdata;

    memset(&usrdata,0,sizeof(USRDATA));    
    if(bgInit == FALSE) {
        pszReason = (PChar)"Adapter is not initialized";
        CLEANUP(VSA_E_NOT_INITIALISED); /* no successful VsaStartup */
    }

    if (p_init == NULL) {
        pszReason = (PChar)"Adapter initialization handle missing ";
        CLEANUP(VSA_E_NULL_PARAM);
    }
    if( p_scanparam ==  NULL) {
        pszReason = (PChar)"Scan parameter structure is null";
        CLEANUP(VSA_E_NULL_PARAM);
    }
    if (p_scanparam->pszObjectName == NULL || *p_scanparam->pszObjectName==0) {
        pszObjName = VSA_BYTE_NAME;
    } else {
        pszObjName = p_scanparam->pszObjectName;
    }
    /*--------------------------------------------------------------------*/
    /* check structure size to protect yourself against different versions*/
    /*--------------------------------------------------------------------*/
    if (p_scanparam->struct_size != sizeof(VSA_SCANPARAM)) {
        pszReason = (PChar)"Scan parameter structure size is not valid";
        CLEANUP(VSA_E_INVALID_PARAM);
    }

    /* check callback structure passed to me */
    rc = registerCallback(p_callback,&usrdata);
    if (rc) {
        pszReason = (PChar)"Callback messages not supported";
        CLEANUP(rc);
    }
    usrdata.heng = p_init->hEngine;

    /*
     * allocate VSA_SCANINFO 
     */
    if (pp_scinfo != NULL) {
        (*pp_scinfo)         = (PVSA_SCANINFO)calloc(1,sizeof(VSA_SCANINFO));
        if ((*pp_scinfo) == NULL) {
            CLEANUP(VSA_E_NO_SPACE);
        }
        (*pp_scinfo)->pContentInfo = NULL;
        p_scaninfo = (*pp_scinfo);
    } else {
        p_scaninfo = (PVSA_SCANINFO)calloc(1,sizeof(VSA_SCANINFO));
        if (p_scaninfo == NULL) {
            CLEANUP(VSA_E_NO_SPACE);
        }
        p_scaninfo->pContentInfo = NULL;
    }
    usrdata.pScanInfo      = p_scaninfo;
    p_scaninfo->struct_size= sizeof(VSA_SCANINFO);
    p_scaninfo->uiJobID    = p_scanparam->uiJobID;
    /* pre-check based of object-/file-name */
    usrdata.tFileType = VS_OT_UNKNOWN;
    getFileType(pszObjName,szExt2,szMimeType,&usrdata.tFileType);
    if((p_scanparam->tActionCode & VSA_AP_CHECKMIMETYPE) == VSA_AP_CHECKMIMETYPE)
    {
        usrdata.bMimeCheck = TRUE;
    }
    if((p_scanparam->tActionCode & VSA_AP_BLOCKACTIVECONTENT) == VSA_AP_BLOCKACTIVECONTENT)
    {
        usrdata.bActiveContent = TRUE;
    }

    /*--------------------------------------------------------------------*/
    /* Comment:                                                           */
    /* Set scan parameter configuration                                   */ 
    /*--------------------------------------------------------------------*/
    rc = vsaSetScanConfig(p_scanparam, p_optparams,&usrdata);
    if (rc)
    {
        pszReason = (PChar)"At least one parameter is invalid!";
        CLEANUP(rc);
    }
    /*--------------------------------------------------------------------*/
    /* example callback to query whether we should start                  */
    /*--------------------------------------------------------------------*/
    _vsa_rc = CB_FUNC( VS_M_ABORTSCAN, (size_t)p_scanparam->uiJobID );
    if (_vsa_rc == VS_CB_NEXT || _vsa_rc == VS_CB_TERMINATE)
        CLEANUP(VSA_E_CBC_TERMINATED);

    _vsa_rc = CB_FUNC( VS_M_OBJECTFOUND, pszObjName );
    if (_vsa_rc == VS_CB_NEXT || _vsa_rc == VS_CB_TERMINATE)
        CLEANUP(VSA_E_CBC_TERMINATED);

    /*--------------------------------------------------------------------*/
    /* Comment:                                                           */
    /* Start here to process the different action types.                  */
    /* First we perform some validation checks                            */
    /*--------------------------------------------------------------------*/
    /* 
    * Scan local file
    */
    switch(p_scanparam->tScanCode)
    { 
    case VSA_SP_FILE:
        {
        /*
         * Scan local file.
         * check first size 
         */
        if(p_scanparam->pszObjectName != NULL)
        {
            rc = getFileSize(pszObjName,&usrdata.lObjectSize);
            if (rc) {
                pszReason = (PChar)"The file could not be opened!";
                CLEANUP( VSA_E_SCAN_FAILED );
            }
            if (usrdata.lObjectSize > sizeof(_buffer))
            {
                lLength = sizeof(_buffer);
            }
            _fp = fopen((const char*)p_scanparam->pszObjectName,"rb");
            if (_fp == NULL)
            {
                pszReason = (PChar)"The file could not be opened!";
                CLEANUP( VSA_E_SCAN_FAILED );
            }
            memset(_buffer,0,sizeof(_buffer));
            lLength =fread(_buffer,sizeof(Char),sizeof(_buffer)-1,_fp);
            _buffer[lLength] = 0;
            fclose(_fp);
        }
        else
        {
            pszReason = (PChar)"The object name was not specified!";
            CLEANUP( VSA_E_SCAN_FAILED );
        }
        pszBuffer = _buffer;
        lBuffer   = lLength;
        }           
    break;
    case VSA_SP_BYTES:
        {
            /*
             * Scan byte buffer
             */
        pszBuffer = p_scanparam->pbByte;
        lBuffer   = p_scanparam->lLength;        
        }
    break;    
    default:
        pszReason = (PChar)"The scan object is not supported; support only files and memory buffer";
        CLEANUP(VSA_E_INVALID_SCANOBJECT);
        /*NOTREACHED*/
        break;
    }
    if(usrdata.bMimeCheck == TRUE || usrdata.bScanAllFiles == TRUE)
    {
        rc = getByteType(pszBuffer,lBuffer,szExt,szMimeType,&usrdata);
        if(rc) CLEANUP(rc);
        if(usrdata.tFileType != usrdata.tObjectType)
        {
            if(usrdata.tFileType == VS_OT_UNKNOWN && usrdata.tObjectType == VS_OT_BINARY)
            {
                /* Here we found an unknown binary object, which is a regular case of the demo VSA
                 * And if we do not want block active content, we can stop here.
                 */
                if(usrdata.bActiveContent == FALSE)
                   CLEANUP(VSA_E_NOT_SCANNED);
            }
            else
            {
                Char err[1024];
                sprintf((char*)err,"Extension (%s) is not compatible to MIME type (%s)",(const char*)szExt2,(const char*)szMimeType);
                addVirusInfo(p_scanparam->uiJobID,
                             p_scanparam->pszObjectName,
                             lBuffer,
                             FALSE,
                             VS_DT_MIMEVALIDATION,
                             VS_VT_NOVIRUS,
                             usrdata.tObjectType,
                             VS_AT_BLOCKED,
                             0,
                             err,
                             (PChar)"The extra content check is active, therefore file extension must be compatible to the detected MIME type",
                             p_scaninfo->uiInfections++,
                             &p_scaninfo->pVirusInfo);
                CLEANUP(VSA_E_BLOCKED_BY_POLICY);
            }
        }
    }
    else
    {
        usrdata.tObjectType = usrdata.tFileType;
    }
    rc = addContentInfo(p_scanparam->uiJobID,
                        p_scanparam->pszObjectName,
                        lBuffer,
                        usrdata.tObjectType,
                        szExt,
                        szMimeType,
                        NULL,
                        p_scaninfo->uiScanned++,
                        &p_scaninfo->pContentInfo);
    if(rc) CLEANUP(rc);
    if(usrdata.bActiveContent == TRUE)
    {
        rc = check4ActiveContent(p_scanparam->uiJobID,
                                 p_scanparam->pszObjectName,
                                 pszBuffer,
                                 lBuffer,
                                 pUsrData);
        if(rc) CLEANUP(rc);
    }
    if(usrdata.bMimeCheck == TRUE)
    {
        rc =  checkContentType(p_scanparam->uiJobID,
                               p_scanparam->pszObjectName,
                               lBuffer,
                               szExt,
                               szMimeType,
                               pUsrData);
        if(rc) CLEANUP(rc);
    }
    rc = scanBuffer(p_scanparam->uiJobID,
                    p_scanparam->pszObjectName,
                    pszBuffer,
                    lBuffer,
                    pUsrData);
    if(rc) CLEANUP(rc);
    /* Exception handling */
cleanup:
    switch(rc)
    {
    case VSA_E_ACTIVECONTENT_FOUND:
    case VSA_E_BLOCKED_BY_POLICY:
    case VSA_E_VIRUS_FOUND: /* do not handle here */
    case VSA_E_CLEAN_OK:
    case VSA_E_CLEAN_FAILED:
    break;
    case VSA_OK:
    break;
    case VSA_E_NULL_PARAM:
    case VSA_E_NO_SPACE:
        SET_VSA_RC( VSA_E_NULL_PARAM );
        addScanError(0,
                     pszObjName,
                     0,
                     2,
                     pszReason,
                     p_scaninfo->uiScanErrors++,
                     &p_scaninfo->pScanError
                     );
        _vsa_rc = CB_FUNC( VS_M_ERROR, pszReason );
        if (_vsa_rc == VS_CB_NEXT || _vsa_rc == VS_CB_TERMINATE)
            SET_VSA_RC(VSA_E_CBC_TERMINATED);
    break;
    case VSA_E_NOT_SUPPORTED:
        SET_VSA_RC( VSA_E_NOT_SUPPORTED );
        addScanError(p_scanparam->uiJobID,
                     pszObjName,
                     usrdata.lObjectSize,
                     10,
                     pszReason,
                     p_scaninfo->uiScanErrors++,
                     &p_scaninfo->pScanError
                     );
        _vsa_rc = CB_FUNC( VS_M_ERROR, pszReason );
        if (_vsa_rc == VS_CB_NEXT || _vsa_rc == VS_CB_TERMINATE)
            SET_VSA_RC(VSA_E_CBC_TERMINATED);
    break;
    case VSA_E_NOT_SCANNED:
        SET_VSA_RC( VSA_E_NOT_SCANNED );
        pUsrData->pScanInfo->uiNotScanned++;
        addScanError(p_scanparam->uiJobID,
                     pszObjName,
                     usrdata.lObjectSize,
                     11,
                     (PChar)"This adapter will only scan EICAR test files",
                     p_scaninfo->uiScanErrors++,
                     &p_scaninfo->pScanError
                     );
        _vsa_rc = CB_FUNC( VS_M_NOTSCANNED, p_scaninfo->pScanError );
        if (_vsa_rc == VS_CB_NEXT || _vsa_rc == VS_CB_TERMINATE)
            SET_VSA_RC(VSA_E_CBC_TERMINATED);
    break;
    case VSA_E_SCAN_FAILED:
        SET_VSA_RC( VSA_E_SCAN_FAILED );
        addScanError(p_scanparam->uiJobID,
                     pszObjName,
                     usrdata.lObjectSize,
                     13,
                     pszReason,
                     p_scaninfo->uiScanErrors++,
                     &p_scaninfo->pScanError
                     );
        _vsa_rc = CB_FUNC( VS_M_ERROR, pszReason );
        if (_vsa_rc == VS_CB_NEXT || _vsa_rc == VS_CB_TERMINATE)
            SET_VSA_RC(VSA_E_CBC_TERMINATED);
    break;
    case VSA_E_CBC_TERMINATED:
        SET_VSA_RC( VSA_E_CBC_TERMINATED );
        addScanError(p_scanparam->uiJobID,
                     pszObjName,
                     usrdata.lObjectSize,
                     17,
                     (PChar)"This current action in callback was terminated",
                     p_scaninfo->uiScanErrors++,
                     &p_scaninfo->pScanError
                     );
    break;
    default:
        if(rc > 0) {
           if(pszReason==NULL) pszReason = (PChar)"Unknown error";
           SET_VSA_RC( rc );
           addScanError(p_scanparam->uiJobID,
                        pszObjName,
                        usrdata.lObjectSize,
                        (int)rc,
                        pszReason,
                        p_scaninfo->uiScanErrors++,
                        &p_scaninfo->pScanError
                        );
           _vsa_rc = CB_FUNC( VS_M_ERROR, pszReason );
           if (_vsa_rc == VS_CB_NEXT || _vsa_rc == VS_CB_TERMINATE)
               SET_VSA_RC(VSA_E_CBC_TERMINATED);
        }
    break;
    }
    if(pUsrData != NULL) {
       if(pUsrData->pszBlockExtensions) free(pUsrData->pszBlockExtensions);
       if(pUsrData->pszBlockMimeTypes)  free(pUsrData->pszBlockMimeTypes);
       if(pUsrData->pszScanExtensions)  free(pUsrData->pszScanExtensions);
       if(pUsrData->pszScanMimeTypes)   free(pUsrData->pszScanMimeTypes);
    }
    if(pp_scinfo == NULL) {
       VsaReleaseScan(&p_scaninfo);
    }
    return (rc);
} /* VsaScan */


/**********************************************************************
 *  VsaReleaseScan()
 *
 *  Description:
 *     Release the dynamically allocated structure VSA_SCANINFO. The address of the
 *     of the handle is required but the handle can also point to NULL. <nl>
 *     "REQ" (required) parameters are mandatory. <nl>
 *     "OPT" (optional) parameters are not mandatory.
 *  
 *  Returncodes:
 *  VSA_OK                   |      Success
 *  VSA_E_NOT_INITIALISED    |      Global initialization not successful
 *  VSA_E_NULL_PARAM         |      NULL pointer provided
 *
 **********************************************************************/
VSA_RC 
DLL_EXPORT VsaReleaseScan(PPVSA_SCANINFO ppscinfo)
{
    UInt i = 0;

    if(bgInit == FALSE)
        return VSA_E_NOT_INITIALISED; /* no successful VsaStartup */
    if(ppscinfo == NULL)
        return VSA_E_NULL_PARAM;
    /*--------------------------------------------------------------------*/
    /* free the VSA_SCANINFO structure and sub structure VSA_SCANINFO     */
    /*--------------------------------------------------------------------*/
    if( ppscinfo != NULL && (*ppscinfo) != NULL && (*ppscinfo)->pVirusInfo != NULL )
    {
        for(i=0 ; i < (UInt)((*ppscinfo)->uiInfections); i++)
        {
            freevirusinfo2(&((*ppscinfo)->pVirusInfo[i]));
        }
        free((*ppscinfo)->pVirusInfo);
        (*ppscinfo)->pVirusInfo = NULL;
    }
    /*--------------------------------------------------------------------*/
    /* free VSA_SCANERROR                                                 */
    /*--------------------------------------------------------------------*/
    if( ppscinfo != NULL && (*ppscinfo) != NULL && (*ppscinfo)->pScanError != NULL)
    {
        for(i=0 ; i < (UInt)((*ppscinfo)->uiScanErrors); i++)
        {
            freescanerror2(&((*ppscinfo)->pScanError[i]));
        }
        free((*ppscinfo)->pScanError);
        (*ppscinfo)->pScanError = NULL;
    }
    /*--------------------------------------------------------------------*/
    /* free VSA_CONTENTINFO                                               */
    /*--------------------------------------------------------------------*/
    if( ppscinfo != NULL && (*ppscinfo) != NULL && (*ppscinfo)->pContentInfo != NULL)
    {
        for(i=0 ; i < (UInt)((*ppscinfo)->uiScanned); i++)
        {
            freecontentinfo2(&((*ppscinfo)->pContentInfo[i]));
        }
        free((*ppscinfo)->pContentInfo);
        (*ppscinfo)->pContentInfo = NULL;
    }
    /*--------------------------------------------------------------------*/
    /* free a allocated pointer if object was repaired/modified           */
    /*--------------------------------------------------------------------*/
    if( ppscinfo != NULL && (*ppscinfo) != NULL && (*ppscinfo)->pbBytesCleaned != NULL)
    {
        free((*ppscinfo)->pbBytesCleaned);
        (*ppscinfo)->pbBytesCleaned = NULL;
    }
    /*--------------------------------------------------------------------*/
    /* free the pointer and reset it, so that the caller receives NULL    */
    /*--------------------------------------------------------------------*/
    if( (*ppscinfo) != NULL) {
        free((*ppscinfo));
        (*ppscinfo) = NULL;
    }        
    return (VSA_OK);

} /* VsaReleaseScan */


/**********************************************************************
 *  VsaEnd()
 *
 *  Description:
 *     Closes the engine instance. Releases also VSA_CONFIG allocated
 *     by VsaGetConfig - assumes that the actual job is done there. <nl>
 *     "REQ" (required) parameters are mandatory. <nl>
 *     "OPT" (optional) parameters are not mandatory.
 *
 *  Returncodes:
 *  VSA_OK                     |    Success
 *  VSA_E_NULL_PARAM           |    NULL pointer provided
 *  VSA_E_NOT_INITIALISED      |    Global initialization not successful
 *  VSA_E_END_FAILED           |    The AV engine could not be closed
 *  VSA_E_IN_PROGRESS          |    Any thread is still running
 *
 **********************************************************************/
VSA_RC 
DLL_EXPORT VsaEnd(PPVSA_INIT pp_init, PPVSA_CONFIG pp_config)
{ 
    VSA_RC        rc    = VSA_OK;
    
    if(bgInit == FALSE)
        CLEANUP( VSA_E_NOT_INITIALISED ); /* no successful VsaStartup */
    if(pp_init == NULL || pp_config == NULL)
        CLEANUP( VSA_E_NULL_PARAM );
    /*--------------------------------------------------------------------*/
    /* free VSA_INIT                                                      */
    /*--------------------------------------------------------------------*/
    if (pp_init != NULL && (*pp_init) != NULL)
    {
        freeVSA_INIT(pp_init);
    }   
        
    freeVSA_CONFIG(pp_config);

cleanup:
    return (rc);
} /* VsaEnd */


/**********************************************************************
 *  VsaCleanup()
 *
 *  Description:
 *     Global cleanup for the adapter. This function will be called at last
 *     before unloading the VSA.
 *
 *  Returncodes:
 *  VSA_OK                     |    Success
 *  VSA_E_NOT_INITIALISED      |    Global initialization not successful
 *  VSA_E_IN_PROGRESS          |    Any thread is still running
 *
 **********************************************************************/
VSA_RC 
DLL_EXPORT VsaCleanup( void )
{
    /* 
     * cleanup
     */
    if(bgInit == FALSE)
        return VSA_E_NOT_INITIALISED; /* no successful VsaStartup */
    
    if(lgRefCounter != (size_t)0)
        return VSA_E_IN_PROGRESS;     /* any instance is still active */
    /*--------------------------------------------------------------------*/
    /* The cleanup will be called process global                          */
    /*--------------------------------------------------------------------*/    
    bgInit = FALSE;
    return VSA_OK;
} /* VsaCleanup */


/**********************************************************************
 **********************************************************************
 *
 *
 *  PRIVATE helper/auxiliary functions, here only for this prototype implementation
 *  If you would built an adapter, then you can decide by yourself, 
 *  which parts you want have here or not.
 *
 *
 **********************************************************************
 **********************************************************************/
static VSA_RC scanBuffer(UInt            uiJobID,
                         PChar           pszObjectName,
                         PByte           pObject,
                         size_t          lObjectSize,
                         USRDATA        *pUsrData)
{
    VSA_RC         rc     = VSA_OK;
    VS_CALLRC _vsa_rc     = VS_CB_OK;
    if( pUsrData == NULL )
        return VSA_E_NULL_PARAM;

    if( pUsrData->tObjectType == VS_OT_SAR )
    {
        if(pUsrData->bScanAllFiles == FALSE && pUsrData->bScanBestEffort == FALSE && pUsrData->bScanCompressed == FALSE)
            CLEANUP(VSA_E_NOT_SCANNED);

        rc = scanCompressed(uiJobID,
                            pszObjectName,
                            pObject,
                            lObjectSize,
                            pUsrData);
    }
    else
    {
        /*
        * Comment:
        * Perform the scan!
        * This example adapter performs a plain memory compare,
        * because this coding should not demonstrate virus scan functionality,
        * but implementing SAP virus scan interface.
        * Normally you should compare the byte signatures!
        */
        if( lObjectSize== EICAR_LENGTH && 0 == (memcmp(pObject, pszEICAR, EICAR_LENGTH)) )
        {
            /* found EICAR */
            rc = addVirusInfo(uiJobID,
                              pszObjectName,
                              lObjectSize,
                              FALSE,
                              VS_DT_KNOWNVIRUS,
                              VS_VT_TEST,
                              pUsrData->tObjectType,
                              VS_AT_NOACTION,
                              0,
                              (PChar)"EICAR-Test-STRING",
                              (PChar)"See www.eicar.org, this is only a test pattern for AV engines",
                              pUsrData->pScanInfo->uiInfections,
                              &(pUsrData->pScanInfo->pVirusInfo));
            if(rc) CLEANUP(rc);
            pUsrData->pScanInfo->uiInfections++;
            SET_VSA_RC( VSA_E_VIRUS_FOUND );
            _vsa_rc = CB_FUNC( VS_M_VIRUS, pUsrData->pScanInfo->pVirusInfo );
            if (_vsa_rc == VS_CB_NEXT || _vsa_rc == VS_CB_TERMINATE) {
                CLEANUP(VSA_E_CBC_TERMINATED);
            }
            rc = VSA_E_VIRUS_FOUND;
        }
        else
        {   /* found Shakespeare text */
            if( lObjectSize== SHAKESPEARE_LENGHT && 0 == (memcmp(pObject, pszShkp, SHAKESPEARE_LENGHT)) )
            {
                SET_VSA_RC( VSA_OK );
                pUsrData->pScanInfo->uiClean++;
                /* found our positive object */
                _vsa_rc = CB_FUNC( VS_M_CLEAN, (size_t)uiJobID );
                if (_vsa_rc == VS_CB_NEXT || _vsa_rc == VS_CB_TERMINATE) {
                    CLEANUP(VSA_E_CBC_TERMINATED);
                }
                rc = VSA_OK;
            }
            else
            {
                rc = VSA_E_NOT_SCANNED;
            }
        }
    }
cleanup:
    return rc;
} /* scanBuffer */

static VSA_RC scanCompressed(UInt            uiJobID,
                             PChar           pszObjectName,
                             PByte           pObject,
                             size_t          lObjectSize,
                             USRDATA        *pUsrData)
{
    VSA_RC          rc      = VSA_OK;
    int             counter = 0;
    size_t          lLength = 0;
    Char            szExt[EXT_LN]  = ".*";
    Byte            _decompr[65536];
    Char            szMimeType[MIME_LN] = "unknown/unknown";
    struct SAREntry *_loc   = NULL,
                    *sentry = ParseEntriesFromBuffer(pObject,(SAP_INT)lObjectSize);

    if(sentry==NULL) {
        if(pUsrData->bMimeCheck == TRUE || pUsrData->bScanAllFiles == TRUE)
        {
            rc = addVirusInfo(uiJobID,
                              pszObjectName,
                              lObjectSize,
                              FALSE,
                              VS_DT_MIMEVALIDATION,
                              VS_VT_CORRUPTED,
                              pUsrData->tObjectType,
                              VS_AT_BLOCKED,
                              0,
                              (PChar)"Corrupted SAR",
                              (PChar)"The archive structure is invalid",
                              pUsrData->pScanInfo->uiInfections,
                              &(pUsrData->pScanInfo->pVirusInfo));
            if(rc) CLEANUP(rc);
            pUsrData->pScanInfo->uiInfections++;
            CLEANUP(VSA_E_BLOCKED_BY_POLICY);
        }
        else
        {
              addScanError(uiJobID,
                           pszObjectName,
                           lObjectSize,
                           13,
                           (PChar)"Corrupted SAR file",
                           pUsrData->pScanInfo->uiScanErrors++,
                           &pUsrData->pScanInfo->pScanError);
              CLEANUP(VSA_E_SCAN_FAILED);
        }
    }
    _loc = sentry;
    while(_loc != NULL) {
          counter++;
          if ((_loc->uncompressed_size < (size_t)EICAR_LENGTH) || (_loc->uncompressed_size > (size_t)SHAKESPEARE_LENGHT)) {
              /* we want only scan the both test objects */
              pUsrData->pScanInfo->uiNotScanned++;
              CLEANUP( VSA_E_NOT_SCANNED );
          }
          _loc = _loc->next;
    }
    
    if(counter != 1) CLEANUP( VSA_E_NOT_SCANNED );
    lLength = ExtractEntryFromBuffer(pObject,(SAP_INT)lObjectSize, 0, _decompr, 65536);
    if(lLength==0)
    {
       addScanError(uiJobID,
                    pszObjectName,
                    lObjectSize,
                    13,
                    (PChar)"Not extracted",
                    pUsrData->pScanInfo->uiScanErrors,
                    &pUsrData->pScanInfo->pScanError);
        pUsrData->pScanInfo->uiScanErrors++;
        pUsrData->pScanInfo->uiNotScanned++;
        CLEANUP( VSA_E_NOT_SCANNED );
    }
    else
    {
        rc = getByteType(_decompr,lLength,szExt,szMimeType,pUsrData);
        if(rc) CLEANUP( rc );
        rc = addContentInfo(uiJobID,
                            sentry->name,
                            lLength,
                            pUsrData->tObjectType,
                            szExt,
                            szMimeType,
                            NULL,
                            pUsrData->pScanInfo->uiScanned++,
                            &pUsrData->pScanInfo->pContentInfo);
        if(rc) CLEANUP(rc);
        /*
        * Comment:
        * Perform the Active Content Check inside of archive
        */
        if(pUsrData->bActiveContent == TRUE && pUsrData->bScanAllFiles == TRUE && pUsrData->bScanCompressed == TRUE)
        {
            rc = check4ActiveContent(uiJobID,
                                     sentry->name,
                                     _decompr,
                                     lLength,
                                     pUsrData);
            if(rc) CLEANUP(rc);
        }
        /*
        * Comment:
        * Perform the MIME Check inside of archive
        */
        if(pUsrData->bMimeCheck == TRUE && pUsrData->bScanAllFiles == TRUE && pUsrData->bScanCompressed == TRUE)
        {
            rc =  checkContentType(uiJobID,
                                   sentry->name,
                                   lLength,
                                   szExt,
                                   szMimeType,
                                   pUsrData);
            if(rc) CLEANUP(rc);
        }
        rc = scanBuffer(uiJobID,
                        sentry->name,
                        _decompr,
                        lLength,
                        pUsrData);
    }
cleanup:
    FreeInfo(sentry);
    return rc;
} /* scanCompressed */

static VSA_RC addContentInfo(UInt              uiJobID,
                             PChar             pszObjectName,
                             size_t            lObjectSize,
                             VS_OBJECTTYPE_T   tContentType,
                             PChar             pszExtension,
                             PChar             pszContentType,
                             PChar             pszCharSet,
                             UInt              lContent,
                             PPVSA_CONTENTINFO pp_content)
{
    size_t len = 0;
    VSA_RC rc  = VSA_OK;
    if( pp_content == NULL )
        return VSA_E_NULL_PARAM;

    if( (*pp_content) == NULL && lContent == 0 ) {
        (*pp_content) = (PVSA_CONTENTINFO)calloc(1,sizeof(VSA_CONTENTINFO));
    } else {
        (*pp_content) = (PVSA_CONTENTINFO)realloc((*pp_content), (lContent + 1) * sizeof(VSA_CONTENTINFO));
    }
    if( (*pp_content) == NULL )
        return VSA_E_NO_SPACE;

    (*pp_content)[lContent].struct_size = sizeof(VSA_CONTENTINFO);
    (*pp_content)[lContent].match_so = 0;
    (*pp_content)[lContent].match_eo = 0;
    (*pp_content)[lContent].uiJobID  = uiJobID;
    (*pp_content)[lContent].lObjectSize = lObjectSize;
    (*pp_content)[lContent].tObjectType = tContentType;
    if(pszObjectName) {
       SETSTRING((*pp_content)[lContent].pszObjectName, pszObjectName);
    }
    if(pszExtension) {
       SETSTRING((*pp_content)[lContent].pszExtension, pszExtension);
    } else {
       SETSTRING((*pp_content)[lContent].pszExtension, ".*");
    }
    if(pszContentType) {
       SETSTRING((*pp_content)[lContent].pszContentType, pszContentType);
    } else {
       SETSTRING((*pp_content)[lContent].pszContentType, "unknown/unknown");
    }
    if(pszCharSet) {
       SETSTRING((*pp_content)[lContent].pszCharSet, pszCharSet);
    } else {
       SETSTRING((*pp_content)[lContent].pszCharSet, "");
    }
cleanup:
    return rc;
} /* addContentInfo */

static VSA_RC addScanError(UInt            uiJobID,
                           PChar           pszObjectName,
                           size_t          lObjectSize,
                           Int             iErrorRC,
                           PChar           pszErrorText,
                           UInt            lError,
                           PPVSA_SCANERROR pp_scanerror)
{
    size_t len = 0;
    VSA_RC rc  = VSA_OK;

    if (pp_scanerror == NULL)
        return VSA_E_NULL_PARAM;

    if( (*pp_scanerror) == NULL && lError == 0 ) {
        (*pp_scanerror) = (PVSA_SCANERROR)calloc(1,sizeof(VSA_SCANERROR));
    } else {
        (*pp_scanerror) = (PVSA_SCANERROR)realloc((*pp_scanerror), (lError + 1) * sizeof(VSA_SCANERROR));
    }
    if( (*pp_scanerror) == NULL )
        return VSA_E_NO_SPACE;
        
    if( (*pp_scanerror) == NULL)
        return VSA_E_NO_SPACE;
  
    (*pp_scanerror)[lError].struct_size = sizeof(VSA_SCANERROR);    
    (*pp_scanerror)[lError].lObjectSize = lObjectSize;
    (*pp_scanerror)[lError].iErrorRC    = iErrorRC;
    (*pp_scanerror)[lError].uiJobID     = uiJobID;
    if(pszErrorText) {
        SETSTRING( (*pp_scanerror)[lError].pszErrorText, pszErrorText );
    } else {
        SETSTRING( (*pp_scanerror)[lError].pszErrorText, "Generic error" );
    }
    if(pszObjectName) {
        SETSTRING( (*pp_scanerror)[lError].pszObjectName, pszObjectName );
    }
    
cleanup:
    return rc;
} /* addScanError */

static VSA_RC addVirusInfo(UInt            uiJobID,
                           PChar           pszObjectName,
                           size_t          lObjectSize,
                           Bool            bRepairable,
                           VS_DETECTTYPE_T tDetectType,
                           VS_VIRUSTYPE_T  tVirusType,
                           VS_OBJECTTYPE_T tObjectType,
                           VS_ACTIONTYPE_T tActionType,
                           UInt            uiVirusID,
                           PChar           pszVirusName,
                           PChar           pszFreeTextInfo,
                           UInt            lInfected,
                           PPVSA_VIRUSINFO pp_virusinfo)
{
    size_t len = 0;
    VSA_RC rc  = VSA_OK;
    if( pp_virusinfo == NULL )
        return VSA_E_NULL_PARAM;

    if( (*pp_virusinfo) == NULL && lInfected == 0 ) {
        (*pp_virusinfo) = (PVSA_VIRUSINFO)calloc(1,sizeof(VSA_VIRUSINFO));
    } else {
        (*pp_virusinfo) = (PVSA_VIRUSINFO)realloc((*pp_virusinfo), (lInfected + 1) * sizeof(VSA_VIRUSINFO));
    }
    if( (*pp_virusinfo) == NULL )
        return VSA_E_NO_SPACE;

    (*pp_virusinfo)[lInfected].struct_size = sizeof(VSA_VIRUSINFO);
    (*pp_virusinfo)[lInfected].bRepairable = bRepairable;
    (*pp_virusinfo)[lInfected].uiVirusID   = uiVirusID;
    (*pp_virusinfo)[lInfected].uiJobID     = uiJobID;
    (*pp_virusinfo)[lInfected].lObjectSize = lObjectSize;
    (*pp_virusinfo)[lInfected].tActionType = tActionType;
    (*pp_virusinfo)[lInfected].tDetectType = tDetectType;
    (*pp_virusinfo)[lInfected].tObjectType = tObjectType;
    (*pp_virusinfo)[lInfected].tVirusType  = tVirusType;
    if(pszObjectName) {
       SETSTRING((*pp_virusinfo)[lInfected].pszObjectName, pszObjectName);
    }
    if(pszVirusName) {
       SETSTRING((*pp_virusinfo)[lInfected].pszVirusName, pszVirusName);
    } else {
       SETSTRING((*pp_virusinfo)[lInfected].pszVirusName, "Generic Infection");
    }
    if(pszFreeTextInfo) {
       SETSTRING((*pp_virusinfo)[lInfected].pszFreeTextInfo, pszFreeTextInfo);
    } else {
       SETSTRING((*pp_virusinfo)[lInfected].pszFreeTextInfo, "");
    }

cleanup:
    return rc;
} /* addVirusInfo */

static VSA_RC vsaSetScanConfig(VSA_SCANPARAM *p_scanparam,VSA_OPTPARAMS *p_optparams, USRDATA *usrdata)
{
    VSA_RC     rc        = VSA_OK;
    Int        i         = 0,
               arraysize = (p_optparams?p_optparams->usOptParams:0);

    if(p_scanparam->struct_size != sizeof(VSA_SCANPARAM))
    {
        addScanError(0,
                     p_scanparam->pszObjectName,
                     0,
                     13,
                     (PChar)"Invalid structure VSA_SCANPARAM passed",
                     usrdata->pScanInfo->uiScanErrors++,
                     &usrdata->pScanInfo->pScanError);
    }
    /*
     * set the optional setting
     */
    for (i=0; i<arraysize; i++)
    {
        switch (p_optparams->pOptParam[i].tCode)
        {/* SCANBESTEFFORT should provide a scan with "best effort" of the engine.
          * And effort stands for security settings, e.g. all files, compressed.
          * This demo adapter has only allFiles and scanCompress as setting, but
          * the idea of scanBestEffort is, that independent of the engine all security
          * relevant features are activated, even if the scan is then slower.
          * Comment to VSI-2: ScanBestEffort can also be used for Active Content and MIME check
          * activation, however this is no requirement and rely on the partner product
          */
        case VS_OP_SCANBESTEFFORT:
             if ((p_optparams->pOptParam[i].pvValue)!=NULL) {
                 usrdata->bScanBestEffort = TRUE;
                 usrdata->bScanAllFiles = TRUE;
                 usrdata->bScanCompressed = TRUE;
             }
        break;
        case VS_OP_SCANALLFILES:
             if ((p_optparams->pOptParam[i].pvValue)!=NULL) {
                 usrdata->bScanAllFiles = TRUE;
             } else {
                 usrdata->bScanAllFiles = FALSE;
             }
        break;
        case VS_OP_SCANEXTRACT:
             if ((p_optparams->pOptParam[i].pvValue)!=NULL) {
                 usrdata->bScanCompressed = TRUE;
             } else {
                 usrdata->bScanCompressed = FALSE;
             }
        break;
        case VS_OP_SCANMIMETYPES:
        case VS_OP_SCANEXTENSIONS:
        case VS_OP_BLOCKMIMETYPES:
        case VS_OP_BLOCKEXTENSIONS:
             rc = vsaSetContentTypeParametes(&p_optparams->pOptParam[i],usrdata);
             if(rc) return rc;
        break;
        default:
        break;
        }
    }
   
    return rc;
} /* vsaSetScanConfig */

static VSA_RC vsaSetContentTypeParametes(VSA_OPTPARAM *param,
                                         USRDATA *pUsrData
                                         )
{
    PChar dest = NULL;
    Bool  mime = FALSE;
    PChar in = (PChar)param->pvValue;
    size_t inlen = 0;
    UInt i = 0;

    if(in == NULL) return VSA_OK;
    inlen = strlen((const char*)in);
    dest = (PChar)calloc(1,inlen * 2);
    if(dest == NULL) return VSA_E_NO_SPACE;

    switch(param->tCode)
    {
        case VS_OP_SCANMIMETYPES:
                pUsrData->pszScanMimeTypes = dest;
                mime = TRUE;
        break;
        case VS_OP_SCANEXTENSIONS:
                pUsrData->pszScanExtensions = dest;
                mime = FALSE;
        break;
        case VS_OP_BLOCKMIMETYPES:
                pUsrData->pszBlockMimeTypes = dest;
                mime = TRUE;
        break;
        case VS_OP_BLOCKEXTENSIONS:
                pUsrData->pszBlockExtensions = dest;
                mime = FALSE;
        break;
        default:
              free(dest);
              return VSA_OK;
    }
    for(i=0; i < inlen; i++)
    {
        if(i==0 && mime==FALSE && in[i] != '.')
        {
            addScanError(0,(PChar)in,0,-1,(PChar)"Invalid extension",pUsrData->pScanInfo->uiScanErrors++,&pUsrData->pScanInfo->pScanError);
            return VSA_E_INVALID_PARAM;
        }
        dest[i] = tolower((int)in[i]);
    }
    if(i>0 && i < (inlen * 2) && dest[i] != ';')
        dest[i] = ';';
    return VSA_OK;
} /* vsaSetContentTypeParametes */

static VSA_RC getFileSize(Char *pszfilename, size_t *size)
{
      VSA_RC rc = VSA_OK;

      struct stat     buf;
        if (pszfilename == NULL)
            return VSA_E_NULL_PARAM;
        
        if (0!=(stat((const char*)pszfilename,&buf)))
            return VSA_E_SCAN_FAILED;
      
        (*size)  = buf.st_size;

        return rc;
} /* getFileSize */

static VSA_RC getFileType(PChar filename, PChar ext,  PChar mimetype, VS_OBJECTTYPE_T *tType)
{
    const char *p = NULL;
    int i;
    memset(ext,0,EXT_LN);
    ext[0] = '*';
    ext[1] = 0;
    if(filename==NULL||*filename==0)
    {
        if(tType==NULL){
           return VSA_OK;
        }
        *tType = VS_OT_UNKNOWN;
    }
    if(tType==NULL) {
        return VSA_OK;
    }
    p = strrchr((const char*)filename,(int)'.');
    if(p != NULL && (strlen((const char*)p) > (size_t)2) )
    {
        p++;
        ext[0] = '*';
        ext[1] = '.';
        for(i=0;i<6 && p[i] != 0;i++)
            ext[i+2] = tolower(p[i]);
        if(
            ( (ext[2] == 'e' && ext[3] == 'x' && ext[4] == 'e')  ||
              (ext[2] == 'b' && ext[3] == 'i' && ext[4] == 'n')  ||
              (ext[2] == 'r' && ext[3] == 'a' && ext[4] == 'w')  ||
              (ext[2] == 'r' && ext[3] == 'e' && ext[4] == 'o')  ||
              (ext[2] == 'c' && ext[3] == 'o' && ext[4] == 'm') )
            )
        {
            strcpy((char *)mimetype, (const char *)"application/octet-stream");
            *tType = VS_OT_BINARY;
        }else if(ext[2] == 't' && ext[3] == 'x' && ext[4] == 't')
        {
            strcpy((char *)mimetype,(const char *)"text/plain");
            *tType = VS_OT_TEXT;
        }else if(ext[2] == 's' && ext[3] == 'a' && ext[4] == 'r')
        {
            strcpy((char *)mimetype,(const char *)"application/sar");
            *tType = VS_OT_SAR;
        }else if(ext[2] == 'z' && ext[3] == 'i' && ext[4] == 'p')
        {
            strcpy((char *)mimetype,(const char *)"application/zip");
            *tType = VS_OT_ZIP;
        }else if(ext[2] == 'r' && ext[3] == 'a' && ext[4] == 'r')
        {
            strcpy((char *)mimetype,(const char *)"application/rar");
            *tType = VS_OT_RAR;
        }else if(ext[2] == 'h' && ext[3] == 't' && ext[4] == 'm')
        {
            strcpy((char *)mimetype,(const char *)"text/html");
            *tType = VS_OT_HTML;
        }else if(ext[2] == 'x' && ext[3] == 'm' && ext[4] == 'l')
        {
            strcpy((char *)mimetype,(const char *)"text/xml");
            *tType = VS_OT_XML;
        }else if(ext[2] == 'x' && ext[3] == 's' && ext[4] == 'l')
        {
            strcpy((char *)mimetype,(const char *)"text/xsl");
            *tType = VS_OT_XSL;
        }else if(ext[2] == 'p' && ext[3] == 'd' && ext[4] == 'f')
        {
            strcpy((char *)mimetype,(const char *)"application/pdf");
            *tType = VS_OT_PDF;
        }else if(ext[2] == 'g' && ext[3] == 'i' && ext[4] == 'f')
        {
            strcpy((char *)mimetype,(const char *)"image/gif");
            *tType = VS_OT_GIF;
        }else if(ext[2] == 'j' && ext[3] == 'p' && ( ext[4] == 'g'  || ext[4] == 'e'))
        {
            strcpy((char *)mimetype,(const char *)"image/jpeg");
            *tType = VS_OT_JPEG;
        }else if(ext[2] == 'p' && ext[3] == 'n' && ext[4] == 'g')
        {
            strcpy((char *)mimetype,(const char *)"image/png");
            *tType = VS_OT_PNG;
        }else if(ext[2] == 's' && ext[3] == 'w' && ext[4] == 'f')
        {
            strcpy((char *)mimetype,(const char *)"application/x-shockwave-flash");
            *tType = VS_OT_FLASH;
        }else if(ext[2] == 'x' && ext[3] == 'a' && ext[4] == 'p')
        {
            strcpy((char *)mimetype,(const char *)"application/x-silverlight");
            *tType = VS_OT_SILVERLIGHT;
        }else if(ext[2] == 'r' && ext[3] == 't' && ext[4] == 'f')
        {
            strcpy((char *)mimetype,(const char *)"test/rtf");
            *tType = VS_OT_RTF;
        }else if(ext[2] == 'p' && ext[3] == 's' && ext[4] == '\0')
        {
            strcpy((char *)mimetype,(const char *)"application/postscript");
            *tType = VS_OT_POSTSCRIPT;
        }else if(ext[2] == 'r' && ext[3] == 'a' && ext[4] == 'r')
        {
            strcpy((char *)mimetype,(const char *)"application/rar");
            *tType = VS_OT_RAR;
        }else if(ext[2] == 'j' && ext[3] == 's' && ext[4] == '\0')
        {
            strcpy((char *)mimetype,(const char *)"application/javascript");
            *tType = VS_OT_JSCRIPT;
        }else if(ext[2] == 'j' && ext[3] == 'a' && ext[4] == 'r')
        {
            strcpy((char *)mimetype,(const char *)"application/x-jar");
            *tType = VS_OT_JAR;
        }else if(ext[2] == 'c' && ext[3] == 'l' && ext[4] == 'a' && ext[5] == 's' && ext[6] == 's')
        {
            strcpy((char *)mimetype,(const char *)"application/x-java-class");
            *tType = VS_OT_JAVA;
        }else if(ext[2] == 'e' && ext[3] == 's' && ext[4] == '\0')
        {
            strcpy((char *)mimetype,(const char *)"application/ecmascript");
            *tType = VS_OT_EMCASCRIPT;
        }else if(ext[2] == 'a' && ext[3] == 'l' && ext[4] == 'f')
        {
            strcpy((char *)mimetype,(const char *)"application/x-alf");
            *tType = VS_OT_ALF;
        }else if(ext[2] == 'o' && ext[3] == 't' && ext[4] == 'f')
        {
            strcpy((char *)mimetype,(const char *)"application/x-otf");
            *tType = VS_OT_OTF;
        }else if(ext[2] == 's' && ext[3] == 'i' && ext[4] == 'm')
        {
            strcpy((char *)mimetype,(const char *)"application/x-sim");
            *tType = VS_OT_SIM;
        }else if(ext[2] == 'x' && ext[3] == 'l' && ext[4] == 's' && ext[5] == 'x')
        {
            strcpy((char *)mimetype,(const char *)"application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
            *tType = VS_OT_MSO;
        }else if(ext[2] == 'x' && ext[3] == 'l' && ext[4] == 's')
        {
            strcpy((char *)mimetype,(const char *)"application/vnd.ms-excel");
            *tType = VS_OT_MSO;
        }else if(ext[2] == 'd' && ext[3] == 'o' && ext[4] == 'c' && ext[5] == 'x')
        {
            strcpy((char *)mimetype,(const char *)"application/vnd.openxmlformats-officedocument.wordprocessingml.document");
            *tType = VS_OT_MSO;
        }else if(ext[2] == 'd' && ext[3] == 'o' && ext[4] == 'c')
        {
            strcpy((char *)mimetype,(const char *)"application/msword");
            *tType = VS_OT_MSO;
        }else if(ext[2] == 'f' && ext[3] == 'l' && ext[4] == 'v')
        {
            strcpy((char *)mimetype,(const char *)"video/x-flv");
            *tType = VS_OT_FLASHVIDEO;
        }else if(ext[2] == 'k' && ext[3] == 'e' && ext[4] == 'p')
        {
            strcpy((char *)mimetype,(const char *)"application/x-kep");
            *tType = VS_OT_KEP;
        }else if(ext[2] == 'i' && ext[3] == 'n' && ext[4] == 'i')
        {
            strcpy((char *)mimetype,(const char *)"application/x-ini");
            *tType = VS_OT_INI;
        }else if(ext[2] == 's' && ext[3] == 'a' && ext[4] == 'p')
        {
            strcpy((char *)mimetype,(const char *)"application/x-sapshortcut");
            *tType = VS_OT_SAPSHORTCUT;
        }else
        {
            strcpy((char *)mimetype,(const char *)"unknown/unknown");
            *tType = VS_OT_UNKNOWN;
        }

    }
    return VSA_OK;
} /* getFileType */

static VSA_RC getByteType(PByte pByte, size_t lByte, PChar ext, PChar mimetype, USRDATA *pUsrData)
{
typedef enum TYPE_STATUS {
    UNKNOWN,
    BEGIN,
    SEARCH,
    LOOKAHEAD
} TYPE_STATUS;

    PByte  ptr;
    /*size_t scanSize = lByte > 65536 ? 65536 : lByte;*/
    Bool   text = TRUE;
    VSA_RC rc  = VSA_OK;
    VS_OBJECTTYPE_T _type = VS_OT_UNKNOWN;
    VS_OBJECTTYPE_T _tEnd = VS_OT_UNKNOWN;
    UInt   i;
    TYPE_STATUS status = BEGIN;

    if(pUsrData == NULL) {
       return VSA_E_NULL_PARAM;
    }
    if(pByte == NULL || lByte == 0) {
        addScanError(0,(PChar)"getByteType",0,-1,(PChar)"No byte array passed",pUsrData->pScanInfo->uiScanErrors++,&pUsrData->pScanInfo->pScanError);
        CLEANUP(VSA_E_INVALID_PARAM);
    }
    for(i=0; i < lByte; i++)
    {
        ptr = pByte+i;
        if(text == TRUE) {
           if(0==isprint((int)*ptr)) {
               int n = (int)*ptr;
               if(n == '\n' || n == '\r' || n == '\t' || n == EOF) continue;
               text = FALSE;
               if(i > 0 && pUsrData->tFileType == VS_OT_UNKNOWN && _type == VS_OT_UNKNOWN) goto cleanup;
           }
        }
        switch( status )
        {
        case BEGIN:
            switch(*ptr)
            {
            case '%':
                if(lByte > 7 && 0==memcmp(ptr,"%PDF-",5)) _type = VS_OT_PDF;
                if(lByte > 15 && 0==memcmp(ptr,"%!PS-Adobe",10)) _type = VS_OT_POSTSCRIPT;
                break;
            case '\004':
                if(lByte > 16 && 0==memcmp(ptr,"\004%!PS-Adobe",11)) _type = VS_OT_POSTSCRIPT;
                break;
            case '<':
                if(lByte > 10 && 0==memcmp(ptr,"<?xml",5)) _type = VS_OT_XML;
                else if(lByte > 10 && 0==memcmp(ptr,"<html",5)) _type = VS_OT_HTML;
                else if(lByte > 10 && 0==memcmp(ptr,"<HTML",5)) _type = VS_OT_HTML;
                else _type  = VS_OT_XHTML;
                status = SEARCH;
                break;
            case '\\':
                if(lByte > 6 && 0==memcmp(ptr,"\\rtf",4)) { _tEnd = _type = VS_OT_RTF; CLEANUP(VSA_OK); }
                break;
            case 'P':
                if(lByte > 6 && 0==memcmp(ptr,"PK\003\004",4))  _type = VS_OT_ZIP;
                else if(lByte > 5 && 0==memcmp(ptr,"P\002\000",3)) { _tEnd = _type = VS_OT_KEP; CLEANUP(VSA_OK); }
                break;
            case 'R':
                if(lByte > 10 && 0==memcmp(ptr,"Rar!",4)) { _tEnd = _type = VS_OT_RAR; CLEANUP(VSA_OK); }
                break;
            case 'C':
                if(lByte > 10 && 0==memcmp(ptr,"CAR 2.0",7)) { _tEnd = _type = VS_OT_SAR; CLEANUP(VSA_OK); }
                if(lByte > 10 && 0==memcmp(ptr,"CWS",3)) { _tEnd = _type = VS_OT_FLASH; CLEANUP(VSA_OK); }
                break;
            case 'F':
                if(lByte > 10 && 0==memcmp(ptr,"FWS",3)) { _tEnd = _type = VS_OT_FLASH; CLEANUP(VSA_OK); }
                if(lByte > 10 && 0==memcmp(ptr,"FLV",3)) { _tEnd = _type = VS_OT_FLASHVIDEO; CLEANUP(VSA_OK); }
                break;
            case '\x89':
                if(lByte > 5 && 0==memcmp(ptr,"\x89PNG",4)) { _tEnd = _type = VS_OT_PNG; CLEANUP(VSA_OK); }
                break;
            case 'G':
                if(lByte > 5 && 0==memcmp(ptr,"GIF8",4)) { _tEnd = _type = VS_OT_GIF; CLEANUP(VSA_OK); }
                if(lByte > 5 && 0==memcmp(ptr,"GIF",3)) { _tEnd = _type = VS_OT_IMAGE; CLEANUP(VSA_OK); }
                break;
            case 'i':
                if(lByte > 5 && 0==memcmp(ptr,"iTut",4)) { _tEnd = _type = VS_OT_SIM; CLEANUP(VSA_OK); }
                break;
            case '\377':
                if(lByte > 5 && 0==memcmp(ptr,"\377\330\377\340",4)) { _tEnd = _type = VS_OT_JPEG; CLEANUP(VSA_OK); }
                if(lByte > 5 && 0==memcmp(ptr,"\377\330\377\356",4)) { _tEnd = _type = VS_OT_JPEG; CLEANUP(VSA_OK); }
                if(lByte > 5 && 0==memcmp(ptr,"\377\330",2)) { _tEnd = _type = VS_OT_IMAGE; CLEANUP(VSA_OK); }
                break;
            case '\xca':
                if(lByte > 5 && 0==memcmp(ptr,"\xca\xfe\xba\xbe",4)) { _tEnd = _type = VS_OT_JAVA; CLEANUP(VSA_OK); }
                break;
            default: break;
            }
            status = SEARCH;
            break;
        case SEARCH:
            switch(*ptr)
            {
            case '%':
                status = LOOKAHEAD;
                break;
            case '<':
                if(_type != VS_OT_UNKNOWN) continue;
                if((lByte - i ) > 5 && 0==memcmp(ptr,"<?xml",5)) { _type = VS_OT_XML; }
                else if(_type == VS_OT_XML && (lByte - i ) > 5 && 0==memcmp(ptr,"</xml>",6)) { _tEnd = VS_OT_XML; CLEANUP(VSA_OK); }
                else if(_type == VS_OT_HTML && (lByte - i ) > 5 && 0==memcmp(ptr,"</html>",7)) { _tEnd = VS_OT_HTML; CLEANUP(VSA_OK); }
                else if(_type == VS_OT_HTML && (lByte - i ) > 5 && 0==memcmp(ptr,"</HTML>",7)) { _tEnd = VS_OT_HTML; CLEANUP(VSA_OK); }
                else _type = VS_OT_XHTML;
                break;
            case '\\':
                if(_type != VS_OT_UNKNOWN) continue;
                if(lByte > 6 && 0==memcmp(ptr,"\\rtf",4)) _type = VS_OT_RTF;
                break;
            case 'M':
                if(_type == VS_OT_ZIP && (pUsrData->tFileType == VS_OT_JAR || pUsrData->tFileType == VS_OT_MSO) ) {
                   if((lByte - i ) > 10 && 0==memcmp(ptr,"META-INF/",9)) { _tEnd = _type = VS_OT_JAR; CLEANUP(VSA_OK); }
                   if((lByte - i ) > 12 && 0==memcmp(ptr,"MANIFEST.MF",11)) { _tEnd = _type = VS_OT_JAR; CLEANUP(VSA_OK); }
                   if((lByte - i ) > 7 && 0==memcmp(ptr,".class",6)) { _tEnd = _type = VS_OT_JAR; CLEANUP(VSA_OK); }
                }
                break;
            case 'A':
                if(_type == VS_OT_ZIP) {
                   if((lByte - i ) > 17 && 0==memcmp(ptr,"AppManifest.xaml",16)) { _tEnd = _type = VS_OT_SILVERLIGHT; CLEANUP(VSA_OK); }
                }
                break;
            case 'c':
                if(_type == VS_OT_ZIP) {
                   if((lByte - i ) > 6 && 0==memcmp(ptr-1,".class",6)) { _tEnd = _type = VS_OT_JAR; CLEANUP(VSA_OK); }
                }
                break;
            case '[':
                if(_type == VS_OT_ZIP) {
                   if((lByte - i ) > 20 && 0==memcmp(ptr,"[Content_Types].xml",19)) { _tEnd = _type = VS_OT_MSO; CLEANUP(VSA_OK); }
                }
                break;
            default: break;
            }
            break;
        case LOOKAHEAD:
            switch(*ptr)
            {
            case '%':
                if( (_type == VS_OT_POSTSCRIPT || _type == VS_OT_PDF )
                    && (lByte - i ) > 3 
                    && 0==memcmp(ptr,"%EOF",4)) 
                {
                    _tEnd = _type;
                }
                break;
            default: status = SEARCH; break;
            }
            break;
        default:
            status = UNKNOWN;
            break;
        }
    }
cleanup:
    if(rc != VSA_OK) {
       strcpy((char *)mimetype,(const char *)"unknown/unknown");
       strcpy((char *)ext,".*");
       pUsrData->tObjectType = VS_OT_UNKNOWN;
       rc = VSA_E_NOT_SCANNED;
    } else {
       if(_tEnd == VS_OT_UNKNOWN && text==TRUE) {
          if(_type ==VS_OT_UNKNOWN && pUsrData->tFileType >= VS_OT_TEXT && pUsrData->tFileType < VS_OT_IMAGE) {
               _tEnd = _type = pUsrData->tFileType;
          } else if(_type >= VS_OT_TEXT && _type < VS_OT_IMAGE) {
            if(_type != VS_OT_UNKNOWN)
               _tEnd = _type; /* in case of textual objects the extension can be overtaken */
            else
               _tEnd = _type = pUsrData->tFileType;
          }
       }
       if(_type != _tEnd) {
           if(text==TRUE) {
             if(_type > VS_OT_UNKNOWN && _type < VS_OT_IMAGE && _tEnd > VS_OT_UNKNOWN && _tEnd < VS_OT_IMAGE)
                _type = pUsrData->tFileType;
             else
                _type = VS_OT_UNKNOWN;
           } else {
             if(_type == VS_OT_ZIP && _tEnd == VS_OT_UNKNOWN)
                 _type = VS_OT_ZIP;
             else
                 _type = VS_OT_BINARY;
           }
       }
       pUsrData->tObjectType = _type;
       switch(_type) {
        case VS_OT_BINARY:
           strcpy((char *)mimetype,(const char *)"application/octet stream");
           strcpy((char *)ext,(const char *)".bin");
           break;
        case VS_OT_PDF:
           strcpy((char *)mimetype,(const char *)"application/pdf");
           strcpy((char *)ext,(const char *)".pdf");
           break;
        case VS_OT_POSTSCRIPT:
           strcpy((char *)mimetype,(const char *)"application/postscript");
           strcpy((char *)ext,(const char *)".ps");
           break;
        case VS_OT_SAR:
           strcpy((char *)mimetype,(const char *)"application/sar");
           strcpy((char *)ext,(const char *)".sar");
           break;
        case VS_OT_RAR:
           strcpy((char *)mimetype,(const char *)"application/rar");
           strcpy((char *)ext,(const char *)".rar");
           break;
        case VS_OT_ZIP:
        case VS_OT_BZIP2:
        case VS_OT_GZIP:
            strcpy((char *)mimetype,(const char *)"application/zip");
            strcpy((char *)ext,(const char *)".zip");
            break;
        case VS_OT_XHTML:
           strcpy((char *)mimetype,(const char *)"application/xhtml+xml");
           strcpy((char *)ext,(const char *)".xhtml");
           break;
        case VS_OT_HTML:
           strcpy((char *)mimetype,(const char *)"text/html");
           strcpy((char *)ext,(const char *)".html");
           break;
        case VS_OT_TEXT:
           strcpy((char *)mimetype,(const char *)"text/plain");
           strcpy((char *)ext,(const char *)".txt");
           break;
        case VS_OT_FLASH:
           strcpy((char *)mimetype,(const char *)"application/x-shockwave-flash");
           strcpy((char *)ext,(const char *)".swf");
           break;
        case VS_OT_FLASHVIDEO:
           strcpy((char *)mimetype,(const char *)"video/x-flv");
           strcpy((char *)ext,(const char *)".flv");
           break;
        case VS_OT_IMAGE:
           strcpy((char *)mimetype,"image/*");
           strcpy((char *)ext,".jpg");
           break;
        case VS_OT_GIF:
            strcpy((char *)mimetype,"image/gif");
            strcpy((char *)ext,".gif");
            break;
        case VS_OT_PNG:
           strcpy((char *)mimetype,(const char *)"image/png");
           strcpy((char *)ext,(const char *)".png");
           break;
        case VS_OT_JPEG:
           strcpy((char *)mimetype,(const char *)"image/jpeg");
           strcpy((char *)ext,(const char *)".jpg");
           break;
        case VS_OT_SILVERLIGHT:
           strcpy((char *)mimetype,(const char *)"application/x-silverlight");
           strcpy((char *)ext,(const char *)".xap");
           break;
        case VS_OT_JSCRIPT:
            strcpy((char *)mimetype,(const char *)"application/javascript");
            strcpy((char *)ext,(const char *)".js");
            break;
        case VS_OT_EMCASCRIPT:
            strcpy((char *)mimetype,(const char *)"application/ecmascript");
            strcpy((char *)ext,(const char *)".es");
            break;
        case VS_OT_JAR:
            strcpy((char *)mimetype,(const char *)"application/x-jar");
            strcpy((char *)ext,(const char *)".jar");
            break;
        case VS_OT_JAVA:
            strcpy((char *)mimetype,(const char *)"application/x-java-class");
            strcpy((char *)ext,(const char *)".class");
            break;
        case VS_OT_ALF:
            strcpy((char *)mimetype,(const char *)"application/x-alf");
            strcpy((char *)ext,(const char *)".alf");
            break;
        case VS_OT_OTF:
            strcpy((char *)mimetype,(const char *)"application/x-otf");
            strcpy((char *)ext,(const char *)".oft");
            break;
        case VS_OT_MSO:
            strcpy((char *)mimetype,(const char *)"application/office");
            strcpy((char *)ext,(const char *)".doc");
            break;
        case VS_OT_KEP:
            strcpy((char *)mimetype,(const char *)"application/x-kep");
            strcpy((char *)ext,(const char *)".kep");
            break;
        case VS_OT_SIM:
            strcpy((char *)mimetype,(const char *)"application/x-sim");
            strcpy((char *)ext,(const char *)".sim");
            break;
        case VS_OT_SAPSHORTCUT:
            strcpy((char *)mimetype,(const char *)"application/x-sapshortcut");
            strcpy((char *)ext,(const char *)".sap");
            break;
        case VS_OT_INI:
            strcpy((char *)mimetype,(const char *)"text/x-ini");
            strcpy((char *)ext,(const char *)".ini");
            break;
        default:
           strcpy((char *)mimetype,(const char *)"unknown/unknown");
           strcpy((char *)ext,(const char *)".*");
           pUsrData->tObjectType = VS_OT_UNKNOWN;
           break;
       }
    }
    return rc;
} /* getByteType */

static char* memstr(char* input, const unsigned char* search, int len_input, int len_search)
{
    int i;
    for(i = 0; (memcmp(input, search, len_search)) && (i != len_input-len_search); input++,i++)
        ;
    return (i == len_input-len_search)?0:input;
} /* memstr */

static VSA_RC check4ActiveContent(UInt            uiJobID,
                                  PChar           pszObjectName,
                                  PByte           pObject,
                                  size_t          lObjectSize,
                                  USRDATA        *pUsrData)
{    
    Byte   c;
    char  *p   = NULL;
    char  *str = (char*)pObject;
    size_t snifflen = lObjectSize > 1024 ? 1024:lObjectSize;

    if(pObject==NULL) return VSA_OK;
    
    if(pUsrData->tObjectType > VS_OT_UNKNOWN && pUsrData->tObjectType < VS_OT_IMAGE)
    {
       c = *(pObject+lObjectSize);
       *(pObject+lObjectSize) = 0;
       p = strstr(str,(const char*)"<script");
       if(p==NULL)
       p = strstr(str,(const char*)"<applet");
       if(p==NULL)
       p = strstr(str,(const char*)"<object");
       if(p==NULL)
       p = strstr(str,(const char*)"<embed");
       *(pObject+lObjectSize) = c;
    }
    else if(pUsrData->tObjectType == VS_OT_PDF)
    {
            c = *(pObject+lObjectSize);
            *(pObject+lObjectSize) = 0;
            p = strstr(str,(const char*)"/JS");
            if(p==NULL)
            p = strstr(str,(const char*)"/OpenAction");
            *(pObject+lObjectSize) = c;
    }
    else if(pUsrData->tObjectType == VS_OT_MSO)
    {
            c = *(pObject+lObjectSize);
            *(pObject+lObjectSize) = 0;
            p = memstr(str,(const unsigned char*)".class", lObjectSize,6);
            if(p==NULL)
            p = memstr(str,(const unsigned char*)"vbaProject.bin", lObjectSize,14);
            *(pObject+lObjectSize) = c;
    }
    else
    {   
        c = *(pObject+snifflen);
        *(pObject+snifflen) = 0;
        p = strstr(str,(const char*)"<script");
        if(p==NULL)
        p = strstr(str,(const char*)"<applet");
        if(p==NULL)
        p = strstr(str,(const char*)"<object");
        if(p==NULL)
        p = strstr(str,(const char*)"<embed");
        *(pObject+snifflen) = c;
    }
    if(p != NULL)
    {
        addVirusInfo(uiJobID,
                     pszObjectName,
                     lObjectSize,
                     FALSE,
                     VS_DT_ACTIVECONTENT,
                     VS_VT_PUA,
                     pUsrData->tObjectType,
                     VS_AT_BLOCKED,
                     0,
                     (PChar)"Embedded script found",
                     (PChar)"Active content block",
                     pUsrData->pScanInfo->uiInfections++,
                     &pUsrData->pScanInfo->pVirusInfo);
        return VSA_E_ACTIVECONTENT_FOUND;
    }
    else
    {
        return VSA_OK;
    }
} /* check4ActiveContent */

static VSA_RC checkContentType(UInt            uiJobID,
                               PChar           pszObjectName,
                               size_t          lObjectSize,
                               PChar           pExtension,
                               PChar           pMimeType,
                               USRDATA        *pUsrData)
{
    char   *p           = NULL;
    VSA_RC  rc          = VSA_OK;
    PChar   errfreename = NULL;
    char   *str         = NULL;
    Char    errname[1024];

    if(pUsrData->pszScanMimeTypes != NULL)
    {
       str = (char*)pUsrData->pszScanMimeTypes;
       p = strstr(str,(const char*)pMimeType);
       if(p==NULL)
       {
           sprintf((char*)errname,"MIME type %s is not allowed (whitelist %s)",pMimeType,(const char*)pUsrData->pszScanMimeTypes);
           errfreename = (PChar)"Check SCANMIMETYPES parameter";
           CLEANUP(VSA_E_BLOCKED_BY_POLICY);
       }
    }
    if(pUsrData->pszBlockMimeTypes != NULL)
    {
       str = (char*)pUsrData->pszBlockMimeTypes;
       p = strstr(str,(const char*)pMimeType);
       if(p!=NULL)
       {
           sprintf((char*)errname,"MIME type %s is not allowed (blacklist %s)",pMimeType,(const char*)pUsrData->pszBlockMimeTypes);
           errfreename = (PChar)"Check BLOCKMIMETYPES parameter";
           CLEANUP(VSA_E_BLOCKED_BY_POLICY);
       }
    }
    if(pUsrData->pszScanExtensions != NULL)
    {
       str = (char*)pUsrData->pszScanExtensions;
       p = strstr(str,(const char*)pExtension);
       if(p==NULL)
       {
           sprintf((char*)errname,"File extension %s is not allowed (whitelist %s)",pExtension,(const char*)pUsrData->pszScanExtensions);
           errfreename = (PChar)"Check SCANEXTENSIONS parameter";
           CLEANUP(VSA_E_BLOCKED_BY_POLICY);
       }
    }
    if(pUsrData->pszBlockExtensions != NULL)
    {
       str = (char*)pUsrData->pszBlockExtensions;
       p = strstr(str,(const char*)pExtension);
       if(p==NULL)
       {
           sprintf((char*)errname,"File extension %s is not allowed (blacklist %s)",pExtension,(const char*)pUsrData->pszBlockExtensions);
           errfreename = (PChar)"Check BLOCKEXTENSIONS parameter";
           CLEANUP(VSA_E_BLOCKED_BY_POLICY);
       }
    }
cleanup:
    if(rc){
       addVirusInfo(uiJobID,
                    pszObjectName,
                    lObjectSize,
                    FALSE,
                    VS_DT_MIMEVALIDATION,
                    VS_VT_NOVIRUS,
                    pUsrData->tObjectType,
                    VS_AT_BLOCKED,
                    0,
                    errname,
                    errfreename,
                    pUsrData->pScanInfo->uiInfections++,
                    &pUsrData->pScanInfo->pVirusInfo);
       rc = VSA_E_BLOCKED_BY_POLICY;
    }
    return rc;
} /* checkContentType */

static VSA_RC registerCallback(VSA_CALLBACK *p_callback, USRDATA *usrdata)
{

    VSA_RC rc = VSA_OK;

    /* initialize the helper structure USRDATA */
    memset(usrdata,0,sizeof(USRDATA));

    if (p_callback != NULL)
    {
        if (p_callback->struct_size != sizeof(VSA_CALLBACK))
            CLEANUP(VSA_E_INVALID_PARAM);
        if (p_callback->pEventCBFP != NULL && p_callback->uiEventMsgFlags != 0)
        {
            if ((UInt)VS_M_ALL > (p_callback->uiEventMsgFlags) )  /* only known messages */
            {
                usrdata->uiMsgFlags = p_callback->uiEventMsgFlags;
            }
            else if ( (p_callback->uiEventMsgFlags) == uigVS_SAP_ALL )
            {
                usrdata->uiMsgFlags = uigVS_SAP_ALL;     /* constant for all messages */
                
            }
            else
            {
                CLEANUP(VSA_E_NOT_SUPPORTED);            /* wished messages not supported */
            }
            usrdata->pvFncptr   = p_callback->pEventCBFP;            
            usrdata->pvUsrdata  = p_callback->pvUsrData;
        }
        if ( p_callback->pEventCBFP       == NULL ||  
             p_callback->uiEventMsgFlags == 0
            )           
        {

            usrdata->pvFncptr          = NULL;
            usrdata->uiMsgFlags        = 0;
                  /* may somebody forgot this feature to set MsgFlags?
                   * anyway you want handle this (trace or error), i deactivate
                   * here the CALLBACK mechanism
                   */
        }
    }       

cleanup:
    return (rc);
} /* registerCallback */

static void freeVSA_INIT(VSA_INIT **pp_init)
{
  if(pp_init != NULL && (*pp_init) != NULL)
  {
    /* cleanup the driver structures */
    if ((*pp_init) != NULL && (*pp_init)->pDriver != NULL){
        int i = 0;
        for (i=0; i<(*pp_init)->usDrivers; i++)
        {
            if ((*pp_init) != NULL && (*pp_init)->pDriver != NULL && (*pp_init)->pDriver[i].pszName != NULL){
                free((*pp_init)->pDriver[i].pszName);   
                (*pp_init)->pDriver[i].pszName = NULL;
            }
        }
        free((*pp_init)->pDriver);
        (*pp_init)->pDriver = NULL;
    }    
    if ((*pp_init) != NULL && (*pp_init)->pszEngineVersionText != NULL)
        free((*pp_init)->pszEngineVersionText);
    if ((*pp_init) != NULL && (*pp_init)->pszErrorText != NULL)
        free((*pp_init)->pszErrorText);

    /* free the rest of the p_init */
    if ((*pp_init) != NULL) {
        free((*pp_init));    
    }
    /* set now to NULL */
    (*pp_init) = NULL;

    /* dec. the ref counter */
    if(lgRefCounter > (size_t)0)
        lgRefCounter--;
  }
} /* freeVSA_INIT */

static void freeVSA_CONFIG(VSA_CONFIG **pp_config)
{
  if(pp_config != NULL && (*pp_config) != NULL)
  {
    if ((*pp_config)->pInitParams != NULL && (*pp_config)->pInitParams->pInitParam != NULL)
    {
        int i=0;
        for(i=0; i<(*pp_config)->pInitParams->usInitParams; i++)
        {
            if ((*pp_config)->pInitParams->pInitParam[i].tType == VS_TYPE_CHAR &&
                (*pp_config)->pInitParams->pInitParam[i].pvValue != NULL)
                    free(((*pp_config)->pInitParams->pInitParam[i].pvValue));
        }
        free((*pp_config)->pInitParams->pInitParam);
    }
    if ((*pp_config)->pInitParams != NULL)
        free((*pp_config)->pInitParams);
    if ((*pp_config)->pOptParams != NULL && (*pp_config)->pOptParams->pOptParam != NULL)
    {
        int i=0;
        for(i=0; i<(*pp_config)->pOptParams->usOptParams; i++)
        {
            if ((*pp_config)->pOptParams->pOptParam[i].tType == VS_TYPE_CHAR &&
                (*pp_config)->pOptParams->pOptParam[i].pvValue != NULL)
                    free((*pp_config)->pOptParams->pOptParam[i].pvValue);
        }
        free((*pp_config)->pOptParams->pOptParam);
    }
    if ((*pp_config)->pOptParams != NULL)
        free((*pp_config)->pOptParams);

    if ((*pp_config)->pAdapterInfo != NULL )
    {
        if((*pp_config)->pAdapterInfo->pszVendorInfo != NULL)
          free((*pp_config)->pAdapterInfo->pszVendorInfo);
        if((*pp_config)->pAdapterInfo->pszAdapterName != NULL)
          free((*pp_config)->pAdapterInfo->pszAdapterName);
        /* free adapter info structure */
        free((*pp_config)->pAdapterInfo);
        (*pp_config)->pAdapterInfo = NULL;
    }
    
    free(*pp_config);
    /* set now to NULL */
    (*pp_config) = NULL;
  }
} /* freeVSA_CONFIG */

static void freecontentinfo(VSA_CONTENTINFO **vsacontent)
{
    if (vsacontent != NULL && (*vsacontent) != NULL) {
      if ((*vsacontent)->pszObjectName != NULL) {
          free((*vsacontent)->pszObjectName);
          (*vsacontent)->pszObjectName = NULL;
      }
      if ((*vsacontent)->pszExtension!= NULL) {
          free((*vsacontent)->pszExtension);
          (*vsacontent)->pszExtension= NULL;
      }
      if ((*vsacontent)->pszContentType!= NULL) {
          free((*vsacontent)->pszContentType);
          (*vsacontent)->pszContentType= NULL;
      }
      if ((*vsacontent)->pszCharSet!= NULL) {
          free((*vsacontent)->pszCharSet);
          (*vsacontent)->pszCharSet= NULL;
      }
      if ((*vsacontent) != NULL) {
          free((*vsacontent));
          (*vsacontent) = NULL;
      }
    }
} /* freecontentinfo */

static void freecontentinfo2(VSA_CONTENTINFO *vsacontent)
{
    if (vsacontent != NULL) {
      if (vsacontent->pszObjectName != NULL) {
          free(vsacontent->pszObjectName);
          vsacontent->pszObjectName = NULL;
      }
      if (vsacontent->pszExtension!= NULL) {
          free(vsacontent->pszExtension);
          vsacontent->pszExtension= NULL;
      }
      if (vsacontent->pszContentType!= NULL) {
          free(vsacontent->pszContentType);
          vsacontent->pszContentType= NULL;
      }
      if (vsacontent->pszCharSet!= NULL) {
          free(vsacontent->pszCharSet);
          vsacontent->pszCharSet= NULL;
      }
    }
} /* freecontentinfo2 */

static void freevirusinfo(VSA_VIRUSINFO **vsainf)
{
  if(vsainf != NULL && (*vsainf) != NULL) {
    if ((*vsainf)->pszVirusName != NULL) {
        free((*vsainf)->pszVirusName);
        (*vsainf)->pszVirusName = NULL;
    }
    if ((*vsainf)->pszObjectName != NULL){
        free((*vsainf)->pszObjectName);
        (*vsainf)->pszObjectName = NULL;
    }
    if ((*vsainf)->pszFreeTextInfo != NULL){
        free((*vsainf)->pszFreeTextInfo);
        (*vsainf)->pszFreeTextInfo = NULL;
    } 
    if ((*vsainf) != NULL) {
        free((*vsainf));
        (*vsainf) = NULL;
    }
  }
} /* freevirusinfo */

static void freevirusinfo2(VSA_VIRUSINFO *vsainf)
{
  if(vsainf != NULL ) {
    if ((*vsainf).pszVirusName != NULL) {
        free((*vsainf).pszVirusName);
        (*vsainf).pszVirusName = NULL;
    }
    if ((*vsainf).pszObjectName != NULL){
        free((*vsainf).pszObjectName);
        (*vsainf).pszObjectName = NULL;
    }
    if ((*vsainf).pszFreeTextInfo != NULL){
        free((*vsainf).pszFreeTextInfo);
        (*vsainf).pszFreeTextInfo = NULL;
    }    
  }
} /* freevirusinfo2 */

static void freescanerror(VSA_SCANERROR **vsierror)
{
  if (vsierror != NULL && (*vsierror) != NULL) {
    if ((*vsierror)->pszObjectName != NULL) {
        free((*vsierror)->pszObjectName);
        (*vsierror)->pszObjectName = NULL;
    }
    if ((*vsierror)->pszErrorText!= NULL) {
        free((*vsierror)->pszErrorText);
        (*vsierror)->pszErrorText= NULL;
    }
    if ((*vsierror) != NULL) {
        free((*vsierror));
        (*vsierror) = NULL;
    }
  }
} /* freescanerror */

static void freescanerror2(VSA_SCANERROR *vsierror)
{
  if (vsierror != NULL ) {
    if ((*vsierror).pszObjectName != NULL) {
        free((*vsierror).pszObjectName);
        (*vsierror).pszObjectName = NULL;
    }
    if ((*vsierror).pszErrorText!= NULL) {
        free((*vsierror).pszErrorText);
        (*vsierror).pszErrorText= NULL;
    }

  }
} /* freescanerror2 */
