import java.util.Iterator;
import java.util.Vector;

import com.sap.idm.vds.*;


/**
 * Common class for all the test cases
 * 
 * @author I054742
 *
 */
abstract public class TestCase {
	
	/* Initial set of illegal attribute names */
	static protected String[] FAKE_ATTRIBUTES = {"ajsdopfaspodfmopasdooookmcj","mbaoujckjjuehfahhsill","lasldifnbaskjdkflufj","nsadf00q34asfdasdfoeeo","jvnjfduyu7eyhfh048u","0a9erja90weasdffeegggg000","0927420930942304923478"};
	// old fake attributes {"ijjaskkjf", "", "FAKEATTRIBUTE084kkhmdnpawbiiksqppah76", "      \r\r\n", "adiikdnfnerynopossiblekdkkk", "jjdskakrqqskiek  kdk fknkdkzk", "superfake alldoo llafsdllf oasldl f","fakestnv0910ijvah09hqegjfkkdiikekkdk098858484","ijjaskkjf", "", "FAKEATTRIBUTE084kkhmdnpawbiiksqppah76", "      \r\r\n", "adiikdnfnerynopossiblekdkkk", "jjdskakrqqskiek  kdk fknkdkzk","hjlasdjl98598599i3ki","0300h09j09j39399494kjjvjf","njfsadfisahdfaheihfweainfakljsdofow","nasjdfoasdnfiaosheiuruuruyyqqqkgjgjhjf"};
	/* Definitive set of illegal attribute names */
	static protected String[] FAKE_ATTRIBUTE_NAMES;
	/* Example values used to create fake attribute values */
	static protected String[] EXAMPLE_VALUES = {"ajsdopfaspodfmopasdooookmcj","mbaoujckjjuehfahhsill","lasldifnbaskjdkflufj","nsadf00q34asfdasdfoeeo","jvnjfduyu7eyhfh048u","0a9erja90weasdffeegggg000","0927420930942304923478"};
	//{"", " ", "\n", "wdfgtgnotasdfhing", ",", "an exaklkijhjdmple", "and no more ldojghjmkiksj"};
	/* Indicates if the set of fake attribute names has been initialized */
	static private boolean initializated = false;
	
	static protected MyHashMap NEEDED_PROPS = new MyHashMap () {{}};
	static MyHashMap getNEEDED_PROPS () {return NEEDED_PROPS;}
	
	static protected final int STATIC_MAX_NUMB_OF_TRIES_FOR_ADDING_NEW_ENTRY = 10;
	
	static private MyHashMap numbOfExecTC = null;
	
	protected MyAttribute att = null;
	protected Entry ent = null;
	protected EntrySet modified = null;
	protected boolean skipTest = false;
	
	public TestCase (MyAttribute att, Entry ent, EntrySet modified) {
		this.att = att;
		this.ent = ent;
		this.modified = modified;
		if (initializated==false) {
			initFakeAttributes();
			resetNumbOfExecTC();
			initializated = true;
		}
	}
	
	abstract void calculateVals ();
	
	abstract void calculateExpected ();
	
	abstract MVDOperationResult executeCase ();
	
	abstract boolean checkEquality();
	
	abstract String getTestCaseId();
	
	abstract int getPriority();
	
	abstract MyHashMap getProps ();
	
	
	final void setParams (MyAttribute att, Entry ent, EntrySet modified) {
		this.att = att;
		this.ent = ent;
		this.modified = modified;
	}
	
	
	final boolean test () {
		this.countTC();
		this.calculateVals();
		if (this.skipTest) {
			this.skipTest = false;
			return true;
		}
		this.calculateExpected();
		this.executeCase();
		return this.checkEquality();
	}
	
	
	final void justExecute () {
		this.countTC();
		this.calculateVals();
		this.calculateExpected();
		this.executeCase();
	}
	
	
	protected void countTC () {
		
		String tcId = this.getClass().getName();
		
		if (numbOfExecTC.containsKey(tcId)==false) {
			numbOfExecTC.put(tcId,new Integer(1));
		}
		else{
			int nOfTC = ((Integer)numbOfExecTC.get(tcId)).intValue();
			numbOfExecTC.put(tcId,new Integer(nOfTC+1));
		}
	}
	
	
	static MyHashMap getNumbOfExecTC () {
		
		return numbOfExecTC;
	}
	
	
	String writePreData () {
		String res = "entry: "+this.ent.writeData();
		res+=" - attribute: "+(this.att==null?null:this.att.writeData());
		return res;
	}
	
	
	/**
     * Creates an entry corresponding with the current one in VDS corresponding with the DN 'dn'
     * @param dn The DN
     * @return The entry
     */
	protected Entry pickUpEntry (String dn) {
    	
    	Entry res = new Entry(dn,new Vector());
    	MVDSearchResults sr = Globals.ldapm.search(dn,LDAPManagement.BASE,Globals.attr,
    											   Globals.urlfilter,Globals.szLimit,Globals.tmLimit,true);
    	res.setAttributes(sr);
    	
    	return res;
    }
    
    
    /**
     * Creates an attribute corresponding with the attribute which name is 'name' and is part of the entry 
     * identify in VDS by DN 'dn'
     * @param dn The DN that identify the entry
     * @param name The attribute name
     * @return The attribute
     */
	protected MyAttribute pickUpAttribute (String dn, String name) {
		
		MVDSearchResults sr = Globals.ldapm.search(dn,LDAPManagement.BASE,Globals.attr,
												   Globals.urlfilter,Globals.szLimit,Globals.tmLimit,true);
		
		if (sr==null) return null;
		if (sr.size()<=0) return null;
		MVDSearchResultEntry sre = (MVDSearchResultEntry)sr.get(0);
		
		return new MyAttribute(name,(sre.getAttrValues(name)!=null?sre.getAttrValues(name):new Vector()),null);
	}
	
	
	/**
	 * Checks if the set of entries 'modified' is the same one that the set of entries directly under the
	 * stating point 'sPoint'
	 * @param modified Set of entries
	 * @param sPoint Starting point
	 * @return 'true' if the expected results matches with the real state in every moment
	 * 		   'false' otherwise
	 */
	protected boolean checkAddAndDeleteEntryResult (EntrySet modified, String sPoint, String dn, String expected) {

		Loggin.Log(Globals.indent()+"Checking if the entry was added or deleted properly");
		
		String collateralLevel = GeneralConfiguration.getCollateralEffectsChecking();
		if (modified!=null && collateralLevel.equalsIgnoreCase("hard")) {
			
			MVDSearchResults sr = Globals.ldapm.search(sPoint,LDAPManagement.ONE_LEVEL,Globals.attr,
													   Globals.urlfilter,Globals.szLimit,Globals.tmLimit,true);
			EntrySet es = new EntrySet();
			es.setEntries(sr);
			
			for (es.startIterator(); es.hasNext(); ) {
				Entry e = es.next();
				MVDSearchResults base_sr = Globals.ldapm.search(e.getDn(),LDAPManagement.BASE,Globals.attr,Globals.urlfilter,
																Globals.szLimit,Globals.tmLimit,true);
				/* Creates the attribute set for each entry */
				e.setAttributes(base_sr);
			}

			if (modified.equalsOnlyDefinitionAndNotAutomatic(es)== false) return false;
		}
		else if (modified!=null && collateralLevel.equalsIgnoreCase("medium")) {
			
			MVDSearchResults sr = Globals.ldapm.search(sPoint,LDAPManagement.ONE_LEVEL,Globals.attr,
													   Globals.urlfilter,Globals.szLimit,Globals.tmLimit,true);
			EntrySet es = new EntrySet();
			es.setEntries(sr);
			
			if (modified.equalsOneLevel(es)==false) return false;
		}
		
		MVDSearchResults sr = Globals.ldapm.search(dn,LDAPManagement.BASE,Globals.attr,Globals.urlfilter,Globals.szLimit,Globals.tmLimit,true);
		if (expected.equalsIgnoreCase("NOTfind")) {
			if (sr==null) return true;
			if (sr.size()==0) return true;
		}
		else if (expected.equalsIgnoreCase("find")) {
			if (sr!=null && sr.size()>0) {
				Entry entSystem = new Entry(dn,new Vector());
				entSystem.setAttributes(sr);
				Entry entCTT = null;
				if (modified!=null) entCTT=modified.getEntry(dn);
				else entCTT = Testing.existsEntryInSystem(dn);
				
				return entSystem.fitsCompletelyConfig(true) && entCTT.equalsOnlyDefinitionAndNotAutomatic(entSystem);
			}
		}
		Error.entryNOTFound(dn,expected.equalsIgnoreCase("NOTfind"));
		return false;
	}
	
	
	static void resetNumbOfExecTC () {
		
		numbOfExecTC = new MyHashMap();
	}
	
	
	static private void initFakeAttributes () {
		
		/* This will be a set of illegal attribute names */
		Vector fakeAttributeNames = new Vector();
		for (int i=0; i<FAKE_ATTRIBUTES.length; i++) {
			String attName = FAKE_ATTRIBUTES[i];
			if (Globals.existsAttName(attName)==false)	{
				fakeAttributeNames.add(FAKE_ATTRIBUTES[i]);
			}
		}
		/* Creates the definitive list of illegal attribute names */
		FAKE_ATTRIBUTE_NAMES = new String[fakeAttributeNames.size()];
		for (int i=0; i<FAKE_ATTRIBUTE_NAMES.length; i++) {
			FAKE_ATTRIBUTE_NAMES[i]=(String)fakeAttributeNames.elementAt(i);
		}
	}

	
	static Entry createEntry (String rDnId, String origDn,EntrySet modified) {
		return createEntry(rDnId,origDn,false,null,modified);
	}
	
	
	static protected Entry createEntry (String rDnId, String origDn, boolean putIllegalVals, Object nothing, EntrySet modified) {
		
		MyHashMap accums = Globals.getLegalDNSpecs();
		String realOrigRdnId = createReverseDnIds(origDn);
		Vector candidates = new Vector();
		
		for (Iterator it=accums.keySet().iterator(); it.hasNext(); ) {
			String accum = (String)it.next();
			EntryDefinition eDef = Globals.getEntDefFromRDNID(accum);
			if (eDef==null) continue;
			if (eDef.matchPossibEnt(realOrigRdnId)) {
				candidates.add(accum);
			}
		}
		if (candidates.size()==0) return null;
		
		for (int k=0; k<STATIC_MAX_NUMB_OF_TRIES_FOR_ADDING_NEW_ENTRY; k++) {
			
			String picked = (String)candidates.elementAt(Globals.randGen.nextInt(candidates.size()));
			Vector vals = (Vector)accums.get(picked);
			String dnId = (String)vals.elementAt(EntryDefinition.L_DN_ID);
			Vector regExps = (Vector)vals.elementAt(EntryDefinition.L_DN_RE);
			Vector attNames = (Vector)vals.elementAt(EntryDefinition.L_DN_ATTS);
			String provDn = dnId+"=justForRdnId,"+origDn;
			Entry provEntry = new Entry(provDn);
			String provRdnId = provEntry.getRdnId();
			String dnVal = "";
			Vector atts = new Vector();
			for (int i=0; i<attNames.size(); i++) {
				String attName = (String)attNames.elementAt(i);
				RegularExpression regExp = (RegularExpression)regExps.elementAt(i);
				Vector v = new Vector();
				try {
					String val = null;
					if (putIllegalVals) val = regExp.randomIllegalString(1); 
					else val = regExp.randomLegalString(Globals.randGen.nextInt());
					if (val==null) return null;
					v.add(val);
					if (dnVal.length()>0) dnVal+=" ";
					dnVal+=val;
				}
				catch (Exception exc) {
					if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in TestCase.createEntry");
				}
				try {
					if (Globals.hasPropertyByRdnId(provRdnId,attName,Globals.PROP_VIRTUAL)==false) {
						MyAttribute att = new MyAttribute(attName,v,null);
						atts.add(att);
					}
				}
				catch (Exception exc) {
					if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in TestCase.createEntry");
				}
			}
			String newDn = dnId+"="+dnVal+","+origDn;
			if (modified.containsEntry(newDn)) continue;
			
			String newRdnId = Preprocessing.getAssociatedRdnId(newDn,false,false);
			MyHashMap attList = Globals.getAttListByRdnId(newRdnId);
			boolean addMay = (Globals.randGen.nextInt(100)<75);
			boolean addAllMay = (Globals.randGen.nextInt(100)<25);
			for (Iterator it=attList.keySet().iterator(); it.hasNext(); ) {
				String attName = (String)it.next();
				try {
					if (addMay && Globals.hasPropertyByRdnId(newRdnId,attName,Globals.PROP_MAY)) {
						if (addAllMay || Globals.randGen.nextInt(100)<50) {
							atts.add(createLegalAtt(null,newRdnId,attName,null,4));
						}
					}
					else if (Globals.hasPropertyByRdnId(newRdnId,attName,Globals.PROP_MUST) &&
							Globals.hasPropertyByRdnId(newRdnId,attName,Globals.PROP_LINKED)==false) {
						atts.add(createLegalAtt(null,newRdnId,attName,null,4));
					}
				}
				catch (Exception exc) {
					if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in TestCase.createEntry");
				}
			}
			return new Entry(newDn,atts);
		}
		return null;
	}
	
	
	static private String createReverseDnIds (String dn) {
    	
		String res = new String();
		
		String parts[] = GenericMethods.collectStrings(dn,' ',' ',",");
		for (int i=parts.length-1; i>=0; i--) {
			String id = GenericMethods.collectStrings(parts[i],' ',' ',"=")[0];
			if (res.length()>0) res+=",";
			res+=id;
		}
		
    	return res;
    }
	
	
	static private MyAttribute createLegalAtt (Entry ent, String rDnId, String attName, Object nothing, int aMaxNVals) {
		
		int maxNVals = aMaxNVals;
		boolean isAppendable = false;
		try {
			if (Globals.hasPropertyByRdnId(rDnId,attName,Globals.PROP_MULTIVALUE)==false) {
				maxNVals = 1;
			}
			isAppendable = Globals.hasPropertyByRdnId(rDnId,attName,Globals.PROP_APPENDABLE);
		}
		catch (Exception exc) {
			if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in TestCase.createLegalAtt");
		}
		int nVals = Globals.randGen.nextInt(maxNVals)+1; 
		Vector vals = new Vector();
		String forAppendable = "";
		for (int i=0; i<nVals; i++) {
			if (isAppendable) forAppendable+=Globals.getLegalValue(rDnId,attName);
			else vals.add(Globals.getLegalValue(rDnId,attName));
		}
		if (isAppendable) vals.add(forAppendable);
		return new MyAttribute(attName,vals,ent);
	}
	
	
	protected MyAttribute createLegalAtt (Entry ent, String rDnId, String attName, int aMaxNVals) {
		return createLegalAtt(ent,rDnId,attName,null,aMaxNVals);
	}
	
	
	/**
	 * Creates a set of modification values for one attribute
	 * @param name Attribute name
	 * @param value Set of values
	 * @param modType Type of modification
	 * @return The set of modification values
	 */
	protected Vector createModifAtt (String name, Vector value, int modType) {
		
		Vector res = new Vector();
		
		if (value.size()==0) {
			MVDModAttrValues v = new MVDModAttrValues();
			v.setAttrName(name);
			v.setModType(modType);
			res.add(v);
		}
		else {
			for (int i=0; i<value.size(); i++) {
				MVDModAttrValues v = new MVDModAttrValues();
				v.setAttrName(name);
				v.add(value.elementAt(i));
				v.setModType(modType);
				res.add(v);
			}
		}
		return res;
	}
}
