import java.util.*;



import javax.naming.directory.*;
import javax.naming.ldap.*;
import javax.naming.*;

import com.sap.idm.vds.*;


/**
 * Class in charge of the LDAP management
 * 
 * @author I054742
 *
 */
class LDAPManagement {
	
	/* Identifier for fake entries */
	static String FAKE_ENTRY_ID = "1889kjdjiuttbbhdyyyyhxnnjakksFAKE_ENTRY_IDjjii4090amiyiokasdjjjdjakksjbbb2200kkjdj";
	
	/* Indicates if for each operation a new context must be created */
	private final boolean RECONNECT = false;
	
	/* Search options */
	static final int BASE = 0;
	static final int ONE_LEVEL = 1;
	static final int SUBTREE = 2;
	
	/* Modify options */
	private final int ADD = 0;
	private final int DELETE = 1;
	private final int REPLACE = 2;
	
	/* LDAP parameters */
	private LdapContext mLdapContext = null;
	private NamingEnumeration ne = null;
	private String server = null;
	private String port = null;
	private String user = null;
	private String password = null;
	private String securityProtocol = null;
	
	/* LDAP summary data */
	static private int numberOfAdditions=0;
	static private int numberOfDeletions=0;
	static private int numberOfModificationsAdd=0;
	static private int numberOfModificationsDelete=0;
	static private int numberOfModificationsReplace=0;
	static private int numberOfSearchBase=0;
	static private int numberOfSearchOnelevel=0;
	static private int numberOfSearchSubtree=0;
	
	static private MyHashMap searchesByEntryType = new MyHashMap();
	
	
	/**
	 * Construct an LDAPManagement object and creates a LDAP context from the received parameters
	 * @param mServer Server
	 * @param mPort Port
	 * @param mUser User
	 * @param mPassword Password
	 * @param aSecurityProtocol Security protocol
	 * @throws Exception An exception will be thrown if some problem is finding while creating the context
	 */
	LDAPManagement (String mServer, String mPort, String mUser, String mPassword,
	        		String aSecurityProtocol) throws Exception {
		try {
			this.createLdapContext(mServer,mPort,mUser,mPassword,aSecurityProtocol);
			this.server=mServer;
			this.port=mPort;
			this.user=mUser;
			this.password=mPassword;
			this.securityProtocol=aSecurityProtocol;
		}
		catch (Exception exc) {
			if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in LDAPManagement.LDAPManagement");
			//Loggin.Log(Globals.indent()+"ERROR: Error creating LDAP Connection"+(exc.getMessage()==null?"":" -> "+exc.getMessage()));
			throw new Exception();
		}
	}
	
	
	LDAPManagement () {}
	
	
	/**
	 * Initializes the summary data
	 */
	static void reset () {
		
		numberOfAdditions=0;
		numberOfDeletions=0;
		numberOfModificationsAdd=0;
		numberOfModificationsDelete=0;
		numberOfModificationsReplace=0;
		numberOfSearchBase=0;
		numberOfSearchOnelevel=0;
		numberOfSearchSubtree=0;
		searchesByEntryType = new MyHashMap();
	}
	
	
	/**
	 * Cleans up the naming global exception variable
	 */
	private void cleanUp()  {
    	try  { if (this.ne != null) this.ne.close();} catch (NamingException exc) {if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in LDAPManagement.cleanUp");}
    	try  { if (this.RECONNECT && this.mLdapContext != null)  this.mLdapContext.close();} catch (NamingException exc) {if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in LDAPManagement.cleanUp");}
	}
	
	
	/**
	 * Creates a LDAP context from the received parameters
	 * @param mServer Server
	 * @param mPort Port
	 * @param mUser User
	 * @param mPassword Password
	 * @param aSecurityProtocol Security protocol
	 * @throws Exception An exception will be thrown if some error is encountered while the process
	 */
	private void createLdapContext (String mServer, String mPort, String mUser, String mPassword,
	        String aSecurityProtocol) throws Exception {

		try {
			Hashtable env = new Hashtable();

			env.put(Context.INITIAL_CONTEXT_FACTORY, "com.sun.jndi.ldap.LdapCtxFactory");
			env.put(Context.SECURITY_AUTHENTICATION, "simple");

			// --- host and port
			env.put(Context.PROVIDER_URL, "ldap://" + mServer + ":" + mPort);
			env.put(Context.SECURITY_PRINCIPAL, mUser);
			env.put(Context.SECURITY_CREDENTIALS, mPassword);

			if (aSecurityProtocol != null) {
			    env.put(Context.SECURITY_PROTOCOL, aSecurityProtocol);
				if(aSecurityProtocol.equalsIgnoreCase("ssl")){
					env.put("java.naming.ldap.factory.socket", "com.sap.idm.vds.util.MxJNDISocketFactory");
				}
			}

			mLdapContext = new InitialLdapContext(env, null);
		} catch (Exception exc) {
			if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in LDAPManagement.createLdapContext");
		    throw exc;
		}
	}
	
	
	/**
	 * Searches using LDAP
	 * @param startingpoint Starting point where the search must be made from
	 * @param scope Search's scope
	 * @param attr Attribute set that the search must give
	 * @param urlfilter Filter used to filter the search elements
	 * @param szLimit Maximum number of entries that the entry will give
	 * @param tmLimit Maximum amount of time that the search can last
	 * @param identation Parameter used for log indentation
	 * @return A entry set as result of the search
	 */
	MVDSearchResults search (String startingpoint, int scope, Vector attr,
						     String urlfilter, int szLimit, int tmLimit, boolean login)  {

		if (scope==BASE) numberOfSearchBase++;
		else if (scope==ONE_LEVEL) numberOfSearchOnelevel++;
		else if (scope==SUBTREE) numberOfSearchSubtree++;
		
		String eId = (new Entry(startingpoint,false,false)).getId();
		if (Globals.existsEnt(eId)==false) eId = FAKE_ENTRY_ID;
		if (searchesByEntryType.containsKey(eId)==false) {
			int nOfSearches[] = {0,0,0};
			searchesByEntryType.put(eId,nOfSearches);
		}
		((int[])searchesByEntryType.get(eId))[scope]++;
		
		MVDSearchResults result = new MVDSearchResults();
	
		// --- add attributes, if any
		SearchControls aConstraints = new SearchControls();
		if (attr.size() > 0) {
			String [] attrIDs = (String [])attr.toArray(new String[0]);
			aConstraints.setReturningAttributes(attrIDs);
		}
		
		aConstraints.setSearchScope(scope);
		aConstraints.setCountLimit(szLimit);
		aConstraints.setTimeLimit(tmLimit);
		
		try  {
			if (this.RECONNECT) this.createLdapContext(this.server,this.port,this.user,this.password,this.securityProtocol);
			ne = mLdapContext.search(startingpoint, urlfilter, aConstraints);
			result.putNamingEnumeration(ne, startingpoint, szLimit);
			result.setOK();
		
		} catch (NamingException exc) {
			if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in LDAPManagement.search");
			int aCode = result.PickTheNamingExceptionCode(exc) ;
			if (login) Loggin.Log(Globals.indent()+"Error given by VDS while searching -> "+aCode+": "+exc.getMessage());
			result.setError(aCode, exc.getMessage());
			this.cleanUp();
			return null;
		
		} catch (Exception exc) {
			if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in LDAPManagement.search");
			int aCode = result.getCode();
			if (login) Loggin.Log(Globals.indent()+"Error given by VDS while searching -> "+aCode+": "+exc.getMessage());
			result.setError(aCode, exc.getMessage());
			this.cleanUp();
			return null;
		}
		this.cleanUp();
		return result;
	}	
	
	
	/**
	 * Modifies using LDAP
	 * @param startingpoint The DN of the entry where the modifications will be made
	 * @param attrNamesAndModValues Set of attributes that will be modified and their values
	 * @param indentation Used for the log
	 * @return The result of the operation
	 */
	MVDOperationResult modify (String startingpoint, Vector attrNamesAndModValues)  {
		
		MVDOperationResult result = new MVDOperationResult();
		MVDModAttrValues oneMvdMod;
		
		try	{
			
			BasicAttribute baOne = null;
			String attrName = "";
			Vector modifications = new Vector();
			
			int ls = attrNamesAndModValues.size();
			
			int modType=-1;
			
			for (int i = 0; i< ls; i++) {
				
				oneMvdMod = (MVDModAttrValues) attrNamesAndModValues.elementAt(i);
				
				// --- fetch attribute name to be modified
				attrName = oneMvdMod.getAttrName();
				baOne = new BasicAttribute(attrName);
				int t = oneMvdMod.size();
				for (int x=0;x < t; x++)  {
					baOne.add(oneMvdMod.elementAt(x));
				}
		
				int aType = oneMvdMod.getModType();
				modType = aType;
				if (aType == 0)
					aType = 1;
				else if (aType == 1)
					aType = 3;
				ModificationItem mi = new ModificationItem(aType, baOne);
				modifications.add(mi);
			}
		
			if (modType==this.ADD) numberOfModificationsAdd++;
			else if (modType==this.DELETE) numberOfModificationsDelete++;
			else if (modType==this.REPLACE) numberOfModificationsReplace++;
			
			int ix = 0;
			ModificationItem[] m = new ModificationItem[modifications.size ()];
			for ( Iterator it = modifications.iterator (); it.hasNext (); ){
				m[ix++] = (ModificationItem) it.next ();
			}
			if (this.RECONNECT) this.createLdapContext(this.server,this.port,this.user,this.password,this.securityProtocol);
			mLdapContext.modifyAttributes(startingpoint,m);
			
			result.setOK();
			this.cleanUp();
			return result;
	
		}
		catch (NamingException exc) {
			if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in LDAPManagement.modify");
			int aCode = result.PickTheNamingExceptionCode(exc);
			Loggin.Log(Globals.indent()+"Error given by VDS while modifying attribute -> "+aCode+": "+exc.getMessage());
			result.setError(aCode, exc.getMessage() );
			this.cleanUp();
			return null;
		}
		catch (Exception exc) {
			if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in LDAPManagement.modify");
			int aCode = result.getCode();
			Loggin.Log(Globals.indent()+"Error given by VDS while modifying attribute -> "+aCode+": "+exc.getMessage());
			result.setError(aCode, exc.getMessage() );
			this.cleanUp();
			return null;
		}
	}
	
	
	/**
	 * Adds using LDAP
	 * @param startingpoint The DN where the entry will be added
	 * @param attrNamesAndValues Set of attributes and their corresponding values for the new entry
	 * @param indentation Used for the log
	 * @return The result of the operation
	 */
	MVDOperationResult add (String startingpoint, HashMap attrNamesAndValues)  {
		
		int index = startingpoint.indexOf("=");
		if (index>=0) {
			String rDnId = startingpoint.substring(0,index);
			if (attrNamesAndValues.containsKey(rDnId)==false) {
				Vector vals = new Vector();
				vals.add(rDnId);
				attrNamesAndValues.put(Certification.DELETE_JDNIKEY,vals);
			}
		}
		
		MVDOperationResult result = new MVDOperationResult();
		numberOfAdditions++;
		
		try {
			
			BasicAttributes bas = new BasicAttributes();
			BasicAttribute baOne = null;
			String attrName = "";
			Vector attrValue = null;
			
			for (Iterator ii = attrNamesAndValues.keySet().iterator() ; ii.hasNext(); ){
				
				attrName = (String) ii.next();
				attrValue = (Vector) attrNamesAndValues.get(attrName);
				baOne = new BasicAttribute(attrName);
				int t = attrValue.size();
				for (int x=0;x < t; x++)  {
					baOne.add(attrValue.elementAt(x));
				}
				bas.put(baOne);
			}
			if (this.RECONNECT) this.createLdapContext(this.server,this.port,this.user,this.password,this.securityProtocol);
			mLdapContext.createSubcontext(startingpoint,bas);
			result.setOK();
		
		} catch (NamingException exc) {
			if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in LDAPManagement.add");
			int aCode = result.PickTheNamingExceptionCode(exc);
			Loggin.Log(Globals.indent()+"Error given by VDS while adding entry -> "+aCode+": "+exc.getMessage());
			result.setError(aCode, exc.getMessage() );
			this.cleanUp();
			return null;
		}
		catch (Exception exc) {
			if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in LDAPManagement.add");
			int aCode = result.getCode();
			Loggin.Log(Globals.indent()+"Error given by VDS while adding entry -> "+aCode+": "+exc.getMessage());
			result.setError(aCode, exc.getMessage() );
			this.cleanUp();
			return null;
		}

		this.cleanUp();
		return result;
	}
	
	
	/**
	 * Deletes using LDAP
	 * @param startingpoint DN of the entry to be deleted
	 * @param identation Indentation used for the log
	 * @return The result of the operation
	 */
	MVDOperationResult delete (String startingpoint)  {

		MVDOperationResult result = new MVDOperationResult();
		numberOfDeletions++;
		
		try  {
			
			if (this.RECONNECT) this.createLdapContext(this.server,this.port,this.user,this.password,this.securityProtocol);
			mLdapContext.destroySubcontext(startingpoint);
			result.setOK();
		
		} catch (NamingException exc) {
			if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in LDAPManagement.delete");
			int aCode = result.PickTheNamingExceptionCode(exc);
			Loggin.Log(Globals.indent()+"Error given by VDS while deleting entry -> "+aCode+": "+exc.getMessage());
			result.setError(aCode, exc.getMessage());
			this.cleanUp();
			return null;
		}
		catch (Exception exc) {
			if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in LDAPManagement.delete");
			int aCode = result.getCode();
			Loggin.Log(Globals.indent()+"Error given by VDS while deleting entry -> "+aCode+": "+exc.getMessage());
			result.setError(aCode, exc.getMessage());
			this.cleanUp();
			return null;
		}
		
		this.cleanUp();
		return result;
	}


	void writeReport () {
		
		Loggin.Log(Globals.indent()+"Internal LDAP Summary");
		Globals.incIndent();
		Loggin.Log(Globals.indent()+numberOfAdditions+" additions");
		Loggin.Log(Globals.indent()+numberOfDeletions+" deletions");
		Loggin.Log(Globals.indent()+(numberOfModificationsAdd+numberOfModificationsDelete+numberOfModificationsReplace)+" modifications");
		Globals.incIndent();
		Loggin.Log(Globals.indent()+numberOfModificationsAdd+" modifications with add option");
		Loggin.Log(Globals.indent()+numberOfModificationsDelete+" modifications with delete option");
		Loggin.Log(Globals.indent()+numberOfModificationsReplace+" modifications with replace option");
		Globals.decIndent();
		Loggin.Log(Globals.indent()+numberOfSearchBase+" searchs with base option");
		Loggin.Log(Globals.indent()+numberOfSearchOnelevel+" searchs with one level option");
		Loggin.Log(Globals.indent()+numberOfSearchSubtree+" searchs with subtree option");
		for (Iterator it=searchesByEntryType.keySet().iterator(); it.hasNext(); ) {
			String id = (String)it.next();
			if (id.equalsIgnoreCase(LDAPManagement.FAKE_ENTRY_ID)) continue;
			int nOfSearches[] = (int[])searchesByEntryType.get(id);
			Loggin.Log(Globals.indent()+"Number of searches for entry type: \""+id+"\"");
			Globals.incIndent();
			Loggin.Log(Globals.indent()+"Base search: "+(nOfSearches[0]));
			Loggin.Log(Globals.indent()+"One level search: "+(nOfSearches[1]));
			Loggin.Log(Globals.indent()+"Subtree search: "+(nOfSearches[2]));
			Globals.decIndent();
		}
		if (searchesByEntryType.containsKey(LDAPManagement.FAKE_ENTRY_ID)) {
			String id = LDAPManagement.FAKE_ENTRY_ID;
			int nOfSearches[] = (int[])searchesByEntryType.get(id);
			Loggin.Log(Globals.indent()+"Number of searches for fake entries type:");
			Globals.incIndent();
			Loggin.Log(Globals.indent()+"Base search: "+(nOfSearches[0]));
			Loggin.Log(Globals.indent()+"One level search: "+(nOfSearches[1]));
			Loggin.Log(Globals.indent()+"Subtree search: "+(nOfSearches[2]));
			Globals.decIndent();
		}
	}
	
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////The next methods basically the summary data ///////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////	
	
	
	static int getNumberOfAdditions () {
		
		return numberOfAdditions;
	}
	
	
	static int getNumberOfDeletions () {
		
		return numberOfDeletions;
	}
	
	
	static int getNumberOfModificationsAdd () {
		
		return numberOfModificationsAdd;
	}
	
	
	static int getNumberOfModificationsDelete () {
		
		return numberOfModificationsDelete;
	}
	

	static int getNumberOfModificationsReplace () {
	
		return numberOfModificationsReplace;
	}

	
	static int getNumberOfSearchBase () {
		
		return numberOfSearchBase;
	}
	
	
	static int getNumberOfSearchOnelevel () {
		
		return numberOfSearchOnelevel;
	}
	
	
	static int getNumberOfSearchSubtree () {
		
		return numberOfSearchSubtree;
	}
	
	
	static MyHashMap getSearchesByEntryType () {
		
		return searchesByEntryType;
	}
}
