import java.util.*;


/**
 * Executes the Instuctions testing
 * 
 * @author I054742
 *
 */
public class InstructionsTesting extends Testing {

	private final int NOT_TESTED = 0;
	private final int NEG_RESULT = -1;
	private final int POS_RESULT = 1;
	
	
	boolean execute () {
		
		Loggin.Log("");
		Loggin.Log(Globals.indent()+"#####################################################################");
		Loggin.Log(Globals.indent()+"############ Starting Instructions Testing Process... ###############");
		Loggin.Log(Globals.indent()+"#####################################################################");
		Loggin.Log("");
		Globals.incIndent();	
		
		try {
			this.testName = "Instructions";
			
			Vector insts = InstructionsFile.getInsts();
			
			this.success = false;
			this.initialize();
			
			for (int i=0; this.timeCondition() && i<insts.size(); i++) {
				
				Instruction inst = (Instruction)insts.elementAt(i);
				
				Loggin.Log("");
				Loggin.Log(Globals.indent()+"Executing instruction: "+inst.getOriginal()+". In line: "+inst.getNLine());
				
				if (this.evaluateInst((Instruction)insts.elementAt(i))==false) {
					Globals.errorMessage = "The expected result was not the one in the system when the instruction: \""+inst.getOriginal()+"\" was executed";
					this.endTime = System.currentTimeMillis();
					return false;
				}
			}
			
			this.endTime = System.currentTimeMillis();
			this.success = true;
			return true;
		}
		catch (Exception exc) {
			if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in InstructionsTesting.execute");
			Loggin.Log(Globals.indent()+"ERROR: Error while executing Instructions Testing Process");
			return false;
		}
		finally {
			this.endTime = System.currentTimeMillis();
			this.times[Globals.TIME_INDEX_FOR_TESTER] = System.currentTimeMillis() - this.times[Globals.TIME_INDEX_FOR_TESTER];
			Globals.decIndent();
			Loggin.Log("");
			Loggin.Log(Globals.indent()+"#####################################################################");
			Loggin.Log(Globals.indent()+"############### Instructions Testing Process ended... ###############");
			Loggin.Log(Globals.indent()+"#####################################################################");
			Loggin.Log("");
		}
	}
	
	
	private boolean evaluateInst(Instruction inst) {
		
		if (inst.getType().equalsIgnoreCase(Instruction.ADD)) return evaluateAdd(inst.getEntry(), inst);
		if (inst.getType().equalsIgnoreCase(Instruction.DEL)) return evaluateDel(inst.getDn());
		if (inst.getType().equalsIgnoreCase(Instruction.MOD_ADD)) return evaluateMod_Add(inst.getDn(),inst.getAttributes());
		if (inst.getType().equalsIgnoreCase(Instruction.MOD_DEL)) return evaluateMod_Del(inst.getDn(),inst.getAttributes());
		if (inst.getType().equalsIgnoreCase(Instruction.MOD_REP)) return evaluateMod_Rep(inst.getDn(),inst.getAttributes());
		
		Loggin.Log(Globals.indent()+"ERROR: Unknown instruction type");
		return false;
	}
	
	
	private boolean evaluateAdd (Entry ent, Instruction inst) {
		
		if (this.existsEntry(ent.getDn())!=null) {
			Loggin.Log(Globals.indent()+"WARNING: The instruction: \""+inst.getOriginal()+"\" will not be executed since there is not test case to adapt it. There is no test case that contemplate the entry duplications");
			return true;
		}
		if (ent.createdCompletelyLegal(true)) {
			Loggin.Log(Globals.indent()+"The instruction will be executed in legal mode");
			return new TestCase_A_LEGAL_DN(ent.pickUpAttRandomly(),ent,null).test();
		}
		
		Loggin.Log(Globals.indent()+"The instruction will be executed in illegal mode since the entry is not completely legal");
		return new TestCase_A_ILLEGAL(ent.pickUpAttRandomly(),ent,null).test();
	}
	
	
	private boolean evaluateDel (String dn) {
		
		Entry ent = this.existsEntry(dn);
		if (ent!=null) {
			Loggin.Log(Globals.indent()+"The instruction will be executed in legal mode");
			return new TestCase_D_EXISTING_ENTRY(null,ent,null).test();
		}
		
		Loggin.Log(Globals.indent()+"The instruction will be executed in illegal mode since the entry does not exists");
		return new TestCase_D_ILLEGAL(null,new Entry(dn),null).test();
	}
	
	
	private int evaluateMod (Entry ent, MyAttribute att, TestCase illegalMTC) {
		
		if (ent==null) {
			Loggin.Log(Globals.indent()+"The instruction will be executed in illegal mode since the entry is null");
			if (illegalMTC.test()) return this.POS_RESULT;
			else return this.NEG_RESULT;
		}
		
		if (Globals.existsAttByRDnId(ent.getRdnId(),att.getName())==false) {
			Loggin.Log(Globals.indent()+"The instruction will be executed in illegal mode since the attribute: \""+att.getName()+"\" has not definition");
			if (illegalMTC.test()) return this.POS_RESULT;
			else return this.NEG_RESULT;
		}
		
		boolean areLegal = true;
		String lastEvalVal = null;
		for (int i=0; areLegal && i<att.getNumbOfValues(); i++) {
			lastEvalVal = att.getValueAt(i);
			areLegal &= Globals.isLegalValue(ent.getRdnId(),att.getName(),lastEvalVal);
		}
		if (areLegal==false) {
			Loggin.Log(Globals.indent()+"The instruction will be executed in illegal mode since the value: \""+lastEvalVal+"\" of the attribute: \""+att.getName()+"\" is illegal");
			if (illegalMTC.test()) return this.POS_RESULT;
			else return this.NEG_RESULT;
		}
		try  {
			if (Globals.hasPropertyByRdnId(ent.getRdnId(),att.getName(),Globals.PROP_MODIFIABLE)==false) {
				Loggin.Log(Globals.indent()+"The instruction will be executed in illegal mode since the attribute: \""+att.getName()+"\" is not modifiable");
				if (illegalMTC.test()) return this.POS_RESULT;
				else return this.NEG_RESULT;
			}
		}
		catch (Exception exc) {
			if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in InstructionsTesting.evaluateMod");
		}
		
		return this.NOT_TESTED;
	}
	
	
	private boolean evaluateMod_Add (String dn, Vector attributes) {
		
		MyAttribute att = (MyAttribute)attributes.elementAt(0);
		Entry ent = this.existsEntry(dn);
		TestCase illegalMTC = new TestCase_M_ILLEGAL(ent.givesAttributeByName(att.getName()),ent,null,att.getValues(),Globals.MODIF_ADD_ATTRS);
		
		int resEvalMod = this.evaluateMod(ent,att,illegalMTC);
		if (resEvalMod!=this.NOT_TESTED) return (resEvalMod==this.POS_RESULT);

		try {
			
			MyAttribute attInEntry = ent.containsAttributeByName(att.getName());
			if (attInEntry==null) {
				if (Globals.hasPropertyByRdnId(ent.getRdnId(),att.getName(),Globals.PROP_MAY)) {
					
					Loggin.Log(Globals.indent()+"The instruction will be executed in legal mode to add a new attribute");
					return new TestCase_MAA_ADD(att,ent,null,att.getValues()).test();
				}
				else {
					
					Loggin.Log(Globals.indent()+"The instruction will be executed in illegal mode since the attribute to be added has not the MAY property");
					return illegalMTC.test();
				}
			}
			
			if (Globals.hasPropertyByRdnId(ent.getRdnId(),att.getName(),Globals.PROP_MULTIVALUE)==false &&
				attInEntry.getNumbOfValues()>0 &&
				Globals.hasPropertyByRdnId(ent.getRdnId(),att.getName(),Globals.PROP_APPENDABLE)==false) {
				
				Loggin.Log(Globals.indent()+"The instruction will be executed in illegal mode since it has not the MULTIVALUE property");
				return illegalMTC.test();
			}
			
			boolean contsSomeVal = false;
			for (int i=0; contsSomeVal==false && i<att.getNumbOfValues(); i++) {
				contsSomeVal |= attInEntry.containsValue(att.getValueAt(i));
			}
			
			if (contsSomeVal && Globals.hasPropertyByRdnId(ent.getRdnId(),att.getName(),Globals.PROP_APPENDABLE)==false &&
				Globals.hasPropertyByRdnId(ent.getRdnId(),att.getName(),Globals.PROP_ALLOWS_DUPLICATE)==false) {
				
				Loggin.Log(Globals.indent()+"The instruction will be executed in illegal mode since the attribute does not allow duplicate values");
				return illegalMTC.test();
			}
			
			Loggin.Log(Globals.indent()+"The instruction will be executed in legal mode");
			return new TestCase_MA_NORMAL_CASE(attInEntry,ent,null,att.getValues()).test();
			
		}
		catch (Exception exc) {
			if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in InstructionsTesting.evaluateMod_Add");
			Error.internalUnexpected();
			return false;
		}
	}
	
	
	private boolean evaluateMod_Del (String dn, Vector attributes) {
		
		MyAttribute att = (MyAttribute)attributes.elementAt(0);
		Entry ent = this.existsEntry(dn);
		TestCase illegalMTC = new TestCase_M_ILLEGAL(ent.givesAttributeByName(att.getName()),ent,null,att.getValues(),Globals.MODIF_ADD_ATTRS);
		
		int resEvalMod = this.evaluateMod(ent,att,illegalMTC);
		if (resEvalMod!=this.NOT_TESTED) return (resEvalMod==this.POS_RESULT);

		try {
			
			MyAttribute attInEntry = ent.containsAttributeByName(att.getName());
			
			if (attInEntry==null) {
				
				Loggin.Log(Globals.indent()+"The instruction will be executed in illegal mode since there is no attribute: \""+att.getName()+"\" in the entry");
				return illegalMTC.test();
			}
			
			if (att.containsAllValues(attInEntry.getValues())) {
				
				Loggin.Log(Globals.indent()+"The instruction will be executed in legal mode");
				return new TestCase_MDA_DELETE(attInEntry,ent,null,att.getValues()).test();
			}
			
			Loggin.Log(Globals.indent()+"The instruction will be executed in legal mode");
			return new TestCase_MD_NORMAL_CASE(attInEntry,ent,null,att.getValues()).test();
			
		}
		catch (Exception exc) {
			if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in InstructionsTesting.evaluateMod_Del");
			Error.internalUnexpected();
			return false;
		}
	}
	
	
	private boolean evaluateMod_Rep (String dn, Vector attributes) {
		
		MyAttribute att = (MyAttribute)attributes.elementAt(0);
		Entry ent = this.existsEntry(dn);
		TestCase illegalMTC = new TestCase_M_ILLEGAL(ent.givesAttributeByName(att.getName()),ent,null,att.getValues(),Globals.MODIF_ADD_ATTRS);
		
		int resEvalMod = this.evaluateMod(ent,att,illegalMTC);
		if (resEvalMod!=this.NOT_TESTED) return (resEvalMod==this.POS_RESULT);

		try {
			
			MyAttribute attInEntry = ent.containsAttributeByName(att.getName());
			
			if (attInEntry==null) {
				
				if (Globals.hasPropertyByRdnId(ent.getRdnId(),att.getName(),Globals.PROP_MAY)) {
					
					Loggin.Log(Globals.indent()+"The instruction will be executed in legal mode to add a new attribute");
					return new TestCase_MAA_REPLACE(att,ent,null,att.getValues()).test();
				}
				else {
					
					Loggin.Log(Globals.indent()+"The instruction will be executed in illegal mode since the attribute to be added has not the MAY property");
					return illegalMTC.test();
				}
			}
			
			if (att.getNumbOfValues()==0) {
				
				Loggin.Log(Globals.indent()+"The instruction will be executed in legal mode");
				return new TestCase_MDA_REPLACE(attInEntry,ent,null).test();
			}
			
			Loggin.Log(Globals.indent()+"The instruction will be executed in legal mode");
			return new TestCase_MR_NORMAL_CASE(attInEntry,ent,null,((MyAttribute)attributes.elementAt(0)).getValues()).test();
			
		}
		catch (Exception exc) {
			if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in InstructionsTesting.evaluateMod_Rep");
			Error.internalUnexpected();
			return false;
		}
	}
	
	
	void writeReport () {
		
		this.writeHeader();
		
		if (this.success==false) {
			Loggin.Log(Globals.indent()+"Instructions testing process was aborted because some error occurred during the process. ERROR: "+Globals.errorMessage);
		}
		else {
			if (this.timeExpired) {
				Loggin.Log(Globals.indent()+"Instructions testing process finished before finishing all the desired operations because the maximum allowed time for the process was reached");
			}
			else {
				Loggin.Log(Globals.indent()+"Instructions testing process finished successfully");
			}
		}
		
		Loggin.Log("");
		this.writeNumbOfTC();
		
		Loggin.Log("");
		Globals.ldapm.writeReport();
		
		Loggin.Log("");
		Loggin.Log(Globals.indent()+"Execution time: "+GenericMethods.expressProperlyTime(this.endTime-this.startTime));
	}
}
