import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.Iterator;
import java.util.Vector;


/**
 * Set of methods for parsing files
 * 
 * @author I054742
 *
 */
class FileParsingMethods {
	
	static protected final String COMMENTARY = "#";
	static protected final String DEFINITION_SEPARATOR = "@";
	static final String TEST_CASES_YES = "testCasesYes";
	static final String TEST_CASES_NO = "testCasesNo";
	
	static protected int nLine = 0;
	
	
	/**
	 * Reads a line from the file 'raf'. Commented lines will be ignored
	 * @param raf The file
	 * @return A read line if it exists
	 * 		   'null' otherwise
	 * @throws IOException
	 */
	static String readLine (RandomAccessFile raf, int toLowerOrUpper) throws IOException {
		
		++nLine;
		while (true) {
			String res = raf.readLine();
			if (res==null) return null;
			res=GenericMethods.cleanTrailingAndLeadingSpaces(res);
			if (res.length()>0 && res.startsWith(COMMENTARY)==false) {
				if (toLowerOrUpper==0) return res;
				else if (toLowerOrUpper<0) return res.toLowerCase();
				else return res.toUpperCase();
			}
			++nLine;
		}
	}
	
	
	/**
	 * Tries making integer numbers from the string 'str'
	 * @param str The string
	 * @param acceptNegatives Indicates if negatives values will be accepted
	 * @param param Parameter identifier related to 'str' (the parameter value)
	 * @param nLine Number of the last read line
	 * @return An integer if the conversion was properly done
	 * @throws Exception
	 */
	static int makeInteger (String str, boolean acceptNegatives, String param, int nLine) throws Exception {
		
		int res = new Integer(str).intValue();
		
		if (res<0 && !acceptNegatives) {
			
			Loggin.Log(Globals.indent()+"ERROR: Negative number not allowed for parameter: \""+param+"\" in line "+nLine);
			throw new Exception();
		}
		
		return res;
	}
	
	
	/**
	 * This method parses a line properly. The parsed lines will be of the form: parameterName = parameterValue
	 * @param line The line to be parsed
	 * @param identifier Expected parameterName in this line
	 * @param legalVals Set of legal values for the parameterValue
	 * @param nLine Number line in the file
	 * @return The parameterValue if everything was correct
	 * @throws Exception It will be thrown if some error occurred
	 */
	static String parseLine (String line, String identifier, String legalVals, int nLine) throws Exception {
		
		if (line==null) {
			Loggin.Log(Globals.indent()+"ERROR: There is some missing line(s). Error after line "+nLine);
			throw new Exception();
		}

		line = GenericMethods.cleanTrailingAndLeadingSpaces(line);
		String id = GenericMethods.cleanTrailingAndLeadingSpaces(line.substring(0,line.indexOf("=")));
		if (id.equalsIgnoreCase(identifier)==false) {
			Loggin.Log(Globals.indent()+"ERROR: Incorrect identifier in line "+nLine+". The expected one is \""+identifier+"\"");
			throw new Exception();
		}
		String val = GenericMethods.cleanTrailingAndLeadingSpaces((line = line.substring(line.indexOf("=")+1,line.length())));
		RegularExpression re = RegularExpression.getRegExpFromId(legalVals);
		if (re==null) {
			if (Certification.DEBUGGING_INFO) System.out.println("Regular Expression didnt compile->"+legalVals+". Line->"+nLine+", identifier->"+id+", value"+val);
			Loggin.Log(Globals.indent()+"ERROR: Critical internal error. Please, restart again");
			throw new Exception();
		}
		if (re.matches(val)==false) {
			Loggin.Log(Globals.indent()+"ERROR: The value \""+val+"\" of the identifier \""+identifier+"\" in line "+nLine+" does not match with the set of legal values given by \""+legalVals+"\"");
			throw new Exception();
		}
		return val;
	}
	
	
	/**
	 * This method parses a line properly. The parsed lines will be of the form: parameterName = parameterValue
	 * @param line The line to be parsed
	 * @param identifier Expected parameterName in this line
	 * @param legalVals Set of legal values for the parameterValue
	 * @param nLine Number line in the file
	 * @return The parameterValue if everything was correct
	 * @throws Exception It will be thrown if some error occurred
	 */
	static String[] parseLinePar (String line, MyHashMap idsLegalVals, int nLine) throws Exception {
		
		if (line==null) {
			Loggin.Log(Globals.indent()+"ERROR: There is some missing line(s). Error after line "+nLine);
			throw new Exception();
		}

		line = GenericMethods.cleanTrailingAndLeadingSpaces(line);
		if (line.length()==0) {
			Loggin.Log(Globals.indent()+"ERROR: Blank lines are not allowed. ");
			throw new Exception();
		}
		String id = GenericMethods.cleanTrailingAndLeadingSpaces(line.substring(0,line.indexOf("=")));
		if (idsLegalVals!=null && idsLegalVals.containsKey(id)==false) {
			Loggin.Log(Globals.indent()+"ERROR: Incorrect identifier in line "+nLine+". The identifier \""+id+"\" is not a legal identifier");
			throw new Exception();
		}
		String val = GenericMethods.cleanTrailingAndLeadingSpaces((line = line.substring(line.indexOf("=")+1,line.length())));
		String legalVals = "\\.*";
		if (idsLegalVals!=null) legalVals = (String)idsLegalVals.get(id);
		RegularExpression re = RegularExpression.getRegExpFromId(legalVals);
		if (re==null) {
			if (Certification.DEBUGGING_INFO) System.out.println("Regular Expression didnt compile->"+legalVals+". Line->"+nLine+", identifier->"+id+", value"+val);
			Loggin.Log(Globals.indent()+"ERROR: Critical internal error. Please, restart again");
			throw new Exception();
		}
		if (re.matches(val)==false) {
			Loggin.Log(Globals.indent()+"ERROR: The value \""+val+"\" of the identifier \""+id+"\" in line "+nLine+" does not match with the set of legal values given by \""+legalVals+"\"");
			throw new Exception();
		}
		return (new String[]{id,val});
	}
	
	
	static Vector parseParts (String id, String val, final MyHashMap SPECIAL_PARSING, final MyHashMap SPECIAL_VALUES) throws Exception {
		
		Vector res = new Vector();
		
		if (SPECIAL_PARSING.containsKey(id)) {
			Vector vParts = GenericMethods.fromArrStringToVector(GenericMethods.collectStrings(val,'{','}',","));
			String values[] = (String[])SPECIAL_VALUES.get(id);
			String value = values[0];
			String listType = values[1];
			MyHashMap parts = GenericMethods.fromVectorToMyHashMap(vParts,value);
			MyHashMap legalIds = (MyHashMap)SPECIAL_PARSING.get(id);
			for (Iterator it = parts.keySet().iterator(); it.hasNext(); ) {
				String key = (String)it.next();
				if (legalIds.containsKey(key)==false) {
					Loggin.Log(Globals.indent()+"ERROR: The identifier \""+key+"\" is not a legal "+listType+". Line: "+nLine);
					throw new Exception();
				}
				res.add(new String[]{key,value});
			}
		}
		else {
			res.add(new String[]{id,val});
		}
		
		return res;
	}
}
