import java.io.File;
import java.io.RandomAccessFile;
import java.util.Iterator;
import java.util.Vector;


/**
 * Class in charge of parsing the file that contains the entries configuration and keeping the information
 * regarding with each entry
 * 
 * @author I054742
 *
 */
class EntryConfiguration extends FileParsingMethods {
	
	/* Set of regular expressions for the possible values of all the parameters */
	static private final String RE_ID = "[-0-9A-Za-z_]+";
	static private final String RE_EXTENDS = "\\["+RE_ID+"(,"+RE_ID+")*\\]";
	static private final String RE_FRDNVAL = "(<"+RE_ID+">)+(\\?|\\*|\\+|)";
	static private final String RE_RDNVAL = "("+RE_FRDNVAL+"|"+RE_ID+")";
	static private final String RE_RDN = RE_ID+"="+RE_FRDNVAL+"(,"+RE_ID+"="+RE_RDNVAL+")*,"+RE_ID+"="+RE_ID;
	static private final String RE_LIST = "\\{"+RE_ID+"(,"+RE_ID+")*\\}";
	static private final String RE_BOOL = "((y|Y)(e|E)(s|S))|((n|N)(o|O))";
	static private final String RE_TC_PROP_ID = "[A-Za-z_]+";
	static private final String RE_TC_PROP_LIST = "\\{"+RE_TC_PROP_ID+"(,"+RE_TC_PROP_ID+")*\\}";
	
	/* Association of parameter identifiers with parameter values */
	static private final MyHashMap IDS_LEGAL_VALUES = new MyHashMap() {{
		
		put(Definition.EXTENDS,RE_EXTENDS);
		put(EntryDefinition.RDN,RE_RDN);
		put(Definition.ID,RE_ID);
		put(TEST_CASES_YES,RE_TC_PROP_LIST);
		put(TEST_CASES_NO,RE_TC_PROP_LIST);
		
		put(Globals.PROP_MUST,RE_LIST);
		put(Globals.PROP_MAY,RE_LIST);
		put(Globals.PROP_AUTOMATIC_MUST,RE_LIST);
		put(Globals.PROP_AUTOMATIC_MAY,RE_LIST);
		put(Globals.PROP_ALLOWS_NO_VALUE,RE_LIST);
		put(Globals.PROP_APPENDABLE,RE_LIST);
		put(Globals.PROP_MULTIVALUE,RE_LIST);
		put(Globals.PROP_ALLOWS_DUPLICATE,RE_LIST);
		put(Globals.PROP_MODIFIABLE,RE_LIST);
		put(Globals.PROP_VIRTUAL,RE_LIST);
	}};
	
	
	/* Parameters that need an special parsing */
	static private MyHashMap SPECIAL_PARSING = new MyHashMap() {{
		put(TEST_CASES_YES,Globals.TEST_CASES);
		put(TEST_CASES_NO,Globals.TEST_CASES);
	}};
	
	/* Special values used for the special parsing */
	static private MyHashMap SPECIAL_VALUES = new MyHashMap() {{
		put(TEST_CASES_YES,new String[]{"yes","test case"});
		put(TEST_CASES_NO,new String[]{"no","test case"});
	}};
	
	/* Configuration file */
	static private String CONFIGURATION_FILE = ".\\conf\\EntryConfiguration.conf";
	
	
	/**
	 * Reads one entry configuration
	 * @param raf Entry configuration file
	 * @return An EntryDefinition object if there is some definition more
	 * 		   'null' otherwise
	 * @throws Exception
	 */
	static private EntryDefinition readEntry (RandomAccessFile raf) throws Exception {
		
		MyHashMap currentParams = new MyHashMap();
		
		while (true) {
			
			String line = readLine(raf,0);
			if (line==null) {
				if (currentParams.size()==0) return null;
				else break;
			}
			if (line.startsWith(DEFINITION_SEPARATOR)) break;
			String idVal[] = parseLinePar(line,IDS_LEGAL_VALUES,nLine);
			Vector idVals = parseParts(idVal[0],idVal[1],SPECIAL_PARSING,SPECIAL_VALUES);
			for (int i=0; i<idVals.size(); i++) {
				idVal = (String[])idVals.elementAt(i);
				if (currentParams.containsKey(idVal[0])) {
					Loggin.Log(Globals.indent()+"ERROR: Duplicate parameter \""+idVal[0]+"\" in line: "+nLine);
					throw new Exception();
				}
				currentParams.put(idVal[0],idVal[1]);
			}
		}
		try {
			return new EntryDefinition(currentParams);
		}
		catch (Exception exc) {
			if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in EntryConfiguration.readEntry");
			throw new Exception();
		}
	}
	
	
	/**
	 * Reads the entry configuration file
	 * @throws Exception
	 */
	static void readFile () throws Exception {
		
		Loggin.Log(Globals.indent()+"Reading entry configuration file...");
		Globals.incIndent();
		
		try {
			File file = new File(CONFIGURATION_FILE);
			if (file.exists()==false || file.isDirectory()) {
				
				Loggin.Log(Globals.indent()+"ERROR: The configuration file "+CONFIGURATION_FILE+" does not exist");
				throw new Exception();
			}
			
			RandomAccessFile raf = new RandomAccessFile(CONFIGURATION_FILE,"rw");
			nLine = 0;
			
			while (true) {
				EntryDefinition eDef = readEntry(raf);
				if (eDef!=null) {
					if (eDef.getRdnId()!=null && Globals.getEntDefByRDNID().containsKey(eDef.getRdnId())) {
						Loggin.Log(Globals.indent()+"ERROR: Two entry definitions with the same RDN: \""+eDef.getRDN()+"\"");
						throw new Exception();
					}
					if (Globals.getEntDefByID().containsKey(eDef.getId())) {
						Loggin.Log(Globals.indent()+"ERROR: Two entry definitions with the same ID: \""+eDef.getId()+"\"");
						throw new Exception();
					}
					Globals.addEntDefByID(eDef.getId(),eDef);
					if (eDef.getRdnId()!=null) {
						if (Globals.getEntDefByRDNID().containsKey(eDef.getRdnId())==false) {
							Globals.addEntDefByRDNID(eDef.getRdnId(),eDef);
						}
						else {
							Loggin.Log(Globals.indent()+"ERROR: There at least are two entries specifications with the same DN specification");
							throw new Exception();
						}
					}
					Globals.addProps(eDef.getId(),eDef.getProperties());
				}
				else break;
			}
			
			raf.close();
			
			Loggin.Log(Globals.indent()+"Entries configuration file read successfully");
		}
		catch (Exception exc) {
			if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in EntryConfiguration.readFile");
			Loggin.Log(Globals.indent()+"ERROR: Error reading entries configuration file");
			throw new Exception();
		}
		finally {
			Globals.decIndent();
			Loggin.Log(Globals.indent()+"Entry configuration file reading finished");
		}
	}
}