import java.util.*;

import com.sap.idm.vds.MVDSearchResultEntry;
import com.sap.idm.vds.MVDSearchResults;


/**
 * Class that represents one entry
 * 
 * @author I054742
 *
 */
class Entry implements Cloneable {

	/* Entry's DN */
	private String dn = null;
	/* Set of attributes that the entry can contain */
	private Vector attributes = null;
	/* Special id for the entry dn */
	private String rDnId = null;
	/* Indicates if the set of attributes is sorted */
	private boolean isSorted = false;
	/* Indicates if the set of attributes is sorted and the set of values of each attribute */
	private boolean isAllSorted = false;
	
	
	Entry (String aDn, boolean createRdn, boolean setAtts) {
		this.dn=aDn;
		this.attributes=new Vector();
		if (setAtts) this.setAttributes();
		if (createRdn) this.rDnId=Preprocessing.getAssociatedRdnId(aDn,false,false);
		this.setEntryAtts();
	}
	
	
	Entry (String aDn) {
		this.dn=aDn;
		this.attributes=new Vector();
		this.setAttributes();
		this.rDnId=Preprocessing.getAssociatedRdnId(aDn,false,false);
		this.setEntryAtts();
	}
	
	
	Entry (String aDn, Vector aAttributes) {
		this.dn=aDn;
		this.attributes=aAttributes;
		this.rDnId=Preprocessing.getAssociatedRdnId(aDn,false,false);
		this.setEntryAtts();
	}
	
	Entry (String aDn, Vector aAttributes, boolean ignoreRdnId) {
		this.dn=aDn;
		this.attributes=aAttributes;
		if (ignoreRdnId==false) this.rDnId=Preprocessing.getAssociatedRdnId(aDn,false,false);
		this.setEntryAtts();
	}
	
	
	
	/**
	 * Set the set attributes of the entry
	 */
	void setEntryAtts () {
		
		if (this.attributes==null) return;
		for (int i=0; i<this.attributes.size(); i++) {
			MyAttribute att = (MyAttribute)this.attributes.elementAt(i);
			att.setRelatedEntry(this);
		}
	}
	
	
	/**
	 * Gives the entry's DN
	 * @return The entry's DN
	 */
	String getDn () {
		
		return this.dn;
	}
	
	
	
	Vector getAttributes ()  {
		
		return this.attributes;
	}
	
	
	
	String getRdnId () {
		
		return this.rDnId;
	}
	
	
	/**
	 * Picks up an attribute of the entry
	 * @return A MyAttribute if the entry contains some
	 * 		   'null' otherwise
	 */
	MyAttribute pickUpAttRandomly () {
		
		if (this.attributes.size()==0) return null;
		return (MyAttribute)this.attributes.elementAt(Globals.randGen.nextInt(this.attributes.size()));
	}

	
	/**
	 * Set the attributes of the entry according to the ones that the system has
	 */
	private void setAttributes () {
		
		MVDSearchResults base_sr = Globals.ldapm.search(this.dn,LDAPManagement.BASE,Globals.attr,Globals.urlfilter,
														Globals.szLimit,Globals.tmLimit,false);
		this.setAttributes(base_sr);
	}
	
	
	/**
	 * Gives the most significant identifier of the entry's DN
	 * @return The most significant identifier
	 */
	String getId () {
		
		return this.dn.substring(0,this.dn.indexOf("="));
	}
	
	
	/**
	 * Gives the most significant identifier of the entry's DN
	 * @return The most significant identifier
	 */
	String getVal () {
		
		int limit = this.dn.length();
		if (this.dn.indexOf(",")>=0) limit = this.dn.indexOf(",");
		return this.dn.substring(this.dn.indexOf("=")+1,limit);
	}
	
	
	/**
	 * Gets the complete DN except the part corresponding to the RDN
	 * @return The complete DN except the part corresponding to the RDN
	 */
	String getSP () {
		int index = this.dn.indexOf(",");
		if (index<0) return "";
		return this.dn.substring(index+1);
	}
	
	
	/**
	 * Gives the value of the attribute which name is 'attrName'
	 * @param attrName The attribute name
	 * @return The attribute values
	 */
	Vector getAttrValue (String attrName) {
		
		for (int i=0; i<this.attributes.size(); i++) {
			MyAttribute a = ((MyAttribute)this.attributes.elementAt(i));
			if (a.getName().equalsIgnoreCase(attrName)) {
				return a.getValues();
			}
		}
		
		return null;
	}
	
	
	/**
	 * Writes the set of attributes of the entry
	 * @return
	 */
	String writeData () {
		String res = this.dn;
		for (int i=0; i<this.attributes.size(); i++) {
			res+=" - att"+(i+1)+" -> "+((MyAttribute)this.attributes.elementAt(i)).writeData();
		}
		return res;
	}
	
	
	/**
	 * Creates the DN of the entry according to the linked attributes
	 */
	boolean adjustDnToLinked () {
		
		MyHashMap legalDnSpecs = Globals.getLegalDNSpecs();
		Vector v = (Vector)legalDnSpecs.get(this.getRdnId());
		Vector atts = (Vector)v.elementAt(EntryDefinition.L_DN_ATTS);
		String newDn = "";
		
		for (int i=0; i<atts.size(); i++) {
			
			String attName = (String)atts.elementAt(i);
			if (newDn.length()>0) newDn+=" ";
			try {
				if (Globals.hasPropertyByRdnId(this.rDnId,attName,Globals.PROP_VIRTUAL)) {
					String val = null;
					int tries = 0;
					do {
						val = Globals.getIllegalValue(this.rDnId,attName);
						if (val==null) continue;
					} while(++tries<100 && val==null);
					if (val==null) newDn+="virtual";
					else newDn+=val;
				}
				else {
					newDn+=(String)this.getAttrValue(attName).elementAt(0);
				}
			}
			catch (Exception exc) {return false;}
		}
		newDn=this.getId()+"="+newDn;
		String oldDn = this.dn;
		int indexOf = oldDn.indexOf(',');
		String restOfDn = "";
		if (indexOf>=0) restOfDn=oldDn.substring(indexOf+1);
		newDn+=","+restOfDn;
		this.dn=newDn;
		this.rDnId=Preprocessing.getAssociatedRdnId(this.dn,false,false);
		this.isAllSorted=this.isSorted=false;
		
		return true;
	}
	
	
	
	/**
	 * Indicates if the entry is sorted by attribute name
	 * @return 'true' if it is sorted
	 * 		   'false' otherwise
	 */
	boolean isSorted () {
		
		return this.isSorted;
	}
	
	
	/**
	 * Indicates if the entry is sorted by attribute name and its attributes have their values sorted
	 * @return 'true' if it is sorted
	 * 		   'false' otherwise
	 */
	boolean isAllSorted () {
		
		return this.isAllSorted;
	}
	
	
	/**
	 * Gives the current number of attributes of this entry
	 * @return The number of attributes
	 */
	int getNumbOfAttrs () {
		
		return this.attributes.size();
	}
	
	
	/**
	 * Gives the attribute in the position 'index'
	 * @param index Position where the required attribute is
	 * @return The attribute at position 'index'
	 */
	MyAttribute getAttributeAt (int index) {
		
		return (MyAttribute)this.attributes.elementAt(index);
	}
	
	
	/**
	 * Sets the set of attributes taking the attributes from 'sr'
	 * @param sr A set of attributes and values
	 */
	void setAttributes (MVDSearchResults sr) {
		
		if (sr==null) return;
		if (sr.size()==0) return;
		MVDSearchResultEntry e = (MVDSearchResultEntry)sr.firstElement();
		HashMap attrs = e.getAttrAndValues();
		for (Iterator it=attrs.keySet().iterator(); it.hasNext(); ) {
			String key = (String)it.next();
			Vector v = (Vector)attrs.get(key);
			this.attributes.add(new MyAttribute(key,v,this));
		}
		this.isSorted=this.isAllSorted=false;
	}
	
	
	/**
	 * Sets the RDN id
	 * @param newIdValue New set DN id value
	 */
	void setRdnId (String newRdnId) {
		
		this.dn = newRdnId+this.dn.substring(this.dn.indexOf("="));
	}
	
	
	/**
	 * Sets the entry DN
	 * @param newDn The new DN
	 */
	void setDn (String newDn) {
		
		this.dn=newDn;
	}
	
	
	/**
	 * If there is an attribute with similar name to the name of 'aA' it will be set to 'aA'
	 * @param aA The attribute
	 */
	void setAttribute (MyAttribute aA) {
		
		for (int i=0; i<this.attributes.size(); i++) {
			
			MyAttribute a = (MyAttribute)this.attributes.elementAt(i);
			if (a.getName().equalsIgnoreCase(aA.getName())) {
				this.attributes.setElementAt(aA,i);
				break;
			}
		}
	}
	
	
	/**
	 * Checks if the entry fits the configuration information
	 * @param log When 'true' then the log will be activated
	 * @return 'true' if there is no irregularities in this entry
	 * 		   'false' otherwise
	 */
	boolean fitsCompletelyConfig (boolean log) {
		
		try  {
			MyHashMap attList = Globals.getAttListByRdnId(this.rDnId);
			for (Iterator attIt = attList.keySet().iterator(); attIt.hasNext(); ) {
				String attName = (String)attIt.next();
				if ((Globals.hasPropertyByRdnId(this.rDnId,attName,Globals.PROP_MUST) || 
					 Globals.hasPropertyByRdnId(this.rDnId,attName,Globals.PROP_AUTOMATIC_MUST))
					 &&
					 this.containsAttributeByName(attName)==null
					 ) {
					if (log) Error.missedMandAttribute(this.dn,attName);
					return false;
				}
			}
		
			for (int i=0; i<this.attributes.size(); i++) {
				MyAttribute att = (MyAttribute)this.attributes.elementAt(i);
				if (attList.containsKey(att.getName())==false) continue;
				for (int j=0; j<att.getNumbOfValues(); j++) {
					String val = att.getValueAt(j);
					if (Globals.isLegalValue(this.rDnId,att.getName(),val)==false) {
						if (log) Error.illegalAttValue(this.dn,att.getName(),val);
						return false;
					}
				}
			}
			
			return true;
		
		}
		catch (Exception exc) {
			if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in Entry.fitsCompletelyConfig");
			Error.internalUnexpected();
			return false;
		}
	}
	
	
	/**
	 * Checks if the entry was created completely legal
	 * @param log When 'true' then the log will be activated
	 * @return 'true' if there is no irregularities in this entry
	 * 		   'false' otherwise
	 */
	boolean createdCompletelyLegal (boolean log) {
		
		try {
			if (this.rDnId==null) return false;
			
			MyHashMap attList = Globals.getAttListByRdnId(this.rDnId);
			
			/* Checks if the entry contains all the mandatory attributes */
			for (Iterator attIt = attList.keySet().iterator(); attIt.hasNext(); ) {
				String attName = (String)attIt.next();
				if ((Globals.hasPropertyByRdnId(this.rDnId,attName,Globals.PROP_MUST)  && this.containsAttributeByName(attName)==null)) {
					if (log) Loggin.Log(Globals.indent()+"The entry has not all the mandatory attributes");
					return false;
				}
			}
			
			/* Checks if every value of every attribute is legal */
			for (int i=0; i<this.attributes.size(); i++) {
				MyAttribute att = (MyAttribute)this.attributes.elementAt(i);
				if (attList.containsKey(att.getName())==false) {
					if (log) Loggin.Log(Globals.indent()+"The attribute: \""+att.getName()+"\" has not definition");
					return false;
				}
				if (Globals.hasPropertyByRdnId(this.rDnId,att.getName(),Globals.PROP_MUST)==false && Globals.hasPropertyByRdnId(this.rDnId,att.getName(),Globals.PROP_MAY)==false) {
					if (log) Loggin.Log(Globals.indent()+"The attribute: \""+att.getName()+"\" neither has the property MUST nor MAY");
					return false;
				}
				for (int j=0; j<att.getNumbOfValues(); j++) {
					String val = att.getValueAt(j);
					if (Globals.isLegalValue(this.rDnId,att.getName(),val)==false) {
						if (log) Loggin.Log(Globals.indent()+"The value: \""+val+"\" of the attribute: \""+att.getName()+"\" is illegal");
						return false;
					}
				}
			}
			
			/* Checks the linked attributes */
			Vector rDnParts = (Vector)((Vector)((EntryDefinition)Globals.getEntDefByRDNID().get(this.rDnId)).getRdnParts()).firstElement();
			Vector linkedAtts = (Vector)rDnParts.elementAt(EntryDefinition.L_DN_ATTS);
			
			if (Globals.hasPropertyByRdnId(this.rDnId,((String)linkedAtts.firstElement()),Globals.PROP_VIRTUAL)) {
				
				Vector regExps = (Vector)rDnParts.elementAt(EntryDefinition.L_DN_RE);
				RegularExpression rexp = (RegularExpression)regExps.firstElement();
				if (rexp.matches(this.getVal())==false) {
					if (log) Loggin.Log(Globals.indent()+"The DN has illegal values");
					return false;
				}
			}
			else {
				String compVal = "";
				for (int i=0; i<linkedAtts.size(); i++) {
					String attName = (String)linkedAtts.elementAt(i);
					MyAttribute att = this.containsAttributeByName(attName);
					if (compVal.length()>0) compVal+=" ";
					compVal+=att.getValueAt(0);
				}
				if (compVal.equals(this.getVal())==false) {
					if (log) Loggin.Log(Globals.indent()+"The DN does not match with the linked attributes");
					return false;
				}
			}
			
			return true;
		}
		catch (Exception exc) {
			if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in Entry.createdCompletelyLegal");
			if (log) Error.internalUnexpected();
			return false;
		}
	}
	
	
	/**
	 * Adds the attribute 'a'
	 * @param a The attribute to be added
	 */
	void addAttribute (MyAttribute a) {
		
		if (a==null) return;
		a.setRelatedEntry(this);
		this.attributes.add(a);
		this.isSorted=this.isAllSorted=false;
	}
	
	
	/**
	 * Tries to remove the attribute 'a'
	 * @param a The attribute
	 * @return 'true' if the attribute existed
	 * 		   'false' otherwise
	 */
	boolean removeAttribute (MyAttribute a) {
		
		for (int i=0; i<this.attributes.size(); i++) {
			
			if (a.getName().equalsIgnoreCase(((MyAttribute)this.attributes.elementAt(i)).getName())) {
				this.attributes.removeElementAt(i);
				return true;
			}
		}
		return false;
	}
	
	
	/**
	 * Checks if the entry contains some attribute as 'a'
	 * @param a The attribute
	 * @return 'true' if it contains the attribute
	 * 		   'false' otherwise
	 */
	boolean containsAttribute (MyAttribute a) {
		
		for (int i=0; i<this.attributes.size(); i++) {
			if (((MyAttribute)this.attributes.elementAt(i)).equals(a)) {
				return true;
			}
		}
		return false;
	}
	
	
	/**
	 * Checks if the entry contains some attribute which name is 'name'
	 * @param name The attribute name
	 * @return 'true' if it contains the attribute
	 * 		   'false' otherwise
	 */
	MyAttribute containsAttributeByName (String name) {
		
		for (int i=0; i<this.attributes.size(); i++) {
			if (((MyAttribute)this.attributes.elementAt(i)).getName().equalsIgnoreCase(name)) {
				return ((MyAttribute)this.attributes.elementAt(i));
			}
		}
		return null;
	}
	
	
	/**
	 * Checks if the entry contains some attribute which name is 'name'
	 * @param name The attribute name
	 * @return 'true' if it contains the attribute
	 * 		   'false' otherwise
	 */
	MyAttribute givesAttributeByName (String name) {
		
		for (int i=0; i<this.attributes.size(); i++) {
			if (((MyAttribute)this.attributes.elementAt(i)).getName().equalsIgnoreCase(name)) {
				return (MyAttribute)this.attributes.elementAt(i);
			}
		}
		return null;
	}
	
	
	
	/**
	 * Checks if the entry contains all the attributes given in 'attrs'. It just look at the attribute name
	 * @param attrs Set of attribute names
	 * @return 'true' if it contains them
	 * 		   'false' otherwise
	 */
	boolean containsAllAttributes (String []attrs) {
		
		for (int i=0; i<attrs.length; i++) {
			if (this.containsAttributeByName(attrs[i])==null) return false; 
		}
		return true;
	}	
	
	
	/**
	 * Sorts the set of attributes by name
	 */
	void sort () {
		
		Collections.sort(this.attributes,new CompareAttributes());
		this.isSorted=true;
	}
	
	
	/**
	 * Sorts the set of attributes by name and the value of the attributes
	 */
	void sortAll () {
		
		if (this.isSorted==false) this.sort();
		for (int i=0; i<this.attributes.size(); i++) {
			MyAttribute a = (MyAttribute)this.attributes.elementAt(i);
			if (a.isSorted()==false) a.sort();
		}
		this.isAllSorted=true;
	}
	
	
	/**
	 * Creates a clone of an entry object
	 */
	public Object clone () {
		
		Entry e = new Entry(this.dn,new Vector());
		for (int i=0; i<this.attributes.size(); i++) {
			MyAttribute a = this.getAttributeAt(i);
			e.addAttribute((MyAttribute)a.clone());
		}
		e.rDnId=this.rDnId;
		return e;
	}
	
	
	/**
	 * Checks if the current entry and the entry 'e' are equals if we look only at their attribute names
	 * @param e The entry to be compared with the one that calls this method
	 * @return 'true' if the entries are equal
	 * 		   'false' otherwise
	 */
	boolean equalsOneLevel (Entry e) {
		
		if (GeneralConfiguration.isDnEqualsByValues()) {
			if (this.equalsDnByValues(e)==false) return false;
		}
		else if (e.getDn().equalsIgnoreCase(this.dn)==false) return false;
		if (this.attributes.size()!=e.getNumbOfAttrs()) return false;
		if (this.isSorted==false) this.sort();
		if (e.isSorted()==false) e.sort();
		for (int i=0; i<this.attributes.size(); i++) {
			MyAttribute a1 = e.getAttributeAt(i);
			MyAttribute a2 = this.getAttributeAt(i);
			if (a1.getName().equalsIgnoreCase(a2.getName())==false) return false;
		}
		return true;
	}
	
	
	/**
	 * Checks if two entry's DNs are equal looking only at their DN values
	 * @param e The entry to be compared with the one that called this method
	 * @return 'true' if the two entry's DNs are equal by value
	 * 		   'false' otherwise
	 */
	boolean equalsDnByValues (Entry entSystem) {
		
		String values1[] = entSystem.getDn().split(",");
		String values2[] = this.dn.split(",");
		
		if (values1.length!=values2.length) return false;
		for (int i=0; i<values1.length; i++) {
			if (values1[i].split("=")[1].equalsIgnoreCase(values2[i].split("=")[1])==false) return false;
		}
		return true;
	}
	
	
	String printAttributes() {
		
		this.sortAll();
		String res = "[";
		for (int i=0; i<this.getNumbOfAttrs(); i++) {
			MyAttribute ma = this.getAttributeAt(i);
			if (i>0) res+=",";
			res+=ma.getName()+"="+ma.getValues();
		}
		return res+"]";
	}
	
	
	/**
	 * Compares the current entry with the entry 'e'. Depending of the entry definition of the type of entry
	 * the entry will be compared by DN values or by the whole DN
	 * @param e The entry to be compared with the current one
	 * @return 0 if both entries are equal
	 * 		   >0 if this>e
	 * 		   <0 if this<e
	 */
	int compare (Entry e) {
		
		if (GeneralConfiguration.isDnEqualsByValues()) {
			return this.compareDnByValues(e);
		}
		else return this.dn.compareToIgnoreCase(e.getDn());
	}
	
	
	/**
	 * Compare the DN's of this entry and the entry 'e' by the values of their DNs
	 * @param e The entry to be compared with this one
	 * @return 0 if both entries are equal
	 * 		   >0 if this>e
	 * 		   <0 if this<e
	 */
	int compareDnByValues (Entry e) {
		
		String values1[] = e.getDn().split(",");
		String values2[] = this.dn.split(",");
		
		if (values1.length>values2.length) {
			String aux[] = values1;
			values1=values2;
			values2=aux;
		}
		for (int i=0; i<values1.length; i++) {
			int val = values2[i].split("=")[1].compareToIgnoreCase(values1[i].split("=")[1]);
			if (val!=0) return val;
		}
		if (values1.length!=values2.length) return -1;
		return 0;
	}
	

	/**
	 * Checks if this entry's DN is the same that 'dn' compared by value
	 * @param dn The DN to be compared with the one of this entry
	 * @return 'true' if they are equal
	 * 		   'false' otherwise
	 */
	boolean equalsDnByValues (String dn) {
		
		String values1[] = dn.split(",");
		String values2[] = this.dn.split(",");
		
		if (values1.length!=values2.length) return false;
		for (int i=0; i<values1.length; i++) {
			if (values1[i].split("=")[1].equalsIgnoreCase(values2[i].split("=")[1])==false) return false;
		}
		return true;
	}
	
	
	/**
	 * Checks if this entry and the entry 'e' are equals. Checking if each of their attributes are equal
	 * @param e The entry to be compared with this one
	 * @return 'true' if both entries are equal
	 * 		   'false' otherwise
	 */
	boolean equals (Entry e) {
		
		boolean res = true;
		
		if (GeneralConfiguration.isDnEqualsByValues()) res = this.equalsDnByValues(e);
		else res = (e.getDn().equalsIgnoreCase(this.dn));
		
		res = (res && this.attributes.size()==e.getNumbOfAttrs());
		if (res && this.isAllSorted==false) this.sortAll();
		if (res && e.isAllSorted()==false) e.sortAll();
		for (int i=0; res && i<this.attributes.size(); i++) {
			MyAttribute a1 = e.getAttributeAt(i);
			MyAttribute a2 = this.getAttributeAt(i);
			res = (a1.equals(a2));
		}
		if (res==false) {
			String msg = Globals.indent()+"The next two entries were expected to be equal but they are not:\n";
			msg+=Globals.indent()+"Expected entry: "+this.printAttributes()+"\n";
			msg+=Globals.indent()+"Entry according to connector: "+e.printAttributes()+"\n";
			Globals.errorMessage = msg;
			Loggin.Log(Globals.errorMessage);
		}
		return res;
	}
	
	
	/**
	 * Checks if this entry and the entry 'e' are equals. Checking if each of their attributes are equal. But in this case only the attributes which have definition and which have not the property automatic will be taken into consideration
	 * @param e The entry to be compared with this one
	 * @return 'true' if both entries are equal
	 * 		   'false' otherwise
	 */
	boolean equalsOnlyDefinitionAndNotAutomatic (Entry entSystem) {
		
		if (GeneralConfiguration.isDnEqualsByValues()) {
			if (this.equalsDnByValues(entSystem)==false) {
				Error.diffEntries(this,entSystem);
				return false;
			}
		}
		else if (entSystem.getDn().equalsIgnoreCase(this.dn)==false) {
			Error.diffEntries(this,entSystem);
			return false;
		}
		
		Vector v = new Vector();
		String eId = ((EntryDefinition)Globals.getEntDefFromRDNID(this.rDnId)).getId();
		for (int i=0; i<this.attributes.size(); i++) {
			MyAttribute ma = (MyAttribute)this.attributes.elementAt(i);
			String attName = ma.getName();
			if (Globals.existsAtt(eId,attName)==false) continue;
			try {
				if (Globals.hasPropertyByEntDefId(eId,attName,Globals.PROP_AUTOMATIC_MAY) || Globals.hasPropertyByEntDefId(eId,attName,Globals.PROP_AUTOMATIC_MUST)) continue;
			}
			catch (Exception exc) {
				if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in Entry.equalsOnlyDefinitionAndNotAutomatic");
			}
			v.add(ma);
		}
		Entry entCTT = new Entry(this.getDn(),v);
		
		v = new Vector();
		eId = ((EntryDefinition)Globals.getEntDefFromRDNID(entSystem.getRdnId())).getId();
		for (int i=0; i<entSystem.getNumbOfAttrs(); i++) {
			MyAttribute ma = entSystem.getAttributeAt(i);
			String attName = ma.getName();
			if (Globals.existsAtt(eId,attName)==false) continue;
			try {
				if (Globals.hasPropertyByEntDefId(eId,attName,Globals.PROP_AUTOMATIC_MAY) || Globals.hasPropertyByEntDefId(eId,attName,Globals.PROP_AUTOMATIC_MUST)) continue;
			}
			catch (Exception exc) {
				if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in Entry.equalsOnlyDefinitionAndNotAutomatic");
			}
			v.add(ma);
		}
		Entry entSystem2 = new Entry(entSystem.getDn(),v);
		
		if (entCTT.getNumbOfAttrs()!=entSystem2.getNumbOfAttrs()) {
			Error.diffEntries(entCTT,entSystem);
			return false;
		}
		if (entCTT.isAllSorted()==false) entCTT.sortAll();
		if (entSystem2.isAllSorted()==false) entSystem2.sortAll();
		for (int i=0; i<entCTT.attributes.size(); i++) {
			MyAttribute a1 = entCTT.getAttributeAt(i);
			MyAttribute a2 = entSystem2.getAttributeAt(i);
			if (a1.equals(a2)==false) {
				Error.diffEntries(entCTT,entSystem);
				return false;
			}
		}
		return true;
	}
}
