import java.util.*;


/**
 * Class in charge of checking the configuration files and preparing the data contained into them
 * @author I054742
 *
 */
public class Configurations {

	/**
	 * Read the configuration files
	 * @return 'true' if the reading was correct
	 * 		   'false' otherwise
	 */
	static boolean readFiles () {
		
		Loggin.Log(Globals.indent()+"Starting reading configuration files...");
		
		try {
			
			Globals.incIndent();
			
			GeneralConfiguration.readFile();
			if (GeneralConfiguration.getTime()==0) {
				Loggin.Log(Globals.indent()+"WARNING: The whole process will be stopped since the maximum allowed time is zero");
				return false;
			}
			Globals.resetRandom(GeneralConfiguration.getSeed());
			if (GeneralConfiguration.getTestType().equalsIgnoreCase(Tester.EXPECTEDS)) return true;
			RegularExpressionFile.readFile();
			EntryConfiguration.readFile();
			AttributeConfiguration.readFile();
			
			return true;
		}
		catch (Exception exc) {
			if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in Configurations.readFiles");
			Loggin.Log(Globals.indent()+"ERROR: Some error was found while reading configuration files");
			return false;
		}
		finally {
			Globals.decIndent();
			Loggin.Log(Globals.indent()+"Configuration files reading finished");
		}
	}
	
	
	/**
	 * Makes a general checking of the information obtained from the configuration files
	 * @return 'true' if no irregularities were found
	 * 		   'false' otherwise
	 */
	static boolean checking () {
		
		Loggin.Log(Globals.indent()+"Starting configuration checking...");
		try {
			
			Globals.incIndent();
			
			if (legalDNSpecsParsing()==false) return false;
			if (attVsEntChecking()==false) return false;
			if (attByEntCollisions()==false) return false;
			if (GeneralConfiguration.getTestType().equalsIgnoreCase(Tester.INSTRUCTIONS)) {
				InstructionsFile.readFile();
			}
			
			return true;
		}
		catch (Exception exc) {
			if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in Configurations.checking");
			Loggin.Log(Globals.indent()+"ERROR: Some error was found while configuration checking");
			return false;
		}
		finally {
			Globals.decIndent();
			Loggin.Log(Globals.indent()+"Configuration checking finished");
		}
	}
	
	
	/**
	 * Checks if every attribute definition has correspondence with some entry definition and the opposite
	 * @return 'true' if no irregularities were found
	 * 		   'false' otherwise
	 */
	static private boolean attVsEntChecking () {
		
		MyHashMap entDefs = Globals.getEntDefByID();
		for (Iterator it=entDefs.keySet().iterator(); it.hasNext(); ) {
			
			String eId = (String)it.next();
			EntryDefinition eDef = (EntryDefinition)entDefs.get(eId);
			MyHashMap attList = eDef.getAttrList();
			for (Iterator itA=attList.keySet().iterator(); itA.hasNext(); ) {
				
				String attName = (String)itA.next();
				String identif = attName+Globals.RDN_SEP+eId;
				Object attDef = Globals.getAttDefByAtt_Separ_Ent().get(identif);
				if (attDef==null) {
					Loggin.Log(Globals.indent()+"ERROR: The attribute: \""+attName+"\" of the entry definition which ID is: \""+eId+"\" has not any associated attribute definition");
					return false;
				}
			}
		}
		return true;
	} 
	
	
	/**
	 * Checks if the collisions between the attribute properties
	 * @return 'true' if no collision was found
	 * 		   'false' otherwise
	 */
	static private boolean attByEntCollisions () {
		
		MyHashMap entDefs = Globals.getEntDefByID();
		/* Pass through each entry */
		for (Iterator it=entDefs.keySet().iterator(); it.hasNext(); ) {
			
			String eId = (String)it.next();
			EntryDefinition eDef = (EntryDefinition)entDefs.get(eId);
			MyHashMap attList = eDef.getAttrList();
			/* For each attribute of each entry */
			for (Iterator itA=attList.keySet().iterator(); itA.hasNext(); ) {
				
				String attName = (String)itA.next();
				MyHashMap propsByEnt = (MyHashMap)Globals.getPropsByAtt().get(attName);
				MyHashMap props = (MyHashMap)propsByEnt.get(eId);
				boolean okMust = false;
				/* Checks if the attribute has at least one of the mandatory properties */
				for (Iterator itM=Globals.getMUST_AT_LEAST_ONE_PROP().keySet().iterator(); itM.hasNext(); ) {
					String idM = (String)itM.next();
					if (props.containsKey(idM)) {
						String yesNo = (String)props.get(idM);
						if (yesNo.equalsIgnoreCase("yes")) {
							okMust = true;
							break;
						}
					}
				}
				/* If no mandatory property then error */
				if (okMust==false) {
					Loggin.Log(Globals.indent()+"ERROR: The attribute: \""+attName+"\" of the entry: \""+eId+"\" has no mandatory property");
					return false;
				}
				/* Checks the property constraints */
				for (Iterator itP=Globals.getPROPERTIES_CONSTRAINTS().keySet().iterator(); itP.hasNext(); ) {
					
					String pId = (String)itP.next();
					if (props.containsKey(pId)) {
						String val = (String)props.get(pId);
						if (val.equalsIgnoreCase("yes")) {
							MyHashMap constraints = (MyHashMap)Globals.getPROPERTIES_CONSTRAINTS().get(pId);
							/* For each constraint checks that the attribute properties are legal */
							for (Iterator itC=constraints.keySet().iterator(); itC.hasNext(); ) {
								
								String propConstrId = (String)itC.next();
								String yesNo = (String)constraints.get(propConstrId);
								boolean correct = true;
								String expectedValue = null;
								if (yesNo.equalsIgnoreCase("yes")) {
									if (props.containsKey(propConstrId)==false) {
										expectedValue = "yes";
										correct = false;
									}
									else {
										String realPropValue = (String)props.get(propConstrId);
										if (realPropValue.equalsIgnoreCase("no")) {
											expectedValue = "yes";
											correct = false;
										}
									}
								}
								else {
									if (props.containsKey(propConstrId)) {
										String realPropValue = (String)props.get(propConstrId);
										if (realPropValue.equals("yes")) {
											expectedValue = "no";
											correct = false;
										}
									}
								}
								if (correct==false) {
									Loggin.Log(Globals.indent()+"ERROR: Incompatibility between the properties of the attribute: \""+attName+"\" of the entry: \""+eId+"\"");
									Loggin.Log(Globals.indent()+"    The property: \""+pId+"\" has value \"yes\" and the property: \""+propConstrId+"\" should have value \""+expectedValue+"\"");
									return false;
								}
							}
						}
					}
				}
			}
		}
		return true;
	}
	
	
	/**
	 * Creates the vector of attributes and the vector of regular expressions of every legal DN specification
	 * @return 'true' if no irregularities were found
	 * 		   'false' otherwise
	 */
	static private boolean legalDNSpecsParsing () {
		
		MyHashMap all = Globals.getLegalDNSpecs();
		/* Goes through the set of all the legal DN specifications */
		for (Iterator it=all.keySet().iterator(); it.hasNext(); ) {
			
			String rDnId = (String)it.next();
			Vector v = (Vector)all.get(rDnId);
			String val = (String)v.elementAt(EntryDefinition.L_DN_VAL);
			EntryDefinition eDef = (EntryDefinition)Globals.getEntDefByRDNID().get(rDnId);
			Vector reVect = new Vector();
			Vector attrVect = new Vector();
			/* Constants values */
			if (val.contains("<")==false) {
				if (Globals.getRegExpDefs().containsKey(val)) {
					Loggin.Log(Globals.indent()+"ERROR: Illegal constant used in the RDN: \""+(String)v.elementAt(5)+"\". It is the same as a regular expression definition");
					return false;
				}
				RegularExpression re = RegularExpression.getRegExpFromId(val);
				if (re==null) {
					Loggin.Log(Globals.indent()+"ERROR: Illegal constant used in the RDN: \""+(String)v.elementAt(5)+"\"");
					return false;
				}
				reVect.add(re);
			}
			/* Linked attributes */
			else {
				/* 0 = not yet defined, 1 = mandatory, 2 = virtual */
				int typeLinked = 0;
				StringTokenizer st = new StringTokenizer(val,">",false);
				/* For each of the linked attributes */
				while (st.hasMoreTokens()) {
					String token = st.nextToken();
					token = GenericMethods.cleanBrackets(token,'<','>');
					MyHashMap attList = eDef.getAttrList();
					if (attList==null || attList.containsKey(token)==false) {
						Loggin.Log(Globals.indent()+"ERROR: The attribute: \""+token+"\" has no definition in the entry with RDN: \""+eDef.getRDN()+"\"");								
						return false;
					}
					String entId = eDef.getId();
					String attName = token;
					String identif = attName+Globals.RDN_SEP+entId;
					AttributeDefinition attDef = (AttributeDefinition)Globals.getAttDefByAtt_Separ_Ent().get(identif);
					try {
						int currTypeLinked = 0;
						
						if (Globals.hasPropertyByEntDefId(entId,attName,Globals.PROP_VIRTUAL)) {
							if (typeLinked==2) {
								Loggin.Log(Globals.indent()+"ERROR: Only one virtual attribute is allowed by entry");
								return false;
							}
							currTypeLinked = 2;
						}
						else if (Globals.hasPropertyByEntDefId(entId,attName,Globals.PROP_AUTOMATIC_MUST)) currTypeLinked = 1;
						
						if (typeLinked==0) typeLinked = currTypeLinked;
						else if (typeLinked!=currTypeLinked) {
							Loggin.Log(Globals.indent()+"ERROR: Not allowed to mix virtual attributes with mandatory attributes for the linked attributes");
							return false;
						}
					}
					catch (Exception exc) {
						if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in Configurations.legalDNSpecsParsing");
						Error.internalUnexpected();
						return false;
					}
					/* Adds a RE to the vector of regular expressions */
					reVect.add(attDef.getLegals());
					/* Adds an attribute name to the vector of attribute names */
					attrVect.add(attName);
					/* Adds the linked attributes to the properties lists */
					MyHashMap props = new MyHashMap();
					props.put(Globals.PROP_LINKED,"yes");
					MyHashMap propsList = new MyHashMap();
					propsList.put(attName,props);
					Globals.addProps(entId,propsList);
				}
				eDef.setLinkedAttrs(attrVect);
			}
			/* Adds the vector of regular expressions */
			v.setElementAt(reVect,EntryDefinition.L_DN_RE);
			/* Adds the vector of attributes */
			v.setElementAt(attrVect,EntryDefinition.L_DN_ATTS);
		}
		
		return true;
	}
}
