import java.util.*;
import java.io.*;


/**
 * Class in charge of parsing the file that contains the attributes configuration and saving the information related to each attribute
 * 
 * @author I054742
 *
 */
class AttributeConfiguration extends FileParsingMethods {
	
	/* Legal identifiers */
	static final String EXTENDS = Definition.EXTENDS;
	static final String ATTRS = AttributeDefinition.ATTRS;
	static final String ID = Definition.ID;
	static final String LEGAL_VALUES = AttributeDefinition.LEGAL_VALUES;
	static final String PROPERTIES_YES = "propertiesYes";
	static final String PROPERTIES_NO = "propertiesNo";
	
	/* Legal values for the values related to each identifier */
	static private final String RE_ID = "[-0-9A-Za-z_]+";
	static private final String RE_ATTR_LIST_ID = "("+RE_ID+"|"+RE_ID+"@"+RE_ID+")";
	static private final String RE_ATTR_LIST = "\\{"+RE_ATTR_LIST_ID+"(,"+RE_ATTR_LIST_ID+")*\\}";
	static private final String RE_EXTENDS = "\\["+RE_ID+"(,"+RE_ID+")*\\]";
	//static private final String RE_BOOL = "((y|Y)(e|E)(s|S))|((n|N)(o|O))";
	static private final String RE_TC_PROP_ID = "[A-Za-z_]+";
	static private final String RE_TC_PROP_LIST = "\\{"+RE_TC_PROP_ID+"(,"+RE_TC_PROP_ID+")*\\}";
	
	/* Relations between identifiers and regular expressions */
	static private final MyHashMap IDS_LEGAL_VALUES = new MyHashMap() {{
		
		put(ID,RE_ID);
		put(EXTENDS,RE_EXTENDS);
		put(ATTRS,RE_ATTR_LIST);
		put(LEGAL_VALUES,RE_ID);
		
		put(TEST_CASES_YES,RE_TC_PROP_LIST);
		put(TEST_CASES_NO,RE_TC_PROP_LIST);
		put(PROPERTIES_YES,RE_TC_PROP_LIST);
		put(PROPERTIES_NO,RE_TC_PROP_LIST);
	}};
	
	/* List of legal test cases and properties */
	static private MyHashMap SPECIAL_PARSING = new MyHashMap() {{
		put(TEST_CASES_YES,Globals.ATT_TEST_CASES);
		put(TEST_CASES_NO,Globals.ATT_TEST_CASES);
		put(PROPERTIES_YES,Globals.PROPERTIES);
		put(PROPERTIES_NO,Globals.PROPERTIES);
	}};
	
	/* Used to speciallist parsing */
	static private MyHashMap SPECIAL_VALUES = new MyHashMap() {{
		put(TEST_CASES_YES,new String[]{"yes","test case"});
		put(TEST_CASES_NO,new String[]{"no","test case"});
		put(PROPERTIES_YES,new String[]{"yes","property"});
		put(PROPERTIES_NO,new String[]{"no","property"});
	}};
	
	
	/* Configuration file */
	static private String CONFIGURATION_FILE = ".\\conf\\AttributeConfiguration.conf";
	
	
	/**
	 * Reads a complete attribute configuration
	 * @param raf Attribute configuration file
	 * @return The next attribute specification
	 * 		   'null' if there is no more attribute definition
	 * @throws Exception Thrown if the current attribute specification is illegal
	 */
	static private AttributeDefinition readAttribute (RandomAccessFile raf) throws Exception {
		
		MyHashMap currentParams = new MyHashMap();
		
		while (true) {
			
			/* Reads one line */
			String line = readLine(raf,0);
			if (line==null) {
				if (currentParams.size()==0) return null;
				else break;
			}
			/* If it finds a separator of attribute specifications then it stops reading */
			if (line.startsWith(DEFINITION_SEPARATOR)) break;
			/* Parses one line giving identifier and value */
			String idVal[] = parseLinePar(line,IDS_LEGAL_VALUES,nLine);
			/* Makes a special parsing for lists of test cases and properties */
			Vector idVals = parseParts(idVal[0],idVal[1],SPECIAL_PARSING,SPECIAL_VALUES);
			/* For each identifier (including list parsing) ... */
			for (int i=0; i<idVals.size(); i++) {
				idVal = (String[])idVals.elementAt(i);
				if (currentParams.containsKey(idVal[0])) {
					Loggin.Log(Globals.indent()+"ERROR: Duplicate parameter \""+idVal[0]+"\" in line: "+nLine);
					throw new Exception();
				}
				currentParams.put(idVal[0],idVal[1]);
			}
		}
		try {
			/* Tries creating a legal attribute definition */
			return new AttributeDefinition(currentParams);
		}
		catch (Exception exc) {
			if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in AttributeConfiguration.readAttribute");
			throw new Exception();
		}
	}
	
	
	/**
	 * Reads the attribute configuration file
	 * @throws Exception
	 */
	static void readFile () throws Exception {
		
		Loggin.Log(Globals.indent()+"Reading attributes configuration file...");
		Globals.incIndent();
		
		try {
			File file = new File(CONFIGURATION_FILE);
			if (file.exists()==false || file.isDirectory()) {
				
				Loggin.Log(Globals.indent()+"ERROR: The configuration file "+CONFIGURATION_FILE+" does not exist");
				throw new Exception();
			}
			
			nLine = 0;
			RandomAccessFile raf = new RandomAccessFile(CONFIGURATION_FILE,"rw");
			
			while (true) {
				/* Reads the next attribute definition */
				AttributeDefinition aDef = readAttribute(raf);
				if (aDef!=null) {
					/* Saves the attribute definition */
					Globals.addAttDefByID(aDef.getId(),aDef);
					MyHashMap atts = aDef.getAttrList();
					if (atts!=null) {
						/* Creates the set of properties and the relations between attribute definitions and entry definitions */
						for (Iterator it=atts.keySet().iterator(); it.hasNext(); ) {
							String attrName = (String)it.next();
							MyHashMap ents = (MyHashMap)atts.get(attrName);
							for (Iterator itE=ents.keySet().iterator(); itE.hasNext(); ) {
								String eId = (String)itE.next();
								MyHashMap props = new MyHashMap();
								props.put(attrName,aDef.getProperties());
								Globals.addProps(eId,props);
								String idWithSep = attrName+Globals.RDN_SEP+eId;
								Globals.addAttDefByAtt_Separ_Ent(idWithSep,aDef);
							}
						}
					}
				}
				else break;
			}
			
			raf.close();
			Loggin.Log(Globals.indent()+"Attributes configuration file read successfully");
		}
		catch (Exception exc) {
			if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in AttributeConfiguration.readFile");
			Loggin.Log(Globals.indent()+"ERROR: Error reading attributes configuration file");
			throw new Exception();
		}
		finally {
			Globals.decIndent();
			Loggin.Log(Globals.indent()+"Attribute configuration file reading finished");
		}
	}
}