/*
*   Author:   Scott Bailey
*   License:  BSD
*
*
*/


-- Returns the length of p1 in seconds
CREATE OR REPLACE FUNCTION duration(period)
RETURNS numeric AS
$$
    SELECT EXTRACT(EPOCH FROM
      (next($1) - first($1)))::numeric;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;


-- Returns date (not timestampTz) of start time
CREATE OR REPLACE FUNCTION start_date(period)
RETURNS date AS
$$
    SELECT first($1)::date;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;


-- Returns date (not timestampTz) of end time
CREATE OR REPLACE FUNCTION end_date(period)
RETURNS date AS
$$
    SELECT next($1)::date;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;


--  Mid point of the period
CREATE OR REPLACE FUNCTION mean_time(period)
RETURNS timestampTz AS
$$
    SELECT first($1) + length($1) / 2;
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;


-- p1.first = p2.first and p1.last < p2.last
CREATE OR REPLACE FUNCTION starts(period, period)
RETURNS boolean AS
$$
    SELECT first($1) = first($2) AND next($1) < next($2);
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;


-- p1.last = p2.last and p1.first > p2.first
CREATE OR REPLACE FUNCTION ends(period, period)
RETURNS boolean AS
$$
    SELECT next($1) = next($2) AND first($1) > first($2);
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;


/*****************************************************************
*                         Boolean Operators
*****************************************************************/

CREATE OPERATOR <<(
  PROCEDURE = before,
  LEFTARG = period,
  RIGHTARG = period
);

CREATE OPERATOR >>(
  PROCEDURE = after,
  LEFTARG = period,
  RIGHTARG = period
);

CREATE OPERATOR |@(
  PROCEDURE = starts,
  LEFTARG = period,
  RIGHTARG = period
);

CREATE OPERATOR @|(
  PROCEDURE = ends,
  LEFTARG = period,
  RIGHTARG = period
);

CREATE OPERATOR ><(
  PROCEDURE = adjacent,
  LEFTARG = period,
  RIGHTARG = period
);

/*****************************************************************
*                         Set Functions
*****************************************************************/


-- Returns the length (interval) of the overlap between p1 and p2
CREATE OR REPLACE FUNCTION period_overlap(period, period)
RETURNS interval AS
$$
    SELECT CASE WHEN overlaps($1, $2)
    THEN LEAST(next($1), next($2)) -
      GREATEST(first($1), first($2)) END;
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;

-- Like period_union except that periods do not
-- need to overlap
CREATE OR REPLACE FUNCTION period_extend(period, period)
RETURNS period AS
$$
    SELECT (LEAST(first($1), first($2)),
    GREATEST(next($1), next($2)))::period;
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;


-- remove any values from p1 that are contained in p2
CREATE OR REPLACE FUNCTION period_except(period, period)
RETURNS period[] AS
$$ 
  SELECT array_agg(p)
  FROM (
    SELECT CASE WHEN first($1) < first($2)
    THEN period(first($1),
    LEAST(next($1), first($2))) END AS p

    UNION ALL
    
    SELECT CASE WHEN next($1) > next($2)
    THEN period(GREATEST(first($1), next($2)),
    next($1))END
  ) sub
  WHERE p IS NOT NULL;
$$ LANGUAGE 'sql' IMMUTABLE;


-- Enumerate all granules of size interval in period
CREATE OR REPLACE FUNCTION enumerate(period, interval)
RETURNS SETOF timestampTz AS
$$
  SELECT first($1) + ($2 * i)
  FROM generate_series(0, 
    floor(duration($1)/duration($2))::int - 1) i
  WHERE $2 > INTERVAL '0 seconds';
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;


-- Enumerate all seconds in period
CREATE OR REPLACE FUNCTION enumerate(period)
RETURNS SETOF timestampTz AS
$$
  SELECT enumerate($1, period_granularity());
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;


/*****************************************************************
*                       period functions
*****************************************************************/


-- pretty print
CREATE OR REPLACE FUNCTION to_char(period)
RETURNS varchar AS
$$
    SELECT '[' || TO_CHAR(first($1), 'YYYY-MM-DD HH24:MI:SS') || 
    ', '  || TO_CHAR(next($1), 'YYYY-MM-DD HH24:MI:SS') || ')';
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;


-- shifts entire period by interval
CREATE OR REPLACE FUNCTION period_shift(period, interval)
RETURNS period AS
$$
    SELECT (first($1) + $2, next($1) + $2)::period;
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;


-- shifts entire period by n units(seconds)
CREATE OR REPLACE FUNCTION period_shift(period, numeric)
RETURNS period AS
$$
    SELECT (first($1) + intv, next($1) + intv)::period
    FROM (
       SELECT period_granularity() * $2 AS intv
    ) sub;
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;


-- add interval from duration of period
-- if interval is negative and longer than length(p1) then 
-- period will have a 0 length
CREATE OR REPLACE FUNCTION period_grow(period, interval)
RETURNS period AS
$$
    SELECT CASE WHEN $2 > INTERVAL '0 second'
      THEN (first($1), next($1) + $2)::period
    WHEN length($1) + $2 > INTERVAL '0 second'
      THEN (first($1), next($1) + $2)::period
    ELSE (first($1), first($1))::period END;
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;


-- add n seconds from the duration of period
CREATE OR REPLACE FUNCTION period_grow(period, numeric)
RETURNS period AS
$$
    SELECT CASE WHEN intv > zero
      THEN (first($1), next($1) + intv)::period
    WHEN length($1) + intv > zero
      THEN (first($1), next($1) + intv)::period
    ELSE (first($1), first($1))::period END
    FROM (
        SELECT INTERVAL '0 second' AS zero,
        period_granularity() * $2 AS intv
    ) sub;
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;


-- subtract interval from the duration of a period
CREATE OR REPLACE FUNCTION period_shrink(period, interval)
RETURNS period AS
$$    SELECT period_grow($1, -$2);
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;


-- subtract n seconds from the duration of a period
CREATE OR REPLACE FUNCTION period_shrink(period, numeric)
RETURNS period AS
$$
    SELECT period_grow($1, -$2);
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;


-- Measures p1 in the context of period ctx
-- returning (x1, x2) as percentages of ctx
CREATE OR REPLACE FUNCTION period_measure(
  subject IN period,
  ctx     IN period,
  x1      OUT  numeric,
  x2      OUT  numeric
) AS
$$
    SELECT (duration(first(subject) - first(ctx))
        / ctx_duration * 100)::numeric,
    (duration(next(subject) - first(ctx))
        / ctx_duration * 100)::numeric
    FROM (
        SELECT period_intersect($1, $2) AS subject,
        $2 AS ctx,
        duration($2) AS ctx_duration
    ) sub
    WHERE duration(subject) > 0 AND ctx_duration > 0;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;


-- Visual representation of p1 within context of ctx
CREATE OR REPLACE FUNCTION period_draw(p1 period, ctx period, width integer)
  RETURNS text AS
$BODY$
    SELECT '|' ||
        repeat(' ', pre) ||
        repeat('*', dur) ||
        repeat(' ', $3 - pre - dur) || '|  (' ||
        round(x1) || ', ' || round(x2) || ')'
    FROM (
        SELECT round(x1 * mult)::int AS pre,
          round((x2 - x1) * mult)::int AS dur,
          x1, x2
        FROM (
            SELECT x1, x2, $3 / 100.0 AS mult
            FROM period_measure($1, $2) s
        ) s1
    ) s2;
$BODY$
  LANGUAGE 'sql' IMMUTABLE STRICT;

/*****************************************************************
*                       period operators
*****************************************************************/

CREATE OPERATOR <+>(
  PROCEDURE = period_extend,
  LEFTARG = period,
  RIGHTARG = period
);


CREATE OPERATOR +(
  PROCEDURE = period_grow,
  LEFTARG = period,
  RIGHTARG = interval
);

CREATE OPERATOR -(
  PROCEDURE = period_shrink,
  LEFTARG = period,
  RIGHTARG = interval
);

CREATE OPERATOR +(
  PROCEDURE = period_grow,
  LEFTARG = period,
  RIGHTARG = numeric
);

CREATE OPERATOR -(
  PROCEDURE = period_shrink,
  LEFTARG = period,
  RIGHTARG = numeric
);

CREATE OPERATOR <->(
  PROCEDURE = period_overlap,
  LEFTARG = period,
  RIGHTARG = period
);

CREATE OPERATOR @@(
  PROCEDURE = mean_time,
  RIGHTARG = period
);
