/*____________________________________________________________________________
	Copyright (C) 1997 Network Associates, Inc. and its affiliates.
	All rights reserved.
	
	$Id: PGPSelectKeysDialog.cp,v 1.7.2.1.2.1 1998/11/12 03:07:19 heller Exp $
____________________________________________________________________________*/

#include <LPushButton.h>
#include <LTableMonoGeometry.h>
#include <LTableArrayStorage.h>
#include <LTableMultiSelector.h>
#include <PP_Messages.h>
#include <UDrawingUtils.h>
#include <UGAColorRamp.h>
#include <URegistrar.h>
#include <UTextTraits.h>

#include "pgpMem.h"
#include "pgpErrors.h"
#include "pgpKeys.h"
#include "pgpUtilities.h"
#include "pgpClientLib.h"
#include "pgpClientPrefs.h"

#include "pflPrefTypes.h"

#include "MacErrors.h"
#include "MacCursors.h"
#include "MacQuickdraw.h"
#include "MacStrings.h"

#include "StSaveHeapZone.h"
#include "StSaveCurResFile.h"
#include "CString.h"

#include "PGPclientLibUtils.h"
#include "PGPclientLibDialogs.h"

#include "PGPSelectKeysDialog.h"


const ResIDT	Txtr_Geneva9				=	4748;

const ResIDT	ppob_SelectKeysID			=	4753;

const PaneIDT	caption_Prompt				=	1003;
const PaneIDT	table_UserIDs				=	1004;
const PaneIDT	button_SelectAll			=	1005;
const PaneIDT	button_SelectNone			=	1006;
const PaneIDT	button_OK					=	'bOK ';

const MessageT	msg_UserIDs					=	table_UserIDs;
const MessageT	msg_SelectAll				=	button_SelectAll;
const MessageT	msg_SelectNone				=	button_SelectNone;

const SInt16	kUserIDColumnWidth			=	232;
const SInt16	kValidityColumnLeft			=	241;
const SInt16	kValidityColumnWidth		=	50;
const SInt16	kValidityColumnIconIndent	=	14;
const SInt16	kCreationColumnLeft			=	310;
const SInt16	kCreationColumnWidth		=	65;
const SInt16	kSizeColumnLeft				=	384;
const SInt16	kSizeColumnWidth			=	75;
const SInt16	kIconWidth					=	16;
const SInt16	kIconHeight					=	16;
const SInt16	kIconIndent					=	2;
const SInt16	kUserIDIndent				=	4;
const SInt16	kVerticalIndent				=	3;

const ResIDT	icmx_ValidKey	 			=	3014;
const ResIDT	icmx_InvalidKey 			=	3015;
const ResIDT	icmx_AxiomaticKey			=	3017;

const ResIDT	acur_BeachBall				=	4747;

const ResIDT	ppat_BarberPattern	 		=	4747;

const SInt16	kBarHeight					=	10;
const SInt16	kMaxValidity				=	3;

const ResIDT	icmx_DHKey					=	3007;
const ResIDT	icmx_RSAKey					=	3000;

const ResIDT	icmx_RevokedDHKey			=	3012;
const ResIDT	icmx_RevokedRSAKey			=	3004;

const ResIDT	icmx_SecretDHKey			=	3008;
const ResIDT	icmx_SecretRSAKey			=	3001;

const ResIDT	icmx_ExpiredDHKey			=	3013;
const ResIDT	icmx_ExpiredRSAKey			=	3005;

static const ResID	icsx_DSASplitKeyPair	=	3018;
static const ResID	icsx_RSASplitKeyPair	=	3019;

	static PGPError
RealPGPSelectKeysDialog(
	PGPContextRef			context,
	PGPSelectKeysSettings	settings,
	PGPPrefRef				clientPrefsRef,
	const char *			prompt,
	PGPKeySetRef			fromSet,							
	PGPKeySetRef			validitySet,
	PGPKeySetRef			*selectedSet)					
{
	CComboError	err;
	DialogRef	skDialog;
	
	
	RegisterClass_( CSelectKeysDialog );
	RegisterClass_( CSelectKeysTable );
	
	// Create standard Macintosh window and overlay PowerPlant onto it
	skDialog = 
			CPGPModalGrafPortView::CreateDialog( ppob_SelectKeysID );
	if( IsntNull( skDialog ) )
	{
		CSelectKeysDialog *skGrafPortView = nil;

		skGrafPortView =
				(CSelectKeysDialog *) GetWRefCon( skDialog );
		pgpAssert( IsntNull( skGrafPortView ) );
		
		err = skGrafPortView->Init( context, settings, clientPrefsRef, prompt,
									fromSet, validitySet, selectedSet);
		if( err.IsntError() )
		{
			ModalFilterUPP	filterUPP;
			MessageT		dismissMessage;
			short			itemHit = 0;
		
			InitCursor();
			
			SetPort( skDialog );
			ShowWindow( skDialog );
			SelectWindow( skDialog );
	
			skGrafPortView->Refresh();
			
			filterUPP = NewModalFilterProc(
						CPGPModalGrafPortView::ModalFilterProcHandler );
			
			skGrafPortView->SetDismissMessage( msg_Nothing );

			while( itemHit == 0 )
			{
				ModalDialog( filterUPP, &itemHit );
			}
		
			dismissMessage = skGrafPortView->GetDismissMessage();
			if( dismissMessage == msg_OK )
			{
				err = skGrafPortView->GetKeys();
			}
			else
			{
				err.pgpErr = kPGPError_UserAbort;
			}
		
			DisposeRoutineDescriptor( filterUPP );
		}
		
		delete skGrafPortView;
		DisposeDialog( skDialog );
	}
	else
	{
		err.pgpErr = kPGPError_OutOfMemory;
	}
	
	return err.ConvertToPGPError();
}

	PGPError
PGPSelectKeysDialog(
	PGPContextRef			context,
	PGPSelectKeysSettings	settings,
	const char *			prompt,
	PGPKeySetRef			fromSet,							
	PGPKeySetRef			validitySet,
	PGPKeySetRef			*selectedSet)					
{
	PGPError			err;
	PGPclientLibState	state;
	
	PGPValidatePtr( selectedSet );
	*selectedSet	= kInvalidPGPKeySetRef;
	PGPValidateParam( IsntNull( context ) );
	PGPValidateParam( IsntNull( fromSet ) );
	
	err = EnterPGPclientLib( context, &state );
	if( IsntPGPError( err ) )
	{
		err = RealPGPSelectKeysDialog( context, settings, state.clientPrefsRef,
					prompt, fromSet, validitySet, selectedSet );
	}
		
	ExitPGPclientLib( &state );
	
	return( err );
}



CSelectKeysDialog::CSelectKeysDialog(
	LStream *	inStream)
		: CPGPModalGrafPortView(inStream)
{
}



CSelectKeysDialog::~CSelectKeysDialog()
{
}



	CComboError
CSelectKeysDialog::Init(
	PGPContextRef			inContext,
	PGPSelectKeysSettings	settings,
	PGPPrefRef				inClientPrefsRef,
	const char *			inPrompt,
	PGPKeySetRef			inFromSet,
	PGPKeySetRef			inValiditySet,
	PGPKeySetRef *			outSelectedSet)
{
	CComboError	err;
	PGPBoolean	inShowRSA;
	
	if (IsntNull( inPrompt ) )
	{
		FindPaneByID(caption_Prompt)->SetDescriptor(CString(inPrompt));
	}
	
	inShowRSA = ( settings & kPGPSelectKeysNoRSAKeys ) == 0;
	
	err = mUserIDTable->SetTableInfo( inContext, inClientPrefsRef,
							inFromSet, inValiditySet, inShowRSA,
							outSelectedSet );
	if( err.IsntError() )
	{	
		if( settings & kPGPSelectKeysImportVariation )
		{
			Str255	buttonName;
			
			GetIndString(	buttonName, kDialogStringListResID,
							kImportButtonNameIndex );
			FindPaneByID(button_OK)->SetDescriptor( buttonName );
			mUserIDTable->SelectAllCells();
		}
		AdjustButtons();
	}
	
	return( err );
}

	CComboError
CSelectKeysDialog::GetKeys(void)
{
	pgpAssert( IsntNull( mUserIDTable ) );
	
	return( mUserIDTable->GetKeys() );
}



	void
CSelectKeysDialog::FinishCreateSelf()
{
	LPushButton *	button;
	
	CPGPModalGrafPortView::FinishCreateSelf();
	
	button = (LPushButton *) FindPaneByID(button_SelectAll);
	button->AddListener(this);

	button = (LPushButton *) FindPaneByID(button_SelectNone);
	button->AddListener(this);
	
	mUserIDTable = (CSelectKeysTable *) FindPaneByID(table_UserIDs);
	pgpAssert( IsntNull( mUserIDTable ) );
	
	mUserIDTable->AddListener(this);
}

	void
CSelectKeysDialog::AdjustButtons(void)
{
	TableIndexT	numRows;
	TableIndexT	numColumns;
	STableCell	theCell(0, 0);
	TableIndexT	numSelectedRows = 0;
	
	mUserIDTable->GetTableSize( numRows, numColumns );
	
	while( mUserIDTable->GetNextSelectedCell( theCell ) )
		++numSelectedRows;
		
	if ( numSelectedRows == 0 )
	{
		mOKButton->Disable();
		FindPaneByID(button_SelectNone)->Disable();
	}
	else
	{
		mOKButton->Enable();
		FindPaneByID(button_SelectNone)->Enable();
	}
	
	if ( numSelectedRows == numRows )
	{
		FindPaneByID(button_SelectAll)->Disable();
	}
	else
	{
		FindPaneByID(button_SelectAll)->Enable();
	}
}

	void
CSelectKeysDialog::ListenToMessage(
	MessageT	inMessage,
	void *		ioParam)
{
	switch (inMessage)
	{
		case msg_UserIDs:
			AdjustButtons();
			break;
		
		case msg_SelectAll:
			mUserIDTable->SelectAllCells();
			AdjustButtons();
			break;

		case msg_SelectNone:
			mUserIDTable->UnselectAllCells();
			AdjustButtons();
			break;
		
		default:
			CPGPModalGrafPortView::ListenToMessage( inMessage, ioParam );
			break;
	}
}

CSelectKeysTable::CSelectKeysTable(
	LStream *	inStream)
		: LTableView(inStream),
		mFromSet(kInvalidPGPKeySetRef),
		mBarberPixPat(nil),
		mKeyIter(kInvalidPGPKeyIterRef),
		mSelectedSet(nil)
{
	FontInfo	fontInfo;
	SPoint32	scrollUnit;
	
	UTextTraits::SetPortTextTraits(Txtr_Geneva9);
	::GetFontInfo(&fontInfo);
	scrollUnit.v = fontInfo.ascent + fontInfo.leading + fontInfo.descent
					+ (kVerticalIndent * 2);
	scrollUnit.h = 1;
	SetTableGeometry(new LTableMonoGeometry(	this,
												mFrameSize.width,
												scrollUnit.v));
	SetTableStorage(new LTableArrayStorage(	this,
											sizeof(PGPKeyRef)));
	SetTableSelector(new LTableMultiSelector(this));
	InsertCols(1, 0);
	SetScrollUnit(scrollUnit);
	mCustomHilite = true;
}



CSelectKeysTable::~CSelectKeysTable()
{
	if (mKeyIter != kInvalidPGPKeyIterRef) {
		PGPFreeKeyIter(mKeyIter);
	}
	if (mBarberPixPat != nil) {
		::DisposePixPat(mBarberPixPat);
	}
}
	

	CComboError
CSelectKeysTable::AddKeysToTable(
	PGPKeySetRef	inFromSet,
	PGPBoolean		inShowRSA )
{
	PGPKeyListRef		keyList;
	CComboError			err;
	PGPInt32			algorithm;
	
	err.pgpErr = PGPOrderKeySet( inFromSet, kPGPUserIDOrdering, &keyList);
	if( err.IsntError() )
	{
		err.pgpErr = PGPNewKeyIter(	keyList, &mKeyIter);
		if( err.IsntError() )
		{
			PGPKeyRef	key;

			while ( ( PGPKeyIterNext( mKeyIter, &key ) == kPGPError_NoErr )
					&& ( key != kInvalidPGPKeyRef ) )
			{
				PGPGetKeyNumber( key, kPGPKeyPropAlgID, &algorithm );
				if( inShowRSA ||
					(algorithm != kPGPPublicKeyAlgorithm_RSA ) )
					InsertRows(	1, LArray::index_Last, &key );
			}
		}
		
		PGPFreeKeyList( keyList );
	}
	
	return( err );
}


	CComboError
CSelectKeysTable::SetTableInfo(
	PGPContextRef	inContext,
	PGPPrefRef		inClientPrefsRef,
	PGPKeySetRef	inFromSet,
	PGPKeySetRef	inValiditySet,
	PGPBoolean		inShowRSA,
	PGPKeySetRef *	outSelectedSet)
{
	CComboError			err;
	PGPUInt32			numKeys;

	// Make sure that there are some keys to show
	err.pgpErr = PGPCountKeys( inFromSet, &numKeys );
	if( err.IsntError() )
	{
		if( numKeys > 0 )
		{
			// Save off our sets
			mSelectedSet 	= outSelectedSet;
			mFromSet 		= inFromSet;
			
			mBarberPixPat = GetPixPat(ppat_BarberPattern);
			pgpAssert( IsntNull( mBarberPixPat ) );
		
			err.pgpErr = PGPGetPrefBoolean(	inClientPrefsRef,
									kPGPPrefDisplayMarginalValidity,
									&mShowMarginalValidity);
			if( err.IsntError() )
			{
				err.pgpErr = PGPGetPrefBoolean(	inClientPrefsRef,
										kPGPPrefMarginalIsInvalid,
										&mMarginalIsInvalid);
			}
			
			if( err.IsntError() )
			{
				err = AddKeysToTable( inFromSet, inShowRSA );
			}

			// Propagate validity
			if( IsntNull( inValiditySet ) && err.IsntError() )
			{
				PGPKeySetRef	combinedSet = kInvalidPGPKeySetRef;
			
				// Create combined set
				err.pgpErr = PGPNewKeySet( inContext, &combinedSet );
				if( err.IsntError() )
				{
					err.pgpErr = PGPAddKeys( inFromSet, combinedSet);
					if( err.IsntError() )
					{
						err.pgpErr = PGPAddKeys( inValiditySet, combinedSet);
					}
				}

				if( err.IsntError() )
				{
					AnimatedCursorRef	cursorRef = nil;

					err.err = Get1AnimatedCursor( acur_BeachBall, &cursorRef );
					pgpAssert( err.IsntError() );

					err.pgpErr = PGPCheckKeyRingSigs( inFromSet, combinedSet,
									TRUE, EventHandler, cursorRef );

					DisposeAnimatedCursor( cursorRef );
				}

				if( err.IsntError() )
				{
					err.pgpErr = PGPPropagateTrust( combinedSet );
				}
				
				PGPFreeKeySet( combinedSet );
			}
		}
		else
		{
			err.pgpErr = kPGPError_BadParams;
		}
	}

	return( err );
}



	CComboError
CSelectKeysTable::GetKeys()
{
	CComboError		err;

	err.pgpErr = PGPNewEmptyKeySet( mFromSet, mSelectedSet );
	if( err.IsntError() )
	{
		STableCell		theCell(0, 0);

		while( GetNextSelectedCell( theCell ) )
		{
			PGPKeySetRef	tempSet;
			PGPKeyRef		key;
			UInt32			len = sizeof( key );
			PGPKeySetRef	singleKeySet = kInvalidPGPKeySetRef;

			tempSet = *mSelectedSet;
			
			GetCellData( theCell, &key, len );
			
			err.pgpErr = PGPNewSingletonKeySet( key, &singleKeySet );
			if( err.IsntError() )
			{
				err.pgpErr = PGPUnionKeySets( tempSet, singleKeySet,
										mSelectedSet);
				
				PGPFreeKeySet( singleKeySet );
			}
			
			PGPFreeKeySet( tempSet );
			
			if( err.IsError() )
				break;
		}
	}

	return( err );
}



	void
CSelectKeysTable::ClickSelf(
	const SMouseDownEvent &	inMouseDown)
{
	LTableView::ClickSelf(inMouseDown);
	BroadcastMessage(GetPaneID());
}



	void
CSelectKeysTable::DrawSelf()
{
	Rect		frame;
	RGBColor	tempColor;
	
	tempColor = UGAColorRamp::GetColor( colorRamp_Gray1 );
	RGBForeColor( &tempColor );
	
	tempColor = UGAColorRamp::GetWhiteColor();
	RGBBackColor( &tempColor );

	CalcLocalFrameRect(frame);
	{
		StDeviceLoop	devLoop( frame );		
		SInt16			depth;

		while( devLoop.NextDepth( depth ) )
		{
			if( depth >= 8 )
				PaintRect(&frame);
			else
				EraseRect(&frame);
		}
	}
	
	LTableView::DrawSelf();
}



	void
CSelectKeysTable::HiliteCellActively(
	const STableCell	&inCell,
	Boolean				/*inHilite*/)
{
	Rect	cellFrame;
	
	if(GetLocalCellRect(inCell, cellFrame) && FocusExposed()) {
		StDeviceLoop	devLoop(cellFrame);		
		SInt16			depth;

		while(devLoop.NextDepth(depth)) {
			StColorPenState saveColorPen;   // Preserve color & pen state
			
			StColorPenState::Normalize();
			if(depth >= 8) {
				::RGBBackColor(&UGAColorRamp::GetColor(colorRamp_Gray2));
				::RGBForeColor(&UGAColorRamp::GetColor(colorRamp_Black));
			} else {
				::RGBBackColor(&UGAColorRamp::GetColor(colorRamp_White));
				::RGBForeColor(&UGAColorRamp::GetColor(colorRamp_Black));
			}
			cellFrame.right = cellFrame.left + kUserIDColumnWidth;
	        UDrawingUtils::SetHiliteModeOn();
			::InvertRect(&cellFrame);
		}
	}
}



	void
CSelectKeysTable::HiliteCellInactively(
	const STableCell	&inCell,
	Boolean			 	/*inHilite*/)
{
	Rect	cellFrame;
	
	if(GetLocalCellRect(inCell, cellFrame) && FocusExposed()) {
		StDeviceLoop	devLoop(cellFrame);		
		SInt16			depth;

		while(devLoop.NextDepth(depth)) {
	        StColorPenState saveColorPen;   // Preserve color & pen state
	        
	        StColorPenState::Normalize();
	        if(depth >= 8) {
				::RGBBackColor(&UGAColorRamp::GetColor(colorRamp_Gray2));
				::RGBForeColor(&UGAColorRamp::GetColor(colorRamp_Black));
			} else {
				::RGBBackColor(&UGAColorRamp::GetColor(colorRamp_White));
				::RGBForeColor(&UGAColorRamp::GetColor(colorRamp_Black));
			}
			::PenMode(srcXor);
			cellFrame.right = cellFrame.left + kUserIDColumnWidth;
			cellFrame.bottom -= 1;
			UDrawingUtils::SetHiliteModeOn();
			::FrameRect(&cellFrame);
		}
	}
}



	void
CSelectKeysTable::DrawCell(
	const STableCell &	inCell,
	const Rect &		inLocalRect)
{
	Rect			theRect = inLocalRect;
	StDeviceLoop	devLoop(inLocalRect);
	SInt16			depth;
	
	while (devLoop.NextDepth(depth)) {
		if (depth >= 8) {
			theRect.right = theRect.left + kUserIDColumnWidth;
			::RGBForeColor(&UGAColorRamp::GetColor(colorRamp_Gray2));
			::PaintRect(&theRect);
			::RGBForeColor(&UGAColorRamp::GetColor(colorRamp_Gray1));
			theRect.left += kUserIDColumnWidth;
			theRect.right = inLocalRect.right;
			::PaintRect(&theRect);
			::RGBForeColor(&UGAColorRamp::GetColor(colorRamp_White));
			::MoveTo(inLocalRect.left, inLocalRect.bottom - 1);
			::LineTo(inLocalRect.right - 1, inLocalRect.bottom - 1);
			::RGBForeColor(&UGAColorRamp::GetColor(colorRamp_Black));
		} else {
			::RGBForeColor(&UGAColorRamp::GetColor(colorRamp_Black));
			::RGBBackColor(&UGAColorRamp::GetColor(colorRamp_White));
			::EraseRect(&inLocalRect);
		}
		
		// Draw the contents
		PGPError		pgpErr;
		PGPKeyRef		key;
		PGPInt32		algorithm;
		SInt16			iconTop = inLocalRect.top + (GetRowHeight(inCell.row)
									- kIconHeight) / 2;
		SInt16			iconRight = inLocalRect.left + kIconIndent
									+ kIconWidth;
		Rect			drawRect = {iconTop, inLocalRect.left + kIconIndent,
									iconTop + kIconHeight, iconRight};
		SInt16			just;
		Str255			string;
		size_t			len;
		PGPTime			creation;
		PGPInt32		bits;
		PGPSubKeyRef	subKey;		
		ResID			iconID;
		PGPBoolean		secretKey;
		PGPBoolean		expiredKey;
		PGPBoolean		revokedKey;
		PGPBoolean		secShared;
		
		len = sizeof(key);
		GetCellData(	inCell,
						&key,
						len);
		just = UTextTraits::SetPortTextTraits(Txtr_Geneva9);
		if (depth >= 8) {
			::RGBBackColor(&UGAColorRamp::GetColor(colorRamp_Gray2));
		}
		
		// Plot the icon
		pgpErr = PGPGetKeyNumber(	key,
									kPGPKeyPropAlgID,
									&algorithm);
		PGPThrowIfPGPErr_(pgpErr);

		pgpErr = PGPGetKeyBoolean(	key,
									kPGPKeyPropIsSecret,
									&secretKey);
		PGPThrowIfPGPErr_(pgpErr);

		pgpErr = PGPGetKeyBoolean(	key,
									kPGPKeyPropIsRevoked,
									&revokedKey);
		PGPThrowIfPGPErr_(pgpErr);

		pgpErr = PGPGetKeyBoolean(	key,
									kPGPKeyPropIsExpired,
									&expiredKey);
		PGPThrowIfPGPErr_(pgpErr);
		
		if( algorithm == kPGPPublicKeyAlgorithm_DSA )
		{
			if( revokedKey )
				iconID = icmx_RevokedDHKey;
			else if( expiredKey )
				iconID = icmx_ExpiredDHKey;
			else if( secretKey )
			{
				PGPGetKeyBoolean(	key,
									kPGPKeyPropIsSecretShared,
									&secShared );
				iconID = (secShared) ? icsx_DSASplitKeyPair : icmx_SecretDHKey;
			}
			else
				iconID = icmx_DHKey;
		}
		else
		{
			if( revokedKey )
				iconID = icmx_RevokedRSAKey;
			else if( expiredKey )
				iconID = icmx_ExpiredRSAKey;
			else if( secretKey )
			{
				PGPGetKeyBoolean(	key,
									kPGPKeyPropIsSecretShared,
									&secShared );
				iconID = (secShared) ? icsx_RSASplitKeyPair : icmx_SecretRSAKey;
			}
			else
				iconID = icmx_RSAKey;
		}
		
		::PlotIconID(	&drawRect,
						atNone,
						ttNone,
						iconID);
								
		// Draw the UserID
		::SetRect(	&drawRect,
					iconRight + kUserIDIndent,
					inLocalRect.top + kVerticalIndent,
					kUserIDColumnWidth,
					inLocalRect.bottom - kVerticalIndent);
		len = sizeof(string) - 1;
		pgpErr = PGPGetPrimaryUserIDNameBuffer(key,
			sizeof( string ) - 1, (char *)&string[1], &len);
		if(pgpErr != kPGPError_BufferTooSmall)
		{
			PGPThrowIfPGPErr_(pgpErr);
		}
		string[0] = len;
		::TruncString(	drawRect.right - drawRect.left,
						string,
						truncMiddle);
		UTextDrawing::DrawWithJustification(	(Ptr) &string[1],
												string[0],
												drawRect,
												just);
								
		// Adjust the background color for last columns
		if (depth >= 8) {
			::RGBBackColor(&UGAColorRamp::GetColor(colorRamp_Gray1));
		}
		
		// Draw the validity
		RGBColor	tempColor;
		RGBColor	saveBackColor;
		Rect		validityBarRect;
		PGPBoolean	isAxiomatic;
		PGPInt32	validity;
				
		pgpErr = PGPGetKeyBoolean(	key,
									kPGPKeyPropIsAxiomatic,
									&isAxiomatic);
		PGPThrowIfPGPErr_(pgpErr);
		pgpErr = PGPGetKeyNumber(	key,
									kPGPKeyPropValidity,
									&validity);
		PGPThrowIfPGPErr_(pgpErr);
		::SetRect(	&drawRect,
					kValidityColumnLeft,
					inLocalRect.top + kVerticalIndent,
					kValidityColumnLeft + kValidityColumnWidth,
					inLocalRect.bottom - kVerticalIndent);
		::GetBackColor(&saveBackColor);
		tempColor = UGAColorRamp::GetBlackColor();
		::RGBForeColor(&tempColor);

		if (mShowMarginalValidity) {
			RGBColor	trustColor = { 52223, 52223, 65535 };
			
			::SetRect(	&validityBarRect,
						drawRect.left,
						drawRect.top,
						drawRect.right,
						drawRect.top + kBarHeight);
			AlignRect(	&drawRect,
						&validityBarRect,
						kAlignAtVerticalCenter);
			::FrameRect(&validityBarRect);
			::InsetRect(	&validityBarRect,
							1,
							1);

			if (isAxiomatic && (mBarberPixPat != nil)) {
				::PenPixPat(mBarberPixPat);
				::PaintRect(&validityBarRect);
				::PenNormal();
			} else {
				SInt16 		width;
				SInt16		barPercent;
				switch (validity) {
					default:
					case kPGPValidity_Unknown:
					case kPGPValidity_Invalid:
					{
						barPercent = 0;
					}
					break;
					
						
					case kPGPValidity_Marginal:
					{
						barPercent = 50;
					}
					break;
					
						
					case kPGPValidity_Complete:
					{
						barPercent = 100;
					}
					break;
				}
				
				if (depth >= 8) {
					::RGBForeColor(&trustColor);
					::PaintRect(&validityBarRect);
				}
				width = validityBarRect.right - validityBarRect.left;
				validityBarRect.right -= (width * (100 - barPercent)) / 100;
				tempColor = UGAColorRamp::GetColor(colorRamp_Gray9);
				::RGBForeColor(&tempColor);
				::PaintRect(&validityBarRect);
			}

		} else {
			ResIDT	iconResID;
			
			if (isAxiomatic)
			{
				iconResID = icmx_AxiomaticKey;
			}
			else if (validity == kPGPValidity_Complete)
			{
				iconResID = icmx_ValidKey;
			}
			else if (validity == kPGPValidity_Marginal)
			{
				if( mMarginalIsInvalid )
				{
					iconResID = icmx_InvalidKey;
				}
				else
				{
					iconResID = icmx_ValidKey;
				}
			}
			else
			{
				iconResID = icmx_InvalidKey;
			}
			
			::SetRect(	&validityBarRect,
						drawRect.left + kValidityColumnIconIndent,
						drawRect.top,
						drawRect.left + kIconWidth
								+ kValidityColumnIconIndent,
						drawRect.top + kIconHeight);
			AlignRect(	&drawRect,
						&validityBarRect,
						kAlignAtVerticalCenter);
			tempColor = UGAColorRamp::GetWhiteColor();
			::RGBBackColor(&tempColor);
			::PlotIconID(	&validityBarRect,
							kAlignNone,
							IsEnabled() ? kTransformNone : kTransformDisabled,
							iconResID);
		}
		tempColor = UGAColorRamp::GetBlackColor();
		::RGBForeColor(&tempColor);
		::RGBBackColor(&saveBackColor);

		// Draw the creation date
		pgpErr = PGPGetKeyTime(	key,
								kPGPKeyPropCreation,
								&creation);
		PGPThrowIfPGPErr_(pgpErr);
		creation = PGPTimeToMacTime(creation);
		::DateString(creation, shortDate, string, nil);
		::SetRect(	&drawRect,
					kCreationColumnLeft,
					inLocalRect.top + kVerticalIndent,
					kCreationColumnLeft + kCreationColumnWidth,
					inLocalRect.bottom - kVerticalIndent);
		UTextDrawing::DrawWithJustification(	(Ptr) &string[1],
												string[0],
												drawRect,
												just);

		// Draw the size
		pgpErr = PGPGetKeyNumber(	key,
									kPGPKeyPropBits,
									&bits);
		PGPThrowIfPGPErr_(pgpErr);
		if (algorithm == kPGPPublicKeyAlgorithm_DSA)
		{
			PGPInt32	bits2;
			Str31		string2;
			
			GetIndString( string, kDialogStringListResID,
					kDSSDHKeySizeFormatStrIndex );
			
			PGPKeyIterSeek(mKeyIter, key);
			pgpErr = PGPKeyIterNextSubKey(mKeyIter, &subKey);
			if (IsntPGPError(pgpErr) && PGPSubKeyRefIsValid(subKey)) {
				
				pgpErr = PGPGetSubKeyNumber(	subKey,
												kPGPKeyPropBits,
												&bits2);
				PGPThrowIfPGPErr_(pgpErr);
			}
			
			::NumToString(bits2, string2);
			PrintPString( string, string, string2 );

			::NumToString(bits, string2);
			PrintPString( string, string, string2 );
		}
		else
		{
			::NumToString(bits, string);
		}
		::SetRect(	&drawRect,
					kSizeColumnLeft,
					inLocalRect.top + kVerticalIndent,
					kSizeColumnLeft + kSizeColumnWidth,
					inLocalRect.bottom - kVerticalIndent);
		UTextDrawing::DrawWithJustification(	(Ptr) &string[1],
												string[0],
												drawRect,
												just);
	}
}



	PGPError
CSelectKeysTable::EventHandler(
	PGPContextRef	context,
	PGPEvent *		event,
	PGPUserValue	userValue)
{
	(void) context;
	(void) event;
	
	if( userValue != 0 )
		AnimateCursor((AnimatedCursorRef) userValue);
	
	return kPGPError_NoErr;
}

