/*____________________________________________________________________________
	Copyright (C) 1997-1998 Network Associates, Inc. and its affiliates.
	All rights reserved.

	$Id: CResultsWindow.cp,v 1.12.8.1 1998/11/12 03:09:25 heller Exp $
____________________________________________________________________________*/

#include <MacWindows.h>

#include <LBevelButton.h>
#include <PP_Messages.h>
#include <UGraphicUtils.h>
#include <UTextTraits.h>

#include "MacQuickdraw.h"

#include "PGPtoolsResources.h"

#include "CResultsTable.h"
#include "CResultsWindow.h"

const PaneIDT	kResultsTablePaneID 			= 5002;
const ResIDT	kColumnTitlesTextTraitsResID	= 135;

const PaneIDT	kNameColumnButtonPaneID			= 'NBtn';
const PaneIDT	kSignerColumnButtonPaneID		= 'SBtn';
const PaneIDT	kValidityColumnButtonPaneID		= 'VBtn';
const PaneIDT	kDateColumnButtonPaneID			= 'DBtn';

const ushort	kIconColumnLeftMargin		= 20;
const ushort	kNameColumnLeftMargin		= 42;
const ushort	kNameColumnRightMargin		= 5;
const ushort	kSignerColumnLeftMargin		= 5;
const ushort	kSignerColumnRightMargin	= 5;
const ushort	kValidityColumnLeftMargin	= 5;
const ushort	kValidityColumnRightMargin	= 5;
const ushort	kDateColumnLeftMargin		= 5;
const ushort	kDateColumnRightMargin		= 5;

const ushort	kMaxNameColumnWidth			= 250;
const ushort	kMinNameColumnWidth			= 80;


static CResultsWindow	*gResultsWindow;

LBevelButton	*CResultsWindow::mNameColumnButton;
LBevelButton	*CResultsWindow::mSignerColumnButton;
LBevelButton	*CResultsWindow::mValidityColumnButton;
LBevelButton	*CResultsWindow::mDateColumnButton;

CResultsWindow::CResultsWindow(LStream *inStream)
	: CPGPtoolsWindow( inStream )
{
	pgpAssert( gResultsWindow == nil );
	
	gResultsWindow 		= this;
	mHaveSignatures		= FALSE;
	mPendingNewSummary	= FALSE;
}

CResultsWindow::~CResultsWindow(void)
{
}

	void
CResultsWindow::FinishCreateSelf(void)
{
	CPGPtoolsWindow::FinishCreateSelf();
	
	SetThemeWindowBackground( GetMacPort(), kThemeListViewBackgroundBrush, FALSE );
	
	mResultsTable = (CResultsTable *) FindPaneByID( kResultsTablePaneID );
	pgpAssertAddrValid( mResultsTable, VoidAlign );

	mNameColumnButton = (LBevelButton *)
							FindPaneByID( kNameColumnButtonPaneID );
	pgpAssertAddrValid( mNameColumnButton, VoidAlign );

	mSignerColumnButton = (LBevelButton *)
							FindPaneByID( kSignerColumnButtonPaneID );
	pgpAssertAddrValid( mSignerColumnButton, VoidAlign );

	mValidityColumnButton = (LBevelButton *)
							FindPaneByID( kValidityColumnButtonPaneID );
	pgpAssertAddrValid( mValidityColumnButton, VoidAlign );

	mDateColumnButton = (LBevelButton *)
							FindPaneByID( kDateColumnButtonPaneID );
	pgpAssertAddrValid( mDateColumnButton, VoidAlign );

	UTextTraits::SetPortTextTraits( kColumnTitlesTextTraitsResID );
	GetFontInfo( &mTitlesFontInfo );
	
	// Force synchronization of column buttons
	ResizeWindowBy( 1, 0 );
	ResizeWindowBy( -1, 0 );	
}

	void
CResultsWindow::Click(SMouseDownEvent &inMouseDown)
{
	LPane	*clickedPane;
	
	// Our columnn titles are buttons which do not respond to clicks.
	// Filter out clicks to these buttons here.
	
	clickedPane = FindSubPaneHitBy( inMouseDown.wherePort.h,
									inMouseDown.wherePort.v);
	if( clickedPane != mNameColumnButton &&
		clickedPane != mSignerColumnButton &&
		clickedPane != mValidityColumnButton &&
		clickedPane != mDateColumnButton )
	{
		CPGPtoolsWindow::Click( inMouseDown );
	}
}

	void
CResultsWindow::ApplyForeAndBackColors(void) const
{
	SetThemeWindowBackground( GetMacPort(),
				kThemeListViewBackgroundBrush, FALSE );
}

	void
CResultsWindow::AttemptClose(void)
{
	Hide();
}

	void
CResultsWindow::DoClose(void)
{
	Hide();
}

	void
CResultsWindow::GetColumnTitlesRect(Rect *titlesRect)
{
	Rect	tableRect;
	
	CalcPortFrameRect( *titlesRect );
	mResultsTable->CalcPortFrameRect( tableRect );

	titlesRect->bottom 	= tableRect.top - 1;
	titlesRect->right	= tableRect.right;
}

	void
CResultsWindow::ResizeFrameBy(
	Int16		inWidthDelta,
	Int16		inHeightDelta,
	Boolean		inRefresh)
{
	// Resize the single column to fit to the width of the window.

	// Force redraw of the column titles.	
	
	CPGPtoolsWindow::ResizeFrameBy( inWidthDelta, inHeightDelta,
				inRefresh );	

	FocusDraw();
	
	if( inWidthDelta != 0 )
	{
		Rect	titleRect;
		Rect	windowRect;
		Rect	validityButtonRect;
		Rect	nameButtonRect;
		Rect	signerButtonRect;
		Rect	newSignerButtonRect;
		ushort	nameWidth;
		
		GetColumnTitlesRect( &titleRect );
		CalcPortFrameRect( windowRect );
		
		windowRect.bottom = titleRect.bottom;
		InvalRect( &windowRect );
		
		mValidityColumnButton->CalcPortFrameRect( validityButtonRect );
		mNameColumnButton->CalcPortFrameRect( nameButtonRect );
		mSignerColumnButton->CalcPortFrameRect( signerButtonRect );

		nameWidth = ( validityButtonRect.left - nameButtonRect.left ) * 0.4;
		if( nameWidth > kMaxNameColumnWidth )
		{
			nameWidth = kMaxNameColumnWidth;
		}
		else if( nameWidth < kMinNameColumnWidth )
		{
			nameWidth = kMinNameColumnWidth;
		}

		nameButtonRect.right = nameButtonRect.left + nameWidth;
		
		newSignerButtonRect 		= signerButtonRect;
		newSignerButtonRect.left 	= nameButtonRect.right;
		newSignerButtonRect.right 	= validityButtonRect.left;
		
		mNameColumnButton->ResizeFrameTo( 
					UGraphicUtils::RectWidth( nameButtonRect ),
					UGraphicUtils::RectHeight( nameButtonRect ), TRUE );
		
		mSignerColumnButton->MoveBy(
					newSignerButtonRect.left - signerButtonRect.left,
					newSignerButtonRect.top - signerButtonRect.top, TRUE );

		mSignerColumnButton->ResizeFrameTo(
					UGraphicUtils::RectWidth( newSignerButtonRect ),
					UGraphicUtils::RectHeight( newSignerButtonRect ), TRUE );
	}
}

	void
CResultsWindow::NewSummary(void)
{
	gResultsWindow->mPendingNewSummary = TRUE;
}

	void
CResultsWindow::AddResult(
	ConstStr255Param		itemName,
	PGPtoolsSignatureData	*signatureData)
{
	SwitchTarget( this );
	Select();
	
	if( mPendingNewSummary )
	{
		mPendingNewSummary = FALSE;
		
		mResultsTable->AddSummaryRow();
	}
	
	mHaveSignatures = TRUE;
	
	mResultsTable->AddProcessedItemRow( itemName, signatureData );

	UpdatePort();
}

	void
CResultsWindow::NewResult(
	ConstStr255Param		itemName,
	PGPtoolsSignatureData	*signatureData)
{
	gResultsWindow->Show();
	gResultsWindow->AddResult( itemName, signatureData );
}

	void
CResultsWindow::IncrementCount(void)
{
	SwitchTarget( this );
	Select();
	Show();
	
	mResultsTable->IncrementVerifiedCount();

	UpdatePort();
}

	void
CResultsWindow::IncrementVerifiedCount(void)
{
	gResultsWindow->IncrementCount();
}

	void
CResultsWindow::CalcColumnBounds(
	const Rect	*cellRect,
	Rect		*iconRect,
	Rect		*nameRect,
	Rect		*signerRect,
	Rect		*validityRect,
	Rect		*signingDateRect)
{
	pgpAssertAddrValid( cellRect, Rect );
	pgpAssertAddrValid( iconRect, Rect );
	pgpAssertAddrValid( nameRect, Rect );
	pgpAssertAddrValid( signerRect, Rect );
	pgpAssertAddrValid( validityRect, Rect );
	pgpAssertAddrValid( signingDateRect, Rect );
	
	// Assumes we're using a 16 pixel wide icon
	*iconRect 		= *cellRect;
	iconRect->left 	+= kIconColumnLeftMargin;
	iconRect->right	= iconRect->left + 16;

	mNameColumnButton->CalcPortFrameRect( *nameRect );
	
	nameRect->top		= cellRect->top;
	nameRect->bottom	= cellRect->bottom;
	nameRect->left 		+= kNameColumnLeftMargin;
	nameRect->right 	-= kNameColumnRightMargin;

	mSignerColumnButton->CalcPortFrameRect( *signerRect );
	
	signerRect->top		= cellRect->top;
	signerRect->bottom	= cellRect->bottom;
	signerRect->left 	+= kSignerColumnLeftMargin;
	signerRect->right 	-= kSignerColumnRightMargin;
	
	mValidityColumnButton->CalcPortFrameRect( *validityRect );
	
	validityRect->top		= cellRect->top;
	validityRect->bottom	= cellRect->bottom;
	validityRect->left 		+= kValidityColumnLeftMargin;
	validityRect->right 	-= kValidityColumnRightMargin;

	mDateColumnButton->CalcPortFrameRect( *signingDateRect );
	
	signingDateRect->top	= cellRect->top;
	signingDateRect->bottom	= cellRect->bottom;
	signingDateRect->left 	+= kDateColumnLeftMargin;
	signingDateRect->right 	-= kDateColumnRightMargin;
}

	void
CResultsWindow::FindCommandStatus(
	CommandT	inCommand,
	Boolean		&outEnabled,
	Boolean		&outUsesMark,
	Char16		&outMark,
	Str255		outName)
{
	switch (inCommand)
	{
		case cmd_Clear:
		{
			TableIndexT	numRows;
			TableIndexT	numColumns;
			
			mResultsTable->GetTableSize( numRows, numColumns );
			outEnabled = ( numRows > 0 );
			break;
		}
			
		default:
			CPGPtoolsWindow::FindCommandStatus(inCommand, outEnabled,
												outUsesMark, outMark, outName);
			break;
	}
}

	Boolean
CResultsWindow::ObeyCommand(
	CommandT	inCommand,
	void		*ioParam)
{
	Boolean		cmdHandled = true;

	switch (inCommand)
	{
		case cmd_Clear:
		{
			TableIndexT	numRows;
			TableIndexT	numColumns;
			
			do
			{
				mResultsTable->GetTableSize( numRows, numColumns );
				if( numRows > 0 )
				{
					mResultsTable->RemoveRows( 1, 1, FALSE );
				}
			} while( numRows != 0 );
			
			mResultsTable->Refresh();
			
			break;
		}
			
		default:
			cmdHandled = CPGPtoolsWindow::ObeyCommand(inCommand, ioParam);
			break;
	}
	
	return cmdHandled;
}

