/////////////////////////////////////////////////////////////////////
////                                                             ////
////  I2F                                                        ////
////  Integer to Floating Point (single precision)               ////
////                                                             ////
////                                                             ////
////  Author: Rudolf Usselmann                                   ////
////          russelmann@hotmail.com                             ////
////                                                             ////
/////////////////////////////////////////////////////////////////////
////                                                             ////
//// Copyright (C) 2000 Rudolf Usselmann                         ////
////                    russelmann@hotmail.com                   ////
////                                                             ////
//// This source file may be used and distributed without        ////
//// restriction provided that this copyright statement is not   ////
//// removed from the file and that any derivative work contains ////
//// the original copyright notice and the associated disclaimer.////
////                                                             ////
//// THIS SOURCE FILE IS PROVIDED "AS IS" AND WITHOUT ANY        ////
//// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, WITHOUT           ////
//// LIMITATION, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND   ////
//// FITNESS FOR A PARTICULAR PURPOSE.                           ////
////                                                             ////
/////////////////////////////////////////////////////////////////////


`timescale 1ns / 10ps

module i2f( int, fp);
input	[31:0]	int;
output	[31:0]	fp;

////////////////////////////////////////////////////////////////////////
//
// Local Wires and registers
//

wire		sign;
reg	[22:0]	fract_out;
wire	[31:0]	fract_in;
wire	[7:0]	exp_out;
wire	[7:0]	exp_in;
reg	[5:0]	exp_adj;
wire	[30:0]	twos_compl;

////////////////////////////////////////////////////////////////////////
//
// Convert Interger to Floating Point
//

assign sign     = int[31];		// Sign
assign exp_in   = 8'h80 + 30;		// BIAS + number of places to move the period
assign fract_in = int[31] ? twos_compl : int;
assign twos_compl  = ~int + 1;		// For negative numbers

////////////////////////////////////////////////////////////////////////
//
// Normalize Result
//

// Count Number of zeros in fract_in from msb to lsb and adjust
// output so there are no leading zeros

always @(fract_in)
   casex(fract_in)	// synopsys full_case parallel_case
	32'b1???????????????????????????????: fract_out = fract_in[30:8];
	32'b01??????????????????????????????: fract_out = fract_in[29:7];
	32'b001?????????????????????????????: fract_out = fract_in[28:6];
	32'b0001????????????????????????????: fract_out = fract_in[27:5];
	32'b00001???????????????????????????: fract_out = fract_in[26:4];
	32'b000001??????????????????????????: fract_out = fract_in[25:3];
	32'b0000001?????????????????????????: fract_out = fract_in[24:2];
	32'b00000001????????????????????????: fract_out = fract_in[23:1];
	32'b000000001???????????????????????: fract_out = fract_in[22:0];
	32'b0000000001??????????????????????: fract_out = { fract_in[21:0], 1'h0 };
	32'b00000000001?????????????????????: fract_out = { fract_in[20:0], 2'h0 };
	32'b000000000001????????????????????: fract_out = { fract_in[19:0], 3'h0 };
	32'b0000000000001???????????????????: fract_out = { fract_in[18:0], 4'h0 };
	32'b00000000000001??????????????????: fract_out = { fract_in[17:0], 5'h0 };
	32'b000000000000001?????????????????: fract_out = { fract_in[16:0], 6'h0 };
	32'b0000000000000001????????????????: fract_out = { fract_in[15:0], 7'h0 };
	32'b00000000000000001???????????????: fract_out = { fract_in[14:0], 8'h0 };
	32'b000000000000000001??????????????: fract_out = { fract_in[13:0], 9'h0 };
	32'b0000000000000000001?????????????: fract_out = { fract_in[12:0], 10'h0 };
	32'b00000000000000000001????????????: fract_out = { fract_in[11:0], 11'h0 };
	32'b000000000000000000001???????????: fract_out = { fract_in[10:0], 12'h0 };
	32'b0000000000000000000001??????????: fract_out = { fract_in[9:0], 13'h0 };
	32'b00000000000000000000001?????????: fract_out = { fract_in[8:0], 14'h0 };
	32'b000000000000000000000001????????: fract_out = { fract_in[7:0], 15'h0 };
	32'b0000000000000000000000001???????: fract_out = { fract_in[6:0], 16'h0 };
	32'b00000000000000000000000001??????: fract_out = { fract_in[5:0], 17'h0 };
	32'b000000000000000000000000001?????: fract_out = { fract_in[4:0], 18'h0 };
	32'b0000000000000000000000000001????: fract_out = { fract_in[3:0], 19'h0 };
	32'b00000000000000000000000000001???: fract_out = { fract_in[2:0], 20'h0 };
	32'b000000000000000000000000000001??: fract_out = { fract_in[1:0], 21'h0 };
	32'b0000000000000000000000000000001?: fract_out = { fract_in[000], 22'h0 };
	32'b0000000000000000000000000000000?: fract_out = 23'h0;
   endcase

// Determine how much to add to exp_out
always @(fract_in)
   casex(fract_in)	// synopsys full_case parallel_case
	32'b1???????????????????????????????: exp_adj =   0;
	32'b01??????????????????????????????: exp_adj =  -1;
	32'b001?????????????????????????????: exp_adj =  -2;
	32'b0001????????????????????????????: exp_adj =  -3;
	32'b00001???????????????????????????: exp_adj =  -4;
	32'b000001??????????????????????????: exp_adj =  -5;
	32'b0000001?????????????????????????: exp_adj =  -6;
	32'b00000001????????????????????????: exp_adj =  -7;
	32'b000000001???????????????????????: exp_adj =  -8;
	32'b0000000001??????????????????????: exp_adj =  -9;
	32'b00000000001?????????????????????: exp_adj =  -10;
	32'b000000000001????????????????????: exp_adj =  -11;
	32'b0000000000001???????????????????: exp_adj =  -12;
	32'b00000000000001??????????????????: exp_adj =  -13;
	32'b000000000000001?????????????????: exp_adj =  -14;
	32'b0000000000000001????????????????: exp_adj =  -15;
	32'b00000000000000001???????????????: exp_adj =  -16;
	32'b000000000000000001??????????????: exp_adj =  -17;
	32'b0000000000000000001?????????????: exp_adj =  -18;
	32'b00000000000000000001????????????: exp_adj =  -19;
	32'b000000000000000000001???????????: exp_adj =  -20;
	32'b0000000000000000000001??????????: exp_adj =  -21;
	32'b00000000000000000000001?????????: exp_adj =  -22;
	32'b000000000000000000000001????????: exp_adj =  -23;
	32'b0000000000000000000000001???????: exp_adj =  -24;
	32'b00000000000000000000000001??????: exp_adj =  -25;
	32'b000000000000000000000000001?????: exp_adj =  -26;
	32'b0000000000000000000000000001????: exp_adj =  -27;
	32'b00000000000000000000000000001???: exp_adj =  -28;
	32'b000000000000000000000000000001??: exp_adj =  -29;
	32'b0000000000000000000000000000001?: exp_adj =  -30;
	32'b00000000000000000000000000000001: exp_adj =  -31;
	32'b00000000000000000000000000000000: exp_adj =    0;
   endcase

assign exp_out = exp_in + {exp_adj[5], exp_adj[5], exp_adj};
assign fp = {sign, exp_out[7:0], fract_out};

endmodule

